// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @author Phil Bradley


#ifndef INCLUDED_core_conformation_util_hh
#define INCLUDED_core_conformation_util_hh


// Unit headers

// Project headers
#include <core/types.hh>
#include <core/conformation/Residue.fwd.hh>
#include <core/conformation/Conformation.fwd.hh>

#include <core/chemical/ResidueConnection.fwd.hh>
#include <core/chemical/VariantType.fwd.hh>
#include <core/chemical/ResidueType.fwd.hh>

#include <core/id/AtomID_Map.fwd.hh>
#include <core/id/NamedAtomID.fwd.hh>
#include <core/id/AtomID.fwd.hh>
#include <core/id/NamedStubID.fwd.hh>
#include <core/kinematics/tree/Atom.fwd.hh>
#include <core/kinematics/AtomTree.fwd.hh>
#include <core/kinematics/FoldTree.fwd.hh>
#include <core/kinematics/AtomPointer.fwd.hh>
#include <core/kinematics/Edge.fwd.hh>

// Utility headers

// C++ headers
#include <iosfwd>

namespace core {
namespace conformation {

void
orient_residue_for_ideal_bond(
	Residue & moving_rsd,
	chemical::ResidueConnection const & moving_connection,
	Residue const & fixed_rsd,
	chemical::ResidueConnection const & fixed_connection,
	Conformation const & conformation // unused
);

/// @brief  Sets the two bond angles and the bond length across the junction, rebuilds dependent atoms (eg O,H)
void
insert_ideal_bonds_at_polymer_junction(
	Size const seqpos,
	Conformation & conformation
);

void
insert_ideal_mainchain_bonds(
	Size const seqpos,
	Conformation & conformation
);


/// @brief  Idealize backbone and sidechain at seqpos
void
idealize_position(
	Size const seqpos,
	Conformation & conformation
);


/// @brief  Return true if position contains an ideal geometry up to some epsilon
///
/// @params seqpos - sequence position
/// @params conformation - conformation object
/// @params theta_epsilon - permitted deviation from ideal bond angles, in Radians
/// @params D_epsilon - permitted deviation from ideal bond length
///
/// @remarks conformation is needed for context of polymer nbrs
bool
is_ideal_position( // Barak 6/30/09
	Size const seqpos,
	Conformation const& conformation,
 	Real theta_epsilon = 0.005, // ~0.29 degrees
	Real D_epsilon = 0.02
);


/// @brief  Fills coords of target_rsd with coords from source_rsd of same atom_name, rebuilds others.
void
copy_residue_coordinates_and_rebuild_missing_atoms(
	Residue const & source_rsd,
	Residue & target_rsd,
	Conformation const & conf
);


///
void
show_atom_tree( kinematics::tree::Atom const & atom, Conformation const & conf, std::ostream & os );

void
replace_conformation_residue_copying_existing_coordinates(
	conformation::Conformation & conformation,
	Size const seqpos,
	chemical::ResidueType const & new_rsd_type
);


void
add_variant_type_to_conformation_residue(
	conformation::Conformation & conformation,
	chemical::VariantType const & variant_type,
	Size const seqpos
);


void
remove_variant_type_from_conformation_residue(
	conformation::Conformation & conformation,
	chemical::VariantType const & variant_type,
	Size const seqpos
);


void
add_lower_terminus_type_to_conformation_residue(
	conformation::Conformation & conformation,
	Size const seqpos
);


void
add_upper_terminus_type_to_conformation_residue(
	conformation::Conformation & conformation,
	Size const seqpos
);


void
remove_lower_terminus_type_from_conformation_residue(
	conformation::Conformation & conformation,
	Size const seqpos
);


void
remove_upper_terminus_type_from_conformation_residue(
	conformation::Conformation & conformation,
	Size const seqpos
);

void
build_tree(
    kinematics::FoldTree const & fold_tree,
	conformation::ResidueCAPs const & residues,
	kinematics::AtomPointer2D & atom_pointer
	);

/// @brief build a sub atom-tree for a jump edge and attach it to main atom-tree
void
build_jump_edge(
	kinematics::Edge const & edge,
	conformation::ResidueCAPs const & residues,
	kinematics::AtomPointer2D & atom_pointer
	);

/// @brief build a sub atom-tree for a polymer edge and attach it to main atom-tree
void
build_polymer_edge(
	kinematics::Edge const & edge,
	conformation::ResidueCAPs const & residues,
	kinematics::AtomPointer2D & atom_pointer
	);

/// @brief build a sub atom-tree for a chemical edge and attach it to main atom-tree
void
build_chemical_edge(
	kinematics::Edge const & edge,
	conformation::ResidueCAPs const & residues,
	kinematics::AtomPointer2D & atom_pointer
	);


/// @brief  build the tree of atoms for this residue, anchored at root_atomno
void
build_residue_tree(
	int const root_atomno,
	conformation::Residue const & rsd,
	kinematics::AtomPointer1D & atom_ptr,
	bool const root_atom_is_jump_atom
	//bool const keep_1st_child_position = false
	);

/// @brief  build_residue_tree function that uses the foldtree info
/// @brief  also used in build_tree to build the residue tree for the root residue
void
build_residue_tree(
	conformation::ResidueCAPs const & residues,
	conformation::Residue const & rsd,
	kinematics::FoldTree const & fold_tree,
	kinematics::AtomPointer1D & atom_ptr
	);

/// @brief  Helper function for conformation routines
void
replace_residue_in_atom_tree(
	conformation::Residue const & new_rsd,
	kinematics::FoldTree const & fold_tree,
	conformation::ResidueCAPs const & residues,
	kinematics::AtomTree & atom_tree
	);

/// @brief  Inserts/ appends new residue subtree into an existing atomtree
/// @note  The foldtree must already have been changed to reflect the new residue
/// @note  The sequence position of the new residue is deduced from new_rsd.seqpos()
/// @note  This function handles renumbering of the atomtree if necessary
void
insert_residue_into_atom_tree(
	conformation::Residue const & new_rsd,
	kinematics::FoldTree const & fold_tree,
	conformation::ResidueCAPs const & residues,
	kinematics::AtomTree & atom_tree
	);

int
get_root_atomno(
	conformation::Residue const & rsd,
	int const dir // +1, -1, or "dir_jump"
	);

/// @brief  Get the atom-index of the atom to which the residue at position seqpos should be anchored.
int
get_anchor_atomno(
	conformation::Residue const & anchor_rsd,
	Size const seqpos,
	kinematics::FoldTree const & fold_tree
	);


/// @brief get anchor atom to which the atom-tree of next residue in the edge is attached.
int
get_anchor_atomno(
	conformation::Residue const & rsd,
	int const dir // forward(1), backward(-1), or "dir_jump"
	);

/// @brief  Use this routine to deduce atom indices of connect atoms in the tree
void
get_anchor_and_root_atoms(
	conformation::Residue const & anchor_rsd,
	conformation::Residue const & root_rsd,
	kinematics::Edge const & edge,
	Size & anchor_atomno,
	Size & root_atomno
	);

/// @brief  Moves the first same-residue child of the jump atom corresponding to edge into first place in the child list
void
promote_sameresidue_child_of_jump_atom(
	kinematics::Edge const & edge,
	conformation::ResidueCAPs const & residues,
	kinematics::AtomPointer2D const & atom_pointer
	);


/// @brief  Moves the first same-residue child of the jump atom corresponding to edge into first place in the child list
void
promote_sameresidue_child_of_jump_atom(
	kinematics::Edge const & edge,
	conformation::ResidueCAPs const & residues,
	kinematics::AtomTree & atom_tree
	);

void
get_chemical_root_and_anchor_atomnos(
	conformation::Residue const & rsd_anchor,
	conformation::Residue const & rsd_root,
	Size & anchor_atom_no,
	Size & root_atom_no
	);

/// @brief set up a map to match mainchain atoms from residue1 to residue2
void
setup_corresponding_atoms(
	id::AtomID_Map< id::AtomID > & atom_map,
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2
);

/// @brief Replace a CYS with a CYD or vice-versa for changing disulfide bonds.
/// @param[in] index Position of the residue to replace.
/// @param[in] cys_type_name3 The 3-letter name of the cys type to use: either CYS
///  or CYD.
/// @param[inout] conf The conformation to modify
/// @return true if the replacement was successful, false otherwise.
bool change_cys_state( Size const index, std::string cys_type_name3, core::conformation::Conformation & conf );

/// @brief Find whether there is a disulfide defined between two residues
bool is_disulfide_bond( core::conformation::Conformation const& conformation, core::Size residueA_pos, core::Size residueB_pos);

/// @brief Generate a list of all disulfide bonds in the conformation
void disulfide_bonds( core::conformation::Conformation const& conformation, utility::vector1< std::pair<core::Size,core::Size> > & disulfides );

/// @brief Introduce cysteines at the specified location and define a disulfide bond between them.
/// @details Does not do the repacking & minimization required to place the disulfide correctly.
void form_disulfide(core::conformation::Conformation & conformation, core::Size lower_res, core::Size upper_res);

id::NamedAtomID
atom_id_to_named_atom_id(
	id::AtomID const & atom_id,
	conformation::Residue const & rsd
);

/* - undefinded, commenting out to make PyRosetta compile...
id::AtomID
named_atom_id_to_atom_id(
	id::AtomID const & atom_id,
	conformation::Residue const & rsd
);
*/

id::AtomID
named_atom_id_to_atom_id(
	id::NamedAtomID const & atom_id,
	conformation::Residue const & rsd
);

id::NamedStubID
stub_id_to_named_stub_id(
	id::StubID const & stub_id,
	conformation::Residue const & rsd
);

/* - undefinded, commenting out to make PyRosetta compile...
id::StubID
named_stub_id_to_named_stub_id(
	id::NamedStubID const & named_stub_id,
	conformation::Residue const & rsd
);
*/
} // conformation
} // core



#endif
