// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file docking_initialization_protocols
/// @brief initialization protocols for docking
/// @detailed
///		This contains the functions that create initial positions for docking
///		You can either randomize partner 1 or partner 2, spin partner 2, or
///		perform a simple perturbation.
/// @author Monica Berrondo


#ifndef INCLUDED_protocols_docking_DockingInitialPerturbation_hh
#define INCLUDED_protocols_docking_DockingInitialPerturbation_hh

#include <protocols/docking/types.hh>
#include <protocols/docking/DockingInitialPerturbation.fwd.hh>

// Package headers
#include <core/pose/Pose.fwd.hh>

#include <core/scoring/ScoreFunction.fwd.hh>

#include <protocols/moves/Mover.hh>

#include <utility/pointer/owning_ptr.hh>
#include <utility/vector1.hh>
#include <string>

namespace protocols {
namespace docking {

/// @brief this mover carries out the initial perturbation phase of the RosettaDock algorithm
/// based on user-inputted command line options
class DockingInitialPerturbation : public moves::Mover
{
public:

	/// @brief Default constructor
	DockingInitialPerturbation();

	/// @brief Constructor with two arguments. The first is the jump number to dock over, the second is a boolean (true
	///		will use slide into contact, false will not).
	DockingInitialPerturbation(
		core::Size const rb_jump,
		bool const slide=true
	);

	/// @brief Constructor with two arguments. The first is the DockJumps, the second is a boolean (true
	///		will use slide into contact, false will not).
	DockingInitialPerturbation(
		DockJumps const movable_jumps,
		bool const slide=true
	);

	//destructor
	~DockingInitialPerturbation();

	// protocol functions
	virtual void apply( core::pose::Pose & pose );
	void apply_body(core::pose::Pose & pose, core::Size jump_number );
	virtual std::string get_name() const;

	/// @brief Calls set_dault, register_from_options and init_from_options
	void init();

	/// @brief Sets members to default values
	void set_default();

	/// @brief Associates relevant options with the DockingInitialPerturbation class
	void register_options();

	/// @brief Assigns user specified values to primitive members using command line options
	void init_from_options();

	/// @brief set functions
	void set_randomize1(bool randomize1){ randomize1_ = randomize1; }
	void set_randomize2(bool randomize2){ randomize2_ = randomize2; }
	void set_dock_pert(utility::vector1< core::Real > dock_pert){
		dock_pert_ = dock_pert;
		if_dock_pert_ = true;
	}
	void set_uniform_trans(core::Real uniform_trans){
		uniform_trans_ = uniform_trans;
		if_uniform_trans_ = true;
	}
	void set_spin( bool spin){ spin_ = spin;}
	void set_center( bool center) { center_at_interface_ = center;}

private:
	/// do slide into context?
	bool slide_;

	// docking
	DockJumps movable_jumps_;

	bool randomize1_;
	bool randomize2_;
	bool spin_;
	bool if_dock_pert_;
	bool if_uniform_trans_;
	bool multiple_jumps_;
	bool center_at_interface_;

	utility::vector1< core::Real > dock_pert_;
	core::Real uniform_trans_;
};

/// @brief Contrary to the name, slides things apart first, then together.
/// OK for proteins, bad for ligands (because they may escape the pocket permanently).
class DockingSlideIntoContact : public moves::Mover
{
public:
	DockingSlideIntoContact();

	// constructor with arguments
	DockingSlideIntoContact( core::Size const rb_jump );

	//destructor
	~DockingSlideIntoContact();

	// protocol functions
	virtual void apply( core::pose::Pose & pose );
	virtual std::string get_name() const;

private:
	core::scoring::ScoreFunctionOP scorefxn_;

	/// which jump to use for docking
	core::Size rb_jump_;
};

/// @brief Slides docking partners together by monitoring fa_rep.
/// @details
///		If partners are already touching, no change is made.
///		Separation will be 1A or less after calling this function.
class FaDockingSlideIntoContact : public moves::Mover
{
public:
	FaDockingSlideIntoContact();
	FaDockingSlideIntoContact( core::Size const rb_jump);

	//destructor
	~FaDockingSlideIntoContact();

	virtual void apply( core::pose::Pose & pose );
	virtual std::string get_name() const;

private:
	core::Size rb_jump_;
	core::scoring::ScoreFunctionOP scorefxn_;
	core::Real tolerance_; ///< how accurate do you want to be?

};

} // docking
} // protocols

#endif
