// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/fldsgn/BluePrintBDR.cc
/// @brief
/// @author Nobuyasu Koga ( nobuyasu@uw.edu ),  Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <protocols/jd2/parser/BluePrint.hh>
#include <protocols/fldsgn/BluePrintBDR.hh>
#include <protocols/fldsgn/BluePrintBDRCreator.hh>

// package headers
#include <protocols/forge/build/BuildInstruction.hh>
#include <protocols/forge/build/BuildManager.hh>
#include <protocols/forge/build/SegmentInsert.hh>
#include <protocols/forge/components/VarLengthBuild.hh>
// AUTO-REMOVED #include <protocols/forge/methods/chainbreak_eval.hh>
#include <protocols/forge/methods/pose_mod.hh>
// AUTO-REMOVED #include <protocols/forge/methods/util.hh>
// AUTO-REMOVED #include <protocols/fldsgn/topology/HSSTriplet.hh> // REQUIRED FOR WINDOWS

// project headers
#include <basic/Tracer.hh>
#include <core/chemical/AA.hh>
#include <core/chemical/ResidueTypeSet.hh>
// AUTO-REMOVED #include <core/chemical/util.hh>
#include <core/import_pose/import_pose.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/pose/Pose.hh>
#include <core/pose/symmetry/util.hh>
#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/dssp/Dssp.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/util/SwitchResidueTypeSet.hh>

#include <protocols/forge/constraints/NtoC_RCG.hh>
#include <protocols/forge/constraints/ConstraintFileRCG.hh>
#include <protocols/fldsgn/SheetConstraintsRCG.hh>
#include <protocols/forge/remodel/RemodelConstraintGenerator.hh>
// AUTO-REMOVED #include <protocols/forge/remodel/RemodelLoopMover.hh>
#include <protocols/forge/build/SegmentRebuild.hh>

#include <protocols/moves/DataMap.hh>
#include <protocols/toolbox/pose_manipulation.hh>
#include <utility/tag/Tag.hh>

// C++ headers
#include <utility>

// boost
// AUTO-REMOVED #include <boost/lexical_cast.hpp>

#include <utility/vector0.hh>
#include <utility/vector1.hh>

namespace protocols {
namespace fldsgn {


static basic::Tracer TR( "protocols.fldsgn.BluePrintBDR" );

std::string
BluePrintBDRCreator::keyname() const
{
	return BluePrintBDRCreator::mover_name();
}

protocols::moves::MoverOP
BluePrintBDRCreator::create_mover() const {
	return new BluePrintBDR;
}

std::string
BluePrintBDRCreator::mover_name()
{
	return "BluePrintBDR";
}


/// @brief default constructor
BluePrintBDR::BluePrintBDR() :
	Super( "BluePrintBDR" ),
	blueprint_( NULL ),
	sfx_( core::scoring::ScoreFunctionFactory::create_score_function( "fldsgn_cen" ) ),
	loop_mover_str_( "RemodelLoopMover" ),
	use_fullmer_( false ),
	num_fragpick_( 200 ),
	use_sequence_bias_( false ),
	use_abego_bias_( false ),
	max_linear_chainbreak_( 0.07 ),
	initialized_( false ),
	ss_from_blueprint_( true ),
	constraints_NtoC_( -1.0 ),
	constraints_sheet_( -1.0 ),
	constraint_file_( "" ),
	dump_pdb_when_fail_( "" ),
	rmdl_attempts_( 1 ),
	use_poly_val_( true )
{}

/// @brief value constructor
BluePrintBDR::BluePrintBDR( String const & filename, bool const ss_from_blueprint ) :
	Super( "BluePrintBDR" ),
	sfx_( core::scoring::ScoreFunctionFactory::create_score_function( "fldsgn_cen" ) ),
	loop_mover_str_( "RemodelLoopMover" ),
	use_fullmer_( false ),
	num_fragpick_( 200 ),
	use_sequence_bias_( false ),
	use_abego_bias_( false ),
	max_linear_chainbreak_( 0.07 ),
	initialized_( false ),
	ss_from_blueprint_( ss_from_blueprint ),
	constraints_NtoC_( -1.0 ),
	constraints_sheet_( -1.0 ),
	constraint_file_( "" ),
	dump_pdb_when_fail_( "" ),
	rmdl_attempts_( 1 ),
	use_poly_val_( true )
{
	set_blueprint( filename );
}

/// @brief value constructor
BluePrintBDR::BluePrintBDR( BluePrintOP const & blueprintOP, bool const ss_from_blueprint ) :
	Super( "BluePrintBDR" ),
	blueprint_( blueprintOP ),
	sfx_( core::scoring::ScoreFunctionFactory::create_score_function( "fldsgn_cen" ) ),
	loop_mover_str_( "RemodelLoopMover" ),
	use_fullmer_( false ),
	num_fragpick_( 200 ),
	use_sequence_bias_( false ),
	use_abego_bias_( false ),
	max_linear_chainbreak_( 0.07 ),
	initialized_( false ),
	ss_from_blueprint_( ss_from_blueprint ),
	constraints_NtoC_( -1.0 ),
	constraints_sheet_( -1.0 ),
	constraint_file_( "" ),
	dump_pdb_when_fail_( "" ),
	rmdl_attempts_( 1 ),
	use_poly_val_( true )
{}

/// @Brief copy constructor
BluePrintBDR::BluePrintBDR( BluePrintBDR const & rval ) :
	//utility::pointer::ReferenceCount(),
	Super( rval ),
	blueprint_( rval.blueprint_ ),
	manager_( rval.manager_ ),
	sfx_( rval.sfx_ ),
	loop_mover_str_( rval.loop_mover_str_ ),
	use_fullmer_( rval.use_fullmer_ ),
	num_fragpick_( rval.num_fragpick_ ),
	use_sequence_bias_( rval.use_sequence_bias_ ),
	use_abego_bias_( rval.use_abego_bias_ ),
	max_linear_chainbreak_( rval.max_linear_chainbreak_ ),
	initialized_( rval.initialized_ ),
	ss_from_blueprint_( rval.ss_from_blueprint_ ),
	constraints_NtoC_( rval.constraints_NtoC_ ),
	constraints_sheet_( rval.constraints_sheet_ ),
	constraint_file_( rval.constraint_file_ ),
	dump_pdb_when_fail_( rval.dump_pdb_when_fail_ ),
	rmdl_attempts_( rval.rmdl_attempts_ ),
	use_poly_val_( rval.use_poly_val_ )
{
	if ( rval.vlb_.get() ) {
		vlb_ = new VarLengthBuild( *rval.vlb_ );
	}
}

/// @brief default destructor
BluePrintBDR::~BluePrintBDR() {}

/// @brief clone this object
BluePrintBDR::MoverOP
BluePrintBDR::clone() const
{
	return new BluePrintBDR( *this );
}

/// @brief create this type of object
BluePrintBDR::MoverOP
BluePrintBDR::fresh_instance() const
{
	return new BluePrintBDR();
}

/// @brief the centroid level score function, default "remodel_cen"
BluePrintBDR::ScoreFunction const &
BluePrintBDR::scorefunction() const
{
	return *sfx_;
}

/// @brief add instruction to the manager of this BluePrintBDR (no copy)
/// @param[in] bi BuildInstruction
void
BluePrintBDR::add_instruction( BuildInstructionOP bi )
{
	manager_.add( bi );

	// additional instruction means we'll need a new re-init the VLB, so
	// go ahead and drop the existing one
	vlb_ = 0;
}


/// @brief create directed dependency between two instructions
void
BluePrintBDR::create_directed_dependency(
	BuildInstructionOP u,
	BuildInstructionOP v
)
{
	manager_.create_directed_dependency( u, v );
}


/// @brief set the centroid level score function
void
BluePrintBDR::scorefunction( ScoreFunction const & sfx )
{
	sfx_ = sfx.clone();
}

/// @brief set the centroid level score function
void
BluePrintBDR::scorefunction( ScoreFunctionOP sfx )
{
	sfx_ = sfx->clone();
}

/// @brief use blueprint
void
BluePrintBDR::set_blueprint( String const & filename )
{
	blueprint_ = new BluePrint( filename );
}

/// @brief use blueprint
void
BluePrintBDR::set_blueprint( BluePrintOP const & blp )
{
	blueprint_ = blp;
}

/// @brief set constraint between N- and C- terminal residues, need to set the weight
void
BluePrintBDR::set_constraints_NtoC( Real const & weight )
{
	constraints_NtoC_ = weight;
}

/// @brief set constraint file
void
BluePrintBDR::set_constraint_file( String const & constraint_file )
{
	constraint_file_ = constraint_file;
}

/// @brief dump pdb when this protocol failed
void
BluePrintBDR::dump_pdb_when_fail( String const & dump_pdb_when_fail )
{
	dump_pdb_when_fail_ = dump_pdb_when_fail;
}

/// @brief set instruction by blueprint
bool
BluePrintBDR::set_instruction_blueprint( Pose const & pose )
{
	using protocols::forge::build::Interval;
	using protocols::forge::build::SegmentRebuild;
	using protocols::forge::build::SegmentInsert;

	bool flag( false ), insert( false );
	String aa, ss, insert_name, ins_sec;
	Size left( 0 ), right( 0 ), count( 0 ), insnum( 0 );
	Pose insert_pose;
	for( Size i=1; i<=blueprint_->total_residue(); i++ ){

		if( blueprint_->resnum( i ) != 0 ){
			count++;
			if( count > pose.total_residue() ){
				TR.Error << "Residue number in blueprint file is more than that of pose!,  pose/blueprint= "
								 << pose.total_residue() << "/" << count << std::endl;
				return false;
			}
		}

		//if( blueprint_->buildtype( i ) != '.' && !flag ) {  // found R or X at first
		if( blueprint_->buildtype( i ) == 'R' && blueprint_->buildtype( i ) != 'I' && !flag ) {  // found R at first

			if( count == 0 || i==1 ){ // N-terminal extensione
				left = 1;
			}else{
				left = blueprint_->resnum( i-1 )+1; // insert in the middle of sequence
				if( left > pose.total_residue() ){  // C-terminal extension
					left = pose.total_residue();
				}
			}
			flag = true;

			// } else if( blueprint_->buildtype( i ) == '.' && flag ){ // add instruction
		} else if( blueprint_->buildtype( i ) != 'R' && blueprint_->buildtype( i ) != 'I' && flag ){ // add instruction

			right = blueprint_->resnum( i )-1;
			if ( right == 0 ) { // at the N-termianl
				right = 1;
			} else if( right < left ){ // insert residues into the successive residue numbers
				right = left;
			}
			runtime_assert( right <= pose.total_residue() );

			if( insert ) {
				if( ins_sec.length() == 1 ) {
					TR << "Secondary structure of insert pose will be given by Dssp" << std::endl;
					Dssp dssp( pose );
					dssp.insert_ss_into_pose( insert_pose );
				} else {
					runtime_assert( insert_pose.total_residue() == ins_sec.length() );
					for( Size j=1; j<=insert_pose.total_residue(); j++ ) {
						insert_pose.set_secstruct( j, ins_sec[ j-1 ] );
					}
				}
				TR << "SegmentInsert left " << left	<< ", right: " << right
					 << ", ss: " << ss << ", aa:" << aa << ", pdb:" << insert_name << std::endl;
				add_instruction( new SegmentInsert( Interval( left, right ), ss, aa, insert_pose ) );
			} else {
				TR << "SegmentRebuild left: " << left << ", right: " << right << ", ss: " << ss << ", aa:" << aa << std::endl;
				add_instruction( new SegmentRebuild( Interval( left, right ), ss, aa ) );
			}

			flag = false;
			insert = false;
			aa = "";
			ss = "";

		} //blurprint_->buildtype()

		if( flag ){
			if( blueprint_->buildtype( i ) == 'I' && !insert ) {
				insert = true;
				insnum ++;
				ins_sec = blueprint_->secstruct( i );
				insert_name = blueprint_->insertion( insnum );
				core::import_pose::pose_from_pdb( insert_pose, insert_name );
				aa += '^';
				ss += '^';
			} else if( blueprint_->buildtype( i ) == 'I' ) {
				ins_sec += blueprint_->secstruct( i );
			} else {
				aa += blueprint_->sequence( i );
				ss += blueprint_->secstruct( i );
			}
		} // flag

	} // blueprint_->total_residue()

	if( flag ){
		if( blueprint_->resnum( blueprint_->total_residue() ) == 0 ) {
			right = pose.total_residue();
		}else{
			right = blueprint_->resnum( blueprint_->total_residue() );
			runtime_assert( right <= pose.total_residue() );
		}
		add_instruction( new SegmentRebuild( Interval( left, right ), ss, aa ) );
		TR << "SegmentRebuild left: " << left << ", right: " << right << ", ss: " << ss << ", aa:" << aa << std::endl;
	}

	if( instruction_size() < 1 ) {
		TR << "There is no instruction in blueprint. " << std::endl;
		return false;
	}

	return true;
} // set_build_instruction


/// @brief apply defined moves to given Pose
void
BluePrintBDR::apply( Pose & pose )
{
	using protocols::moves::MS_SUCCESS;
	using protocols::moves::FAIL_DO_NOT_RETRY;
	using protocols::moves::FAIL_BAD_INPUT;
	using protocols::moves::FAIL_RETRY;

	//set instruction by blueprint file and store secondary structure information
	if( !blueprint_ && !initialized_ ){
		TR << "You need to set a blueprint file" << std::endl;
		set_last_move_status( FAIL_BAD_INPUT );
		return;
	}else if( !initialized_ ){
		if( ! set_instruction_blueprint( pose ) ){
			set_last_move_status( FAIL_BAD_INPUT );
			return;
		}
		// assign secondary structure   ... probably, this is bug, we have to set SS info everytime
		// if( !ss_from_blueprint_ ){
		// Dssp dssp( pose );
		// dssp.insert_ss_into_pose( pose );
		//		}else{
		// insert_ss_into_pose( pose );
		// }
		//initial_pose_ = new Pose( pose );
		initialized_ = true;
	}

	// assign secondary structure
	if( !ss_from_blueprint_ ){
		Dssp dssp( pose );
		dssp.insert_ss_into_pose( pose );
	}else{
		blueprint_->insert_ss_into_pose( pose );
	}

	// do centroid build
	if ( !centroid_build( pose ) ) { // build failed
		set_last_move_status( FAIL_RETRY );

		// dump pdb when failed
		if( dump_pdb_when_fail_ != "" ){
			pose.dump_pdb( dump_pdb_when_fail_ );
		}
		return;
	}

	// if we've gotten to this point, then the structure has been built properly
	set_last_move_status( MS_SUCCESS );
	// output score
	// sfx_->show( TR, pose );

}


std::string
BluePrintBDR::get_name() const {
	return BluePrintBDRCreator::mover_name();
}

/// @brief run the centroid level build stage
/// @return true if loop closed, false otherwise
bool BluePrintBDR::centroid_build(
	Pose & pose
)
{
	using core::scoring::STANDARD_WTS;
	using core::scoring::SCORE12_PATCH;
	using core::scoring::ScoreFunctionOP;
	using core::scoring::ScoreFunctionFactory;
	using protocols::moves::MS_SUCCESS;

	using core::util::switch_to_residue_type_set;
	using protocols::forge::methods::restore_residues;
	using protocols::forge::constraints::NtoC_RCG;
	using protocols::forge::constraints::NtoC_RCGOP;
	using protocols::forge::constraints::ConstraintFileRCG;
	using protocols::forge::constraints::ConstraintFileRCGOP;
	//using protocols::forge::constraints::SheetConstraintsRCG;
	//using protocols::forge::constraints::SheetConstraintsRCGOP;

	using protocols::toolbox::pose_manipulation::construct_poly_XXX_pose;


	// safety, clear the energies object
	pose.energies().clear();

	// make backup Pose for transferring sidechains
	Pose archive_pose = pose;
	Pose modified_archive_pose = archive_pose;
	manager_.modify( modified_archive_pose );

	// ensure modified_archive_pose is completely full-atom, otherwise mismatch
	// will occur when restoring sidechains at the end of the procedure
	bool mod_ap_is_full_atom = true;
	for ( Size i = 1, ie = modified_archive_pose.n_residue(); mod_ap_is_full_atom && i != ie; ++i ) {
		mod_ap_is_full_atom &= ( modified_archive_pose.residue( i ).residue_type_set().name() == core::chemical::FA_STANDARD );
	}

	if ( !mod_ap_is_full_atom ) {
		core::util::switch_to_residue_type_set( modified_archive_pose, core::chemical::FA_STANDARD );
	}

	if( use_poly_val_ ) {
		// flip to poly-ala-gly-pro-disulf pose
		utility::vector1< Size > protein_residues;
		for ( Size i = 1, ie = pose.n_residue(); i <= ie; ++i ) {
			if ( pose.residue( i ).is_protein() ) {
				protein_residues.push_back( i );
			}
		}
		construct_poly_XXX_pose( "VAL", pose, protein_residues, false, true, false );
	}
	/////////////////////////////////////////////////////////////////////////////////////

	// Run VLB to build the new section, if no segments have been added/deleted
	// we use the same VLB so that fragment caching works properly
	if ( !vlb_.get() ) {
		vlb_ = new VarLengthBuild( manager_ );
	}

	// set weight of constraints
	if( constraints_NtoC_ > 0.0 || constraints_sheet_ > 0.0 ){
		Real cst_weight( sfx_->get_weight( core::scoring::atom_pair_constraint ) );
		runtime_assert( cst_weight > 0.0 );
	}

	if( constraints_NtoC_ > 0.0 ){
		NtoC_RCGOP rcg = new NtoC_RCG;
		vlb_->add_rcg( rcg );
	}

	if( constraints_sheet_ > 0.0 ){
		SheetConstraintsRCGOP rcg = new SheetConstraintsRCG( blueprint_, constraints_sheet_ );
		vlb_->add_rcg( rcg );
	}

	if( constraint_file_ != "" ){
		ConstraintFileRCGOP cst = new ConstraintFileRCG( constraint_file_ );
		vlb_->add_rcg( cst );
	}

	vlb_->scorefunction( sfx_ );
	vlb_->vall_memory_usage( protocols::forge::components::VLB_VallMemoryUsage::CLEAR_IF_CACHING_FRAGMENTS );
	vlb_->use_fullmer( use_fullmer_ );
	vlb_->max_linear_chainbreak( max_linear_chainbreak_ );
	vlb_->loop_mover_str( loop_mover_str_ );
	vlb_->num_fragpick( num_fragpick_ );

	if( use_abego_bias_ ) {
		vlb_->set_abego( blueprint_->abego() );
	}

	if ( use_sequence_bias_ ) {
		vlb_->original_sequence( archive_pose.sequence() );
	}

	core::scoring::constraints::ConstraintSet cst ( *pose.constraint_set() );
	cst.show(std::cout);

	vlb_->apply( pose );

	vlb_->clear_rcgs();

	if ( vlb_->get_last_move_status() == MS_SUCCESS ) {

		// record the used manager w/ all mapping info
		manager_ = vlb_->manager();

		// safety, clear all the energies before restoring full-atom residues and
		// scoring
		pose.energies().clear();

		if ( !core::pose::symmetry::is_symmetric( pose ) ) {
			// Swap back original sidechains.  At the moment this is a two step process
			// in case any sidechains from SegmentInsert and the like that aren't in the
			// original archive pose need to be transferred.
			restore_residues( modified_archive_pose, pose );
			restore_residues( manager_.original2modified(), archive_pose, pose );
		} else {
			TR << "Side-chains won't be swapped back to original. " << std::endl;
		}

		return true; // loop closed

	} else {

		pose = archive_pose;

	}

	return false; // false if loop not closed
}

/// @brief parse xml
void
BluePrintBDR::parse_my_tag(
	TagPtr const tag,
	DataMap & data,
	Filters_map const &,
	Movers_map const &,
	Pose const & )
{
	String const blueprint( tag->getOption<std::string>( "blueprint", "" ) );
	if( blueprint == "" ){
		TR << "No input of blueprint file ! " << std::endl;
		runtime_assert( false );
	}
	set_blueprint( blueprint );

	// set secondary structure using blueprint
	ss_from_blueprint_ = tag->getOption<bool>( "ss_from_blueprint", 1 );

	// set scorefxn
	String const sfxn ( tag->getOption<String>( "scorefxn", "" ) );
	if( sfxn != "" ) {
		sfx_ = data.get< ScoreFunction * >( "scorefxns", sfxn );
		TR << "score function, " << sfxn << ", is used. " << std::endl;
	}

	// pick fragment using sequence information
	use_sequence_bias_ = tag->getOption<bool>( "use_sequence_bias", 0 );

	// pick fragment using abego torsion information
	use_abego_bias_ = tag->getOption<bool>( "use_abego_bias", 0 );

	// constraint N- and C- terminal
	constraints_NtoC_ = tag->getOption<Real>( "constraints_NtoC", -1.0 );

	// constraints between Ca atoms in sheet
	constraints_sheet_ = tag->getOption<Real>( "constraints_sheet", -1.0 );

	// set constraint file
	constraint_file_ = tag->getOption<String>( "constraint_file", "" );

	// dump pdb when the protocol fail
	dump_pdb_when_fail_ = tag->getOption<String>( "dump_pdb_when_fail", "" );

	// loop mover for rebuilding loops
	loop_mover_str_ = tag->getOption<String>( "loop_mover", "RemodelLoopMover" );

	// number of allowed_closure_attempts_ of RemodelLoopMover
	rmdl_attempts_ = tag->getOption<Size>( "rmdl_attempts", 1 );

	// entire sequence except for rebuilding parts become poly-Val ( default true )
	use_poly_val_ = tag->getOption<bool>( "use_poly_val", 1 );

}


} // namespace fldsgn
} // namespace protocols
