// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/protein_interface_design/movers/TaskAwareCsts.cc
/// @brief
/// @author Sarel Fleishman (sarelf@u.washington.edu)

// Unit headers
#include <protocols/protein_interface_design/movers/TaskAwareCsts.hh>
#include <protocols/protein_interface_design/movers/TaskAwareCstsCreator.hh>
#include <boost/foreach.hpp>
#define foreach BOOST_FOREACH
// Package headers
#include <core/pose/Pose.hh>
#include <core/conformation/util.hh>
#include <core/conformation/Conformation.hh>
#include <core/pack/task/TaskFactory.hh>
#include <basic/Tracer.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <utility/tag/Tag.hh>
#include <utility/vector1.hh>
#include <protocols/moves/DataMap.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/rosetta_scripts/util.hh>
//Auto Headers
#include <core/conformation/Residue.hh>
#include <numeric/xyzVector.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/id/AtomID.hh>

namespace protocols {
namespace protein_interface_design {
namespace movers {

static basic::Tracer TR( "protocols.protein_interface_design.movers.TaskAwareCsts" );
std::string
TaskAwareCstsCreator::keyname() const
{
	return TaskAwareCstsCreator::mover_name();
}

protocols::moves::MoverOP
TaskAwareCstsCreator::create_mover() const {
	return new TaskAwareCsts;
}

std::string
TaskAwareCstsCreator::mover_name()
{
	return "TaskAwareCsts";
}

TaskAwareCsts::TaskAwareCsts() :
	Mover( TaskAwareCstsCreator::mover_name() ),
	task_factory_( NULL ),
	cst_type_( "coordinate" )
{
}


TaskAwareCsts::~TaskAwareCsts() {}

void
TaskAwareCsts::apply( core::pose::Pose & pose )
{
	using namespace protocols::rosetta_scripts;
	using namespace core::scoring::constraints;
	using core::id::AtomID;

	ConstraintCOPs cst;
	utility::vector1< core::Size > const designable( residue_packer_states( pose, task_factory(), true/*designable*/, false/*packable*/ ) );
	runtime_assert( designable.size() );
	AtomID const anchor_atom( AtomID( pose.residue( designable[ 1 ] ).atom_index( "CA" ), designable[ 1 ] ) ); // anchor to first designable CA
	HarmonicFuncOP coord_cst_func( new HarmonicFunc( 0.0, 1.0/*sd*/ ) ); // hardwired for now
	TR<<"Adding constraints to pose: ";
	foreach( core::Size const resid, designable ){
		core::conformation::Residue const rsd_i( pose.residue( resid ) );
		if( cst_type_ == "coordinate" ){
			cst.push_back( new CoordinateConstraint( AtomID( rsd_i.atom_index( "CA" ), resid ), anchor_atom, rsd_i.xyz( "CA" ), coord_cst_func ) );
			TR<<"coordinate cst at resid: "<<resid<<'\n';
		}
	}
	pose.add_constraints( cst );
	TR.flush();
}

std::string
TaskAwareCsts::get_name() const {
	return TaskAwareCstsCreator::mover_name();
}

void
TaskAwareCsts::parse_my_tag( TagPtr const tag, protocols::moves::DataMap &data, protocols::filters::Filters_map const &, protocols::moves::Movers_map const &, core::pose::Pose const & pose )
{
	using namespace protocols::rosetta_scripts;
	task_factory( parse_task_operations( tag, data ) );
	cst_type( tag->getOption< std::string >( "cst_type", "coordinate" ) );
}

protocols::moves::MoverOP
TaskAwareCsts::clone() const {
    return( protocols::moves::MoverOP( new TaskAwareCsts( *this ) ));
}

core::pack::task::TaskFactoryOP
TaskAwareCsts::task_factory() const{ return task_factory_; }

void
TaskAwareCsts::task_factory( core::pack::task::TaskFactoryOP tf ){ task_factory_ = tf; }

} //movers
} //protein_interface_design
} //protocols
