// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file StepWiseProteinPoseMinimizer
/// @brief Not particularly fancy, just minimizes a list of poses.
/// @detailed
/// @author Rhiju Das


//////////////////////////////////
#include <protocols/swa/protein/StepWiseProteinPoseMinimizer.hh>
#include <protocols/swa/protein/StepWiseProteinUtil.hh>
#include <protocols/swa/StepWiseUtil.hh>

//////////////////////////////////
#include <core/conformation/Conformation.hh>
#include <core/conformation/util.hh>
#include <core/types.hh>
//#include <core/io/silent/BinaryProteinSilentStruct.hh>
#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/SilentFileData.fwd.hh>
#include <core/id/TorsionID.hh>
#include <core/pose/Pose.hh>
#include <core/pose/util.hh>
#include <core/scoring/constraints/util.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/EnergyGraph.hh>
// AUTO-REMOVED #include <core/scoring/rms_util.hh>
// AUTO-REMOVED #include <core/scoring/rms_util.tmpl.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/kinematics/MoveMap.hh>
#include <ObjexxFCL/format.hh>

//#include <utility/basic_sys_util.hh>
// AUTO-REMOVED #include <time.h>


#include <string>

//Auto Headers
#include <utility/vector1.hh>
using namespace core;
using core::Real;
using ObjexxFCL::fmt::F;

namespace protocols {
namespace swa {
namespace protein {


  //////////////////////////////////////////////////////////////////////////
  //constructor!
  StepWiseProteinPoseMinimizer::StepWiseProteinPoseMinimizer( PoseList & pose_list, utility::vector1< Size > const & moving_residues ):
		Mover(),
    pose_list_( pose_list ),
    moving_residues_( moving_residues ),
		move_takeoff_torsions_( true ),
		rescore_only_( false ),
		move_jumps_between_chains_( false ),
		silent_file_( "" ),
    fa_scorefxn_( core::scoring::getScoreFunction() ),
		min_type_( "dfpmin_armijo_nonmonotone" ), // used to be dfpmin
		min_tolerance_( 0.000025 ) // used to be 0.00000025
  {
		Mover::type("StepWiseProteinPoseMinimizer");
  }

  //////////////////////////////////////////////////////////////////////////
  //destructor
  StepWiseProteinPoseMinimizer::~StepWiseProteinPoseMinimizer()
  {}

	/////////////////////
	std::string
	StepWiseProteinPoseMinimizer::get_name() const {
		return "StepWiseProteinPoseMinimizer";
	}


  //////////////////////////////////////////////////////////////////////////
  void
  StepWiseProteinPoseMinimizer::apply( core::pose::Pose & pose )
  {
    using namespace core::optimization;
    using namespace core::scoring;
    using namespace core::scoring::constraints;
    using namespace core::pose;

		clock_t const time_start( clock() );

		ConstraintSetOP cst_set = pose.constraint_set()->clone();

		utility::vector1< std::pair<core::Size,core::Size> > disulfides;
		core::conformation::disulfide_bonds(pose.conformation(), disulfides);

    AtomTreeMinimizer minimizer;
    bool const use_nblist( true );
    MinimizerOptions options( min_type_, min_tolerance_, use_nblist, false, false );
    options.nblist_auto_update( true );

    kinematics::MoveMap mm_start, mm;
		std::cout << "MOVE TAKEOFF TORSIONS: " << move_takeoff_torsions_ << std::endl;
		protocols::swa::Figure_out_moving_residues( mm_start, pose, fixed_res_, move_takeoff_torsions_, move_jumps_between_chains_ );
		mm = mm_start;

		//		using namespace core::id;
		//		for ( Size i = 1; i <= pose.total_residue(); i++ ){
		//			std::cout << " MM: " << i << ' ' << mm.get( TorsionID( i, BB, 1 ) ) << ' ' << mm.get( TorsionID( i, BB, 2 ) ) << ' ' << mm.get( TorsionID( i, BB, 3 ) ) << std::endl;
		//		}

    Size count( 1 );
		Real const original_coordinate_cst_weight = fa_scorefxn_->get_weight( coordinate_constraint );

		sfd_ = new core::io::silent::SilentFileData;

    for ( PoseList::iterator iter = pose_list_.begin(); iter != pose_list_.end(); iter++ ) {

      std::cout << "Minimizing decoy " << count++ << " out of " << pose_list_.size() << std::endl;

      PoseOP & pose_op( iter->second );
      pose = *pose_op; // This copy is to allow for easy graphic visualization.

			// Following are necessary because poses from clustering went thorugh silent struct and lost their constraints & disulfide information.
			pose.constraint_set( cst_set );
			pose.conformation().fix_disulfides( disulfides );

			Real const score_original = (*fa_scorefxn_)( pose );

			// The movemap has all dofs for "non-fixed residues" free to move.
			// We can also let sidechains minimize in fixed-residues -- for
			// speed only look at neighbors of moving residues.
			mm = mm_start;
			let_neighboring_chis_minimize( mm, pose );

			if ( !rescore_only_ ){

				// One minimize with loose coordinate tethers to make sure the pose doesn't blow up.
				core::scoring::constraints::add_coordinate_constraints( pose );
				if ( fa_scorefxn_->has_zero_weight( coordinate_constraint) ) fa_scorefxn_->set_weight( coordinate_constraint, 1.0 );
				minimizer.run( pose, mm, *fa_scorefxn_, options );

				// Now a regular minimize.
				pose.constraint_set( cst_set ); // return original constraints (no added coordinate constraints)
				fa_scorefxn_->set_weight( coordinate_constraint, original_coordinate_cst_weight );

				// for poses with chainbreaks, do an initial minimization with a weak linear_chainbreak term. (anneal it in.)
				if ( pose_has_chainbreak( pose ) ){

					Real const linear_chainbreak_weight_original = fa_scorefxn_->get_weight( linear_chainbreak );
					if ( linear_chainbreak_weight_original < 20.0 ) std::cout << "WARNING!! Your linear_chainbreak weight is " << F(8,3,linear_chainbreak_weight_original ) << ", which is less than recommended (20.0) " << std::endl;

					fa_scorefxn_->set_weight( linear_chainbreak, linear_chainbreak_weight_original * 0.25 );
					minimizer.run( pose, mm, *fa_scorefxn_, options );
					fa_scorefxn_->set_weight( linear_chainbreak, linear_chainbreak_weight_original );
				}


				minimizer.run( pose, mm, *fa_scorefxn_, options );

			}

			setPoseExtraScores( pose, "score_orig", score_original );
      std::string const & tag( iter->first );
      protocols::swa::protein::output_silent_struct( pose, get_native_pose(), silent_file_, tag, sfd_, calc_rms_res_ );

			std::cout << "Score minimized from " <<F(8,3, score_original) << " to " << F(8,3,(*fa_scorefxn_)( pose )) << std::endl;

			// Running into file locking issues
			//			utility::sys_sleep( 0.5 );
			//exit( 0 );

      // Might was well replace pose in original list.
      *pose_op = pose;
    }

		std::cout << "Total time in StepWiseProteinPoseMinimizer: " <<
			static_cast<Real>(clock() - time_start) / CLOCKS_PER_SEC << std::endl;

  }

  //////////////////////////////////////////////////////////////////////////
	void
	StepWiseProteinPoseMinimizer::let_neighboring_chis_minimize(
																															core::kinematics::MoveMap & mm,
																															core::pose::Pose & pose ){

		using namespace core::scoring;

		(*fa_scorefxn_)( pose );
		EnergyGraph const & energy_graph( pose.energies().energy_graph() );

		for ( Size n = 1; n <= moving_residues_.size(); n++ ) {

			Size const i = moving_residues_[ n ];

			for( graph::Graph::EdgeListConstIter
						 iter = energy_graph.get_node( i )->const_edge_list_begin();
					 iter != energy_graph.get_node( i )->const_edge_list_end();
					 ++iter ){

				Size j( (*iter)->get_other_ind( i ) );
				if ( pose.residue(j).has_variant_type( "VIRTUAL_RESIDUE" ) ) continue;

				if ( pose.residue(j).is_protein() ){
					mm.set_chi( j, true );
				} else if ( pose.residue(j).is_RNA() ){
					mm.set( id::TorsionID( j, id::CHI, 4), true ); // 2'-OH.
				}

			}
		}

	}

  //////////////////////////////////////////////////////////////////////////
	bool
	StepWiseProteinPoseMinimizer::pose_has_chainbreak( pose::Pose const & pose ){
		// this is pretty conservative -- actually  there might be
		// cases where the pose has a chainbreak but the minimized dofs would
		// not affect the relative positions of the chainbreak residues.
		for ( Size i = 1; i <= pose.total_residue(); i++ ){
			if ( pose.residue_type(i).has_variant_type( "CUTPOINT_UPPER" ) ) return true;
			if ( pose.residue_type(i).has_variant_type( "CUTPOINT_LOWER" ) ) return true;
		}
		return false;
	}

  //////////////////////////////////////////////////////////////////////////
  void
  StepWiseProteinPoseMinimizer::set_silent_file( std::string const & silent_file ){
    silent_file_ = silent_file;
  }

  //////////////////////////////////////////////////////////////////////////
  void
  StepWiseProteinPoseMinimizer::set_min_type( std::string const & min_type ){
    min_type_ = min_type;
  }

  //////////////////////////////////////////////////////////////////////////
  void
  StepWiseProteinPoseMinimizer::set_min_tolerance( Real const & min_tolerance ){
    min_tolerance_ = min_tolerance;
  }

  //////////////////////////////////////////////////////////////////////////
	void
	StepWiseProteinPoseMinimizer::set_scorefxn( core::scoring::ScoreFunctionOP const & scorefxn ){
		fa_scorefxn_ = scorefxn;
	}

  //////////////////////////////////////////////////////////////////////////
	void
	StepWiseProteinPoseMinimizer::set_fixed_res( utility::vector1< core::Size > const & fixed_res ){
		fixed_res_ = fixed_res;
	}

  //////////////////////////////////////////////////////////////////////////
	void
	StepWiseProteinPoseMinimizer::set_calc_rms_res( utility::vector1< core::Size > const & calc_rms_res ){
		calc_rms_res_ = calc_rms_res;
	}

  //////////////////////////////////////////////////////////////////////////
	core::io::silent::SilentFileDataOP &
	StepWiseProteinPoseMinimizer::silent_file_data(){
		return sfd_;
	}

	//	void
	//	StepWiseProteinPoseMinimizer::set_constraint_set( core::scoring::constraints::ConstraintSetOP const & cst_set ){
	//		cst_set_ = cst_set;
	//	}


}
}
}
