// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   test/util/pose_funcs.hh
/// @brief  common utility functions for getting a pose into a unit test.
/// @author Andrew Leaver-Fay

#ifndef INCLUDED_util_pose_funcs_HH
#define INCLUDED_util_pose_funcs_HH

#include <core/chemical/ChemicalManager.hh>

#include <core/io/pdb/pose_io.hh>
#include <core/io/pdb/pdb_dynamic_reader.hh>

#include <core/import_pose/import_pose.hh>
#include <core/io/pdb/file_data.hh>

#include <core/pose/Pose.hh>

#include <string>


std::string
ubq_twores_string();

std::string
test_in_pdb_string();

std::string
trp_cage_ideal();

std::string
pdb_string_1ten();

inline
core::pose::Pose
fullatom_pose_from_string(
	std::string const & pdbstring
)
{
	using namespace core::chemical;
	using namespace core::pose;

	Pose pose;
	core::io::pdb::FileData fd = core::io::pdb::PDB_DReader::createFileData( pdbstring );
	ResidueTypeSetCAP residue_set
		( ChemicalManager::get_instance()->residue_type_set( FA_STANDARD ) );
	core::import_pose::build_pose( fd, pose, *residue_set);
	return pose;

}

inline
core::pose::PoseOP
fullatom_poseop_from_string(
	std::string const & pdbstring
)
{
	using namespace core::chemical;
	using namespace core::pose;

	PoseOP pose = new Pose;
	core::io::pdb::FileData fd = core::io::pdb::PDB_DReader::createFileData( pdbstring );
	ResidueTypeSetCAP residue_set
		( ChemicalManager::get_instance()->residue_type_set( FA_STANDARD ) );
	core::import_pose::build_pose( fd,*pose, *residue_set);
	return pose;
}

inline
core::pose::Pose
create_twores_1ubq_pose() {
	return fullatom_pose_from_string( ubq_twores_string() );
}

inline
core::pose::PoseOP
create_twores_1ubq_poseop() {
	return fullatom_poseop_from_string( ubq_twores_string() );
}


inline
core::pose::Pose
create_test_in_pdb_pose() {
	return fullatom_pose_from_string( test_in_pdb_string() );
}

inline
core::pose::Pose
create_1ten_pdb_pose() {
	return fullatom_pose_from_string( pdb_string_1ten() );
}


inline
core::pose::PoseOP
create_test_in_pdb_poseop() {
	return fullatom_poseop_from_string( test_in_pdb_string() );
}

inline
core::pose::Pose
create_trpcage_ideal_pose() {
	return fullatom_pose_from_string( trp_cage_ideal() );
}

inline
core::pose::PoseOP
create_trpcage_ideal_poseop() {
	return fullatom_poseop_from_string( trp_cage_ideal() );
}


inline
std::string
ubq_twores_string() {

	return
		"ATOM      1  N   MET     1      27.340  24.430   2.614  1.00  9.67      1UBQ  71\n"
		"ATOM      2  CA  MET     1      26.266  25.413   2.842  1.00 10.38      1UBQ  72\n"
		"ATOM      3  C   MET     1      26.913  26.639   3.531  1.00  9.62      1UBQ  73\n"
		"ATOM      4  O   MET     1      27.886  26.463   4.263  1.00  9.62      1UBQ  74\n"
		"ATOM      5  CB  MET     1      25.112  24.880   3.649  1.00 13.77      1UBQ  75\n"
		"ATOM      6  CG  MET     1      25.353  24.860   5.134  1.00 16.29      1UBQ  76\n"
		"ATOM      7  SD  MET     1      23.930  23.959   5.904  1.00 17.17      1UBQ  77\n"
		"ATOM      8  CE  MET     1      24.447  23.984   7.620  1.00 16.11      1UBQ  78\n"
		"ATOM      9  N   GLN     2      26.335  27.770   3.258  1.00  9.27      1UBQ  79\n"
		"ATOM     10  CA  GLN     2      26.850  29.021   3.898  1.00  9.07      1UBQ  80\n"
		"ATOM     11  C   GLN     2      26.100  29.253   5.202  1.00  8.72      1UBQ  81\n"
		"ATOM     12  O   GLN     2      24.865  29.024   5.330  1.00  8.22      1UBQ  82\n"
		"ATOM     13  CB  GLN     2      26.733  30.148   2.905  1.00 14.46      1UBQ  83\n"
		"ATOM     14  CG  GLN     2      26.882  31.546   3.409  1.00 17.01      1UBQ  84\n"
		"ATOM     15  CD  GLN     2      26.786  32.562   2.270  1.00 20.10      1UBQ  85\n"
		"ATOM     16  OE1 GLN     2      27.783  33.160   1.870  1.00 21.89      1UBQ  86\n"
		"ATOM     17  NE2 GLN     2      25.562  32.733   1.806  1.00 19.49      1UBQ  87\n";
}

// For those playing along at home, this looks like it's the protien portion of 1BL0
// (MarA, the alpha-CTD of RNA polymerase). That's chain A, residues 9-124, inclusive,
// after hydrogen addition and flip_HNQ.
inline
std::string
test_in_pdb_string()
{
	return
		"ATOM   1525  N   ASP    49       0.229  36.012  74.172  1.00 49.53\n"
		"ATOM   1526  CA  ASP    49       0.041  35.606  75.594  1.00 50.07\n"
		"ATOM   1527  C   ASP    49      -0.096  36.849  76.498  1.00 50.51\n"
		"ATOM   1528  O   ASP    49      -0.951  36.895  77.382  1.00 49.49\n"
		"ATOM   1529  CB  ASP    49       1.225  34.718  76.092  1.00 49.72\n"
		"ATOM   1530  CG  ASP    49       2.159  34.156  74.999  1.00 50.62\n"
		"ATOM   1531  OD1 ASP    49       1.688  33.361  74.151  1.00 49.24\n"
		"ATOM   1532  OD2 ASP    49       3.378  34.497  75.007  1.00 49.24\n"
		"ATOM   1533  H   ASP    49       0.224  36.191  73.178  1.00 50.31\n"
		"ATOM   1534  HA  ASP    49      -0.884  35.037  75.696  1.00 50.09\n"
		"ATOM   1535 1HB  ASP    49       1.839  35.199  76.854  1.00 50.09\n"
		"ATOM   1536 2HB  ASP    49       0.670  33.892  76.539  1.00 50.09\n"
		"ATOM   1537  N   ALA    50       0.764  37.858  76.239  1.00 49.53\n"
		"ATOM   1538  CA  ALA    50       0.778  39.078  77.028  1.00 50.07\n"
		"ATOM   1539  C   ALA    50      -0.624  39.484  77.368  1.00 50.51\n"
		"ATOM   1540  O   ALA    50      -0.925  39.776  78.527  1.00 49.49\n"
		"ATOM   1541  CB  ALA    50       1.474  40.216  76.301  1.00 49.73\n"
		"ATOM   1542  H   ALA    50       1.417  37.762  75.474  1.00 50.31\n"
		"ATOM   1543  HA  ALA    50       1.307  38.885  77.961  1.00 50.09\n"
		"ATOM   1544 1HB  ALA    50       1.461  41.108  76.927  1.00 50.09\n"
		"ATOM   1545 2HB  ALA    50       2.506  39.936  76.090  1.00 50.09\n"
		"ATOM   1546 3HB  ALA    50       0.955  40.422  75.367  1.00 50.09\n"
		"ATOM   1547  N   ILE    51      -1.497  39.472  76.368  1.00 49.53\n"
		"ATOM   1548  CA  ILE    51      -2.894  39.862  76.581  1.00 50.07\n"
		"ATOM   1549  C   ILE    51      -3.681  38.874  77.494  1.00 50.51\n"
		"ATOM   1550  O   ILE    51      -4.440  39.307  78.357  1.00 49.49\n"
		"ATOM   1551  CB  ILE    51      -3.640  40.060  75.221  1.00 49.91\n"
		"ATOM   1552  CG1 ILE    51      -5.143  40.243  75.468  1.00 49.82\n"
		"ATOM   1553  CG2 ILE    51      -3.373  38.888  74.282  1.00 49.73\n"
		"ATOM   1554  CD1 ILE    51      -5.960  38.974  75.360  1.00 49.73\n"
		"ATOM   1555  H   ILE    51      -1.194  39.190  75.447  1.00 50.31\n"
		"ATOM   1556  HA  ILE    51      -2.951  40.808  77.118  1.00 50.09\n"
		"ATOM   1557  HB  ILE    51      -3.282  40.975  74.750  1.00 50.09\n"
		"ATOM   1558 1HG2 ILE    51      -3.902  39.046  73.343  1.00 50.09\n"
		"ATOM   1559 2HG2 ILE    51      -2.304  38.814  74.088  1.00 50.09\n"
		"ATOM   1560 3HG2 ILE    51      -3.723  37.964  74.744  1.00 50.09\n"
		"ATOM   1561 1HG1 ILE    51      -5.259  40.657  76.469  1.00 50.09\n"
		"ATOM   1562 2HG1 ILE    51      -5.506  40.963  74.734  1.00 50.09\n"
		"ATOM   1563 1HD1 ILE    51      -7.010  39.199  75.549  1.00 50.09\n"
		"ATOM   1564 2HD1 ILE    51      -5.854  38.556  74.358  1.00 50.09\n"
		"ATOM   1565 3HD1 ILE    51      -5.608  38.250  76.094  1.00 50.09\n"
		"ATOM   1566  N   THR    52      -3.486  37.570  77.328  1.00 49.53\n"
		"ATOM   1567  CA  THR    52      -4.225  36.591  78.124  1.00 50.07\n"
		"ATOM   1568  C   THR    52      -3.547  36.333  79.464  1.00 50.51\n"
		"ATOM   1569  O   THR    52      -4.214  36.142  80.489  1.00 49.49\n"
		"ATOM   1570  CB  THR    52      -4.343  35.276  77.345  1.00 50.14\n"
		"ATOM   1571  OG1 THR    52      -4.907  34.235  78.157  1.00 49.34\n"
		"ATOM   1572  CG2 THR    52      -2.968  34.868  76.863  1.00 49.73\n"
		"ATOM   1573  H   THR    52      -2.817  37.250  76.642  1.00 50.31\n"
		"ATOM   1574  HG1 THR    52      -4.968  33.426  77.643  1.00 50.43\n"
		"ATOM   1575  HA  THR    52      -5.218  36.977  78.354  1.00 50.09\n"
		"ATOM   1576  HB  THR    52      -4.999  35.429  76.488  1.00 50.09\n"
		"ATOM   1577 1HG2 THR    52      -3.041  33.933  76.307  1.00 50.09\n"
		"ATOM   1578 2HG2 THR    52      -2.564  35.646  76.215  1.00 50.09\n"
		"ATOM   1579 3HG2 THR    52      -2.309  34.731  77.720  1.00 50.09\n"
		"ATOM   1580  N   ILE    53      -2.220  36.307  79.463  1.00 49.53\n"
		"ATOM   1581  CA  ILE    53      -1.508  36.121  80.712  1.00 50.07\n"
		"ATOM   1582  C   ILE    53      -1.765  37.303  81.617  1.00 50.51\n"
		"ATOM   1583  O   ILE    53      -1.912  37.119  82.808  1.00 49.49\n"
		"ATOM   1584  CB  ILE    53       0.010  35.998  80.515  1.00 49.91\n"
		"ATOM   1585  CG1 ILE    53       0.349  34.580  80.084  1.00 49.82\n"
		"ATOM   1586  CG2 ILE    53       0.744  36.360  81.808  1.00 49.73\n"
		"ATOM   1587  CD1 ILE    53      -0.241  33.514  80.988  1.00 49.73\n"
		"ATOM   1588  H   ILE    53      -1.710  36.416  78.598  1.00 50.31\n"
		"ATOM   1589  HA  ILE    53      -1.892  35.254  81.247  1.00 50.09\n"
		"ATOM   1590  HB  ILE    53       0.319  36.663  79.709  1.00 50.09\n"
		"ATOM   1591 1HG2 ILE    53       1.819  36.268  81.654  1.00 50.09\n"
		"ATOM   1592 2HG2 ILE    53       0.504  37.385  82.088  1.00 50.09\n"
		"ATOM   1593 3HG2 ILE    53       0.433  35.684  82.605  1.00 50.09\n"
		"ATOM   1594 1HG1 ILE    53      -0.029  34.450  79.070  1.00 50.09\n"
		"ATOM   1595 2HG1 ILE    53       1.436  34.494  80.079  1.00 50.09\n"
		"ATOM   1596 1HD1 ILE    53       0.042  32.527  80.621  1.00 50.09\n"
		"ATOM   1597 2HD1 ILE    53       0.138  33.645  82.002  1.00 50.09\n"
		"ATOM   1598 3HD1 ILE    53      -1.327  33.600  80.993  1.00 50.09\n"
		"ATOM   1599  N   HIS    54      -1.792  38.521  81.023  1.00 49.53\n"
		"ATOM   1600  CA  HIS    54      -2.050  39.765  81.771  1.00 50.07\n"
		"ATOM   1601  C   HIS    54      -3.436  39.741  82.387  1.00 50.51\n"
		"ATOM   1602  O   HIS    54      -3.727  40.494  83.318  1.00 49.49\n"
		"ATOM   1603  CB  HIS    54      -1.906  41.013  80.883  1.00 49.92\n"
		"ATOM   1604  CG  HIS    54      -0.503  41.542  80.888  1.00 50.22\n"
		"ATOM   1605  ND1 HIS    54       0.606  40.717  81.095  1.00 49.30\n"
		"ATOM   1606  CD2 HIS    54       0.005  42.792  80.662  1.00 49.95\n"
		"ATOM   1607  CE1 HIS    54       1.718  41.419  80.982  1.00 50.25\n"
		"ATOM   1608  NE2 HIS    54       1.392  42.675  80.722  1.00 49.64\n"
		"ATOM   1609  H   HIS    54      -1.629  38.579  80.028  1.00 50.31\n"
		"ATOM   1610  HE2 HIS    54       2.044  43.435  80.587  1.00 50.32\n"
		"ATOM   1611  HA  HIS    54      -1.341  39.851  82.594  1.00 50.09\n"
		"ATOM   1612 1HB  HIS    54      -2.163  40.772  79.851  1.00 50.09\n"
		"ATOM   1613 2HB  HIS    54      -2.559  41.807  81.243  1.00 50.09\n"
		"ATOM   1614  HE1 HIS    54       2.690  40.941  81.102  1.00 50.13\n"
		"ATOM   1615  HD2 HIS    54      -0.463  43.757  80.465  1.00 50.09\n"
		"ATOM   1616  N   SER    55      -4.279  38.850  81.877  1.00 49.53\n"
		"ATOM   1617  CA  SER    55      -5.622  38.681  82.392  1.00 50.07\n"
		"ATOM   1618  C   SER    55      -5.672  37.661  83.550  1.00 50.51\n"
		"ATOM   1619  O   SER    55      -6.102  38.000  84.643  1.00 49.49\n"
		"ATOM   1620  CB  SER    55      -6.564  38.263  81.271  1.00 50.05\n"
		"ATOM   1621  OG  SER    55      -7.690  37.609  81.821  1.00 49.34\n"
		"ATOM   1622  H   SER    55      -3.974  38.273  81.106  1.00 50.31\n"
		"ATOM   1623  HG  SER    55      -8.284  37.346  81.114  1.00 50.43\n"
		"ATOM   1624  HA  SER    55      -6.084  39.616  82.712  1.00 50.09\n"
		"ATOM   1625 1HB  SER    55      -6.874  39.154  80.726  1.00 50.09\n"
		"ATOM   1626 2HB  SER    55      -6.030  37.594  80.597  1.00 50.09\n"
		"ATOM   1627  N   ILE    56      -5.251  36.411  83.296  1.00 49.53\n"
		"ATOM   1628  CA  ILE    56      -5.161  35.383  84.352  1.00 50.07\n"
		"ATOM   1629  C   ILE    56      -4.589  35.961  85.675  1.00 50.51\n"
		"ATOM   1630  O   ILE    56      -4.994  35.580  86.770  1.00 49.49\n"
		"ATOM   1631  CB  ILE    56      -4.289  34.199  83.887  1.00 49.91\n"
		"ATOM   1632  CG1 ILE    56      -4.831  32.940  84.495  1.00 49.82\n"
		"ATOM   1633  CG2 ILE    56      -2.840  34.402  84.273  1.00 49.73\n"
		"ATOM   1634  CD1 ILE    56      -6.327  32.842  84.380  1.00 49.73\n"
		"ATOM   1635  H   ILE    56      -4.987  36.166  82.352  1.00 50.31\n"
		"ATOM   1636  HA  ILE    56      -6.153  35.022  84.621  1.00 50.09\n"
		"ATOM   1637  HB  ILE    56      -4.368  34.103  82.805  1.00 50.09\n"
		"ATOM   1638 1HG2 ILE    56      -2.250  33.551  83.933  1.00 50.09\n"
		"ATOM   1639 2HG2 ILE    56      -2.464  35.313  83.809  1.00 50.09\n"
		"ATOM   1640 3HG2 ILE    56      -2.760  34.488  85.357  1.00 50.09\n"
		"ATOM   1641 1HG1 ILE    56      -4.360  32.103  83.981  1.00 50.09\n"
		"ATOM   1642 2HG1 ILE    56      -4.535  32.934  85.544  1.00 50.09\n"
		"ATOM   1643 1HD1 ILE    56      -6.668  31.912  84.836  1.00 50.09\n"
		"ATOM   1644 2HD1 ILE    56      -6.788  33.687  84.892  1.00 50.09\n"
		"ATOM   1645 3HD1 ILE    56      -6.613  32.855  83.329  1.00 50.09\n"
		"ATOM   1646  N   LEU    57      -3.636  36.868  85.544  1.00 49.53\n"
		"ATOM   1647  CA  LEU    57      -3.105  37.610  86.674  1.00 50.07\n"
		"ATOM   1648  C   LEU    57      -4.271  38.274  87.507  1.00 50.51\n"
		"ATOM   1649  O   LEU    57      -4.384  38.078  88.731  1.00 49.49\n"
		"ATOM   1650  CB  LEU    57      -2.136  38.705  86.149  1.00 49.82\n"
		"ATOM   1651  CG  LEU    57      -0.607  38.429  86.152  1.00 49.91\n"
		"ATOM   1652  CD1 LEU    57      -0.286  37.060  86.720  1.00 49.73\n"
		"ATOM   1653  CD2 LEU    57      -0.040  38.558  84.753  1.00 49.73\n"
		"ATOM   1654  H   LEU    57      -3.263  37.052  84.623  1.00 50.31\n"
		"ATOM   1655  HA  LEU    57      -2.588  36.928  87.348  1.00 50.09\n"
		"ATOM   1656 1HB  LEU    57      -2.476  38.762  85.116  1.00 50.09\n"
		"ATOM   1657 2HB  LEU    57      -2.305  39.663  86.640  1.00 50.09\n"
		"ATOM   1658  HG  LEU    57      -0.138  39.211  86.750  1.00 50.09\n"
		"ATOM   1659 1HD1 LEU    57       0.793  36.903  86.706  1.00 50.09\n"
		"ATOM   1660 2HD1 LEU    57      -0.647  36.998  87.747  1.00 50.09\n"
		"ATOM   1661 3HD1 LEU    57      -0.771  36.293  86.117  1.00 50.09\n"
		"ATOM   1662 1HD2 LEU    57       1.032  38.362  84.776  1.00 50.09\n"
		"ATOM   1663 2HD2 LEU    57      -0.527  37.838  84.095  1.00 50.09\n"
		"ATOM   1664 3HD2 LEU    57      -0.216  39.567  84.380  1.00 50.09\n"
		"ATOM   1665  N   ASP    58      -5.155  39.059  86.852  1.00 49.53\n"
		"ATOM   1666  CA  ASP    58      -6.240  39.725  87.589  1.00 50.07\n"
		"ATOM   1667  C   ASP    58      -7.134  38.708  88.198  1.00 50.51\n"
		"ATOM   1668  O   ASP    58      -7.532  38.824  89.346  1.00 49.49\n"
		"ATOM   1669  CB  ASP    58      -7.079  40.634  86.683  1.00 49.72\n"
		"ATOM   1670  CG  ASP    58      -6.281  41.787  86.122  1.00 50.62\n"
		"ATOM   1671  OD1 ASP    58      -5.611  41.580  85.091  1.00 49.24\n"
		"ATOM   1672  OD2 ASP    58      -6.311  42.897  86.712  1.00 49.24\n"
		"ATOM   1673  H   ASP    58      -5.070  39.189  85.854  1.00 50.31\n"
		"ATOM   1674  HA  ASP    58      -5.823  40.317  88.404  1.00 50.09\n"
		"ATOM   1675 1HB  ASP    58      -7.591  40.098  85.884  1.00 50.09\n"
		"ATOM   1676 2HB  ASP    58      -7.811  41.000  87.404  1.00 50.09\n"
		"ATOM   1677  N   TRP    59      -7.438  37.691  87.434  1.00 49.53\n"
		"ATOM   1678  CA  TRP    59      -8.307  36.699  87.982  1.00 50.07\n"
		"ATOM   1679  C   TRP    59      -7.701  36.101  89.223  1.00 50.51\n"
		"ATOM   1680  O   TRP    59      -8.432  35.736  90.140  1.00 49.49\n"
		"ATOM   1681  CB  TRP    59      -8.620  35.581  87.000  1.00 49.82\n"
		"ATOM   1682  CG  TRP    59      -9.535  34.551  87.595  1.00 49.97\n"
		"ATOM   1683  CD1 TRP    59     -10.902  34.633  87.706  1.00 50.03\n"
		"ATOM   1684  CD2 TRP    59      -9.170  33.258  88.106  1.00 49.98\n"
		"ATOM   1685  NE1 TRP    59     -11.406  33.470  88.256  1.00 49.39\n"
		"ATOM   1686  CE2 TRP    59     -10.362  32.614  88.513  1.00 50.13\n"
		"ATOM   1687  CE3 TRP    59      -7.959  32.591  88.270  1.00 49.88\n"
		"ATOM   1688  CZ2 TRP    59     -10.372  31.333  89.063  1.00 49.88\n"
		"ATOM   1689  CZ3 TRP    59      -7.970  31.314  88.814  1.00 49.88\n"
		"ATOM   1690  CH2 TRP    59      -9.171  30.699  89.203  1.00 49.88\n"
		"ATOM   1691  H   TRP    59      -7.073  37.611  86.496  1.00 50.31\n"
		"ATOM   1692  HE1 TRP    59     -12.381  33.280  88.440  1.00 50.38\n"
		"ATOM   1693  HD1 TRP    59     -11.370  35.554  87.363  1.00 50.12\n"
		"ATOM   1694  HZ2 TRP    59     -11.326  30.893  89.354  1.00 50.12\n"
		"ATOM   1695  HH2 TRP    59      -9.142  29.695  89.626  1.00 50.12\n"
		"ATOM   1696  HZ3 TRP    59      -7.016  30.800  88.932  1.00 50.12\n"
		"ATOM   1697  HE3 TRP    59      -7.000  33.024  87.988  1.00 50.12\n"
		"ATOM   1698  HA  TRP    59      -9.252  37.156  88.276  1.00 50.09\n"
		"ATOM   1699 1HB  TRP    59      -9.113  35.983  86.114  1.00 50.09\n"
		"ATOM   1700 2HB  TRP    59      -7.702  35.072  86.705  1.00 50.09\n"
		"ATOM   1701  N   ILE    60      -6.366  35.987  89.259  1.00 49.53\n"
		"ATOM   1702  CA  ILE    60      -5.731  35.400  90.422  1.00 50.07\n"
		"ATOM   1703  C   ILE    60      -5.867  36.305  91.637  1.00 50.51\n"
		"ATOM   1704  O   ILE    60      -6.307  35.859  92.684  1.00 49.49\n"
		"ATOM   1705  CB  ILE    60      -4.239  35.121  90.193  1.00 49.91\n"
		"ATOM   1706  CG1 ILE    60      -4.066  33.948  89.214  1.00 49.82\n"
		"ATOM   1707  CG2 ILE    60      -3.582  34.802  91.529  1.00 49.73\n"
		"ATOM   1708  CD1 ILE    60      -2.645  33.725  88.744  1.00 49.73\n"
		"ATOM   1709  H   ILE    60      -5.807  36.308  88.482  1.00 50.31\n"
		"ATOM   1710  HA  ILE    60      -6.232  34.476  90.708  1.00 50.09\n"
		"ATOM   1711  HB  ILE    60      -3.782  35.993  89.726  1.00 50.09\n"
		"ATOM   1712 1HG2 ILE    60      -2.522  34.602  91.375  1.00 50.09\n"
		"ATOM   1713 2HG2 ILE    60      -3.696  35.650  92.202  1.00 50.09\n"
		"ATOM   1714 3HG2 ILE    60      -4.056  33.924  91.968  1.00 50.09\n"
		"ATOM   1715 1HG1 ILE    60      -4.420  33.050  89.718  1.00 50.09\n"
		"ATOM   1716 2HG1 ILE    60      -4.699  34.151  88.350  1.00 50.09\n"
		"ATOM   1717 1HD1 ILE    60      -2.617  32.879  88.057  1.00 50.09\n"
		"ATOM   1718 2HD1 ILE    60      -2.285  34.619  88.233  1.00 50.09\n"
		"ATOM   1719 3HD1 ILE    60      -2.006  33.516  89.601  1.00 50.09\n"
		"ATOM   1720  N   GLU    61      -5.495  37.573  91.501  1.00 49.53\n"
		"ATOM   1721  CA  GLU    61      -5.566  38.475  92.638  1.00 50.07\n"
		"ATOM   1722  C   GLU    61      -7.019  38.636  93.045  1.00 50.51\n"
		"ATOM   1723  O   GLU    61      -7.358  38.884  94.203  1.00 49.49\n"
		"ATOM   1724  CB  GLU    61      -4.954  39.832  92.269  1.00 49.82\n"
		"ATOM   1725  CG  GLU    61      -3.662  40.227  92.990  1.00 49.72\n"
		"ATOM   1726  CD  GLU    61      -2.468  39.361  92.642  1.00 50.62\n"
		"ATOM   1727  OE1 GLU    61      -2.071  39.280  91.445  1.00 49.24\n"
		"ATOM   1728  OE2 GLU    61      -1.920  38.765  93.564  1.00 49.24\n"
		"ATOM   1729  H   GLU    61      -5.163  37.910  90.608  1.00 50.31\n"
		"ATOM   1730  HA  GLU    61      -5.031  38.045  93.485  1.00 50.09\n"
		"ATOM   1731 1HB  GLU    61      -4.752  39.810  91.198  1.00 50.09\n"
		"ATOM   1732 2HB  GLU    61      -5.708  40.590  92.481  1.00 50.09\n"
		"ATOM   1733 1HG  GLU    61      -3.376  41.266  92.829  1.00 50.09\n"
		"ATOM   1734 2HG  GLU    61      -3.899  40.075  94.042  1.00 50.09\n"
		"ATOM   1735  N   ASP    62      -7.880  38.492  92.083  1.00 49.53\n"
		"ATOM   1736  CA  ASP    62      -9.309  38.541  92.343  1.00 50.07\n"
		"ATOM   1737  C   ASP    62      -9.760  37.212  92.958  1.00 50.51\n"
		"ATOM   1738  O   ASP    62     -10.950  36.913  93.029  1.00 49.49\n"
		"ATOM   1739  CB  ASP    62     -10.045  38.809  91.021  1.00 49.72\n"
		"ATOM   1740  CG  ASP    62     -10.473  40.270  90.876  1.00 50.62\n"
		"ATOM   1741  OD1 ASP    62     -10.859  40.885  91.904  1.00 49.24\n"
		"ATOM   1742  OD2 ASP    62     -10.429  40.805  89.735  1.00 49.24\n"
		"ATOM   1743  H   ASP    62      -7.551  38.344  91.140  1.00 50.31\n"
		"ATOM   1744  HA  ASP    62      -9.526  39.326  93.068  1.00 50.09\n"
		"ATOM   1745 1HB  ASP    62      -9.491  38.482  90.141  1.00 50.09\n"
		"ATOM   1746 2HB  ASP    62     -10.920  38.173  91.157  1.00 50.09\n"
		"ATOM   1747  N   ASN    63      -8.790  36.416  93.443  1.00 49.53\n"
		"ATOM   1748  CA  ASN    63      -9.118  35.122  94.007  1.00 50.07\n"
		"ATOM   1749  C   ASN    63      -8.308  34.728  95.263  1.00 50.51\n"
		"ATOM   1750  O   ASN    63      -8.874  34.790  96.361  1.00 49.49\n"
		"ATOM   1751  CB  ASN    63      -9.043  34.036  92.934  1.00 49.82\n"
		"ATOM   1752  CG  ASN    63     -10.388  33.780  92.254  1.00 50.55\n"
		"ATOM   1753  OD1 ASN    63     -11.292  33.172  92.844  1.00 49.45\n"
		"ATOM   1754  ND2 ASN    63     -10.532  34.242  91.012  1.00 49.38\n"
		"ATOM   1755  H   ASN    63      -7.828  36.722  93.414  1.00 50.31\n"
		"ATOM   1756 1HD2 ASN    63     -11.393  34.101  90.523  1.00 50.32\n"
		"ATOM   1757 2HD2 ASN    63      -9.780  34.730  90.569  1.00 50.30\n"
		"ATOM   1758  HA  ASN    63     -10.157  35.116  94.340  1.00 50.09\n"
		"ATOM   1759 1HB  ASN    63      -8.247  34.042  92.188  1.00 50.09\n"
		"ATOM   1760 2HB  ASN    63      -8.856  33.244  93.659  1.00 50.09\n"
		"ATOM   1761  N   LEU    64      -7.003  34.350  95.141  1.00 49.53\n"
		"ATOM   1762  CA  LEU    64      -6.217  33.806  96.304  1.00 50.07\n"
		"ATOM   1763  C   LEU    64      -6.992  33.866  97.582  1.00 50.51\n"
		"ATOM   1764  O   LEU    64      -7.299  32.833  98.164  1.00 49.49\n"
		"ATOM   1765  CB  LEU    64      -4.852  34.501  96.542  1.00 49.82\n"
		"ATOM   1766  CG  LEU    64      -4.570  35.855  95.916  1.00 49.91\n"
		"ATOM   1767  CD1 LEU    64      -4.025  36.780  96.993  1.00 49.73\n"
		"ATOM   1768  CD2 LEU    64      -3.542  35.707  94.823  1.00 49.73\n"
		"ATOM   1769  H   LEU    64      -6.550  34.438  94.243  1.00 50.31\n"
		"ATOM   1770  HA  LEU    64      -5.996  32.753  96.134  1.00 50.09\n"
		"ATOM   1771 1HB  LEU    64      -4.904  34.629  97.622  1.00 50.09\n"
		"ATOM   1772 2HB  LEU    64      -4.017  33.846  96.294  1.00 50.09\n"
		"ATOM   1773  HG  LEU    64      -5.509  36.221  95.499  1.00 50.09\n"
		"ATOM   1774 1HD1 LEU    64      -3.816  37.759  96.561  1.00 50.09\n"
		"ATOM   1775 2HD1 LEU    64      -4.762  36.885  97.789  1.00 50.09\n"
		"ATOM   1776 3HD1 LEU    64      -3.106  36.361  97.401  1.00 50.09\n"
		"ATOM   1777 1HD2 LEU    64      -3.344  36.682  94.376  1.00 50.09\n"
		"ATOM   1778 2HD2 LEU    64      -2.620  35.305  95.242  1.00 50.09\n"
		"ATOM   1779 3HD2 LEU    64      -3.919  35.028  94.058  1.00 50.09\n"
		"ATOM   1780  N   GLU    65      -7.322  35.098  97.965  1.00 49.53\n"
		"ATOM   1781  CA  GLU    65      -8.069  35.444  99.176  1.00 50.07\n"
		"ATOM   1782  C   GLU    65      -9.086  34.406  99.626  1.00 50.51\n"
		"ATOM   1783  O   GLU    65      -9.742  34.595 100.630  1.00 49.49\n"
		"ATOM   1784  CB  GLU    65      -8.744  36.813  99.025  1.00 49.82\n"
		"ATOM   1785  CG  GLU    65      -8.250  37.846 100.037  1.00 49.72\n"
		"ATOM   1786  CD  GLU    65      -9.095  37.867 101.296  1.00 50.62\n"
		"ATOM   1787  OE1 GLU    65     -10.008  37.019 101.401  1.00 49.24\n"
		"ATOM   1788  OE2 GLU    65      -8.852  38.730 102.174  1.00 49.24\n"
		"ATOM   1789  H   GLU    65      -7.032  35.860  97.369  1.00 50.31\n"
		"ATOM   1790  HA  GLU    65      -7.384  35.539 100.019  1.00 50.09\n"
		"ATOM   1791 1HB  GLU    65      -8.540  37.158  98.011  1.00 50.09\n"
		"ATOM   1792 2HB  GLU    65      -9.816  36.655  99.147  1.00 50.09\n"
		"ATOM   1793 1HG  GLU    65      -7.205  37.714 100.317  1.00 50.09\n"
		"ATOM   1794 2HG  GLU    65      -8.368  38.794  99.514  1.00 50.09\n"
		"ATOM   1795  N   SER    66      -9.243  33.333  98.901  1.00 49.53\n"
		"ATOM   1796  CA  SER    66     -10.067  32.262  99.383  1.00 50.07\n"
		"ATOM   1797  C   SER    66      -9.822  30.990  98.618  1.00 50.51\n"
		"ATOM   1798  O   SER    66      -9.717  29.913  99.189  1.00 49.49\n"
		"ATOM   1799  CB  SER    66     -11.559  32.633  99.328  1.00 50.05\n"
		"ATOM   1800  OG  SER    66     -12.253  32.160 100.479  1.00 49.34\n"
		"ATOM   1801  H   SER    66      -8.786  33.256  98.004  1.00 50.31\n"
		"ATOM   1802  HG  SER    66     -11.869  32.552 101.267  1.00 50.43\n"
		"ATOM   1803  HA  SER    66      -9.954  32.076 100.452  1.00 50.09\n"
		"ATOM   1804 1HB  SER    66     -11.646  33.718  99.274  1.00 50.09\n"
		"ATOM   1805 2HB  SER    66     -11.997  32.190  98.435  1.00 50.09\n"
		"ATOM   1806  N   PRO    67      -9.651  31.074  97.320  1.00 49.71\n"
		"ATOM   1807  CA  PRO    67      -9.453  29.902  96.511  1.00 50.02\n"
		"ATOM   1808  C   PRO    67      -8.188  29.194  96.819  1.00 50.51\n"
		"ATOM   1809  O   PRO    67      -7.238  29.786  97.327  1.00 49.49\n"
		"ATOM   1810  CB  PRO    67      -9.427  30.417  95.080  1.00 49.82\n"
		"ATOM   1811  CG  PRO    67      -9.000  31.827  95.204  1.00 49.82\n"
		"ATOM   1812  CD  PRO    67      -9.563  32.300  96.518  1.00 50.00\n"
		"ATOM   1813 1HD  PRO    67      -8.959  32.854  97.005  1.00 50.09\n"
		"ATOM   1814 2HD  PRO    67     -10.471  32.581  96.457  1.00 50.09\n"
		"ATOM   1815 1HG  PRO    67      -8.172  31.843  95.381  1.00 50.09\n"
		"ATOM   1816 2HG  PRO    67      -9.527  32.333  94.617  1.00 50.09\n"
		"ATOM   1817 1HB  PRO    67      -8.714  29.952  94.745  1.00 50.09\n"
		"ATOM   1818 2HB  PRO    67     -10.271  30.412  94.886  1.00 50.09\n"
		"ATOM   1819  HA  PRO    67     -10.199  29.257  96.654  1.00 50.09\n"
		"ATOM   1820  N   LEU    68      -8.208  27.928  96.496  1.00 49.53\n"
		"ATOM   1821  CA  LEU    68      -7.036  27.110  96.542  1.00 50.07\n"
		"ATOM   1822  C   LEU    68      -6.231  27.367  95.275  1.00 50.51\n"
		"ATOM   1823  O   LEU    68      -5.145  26.826  95.101  1.00 49.49\n"
		"ATOM   1824  CB  LEU    68      -7.421  25.629  96.640  1.00 49.82\n"
		"ATOM   1825  CG  LEU    68      -8.788  25.313  97.268  1.00 49.91\n"
		"ATOM   1826  CD1 LEU    68      -8.828  23.858  97.742  1.00 49.73\n"
		"ATOM   1827  CD2 LEU    68      -9.085  26.261  98.424  1.00 49.73\n"
		"ATOM   1828  H   LEU    68      -9.081  27.513  96.204  1.00 50.31\n"
		"ATOM   1829  HA  LEU    68      -6.423  27.399  97.395  1.00 50.09\n"
		"ATOM   1830 1HB  LEU    68      -7.441  25.381  95.580  1.00 50.09\n"
		"ATOM   1831 2HB  LEU    68      -6.647  25.043  97.136  1.00 50.09\n"
		"ATOM   1832  HG  LEU    68      -9.545  25.501  96.507  1.00 50.09\n"
		"ATOM   1833 1HD1 LEU    68      -9.801  23.646  98.185  1.00 50.09\n"
		"ATOM   1834 2HD1 LEU    68      -8.664  23.194  96.893  1.00 50.09\n"
		"ATOM   1835 3HD1 LEU    68      -8.049  23.696  98.486  1.00 50.09\n"
		"ATOM   1836 1HD2 LEU    68     -10.058  26.018  98.852  1.00 50.09\n"
		"ATOM   1837 2HD2 LEU    68      -8.315  26.156  99.189  1.00 50.09\n"
		"ATOM   1838 3HD2 LEU    68      -9.096  27.288  98.059  1.00 50.09\n"
		"ATOM   1839  N   SER    69      -6.761  28.204  94.401  1.00 49.53\n"
		"ATOM   1840  CA  SER    69      -6.056  28.547  93.160  1.00 50.07\n"
		"ATOM   1841  C   SER    69      -4.901  27.574  92.780  1.00 50.51\n"
		"ATOM   1842  O   SER    69      -3.756  27.977  92.642  1.00 49.49\n"
		"ATOM   1843  CB  SER    69      -5.523  29.978  93.243  1.00 50.05\n"
		"ATOM   1844  OG  SER    69      -6.227  30.803  92.334  1.00 49.34\n"
		"ATOM   1845  H   SER    69      -7.665  28.614  94.589  1.00 50.31\n"
		"ATOM   1846  HG  SER    69      -5.889  31.700  92.389  1.00 50.43\n"
		"ATOM   1847  HA  SER    69      -6.717  28.635  92.297  1.00 50.09\n"
		"ATOM   1848 1HB  SER    69      -5.658  30.341  94.262  1.00 50.09\n"
		"ATOM   1849 2HB  SER    69      -4.462  29.970  92.999  1.00 50.09\n"
		"ATOM   1850  N   LEU    70      -5.226  26.297  92.596  1.00 49.53\n"
		"ATOM   1851  CA  LEU    70      -4.238  25.315  92.173  1.00 50.07\n"
		"ATOM   1852  C   LEU    70      -4.056  25.388  90.672  1.00 50.51\n"
		"ATOM   1853  O   LEU    70      -4.921  25.875  89.981  1.00 49.49\n"
		"ATOM   1854  CB  LEU    70      -4.694  23.922  92.509  1.00 49.82\n"
		"ATOM   1855  CG  LEU    70      -6.080  23.572  92.014  1.00 49.91\n"
		"ATOM   1856  CD1 LEU    70      -6.113  22.119  91.581  1.00 49.73\n"
		"ATOM   1857  CD2 LEU    70      -7.133  23.847  93.100  1.00 49.73\n"
		"ATOM   1858  H   LEU    70      -6.179  26.002  92.753  1.00 50.31\n"
		"ATOM   1859  HA  LEU    70      -3.275  25.548  92.626  1.00 50.09\n"
		"ATOM   1860 1HB  LEU    70      -3.944  23.353  91.962  1.00 50.09\n"
		"ATOM   1861 2HB  LEU    70      -4.602  23.713  93.575  1.00 50.09\n"
		"ATOM   1862  HG  LEU    70      -6.306  24.242  91.184  1.00 50.09\n"
		"ATOM   1863 1HD1 LEU    70      -7.112  21.868  91.224  1.00 50.09\n"
		"ATOM   1864 2HD1 LEU    70      -5.392  21.962  90.778  1.00 50.09\n"
		"ATOM   1865 3HD1 LEU    70      -5.859  21.482  92.427  1.00 50.09\n"
		"ATOM   1866 1HD2 LEU    70      -8.122  23.588  92.721  1.00 50.09\n"
		"ATOM   1867 2HD2 LEU    70      -6.913  23.245  93.982  1.00 50.09\n"
		"ATOM   1868 3HD2 LEU    70      -7.113  24.903  93.368  1.00 50.09\n"
		"ATOM   1869  N   GLU    71      -2.962  24.851  90.161  1.00 49.53\n"
		"ATOM   1870  CA  GLU    71      -2.651  24.912  88.724  1.00 50.07\n"
		"ATOM   1871  C   GLU    71      -3.840  24.473  87.846  1.00 50.51\n"
		"ATOM   1872  O   GLU    71      -4.072  25.021  86.741  1.00 49.49\n"
		"ATOM   1873  CB  GLU    71      -1.438  24.009  88.483  1.00 49.82\n"
		"ATOM   1874  CG  GLU    71      -0.463  24.515  87.450  1.00 49.72\n"
		"ATOM   1875  CD  GLU    71       0.346  25.637  88.023  1.00 50.62\n"
		"ATOM   1876  OE1 GLU    71      -0.220  26.739  88.165  1.00 49.24\n"
		"ATOM   1877  OE2 GLU    71       1.532  25.411  88.360  1.00 49.24\n"
		"ATOM   1878  H   GLU    71      -2.316  24.382  90.779  1.00 50.31\n"
		"ATOM   1879  HA  GLU    71      -2.431  25.941  88.437  1.00 50.09\n"
		"ATOM   1880 1HB  GLU    71      -0.924  23.905  89.439  1.00 50.09\n"
		"ATOM   1881 2HB  GLU    71      -1.821  23.038  88.168  1.00 50.09\n"
		"ATOM   1882 1HG  GLU    71       0.193  23.739  87.056  1.00 50.09\n"
		"ATOM   1883 2HG  GLU    71      -1.105  24.886  86.652  1.00 50.09\n"
		"ATOM   1884  N   LYS    72      -4.567  23.455  88.327  1.00 49.53\n"
		"ATOM   1885  CA  LYS    72      -5.720  22.948  87.615  1.00 50.07\n"
		"ATOM   1886  C   LYS    72      -6.789  24.024  87.585  1.00 50.51\n"
		"ATOM   1887  O   LYS    72      -7.122  24.526  86.529  1.00 49.49\n"
		"ATOM   1888  CB  LYS    72      -6.247  21.687  88.284  1.00 49.82\n"
		"ATOM   1889  CG  LYS    72      -7.048  20.818  87.352  1.00 49.82\n"
		"ATOM   1890  CD  LYS    72      -6.380  19.471  87.156  1.00 49.82\n"
		"ATOM   1891  CE  LYS    72      -7.126  18.413  87.943  1.00 50.21\n"
		"ATOM   1892  NZ  LYS    72      -8.492  18.900  88.310  1.00 49.70\n"
		"ATOM   1893  H   LYS    72      -4.304  23.032  89.206  1.00 50.31\n"
		"ATOM   1894 1HZ  LYS    72      -8.975  18.184  88.834  1.00 50.33\n"
		"ATOM   1895 2HZ  LYS    72      -8.412  19.734  88.874  1.00 50.33\n"
		"ATOM   1896 3HZ  LYS    72      -9.011  19.111  87.470  1.00 50.33\n"
		"ATOM   1897  HA  LYS    72      -5.447  22.723  86.583  1.00 50.09\n"
		"ATOM   1898 1HB  LYS    72      -5.382  21.139  88.658  1.00 50.09\n"
		"ATOM   1899 2HB  LYS    72      -6.865  22.005  89.124  1.00 50.09\n"
		"ATOM   1900 1HG  LYS    72      -8.042  20.676  87.779  1.00 50.09\n"
		"ATOM   1901 2HG  LYS    72      -7.135  21.327  86.393  1.00 50.09\n"
		"ATOM   1902 1HD  LYS    72      -6.390  19.230  86.092  1.00 50.09\n"
		"ATOM   1903 2HD  LYS    72      -5.349  19.541  87.501  1.00 50.09\n"
		"ATOM   1904 1HE  LYS    72      -7.199  17.517  87.328  1.00 50.05\n"
		"ATOM   1905 2HE  LYS    72      -6.553  18.189  88.843  1.00 50.05\n"
		"ATOM   1906  N   VAL    73      -7.312  24.398  88.749  1.00 49.53\n"
		"ATOM   1907  CA  VAL    73      -8.350  25.442  88.813  1.00 50.07\n"
		"ATOM   1908  C   VAL    73      -7.917  26.651  87.998  1.00 50.51\n"
		"ATOM   1909  O   VAL    73      -8.727  27.333  87.370  1.00 49.49\n"
		"ATOM   1910  CB  VAL    73      -8.655  25.868  90.292  1.00 49.91\n"
		"ATOM   1911  CG1 VAL    73      -9.375  27.207  90.366  1.00 49.73\n"
		"ATOM   1912  CG2 VAL    73      -9.514  24.804  90.975  1.00 49.73\n"
		"ATOM   1913  H   VAL    73      -6.993  23.960  89.602  1.00 50.31\n"
		"ATOM   1914  HA  VAL    73      -9.282  25.103  88.360  1.00 50.09\n"
		"ATOM   1915  HB  VAL    73      -7.715  25.947  90.839  1.00 50.09\n"
		"ATOM   1916 1HG1 VAL    73      -9.565  27.461  91.409  1.00 50.09\n"
		"ATOM   1917 2HG1 VAL    73      -8.754  27.979  89.912  1.00 50.09\n"
		"ATOM   1918 3HG1 VAL    73     -10.322  27.141  89.831  1.00 50.09\n"
		"ATOM   1919 1HG2 VAL    73      -9.721  25.106  92.001  1.00 50.09\n"
		"ATOM   1920 2HG2 VAL    73     -10.453  24.693  90.433  1.00 50.09\n"
		"ATOM   1921 3HG2 VAL    73      -8.981  23.853  90.978  1.00 50.09\n"
		"ATOM   1922  N   SER    74      -6.631  26.905  87.977  1.00 49.53\n"
		"ATOM   1923  CA  SER    74      -6.139  28.037  87.226  1.00 50.07\n"
		"ATOM   1924  C   SER    74      -6.131  27.689  85.747  1.00 50.51\n"
		"ATOM   1925  O   SER    74      -6.307  28.562  84.885  1.00 49.49\n"
		"ATOM   1926  CB  SER    74      -4.751  28.381  87.721  1.00 50.05\n"
		"ATOM   1927  OG  SER    74      -4.488  29.734  87.480  1.00 49.34\n"
		"ATOM   1928  H   SER    74      -5.988  26.313  88.483  1.00 50.31\n"
		"ATOM   1929  HG  SER    74      -3.607  29.949  87.795  1.00 50.43\n"
		"ATOM   1930  HA  SER    74      -6.699  28.955  87.410  1.00 50.09\n"
		"ATOM   1931 1HB  SER    74      -4.701  28.172  88.789  1.00 50.09\n"
		"ATOM   1932 2HB  SER    74      -4.028  27.758  87.196  1.00 50.09\n"
		"ATOM   1933  N   GLU    75      -5.938  26.393  85.474  1.00 49.53\n"
		"ATOM   1934  CA  GLU    75      -5.984  25.868  84.118  1.00 50.07\n"
		"ATOM   1935  C   GLU    75      -7.320  26.235  83.458  1.00 50.51\n"
		"ATOM   1936  O   GLU    75      -7.320  26.861  82.413  1.00 49.49\n"
		"ATOM   1937  CB  GLU    75      -5.811  24.340  84.134  1.00 49.82\n"
		"ATOM   1938  CG  GLU    75      -6.734  23.592  83.151  1.00 49.72\n"
		"ATOM   1939  CD  GLU    75      -7.507  22.395  83.747  1.00 50.62\n"
		"ATOM   1940  OE1 GLU    75      -6.925  21.288  83.832  1.00 49.24\n"
		"ATOM   1941  OE2 GLU    75      -8.707  22.554  84.103  1.00 49.24\n"
		"ATOM   1942  H   GLU    75      -5.754  25.757  86.237  1.00 50.31\n"
		"ATOM   1943  HA  GLU    75      -5.195  26.323  83.518  1.00 50.09\n"
		"ATOM   1944 1HB  GLU    75      -4.770  24.135  83.885  1.00 50.09\n"
		"ATOM   1945 2HB  GLU    75      -6.014  24.007  85.152  1.00 50.09\n"
		"ATOM   1946 1HG  GLU    75      -7.469  24.237  82.670  1.00 50.09\n"
		"ATOM   1947 2HG  GLU    75      -6.047  23.203  82.400  1.00 50.09\n"
		"ATOM   1948  N   ARG    76      -8.447  25.836  84.110  1.00 49.53\n"
		"ATOM   1949  CA  ARG    76      -9.820  26.077  83.627  1.00 50.07\n"
		"ATOM   1950  C   ARG    76      -9.881  27.308  82.782  1.00 50.51\n"
		"ATOM   1951  O   ARG    76     -10.401  27.301  81.680  1.00 49.49\n"
		"ATOM   1952  CB  ARG    76     -10.806  26.278  84.783  1.00 49.82\n"
		"ATOM   1953  CG  ARG    76     -11.022  27.748  85.191  1.00 49.82\n"
		"ATOM   1954  CD  ARG    76     -12.354  27.983  85.903  1.00 50.20\n"
		"ATOM   1955  NE  ARG    76     -12.372  27.454  87.283  1.00 49.30\n"
		"ATOM   1956  CZ  ARG    76     -13.007  28.028  88.332  1.00 50.64\n"
		"ATOM   1957  NH1 ARG    76     -13.685  29.174  88.170  1.00 49.20\n"
		"ATOM   1958  NH2 ARG    76     -12.955  27.461  89.538  1.00 49.20\n"
		"ATOM   1959  H   ARG    76      -8.335  25.342  84.984  1.00 50.31\n"
		"ATOM   1960 1HH1 ARG    76     -13.723  29.611  87.260  1.00 50.46\n"
		"ATOM   1961 2HH1 ARG    76     -14.155  29.595  88.958  1.00 50.46\n"
		"ATOM   1962 1HH2 ARG    76     -12.440  26.601  89.667  1.00 50.46\n"
		"ATOM   1963 2HH2 ARG    76     -13.429  27.893  90.317  1.00 50.46\n"
		"ATOM   1964  HE  ARG    76     -11.939  26.621  87.660  1.00 50.44\n"
		"ATOM   1965  HA  ARG    76     -10.140  25.248  82.995  1.00 50.09\n"
		"ATOM   1966 1HB  ARG    76     -11.755  25.845  84.470  1.00 50.09\n"
		"ATOM   1967 2HB  ARG    76     -10.415  25.720  85.634  1.00 50.09\n"
		"ATOM   1968 1HG  ARG    76     -10.214  28.045  85.860  1.00 50.09\n"
		"ATOM   1969 2HG  ARG    76     -10.994  28.364  84.292  1.00 50.09\n"
		"ATOM   1970 1HD  ARG    76     -12.548  29.054  85.947  1.00 50.09\n"
		"ATOM   1971 2HD  ARG    76     -13.148  27.493  85.340  1.00 50.09\n"
		"ATOM   1972  N   SER    77      -9.376  28.378  83.293  1.00 49.53\n"
		"ATOM   1973  CA  SER    77      -9.383  29.551  82.511  1.00 50.07\n"
		"ATOM   1974  C   SER    77      -8.232  29.533  81.483  1.00 50.51\n"
		"ATOM   1975  O   SER    77      -7.062  29.764  81.806  1.00 49.49\n"
		"ATOM   1976  CB  SER    77      -9.358  30.706  83.477  1.00 50.05\n"
		"ATOM   1977  OG  SER    77     -10.324  30.446  84.477  1.00 49.34\n"
		"ATOM   1978  H   SER    77      -8.987  28.376  84.225  1.00 50.31\n"
		"ATOM   1979  HG  SER    77     -10.331  31.168  85.110  1.00 50.43\n"
		"ATOM   1980  HA  SER    77     -10.334  29.729  82.006  1.00 50.09\n"
		"ATOM   1981 1HB  SER    77      -8.358  30.781  83.904  1.00 50.09\n"
		"ATOM   1982 2HB  SER    77      -9.589  31.621  82.933  1.00 50.09\n"
		"ATOM   1983  N   GLY    78      -8.576  29.184  80.234  1.00 49.53\n"
		"ATOM   1984  CA  GLY    78      -7.579  29.159  79.194  1.00 49.98\n"
		"ATOM   1985  C   GLY    78      -7.272  27.763  78.664  1.00 50.51\n"
		"ATOM   1986  O   GLY    78      -8.173  26.954  78.410  1.00 49.49\n"
		"ATOM   1987  H   GLY    78      -9.533  28.939  80.023  1.00 50.31\n"
		"ATOM   1988 1HA  GLY    78      -7.925  29.762  78.354  1.00 50.09\n"
		"ATOM   1989 2HA  GLY    78      -6.650  29.576  79.581  1.00 50.09\n"
		"ATOM   1990  N   TYR    79      -5.978  27.483  78.542  1.00 49.53\n"
		"ATOM   1991  CA  TYR    79      -5.468  26.247  77.973  1.00 50.07\n"
		"ATOM   1992  C   TYR    79      -5.170  25.164  79.020  1.00 50.51\n"
		"ATOM   1993  O   TYR    79      -5.774  25.163  80.100  1.00 49.49\n"
		"ATOM   1994  CB  TYR    79      -4.198  26.582  77.211  1.00 49.82\n"
		"ATOM   1995  CG  TYR    79      -4.281  27.874  76.440  1.00 50.00\n"
		"ATOM   1996  CD1 TYR    79      -4.537  29.094  77.073  1.00 49.88\n"
		"ATOM   1997  CD2 TYR    79      -4.065  27.874  75.064  1.00 49.88\n"
		"ATOM   1998  CE1 TYR    79      -4.614  30.274  76.339  1.00 49.88\n"
		"ATOM   1999  CE2 TYR    79      -4.140  29.053  74.326  1.00 49.88\n"
		"ATOM   2000  CZ  TYR    79      -4.400  30.255  74.967  1.00 50.11\n"
		"ATOM   2001  OH  TYR    79      -4.511  31.421  74.230  1.00 49.46\n"
		"ATOM   2002  H   TYR    79      -5.311  28.169  78.864  1.00 50.31\n"
		"ATOM   2003  HH  TYR    79      -4.391  31.282  73.287  1.00 50.43\n"
		"ATOM   2004  HD1 TYR    79      -4.677  29.106  78.154  1.00 50.12\n"
		"ATOM   2005  HE1 TYR    79      -4.841  31.217  76.835  1.00 50.12\n"
		"ATOM   2006  HE2 TYR    79      -3.995  29.030  73.246  1.00 50.12\n"
		"ATOM   2007  HD2 TYR    79      -3.835  26.931  74.567  1.00 50.12\n"
		"ATOM   2008  HA  TYR    79      -6.214  25.809  77.310  1.00 50.09\n"
		"ATOM   2009 1HB  TYR    79      -3.390  26.642  77.941  1.00 50.09\n"
		"ATOM   2010 2HB  TYR    79      -4.007  25.757  76.525  1.00 50.09\n"
		"ATOM   2011  N   SER    80      -4.233  24.237  78.666  1.00 49.53\n"
		"ATOM   2012  CA  SER    80      -3.769  23.115  79.519  1.00 50.07\n"
		"ATOM   2013  C   SER    80      -2.790  23.622  80.552  1.00 50.51\n"
		"ATOM   2014  O   SER    80      -2.050  24.589  80.304  1.00 49.49\n"
		"ATOM   2015  CB  SER    80      -3.058  22.051  78.673  1.00 50.05\n"
		"ATOM   2016  OG  SER    80      -1.808  22.551  78.205  1.00 49.34\n"
		"ATOM   2017  H   SER    80      -3.821  24.322  77.748  1.00 50.31\n"
		"ATOM   2018  HG  SER    80      -1.958  23.333  77.668  1.00 50.43\n"
		"ATOM   2019  HA  SER    80      -4.583  22.569  79.997  1.00 50.09\n"
		"ATOM   2020 1HB  SER    80      -2.898  21.167  79.289  1.00 50.09\n"
		"ATOM   2021 2HB  SER    80      -3.695  21.793  77.828  1.00 50.09\n"
		"ATOM   2022  N   LYS    81      -2.737  22.957  81.687  1.00 49.53\n"
		"ATOM   2023  CA  LYS    81      -1.908  23.455  82.749  1.00 50.07\n"
		"ATOM   2024  C   LYS    81      -0.489  23.682  82.282  1.00 50.51\n"
		"ATOM   2025  O   LYS    81       0.085  24.714  82.578  1.00 49.49\n"
		"ATOM   2026  CB  LYS    81      -1.918  22.501  83.903  1.00 49.82\n"
		"ATOM   2027  CG  LYS    81      -1.565  21.101  83.513  1.00 49.82\n"
		"ATOM   2028  CD  LYS    81      -1.543  20.213  84.743  1.00 49.82\n"
		"ATOM   2029  CE  LYS    81      -2.975  19.894  85.145  1.00 50.21\n"
		"ATOM   2030  NZ  LYS    81      -3.789  19.550  83.977  1.00 49.70\n"
		"ATOM   2031  H   LYS    81      -3.272  22.109  81.809  1.00 50.31\n"
		"ATOM   2032 1HZ  LYS    81      -4.733  19.344  84.273  1.00 50.33\n"
		"ATOM   2033 2HZ  LYS    81      -3.801  20.329  83.333  1.00 50.33\n"
		"ATOM   2034 3HZ  LYS    81      -3.395  18.740  83.519  1.00 50.33\n"
		"ATOM   2035  HA  LYS    81      -2.281  24.423  83.085  1.00 50.09\n"
		"ATOM   2036 1HB  LYS    81      -1.202  22.877  84.634  1.00 50.09\n"
		"ATOM   2037 2HB  LYS    81      -2.920  22.527  84.331  1.00 50.09\n"
		"ATOM   2038 1HG  LYS    81      -2.310  20.742  82.802  1.00 50.09\n"
		"ATOM   2039 2HG  LYS    81      -0.585  21.111  83.038  1.00 50.09\n"
		"ATOM   2040 1HD  LYS    81      -0.997  19.300  84.502  1.00 50.09\n"
		"ATOM   2041 2HD  LYS    81      -1.028  20.742  85.544  1.00 50.09\n"
		"ATOM   2042 1HE  LYS    81      -2.956  19.057  85.842  1.00 50.05\n"
		"ATOM   2043 2HE  LYS    81      -3.393  20.770  85.642  1.00 50.05\n"
		"ATOM   2044  N   TRP    82       0.041  22.728  81.517  1.00 49.53\n"
		"ATOM   2045  CA  TRP    82       1.403  22.793  81.024  1.00 50.07\n"
		"ATOM   2046  C   TRP    82       1.617  24.006  80.121  1.00 50.51\n"
		"ATOM   2047  O   TRP    82       2.671  24.648  80.158  1.00 49.49\n"
		"ATOM   2048  CB  TRP    82       1.711  21.507  80.270  1.00 49.82\n"
		"ATOM   2049  CG  TRP    82       1.392  20.253  81.037  1.00 49.97\n"
		"ATOM   2050  CD1 TRP    82       2.139  19.715  82.044  1.00 50.03\n"
		"ATOM   2051  CD2 TRP    82       0.309  19.304  80.784  1.00 49.98\n"
		"ATOM   2052  NE1 TRP    82       1.610  18.499  82.437  1.00 49.39\n"
		"ATOM   2053  CE2 TRP    82       0.493  18.218  81.679  1.00 50.13\n"
		"ATOM   2054  CE3 TRP    82      -0.792  19.256  79.895  1.00 49.88\n"
		"ATOM   2055  CZ2 TRP    82      -0.372  17.113  81.721  1.00 49.88\n"
		"ATOM   2056  CZ3 TRP    82      -1.644  18.143  79.931  1.00 49.88\n"
		"ATOM   2057  CH2 TRP    82      -1.431  17.095  80.843  1.00 49.88\n"
		"ATOM   2058  H   TRP    82      -0.527  21.930  81.272  1.00 50.31\n"
		"ATOM   2059  HE1 TRP    82       1.980  17.904  83.165  1.00 50.38\n"
		"ATOM   2060  HD1 TRP    82       3.001  20.289  82.378  1.00 50.12\n"
		"ATOM   2061  HZ2 TRP    82      -0.176  16.318  82.441  1.00 50.12\n"
		"ATOM   2062  HH2 TRP    82      -2.120  16.250  80.853  1.00 50.12\n"
		"ATOM   2063  HZ3 TRP    82      -2.478  18.114  79.229  1.00 50.12\n"
		"ATOM   2064  HE3 TRP    82      -1.013  20.048  79.181  1.00 50.12\n"
		"ATOM   2065  HA  TRP    82       2.094  22.905  81.860  1.00 50.09\n"
		"ATOM   2066 1HB  TRP    82       1.128  21.460  79.350  1.00 50.09\n"
		"ATOM   2067 2HB  TRP    82       2.773  21.458  80.030  1.00 50.09\n"
		"ATOM   2068  N   HIS    83       0.618  24.326  79.312  1.00 49.53\n"
		"ATOM   2069  CA  HIS    83       0.721  25.471  78.403  1.00 50.07\n"
		"ATOM   2070  C   HIS    83       0.637  26.798  79.141  1.00 50.51\n"
		"ATOM   2071  O   HIS    83       1.546  27.622  79.035  1.00 49.49\n"
		"ATOM   2072  CB  HIS    83      -0.373  25.401  77.328  1.00 49.92\n"
		"ATOM   2073  CG  HIS    83      -0.225  26.481  76.274  1.00 50.22\n"
		"ATOM   2074  ND1 HIS    83       0.927  26.647  75.520  1.00 49.30\n"
		"ATOM   2075  CD2 HIS    83      -1.075  27.450  75.867  1.00 49.95\n"
		"ATOM   2076  CE1 HIS    83       0.774  27.670  74.708  1.00 50.25\n"
		"ATOM   2077  NE2 HIS    83      -0.441  28.172  74.903  1.00 49.64\n"
		"ATOM   2078  H   HIS    83      -0.227  23.773  79.322  1.00 50.31\n"
		"ATOM   2079  HE2 HIS    83      -0.838  28.963  74.417  1.00 50.32\n"
		"ATOM   2080  HA  HIS    83       1.693  25.461  77.910  1.00 50.09\n"
		"ATOM   2081 1HB  HIS    83      -0.333  24.441  76.812  1.00 50.09\n"
		"ATOM   2082 2HB  HIS    83      -1.355  25.527  77.783  1.00 50.09\n"
		"ATOM   2083  HE1 HIS    83       1.576  27.966  74.032  1.00 50.13\n"
		"ATOM   2084  HD2 HIS    83      -2.090  27.717  76.163  1.00 50.09\n"
		"ATOM   2085  N   LEU    84      -0.485  27.015  79.840  1.00 49.53\n"
		"ATOM   2086  CA  LEU    84      -0.656  28.197  80.681  1.00 50.07\n"
		"ATOM   2087  C   LEU    84       0.649  28.478  81.407  1.00 50.51\n"
		"ATOM   2088  O   LEU    84       1.092  29.605  81.502  1.00 49.49\n"
		"ATOM   2089  CB  LEU    84      -1.756  27.952  81.708  1.00 49.82\n"
		"ATOM   2090  CG  LEU    84      -2.153  29.146  82.577  1.00 49.91\n"
		"ATOM   2091  CD1 LEU    84      -2.267  30.394  81.727  1.00 49.73\n"
		"ATOM   2092  CD2 LEU    84      -3.463  28.851  83.293  1.00 49.73\n"
		"ATOM   2093  H   LEU    84      -1.235  26.341  79.784  1.00 50.31\n"
		"ATOM   2094  HA  LEU    84      -0.877  29.061  80.056  1.00 50.09\n"
		"ATOM   2095 1HB  LEU    84      -2.574  27.702  81.035  1.00 50.09\n"
		"ATOM   2096 2HB  LEU    84      -1.532  27.094  82.342  1.00 50.09\n"
		"ATOM   2097  HG  LEU    84      -1.386  29.258  83.343  1.00 50.09\n"
		"ATOM   2098 1HD1 LEU    84      -2.550  31.238  82.356  1.00 50.09\n"
		"ATOM   2099 2HD1 LEU    84      -1.307  30.601  81.254  1.00 50.09\n"
		"ATOM   2100 3HD1 LEU    84      -3.026  30.243  80.960  1.00 50.09\n"
		"ATOM   2101 1HD2 LEU    84      -3.740  29.706  83.911  1.00 50.09\n"
		"ATOM   2102 2HD2 LEU    84      -4.246  28.665  82.558  1.00 50.09\n"
		"ATOM   2103 3HD2 LEU    84      -3.343  27.971  83.925  1.00 50.09\n"
		"ATOM   2104  N   GLN    85       1.267  27.429  81.914  1.00 49.53\n"
		"ATOM   2105  CA  GLN    85       2.511  27.563  82.614  1.00 50.07\n"
		"ATOM   2106  C   GLN    85       3.587  28.077  81.684  1.00 50.51\n"
		"ATOM   2107  O   GLN    85       4.329  29.012  82.027  1.00 49.49\n"
		"ATOM   2108  CB  GLN    85       2.964  26.230  83.155  1.00 49.82\n"
		"ATOM   2109  CG  GLN    85       2.138  25.673  84.282  1.00 49.82\n"
		"ATOM   2110  CD  GLN    85       2.839  24.509  84.938  1.00 50.55\n"
		"ATOM   2111  OE1 GLN    85       3.901  24.679  85.562  1.00 49.45\n"
		"ATOM   2112  NE2 GLN    85       2.265  23.318  84.785  1.00 49.38\n"
		"ATOM   2113  H   GLN    85       0.856  26.513  81.807  1.00 50.31\n"
		"ATOM   2114 1HE2 GLN    85       2.681  22.505  85.196  1.00 50.32\n"
		"ATOM   2115 2HE2 GLN    85       1.418  23.234  84.260  1.00 50.30\n"
		"ATOM   2116  HA  GLN    85       2.404  28.298  83.411  1.00 50.09\n"
		"ATOM   2117 1HB  GLN    85       2.948  25.529  82.320  1.00 50.09\n"
		"ATOM   2118 2HB  GLN    85       3.990  26.362  83.498  1.00 50.09\n"
		"ATOM   2119 1HG  GLN    85       1.693  26.301  85.054  1.00 50.09\n"
		"ATOM   2120 2HG  GLN    85       1.352  25.288  83.631  1.00 50.09\n"
		"ATOM   2121  N   ARG    86       3.719  27.430  80.521  1.00 49.53\n"
		"ATOM   2122  CA  ARG    86       4.715  27.850  79.506  1.00 50.07\n"
		"ATOM   2123  C   ARG    86       4.566  29.351  79.220  1.00 50.51\n"
		"ATOM   2124  O   ARG    86       5.516  30.112  79.410  1.00 49.49\n"
		"ATOM   2125  CB  ARG    86       4.539  27.033  78.188  1.00 49.82\n"
		"ATOM   2126  CG  ARG    86       5.782  26.919  77.261  1.00 49.82\n"
		"ATOM   2127  CD  ARG    86       5.438  26.302  75.890  1.00 50.20\n"
		"ATOM   2128  NE  ARG    86       5.109  24.864  75.965  1.00 49.30\n"
		"ATOM   2129  CZ  ARG    86       6.013  23.876  76.105  1.00 50.64\n"
		"ATOM   2130  NH1 ARG    86       7.313  24.155  76.144  1.00 49.20\n"
		"ATOM   2131  NH2 ARG    86       5.619  22.610  76.193  1.00 49.20\n"
		"ATOM   2132  H   ARG    86       3.126  26.636  80.329  1.00 50.31\n"
		"ATOM   2133 1HH1 ARG    86       7.624  25.113  76.069  1.00 50.46\n"
		"ATOM   2134 2HH1 ARG    86       7.984  23.408  76.249  1.00 50.46\n"
		"ATOM   2135 1HH2 ARG    86       4.634  22.387  76.155  1.00 50.46\n"
		"ATOM   2136 2HH2 ARG    86       6.305  21.877  76.297  1.00 50.46\n"
		"ATOM   2137  HE  ARG    86       4.208  24.406  75.927  1.00 50.44\n"
		"ATOM   2138  HA  ARG    86       5.723  27.694  79.891  1.00 50.09\n"
		"ATOM   2139 1HB  ARG    86       4.234  26.028  78.475  1.00 50.09\n"
		"ATOM   2140 2HB  ARG    86       3.737  27.509  77.624  1.00 50.09\n"
		"ATOM   2141 1HG  ARG    86       6.193  27.917  77.107  1.00 50.09\n"
		"ATOM   2142 2HG  ARG    86       6.526  26.293  77.755  1.00 50.09\n"
		"ATOM   2143 1HD  ARG    86       4.578  26.824  75.473  1.00 50.09\n"
		"ATOM   2144 2HD  ARG    86       6.292  26.420  75.225  1.00 50.09\n"
		"ATOM   2145  N   MET    87       3.346  29.767  78.807  1.00 49.53\n"
		"ATOM   2146  CA  MET    87       3.064  31.156  78.453  1.00 50.07\n"
		"ATOM   2147  C   MET    87       3.449  32.114  79.566  1.00 50.51\n"
		"ATOM   2148  O   MET    87       4.280  33.001  79.346  1.00 49.49\n"
		"ATOM   2149  CB  MET    87       1.580  31.342  78.150  1.00 49.82\n"
		"ATOM   2150  CG  MET    87       1.055  30.817  76.822  1.00 49.86\n"
		"ATOM   2151  SD  MET    87      -0.740  30.784  76.802  1.00 49.91\n"
		"ATOM   2152  CE  MET    87      -1.061  32.372  77.546  1.00 49.78\n"
		"ATOM   2153  H   MET    87       2.600  29.089  78.741  1.00 50.31\n"
		"ATOM   2154  HA  MET    87       3.656  31.446  77.585  1.00 50.09\n"
		"ATOM   2155 1HB  MET    87       1.029  30.843  78.946  1.00 50.09\n"
		"ATOM   2156 2HB  MET    87       1.380  32.414  78.185  1.00 50.09\n"
		"ATOM   2157 1HG  MET    87       1.423  31.468  76.030  1.00 50.09\n"
		"ATOM   2158 2HG  MET    87       1.446  29.810  76.677  1.00 50.09\n"
		"ATOM   2159 1HE  MET    87      -2.137  32.530  77.617  1.00 50.09\n"
		"ATOM   2160 2HE  MET    87      -0.622  32.402  78.544  1.00 50.09\n"
		"ATOM   2161 3HE  MET    87      -0.619  33.157  76.932  1.00 50.09\n"
		"ATOM   2162  N   PHE    88       2.854  31.884  80.754  1.00 49.53\n"
		"ATOM   2163  CA  PHE    88       3.043  32.680  81.977  1.00 50.07\n"
		"ATOM   2164  C   PHE    88       4.511  32.796  82.391  1.00 50.51\n"
		"ATOM   2165  O   PHE    88       4.938  33.849  82.862  1.00 49.49\n"
		"ATOM   2166  CB  PHE    88       2.238  32.037  83.098  1.00 49.82\n"
		"ATOM   2167  CG  PHE    88       2.149  32.828  84.363  1.00 50.00\n"
		"ATOM   2168  CD1 PHE    88       3.211  32.852  85.282  1.00 49.88\n"
		"ATOM   2169  CD2 PHE    88       0.969  33.481  84.683  1.00 49.88\n"
		"ATOM   2170  CE1 PHE    88       3.094  33.511  86.488  1.00 49.88\n"
		"ATOM   2171  CE2 PHE    88       0.843  34.150  85.895  1.00 49.88\n"
		"ATOM   2172  CZ  PHE    88       1.903  34.159  86.805  1.00 49.88\n"
		"ATOM   2173  H   PHE    88       2.224  31.097  80.809  1.00 50.31\n"
		"ATOM   2174  HD1 PHE    88       4.142  32.341  85.034  1.00 50.12\n"
		"ATOM   2175  HE1 PHE    88       3.931  33.522  87.186  1.00 50.12\n"
		"ATOM   2176  HZ  PHE    88       1.797  34.670  87.761  1.00 50.12\n"
		"ATOM   2177  HE2 PHE    88      -0.084  34.669  86.139  1.00 50.12\n"
		"ATOM   2178  HD2 PHE    88       0.142  33.464  83.973  1.00 50.12\n"
		"ATOM   2179  HA  PHE    88       2.703  33.703  81.809  1.00 50.09\n"
		"ATOM   2180 1HB  PHE    88       1.209  31.877  82.777  1.00 50.09\n"
		"ATOM   2181 2HB  PHE    88       2.682  31.083  83.378  1.00 50.09\n"
		"ATOM   2182  N   LYS    89       5.283  31.715  82.261  1.00 49.53\n"
		"ATOM   2183  CA  LYS    89       6.700  31.794  82.601  1.00 50.07\n"
		"ATOM   2184  C   LYS    89       7.402  32.718  81.653  1.00 50.51\n"
		"ATOM   2185  O   LYS    89       8.147  33.616  82.059  1.00 49.49\n"
		"ATOM   2186  CB  LYS    89       7.410  30.431  82.516  1.00 49.82\n"
		"ATOM   2187  CG  LYS    89       8.931  30.518  82.823  1.00 49.82\n"
		"ATOM   2188  CD  LYS    89       9.194  31.335  84.065  1.00 49.82\n"
		"ATOM   2189  CE  LYS    89      10.413  30.852  84.835  1.00 50.21\n"
		"ATOM   2190  NZ  LYS    89      10.421  29.372  85.043  1.00 49.70\n"
		"ATOM   2191  H   LYS    89       4.888  30.847  81.929  1.00 50.31\n"
		"ATOM   2192 1HZ  LYS    89      11.249  29.109  85.558  1.00 50.33\n"
		"ATOM   2193 2HZ  LYS    89       9.600  29.100  85.565  1.00 50.33\n"
		"ATOM   2194 3HZ  LYS    89      10.419  28.906  84.147  1.00 50.33\n"
		"ATOM   2195  HA  LYS    89       6.817  32.210  83.602  1.00 50.09\n"
		"ATOM   2196 1HB  LYS    89       6.920  29.771  83.232  1.00 50.09\n"
		"ATOM   2197 2HB  LYS    89       7.253  30.051  81.506  1.00 50.09\n"
		"ATOM   2198 1HG  LYS    89       9.314  29.507  82.964  1.00 50.09\n"
		"ATOM   2199 2HG  LYS    89       9.430  30.979  81.972  1.00 50.09\n"
		"ATOM   2200 1HD  LYS    89       9.351  32.374  83.771  1.00 50.09\n"
		"ATOM   2201 2HD  LYS    89       8.321  31.271  84.714  1.00 50.09\n"
		"ATOM   2202 1HE  LYS    89      11.304  31.136  84.277  1.00 50.05\n"
		"ATOM   2203 2HE  LYS    89      10.421  31.346  85.807  1.00 50.05\n"
		"ATOM   2204  N   LYS    90       7.213  32.447  80.372  1.00 49.53\n"
		"ATOM   2205  CA  LYS    90       7.844  33.238  79.354  1.00 50.07\n"
		"ATOM   2206  C   LYS    90       7.344  34.647  79.429  1.00 50.51\n"
		"ATOM   2207  O   LYS    90       8.073  35.562  79.134  1.00 49.49\n"
		"ATOM   2208  CB  LYS    90       7.576  32.641  77.974  1.00 49.82\n"
		"ATOM   2209  CG  LYS    90       6.423  33.285  77.204  1.00 49.82\n"
		"ATOM   2210  CD  LYS    90       6.110  32.525  75.910  1.00 49.82\n"
		"ATOM   2211  CE  LYS    90       6.246  33.399  74.663  1.00 50.21\n"
		"ATOM   2212  NZ  LYS    90       7.298  32.902  73.721  1.00 49.70\n"
		"ATOM   2213  H   LYS    90       6.618  31.675  80.107  1.00 50.31\n"
		"ATOM   2214 1HZ  LYS    90       7.347  33.513  72.918  1.00 50.33\n"
		"ATOM   2215 2HZ  LYS    90       8.193  32.891  74.190  1.00 50.33\n"
		"ATOM   2216 3HZ  LYS    90       7.065  31.967  73.419  1.00 50.33\n"
		"ATOM   2217  HA  LYS    90       8.921  33.267  79.523  1.00 50.09\n"
		"ATOM   2218 1HB  LYS    90       8.495  32.747  77.396  1.00 50.09\n"
		"ATOM   2219 2HB  LYS    90       7.360  31.582  78.120  1.00 50.09\n"
		"ATOM   2220 1HG  LYS    90       5.539  33.288  77.843  1.00 50.09\n"
		"ATOM   2221 2HG  LYS    90       6.699  34.311  76.961  1.00 50.09\n"
		"ATOM   2222 1HD  LYS    90       6.799  31.683  75.826  1.00 50.09\n"
		"ATOM   2223 2HD  LYS    90       5.088  32.151  75.965  1.00 50.09\n"
		"ATOM   2224 1HE  LYS    90       5.286  33.413  74.149  1.00 50.05\n"
		"ATOM   2225 2HE  LYS    90       6.502  34.410  74.981  1.00 50.05\n"
		"ATOM   2226  N   GLU    91       6.100  34.801  79.856  1.00 49.53\n"
		"ATOM   2227  CA  GLU    91       5.453  36.103  79.942  1.00 50.07\n"
		"ATOM   2228  C   GLU    91       5.917  36.949  81.133  1.00 50.51\n"
		"ATOM   2229  O   GLU    91       6.071  38.167  80.965  1.00 49.49\n"
		"ATOM   2230  CB  GLU    91       3.931  35.931  80.026  1.00 49.82\n"
		"ATOM   2231  CG  GLU    91       3.190  36.123  78.720  1.00 49.72\n"
		"ATOM   2232  CD  GLU    91       3.725  37.270  77.863  1.00 50.62\n"
		"ATOM   2233  OE1 GLU    91       3.665  38.447  78.303  1.00 49.24\n"
		"ATOM   2234  OE2 GLU    91       4.194  36.995  76.735  1.00 49.24\n"
		"ATOM   2235  H   GLU    91       5.578  33.982  80.133  1.00 50.31\n"
		"ATOM   2236  HA  GLU    91       5.696  36.696  79.059  1.00 50.09\n"
		"ATOM   2237 1HB  GLU    91       3.745  34.923  80.396  1.00 50.09\n"
		"ATOM   2238 2HB  GLU    91       3.567  36.658  80.752  1.00 50.09\n"
		"ATOM   2239 1HG  GLU    91       3.165  35.227  78.100  1.00 50.09\n"
		"ATOM   2240 2HG  GLU    91       2.178  36.374  79.034  1.00 50.09\n"
		"ATOM   2241  N   THR    92       6.093  36.293  82.337  1.00 49.53\n"
		"ATOM   2242  CA  THR    92       6.464  36.945  83.640  1.00 50.07\n"
		"ATOM   2243  C   THR    92       7.851  36.527  84.240  1.00 50.51\n"
		"ATOM   2244  O   THR    92       8.371  37.212  85.143  1.00 49.49\n"
		"ATOM   2245  CB  THR    92       5.345  36.706  84.735  1.00 50.14\n"
		"ATOM   2246  OG1 THR    92       5.383  35.355  85.270  1.00 49.34\n"
		"ATOM   2247  CG2 THR    92       3.986  36.922  84.125  1.00 49.73\n"
		"ATOM   2248  H   THR    92       5.961  35.292  82.342  1.00 50.31\n"
		"ATOM   2249  HG1 THR    92       4.691  35.252  85.928  1.00 50.43\n"
		"ATOM   2250  HA  THR    92       6.547  38.023  83.502  1.00 50.09\n"
		"ATOM   2251  HB  THR    92       5.507  37.396  85.563  1.00 50.09\n"
		"ATOM   2252 1HG2 THR    92       3.219  36.757  84.881  1.00 50.09\n"
		"ATOM   2253 2HG2 THR    92       3.914  37.943  83.750  1.00 50.09\n"
		"ATOM   2254 3HG2 THR    92       3.842  36.222  83.303  1.00 50.09\n"
		"ATOM   2255  N   GLY    93       8.447  35.419  83.712  1.00 49.53\n"
		"ATOM   2256  CA  GLY    93       9.714  34.861  84.230  1.00 49.98\n"
		"ATOM   2257  C   GLY    93       9.525  34.203  85.609  1.00 50.51\n"
		"ATOM   2258  O   GLY    93      10.479  33.818  86.286  1.00 49.49\n"
		"ATOM   2259  H   GLY    93       8.002  34.959  82.931  1.00 50.31\n"
		"ATOM   2260 1HA  GLY    93      10.085  34.111  83.531  1.00 50.09\n"
		"ATOM   2261 2HA  GLY    93      10.445  35.663  84.322  1.00 50.09\n"
		"ATOM   2262  N   HIS    94       8.261  34.075  85.988  1.00 49.53\n"
		"ATOM   2263  CA  HIS    94       7.832  33.563  87.281  1.00 50.07\n"
		"ATOM   2264  C   HIS    94       7.003  32.307  87.087  1.00 50.51\n"
		"ATOM   2265  O   HIS    94       6.021  32.348  86.366  1.00 49.49\n"
		"ATOM   2266  CB  HIS    94       6.916  34.594  87.923  1.00 49.92\n"
		"ATOM   2267  CG  HIS    94       7.476  35.444  89.007  1.00 50.22\n"
		"ATOM   2268  ND1 HIS    94       7.701  36.795  88.831  1.00 49.30\n"
		"ATOM   2269  CD2 HIS    94       7.642  35.201  90.325  1.00 49.95\n"
		"ATOM   2270  CE1 HIS    94       7.966  37.346  90.000  1.00 50.25\n"
		"ATOM   2271  NE2 HIS    94       7.936  36.400  90.923  1.00 49.64\n"
		"ATOM   2272  H   HIS    94       7.545  34.353  85.332  1.00 50.31\n"
		"ATOM   2273  HE2 HIS    94       8.101  36.530  91.911  1.00 50.32\n"
		"ATOM   2274  HA  HIS    94       8.704  33.286  87.873  1.00 50.09\n"
		"ATOM   2275 1HB  HIS    94       6.568  35.306  87.174  1.00 50.09\n"
		"ATOM   2276 2HB  HIS    94       6.059  34.101  88.381  1.00 50.09\n"
		"ATOM   2277  HE1 HIS    94       8.162  38.415  90.078  1.00 50.13\n"
		"ATOM   2278  HD2 HIS    94       7.584  34.291  90.923  1.00 50.09\n"
		"ATOM   2279  N   SER    95       7.378  31.205  87.735  1.00 49.53\n"
		"ATOM   2280  CA  SER    95       6.572  29.989  87.637  1.00 50.07\n"
		"ATOM   2281  C   SER    95       5.179  30.261  88.222  1.00 50.51\n"
		"ATOM   2282  O   SER    95       5.072  30.812  89.298  1.00 49.49\n"
		"ATOM   2283  CB  SER    95       7.211  28.846  88.435  1.00 50.05\n"
		"ATOM   2284  OG  SER    95       8.593  29.069  88.643  1.00 49.34\n"
		"ATOM   2285  H   SER    95       8.219  31.210  88.294  1.00 50.31\n"
		"ATOM   2286  HG  SER    95       8.963  28.338  89.144  1.00 50.43\n"
		"ATOM   2287  HA  SER    95       6.502  29.597  86.622  1.00 50.09\n"
		"ATOM   2288 1HB  SER    95       6.709  28.772  89.399  1.00 50.09\n"
		"ATOM   2289 2HB  SER    95       7.075  27.917  87.884  1.00 50.09\n"
		"ATOM   2290  N   LEU    96       4.113  29.885  87.533  1.00 49.53\n"
		"ATOM   2291  CA  LEU    96       2.783  30.133  88.066  1.00 50.07\n"
		"ATOM   2292  C   LEU    96       2.654  29.647  89.552  1.00 50.51\n"
		"ATOM   2293  O   LEU    96       2.287  30.418  90.455  1.00 49.49\n"
		"ATOM   2294  CB  LEU    96       1.772  29.435  87.153  1.00 49.82\n"
		"ATOM   2295  CG  LEU    96       0.483  30.161  86.787  1.00 49.91\n"
		"ATOM   2296  CD1 LEU    96      -0.685  29.255  87.066  1.00 49.73\n"
		"ATOM   2297  CD2 LEU    96       0.343  31.451  87.560  1.00 49.73\n"
		"ATOM   2298  H   LEU    96       4.223  29.427  86.640  1.00 50.31\n"
		"ATOM   2299  HA  LEU    96       2.589  31.205  88.078  1.00 50.09\n"
		"ATOM   2300 1HB  LEU    96       2.382  29.335  86.257  1.00 50.09\n"
		"ATOM   2301 2HB  LEU    96       1.509  28.445  87.525  1.00 50.09\n"
		"ATOM   2302  HG  LEU    96       0.548  30.425  85.731  1.00 50.09\n"
		"ATOM   2303 1HD1 LEU    96      -1.612  29.766  86.807  1.00 50.09\n"
		"ATOM   2304 2HD1 LEU    96      -0.593  28.348  86.469  1.00 50.09\n"
		"ATOM   2305 3HD1 LEU    96      -0.698  28.994  88.124  1.00 50.09\n"
		"ATOM   2306 1HD2 LEU    96      -0.586  31.946  87.276  1.00 50.09\n"
		"ATOM   2307 2HD2 LEU    96       0.328  31.236  88.629  1.00 50.09\n"
		"ATOM   2308 3HD2 LEU    96       1.185  32.105  87.333  1.00 50.09\n"
		"ATOM   2309  N   GLY    97       3.012  28.390  89.811  1.00 49.53\n"
		"ATOM   2310  CA  GLY    97       2.832  27.829  91.151  1.00 49.98\n"
		"ATOM   2311  C   GLY    97       3.367  28.685  92.318  1.00 50.51\n"
		"ATOM   2312  O   GLY    97       2.628  28.991  93.259  1.00 49.49\n"
		"ATOM   2313  H   GLY    97       3.410  27.821  89.078  1.00 50.31\n"
		"ATOM   2314 1HA  GLY    97       1.768  27.682  91.335  1.00 50.09\n"
		"ATOM   2315 2HA  GLY    97       3.346  26.870  91.208  1.00 50.09\n"
		"ATOM   2316  N   GLN    98       4.657  29.060  92.226  1.00 49.53\n"
		"ATOM   2317  CA  GLN    98       5.394  29.812  93.251  1.00 50.07\n"
		"ATOM   2318  C   GLN    98       4.993  31.291  93.316  1.00 50.51\n"
		"ATOM   2319  O   GLN    98       5.218  31.980  94.321  1.00 49.49\n"
		"ATOM   2320  CB  GLN    98       6.915  29.642  93.000  1.00 49.82\n"
		"ATOM   2321  CG  GLN    98       7.717  30.918  92.877  1.00 49.82\n"
		"ATOM   2322  CD  GLN    98       9.184  30.649  92.591  1.00 50.55\n"
		"ATOM   2323  OE1 GLN    98       9.821  31.391  91.815  1.00 49.45\n"
		"ATOM   2324  NE2 GLN    98       9.730  29.591  93.208  1.00 49.38\n"
		"ATOM   2325  H   GLN    98       5.157  28.804  91.387  1.00 50.31\n"
		"ATOM   2326 1HE2 GLN    98      10.693  29.365  93.057  1.00 50.32\n"
		"ATOM   2327 2HE2 GLN    98       9.175  29.027  93.820  1.00 50.30\n"
		"ATOM   2328  HA  GLN    98       5.168  29.404  94.236  1.00 50.09\n"
		"ATOM   2329 1HB  GLN    98       7.308  29.059  93.833  1.00 50.09\n"
		"ATOM   2330 2HB  GLN    98       7.020  29.072  92.077  1.00 50.09\n"
		"ATOM   2331 1HG  GLN    98       7.400  31.756  92.257  1.00 50.09\n"
		"ATOM   2332 2HG  GLN    98       7.613  31.187  93.929  1.00 50.09\n"
		"ATOM   2333  N   TYR    99       4.393  31.779  92.247  1.00 49.53\n"
		"ATOM   2334  CA  TYR    99       3.892  33.148  92.211  1.00 50.07\n"
		"ATOM   2335  C   TYR    99       2.591  33.218  93.018  1.00 50.51\n"
		"ATOM   2336  O   TYR    99       2.417  34.071  93.893  1.00 49.49\n"
		"ATOM   2337  CB  TYR    99       3.648  33.572  90.763  1.00 49.82\n"
		"ATOM   2338  CG  TYR    99       2.899  34.858  90.662  1.00 50.00\n"
		"ATOM   2339  CD1 TYR    99       3.531  36.080  90.902  1.00 49.88\n"
		"ATOM   2340  CD2 TYR    99       1.539  34.842  90.408  1.00 49.88\n"
		"ATOM   2341  CE1 TYR    99       2.797  37.280  90.891  1.00 49.88\n"
		"ATOM   2342  CE2 TYR    99       0.799  36.015  90.390  1.00 49.88\n"
		"ATOM   2343  CZ  TYR    99       1.422  37.234  90.638  1.00 50.11\n"
		"ATOM   2344  OH  TYR    99       0.685  38.381  90.639  1.00 49.46\n"
		"ATOM   2345  H   TYR    99       4.278  31.190  91.434  1.00 50.31\n"
		"ATOM   2346  HH  TYR    99      -0.246  38.229  90.458  1.00 50.43\n"
		"ATOM   2347  HD1 TYR    99       4.603  36.088  91.100  1.00 50.12\n"
		"ATOM   2348  HE1 TYR    99       3.295  38.232  91.076  1.00 50.12\n"
		"ATOM   2349  HE2 TYR    99      -0.271  35.981  90.182  1.00 50.12\n"
		"ATOM   2350  HD2 TYR    99       1.050  33.886  90.221  1.00 50.12\n"
		"ATOM   2351  HA  TYR    99       4.612  33.818  92.682  1.00 50.09\n"
		"ATOM   2352 1HB  TYR    99       4.624  33.668  90.286  1.00 50.09\n"
		"ATOM   2353 2HB  TYR    99       3.086  32.771  90.283  1.00 50.09\n"
		"ATOM   2354  N   ILE   100       1.695  32.279  92.779  1.00 49.53\n"
		"ATOM   2355  CA  ILE   100       0.478  32.298  93.534  1.00 50.07\n"
		"ATOM   2356  C   ILE   100       0.763  32.191  94.979  1.00 50.51\n"
		"ATOM   2357  O   ILE   100       0.170  32.889  95.759  1.00 49.49\n"
		"ATOM   2358  CB  ILE   100      -0.449  31.162  93.170  1.00 49.91\n"
		"ATOM   2359  CG1 ILE   100      -0.912  31.345  91.760  1.00 49.82\n"
		"ATOM   2360  CG2 ILE   100      -1.622  31.126  94.117  1.00 49.73\n"
		"ATOM   2361  CD1 ILE   100      -1.670  30.159  91.242  1.00 49.73\n"
		"ATOM   2362  H   ILE   100       1.863  31.567  92.083  1.00 50.31\n"
		"ATOM   2363  HA  ILE   100      -0.032  33.254  93.417  1.00 50.09\n"
		"ATOM   2364  HB  ILE   100       0.104  30.224  93.212  1.00 50.09\n"
		"ATOM   2365 1HG2 ILE   100      -2.284  30.304  93.845  1.00 50.09\n"
		"ATOM   2366 2HG2 ILE   100      -1.263  30.981  95.135  1.00 50.09\n"
		"ATOM   2367 3HG2 ILE   100      -2.169  32.067  94.056  1.00 50.09\n"
		"ATOM   2368 1HG1 ILE   100      -1.545  32.231  91.739  1.00 50.09\n"
		"ATOM   2369 2HG1 ILE   100      -0.026  31.518  91.148  1.00 50.09\n"
		"ATOM   2370 1HD1 ILE   100      -1.982  30.346  90.214  1.00 50.09\n"
		"ATOM   2371 2HD1 ILE   100      -1.031  29.276  91.271  1.00 50.09\n"
		"ATOM   2372 3HD1 ILE   100      -2.550  29.990  91.861  1.00 50.09\n"
		"ATOM   2373  N   ARG   101       1.649  31.285  95.327  1.00 49.53\n"
		"ATOM   2374  CA  ARG   101       1.969  31.067  96.716  1.00 50.07\n"
		"ATOM   2375  C   ARG   101       2.683  32.278  97.274  1.00 50.51\n"
		"ATOM   2376  O   ARG   101       2.470  32.679  98.414  1.00 49.49\n"
		"ATOM   2377  CB  ARG   101       2.820  29.809  96.859  1.00 49.82\n"
		"ATOM   2378  CG  ARG   101       2.585  29.023  98.159  1.00 49.82\n"
		"ATOM   2379  CD  ARG   101       2.761  27.516  97.993  1.00 50.20\n"
		"ATOM   2380  NE  ARG   101       3.613  27.133  96.887  1.00 49.30\n"
		"ATOM   2381  CZ  ARG   101       4.871  27.499  96.883  1.00 50.64\n"
		"ATOM   2382  NH1 ARG   101       5.299  28.242  97.908  1.00 49.20\n"
		"ATOM   2383  NH2 ARG   101       5.700  27.170  95.893  1.00 49.20\n"
		"ATOM   2384  H   ARG   101       2.110  30.736  94.615  1.00 50.31\n"
		"ATOM   2385 1HH1 ARG   101       4.658  28.498  98.646  1.00 50.46\n"
		"ATOM   2386 2HH1 ARG   101       6.262  28.544  97.941  1.00 50.46\n"
		"ATOM   2387 1HH2 ARG   101       5.360  26.621  95.115  1.00 50.46\n"
		"ATOM   2388 2HH2 ARG   101       6.663  27.471  95.924  1.00 50.46\n"
		"ATOM   2389  HE  ARG   101       3.392  26.595  96.060  1.00 50.44\n"
		"ATOM   2390  HA  ARG   101       1.051  30.940  97.291  1.00 50.09\n"
		"ATOM   2391 1HB  ARG   101       2.593  29.173  96.005  1.00 50.09\n"
		"ATOM   2392 2HB  ARG   101       3.863  30.124  96.811  1.00 50.09\n"
		"ATOM   2393 1HG  ARG   101       3.294  29.373  98.909  1.00 50.09\n"
		"ATOM   2394 2HG  ARG   101       1.568  29.217  98.502  1.00 50.09\n"
		"ATOM   2395 1HD  ARG   101       3.202  27.109  98.902  1.00 50.09\n"
		"ATOM   2396 2HD  ARG   101       1.785  27.062  97.828  1.00 50.09\n"
		"ATOM   2397  N   SER   102       3.532  32.875  96.470  1.00 49.53\n"
		"ATOM   2398  CA  SER   102       4.252  34.069  96.928  1.00 50.07\n"
		"ATOM   2399  C   SER   102       3.267  35.180  97.212  1.00 50.51\n"
		"ATOM   2400  O   SER   102       3.480  35.969  98.109  1.00 49.49\n"
		"ATOM   2401  CB  SER   102       5.265  34.518  95.878  1.00 50.05\n"
		"ATOM   2402  OG  SER   102       6.241  33.506  95.685  1.00 49.34\n"
		"ATOM   2403  H   SER   102       3.689  32.514  95.540  1.00 50.31\n"
		"ATOM   2404  HG  SER   102       6.874  33.795  95.024  1.00 50.43\n"
		"ATOM   2405  HA  SER   102       4.888  33.885  97.795  1.00 50.09\n"
		"ATOM   2406 1HB  SER   102       4.737  34.711  94.945  1.00 50.09\n"
		"ATOM   2407 2HB  SER   102       5.741  35.436  96.221  1.00 50.09\n"
		"ATOM   2408  N   ARG   103       2.183  35.210  96.443  1.00 49.53\n"
		"ATOM   2409  CA  ARG   103       1.161  36.211  96.618  1.00 50.07\n"
		"ATOM   2410  C   ARG   103       0.315  35.949  97.869  1.00 50.51\n"
		"ATOM   2411  O   ARG   103      -0.058  36.882  98.575  1.00 49.49\n"
		"ATOM   2412  CB  ARG   103       0.271  36.263  95.391  1.00 49.82\n"
		"ATOM   2413  CG  ARG   103       1.008  36.670  94.147  1.00 49.82\n"
		"ATOM   2414  CD  ARG   103       0.816  38.148  93.868  1.00 50.20\n"
		"ATOM   2415  NE  ARG   103       1.147  38.936  95.039  1.00 49.30\n"
		"ATOM   2416  CZ  ARG   103       0.296  39.640  95.790  1.00 50.64\n"
		"ATOM   2417  NH1 ARG   103      -0.991  39.708  95.500  1.00 49.20\n"
		"ATOM   2418  NH2 ARG   103       0.755  40.290  96.842  1.00 49.20\n"
		"ATOM   2419  H   ARG   103       2.073  34.515  95.719  1.00 50.31\n"
		"ATOM   2420 1HH1 ARG   103      -1.351  39.219  94.693  1.00 50.46\n"
		"ATOM   2421 2HH1 ARG   103      -1.610  40.249  96.087  1.00 50.46\n"
		"ATOM   2422 1HH2 ARG   103       1.740  40.249  97.067  1.00 50.46\n"
		"ATOM   2423 2HH2 ARG   103       0.122  40.826  97.416  1.00 50.46\n"
		"ATOM   2424  HE  ARG   103       2.044  39.089  95.481  1.00 50.44\n"
		"ATOM   2425  HA  ARG   103       1.623  37.188  96.762  1.00 50.09\n"
		"ATOM   2426 1HB  ARG   103      -0.157  35.269  95.266  1.00 50.09\n"
		"ATOM   2427 2HB  ARG   103      -0.524  36.977  95.606  1.00 50.09\n"
		"ATOM   2428 1HG  ARG   103       2.068  36.456  94.286  1.00 50.09\n"
		"ATOM   2429 2HG  ARG   103       0.626  36.084  93.311  1.00 50.09\n"
		"ATOM   2430 1HD  ARG   103       1.463  38.439  93.041  1.00 50.09\n"
		"ATOM   2431 2HD  ARG   103      -0.224  38.325  93.595  1.00 50.09\n"
		"ATOM   2432  N   LYS   104      -0.002  34.676  98.144  1.00 49.53\n"
		"ATOM   2433  CA  LYS   104      -0.775  34.322  99.354  1.00 50.07\n"
		"ATOM   2434  C   LYS   104       0.050  34.557 100.628  1.00 50.51\n"
		"ATOM   2435  O   LYS   104      -0.469  35.015 101.668  1.00 49.49\n"
		"ATOM   2436  CB  LYS   104      -1.201  32.871  99.266  1.00 49.82\n"
		"ATOM   2437  CG  LYS   104      -2.263  32.611  98.216  1.00 49.82\n"
		"ATOM   2438  CD  LYS   104      -2.696  31.161  98.328  1.00 49.82\n"
		"ATOM   2439  CE  LYS   104      -3.951  30.868  97.522  1.00 50.21\n"
		"ATOM   2440  NZ  LYS   104      -4.381  29.454  97.743  1.00 49.70\n"
		"ATOM   2441  H   LYS   104       0.293  33.945  97.513  1.00 50.31\n"
		"ATOM   2442 1HZ  LYS   104      -5.215  29.268  97.204  1.00 50.33\n"
		"ATOM   2443 2HZ  LYS   104      -3.644  28.828  97.451  1.00 50.33\n"
		"ATOM   2444 3HZ  LYS   104      -4.575  29.311  98.724  1.00 50.33\n"
		"ATOM   2445  HA  LYS   104      -1.654  34.962  99.431  1.00 50.09\n"
		"ATOM   2446 1HB  LYS   104      -0.307  32.289  99.039  1.00 50.09\n"
		"ATOM   2447 2HB  LYS   104      -1.578  32.588 100.249  1.00 50.09\n"
		"ATOM   2448 1HG  LYS   104      -3.101  33.285  98.397  1.00 50.09\n"
		"ATOM   2449 2HG  LYS   104      -1.837  32.817  97.234  1.00 50.09\n"
		"ATOM   2450 1HD  LYS   104      -1.885  30.527  97.966  1.00 50.09\n"
		"ATOM   2451 2HD  LYS   104      -2.889  30.936  99.377  1.00 50.09\n"
		"ATOM   2452 1HE  LYS   104      -4.731  31.557  97.842  1.00 50.05\n"
		"ATOM   2453 2HE  LYS   104      -3.727  31.037  96.469  1.00 50.05\n"
		"ATOM   2454  N   MET   105       1.364  34.253 100.552  1.00 49.53\n"
		"ATOM   2455  CA  MET   105       2.231  34.458 101.703  1.00 50.07\n"
		"ATOM   2456  C   MET   105       2.406  35.945 101.918  1.00 50.51\n"
		"ATOM   2457  O   MET   105       2.314  36.435 103.021  1.00 49.49\n"
		"ATOM   2458  CB  MET   105       3.577  33.718 101.501  1.00 49.82\n"
		"ATOM   2459  CG  MET   105       3.470  32.190 101.509  1.00 49.86\n"
		"ATOM   2460  SD  MET   105       2.719  31.582 102.980  1.00 49.91\n"
		"ATOM   2461  CE  MET   105       1.215  30.979 102.282  1.00 49.78\n"
		"ATOM   2462  H   MET   105       1.746  33.882  99.694  1.00 50.31\n"
		"ATOM   2463  HA  MET   105       1.765  34.045 102.597  1.00 50.09\n"
		"ATOM   2464 1HB  MET   105       3.980  34.048 100.545  1.00 50.09\n"
		"ATOM   2465 2HB  MET   105       4.240  34.039 102.306  1.00 50.09\n"
		"ATOM   2466 1HG  MET   105       2.876  31.887 100.648  1.00 50.09\n"
		"ATOM   2467 2HG  MET   105       4.476  31.779 101.420  1.00 50.09\n"
		"ATOM   2468 1HE  MET   105       0.595  30.549 103.069  1.00 50.09\n"
		"ATOM   2469 2HE  MET   105       0.680  31.801 101.805  1.00 50.09\n"
		"ATOM   2470 3HE  MET   105       1.440  30.213 101.539  1.00 50.09\n"
		"ATOM   2471  N   THR   106       2.580  36.701 100.859  1.00 49.53\n"
		"ATOM   2472  CA  THR   106       2.738  38.126 101.108  1.00 50.07\n"
		"ATOM   2473  C   THR   106       1.439  38.743 101.679  1.00 50.51\n"
		"ATOM   2474  O   THR   106       1.472  39.575 102.592  1.00 49.49\n"
		"ATOM   2475  CB  THR   106       3.199  38.855  99.849  1.00 50.14\n"
		"ATOM   2476  OG1 THR   106       2.860  40.209  99.955  1.00 49.34\n"
		"ATOM   2477  CG2 THR   106       2.535  38.288  98.638  1.00 49.73\n"
		"ATOM   2478  H   THR   106       2.603  36.316  99.926  1.00 50.31\n"
		"ATOM   2479  HG1 THR   106       3.148  40.673  99.165  1.00 50.43\n"
		"ATOM   2480  HA  THR   106       3.511  38.286 101.860  1.00 50.09\n"
		"ATOM   2481  HB  THR   106       4.281  38.762  99.762  1.00 50.09\n"
		"ATOM   2482 1HG2 THR   106       2.878  38.822  97.752  1.00 50.09\n"
		"ATOM   2483 2HG2 THR   106       2.787  37.232  98.546  1.00 50.09\n"
		"ATOM   2484 3HG2 THR   106       1.455  38.397  98.732  1.00 50.09\n"
		"ATOM   2485  N   GLU   107       0.290  38.291 101.198  1.00 49.53\n"
		"ATOM   2486  CA  GLU   107      -1.001  38.825 101.691  1.00 50.07\n"
		"ATOM   2487  C   GLU   107      -1.317  38.410 103.145  1.00 50.51\n"
		"ATOM   2488  O   GLU   107      -1.900  39.180 103.895  1.00 49.49\n"
		"ATOM   2489  CB  GLU   107      -2.128  38.356 100.788  1.00 49.82\n"
		"ATOM   2490  CG  GLU   107      -2.208  39.113  99.474  1.00 49.72\n"
		"ATOM   2491  CD  GLU   107      -3.455  39.963  99.354  1.00 50.62\n"
		"ATOM   2492  OE1 GLU   107      -4.579  39.397  99.433  1.00 49.24\n"
		"ATOM   2493  OE2 GLU   107      -3.308  41.195  99.164  1.00 49.24\n"
		"ATOM   2494  H   GLU   107       0.294  37.574 100.487  1.00 50.31\n"
		"ATOM   2495  HA  GLU   107      -0.971  39.915 101.699  1.00 50.09\n"
		"ATOM   2496 1HB  GLU   107      -1.961  37.296 100.595  1.00 50.09\n"
		"ATOM   2497 2HB  GLU   107      -3.056  38.482 101.345  1.00 50.09\n"
		"ATOM   2498 1HG  GLU   107      -1.341  39.746  99.285  1.00 50.09\n"
		"ATOM   2499 2HG  GLU   107      -2.250  38.323  98.725  1.00 50.09\n"
		"ATOM   2500  N   ILE   108      -0.988  37.168 103.511  1.00 49.53\n"
		"ATOM   2501  CA  ILE   108      -1.193  36.702 104.873  1.00 50.07\n"
		"ATOM   2502  C   ILE   108      -0.270  37.444 105.805  1.00 50.51\n"
		"ATOM   2503  O   ILE   108      -0.553  37.620 106.977  1.00 49.49\n"
		"ATOM   2504  CB  ILE   108      -0.864  35.214 104.989  1.00 49.91\n"
		"ATOM   2505  CG1 ILE   108      -1.948  34.435 104.272  1.00 49.82\n"
		"ATOM   2506  CG2 ILE   108      -0.733  34.789 106.443  1.00 49.73\n"
		"ATOM   2507  CD1 ILE   108      -1.760  32.934 104.245  1.00 49.73\n"
		"ATOM   2508  H   ILE   108      -0.589  36.540 102.828  1.00 50.31\n"
		"ATOM   2509  HA  ILE   108      -2.201  36.939 105.212  1.00 50.09\n"
		"ATOM   2510  HB  ILE   108       0.079  35.018 104.480  1.00 50.09\n"
		"ATOM   2511 1HG2 ILE   108      -0.499  33.726 106.492  1.00 50.09\n"
		"ATOM   2512 2HG2 ILE   108       0.066  35.358 106.918  1.00 50.09\n"
		"ATOM   2513 3HG2 ILE   108      -1.672  34.978 106.964  1.00 50.09\n"
		"ATOM   2514 1HG1 ILE   108      -2.891  34.659 104.769  1.00 50.09\n"
		"ATOM   2515 2HG1 ILE   108      -1.984  34.800 103.245  1.00 50.09\n"
		"ATOM   2516 1HD1 ILE   108      -2.589  32.472 103.708  1.00 50.09\n"
		"ATOM   2517 2HD1 ILE   108      -0.823  32.693 103.741  1.00 50.09\n"
		"ATOM   2518 3HD1 ILE   108      -1.732  32.552 105.264  1.00 50.09\n"
		"ATOM   2519  N   ALA   109       0.856  37.863 105.283  1.00 49.53\n"
		"ATOM   2520  CA  ALA   109       1.836  38.589 106.076  1.00 50.07\n"
		"ATOM   2521  C   ALA   109       1.370  39.997 106.447  1.00 50.51\n"
		"ATOM   2522  O   ALA   109       1.648  40.463 107.546  1.00 49.49\n"
		"ATOM   2523  CB  ALA   109       3.151  38.686 105.314  1.00 49.73\n"
		"ATOM   2524  H   ALA   109       1.048  37.678 104.309  1.00 50.31\n"
		"ATOM   2525  HA  ALA   109       1.991  38.055 107.014  1.00 50.09\n"
		"ATOM   2526 1HB  ALA   109       3.879  39.231 105.915  1.00 50.09\n"
		"ATOM   2527 2HB  ALA   109       3.528  37.684 105.108  1.00 50.09\n"
		"ATOM   2528 3HB  ALA   109       2.989  39.213 104.375  1.00 50.09\n"
		"ATOM   2529  N   GLN   110       0.713  40.676 105.514  1.00 49.53\n"
		"ATOM   2530  CA  GLN   110       0.196  42.023 105.715  1.00 50.07\n"
		"ATOM   2531  C   GLN   110      -0.997  42.047 106.647  1.00 50.51\n"
		"ATOM   2532  O   GLN   110      -1.212  43.035 107.309  1.00 49.49\n"
		"ATOM   2533  CB  GLN   110      -0.248  42.602 104.381  1.00 49.82\n"
		"ATOM   2534  CG  GLN   110       0.666  42.253 103.236  1.00 49.82\n"
		"ATOM   2535  CD  GLN   110       0.554  43.249 102.114  1.00 50.55\n"
		"ATOM   2536  OE1 GLN   110       1.152  44.324 102.170  1.00 49.45\n"
		"ATOM   2537  NE2 GLN   110      -0.212  42.891 101.082  1.00 49.38\n"
		"ATOM   2538  H   GLN   110       0.564  40.235 104.618  1.00 50.31\n"
		"ATOM   2539 1HE2 GLN   110      -0.326  43.512 100.304  1.00 50.32\n"
		"ATOM   2540 2HE2 GLN   110      -0.672  42.003 101.083  1.00 50.30\n"
		"ATOM   2541  HA  GLN   110       0.959  42.640 106.189  1.00 50.09\n"
		"ATOM   2542 1HB  GLN   110      -1.250  42.219 104.189  1.00 50.09\n"
		"ATOM   2543 2HB  GLN   110      -0.291  43.684 104.506  1.00 50.09\n"
		"ATOM   2544 1HG  GLN   110       1.718  42.006 103.375  1.00 50.09\n"
		"ATOM   2545 2HG  GLN   110       0.117  41.352 102.958  1.00 50.09\n"
		"ATOM   2546  N   LYS   111      -1.788  40.972 106.646  1.00 49.53\n"
		"ATOM   2547  CA  LYS   111      -2.943  40.866 107.531  1.00 50.07\n"
		"ATOM   2548  C   LYS   111      -2.489  40.499 108.940  1.00 50.51\n"
		"ATOM   2549  O   LYS   111      -3.126  40.876 109.924  1.00 49.49\n"
		"ATOM   2550  CB  LYS   111      -3.916  39.775 107.052  1.00 49.82\n"
		"ATOM   2551  CG  LYS   111      -4.908  40.193 106.001  1.00 49.82\n"
		"ATOM   2552  CD  LYS   111      -5.789  39.016 105.616  1.00 49.82\n"
		"ATOM   2553  CE  LYS   111      -6.444  39.237 104.259  1.00 50.21\n"
		"ATOM   2554  NZ  LYS   111      -6.102  40.568 103.700  1.00 49.70\n"
		"ATOM   2555  H   LYS   111      -1.581  40.211 106.015  1.00 50.31\n"
		"ATOM   2556 1HZ  LYS   111      -6.552  40.682 102.803  1.00 50.33\n"
		"ATOM   2557 2HZ  LYS   111      -6.421  41.291 104.330  1.00 50.33\n"
		"ATOM   2558 3HZ  LYS   111      -5.101  40.638 103.586  1.00 50.33\n"
		"ATOM   2559  HA  LYS   111      -3.447  41.831 107.597  1.00 50.09\n"
		"ATOM   2560 1HB  LYS   111      -3.311  38.959 106.655  1.00 50.09\n"
		"ATOM   2561 2HB  LYS   111      -4.463  39.427 107.928  1.00 50.09\n"
		"ATOM   2562 1HG  LYS   111      -5.518  41.002 106.403  1.00 50.09\n"
		"ATOM   2563 2HG  LYS   111      -4.360  40.551 105.130  1.00 50.09\n"
		"ATOM   2564 1HD  LYS   111      -5.172  38.117 105.581  1.00 50.09\n"
		"ATOM   2565 2HD  LYS   111      -6.561  38.896 106.376  1.00 50.09\n"
		"ATOM   2566 1HE  LYS   111      -6.101  38.456 103.582  1.00 50.05\n"
		"ATOM   2567 2HE  LYS   111      -7.524  39.159 104.384  1.00 50.05\n"
		"ATOM   2568  N   LEU   112      -1.412  39.739 109.050  1.00 49.53\n"
		"ATOM   2569  CA  LEU   112      -0.961  39.412 110.370  1.00 50.07\n"
		"ATOM   2570  C   LEU   112      -0.521  40.672 111.085  1.00 50.51\n"
		"ATOM   2571  O   LEU   112      -0.493  40.751 112.308  1.00 49.49\n"
		"ATOM   2572  CB  LEU   112       0.217  38.490 110.320  1.00 49.82\n"
		"ATOM   2573  CG  LEU   112      -0.029  37.103 109.843  1.00 49.91\n"
		"ATOM   2574  CD1 LEU   112       1.311  36.430 109.608  1.00 49.73\n"
		"ATOM   2575  CD2 LEU   112      -0.829  36.322 110.867  1.00 49.73\n"
		"ATOM   2576  H   LEU   112      -0.925  39.401 108.232  1.00 50.31\n"
		"ATOM   2577  HA  LEU   112      -1.785  38.987 110.942  1.00 50.09\n"
		"ATOM   2578 1HB  LEU   112       0.826  39.020 109.590  1.00 50.09\n"
		"ATOM   2579 2HB  LEU   112       0.743  38.454 111.274  1.00 50.09\n"
		"ATOM   2580  HG  LEU   112      -0.617  37.175 108.928  1.00 50.09\n"
		"ATOM   2581 1HD1 LEU   112       1.150  35.411 109.257  1.00 50.09\n"
		"ATOM   2582 2HD1 LEU   112       1.871  36.987 108.857  1.00 50.09\n"
		"ATOM   2583 3HD1 LEU   112       1.875  36.408 110.540  1.00 50.09\n"
		"ATOM   2584 1HD2 LEU   112      -1.000  35.310 110.499  1.00 50.09\n"
		"ATOM   2585 2HD2 LEU   112      -0.276  36.280 111.806  1.00 50.09\n"
		"ATOM   2586 3HD2 LEU   112      -1.788  36.814 111.032  1.00 50.09\n"
		"ATOM   2587  N   LYS   113      -0.180  41.660 110.317  1.00 49.53\n"
		"ATOM   2588  CA  LYS   113       0.328  42.864 110.906  1.00 50.07\n"
		"ATOM   2589  C   LYS   113      -0.659  44.008 110.868  1.00 50.51\n"
		"ATOM   2590  O   LYS   113      -0.513  44.945 111.634  1.00 49.49\n"
		"ATOM   2591  CB  LYS   113       1.634  43.294 110.203  1.00 49.82\n"
		"ATOM   2592  CG  LYS   113       2.825  42.349 110.444  1.00 49.82\n"
		"ATOM   2593  CD  LYS   113       4.091  43.100 110.919  1.00 49.82\n"
		"ATOM   2594  CE  LYS   113       4.116  43.292 112.431  1.00 50.21\n"
		"ATOM   2595  NZ  LYS   113       3.715  44.680 112.816  1.00 49.70\n"
		"ATOM   2596  H   LYS   113      -0.271  41.582 109.314  1.00 50.31\n"
		"ATOM   2597 1HZ  LYS   113       3.743  44.772 113.822  1.00 50.33\n"
		"ATOM   2598 2HZ  LYS   113       4.353  45.343 112.399  1.00 50.33\n"
		"ATOM   2599 3HZ  LYS   113       2.778  44.863 112.487  1.00 50.33\n"
		"ATOM   2600  HA  LYS   113       0.541  42.697 111.962  1.00 50.09\n"
		"ATOM   2601 1HB  LYS   113       1.420  43.344 109.135  1.00 50.09\n"
		"ATOM   2602 2HB  LYS   113       1.882  44.290 110.570  1.00 50.09\n"
		"ATOM   2603 1HG  LYS   113       2.541  41.620 111.204  1.00 50.09\n"
		"ATOM   2604 2HG  LYS   113       3.051  41.831 109.512  1.00 50.09\n"
		"ATOM   2605 1HD  LYS   113       4.969  42.526 110.619  1.00 50.09\n"
		"ATOM   2606 2HD  LYS   113       4.116  44.078 110.438  1.00 50.09\n"
		"ATOM   2607 1HE  LYS   113       3.429  42.575 112.878  1.00 50.05\n"
		"ATOM   2608 2HE  LYS   113       5.128  43.092 112.784  1.00 50.05\n"
		"ATOM   2609  N   GLU   114      -1.645  43.955 109.985  1.00 49.53\n"
		"ATOM   2610  CA  GLU   114      -2.597  45.075 109.876  1.00 50.07\n"
		"ATOM   2611  C   GLU   114      -3.945  44.717 110.499  1.00 50.51\n"
		"ATOM   2612  O   GLU   114      -4.902  45.486 110.398  1.00 49.49\n"
		"ATOM   2613  CB  GLU   114      -2.824  45.448 108.402  1.00 49.82\n"
		"ATOM   2614  CG  GLU   114      -1.855  46.481 107.835  1.00 49.72\n"
		"ATOM   2615  CD  GLU   114      -0.439  45.953 107.630  1.00 50.62\n"
		"ATOM   2616  OE1 GLU   114       0.288  45.747 108.639  1.00 49.24\n"
		"ATOM   2617  OE2 GLU   114      -0.044  45.764 106.463  1.00 49.24\n"
		"ATOM   2618  H   GLU   114      -1.744  43.147 109.387  1.00 50.31\n"
		"ATOM   2619  HA  GLU   114      -2.220  45.936 110.428  1.00 50.09\n"
		"ATOM   2620 1HB  GLU   114      -2.739  44.528 107.824  1.00 50.09\n"
		"ATOM   2621 2HB  GLU   114      -3.840  45.834 108.324  1.00 50.09\n"
		"ATOM   2622 1HG  GLU   114      -2.189  46.918 106.894  1.00 50.09\n"
		"ATOM   2623 2HG  GLU   114      -1.826  47.253 108.602  1.00 50.09\n"
		"ATOM   2624  N   SER   115      -4.011  43.552 111.141  1.00 49.53\n"
		"ATOM   2625  CA  SER   115      -5.258  43.075 111.679  1.00 50.07\n"
		"ATOM   2626  C   SER   115      -5.017  41.967 112.645  1.00 50.51\n"
		"ATOM   2627  O   SER   115      -3.910  41.433 112.698  1.00 49.49\n"
		"ATOM   2628  CB  SER   115      -6.135  42.480 110.593  1.00 50.05\n"
		"ATOM   2629  OG  SER   115      -6.108  41.054 110.696  1.00 49.34\n"
		"ATOM   2630  H   SER   115      -3.175  42.996 111.252  1.00 50.31\n"
		"ATOM   2631  HG  SER   115      -6.662  40.674 110.010  1.00 50.43\n"
		"ATOM   2632  HA  SER   115      -5.858  43.859 112.143  1.00 50.09\n"
		"ATOM   2633 1HB  SER   115      -7.151  42.851 110.724  1.00 50.09\n"
		"ATOM   2634 2HB  SER   115      -5.755  42.799 109.623  1.00 50.09\n"
		"ATOM   2635  N   ASN   116      -6.107  41.554 113.332  1.00 49.53\n"
		"ATOM   2636  CA  ASN   116      -6.011  40.493 114.300  1.00 50.07\n"
		"ATOM   2637  C   ASN   116      -6.730  39.205 113.922  1.00 50.51\n"
		"ATOM   2638  O   ASN   116      -6.919  38.324 114.749  1.00 49.49\n"
		"ATOM   2639  CB  ASN   116      -6.428  40.944 115.671  1.00 49.82\n"
		"ATOM   2640  CG  ASN   116      -5.572  40.161 116.640  1.00 50.55\n"
		"ATOM   2641  OD1 ASN   116      -4.528  40.657 117.123  1.00 49.45\n"
		"ATOM   2642  ND2 ASN   116      -6.027  38.959 116.989  1.00 49.38\n"
		"ATOM   2643  H   ASN   116      -6.999  41.996 113.163  1.00 50.31\n"
		"ATOM   2644 1HD2 ASN   116      -5.504  38.393 117.628  1.00 50.32\n"
		"ATOM   2645 2HD2 ASN   116      -6.890  38.622 116.613  1.00 50.30\n"
		"ATOM   2646  HA  ASN   116      -4.968  40.197 114.422  1.00 50.09\n"
		"ATOM   2647 1HB  ASN   116      -6.532  42.005 115.902  1.00 50.09\n"
		"ATOM   2648 2HB  ASN   116      -7.417  40.492 115.588  1.00 50.09\n"
		"ATOM   2649  N   GLU   117      -7.091  39.067 112.665  1.00 49.53\n"
		"ATOM   2650  CA  GLU   117      -7.726  37.828 112.224  1.00 50.07\n"
		"ATOM   2651  C   GLU   117      -6.881  36.657 112.673  1.00 50.51\n"
		"ATOM   2652  O   GLU   117      -5.720  36.579 112.340  1.00 49.49\n"
		"ATOM   2653  CB  GLU   117      -7.827  37.801 110.710  1.00 49.82\n"
		"ATOM   2654  CG  GLU   117      -8.850  38.755 110.129  1.00 49.72\n"
		"ATOM   2655  CD  GLU   117      -8.554  39.132 108.692  1.00 50.62\n"
		"ATOM   2656  OE1 GLU   117      -8.204  38.226 107.891  1.00 49.24\n"
		"ATOM   2657  OE2 GLU   117      -8.673  40.338 108.365  1.00 49.24\n"
		"ATOM   2658  H   GLU   117      -6.930  39.816 112.007  1.00 50.31\n"
		"ATOM   2659  HA  GLU   117      -8.705  37.727 112.693  1.00 50.09\n"
		"ATOM   2660 1HB  GLU   117      -6.840  38.050 110.321  1.00 50.09\n"
		"ATOM   2661 2HB  GLU   117      -8.083  36.780 110.426  1.00 50.09\n"
		"ATOM   2662 1HG  GLU   117      -9.872  38.381 110.185  1.00 50.09\n"
		"ATOM   2663 2HG  GLU   117      -8.755  39.643 110.752  1.00 50.09\n"
		"ATOM   2664  N   PRO   118      -7.425  35.753 113.463  1.00 49.71\n"
		"ATOM   2665  CA  PRO   118      -6.683  34.585 113.929  1.00 50.02\n"
		"ATOM   2666  C   PRO   118      -6.037  33.814 112.776  1.00 50.51\n"
		"ATOM   2667  O   PRO   118      -6.615  33.686 111.689  1.00 49.49\n"
		"ATOM   2668  CB  PRO   118      -7.705  33.732 114.675  1.00 49.82\n"
		"ATOM   2669  CG  PRO   118      -9.035  34.389 114.451  1.00 49.82\n"
		"ATOM   2670  CD  PRO   118      -8.775  35.815 114.013  1.00 50.00\n"
		"ATOM   2671 1HD  PRO   118      -9.314  36.086 113.275  1.00 50.09\n"
		"ATOM   2672 2HD  PRO   118      -8.682  36.423 114.741  1.00 50.09\n"
		"ATOM   2673 1HG  PRO   118      -9.368  34.084 113.735  1.00 50.09\n"
		"ATOM   2674 2HG  PRO   118      -9.408  34.564 115.293  1.00 50.09\n"
		"ATOM   2675 1HB  PRO   118      -7.701  32.975 114.162  1.00 50.09\n"
		"ATOM   2676 2HB  PRO   118      -7.479  33.882 115.497  1.00 50.09\n"
		"ATOM   2677  HA  PRO   118      -5.959  34.866 114.553  1.00 50.09\n"
		"ATOM   2678  N   ILE   119      -4.836  33.309 113.008  1.00 49.53\n"
		"ATOM   2679  CA  ILE   119      -4.158  32.615 111.956  1.00 50.07\n"
		"ATOM   2680  C   ILE   119      -4.893  31.344 111.483  1.00 50.51\n"
		"ATOM   2681  O   ILE   119      -4.709  30.912 110.342  1.00 49.49\n"
		"ATOM   2682  CB  ILE   119      -2.708  32.263 112.342  1.00 49.91\n"
		"ATOM   2683  CG1 ILE   119      -2.620  30.817 112.788  1.00 49.82\n"
		"ATOM   2684  CG2 ILE   119      -2.180  33.184 113.433  1.00 49.73\n"
		"ATOM   2685  CD1 ILE   119      -1.224  30.444 113.133  1.00 49.73\n"
		"ATOM   2686  H   ILE   119      -4.404  33.412 113.915  1.00 50.31\n"
		"ATOM   2687  HA  ILE   119      -4.110  33.231 111.059  1.00 50.09\n"
		"ATOM   2688  HB  ILE   119      -2.073  32.361 111.462  1.00 50.09\n"
		"ATOM   2689 1HG2 ILE   119      -1.156  32.906 113.680  1.00 50.09\n"
		"ATOM   2690 2HG2 ILE   119      -2.201  34.214 113.081  1.00 50.09\n"
		"ATOM   2691 3HG2 ILE   119      -2.805  33.092 114.322  1.00 50.09\n"
		"ATOM   2692 1HG1 ILE   119      -3.275  30.703 113.651  1.00 50.09\n"
		"ATOM   2693 2HG1 ILE   119      -2.992  30.202 111.968  1.00 50.09\n"
		"ATOM   2694 1HD1 ILE   119      -1.193  29.401 113.449  1.00 50.09\n"
		"ATOM   2695 2HD1 ILE   119      -0.584  30.577 112.260  1.00 50.09\n"
		"ATOM   2696 3HD1 ILE   119      -0.867  31.078 113.944  1.00 50.09\n"
		"ATOM   2697  N   LEU   120      -5.708  30.711 112.336  1.00 49.53\n"
		"ATOM   2698  CA  LEU   120      -6.486  29.580 111.819  1.00 50.07\n"
		"ATOM   2699  C   LEU   120      -7.497  30.134 110.833  1.00 50.51\n"
		"ATOM   2700  O   LEU   120      -7.762  29.553 109.799  1.00 49.49\n"
		"ATOM   2701  CB  LEU   120      -7.236  28.813 112.933  1.00 49.82\n"
		"ATOM   2702  CG  LEU   120      -7.489  27.302 112.714  1.00 49.91\n"
		"ATOM   2703  CD1 LEU   120      -8.809  26.902 113.350  1.00 49.73\n"
		"ATOM   2704  CD2 LEU   120      -7.509  26.938 111.233  1.00 49.73\n"
		"ATOM   2705  H   LEU   120      -5.784  31.002 113.300  1.00 50.31\n"
		"ATOM   2706  HA  LEU   120      -5.826  28.897 111.284  1.00 50.09\n"
		"ATOM   2707 1HB  LEU   120      -6.525  28.943 113.747  1.00 50.09\n"
		"ATOM   2708 2HB  LEU   120      -8.178  29.297 113.193  1.00 50.09\n"
		"ATOM   2709  HG  LEU   120      -6.653  26.765 113.162  1.00 50.09\n"
		"ATOM   2710 1HD1 LEU   120      -8.981  25.837 113.193  1.00 50.09\n"
		"ATOM   2711 2HD1 LEU   120      -8.775  27.109 114.420  1.00 50.09\n"
		"ATOM   2712 3HD1 LEU   120      -9.619  27.471 112.895  1.00 50.09\n"
		"ATOM   2713 1HD2 LEU   120      -7.689  25.868 111.124  1.00 50.09\n"
		"ATOM   2714 2HD2 LEU   120      -8.303  27.493 110.732  1.00 50.09\n"
		"ATOM   2715 3HD2 LEU   120      -6.549  27.192 110.782  1.00 50.09\n"
		"ATOM   2716  N   TYR   121      -8.052  31.281 111.121  1.00 49.53\n"
		"ATOM   2717  CA  TYR   121      -9.009  31.779 110.185  1.00 50.07\n"
		"ATOM   2718  C   TYR   121      -8.339  32.213 108.899  1.00 50.51\n"
		"ATOM   2719  O   TYR   121      -8.972  32.272 107.842  1.00 49.49\n"
		"ATOM   2720  CB  TYR   121      -9.783  32.941 110.782  1.00 49.82\n"
		"ATOM   2721  CG  TYR   121     -10.108  34.003 109.762  1.00 50.00\n"
		"ATOM   2722  CD1 TYR   121      -9.154  34.964 109.381  1.00 49.88\n"
		"ATOM   2723  CD2 TYR   121     -11.342  33.997 109.104  1.00 49.88\n"
		"ATOM   2724  CE1 TYR   121      -9.432  35.910 108.392  1.00 49.88\n"
		"ATOM   2725  CE2 TYR   121     -11.637  34.940 108.108  1.00 49.88\n"
		"ATOM   2726  CZ  TYR   121     -10.672  35.888 107.743  1.00 50.11\n"
		"ATOM   2727  OH  TYR   121     -10.952  36.815 106.754  1.00 49.46\n"
		"ATOM   2728  H   TYR   121      -7.817  31.789 111.962  1.00 50.31\n"
		"ATOM   2729  HH  TYR   121     -10.222  37.415 106.585  1.00 50.43\n"
		"ATOM   2730  HD1 TYR   121      -8.181  34.964 109.874  1.00 50.12\n"
		"ATOM   2731  HE1 TYR   121      -8.689  36.661 108.124  1.00 50.12\n"
		"ATOM   2732  HE2 TYR   121     -12.613  34.933 107.623  1.00 50.12\n"
		"ATOM   2733  HD2 TYR   121     -12.081  33.244 109.377  1.00 50.12\n"
		"ATOM   2734  HA  TYR   121      -9.712  30.989 109.919  1.00 50.09\n"
		"ATOM   2735 1HB  TYR   121     -10.702  32.533 111.206  1.00 50.09\n"
		"ATOM   2736 2HB  TYR   121      -9.170  33.361 111.579  1.00 50.09\n"
		"ATOM   2737  N   LEU   122      -7.076  32.584 109.012  1.00 49.53\n"
		"ATOM   2738  CA  LEU   122      -6.329  33.016 107.829  1.00 50.07\n"
		"ATOM   2739  C   LEU   122      -5.943  31.788 107.008  1.00 50.51\n"
		"ATOM   2740  O   LEU   122      -5.918  31.811 105.800  1.00 49.49\n"
		"ATOM   2741  CB  LEU   122      -5.072  33.749 108.239  1.00 49.82\n"
		"ATOM   2742  CG  LEU   122      -5.259  35.154 108.825  1.00 49.91\n"
		"ATOM   2743  CD1 LEU   122      -3.940  35.657 109.391  1.00 49.73\n"
		"ATOM   2744  CD2 LEU   122      -5.725  36.098 107.751  1.00 49.73\n"
		"ATOM   2745  H   LEU   122      -6.625  32.570 109.916  1.00 50.31\n"
		"ATOM   2746  HA  LEU   122      -6.965  33.642 107.204  1.00 50.09\n"
		"ATOM   2747 1HB  LEU   122      -4.719  33.073 109.016  1.00 50.09\n"
		"ATOM   2748 2HB  LEU   122      -4.343  33.784 107.430  1.00 50.09\n"
		"ATOM   2749  HG  LEU   122      -6.023  35.083 109.599  1.00 50.09\n"
		"ATOM   2750 1HD1 LEU   122      -4.079  36.655 109.806  1.00 50.09\n"
		"ATOM   2751 2HD1 LEU   122      -3.600  34.982 110.177  1.00 50.09\n"
		"ATOM   2752 3HD1 LEU   122      -3.195  35.695 108.597  1.00 50.09\n"
		"ATOM   2753 1HD2 LEU   122      -5.856  37.094 108.175  1.00 50.09\n"
		"ATOM   2754 2HD2 LEU   122      -4.983  36.137 106.953  1.00 50.09\n"
		"ATOM   2755 3HD2 LEU   122      -6.675  35.749 107.346  1.00 50.09\n"
		"ATOM   2756  N   ALA   123      -5.607  30.714 107.706  1.00 49.53\n"
		"ATOM   2757  CA  ALA   123      -5.271  29.468 107.047  1.00 50.07\n"
		"ATOM   2758  C   ALA   123      -6.415  29.005 106.157  1.00 50.51\n"
		"ATOM   2759  O   ALA   123      -6.223  28.764 104.974  1.00 49.49\n"
		"ATOM   2760  CB  ALA   123      -4.980  28.419 108.105  1.00 49.73\n"
		"ATOM   2761  H   ALA   123      -5.585  30.764 108.715  1.00 50.31\n"
		"ATOM   2762  HA  ALA   123      -4.403  29.633 106.408  1.00 50.09\n"
		"ATOM   2763 1HB  ALA   123      -4.726  27.476 107.621  1.00 50.09\n"
		"ATOM   2764 2HB  ALA   123      -4.144  28.747 108.723  1.00 50.09\n"
		"ATOM   2765 3HB  ALA   123      -5.860  28.279 108.730  1.00 50.09\n"
		"ATOM   2766  N   GLU   124      -7.613  28.879 106.747  1.00 49.53\n"
		"ATOM   2767  CA  GLU   124      -8.874  28.431 106.075  1.00 50.07\n"
		"ATOM   2768  C   GLU   124      -9.294  29.373 104.931  1.00 50.51\n"
		"ATOM   2769  O   GLU   124      -9.737  28.931 103.876  1.00 49.49\n"
		"ATOM   2770  CB  GLU   124      -9.989  28.367 107.159  1.00 49.82\n"
		"ATOM   2771  CG  GLU   124     -11.363  27.852 106.730  1.00 49.72\n"
		"ATOM   2772  CD  GLU   124     -12.364  27.835 107.894  1.00 50.62\n"
		"ATOM   2773  OE1 GLU   124     -11.980  28.204 109.031  1.00 49.24\n"
		"ATOM   2774  OE2 GLU   124     -13.528  27.455 107.676  1.00 49.24\n"
		"ATOM   2775  H   GLU   124      -7.669  29.106 107.730  1.00 50.31\n"
		"ATOM   2776  HA  GLU   124      -8.722  27.453 105.617  1.00 50.09\n"
		"ATOM   2777 1HB  GLU   124      -9.623  27.718 107.954  1.00 50.09\n"
		"ATOM   2778 2HB  GLU   124     -10.117  29.378 107.545  1.00 50.09\n"
		"ATOM   2779 1HG  GLU   124     -11.796  28.410 105.900  1.00 50.09\n"
		"ATOM   2780 2HG  GLU   124     -11.166  26.829 106.414  1.00 50.09\n"
		"ATOM   2781  N   ARG   125      -9.148  30.660 105.159  1.00 49.53\n"
		"ATOM   2782  CA  ARG   125      -9.548  31.678 104.213  1.00 50.07\n"
		"ATOM   2783  C   ARG   125      -8.638  31.756 102.980  1.00 50.51\n"
		"ATOM   2784  O   ARG   125      -9.054  32.220 101.901  1.00 49.49\n"
		"ATOM   2785  CB  ARG   125      -9.566  33.013 104.951  1.00 49.82\n"
		"ATOM   2786  CG  ARG   125     -10.101  34.204 104.195  1.00 49.82\n"
		"ATOM   2787  CD  ARG   125      -8.994  35.076 103.619  1.00 50.20\n"
		"ATOM   2788  NE  ARG   125      -8.693  36.272 104.439  1.00 49.30\n"
		"ATOM   2789  CZ  ARG   125      -9.510  37.312 104.680  1.00 50.64\n"
		"ATOM   2790  NH1 ARG   125     -10.745  37.333 104.201  1.00 49.20\n"
		"ATOM   2791  NH2 ARG   125      -9.085  38.328 105.432  1.00 49.20\n"
		"ATOM   2792  H   ARG   125      -8.738  30.951 106.035  1.00 50.31\n"
		"ATOM   2793 1HH1 ARG   125     -11.082  36.559 103.647  1.00 50.46\n"
		"ATOM   2794 2HH1 ARG   125     -11.344  38.123 104.392  1.00 50.46\n"
		"ATOM   2795 1HH2 ARG   125      -8.151  38.311 105.817  1.00 50.46\n"
		"ATOM   2796 2HH2 ARG   125      -9.698  39.109 105.613  1.00 50.46\n"
		"ATOM   2797  HE  ARG   125      -7.851  36.511 104.945  1.00 50.44\n"
		"ATOM   2798  HA  ARG   125     -10.542  31.452 103.826  1.00 50.09\n"
		"ATOM   2799 1HB  ARG   125     -10.174  32.875 105.844  1.00 50.09\n"
		"ATOM   2800 2HB  ARG   125      -8.538  33.229 105.242  1.00 50.09\n"
		"ATOM   2801 1HG  ARG   125     -10.727  33.844 103.378  1.00 50.09\n"
		"ATOM   2802 2HG  ARG   125     -10.704  34.804 104.877  1.00 50.09\n"
		"ATOM   2803 1HD  ARG   125      -8.082  34.485 103.542  1.00 50.09\n"
		"ATOM   2804 2HD  ARG   125      -9.291  35.416 102.628  1.00 50.09\n"
		"ATOM   2805  N   TYR   126      -7.383  31.349 103.091  1.00 49.53\n"
		"ATOM   2806  CA  TYR   126      -6.566  31.404 101.883  1.00 50.07\n"
		"ATOM   2807  C   TYR   126      -6.350  30.051 101.260  1.00 50.51\n"
		"ATOM   2808  O   TYR   126      -5.521  29.940 100.363  1.00 49.49\n"
		"ATOM   2809  CB  TYR   126      -5.221  32.054 102.139  1.00 49.82\n"
		"ATOM   2810  CG  TYR   126      -5.320  33.540 102.144  1.00 50.00\n"
		"ATOM   2811  CD1 TYR   126      -5.633  34.253 103.309  1.00 49.88\n"
		"ATOM   2812  CD2 TYR   126      -5.117  34.248 100.970  1.00 49.88\n"
		"ATOM   2813  CE1 TYR   126      -5.737  35.644 103.284  1.00 49.88\n"
		"ATOM   2814  CE2 TYR   126      -5.218  35.637 100.944  1.00 49.88\n"
		"ATOM   2815  CZ  TYR   126      -5.522  36.329 102.095  1.00 50.11\n"
		"ATOM   2816  OH  TYR   126      -5.553  37.696 102.044  1.00 49.46\n"
		"ATOM   2817  H   TYR   126      -7.009  31.017 103.969  1.00 50.31\n"
		"ATOM   2818  HH  TYR   126      -5.350  38.044 101.172  1.00 50.43\n"
		"ATOM   2819  HD1 TYR   126      -5.795  33.702 104.235  1.00 50.12\n"
		"ATOM   2820  HE1 TYR   126      -5.984  36.194 104.191  1.00 50.12\n"
		"ATOM   2821  HE2 TYR   126      -5.057  36.178 100.011  1.00 50.12\n"
		"ATOM   2822  HD2 TYR   126      -4.876  33.699 100.059  1.00 50.12\n"
		"ATOM   2823  HA  TYR   126      -7.069  32.002 101.123  1.00 50.09\n"
		"ATOM   2824 1HB  TYR   126      -4.865  31.691 103.104  1.00 50.09\n"
		"ATOM   2825 2HB  TYR   126      -4.546  31.718 101.353  1.00 50.09\n"
		"ATOM   2826  N   GLY   127      -7.028  29.023 101.775  1.00 49.53\n"
		"ATOM   2827  CA  GLY   127      -7.042  27.743 101.060  1.00 49.98\n"
		"ATOM   2828  C   GLY   127      -6.239  26.641 101.678  1.00 50.51\n"
		"ATOM   2829  O   GLY   127      -6.048  25.601 101.056  1.00 49.49\n"
		"ATOM   2830  H   GLY   127      -7.526  29.126 102.648  1.00 50.31\n"
		"ATOM   2831 1HA  GLY   127      -8.069  27.382 100.996  1.00 50.09\n"
		"ATOM   2832 2HA  GLY   127      -6.647  27.892 100.056  1.00 50.09\n"
		"ATOM   2833  N   PHE   128      -5.792  26.847 102.897  1.00 49.53\n"
		"ATOM   2834  CA  PHE   128      -4.975  25.882 103.571  1.00 50.07\n"
		"ATOM   2835  C   PHE   128      -5.853  24.948 104.379  1.00 50.51\n"
		"ATOM   2836  O   PHE   128      -6.775  25.392 105.054  1.00 49.49\n"
		"ATOM   2837  CB  PHE   128      -3.984  26.611 104.477  1.00 49.82\n"
		"ATOM   2838  CG  PHE   128      -2.987  27.472 103.714  1.00 50.00\n"
		"ATOM   2839  CD1 PHE   128      -1.837  26.913 103.165  1.00 49.88\n"
		"ATOM   2840  CD2 PHE   128      -3.225  28.805 103.454  1.00 49.88\n"
		"ATOM   2841  CE1 PHE   128      -0.972  27.633 102.348  1.00 49.88\n"
		"ATOM   2842  CE2 PHE   128      -2.370  29.546 102.631  1.00 49.88\n"
		"ATOM   2843  CZ  PHE   128      -1.245  28.964 102.084  1.00 49.88\n"
		"ATOM   2844  H   PHE   128      -6.030  27.707 103.370  1.00 50.31\n"
		"ATOM   2845  HD1 PHE   128      -1.598  25.871 103.378  1.00 50.12\n"
		"ATOM   2846  HE1 PHE   128      -0.093  27.148 101.925  1.00 50.12\n"
		"ATOM   2847  HZ  PHE   128      -0.575  29.544 101.451  1.00 50.12\n"
		"ATOM   2848  HE2 PHE   128      -2.589  30.592 102.417  1.00 50.12\n"
		"ATOM   2849  HD2 PHE   128      -4.097  29.287 103.899  1.00 50.12\n"
		"ATOM   2850  HA  PHE   128      -4.441  25.274 102.840  1.00 50.09\n"
		"ATOM   2851 1HB  PHE   128      -4.516  27.274 105.159  1.00 50.09\n"
		"ATOM   2852 2HB  PHE   128      -3.403  25.891 105.052  1.00 50.09\n"
		"ATOM   2853  N   GLU   129      -5.542  23.664 104.319  1.00 49.53\n"
		"ATOM   2854  CA  GLU   129      -6.325  22.669 104.997  1.00 50.07\n"
		"ATOM   2855  C   GLU   129      -6.185  22.744 106.497  1.00 50.51\n"
		"ATOM   2856  O   GLU   129      -6.879  22.016 107.226  1.00 49.49\n"
		"ATOM   2857  CB  GLU   129      -5.925  21.275 104.524  1.00 49.82\n"
		"ATOM   2858  CG  GLU   129      -6.571  20.826 103.210  1.00 49.72\n"
		"ATOM   2859  CD  GLU   129      -8.097  20.919 103.215  1.00 50.62\n"
		"ATOM   2860  OE1 GLU   129      -8.765  20.008 103.782  1.00 49.24\n"
		"ATOM   2861  OE2 GLU   129      -8.632  21.901 102.637  1.00 49.24\n"
		"ATOM   2862  H   GLU   129      -4.735  23.376 103.784  1.00 50.31\n"
		"ATOM   2863  HA  GLU   129      -7.384  22.827 104.789  1.00 50.09\n"
		"ATOM   2864 1HB  GLU   129      -4.841  21.278 104.406  1.00 50.09\n"
		"ATOM   2865 2HB  GLU   129      -6.203  20.578 105.315  1.00 50.09\n"
		"ATOM   2866 1HG  GLU   129      -6.203  21.367 102.339  1.00 50.09\n"
		"ATOM   2867 2HG  GLU   129      -6.289  19.777 103.131  1.00 50.09\n"
		"ATOM   2868  N   SER   130      -5.276  23.588 106.981  1.00 49.53\n"
		"ATOM   2869  CA  SER   130      -5.104  23.721 108.443  1.00 50.07\n"
		"ATOM   2870  C   SER   130      -4.076  24.796 108.854  1.00 50.51\n"
		"ATOM   2871  O   SER   130      -3.457  25.456 108.016  1.00 49.49\n"
		"ATOM   2872  CB  SER   130      -4.725  22.378 109.043  1.00 50.05\n"
		"ATOM   2873  OG  SER   130      -3.341  22.185 108.978  1.00 49.34\n"
		"ATOM   2874  H   SER   130      -4.706  24.135 106.352  1.00 50.31\n"
		"ATOM   2875  HG  SER   130      -3.119  21.333 109.361  1.00 50.43\n"
		"ATOM   2876  HA  SER   130      -6.037  23.918 108.973  1.00 50.09\n"
		"ATOM   2877 1HB  SER   130      -5.052  22.356 110.082  1.00 50.09\n"
		"ATOM   2878 2HB  SER   130      -5.233  21.591 108.487  1.00 50.09\n"
		"ATOM   2879  N   GLN   131      -3.927  24.985 110.142  1.00 49.53\n"
		"ATOM   2880  CA  GLN   131      -3.003  25.960 110.658  1.00 50.07\n"
		"ATOM   2881  C   GLN   131      -1.572  25.495 110.473  1.00 50.51\n"
		"ATOM   2882  O   GLN   131      -0.714  26.260 110.070  1.00 49.49\n"
		"ATOM   2883  CB  GLN   131      -3.264  26.149 112.125  1.00 49.82\n"
		"ATOM   2884  CG  GLN   131      -2.508  27.325 112.698  1.00 49.82\n"
		"ATOM   2885  CD  GLN   131      -2.749  27.500 114.167  1.00 50.55\n"
		"ATOM   2886  OE1 GLN   131      -2.157  28.369 114.826  1.00 49.45\n"
		"ATOM   2887  NE2 GLN   131      -3.618  26.670 114.697  1.00 49.38\n"
		"ATOM   2888  H   GLN   131      -4.474  24.433 110.788  1.00 50.31\n"
		"ATOM   2889 1HE2 GLN   131      -3.831  26.725 115.673  1.00 50.32\n"
		"ATOM   2890 2HE2 GLN   131      -4.067  25.984 114.125  1.00 50.30\n"
		"ATOM   2891  HA  GLN   131      -3.105  26.891 110.100  1.00 50.09\n"
		"ATOM   2892 1HB  GLN   131      -4.339  26.291 112.238  1.00 50.09\n"
		"ATOM   2893 2HB  GLN   131      -2.966  25.224 112.619  1.00 50.09\n"
		"ATOM   2894 1HG  GLN   131      -1.448  27.492 112.511  1.00 50.09\n"
		"ATOM   2895 2HG  GLN   131      -3.092  28.073 112.161  1.00 50.09\n"
		"ATOM   2896  N   GLN   132      -1.345  24.217 110.781  1.00 49.53\n"
		"ATOM   2897  CA  GLN   132      -0.053  23.575 110.664  1.00 50.07\n"
		"ATOM   2898  C   GLN   132       0.468  23.671 109.261  1.00 50.51\n"
		"ATOM   2899  O   GLN   132       1.634  23.960 109.054  1.00 49.49\n"
		"ATOM   2900  CB  GLN   132      -0.152  22.121 111.074  1.00 49.82\n"
		"ATOM   2901  CG  GLN   132      -0.602  21.950 112.533  1.00 49.82\n"
		"ATOM   2902  CD  GLN   132      -2.072  22.310 112.757  1.00 50.55\n"
		"ATOM   2903  OE1 GLN   132      -2.837  22.526 111.818  1.00 49.45\n"
		"ATOM   2904  NE2 GLN   132      -2.474  22.376 114.012  1.00 49.38\n"
		"ATOM   2905  H   GLN   132      -2.121  23.664 111.116  1.00 50.31\n"
		"ATOM   2906 1HE2 GLN   132      -3.425  22.607 114.222  1.00 50.32\n"
		"ATOM   2907 2HE2 GLN   132      -1.829  22.194 114.754  1.00 50.30\n"
		"ATOM   2908  HA  GLN   132       0.668  24.087 111.301  1.00 50.09\n"
		"ATOM   2909 1HB  GLN   132      -0.863  21.650 110.395  1.00 50.09\n"
		"ATOM   2910 2HB  GLN   132       0.835  21.684 110.925  1.00 50.09\n"
		"ATOM   2911 1HG  GLN   132      -0.397  21.050 113.112  1.00 50.09\n"
		"ATOM   2912 2HG  GLN   132       0.012  22.769 112.909  1.00 50.09\n"
		"ATOM   2913  N   THR   133      -0.391  23.430 108.288  1.00 49.53\n"
		"ATOM   2914  CA  THR   133       0.006  23.527 106.887  1.00 50.07\n"
		"ATOM   2915  C   THR   133       0.429  24.954 106.531  1.00 50.51\n"
		"ATOM   2916  O   THR   133       1.401  25.162 105.820  1.00 49.49\n"
		"ATOM   2917  CB  THR   133      -1.149  23.057 105.971  1.00 50.14\n"
		"ATOM   2918  OG1 THR   133      -1.089  21.634 105.817  1.00 49.34\n"
		"ATOM   2919  CG2 THR   133      -1.057  23.713 104.611  1.00 49.73\n"
		"ATOM   2920  H   THR   133      -1.341  23.174 108.517  1.00 50.31\n"
		"ATOM   2921  HG1 THR   133      -1.806  21.344 105.248  1.00 50.43\n"
		"ATOM   2922  HA  THR   133       0.874  22.893 106.705  1.00 50.09\n"
		"ATOM   2923  HB  THR   133      -2.099  23.324 106.433  1.00 50.09\n"
		"ATOM   2924 1HG2 THR   133      -1.880  23.367 103.985  1.00 50.09\n"
		"ATOM   2925 2HG2 THR   133      -1.116  24.795 104.725  1.00 50.09\n"
		"ATOM   2926 3HG2 THR   133      -0.109  23.448 104.144  1.00 50.09\n"
		"ATOM   2927  N   LEU   134      -0.315  25.932 107.027  1.00 49.53\n"
		"ATOM   2928  CA  LEU   134       0.008  27.331 106.758  1.00 50.07\n"
		"ATOM   2929  C   LEU   134       1.210  27.747 107.561  1.00 50.51\n"
		"ATOM   2930  O   LEU   134       2.000  28.531 107.079  1.00 49.49\n"
		"ATOM   2931  CB  LEU   134      -1.131  28.250 107.127  1.00 49.82\n"
		"ATOM   2932  CG  LEU   134      -0.800  29.731 107.162  1.00 49.91\n"
		"ATOM   2933  CD1 LEU   134      -0.481  30.242 105.798  1.00 49.73\n"
		"ATOM   2934  CD2 LEU   134      -1.982  30.505 107.732  1.00 49.73\n"
		"ATOM   2935  H   LEU   134      -1.116  25.707 107.599  1.00 50.31\n"
		"ATOM   2936  HA  LEU   134       0.270  27.450 105.707  1.00 50.09\n"
		"ATOM   2937 1HB  LEU   134      -1.803  28.041 106.297  1.00 50.09\n"
		"ATOM   2938 2HB  LEU   134      -1.605  27.951 108.062  1.00 50.09\n"
		"ATOM   2939  HG  LEU   134       0.051  29.857 107.832  1.00 50.09\n"
		"ATOM   2940 1HD1 LEU   134      -0.248  31.305 105.853  1.00 50.09\n"
		"ATOM   2941 2HD1 LEU   134       0.379  29.703 105.399  1.00 50.09\n"
		"ATOM   2942 3HD1 LEU   134      -1.339  30.093 105.144  1.00 50.09\n"
		"ATOM   2943 1HD2 LEU   134      -1.742  31.568 107.757  1.00 50.09\n"
		"ATOM   2944 2HD2 LEU   134      -2.859  30.345 107.105  1.00 50.09\n"
		"ATOM   2945 3HD2 LEU   134      -2.192  30.157 108.744  1.00 50.09\n"
		"ATOM   2946  N   THR   135       1.267  27.295 108.813  1.00 49.53\n"
		"ATOM   2947  CA  THR   135       2.388  27.553 109.661  1.00 50.07\n"
		"ATOM   2948  C   THR   135       3.685  27.105 108.973  1.00 50.51\n"
		"ATOM   2949  O   THR   135       4.640  27.866 108.906  1.00 49.49\n"
		"ATOM   2950  CB  THR   135       2.234  26.802 110.985  1.00 50.14\n"
		"ATOM   2951  OG1 THR   135       0.964  27.078 111.572  1.00 49.34\n"
		"ATOM   2952  CG2 THR   135       3.322  27.200 111.963  1.00 49.73\n"
		"ATOM   2953  H   THR   135       0.494  26.754 109.174  1.00 50.31\n"
		"ATOM   2954  HG1 THR   135       0.887  28.021 111.737  1.00 50.43\n"
		"ATOM   2955  HA  THR   135       2.477  28.625 109.839  1.00 50.09\n"
		"ATOM   2956  HB  THR   135       2.302  25.731 110.795  1.00 50.09\n"
		"ATOM   2957 1HG2 THR   135       3.190  26.652 112.896  1.00 50.09\n"
		"ATOM   2958 2HG2 THR   135       4.297  26.964 111.537  1.00 50.09\n"
		"ATOM   2959 3HG2 THR   135       3.260  28.270 112.159  1.00 50.09\n"
		"ATOM   2960  N   ARG   136       3.731  25.887 108.437  1.00 49.53\n"
		"ATOM   2961  CA  ARG   136       4.965  25.424 107.745  1.00 50.07\n"
		"ATOM   2962  C   ARG   136       5.257  26.156 106.433  1.00 50.51\n"
		"ATOM   2963  O   ARG   136       6.412  26.428 106.126  1.00 49.49\n"
		"ATOM   2964  CB  ARG   136       4.903  23.935 107.482  1.00 49.82\n"
		"ATOM   2965  CG  ARG   136       5.736  23.459 106.359  1.00 49.82\n"
		"ATOM   2966  CD  ARG   136       5.709  21.955 106.335  1.00 50.20\n"
		"ATOM   2967  NE  ARG   136       4.485  21.437 105.719  1.00 49.30\n"
		"ATOM   2968  CZ  ARG   136       4.123  20.157 105.825  1.00 50.64\n"
		"ATOM   2969  NH1 ARG   136       4.880  19.325 106.531  1.00 49.20\n"
		"ATOM   2970  NH2 ARG   136       3.017  19.704 105.251  1.00 49.20\n"
		"ATOM   2971  H   ARG   136       2.928  25.277 108.500  1.00 50.31\n"
		"ATOM   2972 1HH1 ARG   136       5.718  19.666 106.980  1.00 50.46\n"
		"ATOM   2973 2HH1 ARG   136       4.614  18.355 106.616  1.00 50.46\n"
		"ATOM   2974 1HH2 ARG   136       2.432  20.335 104.721  1.00 50.46\n"
		"ATOM   2975 2HH2 ARG   136       2.765  18.731 105.345  1.00 50.46\n"
		"ATOM   2976  HE  ARG   136       3.787  21.919 105.169  1.00 50.44\n"
		"ATOM   2977  HA  ARG   136       5.834  25.613 108.375  1.00 50.09\n"
		"ATOM   2978 1HB  ARG   136       5.220  23.437 108.397  1.00 50.09\n"
		"ATOM   2979 2HB  ARG   136       3.860  23.691 107.279  1.00 50.09\n"
		"ATOM   2980 1HG  ARG   136       5.333  23.866 105.431  1.00 50.09\n"
		"ATOM   2981 2HG  ARG   136       6.753  23.824 106.504  1.00 50.09\n"
		"ATOM   2982 1HD  ARG   136       6.565  21.595 105.766  1.00 50.09\n"
		"ATOM   2983 2HD  ARG   136       5.770  21.584 107.357  1.00 50.09\n"
		"ATOM   2984  N   THR   137       4.235  26.444 105.639  1.00 49.53\n"
		"ATOM   2985  CA  THR   137       4.451  27.186 104.390  1.00 50.07\n"
		"ATOM   2986  C   THR   137       4.967  28.625 104.697  1.00 50.51\n"
		"ATOM   2987  O   THR   137       5.887  29.145 104.053  1.00 49.49\n"
		"ATOM   2988  CB  THR   137       3.118  27.287 103.583  1.00 50.14\n"
		"ATOM   2989  OG1 THR   137       3.037  26.246 102.628  1.00 49.34\n"
		"ATOM   2990  CG2 THR   137       3.070  28.574 102.791  1.00 49.73\n"
		"ATOM   2991  H   THR   137       3.303  26.151 105.894  1.00 50.31\n"
		"ATOM   2992  HG1 THR   137       3.067  25.398 103.077  1.00 50.43\n"
		"ATOM   2993  HA  THR   137       5.222  26.694 103.797  1.00 50.09\n"
		"ATOM   2994  HB  THR   137       2.279  27.191 104.272  1.00 50.09\n"
		"ATOM   2995 1HG2 THR   137       2.133  28.624 102.237  1.00 50.09\n"
		"ATOM   2996 2HG2 THR   137       3.136  29.422 103.472  1.00 50.09\n"
		"ATOM   2997 3HG2 THR   137       3.906  28.602 102.094  1.00 50.09\n"
		"ATOM   2998  N   PHE   138       4.297  29.288 105.626  1.00 49.53\n"
		"ATOM   2999  CA  PHE   138       4.700  30.621 106.021  1.00 50.07\n"
		"ATOM   3000  C   PHE   138       6.160  30.590 106.449  1.00 50.51\n"
		"ATOM   3001  O   PHE   138       6.966  31.386 105.974  1.00 49.49\n"
		"ATOM   3002  CB  PHE   138       3.852  31.134 107.191  1.00 49.82\n"
		"ATOM   3003  CG  PHE   138       3.782  32.634 107.291  1.00 50.00\n"
		"ATOM   3004  CD1 PHE   138       2.918  33.338 106.473  1.00 49.88\n"
		"ATOM   3005  CD2 PHE   138       4.569  33.342 108.204  1.00 49.88\n"
		"ATOM   3006  CE1 PHE   138       2.800  34.716 106.577  1.00 49.88\n"
		"ATOM   3007  CE2 PHE   138       4.454  34.723 108.310  1.00 49.88\n"
		"ATOM   3008  CZ  PHE   138       3.582  35.415 107.483  1.00 49.88\n"
		"ATOM   3009  H   PHE   138       3.495  28.859 106.065  1.00 50.31\n"
		"ATOM   3010  HD1 PHE   138       2.322  32.797 105.737  1.00 50.12\n"
		"ATOM   3011  HE1 PHE   138       2.091  35.251 105.946  1.00 50.12\n"
		"ATOM   3012  HZ  PHE   138       3.511  36.500 107.542  1.00 50.12\n"
		"ATOM   3013  HE2 PHE   138       5.050  35.266 109.043  1.00 50.12\n"
		"ATOM   3014  HD2 PHE   138       5.274  32.798 108.833  1.00 50.12\n"
		"ATOM   3015  HA  PHE   138       4.620  31.299 105.170  1.00 50.09\n"
		"ATOM   3016 1HB  PHE   138       2.825  30.784 107.092  1.00 50.09\n"
		"ATOM   3017 2HB  PHE   138       4.265  30.782 108.135  1.00 50.09\n"
		"ATOM   3018  N   LYS   139       6.487  29.631 107.335  1.00 49.53\n"
		"ATOM   3019  CA  LYS   139       7.838  29.472 107.938  1.00 50.07\n"
		"ATOM   3020  C   LYS   139       8.906  29.290 106.866  1.00 50.51\n"
		"ATOM   3021  O   LYS   139      10.009  29.798 106.982  1.00 49.49\n"
		"ATOM   3022  CB  LYS   139       7.885  28.253 108.889  1.00 49.82\n"
		"ATOM   3023  CG  LYS   139       9.305  27.828 109.272  1.00 49.82\n"
		"ATOM   3024  CD  LYS   139       9.712  28.345 110.616  1.00 49.82\n"
		"ATOM   3025  CE  LYS   139      11.156  27.951 110.899  1.00 50.21\n"
		"ATOM   3026  NZ  LYS   139      11.977  29.101 111.364  1.00 49.70\n"
		"ATOM   3027  H   LYS   139       5.768  28.976 107.608  1.00 50.31\n"
		"ATOM   3028 1HZ  LYS   139      12.923  28.791 111.537  1.00 50.33\n"
		"ATOM   3029 2HZ  LYS   139      11.983  29.820 110.654  1.00 50.33\n"
		"ATOM   3030 3HZ  LYS   139      11.583  29.473 112.216  1.00 50.33\n"
		"ATOM   3031  HA  LYS   139       8.104  30.374 108.490  1.00 50.09\n"
		"ATOM   3032 1HB  LYS   139       7.326  28.525 109.785  1.00 50.09\n"
		"ATOM   3033 2HB  LYS   139       7.378  27.432 108.382  1.00 50.09\n"
		"ATOM   3034 1HG  LYS   139       9.349  26.738 109.283  1.00 50.09\n"
		"ATOM   3035 2HG  LYS   139       9.996  28.211 108.521  1.00 50.09\n"
		"ATOM   3036 1HD  LYS   139       9.607  29.431 110.617  1.00 50.09\n"
		"ATOM   3037 2HD  LYS   139       9.049  27.917 111.368  1.00 50.09\n"
		"ATOM   3038 1HE  LYS   139      11.155  27.178 111.666  1.00 50.05\n"
		"ATOM   3039 2HE  LYS   139      11.587  27.551 109.981  1.00 50.05\n"
		"ATOM   3040  N   ASN   140       8.521  28.545 105.847  1.00 49.53\n"
		"ATOM   3041  CA  ASN   140       9.339  28.226 104.704  1.00 50.07\n"
		"ATOM   3042  C   ASN   140       9.565  29.455 103.786  1.00 50.51\n"
		"ATOM   3043  O   ASN   140      10.618  29.624 103.165  1.00 49.49\n"
		"ATOM   3044  CB  ASN   140       8.644  27.104 103.959  1.00 49.82\n"
		"ATOM   3045  CG  ASN   140       8.765  25.816 104.715  1.00 50.55\n"
		"ATOM   3046  OD1 ASN   140       9.066  25.793 105.926  1.00 49.45\n"
		"ATOM   3047  ND2 ASN   140       8.549  24.721 103.999  1.00 49.38\n"
		"ATOM   3048  H   ASN   140       7.583  28.170 105.870  1.00 50.31\n"
		"ATOM   3049 1HD2 ASN   140       8.611  23.821 104.432  1.00 50.32\n"
		"ATOM   3050 2HD2 ASN   140       8.325  24.795 103.028  1.00 50.30\n"
		"ATOM   3051  HA  ASN   140      10.330  27.911 105.035  1.00 50.09\n"
		"ATOM   3052 1HB  ASN   140       7.639  27.239 103.558  1.00 50.09\n"
		"ATOM   3053 2HB  ASN   140       9.365  27.114 103.142  1.00 50.09\n"
		"ATOM   3054  N   TYR   141       8.601  30.339 103.740  1.00 49.53\n"
		"ATOM   3055  CA  TYR   141       8.693  31.482 102.847  1.00 50.07\n"
		"ATOM   3056  C   TYR   141       9.343  32.665 103.531  1.00 50.51\n"
		"ATOM   3057  O   TYR   141      10.003  33.456 102.854  1.00 49.49\n"
		"ATOM   3058  CB  TYR   141       7.291  31.885 102.357  1.00 49.82\n"
		"ATOM   3059  CG  TYR   141       7.275  33.026 101.398  1.00 50.00\n"
		"ATOM   3060  CD1 TYR   141       7.414  34.342 101.807  1.00 49.88\n"
		"ATOM   3061  CD2 TYR   141       7.125  32.776 100.046  1.00 49.88\n"
		"ATOM   3062  CE1 TYR   141       7.391  35.386 100.891  1.00 49.88\n"
		"ATOM   3063  CE2 TYR   141       7.101  33.792  99.112  1.00 49.88\n"
		"ATOM   3064  CZ  TYR   141       7.236  35.099  99.540  1.00 50.11\n"
		"ATOM   3065  OH  TYR   141       7.133  36.111  98.645  1.00 49.46\n"
		"ATOM   3066  H   TYR   141       7.788  30.226 104.329  1.00 50.31\n"
		"ATOM   3067  HH  TYR   141       7.242  36.978  99.044  1.00 50.43\n"
		"ATOM   3068  HD1 TYR   141       7.542  34.552 102.869  1.00 50.12\n"
		"ATOM   3069  HE1 TYR   141       7.494  36.417 101.228  1.00 50.12\n"
		"ATOM   3070  HE2 TYR   141       6.976  33.557  98.055  1.00 50.12\n"
		"ATOM   3071  HD2 TYR   141       7.022  31.744  99.710  1.00 50.12\n"
		"ATOM   3072  HA  TYR   141       9.323  31.234 101.992  1.00 50.09\n"
		"ATOM   3073 1HB  TYR   141       6.852  31.009 101.878  1.00 50.09\n"
		"ATOM   3074 2HB  TYR   141       6.706  32.149 103.237  1.00 50.09\n"
		"ATOM   3075  N   PHE   142       9.150  32.781 104.867  1.00 49.53\n"
		"ATOM   3076  CA  PHE   142       9.645  33.927 105.667  1.00 50.07\n"
		"ATOM   3077  C   PHE   142      10.725  33.563 106.690  1.00 50.51\n"
		"ATOM   3078  O   PHE   142      11.448  34.442 107.155  1.00 49.49\n"
		"ATOM   3079  CB  PHE   142       8.476  34.561 106.413  1.00 49.82\n"
		"ATOM   3080  CG  PHE   142       7.651  35.442 105.551  1.00 50.00\n"
		"ATOM   3081  CD1 PHE   142       8.181  36.621 105.008  1.00 49.88\n"
		"ATOM   3082  CD2 PHE   142       6.328  35.118 105.308  1.00 49.88\n"
		"ATOM   3083  CE1 PHE   142       7.401  37.453 104.219  1.00 49.88\n"
		"ATOM   3084  CE2 PHE   142       5.533  35.960 104.512  1.00 49.88\n"
		"ATOM   3085  CZ  PHE   142       6.068  37.129 103.976  1.00 49.88\n"
		"ATOM   3086  H   PHE   142       8.643  32.049 105.343  1.00 50.31\n"
		"ATOM   3087  HD1 PHE   142       9.220  36.881 105.212  1.00 50.12\n"
		"ATOM   3088  HE1 PHE   142       7.832  38.358 103.791  1.00 50.12\n"
		"ATOM   3089  HZ  PHE   142       5.448  37.788 103.371  1.00 50.12\n"
		"ATOM   3090  HE2 PHE   142       4.493  35.701 104.312  1.00 50.12\n"
		"ATOM   3091  HD2 PHE   142       5.915  34.205 105.740  1.00 50.12\n"
		"ATOM   3092  HA  PHE   142      10.115  34.661 105.011  1.00 50.09\n"
		"ATOM   3093 1HB  PHE   142       7.817  33.787 106.804  1.00 50.09\n"
		"ATOM   3094 2HB  PHE   142       8.845  35.173 107.235  1.00 50.09\n"
		"ATOM   3095  N   ASP   143      10.813  32.285 107.054  1.00 49.53\n"
		"ATOM   3096  CA  ASP   143      11.799  31.821 108.050  1.00 50.07\n"
		"ATOM   3097  C   ASP   143      11.418  32.248 109.480  1.00 50.51\n"
		"ATOM   3098  O   ASP   143      12.236  32.268 110.408  1.00 49.49\n"
		"ATOM   3099  CB  ASP   143      13.204  32.299 107.675  1.00 49.72\n"
		"ATOM   3100  CG  ASP   143      13.749  31.592 106.433  1.00 50.62\n"
		"ATOM   3101  OD1 ASP   143      13.813  30.331 106.421  1.00 49.24\n"
		"ATOM   3102  OD2 ASP   143      14.094  32.306 105.456  1.00 49.24\n"
		"ATOM   3103  H   ASP   143      10.186  31.613 106.636  1.00 50.31\n"
		"ATOM   3104  HA  ASP   143      11.831  30.731 108.058  1.00 50.09\n"
		"ATOM   3105 1HB  ASP   143      13.280  33.381 107.566  1.00 50.09\n"
		"ATOM   3106 2HB  ASP   143      13.757  31.982 108.560  1.00 50.09\n"
		"ATOM   3107  N   VAL   144      10.133  32.556 109.620  1.00 49.53\n"
		"ATOM   3108  CA  VAL   144       9.498  32.972 110.853  1.00 50.07\n"
		"ATOM   3109  C   VAL   144       8.046  32.506 110.793  1.00 50.51\n"
		"ATOM   3110  O   VAL   144       7.376  32.797 109.813  1.00 49.49\n"
		"ATOM   3111  CB  VAL   144       9.492  34.504 110.995  1.00 49.91\n"
		"ATOM   3112  CG1 VAL   144       8.967  34.868 112.371  1.00 49.73\n"
		"ATOM   3113  CG2 VAL   144      10.878  35.074 110.817  1.00 49.73\n"
		"ATOM   3114  H   VAL   144       9.550  32.495 108.797  1.00 50.31\n"
		"ATOM   3115  HA  VAL   144       9.951  32.484 111.717  1.00 50.09\n"
		"ATOM   3116  HB  VAL   144       8.851  34.929 110.222  1.00 50.09\n"
		"ATOM   3117 1HG1 VAL   144       8.958  35.953 112.482  1.00 50.09\n"
		"ATOM   3118 2HG1 VAL   144       7.954  34.484 112.487  1.00 50.09\n"
		"ATOM   3119 3HG1 VAL   144       9.612  34.431 113.134  1.00 50.09\n"
		"ATOM   3120 1HG2 VAL   144      10.842  36.158 110.922  1.00 50.09\n"
		"ATOM   3121 2HG2 VAL   144      11.542  34.657 111.575  1.00 50.09\n"
		"ATOM   3122 3HG2 VAL   144      11.252  34.818 109.826  1.00 50.09\n"
		"ATOM   3123  N   PRO   145       7.539  31.768 111.797  1.00 49.71\n"
		"ATOM   3124  CA  PRO   145       6.162  31.291 111.803  1.00 50.02\n"
		"ATOM   3125  C   PRO   145       5.283  32.490 111.826  1.00 50.51\n"
		"ATOM   3126  O   PRO   145       5.778  33.584 112.164  1.00 49.49\n"
		"ATOM   3127  CB  PRO   145       6.043  30.486 113.059  1.00 49.82\n"
		"ATOM   3128  CG  PRO   145       7.451  30.099 113.329  1.00 49.82\n"
		"ATOM   3129  CD  PRO   145       8.244  31.336 112.999  1.00 50.00\n"
		"ATOM   3130 1HD  PRO   145       8.124  32.057 113.611  1.00 50.09\n"
		"ATOM   3131 2HD  PRO   145       9.126  31.167 112.680  1.00 50.09\n"
		"ATOM   3132 1HG  PRO   145       7.563  30.060 114.168  1.00 50.09\n"
		"ATOM   3133 2HG  PRO   145       7.727  29.537 112.631  1.00 50.09\n"
		"ATOM   3134 1HB  PRO   145       5.820  31.139 113.659  1.00 50.09\n"
		"ATOM   3135 2HB  PRO   145       5.619  29.788 112.770  1.00 50.09\n"
		"ATOM   3136  HA  PRO   145       5.985  30.745 110.988  1.00 50.09\n"
		"ATOM   3137  N   PRO   146       3.966  32.288 111.597  1.00 49.71\n"
		"ATOM   3138  CA  PRO   146       2.975  33.373 111.528  1.00 50.02\n"
		"ATOM   3139  C   PRO   146       2.782  34.199 112.788  1.00 50.51\n"
		"ATOM   3140  O   PRO   146       2.801  35.424 112.718  1.00 49.49\n"
		"ATOM   3141  CB  PRO   146       1.682  32.667 111.087  1.00 49.82\n"
		"ATOM   3142  CG  PRO   146       2.135  31.336 110.561  1.00 49.82\n"
		"ATOM   3143  CD  PRO   146       3.310  30.979 111.448  1.00 50.00\n"
		"ATOM   3144 1HD  PRO   146       3.064  30.740 112.338  1.00 50.09\n"
		"ATOM   3145 2HD  PRO   146       3.977  30.454 111.016  1.00 50.09\n"
		"ATOM   3146 1HG  PRO   146       1.573  30.748 110.797  1.00 50.09\n"
		"ATOM   3147 2HG  PRO   146       2.580  31.490 109.751  1.00 50.09\n"
		"ATOM   3148 1HB  PRO   146       1.289  32.510 111.898  1.00 50.09\n"
		"ATOM   3149 2HB  PRO   146       1.450  33.153 110.409  1.00 50.09\n"
		"ATOM   3150  HA  PRO   146       3.229  34.024 110.818  1.00 50.09\n"
		"ATOM   3151  N   HIS   147       2.604  33.566 113.934  1.00 49.53\n"
		"ATOM   3152  CA  HIS   147       2.365  34.345 115.128  1.00 50.07\n"
		"ATOM   3153  C   HIS   147       3.572  35.200 115.502  1.00 50.51\n"
		"ATOM   3154  O   HIS   147       3.435  36.420 115.736  1.00 49.49\n"
		"ATOM   3155  CB  HIS   147       1.928  33.435 116.279  1.00 49.92\n"
		"ATOM   3156  CG  HIS   147       1.459  34.232 117.446  1.00 50.22\n"
		"ATOM   3157  ND1 HIS   147       1.712  33.862 118.745  1.00 49.30\n"
		"ATOM   3158  CD2 HIS   147       0.924  35.480 117.504  1.00 49.95\n"
		"ATOM   3159  CE1 HIS   147       1.360  34.839 119.544  1.00 50.25\n"
		"ATOM   3160  NE2 HIS   147       0.880  35.836 118.821  1.00 49.64\n"
		"ATOM   3161  H   HIS   147       2.634  32.557 113.975  1.00 50.31\n"
		"ATOM   3162  HE2 HIS   147       0.535  36.718 119.172  1.00 50.32\n"
		"ATOM   3163  HA  HIS   147       1.544  35.040 114.954  1.00 50.09\n"
		"ATOM   3164 1HB  HIS   147       1.106  32.793 115.960  1.00 50.09\n"
		"ATOM   3165 2HB  HIS   147       2.764  32.818 116.608  1.00 50.09\n"
		"ATOM   3166  HE1 HIS   147       1.485  34.746 120.623  1.00 50.13\n"
		"ATOM   3167  HD2 HIS   147       0.562  36.167 116.739  1.00 50.09\n"
		"ATOM   3168  N   LYS   148       4.753  34.560 115.577  1.00 49.53\n"
		"ATOM   3169  CA  LYS   148       6.008  35.286 115.845  1.00 50.07\n"
		"ATOM   3170  C   LYS   148       6.165  36.443 114.836  1.00 50.51\n"
		"ATOM   3171  O   LYS   148       6.536  37.544 115.192  1.00 49.49\n"
		"ATOM   3172  CB  LYS   148       7.197  34.307 115.791  1.00 49.82\n"
		"ATOM   3173  CG  LYS   148       8.514  34.910 116.270  1.00 49.82\n"
		"ATOM   3174  CD  LYS   148       9.489  33.890 116.866  1.00 49.82\n"
		"ATOM   3175  CE  LYS   148      10.163  33.050 115.797  1.00 50.21\n"
		"ATOM   3176  NZ  LYS   148      10.931  31.926 116.414  1.00 49.70\n"
		"ATOM   3177  H   LYS   148       4.781  33.559 115.447  1.00 50.31\n"
		"ATOM   3178 1HZ  LYS   148      11.372  31.380 115.687  1.00 50.33\n"
		"ATOM   3179 2HZ  LYS   148      11.639  32.300 117.030  1.00 50.33\n"
		"ATOM   3180 3HZ  LYS   148      10.302  31.338 116.942  1.00 50.33\n"
		"ATOM   3181  HA  LYS   148       5.973  35.728 116.842  1.00 50.09\n"
		"ATOM   3182 1HB  LYS   148       6.935  33.452 116.415  1.00 50.09\n"
		"ATOM   3183 2HB  LYS   148       7.297  33.982 114.755  1.00 50.09\n"
		"ATOM   3184 1HG  LYS   148       9.003  35.391 115.422  1.00 50.09\n"
		"ATOM   3185 2HG  LYS   148       8.296  35.656 117.034  1.00 50.09\n"
		"ATOM   3186 1HD  LYS   148      10.252  34.428 117.431  1.00 50.09\n"
		"ATOM   3187 2HD  LYS   148       8.937  33.233 117.538  1.00 50.09\n"
		"ATOM   3188 1HE  LYS   148       9.393  32.657 115.135  1.00 50.05\n"
		"ATOM   3189 2HE  LYS   148      10.835  33.694 115.230  1.00 50.05\n"
		"ATOM   3190  N   TYR   149       5.829  36.215 113.570  1.00 49.53\n"
		"ATOM   3191  CA  TYR   149       5.886  37.295 112.581  1.00 50.07\n"
		"ATOM   3192  C   TYR   149       4.871  38.390 112.914  1.00 50.51\n"
		"ATOM   3193  O   TYR   149       5.083  39.574 112.633  1.00 49.49\n"
		"ATOM   3194  CB  TYR   149       5.594  36.737 111.177  1.00 49.82\n"
		"ATOM   3195  CG  TYR   149       5.772  37.731 110.018  1.00 50.00\n"
		"ATOM   3196  CD1 TYR   149       4.773  38.671 109.742  1.00 49.88\n"
		"ATOM   3197  CD2 TYR   149       6.918  37.748 109.203  1.00 49.88\n"
		"ATOM   3198  CE1 TYR   149       4.898  39.587 108.704  1.00 49.88\n"
		"ATOM   3199  CE2 TYR   149       7.041  38.656 108.172  1.00 49.88\n"
		"ATOM   3200  CZ  TYR   149       6.024  39.579 107.924  1.00 50.11\n"
		"ATOM   3201  OH  TYR   149       6.121  40.501 106.870  1.00 49.46\n"
		"ATOM   3202  H   TYR   149       5.533  35.292 113.288  1.00 50.31\n"
		"ATOM   3203  HH  TYR   149       6.937  40.415 106.370  1.00 50.43\n"
		"ATOM   3204  HD1 TYR   149       3.874  38.680 110.359  1.00 50.12\n"
		"ATOM   3205  HE1 TYR   149       4.106  40.310 108.510  1.00 50.12\n"
		"ATOM   3206  HE2 TYR   149       7.935  38.654 107.547  1.00 50.12\n"
		"ATOM   3207  HD2 TYR   149       7.713  37.028 109.398  1.00 50.12\n"
		"ATOM   3208  HA  TYR   149       6.871  37.763 112.600  1.00 50.09\n"
		"ATOM   3209 1HB  TYR   149       6.267  35.893 111.019  1.00 50.09\n"
		"ATOM   3210 2HB  TYR   149       4.563  36.385 111.179  1.00 50.09\n"
		"ATOM   3211  N   ARG   150       3.754  37.977 113.487  1.00 49.53\n"
		"ATOM   3212  CA  ARG   150       2.701  38.907 113.813  1.00 50.07\n"
		"ATOM   3213  C   ARG   150       3.138  39.882 114.889  1.00 50.51\n"
		"ATOM   3214  O   ARG   150       2.820  41.073 114.831  1.00 49.49\n"
		"ATOM   3215  CB  ARG   150       1.454  38.134 114.269  1.00 49.82\n"
		"ATOM   3216  CG  ARG   150       0.543  38.886 115.247  1.00 49.82\n"
		"ATOM   3217  CD  ARG   150      -0.784  39.265 114.619  1.00 50.20\n"
		"ATOM   3218  NE  ARG   150      -1.718  38.151 114.657  1.00 49.30\n"
		"ATOM   3219  CZ  ARG   150      -2.734  38.028 113.836  1.00 50.64\n"
		"ATOM   3220  NH1 ARG   150      -2.976  38.975 112.955  1.00 49.20\n"
		"ATOM   3221  NH2 ARG   150      -3.514  36.969 113.903  1.00 49.20\n"
		"ATOM   3222  H   ARG   150       3.636  36.997 113.700  1.00 50.31\n"
		"ATOM   3223 1HH1 ARG   150      -2.379  39.789 112.916  1.00 50.46\n"
		"ATOM   3224 2HH1 ARG   150      -3.757  38.884 112.322  1.00 50.46\n"
		"ATOM   3225 1HH2 ARG   150      -3.327  36.251 114.589  1.00 50.46\n"
		"ATOM   3226 2HH2 ARG   150      -4.294  36.880 113.270  1.00 50.46\n"
		"ATOM   3227  HE  ARG   150      -1.740  37.349 115.272  1.00 50.44\n"
		"ATOM   3228  HA  ARG   150       2.453  39.506 112.936  1.00 50.09\n"
		"ATOM   3229 1HB  ARG   150       0.888  37.886 113.372  1.00 50.09\n"
		"ATOM   3230 2HB  ARG   150       1.804  37.217 114.742  1.00 50.09\n"
		"ATOM   3231 1HG  ARG   150       0.358  38.247 116.110  1.00 50.09\n"
		"ATOM   3232 2HG  ARG   150       1.056  39.793 115.570  1.00 50.09\n"
		"ATOM   3233 1HD  ARG   150      -1.208  40.106 115.167  1.00 50.09\n"
		"ATOM   3234 2HD  ARG   150      -0.615  39.555 113.583  1.00 50.09\n"
		"ATOM   3235  N   MET   151       3.875  39.385 115.860  1.00 49.53\n"
		"ATOM   3236  CA  MET   151       4.284  40.194 117.016  1.00 50.07\n"
		"ATOM   3237  C   MET   151       5.677  40.815 116.907  1.00 50.51\n"
		"ATOM   3238  O   MET   151       6.343  40.914 117.918  1.00 49.49\n"
		"ATOM   3239  CB  MET   151       4.236  39.329 118.285  1.00 49.82\n"
		"ATOM   3240  CG  MET   151       2.992  38.488 118.432  1.00 49.86\n"
		"ATOM   3241  SD  MET   151       1.837  39.192 119.621  1.00 49.91\n"
		"ATOM   3242  CE  MET   151       0.988  40.439 118.707  1.00 49.78\n"
		"ATOM   3243  H   MET   151       4.168  38.420 115.809  1.00 50.31\n"
		"ATOM   3244  HA  MET   151       3.612  41.044 117.131  1.00 50.09\n"
		"ATOM   3245 1HB  MET   151       5.109  38.679 118.258  1.00 50.09\n"
		"ATOM   3246 2HB  MET   151       4.313  40.008 119.135  1.00 50.09\n"
		"ATOM   3247 1HG  MET   151       2.512  38.418 117.456  1.00 50.09\n"
		"ATOM   3248 2HG  MET   151       3.292  37.494 118.762  1.00 50.09\n"
		"ATOM   3249 1HE  MET   151       0.257  40.929 119.351  1.00 50.09\n"
		"ATOM   3250 2HE  MET   151       1.705  41.177 118.345  1.00 50.09\n"
		"ATOM   3251 3HE  MET   151       0.476  39.983 117.859  1.00 50.09\n"
		"ATOM   3252  N   THR   152       6.116  41.245 115.727  1.00 49.53\n"
		"ATOM   3253  CA  THR   152       7.441  41.840 115.626  1.00 50.07\n"
		"ATOM   3254  C   THR   152       7.434  43.036 114.737  1.00 50.51\n"
		"ATOM   3255  O   THR   152       6.933  42.973 113.629  1.00 49.49\n"
		"ATOM   3256  CB  THR   152       8.415  40.863 115.047  1.00 50.14\n"
		"ATOM   3257  OG1 THR   152       7.865  40.351 113.822  1.00 49.34\n"
		"ATOM   3258  CG2 THR   152       8.645  39.744 116.050  1.00 49.73\n"
		"ATOM   3259  H   THR   152       5.534  41.159 114.906  1.00 50.31\n"
		"ATOM   3260  HG1 THR   152       7.029  39.915 114.005  1.00 50.43\n"
		"ATOM   3261  HA  THR   152       7.764  42.189 116.607  1.00 50.09\n"
		"ATOM   3262  HB  THR   152       9.353  41.378 114.840  1.00 50.09\n"
		"ATOM   3263 1HG2 THR   152       9.354  39.027 115.637  1.00 50.09\n"
		"ATOM   3264 2HG2 THR   152       9.045  40.161 116.974  1.00 50.09\n"
		"ATOM   3265 3HG2 THR   152       7.700  39.243 116.257  1.00 50.09\n"
		"ATOM   3266  N   ASN   153       8.015  44.125 115.224  1.00 49.53\n"
		"ATOM   3267  CA  ASN   153       7.992  45.375 114.481  1.00 50.07\n"
		"ATOM   3268  C   ASN   153       9.006  45.363 113.346  1.00 50.51\n"
		"ATOM   3269  O   ASN   153       8.613  45.402 112.177  1.00 49.49\n"
		"ATOM   3270  CB  ASN   153       8.251  46.591 115.417  1.00 49.82\n"
		"ATOM   3271  CG  ASN   153       7.438  46.628 116.731  1.00 50.55\n"
		"ATOM   3272  OD1 ASN   153       7.919  46.219 117.807  1.00 49.45\n"
		"ATOM   3273  ND2 ASN   153       6.183  47.089 116.638  1.00 49.38\n"
		"ATOM   3274  H   ASN   153       8.478  44.085 116.121  1.00 50.31\n"
		"ATOM   3275 1HD2 ASN   153       5.607  47.138 117.455  1.00 50.32\n"
		"ATOM   3276 2HD2 ASN   153       5.823  47.385 115.754  1.00 50.30\n"
		"ATOM   3277  HA  ASN   153       7.013  45.512 114.018  1.00 50.09\n"
		"ATOM   3278 1HB  ASN   153       9.268  46.904 115.657  1.00 50.09\n"
		"ATOM   3279 2HB  ASN   153       7.807  47.303 114.721  1.00 50.09\n"
		"ATOM   3280  N   MET   154      10.300  45.281 113.707  1.00 49.53\n"
		"ATOM   3281  CA  MET   154      11.418  45.341 112.752  1.00 50.07\n"
		"ATOM   3282  C   MET   154      11.136  44.544 111.504  1.00 50.51\n"
		"ATOM   3283  O   MET   154      11.402  45.021 110.390  1.00 49.49\n"
		"ATOM   3284  CB  MET   154      12.688  44.829 113.409  1.00 49.82\n"
		"ATOM   3285  CG  MET   154      13.624  45.920 113.901  1.00 49.86\n"
		"ATOM   3286  SD  MET   154      14.758  45.297 115.167  1.00 49.91\n"
		"ATOM   3287  CE  MET   154      13.845  45.684 116.675  1.00 49.78\n"
		"ATOM   3288  H   MET   154      10.516  45.172 114.688  1.00 50.31\n"
		"ATOM   3289  HA  MET   154      11.574  46.369 112.426  1.00 50.09\n"
		"ATOM   3290 1HB  MET   154      12.382  44.206 114.248  1.00 50.09\n"
		"ATOM   3291 2HB  MET   154      13.203  44.215 112.669  1.00 50.09\n"
		"ATOM   3292 1HG  MET   154      14.191  46.291 113.048  1.00 50.09\n"
		"ATOM   3293 2HG  MET   154      13.018  46.726 114.313  1.00 50.09\n"
		"ATOM   3294 1HE  MET   154      14.419  45.358 117.543  1.00 50.09\n"
		"ATOM   3295 2HE  MET   154      13.678  46.760 116.733  1.00 50.09\n"
		"ATOM   3296 3HE  MET   154      12.885  45.168 116.661  1.00 50.09\n"
		"ATOM   3297  N   GLN   155      10.630  43.303 111.709  1.00 49.53\n"
		"ATOM   3298  CA  GLN   155      10.223  42.445 110.588  1.00 50.07\n"
		"ATOM   3299  C   GLN   155       9.422  43.336 109.643  1.00 50.51\n"
		"ATOM   3300  O   GLN   155       8.369  43.853 110.051  1.00 49.49\n"
		"ATOM   3301  CB  GLN   155       9.356  41.249 111.087  1.00 49.82\n"
		"ATOM   3302  CG  GLN   155       9.832  39.866 110.620  1.00 49.82\n"
		"ATOM   3303  CD  GLN   155      11.073  39.332 111.347  1.00 50.55\n"
		"ATOM   3304  OE1 GLN   155      12.199  39.486 110.874  1.00 49.45\n"
		"ATOM   3305  NE2 GLN   155      10.874  38.714 112.513  1.00 49.38\n"
		"ATOM   3306  H   GLN   155      10.531  42.961 112.654  1.00 50.31\n"
		"ATOM   3307 1HE2 GLN   155      11.652  38.348 113.025  1.00 50.32\n"
		"ATOM   3308 2HE2 GLN   155       9.947  38.617 112.876  1.00 50.30\n"
		"ATOM   3309  HA  GLN   155      11.108  42.081 110.067  1.00 50.09\n"
		"ATOM   3310 1HB  GLN   155       9.365  41.287 112.176  1.00 50.09\n"
		"ATOM   3311 2HB  GLN   155       8.342  41.423 110.726  1.00 50.09\n"
		"ATOM   3312 1HG  GLN   155       9.158  39.017 110.509  1.00 50.09\n"
		"ATOM   3313 2HG  GLN   155      10.144  40.211 109.633  1.00 50.09\n"
		"ATOM   3314  N   GLY   156       9.964  43.556 108.412  1.00 49.53\n"
		"ATOM   3315  CA  GLY   156       9.338  44.461 107.458  1.00 49.98\n"
		"ATOM   3316  C   GLY   156       8.801  43.837 106.169  1.00 50.51\n"
		"ATOM   3317  O   GLY   156       8.453  42.645 106.099  1.00 49.49\n"
		"ATOM   3318  H   GLY   156      10.818  43.081 108.156  1.00 50.31\n"
		"ATOM   3319 1HA  GLY   156       8.489  44.950 107.935  1.00 50.09\n"
		"ATOM   3320 2HA  GLY   156      10.064  45.213 107.152  1.00 50.09\n"
		"ATOM   3321  N   GLU   157       8.730  44.699 105.149  1.00 49.53\n"
		"ATOM   3322  CA  GLU   157       8.201  44.363 103.841  1.00 50.07\n"
		"ATOM   3323  C   GLU   157       9.052  43.314 103.154  1.00 50.51\n"
		"ATOM   3324  O   GLU   157       8.747  42.848 102.062  1.00 49.49\n"
		"ATOM   3325  CB  GLU   157       8.124  45.629 102.993  1.00 49.82\n"
		"ATOM   3326  CG  GLU   157       7.317  45.455 101.724  1.00 49.72\n"
		"ATOM   3327  CD  GLU   157       6.152  46.422 101.621  1.00 50.62\n"
		"ATOM   3328  OE1 GLU   157       5.642  46.875 102.672  1.00 49.24\n"
		"ATOM   3329  OE2 GLU   157       5.745  46.727 100.478  1.00 49.24\n"
		"ATOM   3330  H   GLU   157       9.065  45.640 105.298  1.00 50.31\n"
		"ATOM   3331  HA  GLU   157       7.205  43.932 103.945  1.00 50.09\n"
		"ATOM   3332 1HB  GLU   157       7.674  46.402 103.616  1.00 50.09\n"
		"ATOM   3333 2HB  GLU   157       9.148  45.910 102.746  1.00 50.09\n"
		"ATOM   3334 1HG  GLU   157       7.913  45.539 100.816  1.00 50.09\n"
		"ATOM   3335 2HG  GLU   157       6.924  44.443 101.807  1.00 50.09\n"
		"ATOM   3336  N   SER   158      10.131  42.982 103.824  1.00 49.53\n"
		"ATOM   3337  CA  SER   158      11.053  41.970 103.364  1.00 50.07\n"
		"ATOM   3338  C   SER   158      10.313  40.729 102.782  1.00 50.51\n"
		"ATOM   3339  O   SER   158       9.466  40.127 103.446  1.00 49.49\n"
		"ATOM   3340  CB  SER   158      11.958  41.571 104.547  1.00 50.05\n"
		"ATOM   3341  OG  SER   158      12.808  40.485 104.253  1.00 49.34\n"
		"ATOM   3342  H   SER   158      10.324  43.453 104.696  1.00 50.31\n"
		"ATOM   3343  HG  SER   158      13.345  40.282 105.023  1.00 50.43\n"
		"ATOM   3344  HA  SER   158      11.772  42.340 102.632  1.00 50.09\n"
		"ATOM   3345 1HB  SER   158      12.574  42.428 104.818  1.00 50.09\n"
		"ATOM   3346 2HB  SER   158      11.328  41.297 105.392  1.00 50.09\n"
		"ATOM   3347  N   ARG   159      10.659  40.390 101.508  1.00 49.53\n"
		"ATOM   3348  CA  ARG   159      10.188  39.236 100.704  1.00 50.07\n"
		"ATOM   3349  C   ARG   159       8.774  39.370 100.143  1.00 50.51\n"
		"ATOM   3350  O   ARG   159       8.233  38.433  99.543  1.00 49.49\n"
		"ATOM   3351  CB  ARG   159      10.357  37.915 101.474  1.00 49.82\n"
		"ATOM   3352  CG  ARG   159      11.602  37.882 102.354  1.00 49.82\n"
		"ATOM   3353  CD  ARG   159      12.002  36.451 102.679  1.00 50.20\n"
		"ATOM   3354  NE  ARG   159      13.017  36.332 103.763  1.00 49.30\n"
		"ATOM   3355  CZ  ARG   159      13.490  35.145 104.178  1.00 50.64\n"
		"ATOM   3356  NH1 ARG   159      13.014  34.024 103.627  1.00 49.20\n"
		"ATOM   3357  NH2 ARG   159      14.408  35.065 105.120  1.00 49.20\n"
		"ATOM   3358  H   ARG   159      11.321  41.003 101.053  1.00 50.31\n"
		"ATOM   3359 1HH1 ARG   159      12.307  34.079 102.907  1.00 50.46\n"
		"ATOM   3360 2HH1 ARG   159      13.363  33.127 103.933  1.00 50.46\n"
		"ATOM   3361 1HH2 ARG   159      14.768  35.910 105.543  1.00 50.46\n"
		"ATOM   3362 2HH2 ARG   159      14.747  34.161 105.415  1.00 50.46\n"
		"ATOM   3363  HE  ARG   159      13.468  37.056 104.306  1.00 50.44\n"
		"ATOM   3364  HA  ARG   159      10.822  39.116  99.825  1.00 50.09\n"
		"ATOM   3365 1HB  ARG   159       9.465  37.787 102.085  1.00 50.09\n"
		"ATOM   3366 2HB  ARG   159      10.406  37.118 100.731  1.00 50.09\n"
		"ATOM   3367 1HG  ARG   159      12.415  38.378 101.824  1.00 50.09\n"
		"ATOM   3368 2HG  ARG   159      11.388  38.422 103.277  1.00 50.09\n"
		"ATOM   3369 1HD  ARG   159      11.117  35.899 102.995  1.00 50.09\n"
		"ATOM   3370 2HD  ARG   159      12.419  35.988 101.785  1.00 50.09\n"
		"ATOM   3371  N   PHE   160       8.185  40.525 100.287  1.00 49.53\n"
		"ATOM   3372  CA  PHE   160       6.858  40.682  99.767  1.00 50.07\n"
		"ATOM   3373  C   PHE   160       6.881  40.672  98.246  1.00 50.51\n"
		"ATOM   3374  O   PHE   160       7.744  41.291  97.634  1.00 49.49\n"
		"ATOM   3375  CB  PHE   160       6.252  42.004 100.218  1.00 49.82\n"
		"ATOM   3376  CG  PHE   160       5.610  42.000 101.595  1.00 50.00\n"
		"ATOM   3377  CD1 PHE   160       5.928  41.026 102.538  1.00 49.88\n"
		"ATOM   3378  CD2 PHE   160       4.711  42.986 101.950  1.00 49.88\n"
		"ATOM   3379  CE1 PHE   160       5.367  41.040 103.806  1.00 49.88\n"
		"ATOM   3380  CE2 PHE   160       4.130  43.016 103.228  1.00 49.88\n"
		"ATOM   3381  CZ  PHE   160       4.457  42.051 104.153  1.00 49.88\n"
		"ATOM   3382  H   PHE   160       8.652  41.290 100.753  1.00 50.31\n"
		"ATOM   3383  HD1 PHE   160       6.633  40.238 102.272  1.00 50.12\n"
		"ATOM   3384  HE1 PHE   160       5.634  40.268 104.528  1.00 50.12\n"
		"ATOM   3385  HZ  PHE   160       4.013  42.068 105.147  1.00 50.12\n"
		"ATOM   3386  HE2 PHE   160       3.421  43.802 103.487  1.00 50.12\n"
		"ATOM   3387  HD2 PHE   160       4.450  43.755 101.222  1.00 50.12\n"
		"ATOM   3388  HA  PHE   160       6.236  39.843 100.081  1.00 50.09\n"
		"ATOM   3389 1HB  PHE   160       7.020  42.776 100.255  1.00 50.09\n"
		"ATOM   3390 2HB  PHE   160       5.465  42.309  99.529  1.00 50.09\n"
		"ATOM   3391  N   LEU   161       5.947  39.950  97.628  1.00 49.53\n"
		"ATOM   3392  CA  LEU   161       5.788  39.997  96.171  1.00 50.07\n"
		"ATOM   3393  C   LEU   161       4.471  40.725  95.808  1.00 50.51\n"
		"ATOM   3394  O   LEU   161       3.406  40.463  96.372  1.00 49.49\n"
		"ATOM   3395  CB  LEU   161       5.781  38.579  95.617  1.00 49.82\n"
		"ATOM   3396  CG  LEU   161       5.535  38.418  94.123  1.00 49.91\n"
		"ATOM   3397  CD1 LEU   161       6.505  39.294  93.321  1.00 49.73\n"
		"ATOM   3398  CD2 LEU   161       5.719  36.954  93.770  1.00 49.73\n"
		"ATOM   3399  H   LEU   161       5.337  39.359  98.174  1.00 50.31\n"
		"ATOM   3400  HA  LEU   161       6.603  40.572  95.733  1.00 50.09\n"
		"ATOM   3401 1HB  LEU   161       6.807  38.294  95.847  1.00 50.09\n"
		"ATOM   3402 2HB  LEU   161       5.094  37.935  96.165  1.00 50.09\n"
		"ATOM   3403  HG  LEU   161       4.500  38.703  93.933  1.00 50.09\n"
		"ATOM   3404 1HD1 LEU   161       6.315  39.167  92.255  1.00 50.09\n"
		"ATOM   3405 2HD1 LEU   161       6.360  40.340  93.592  1.00 50.09\n"
		"ATOM   3406 3HD1 LEU   161       7.530  39.000  93.544  1.00 50.09\n"
		"ATOM   3407 1HD2 LEU   161       5.547  36.813  92.702  1.00 50.09\n"
		"ATOM   3408 2HD2 LEU   161       6.734  36.644  94.018  1.00 50.09\n"
		"ATOM   3409 3HD2 LEU   161       5.007  36.351  94.334  1.00 50.09\n"
		"ATOM   3410  N   HIS   162       4.501  41.638  94.867  1.00 49.53\n"
		"ATOM   3411  CA  HIS   162       3.274  42.322  94.568  1.00 50.07\n"
		"ATOM   3412  C   HIS   162       2.662  41.920  93.209  1.00 50.51\n"
		"ATOM   3413  O   HIS   162       3.379  41.580  92.302  1.00 49.49\n"
		"ATOM   3414  CB  HIS   162       3.522  43.802  94.695  1.00 49.92\n"
		"ATOM   3415  CG  HIS   162       3.616  44.227  96.128  1.00 50.22\n"
		"ATOM   3416  ND1 HIS   162       2.610  43.992  97.043  1.00 49.30\n"
		"ATOM   3417  CD2 HIS   162       4.631  44.803  96.829  1.00 49.95\n"
		"ATOM   3418  CE1 HIS   162       2.992  44.406  98.250  1.00 50.25\n"
		"ATOM   3419  NE2 HIS   162       4.218  44.899  98.152  1.00 49.64\n"
		"ATOM   3420  H   HIS   162       5.355  41.852  94.371  1.00 50.31\n"
		"ATOM   3421  HE2 HIS   162       4.770  45.283  98.905  1.00 50.32\n"
		"ATOM   3422  HA  HIS   162       2.521  42.065  95.312  1.00 50.09\n"
		"ATOM   3423 1HB  HIS   162       4.461  44.069  94.209  1.00 50.09\n"
		"ATOM   3424 2HB  HIS   162       2.705  44.360  94.237  1.00 50.09\n"
		"ATOM   3425  HE1 HIS   162       2.327  44.307  99.108  1.00 50.13\n"
		"ATOM   3426  HD2 HIS   162       5.618  45.164  96.541  1.00 50.09\n"
		"ATOM   3427  N   PRO   163       1.298  42.026  93.123  1.00 49.71\n"
		"ATOM   3428  CA  PRO   163       0.385  41.687  91.998  1.00 50.02\n"
		"ATOM   3429  C   PRO   163       0.805  41.232  90.555  1.00 50.51\n"
		"ATOM   3430  O   PRO   163      -0.085  40.827  89.762  1.00 49.49\n"
		"ATOM   3431  CB  PRO   163      -0.571  42.865  92.010  1.00 49.82\n"
		"ATOM   3432  CG  PRO   163      -0.800  43.055  93.467  1.00 49.82\n"
		"ATOM   3433  CD  PRO   163       0.463  42.638  94.198  1.00 50.00\n"
		"ATOM   3434 1HD  PRO   163       0.982  43.378  94.502  1.00 50.09\n"
		"ATOM   3435 2HD  PRO   163       0.327  41.914  94.801  1.00 50.09\n"
		"ATOM   3436 1HG  PRO   163      -0.787  43.885  93.635  1.00 50.09\n"
		"ATOM   3437 2HG  PRO   163      -1.364  42.364  93.756  1.00 50.09\n"
		"ATOM   3438 1HB  PRO   163      -0.017  43.536  91.728  1.00 50.09\n"
		"ATOM   3439 2HB  PRO   163      -1.280  42.508  91.664  1.00 50.09\n"
		"ATOM   3440  HA  PRO   163      -0.206  40.927  92.254  1.00 50.09\n"
		"ATOM   3441  N   LEU   164       2.075  41.220  90.148  1.00 49.53\n"
		"ATOM   3442  CA  LEU   164       2.288  40.762  88.748  1.00 50.07\n"
		"ATOM   3443  C   LEU   164       3.536  39.874  88.492  1.00 50.51\n"
		"ATOM   3444  O   LEU   164       3.391  38.639  88.590  1.00 49.49\n"
		"ATOM   3445  CB  LEU   164       2.230  41.972  87.793  1.00 49.82\n"
		"ATOM   3446  CG  LEU   164       1.098  42.990  88.046  1.00 49.91\n"
		"ATOM   3447  CD1 LEU   164       1.621  44.395  87.843  1.00 49.73\n"
		"ATOM   3448  CD2 LEU   164      -0.106  42.717  87.139  1.00 49.73\n"
		"ATOM   3449  H   LEU   164       2.838  41.502  90.747  1.00 50.31\n"
		"ATOM   3450  HA  LEU   164       1.453  40.134  88.438  1.00 50.09\n"
		"ATOM   3451 1HB  LEU   164       3.191  42.428  88.026  1.00 50.09\n"
		"ATOM   3452 2HB  LEU   164       2.219  41.662  86.748  1.00 50.09\n"
		"ATOM   3453  HG  LEU   164       0.748  42.838  89.067  1.00 50.09\n"
		"ATOM   3454 1HD1 LEU   164       0.819  45.112  88.023  1.00 50.09\n"
		"ATOM   3455 2HD1 LEU   164       2.438  44.584  88.540  1.00 50.09\n"
		"ATOM   3456 3HD1 LEU   164       1.983  44.504  86.821  1.00 50.09\n"
		"ATOM   3457 1HD2 LEU   164      -0.887  43.450  87.342  1.00 50.09\n"
		"ATOM   3458 2HD2 LEU   164       0.200  42.791  86.095  1.00 50.09\n"
		"ATOM   3459 3HD2 LEU   164      -0.490  41.716  87.335  1.00 50.09\n"
		"TER\n";

}

inline
std::string
trp_cage_ideal()
{
	return
		"ATOM      1  N   ASN A   1      -8.725   4.221  -0.522  1.00  0.00              \n"
		"ATOM      2  CA  ASN A   1      -8.590   3.159  -1.511  1.00  0.00              \n"
		"ATOM      3  C   ASN A   1      -7.130   2.937  -1.885  1.00  0.00              \n"
		"ATOM      4  O   ASN A   1      -6.662   1.800  -1.952  1.00  0.00              \n"
		"ATOM      5  CB  ASN A   1      -9.411   3.449  -2.754  1.00  0.00              \n"
		"ATOM      6  CG  ASN A   1     -10.882   3.189  -2.582  1.00  0.00              \n"
		"ATOM      7  OD1 ASN A   1     -11.326   2.684  -1.545  1.00  0.00              \n"
		"ATOM      8  ND2 ASN A   1     -11.628   3.456  -3.624  1.00  0.00              \n"
		"ATOM      9 1H   ASN A   1      -9.265   5.035  -0.734  1.00  0.00              \n"
		"ATOM     10 2H   ASN A   1      -9.137   4.026   0.368  1.00  0.00              \n"
		"ATOM     11 3H   ASN A   1      -7.913   4.695  -0.183  1.00  0.00              \n"
		"ATOM     12  HA  ASN A   1      -8.948   2.216  -1.094  1.00  0.00              \n"
		"ATOM     13 1HB  ASN A   1      -9.274   4.379  -3.309  1.00  0.00              \n"
		"ATOM     14 2HB  ASN A   1      -8.971   2.624  -3.315  1.00  0.00              \n"
		"ATOM     15 1HD2 ASN A   1     -12.616   3.307  -3.581  1.00  0.00              \n"
		"ATOM     16 2HD2 ASN A   1     -11.210   3.808  -4.461  1.00  0.00              \n"
		"ATOM     17  N   LEU A   2      -6.414   4.030  -2.127  1.00  0.00              \n"
		"ATOM     18  CA  LEU A   2      -4.993   3.959  -2.449  1.00  0.00              \n"
		"ATOM     19  C   LEU A   2      -4.238   3.117  -1.429  1.00  0.00              \n"
		"ATOM     20  O   LEU A   2      -3.492   2.207  -1.791  1.00  0.00              \n"
		"ATOM     21  CB  LEU A   2      -4.396   5.370  -2.524  1.00  0.00              \n"
		"ATOM     22  CG  LEU A   2      -4.849   6.329  -1.416  1.00  0.00              \n"
		"ATOM     23  CD1 LEU A   2      -3.657   6.742  -0.563  1.00  0.00              \n"
		"ATOM     24  CD2 LEU A   2      -5.514   7.548  -2.039  1.00  0.00              \n"
		"ATOM     25  H   LEU A   2      -6.865   4.932  -2.088  1.00  0.00              \n"
		"ATOM     26  HA  LEU A   2      -4.862   3.469  -3.413  1.00  0.00              \n"
		"ATOM     27 1HB  LEU A   2      -3.343   5.123  -2.397  1.00  0.00              \n"
		"ATOM     28 2HB  LEU A   2      -4.548   5.822  -3.504  1.00  0.00              \n"
		"ATOM     29  HG  LEU A   2      -5.599   5.807  -0.822  1.00  0.00              \n"
		"ATOM     30 1HD1 LEU A   2      -3.988   7.423   0.221  1.00  0.00              \n"
		"ATOM     31 2HD1 LEU A   2      -3.211   5.857  -0.109  1.00  0.00              \n"
		"ATOM     32 3HD1 LEU A   2      -2.918   7.241  -1.188  1.00  0.00              \n"
		"ATOM     33 1HD2 LEU A   2      -5.836   8.229  -1.250  1.00  0.00              \n"
		"ATOM     34 2HD2 LEU A   2      -4.804   8.057  -2.691  1.00  0.00              \n"
		"ATOM     35 3HD2 LEU A   2      -6.380   7.233  -2.621  1.00  0.00              \n"
		"ATOM     36  N   TYR A   3      -4.437   3.425  -0.152  1.00  0.00              \n"
		"ATOM     37  CA  TYR A   3      -3.717   2.748   0.920  1.00  0.00              \n"
		"ATOM     38  C   TYR A   3      -4.122   1.283   1.017  1.00  0.00              \n"
		"ATOM     39  O   TYR A   3      -3.317   0.430   1.389  1.00  0.00              \n"
		"ATOM     40  CB  TYR A   3      -3.963   3.451   2.257  1.00  0.00              \n"
		"ATOM     41  CG  TYR A   3      -2.813   3.335   3.232  1.00  0.00              \n"
		"ATOM     42  CD1 TYR A   3      -2.729   2.264   4.110  1.00  0.00              \n"
		"ATOM     43  CD2 TYR A   3      -1.813   4.295   3.270  1.00  0.00              \n"
		"ATOM     44  CE1 TYR A   3      -1.680   2.153   5.003  1.00  0.00              \n"
		"ATOM     45  CE2 TYR A   3      -0.760   4.194   4.158  1.00  0.00              \n"
		"ATOM     46  CZ  TYR A   3      -0.697   3.121   5.024  1.00  0.00              \n"
		"ATOM     47  OH  TYR A   3       0.351   3.015   5.910  1.00  0.00              \n"
		"ATOM     48  H   TYR A   3      -5.104   4.146   0.082  1.00  0.00              \n"
		"ATOM     49  HA  TYR A   3      -2.646   2.761   0.713  1.00  0.00              \n"
		"ATOM     50 1HB  TYR A   3      -4.149   4.504   2.039  1.00  0.00              \n"
		"ATOM     51 2HB  TYR A   3      -4.857   3.007   2.694  1.00  0.00              \n"
		"ATOM     52  HD1 TYR A   3      -3.509   1.503   4.088  1.00  0.00              \n"
		"ATOM     53  HD2 TYR A   3      -1.869   5.140   2.584  1.00  0.00              \n"
		"ATOM     54  HE1 TYR A   3      -1.627   1.306   5.688  1.00  0.00              \n"
		"ATOM     55  HE2 TYR A   3       0.016   4.960   4.173  1.00  0.00              \n"
		"ATOM     56  HH  TYR A   3       1.121   3.522   5.643  1.00  0.00              \n"
		"ATOM     57  N   ILE A   4      -5.376   0.998   0.681  1.00  0.00              \n"
		"ATOM     58  CA  ILE A   4      -5.853  -0.377   0.598  1.00  0.00              \n"
		"ATOM     59  C   ILE A   4      -5.083  -1.165  -0.455  1.00  0.00              \n"
		"ATOM     60  O   ILE A   4      -4.699  -2.312  -0.228  1.00  0.00              \n"
		"ATOM     61  CB  ILE A   4      -7.356  -0.435   0.272  1.00  0.00              \n"
		"ATOM     62  CG1 ILE A   4      -8.126   0.578   1.123  1.00  0.00              \n"
		"ATOM     63  CG2 ILE A   4      -7.895  -1.840   0.494  1.00  0.00              \n"
		"ATOM     64  CD1 ILE A   4      -7.917   0.410   2.611  1.00  0.00              \n"
		"ATOM     65  H   ILE A   4      -6.015   1.754   0.478  1.00  0.00              \n"
		"ATOM     66  HA  ILE A   4      -5.664  -0.906   1.532  1.00  0.00              \n"
		"ATOM     67  HB  ILE A   4      -7.505  -0.148  -0.768  1.00  0.00              \n"
		"ATOM     68 1HG1 ILE A   4      -7.799   1.572   0.820  1.00  0.00              \n"
		"ATOM     69 2HG1 ILE A   4      -9.185   0.458   0.890  1.00  0.00              \n"
		"ATOM     70 1HG2 ILE A   4      -8.959  -1.863   0.259  1.00  0.00              \n"
		"ATOM     71 2HG2 ILE A   4      -7.366  -2.539  -0.153  1.00  0.00              \n"
		"ATOM     72 3HG2 ILE A   4      -7.747  -2.127   1.535  1.00  0.00              \n"
		"ATOM     73 1HD1 ILE A   4      -8.495   1.163   3.148  1.00  0.00              \n"
		"ATOM     74 2HD1 ILE A   4      -8.246  -0.584   2.916  1.00  0.00              \n"
		"ATOM     75 3HD1 ILE A   4      -6.861   0.530   2.846  1.00  0.00              \n"
		"ATOM     76  N   GLN A   5      -4.861  -0.541  -1.607  1.00  0.00              \n"
		"ATOM     77  CA  GLN A   5      -4.136  -1.183  -2.697  1.00  0.00              \n"
		"ATOM     78  C   GLN A   5      -2.671  -1.395  -2.335  1.00  0.00              \n"
		"ATOM     79  O   GLN A   5      -2.035  -2.336  -2.808  1.00  0.00              \n"
		"ATOM     80  CB  GLN A   5      -4.235  -0.343  -3.974  1.00  0.00              \n"
		"ATOM     81  CG  GLN A   5      -4.195  -1.155  -5.257  1.00  0.00              \n"
		"ATOM     82  CD  GLN A   5      -5.514  -1.846  -5.547  1.00  0.00              \n"
		"ATOM     83  OE1 GLN A   5      -5.671  -3.045  -5.301  1.00  0.00              \n"
		"ATOM     84  NE2 GLN A   5      -6.471  -1.092  -6.077  1.00  0.00              \n"
		"ATOM     85  H   GLN A   5      -5.203   0.401  -1.731  1.00  0.00              \n"
		"ATOM     86  HA  GLN A   5      -4.555  -2.171  -2.883  1.00  0.00              \n"
		"ATOM     87 1HB  GLN A   5      -5.174   0.208  -3.915  1.00  0.00              \n"
		"ATOM     88 2HB  GLN A   5      -3.401   0.358  -3.956  1.00  0.00              \n"
		"ATOM     89 1HG  GLN A   5      -3.813  -0.756  -6.196  1.00  0.00              \n"
		"ATOM     90 2HG  GLN A   5      -3.496  -1.897  -4.868  1.00  0.00              \n"
		"ATOM     91 1HE2 GLN A   5      -7.363  -1.492  -6.290  1.00  0.00              \n"
		"ATOM     92 2HE2 GLN A   5      -6.301  -0.124  -6.262  1.00  0.00              \n"
		"ATOM     93  N   TRP A   6      -2.142  -0.514  -1.493  1.00  0.00              \n"
		"ATOM     94  CA  TRP A   6      -0.771  -0.642  -1.012  1.00  0.00              \n"
		"ATOM     95  C   TRP A   6      -0.643  -1.780  -0.009  1.00  0.00              \n"
		"ATOM     96  O   TRP A   6       0.338  -2.524  -0.021  1.00  0.00              \n"
		"ATOM     97  CB  TRP A   6      -0.303   0.670  -0.380  1.00  0.00              \n"
		"ATOM     98  CG  TRP A   6       1.078   0.598   0.197  1.00  0.00              \n"
		"ATOM     99  CD1 TRP A   6       2.251   0.573  -0.497  1.00  0.00              \n"
		"ATOM    100  CD2 TRP A   6       1.430   0.541   1.584  1.00  0.00              \n"
		"ATOM    101  NE1 TRP A   6       3.312   0.503   0.371  1.00  0.00              \n"
		"ATOM    102  CE2 TRP A   6       2.834   0.484   1.656  1.00  0.00              \n"
		"ATOM    103  CE3 TRP A   6       0.692   0.533   2.774  1.00  0.00              \n"
		"ATOM    104  CZ2 TRP A   6       3.514   0.418   2.862  1.00  0.00              \n"
		"ATOM    105  CZ3 TRP A   6       1.374   0.469   3.984  1.00  0.00              \n"
		"ATOM    106  CH2 TRP A   6       2.745   0.414   4.026  1.00  0.00              \n"
		"ATOM    107  H   TRP A   6      -2.703   0.264  -1.177  1.00  0.00              \n"
		"ATOM    108  HA  TRP A   6      -0.109  -0.883  -1.844  1.00  0.00              \n"
		"ATOM    109 1HB  TRP A   6      -0.288   1.465  -1.126  1.00  0.00              \n"
		"ATOM    110 2HB  TRP A   6      -0.968   0.952   0.436  1.00  0.00              \n"
		"ATOM    111  HD1 TRP A   6       2.182   0.611  -1.583  1.00  0.00              \n"
		"ATOM    112  HE1 TRP A   6       4.287   0.471   0.107  1.00  0.00              \n"
		"ATOM    113  HE3 TRP A   6      -0.396   0.578   2.798  1.00  0.00              \n"
		"ATOM    114  HZ2 TRP A   6       4.603   0.375   2.850  1.00  0.00              \n"
		"ATOM    115  HZ3 TRP A   6       0.788   0.463   4.903  1.00  0.00              \n"
		"ATOM    116  HH2 TRP A   6       3.241   0.364   4.996  1.00  0.00              \n"
		"ATOM    117  N   LEU A   7      -1.639  -1.911   0.861  1.00  0.00              \n"
		"ATOM    118  CA  LEU A   7      -1.604  -2.911   1.921  1.00  0.00              \n"
		"ATOM    119  C   LEU A   7      -1.768  -4.317   1.357  1.00  0.00              \n"
		"ATOM    120  O   LEU A   7      -1.107  -5.255   1.802  1.00  0.00              \n"
		"ATOM    121  CB  LEU A   7      -2.696  -2.622   2.959  1.00  0.00              \n"
		"ATOM    122  CG  LEU A   7      -2.324  -1.584   4.025  1.00  0.00              \n"
		"ATOM    123  CD1 LEU A   7      -3.503  -1.351   4.961  1.00  0.00              \n"
		"ATOM    124  CD2 LEU A   7      -1.107  -2.065   4.801  1.00  0.00              \n"
		"ATOM    125  H   LEU A   7      -2.441  -1.301   0.785  1.00  0.00              \n"
		"ATOM    126  HA  LEU A   7      -0.633  -2.887   2.414  1.00  0.00              \n"
		"ATOM    127 1HB  LEU A   7      -3.465  -2.221   2.300  1.00  0.00              \n"
		"ATOM    128 2HB  LEU A   7      -3.060  -3.536   3.428  1.00  0.00              \n"
		"ATOM    129  HG  LEU A   7      -2.046  -0.668   3.503  1.00  0.00              \n"
		"ATOM    130 1HD1 LEU A   7      -3.229  -0.613   5.715  1.00  0.00              \n"
		"ATOM    131 2HD1 LEU A   7      -4.355  -0.985   4.389  1.00  0.00              \n"
		"ATOM    132 3HD1 LEU A   7      -3.768  -2.287   5.451  1.00  0.00              \n"
		"ATOM    133 1HD2 LEU A   7      -0.843  -1.326   5.558  1.00  0.00              \n"
		"ATOM    134 2HD2 LEU A   7      -1.335  -3.015   5.284  1.00  0.00              \n"
		"ATOM    135 3HD2 LEU A   7      -0.268  -2.198   4.117  1.00  0.00              \n"
		"ATOM    136  N   LYS A   8      -2.654  -4.456   0.377  1.00  0.00              \n"
		"ATOM    137  CA  LYS A   8      -2.958  -5.759  -0.202  1.00  0.00              \n"
		"ATOM    138  C   LYS A   8      -1.734  -6.357  -0.886  1.00  0.00              \n"
		"ATOM    139  O   LYS A   8      -1.670  -7.563  -1.121  1.00  0.00              \n"
		"ATOM    140  CB  LYS A   8      -4.114  -5.647  -1.197  1.00  0.00              \n"
		"ATOM    141  CG  LYS A   8      -3.764  -4.921  -2.489  1.00  0.00              \n"
		"ATOM    142  CD  LYS A   8      -3.596  -5.896  -3.644  1.00  0.00              \n"
		"ATOM    143  CE  LYS A   8      -3.452  -5.165  -4.971  1.00  0.00              \n"
		"ATOM    144  NZ  LYS A   8      -4.562  -5.489  -5.908  1.00  0.00              \n"
		"ATOM    145  H   LYS A   8      -3.129  -3.638   0.023  1.00  0.00              \n"
		"ATOM    146  HA  LYS A   8      -3.244  -6.456   0.587  1.00  0.00              \n"
		"ATOM    147 1HB  LYS A   8      -4.436  -6.663  -1.428  1.00  0.00              \n"
		"ATOM    148 2HB  LYS A   8      -4.921  -5.118  -0.691  1.00  0.00              \n"
		"ATOM    149 1HG  LYS A   8      -4.566  -4.218  -2.719  1.00  0.00              \n"
		"ATOM    150 2HG  LYS A   8      -2.835  -4.373  -2.338  1.00  0.00              \n"
		"ATOM    151 1HD  LYS A   8      -2.706  -6.500  -3.463  1.00  0.00              \n"
		"ATOM    152 2HD  LYS A   8      -4.471  -6.545  -3.681  1.00  0.00              \n"
		"ATOM    153 1HE  LYS A   8      -3.447  -4.094  -4.772  1.00  0.00              \n"
		"ATOM    154 2HE  LYS A   8      -2.502  -5.456  -5.420  1.00  0.00              \n"
		"ATOM    155 1HZ  LYS A   8      -4.429  -4.986  -6.774  1.00  0.00              \n"
		"ATOM    156 2HZ  LYS A   8      -4.567  -6.483  -6.094  1.00  0.00              \n"
		"ATOM    157 3HZ  LYS A   8      -5.442  -5.220  -5.494  1.00  0.00              \n"
		"ATOM    158  N   ASP A   9      -0.765  -5.505  -1.203  1.00  0.00              \n"
		"ATOM    159  CA  ASP A   9       0.475  -5.953  -1.824  1.00  0.00              \n"
		"ATOM    160  C   ASP A   9       1.498  -6.368  -0.774  1.00  0.00              \n"
		"ATOM    161  O   ASP A   9       2.551  -6.915  -1.102  1.00  0.00              \n"
		"ATOM    162  CB  ASP A   9       1.056  -4.854  -2.718  1.00  0.00              \n"
		"ATOM    163  CG  ASP A   9       0.677  -4.974  -4.188  1.00  0.00              \n"
		"ATOM    164  OD1 ASP A   9      -0.492  -5.085  -4.473  1.00  0.00              \n"
		"ATOM    165  OD2 ASP A   9       1.536  -4.796  -5.018  1.00  0.00              \n"
		"ATOM    166  H   ASP A   9      -0.891  -4.522  -1.009  1.00  0.00              \n"
		"ATOM    167  HA  ASP A   9       0.285  -6.835  -2.436  1.00  0.00              \n"
		"ATOM    168 1HB  ASP A   9       0.833  -3.846  -2.367  1.00  0.00              \n"
		"ATOM    169 2HB  ASP A   9       2.121  -5.054  -2.600  1.00  0.00              \n"
		"ATOM    170  N   GLY A  10       1.182  -6.104   0.489  1.00  0.00              \n"
		"ATOM    171  CA  GLY A  10       2.000  -6.578   1.599  1.00  0.00              \n"
		"ATOM    172  C   GLY A  10       2.650  -5.417   2.340  1.00  0.00              \n"
		"ATOM    173  O   GLY A  10       3.547  -5.616   3.160  1.00  0.00              \n"
		"ATOM    174  H   GLY A  10       0.354  -5.560   0.686  1.00  0.00              \n"
		"ATOM    175 1HA  GLY A  10       1.370  -7.135   2.293  1.00  0.00              \n"
		"ATOM    176 2HA  GLY A  10       2.780  -7.233   1.212  1.00  0.00              \n"
		"ATOM    177  N   GLY A  11       2.192  -4.205   2.048  1.00  0.00              \n"
		"ATOM    178  CA  GLY A  11       2.664  -3.018   2.750  1.00  0.00              \n"
		"ATOM    179  C   GLY A  11       4.171  -2.851   2.600  1.00  0.00              \n"
		"ATOM    180  O   GLY A  11       4.694  -2.821   1.485  1.00  0.00              \n"
		"ATOM    181  H   GLY A  11       1.499  -4.103   1.320  1.00  0.00              \n"
		"ATOM    182 1HA  GLY A  11       2.167  -2.139   2.339  1.00  0.00              \n"
		"ATOM    183 2HA  GLY A  11       2.421  -3.109   3.808  1.00  0.00              \n"
		"ATOM    184  N   PRO A  12       4.865  -2.744   3.727  1.00  0.00              \n"
		"ATOM    185  CA  PRO A  12       6.309  -2.546   3.722  1.00  0.00              \n"
		"ATOM    186  C   PRO A  12       7.022  -3.711   3.047  1.00  0.00              \n"
		"ATOM    187  O   PRO A  12       8.105  -3.546   2.484  1.00  0.00              \n"
		"ATOM    188  CB  PRO A  12       6.676  -2.428   5.205  1.00  0.00              \n"
		"ATOM    189  CG  PRO A  12       5.407  -2.017   5.870  1.00  0.00              \n"
		"ATOM    190  CD  PRO A  12       4.311  -2.686   5.085  1.00  0.00              \n"
		"ATOM    191  HA  PRO A  12       6.615  -1.657   3.152  1.00  0.00              \n"
		"ATOM    192 1HB  PRO A  12       7.046  -3.383   5.606  1.00  0.00              \n"
		"ATOM    193 2HB  PRO A  12       7.469  -1.683   5.366  1.00  0.00              \n"
		"ATOM    194 1HG  PRO A  12       5.389  -2.335   6.923  1.00  0.00              \n"
		"ATOM    195 2HG  PRO A  12       5.289  -0.924   5.861  1.00  0.00              \n"
		"ATOM    196 1HD  PRO A  12       4.079  -3.692   5.467  1.00  0.00              \n"
		"ATOM    197 2HD  PRO A  12       3.373  -2.111   5.106  1.00  0.00              \n"
		"ATOM    198  N   SER A  13       6.409  -4.888   3.105  1.00  0.00              \n"
		"ATOM    199  CA  SER A  13       7.062  -6.115   2.667  1.00  0.00              \n"
		"ATOM    200  C   SER A  13       6.931  -6.301   1.160  1.00  0.00              \n"
		"ATOM    201  O   SER A  13       7.040  -7.416   0.651  1.00  0.00              \n"
		"ATOM    202  CB  SER A  13       6.478  -7.308   3.399  1.00  0.00              \n"
		"ATOM    203  OG  SER A  13       6.685  -7.229   4.782  1.00  0.00              \n"
		"ATOM    204  H   SER A  13       5.465  -4.933   3.462  1.00  0.00              \n"
		"ATOM    205  HA  SER A  13       8.112  -6.175   2.958  1.00  0.00              \n"
		"ATOM    206 1HB  SER A  13       5.407  -7.347   3.202  1.00  0.00              \n"
		"ATOM    207 2HB  SER A  13       6.949  -8.214   3.021  1.00  0.00              \n"
		"ATOM    208  HG  SER A  13       7.620  -7.337   4.973  1.00  0.00              \n"
		"ATOM    209  N   SER A  14       6.697  -5.202   0.452  1.00  0.00              \n"
		"ATOM    210  CA  SER A  14       6.426  -5.256  -0.980  1.00  0.00              \n"
		"ATOM    211  C   SER A  14       7.309  -4.280  -1.746  1.00  0.00              \n"
		"ATOM    212  O   SER A  14       7.028  -3.945  -2.897  1.00  0.00              \n"
		"ATOM    213  CB  SER A  14       4.962  -4.963  -1.245  1.00  0.00              \n"
		"ATOM    214  OG  SER A  14       4.468  -5.692  -2.335  1.00  0.00              \n"
		"ATOM    215  H   SER A  14       6.708  -4.305   0.917  1.00  0.00              \n"
		"ATOM    216  HA  SER A  14       6.526  -6.257  -1.401  1.00  0.00              \n"
		"ATOM    217 1HB  SER A  14       4.388  -5.223  -0.355  1.00  0.00              \n"
		"ATOM    218 2HB  SER A  14       4.850  -3.900  -1.449  1.00  0.00              \n"
		"ATOM    219  HG  SER A  14       3.773  -6.283  -2.035  1.00  0.00              \n"
		"ATOM    220  N   GLY A  15       8.378  -3.826  -1.101  1.00  0.00              \n"
		"ATOM    221  CA  GLY A  15       9.420  -3.068  -1.783  1.00  0.00              \n"
		"ATOM    222  C   GLY A  15       8.902  -1.713  -2.247  1.00  0.00              \n"
		"ATOM    223  O   GLY A  15       9.405  -1.144  -3.215  1.00  0.00              \n"
		"ATOM    224  H   GLY A  15       8.472  -4.011  -0.112  1.00  0.00              \n"
		"ATOM    225 1HA  GLY A  15      10.255  -2.914  -1.098  1.00  0.00              \n"
		"ATOM    226 2HA  GLY A  15       9.762  -3.633  -2.649  1.00  0.00              \n"
		"ATOM    227  N   ARG A  16       7.893  -1.201  -1.550  1.00  0.00              \n"
		"ATOM    228  CA  ARG A  16       7.231   0.033  -1.954  1.00  0.00              \n"
		"ATOM    229  C   ARG A  16       6.779   0.838  -0.742  1.00  0.00              \n"
		"ATOM    230  O   ARG A  16       6.208   0.291   0.201  1.00  0.00              \n"
		"ATOM    231  CB  ARG A  16       6.079  -0.221  -2.915  1.00  0.00              \n"
		"ATOM    232  CG  ARG A  16       5.003  -1.158  -2.390  1.00  0.00              \n"
		"ATOM    233  CD  ARG A  16       3.978  -1.538  -3.395  1.00  0.00              \n"
		"ATOM    234  NE  ARG A  16       4.238  -2.796  -4.076  1.00  0.00              \n"
		"ATOM    235  CZ  ARG A  16       4.852  -2.907  -5.270  1.00  0.00              \n"
		"ATOM    236  NH1 ARG A  16       5.236  -1.841  -5.936  1.00  0.00              \n"
		"ATOM    237  NH2 ARG A  16       5.037  -4.118  -5.767  1.00  0.00              \n"
		"ATOM    238  H   ARG A  16       7.576  -1.680  -0.719  1.00  0.00              \n"
		"ATOM    239  HA  ARG A  16       7.931   0.665  -2.503  1.00  0.00              \n"
		"ATOM    240 1HB  ARG A  16       5.634   0.747  -3.141  1.00  0.00              \n"
		"ATOM    241 2HB  ARG A  16       6.510  -0.643  -3.823  1.00  0.00              \n"
		"ATOM    242 1HG  ARG A  16       5.481  -2.072  -2.036  1.00  0.00              \n"
		"ATOM    243 2HG  ARG A  16       4.494  -0.669  -1.558  1.00  0.00              \n"
		"ATOM    244 1HD  ARG A  16       3.012  -1.629  -2.899  1.00  0.00              \n"
		"ATOM    245 2HD  ARG A  16       3.923  -0.760  -4.156  1.00  0.00              \n"
		"ATOM    246  HE  ARG A  16       4.019  -3.741  -3.790  1.00  0.00              \n"
		"ATOM    247 1HH1 ARG A  16       5.071  -0.921  -5.551  1.00  0.00              \n"
		"ATOM    248 2HH1 ARG A  16       5.694  -1.945  -6.830  1.00  0.00              \n"
		"ATOM    249 1HH2 ARG A  16       4.720  -4.926  -5.250  1.00  0.00              \n"
		"ATOM    250 2HH2 ARG A  16       5.494  -4.229  -6.660  1.00  0.00              \n"
		"ATOM    251  N   PRO A  17       7.039   2.141  -0.774  1.00  0.00              \n"
		"ATOM    252  CA  PRO A  17       6.692   3.018   0.338  1.00  0.00              \n"
		"ATOM    253  C   PRO A  17       5.201   3.328   0.352  1.00  0.00              \n"
		"ATOM    254  O   PRO A  17       4.512   3.156  -0.654  1.00  0.00              \n"
		"ATOM    255  CB  PRO A  17       7.539   4.273   0.103  1.00  0.00              \n"
		"ATOM    256  CG  PRO A  17       7.704   4.338  -1.377  1.00  0.00              \n"
		"ATOM    257  CD  PRO A  17       7.796   2.906  -1.832  1.00  0.00              \n"
		"ATOM    258  HA  PRO A  17       6.897   2.563   1.319  1.00  0.00              \n"
		"ATOM    259 1HB  PRO A  17       7.040   5.174   0.488  1.00  0.00              \n"
		"ATOM    260 2HB  PRO A  17       8.513   4.203   0.609  1.00  0.00              \n"
		"ATOM    261 1HG  PRO A  17       6.852   4.849  -1.850  1.00  0.00              \n"
		"ATOM    262 2HG  PRO A  17       8.609   4.900  -1.652  1.00  0.00              \n"
		"ATOM    263 1HD  PRO A  17       7.342   2.754  -2.822  1.00  0.00              \n"
		"ATOM    264 2HD  PRO A  17       8.837   2.557  -1.899  1.00  0.00              \n"
		"ATOM    265  N   PRO A  18       4.707   3.784   1.498  1.00  0.00              \n"
		"ATOM    266  CA  PRO A  18       3.287   4.074   1.659  1.00  0.00              \n"
		"ATOM    267  C   PRO A  18       2.903   5.361   0.940  1.00  0.00              \n"
		"ATOM    268  O   PRO A  18       3.672   6.321   0.912  1.00  0.00              \n"
		"ATOM    269  CB  PRO A  18       3.097   4.182   3.175  1.00  0.00              \n"
		"ATOM    270  CG  PRO A  18       4.432   4.602   3.688  1.00  0.00              \n"
		"ATOM    271  CD  PRO A  18       5.435   3.940   2.781  1.00  0.00              \n"
		"ATOM    272  HA  PRO A  18       2.642   3.299   1.219  1.00  0.00              \n"
		"ATOM    273 1HB  PRO A  18       2.322   4.919   3.433  1.00  0.00              \n"
		"ATOM    274 2HB  PRO A  18       2.787   3.221   3.612  1.00  0.00              \n"
		"ATOM    275 1HG  PRO A  18       4.540   5.696   3.666  1.00  0.00              \n"
		"ATOM    276 2HG  PRO A  18       4.575   4.286   4.732  1.00  0.00              \n"
		"ATOM    277 1HD  PRO A  18       6.337   4.554   2.640  1.00  0.00              \n"
		"ATOM    278 2HD  PRO A  18       5.766   2.966   3.168  1.00  0.00              \n"
		"ATOM    279  N   PRO A  19       1.708   5.373   0.359  1.00  0.00              \n"
		"ATOM    280  CA  PRO A  19       1.211   6.549  -0.346  1.00  0.00              \n"
		"ATOM    281  C   PRO A  19       0.757   7.626   0.631  1.00  0.00              \n"
		"ATOM    282  O   PRO A  19      -0.091   7.381   1.489  1.00  0.00              \n"
		"ATOM    283  CB  PRO A  19       0.053   6.015  -1.195  1.00  0.00              \n"
		"ATOM    284  CG  PRO A  19      -0.464   4.848  -0.425  1.00  0.00              \n"
		"ATOM    285  CD  PRO A  19       0.744   4.238   0.235  1.00  0.00              \n"
		"ATOM    286  HA  PRO A  19       1.983   7.033  -0.963  1.00  0.00              \n"
		"ATOM    287 1HB  PRO A  19      -0.728   6.777  -1.336  1.00  0.00              \n"
		"ATOM    288 2HB  PRO A  19       0.393   5.712  -2.196  1.00  0.00              \n"
		"ATOM    289 1HG  PRO A  19      -1.207   5.162   0.323  1.00  0.00              \n"
		"ATOM    290 2HG  PRO A  19      -0.961   4.124  -1.087  1.00  0.00              \n"
		"ATOM    291 1HD  PRO A  19       0.510   3.821   1.226  1.00  0.00              \n"
		"ATOM    292 2HD  PRO A  19       1.177   3.422  -0.363  1.00  0.00              \n"
		"ATOM    293  N   SER A  20       1.326   8.819   0.495  1.00  0.00              \n"
		"ATOM    294  CA  SER A  20       0.866   9.978   1.251  1.00  0.00              \n"
		"ATOM    295  C   SER A  20      -0.402  10.563   0.644  1.00  0.00              \n"
		"ATOM    296  O   SER A  20      -1.465  10.055   0.870  1.00  0.00              \n"
		"ATOM    297  OXT SER A  20      -0.338  11.533  -0.060  1.00  0.00              \n"
		"ATOM    298  CB  SER A  20       1.958  11.029   1.310  1.00  0.00              \n"
		"ATOM    299  OG  SER A  20       3.116  10.550   1.937  1.00  0.00              \n"
		"ATOM    300  H   SER A  20       2.098   8.925  -0.148  1.00  0.00              \n"
		"ATOM    301  HA  SER A  20       0.709   9.769   2.310  1.00  0.00              \n"
		"ATOM    302 1HB  SER A  20       2.205  11.332   0.293  1.00  0.00              \n"
		"ATOM    303 2HB  SER A  20       1.584  11.889   1.864  1.00  0.00              \n"
		"ATOM    304  HG  SER A  20       2.871  10.016   2.696  1.00  0.00              \n"
		"TER\n";
}

inline
std::string
pdb_string_1ten() {
	return
	"ATOM      1  C   ARG A 802      31.142  34.383  12.972  1.00 35.79           C  \n"
	"ATOM      2  O   ARG A 802      32.275  34.066  13.343  1.00 36.69           O  \n"
	"ATOM      3  N   LEU A 803      30.980  35.302  12.030  1.00 32.50           N  \n"
	"ATOM      4  CA  LEU A 803      32.048  36.234  11.756  1.00 31.31           C  \n"
	"ATOM      5  C   LEU A 803      32.230  37.196  12.910  1.00 28.72           C  \n"
	"ATOM      6  O   LEU A 803      31.296  37.650  13.562  1.00 27.63           O  \n"
	"ATOM      7  CB  LEU A 803      31.771  37.027  10.494  1.00 33.95           C  \n"
	"ATOM      8  CG  LEU A 803      31.960  36.329   9.177  1.00 33.82           C  \n"
	"ATOM      9  CD1 LEU A 803      31.324  37.143   8.089  1.00 33.55           C  \n"
	"ATOM     10  CD2 LEU A 803      33.439  36.146   8.922  1.00 36.14           C  \n"
	"ATOM     11  H   LEU A 803      30.105  35.412  11.606  1.00  0.00           H  \n"
	"ATOM     12  N   ASP A 804      33.483  37.494  13.121  1.00 28.42           N  \n"
	"ATOM     13  CA  ASP A 804      33.930  38.405  14.151  1.00 32.74           C  \n"
	"ATOM     14  C   ASP A 804      33.677  39.850  13.665  1.00 31.63           C  \n"
	"ATOM     15  O   ASP A 804      33.906  40.129  12.481  1.00 26.95           O  \n"
	"ATOM     16  CB  ASP A 804      35.394  38.035  14.273  1.00 38.31           C  \n"
	"ATOM     17  CG  ASP A 804      36.242  38.978  15.082  1.00 46.29           C  \n"
	"ATOM     18  OD1 ASP A 804      36.224  38.901  16.305  1.00 55.21           O  \n"
	"ATOM     19  OD2 ASP A 804      37.087  39.848  14.433  1.00 54.50           O  \n"
	"ATOM     20  H   ASP A 804      34.144  37.116  12.512  1.00  0.00           H  \n"
	"ATOM     21  HD2 ASP A 804      36.874  40.753  14.738  1.00  0.00           H  \n"
	"ATOM     22  N   ALA A 805      33.146  40.772  14.479  1.00 30.18           N  \n"
	"ATOM     23  CA  ALA A 805      32.911  42.160  14.064  1.00 24.86           C  \n"
	"ATOM     24  C   ALA A 805      34.223  42.938  14.027  1.00 22.17           C  \n"
	"ATOM     25  O   ALA A 805      35.210  42.440  14.584  1.00 22.29           O  \n"
	"ATOM     26  CB  ALA A 805      31.991  42.833  15.070  1.00 25.78           C  \n"
	"ATOM     27  H   ALA A 805      32.942  40.508  15.402  1.00  0.00           H  \n"
	"ATOM     28  N   PRO A 806      34.381  44.095  13.359  1.00 21.44           N  \n"
	"ATOM     29  CA  PRO A 806      35.651  44.839  13.353  1.00 21.43           C  \n"
	"ATOM     30  C   PRO A 806      35.909  45.321  14.792  1.00 20.14           C  \n"
	"ATOM     31  O   PRO A 806      34.970  45.650  15.537  1.00 18.39           O  \n"
	"ATOM     32  CB  PRO A 806      35.415  45.973  12.353  1.00 21.28           C  \n"
	"ATOM     33  CG  PRO A 806      34.354  45.388  11.441  1.00 22.07           C  \n"
	"ATOM     34  CD  PRO A 806      33.420  44.676  12.421  1.00 19.38           C  \n"
	"ATOM     35  N   SER A 807      37.161  45.394  15.229  1.00 20.48           N  \n"
	"ATOM     36  CA  SER A 807      37.434  45.777  16.605  1.00 21.07           C  \n"
	"ATOM     37  C   SER A 807      38.076  47.149  16.663  1.00 17.37           C  \n"
	"ATOM     38  O   SER A 807      38.512  47.683  15.647  1.00 18.23           O  \n"
	"ATOM     39  CB  SER A 807      38.324  44.686  17.206  1.00 22.47           C  \n"
	"ATOM     40  OG  SER A 807      38.572  44.759  18.608  1.00 29.52           O  \n"
	"ATOM     41  H   SER A 807      37.907  45.251  14.604  1.00  0.00           H  \n"
	"ATOM     42  HG  SER A 807      37.792  44.417  19.086  1.00  0.00           H  \n"
	"ATOM     43  N   GLN A 808      38.143  47.713  17.867  1.00 17.96           N  \n"
	"ATOM     44  CA  GLN A 808      38.721  49.017  18.148  1.00 20.95           C  \n"
	"ATOM     45  C   GLN A 808      38.318  50.090  17.126  1.00 18.40           C  \n"
	"ATOM     46  O   GLN A 808      39.135  50.696  16.416  1.00 15.34           O  \n"
	"ATOM     47  CB  GLN A 808      40.223  48.876  18.266  1.00 19.39           C  \n"
	"ATOM     48  CG  GLN A 808      40.818  50.021  19.072  1.00 31.72           C  \n"
	"ATOM     49  CD  GLN A 808      42.316  49.902  19.329  1.00 37.78           C  \n"
	"ATOM     50  OE1 GLN A 808      42.850  50.452  20.289  1.00 43.29           O  \n"
	"ATOM     51  NE2 GLN A 808      43.100  49.209  18.511  1.00 39.16           N  \n"
	"ATOM     52  H   GLN A 808      37.809  47.198  18.637  1.00  0.00           H  \n"
	"ATOM     53 HE21 GLN A 808      44.041  49.168  18.766  1.00  0.00           H  \n"
	"ATOM     54 HE22 GLN A 808      42.721  48.781  17.718  1.00  0.00           H  \n"
	"ATOM     55  N   ILE A 809      37.001  50.280  17.055  1.00 15.30           N  \n"
	"ATOM     56  CA  ILE A 809      36.423  51.242  16.125  1.00 16.06           C  \n"
	"ATOM     57  C   ILE A 809      36.532  52.558  16.853  1.00 14.86           C  \n"
	"ATOM     58  O   ILE A 809      36.205  52.656  18.043  1.00 14.95           O  \n"
	"ATOM     59  CB  ILE A 809      34.948  50.902  15.771  1.00 14.49           C  \n"
	"ATOM     60  CG1 ILE A 809      34.855  49.511  15.157  1.00 14.99           C  \n"
	"ATOM     61  CG2 ILE A 809      34.385  51.972  14.830  1.00 14.77           C  \n"
	"ATOM     62  CD1 ILE A 809      33.439  48.935  15.128  1.00 16.60           C  \n"
	"ATOM     63  H   ILE A 809      36.406  49.869  17.717  1.00  0.00           H  \n"
	"ATOM     64  N   GLU A 810      37.106  53.536  16.155  1.00 12.69           N  \n"
	"ATOM     65  CA  GLU A 810      37.286  54.854  16.724  1.00 13.69           C  \n"
	"ATOM     66  C   GLU A 810      36.910  55.878  15.674  1.00 13.18           C  \n"
	"ATOM     67  O   GLU A 810      37.171  55.652  14.482  1.00 17.48           O  \n"
	"ATOM     68  CB  GLU A 810      38.750  55.059  17.136  1.00 17.21           C  \n"
	"ATOM     69  CG  GLU A 810      39.239  54.069  18.214  1.00 20.12           C  \n"
	"ATOM     70  CD  GLU A 810      40.677  54.266  18.625  1.00 26.18           C  \n"
	"ATOM     71  OE1 GLU A 810      40.989  54.189  19.814  1.00 35.06           O  \n"
	"ATOM     72  OE2 GLU A 810      41.637  54.615  17.708  1.00 30.01           O  \n"
	"ATOM     73  H   GLU A 810      37.351  53.380  15.218  1.00  0.00           H  \n"
	"ATOM     74  HE2 GLU A 810      41.563  53.926  17.016  1.00  0.00           H  \n"
	"ATOM     75  N   VAL A 811      36.257  56.963  16.076  1.00 11.17           N  \n"
	"ATOM     76  CA  VAL A 811      35.899  58.053  15.176  1.00 13.39           C  \n"
	"ATOM     77  C   VAL A 811      36.909  59.139  15.571  1.00 14.32           C  \n"
	"ATOM     78  O   VAL A 811      37.053  59.374  16.756  1.00 14.12           O  \n"
	"ATOM     79  CB  VAL A 811      34.442  58.504  15.411  1.00 11.38           C  \n"
	"ATOM     80  CG1 VAL A 811      34.115  59.675  14.484  1.00  9.55           C  \n"
	"ATOM     81  CG2 VAL A 811      33.492  57.342  15.154  1.00 12.63           C  \n"
	"ATOM     82  H   VAL A 811      36.063  57.098  17.037  1.00  0.00           H  \n"
	"ATOM     83  N   LYS A 812      37.690  59.723  14.674  1.00 13.41           N  \n"
	"ATOM     84  CA  LYS A 812      38.688  60.715  15.080  1.00 13.76           C  \n"
	"ATOM     85  C   LYS A 812      38.645  61.826  14.048  1.00 12.39           C  \n"
	"ATOM     86  O   LYS A 812      37.931  61.726  13.025  1.00 13.26           O  \n"
	"ATOM     87  CB  LYS A 812      40.091  60.121  14.990  1.00 19.53           C  \n"
	"ATOM     88  CG  LYS A 812      40.383  58.790  15.662  1.00 24.57           C  \n"
	"ATOM     89  CD  LYS A 812      41.680  58.290  15.059  1.00 28.59           C  \n"
	"ATOM     90  CE  LYS A 812      42.054  56.843  15.393  1.00 30.10           C  \n"
	"ATOM     91  NZ  LYS A 812      43.372  56.578  14.848  1.00 36.98           N  \n"
	"ATOM     92  H   LYS A 812      37.528  59.607  13.718  1.00  0.00           H  \n"
	"ATOM     93  HZ1 LYS A 812      43.371  56.797  13.826  1.00  0.00           H  \n"
	"ATOM     94  HZ2 LYS A 812      44.087  57.217  15.259  1.00  0.00           H  \n"
	"ATOM     95  HZ3 LYS A 812      43.659  55.595  14.983  1.00  0.00           H  \n"
	"ATOM     96  N   ASP A 813      39.388  62.903  14.280  1.00  8.14           N  \n"
	"ATOM     97  CA  ASP A 813      39.518  64.032  13.365  1.00 12.00           C  \n"
	"ATOM     98  C   ASP A 813      38.178  64.622  12.932  1.00 13.61           C  \n"
	"ATOM     99  O   ASP A 813      37.945  64.914  11.736  1.00 13.01           O  \n"
	"ATOM    100  CB  ASP A 813      40.355  63.578  12.110  1.00 14.38           C  \n"
	"ATOM    101  CG  ASP A 813      41.749  63.043  12.429  1.00 13.29           C  \n"
	"ATOM    102  OD1 ASP A 813      42.418  63.623  13.288  1.00 15.38           O  \n"
	"ATOM    103  OD2 ASP A 813      42.157  62.038  11.846  1.00 17.44           O  \n"
	"ATOM    104  H   ASP A 813      39.837  62.952  15.152  1.00  0.00           H  \n"
	"ATOM    105  N   VAL A 814      37.273  64.826  13.895  1.00 13.49           N  \n"
	"ATOM    106  CA  VAL A 814      35.930  65.326  13.600  1.00 10.18           C  \n"
	"ATOM    107  C   VAL A 814      35.933  66.838  13.484  1.00 11.87           C  \n"
	"ATOM    108  O   VAL A 814      36.534  67.538  14.294  1.00 14.52           O  \n"
	"ATOM    109  CB  VAL A 814      34.910  64.840  14.696  1.00 11.44           C  \n"
	"ATOM    110  CG1 VAL A 814      33.499  65.346  14.439  1.00  9.08           C  \n"
	"ATOM    111  CG2 VAL A 814      34.895  63.328  14.841  1.00 11.82           C  \n"
	"ATOM    112  H   VAL A 814      37.518  64.675  14.844  1.00  0.00           H  \n"
	"ATOM    113  N   THR A 815      35.329  67.370  12.434  1.00 13.32           N  \n"
	"ATOM    114  CA  THR A 815      35.230  68.796  12.241  1.00 12.46           C  \n"
	"ATOM    115  C   THR A 815      33.731  69.041  12.115  1.00 12.09           C  \n"
	"ATOM    116  O   THR A 815      32.941  68.093  12.168  1.00 14.31           O  \n"
	"ATOM    117  CB  THR A 815      35.989  69.280  10.965  1.00 11.09           C  \n"
	"ATOM    118  OG1 THR A 815      35.311  68.809   9.808  1.00  9.35           O  \n"
	"ATOM    119  CG2 THR A 815      37.423  68.792  10.933  1.00  9.01           C  \n"
	"ATOM    120  H   THR A 815      34.790  66.794  11.854  1.00  0.00           H  \n"
	"ATOM    121  HG1 THR A 815      35.868  68.166   9.347  1.00  0.00           H  \n"
	"ATOM    122  N   ASP A 816      33.264  70.266  11.904  1.00 12.67           N  \n"
	"ATOM    123  CA  ASP A 816      31.846  70.545  11.749  1.00 15.20           C  \n"
	"ATOM    124  C   ASP A 816      31.233  69.751  10.606  1.00 13.15           C  \n"
	"ATOM    125  O   ASP A 816      30.045  69.434  10.645  1.00 13.30           O  \n"
	"ATOM    126  CB  ASP A 816      31.606  72.041  11.471  1.00 23.64           C  \n"
	"ATOM    127  CG  ASP A 816      31.724  72.860  12.736  1.00 30.18           C  \n"
	"ATOM    128  OD1 ASP A 816      32.830  73.283  13.058  1.00 37.51           O  \n"
	"ATOM    129  OD2 ASP A 816      30.726  73.052  13.429  1.00 38.56           O  \n"
	"ATOM    130  H   ASP A 816      33.886  71.024  11.900  1.00  0.00           H  \n"
	"ATOM    131  N   THR A 817      32.005  69.385   9.578  1.00  8.73           N  \n"
	"ATOM    132  CA  THR A 817      31.443  68.664   8.456  1.00  8.91           C  \n"
	"ATOM    133  C   THR A 817      32.154  67.389   8.053  1.00  8.28           C  \n"
	"ATOM    134  O   THR A 817      31.733  66.807   7.045  1.00  9.12           O  \n"
	"ATOM    135  CB  THR A 817      31.362  69.633   7.239  1.00  8.54           C  \n"
	"ATOM    136  OG1 THR A 817      32.694  70.017   6.875  1.00 14.52           O  \n"
	"ATOM    137  CG2 THR A 817      30.513  70.871   7.491  1.00  8.29           C  \n"
	"ATOM    138  H   THR A 817      32.955  69.625   9.570  1.00  0.00           H  \n"
	"ATOM    139  HG1 THR A 817      32.998  70.704   7.488  1.00  0.00           H  \n"
	"ATOM    140  N   THR A 818      33.207  66.909   8.733  1.00 10.30           N  \n"
	"ATOM    141  CA  THR A 818      33.927  65.691   8.340  1.00 10.76           C  \n"
	"ATOM    142  C   THR A 818      34.251  64.808   9.552  1.00 11.80           C  \n"
	"ATOM    143  O   THR A 818      34.187  65.281  10.691  1.00 10.41           O  \n"
	"ATOM    144  CB  THR A 818      35.293  65.979   7.584  1.00 10.75           C  \n"
	"ATOM    145  OG1 THR A 818      36.203  66.587   8.504  1.00 11.90           O  \n"
	"ATOM    146  CG2 THR A 818      35.169  66.893   6.367  1.00  9.14           C  \n"
	"ATOM    147  H   THR A 818      33.514  67.361   9.547  1.00  0.00           H  \n"
	"ATOM    148  HG1 THR A 818      36.662  65.862   8.948  1.00  0.00           H  \n"
	"ATOM    149  N   ALA A 819      34.632  63.551   9.337  1.00 10.93           N  \n"
	"ATOM    150  CA  ALA A 819      35.012  62.617  10.390  1.00 12.60           C  \n"
	"ATOM    151  C   ALA A 819      35.815  61.452   9.784  1.00 14.53           C  \n"
	"ATOM    152  O   ALA A 819      35.602  61.110   8.610  1.00 13.95           O  \n"
	"ATOM    153  CB  ALA A 819      33.747  62.039  11.050  1.00  6.99           C  \n"
	"ATOM    154  H   ALA A 819      34.658  63.206   8.415  1.00  0.00           H  \n"
	"ATOM    155  N   LEU A 820      36.746  60.821  10.504  1.00 10.43           N  \n"
	"ATOM    156  CA  LEU A 820      37.504  59.697  10.009  1.00 12.40           C  \n"
	"ATOM    157  C   LEU A 820      37.110  58.496  10.869  1.00  9.68           C  \n"
	"ATOM    158  O   LEU A 820      37.100  58.624  12.092  1.00 14.45           O  \n"
	"ATOM    159  CB  LEU A 820      39.021  59.948  10.142  1.00 10.05           C  \n"
	"ATOM    160  CG  LEU A 820      39.922  58.790   9.673  1.00 14.12           C  \n"
	"ATOM    161  CD1 LEU A 820      39.897  58.630   8.164  1.00 13.54           C  \n"
	"ATOM    162  CD2 LEU A 820      41.363  59.020  10.120  1.00 13.90           C  \n"
	"ATOM    163  H   LEU A 820      36.925  61.102  11.428  1.00  0.00           H  \n"
	"ATOM    164  N   ILE A 821      36.720  57.354  10.305  1.00  9.63           N  \n"
	"ATOM    165  CA  ILE A 821      36.330  56.169  11.068  1.00 10.98           C  \n"
	"ATOM    166  C   ILE A 821      37.494  55.199  10.866  1.00 10.72           C  \n"
	"ATOM    167  O   ILE A 821      37.958  55.022   9.726  1.00 11.83           O  \n"
	"ATOM    168  CB  ILE A 821      35.029  55.510  10.500  1.00 16.02           C  \n"
	"ATOM    169  CG1 ILE A 821      33.876  56.472  10.146  1.00 11.93           C  \n"
	"ATOM    170  CG2 ILE A 821      34.566  54.433  11.488  1.00 13.06           C  \n"
	"ATOM    171  CD1 ILE A 821      33.431  57.417  11.280  1.00 16.30           C  \n"
	"ATOM    172  H   ILE A 821      36.717  57.294   9.330  1.00  0.00           H  \n"
	"ATOM    173  N   THR A 822      38.023  54.562  11.905  1.00 12.38           N  \n"
	"ATOM    174  CA  THR A 822      39.146  53.637  11.767  1.00 15.12           C  \n"
	"ATOM    175  C   THR A 822      38.766  52.382  12.523  1.00 15.79           C  \n"
	"ATOM    176  O   THR A 822      37.884  52.469  13.387  1.00 17.25           O  \n"
	"ATOM    177  CB  THR A 822      40.493  54.193  12.339  1.00 15.96           C  \n"
	"ATOM    178  OG1 THR A 822      40.313  54.637  13.683  1.00 17.80           O  \n"
	"ATOM    179  CG2 THR A 822      41.019  55.348  11.525  1.00 16.24           C  \n"
	"ATOM    180  H   THR A 822      37.583  54.627  12.780  1.00  0.00           H  \n"
	"ATOM    181  HG1 THR A 822      39.457  55.095  13.712  1.00  0.00           H  \n"
	"ATOM    182  N   TRP A 823      39.348  51.232  12.218  1.00 13.01           N  \n"
	"ATOM    183  CA  TRP A 823      39.036  50.027  12.959  1.00 16.49           C  \n"
	"ATOM    184  C   TRP A 823      40.183  49.063  12.735  1.00 16.02           C  \n"
	"ATOM    185  O   TRP A 823      41.101  49.366  11.962  1.00 16.14           O  \n"
	"ATOM    186  CB  TRP A 823      37.703  49.396  12.484  1.00 13.76           C  \n"
	"ATOM    187  CG  TRP A 823      37.620  49.159  10.975  1.00 15.92           C  \n"
	"ATOM    188  CD1 TRP A 823      38.044  47.982  10.405  1.00 17.09           C  \n"
	"ATOM    189  CD2 TRP A 823      37.170  50.062  10.043  1.00 18.82           C  \n"
	"ATOM    190  NE1 TRP A 823      37.874  48.141   9.124  1.00 19.44           N  \n"
	"ATOM    191  CE2 TRP A 823      37.362  49.355   8.865  1.00 19.88           C  \n"
	"ATOM    192  CE3 TRP A 823      36.654  51.348  10.027  1.00 16.48           C  \n"
	"ATOM    193  CZ2 TRP A 823      37.039  49.915   7.646  1.00 17.86           C  \n"
	"ATOM    194  CZ3 TRP A 823      36.335  51.913   8.809  1.00 15.47           C  \n"
	"ATOM    195  CH2 TRP A 823      36.529  51.197   7.637  1.00 17.86           C  \n"
	"ATOM    196  H   TRP A 823      39.992  51.164  11.476  1.00  0.00           H  \n"
	"ATOM    197  HE1 TRP A 823      38.082  47.450   8.450  1.00  0.00           H  \n"
	"ATOM    198  N   PHE A 824      40.125  47.924  13.419  1.00 20.34           N  \n"
	"ATOM    199  CA  PHE A 824      41.086  46.843  13.249  1.00 25.78           C  \n"
	"ATOM    200  C   PHE A 824      40.367  45.784  12.451  1.00 23.49           C  \n"
	"ATOM    201  O   PHE A 824      39.194  45.492  12.724  1.00 29.89           O  \n"
	"ATOM    202  CB  PHE A 824      41.502  46.188  14.554  1.00 28.05           C  \n"
	"ATOM    203  CG  PHE A 824      42.754  46.796  15.160  1.00 36.99           C  \n"
	"ATOM    204  CD1 PHE A 824      43.161  48.087  14.846  1.00 41.33           C  \n"
	"ATOM    205  CD2 PHE A 824      43.490  46.042  16.050  1.00 42.54           C  \n"
	"ATOM    206  CE1 PHE A 824      44.296  48.620  15.420  1.00 43.56           C  \n"
	"ATOM    207  CE2 PHE A 824      44.625  46.579  16.624  1.00 47.02           C  \n"
	"ATOM    208  CZ  PHE A 824      45.028  47.865  16.312  1.00 46.87           C  \n"
	"ATOM    209  H   PHE A 824      39.354  47.764  13.994  1.00  0.00           H  \n"
	"ATOM    210  N   LYS A 825      41.049  45.215  11.473  1.00 27.19           N  \n"
	"ATOM    211  CA  LYS A 825      40.490  44.163  10.636  1.00 29.67           C  \n"
	"ATOM    212  C   LYS A 825      39.999  42.967  11.457  1.00 31.44           C  \n"
	"ATOM    213  O   LYS A 825      40.664  42.617  12.439  1.00 31.93           O  \n"
	"ATOM    214  CB  LYS A 825      41.579  43.673   9.685  1.00 33.46           C  \n"
	"ATOM    215  CG  LYS A 825      41.127  42.540   8.777  1.00 36.99           C  \n"
	"ATOM    216  CD  LYS A 825      42.197  42.072   7.834  1.00 45.14           C  \n"
	"ATOM    217  CE  LYS A 825      41.552  40.986   7.001  1.00 50.20           C  \n"
	"ATOM    218  NZ  LYS A 825      42.320  40.799   5.793  1.00 57.77           N  \n"
	"ATOM    219  H   LYS A 825      41.983  45.475  11.347  1.00  0.00           H  \n"
	"ATOM    220  HZ1 LYS A 825      43.325  40.616   5.927  1.00  0.00           H  \n"
	"ATOM    221  HZ2 LYS A 825      41.904  40.037   5.212  1.00  0.00           H  \n"
	"ATOM    222  HZ3 LYS A 825      42.214  41.603   5.128  1.00  0.00           H  \n"
	"ATOM    223  N   PRO A 826      38.847  42.347  11.143  1.00 32.88           N  \n"
	"ATOM    224  CA  PRO A 826      38.320  41.193  11.857  1.00 36.58           C  \n"
	"ATOM    225  C   PRO A 826      39.339  40.056  11.748  1.00 42.32           C  \n"
	"ATOM    226  O   PRO A 826      40.110  39.988  10.779  1.00 42.32           O  \n"
	"ATOM    227  CB  PRO A 826      37.041  40.862  11.117  1.00 33.22           C  \n"
	"ATOM    228  CG  PRO A 826      36.575  42.228  10.726  1.00 32.11           C  \n"
	"ATOM    229  CD  PRO A 826      37.851  42.848  10.206  1.00 32.37           C  \n"
	"ATOM    230  N   LEU A 827      39.347  39.168  12.754  1.00 45.97           N  \n"
	"ATOM    231  CA  LEU A 827      40.265  38.026  12.789  1.00 49.24           C  \n"
	"ATOM    232  C   LEU A 827      40.072  37.059  11.620  1.00 49.85           C  \n"
	"ATOM    233  O   LEU A 827      41.005  36.627  10.940  1.00 48.25           O  \n"
	"ATOM    234  CB  LEU A 827      40.111  37.270  14.111  1.00 49.36           C  \n"
	"ATOM    235  CG  LEU A 827      40.668  37.836  15.429  1.00 54.77           C  \n"
	"ATOM    236  CD1 LEU A 827      42.155  38.111  15.292  1.00 55.29           C  \n"
	"ATOM    237  CD2 LEU A 827      39.971  39.114  15.869  1.00 58.43           C  \n"
	"ATOM    238  H   LEU A 827      38.740  39.336  13.501  1.00  0.00           H  \n"
	"ATOM    239  N   ALA A 828      38.801  36.713  11.413  1.00 50.82           N  \n"
	"ATOM    240  CA  ALA A 828      38.405  35.813  10.351  1.00 51.43           C  \n"
	"ATOM    241  C   ALA A 828      38.563  36.488   8.994  1.00 52.97           C  \n"
	"ATOM    242  O   ALA A 828      38.393  37.704   8.860  1.00 54.87           O  \n"
	"ATOM    243  CB  ALA A 828      36.926  35.485  10.514  1.00 49.87           C  \n"
	"ATOM    244  H   ALA A 828      38.120  37.203  11.906  1.00  0.00           H  \n"
	"ATOM    245  N   GLU A 829      38.897  35.694   7.975  1.00 52.39           N  \n"
	"ATOM    246  CA  GLU A 829      39.005  36.188   6.608  1.00 49.85           C  \n"
	"ATOM    247  C   GLU A 829      37.581  36.602   6.218  1.00 43.90           C  \n"
	"ATOM    248  O   GLU A 829      36.592  35.958   6.622  1.00 41.36           O  \n"
	"ATOM    249  CB  GLU A 829      39.573  35.063   5.761  1.00 56.98           C  \n"
	"ATOM    250  CG  GLU A 829      39.444  35.250   4.251  1.00 65.94           C  \n"
	"ATOM    251  CD  GLU A 829      38.492  34.230   3.642  1.00 70.23           C  \n"
	"ATOM    252  OE1 GLU A 829      37.327  34.190   4.062  1.00 71.57           O  \n"
	"ATOM    253  OE2 GLU A 829      38.954  33.458   2.802  1.00 70.76           O  \n"
	"ATOM    254  H   GLU A 829      39.128  34.763   8.148  1.00  0.00           H  \n"
	"ATOM    255  N   ILE A 830      37.472  37.687   5.469  1.00 36.72           N  \n"
	"ATOM    256  CA  ILE A 830      36.194  38.269   5.113  1.00 31.51           C  \n"
	"ATOM    257  C   ILE A 830      36.270  38.842   3.690  1.00 31.31           C  \n"
	"ATOM    258  O   ILE A 830      37.359  39.171   3.209  1.00 32.73           O  \n"
	"ATOM    259  CB  ILE A 830      35.938  39.449   6.122  1.00 24.90           C  \n"
	"ATOM    260  CG1 ILE A 830      37.130  40.405   6.106  1.00 24.50           C  \n"
	"ATOM    261  CG2 ILE A 830      35.705  38.985   7.556  1.00 21.93           C  \n"
	"ATOM    262  CD1 ILE A 830      36.936  41.816   6.647  1.00 25.95           C  \n"
	"ATOM    263  H   ILE A 830      38.251  38.034   4.984  1.00  0.00           H  \n"
	"ATOM    264  N   ASP A 831      35.158  39.011   2.978  1.00 32.25           N  \n"
	"ATOM    265  CA  ASP A 831      35.179  39.628   1.649  1.00 31.19           C  \n"
	"ATOM    266  C   ASP A 831      35.267  41.137   1.808  1.00 29.45           C  \n"
	"ATOM    267  O   ASP A 831      35.931  41.863   1.051  1.00 30.10           O  \n"
	"ATOM    268  CB  ASP A 831      33.902  39.441   0.881  1.00 33.83           C  \n"
	"ATOM    269  CG  ASP A 831      33.328  38.053   0.840  1.00 35.78           C  \n"
	"ATOM    270  OD1 ASP A 831      34.039  37.059   0.941  1.00 35.03           O  \n"
	"ATOM    271  OD2 ASP A 831      32.111  37.987   0.787  1.00 39.79           O  \n"
	"ATOM    272  H   ASP A 831      34.315  38.639   3.310  1.00  0.00           H  \n"
	"ATOM    273  N   GLY A 832      34.564  41.653   2.811  1.00 27.64           N  \n"
	"ATOM    274  CA  GLY A 832      34.556  43.073   3.065  1.00 24.19           C  \n"
	"ATOM    275  C   GLY A 832      33.848  43.426   4.362  1.00 23.63           C  \n"
	"ATOM    276  O   GLY A 832      33.495  42.554   5.171  1.00 17.74           O  \n"
	"ATOM    277  H   GLY A 832      34.041  41.056   3.388  1.00  0.00           H  \n"
	"ATOM    278  N   ILE A 833      33.625  44.736   4.469  1.00 21.38           N  \n"
	"ATOM    279  CA  ILE A 833      33.006  45.395   5.604  1.00 18.75           C  \n"
	"ATOM    280  C   ILE A 833      31.911  46.302   5.064  1.00 16.41           C  \n"
	"ATOM    281  O   ILE A 833      32.144  46.992   4.073  1.00 18.30           O  \n"
	"ATOM    282  CB  ILE A 833      34.098  46.287   6.293  1.00 20.88           C  \n"
	"ATOM    283  CG1 ILE A 833      35.264  45.461   6.805  1.00 26.05           C  \n"
	"ATOM    284  CG2 ILE A 833      33.542  47.163   7.426  1.00 16.62           C  \n"
	"ATOM    285  CD1 ILE A 833      34.862  44.535   7.970  1.00 27.78           C  \n"
	"ATOM    286  H   ILE A 833      33.803  45.297   3.682  1.00  0.00           H  \n"
	"ATOM    287  N   GLU A 834      30.730  46.331   5.644  1.00 10.55           N  \n"
	"ATOM    288  CA  GLU A 834      29.693  47.252   5.225  1.00 12.27           C  \n"
	"ATOM    289  C   GLU A 834      29.656  48.312   6.334  1.00 12.26           C  \n"
	"ATOM    290  O   GLU A 834      29.608  47.975   7.530  1.00 10.91           O  \n"
	"ATOM    291  CB  GLU A 834      28.356  46.573   5.153  1.00 14.77           C  \n"
	"ATOM    292  CG  GLU A 834      28.330  45.513   4.067  1.00 28.31           C  \n"
	"ATOM    293  CD  GLU A 834      26.988  44.841   3.825  1.00 35.86           C  \n"
	"ATOM    294  OE1 GLU A 834      25.982  45.152   4.467  1.00 35.82           O  \n"
	"ATOM    295  OE2 GLU A 834      26.939  43.951   2.984  1.00 42.42           O  \n"
	"ATOM    296  H   GLU A 834      30.549  45.753   6.413  1.00  0.00           H  \n"
	"ATOM    297  N   LEU A 835      29.801  49.584   5.964  1.00 11.11           N  \n"
	"ATOM    298  CA  LEU A 835      29.762  50.672   6.915  1.00 11.37           C  \n"
	"ATOM    299  C   LEU A 835      28.491  51.478   6.693  1.00 10.41           C  \n"
	"ATOM    300  O   LEU A 835      28.277  51.910   5.565  1.00 10.24           O  \n"
	"ATOM    301  CB  LEU A 835      30.998  51.553   6.746  1.00 13.25           C  \n"
	"ATOM    302  CG  LEU A 835      31.246  52.751   7.685  1.00 17.99           C  \n"
	"ATOM    303  CD1 LEU A 835      32.730  53.056   7.701  1.00 19.93           C  \n"
	"ATOM    304  CD2 LEU A 835      30.504  54.022   7.246  1.00 16.77           C  \n"
	"ATOM    305  H   LEU A 835      29.863  49.790   5.010  1.00  0.00           H  \n"
	"ATOM    306  N   THR A 836      27.614  51.697   7.678  1.00  8.18           N  \n"
	"ATOM    307  CA  THR A 836      26.435  52.500   7.470  1.00  5.92           C  \n"
	"ATOM    308  C   THR A 836      26.505  53.697   8.393  1.00  8.12           C  \n"
	"ATOM    309  O   THR A 836      26.776  53.522   9.582  1.00  8.29           O  \n"
	"ATOM    310  CB  THR A 836      25.185  51.670   7.769  1.00  7.14           C  \n"
	"ATOM    311  OG1 THR A 836      25.226  50.576   6.859  1.00 11.33           O  \n"
	"ATOM    312  CG2 THR A 836      23.865  52.437   7.623  1.00  4.69           C  \n"
	"ATOM    313  H   THR A 836      27.752  51.278   8.547  1.00  0.00           H  \n"
	"ATOM    314  HG1 THR A 836      25.066  50.927   5.969  1.00  0.00           H  \n"
	"ATOM    315  N   TYR A 837      26.241  54.908   7.916  1.00  7.71           N  \n"
	"ATOM    316  CA  TYR A 837      26.257  56.087   8.770  1.00  7.07           C  \n"
	"ATOM    317  C   TYR A 837      25.055  56.956   8.467  1.00  9.72           C  \n"
	"ATOM    318  O   TYR A 837      24.507  56.868   7.353  1.00  9.64           O  \n"
	"ATOM    319  CB  TYR A 837      27.547  56.891   8.558  1.00  7.76           C  \n"
	"ATOM    320  CG  TYR A 837      27.714  57.569   7.190  1.00  9.63           C  \n"
	"ATOM    321  CD1 TYR A 837      28.327  56.890   6.152  1.00  9.64           C  \n"
	"ATOM    322  CD2 TYR A 837      27.290  58.878   7.023  1.00 10.90           C  \n"
	"ATOM    323  CE1 TYR A 837      28.525  57.531   4.951  1.00 10.98           C  \n"
	"ATOM    324  CE2 TYR A 837      27.474  59.526   5.826  1.00 10.08           C  \n"
	"ATOM    325  CZ  TYR A 837      28.099  58.838   4.804  1.00 15.16           C  \n"
	"ATOM    326  OH  TYR A 837      28.318  59.491   3.602  1.00 17.95           O  \n"
	"ATOM    327  H   TYR A 837      25.973  55.020   6.977  1.00  0.00           H  \n"
	"ATOM    328  HH  TYR A 837      27.499  59.469   3.079  1.00  0.00           H  \n"
	"ATOM    329  N   GLY A 838      24.626  57.773   9.419  1.00  6.97           N  \n"
	"ATOM    330  CA  GLY A 838      23.536  58.704   9.194  1.00  8.10           C  \n"
	"ATOM    331  C   GLY A 838      23.238  59.471  10.472  1.00  7.81           C  \n"
	"ATOM    332  O   GLY A 838      23.792  59.143  11.528  1.00  7.59           O  \n"
	"ATOM    333  H   GLY A 838      25.027  57.731  10.315  1.00  0.00           H  \n"
	"ATOM    334  N   ILE A 839      22.361  60.468  10.445  1.00  8.49           N  \n"
	"ATOM    335  CA  ILE A 839      21.988  61.251  11.621  1.00  7.91           C  \n"
	"ATOM    336  C   ILE A 839      21.283  60.262  12.546  1.00  7.41           C  \n"
	"ATOM    337  O   ILE A 839      20.314  59.598  12.162  1.00  7.89           O  \n"
	"ATOM    338  CB  ILE A 839      21.086  62.452  11.220  1.00 12.25           C  \n"
	"ATOM    339  CG1 ILE A 839      21.877  63.447  10.351  1.00  8.61           C  \n"
	"ATOM    340  CG2 ILE A 839      20.501  63.133  12.468  1.00  4.71           C  \n"
	"ATOM    341  CD1 ILE A 839      20.922  64.381   9.600  1.00 13.39           C  \n"
	"ATOM    342  H   ILE A 839      21.881  60.630   9.611  1.00  0.00           H  \n"
	"ATOM    343  N   LYS A 840      21.747  60.192  13.787  1.00  7.92           N  \n"
	"ATOM    344  CA  LYS A 840      21.264  59.214  14.728  1.00 10.98           C  \n"
	"ATOM    345  C   LYS A 840      19.771  59.071  14.863  1.00 11.87           C  \n"
	"ATOM    346  O   LYS A 840      19.270  57.956  14.688  1.00 14.51           O  \n"
	"ATOM    347  CB  LYS A 840      21.884  59.441  16.094  1.00 14.46           C  \n"
	"ATOM    348  CG  LYS A 840      21.627  58.227  16.993  1.00 22.27           C  \n"
	"ATOM    349  CD  LYS A 840      22.248  58.320  18.385  1.00 29.43           C  \n"
	"ATOM    350  CE  LYS A 840      22.141  56.951  19.091  1.00 36.76           C  \n"
	"ATOM    351  NZ  LYS A 840      22.918  55.894  18.438  1.00 41.95           N  \n"
	"ATOM    352  H   LYS A 840      22.335  60.913  14.114  1.00  0.00           H  \n"
	"ATOM    353  HZ1 LYS A 840      23.881  56.206  18.220  1.00  0.00           H  \n"
	"ATOM    354  HZ2 LYS A 840      23.091  55.046  19.020  1.00  0.00           H  \n"
	"ATOM    355  HZ3 LYS A 840      22.528  55.520  17.543  1.00  0.00           H  \n"
	"ATOM    356  N   ASP A 841      19.071  60.160  15.065  1.00 13.50           N  \n"
	"ATOM    357  CA  ASP A 841      17.640  60.066  15.255  1.00 17.49           C  \n"
	"ATOM    358  C   ASP A 841      16.787  60.279  14.026  1.00 19.86           C  \n"
	"ATOM    359  O   ASP A 841      15.577  60.537  14.131  1.00 22.77           O  \n"
	"ATOM    360  CB AASP A 841      17.240  61.070  16.332  0.71 21.06           C  \n"
	"ATOM    361  CB BASP A 841      17.175  60.989  16.385  0.29 15.41           C  \n"
	"ATOM    362  CG AASP A 841      17.995  60.816  17.617  0.71 24.64           C  \n"
	"ATOM    363  CG BASP A 841      17.819  62.356  16.342  0.29 13.51           C  \n"
	"ATOM    364  OD1AASP A 841      17.575  59.974  18.400  0.71 23.53           O  \n"
	"ATOM    365  OD1BASP A 841      17.875  62.993  15.295  0.29 10.36           O  \n"
	"ATOM    366  OD2AASP A 841      19.044  61.431  17.787  0.71 29.55           O  \n"
	"ATOM    367  OD2BASP A 841      18.405  62.741  17.344  0.29 15.32           O  \n"
	"ATOM    368  H   ASP A 841      19.506  61.024  15.252  1.00  0.00           H  \n"
	"ATOM    369  N   VAL A 842      17.364  60.181  12.827  1.00 14.83           N  \n"
	"ATOM    370  CA  VAL A 842      16.567  60.282  11.607  1.00 14.80           C  \n"
	"ATOM    371  C   VAL A 842      16.652  58.852  11.058  1.00 15.79           C  \n"
	"ATOM    372  O   VAL A 842      17.617  58.535  10.353  1.00 16.41           O  \n"
	"ATOM    373  CB  VAL A 842      17.112  61.360  10.621  1.00 15.27           C  \n"
	"ATOM    374  CG1 VAL A 842      16.349  61.346   9.280  1.00 10.93           C  \n"
	"ATOM    375  CG2 VAL A 842      16.960  62.760  11.253  1.00 14.79           C  \n"
	"ATOM    376  H   VAL A 842      18.335  60.064  12.773  1.00  0.00           H  \n"
	"ATOM    377  N   PRO A 843      15.695  57.939  11.290  1.00 14.52           N  \n"
	"ATOM    378  CA  PRO A 843      15.725  56.562  10.816  1.00 15.91           C  \n"
	"ATOM    379  C   PRO A 843      15.960  56.346   9.335  1.00 13.98           C  \n"
	"ATOM    380  O   PRO A 843      16.609  55.386   8.925  1.00 15.03           O  \n"
	"ATOM    381  CB  PRO A 843      14.385  56.004  11.247  1.00 16.78           C  \n"
	"ATOM    382  CG  PRO A 843      14.004  56.847  12.439  1.00 15.20           C  \n"
	"ATOM    383  CD  PRO A 843      14.433  58.212  11.976  1.00 15.21           C  \n"
	"ATOM    384  N   GLY A 844      15.463  57.243   8.509  1.00 14.64           N  \n"
	"ATOM    385  CA  GLY A 844      15.635  57.077   7.093  1.00 13.97           C  \n"
	"ATOM    386  C   GLY A 844      16.969  57.577   6.591  1.00 14.50           C  \n"
	"ATOM    387  O   GLY A 844      17.236  57.323   5.417  1.00 13.32           O  \n"
	"ATOM    388  H   GLY A 844      14.951  58.004   8.857  1.00  0.00           H  \n"
	"ATOM    389  N   ASP A 845      17.791  58.268   7.388  1.00 11.02           N  \n"
	"ATOM    390  CA  ASP A 845      19.058  58.768   6.894  1.00 10.05           C  \n"
	"ATOM    391  C   ASP A 845      20.106  57.698   7.084  1.00 10.44           C  \n"
	"ATOM    392  O   ASP A 845      20.731  57.629   8.153  1.00 11.31           O  \n"
	"ATOM    393  CB  ASP A 845      19.518  60.040   7.617  1.00  8.18           C  \n"
	"ATOM    394  CG  ASP A 845      20.855  60.558   7.113  1.00 11.60           C  \n"
	"ATOM    395  OD1 ASP A 845      21.324  60.248   6.015  1.00 13.89           O  \n"
	"ATOM    396  OD2 ASP A 845      21.524  61.256   7.854  1.00 12.09           O  \n"
	"ATOM    397  H   ASP A 845      17.585  58.406   8.333  1.00  0.00           H  \n"
	"ATOM    398  N   ARG A 846      20.290  56.799   6.118  1.00  9.21           N  \n"
	"ATOM    399  CA  ARG A 846      21.296  55.751   6.251  1.00  8.99           C  \n"
	"ATOM    400  C   ARG A 846      22.081  55.502   4.956  1.00  8.68           C  \n"
	"ATOM    401  O   ARG A 846      21.512  55.012   3.964  1.00 11.48           O  \n"
	"ATOM    402  CB  ARG A 846      20.643  54.424   6.749  1.00 10.70           C  \n"
	"ATOM    403  CG  ARG A 846      20.019  54.404   8.167  1.00 14.69           C  \n"
	"ATOM    404  CD  ARG A 846      21.037  54.749   9.279  1.00 14.76           C  \n"
	"ATOM    405  NE  ARG A 846      20.461  54.805  10.628  1.00 19.43           N  \n"
	"ATOM    406  CZ  ARG A 846      20.182  55.948  11.309  1.00 15.79           C  \n"
	"ATOM    407  NH1 ARG A 846      20.411  57.161  10.777  1.00  9.45           N  \n"
	"ATOM    408  NH2 ARG A 846      19.755  55.863  12.581  1.00 18.32           N  \n"
	"ATOM    409  H   ARG A 846      19.779  56.900   5.279  1.00  0.00           H  \n"
	"ATOM    410  HE  ARG A 846      20.248  53.921  11.025  1.00  0.00           H  \n"
	"ATOM    411 HH11 ARG A 846      20.723  57.247   9.823  1.00  0.00           H  \n"
	"ATOM    412 HH12 ARG A 846      20.159  58.020  11.237  1.00  0.00           H  \n"
	"ATOM    413 HH21 ARG A 846      19.673  54.961  13.014  1.00  0.00           H  \n"
	"ATOM    414 HH22 ARG A 846      19.515  56.655  13.151  1.00  0.00           H  \n"
	"ATOM    415  N   THR A 847      23.360  55.837   4.897  1.00  8.06           N  \n"
	"ATOM    416  CA  THR A 847      24.165  55.553   3.703  1.00  6.92           C  \n"
	"ATOM    417  C   THR A 847      25.050  54.350   4.019  1.00  6.22           C  \n"
	"ATOM    418  O   THR A 847      25.714  54.354   5.063  1.00  7.00           O  \n"
	"ATOM    419  CB  THR A 847      25.062  56.745   3.336  1.00  4.86           C  \n"
	"ATOM    420  OG1 THR A 847      24.223  57.872   3.185  1.00  9.32           O  \n"
	"ATOM    421  CG2 THR A 847      25.852  56.515   2.049  1.00  9.16           C  \n"
	"ATOM    422  H   THR A 847      23.782  56.312   5.644  1.00  0.00           H  \n"
	"ATOM    423  HG1 THR A 847      23.818  58.054   4.042  1.00  0.00           H  \n"
	"ATOM    424  N   THR A 848      25.088  53.319   3.177  1.00  7.81           N  \n"
	"ATOM    425  CA  THR A 848      25.939  52.153   3.388  1.00  7.81           C  \n"
	"ATOM    426  C   THR A 848      27.044  52.159   2.324  1.00 10.27           C  \n"
	"ATOM    427  O   THR A 848      26.754  52.369   1.142  1.00  9.38           O  \n"
	"ATOM    428  CB  THR A 848      25.129  50.844   3.256  1.00  9.17           C  \n"
	"ATOM    429  OG1 THR A 848      24.111  50.895   4.273  1.00 11.64           O  \n"
	"ATOM    430  CG2 THR A 848      26.001  49.576   3.340  1.00  8.74           C  \n"
	"ATOM    431  H   THR A 848      24.550  53.357   2.358  1.00  0.00           H  \n"
	"ATOM    432  HG1 THR A 848      23.351  51.372   3.901  1.00  0.00           H  \n"
	"ATOM    433  N   ILE A 849      28.290  51.960   2.734  1.00 10.29           N  \n"
	"ATOM    434  CA  ILE A 849      29.460  51.909   1.869  1.00  9.65           C  \n"
	"ATOM    435  C   ILE A 849      29.962  50.473   1.998  1.00 11.80           C  \n"
	"ATOM    436  O   ILE A 849      30.058  49.949   3.116  1.00  9.32           O  \n"
	"ATOM    437  CB  ILE A 849      30.581  52.861   2.371  1.00 10.01           C  \n"
	"ATOM    438  CG1 ILE A 849      30.121  54.280   2.714  1.00 13.89           C  \n"
	"ATOM    439  CG2 ILE A 849      31.719  52.863   1.339  1.00 10.11           C  \n"
	"ATOM    440  CD1 ILE A 849      29.570  55.163   1.577  1.00 11.78           C  \n"
	"ATOM    441  H   ILE A 849      28.443  51.772   3.681  1.00  0.00           H  \n"
	"ATOM    442  N   ASP A 850      30.302  49.802   0.900  1.00 13.67           N  \n"
	"ATOM    443  CA  ASP A 850      30.834  48.453   0.969  1.00 17.41           C  \n"
	"ATOM    444  C   ASP A 850      32.335  48.585   0.762  1.00 14.28           C  \n"
	"ATOM    445  O   ASP A 850      32.799  49.064  -0.279  1.00 14.36           O  \n"
	"ATOM    446  CB  ASP A 850      30.248  47.536  -0.107  1.00 20.31           C  \n"
	"ATOM    447  CG  ASP A 850      28.732  47.381  -0.105  1.00 27.34           C  \n"
	"ATOM    448  OD1 ASP A 850      28.046  47.471   0.911  1.00 29.74           O  \n"
	"ATOM    449  OD2 ASP A 850      28.194  47.180  -1.186  1.00 35.54           O  \n"
	"ATOM    450  H   ASP A 850      30.172  50.220   0.024  1.00  0.00           H  \n"
	"ATOM    451  N   LEU A 851      33.124  48.236   1.768  1.00 12.40           N  \n"
	"ATOM    452  CA  LEU A 851      34.579  48.324   1.710  1.00 13.21           C  \n"
	"ATOM    453  C   LEU A 851      35.183  46.974   1.358  1.00 15.47           C  \n"
	"ATOM    454  O   LEU A 851      34.539  45.934   1.570  1.00 15.52           O  \n"
	"ATOM    455  CB  LEU A 851      35.218  48.733   3.055  1.00 12.28           C  \n"
	"ATOM    456  CG  LEU A 851      35.002  50.093   3.711  1.00 20.11           C  \n"
	"ATOM    457  CD1 LEU A 851      35.169  51.231   2.734  1.00 17.21           C  \n"
	"ATOM    458  CD2 LEU A 851      33.629  50.153   4.337  1.00 21.53           C  \n"
	"ATOM    459  H   LEU A 851      32.691  47.891   2.569  1.00  0.00           H  \n"
	"ATOM    460  N   THR A 852      36.400  46.947   0.826  1.00 16.81           N  \n"
	"ATOM    461  CA  THR A 852      37.051  45.689   0.531  1.00 21.53           C  \n"
	"ATOM    462  C   THR A 852      37.814  45.251   1.785  1.00 22.57           C  \n"
	"ATOM    463  O   THR A 852      38.065  46.070   2.673  1.00 19.44           O  \n"
	"ATOM    464  CB  THR A 852      37.862  45.821  -0.770  1.00 22.08           C  \n"
	"ATOM    465  OG1 THR A 852      38.573  47.042  -0.776  1.00 25.08           O  \n"
	"ATOM    466  CG2 THR A 852      36.913  45.793  -1.973  1.00 27.71           C  \n"
	"ATOM    467  H   THR A 852      36.912  47.774   0.639  1.00  0.00           H  \n"
	"ATOM    468  HG1 THR A 852      39.491  46.942  -0.455  1.00  0.00           H  \n"
	"ATOM    469  N   GLU A 853      38.214  43.987   1.950  1.00 28.76           N  \n"
	"ATOM    470  CA  GLU A 853      38.853  43.515   3.183  1.00 33.54           C  \n"
	"ATOM    471  C   GLU A 853      40.139  44.153   3.697  1.00 33.64           C  \n"
	"ATOM    472  O   GLU A 853      40.480  44.086   4.888  1.00 36.40           O  \n"
	"ATOM    473  CB  GLU A 853      38.949  41.994   3.091  1.00 38.58           C  \n"
	"ATOM    474  CG  GLU A 853      40.007  41.380   2.205  1.00 46.84           C  \n"
	"ATOM    475  CD  GLU A 853      41.133  40.874   3.085  1.00 55.71           C  \n"
	"ATOM    476  OE1 GLU A 853      41.016  39.809   3.702  1.00 63.17           O  \n"
	"ATOM    477  OE2 GLU A 853      42.119  41.581   3.293  1.00 59.46           O  \n"
	"ATOM    478  H   GLU A 853      38.134  43.352   1.204  1.00  0.00           H  \n"
	"ATOM    479  N   ASP A 854      40.804  44.779   2.747  1.00 33.51           N  \n"
	"ATOM    480  CA  ASP A 854      42.056  45.524   2.852  1.00 35.90           C  \n"
	"ATOM    481  C   ASP A 854      41.894  46.917   3.459  1.00 35.74           C  \n"
	"ATOM    482  O   ASP A 854      42.824  47.469   4.061  1.00 32.24           O  \n"
	"ATOM    483  CB  ASP A 854      42.626  45.668   1.440  1.00 45.65           C  \n"
	"ATOM    484  CG  ASP A 854      41.580  46.105   0.417  1.00 53.18           C  \n"
	"ATOM    485  OD1 ASP A 854      40.799  45.257  -0.027  1.00 60.46           O  \n"
	"ATOM    486  OD2 ASP A 854      41.404  47.287   0.135  1.00 60.36           O  \n"
	"ATOM    487  H   ASP A 854      40.442  44.671   1.845  1.00  0.00           H  \n"
	"ATOM    488  N   GLU A 855      40.712  47.542   3.355  1.00 31.38           N  \n"
	"ATOM    489  CA  GLU A 855      40.477  48.866   3.912  1.00 26.62           C  \n"
	"ATOM    490  C   GLU A 855      40.263  48.859   5.428  1.00 24.51           C  \n"
	"ATOM    491  O   GLU A 855      39.541  48.035   5.995  1.00 21.34           O  \n"
	"ATOM    492  CB  GLU A 855      39.300  49.497   3.183  1.00 26.33           C  \n"
	"ATOM    493  CG  GLU A 855      39.711  49.902   1.778  1.00 27.68           C  \n"
	"ATOM    494  CD  GLU A 855      38.584  50.493   0.961  1.00 24.13           C  \n"
	"ATOM    495  OE1 GLU A 855      37.745  49.749   0.468  1.00 24.54           O  \n"
	"ATOM    496  OE2 GLU A 855      38.516  51.712   0.815  1.00 30.84           O  \n"
	"ATOM    497  H   GLU A 855      40.006  47.121   2.820  1.00  0.00           H  \n"
	"ATOM    498  N   ASN A 856      40.914  49.777   6.129  1.00 23.82           N  \n"
	"ATOM    499  CA  ASN A 856      40.804  49.853   7.585  1.00 27.25           C  \n"
	"ATOM    500  C   ASN A 856      40.379  51.241   8.064  1.00 23.91           C  \n"
	"ATOM    501  O   ASN A 856      40.374  51.483   9.271  1.00 24.48           O  \n"
	"ATOM    502  CB  ASN A 856      42.151  49.408   8.273  1.00 30.98           C  \n"
	"ATOM    503  CG  ASN A 856      43.123  50.519   8.723  1.00 40.58           C  \n"
	"ATOM    504  OD1 ASN A 856      43.828  51.180   7.935  1.00 42.65           O  \n"
	"ATOM    505  ND2 ASN A 856      43.171  50.817  10.022  1.00 44.67           N  \n"
	"ATOM    506  H   ASN A 856      41.518  50.380   5.652  1.00  0.00           H  \n"
	"ATOM    507 HD21 ASN A 856      43.806  51.481  10.344  1.00  0.00           H  \n"
	"ATOM    508 HD22 ASN A 856      42.539  50.338  10.609  1.00  0.00           H  \n"
	"ATOM    509  N   GLN A 857      40.019  52.186   7.185  1.00 20.89           N  \n"
	"ATOM    510  CA  GLN A 857      39.607  53.523   7.586  1.00 18.40           C  \n"
	"ATOM    511  C   GLN A 857      38.691  54.089   6.510  1.00 19.42           C  \n"
	"ATOM    512  O   GLN A 857      38.736  53.584   5.378  1.00 16.27           O  \n"
	"ATOM    513  CB  GLN A 857      40.776  54.487   7.817  1.00 24.50           C  \n"
	"ATOM    514  CG  GLN A 857      41.607  54.952   6.618  1.00 33.30           C  \n"
	"ATOM    515  CD  GLN A 857      42.680  55.984   6.971  1.00 37.62           C  \n"
	"ATOM    516  OE1 GLN A 857      43.495  55.797   7.877  1.00 40.64           O  \n"
	"ATOM    517  NE2 GLN A 857      42.779  57.102   6.263  1.00 39.30           N  \n"
	"ATOM    518  H   GLN A 857      39.914  51.970   6.234  1.00  0.00           H  \n"
	"ATOM    519 HE21 GLN A 857      43.467  57.748   6.525  1.00  0.00           H  \n"
	"ATOM    520 HE22 GLN A 857      42.160  57.248   5.517  1.00  0.00           H  \n"
	"ATOM    521  N   TYR A 858      37.870  55.097   6.823  1.00 13.62           N  \n"
	"ATOM    522  CA  TYR A 858      36.981  55.685   5.849  1.00 12.80           C  \n"
	"ATOM    523  C   TYR A 858      36.767  57.123   6.273  1.00 11.56           C  \n"
	"ATOM    524  O   TYR A 858      36.485  57.392   7.436  1.00 10.10           O  \n"
	"ATOM    525  CB  TYR A 858      35.643  54.903   5.731  1.00 11.12           C  \n"
	"ATOM    526  CG  TYR A 858      34.842  55.389   4.519  1.00 19.40           C  \n"
	"ATOM    527  CD1 TYR A 858      35.290  55.143   3.227  1.00 15.62           C  \n"
	"ATOM    528  CD2 TYR A 858      33.690  56.127   4.715  1.00 17.45           C  \n"
	"ATOM    529  CE1 TYR A 858      34.588  55.639   2.137  1.00 20.62           C  \n"
	"ATOM    530  CE2 TYR A 858      32.990  56.623   3.631  1.00 23.63           C  \n"
	"ATOM    531  CZ  TYR A 858      33.442  56.377   2.349  1.00 22.45           C  \n"
	"ATOM    532  OH  TYR A 858      32.730  56.878   1.267  1.00 28.40           O  \n"
	"ATOM    533  H   TYR A 858      37.838  55.426   7.749  1.00  0.00           H  \n"
	"ATOM    534  HH  TYR A 858      32.042  57.494   1.582  1.00  0.00           H  \n"
	"ATOM    535  N   SER A 859      36.939  58.056   5.335  1.00  8.83           N  \n"
	"ATOM    536  CA  SER A 859      36.812  59.470   5.594  1.00 10.54           C  \n"
	"ATOM    537  C   SER A 859      35.442  59.903   5.113  1.00 11.82           C  \n"
	"ATOM    538  O   SER A 859      35.106  59.639   3.956  1.00 15.52           O  \n"
	"ATOM    539  CB  SER A 859      37.965  60.136   4.821  1.00  9.40           C  \n"
	"ATOM    540  OG  SER A 859      38.098  61.521   5.062  1.00 17.67           O  \n"
	"ATOM    541  H   SER A 859      37.134  57.777   4.409  1.00  0.00           H  \n"
	"ATOM    542  HG  SER A 859      38.194  61.684   6.015  1.00  0.00           H  \n"
	"ATOM    543  N   ILE A 860      34.645  60.560   5.951  1.00 11.85           N  \n"
	"ATOM    544  CA  ILE A 860      33.295  61.002   5.610  1.00 14.08           C  \n"
	"ATOM    545  C   ILE A 860      33.272  62.510   5.503  1.00 11.41           C  \n"
	"ATOM    546  O   ILE A 860      33.839  63.163   6.384  1.00 13.40           O  \n"
	"ATOM    547  CB  ILE A 860      32.291  60.622   6.726  1.00 14.11           C  \n"
	"ATOM    548  CG1 ILE A 860      32.320  59.156   7.091  1.00 17.66           C  \n"
	"ATOM    549  CG2 ILE A 860      30.899  60.998   6.253  1.00 16.12           C  \n"
	"ATOM    550  CD1 ILE A 860      31.398  58.866   8.295  1.00 20.42           C  \n"
	"ATOM    551  H   ILE A 860      34.997  60.807   6.831  1.00  0.00           H  \n"
	"ATOM    552  N   GLY A 861      32.583  63.100   4.529  1.00 12.61           N  \n"
	"ATOM    553  CA  GLY A 861      32.504  64.551   4.379  1.00 11.73           C  \n"
	"ATOM    554  C   GLY A 861      31.059  64.952   4.143  1.00 12.09           C  \n"
	"ATOM    555  O   GLY A 861      30.165  64.112   4.008  1.00 10.68           O  \n"
	"ATOM    556  H   GLY A 861      32.064  62.552   3.897  1.00  0.00           H  \n"
	"ATOM    557  N   ASN A 862      30.845  66.254   4.081  1.00 12.35           N  \n"
	"ATOM    558  CA  ASN A 862      29.538  66.887   3.870  1.00 13.78           C  \n"
	"ATOM    559  C   ASN A 862      28.502  66.562   4.927  1.00 14.52           C  \n"
	"ATOM    560  O   ASN A 862      27.315  66.411   4.649  1.00 14.60           O  \n"
	"ATOM    561  CB  ASN A 862      28.940  66.590   2.497  1.00 14.01           C  \n"
	"ATOM    562  CG  ASN A 862      29.783  67.172   1.392  1.00 18.50           C  \n"
	"ATOM    563  OD1 ASN A 862      29.933  68.381   1.212  1.00 20.81           O  \n"
	"ATOM    564  ND2 ASN A 862      30.425  66.284   0.658  1.00 25.13           N  \n"
	"ATOM    565  H   ASN A 862      31.627  66.833   4.202  1.00  0.00           H  \n"
	"ATOM    566 HD21 ASN A 862      30.950  66.593  -0.104  1.00  0.00           H  \n"
	"ATOM    567 HD22 ASN A 862      30.342  65.338   0.914  1.00  0.00           H  \n"
	"ATOM    568  N   LEU A 863      28.942  66.473   6.179  1.00 10.70           N  \n"
	"ATOM    569  CA  LEU A 863      28.040  66.199   7.285  1.00  9.04           C  \n"
	"ATOM    570  C   LEU A 863      27.491  67.536   7.734  1.00 11.91           C  \n"
	"ATOM    571  O   LEU A 863      28.006  68.591   7.348  1.00 14.52           O  \n"
	"ATOM    572  CB  LEU A 863      28.803  65.554   8.432  1.00 10.57           C  \n"
	"ATOM    573  CG  LEU A 863      29.495  64.236   8.082  1.00 12.62           C  \n"
	"ATOM    574  CD1 LEU A 863      30.329  63.787   9.266  1.00 12.56           C  \n"
	"ATOM    575  CD2 LEU A 863      28.458  63.188   7.640  1.00 12.25           C  \n"
	"ATOM    576  H   LEU A 863      29.885  66.661   6.358  1.00  0.00           H  \n"
	"ATOM    577  N   LYS A 864      26.437  67.492   8.537  1.00 15.33           N  \n"
	"ATOM    578  CA  LYS A 864      25.754  68.651   9.084  1.00 13.59           C  \n"
	"ATOM    579  C   LYS A 864      26.412  69.072  10.382  1.00 12.38           C  \n"
	"ATOM    580  O   LYS A 864      26.680  68.205  11.207  1.00 11.09           O  \n"
	"ATOM    581  CB  LYS A 864      24.312  68.266   9.411  1.00 17.95           C  \n"
	"ATOM    582  CG  LYS A 864      23.248  68.787   8.460  1.00 29.96           C  \n"
	"ATOM    583  CD  LYS A 864      22.623  70.080   8.980  1.00 37.98           C  \n"
	"ATOM    584  CE  LYS A 864      21.663  69.776  10.126  1.00 43.87           C  \n"
	"ATOM    585  NZ  LYS A 864      21.113  70.999  10.681  1.00 48.77           N  \n"
	"ATOM    586  H   LYS A 864      26.131  66.610   8.842  1.00  0.00           H  \n"
	"ATOM    587  HZ1 LYS A 864      21.895  71.612  11.005  1.00  0.00           H  \n"
	"ATOM    588  HZ2 LYS A 864      20.527  71.522  10.000  1.00  0.00           H  \n"
	"ATOM    589  HZ3 LYS A 864      20.537  70.755  11.502  1.00  0.00           H  \n"
	"ATOM    590  N   PRO A 865      26.705  70.342  10.637  1.00 14.84           N  \n"
	"ATOM    591  CA  PRO A 865      27.333  70.808  11.873  1.00 15.64           C  \n"
	"ATOM    592  C   PRO A 865      26.416  70.597  13.072  1.00 16.58           C  \n"
	"ATOM    593  O   PRO A 865      25.177  70.588  12.915  1.00 16.52           O  \n"
	"ATOM    594  CB  PRO A 865      27.535  72.294  11.617  1.00 16.88           C  \n"
	"ATOM    595  CG  PRO A 865      27.693  72.372  10.123  1.00 19.33           C  \n"
	"ATOM    596  CD  PRO A 865      26.606  71.422   9.653  1.00 17.72           C  \n"
	"ATOM    597  N   ASP A 866      27.051  70.429  14.235  1.00 13.83           N  \n"
	"ATOM    598  CA  ASP A 866      26.409  70.250  15.543  1.00 17.06           C  \n"
	"ATOM    599  C   ASP A 866      25.309  69.202  15.506  1.00 15.37           C  \n"
	"ATOM    600  O   ASP A 866      24.170  69.384  15.927  1.00 13.76           O  \n"
	"ATOM    601  CB  ASP A 866      25.908  71.623  16.042  1.00 21.16           C  \n"
	"ATOM    602  CG  ASP A 866      25.224  71.613  17.408  1.00 28.69           C  \n"
	"ATOM    603  OD1 ASP A 866      25.687  70.920  18.321  1.00 31.76           O  \n"
	"ATOM    604  OD2 ASP A 866      24.201  72.283  17.554  1.00 31.64           O  \n"
	"ATOM    605  H   ASP A 866      28.035  70.448  14.242  1.00  0.00           H  \n"
	"ATOM    606  N   THR A 867      25.646  68.065  14.921  1.00 13.04           N  \n"
	"ATOM    607  CA  THR A 867      24.668  67.010  14.754  1.00 13.37           C  \n"
	"ATOM    608  C   THR A 867      25.251  65.688  15.228  1.00 11.97           C  \n"
	"ATOM    609  O   THR A 867      26.426  65.395  14.964  1.00 12.38           O  \n"
	"ATOM    610  CB  THR A 867      24.271  66.961  13.226  1.00 11.00           C  \n"
	"ATOM    611  OG1 THR A 867      23.678  68.213  12.836  1.00 11.03           O  \n"
	"ATOM    612  CG2 THR A 867      23.265  65.857  12.924  1.00  9.43           C  \n"
	"ATOM    613  H   THR A 867      26.572  67.894  14.645  1.00  0.00           H  \n"
	"ATOM    614  HG1 THR A 867      24.133  68.908  13.335  1.00  0.00           H  \n"
	"ATOM    615  N   GLU A 868      24.433  64.911  15.938  1.00 11.56           N  \n"
	"ATOM    616  CA  GLU A 868      24.824  63.603  16.408  1.00 12.78           C  \n"
	"ATOM    617  C   GLU A 868      24.643  62.534  15.313  1.00 10.94           C  \n"
	"ATOM    618  O   GLU A 868      23.590  62.422  14.682  1.00 12.56           O  \n"
	"ATOM    619  CB  GLU A 868      23.980  63.281  17.607  1.00 19.23           C  \n"
	"ATOM    620  CG  GLU A 868      24.591  62.064  18.270  1.00 28.26           C  \n"
	"ATOM    621  CD  GLU A 868      24.094  61.788  19.668  1.00 34.88           C  \n"
	"ATOM    622  OE1 GLU A 868      22.895  61.640  19.874  1.00 36.98           O  \n"
	"ATOM    623  OE2 GLU A 868      24.929  61.697  20.559  1.00 41.79           O  \n"
	"ATOM    624  H   GLU A 868      23.527  65.223  16.152  1.00  0.00           H  \n"
	"ATOM    625  N   TYR A 869      25.645  61.736  15.017  1.00  8.26           N  \n"
	"ATOM    626  CA  TYR A 869      25.590  60.720  13.990  1.00  9.47           C  \n"
	"ATOM    627  C   TYR A 869      25.861  59.344  14.568  1.00  9.76           C  \n"
	"ATOM    628  O   TYR A 869      26.470  59.192  15.626  1.00 10.27           O  \n"
	"ATOM    629  CB  TYR A 869      26.684  60.963  12.946  1.00  8.78           C  \n"
	"ATOM    630  CG  TYR A 869      26.364  62.034  11.911  1.00 10.34           C  \n"
	"ATOM    631  CD1 TYR A 869      26.558  63.364  12.212  1.00  6.19           C  \n"
	"ATOM    632  CD2 TYR A 869      25.839  61.644  10.692  1.00  7.35           C  \n"
	"ATOM    633  CE1 TYR A 869      26.214  64.325  11.293  1.00  8.90           C  \n"
	"ATOM    634  CE2 TYR A 869      25.488  62.607   9.777  1.00 10.84           C  \n"
	"ATOM    635  CZ  TYR A 869      25.673  63.934  10.083  1.00 12.02           C  \n"
	"ATOM    636  OH  TYR A 869      25.295  64.899   9.173  1.00 15.03           O  \n"
	"ATOM    637  H   TYR A 869      26.428  61.776  15.601  1.00  0.00           H  \n"
	"ATOM    638  HH  TYR A 869      24.783  64.456   8.463  1.00  0.00           H  \n"
	"ATOM    639  N   GLU A 870      25.394  58.322  13.875  1.00  8.96           N  \n"
	"ATOM    640  CA  GLU A 870      25.653  56.962  14.285  1.00 10.66           C  \n"
	"ATOM    641  C   GLU A 870      26.367  56.262  13.110  1.00 11.05           C  \n"
	"ATOM    642  O   GLU A 870      26.084  56.564  11.938  1.00  8.63           O  \n"
	"ATOM    643  CB  GLU A 870      24.341  56.269  14.573  1.00 11.01           C  \n"
	"ATOM    644  CG  GLU A 870      24.604  54.846  15.035  1.00 23.06           C  \n"
	"ATOM    645  CD  GLU A 870      23.366  54.153  15.541  1.00 34.04           C  \n"
	"ATOM    646  OE1 GLU A 870      22.480  53.827  14.749  1.00 36.83           O  \n"
	"ATOM    647  OE2 GLU A 870      23.294  53.958  16.754  1.00 37.73           O  \n"
	"ATOM    648  H   GLU A 870      24.943  58.469  13.016  1.00  0.00           H  \n"
	"ATOM    649  N   VAL A 871      27.321  55.373  13.399  1.00 10.56           N  \n"
	"ATOM    650  CA  VAL A 871      28.029  54.593  12.402  1.00  9.38           C  \n"
	"ATOM    651  C   VAL A 871      27.868  53.142  12.863  1.00 12.68           C  \n"
	"ATOM    652  O   VAL A 871      27.893  52.850  14.067  1.00 12.47           O  \n"
	"ATOM    653  CB AVAL A 871      29.502  55.082  12.306  0.46 13.39           C  \n"
	"ATOM    654  CB BVAL A 871      29.532  54.948  12.238  0.54 13.72           C  \n"
	"ATOM    655  CG1AVAL A 871      30.296  54.941  13.605  0.46 15.68           C  \n"
	"ATOM    656  CG1BVAL A 871      29.649  56.368  11.725  0.54 11.90           C  \n"
	"ATOM    657  CG2AVAL A 871      30.208  54.347  11.185  0.46 11.19           C  \n"
	"ATOM    658  CG2BVAL A 871      30.361  54.804  13.518  0.54 16.59           C  \n"
	"ATOM    659  H   VAL A 871      27.491  55.168  14.345  1.00  0.00           H  \n"
	"ATOM    660  N   SER A 872      27.683  52.216  11.932  1.00  9.02           N  \n"
	"ATOM    661  CA  SER A 872      27.502  50.812  12.215  1.00 10.57           C  \n"
	"ATOM    662  C   SER A 872      28.403  50.060  11.233  1.00 11.05           C  \n"
	"ATOM    663  O   SER A 872      28.471  50.454  10.065  1.00 10.04           O  \n"
	"ATOM    664  CB  SER A 872      26.042  50.508  11.940  1.00 12.23           C  \n"
	"ATOM    665  OG  SER A 872      25.671  49.148  12.124  1.00 27.11           O  \n"
	"ATOM    666  H   SER A 872      27.706  52.472  10.985  1.00  0.00           H  \n"
	"ATOM    667  HG  SER A 872      25.913  48.676  11.313  1.00  0.00           H  \n"
	"ATOM    668  N   LEU A 873      29.112  49.018  11.672  1.00  8.98           N  \n"
	"ATOM    669  CA  LEU A 873      29.978  48.227  10.817  1.00 10.13           C  \n"
	"ATOM    670  C   LEU A 873      29.683  46.736  10.983  1.00 10.63           C  \n"
	"ATOM    671  O   LEU A 873      29.476  46.238  12.104  1.00  9.04           O  \n"
	"ATOM    672  CB  LEU A 873      31.444  48.474  11.137  1.00 11.93           C  \n"
	"ATOM    673  CG  LEU A 873      32.103  49.770  10.733  1.00 14.87           C  \n"
	"ATOM    674  CD1 LEU A 873      31.877  50.840  11.763  1.00 18.30           C  \n"
	"ATOM    675  CD2 LEU A 873      33.580  49.548  10.643  1.00 16.58           C  \n"
	"ATOM    676  H   LEU A 873      29.031  48.754  12.613  1.00  0.00           H  \n"
	"ATOM    677  N   ILE A 874      29.672  46.003   9.867  1.00 10.56           N  \n"
	"ATOM    678  CA  ILE A 874      29.419  44.559   9.806  1.00 12.49           C  \n"
	"ATOM    679  C   ILE A 874      30.494  43.913   8.914  1.00 16.61           C  \n"
	"ATOM    680  O   ILE A 874      30.848  44.479   7.865  1.00 13.19           O  \n"
	"ATOM    681  CB  ILE A 874      28.026  44.368   9.167  1.00 14.01           C  \n"
	"ATOM    682  CG1 ILE A 874      26.930  44.735  10.154  1.00 21.05           C  \n"
	"ATOM    683  CG2 ILE A 874      27.773  42.959   8.681  1.00 19.36           C  \n"
	"ATOM    684  CD1 ILE A 874      25.631  45.274   9.496  1.00 22.63           C  \n"
	"ATOM    685  H   ILE A 874      29.827  46.468   9.013  1.00  0.00           H  \n"
	"ATOM    686  N   SER A 875      31.068  42.780   9.336  1.00 17.09           N  \n"
	"ATOM    687  CA  SER A 875      32.034  42.018   8.544  1.00 16.34           C  \n"
	"ATOM    688  C   SER A 875      31.186  41.105   7.671  1.00 16.98           C  \n"
	"ATOM    689  O   SER A 875      30.132  40.620   8.115  1.00 15.92           O  \n"
	"ATOM    690  CB  SER A 875      32.873  41.149   9.438  1.00 14.15           C  \n"
	"ATOM    691  OG  SER A 875      33.577  41.975  10.339  1.00 25.98           O  \n"
	"ATOM    692  H   SER A 875      30.865  42.444  10.230  1.00  0.00           H  \n"
	"ATOM    693  HG  SER A 875      33.914  41.378  11.021  1.00  0.00           H  \n"
	"ATOM    694  N   ARG A 876      31.627  40.810   6.453  1.00 19.90           N  \n"
	"ATOM    695  CA  ARG A 876      30.865  39.986   5.537  1.00 21.50           C  \n"
	"ATOM    696  C   ARG A 876      31.734  38.921   4.929  1.00 23.24           C  \n"
	"ATOM    697  O   ARG A 876      32.933  39.122   4.736  1.00 19.73           O  \n"
	"ATOM    698  CB  ARG A 876      30.415  40.846   4.372  1.00 27.22           C  \n"
	"ATOM    699  CG  ARG A 876      29.324  41.870   4.614  1.00 31.10           C  \n"
	"ATOM    700  CD  ARG A 876      28.008  41.151   4.878  1.00 43.18           C  \n"
	"ATOM    701  NE  ARG A 876      27.692  40.175   3.836  1.00 49.41           N  \n"
	"ATOM    702  CZ  ARG A 876      26.707  40.352   2.952  1.00 53.23           C  \n"
	"ATOM    703  NH1 ARG A 876      25.986  41.478   2.930  1.00 58.27           N  \n"
	"ATOM    704  NH2 ARG A 876      26.499  39.403   2.037  1.00 59.37           N  \n"
	"ATOM    705  H   ARG A 876      32.495  41.154   6.150  1.00  0.00           H  \n"
	"ATOM    706  HE  ARG A 876      28.246  39.362   3.786  1.00  0.00           H  \n"
	"ATOM    707 HH11 ARG A 876      26.306  42.285   3.460  1.00  0.00           H  \n"
	"ATOM    708 HH12 ARG A 876      25.298  41.689   2.243  1.00  0.00           H  \n"
	"ATOM    709 HH21 ARG A 876      27.131  38.615   1.986  1.00  0.00           H  \n"
	"ATOM    710 HH22 ARG A 876      25.803  39.456   1.329  1.00  0.00           H  \n"
	"ATOM    711  N   ARG A 877      31.133  37.780   4.662  1.00 28.84           N  \n"
	"ATOM    712  CA  ARG A 877      31.806  36.695   3.983  1.00 35.33           C  \n"
	"ATOM    713  C   ARG A 877      30.657  35.885   3.421  1.00 33.94           C  \n"
	"ATOM    714  O   ARG A 877      29.960  35.146   4.115  1.00 33.47           O  \n"
	"ATOM    715  CB  ARG A 877      32.678  35.813   4.850  1.00 44.13           C  \n"
	"ATOM    716  CG  ARG A 877      33.496  34.901   3.926  1.00 55.47           C  \n"
	"ATOM    717  CD  ARG A 877      33.965  33.636   4.629  1.00 63.76           C  \n"
	"ATOM    718  NE  ARG A 877      34.928  33.924   5.681  1.00 69.51           N  \n"
	"ATOM    719  CZ  ARG A 877      34.820  33.442   6.927  1.00 72.51           C  \n"
	"ATOM    720  NH1 ARG A 877      33.793  32.651   7.291  1.00 71.84           N  \n"
	"ATOM    721  NH2 ARG A 877      35.761  33.775   7.811  1.00 74.59           N  \n"
	"ATOM    722  H   ARG A 877      30.195  37.653   4.921  1.00  0.00           H  \n"
	"ATOM    723  HE  ARG A 877      35.728  34.439   5.374  1.00  0.00           H  \n"
	"ATOM    724 HH11 ARG A 877      33.100  32.414   6.607  1.00  0.00           H  \n"
	"ATOM    725 HH12 ARG A 877      33.685  32.286   8.216  1.00  0.00           H  \n"
	"ATOM    726 HH21 ARG A 877      36.454  34.433   7.478  1.00  0.00           H  \n"
	"ATOM    727 HH22 ARG A 877      35.762  33.503   8.770  1.00  0.00           H  \n"
	"ATOM    728  N   GLY A 878      30.422  36.123   2.148  1.00 37.06           N  \n"
	"ATOM    729  CA  GLY A 878      29.393  35.476   1.390  1.00 35.84           C  \n"
	"ATOM    730  C   GLY A 878      28.034  35.812   1.932  1.00 37.77           C  \n"
	"ATOM    731  O   GLY A 878      27.569  36.951   1.991  1.00 40.42           O  \n"
	"ATOM    732  H   GLY A 878      30.996  36.768   1.692  1.00  0.00           H  \n"
	"ATOM    733  N   ASP A 879      27.460  34.707   2.326  1.00 39.84           N  \n"
	"ATOM    734  CA  ASP A 879      26.134  34.605   2.900  1.00 43.31           C  \n"
	"ATOM    735  C   ASP A 879      26.096  35.091   4.356  1.00 43.28           C  \n"
	"ATOM    736  O   ASP A 879      25.087  35.607   4.855  1.00 44.37           O  \n"
	"ATOM    737  CB  ASP A 879      25.693  33.121   2.749  1.00 46.71           C  \n"
	"ATOM    738  CG  ASP A 879      26.690  32.027   3.181  1.00 49.77           C  \n"
	"ATOM    739  OD1 ASP A 879      27.798  31.954   2.635  1.00 51.33           O  \n"
	"ATOM    740  OD2 ASP A 879      26.363  31.235   4.063  1.00 51.76           O  \n"
	"ATOM    741  H   ASP A 879      27.979  33.877   2.271  1.00  0.00           H  \n"
	"ATOM    742  N   MET A 880      27.249  35.022   5.014  1.00 41.22           N  \n"
	"ATOM    743  CA  MET A 880      27.410  35.400   6.402  1.00 39.61           C  \n"
	"ATOM    744  C   MET A 880      27.820  36.839   6.689  1.00 36.68           C  \n"
	"ATOM    745  O   MET A 880      28.495  37.503   5.891  1.00 33.40           O  \n"
	"ATOM    746  CB AMET A 880      28.439  34.472   7.008  0.89 41.14           C  \n"
	"ATOM    747  CB BMET A 880      28.391  34.459   7.109  0.11 39.22           C  \n"
	"ATOM    748  CG AMET A 880      27.912  33.055   7.065  0.89 47.79           C  \n"
	"ATOM    749  CG BMET A 880      27.877  33.042   7.361  0.11 39.86           C  \n"
	"ATOM    750  SD AMET A 880      29.216  31.813   7.738  0.89 53.61           S  \n"
	"ATOM    751  SD BMET A 880      26.330  32.907   8.520  0.11 39.42           S  \n"
	"ATOM    752  CE AMET A 880      30.349  31.907   6.200  0.89 51.20           C  \n"
	"ATOM    753  CE BMET A 880      27.193  33.413  10.146  0.11 39.53           C  \n"
	"ATOM    754  H   MET A 880      28.011  34.678   4.508  1.00  0.00           H  \n"
	"ATOM    755  N   SER A 881      27.425  37.307   7.873  1.00 31.68           N  \n"
	"ATOM    756  CA  SER A 881      27.731  38.636   8.338  1.00 27.69           C  \n"
	"ATOM    757  C   SER A 881      27.986  38.542   9.834  1.00 27.01           C  \n"
	"ATOM    758  O   SER A 881      27.564  37.568  10.485  1.00 29.15           O  \n"
	"ATOM    759  CB  SER A 881      26.555  39.545   8.040  1.00 25.99           C  \n"
	"ATOM    760  OG  SER A 881      25.335  39.025   8.534  1.00 31.33           O  \n"
	"ATOM    761  H   SER A 881      26.860  36.776   8.466  1.00  0.00           H  \n"
	"ATOM    762  HG  SER A 881      24.864  39.754   8.972  1.00  0.00           H  \n"
	"ATOM    763  N   SER A 882      28.766  39.469  10.398  1.00 22.98           N  \n"
	"ATOM    764  CA  SER A 882      28.974  39.469  11.825  1.00 15.61           C  \n"
	"ATOM    765  C   SER A 882      27.767  40.184  12.424  1.00 17.85           C  \n"
	"ATOM    766  O   SER A 882      26.812  40.611  11.746  1.00 14.61           O  \n"
	"ATOM    767  CB  SER A 882      30.219  40.261  12.133  1.00 15.24           C  \n"
	"ATOM    768  OG  SER A 882      30.145  41.646  11.799  1.00 15.66           O  \n"
	"ATOM    769  H   SER A 882      29.145  40.197   9.861  1.00  0.00           H  \n"
	"ATOM    770  HG  SER A 882      29.742  42.115  12.552  1.00  0.00           H  \n"
	"ATOM    771  N   ASN A 883      27.798  40.330  13.747  1.00 19.96           N  \n"
	"ATOM    772  CA  ASN A 883      26.788  41.113  14.432  1.00 17.59           C  \n"
	"ATOM    773  C   ASN A 883      27.263  42.558  14.208  1.00 13.19           C  \n"
	"ATOM    774  O   ASN A 883      28.468  42.771  14.026  1.00 15.25           O  \n"
	"ATOM    775  CB  ASN A 883      26.786  40.821  15.915  1.00 18.46           C  \n"
	"ATOM    776  CG  ASN A 883      26.151  39.485  16.285  1.00 21.02           C  \n"
	"ATOM    777  OD1 ASN A 883      25.086  39.094  15.787  1.00 23.56           O  \n"
	"ATOM    778  ND2 ASN A 883      26.849  38.745  17.119  1.00 18.49           N  \n"
	"ATOM    779  H   ASN A 883      28.536  39.951  14.270  1.00  0.00           H  \n"
	"ATOM    780 HD21 ASN A 883      26.383  37.948  17.471  1.00  0.00           H  \n"
	"ATOM    781 HD22 ASN A 883      27.748  39.008  17.393  1.00  0.00           H  \n"
	"ATOM    782  N   PRO A 884      26.406  43.573  14.108  1.00 13.25           N  \n"
	"ATOM    783  CA  PRO A 884      26.848  44.943  13.907  1.00 12.89           C  \n"
	"ATOM    784  C   PRO A 884      27.418  45.545  15.198  1.00 11.29           C  \n"
	"ATOM    785  O   PRO A 884      26.963  45.205  16.310  1.00 10.01           O  \n"
	"ATOM    786  CB  PRO A 884      25.578  45.659  13.471  1.00 11.53           C  \n"
	"ATOM    787  CG  PRO A 884      24.442  44.849  14.034  1.00 14.55           C  \n"
	"ATOM    788  CD  PRO A 884      24.956  43.425  13.925  1.00 11.15           C  \n"
	"ATOM    789  N   ALA A 885      28.454  46.369  15.022  1.00 10.52           N  \n"
	"ATOM    790  CA  ALA A 885      29.066  47.130  16.115  1.00 10.83           C  \n"
	"ATOM    791  C   ALA A 885      28.778  48.599  15.763  1.00 11.74           C  \n"
	"ATOM    792  O   ALA A 885      29.041  49.034  14.623  1.00  9.97           O  \n"
	"ATOM    793  CB  ALA A 885      30.550  46.881  16.177  1.00  7.37           C  \n"
	"ATOM    794  H   ALA A 885      28.815  46.486  14.117  1.00  0.00           H  \n"
	"ATOM    795  N   LYS A 886      28.216  49.370  16.710  1.00  8.97           N  \n"
	"ATOM    796  CA  LYS A 886      27.808  50.757  16.520  1.00 10.90           C  \n"
	"ATOM    797  C   LYS A 886      28.477  51.720  17.469  1.00 12.51           C  \n"
	"ATOM    798  O   LYS A 886      28.714  51.382  18.639  1.00 12.85           O  \n"
	"ATOM    799  CB  LYS A 886      26.341  50.962  16.831  1.00 12.08           C  \n"
	"ATOM    800  CG  LYS A 886      25.421  50.167  15.950  1.00 22.29           C  \n"
	"ATOM    801  CD  LYS A 886      24.055  50.289  16.560  1.00 26.33           C  \n"
	"ATOM    802  CE  LYS A 886      23.102  49.741  15.525  1.00 36.10           C  \n"
	"ATOM    803  NZ  LYS A 886      22.017  49.040  16.166  1.00 42.39           N  \n"
	"ATOM    804  H   LYS A 886      28.165  49.015  17.613  1.00  0.00           H  \n"
	"ATOM    805  HZ1 LYS A 886      21.492  49.596  16.872  1.00  0.00           H  \n"
	"ATOM    806  HZ2 LYS A 886      21.397  48.576  15.475  1.00  0.00           H  \n"
	"ATOM    807  HZ3 LYS A 886      22.400  48.259  16.750  1.00  0.00           H  \n"
	"ATOM    808  N   GLU A 887      28.674  52.943  16.976  1.00 10.17           N  \n"
	"ATOM    809  CA  GLU A 887      29.283  54.027  17.733  1.00 11.87           C  \n"
	"ATOM    810  C   GLU A 887      28.579  55.306  17.334  1.00 10.66           C  \n"
	"ATOM    811  O   GLU A 887      27.995  55.388  16.243  1.00 10.50           O  \n"
	"ATOM    812  CB  GLU A 887      30.751  54.189  17.406  1.00 14.32           C  \n"
	"ATOM    813  CG  GLU A 887      31.522  52.926  17.755  1.00 23.67           C  \n"
	"ATOM    814  CD  GLU A 887      32.873  53.105  18.400  1.00 27.37           C  \n"
	"ATOM    815  OE1 GLU A 887      33.302  54.234  18.608  1.00 35.92           O  \n"
	"ATOM    816  OE2 GLU A 887      33.517  52.107  18.709  1.00 29.61           O  \n"
	"ATOM    817  H   GLU A 887      28.381  53.142  16.060  1.00  0.00           H  \n"
	"ATOM    818  N   THR A 888      28.564  56.287  18.217  1.00  8.99           N  \n"
	"ATOM    819  CA  THR A 888      27.961  57.585  17.954  1.00 13.10           C  \n"
	"ATOM    820  C   THR A 888      29.061  58.652  17.974  1.00 11.93           C  \n"
	"ATOM    821  O   THR A 888      30.110  58.400  18.571  1.00 13.02           O  \n"
	"ATOM    822  CB ATHR A 888      26.883  57.739  19.054  0.54 15.21           C  \n"
	"ATOM    823  CB BTHR A 888      26.956  58.020  19.036  0.46 14.16           C  \n"
	"ATOM    824  OG1ATHR A 888      25.892  56.702  18.873  0.54 17.08           O  \n"
	"ATOM    825  OG1BTHR A 888      27.469  57.741  20.327  0.46 19.15           O  \n"
	"ATOM    826  CG2ATHR A 888      26.220  59.086  18.977  0.54 17.13           C  \n"
	"ATOM    827  CG2BTHR A 888      25.697  57.235  18.907  0.46 15.24           C  \n"
	"ATOM    828  H   THR A 888      29.033  56.197  19.078  1.00  0.00           H  \n"
	"ATOM    829  HG1ATHR A 888      26.364  55.864  18.773  0.54  0.00           H  \n"
	"ATOM    830  HG1BTHR A 888      28.360  58.116  20.435  0.46  0.00           H  \n"
	"ATOM    831  N   PHE A 889      28.907  59.806  17.341  1.00 10.28           N  \n"
	"ATOM    832  CA  PHE A 889      29.897  60.877  17.375  1.00  9.41           C  \n"
	"ATOM    833  C   PHE A 889      29.111  62.169  17.122  1.00 11.03           C  \n"
	"ATOM    834  O   PHE A 889      27.985  62.077  16.597  1.00  9.16           O  \n"
	"ATOM    835  CB  PHE A 889      31.014  60.674  16.314  1.00  8.64           C  \n"
	"ATOM    836  CG  PHE A 889      30.632  60.708  14.835  1.00 10.20           C  \n"
	"ATOM    837  CD1 PHE A 889      30.251  59.545  14.179  1.00 12.46           C  \n"
	"ATOM    838  CD2 PHE A 889      30.738  61.900  14.124  1.00 11.03           C  \n"
	"ATOM    839  CE1 PHE A 889      29.992  59.572  12.825  1.00 10.88           C  \n"
	"ATOM    840  CE2 PHE A 889      30.472  61.916  12.769  1.00  7.92           C  \n"
	"ATOM    841  CZ  PHE A 889      30.106  60.756  12.121  1.00  6.96           C  \n"
	"ATOM    842  H   PHE A 889      28.065  59.995  16.868  1.00  0.00           H  \n"
	"ATOM    843  N   THR A 890      29.638  63.367  17.361  1.00 10.21           N  \n"
	"ATOM    844  CA  THR A 890      28.888  64.590  17.129  1.00  9.60           C  \n"
	"ATOM    845  C   THR A 890      29.789  65.518  16.332  1.00 10.62           C  \n"
	"ATOM    846  O   THR A 890      30.986  65.501  16.532  1.00 11.80           O  \n"
	"ATOM    847  CB ATHR A 890      28.606  65.132  18.547  0.28 10.72           C  \n"
	"ATOM    848  CB BTHR A 890      28.378  65.243  18.462  0.72 11.80           C  \n"
	"ATOM    849  OG1ATHR A 890      27.625  64.247  19.079  0.28  8.97           O  \n"
	"ATOM    850  OG1BTHR A 890      27.524  66.318  18.056  0.72 12.94           O  \n"
	"ATOM    851  CG2ATHR A 890      28.145  66.571  18.595  0.28 10.15           C  \n"
	"ATOM    852  CG2BTHR A 890      29.474  65.701  19.418  0.72  7.66           C  \n"
	"ATOM    853  H   THR A 890      30.579  63.488  17.652  1.00  0.00           H  \n"
	"ATOM    854  HG1ATHR A 890      27.458  64.426  20.019  0.50  0.00           H  \n"
	"ATOM    855  HG1BTHR A 890      27.026  66.721  18.797  0.50  0.00           H  \n"
	"ATOM    856  N   THR A 891      29.279  66.246  15.341  1.00 10.80           N  \n"
	"ATOM    857  CA  THR A 891      30.109  67.194  14.602  1.00 11.54           C  \n"
	"ATOM    858  C   THR A 891      30.016  68.571  15.274  1.00 12.68           C  \n"
	"ATOM    859  O   THR A 891      29.317  68.719  16.280  1.00 12.09           O  \n"
	"ATOM    860  CB  THR A 891      29.611  67.270  13.145  1.00 10.50           C  \n"
	"ATOM    861  OG1 THR A 891      28.241  67.613  13.276  1.00  9.51           O  \n"
	"ATOM    862  CG2 THR A 891      29.759  65.984  12.338  1.00  5.19           C  \n"
	"ATOM    863  OXT THR A 891      30.619  69.506  14.762  1.00 17.59           O  \n"
	"ATOM    864  H   THR A 891      28.330  66.160  15.113  1.00  0.00           H  \n"
	"ATOM    865  HG1 THR A 891      27.903  67.602  12.375  1.00  0.00           H  \n"
	"TER     866      THR A 891                                                      \n"
	"END                                                                             \n";
}

#endif
