
#!/usr/bin/python

# (c) Copyright Rosetta Commons Member Institutions.
# (c) This file is part of the Rosetta software suite and is made available under license.
# (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
# (c) For more information, see http://www.rosettacommons.org. Questions about this can be
# (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

## @file   /GUIs/pyrosetta_toolkit/window_main/IO/GUIInput.py
## @brief  Class responsible for managing input variables of the GUI.
## @author Jared Adolf-Bryfogle (jadolfbr@gmail.com)
## @author Steven Combs (steven.combs1@gmail.com)

#Rosetta Imports
from rosetta import *

#Python Imports
import urllib2
import os.path
import re

#Tkinter Imports
from Tkinter import StringVar
import tkFileDialog
import tkSimpleDialog

#Toolkit Imports
from window_main import global_variables
from modules.RegionalScoring import RegionalScoring
from modules.Region import Region
from modules.Region import Regions
from modules.tools import input as input_tools
from window_modules.clean_pdb.FixPDBWindow import FixPDBWindow
from window_modules.options_system.OptionSystemManager import OptionSystemManager
#from pyrosetta_toolkit import main_window

class GUIInput:
    def __init__(self, toolkit):
        self.toolkit = toolkit; #Basically an AP of the toolkit
        self.pose = self.toolkit.pose
        self.pdb_path = StringVar(); self.pdb_path.set("0");
        self.PDBLIST = StringVar(); self.PDBLIST.set("")
        
        self.region_start=StringVar(); #Start of Region
        self.region_end=StringVar(); #End of Region
        self.region_chain=StringVar(); #Chain of Region
        self.region_sequence=StringVar(); #Sequence in Entry
        self.loops_as_strings = []; #Array of Regions: start:end:chain
        self.regions = Regions(); #This will replace loops_as_strings
        self.loops = Loops()
        
        #These are set if a user selects a residue from the sequence
        self.residue_string = StringVar(); #String that is displayed when individual reside is selected
        self.residue_resnum = StringVar();
        self.residue_rosetta_resnum = StringVar();
        self.residue_chain = StringVar()
        
        self.constraint_file_path = StringVar(); #Path to constraint file if loaded through GUI (Not options system).
        self.param_pathlist_file = ""; #Path to a file which lists paths to all params to use.  One on each line
        self.param_paths = []; #Array of parameter paths.
        self.loaded_paths = [];  #Since something about loading a new residuetypeset is global, with horrible exception handling, WE need to keep track of it.
        self.nonstandard_ResidueTypeSet = ""; #This is set through the ncaa window or loading a param path file.  
        
        self.options_manager= OptionSystemManager(); #This is due to Protocols needing Rosetta to be reinitialized without loosing already set options -  to set the seed up before multiprocessing runs!
        
        self.options_manager= OptionSystemManager(); #This is due to Protocols needing Rosetta to be reinitialized without loosing already set options -  to set the seed up before multiprocessing runs!
        
        self.pdb_url = "http://www.rcsb.org/pdb/files"
        #if 0: self.toolkit = main_window()
        
#### POSE INPUT ####

    def choose_load_pose(self, message="Load Pose"):
        """
        Loads a Pose through the tk File Dialog
        """
        infilename = tkFileDialog.askopenfilename(initialdir=global_variables.current_directory, title=message)
        if not infilename:return
        
        global_variables.current_directory= os.path.dirname(infilename)
        print global_variables.current_directory
        self.load_pose(infilename)
        
    def fetch_pdb(self):
        """
        Fetches the PDB, opens FixPDBWindow to allow the user to clean the PDB before trying to load it.
        """
        #Create default directory
        outpath = self.toolkit.toolkit_home+"/PDBs"
        if not os.path.exists(outpath):os.mkdir(outpath)
        global_variables.current_directory = outpath
        
        #Ask for PDB id.
        pdbID = tkSimpleDialog.askstring(title="Fetch pdb", prompt="Please enter PDB ID.")
        if not pdbID: return
        
        #Open and Write the PDB
        FILE = urllib2.urlopen(self.pdb_url+'/'+pdbID.lower()+'.pdb')
        OUTFILE = open(outpath+'/'+pdbID.upper()+'.pdb', 'w')
        for line in FILE:
            OUTFILE.write(line)
        OUTFILE.close()
        
        fetched_pdb = outpath+'/'+pdbID.upper()+'.pdb'
        print "PDB saved to pyrosetta_toolkit/PDBs"
        cleaner = FixPDBWindow(self, self.toolkit.score_class, self.toolkit.pose, fetched_pdb)
        cleaner.runfixPDBWindow(self.toolkit._tk_, 0, 0)
    
    def select_pose_then_launch_fixpdb(self):
        """
        This way of loading a pose asks the user to open a PDB, then launches the fixPDBWindow as per Labonte's suggestion.
        """
        
        infilename = tkFileDialog.askopenfilename(initialdir=global_variables.current_directory, title="Select PDB file")
        if not infilename:return
        
        global_variables.current_directory= os.path.dirname(infilename)
        print global_variables.current_directory
        cleaner = FixPDBWindow(self, self.toolkit.score_class, self.toolkit.pose, infilename)
        cleaner.cleaned_pdb_path.set(infilename)
        cleaner.runfixPDBWindow(self.toolkit._tk_, 0, 0)
        cleaner.enable_load()
        
    def load_pose(self, path):
        """
        Load a pose into the toolkit.pose variable.  Setup nessessary variables/etc for objects and window objects of the toolkit.  Can have NCAA that have been enabled.
        """
        self.pdb_path.set(path)
        print self.pdb_path.get()
        if self.nonstandard_ResidueTypeSet:
            self.toolkit.pose.assign(pose_from_pdb(self.nonstandard_ResidueTypeSet, path))
        else:
            pose_from_pdb(self.toolkit.pose, self.pdb_path.get())
        self.toolkit.native_pose.assign(self.toolkit.pose); #Set native pose for RMSD.

        print self.toolkit.pose
        self.toolkit.pymol_class.SendNewPose()
        self.regional_score_class = RegionalScoring(self.toolkit.pose, self.toolkit.score_class.score);
        pdbname = os.path.basename(self.pdb_path.get())
        pdbname = pdbname.split(".")[0]
        self.toolkit.output_class.outname.set(pdbname)
        self.toolkit.output_class.outdir.set(os.path.dirname(self.pdb_path.get()))
        self.toolkit.DesignDic = dict()
        self.region_sequence.set(self.toolkit.pose.sequence())
    
    def return_loaded_pose(self, path):
        """
        Load and return a pose.  Can have NCAA that have been enabled.
        """
        p = Pose()
        if self.nonstandard_ResidueTypeSet:
            p.assign(pose_from_pdb(self.nonstandard_ResidueTypeSet, path))
        else:
            pose_from_pdb(p, self.pdb_path.get())
        return p
    
    def set_PDBLIST(self):
        infilename = tkFileDialog.askopenfilename(initialdir=global_variables.current_directory,title='Open PDBLIST')
        if not infilename:return
        global_variables.current_directory =os.path.dirname(infilename)
        print "PDBLIST set"
        self.PDBLIST.set(infilename)
    
    def load_param_list(self):
        """
        Loads paths from param path files into an array.  Creates a residue type set from the params.
        """
        infilename = tkFileDialog.askopenfilename(initialdir=global_variables.current_directory,title='Open param pathList file')
        if not infilename: return
        self.param_pathlist_file = infilename
        global_variables.current_directory =os.path.dirname(infilename)
        FILE = open(infilename, 'r')
        for line in FILE:
            if re.search("#", line): continue
            line = line.strip()
            self.param_paths.append(line)
        
        self.nonstandard_ResidueTypeSet, self.loaded_paths = input_tools.get_residuetypeset_from_path_array(self.param_paths, self.loaded_paths)
        FILE.close()
        
    def load_loop(self):
        """
        Returns a loops_as_strings array to be used by the InputFrame
        """
        
        if not self.toolkit.pose.total_residue():
            print "\nNumbering conversion requires a pose to be loaded...please load a pose.."
            return
        
        infilename = tkFileDialog.askopenfilename(initialdir=global_variables.current_directory,title='Open Loop file')
        if not infilename: return
        global_variables.current_directory =os.path.dirname(infilename)
        FILE = open(infilename, 'r')
        loops_as_strings = []
        for line in FILE:
            print line
            line = line.strip()
            lineSP = line.split()
            start = lineSP[1]
            end = lineSP[2]
            chain_start = self.toolkit.pose.pdb_info().chain(int(start))
            chain_end = self.toolkit.pose.pdb_info().chain(int(end))
            if chain_start != chain_end:
                print "Invalid loop for GUI.  Start and end residues are on different chains.\n"
                return
            loop_string = repr(self.toolkit.pose.pdb_info().number(int(start)))+":"+repr(self.toolkit.pose.pdb_info().number(int(end)))+":"+chain_start
            loops_as_strings.append(loop_string)
        FILE.close()
        return loops_as_strings

###Region Selection####
    def set_residue_of_interest(self, resnum, chain, rosetta_resnum):
        """
        Sets current individual residue information.
        All Strings.
        """
        self.residue_resnum.set(resnum)
        self.residue_chain.set(chain)
        self.residue_rosetta_resnum.set(rosetta_resnum)
        
    def return_region_from_entry(self):
        """
        Returns a region of the currently set region parameters in the region entry boxes.
        """
        if not self.region_chain.get():return
        looFull = self.region_start.get()+ ":"+ self.region_end.get()+":"+self.region_chain.get().upper()
        start = looFull.split(":")[0]; end = looFull.split(":")[1]
        
         #Chain
        if (start == "" and end==""):
            region = Region(self.region_chain.get().upper(), None, None)
        #Nter
        elif start=="":
            region = Region(self.region_chain.get().upper(), None, int(end))
        #Cter
        elif end=="":
            region = Region(self.region_chain.get().upper(), int(start), None)
        #Loop
        else: 
            region = Region(self.region_chain.get().upper(), int(start), int(end))
        
        
        return region        