// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.5 $
//  $Date: 2006/02/20 13:26:56 $
//  $Author: plato $

#ifndef ADDITIONAL_BACKGROUND_NODES_INTERACTION_GRAPH_H
#define ADDITIONAL_BACKGROUND_NODES_INTERACTION_GRAPH_H

//Rosetta Headers
//#include "PrecomputedInteractionGraph.h"

//STL Headers
#include <list>
#include <vector>

namespace pack{

template < typename V, typename E, typename G > class FirstClassNode;
template < typename V, typename E, typename G > class FirstClassEdge;
template < typename V, typename E, typename G > class BackgroundNode;
template < typename V, typename E, typename G > class BackgroundToFirstClassEdge;
template < typename V, typename E, typename G > class AdditionalBackgroundNodesInteractionGraph;

template < typename V, typename E, typename G >
class FirstClassNode : public V
{
public:
	typedef typename std::list< BackgroundToFirstClassEdge< V, E, G > * >::iterator   BackgroundEdgeListIter;
	typedef typename std::list< BackgroundToFirstClassEdge< V, E, G > * >             BackgroundEdgeList;
	typedef typename std::vector< BackgroundToFirstClassEdge< V, E, G > * >           BackgroundEdgeVector;
	typedef typename std::vector< BackgroundToFirstClassEdge< V, E, G > * >::iterator BackgroundEdgeVectorIter;
	typedef typename std::vector< BackgroundNode< V, E, G > * >                       BackgroundNodeVector;


public:
	virtual ~FirstClassNode();
	FirstClassNode( G * owner, int node_id, int num_states);


	BackgroundEdgeListIter
	add_background_edge(BackgroundToFirstClassEdge< V, E, G > * edge );

	void drop_background_edge
	(
		BackgroundEdgeListIter edge
	);

	virtual unsigned int count_dynamic_memory() const;

protected:

	inline
	int get_num_edges_to_background_nodes() const
	{
		return num_edges_to_bg_nodes_;
	}

	inline
	BackgroundToFirstClassEdge< V, E, G >* get_edge_to_bg_node( int index ) const
	{
		assert( bg_edge_vector_up_to_date_ );
		return bg_edge_vector_[ index ];
	}

	inline
	int get_index_of_adjacent_background_node( int index ) const
	{
		assert( bg_edge_vector_up_to_date_ );
		return adjacent_bg_node_indices_[ index ];
	}

	inline
	BackgroundNode< V, E, G > * get_adjacent_background_node( int index ) const
	{	assert( bg_edge_vector_up_to_date_ );
		return bg_node_vector_[ index ];
	}

	inline
	bool get_bg_edge_vector_up_to_date_() const
	{	return bg_edge_vector_up_to_date_;}

	void update_bg_edge_vector();

private:
	int num_edges_to_bg_nodes_;
	BackgroundEdgeList bg_edge_list_;
	BackgroundEdgeVector bg_edge_vector_;
	std::vector< int > adjacent_bg_node_indices_;
	BackgroundNodeVector bg_node_vector_;
	bool bg_edge_vector_up_to_date_;

	//no default constructor, uncopyable
	FirstClassNode();
	FirstClassNode( FirstClassNode< V, E, G > const & );
	FirstClassNode< V, E, G > & operator = ( FirstClassNode< V, E, G > const & );
};

template < typename V, typename E, typename G >
class FirstClassEdge : public E
{
public:
	virtual ~FirstClassEdge();
	FirstClassEdge(
		G * owner,
		int first_node_ind,
		int second_node_ind
	);

	virtual unsigned int count_dynamic_memory() const { return E::count_dynamic_memory(); }

private:
	//no default constructor, uncopyable
	FirstClassEdge();
	FirstClassEdge( FirstClassEdge< V, E, G > const & );
	FirstClassEdge< V, E, G > & operator = ( FirstClassEdge< V, E, G > const & );

};

template < typename V, typename E, typename G >
class BackgroundNode
{
public:
	typedef typename std::list< BackgroundToFirstClassEdge< V, E, G > * >::iterator   BackgroundEdgeListIter;
	typedef typename std::list< BackgroundToFirstClassEdge< V, E, G > * >             BackgroundEdgeList;
	typedef typename std::vector< BackgroundToFirstClassEdge< V, E, G > * >           BackgroundEdgeVector;
	typedef typename std::vector< BackgroundToFirstClassEdge< V, E, G > * >::iterator BackgroundEdgeVectorIter;
	typedef typename std::vector< FirstClassNode< V, E, G > * >                       FirstClassNodeVector;

public:
	virtual ~BackgroundNode();
	BackgroundNode(
		AdditionalBackgroundNodesInteractionGraph< V, E, G >* owner,
		int node_index
	);

	BackgroundEdgeListIter
	add_edge( BackgroundToFirstClassEdge< V, E, G > * edge_ptr);

	void drop_edge( BackgroundEdgeListIter edge);

	BackgroundToFirstClassEdge< V, E, G > * find_edge( int fc_node_index );

	virtual void prepare_for_simulated_annealing() = 0;
	virtual void print() const = 0;

	virtual unsigned int count_static_memory() const = 0;
	virtual unsigned int count_dynamic_memory() const;

protected:
	void update_edge_vector();
	inline
	int get_node_index() const
	{
		return node_index_;
	}

	inline
	int get_num_incident_edges() const
	{
		return num_incident_edges_;
	}

	inline
	BackgroundToFirstClassEdge< V, E, G > * get_incident_edge( int index ) const
	{
		assert( edge_vector_up_to_date_ );
		return edge_vector_[ index ];
	}

	inline
	int get_index_of_adjacent_first_class_node( int index ) const
	{
		assert( edge_vector_up_to_date_ );
		return adjacent_first_class_node_indices_[ index ];
	}

	inline
	FirstClassNode< V, E, G> * get_adjacent_first_class_node( int index ) const
	{
		assert( edge_vector_up_to_date_ );
		return adjacent_nodes_[ index ];
	}

	inline
	bool get_edge_vector_up_to_date() const
	{
		return edge_vector_up_to_date_;
	}

private:
	int node_index_;
	int num_incident_edges_;
	BackgroundEdgeList edge_list_;
	BackgroundEdgeVector edge_vector_;
	std::vector< int > adjacent_first_class_node_indices_;
	FirstClassNodeVector adjacent_nodes_;
	bool edge_vector_up_to_date_;
	AdditionalBackgroundNodesInteractionGraph< V, E, G > * owner_;

	//no default constructor, uncopyable
	BackgroundNode();
	BackgroundNode( BackgroundNode< V, E, G > const & );
	BackgroundNode< V, E, G > & operator = ( BackgroundNode< V, E, G > const & );
};


template < typename V, typename E, typename G >
class BackgroundToFirstClassEdge
{
public:
	typedef typename std::list< BackgroundToFirstClassEdge< V, E, G > * >::iterator BackgroundEdgeListIter;

public:
	virtual ~BackgroundToFirstClassEdge();
	BackgroundToFirstClassEdge(
		AdditionalBackgroundNodesInteractionGraph< V, E, G > * owner,
		int fc_node_index,
		int bg_node_index);

	int get_other_ind( FirstClassNode< V, E, G >* ) const;
	int get_other_ind( BackgroundNode< V, E, G >* ) const;

	BackgroundNode< V, E, G > * get_other_node( FirstClassNode< V, E, G >*  ) const;
	FirstClassNode< V, E, G > * get_other_node( BackgroundNode< V, E, G >*  ) const;

	void set_pos_in_owners_list( BackgroundEdgeListIter );
	void set_pos_in_node_edgevector( FirstClassNode< V, E, G > * caller, int pos );
	void set_pos_in_node_edgevector( BackgroundNode< V, E, G > * caller, int pos);
	bool same_edge( int fc_node_index, int bg_node_index ) const;

	virtual void prepare_for_simulated_annealing() = 0;

	virtual unsigned int count_static_memory() const = 0;
	virtual unsigned int count_dynamic_memory() const;

protected:
	inline
	int get_first_class_node_index() const
	{	return first_class_node_index_;}

	inline
	int get_background_node_index() const
	{	return background_node_index_;}

	inline
	FirstClassNode< V, E, G > * get_first_class_node() const
	{	return first_class_node_;}

	inline
	BackgroundNode< V, E, G > * get_background_node() const
	{	return background_node_;}

	inline
	AdditionalBackgroundNodesInteractionGraph< V, E, G > * get_owner() const
	{	return owner_;}


private:
	int first_class_node_index_;
	int background_node_index_;
	FirstClassNode< V, E, G > * first_class_node_;
	BackgroundNode< V, E, G > * background_node_;
	int pos_in_fc_edge_vector_;
	int pos_in_bg_edge_vector_;
	BackgroundEdgeListIter pos_in_fc_nodes_edge_list_;
	BackgroundEdgeListIter pos_in_bg_nodes_edge_list_;
	BackgroundEdgeListIter pos_in_owners_edge_list_;
	AdditionalBackgroundNodesInteractionGraph< V, E, G > * owner_;

	//no default constructor, uncopyable
	BackgroundToFirstClassEdge();
	BackgroundToFirstClassEdge(BackgroundToFirstClassEdge< V, E, G > const &);
	BackgroundToFirstClassEdge< V, E, G > & operator = (BackgroundToFirstClassEdge< V, E, G > const & );
};

template < typename V, typename E, typename G >
class AdditionalBackgroundNodesInteractionGraph : public G
{
public:
	typedef typename std::vector< BackgroundNode< V, E, G > * >                           BackgroundNodeVector;
	typedef typename std::list< BackgroundToFirstClassEdge< V, E, G > * >                 BackgroundEdgeList;
	typedef typename std::list< BackgroundToFirstClassEdge< V, E, G > * >::iterator       BackgroundEdgeListIter;
	typedef typename std::list< BackgroundToFirstClassEdge< V, E, G > * >::const_iterator BackgroundEdgeListConstIter;

public:
	AdditionalBackgroundNodesInteractionGraph( int num_nodes );
	virtual ~AdditionalBackgroundNodesInteractionGraph();

	friend class BackgroundToFirstClassEdge< V, E, G >;

	virtual unsigned int getTotalMemoryUsage() const;
	virtual unsigned int count_dynamic_memory() const;

protected:

	void drop_background_edge( BackgroundEdgeListIter iter);

	//Factory methods for background nodes and edges
	virtual BackgroundNode< V, E, G > *
	create_background_node( int bg_node_index ) = 0;

	virtual BackgroundToFirstClassEdge< V, E, G > *
	create_background_edge( int fc_node_index, int bg_node_index ) = 0;

	//call at most once
	void set_num_background_nodes( int num_bg_nodes );

	inline
	int get_num_background_nodes() const
	{	return num_bg_nodes_; }

	void add_background_edge( int first_class_node_index, int bg_node_index );

	BackgroundToFirstClassEdge< V, E, G > *
	find_background_edge(
		int first_class_node_index,
		int bg_node_index
	) const;

	inline
	int get_num_bg_edges() const
	{	return bg_to_fc_edge_list_.size(); }

	inline
	BackgroundNode< V, E, G > * get_background_node( int index ) const
	{	assert( index > 0 && index <= num_bg_nodes_ );
		return bg_nodes_[ index ];
	}

	void
	prepare_for_simulated_annealing();

private:
	int num_bg_nodes_;
	BackgroundNodeVector bg_nodes_;
	BackgroundEdgeList bg_to_fc_edge_list_;

	mutable BackgroundToFirstClassEdge< V, E, G > * focused_bg_edge_;

	//no default constructor, uncopyable
	AdditionalBackgroundNodesInteractionGraph();
	AdditionalBackgroundNodesInteractionGraph(
		AdditionalBackgroundNodesInteractionGraph< V, E, G > const & );
	AdditionalBackgroundNodesInteractionGraph< V, E, G > & operator =
		(AdditionalBackgroundNodesInteractionGraph< V, E, G > const & );
};


//----------------------------------------------------------------------------//
//------------------------- First Class Node Class ---------------------------//
//----------------------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin FirstClassNode::FirstClassNode
///
/// @brief
/// FirstClassNode constructor
///
/// @detailed
///
/// @param
/// owner - [in] - the owning interaction graph
/// node_id - [in] - the index for this node amongst its owners set
/// num_states - [in] - the number of states for this node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
FirstClassNode< V, E, G > ::FirstClassNode
(
	G * owner,
	int node_id,
	int num_states
) :
	V( owner, node_id, num_states ),
	num_edges_to_bg_nodes_(0),
	bg_edge_vector_up_to_date_(false)
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin FirstClassNode::~FirstClassNode
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
FirstClassNode< V, E, G > ::~FirstClassNode()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin FirstClassNode::add_background_edge
///
/// @brief
/// adds a BackgroundToFirstClassEdge to the node's list of such edges
/// and returns an iterator to that list position.
///
/// @detailed
/// edge addition invalidates the bg_edge_vector_up_to_date_ flag
///
/// @param
/// edge_ptr - [in] - a pointer to the BG2FCEdge
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
typename std::list< BackgroundToFirstClassEdge< V, E, G > * >::iterator
FirstClassNode< V, E, G >::add_background_edge(
	BackgroundToFirstClassEdge< V, E, G > * edge_ptr
)
{
   ++num_edges_to_bg_nodes_;
   bg_edge_vector_up_to_date_ = false;
	return bg_edge_list_.insert( bg_edge_list_.end(), edge_ptr );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin FirstClassNode::drop_background_edge
///
/// @brief
/// removes an edge from the node's BackgroundToFirstClassEdge list
///
/// @detailed
/// invalidates bg_edge_vector_up_to_date_ flag
///
/// @param
/// edge - [in] - the iterator to the edge; in order to guarantee constant
///   time edge deletion, each edge keeps iterators to its position in the
///   edge lists of the nodes it's incident upon.  It hands these iterators
///   back to the nodes when it wants to delete itself.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
FirstClassNode< V, E, G >::drop_background_edge
(
	BackgroundEdgeListIter edge
)
{
	--num_edges_to_bg_nodes_;
	bg_edge_list_.erase( edge );
	bg_edge_vector_up_to_date_ = false;
}

template < typename V, typename E, typename G >
unsigned int
FirstClassNode< V, E, G >::count_dynamic_memory() const
{
	//std::cout << "calling FirstClassNode< V, E, G >::count_dynamic_memory()" << std::endl;
	unsigned int total_memory = V::count_dynamic_memory();
	total_memory += 4 * bg_edge_list_.size() * sizeof( BackgroundToFirstClassEdge< V, E, G > * );
	total_memory += bg_edge_vector_.size() * sizeof( BackgroundToFirstClassEdge< V, E, G > * );
	total_memory += adjacent_bg_node_indices_.size() * sizeof ( int );
	total_memory += bg_node_vector_.size() * sizeof( BackgroundNode< V, E, G > * );
	return total_memory;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin FirstClassNode::update_bg_edge_vector
///
/// @brief
/// updates the vector of pointers to background edges to correspond to the
/// list of background edges.  Informs its incident edges of their position
/// in its vector; when an edge invokes a method on a vertex, it can tell that
/// vertex which edge it is in the eyes of that vertex.  e.g.
/// "I'm your third edge."
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
FirstClassNode< V, E, G >::update_bg_edge_vector()
{
	bg_edge_vector_.resize( num_edges_to_bg_nodes_ + 1);
	adjacent_bg_node_indices_.resize( num_edges_to_bg_nodes_ + 1);
	bg_node_vector_.resize(num_edges_to_bg_nodes_ + 1);

	BackgroundEdgeVectorIter position1 = bg_edge_vector_.begin();
	++position1;

	std::copy( bg_edge_list_.begin(), bg_edge_list_.end(),
		position1 );

	for (int ii = 1; ii <= num_edges_to_bg_nodes_; ++ii)
	{
		bg_edge_vector_[ii]->set_pos_in_node_edgevector( this, ii );
	   adjacent_bg_node_indices_[ ii ] =
	   	bg_edge_vector_[ii]->get_other_ind( this );

	   bg_node_vector_[ ii ] =
	   	bg_edge_vector_[ii]->get_other_node( this );
	}

	bg_edge_vector_up_to_date_ = true;
	return;
}

//----------------------------------------------------------------------------//
//------------------------- First Class Edge Class ---------------------------//
//----------------------------------------------------------------------------//


////////////////////////////////////////////////////////////////////////////////
/// @begin FirstClassEdge::FirstClassEdge
///
/// @brief
/// main constructor.
///
/// @detailed
/// This Class is almost empty; a first class edge connects two first class
/// vertices, and is ignorant of the existence of both second class nodes
/// and second-to-first-class edges.  I cannot think of any data that a first
/// class edge should hold that is not held in a PrecomputedPairEnergiesEdge.
/// Nor can I think of any methods that the edge should implement.  However,
/// I include the class in the inheritance heirarchy just in case I come up
/// with something later.
///
/// @param
/// owner - [in] - the owning interaction graph
/// first_node_index - [in] - the index of the lower-indexed first class node
/// second_node_index - [in] - the index of the higher-indexed first class node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
FirstClassEdge< V, E, G >::FirstClassEdge(
	G* owner,
	int first_node_ind,
	int second_node_ind
) : E ( owner, first_node_ind, second_node_ind )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin FirstClassEdge::~FirstClassEdge
///
/// @brief
/// virtual destructor
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
FirstClassEdge< V, E, G >::~FirstClassEdge() {}

//----------------------------------------------------------------------------//
//------------------- Second Class (Background) Node Class -------------------//
//----------------------------------------------------------------------------//


////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundNode::BackgroundNode
///
/// @brief
/// main constructor - no default or copy constructors; no operator =.
///
/// @detailed
/// I use "background node" synonymously with "second class node".
/// This kind of node contributes to the energy of the system.  It does not
/// have any assignable states -- the energy it contributes is a function of
/// the states of the first class vertices that surround it.
/// A simulated annealer is unaware of the presence of this kind of node;
/// the owning interaction graph obscures it from view of the annealer.
///
/// @param
/// owner - [in] - the owning graph
/// node_index - [in] - the index of the background node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
BackgroundNode< V, E, G >::BackgroundNode
(
	AdditionalBackgroundNodesInteractionGraph< V, E, G > * owner,
	int node_index
) :
	node_index_( node_index ),
	num_incident_edges_( 0 ),
	edge_vector_up_to_date_( false ),
	owner_( owner )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundNode::~BackgroundNode
///
/// @brief
/// virtual destructor
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
BackgroundNode< V, E, G >::~BackgroundNode() {}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundNode::add_edge
///
/// @brief
/// adds a BackgroundToFirstClassEdge to the edge list for this node.  Returns
/// an iterator to the new list element.
///
/// @detailed
/// Invalidates edge_vector_up_to_date_ flag
///
/// @param
/// edge_ptr - [in] - pointer to the edge being added
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
typename std::list< BackgroundToFirstClassEdge< V, E, G > * >::iterator
BackgroundNode< V, E, G > ::add_edge(
	BackgroundToFirstClassEdge< V, E, G > * edge_ptr
)
{
	++num_incident_edges_;
	edge_vector_up_to_date_ = false;
	return edge_list_.insert( edge_list_.end(),  edge_ptr );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundNode::drop_edge
///
/// @brief
/// removes an edge from the nodes edge list in constant time
///
/// @detailed
/// Invalidates edge_vector_up_to_date_ flag
///
/// @param
/// edge - [in] - the iterator to the edge's position in node's edge list
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
BackgroundNode< V, E, G >::drop_edge
(
	BackgroundEdgeListIter edge
)
{
	--num_incident_edges_;
	edge_vector_up_to_date_ = false;
	edge_list_.erase( edge );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundNode::update_edge_vector
///
/// @brief
/// vectorizes the edge list representation; informs each of the incident
/// BackgroundToFirstClassEdges of their position in the node's edge vector.
///
/// @detailed
/// sets edge_vector_up_to_date_ flag to true
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void BackgroundNode< V, E, G >::update_edge_vector()
{
	edge_vector_.resize( num_incident_edges_ + 1);
	adjacent_first_class_node_indices_.resize( num_incident_edges_ + 1);
	adjacent_nodes_.resize(num_incident_edges_ + 1);

	BackgroundEdgeVectorIter position1 = edge_vector_.begin();
	++position1;

	std::copy( edge_list_.begin(), edge_list_.end(),
		position1 );

	for (int ii = 1; ii <= num_incident_edges_; ++ii)
	{
		edge_vector_[ii]->set_pos_in_node_edgevector( this, ii );
	   adjacent_first_class_node_indices_[ ii ] =
	   	edge_vector_[ii]->get_other_ind( this );

	   adjacent_nodes_[ ii ] =
	   	edge_vector_[ii]->get_other_node( this );
	}

	edge_vector_up_to_date_ = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundNode::find_edge
///
/// @brief
/// Linear time edge lookup function
///
/// @detailed
///
/// @param
/// fc_node_index - [in] - the index of the first class node that is on the
///   other end of the sought edge
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
BackgroundToFirstClassEdge< V, E, G > *
BackgroundNode< V, E, G > ::find_edge( int fc_node_index )
{
	for (BackgroundEdgeListIter iter = edge_list_.begin();
		iter != edge_list_.end(); ++iter )
	{
		if ((*iter)->same_edge( fc_node_index, node_index_ ))
			return *iter;
	}
	return 0;
}


template < typename V, typename E, typename G >
unsigned int
BackgroundNode< V, E, G >::count_dynamic_memory() const
{
	//std::cout << "calling BackgroundNode< V, E, G >::count_dynamic_memory()" << std::endl;
	unsigned int total_memory = 0;
	total_memory += adjacent_first_class_node_indices_.size() * sizeof ( int );
	total_memory += adjacent_nodes_.size() * sizeof ( FirstClassNode< V, E, G > * );
	total_memory += 4 * edge_list_.size() * sizeof ( BackgroundToFirstClassEdge< V, E, G > * );
	total_memory += edge_vector_.size() * sizeof ( BackgroundToFirstClassEdge< V, E, G > * );

	return 0;
}


//----------------------------------------------------------------------------//
//----------------- Second Class To First Class Edge Class -------------------//
//----------------------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundToFirstClassEdge::BackgroundToFirstClassEdge
///
/// @brief
/// main constructor - no default or copy constructors; no operator =.
///
/// @detailed
/// This class of edge connects first class and second class nodes.  This class
/// of edge is very unlikely to hold the same kind of data in a (concrete)
/// derived class.  E.g. the SASABackgroundEdge holds pre-computed rotamer
/// sphere overlaps for each rotamer on a SASANode and the single (fixed)
/// rotamer on a background residue.  The first class edge does not attempt
/// to store precomputed sphere overlaps as they would be prohibatively
/// expensive.
///
/// @param
/// owner - [in] - the owning graph
/// fc_node_index - [in] - the index of the first class node
/// bg_node_index - [in] - the index of the second class node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
BackgroundToFirstClassEdge< V, E, G >::BackgroundToFirstClassEdge
(
	AdditionalBackgroundNodesInteractionGraph< V, E, G > * owner,
	int fc_node_index,
	int bg_node_index
) :
	first_class_node_index_( fc_node_index ),
	background_node_index_( bg_node_index),
	owner_( owner )
{
	first_class_node_ = ( FirstClassNode< V, E, G > *) (owner_->get_node( first_class_node_index_ ));
	background_node_ = owner_->get_background_node( background_node_index_);
	pos_in_fc_nodes_edge_list_ = first_class_node_->add_background_edge(this);
	pos_in_bg_nodes_edge_list_ = background_node_->add_edge(this);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundToFirstClassEdge::~BackgroundToFirstClassEdge
///
/// @brief
/// virtual destructor.  The edge removes itself from the graph by informing
/// the two vertices its incident upon to drop it from their edge lists.
/// Constant time edge removal.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
BackgroundToFirstClassEdge< V, E, G >::~BackgroundToFirstClassEdge()
{
	first_class_node_->drop_background_edge( pos_in_fc_nodes_edge_list_ );
	background_node_->drop_edge( pos_in_bg_nodes_edge_list_ );
	owner_->drop_background_edge( pos_in_owners_edge_list_ );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundToFirstClassEdge::get_other_ind
///
/// @brief
/// returns the index of the second class node
///
/// @detailed
/// A first class vertex may request a BG2FCEdge for the index of the
/// background node the edge connects it to by invoking
/// edge->get_other_ind( this );  The this pointer simply tells the compiler
/// which of the two overloaded get_other_ind() methods to invoke.
///
/// @param
/// (unnamed parameter) - [in] - First Class nodes want information about
///   second class nodes when they refer to "the other node".  The compiler
///   resolves which of the two overloaded methods to invoke.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
int
BackgroundToFirstClassEdge< V, E, G >::get_other_ind( FirstClassNode< V, E, G > * ) const
{ return background_node_index_;}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundToFirstClassEdge::get_other_ind
///
/// @brief
/// returns the index of the first class node
///
/// @detailed
/// A second class vertex may request a BG2FCEdge for the index of the
/// first class node the edge connects it to by invoking
/// edge->get_other_ind( this );  The this pointer simply tells the compiler
/// which of the two overloaded get_other_ind() methods to invoke.
///
/// @param
/// (unnamed parameter) - [in] - second class nodes want information about
///   first class nodes when they refer to "the other node".  The compiler
///   resolves which of the two overloaded methods to invoke.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
int
BackgroundToFirstClassEdge< V, E, G >::get_other_ind(  BackgroundNode< V, E, G > * ) const
{	return first_class_node_index_;}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundToFirstClassEdge::get_other_node
///
/// @brief
/// returns a pointer to the second class node
///
/// @detailed
/// unnamed parameter is usually the this pointer where the method is invoked
/// inside a method of a first class node.  the pointer type must be resolved
/// at compile time.  Same ideas here as in get_other_ind()
///
/// @param
/// (unnamed parameter) - [in] - First Class nodes want information about
///   second class nodes when they refer to "the other node".  The compiler
///   resolves which of the two overloaded methods to invoke.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
BackgroundNode< V, E, G > *
BackgroundToFirstClassEdge< V, E, G >::get_other_node(
	FirstClassNode< V, E, G > *
) const
{
	return background_node_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundToFirstClassEdge::get_other_node
///
/// @brief
/// returns a pointer to the first class node
///
/// @detailed
/// unnamed parameter is usually the this pointer where the method is invoked
/// inside a method of a second class node.  the pointer type must be resolved
/// at compile time.  Same ideas here as in get_other_ind()
///
/// @param
/// (unnamed parameter) - [in] - second class nodes want information about
///   first class nodes when they refer to "the other node".  The compiler
///   resolves which of the two overloaded methods to invoke.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
FirstClassNode< V, E, G > *
BackgroundToFirstClassEdge< V, E, G > ::get_other_node(
	BackgroundNode< V, E, G > *
) const
{
	return first_class_node_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundToFirstClassEdge::set_pos_in_owners_list
///
/// @brief
/// stores the iterator to this edge in the owning graph's list of
/// background-to-first-class edges.
///
/// @detailed
/// required for constant time edge deletion
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
BackgroundToFirstClassEdge< V, E, G >::set_pos_in_owners_list(
	BackgroundEdgeListIter iter
)
{
	pos_in_owners_edge_list_ = iter;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundToFirstClassEdge::set_pos_in_node_edgevector
///
/// @brief
/// stores the index of this edge in its first class node's edge vector
///
/// @detailed
/// again, first (unnamed) parameter is the this pointer where the method has
/// been invoked inside a method of the first class node.
///
/// @param
/// (unnamed parameter) - [in] - pointer identifying the type of the node that
///   invoked this overloaded method.
/// pos - [in] - the position of this node in the edge vector of the first
///   class node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
BackgroundToFirstClassEdge< V, E, G >::set_pos_in_node_edgevector
(
	FirstClassNode< V, E, G > *,
	int pos
)
{
	pos_in_fc_edge_vector_ = pos;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundToFirstClassEdge::set_pos_in_node_edgevector
///
/// @brief
/// stores the index of this edge in its second class node's edge vector
///
/// @detailed
/// again, first (unnamed) parameter is the this pointer where the method has
/// been invoked inside a method of the second class node.
///
/// @param
/// (unnamed parameter) - [in] - pointer identifying the type of the node that
///   invoked this overloaded method.
/// pos - [in] - the position of this node in the edge vector of the second
///   class node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
BackgroundToFirstClassEdge< V, E, G >::set_pos_in_node_edgevector
(
	BackgroundNode< V, E, G > *,
	int pos
)
{
	pos_in_bg_edge_vector_ = pos;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin BackgroundToFirstClassEdge::same_edge
///
/// @brief
/// returns true if this node is incident upon the two input vertex indices
///
/// @detailed
///
/// @param
/// fc_node_index - [in] - the index of the first class node
/// bg_node_index - [in] - the index of the second class node
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// This graph setup will not support multigraphs: graphs that include multiple
/// edges to the same vertices.  In my original dynamic programming algorithm
/// I didn't want to support multigraphs either; but in the adaptive dynamic
/// programming implementation, multiple copies of "the same" edge ended up
/// saving time -- though maybe not memory.  So, this framework here will not
/// readily support adaptive dynamic programming; but that's ok.
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
bool
BackgroundToFirstClassEdge< V, E, G >::
same_edge( int fc_node_index, int bg_node_index ) const
{
	return (first_class_node_index_ == fc_node_index &&
		background_node_index_ == bg_node_index);
}

template < typename V, typename E, typename G >
unsigned int
BackgroundToFirstClassEdge< V, E, G >::
count_dynamic_memory() const
{
	//std::cout << "calling BackgroundToFirstClassEdge< V, E, G >::count_dynamic_memory()" << std::endl;
	return 0;
}


//----------------------------------------------------------------------------//
//----------- Additional Background Nodes Interaction Graph Class ------------//
//----------------------------------------------------------------------------//


////////////////////////////////////////////////////////////////////////////////
/// @begin AdditionalBackgroundNodesInteractionGraph::
///    AdditionalBackgroundNodesInteractionGraph
///
/// @brief
/// main constructor; no default or copy constructors; no operator =
///
/// @detailed
///
/// @param
/// num_nodes - [in] - the number of first class nodes
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
AdditionalBackgroundNodesInteractionGraph< V, E, G >::
AdditionalBackgroundNodesInteractionGraph( int num_nodes )
:
	G( num_nodes ),
	num_bg_nodes_( -1 ),
	focused_bg_edge_( 0 )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin AdditionalBackgroundNodesInteractionGraph::
///    ~AdditionalBackgroundNodesInteractionGraph
///
/// @brief
/// virtual destructor
///
/// @detailed
/// deallocates each BackgroundToFirstClassEdge. Then deallocates each background
/// node.  Order is important.  The edges assume that its vertices still exist
/// at the time it is removed.  This destructor enforces that.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
AdditionalBackgroundNodesInteractionGraph< V, E, G >::
~AdditionalBackgroundNodesInteractionGraph()
{
	for ( BackgroundEdgeListIter iter = bg_to_fc_edge_list_.begin(); iter != bg_to_fc_edge_list_.end();
		//note: no increment statement here
	)
	{
		BackgroundEdgeListIter next_iter = iter;
		next_iter++;
		//edges delete themselves from this list, invalidating iterators, so
		//get the next iterator before calling delete
		delete (*iter);
		iter = next_iter;
	}

	for (int ii = 1; ii <= num_bg_nodes_; ++ii)
	{
		delete bg_nodes_[ ii ];
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin AdditionalBackgroundNodesInteractionGraph::drop_background_edge
///
/// @brief
/// Constant time edge removal.
///
/// @detailed
///
/// @param
/// iter - [in] - the iterator to the position in the graph's edge list
///   for the edge that is removing itself.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void AdditionalBackgroundNodesInteractionGraph< V, E, G > ::drop_background_edge
(
	BackgroundEdgeListIter iter
)
{
	bg_to_fc_edge_list_.erase( iter );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin AdditionalBackgroundNodesInteractionGraph::set_num_background_nodes
///
/// @brief
/// sets the number of background nodes in the graph.  Should be called
/// no more than once.  Some problem instances do not require background nodes.
///
/// @detailed
/// Allocates the background nodes using a factor method.  Their indices start
/// from 1.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
AdditionalBackgroundNodesInteractionGraph< V, E, G >::
set_num_background_nodes( int num_bg_nodes )
{
	assert( num_bg_nodes_ == -1 && num_bg_nodes >= 0); //call this method at most once
	num_bg_nodes_ = num_bg_nodes;

	if ( num_bg_nodes_ == 0 ) return;

	bg_nodes_.resize( num_bg_nodes_ + 1);
	for (int ii = 1; ii <= num_bg_nodes_; ++ii)
	{
		bg_nodes_[ ii ] = create_background_node( ii );
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin AdditionalBackgroundNodesInteractionGraph::add_background_edge
///
/// @brief
/// adds a BackgroundToFirstClassEdge to the graph and performs the
/// requist bookkeepking.
///
/// @detailed
///
/// @param
/// first_class_node_index - [in] - the index of the first class node
/// bg_node_index - [in] - the index of the second class node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
AdditionalBackgroundNodesInteractionGraph< V, E, G >::
add_background_edge
(
	int first_class_node_index,
	int bg_node_index
)
{
	BackgroundToFirstClassEdge< V, E, G > * new_edge =
		create_background_edge(first_class_node_index, bg_node_index);

	bg_to_fc_edge_list_.push_front( new_edge );
	new_edge->set_pos_in_owners_list( bg_to_fc_edge_list_.begin() );
	focused_bg_edge_ = new_edge;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin AdditionalBackgroundNodesInteractionGraph::find_background_edge
///
/// @brief
/// returns a pointer to the background edge, identified by the indices of
/// the first and second class nodes it connects.  Returns the null pointer if
/// the edge does not exist.  Possibly linear time operation (in the number of
/// BackgroundToFirstClassEdges).
///
/// @detailed
/// The graph keeps a pointer to a "focused edge" so that repeated requests for
/// the same edge take constant time.
///
/// @param
/// first_class_node_index - [in] - the index of the first class node
/// bg_node_index - [in] - the index of the second class node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
BackgroundToFirstClassEdge< V, E, G > *
AdditionalBackgroundNodesInteractionGraph< V, E, G >::
find_background_edge
(
	int first_class_node_index,
	int bg_node_index
) const
{
	if (focused_bg_edge_ != 0 &&
		! focused_bg_edge_->same_edge( first_class_node_index, bg_node_index ) )
	{
		focused_bg_edge_ = bg_nodes_[ bg_node_index ]->find_edge( first_class_node_index );
	}
	return focused_bg_edge_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin AdditionalBackgroundNodesInteractionGraph::
///   prepare_for_simulated_annealing
///
/// @brief
/// invokes prepare_for_simulated_annealing on each of the
/// BackgroundToFirstClassEdges and then invokes prepare_for_simulated_annealing
/// on each of the BackgroundNodes.
///
/// @detailed
/// A BackgroundToFirstClassEdges may decide to delete itself before sim
/// annealing begins.  Since the second class node will likely update its
/// edge vector in its (virtual) call to prepare_for_sim_annealing, any edges
/// that should be removed should be removed before the second class nodes
/// update their edge vectors.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// does not invoke InteractionGraphBase::prepare_for_simulated_annealing()
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
AdditionalBackgroundNodesInteractionGraph< V, E, G >::
prepare_for_simulated_annealing()
{
	for ( BackgroundEdgeListIter iter
		= bg_to_fc_edge_list_.begin(); iter != bg_to_fc_edge_list_.end();
		//note: no increment statement here
	)
	{
		BackgroundEdgeListIter next_iter = iter;
		next_iter++;
		//edges sometimes delete themselves, invalidating iterators, so
		//get the next iterator before calling prepare_for_simulated_annealing
		(*iter)->prepare_for_simulated_annealing();
		iter = next_iter;
	}

	for (int ii = 1; ii <= num_bg_nodes_; ++ii)
	{
		bg_nodes_[ ii ]->prepare_for_simulated_annealing();
	}
	return;
}

template < typename V, typename E, typename G >
unsigned int
AdditionalBackgroundNodesInteractionGraph< V, E, G >::
getTotalMemoryUsage() const
{
	//std::cout << "calling AdditionalBackgroundNodesInteractionGraph< V, E, G >::getTotalMemoryUsage() const" << std::endl;

	unsigned int total_memory = G::getTotalMemoryUsage();
	for (int ii = 1; ii <= get_num_background_nodes(); ++ii)
	{
		total_memory += bg_nodes_[ ii ]->count_dynamic_memory();
		total_memory += bg_nodes_[ ii ]->count_static_memory();
	}
	for ( BackgroundEdgeListConstIter iter = bg_to_fc_edge_list_.begin();
			iter != bg_to_fc_edge_list_.end(); ++iter )
	{
		total_memory += (*iter)->count_dynamic_memory();
		total_memory += (*iter)->count_static_memory();
	}

	return total_memory;
}

template < typename V, typename E, typename G >
unsigned int
AdditionalBackgroundNodesInteractionGraph< V, E, G >::
count_dynamic_memory() const
{
	unsigned int total_memory = G::count_dynamic_memory();
	total_memory += bg_nodes_.size() * sizeof( BackgroundNode< V, E, G > * );
	total_memory += 4 * bg_to_fc_edge_list_.size() * sizeof( BackgroundToFirstClassEdge< V, E, G > * );
	return total_memory;
}


}//end namespace pack

#endif

