// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 18466 $
//  $Date: 2007-11-16 23:05:47 +0200 (Fri, 16 Nov 2007) $
//  $Author: yab $


#ifndef INCLUDED_DesignMap_H
#define INCLUDED_DesignMap_H


// Rosetta headers
#include "aaproperties_pack.h"
//#include "misc.h"
#include "param.h"

// utility headers
#include <utility/vector1.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray1D.hh>

// C++ headers
#include <cstdlib>
#include <cstdio>
#include <iostream>



class DesignMap{
 private:
  //the private class of Residue_To_Vary
  //which keeps the available residues to design
  //as well as the repack_residue for each amino acid position
  class  DesignMapPerResidue{
  public:
    //the default constructor

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin DesignMapPerResidue()
///
/// @brief
///
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
    inline
		DesignMapPerResidue():
			repack_residue(false),
      number_of_available_aa(0),
      the_native_aa(0){
			using namespace param;
      int max_aa = MAX_AA()();
      residues.assign(max_aa, false);       //initialize the vector: in default, the position is fixed
			allowed_variant_types.dimension(aaproperties_pack::number_aav_type);
			allowed_variant_types=false;
			allowed_variant_types(aaproperties_pack::aav_base) = true;
			number_of_available_aav=1;
    }

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin  DesignMapPerResidue(int native_aa)
///
/// @brief
/// where the native amino acids is from aan(this_postion)
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
    inline
      DesignMapPerResidue(int native_aa):
      repack_residue(false),
      number_of_available_aa(0),
      number_of_available_aav(0),
      the_native_aa(native_aa){    // initialize the native aa
				using namespace param;
				int max_aa = MAX_AA()();
				residues.assign(max_aa, false);
				allowed_variant_types.dimension(aaproperties_pack::number_aav_type);
				allowed_variant_types=false;
				allowed_variant_types(aaproperties_pack::aav_base) = true;
				number_of_available_aav=1;
			}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin DesignMapPerResidue(const DesignMapPerResidue& rec)
///
/// @brief
/// the copy constructure
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
    inline
      DesignMapPerResidue(const DesignMapPerResidue& rec):
      repack_residue(rec.repack_residue),
      number_of_available_aa(rec.number_of_available_aa),
      number_of_available_aav(rec.number_of_available_aav),
      the_native_aa(rec.the_native_aa){
				using namespace param;
				for(int i=1; i<=MAX_AA()(); ++i){
					residues.push_back(rec.residues[i]); // Copy the rec.residues' value.
				}
				allowed_variant_types.dimension(aaproperties_pack::number_aav_type);
				for(int i=1; i<=aaproperties_pack::number_aav_type; ++i){
					allowed_variant_types(i)=rec.allowed_variant_types(i);
				}
			}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin  ~DesignMapPerResidue()
///
/// @brief
/// Destructor
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
	inline ~DesignMapPerResidue(){}

    //Set the native amino acide from aan(), which
    //loop each residue pistions and return the native amino acid.
    void set_native_aa(int native_aa);

    //enable an given residues to be available for redesign
    void set_residue(int the_aa);

    //disable this position and if repack_residue is true, set the native aa be true
    void disable_residue(int the_aa);

    //disable a particular variant type at this position
    void enable_variant_type_per_residue(int the_variant_type);

    //disable a particular variant type at this position
    void disable_variant_type_per_residue(int the_variant_type);

    //check whether a particular variant type is allowed at this position
    bool variant_type_is_allowed_per_residue(int the_variant_type) const;

    //fix the sidechain at the position
    void fix_sidechain_of_residue();

    //require the base rotamer at this position (eg. no rotameric water)
    void require_base_rotamer_per_residue();

    //return whether a given aa is available for redesign
    bool design_map_per_residue(int const the_aa) const;

    //return whether the residue is to be fixed
    bool repack_per_residue() const;

    //return whether any variants are allowed for this residue
    bool use_base_rotamer_per_residue() const;

    //Return the (stored) native amino acid
    inline int get_native_aa() const { return the_native_aa; };

		// return the number of allowed aa at this position
    inline int num_allowed_aa_per_residue() const { return number_of_available_aa; };

		// return the number of allowed aav at this position
    inline int num_allowed_aav_per_residue() const { return number_of_available_aav; };

		// return a random choice from the allowed amino acid types
		int random_allowed_aa() const;

  private:
    utility::vector1<bool> residues;
    bool repack_residue;
		FArray1D_bool allowed_variant_types;
    int number_of_available_aa;
    int number_of_available_aav;
    int the_native_aa;
  };

  //==========================================================
  // End of the private class DesignMapPerResidue
  //==========================================================

 public:

////////////////////////////////////////////////////////////////////////////////
/// @begin   DesignMap(int nResidues, FArray1D_int const & res)
///
/// @brief
/// DesignMap constructor, native amino acids array is required
///
/// @detailed
///
/// @param[in]   nResidues
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors yiliu
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

   inline
	DesignMap(int nResidues, FArray1DB_int const & res)
	:
		map( nResidues ),
		default_variant_type( aaproperties_pack::number_aav_type, false )
	{
		for(int i=1; i<=nResidues; i++){
			map[i].set_native_aa(res(i));
		}
		default_variant_type(aaproperties_pack::aav_base) = true;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin   DesignMap(int nResidues, utility::vector1<int>& native_residues)
///
/// @brief
/// DesignMap constructor, native amino acids array is required
///
/// @detailed
///
/// @param[in]   nResidues
/// @param[in]   native_residues
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors yiliu
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	inline
	DesignMap( int nResidues, utility::vector1<int> const & native_residues )
	:
		map( nResidues ),
		default_variant_type( aaproperties_pack::number_aav_type, false )
	{
		for(int i=1; i<=nResidues; ++i){
			map[i].set_native_aa(native_residues[i]);
		}
		default_variant_type(aaproperties_pack::aav_base) = true;
	}

///////////////////////////////////////////////////////////////////////////////
/// @begin   DesignMap(const DesignMap& old)
///
/// @brief
/// DesignMap copy constructor
///
/// @detailed
///
/// @param[in]   old
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors yiliu
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	inline
	DesignMap( DesignMap const & old) :
		map( old.map ),
		default_variant_type( old.default_variant_type )
	{
	}

	inline
	DesignMap &
	operator = ( DesignMap const & right)
	{
		if (this== &right) return *this;
		int size=right.map.size();
		if( map.size() > 0 )
		{
			map.clear();
		}
		if ( right.map.size() > 0 )
		{
			map.reserve( right.map.size() ); //avoid O(n lg n) re-allocations
		}
		for( int i=1; i <= size; ++i)
		{
			map.push_back(right.map[i]);
		}
		default_variant_type.dimension(right.default_variant_type.size() );
		default_variant_type = right.default_variant_type;
		return *this;
	}


///////////////////////////////////////////////////////////////////////////////
/// @begin  ~DesignMap
///
/// @brief
/// DesignMap destructor
///
/// @detailed
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors yiliu
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	inline
  ~DesignMap(){}

  //===================================================================
  //set the design map for a given residue to match a different residue
  //===================================================================
  void copy_res(const int source_residue, const int destination_residue);

  //===============================================================
  //set the design map for a given residue and a given amino acid
  //===============================================================
  void set(const int the_residue, const int the_aa);


  //=================================================
  //set the design map by giving a design_matrix
  //=================================================
  void set(const int nResidues, FArray2D<bool>& design_matrix);


  //================================================
  //disable the aa in the residue position
  //================================================
  void disable(const int the_residue, const int the_aa);

  //disable all
  void disable_all();


  //====================================
  //fix sidechain and variant at all positions
  //====================================
  void fix_completely();

  //====================================
  //fix sidechain and variant at the position
  //====================================
  void fix_completely(const int the_residue);

  //====================================
  //fix sidechain at the position
  //====================================
  void fix_sidechain_allowing_variants(const int the_residue);

  //===================================================
  //only allow the base rotamer (ie. disallow all variants) at all seqpos
  //===================================================
  void require_base_rotamer();

  //===================================================
  //only allow the base rotamer (ie. disallow all variants) at a particular seqpos
  //===================================================
  void require_base_rotamer(const int the_residue);

  //==================================
  //return design_map -- sidechain
  //==================================
  bool get(const int the_residue, const int the_aa) const;

  //====================================================================
  // return whether any rotamers exist for given aa
  // (either heavy atoms move or variants)
  //====================================================================
  bool has_rotamers(int const the_residue, const int the_aa) const;

  //==================================
  //return repack_residue -- protamers
  //==================================
  bool repack_residue(int const the_residue) const;

  //==================================
  //return use_base_rotamer for the specified residue
  //==================================
  bool use_base_rotamer(int const the_residue) const;

  void print_dm() const;
  void print_dm(int theRES) const;

  //==================================
  //Covert for namespace members
  //==================================
  void convert_to_dm(FArray1D_bool const & protamers, FArray2D_bool const & sidechain);
  void convert_to_ns(FArray1D_bool & protamers);

  //=========================================
  // set repack residue in special situations
  //=========================================
  void all_true_repack_residue();

  //==================================
  // allow a variant type at all positions
  //==================================
  void allow_variant_type( int const variant_type);

  //==================================
  // disallow a variant type at all positions
  //==================================
  void disallow_variant_type( int const variant_type);

  //==================================
  // allow a variant type at a specific residue
  //==================================
  void allow_variant_type_at_residue( int const variant_type, int const seqpos);

  //==================================
  // disallow a variant type at a specific residue
  //==================================
  void disallow_variant_type_at_residue(const int variant_type, const int seqpos);

  //==================================
  // disallow a variant type at a specific residue
  //==================================
  bool variant_type_is_allowed( int const variant_type, int const seqpos) const;

  //==================================
  // set the flags which will apply in the general case
  //==================================
  void set_variant_defaults( FArray1DB_bool const & in_default_variant_array );

  //==================================
  // set a specified residue to use the default allowed variants
  //==================================
  void apply_variant_defaults(int const the_residue);

  //==================================
  // set all residues to use the default allowed variants
  //==================================
  void apply_variant_defaults();

  //==================================
  // return the number of allowed aa at the specified seqpos
  //==================================
	int num_allowed_aa(int const the_residue) const;

  //==================================
  // return the number of allowed variant types at the specified seqpos
  //==================================
	int num_allowed_aav(int const the_residue) const;

  //==================================
  // return a random choice from the allowed amino acid types at seqpos
  //==================================
	int random_allowed_aa( int const seqpos ) const;

	int size() const { return map.size(); };

 private:
	utility::vector1<DesignMapPerResidue> map;
	FArray1D_bool default_variant_type;

};


#endif //INCLUDE_DesignMap_H
