// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

#include "DirectedSimAnnealer.h"
#include "InteractionGraphBase.h"
#include "PDInteractionGraph.h"
#include "ligand_ns.h"
#include "misc.h"
#include "random_numbers.h"
#include "param.h"
#include "RotamerSet.h"
//#include "pKa_mode.h" // !!! not supporting pKa_mode initially

#include <ObjexxFCL/Fmath.hh>

#include <iostream>
#include <limits>


#define REPORT1(x1)                              #x1 " " << x1
#define REPORT2(x1,x2)                           #x1 " " << x1 << "\t" #x2 " " << x2
#define REPORT3(x1,x2,x3)                        #x1 " " << x1 << "\t" #x2 " " << x2 << "\t" #x3 " " << x3
#define REPORT4(x1,x2,x3,x4)                     #x1 " " << x1 << "\t" #x2 " " << x2 << "\t" #x3 " " << x3 << "\t" #x4 " " << x4
#define REPORT5(x1,x2,x3,x4,x5)                  #x1 " " << x1 << "\t" #x2 " " << x2 << "\t" #x3 " " << x3 << "\t" #x4 " " << x4 << "\t" #x5 " " << x5
#define REPORT6(x1,x2,x3,x4,x5,x6)               #x1 " " << x1 << "\t" #x2 " " << x2 << "\t" #x3 " " << x3 << "\t" #x4 " " << x4 << "\t" #x5 " " << x5 << "\t" #x6 " " << x6
#define REPORT7(x1,x2,x3,x4,x5,x6,x7)            #x1 " " << x1 << "\t" #x2 " " << x2 << "\t" #x3 " " << x3 << "\t" #x4 " " << x4 << "\t" #x5 " " << x5 << "\t" #x6 " " << x6 << "\t" #x7 " " << x7
#define REPORT8(x1,x2,x3,x4,x5,x6,x7,x8)         #x1 " " << x1 << "\t" #x2 " " << x2 << "\t" #x3 " " << x3 << "\t" #x4 " " << x4 << "\t" #x5 " " << x5 << "\t" #x6 " " << x6 << "\t" #x7 " " << x7 << "\t" #x8 " " << x8


using namespace std;

namespace pack{

  namespace directed_design {

    using namespace std;

    // ____________________ Entry ____________________

    istream& operator>>(istream& in, Entry& e) {

      string s;
      in >> s;

      if( s == "weight" ) {
	e.type = Entry::WEIGHT;
	in >> e.w;
      }
      else if( s == "residue" ) {
	e.type = Entry::RESIDUE;
	in >> e.a;
      }
      else if( s == "respair" ) {
	e.type = Entry::RESPAIR;
	in >> e.a >> e.b;
      }
      else {
	e.type = Entry::UNDEF;
      }

      in.ignore(INT_MAX,'\n');

      return in;

    } // operator>>

    ostream& operator<<(ostream& out, const Entry& e) {

      switch( e.type ) {
      case Entry::WEIGHT:
	return out << "weight " << e.w;
      case Entry::RESIDUE:
	return out << "residue " << e.a;
      case Entry::RESPAIR:
	return out << "respair " << e.a << " " << e.b;
      case Entry::UNDEF:
      default:
	return out << "undef";
      } // switch

    } // operator<<

    // ____________________ File ____________________


    namespace {

      void open(ifstream& fin, const string& filename) {
	fin.close();
	fin.clear();
	fin.open(filename.c_str());
	if( !fin ) {
	}
      } // open

    } // namespace

    File::File() {
    } // File::File

    File::File(const string& filename) {
      ifstream fin;
      open(fin,filename);
      fin >> *this;
    } // File::File




    istream& operator>>(istream& in, File& f) {
      f.vEntries.clear();
      while( in.peek() != EOF ) {
	Entry e;
	in >> e;
	f.vEntries.push_back(e);
      }
      return in;
    } // operator>>

    ostream& operator<<(ostream& out, const File& f) {
      for( size_t i = 0; i < f.vEntries.size(); ++i ) {
	out << f.vEntries[i] << "\n";
      }
      return out;
    } // operator>>


    // ____________________ init_weights ____________________

    int get_molten_res_index( const int res_index, RotamerSet const & rotamer_set ) {
      //assert( false );
      //     for( int ii = 1; ii <= rotamer_set.effective_numres(); ++ii ) {
      //       cout << REPORT4(res_index,ii,rotamer_set.resid_2_moltenres(ii),rotamer_set.effective_numres()) << endl;
      //     } // i
      //cout << __func__ << " " << REPORT2(res_index,rotamer_set.resid_2_moltenres(res_index) ) << endl;
      return rotamer_set.resid_2_moltenres(res_index);
    } // File::get_molten_res_index

    void init_weights( File const &file,
		       int const num_moltenres,
		       RotamerSet const & rotamer_set,
		       FArray2D_float& weights) {

      //cout << "init_weights " << endl;

      // Step 0 - resize weights
      // !!! I would like for this to initialize weights to dim
      // [1..nmr]x[1..nmr] and fill it with 1's
      weights.dimension(num_moltenres,num_moltenres,1);

      // Step 1 - Determine the weight by which to upweight weights
      double weight(0.);
      bool b_found_weight = false;

      for( vector<Entry>::const_iterator i = file.vEntries.begin(); i != file.vEntries.end(); ++i ) {
	const Entry& e = *i;
	if( e.type == Entry::WEIGHT ) {
	  assert( !b_found_weight );
	  b_found_weight = true;
	  weight = e.w;
	}
      } // i
      assert( b_found_weight );

      // Step 2 - For all residues and respairs, set the weight to the appropriate value
      for( vector<Entry>::const_iterator i = file.vEntries.begin(); i != file.vEntries.end(); ++i ) {
	const Entry& e = *i;
	cout << "init_weights:: directed_design::Entry " << e << endl;

	if( e.type == Entry::RESIDUE ) {

	  // Step 2.a - if the entry is a residue, upweight all rows & columns corresponding to that entry
	  const int i = get_molten_res_index(e.a,rotamer_set);

	  if( i == 0 ) {
	    cout << "init_weights:: WARNING - residue " << e.a << " is not being repacked, bias will have no effect" << endl;
	  }

	  if( i != 0 ) {
	    for( int j = 1; j <= num_moltenres ; ++j ) {

	      weights(i,j) = weight;
	      weights(j,i) = weight;

	    } // j

	  }

	}
	else if( e.type == Entry::RESPAIR ) {

	  // Step 2.b - if the entry is a respair, upweight the corresponding matrix entry
	  const int i = get_molten_res_index(e.a,rotamer_set);
	  const int j = get_molten_res_index(e.b,rotamer_set);

	  if( i == 0 ) {
	    cout << "init_weights:: WARNING - residue " << e.a << " is not being repacked, bias will have no effect" << endl;
	  }

	  if( j == 0 ) {
	    cout << "init_weights:: WARNING - residue " << e.b << " is not being repacked, bias will have no effect" << endl;
	  }

	  if( i != 0 && j != 0 ) {
	    weights(i,j) = weight;
	    weights(j,i) = weight;
	  }

	}

      } // i

      for( int i = 1; i <= num_moltenres; ++i ) {
	cout << "init_weights:: weights ";
	for( int j = 1; j <= num_moltenres; ++j ) {
	  cout << weights(i,j) << " ";
	} // j
	cout << endl;
      } // i

    } // init_weights

  } // namespace directed_design

  // ____________________ DirectedSimAnnealer ____________________

  ////////////////////////////////////////////////////////////////////////////////
  /// @begin DirectedSimAnnealer::DirectedSimAnnealer()
  ///
  /// @brief
  /// constructor
  ///
  /// @detailed
  ///
  /// @global_read
  ///
  /// @global_write
  ///
  /// @remarks
  ///
  /// @references
  ///
  /// @authors
  ///
  /// @last_modified

  ////////////////////////////////////////////////////////////////////////////////
  DirectedSimAnnealer::DirectedSimAnnealer( std::vector<int> & rot_to_pack,
					    FArray1D_int & bestrotamer_at_seqpos,
					    float & bestenergy,
					    bool start_with_current, // start simulation with current rotamers
					    pack::InteractionGraphBase * ig,
					    const RotamerSet * p_rotamer_set,
					    FArray1DB_int & current_rot_index,
					    bool calc_rot_freq,
					    FArray1D_float & rot_freq
					    ):
    SimAnnealerBase((int) rot_to_pack.size(),
		    bestrotamer_at_seqpos,
		    bestenergy,
		    start_with_current, // start simulation with current rotamers
		    p_rotamer_set,
		    current_rot_index,
		    calc_rot_freq,
		    rot_freq
		    ),
    ig_(ig),
    rot_to_pack_( rot_to_pack)

  {
    // pass

  } // DirectedSimAnnealer::DirectedSimAnnealer

  DirectedSimAnnealer::DirectedSimAnnealer(FArray1D_int & bestrotamer_at_seqpos,
					   float & bestenergy,
					   bool start_with_current, // start simulation with current rotamers
					   pack::InteractionGraphBase * ig,
					   const RotamerSet * p_rotamer_set,
					   FArray1DB_int & current_rot_index,
					   bool calc_rot_freq,
					   FArray1D_float & rot_freq
					   ):
    SimAnnealerBase((ig->get_num_total_states()),
		    bestrotamer_at_seqpos,
		    bestenergy,
		    start_with_current, // start simulation with current rotamers
		    p_rotamer_set,
		    current_rot_index,
		    calc_rot_freq,
		    rot_freq
		    ),
    ig_(ig),
    rot_to_pack_( ig_->get_num_total_states() )

  {
    for (unsigned int ii =1; ii <= rot_to_pack_.size(); ++ii) {
      rot_to_pack_[ ii-1 ] = ii;
    }
  } // DirectedSimAnnealder::DirectedSimAnnealer

  DirectedSimAnnealer::~DirectedSimAnnealer()
  {
  } // DirectedSimAnnealer::~DirectedSimAnnealer

  ////////////////////////////////////////////////////////////////////////////////
  /// @begin DirectedSimAnnealer::run
  ///
  /// @brief Runs a "directed" sim annealing - a protocol in which
  /// two-body interaction energies between user specified positions
  /// are upweighted during rotamer changes that affect the sequence,
  /// but not during rotamer changes that affect only the
  /// conformation. This causes the rotamer set returned to be biased
  /// towards specific interactions in terms of their sequence
  /// composition, but not in terms of their conformation. The
  /// conformation returned will be no different than if you had just
  /// repacked that sequence. Essentially trades total energy (mostly
  /// reference energy?) for energy involved in specific interactions
  /// (useful?! for ligands, interfaces, hotspots).
  ///
  /// @detailed
  ///
  /// @global_read
  ///
  /// @global_write
  ///
  /// @remarks
  ///
  /// @references
  ///
  /// @authors
  ///
  /// @last_modified
  ////////////////////////////////////////////////////////////////////////////////

  void DirectedSimAnnealer::run() {

    // A value larger than of the possible energy values.
    const float huge_value = numeric_limits<float>::max();

    PDInteractionGraph* pdig = dynamic_cast<PDInteractionGraph*>(ig_);
    assert( pdig );


    const int N = pdig->get_num_nodes();  // number of molten res
    const int N_rot = pdig->get_num_total_states(); // total number of rotamers

    cout << REPORT2(N,N_rot) << endl;

    // the current state and energy
    FArray1D_int state_cur(N,0);
    float E_cur = huge_value; // !!! can be either weighted or unweighted

    // the best weighted state
    FArray1D_int state_best_weighted(N,0);
    float E_best_weighted = huge_value; // !!! always the weighted energy

    // the best unweighted state
    FArray1D_int state_best_unweighted(N,0);
    float E_best_unweighted = huge_value; // !!! always the unweighted energy

    // the state kept
    FArray1D_int state_outer(N,0); // always set to a state which is a minimum of the unweighted energy
    float E_outer = huge_value; // always set to a weighted energy

    // integer maps required

    FArray1D_int index_2_resid(N,0); // maps [1,N] => residue number
    FArray1D_int index_2_rotoffset(N,0); // maps [1,N] => offset for global rotamer index
    FArray1D_int rotid_2_index(N_rot,0); // maps {global rotamer index} => [1,N]
    FArray1D_int rotid_2_state(N_rot,0); // maps {global rotamer index} => [1,get_num_states(index)]

    FArray1D_float loopenergy(maxouteriterations,0.0); // the loopenergy, wtf?

    // Step 0 - initialization

    pdig->prepare_for_simulated_annealing();
    pdig->blanket_assign_state_0();

    for( int resid = 1; resid < p_rotamer_set_->effective_numres(); ++resid ) {
      const int index = p_rotamer_set_->resid_2_moltenres(resid);

      if( index == 0 ) {
	// pass
      }
      else if( 1 <= index && index <= misc::total_residue ) {
	index_2_resid( index ) = resid;
      }
      else if( index - misc::total_residue <= static_cast<int>(ligand::ligand_ptr_vector.size()) ) {
	index_2_resid( index ) = - ( index - misc::total_residue );
      }

      if( index ) {
	//	cout << "DirectedSimAnnealer::run" << " - " << REPORT3(resid,index,index_2_resid(index)) << endl;
      }

    } // resid


    for( int index = 1; index <= N; ++index ) {

      if( index_2_resid(index) > 0 ) {
	index_2_rotoffset(index) = p_rotamer_set_->rotindex_offsets( index_2_resid(index) );
      }
      else {
	index_2_rotoffset(index) = 0;
      }

      //cout << "DirectedSimAnnealer::run" <<  " - " << REPORT3(index,index_2_resid(index),index_2_rotoffset(index)) << endl;

    } // index


    // !!! assumes a lot about how the rotid map is arranged
    int rotid = 1;
    for( int index = 1; index <= N; ++index ) {
      for( int state = 1; state <= pdig->get_num_states_for_node(index) ; ++state ) {
	rotid_2_index(rotid) = index;
	rotid_2_state(rotid) = state;

	//cout << "DirectedSimAnnealer::run" <<  " - " << REPORT3(rotid,rotid_2_index(rotid),rotid_2_state(state)) << endl;

	++rotid;

      } // state
    } // index

    // Step 0.c - initialize the bias matrix
    //     cout << "WeightedSimAnnealer::run - NUMBER OF MOLTEN RESIDUE = " << REPORT1(N) << endl;
    //     cout << "WeightedSimAnnealer::run - READING '" << filename << "'" << endl;
    //     File bias_file(filename);
    //     cout << "WeightedSimAnnealer::run - FILE " << bias_file << endl;
    //bias_file.initialize(N,*p_rotamer_set_,weights_);

    init_weights(file_,N,*p_rotamer_set_,weights_);

    // <boilerplate>
    if( num_of_rot_to_pack_ == 0 ) return;
    setup_iterations();
    const int outeriterations = get_outeriterations();
    // </boilerplate>

    // Step 0.d start with a random configuration
    for( int index = 1; index <= N; ++index ) {
      state_outer(index) = random_range(1,pdig->get_num_states_for_node(index));
    }

    pdig->set_network_state(state_cur,weights_);
    E_outer = huge_value; // !! we always want to get rid of this completely randomly generated state

    // Step 1 - begin the outer loop
    for( int nn = 1; nn <= outeriterations ; ++nn ) {

      setup_temperature(loopenergy,nn);

      const int inneriterations = get_inneriterations();
      pdig->set_errorfull_deltaE_threshold( std::sqrt( get_temperature() ) );

      // Step 1.a - conduct weighted sequence search
      // Step 1.a.i - reload state_outer
      state_cur = state_outer;
      E_cur = pdig->set_network_state(state_cur,weights_); // !!! use bias energy initially

      for( int n = 1; n <= inneriterations; ++n ) {
	const int rotid = pick_a_rotamer(n,nn);
	if( rotid != -1 ) {
	  const int index      = rotid_2_index(rotid);
	  const int state_next = rotid_2_state(rotid);
	  const int state_prev = state_cur(index);

	  if( state_next != state_prev ) {

	    const int rotoffset = index_2_rotoffset(index);
	    const int rotid_prev = rotoffset + state_prev;
	    const int rotid_next = rotoffset + state_next; // really the same as rotid, but this looks nicer
	    const int aa_prev = p_rotamer_set_->report_aa(rotid_prev);
	    const int aa_next = p_rotamer_set_->report_aa(rotid_next);

	    float deltaE_unweighted, prevE_unweighted, deltaE_weighted, prevE_weighted;
	    pdig->consider_substitution_weighted( index, state_next,
					deltaE_unweighted, prevE_unweighted,
					deltaE_weighted,   prevE_weighted,
					weights_ );


	    if( aa_prev == aa_next && pass_metropolis(prevE_unweighted,deltaE_unweighted) || // aa HAS NOT changed => dE_unweighted
		aa_prev != aa_next && pass_metropolis(prevE_weighted,  deltaE_weighted  )) { // aa HAS changed => dE_weighted

	      // accept the substitution
	      state_cur(index) = state_next;
	      E_cur = pdig->commit_considered_substitution_weighted(weights_);

	      // remember the state with the best weighted energy
	      if( E_cur < E_best_weighted ) {
		E_best_weighted = E_cur;
		state_best_weighted = state_cur;
	      }

	    } // metropolis

	  } // state_next != state_prev

	} // rotid != -1

      } // inneriterations


      // Step 1.c - reload the best weighted state at this point

      state_cur = state_best_weighted;
      E_cur = pdig->set_network_state(state_cur); // !!! switch to using the unweighted energy here

      // Step 1.d - conduct an unweighted conformation search maintaining fixed sequence
      for( int n = 1; n <= inneriterations; ++n ) {
	const int rotid = pick_a_rotamer(n,nn);
	if( rotid != -1 ) {

	  const int index = rotid_2_index(rotid);
	  const int state_next = rotid_2_state(rotid);
	  const int state_prev = state_cur(index);

	  if( state_next != state_prev ) {

	    const int rotoffset = index_2_rotoffset(index);
	    const int rotid_prev = rotoffset + state_prev;
	    const int rotid_next = rotoffset + state_next; // really the same as rotid, but this looks nicer
	    const int aa_prev = p_rotamer_set_->report_aa(rotid_prev);
	    const int aa_next = p_rotamer_set_->report_aa(rotid_next);

	    if( aa_prev == aa_next ) { // !!! maintain fixed sequence at this point
	      float deltaE_unweighted, prevE_unweighted;
	      pdig->consider_substitution( index, state_next,
					  deltaE_unweighted, prevE_unweighted );

	      if( pass_metropolis(prevE_unweighted,deltaE_unweighted) ) {
		// accept this state

		// accept the substitution
		state_cur(index) = state_next;
		E_cur = pdig->commit_considered_substitution();//weights_);

		// remember the state with the best unweighted energy
		if( E_cur < E_best_unweighted ) {
		  E_best_unweighted = E_cur;
		  state_best_unweighted = state_cur;
		}

	      }  // metropolis

	    } // aa_prev == aa_next

	  } // state_next != state_prev

	} // rotid

      } // inneriterations


      // Step 1.d - now measure the WEIGHTED energy of the best UNWEIGHTED state
      E_cur = pdig->set_network_state( state_best_unweighted, weights_ );
      if( E_cur < E_outer ) {
	E_outer = E_cur;
	state_outer = state_best_unweighted;
      }

      // Step 1.e - set the loop energy
      //loopenergy(nn) = E_outer; // !!! NB: should really use different temperatures
      loopenergy(nn) = E_best_unweighted;

      cout << "DirectedSimAnnealer::run " << REPORT4(E_cur,E_best_weighted,E_best_unweighted,E_outer) << endl;


    } // outeriterations



    // Step 2.a - reload state_outer
    E_outer = pdig->set_network_state(state_outer); // use the unweighted energy again
    E_best_unweighted = huge_value;

    // Step 2.b - make absolutely sure that we're in the lowest true-energy
    // conformation of this state by doing another round of outer x
    // inner iterations steps with unweighted energy

    clear(); // this resets the jump_ counter about which some unfortunate assumptions are made in SimAnnealBase

    for( int nn = 1; nn <= outeriterations ; ++nn ) {

      setup_temperature(loopenergy,nn);

      const int inneriterations = get_inneriterations();
      pdig->set_errorfull_deltaE_threshold( std::sqrt( get_temperature() ) );

      for( int n = 1; n <= inneriterations; ++n ) {
	const int rotid = pick_a_rotamer(n,nn);
	if( rotid != -1 ) {

	  const int index = rotid_2_index(rotid);
	  const int state_next = rotid_2_state(rotid);
	  const int state_prev = state_cur(index);

	  if( state_next != state_prev ) {

	    const int rotoffset = index_2_rotoffset(index);
	    const int rotid_prev = rotoffset + state_prev;
	    const int rotid_next = rotoffset + state_next; // really the same as rotid, but this looks nicer
	    const int aa_prev = p_rotamer_set_->report_aa(rotid_prev);
	    const int aa_next = p_rotamer_set_->report_aa(rotid_next);

	    if( aa_prev == aa_next ) { // !!! still maintain fixed sequence at this point
	      float deltaE_unweighted, prevE_unweighted;
	      pdig->consider_substitution( index, state_next,
					   deltaE_unweighted, prevE_unweighted );

	      if( pass_metropolis(prevE_unweighted,deltaE_unweighted) ) {
		// accept this state

		// accept the substitution
		state_cur(index) = state_next;
		E_cur = pdig->commit_considered_substitution();//weights_);

		// remember the state with the best unweighted energy
		if( E_cur < E_outer ) {
		  E_outer = E_cur;
		  state_outer = state_cur;
		}

	      }  // metropolis

	    } // aa_prev == aa_next

	  } // state_next != state_prev

	} // rotid

      } // inneriterations

      loopenergy(nn) = E_cur;

      cout << "DirectedSimAnnealer::run " << REPORT3(E_cur,E_best_unweighted,E_outer) << endl;

    }  // outeriterations

    E_outer = pdig->set_network_state(state_outer); // use the unweighted energy again



    // Step 3 - make sure no vertices are unassigned
    if( pdig->any_vertex_state_unassigned() ) {
      std::cerr << "Critical error -- In DirectedSimAnnealer, one or more vertex states unassigned at annealing's completion." << std::endl;
      std::cerr << "Critical error -- assignment and energy of assignment meaningless" << std::endl;
      assert( ! pdig->any_vertex_state_unassigned() );
    }

    // Step 4 - convert state_outer to best_rotamer_at_seqpos
    for( int index = 1; index <= N; ++index ) {
      const int resid = index_2_resid(index);
      if( resid > 0 ) {
	bestrotamer_at_seqpos_(resid) = index_2_rotoffset(index) + state_outer(index);
      }
      else if( resid < 0 ) {
	const int resid_lig = -resid - 1; assert( resid_lig > 0 );
	ligand::ligand_ptr_vector[ resid_lig ]->change_to_ligand_conformation( state_outer(index) - 1 );
	ligand::ligand_ptr_vector[ resid_lig ]->set_best_coordinates();
      }
      else {
	assert( false );
      }
    } // index

    // Step 5 - done!

  } // DirectedSimAnnealer::run


  ////////////////////////////////////////////////////////////////////////////////
  /// @begin DirectedSimAnnealer::pick_a_rotamer()
  ///
  /// @brief pick a rotamer from a list
  ///
  /// @detailed:if no rotamer is available, return a nonsense number, like -1
  ///
  /// @global_read
  ///
  /// @global_write
  ///
  /// @remarks
  ///
  /// @references
  ///
  /// @authors
  ///
  /// @last_modified
  ////////////////////////////////////////////////////////////////////////////////
  int DirectedSimAnnealer::pick_a_rotamer
  (
   int innercycle,
   int outercycle
   )
  {
    using namespace misc;

    bool start_with_current = get_start_with_current();
    int ranrotamer = -1;
    int num = 0;
    //bk if quench cycle, pass through all rotamers before
    //bk repeating a rotamer
    if ( quench() ){
      num =   mod(innercycle - 1, num_of_rot_to_pack_);
      if (num == 0){
	std::random_shuffle(rot_to_pack_.begin(),rot_to_pack_.end());
      }
      ranrotamer = rot_to_pack_.at(num);
      //bk if start of run and start_with_current is true then first nres
      //bk iterations will be used to place the current rotamers
    }else if (start_with_current && outercycle == 1 && innercycle <= total_residue){
      if (current_rot_index_(innercycle) != -1 ){
	ranrotamer = current_rot_index_(innercycle);
      }else{
	return -1;
      }
    }else{
      ranrotamer = rot_to_pack_.at(static_cast<int>( num_of_rot_to_pack_ * ran3() ));
    }
    return ranrotamer;
  }

  ////////////////////////////////////////////////////////////////////////////////
  /// @begin DirectedSimAnnealer::setup_bias
  ///
  /// @brief initialize the weights_ and bias_weight
  ///
  /// @detailed:
  ///
  /// @global_read
  ///
  /// @global_write
  ///
  /// @remarks
  ///
  /// @references
  ///
  /// @authors
  ///
  /// @last_modified
  ////////////////////////////////////////////////////////////////////////////////

  void DirectedSimAnnealer::set_weight_file(directed_design::File& file) {
    file_ = file;
  } // DirectedSimAnnealer::set_weight_file


}//end of namespace






