// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7630 $
//  $Date: 2006-03-10 19:37:52 +0200 (Fri, 10 Mar 2006) $
//  $Author: stuartm $
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/...
// (C) 199x-2005 University of Washington
// (C) 199x-2005 University of California Santa Cruz
// (C) 199x-2005 University of California San Francisco
// (C) 199x-2005 Johns Hopkins University
// (C) 199x-2005 University of North Carolina, Chapel Hill
// (C) 199x-2005 Vanderbilt University

/// @file   Dunbrack4D.h
///
/// @brief  dun_chi and dun_sd array wrapper supporting variable numbers of chi angles
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


#ifndef INCLUDED_rosetta_Dunbrack4D_H
#define INCLUDED_rosetta_Dunbrack4D_H


// Rosetta headers
#include "dunbrack_pack.h"
#include "param.h"

// Other project headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/ChunkVector.hh>


namespace dunbrack_pack {


/// @brief dun_chi and dun_sd array wrapper class
///
/// @remarks
///  This class provides a 4D array-like interface to a compressed array representation.
///  The purpose is to save memory in the large dunbrack_pack dun_chi and dun_sd arrays.
///  The uncompressed array uses 19,740,672 bytes.
///  The total size of this representation is ~8MB on a 32-bit platform (ChunkVector has
///   a higher overhead on 64-bit platforms but that has a small impact on this size).
///  Compression is achieved because most amino acids have fewer than 4 chi angles.
///  ChunkVector is used because it has a known small excess capacity, efficient growth
///   and shrink operations, and lower overhead and faster lookup than std::deque.
///  The Chunk size was chosen to balance excess capacity during accumulation (before
///   shrink is run) and memory overhead.
///  For the dunbrack library used as of 2005/10 the array of indices was measured to
///   have a density (fraction of active values) of ~.357, which is too high to make the
///   lookup slowdown of a packed representation worth it.
///  For the dunbrack library used as of 2005/10 the array of values was measured to have
///   1,417,824 values and use 11,664 Chunks (or ~9 Chunks per (phi,psi) bin).
///  The 14 lower bits in a key tell the start index in the ChunkVector for that
///   ( iphi, ipsi, irot ) and the 2 upper bits tell the number of chi angle entries.
///  The value returned by the subscript operator for an irot not in the library or an
///   ichi greater that the nchi used for the library is zero.
class Dunbrack4D
{


private: // Types


	typedef  unsigned short int  ushort; // Assumed to be 16 bits
	typedef  ChunkVector_float::size_type  size_type;


public: // Creation

	/// @brief Default constructor
	inline
	Dunbrack4D() :
		keys_( nbins(), nbins(), max_packed_rotno(), npos ),
		values_( nbins(), nbins(), ChunkVector_float( 0, 7 ) ) // Chunk size == 2^7 == 128
	{}


	/// @brief Destructor
	inline
	~Dunbrack4D()
	{}


private: // Creation


	/// @ brief Copy constructor
	Dunbrack4D( Dunbrack4D const & ); // Undefined


private: // Methods: assignment operators


	/// @brief Copy assignment
	Dunbrack4D &
	operator=( Dunbrack4D const & ); // Undefined


public: // Methods


	/// @brief Set values for ( iphi, ipsi, irot )
	inline
	Dunbrack4D &
	set(
		int const iphi,
		int const ipsi,
		int const irot,
		int const nchi,
		FArray1D_float const & v
	)
	{
		assert( ( nchi > 0 ) && ( nchi <= param::MAX_CHI ) );
		ushort & key( keys_( iphi, ipsi, irot ) );
		ChunkVector_float & vec( values_( iphi, ipsi ) );
		if ( key == npos ) { // New entry
			assert( ( vec.size() < static_cast< size_type >( index_mask ) ) );
			ushort const index = vec.size();
			key = ( ushort( nchi - 1 ) << nchi_shift ) | index;
			for ( int ichi = 1; ichi <= nchi; ++ichi ) {
				vec.push_back( v( ichi ) );
			}
		} else { // Overwrite existing values
			assert( ( nchi == ( ( key & nchi_mask ) >> nchi_shift ) + 1 ) ); // nchi must not change
			ushort const index = ( key & index_mask );
			for ( int ichi = 1; ichi <= nchi; ++ichi ) {
				vec[ index + ushort( ichi - 1 ) ] = v( ichi );
			}
		}
		return *this;
	}


	/// @brief Shrink the vectors of values to right-size
	inline
	Dunbrack4D &
	shrink()
	{
		for ( int ipsi = 1; ipsi <= nbins()(); ++ipsi ) {
			for ( int iphi = 1; iphi <= nbins()(); ++iphi ) {
				values_( iphi, ipsi ).shrink();
			}
		}
		return *this;
	}


public: // Properties


	/// @brief Subscript: Dunbrack4D( iphi, ipsi, irot, ichi ) const
	/// @note  Returns zero for inactive irot or ichi greater than nchi in array
	inline
	float
	operator ()(
		int const iphi,
		int const ipsi,
		int const irot,
		int const ichi
	) const
	{
		ushort const & key( keys_( iphi, ipsi, irot ) );
		assert( ( ichi > 0 ) && ( ichi <= param::MAX_CHI ) );
		return ( ( ( key != npos ) &&
			( ushort( ichi ) <= ( ( key & nchi_mask ) >> nchi_shift ) + 1 ) )
			? values_( iphi, ipsi )[ ( key & index_mask ) + ushort( ichi - 1 ) ]
			: 0.0f
		);
	}


private: // Fields


	/// @brief Uninitialized key vec
	static ushort const npos;

	/// @brief Mask to select the 14 bits of the index (and max index vec + 1)
	static ushort const index_mask;

	/// @brief Mask to select the 2 bits of the nchi vec
	static ushort const nchi_mask;

	/// @brief Shift to extract the nchi vec
	static ushort const nchi_shift;

	/// @brief Keys with index of first ( iphi, ipsi, irot ) entry and nchi vec
	FArray3D< ushort > keys_;

	/// @brief Packed vector of values for each ( iphi, ipsi )
	/// @note  Not necessarily ordered by the rotamer number
	FArray2D< ChunkVector_float > values_;


}; // Dunbrack4D


} // namespace dunbrack_pack


#endif // INCLUDED_rosetta_Dunbrack4D_H
