// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7630 $
//  $Date: 2006-03-10 19:37:52 +0200 (Fri, 10 Mar 2006) $
//  $Author: stuartm $
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/...
// (C) 199x-2005 University of Washington
// (C) 199x-2005 University of California Santa Cruz
// (C) 199x-2005 University of California San Francisco
// (C) 199x-2005 Johns Hopkins University
// (C) 199x-2005 University of North Carolina, Chapel Hill
// (C) 199x-2005 Vanderbilt University

/// @file   Dunbrack5D.h
///
/// @brief  dun_group_rotno_to_rotno array wrapper
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


#ifndef INCLUDED_rosetta_Dunbrack5D_H
#define INCLUDED_rosetta_Dunbrack5D_H


// Rosetta headers
#include "dunbrack_pack.h"
#include "param.h"
#include "param_aa.h"


// Other project headers
#include <ObjexxFCL/byte.hh>
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/Fmath.hh>

// Standard library headers
#include <cmath>


namespace dunbrack_pack {


/// @brief dun_group_rotno_to_rotno array wrapper class
///
/// @remarks
///  This class provides a 5D array-like interface to a compressed array representation.
///  The purpose is to save memory in the dunbrack_pack dun_group_rotno_to_rotno array.
///  The uncompressed array uses 11,757,312 bytes.
///  The total size of this representation is ~2MB.
///  Compression is achieved by encoding the values for all chi angles into a single byte.
///  The rotno values { 1, 2, 3, 4 } are mapped to the 2-bit values { 00, 01, 10, 11 }
///   except for ASN which also has rotno values of 5 and 6: ASN's 2 chi angles are each
///   mapped to 4-bit values.
///  The uniform zero last chi angles for SER, THR, and TYR are dealt with by using a
///   nchi_pos array to catch those and return a zero.
class Dunbrack5D
{


private: // Types


	typedef  unsigned char  ubyte;


public: // Creation

	/// @brief Default constructor
	inline
	Dunbrack5D() :
		values_( nbins(), nbins(), // phi x psi bins
//KMa phospho_ser
param::MAX_AA_PLUS(), // Only the 20 amino acids have entries: Rest are zero
			nint( std::pow( static_cast< double >( rot_per_chi ), param::MAX_CHI ) ),
			 // max_rot_per_bin recomputed to avoid initialization order problems
			ubyte( 0 ) // Initialize to zero
		)
	{}


	/// @brief Destructor
	inline
	~Dunbrack5D()
	{}


private: // Creation


	/// @ brief Copy constructor
	Dunbrack5D( Dunbrack5D const & ); // Undefined


private: // Methods: assignment operators


	/// @brief Copy assignment
	Dunbrack5D &
	operator=( Dunbrack5D const & ); // Undefined


public: // Methods


	/// @brief Set values for ( iphi, ipsi, aa, irot )
	/// @note  Don't call for aa > 20 (assert will catch this in debug build)
	inline
	Dunbrack5D &
	set(
		int const iphi,
		int const ipsi,
		int const aa,
		int const irot,
		FArray1D_byte const & rotno
	)
	{
		values_( iphi, ipsi, aa, irot ) = ubyte( 0 );
		if ( aa == param_aa::aa_asn ) { // ASN case: 4 bits/chi
			for ( int ichi = 1, nchi_pos_aa = nchi_pos[ aa ]; ichi <= nchi_pos_aa; ++ichi ) {
				assert( ( rotno( ichi ) > byte( 0 ) ) && ( rotno( ichi ) <= byte( 6 ) ) );
				values_( iphi, ipsi, aa, irot ) = values_( iphi, ipsi, aa, irot ) |
				 ( ubyte( rotno( ichi ) - byte( 1 ) ) << asn_chi_shift[ ichi ] );
			}
		} else { // Non-ASN case
			for ( int ichi = 1, nchi_pos_aa = nchi_pos[ aa ]; ichi <= nchi_pos_aa; ++ichi ) {
				assert( ( rotno( ichi ) > byte( 0 ) ) && ( rotno( ichi ) <= byte( 4 ) ) );
				values_( iphi, ipsi, aa, irot ) = values_( iphi, ipsi, aa, irot ) |
				 ( ubyte( rotno( ichi ) - byte( 1 ) ) << chi_shift[ ichi ] );
			}
		}
		return *this;
	}


public: // Properties


	/// @brief Subscript: Dunbrack5D( iphi, ipsi, aa, irot, ichi ) const
	inline
	ubyte
	operator ()(
		int const iphi,
		int const ipsi,
		int const aa,
		int const irot,
		int const ichi
	) const
	{
		assert( ( aa <= param::MAX_AA_PLUS() ) );
		assert( ( ichi > 0 ) && ( ichi <= param::MAX_CHI ) );
		if ( aa == param_aa::aa_asn ) { // ASN case: 4 bits/chi
			return ( ( aa <= param::MAX_AA_PLUS() ) && ( ichi <= nchi_pos[ aa ] ) ? //KMa phospho_ser
			 ( ( values_( iphi, ipsi, aa, irot ) & asn_chi_mask[ ichi ] ) >> asn_chi_shift[ ichi ] ) +
			 ubyte( 1 ) :
			 ubyte( 0 ) );
		} else { // Non-ASN case
			return ( ( aa <= param::MAX_AA_PLUS() ) && ( ichi <= nchi_pos[ aa ] ) ? //KMa phospho_ser
			 ( ( values_( iphi, ipsi, aa, irot ) & chi_mask[ ichi ] ) >> chi_shift[ ichi ] ) +
			 ubyte( 1 ) :
			 ubyte( 0 ) );
		}
	}


private: // Fields


	/// @brief Number of positive chi angles for each amino acid in Dunbrack library
	static ubyte const nchi_pos[ 22 ];//KMa phospho_ser

	/// @brief Masks to select the chi angles' 2 bits
	static ubyte const chi_mask[ 5 ];

	/// @brief ASN masks to select the chi angles' 4 bits
	static ubyte const asn_chi_mask[ 3 ];

	/// @brief Shifts to extract the chi angles' values
	static ubyte const chi_shift[ 5 ];

	/// @brief ASN shifts to extract the chi angles' values
	static ubyte const asn_chi_shift[ 3 ];

	/// @brief Encoded values for each ( phi, psi, aa, rotamer )
	FArray4D< ubyte > values_;


}; // Dunbrack5D


} // namespace dunbrack_pack


#endif // INCLUDED_rosetta_Dunbrack5D_H
