// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15424 $
//  $Date: 2007-06-12 21:26:56 +0300 (Tue, 12 Jun 2007) $
//  $Author: bcorreia $

#ifndef INTERACTION_GRAPH_BASE_CLASS_H
#define INTERACTION_GRAPH_BASE_CLASS_H

#include "InteractionGraph.h"

// STL Headers
#include <iosfwd>
#include <vector>
#include <list>
#include <assert.h>

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>

namespace pack {

class NodeBase;
class EdgeBase;
class InteractionGraphBase;

extern bool output_interaction_graph_memory_usage;

class NodeBase
{
public:
	virtual ~NodeBase();
	NodeBase(InteractionGraphBase*, int node_id, int num_states);
	int get_num_states() const;
	std::list< EdgeBase* >::iterator add_edge(EdgeBase* edge_ptr);
	void drop_edge(std::list< EdgeBase* >::iterator edge_iterator);
	void drop_all_edges();
	EdgeBase* find_edge(int other_node_index) const;

	virtual void assign_zero_state() = 0; // apl - new 3.20.07 -- remove comment when done
	virtual void prepare_for_simulated_annealing() = 0;
	virtual void add_to_one_body_energies( FArray1DB_float & energy1b ) = 0;
	virtual void add_to_one_body_energy( int state, float energy ) = 0; // apl - new 3.20.07 -- remove comment when done
	virtual void zero_one_body_energies() = 0;
	virtual void print() const = 0;
	virtual bool state_unassigned() const = 0;
	virtual float get_totalE() const = 0;

	void depth_first_connected_component_counting();

	virtual unsigned int count_static_memory() const = 0;
	virtual unsigned int count_dynamic_memory() const;

protected:
	void update_edge_vector();

	//Read access to private data granted to derived classes
	//inlined for efficiency
	inline
	int get_node_index() const
	{
		return node_index_;
	}

	inline
	int get_num_incident_edges() const
	{
		return num_incident_edges_;
	}

	inline
	int get_num_edges_to_smaller_indexed_nodes() const
	{
		return num_edges_to_smaller_indexed_nodes_;
	}

	inline
	int get_num_edges_to_larger_indexed_nodes() const
	{
		return num_edges_to_larger_indexed_nodes_;
	}

	inline
	EdgeBase* get_incident_edge( int index ) const
	{
		assert( edge_vector_up_to_date_ );
		return incident_edge_vector_[ index ];
	}

	inline
	int get_index_of_adjacent_node( int index ) const
	{
		assert( edge_vector_up_to_date_ );
		return adjacent_node_ind_[ index ];
	}

	inline
	NodeBase* get_adjacent_node( int index ) const
	{
		assert( edge_vector_up_to_date_ );
		return adjacent_node_[ index ];
	}

	inline
	bool get_edge_vector_up_to_date() const
	{
		return edge_vector_up_to_date_;
	}

	inline
	InteractionGraphBase* get_owner() const
	{
		return owner_;
	}

private:

	int node_index_;
	int num_states_;
	int num_incident_edges_;
	int num_edges_to_smaller_indexed_nodes_;
	int num_edges_to_larger_indexed_nodes_;
	std::list< EdgeBase* > incident_edge_list_;
	std::vector< EdgeBase* > incident_edge_vector_;
	std::vector< int > adjacent_node_ind_;
	std::vector< NodeBase* > adjacent_node_;
	bool edge_vector_up_to_date_;
	InteractionGraphBase* owner_;

	//no default constructor, uncopyable
	NodeBase();
	NodeBase( NodeBase const & );
	NodeBase & operator = ( NodeBase & );
};

class EdgeBase
{
public:
	virtual ~EdgeBase();
	EdgeBase(InteractionGraphBase* owner, int first_node_ind, int second_node_ind);

	int get_other_ind(int callers_index) const;
	NodeBase* get_other_node(int callers_index) const;
	int get_first_node_ind() const;
	int get_second_node_ind() const;
	void set_pos_in_owners_list( std::list< EdgeBase* >::iterator edge_iterator);
	void set_pos_in_node_edgevector(int callers_index, int position);

	bool same_edge(int node1, int node2) const;

	virtual void declare_energies_final() = 0;
	virtual void prepare_for_simulated_annealing() = 0;

	virtual unsigned int count_static_memory() const = 0;
	virtual unsigned int count_dynamic_memory() const;

protected:

	//Read access to private data granted to derived classes
	inline
	int get_node_index( int index ) const
	{
		assert( index == 0 || index == 1 );
		return node_indices_[ index ];
	}

	inline
	int get_num_states_for_node( int index ) const
	{
		assert( index == 0 || index == 1 );
		return num_node_states_[ index ];
	}

	inline
	NodeBase* get_node( int index ) const
	{
		assert( index == 0 || index == 1 );
		return nodes_[ index ];
	}

	inline
	int get_edges_position_in_nodes_edge_vector( int index ) const
	{
		assert( index == 0 || index == 1 );
		return pos_in_nodes_edge_vector_[ index ];
	}

	inline
	InteractionGraphBase* get_owner() const
	{
		return owner_;
	}


private:
	int node_indices_[2];
	int num_node_states_[2];
	NodeBase* nodes_[2];
	int pos_in_nodes_edge_vector_[2];
	std::list< EdgeBase* >::iterator pos_in_nodes_edge_list_[2];
	std::list< EdgeBase* >::iterator pos_in_owners_edge_list_;
	InteractionGraphBase* owner_;

	//no default constructor, uncopyable
	EdgeBase();
	EdgeBase( EdgeBase const & );
	EdgeBase & operator = ( EdgeBase & );

};

class InteractionGraphBase : public virtual InteractionGraph
{
public:
	virtual ~InteractionGraphBase();
	InteractionGraphBase(int num_nodes);

	virtual int get_num_nodes_v() const
	{
		return get_num_nodes();
	}

	inline
	int get_num_nodes() const
	{
		return num_ig_nodes_;
	}

	void set_num_states_for_node(int node, int num_states);
	int  get_num_states_for_node(int node) const;
	float get_node_totalE( int node_index ) const;
	int  get_num_total_states() const {return num_total_states_;}
	virtual float get_one_body_energy_for_node_state( int node, int state) = 0;
	void add_edge(int node1, int node2);
	bool get_edge_exists(int node1, int node2);
	void drop_all_edges_for_node( int node );

	void print_vertices() const;
	void output_connectivity(std::ostream & os) const;
	void output_dimacs(std::ostream & os) const;

	virtual void prepare_for_simulated_annealing();

	virtual void  blanket_assign_state_0() = 0;
	virtual float set_state_for_node(int node_ind, int new_state) = 0;
	virtual float set_network_state( FArray1DB_int & node_states) = 0;
	virtual void consider_substitution(
		int node_ind,
		int new_state,
		float & delta_energy,
		float & prev_energy_for_node) = 0;
	virtual float  commit_considered_substitution() = 0;
	virtual float get_energy_current_state_assignment() = 0;

	virtual int get_edge_memory_usage() const = 0;
	virtual void print_current_state_assignment() const = 0;
	virtual void set_errorfull_deltaE_threshold( float deltaE ) = 0;

	bool any_vertex_state_unassigned() const;

	void add_to_one_body_energies( FArray1DB_float & one_body_energies );
	void update_one_body_energies( FArray1DB_float & old_energy1b, FArray1DB_float & new_energy1b);
	void zero_one_body_energies_for_node( int node );

	void set_number_of_energy_sum_vertex_groups( int num_groups );
	void set_vertex_member_of_group( int vertex, int group );
	void print_vertex_groups();
	virtual float get_energy_sum_for_vertex_group( int group_id ) = 0;
	int count_connected_components_and_initialize_vertex_groups();
	void note_vertex_reached( int node_index );
	bool vertex_already_reached( int node_index );

	inline
	bool get_vertex_member_of_energy_sum_group( int node_index, int group_id )
	{
		assert(  num_energy_sum_groups_ != -1 &&
			node_index > 0 && node_index <= num_ig_nodes_ &&
			group_id > 0 && group_id <= num_energy_sum_groups_);
		return energy_sum_group_membership_( node_index, group_id );
	}

	virtual unsigned int getTotalMemoryUsage() const;

	virtual unsigned int count_static_memory() const = 0;
	virtual unsigned int count_dynamic_memory() const;

	friend class NodeBase;
	friend class EdgeBase;

protected:

	void drop_edge(std::list< EdgeBase* >::iterator edge);
	EdgeBase* find_edge(int node1, int node2) const;

	virtual NodeBase* create_new_node( int node_index, int num_states) = 0;
	virtual EdgeBase* create_new_edge( int index1, int index2) = 0;

	inline
	NodeBase* get_node( int index ) const
	{
		assert( index > 0 && index <= num_ig_nodes_ );
		return ig_nodes_[ index ];
	}

	inline
	int get_num_edges() const
	{
		return ig_edge_list_.size();
	}

	inline
	std::list< EdgeBase* >::iterator get_edge_list_begin()
	{
		return ig_edge_list_.begin();
	}

	inline
	std::list< EdgeBase* >::const_iterator get_edge_list_begin() const
	{
		return ig_edge_list_.begin();
	}

	inline
	std::list< EdgeBase* >::const_iterator get_edge_list_end() const
	{
		return ig_edge_list_.end();
	}

	inline
	int get_node_state_offset( int index ) const
	{
		assert( index > 0 && index <= num_ig_nodes_ );
		return node_state_offsets_[ index ];
	}

private:
	int num_ig_nodes_;
	std::vector< NodeBase* > ig_nodes_;
	std::list< EdgeBase* > ig_edge_list_;

	std::vector< int > node_state_offsets_;
	int num_total_states_;
	mutable EdgeBase* focused_edge_;

	int num_energy_sum_groups_;
	FArray2D_bool energy_sum_group_membership_;
	FArray1D_int component_membership_;

	//no default constructor, uncopyable
	InteractionGraphBase();
	InteractionGraphBase( InteractionGraphBase const &);
	InteractionGraphBase & operator = (InteractionGraphBase const & );
};

} //end namespace pack

#endif //INTERACTION_GRAPH_BASE_CLASS_H
