// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13214 $
//  $Date: 2007-03-02 01:03:10 -0500 (Fri, 02 Mar 2007) $
//  $Author: csmith $

#ifndef INCLUDED_IntervalSet_HH
#define INCLUDED_IntervalSet_HH


// Unit headers
#include <IntervalSet.fwd.hh>

// Rosetta Headers
#include "random_numbers.h"

// Utility Headers
#include <utility/vector0.hh>

// C++ Headers
#include <ostream>


template <typename T>
class IntervalSet {

private:

	utility::vector0<T> endpoints_;

public:

	inline
	IntervalSet()
	{}

	inline
	IntervalSet(
		T start,
		T end
	):
		endpoints_(2)
	{
		assert(start <= end);

		endpoints_[0] = start;
		endpoints_[1] = end;
	}

	inline
	IntervalSet(
		T start1,
		T end1,
		T start2,
		T end2
	):
		endpoints_(4)
	{
		assert(start1 <= end1 && end1 <= start2 && start2 <= end2);

		endpoints_[0] = start1;
		endpoints_[1] = end1;
		endpoints_[2] = start2;
		endpoints_[3] = end2;
	}

	utility::vector0<T> const &
	endpoints() const
	{
		return endpoints_;
	}

	inline
	void
	clear()
	{
		endpoints_.clear();
	}

	inline
	void
	set(
		T start,
		T end
	)
	{
		endpoints_.resize(2);

		assert(start <= end);

		endpoints_[0] = start;
		endpoints_[1] = end;
	}

	inline
	void
	set(
		T start1,
		T end1,
		T start2,
		T end2
	)
	{
		endpoints_.resize(4);

		assert(start1 <= end1 && end1 <= start2 && start2 <= end2);

		endpoints_[0] = start1;
		endpoints_[1] = end1;
		endpoints_[2] = start2;
		endpoints_[3] = end2;
	}

	inline
	void
	push_back(
		T start,
		T end
	)
	{
		int const size = endpoints_.size();
		endpoints_.resize(size+2);

		assert(start <= end && (size == 0 || start >= endpoints_[size-1]));

		endpoints_[size] = start;
		endpoints_[size+1] = end;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin IntervalSet<T>::length
///
/// @brief
/// calculate the total length of all the intervals
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
/// total length
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
	inline
	T
	length()
	{
		T len = 0;
		int const size = endpoints_.size();
		for (int i = 0; i < size; i += 2) {
			len += endpoints_[i+1] - endpoints_[i];
		}

	return len;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin IntervalSet<T>::is_inside
///
/// @brief
/// determine if a point is within one of the intervals
///
/// @detailed
///
/// @param[in] point - numeric value to test
///
/// @global_read
///
/// @global_write
///
/// @return
/// boolean value with result
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
	inline
	bool
	is_inside(
		T point
	)
	{
		int const size = endpoints_.size();
		for (int i = 0; i < size; i += 2) {
			if (point >= endpoints_[i] && point <= endpoints_[i+1]) {
				return true;
			}
		}

		return false;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin IntervalSet<T>::operator&
///
/// @brief
/// calculate the intersection of two IntervalSets
///
/// @detailed
///
/// @param[in] right - the second Interval set
///
/// @global_read
///
/// @global_write
///
/// @return
/// a new IntervalSet
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
	IntervalSet
	operator&(
		IntervalSet const & right
	)
	{
		IntervalSet<T> newintervals;
		utility::vector0<T> const & endpoints_right = right.endpoints();
		int const size1 = endpoints_.size();
		int const size2 = endpoints_right.size();
		int i1 = 0;
		int i2 = 0;

		while (i1 < size1 && i2 < size2) {

			// Determine which interval starts first
			if (endpoints_[i1] < endpoints_right[i2]) {

				if (endpoints_[i1+1] < endpoints_right[i2]) {
					// If there's no overlap, advance to the next endpoints_ interval
					i1 += 2;
				} else if (endpoints_[i1+1] < endpoints_right[i2+1]) {
					// If there's some overlap, push the intersection
					newintervals.push_back(endpoints_right[i2], endpoints_[i1+1]);
					// Advance to the next endpoints_ interval
					i1 += 2;
				} else {
					// If there's total overlap, push the whole endpoints_right interval
					newintervals.push_back(endpoints_right[i2], endpoints_right[i2+1]);
					// Advance to the next endpoints_right interval
					i2 += 2;
				}
			} else {

				if (endpoints_right[i2+1] < endpoints_[i1]) {
					// If there's no overlap, advance to the next endpoints_right interval
					i2 += 2;
				} else if (endpoints_right[i2+1] < endpoints_[i1+1]) {
					// If there's some overlap, push the intersection
					newintervals.push_back(endpoints_[i1], endpoints_right[i2+1]);
					// Advance to the next endpoints_right interval
					i2 += 2;
				} else {
					// If there's total overlap, push the whole endpoints_ interval
					newintervals.push_back(endpoints_[i1], endpoints_[i1+1]);
					// Advance to the next endpoints_ interval
					i1 += 2;
				}
			}
		}

		return newintervals;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin random_point
///
/// @brief
/// pick a random number uniformly from all the intervals
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
/// random number
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
	T
	random_point()
	{
		T const len = length();
		int const size = endpoints_.size();
		int i = 0;

		T randnum = endpoints_[i] + ran3()*len;

		while (i+3 < size && randnum > endpoints_[i+1]) {
			randnum += endpoints_[i+2] - endpoints_[i+1];
			i += 2;
		}

		if (randnum == endpoints_[i+1] && i+3 < size) {
			int j = 1;
			while (i+2*j+3 < size && endpoints_[i+2*j+2] == endpoints_[i+2*j+3]) j++;
			int randpoint = random_range(0, j);
			if (randpoint > 0) randnum = endpoints_[i+2*randpoint];
		}

		return randnum;
	}
};

template <typename T>
inline
std::ostream &
operator<<(
	std::ostream & output,
	const IntervalSet<T> & interval
)
{
	utility::vector0<T> const & endpoints = interval.endpoints();

	for (int i = 0; i < (signed)endpoints.size(); i += 2) {
		if (i != 0) output << " ";
		output << "[" << endpoints[i] << ", " << endpoints[i+1] << "]";
	}

	return output;
}


#endif // INCLUDED_IntervalSet_HH
