// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

#include "after_opts.h"
#include "InteractionGraphSupport.h"
#include "LinearMemoryInteractionGraph.h"
#include "rotamer_trie_calc_energies.h"

#include <iostream>

namespace pack {

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::LinearMemNode(InteractionGraphBase *, int, int)
///
/// @brief
/// main constructor, no default or copy constructors
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
LinearMemNode::LinearMemNode(
	InteractionGraphBase * owner,
	int node_id,
	int num_states
) :
	OnTheFlyNode( owner, node_id, num_states ),
	num_states_to_keep_in_recent_history_( 0 ),
	curr_num_in_recent_history_( 0 ),
	head_of_rh_queue_ptr_( 0 ),
	end_of_rh_queue_ptr_( 0 ),
	states_2_recent_history_( num_states, 0 ),
	current_state_( 0 ),
	curr_state_one_body_energy_( 0.0f ),
	curr_state_total_energy_( 0.0f ),
	alternate_state_( 0 ),
	alternate_state_one_body_energy_( 0 ),
	alternate_state_total_energy_( 0 ),
	alternate_state_is_being_considered_( false ),
	already_prepped_for_simA_( false ),
	accepted_rejected_substitution_history_( ACCEPTANCE_REJECTION_HISTORY_LENGTH, 0 ),
	accepted_history_head_( 1 ),
	num_recently_accepted_( 0 ),
	filled_substitution_history_( false )
{
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::~LinearMemNode()
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
LinearMemNode::~LinearMemNode()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::prepare_for_simulated_annealing
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemNode::prepare_for_simulated_annealing()
{
	if (! get_edge_vector_up_to_date() ) update_internal_vectors();
	// mark out-of-date anything where coordinates have changed
	for (int ii = 1; ii <= get_num_states(); ++ii)
	{
		if ( already_prepped_for_simA_ && ! get_coordinates_current( ii ) )
		{
			for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
			{
				get_incident_linmem_edge(ii)->reset_state_energies(
					get_node_index(),
					ii,
					states_2_recent_history_( ii )
				);
			}
		}
		mark_coordinates_current( ii );
	}
	already_prepped_for_simA_ = true;

	num_recently_accepted_ =  0;
	filled_substitution_history_ = false;
	accepted_history_head_ = 1;

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::print
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemNode::print() const
{

	std::cerr << "LinearMemNode " << get_node_index() << " with " << get_num_states() << " states" << std::endl;
	std::cerr << "curr_state " << current_state_ << " ";
	std::cerr << "curr_state_sparse_mat_info_ ";
	std::cerr << curr_state_sparse_mat_info_.get_aa_type() << " ";
	std::cerr << curr_state_sparse_mat_info_.get_state_ind_for_this_aa_type() << " ";
	std::cerr << "Curr One Body Energy: " << curr_state_one_body_energy_ << std::endl;
	std::cerr << "Curr Two Body Energies:";
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		std::cerr << " " << get_index_of_adjacent_node(ii) << ":" << curr_state_two_body_energies_[ ii ];
	}
	std::cerr << std::endl;

	if ( ! alternate_state_is_being_considered_ ) return;
	std::cerr << "Alt One Body Energy: " << alternate_state_one_body_energy_ << std::endl;
	std::cerr << "Alt Two Body Energies:";
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		std::cerr << " " << get_index_of_adjacent_node(ii) << ":" << alternate_state_two_body_energies_[ ii ];
	}
	std::cerr << std::endl  << "-----------------" << std::endl;


}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::getMemoryUsageInBytes
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
unsigned int
LinearMemNode::count_static_memory() const
{
	return sizeof( LinearMemNode );
}

unsigned int
LinearMemNode::count_dynamic_memory() const
{
	unsigned int total_memory = OnTheFlyNode::count_dynamic_memory();

	total_memory += recent_history_queue_.size() * sizeof( history_queue_struct );
	total_memory += states_2_recent_history_.size() * sizeof( int );
	total_memory += aa_neighbors_for_edges_.size() * sizeof( unsigned char );
	total_memory += neighbors_curr_state_.size() * sizeof( int );
	total_memory += neighbors_state_recent_history_index_.size() * sizeof( int );
	total_memory += neighbors_curr_state_sparse_info_.size() * sizeof( SparseMatrixIndex );

	total_memory += curr_state_two_body_energies_.size() * sizeof( float );
	total_memory += alternate_state_two_body_energies_.size() * sizeof( float );

	return total_memory;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::assign_zero_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemNode::assign_zero_state()
{
	current_state_ = 0;
	alternate_state_ = 0;
	alternate_state_is_being_considered_ = false;

	curr_state_one_body_energy_ = 0.0f;
	//fills from [1] to end
	std::vector< float >::iterator position1 = curr_state_two_body_energies_.begin();
	++position1;
	std::fill( position1,
		curr_state_two_body_energies_.end(),
		0.0f);
	curr_state_total_energy_ = 0.0f;

	for (int ii = 1; ii <= get_num_incident_edges(); ++ii )
	{
		get_incident_linmem_edge(ii)->
			acknowledge_state_zeroed( get_node_index() );
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::assign_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemNode::assign_state(int new_state)
{
	assert( new_state >= 0 && new_state <= get_num_states());

	if (new_state == 0) assign_zero_state();
	else
	{
		//std::cerr << "assign_state: node -  " << get_node_index() <<
		// " new state " << new_state << "...";
		current_state_ = new_state;
		curr_state_sparse_mat_info_ =
			get_sparse_mat_info_for_state( current_state_ );
		curr_state_one_body_energy_ = get_one_body_energy( current_state_ );
		curr_state_total_energy_ = curr_state_one_body_energy_;
		alternate_state_is_being_considered_ = false;
		int bumped_recent_history_index = update_recent_history( current_state_ );

		for (int ii = 1; ii <= get_num_incident_edges(); ++ii )
		{
			get_incident_linmem_edge(ii)->acknowledge_state_change(
				get_node_index(),
				current_state_,
				curr_state_sparse_mat_info_,
				bumped_recent_history_index,
				head_of_rh_queue_ptr_,
				curr_state_two_body_energies_[ii]
			);

			curr_state_total_energy_ += curr_state_two_body_energies_[ ii ];
		}
		//std::cerr<< "..done" << std::endl;
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::partial_assign_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemNode::partial_assign_state( int new_state )
{
	if (new_state == 0 )
	{
		assign_zero_state();
		return;
	}

	current_state_ = new_state;
	curr_state_sparse_mat_info_ =
		get_sparse_mat_info_for_state( current_state_ );
	int bumped_recent_history_index = update_recent_history( current_state_ );

	for (int ii = 1; ii <= get_num_incident_edges(); ++ii )
	{
		get_incident_linmem_edge(ii)->acknowledge_partial_state_change(
			get_node_index(),
			current_state_,
			curr_state_sparse_mat_info_,
			bumped_recent_history_index,
			head_of_rh_queue_ptr_
		);
	}
	alternate_state_is_being_considered_ = false;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::complete_state_assignment
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void LinearMemNode::complete_state_assignment()
{
	if ( current_state_ == 0 ) return;

	curr_state_total_energy_ = curr_state_one_body_energy_ =
		get_one_body_energy( current_state_ );
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		curr_state_two_body_energies_[ ii ] =
			get_incident_linmem_edge( ii )->
			get_energy_following_partial_state_assignment();
		curr_state_total_energy_ += curr_state_two_body_energies_[ ii ];
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::project_deltaE_for_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
LinearMemNode::project_deltaE_for_substitution
(
	int alternate_state,
	float & prev_node_energy
)
{
	alternate_state_is_being_considered_ = true;
	//procrastinated_ = false;
	//std::cerr << "proj_deltaE: node -  " << get_node_index()
	// << " alt state " << alternate_state << "...";

	alternate_state_ = alternate_state;
	int alternate_state_recent_history_index = states_2_recent_history_( alternate_state_ );

	bool store_rpes = num_recently_accepted_ < THRESHOLD_ACCEPTANCE_RATE_FOR_RPE_STORAGE;

	alt_state_sparse_mat_info_ = get_sparse_mat_info_for_state( alternate_state );
	alternate_state_one_body_energy_ = get_one_body_energy( alternate_state );
	alternate_state_total_energy_ = alternate_state_one_body_energy_;
	prev_node_energy = curr_state_total_energy_;

	int aa_neighb_linear_index_offset = aa_neighbors_for_edges_.
		index(1, 1, alt_state_sparse_mat_info_.get_aa_type() ) - 1;

	for (int ii = 1; ii <= get_num_incident_edges();
		++ii, aa_neighb_linear_index_offset += get_num_aa_types())
	{

		if ( neighbors_curr_state_[ ii ] != 0 &&
			aa_neighbors_for_edges_[ aa_neighb_linear_index_offset
			+ neighbors_curr_state_sparse_info_[ ii ].get_aa_type() ] )
		{

			FArray1Da_float neighbor_curr_rotamer_actcoords(
				get_adjacent_linmem_node( ii )->
				get_curr_rotamer_actcoords(), 3 );

			alternate_state_two_body_energies_[ ii ] =
				get_incident_linmem_edge( ii )->get_energy_for_alt_state(
				store_rpes,
				get_node_index(),
				alternate_state_,
				alternate_state_recent_history_index,
				neighbors_curr_state_[ ii ],
				neighbors_state_recent_history_index_[ ii ]
				);
		}
		else
		{
			alternate_state_two_body_energies_[ ii ] = 0;
		}

		alternate_state_total_energy_ += alternate_state_two_body_energies_[ ii ];
	}


	return alternate_state_total_energy_ - curr_state_total_energy_;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::commit_considered_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemNode::commit_considered_substitution()
{
	assert( alternate_state_is_being_considered_ );

	current_state_ = alternate_state_;
	curr_state_sparse_mat_info_ = alt_state_sparse_mat_info_;
	curr_state_one_body_energy_ = alternate_state_one_body_energy_;
	curr_state_total_energy_ = alternate_state_total_energy_;

	//copies from [1] to end
	std::vector< float >::iterator alt_position1 = alternate_state_two_body_energies_.begin();
	++alt_position1;
	std::vector< float >::iterator curr_position1 = curr_state_two_body_energies_.begin();
	++curr_position1;

	std::copy( alt_position1,
		alternate_state_two_body_energies_.end(),
		 curr_position1 );

	int bumped_recent_history_index = update_recent_history( current_state_ );

	for ( int ii = 1; ii <= get_num_incident_edges(); ++ii )
	{
		get_incident_linmem_edge(ii)->acknowledge_substitution(
			get_node_index(),
			curr_state_two_body_energies_[ ii ],
			current_state_,
			curr_state_sparse_mat_info_,
			bumped_recent_history_index,
			head_of_rh_queue_ptr_,
			neighbors_curr_state_[ ii ]
		);
	}

	alternate_state_is_being_considered_ = false;

	++accepted_history_head_;
	if (accepted_history_head_ > ACCEPTANCE_REJECTION_HISTORY_LENGTH )
	{
		accepted_history_head_ = 1;
		filled_substitution_history_ = true;
	}
	if ( ! filled_substitution_history_ ||
		accepted_rejected_substitution_history_( accepted_history_head_ ) == REJECTED )
	{
		++num_recently_accepted_;
	}
	accepted_rejected_substitution_history_( accepted_history_head_ ) = ACCEPTED;

	return;
}

void
LinearMemNode::acknowledge_last_substititon_not_committed()
{
	alternate_state_is_being_considered_ = false;
	++accepted_history_head_;
	if (accepted_history_head_ > ACCEPTANCE_REJECTION_HISTORY_LENGTH )
	{
		accepted_history_head_ = 1;
		filled_substitution_history_ = true;
	}
	if ( filled_substitution_history_ &&
		accepted_rejected_substitution_history_( accepted_history_head_ ) == ACCEPTED )
	{
		--num_recently_accepted_;
	}
	accepted_rejected_substitution_history_( accepted_history_head_ ) = REJECTED;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::compute_pair_energy_for_current_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
LinearMemNode::compute_pair_energy_for_current_state(
	int edge_making_energy_request
)
{
	if ( aa_neighbors_for_edges_(
		neighbors_curr_state_sparse_info_[ edge_making_energy_request ].get_aa_type(),
		edge_making_energy_request,
		curr_state_sparse_mat_info_.get_aa_type() ) )
	{

		FArray1Da_float curr_rotamer_actcoords(
			get_rotamer_actcoords( current_state_ ), 3 );

		FArray1Da_float neighbor_curr_rotamer_actcoords(
			get_adjacent_linmem_node( edge_making_energy_request )->
			get_curr_rotamer_actcoords(), 3 );

		return compute_rotamer_pair_energy(
			edge_making_energy_request,
			current_state_,
			neighbors_curr_state_[ edge_making_energy_request ],
			curr_state_sparse_mat_info_.get_aa_type(),
			neighbors_curr_state_sparse_info_[ edge_making_energy_request ].get_aa_type(),
			get_rotamer( current_state_ ),
			get_adjacent_linmem_node( edge_making_energy_request )->get_current_rotamer(),
			curr_rotamer_actcoords,
			neighbor_curr_rotamer_actcoords
		);
	}
	else
	{
		return 0;
	}

}

float
LinearMemNode::compute_pair_energy_for_alternate_state(
	int edge_making_energy_request
)
{
	FArray1Da_float alt_rotamer_actcoords(
		get_rotamer_actcoords( alternate_state_ ), 3 );

	FArray1Da_float neighbor_curr_rotamer_actcoords(
		get_adjacent_linmem_node( edge_making_energy_request )->
		get_curr_rotamer_actcoords(), 3 );

	return compute_rotamer_pair_energy(
		edge_making_energy_request,
		alternate_state_,
		neighbors_curr_state_[ edge_making_energy_request ],
		alt_state_sparse_mat_info_.get_aa_type(),
		neighbors_curr_state_sparse_info_[ edge_making_energy_request ].get_aa_type(),
		get_rotamer( alternate_state_ ),
		get_adjacent_linmem_node( edge_making_energy_request )->get_current_rotamer(),
		alt_rotamer_actcoords,
		neighbor_curr_rotamer_actcoords );
}




////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::acknowledge_neighbors_state_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemNode::acknowledge_neighbors_state_substitution(
	int edge_to_altered_neighbor,
	float new_edge_energy,
	int other_node_new_state,
	SparseMatrixIndex const & other_node_new_state_sparse_info,
	int other_node_recent_history_index
)
{
	curr_state_total_energy_ +=
		new_edge_energy - curr_state_two_body_energies_[ edge_to_altered_neighbor ];
	curr_state_two_body_energies_[ edge_to_altered_neighbor ] = new_edge_energy;
	neighbors_curr_state_[ edge_to_altered_neighbor ] = other_node_new_state;
	neighbors_curr_state_sparse_info_[ edge_to_altered_neighbor ]  =
		other_node_new_state_sparse_info;
	neighbors_state_recent_history_index_[ edge_to_altered_neighbor ] =
		other_node_recent_history_index;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::acknowledge_neighbors_partial_state_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemNode::acknowledge_neighbors_partial_state_substitution(
	int edge_to_altered_neighbor,
	int other_node_new_state,
	SparseMatrixIndex const & other_node_new_state_sparse_info,
	int other_state_recent_history_index
)
{
	curr_state_total_energy_ = 0;
	curr_state_two_body_energies_[ edge_to_altered_neighbor ] = 0;
	neighbors_curr_state_[ edge_to_altered_neighbor ] = other_node_new_state;
	neighbors_curr_state_sparse_info_[ edge_to_altered_neighbor ]  =
		other_node_new_state_sparse_info;
	neighbors_state_recent_history_index_[ edge_to_altered_neighbor ] =
		other_state_recent_history_index;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::get_sparse_mat_info_for_curr_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
SparseMatrixIndex const &
LinearMemNode::get_sparse_mat_info_for_curr_state() const
{
	return get_sparse_mat_info_for_state( current_state_ );
}

void LinearMemNode::set_recent_history_size(
	int num_states_to_maintain_in_recent_history
)
{
	num_states_to_keep_in_recent_history_ =
		num_states_to_maintain_in_recent_history;
	recent_history_queue_.dimension( num_states_to_keep_in_recent_history_ );
	for (int ii = 1; ii <= num_states_to_keep_in_recent_history_; ++ii)
	{
		recent_history_queue_(ii).more_recent_ptr = 0;
		recent_history_queue_(ii).state_in_rh = 0;
		recent_history_queue_(ii).more_ancient_ptr = 0;
	}
	curr_num_in_recent_history_ = 0;
	head_of_rh_queue_ptr_ = 0;
	end_of_rh_queue_ptr_ = 0;
}

int
LinearMemNode::get_recent_history_size() const
{
	return num_states_to_keep_in_recent_history_;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::print_internal_energies
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemNode::print_internal_energies()
{
	std::cerr << "curr_state " << current_state_ << " ";
	std::cerr << "curr_state_sparse_mat_info_ ";
	std::cerr << curr_state_sparse_mat_info_.get_aa_type() << " ";
	std::cerr << curr_state_sparse_mat_info_.get_state_ind_for_this_aa_type() << " ";
	std::cerr << "curr_state_one_body_energy_ ";
	std::cerr << curr_state_one_body_energy_ << " ";
	std::cerr << "curr_state_total_energy_" << curr_state_total_energy_ << " ";
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		std::cerr << "(" << get_index_of_adjacent_node(ii) << ":" << curr_state_two_body_energies_[ ii ] << ") ";
	}
	std::cerr << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::update_internal_energy_sums
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemNode::update_internal_energy_sums()
{
	assert( get_edge_vector_up_to_date() );
	curr_state_total_energy_ = 0;
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		curr_state_total_energy_ +=
			get_incident_linmem_edge(ii)->get_current_two_body_energy();
	}
	curr_state_total_energy_ += curr_state_one_body_energy_;
	return;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::get_curr_rotamer_actcoords
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float &
LinearMemNode::get_curr_rotamer_actcoords()
{
	return get_rotamer_actcoords( current_state_ );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::compute_rotamer_pair_energy
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
/*
float
LinearMemNode::compute_rotamer_pair_energy(
	int rot1_aa,
	int rot2_aa,
	rotamer_trie const & rotamer1,
	rotamer_trie const & rotamer2,
	FArray1Da_float & rotamer1_actcoords,
	FArray1Da_float & rotamer2_actcoords
)
{
	++num_rpe_calcs;
	return get_sc_scE_trie( rot1_aa, rot2_aa,
		rotamer1, rotamer2, rotamer1_actcoords, rotamer2_actcoords );

}
*/

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemNode::update_internal_vectors
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void LinearMemNode::update_internal_vectors()
{
	NodeBase::update_edge_vector();
	neighbors_curr_state_.resize( get_num_incident_edges() + 1);
	neighbors_curr_state_sparse_info_.resize( get_num_incident_edges() + 1);
	neighbors_state_recent_history_index_.resize( get_num_incident_edges() + 1 );

	aa_neighbors_for_edges_.dimension(
		get_num_aa_types(), get_num_incident_edges(), get_num_aa_types());

	//copy sparse aa-neighbor info from edges
	int count_neighbs_with_higher_indices = 0;
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		neighbors_curr_state_sparse_info_[ii].set_aa_type( 1 );
		neighbors_state_recent_history_index_[ii] = 0;

		FArray2D< unsigned char > const & edge_aa_neighbs =
			get_incident_linmem_edge(ii)->get_sparse_aa_neighbor_info();

		if ( get_node_index() < get_index_of_adjacent_node(ii) )
		{  ++count_neighbs_with_higher_indices;
			for ( int jj = 1; jj <= get_num_aa_types(); ++jj )
			{
				for ( int kk = 1; kk <= get_num_aa_types(); ++kk )
				{
					aa_neighbors_for_edges_(kk, ii, jj) = edge_aa_neighbs(kk, jj);
				}
			}
		}
		else
		{
			for ( int jj = 1; jj <= get_num_aa_types(); ++jj )
			{
				for ( int kk = 1; kk <= get_num_aa_types(); ++kk )
				{
					aa_neighbors_for_edges_(kk, ii, jj) =
						edge_aa_neighbs(jj, kk);
				}
			}
		}
	}

	curr_state_two_body_energies_.resize( get_num_incident_edges() + 1);
	alternate_state_two_body_energies_.resize( get_num_incident_edges() + 1);
	return;
}

// @ brief - allow derived class to "drive" through the deltaE calculation
void
LinearMemNode::calc_deltaEpd( int alternate_state )
{
	float dummy(0.0f);
	project_deltaE_for_substitution( alternate_state, dummy );
}


int
LinearMemNode::update_recent_history( int state )
{
	if ( states_2_recent_history_( state ) != 0 )
	{
		//already stored in recent history -- nothing gets bumped
		int const state_rh_id = states_2_recent_history_( state );
		if ( head_of_rh_queue_ptr_ == state_rh_id ) return 0;

		int const anc_id = recent_history_queue_( state_rh_id ).more_ancient_ptr;
		int const rec_id = recent_history_queue_( state_rh_id ).more_recent_ptr;

		recent_history_queue_( rec_id ).more_ancient_ptr = anc_id;
		if ( state_rh_id != end_of_rh_queue_ptr_ )
		{
			recent_history_queue_( anc_id ).more_recent_ptr = rec_id;
		}
		else
		{
			end_of_rh_queue_ptr_ = rec_id;
		}

		recent_history_queue_( head_of_rh_queue_ptr_ ).more_recent_ptr = state_rh_id;
		recent_history_queue_( state_rh_id ).more_ancient_ptr = head_of_rh_queue_ptr_;
		recent_history_queue_( state_rh_id ).more_recent_ptr = 0;
		head_of_rh_queue_ptr_ = state_rh_id;

		return 0;
	}
	else if ( curr_num_in_recent_history_ < num_states_to_keep_in_recent_history_ )
	{
		++curr_num_in_recent_history_;
		if ( curr_num_in_recent_history_ == 1 ) end_of_rh_queue_ptr_ = 1;

		states_2_recent_history_( state ) = curr_num_in_recent_history_;
		recent_history_queue_( curr_num_in_recent_history_ ).state_in_rh = state;

		if ( head_of_rh_queue_ptr_ != 0 )
		{
			recent_history_queue_( head_of_rh_queue_ptr_ ).more_recent_ptr = curr_num_in_recent_history_;
		}
		recent_history_queue_( curr_num_in_recent_history_ ).more_ancient_ptr = head_of_rh_queue_ptr_;
		recent_history_queue_( curr_num_in_recent_history_ ).more_recent_ptr = 0;
		head_of_rh_queue_ptr_ = curr_num_in_recent_history_;

	}
	else
	{
		//not in recent history, something gets bumped

		if ( num_states_to_keep_in_recent_history_ == 1 )
		{
			states_2_recent_history_( recent_history_queue_( 1 ).state_in_rh ) = 0;
			states_2_recent_history_( state ) = 1;
			recent_history_queue_( 1 ).state_in_rh = state;
			return 1;
		}
		else
		{
			int const prev_end = end_of_rh_queue_ptr_;
			int const one_before_end = recent_history_queue_( prev_end ).more_recent_ptr;
			recent_history_queue_( one_before_end ).more_ancient_ptr = 0;
			end_of_rh_queue_ptr_ = one_before_end;

			states_2_recent_history_( recent_history_queue_( prev_end ).state_in_rh ) = 0;

			states_2_recent_history_( state ) = prev_end;
			recent_history_queue_( prev_end ).state_in_rh = state;


			recent_history_queue_( prev_end ).more_recent_ptr = 0;
			recent_history_queue_( prev_end ).more_ancient_ptr = head_of_rh_queue_ptr_;
			recent_history_queue_( head_of_rh_queue_ptr_ ).more_recent_ptr = prev_end;
			head_of_rh_queue_ptr_ = prev_end;

			return prev_end;
		}
	}
	return 0; //control of flow never reaches here; makes compiler happy
}


//-----------------------------------------------------------------//

float const LinearMemEdge::NOT_YET_COMPUTED_ENERGY = -1234;


////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::LinearMemEdge
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
LinearMemEdge::LinearMemEdge(
	InteractionGraphBase* owner,
	int first_node_ind,
	int second_node_ind
):
	OnTheFlyEdge( owner, first_node_ind, second_node_ind),
	sparse_aa_neighbors_(
		get_linmem_ig_owner()->get_num_aatypes(),
		get_linmem_ig_owner()->get_num_aatypes(),
		(unsigned char) 0 ),
	curr_state_energy_( 0.0f ),
	partial_state_assignment_( false ),
	preped_for_sim_annealing_( false )
{
	store_rpes_[ 0 ] = store_rpes_[ 1 ] = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::~LinearMemEdge()
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
LinearMemEdge::~LinearMemEdge()
{
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::set_sparse_aa_info
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemEdge::set_sparse_aa_info(
	FArray2DB_bool const & aa_neighbors
)
{
	for (int ii = 1; ii <= get_linmem_ig_owner()->get_num_aatypes(); ++ii)
	{
		for (int jj = 1; jj <= get_linmem_ig_owner()->get_num_aatypes(); ++jj)
		{
			if ( aa_neighbors( jj, ii ) )
			{
				sparse_aa_neighbors_( jj, ii ) = 1;
			}
			else
			{
				sparse_aa_neighbors_( jj, ii ) = 0;
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::declare_energies_final
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemEdge::declare_energies_final()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::prepare_for_simulated_annealing
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemEdge::prepare_for_simulated_annealing()
{
	for (int ii = 0; ii < 2; ++ii )
	{
		if ( ! store_rpes_[ ii ] ) wipe( ii );
		store_rpes_[ ii ] = true;
	}

	if ( preped_for_sim_annealing_ ) return;

	for (int ii = 0; ii < 2; ++ii)
	{
		int const other = ! ii;
		stored_rpes_[ ii ].dimension( get_num_states_for_node( other ),
			get_linmem_node( ii )->get_recent_history_size() );
		stored_rpes_[ ii ] = NOT_YET_COMPUTED_ENERGY;
	}

	preped_for_sim_annealing_ = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::getMemoryUsageInBytes
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

unsigned int
LinearMemEdge::count_static_memory() const
{
	return sizeof( LinearMemEdge );
}


unsigned int
LinearMemEdge::count_dynamic_memory() const
{
	unsigned int total_memory = OnTheFlyEdge::count_dynamic_memory();
	total_memory += sparse_aa_neighbors_.size() * sizeof( unsigned char );
	total_memory += stored_rpes_[ 0 ].size() * sizeof( float );
	total_memory += stored_rpes_[ 1 ].size() * sizeof( float );

	return total_memory;
}




////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::get_current_two_body_energy
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
LinearMemEdge::get_current_two_body_energy() const
{
	return curr_state_energy_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::acknowledge_state_change
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemEdge::acknowledge_state_change(
	int node_ind,
	int new_state,
	SparseMatrixIndex const & new_state_sparse_info,
	int bumped_recent_history_index,
	int new_state_recent_history_index,
	float & new_energy
)
{
	int node_substituted =  ( node_ind == get_node_index(0) ? 0 : 1);
	int node_not_substituted = ! node_substituted;

	handle_bumped_recent_history_state_for_node(
		node_substituted,
		node_not_substituted,
		bumped_recent_history_index );

	curr_state_energy_ = get_linmem_node( 0 )->
		compute_pair_energy_for_current_state(
		get_edges_position_in_nodes_edge_vector( 0 ) );

	store_curr_state_energy();

	new_energy = curr_state_energy_;

	get_linmem_node( node_not_substituted )->
		acknowledge_neighbors_state_substitution
		(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		curr_state_energy_,
		new_state,
		new_state_sparse_info,
		new_state_recent_history_index
		);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::acknowledge_state_zeroed
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemEdge::acknowledge_state_zeroed( int node_ind )
{
	int node_substituted = ( node_ind == get_node_index(0) ? 0 : 1);
	int node_not_substituted = ! node_substituted;

	curr_state_energy_ = 0;
	SparseMatrixIndex dummy_sparse_info;
	dummy_sparse_info.set_aa_type( 1 );
	dummy_sparse_info.set_state_ind_for_this_aa_type(1);

	get_linmem_node( node_not_substituted )->
		acknowledge_neighbors_state_substitution
		(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		curr_state_energy_,
		0,
		dummy_sparse_info,
		0
		);
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::acknowledge_partial_state_change
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void LinearMemEdge::acknowledge_partial_state_change(
	int node_ind,
	int new_state,
	SparseMatrixIndex const & new_state_sparse_info,
	int bumped_recent_history_index,
	int new_state_recent_history_index
)
{
	int node_substituted =  ( node_ind == get_node_index(0) ? 0 : 1);
	int node_not_substituted = ! node_substituted;

	handle_bumped_recent_history_state_for_node(
		node_substituted,
		node_not_substituted,
		bumped_recent_history_index );

	curr_state_energy_ = 0;

	get_linmem_node( node_not_substituted )->
		acknowledge_neighbors_partial_state_substitution
		(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		new_state,
		new_state_sparse_info,
		new_state_recent_history_index
		);
	partial_state_assignment_ = true;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::get_energy_following_partial_state_assignment
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
LinearMemEdge::get_energy_following_partial_state_assignment()
{
	if (partial_state_assignment_
		&& get_linmem_node(0)->get_current_state() != 0
		&& get_linmem_node(1)->get_current_state() != 0)
	{

		curr_state_energy_ = get_linmem_node( 0 )->
			compute_pair_energy_for_current_state(
			get_edges_position_in_nodes_edge_vector( 0 ) );
		partial_state_assignment_ = false;
		store_curr_state_energy();
	}
	return curr_state_energy_;
}

void
LinearMemEdge::reset_state_energies(
	int node_index,
	int state,
	int recent_history_id
)
{
	int node_with_reset_state = node_index == get_node_index( 0 ) ? 0 : 1;
	int other = ! node_with_reset_state;

	if ( recent_history_id != 0 )
	{
		FArray1Da_float row_to_wipe(
			stored_rpes_[ node_with_reset_state ]( 1, recent_history_id ),
			get_num_states_for_node( other ) );
		row_to_wipe = NOT_YET_COMPUTED_ENERGY;
	}

	for (unsigned int ii = 1; ii <= stored_rpes_[ other ].size2(); ++ii)
	{
		stored_rpes_[ other ]( state, ii ) = NOT_YET_COMPUTED_ENERGY;
	}

}

float
LinearMemEdge::get_energy_for_alt_state
(
	bool store_rpes,
	int changing_node_index,
	int alternate_state,
	int alternate_state_recent_history_index,
	int other_node_curr_state,
	int other_node_state_recent_history_index
)
{
	assert( other_node_curr_state != 0 );

	int const node_changing = changing_node_index == get_node_index( 0 ) ? 0 : 1;
	int const node_not_changing = ! node_changing;

	if ( store_rpes && ! store_rpes_[ node_changing ] )
	{
		wipe( node_changing );
	}
	store_rpes_[ node_changing ] = store_rpes;

	if ( store_rpes_[ node_changing ] && alternate_state_recent_history_index != 0 )
	{
		alt_state_energy_ = stored_rpes_[ node_changing ]
			( other_node_curr_state, alternate_state_recent_history_index );
		if (alt_state_energy_ != NOT_YET_COMPUTED_ENERGY ) return alt_state_energy_;
	}

	if ( store_rpes_[ node_not_changing ] )
	{
		alt_state_energy_ = stored_rpes_[ node_not_changing]
			( alternate_state, other_node_state_recent_history_index );
		if ( alt_state_energy_ != NOT_YET_COMPUTED_ENERGY ) return alt_state_energy_;
	}

	alt_state_energy_ = get_linmem_node( node_changing )->
		compute_pair_energy_for_alternate_state(
		get_edges_position_in_nodes_edge_vector( node_changing ));

	if ( store_rpes_[ node_changing ] && alternate_state_recent_history_index != 0 )
	{
		stored_rpes_[ node_changing ]
			( other_node_curr_state, alternate_state_recent_history_index )
			= alt_state_energy_;
	}
	if ( store_rpes_[ node_not_changing ] )
	{
		stored_rpes_[ node_not_changing ]
			( alternate_state, other_node_state_recent_history_index )
			= alt_state_energy_;
	}

	return alt_state_energy_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::get_two_body_table_size
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int LinearMemEdge::get_two_body_table_size() const
{
	return ( stored_rpes_[ 0 ].size() + stored_rpes_[ 1 ].size() );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::acknowledge_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
void
LinearMemEdge::acknowledge_substitution(
	int substituted_node_index,
	float const curr_state_energy,
	int nodes_new_state,
	SparseMatrixIndex const & nodes_new_state_sparse_info,
	int bumped_recent_history_index,
	int new_state_recent_history_index,
	int neighbors_curr_state
)
{
	int node_substituted = substituted_node_index == get_node_index(0) ? 0 : 1;
	int node_not_substituted = ! node_substituted;

	handle_bumped_recent_history_state_for_node(
		node_substituted,
		node_not_substituted,
		bumped_recent_history_index );

	curr_state_energy_ = curr_state_energy;
	if ( neighbors_curr_state != 0 )
	{
		stored_rpes_[ node_substituted ]
			( neighbors_curr_state, new_state_recent_history_index ) =
			curr_state_energy_;
	}

	get_linmem_node( node_not_substituted )->
		acknowledge_neighbors_state_substitution
		(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		curr_state_energy_,
		nodes_new_state,
		nodes_new_state_sparse_info,
		new_state_recent_history_index
	);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::get_sparse_aa_neighbor_info
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
FArray2D< unsigned char > const &
LinearMemEdge::get_sparse_aa_neighbor_info( )
{
	return sparse_aa_neighbors_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemEdge::print_current_energy
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemEdge::print_current_energy() const
{
	std::cerr << "LinearMemEdge: " << get_node_index( 0 ) << "/" << get_node_index( 1 );
	std::cerr << " energy= " << curr_state_energy_ << std::endl;
}


void
LinearMemEdge::handle_bumped_recent_history_state_for_node
(
	int node_substituted,
	int node_not_substituted,
	int bumped_recent_history_index
)
{
	if ( ! store_rpes_[ node_substituted ] || bumped_recent_history_index == 0 ) return;

	FArray1Da_float row_to_reset(
		stored_rpes_[ node_substituted ](1, bumped_recent_history_index ),
		get_num_states_for_node( node_not_substituted ) );
	row_to_reset = NOT_YET_COMPUTED_ENERGY;
}

void
LinearMemEdge::store_curr_state_energy()
{
	int curr_states[ 2 ];
	int recent_history_ids[ 2 ];
	for (int ii = 0; ii < 2; ++ii)
	{
		curr_states[ ii ] = get_linmem_node( ii )->get_current_state();
		recent_history_ids[ ii ] = get_linmem_node( ii )->get_curr_state_recent_state_id();
		if ( curr_states[ ii ] == 0 ) return;
	}

	for (int ii = 0; ii < 2; ++ii )
	{
		int const other = ! ii;
		if ( store_rpes_[ ii ] )
		{
			stored_rpes_[ ii ]( curr_states[ other ], recent_history_ids[ ii ] ) = curr_state_energy_;
		}
	}

}

void
LinearMemEdge::wipe( int node )
{
	stored_rpes_[ node ] = NOT_YET_COMPUTED_ENERGY;
}

//-------------------------------------------------------------------//


////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::LinearMemoryInteractionGraph
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
LinearMemoryInteractionGraph::LinearMemoryInteractionGraph(
	int numNodes
) : OnTheFlyInteractionGraph( numNodes ),
	first_time_prepping_for_simA_( true ),
	have_not_committed_last_substitution_( false )
{
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::~LinearMemoryInteractionGraph
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
LinearMemoryInteractionGraph::~LinearMemoryInteractionGraph()
{
}


////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::blanket_assign_state_0
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemoryInteractionGraph::blanket_assign_state_0()
{
	have_not_committed_last_substitution_ = false;
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		get_linmem_node( ii )->assign_zero_state();
	}
	total_energy_current_state_assignment_ = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::set_state_for_node
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
LinearMemoryInteractionGraph::set_state_for_node(int node_ind, int new_state)
{
	have_not_committed_last_substitution_ = false;
	get_linmem_node( node_ind )->assign_state( new_state );
	update_internal_energy_totals();
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::set_network_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
LinearMemoryInteractionGraph::set_network_state( FArray1DB_int & node_states)
{
	have_not_committed_last_substitution_ = false;
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		get_linmem_node( ii )->partial_assign_state( node_states( ii ) );
	}
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		get_linmem_node( ii )->complete_state_assignment();
	}
	update_internal_energy_totals();
	//std::cerr << "Set Network State Finished" << std::endl;
	//print_current_state_assignment();
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::consider_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemoryInteractionGraph::consider_substitution(
	int node_ind,
	int new_state,
	float & delta_energy,
	float & prev_energy_for_node
)
{
	if (have_not_committed_last_substitution_)
	{
		get_linmem_node( node_considering_alt_state_ )->
			acknowledge_last_substititon_not_committed();
	}

	node_considering_alt_state_ = node_ind;

	delta_energy = get_linmem_node( node_ind )->
		project_deltaE_for_substitution( new_state, prev_energy_for_node );

	total_energy_alternate_state_assignment_ =
		total_energy_current_state_assignment_ + delta_energy;
	have_not_committed_last_substitution_ = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::commit_considered_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
LinearMemoryInteractionGraph::commit_considered_substitution()
{
	have_not_committed_last_substitution_ = false;
	get_linmem_node( node_considering_alt_state_ )->commit_considered_substitution();

	total_energy_current_state_assignment_ =
		total_energy_alternate_state_assignment_;

	++num_commits_since_last_update_;
	if (num_commits_since_last_update_ == COMMIT_LIMIT_BETWEEN_UPDATES)
	{
		update_internal_energy_totals();
	}

	return total_energy_alternate_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::get_energy_current_state_assignment
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
LinearMemoryInteractionGraph::get_energy_current_state_assignment()
{
	//std::cerr << "Num rotamer pair energy calculations performed: " << LinearMemNode::num_rpe_calcs << std::endl;
	update_internal_energy_totals();
	return total_energy_current_state_assignment_;
}

// @ breif O(1) total energy report.  Protected read access for derived classes.
float LinearMemoryInteractionGraph::get_energy_PD_current_state_assignment()
{
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::get_edge_memory_usage
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
LinearMemoryInteractionGraph::get_edge_memory_usage() const
{
	int sum = 0;
	for (std::list< EdgeBase* >::const_iterator iter = get_edge_list_begin();
		iter != get_edge_list_end(); ++iter)
	{
		sum += ((LinearMemEdge*) *iter)->get_two_body_table_size();
	}
	return sum;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::print_current_state_assignment
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemoryInteractionGraph::print_current_state_assignment() const
{
	std::cerr << "State Assignment: " << std::endl;
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		std::cerr << "Node " << ii << " state " << get_linmem_node(ii)->get_current_state() << std::endl;
		get_linmem_node(ii)->print();
	}

	for (std::list< EdgeBase* >::const_iterator iter = get_edge_list_begin();
		iter != get_edge_list_end(); ++iter)
	{
		((LinearMemEdge*) (*iter))->print_current_energy();
	}
	std::cerr << "Energy: " << total_energy_current_state_assignment_ << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::set_errorfull_deltaE_threshold
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemoryInteractionGraph::set_errorfull_deltaE_threshold( float )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::get_energy_sum_for_vertex_group
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
LinearMemoryInteractionGraph::get_energy_sum_for_vertex_group( int group_id )
{
	float esum = 0;
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		if ( get_vertex_member_of_energy_sum_group( ii, group_id ) )
		{
			esum += get_linmem_node( ii )->get_one_body_energy_current_state();
		}
	}

	for ( std::list< EdgeBase* >::iterator edge_iter = get_edge_list_begin();
		edge_iter != get_edge_list_end(); ++edge_iter)
	{
		int first_node_ind = (*edge_iter)->get_first_node_ind();
		int second_node_ind = (*edge_iter)->get_second_node_ind();

		if ( get_vertex_member_of_energy_sum_group( first_node_ind, group_id )
			&& get_vertex_member_of_energy_sum_group( second_node_ind, group_id ))
		{
			esum += ((LinearMemEdge*) (*edge_iter))->get_current_two_body_energy();
		}
	}

	return esum;
}

void
LinearMemoryInteractionGraph::prepare_for_simulated_annealing()
{
	if ( first_time_prepping_for_simA_ )
	{
		set_recent_history_sizes();
		first_time_prepping_for_simA_ = false;
	}
	InteractionGraphBase::prepare_for_simulated_annealing();

}

bool
LinearMemoryInteractionGraph::build_sc_only_rotamer() const
{
	return true;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::getMemoryUsageInBytes()
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
unsigned int
LinearMemoryInteractionGraph::count_static_memory() const
{
	return sizeof( LinearMemoryInteractionGraph );
}

unsigned int
LinearMemoryInteractionGraph::count_dynamic_memory() const
{
	unsigned int total_memory = OnTheFlyInteractionGraph::count_dynamic_memory();
	return total_memory;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::create_new_node( int node_index, int num_states)
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
NodeBase*
LinearMemoryInteractionGraph::create_new_node( int node_index, int num_states)
{
	return new LinearMemNode( this, node_index, num_states );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::create_new_edge( int index1, int index2)
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
EdgeBase*
LinearMemoryInteractionGraph::create_new_edge( int index1, int index2)
{
	return new LinearMemEdge( this, index1, index2 );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LinearMemoryInteractionGraph::update_internal_energy_totals
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
LinearMemoryInteractionGraph::update_internal_energy_totals()
{
	total_energy_current_state_assignment_ = 0;

	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		total_energy_current_state_assignment_ += get_linmem_node( ii )->
			get_one_body_energy_current_state();
	}

	for (std::list<EdgeBase*>::iterator iter = get_edge_list_begin();
		iter != get_edge_list_end(); ++iter)
	{
		total_energy_current_state_assignment_ +=
			((LinearMemEdge*) *iter)->get_current_two_body_energy();
	}

	num_commits_since_last_update_ = 0;
	return;
}

void
LinearMemoryInteractionGraph::set_recent_history_sizes()
{
	static int const LinMemIG_history_size( get_cmdline_history_size() );

	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		get_linmem_node( ii )->set_recent_history_size( LinMemIG_history_size );
	}
}

int
LinearMemoryInteractionGraph::get_cmdline_history_size()
{
	int const default_history_size = 20;
	int LinMemIG_history_size;
	if ( linmem_ig )
	{
		optional_positive_intafteroption( "linmem_ig", default_history_size, LinMemIG_history_size );
	}
	return LinMemIG_history_size;
}

} // end namespace pack
