// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision:
//  $Date:
//  $Author: yab $


// Rosetta headers
#include "PackerTask.h"
#include "aa_name_conversion.h"
#include "pack.h"
#include "add_pser.h"
#include "DesignMap.h"
#include "aaproperties_pack.h"
#include "design.h"
#include "design_structure.h"
#include "disulfides.h"
#include "dna_ns.h"
#include "favor_residue_ns.h"
#include "files_paths.h"
#include "input_pdb.h"
#include "interface.h"
#include "ligand.h"
#include "make_pdb.h"
#include "maps.h"
#include "maps_ns.h"
#include "minimize.h"
#include "misc.h"
#include "multistate_design.h"
#include "namespace_rotamers_setting.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_rotamer_trie.h"
#include "param_torsion.h"
#include "pdb.h"
#include "pH_main.h"
#include "pose.h"
#include "RotamerOptions.h"
#include "rotamer_trial_energies.h"
#include "rotamer_trials.h"
#include "rotamer_trie_calc_energies.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "start.h"
#include "symmetric_design.h"
#include "termini.h"

// ObjexxFCL headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4Da.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/FArray.io.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/constants.hh>
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/vector1.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <ctime>
#include <fstream>
#include <iostream>
#include <string>


using namespace std;


PackerTask::PackerTask( pose_ns::Pose const & pose ) :
	design_map( pose.total_residue(), pose.res() ),
	mode_for_packer( "design" ),
	optimize_h( false ),
	rotamerize_flag( design::rotamerize ),
	allow_repack_for_packer( pose.total_residue(), true ),
	make_output_file_for_packer(false),
	nloop( design::ndruns ),
	perturb_ligand_rotamers(0.0, 0.0),
	include_current_for_packer( false ),
	include_extra_for_packer( false ),
	extra_rot_for_packer( param::MAX_CHI, pose.total_residue(), 0 ),
	extra_chi_for_packer( param::MAX_CHI, pose.total_residue(), 0.0f ),
	resfile_options( pose.total_residue() ),
	total_residue_( pose.total_residue() ),
	res_( pose.res() )
{ residue_weight_map_.set_numres( total_residue_ ); }

PackerTask::PackerTask(
	pose_ns::Pose const & pose,
	std::string const & mode_in,
	bool const make_output_file_in,
	FArray1D_bool const & allow_repack_in
) :
	design_map( pose.total_residue(), pose.res() ),
	mode_for_packer( mode_in ),
	optimize_h( mode_for_packer == "optimizeH" ? true : false ),
	rotamerize_flag( design::rotamerize ),
	allow_repack_for_packer( pose.total_residue(), false ),
	make_output_file_for_packer( make_output_file_in ),
	nloop( ! make_output_file_for_packer ? 1 : design::ndruns ),
	perturb_ligand_rotamers( 0.0, 0.0 ),
	include_current_for_packer( false ),
	include_extra_for_packer( false ),
	extra_rot_for_packer( param::MAX_CHI, pose.total_residue(), 0 ),
	extra_chi_for_packer( param::MAX_CHI, pose.total_residue(), 0.0f ),
	resfile_options( pose.total_residue() ),
	total_residue_( pose.total_residue() ),
	res_( pose.res() )
{
	for ( int i=1; i<=total_residue_; i++){
		allow_repack_for_packer[ i ] = allow_repack_in(i);
	}
	residue_weight_map_.set_numres( total_residue_ );
}

PackerTask::PackerTask( PackerTask const & last_task)
	:
	design_map( last_task.design_map ),
	mode_for_packer( last_task.mode_for_packer ),
	optimize_h( last_task.optimize_h ),
	rotamerize_flag( last_task.rotamerize_flag ),
	allow_repack_for_packer( last_task.allow_repack_for_packer ),
	make_output_file_for_packer( last_task.make_output_file_for_packer ),
	nloop( last_task.nloop ),
	perturb_ligand_rotamers( last_task.perturb_ligand_rotamers),
	include_current_for_packer( last_task.include_current_for_packer ),
	include_extra_for_packer( last_task.include_extra_for_packer ),
	extra_rot_for_packer( last_task.extra_rot_for_packer ),
	extra_chi_for_packer( last_task.extra_chi_for_packer ),
	resfile_options( last_task.resfile_options ),
	total_residue_( last_task.total_residue_ ),
	res_( last_task.res_ )
{
	residue_weight_map_.set_numres( total_residue_ );
}

PackerTask::PackerTask( int total_residue, FArray1D_int const & res )
:
	design_map( total_residue, res ),
	mode_for_packer( "design" ),
	optimize_h( false ),
	rotamerize_flag( design::rotamerize ),
	allow_repack_for_packer( total_residue, true ),
	make_output_file_for_packer(false),
	nloop( design::ndruns ),
	perturb_ligand_rotamers( 0.0, 0.0 ),
	include_current_for_packer( false ),
	include_extra_for_packer( false ),
	extra_rot_for_packer( param::MAX_CHI, total_residue, 0 ),
	extra_chi_for_packer( param::MAX_CHI, total_residue, 0.0f ),
	resfile_options( total_residue ),
	total_residue_( total_residue ),
	res_( res )
{
	residue_weight_map_.set_numres( total_residue_ );
}

void
PackerTask::set_designmap( DesignMap const & design_map_in){
	design_map = design_map_in;
}

DesignMap const &
PackerTask::get_designmap() const {
	return design_map;
}

DesignMap &
PackerTask::get_designmap() {
	return design_map;
}


void
PackerTask::set_mode(const std::string &mode_in){
	mode_for_packer = mode_in;
}

std::string const &
PackerTask::get_mode() const
{
	return mode_for_packer;
}

void
PackerTask::set_allow_repack(FArray1DB_bool const & allow_repack_in){
	//using namespace param;
	if(allow_repack_for_packer.size()>0){
		allow_repack_for_packer.clear();
	}
	for ( int i=1; i<=total_residue_; i++){
		allow_repack_for_packer.push_back(allow_repack_in(i));
	}
}

utility::vector1<bool> const &
PackerTask::get_allow_repack() const {
	return allow_repack_for_packer;    // return the allow_repack and throws range error exception if it is out of range.
}



////////////////////////////////////////////////////////////////////////////////
/// @begin setup_residues_to_vary
///
/// @brief
///
/// @detailed
///
/// @param	mode - in -
/// @param	optimize_h - [in/out]? -
/// @param	nloop - [in/out]? -
/// @param	allow_repack - [in/out]? -
/// @param	make_output_file - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified yiliu 10-07-05
/////////////////////////////////////////////////////////////////////////////////
void PackerTask::
setup_residues_to_vary()
{

	using namespace design;
	using namespace design_sym;
	using namespace files_paths;
	using namespace interface;
	using namespace param;
	using namespace param_aa;
	using namespace runlevel_ns;
	using namespace termini_ns;
	using namespace utility;

	// jk Use the assorted flags to determine which variants apply in general
	FArray1D_bool default_variant_setting( aaproperties_pack::number_aav_type, false );
	default_variant_setting(aaproperties_pack::aav_base)=true;
	if ( design::try_both_his_tautomers ) {
		default_variant_setting(aaproperties_pack::aav_his_taut)=true;
	}
	if ( use_N_terminus ) {
		default_variant_setting(aaproperties_pack::aav_Nterm)=true;
		default_variant_setting(aaproperties_pack::aav_base)=false;
	}
	if ( use_C_terminus ) {
		default_variant_setting(aaproperties_pack::aav_Cterm)=true;
		default_variant_setting(aaproperties_pack::aav_base)=false;
	}
	if ( design::explicit_h2o || design::hydrate_dna ) {
		default_variant_setting(aaproperties_pack::aav_water)=true;
	}
	if ( get_pH_packing_flag() ) {
		default_variant_setting(aaproperties_pack::aav_pH)=true;
	}
	design_map.set_variant_defaults(default_variant_setting);

	//Initialize design_map --reset
	// jk Note: sets variants to the defaults (determined above)
	for ( int seqpos = 1; seqpos <= total_residue_; ++seqpos ) {
		design_map.fix_sidechain_allowing_variants( seqpos );
	}

	// jk Require that total_residue is divisible by num_clones+1
	if ( (total_residue_ % (num_clones+1)) != 0 ) {
		std::cout << "Problem setting up symmetric design!" << std::endl;
		std::cout << "Total number of residues is not divisible by sym_design_copies!" << std::endl;
		std::cerr << "Problem setting up symmetric design!" << std::endl;
		std::cerr << "Total number of residues is not divisible by sym_design_copies!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// jk Setup "independent" positions first, then copy to symmetry-related positions
	int const total_residue_indep=total_residue_/(num_clones+1);

	//bk set up which residue positions to is_protein run over
	if ( mode_for_packer == "optimizeH" ) {

		optimizeH_mode();

	} else if ( mode_for_packer == "packrot" ) {

		packrot_mode();

	} else if ( mode_for_packer == "design" && use_design_matrix ) {

		design_matrix_mode();

  } else if ( mode_for_packer == "design" && ( design_inter || build_Ala_interface ) ) {
		design_inter_mode();

	} else if ( mode_for_packer == "design" && natrot ) {

		design_natrot_mode();

	} else if ( mode_for_packer == "design" && mut_list ) {

		design_mutlist_mode();

		//mj if ligand_flag treat as ALLAA for binding site
		//mj and NATAA for non-binding site
	} else if ( mode_for_packer == "design" && get_ligand_flag() && ! pack_in_parallel ) {

		design_ligand_mode();

	} else if ( mode_for_packer == "design" ) {

		if ( resfile == "none" ) {
			if ( dna_interface ) auto_dnaint_mode();
			else design_mode();
		}
		else design_natrot_mode();

	} else if ( mode_for_packer	== "packcys" ) {

		packcys_mode();

	} else {
		std::cout << "pack called with non-standard mode " << mode_for_packer <<
		 std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

  //jk disallow Cys and Gly, unless present in the current structure
  if ( no_new_CG ) {
    for ( int seqpos = 1; seqpos <= total_residue_; ++seqpos ) {
      if ( res_(seqpos) != aa_cys ) design_map.disable(seqpos,aa_cys);
      if ( res_(seqpos) != aa_gly ) design_map.disable(seqpos,aa_gly);
    }
  }

	if ( ( mode_for_packer == "design" ) && ( resfile != "none" ) && ! use_design_matrix ) {
		load_resfile();
  }

	//jjh exclude nucleic acids except for hydrogen optimization
	//ja turn off if expecting DNA rotamers
	for ( int seqpos = 1; seqpos <= total_residue_; ++seqpos ) {
		if ( is_NA(res_(seqpos)) && ( mode_for_packer != "optimizeH" ) &&
			   !design::hydrate_dna && !active_rotamer_options.dna_rotamers_exist ) {
			design_map.fix_sidechain_allowing_variants( seqpos );
		}
	}

	///jjh Allow water variants on DNA when hydrate_dna is true
	if( design::hydrate_dna && !explicit_h2o &&
			(mode_for_packer == "design" || mode_for_packer == "packrot")) {
		for ( int seqpos = 1; seqpos <= total_residue_; ++seqpos ) {
			if ( !is_DNA(res_(seqpos)) ) {
				design_map.disallow_variant_type_at_residue( aaproperties_pack::aav_water, seqpos );
//			design_map.allow_variant_type_at_residue( aaproperties_pack::aav_base, seqpos );
			}
		}
	}

	// Restrict "terminus" variants to the termini, His tautomers to places where His is allowed
	for ( int seqpos = 1; seqpos <= total_residue_; ++seqpos ) {
		if ( ! is_N_terminus(seqpos) ) {
			design_map.disallow_variant_type_at_residue(aaproperties_pack::aav_Nterm, seqpos);
		}
		if ( ! is_C_terminus(seqpos) ) {
			design_map.disallow_variant_type_at_residue(aaproperties_pack::aav_Cterm, seqpos);
		}
		if ( ! design_map.get(seqpos,aa_his) ) {
			design_map.disallow_variant_type_at_residue(aaproperties_pack::aav_his_taut, seqpos);
		}
	}

	// jk If using symmetry, copy design map contents to symmetric positions
	if ( num_clones > 0 ) {
		for ( int i = 1; i <= total_residue_; ++i ) {
			if (clone_follows(i) != 0) {
				design_map.copy_res(clone_follows(i),i);
			}
		}
	}

	//apl THIS SHOULD GO
	set_design_matrix(design_map,total_residue_indep);
}



////////////////////////////////////////////////////////////////////////////////
/// @begin load_resfile
///
/// @brief
///
/// Performs the file i/o to read in the resfile, which specifies the
/// sequence positions that are to be repacked/designed.
///
/// @detailed
///
/// @param[in]	 resfile - in
/// This paramter gives the file to read in.
///
/// @param[in]	 whichres - in
/// @param[in]	 allow_repack_for_packer - in	instead of namespace whichres
/// This array is a mask for the rest of the resfile selection stuff -
/// if whichres is false for a position, it is held fixed.	This is probably
/// used to cycle through structural positions and design each in turn.
///
///
/// @global_read
///
/// A minimal amount of global data is accessed, such as amino acid
/// numbers and characteristics (polar/apolar);
///
/// @global_write
///
/// No global data is altered by the function.
///
/// @remarks
///
/// The documentation for the format of a resfile is elsewhere (Brian's
/// readme.design, I think).
///
/// @references
///
/// @authors
/// zforce am 11-04
/// jjh 8-19-03
/// yiliu 10-7-05
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void PackerTask::
load_resfile( std::string filepath )
{
	using namespace design;
	using namespace design_sym;
	using namespace files_paths;
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;

	if ( filepath == "none" ) return;

	std::cout << "Loading design/repack settings from resfile " << filepath <<
	 std::endl;

	data_x.clear();
	if ( filepath == "auto" ) data_x.open( start_path + start_file + ".resfile" );
	else data_x.open( filepath );

	if ( !data_x ) {
		std::cout << "Open failed for file: " << data_x.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//bk read through header information
//	for ( int i = 1; i <= 26; ++i ) data_x >> skip;

	//ja key on " start" instead of starting at arbitrary line #26
	std::string header = "header";
	while ( header != " start" && !data_x.eof() ) {
		data_x >> bite( 6, header ) >> skip;
//	std::cout << header << std::endl;
	}

	if ( header != " start" ) {
		std::cout << "Error: could't find start of resfile" << std::endl;
	}

	int seqpos;
	std::string charid;
	FArray1D_char aachar( MAX_AA(), ' ' );

	for ( int ii = 1; ii <= (total_residue_/(num_clones+1)); ++ii ) {
		data_x >> skip(3) >> bite(4,seqpos) >> skip(6) >> bite(5,charid) >> skip(2);

		for ( int i = 1, ie = MAX_AA(); i <= ie; ++i ) {
			data_x >> bite( aachar(i) );

			// allow comments after this point
			if ( aachar(i) == '#' ) { aachar(i) = ' '; break; }
		}

		data_x >> skip;
		//		if ( seqpos != ii ) {
		//			std::cout << "problem reading resfile" << std::endl;
		//			std::cout << "seqpos is " << seqpos << " and ii is " << ii << std::endl;
		//			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		//		}
		resfile_options[seqpos] = charid;
		//bk if whichres false then overide resfile and place native rotamer
		if(!allow_repack_for_packer[seqpos]){
			design_map.fix_completely( seqpos );

//bk NATRO, use native amino acid and rotamer
		} else if ( charid == "NATRO" ) {
			design_map.fix_sidechain_allowing_variants( seqpos );

//bk ALLAA, allow all amino acids
		} else if ( charid == "ALLAA" ) {
			for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) design_map.set( seqpos, aa );

//jk NATVA, use native amino acid and rotamer AND variant
		} else if ( charid == "NATVA" ) {
			design_map.fix_completely( seqpos );

//bk NATAA, use native amino acid, but allow different rotamers
		} else if ( charid == "NATAA" ) {
			design_map.fix_sidechain_allowing_variants( seqpos );
			design_map.set( seqpos, res_(seqpos) );

//bk PIKAA, read off the amino acids to be included
		} else if ( charid == "PIKAA" ) {
			design_map.fix_sidechain_allowing_variants( seqpos );
			for ( int j = 1, je = MAX_AA(); j <= je; ++j ) {
				if ( aachar(j) != ' ' ) {
					int aa;
					num_from_res1(aachar(j),aa);
					design_map.set( seqpos, aa );
				}
			}

		} else if ( charid == "ADDAA" ) {
			for ( int j = 1, je = MAX_AA(); j <= je; ++j ) {
				if ( aachar(j) != ' ' ) {
					int aa;
					num_from_res1(aachar(j),aa);
					design_map.set( seqpos, aa );
				}
			}

		} else if ( charid == "NOTAA" ) {
			if ( design_map.num_allowed_aa(seqpos) > 0 ) {
				for ( int j = 1, je = MAX_AA(); j <= je; ++j ) {
					if ( aachar(j) != ' ' ) {
						int aa;
						num_from_res1(aachar(j),aa);
						design_map.disable(seqpos,aa);
					}
				}
				if ( ( design_map.num_allowed_aa(seqpos) == 0 ) && ( build_Ala_interface ) ) {
					design_map.set(seqpos,aa_ala);
				}
			}

//bk POLAR, use only polar amino acids
		} else if ( charid == "POLAR" ) {
			for ( int i = 1; i <= MAX_AUTH_AA; ++i ) {
				if ( aa_is_polar(i) ) design_map.set( seqpos, i );
			}

//bk APOLA, use apolar amino acids
		} else if ( charid == "APOLA" ) {
			for ( int i = 1; i <= MAX_AUTH_AA; ++i ) {
				if ( aa_is_nonpolar(i) ) design_map.set( seqpos, i );
			}

//cmd APOGR, Grow out apolar amino acids at apolar positions
		} else if ( charid == "APOGR" ) {
			design_map.fix_sidechain_allowing_variants( seqpos );
			if ( aa_is_nonpolar(res_(seqpos)) ) {
				for ( int j = 1; j <= MAX_AUTH_AA; ++j ) {
					if ( aa_is_nonpolar(j) && nheavyatoms(j,1) >=
							 nheavyatoms(res_(seqpos),1) ) {
						design_map.set( seqpos, j );
					}
				}
			}

//ja automatic behavior: all 20 amino acids, pending later criteria
		} else if ( charid == "AUTOM" ) {
			design_map.fix_sidechain_allowing_variants( seqpos );
			automatic_behavior( seqpos ) = true;
			for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) design_map.set( seqpos, aa );

//ja automatic behavior: native rotamers only, unless amino acids are given
		} else if ( charid == "AUTOP" ) {
			design_map.fix_sidechain_allowing_variants( seqpos );
			automatic_behavior( seqpos ) = true;
			if ( aachar(1) == ' ' ) { // just native
				//design_map.set( seqpos, misc::res_(seqpos) );
				design_map.set( seqpos, res_(seqpos) );
			} else { // you pick
				for ( int j = 1, je = aachar.size1(); j <= je; ++j ) {
					if ( aachar(j) != ' ' ) {
						int aa;
						num_from_res1(aachar(j),aa);
						design_map.set( seqpos, aa );
					}
				}
			}

		// DNA resfile options
		} else if ( charid == "ALLNA" || charid == "PIKNA" ||
		            charid == "PIKBA" || charid == "DESBA" ) {
			if ( !dna_interface || !is_NA( res_(seqpos) ) ) {
				bad_resfile_flag( charid, seqpos, "not NA, or not in DNA mode" );
			} else {
				// build all DNA rotamers
				if ( charid == "ALLNA" ) {
					for ( int na(1); na <= MAX_AA(); ++na ) {
						if ( is_DNA(na) ) design_map.set( seqpos, na );
					}
				// select particular DNA type(s)
				// PIKBA signals direct mutation; PIKNA rotamerizes the nucleotide
				} else {
					for ( int j(1), je = aachar.size1(); j <= je; ++j ) {
						if ( aachar(j) == ' ' ) continue;
						if ( (aachar(j) != 'A') && (aachar(j) != 'C') &&
						     (aachar(j) != 'G') && (aachar(j) != 'T') ) {
							std::cout << aachar(j) << " is not DNA! Skipping" << std::endl;
						} else {
							// convert to three-letter code ('  A') and then look up aa num
							std::string na_name( 3, ' ' );
							na_name[2] = aachar(j);
							int aa;
							num_from_res3( na_name, aa );
							design_map.set( seqpos, aa );
						}
					}
				}
				if ( charid == "ALLNA" || charid == "PIKNA" ) {
					active_rotamer_options.dna_rotamers_exist = true;
					std::cout << "DNA rotamers at pos " << pdb::pdb_res_num(seqpos) <<
					 std::endl;
				}
			}
			// (note: DESBA is passive for now, but triggers behavior later on)

		// Unknown instruction
		} else {
			// jk Ignore "DEFLT" keyword
			if ( charid != "DEFLT" ) {
				bad_resfile_flag( charid, seqpos, "flag not recognized" );
			}
		}

		//bs ensure that cys never used unless in native or chosen by PIKAA.
		//cmd except cysteines are allowed at dna interfaces
		if ( charid != "PIKAA" && charid != "NATAA" && charid != "NATRO" ) {
			if ( !dna_interface && !get_ligand_flag() ) {
				design_map.disable( seqpos, aa_cys );
			}
		}
	}
	data_x.close();
	data_x.clear();

	//ja make sure that we do not mistakenly design with phosphoserines
	// perhaps phosphoserine would be better represented as a serine variant...?
	if ( MAX_AA() > 20 && !add_phosphoser::aa_pser ) {
		design_map.disable(seqpos,aa_sep);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin bad_resfile_flag
///
/// @brief
/// exception upon bad flag found in resfile
///
/// @author
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
void
PackerTask::bad_resfile_flag(
	std::string const charid,
	int const seqpos,
	std::string const message
)
{
	using namespace pdb;

	std::cout << "Error: invalid " << charid << " flag at seqpos " <<
	 seqpos << " (pdb " << pdb_res_num(seqpos) << "): " << message << std::endl;

	utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );
}

/////////////////////////////////////////////////////
//set Task
void
PackerTask::set_task(
	std::string const & mode_in,
	bool const  make_output_file_in,
	FArray1DB_bool const & allow_repack_in,
	bool const include_current_in,
	bool const update_favored_residues_in
)
{
	//static FArray2D_int const extra_rot( param::MAX_CHI, param::MAX_RES(), 0 );
	//static FArray2D_float const extra_chi( param::MAX_CHI, param::MAX_RES(), 0.0f );

	//set_task( mode_in, make_output_file_in, allow_repack_in, include_current_in,
   //         false, extra_rot, extra_chi );
	using namespace param;

	mode_for_packer = mode_in;
	make_output_file_for_packer = make_output_file_in;

	for ( int i=1; i<=total_residue_; i++){
		allow_repack_for_packer[ i ] = allow_repack_in(i);
	}

	//ja update what is considered "native", for favor_native_residue bonus.  If
	// doing multiple cycles of design, you probably DO NOT want this for any runs
	// after the first one, so call set_task() for those with
	// update_favored_residues_in = false.
	include_current_for_packer = include_current_in;
		if ( update_favored_residues_in ) {
		for ( int pos = 1; pos <= total_residue_; ++pos ) {
			//favor_residue::res_native(pos) = misc::res(pos);
			favor_residue::res_native(pos) = res_(pos);
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
//set Task
void PackerTask::set_task(
	std::string const & mode_in,
	bool const make_output_file_in,
	FArray1DB_bool const & allow_repack_in,
	bool const include_current_in,
	bool const include_extra_in,
	FArray2DB_int const & extra_rot_in,
	FArray2DB_float const & extra_chi_in,
	bool const update_favored_residues_in // default true
){
	set_task( mode_in, make_output_file_in, allow_repack_in,
		include_current_in, update_favored_residues_in );

	include_extra_for_packer = include_extra_in;
	extra_rot_for_packer = extra_rot_in;
	extra_chi_for_packer = extra_chi_in;

}



////////////////////////////////////////////////////////////////////////////////
/// @begin optimizeH_mode
///
/// @brief
///  perform the behaviors when the mode_for_packer == optimizeH
///
/// @detailed
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified yiliu 10-31-06
/////////////////////////////////////////////////////////////////////////////////
void PackerTask::
optimizeH_mode(){
	using namespace design_sym;
	using namespace param;
	using namespace param_aa;

	optimize_h = true;
	// jk Setup "independent" positions first, then copy to symmetry-related positions
	int const total_residue_indep=total_residue_/(num_clones+1);
	nloop = 1;
	for ( int seqpos = 1; seqpos <= total_residue_indep; ++seqpos ) {
		if ( !is_protein(res_(seqpos)) && !is_nonnatural(res_(seqpos)) ) continue;

		for ( int aa = 1, aae = MAX_AA(); aa <= aae; ++aa ) {
			if (aa == res_(seqpos)) design_map.set( seqpos, aa );
			else design_map.disable( seqpos, aa );
		}
	}
	// jk Don't allow water during optimizeH
	design_map.disallow_variant_type(aaproperties_pack::aav_water);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin packrot_mode
///
/// @brief
///  perform the behaviors when the mode_for_packer == packrot
///
/// @detailed
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified yiliu 10-31-06
/////////////////////////////////////////////////////////////////////////////////
void PackerTask::packrot_mode()
{
	using namespace design_sym;
	using namespace disulfides::BOUNDARY;
	using namespace param;
	using namespace param_aa;

	if ( !make_output_file_for_packer) nloop = 1;
	for ( int seqpos = 1; seqpos <= total_residue_; ++seqpos )
	{
		if ( ( get_norepack_disulf() && cys_res_in_disulf(seqpos) ) ||
				 ! allow_repack_for_packer[seqpos] ||
				 ( num_clones > 0 && clone_follows(seqpos) > 0 ) ) {

			// jk Don't allow rotamer variants for fixed pieces during packrot
			design_map.fix_completely( seqpos );

		} else {
			if ( !is_protein(res_(seqpos)) && !is_nonnatural(res_(seqpos)) && !is_ligand(res_(seqpos))) continue;
			for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
				if ( !is_protein(aa) && !is_nonnatural(aa) && !is_ligand(aa)) continue;
				else if (aa==res_(seqpos)) design_map.set( seqpos, aa );
				else design_map.disable( seqpos, aa );
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////////
/// @begin design_matrix_mode
///
/// @brief
///  perform the behaviors when the mode_for_packer ==  "design" && use_design_matrix
///
/// @detailed
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified yiliu 10-31-06
//////////////////////////////////////////////////////////////////////////////////////
void PackerTask::design_matrix_mode(){

	using namespace design;
	using namespace design_sym;
	using namespace param;
	using namespace param_aa;

	// jk Setup "independent" positions first, then copy to symmetry-related positions
	int const total_residue_indep=total_residue_/(num_clones+1);

	for ( int seqpos = 1; seqpos <= total_residue_indep; ++seqpos ) {
		design_map.fix_sidechain_allowing_variants( seqpos );
		for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
			if ( !is_protein(aa) && !is_nonnatural(aa) ) continue;
			else if( design_matrix(aa,seqpos) ) design_map.set( seqpos, aa );
			else design_map.disable( seqpos, aa );
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////////
/// @begin design_natrot_mode
///
/// @brief
///  perform the behaviors when the mode_for_packer ==  "design" && natrot
///
/// @detailed
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified yiliu 10-31-06
//////////////////////////////////////////////////////////////////////////////////////
void PackerTask::
design_natrot_mode(){

	using namespace design_sym;

	// jk Setup "independent" positions first, then copy to symmetry-related positions
	int const total_residue_indep=total_residue_/(num_clones+1);

	std::cout << "all residues held in native coordinates " <<
		total_residue_indep << std::endl;

	for ( int seqpos = 1; seqpos <= total_residue_indep; ++seqpos ) {
		design_map.fix_sidechain_allowing_variants( seqpos );
	}
}


//////////////////////////////////////////////////////////////////////////////////////
/// @begin design_mutlist_mode
///
/// @brief
///  perform the behaviors when the mode_for_packer ==  "design" && mutlist
///
/// @detailed
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified yiliu 10-31-06
//////////////////////////////////////////////////////////////////////////////////////
void PackerTask::
design_mutlist_mode(){
	using namespace design;
	using namespace design_sym;
	using namespace param;
	using namespace param_aa;

	// jk Setup "independent" positions first, then copy to symmetry-related positions
	int const total_residue_indep=total_residue_/(num_clones+1);

	for ( int seqpos = 1; seqpos <= total_residue_indep; ++seqpos ) {
		for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
			if ( !is_protein(aa) && !is_nonnatural(aa) ) continue;
			if (design_matrix(aa,seqpos)) design_map.set( seqpos, aa );
				else design_map.disable( seqpos, aa );
		}
	}
}



//////////////////////////////////////////////////////////////////////////////////////
/// @begin design_inter_mode
///
/// @brief
///  perform the behaviors when the mode_for_packer ==  "design" && design_inter
///
/// @detailed
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified yiliu 10-31-06
//////////////////////////////////////////////////////////////////////////////////////
void PackerTask::
design_inter_mode(){

	using namespace design;
	using namespace design_sym;
	using namespace param;
	using namespace param_aa;

	// jk Setup "independent" positions first, then copy to symmetry-related positions
	int const total_residue_indep=total_residue_/(num_clones+1);

	for ( int seqpos = 1; seqpos <= total_residue_indep; ++seqpos ) {
		if ( allow_repack_for_packer[seqpos] ) {
			if ( is_protein(res_(seqpos)) || is_nonnatural(res_(seqpos)) ) {

				// Check whether this seqpos is on a "repack only" chain
				bool vary_sequence_at_seqpos = true;
				if ( repack_inter ) {
					vary_sequence_at_seqpos = false;
				} else {
					if ( fix_target_seq != 0 ) {
						if ( seqpos <= misc::domain_end(fix_target_seq) ) {
							if ( fix_target_seq == 1 ) {
								vary_sequence_at_seqpos = false;
							} else if ( seqpos > misc::domain_end(fix_target_seq-1) ) {
								vary_sequence_at_seqpos = false;
							}
						}
					}
				}

				if ( vary_sequence_at_seqpos ) {

					// Allow the sequence to vary at this seqpos

					if ( build_Ala_interface ) {
						// jk allow only Ala if we're building an Ala interface,
						// jk if the current residue is not Gly or Pro
						if ( ( res_(seqpos) == aa_gly ) || ( res_(seqpos) == aa_pro ) ) {
							design_map.fix_sidechain_allowing_variants(seqpos);
							design_map.set(seqpos,res_(seqpos));
						} else {
							design_map.fix_sidechain_allowing_variants(seqpos);
							design_map.set(seqpos,aa_ala);
						}

					} else {
						// Use all AAs at this position
						for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
							design_map.set(seqpos,aa);
						}
						// jk don't put in new Cys except for dna interface
						if ( !dna_interface && !get_ligand_flag() ) {
							design_map.disable(seqpos,aa_cys);
						}
					}

				} else {

					// Repack only at this seqpos
					for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
						if ( is_protein(aa) || is_nonnatural(aa) ) {
							if ( aa == res_(seqpos) ) design_map.set(seqpos,aa);
							else design_map.disable(seqpos,aa);
						}
					}

				}

			}
		} else {
			// Use the native rotamer here, don't allow rotamer variants
			design_map.fix_completely(seqpos);
		}
	}

}


//////////////////////////////////////////////////////////////////////////////////////
/// @begin design_ligand_mode
///
/// @brief
///  perform the behaviors when the mode_for_packer ==  "design" &&
///                                        get_ligand_flag() && ! pack_in_parallel )
///
/// @detailed
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified yiliu 10-31-06
//////////////////////////////////////////////////////////////////////////////////////
void PackerTask::
design_ligand_mode(){

	using namespace design_sym;
	using namespace interface;
	using namespace param;
	using namespace param_aa;
	using namespace runlevel_ns;


	// jk Setup "independent" positions first, then copy to symmetry-related positions
	int const total_residue_indep=total_residue_/(num_clones+1);

	for ( int seqpos = 1; seqpos <= total_residue_indep; ++seqpos ) {
		if ( allow_repack_for_packer[seqpos] && int_res(seqpos) ) {
			for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
				if ( is_protein(aa) || is_nonnatural(aa) ) {
					if( int_res(seqpos) || aa == res_(seqpos) ){
						design_map.set( seqpos, aa );
					}
					else design_map.disable( seqpos, aa );
				}
			}
		} else {
			design_map.fix_completely( seqpos );
		}
		if ( runlevel > standard ) std::cout << SS( seqpos ) << ' ' <<
																 SS( allow_repack_for_packer[seqpos] ) << ' ' << SS( int_res(seqpos) ) << std::endl;
	}
	//mj end
}


//////////////////////////////////////////////////////////////////////////////////////
/// @begin design_mode
///
/// @brief
///  perform the behaviors when the mode_for_packer ==  "design"
///
/// @detailed
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified yiliu 10-31-06
//////////////////////////////////////////////////////////////////////////////////////
void PackerTask::design_mode(){

	using namespace design;
	using namespace design_sym;
	using namespace param;
	using namespace param_aa;


	// jk Setup "independent" positions first, then copy to symmetry-related positions
	int const total_residue_indep=total_residue_/(num_clones+1);
	for ( int seqpos = 1; seqpos <= total_residue_indep; ++seqpos ) {
			if ( allow_repack_for_packer[seqpos] ){
				if ( is_protein(res_(seqpos)) || is_nonnatural(res_(seqpos)) ){
					if ( res_(seqpos) == aa_cys ) {
						if ( !dna_interface && !get_ligand_flag() || ( pack_in_parallel && get_ligand_flag() ) ) {
							design_map.fix_completely( seqpos );
						}
					} else {
						for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
							if ( is_protein(aa) || is_nonnatural(aa) ) {
								design_map.set( seqpos, aa );
								if ( aa == aa_cys ) { // for now not designing disulfides
									if ( !dna_interface && !get_ligand_flag() || ( pack_in_parallel && get_ligand_flag() ) ) {
										design_map.disable( seqpos, aa );
									}
								}
							}
						}
					}
				}

			} else {
				design_map.fix_completely( seqpos );
			}
	}
}

//////////////////////////////////////////////////////////////////////////////////////
/// @begin packcys_mode
///
/// @brief
///  perform the behaviors when the mode_for_packer == "packcys"
///
/// @detailed
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified yiliu 10-31-06
//////////////////////////////////////////////////////////////////////////////////////
void PackerTask::
packcys_mode(){

	using namespace design_sym;
	using namespace param;
	using namespace param_aa;

	// jk Setup "independent" positions first, then copy to symmetry-related positions
	int const total_residue_indep = total_residue_/(num_clones+1);

	for ( int seqpos = 1; seqpos <= total_residue_indep; ++seqpos ) {
		if ( res_(seqpos) != aa_cys ) {
			//res_(seqpos) = aa_gly; //WHAT?!
			design_map.fix_sidechain_allowing_variants( seqpos );
		} else {
			for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
				if ( !is_protein(aa) && !is_nonnatural(aa) ) continue;
				if (aa == res_(seqpos)) design_map.set( seqpos, aa );
				else design_map.disable( seqpos, aa );
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////////
/// @begin auto_dnaint_mode
///
/// @brief
/// set up designmap for design at rosetta-determined dna interface
///
/// @authors
/// ashworth
///
//////////////////////////////////////////////////////////////////////////////////////
void PackerTask::
auto_dnaint_mode(){

	using namespace design_sym;
	using namespace param;
	using namespace param_aa;

	// jk Setup "independent" positions first, then copy to symmetry-related positions
	int const total_residue_indep=total_residue_/(num_clones+1);

	for ( int seqpos = 1; seqpos <= total_residue_indep; ++seqpos ) {
		if ( is_protein( res_(seqpos) ) ) {
			design_map.fix_sidechain_allowing_variants( seqpos );
			design::automatic_behavior( seqpos ) = true;
			for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) design_map.set( seqpos, aa );
		}
	}
}
