// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

//xa   Implementation for new simulated annealing routine that will optimize
//xa   a single sequence for multiple structures (Ambroggio & Kuhlman 2006)

// Rosetta Headers
#include "design.h"              // needed for conv_limit_mod
#include "files_paths.h"         // needed for reading in equiv_resfile
#include "EquivResfileReader.h"
#include "InteractionGraphBase.h" // needed for calling ig functions
#include "FixbbSimAnnealer.h"
#include "misc.h"                // needed for res_num_from_pdb_res_num_chain
#include "MultiCoolAnnealer.h"
#include "pack.h"                // needed for rotindex
#include "ParallelSimAnneal.h"
#include "param.h"               // needed for MAX_RES
#include "pdb.h"                 // needed for res_num_from_pdb_res_num_chain
#include "ResidueData.h"
#include "RotamerSet.h"
#include "SimAnnealerBase.h"
#include "XUtilities.cc"

// C++ Headers
#include <fstream>
#include <iostream>
#include <set>

//Utility Headers
#include <utility/basic_sys_util.hh>

namespace pack{



  ////////////////////////////////////////////////////////////////////////////////

  ParallelSimAnnealBase::ParallelSimAnnealBase(
				       const RotamerSet * p_rotamer_set,
				       FArray2DB_bool & neighborlist,
				       FArray1D_int & bestrotamer_at_seqpos,
				       float & bestenergy,
				       bool start_with_current,
				       pack::InteractionGraphBase * ig,
				       FArray1DB_int & current_rot_index,
				       bool calc_rot_freq,
				       FArray1D_float & rot_freq
				       ):
    SimAnnealerBase(
		    num_of_rot_to_pack_,
		    bestrotamer_at_seqpos,
		    bestenergy,
		    start_with_current, // start simulation with current rotamers
		    p_rotamer_set_,
		    current_rot_index,
		    calc_rot_freq,
		    rot_freq
		    ), ig_(ig),
    neighborlist_(neighborlist),
    p_rotamer_set_(p_rotamer_set)
  {
    using namespace param;
    rot_to_pack.reserve( p_rotamer_set_->nrotamers() );
    start_with_current_ = false;
    mutant_sequence_positions.clear();
    previous_energy = bestenergy_ = delta_energy = energy_of_best_run = 0;

    read_equiv_resfile();
    create_residue_data_map();
    add_neighbor_info_to_master_sequence_map();
    create_list_of_key_sequence_positions();
    best_rotamers_ever.clear();
    for( unsigned int i = 1; i <= bestrotamer_at_seqpos_.size(); i++ ) best_rotamers_ever.push_back( bestrotamer_at_seqpos_(i) );
  }

  ////////////////////////////////////////////////////////////////////////////////

  ParallelSimAnnealBase::~ParallelSimAnnealBase()
  {}

  void ParallelSimAnnealBase::print_iteration_status()
  {
    cout << "Current temperature:\t" << get_temperature() << " Lowest energy:\t" << energy_of_best_run << endl;
    cout << "Current energy:\t" << bestenergy_ << " Previous energy:\t" << previous_energy << " Delta energy:\t" << delta_energy << endl;
  }

  void ParallelSimAnnealBase::fill_rot_to_pack_for_current_sequence()
  {
    rot_to_pack.clear();
    list<int> rotamers_for_seq_pos;
    std::map<int, ResidueData>::iterator intResDataMap_iter;

    for( intResDataMap_iter = master_sequence_map.begin(); intResDataMap_iter != master_sequence_map.end(); intResDataMap_iter++)
      if( intResDataMap_iter->second.return_possible_rotamers_for_current_amino_acid( rotamers_for_seq_pos ) )
	copy( rotamers_for_seq_pos.begin(), rotamers_for_seq_pos.end(), back_inserter( rot_to_pack ) );
  }

  void ParallelSimAnnealBase::add_current_rotamers_and_start_with_them()
  {
    int current_rotamer;
    std::map<int, ResidueData>::iterator intResDataMap_iter;

    for( intResDataMap_iter = master_sequence_map.begin(); intResDataMap_iter != master_sequence_map.end(); intResDataMap_iter++)
      if( intResDataMap_iter->second.get_current_rotamer( current_rotamer ) )
	current_rot_index_( intResDataMap_iter->first ) = current_rotamer;

    start_with_current_ = true;
  }

  void ParallelSimAnnealBase::add_current_rotamers_for_packing()
  {
    std::map<int, ResidueData>::iterator intResDataMap_iter;
    int current_rotamer;

    for( intResDataMap_iter = master_sequence_map.begin(); intResDataMap_iter != master_sequence_map.end(); intResDataMap_iter++)
      if( intResDataMap_iter->second.get_current_rotamer( current_rotamer ) )
	if( find( rot_to_pack.begin(), rot_to_pack.end(), current_rotamer ) == rot_to_pack.end() )
	  rot_to_pack.push_back( current_rotamer );
  }

  void ParallelSimAnnealBase::print_current_DNA_sequence()
  {
    string current_codon;
    std::map<int, ResidueData>::iterator intResDataMap_iter;
    for( intResDataMap_iter = master_sequence_map.begin(); intResDataMap_iter != master_sequence_map.end(); intResDataMap_iter++)
      if( intResDataMap_iter->second.get_curr_codon( current_codon ) ) cout << current_codon;
    cout << endl;
  }

  void ParallelSimAnnealBase::print_current_amino_acid_sequence()
  {
    int current_amino_acid;
    std::map<int, ResidueData>::iterator intResDataMap_iter;
    for( intResDataMap_iter = master_sequence_map.begin(); intResDataMap_iter != master_sequence_map.end(); intResDataMap_iter++)
      if( intResDataMap_iter->second.get_curr_aa( current_amino_acid ) ) cout << " " << ResidueData::AminoAcidLookup.find( current_amino_acid )->second << " ";
    cout << endl;
  }

  void ParallelSimAnnealBase::print_current_sequences()
  {
    print_current_DNA_sequence();
    print_current_amino_acid_sequence();
  }

void ParallelSimAnnealBase::pack_set_of_rotamers()
{
		if( rot_to_pack.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

		SimAnnealerBase * sa;
		if ( use_multi_cool_annealer )
		{
			sa = new MultiCoolAnnealer(  rot_to_pack, bestrotamer_at_seqpos_, bestenergy_,
				start_with_current_, ig_, p_rotamer_set_,
				current_rot_index_, calc_rot_freq_, rot_freq_);
		}
		else {
			sa = new FixbbSimAnnealer(  rot_to_pack, bestrotamer_at_seqpos_, bestenergy_,
				start_with_current_, ig_, p_rotamer_set_,
				current_rot_index_, calc_rot_freq_, rot_freq_);
		}

		sa->run();

		delete sa;
		rot_to_pack.clear();
		start_with_current_ = false;
}

  void ParallelSimAnnealBase::accept_new_sequence_from_packing_run()
  {
    std::map<int, ResidueData>::iterator intResDataMap_iter;
    for( intResDataMap_iter = master_sequence_map.begin(); intResDataMap_iter != master_sequence_map.end(); intResDataMap_iter++)
      intResDataMap_iter->second.accept_new_state( bestrotamer_at_seqpos_( intResDataMap_iter->first ) );

    if( bestenergy_ < energy_of_best_run ){
      best_rotamers_ever.clear();
      for( unsigned int i = 1; i <= bestrotamer_at_seqpos_.size(); i++ ) best_rotamers_ever.push_back( bestrotamer_at_seqpos_(i) );
      energy_of_best_run = bestenergy_;

      for( intResDataMap_iter = master_sequence_map.begin(); intResDataMap_iter != master_sequence_map.end(); intResDataMap_iter++)
	intResDataMap_iter->second.set_best_state();
    }
     previous_energy = bestenergy_;
  }

  void ParallelSimAnnealBase::output_best_rotamers()
  {
    if( bestrotamer_at_seqpos_.size() != best_rotamers_ever.size() || best_rotamers_ever.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);    for( unsigned int i = 1; i <= bestrotamer_at_seqpos_.size(); i++ ) bestrotamer_at_seqpos_(i) = best_rotamers_ever.at( i - 1 );
  }

  void ParallelSimAnnealBase::add_neighbor_info_to_master_sequence_map()
  {

    std::list<int> neighbor_list;
    std::map<int, ResidueData>::iterator intResDataMap_iter1, intResDataMap_iter2;

    for( intResDataMap_iter1 = master_sequence_map.begin(); intResDataMap_iter1 != master_sequence_map.end(); intResDataMap_iter1++){
      neighbor_list.clear();
      for( intResDataMap_iter2 = master_sequence_map.begin(); intResDataMap_iter2 != master_sequence_map.end(); intResDataMap_iter2++){
	if( intResDataMap_iter1->first != intResDataMap_iter2->first && neighborlist_( intResDataMap_iter1->first, intResDataMap_iter2->first) )
	  neighbor_list.push_back( intResDataMap_iter2->first );
      }
      intResDataMap_iter1->second.fill_neighbor_list( neighbor_list );
    }

  }

  void ParallelSimAnnealBase::create_residue_data_map()
  {

    using namespace misc;

    if( !master_sequence_map.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    std::map<int, std::list<int> > possible_amino_acids_with_rotamers_for_seq_pos;

    for( int res_num = 1; res_num <= total_residue; res_num++){
      possible_amino_acids_with_rotamers_for_seq_pos.clear();
      for(int aa = 1; aa <= 20; aa++)
	for(int rot_num = 1; rot_num <= p_rotamer_set_->nrotamers(); rot_num++)
	  if( ( p_rotamer_set_->report_seqpos(rot_num) == int(res_num) ) && ( p_rotamer_set_->report_aa(rot_num) == aa ) )
	    possible_amino_acids_with_rotamers_for_seq_pos[ aa ].push_back( rot_num );
      if( possible_amino_acids_with_rotamers_for_seq_pos.empty() )
	possible_amino_acids_with_rotamers_for_seq_pos[ res( res_num ) ];
      master_sequence_map.insert( pair<int, std::map<int, std::list<int> > >( res_num, possible_amino_acids_with_rotamers_for_seq_pos ) );
    }
    add_neighbor_info_to_master_sequence_map();
  }

  void ParallelSimAnnealBase::create_list_of_key_sequence_positions()
  {

    if( equivalent_residues.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

    list<int>::iterator list_oI_iter;
    set<int> non_key_sequence_positions;

    for(unsigned int i = 0; i < equivalent_residues.size(); i++)
      for( list_oI_iter = equivalent_residues.at(i).begin(); list_oI_iter != equivalent_residues.at(i).end(); list_oI_iter++)
	if( list_oI_iter != equivalent_residues.at(i).begin() ) non_key_sequence_positions.insert( * list_oI_iter );

    std::map<int, ResidueData>::iterator intResDataMap_iter;

    for( intResDataMap_iter = master_sequence_map.begin(); intResDataMap_iter != master_sequence_map.end(); intResDataMap_iter++ )
      if( non_key_sequence_positions.find(intResDataMap_iter->first) ==  non_key_sequence_positions.end() )
	key_sequence_positions.push_back( intResDataMap_iter->first );

    key_sequence_positions.sort();

//     cout << "\nKey sequence positions:\n";
//     for( list_oI_iter = key_sequence_positions.begin(); list_oI_iter != key_sequence_positions.end(); list_oI_iter++)
//       cout << * list_oI_iter << " ";
//     cout << endl;

  }

  void ParallelSimAnnealBase::make_a_random_mutation()
  {
    if( key_sequence_positions.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

    mutant_sequence_positions.clear();

    mutant_sequence_positions.push_back( XUtilities::random_element_from_container( key_sequence_positions ) );
    master_sequence_map.find( mutant_sequence_positions.front() )->second.mutate_codon();
    master_sequence_map.find( mutant_sequence_positions.front() )->second.get_curr_aa( amino_acid_mutated_to );
  }

  void ParallelSimAnnealBase::make_a_sequential_substitution()
  {
    mutant_sequence_positions.resize( 1 );

    if( key_sequence_positions_vector.empty() ) reset_sequence_position_vector_for_sequential_mutations();

    if( possible_amino_acids_vector.empty() ){
      * mutant_sequence_positions.begin() = key_sequence_positions_vector.back();
      key_sequence_positions_vector.pop_back();
      reset_amino_acid_vector_for_sequential_mutations();
    }

    amino_acid_mutated_to = possible_amino_acids_vector.back();
    possible_amino_acids_vector.pop_back();

    master_sequence_map.find( mutant_sequence_positions.front() )->second.set_curr_aa( amino_acid_mutated_to );
    //    std::cout << "aa mutated to " << amino_acid_mutated_to << std::endl;
  }

  void ParallelSimAnnealBase::make_a_sequential_mutation()
  {
    mutant_sequence_positions.resize( 1 );

    if( key_sequence_positions_vector.empty() ) reset_sequence_position_vector_for_sequential_mutations();

    * mutant_sequence_positions.begin() = key_sequence_positions_vector.back();
    key_sequence_positions_vector.pop_back();

    master_sequence_map.find( mutant_sequence_positions.front() )->second.mutate_codon();
    master_sequence_map.find( mutant_sequence_positions.front() )->second.get_curr_aa( amino_acid_mutated_to );
  }

  void ParallelSimAnnealBase::make_a_cluster_of_random_mutations( const double & percent_of_key_residues_to_mutate_input )
  {
    reset_sequence_position_vector_for_sequential_mutations();

    int current_mutant_sequence_number;
    int number_of_of_key_residues_to_mutate = int( percent_of_key_residues_to_mutate_input * double( key_sequence_positions_vector.size() ) );

    for( int i = 0; i < number_of_of_key_residues_to_mutate; i++ ){
      cout << i << " ";
      current_mutant_sequence_number = key_sequence_positions_vector.at( i );
      cout << current_mutant_sequence_number << endl;
      master_sequence_map.find( current_mutant_sequence_number )->second.mutate_codon();
      cout << "finished\n";
      mutant_sequence_positions.push_back( current_mutant_sequence_number );
    }
  }

  void ParallelSimAnnealBase::reset_amino_acid_vector_for_sequential_mutations()
  {
    list<int> possible_amino_acids_list;
    master_sequence_map.find( mutant_sequence_positions.front() )->second.return_possible_amino_acids( possible_amino_acids_list );
    copy( possible_amino_acids_list.begin(), possible_amino_acids_list.end(), back_inserter( possible_amino_acids_vector ) );
    random_shuffle( possible_amino_acids_vector.begin(), possible_amino_acids_vector.end() );
  }

  void ParallelSimAnnealBase::reset_sequence_position_vector_for_sequential_mutations()
  {
    copy( key_sequence_positions.begin(), key_sequence_positions.end(), back_inserter( key_sequence_positions_vector ) );
    random_shuffle( key_sequence_positions_vector.begin(), key_sequence_positions_vector.end() );
  }

  void ParallelSimAnnealBase::read_equiv_resfile()
  {

    using namespace files_paths;

    if (equiv_resfile == "none"){
      std::cout << "Please supply a file using the -equiv_resfile option, exiting.\n";
      utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }

    EquivResfileReader EqRsRd;
    EqRsRd.read_equivalent_residue_file(equiv_resfile, equivalent_residues);

  }

  void ParallelSimAnnealBase::revert_to_best_sequence()
  {
    std::map<int, ResidueData>::iterator intResDataMap_iter;

    for( intResDataMap_iter = master_sequence_map.begin(); intResDataMap_iter != master_sequence_map.end(); intResDataMap_iter++)
      intResDataMap_iter->second.revert_to_best_state();
  }

  void ParallelSimAnnealBase::reject_mutations()
  {
    if( mutant_sequence_positions.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    list<int>::iterator mutants_iter;

    for( mutants_iter = mutant_sequence_positions.begin(); mutants_iter != mutant_sequence_positions.end(); mutants_iter++ )
      master_sequence_map.find( * mutants_iter )->second.revert_residue();
  }

  int ParallelSimAnnealBase::number_of_possible_mutations()
  {
    if( key_sequence_positions.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

    int  number_of_possible_mutations = 0;
    list<int>::iterator list_oI_iter;

    for( list_oI_iter = key_sequence_positions.begin(); list_oI_iter != key_sequence_positions.end(); list_oI_iter++)
      number_of_possible_mutations += master_sequence_map.find( * list_oI_iter )->second.number_of_possible_amino_acids();

    return number_of_possible_mutations;
  }

  void ParallelSimAnnealBase::add_rotamers_to_pack_for_residue_list( const list<int> & residue_list_input )
  {
    list<int>::const_iterator list_oI_iter;
    list<int> rotamers_for_seq_pos;

    for( list_oI_iter = residue_list_input.begin(); list_oI_iter != residue_list_input.end(); list_oI_iter++)
      if( master_sequence_map.find( * list_oI_iter )->second.return_possible_rotamers_for_current_amino_acid( rotamers_for_seq_pos ) )
	copy( rotamers_for_seq_pos.begin(), rotamers_for_seq_pos.end(), back_inserter( rot_to_pack) );
  }

  void ParallelSimAnnealBase::add_rotamers_to_pack_for_neighbors_of_mutated_residues()
  {
    list<int>::iterator mutant_seq_iter;;
    list<int> all_neighbors, neighbors_mutant;

    for( mutant_seq_iter = mutant_sequence_positions.begin(); mutant_seq_iter != mutant_sequence_positions.end(); mutant_seq_iter++ ){
      master_sequence_map.find( * mutant_seq_iter )->second.get_neighbors( neighbors_mutant );
      all_neighbors.merge( neighbors_mutant );
      all_neighbors.unique();
    }
    add_rotamers_to_pack_for_residue_list( all_neighbors );
  }

} //end of namespace
