// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.2 $
//  $Date: 2007/03/26 13:02:54 $
//  $Author: plato $


// Rosetta headers
#include "aaproperties_pack.h"
#include "fullatom_extra_props.h"
#include "fullatom_sasa_ns.h"
#include "param.h"
#include "param_pack.h"
#include "param_rotamer_trie.h"
#include "RotamerDots.h"
#include "util_vector.h"
#include "void.h"
#include "void_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/byte.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/constants.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cstdlib>
#include <fstream>
#include <iostream>


namespace pack
{

//----------------------------------------------------------------------------//
//----------------- Rotamer Coordinates For SASApack Class -------------------//
//----------------------------------------------------------------------------//


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerCoords::RotamerCoords
///
/// @brief
/// default constructor
///
/// @detailed
///
/// @param
///
/// @global_read
/// param::MAX_ATOM()
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerCoords::RotamerCoords() :
	coords_( param::MAX_ATOM() ),
	aa_( 0 ),
	aa_variant_( 0 ),
	natoms_( 0 )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerCoords::~RotamerCoords
///
/// @brief
/// destructor
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerCoords::~RotamerCoords() {}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerCoords::RotamerCoords
///
/// @brief
/// copy constructor -- deep copy
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerCoords::RotamerCoords( RotamerCoords const & rhs) :
	coords_( rhs.coords_ ),
	aa_( rhs.aa_ ),
	aa_variant_( rhs.aa_variant_ ),
	natoms_( rhs.natoms_ )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerCoords::RotamerCoords
///
/// @brief
/// constructor from coordinates; coordinates are ordered in fullatom order
///
/// @detailed
/// This constructor takes coordinates in their input order and reorders
/// them in the order defined in param_rotamer_trie.
///
/// @param
/// aa - [in] - the amino acid type for the rotamer
/// aav - [in] - the amino acid variant for the rotamer
/// coords - [in] - the input coordinates, in fullatom ordering
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerCoords::RotamerCoords(
	int aa,
	int aav,
	FArray2DB_float const & coords
)	:
	coords_( param::MAX_ATOM() ),
	aa_( aa ),
	aa_variant_( aav )
{
	natoms_ = aaproperties_pack::natoms( aa_ , aa_variant_ );
	initialize_coords_trie_order( coords );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerCoords::initialize_coords_trie_order
///
/// @brief
/// Converts the coordinates in fullatom order to the trie ordering
/// and computes subtree-bounding-sphere radii
///
/// @detailed
/// The subtree-bouding-sphere radius for an atom is the minimal radius
/// for a sphere centered at the atom that encloses the (vdw + 1.4 A) spheres
/// for each atom that follows that atom in the trie ordering.  "Tree" and
/// "subtree" are accurate, since every path is a tree.  "Path" and
/// "subpath" would describe the situation more precisely, if "subpath" were
/// a word.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// The bounding sphere need not center on the atom; when I implemented the
/// trie, it was important since storing the coordinate for the center of
/// this extra sphere would have cost an additional 12 bytes per atom.  In
/// the RotamerDots implementation, however, the memory usage per atom is less
/// important.  I doubt, however, that tighter bounding spheres will improve
/// running time dramatically.  I would project a <1% running time improvement.
/// Very little time is spent computing square distances for spheres that do
/// not overlap.
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerCoords::initialize_coords_trie_order
(
	FArray2DB_float const & coords
)
{
	//using namespace param_rotamer_trie; //apl fix coming shortly

	//int aa_variant_trie = get_variant_according_to_trie( aa_, aa_variant_ );
	for (int ii = 1; ii <= natoms_; ++ii)
	{
		int ii_trie_order = param_rotamer_trie::fullatom_2_trie_order( ii, aa_, aa_variant_ );
		for (int jj = 0, ii_linear_index = coords.index(1, ii ); jj < 3; ++jj, ++ii_linear_index )
		{
			coords_( ii_trie_order ).coord_[ jj ] = coords[ ii_linear_index ];
		}

		coords_( ii_trie_order ).atom_type_ =
			aaproperties_pack::fullatom_type( ii, aa_, aa_variant_ );
		coords_( ii_trie_order ).atom_radius_ =
			fullatom_ex_props::aradii( coords_( ii_trie_order ).atom_type_);
	}

	//initialize subtree interaction spheres
	for (int ii = 1; ii <= natoms_; ++ii)
	{
		float dist_to_furthest_shell = 0.0f;

		for (int jj = ii+1; jj <= natoms_; ++jj)
		{
			float jj_atom_radius = coords_( jj ).atom_radius_;
			float dist = std::sqrt( vec_dist2(
				get_atom_coords( ii ),
				get_atom_coords( jj )) );

			if ( jj_atom_radius + dist > dist_to_furthest_shell )
			{
				dist_to_furthest_shell = jj_atom_radius + dist;
			}
		}

		coords_(ii).subtree_interaction_sphere_radius_ =
			2*void_ns::probe_radius_large + dist_to_furthest_shell;
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerCoords::set_coords
///
/// @brief
/// sets the aa,aav, and coordinate information for the object;
/// coordinate information should be in fullatom order.  The object will
/// convert that order to the trie order
///
/// @detailed
///
/// @param
/// aa - [in] - the amino acid
/// aav - [in] - the amino acid variant
/// coords - [in] - the input coordinates, in fullatom order
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerCoords::set_coords( int aa, int aav, FArray2DB_float const & coords )
{
	aa_ = aa;
	aa_variant_ = aav;
	natoms_ = aaproperties_pack::natoms( aa_ , aa_variant_ );
	initialize_coords_trie_order( coords );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerCoords::get_num_shared_atoms
///
/// @brief
/// returns the number of atoms that are the same between this rotamer
/// and other.
///
/// @detailed
/// The ordering defined in param_rotame_trie allows for an output sensitive
/// linear-time determination of the number of shared atoms between two
/// rotamers.  Atoms are ordered in increasing distance from the backbone,
/// so that shared atoms between two rotamers are at the beginning.
///
/// @param
/// other - [in] - the other RotamerCoords object to compare with this
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
RotamerCoords::get_num_shared_atoms( RotamerCoords const & other ) const
{
	int fewer_atoms = ( natoms_ < other.natoms_ ? natoms_ : other.natoms_ );
	for (int ii = 1; ii <= fewer_atoms; ++ii)
	{
		if ( coords_(ii).atom_type_ != other.coords_(ii).atom_type_)
		{
			return ii - 1;
		}
		else
		{
			if (coords_(ii).coord_[0] != other.coords_(ii).coord_[0] ||
				coords_(ii).coord_[1] != other.coords_(ii).coord_[1] ||
				coords_(ii).coord_[2] != other.coords_(ii).coord_[2] )
			{
				return ii - 1;
			}
		}
	}
	return fewer_atoms;
}

//----------------------------------------------------------------------------//
//----------------------------- Dot Sphere Class -----------------------------//
//----------------------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::DotSphere
///
/// @brief
/// default constructor
///
/// @detailed
/// initializes all dot counts to zero
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
DotSphere::DotSphere()
:
	dots_coverage_count_(),
	num_covered_( 0 ), num_covered_current_( false )
{
	zero();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::~DotSphere
///
/// @brief
/// destructor
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
DotSphere::~DotSphere()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::DotSphere
///
/// @brief
/// copy constructor
///
/// @detailed
///
/// @param
/// rhs - [in] - the DotSphere object to copy
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// memcpy is much faster than the FArray operator =
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
DotSphere::DotSphere( DotSphere const & rhs )
:
	num_covered_( rhs.num_covered_ ),
	num_covered_current_( rhs.num_covered_current_ )
{
	memcpy( dots_coverage_count_, rhs.dots_coverage_count_, NUM_COUNTS_TO_ALLOCATE);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::zero
///
/// @brief
/// sets the dot coverage counts to zero for all dots
///
/// @detailed
/// memset is fast -- a lot of time is spent in this function so
/// I'm using c-style arrays instead of the FArrays
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
void DotSphere::zero()
{
	if (num_covered_current_ && num_covered_ == 0) return;

	memset(dots_coverage_count_, 0, NUM_COUNTS_TO_ALLOCATE );
	num_covered_ = 0;
	num_covered_current_ = true;
}

void
DotSphere::non_inlined_zero()
{	zero();}


////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::get_total_dots
///
/// @brief
/// returns the number of dots on the sphere's surface
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
int DotSphere::get_total_dots() const
{
	return NUM_DOTS_TOTAL;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::increment_count
///
/// @brief
/// increment the count for the dots using an input ubyte array.
///
/// @detailed
/// Each bit in this ubyte array corresponds to a single dot on the surface of
/// this sphere.  Dot covereage counts are incremented for dots whose
/// corresponding bits in the ubyte array are '1'.
///
/// @param
/// overlap_mask - [in] - array of bytes representing which dot coverage counts
///   to increment
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
void
DotSphere::increment_count( FArray1DB_ubyte const & overlap_mask )
{

	for ( int bb = 0; bb < NUM_BYTES_IN_DOT_SPHERE_OVERLAP_ARRAYS; ++bb )
	{
		const int bb8 = bb*8;
		dots_coverage_count_[ bb8 + 0 ] += static_cast <unsigned char> ( static_cast< bool> (overlap_mask[ bb ] & 0x01));
		dots_coverage_count_[ bb8 + 1 ] += static_cast <unsigned char> ( static_cast< bool> (overlap_mask[ bb ] & 0x02));
		dots_coverage_count_[ bb8 + 2 ] += static_cast <unsigned char> ( static_cast< bool> (overlap_mask[ bb ] & 0x04));
		dots_coverage_count_[ bb8 + 3 ] += static_cast <unsigned char> ( static_cast< bool> (overlap_mask[ bb ] & 0x08));
		dots_coverage_count_[ bb8 + 4 ] += static_cast <unsigned char> ( static_cast< bool> (overlap_mask[ bb ] & 0x10));
		dots_coverage_count_[ bb8 + 5 ] += static_cast <unsigned char> ( static_cast< bool> (overlap_mask[ bb ] & 0x20));
		dots_coverage_count_[ bb8 + 6 ] += static_cast <unsigned char> ( static_cast< bool> (overlap_mask[ bb ] & 0x40));
		dots_coverage_count_[ bb8 + 7 ] += static_cast <unsigned char> ( static_cast< bool> (overlap_mask[ bb ] & 0x80));

		//std::cerr << "dots_coverage_count_( bb8 + 1 )" << (int) dots_coverage_count_( bb8 + 1 ) << std::endl;
		//std::cerr << "dots_coverage_count_( bb8 + 2 )" << (int) dots_coverage_count_( bb8 + 2 ) << std::endl;
		//std::cerr << "dots_coverage_count_( bb8 + 3 )" << (int) dots_coverage_count_( bb8 + 3 ) << std::endl;
		//std::cerr << "dots_coverage_count_( bb8 + 4 )" << (int) dots_coverage_count_( bb8 + 4 ) << std::endl;
		//std::cerr << "dots_coverage_count_( bb8 + 5 )" << (int) dots_coverage_count_( bb8 + 5 ) << std::endl;
		//std::cerr << "dots_coverage_count_( bb8 + 6 )" << (int) dots_coverage_count_( bb8 + 6 ) << std::endl;
		//std::cerr << "dots_coverage_count_( bb8 + 7 )" << (int) dots_coverage_count_( bb8 + 7 ) << std::endl;
		//std::cerr << "dots_coverage_count_( bb8 + 8 )" << (int) dots_coverage_count_( bb8 + 8 ) << std::endl;

	}

	num_covered_current_ = false;
}

void DotSphere::non_inlined_increment_count( FArray1DB_ubyte const & ubytearray )
{	increment_count( ubytearray );}

//////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::get_num_uncovered
///
/// @brief
/// returns the number of dots whose coverage count is 0
///
/// @detailed
/// if the coverage count has not been modified since the last time the
/// number of covered dots was counted, then the method uses the cached
/// result
///
/// @authors apl glb
//////////////////////////////////////////////////////////////////////////////
inline
int
DotSphere::get_num_uncovered() const
{
	if ( ! num_covered_current_ )
	{
		count_num_covered();
	}
	return NUM_DOTS_TOTAL - num_covered_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::count_num_covered
///
/// @brief
/// iterates across all dots and stores the number with a non-zero coverage
/// count for later use
///
/// @detailed
/// both num_covered_ and num_covered_current_ are declared
/// mutable so that they may be modified in this const method
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
void
DotSphere::count_num_covered() const
{
	num_covered_ = 0;
	all_dot_counts_two_or_more_ = true;

	for( int ii = 0; ii <  NUM_DOTS_TOTAL; ++ii ){
		if( dots_coverage_count_[ii] != 0 ){
			num_covered_++;
		}
		all_dot_counts_two_or_more_ = all_dot_counts_two_or_more_ && ( dots_coverage_count_[ ii ] > 1);
	}
	num_covered_current_ = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::get_num_covered
///
/// @brief
/// returns the number ofdots with a non-zero coverage count
///
/// @detailed
/// if the coverage count has not been modified since the last time the
/// number of covered dots was counted, then the method uses the cached
/// result
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
int
DotSphere::get_num_covered() const
{
	if ( ! num_covered_current_ )
	{
		count_num_covered();
	}
	return num_covered_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::operator -=
///
/// @brief
/// decrements the coverage count for this sphere by the coverage count
/// of the rhs sphere
///
/// @detailed
///
/// @param
/// rhs - [in] - the dot coverage counts to subtract
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
DotSphere const & DotSphere::operator -= ( DotSphere const & rhs )
{
	num_covered_current_ = false;
	for (int ii = 0; ii < NUM_COUNTS_TO_ALLOCATE; ++ii)
	{
		dots_coverage_count_[ ii ] -= rhs.dots_coverage_count_[ ii ];
	}
	return * this;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::operator +=
///
/// @brief
/// increments the coverage count for this sphere by the coverage count of the
/// rhs sphere
///
/// @detailed
///
/// @param
///
/// @global_read
/// rhs - [in] - the dot coverage counts to add
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
DotSphere const & DotSphere::operator += ( DotSphere const & rhs )
{
	num_covered_current_ = false;
	for (int ii = 0; ii < NUM_COUNTS_TO_ALLOCATE; ++ii)
	{
		dots_coverage_count_[ ii ] += rhs.dots_coverage_count_[ ii ];
	}
	return * this;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin DotSphere::print
///
/// @brief
/// writes coverage counts to the output stream
///
/// @detailed
///
/// @param
/// os - [in/out] - the output stream to write to
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void DotSphere::print( std::ostream & os) const
{
	for( int ii = 0; ii < NUM_COUNTS_TO_ALLOCATE; ++ii ){
		if( dots_coverage_count_[ii] < 10 ){
			os << (int) dots_coverage_count_[ii];
		}
		else os << "9";

		if (ii % 8 == 7) os << " ";
		if (ii % 40 == 39) os << std::endl;
	}
	os << get_num_covered();
	return;
}


bool DotSphere::get_dot_covered( int dot_index )
{
	assert( dot_index > 0 && dot_index <= NUM_DOTS_TOTAL );
	return (dots_coverage_count_[ dot_index - 1 ] != 0);
}

//note loss of information; counts > 1 are truncated
void DotSphere::write_to_compact_array( FArray1DB_ubyte & compact ) const
{
	for (int ii = 0; ii < NUM_BYTES_IN_DOT_SPHERE_OVERLAP_ARRAYS; ++ii)
	{
		int bb8 = ii*8;
		if ( dots_coverage_count_[ bb8 + 0 ] > 0 ) compact[ ii ] |= static_cast< ubyte > (0x01);
		if ( dots_coverage_count_[ bb8 + 1 ] > 0 ) compact[ ii ] |= static_cast< ubyte > (0x02);
		if ( dots_coverage_count_[ bb8 + 2 ] > 0 ) compact[ ii ] |= static_cast< ubyte > (0x04);
		if ( dots_coverage_count_[ bb8 + 3 ] > 0 ) compact[ ii ] |= static_cast< ubyte > (0x08);
		if ( dots_coverage_count_[ bb8 + 4 ] > 0 ) compact[ ii ] |= static_cast< ubyte > (0x10);
		if ( dots_coverage_count_[ bb8 + 5 ] > 0 ) compact[ ii ] |= static_cast< ubyte > (0x20);
		if ( dots_coverage_count_[ bb8 + 6 ] > 0 ) compact[ ii ] |= static_cast< ubyte > (0x40);
		if ( dots_coverage_count_[ bb8 + 7 ] > 0 ) compact[ ii ] |= static_cast< ubyte > (0x80);
	}
}

void DotSphere::invert()
{
	if ( num_covered_current_ )
	{
		num_covered_ = NUM_DOTS_TOTAL - num_covered_;
	}

	for (int ii = 0; ii < NUM_DOTS_TOTAL; ++ii)
	{
		if (dots_coverage_count_[ ii ] == 0)
			dots_coverage_count_[ii] = 1;
		else
			dots_coverage_count_[ii] = 0;
	}
}

void
DotSphere::double_counts()
{

	for (int ii = 0; ii < NUM_DOTS_TOTAL; ++ii)
	{
		dots_coverage_count_[ ii ] *= 2;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin operator << ( ostream, DotSphere)
///
/// @brief
/// invokes print on the input DotSphere object
///
/// @detailed
///
/// @param
/// os - [in/out] - the output stream to write to
/// ds - [in] - the DotSphere object to write
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
std::ostream & operator << ( std::ostream & os, DotSphere const & ds )
{
	ds.print(os);
	return os;
}

//----------------------------------------------------------------------------//
//---------------------------- Rotamer Dots Class ----------------------------//
//----------------------------------------------------------------------------//
bool RotamerDots::dot_sphere_coordinates_initialized_ = false;
FArray1D< numeric::xyzVector< float > > RotamerDots::dot_sphere_coordinates_( 0 );

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::RotamerDots
///
/// @brief
/// default constructor
///
/// @detailed
/// allocates enough space for as many atoms as the RotamerDots object might
/// hold.  A RotamerDots object will hold rotamers from different amino
/// acids during the course of its lifetime: to avoid new/delete, adequate
/// space gets allocated upfront
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerDots::RotamerDots() :
	coords_(),
	num_atoms_(0),
	atom_counts_large_probe_( param::MAX_ATOM() ),
	atom_counts_small_probe_( param::MAX_ATOM() ),
	spheres_doubly_covered_( 2, param::MAX_ATOM(), false ),
	incomplete_copy_ignoring_doubly_counted_spheres_( false ),
	original_(0),
	decrement_by_(0),
	increment_by_(0),
	score_(0),
	score_is_current_( false )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::~RotamerDots
///
/// @brief
/// destructor
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerDots::~RotamerDots()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::RotamerDots
///
/// @brief
/// copy constructor
///
/// @detailed
///
/// @param
/// rhs - [in] - the RotamerDots object to copy
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerDots::RotamerDots( RotamerDots const & rhs )
:
	coords_( rhs.coords_ ),
	num_atoms_( rhs.num_atoms_ ),
	atom_counts_large_probe_( rhs.atom_counts_large_probe_ ),
	atom_counts_small_probe_( rhs.atom_counts_small_probe_ ),
	spheres_doubly_covered_( 2, param::MAX_ATOM(), false ),
	incomplete_copy_ignoring_doubly_counted_spheres_( false ),
	original_(0),
	decrement_by_(0),
	increment_by_(0),
	score_( rhs.score_ ),
	score_is_current_( rhs.score_is_current_ )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::set_coords
///
/// @brief
/// stores the RotamerCoords object for this
///
/// @detailed
///
/// @param
/// coords - [in] - the RotamerCoords object to store
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerDots::set_coords( RotamerCoords const & coords )
{
	coords_ = coords;
	num_atoms_ = coords_.get_num_atoms();
	score_is_current_ = false;
}

void
RotamerDots::print() const
{
	using namespace param_rotamer_trie;
	for (int ii = 1; ii <= get_num_atoms(); ++ii)
	{
		//int ii_fullatom = trie_order_2_fullatom( ii, coords_.get_residue(), coords_.get_aav() );

		std::cerr << "( ";
		std::cerr << coords_.get_atom_coords( ii )[0] << ", ";
		std::cerr << coords_.get_atom_coords( ii )[1] << ", ";
		std::cerr << coords_.get_atom_coords( ii )[2] << ") " << coords_.get_atom_radius( ii ) << "\n";
		atom_counts_small_probe_( ii ).print( std::cerr );
		std::cerr << "\n";
		atom_counts_large_probe_( ii ).print( std::cerr );
		std::cerr << std::endl;
	}
	std::cerr << get_score() << std::endl;
}

void RotamerDots::copy( RotamerDots const & rhs )
{
	if ( rhs.incomplete_copy_ignoring_doubly_counted_spheres_ && this == rhs.original_)
	{
		//std::cout << "procrastinated copy now being performed" << std::endl;
		copy_non_procrastinated_dot_spheres( rhs );
		complete_procrastinated_decrement( rhs );
		complete_procrastinated_increment( rhs );
		spheres_doubly_covered_ = false; // spheres_doubly_covered_ only holds true when incomplete_copy_ignoring_duob is true;
		detect_double_covereage_in_procrastinated_spheres( rhs );
		score_ = rhs.score_;
		score_is_current_ = rhs.score_is_current_;

		rhs.incomplete_copy_ignoring_doubly_counted_spheres_ = false;
		rhs.original_ = 0;
		rhs.decrement_by_ = 0;
		rhs.increment_by_ = 0;

		original_ = 0;
		decrement_by_ = 0;
		increment_by_ = 0;
		incomplete_copy_ignoring_doubly_counted_spheres_ = false;

	}
	else
	{
		copy_standard( rhs );
	}

}

void RotamerDots::copy_standard(RotamerDots const & rhs)
{
	coords_ = rhs.coords_;
	num_atoms_ = rhs.num_atoms_;
	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		atom_counts_small_probe_(ii) = rhs.atom_counts_small_probe_(ii);
	}
	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		atom_counts_large_probe_(ii) = rhs.atom_counts_large_probe_(ii);
	}
	score_ = rhs.score_;
	score_is_current_ = rhs.score_is_current_;
	spheres_doubly_covered_ = false;
	incomplete_copy_ignoring_doubly_counted_spheres_ = false;
	original_ = 0;
	decrement_by_ = 0;
	increment_by_ = 0;
}

void
RotamerDots::copy_non_procrastinated_dot_spheres( RotamerDots const & rhs )
{
	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		if ( ! rhs.spheres_doubly_covered_(1, ii ) )
		{
			atom_counts_small_probe_( ii ) = rhs.atom_counts_small_probe_( ii );
		}
	}
	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		if ( ! rhs.spheres_doubly_covered_(2, ii ) )
		{
			atom_counts_large_probe_( ii ) = rhs.atom_counts_large_probe_( ii );
		}
	}

}

void
RotamerDots::complete_procrastinated_decrement( RotamerDots const & rhs )
{
	assert( rhs.decrement_by_ && ! rhs.decrement_by_->partial_copy_ );
	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		if ( rhs.spheres_doubly_covered_(1, ii ) && rhs.decrement_by_->non_zero_overlap_( ii ))
		{
			atom_counts_small_probe_( ii ) -= rhs.decrement_by_->atom_counts_small_probe_( ii );
			//std::cout << "Procrastinated decrement small " << ii << std::endl;
			//std::cout << rhs.decrement_by_->atom_counts_small_probe_( ii );
		}
	}
	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		if ( rhs.spheres_doubly_covered_(2, ii ) && rhs.decrement_by_->non_zero_overlap_( ii ))
		{
			atom_counts_large_probe_( ii )  -= rhs.decrement_by_->atom_counts_large_probe_( ii );
			//std::cout << "Procrastinated decrement large " << ii << std::endl;
			//std::cout << rhs.decrement_by_->atom_counts_large_probe_( ii );
		}
	}
	//std::cerr << "Procrastinated decrement" << std::endl;
	//print();
}

void
RotamerDots::complete_procrastinated_increment( RotamerDots const & rhs )
{
	assert( rhs.increment_by_ && ! rhs.increment_by_->partial_copy_);
	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		if ( rhs.spheres_doubly_covered_(1, ii ) && rhs.increment_by_->non_zero_overlap_( ii ))
		{
			atom_counts_small_probe_( ii ) += rhs.increment_by_->atom_counts_small_probe_( ii );
			//std::cout << "Procrastinated increment small " << ii << std::endl;
			//std::cout << rhs.increment_by_->atom_counts_small_probe_( ii );

		}
	}
	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		if ( rhs.spheres_doubly_covered_(2, ii ) && rhs.increment_by_->non_zero_overlap_( ii ))
		{
			atom_counts_large_probe_( ii ) += rhs.increment_by_->atom_counts_large_probe_( ii );
			//std::cout << "Procrastinated increment large " << ii << std::endl;
			//std::cout << rhs.increment_by_->atom_counts_large_probe_( ii );
		}
	}
	//std::cerr << "Procrastinated increment" << std::endl;
	//print();

}

void
RotamerDots::detect_double_covereage_in_procrastinated_spheres( RotamerDots const & rhs )
{

	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		if ( rhs.spheres_doubly_covered_(1, ii ) &&
			(rhs.decrement_by_->non_zero_overlap_( ii ) || rhs.increment_by_->non_zero_overlap_( ii )))
		{
			atom_counts_small_probe_( ii ).get_num_covered();
		}
	}
	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		if ( rhs.spheres_doubly_covered_(2, ii ) &&
			(rhs.decrement_by_->non_zero_overlap_( ii ) || rhs.increment_by_->non_zero_overlap_( ii )))
		{
			atom_counts_large_probe_( ii ).get_num_covered();
		}
	}
}

void
RotamerDots::copy_spheres_not_doubly_covered( RotamerDots * other )
{
	incomplete_copy_ignoring_doubly_counted_spheres_ = false;
	original_ = 0;
	decrement_by_ = 0;
	increment_by_ = 0;
	for (int ii = 1; ii <= other->num_atoms_; ++ii)
	{
		if ( other->atom_counts_large_probe_( ii ).get_all_counts_two_or_more() )
		{
			incomplete_copy_ignoring_doubly_counted_spheres_ = true;
			//std::cerr << "Found Doubly counted sphere: " << ii << std::endl;
			//std::cerr << other->atom_counts_large_probe_( ii ) << std::endl;
			break;
		}
		if ( other->atom_counts_small_probe_( ii ).get_all_counts_two_or_more() )
		{
			incomplete_copy_ignoring_doubly_counted_spheres_ = true;
			//incomplete_copy_ignoring_doubly_counted_spheres_ = true;
			//std::cerr << "Found Doubly counted sphere: " << ii << std::endl;
			//std::cerr << other->atom_counts_small_probe_( ii ) << std::endl;
			break;
		}
	}
	if ( ! incomplete_copy_ignoring_doubly_counted_spheres_ )
	{
		(*this) = *other;
	}
	else
	{
		original_ = other;
		num_atoms_ = other->num_atoms_;
		coords_ = other->coords_;
		for (int ii = 1; ii <= num_atoms_; ++ii)
		{
			spheres_doubly_covered_( 1, ii ) = other->atom_counts_small_probe_( ii ).get_all_counts_two_or_more();
			spheres_doubly_covered_( 2, ii ) = other->atom_counts_large_probe_( ii ).get_all_counts_two_or_more();
		}

		for (int ii = 1; ii <= num_atoms_; ++ii)
		{
			if ( ! spheres_doubly_covered_( 1, ii ) )
			{
				atom_counts_small_probe_(ii) = other->atom_counts_small_probe_( ii );
			}
			//else
			//{
			//	std::cout << "doubly covered small, " << ii << std::endl << other->atom_counts_small_probe_( ii );
			//	std::cout << "doubly covered? " << other->atom_counts_small_probe_( ii ).get_all_counts_two_or_more();
			//	std::cout << std::endl;
			//	other->atom_counts_small_probe_( ii ).count_num_covered();
			//	std::cout << "Recount doubly covered? " << other->atom_counts_small_probe_( ii ).get_all_counts_two_or_more();
			//	std::cout << std::endl;
			//}
		}

		for (int ii = 1; ii <= num_atoms_; ++ii)
		{
			if ( ! spheres_doubly_covered_( 2, ii ) )
			{
				atom_counts_large_probe_(ii) = other->atom_counts_large_probe_( ii );
			}
			//else
			///{
			//	std::cout << "doubly covered large: , " << ii << "; " << other->atom_counts_large_probe_( ii );
			//	std::cout << "doubly covered? " << other->atom_counts_large_probe_( ii ).get_all_counts_two_or_more();
			//	std::cout << std::endl;
			//	other->atom_counts_large_probe_( ii ).count_num_covered();
			//	std::cout << "Recount doubly covered? " << other->atom_counts_large_probe_( ii ).get_all_counts_two_or_more();
			//	std::cout << std::endl;
			//}
		}
		score_ = other->score_;
		score_is_current_ = other->score_is_current_;
	}
}



////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::overlaps
///
/// @brief
/// Returns true if this has any sphere overlap with other
///
/// @detailed
/// uses RotamerCoords subtree bounding spheres to quit as early as possible
///
/// @param
/// other - [in] - the other RotamerDots object
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
RotamerDots::overlaps( RotamerDots const & other ) const
{
	using namespace void_ns;
	using namespace fullatom_ex_props;

	float dist_sq,at1_radius,at2_radius;

	for( int ii = 1; ii <= get_num_atoms(); ++ii){
		for( int jj = 1; jj <= other.get_num_atoms(); ++jj){

			at1_radius = coords_.get_atom_radius( ii );
			at2_radius = other.coords_.get_atom_radius( jj );

			dist_sq = vec_dist2(
			 coords_.get_atom_coords( ii ),
			 other.coords_.get_atom_coords( jj )
			);

			at1_radius += probe_radius_large;
			at2_radius += probe_radius_large;

			//exit if large probe radii touch
			if ( dist_sq < (at1_radius + at2_radius)*(at1_radius + at2_radius) ){
				return true;
			}

			//break if jj's subtree bounding sphere does not overlap with ii's
			//(vdw + probe_radius_large) sphere.
			at2_radius = other.coords_.get_subtree_interaction_sphere_radius( jj ) ;
			if ( dist_sq > (at1_radius + at2_radius)*(at1_radius + at2_radius) ){
				break;
			}


		}
	}
	return false;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::increment_both
///
/// @brief
/// Add rotamer coverage counts for dots on both this and other.
/// sets score_is_current_ to false on both this and rhs
///
/// @detailed
///
/// @param
///
//////////////////////////////////////////////////////////////////////////////
void
RotamerDots::increment_both(
	RotamerDots & other
)
{
	static RotamerDotsCache others_dots_covered_by_this;
	static RotamerDotsCache this_dots_covered_by_other;

	increment_both_and_cache( other,
		others_dots_covered_by_this,
		this_dots_covered_by_other);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::increment_both_and_cache
///
/// @brief
/// Add rotamer coverage counts for dots on both this and other.
/// Cache's the overlap this and other have with each other for greater
/// efficiency.
///
/// @detailed
/// Computes the overlap for all the atoms on this with the subset of atoms
/// [num_atoms_same + 1 .. other_rotamer.num_atoms_] on other_rotamer.
/// The four "mask" ubyte arrays hold dot coverage information for the
/// atoms of this in their overlap with the subset of atms
/// [1 .. num_atoms_same] on other_rotamer.  Writes the atom pair overlap
/// mask indices to the alt_atom_overlap_mask_inds table for later
/// reuse.  This method passes the non_zero_overlap member arrays to
/// the RotamerDotsCache objects to the get_res_res_overlap subroutine.
/// These arrays must hold true for any atom for which a sphere has
/// overlapped with another atom.  Data in these arrays are used
/// by the increment_count_for_some() methods.
///
/// @param
/// other_rotamer - [in/out] - the RotamerDots object for which to increment
///   dot coverage counts
/// other_dots_covered_by_this - [out] - the dot coverage cache represnting
///   dots on the surface of other_rotamer that are covered by this rotamer
///   This object already holds the coverage for all atoms of this on the
///   the first num_atoms_same atoms of other_rotamer.
/// this_dots_covered_by_other - [in/out] - the dot coverage cache representing
///   dots on the surface of this rotamer that are covered by this rotamer.
///   This object already holds the coverage for all atoms of this produced
///   by the first num_atoms_same atoms of other_rotamer.
/// alt_atom_overlap_mask_inds - [out] - the table to write sphere overlap
///   information for all pairs of atoms.
/// mask_this_covered_by_other_small - [in/out] - Compact dot coverage data
/// mask_other_covered_by_this_small - [in/out] - for the spheres.  Preped
/// mask_this_covered_by_other_large - [in/out] - to hold saved data by
/// mask_other_covered_by_this_large - [in/out] - the SASAEdge.
/// num_atoms_same - [in] - the number of atoms on other_rotamer to
///   avoid computing sphere overlap information for; data for these atoms
///   has been re-used from cached information and spread throughout the
///   rest of the input/output parameters.
///
/// @remarks
///   fast for what its doing; slow in comparison not computing anything
///
//////////////////////////////////////////////////////////////////////////////

void
RotamerDots::increment_both_and_cache(
	RotamerDots & other_rotamer,
	RotamerDotsCache & others_dots_covered_by_this,
	RotamerDotsCache & this_dots_covered_by_other,
	FArray3DB< short > & alt_atom_overlap_mask_inds,
	FArray2DB< ubyte > & mask_this_covered_by_other_small,
	FArray2DB< ubyte > & mask_other_covered_by_this_small,
	FArray2DB< ubyte > & mask_this_covered_by_other_large,
	FArray2DB< ubyte > & mask_other_covered_by_this_large,
	int num_atoms_same
)
{
	get_res_res_overlap(
		coords_,
		other_rotamer.coords_,
		mask_this_covered_by_other_small,
		mask_this_covered_by_other_large,
		mask_other_covered_by_this_small,
		mask_other_covered_by_this_large,
		alt_atom_overlap_mask_inds,
		num_atoms_same,
		this_dots_covered_by_other.non_zero_overlap_,
		others_dots_covered_by_this.non_zero_overlap_
	);

	this_dots_covered_by_other.increment_count_for_some(
		mask_this_covered_by_other_small,
		mask_this_covered_by_other_large,
		get_num_atoms() );

	others_dots_covered_by_this.increment_count_for_some(
		mask_other_covered_by_this_small,
		mask_other_covered_by_this_large,
		other_rotamer.get_num_atoms(),
		num_atoms_same );

	increment_from_cached( this_dots_covered_by_other );
	other_rotamer.increment_from_cached( others_dots_covered_by_this );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::get_overlap_cache
///
/// @brief
/// computes the overlap each rotamer (this & other) induce on each other and
/// stores that information in the RotamerDotsCache objects
///
/// @detailed
///
/// @param
/// other - [in] - the other RotamerDots object
/// other_dots_covered_by_this - [out] - the Cache for the dots on the surface
///   of other that are covered by the atoms on this rotamer
/// this_dots_covered_by_other - [out] - the Cache for the dots on the surface
///   of this that are covered by the atoms on the other rotamer
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerDots::get_overlap_cache
(
	RotamerDots const & other,
	RotamerDotsCache & others_dots_covered_by_this,
	RotamerDotsCache & this_dots_covered_by_other
) const
{
	using namespace fullatom_sasa;
	using namespace param;

	//apl static so that they will be allocated exactly once - we don't want to use
	//apl new and delete in the inner most loop.
	static FArray2D_ubyte this_covered_by_other_small( nbytes, MAX_ATOM());
	static FArray2D_ubyte this_covered_by_other_large( nbytes, MAX_ATOM());
	static FArray2D_ubyte other_covered_by_this_small( nbytes, MAX_ATOM());
	static FArray2D_ubyte other_covered_by_this_large( nbytes, MAX_ATOM());

	FArray2Da_ubyte this_covered_by_other_small_proxy(
		this_covered_by_other_small, nbytes, get_num_atoms() );
	FArray2Da_ubyte this_covered_by_other_large_proxy(
		this_covered_by_other_large, nbytes, get_num_atoms() );
	FArray2D_ubyte other_covered_by_this_small_proxy(
		other_covered_by_this_small, nbytes, other.get_num_atoms());
	FArray2D_ubyte other_covered_by_this_large_proxy(
		other_covered_by_this_large, nbytes,  other.get_num_atoms());


	others_dots_covered_by_this.zero();
	this_dots_covered_by_other.zero();

	get_res_res_overlap(
		coords_, other.coords_,
		this_covered_by_other_small_proxy, this_covered_by_other_large_proxy,
		other_covered_by_this_small_proxy, other_covered_by_this_large_proxy,
		this_dots_covered_by_other.non_zero_overlap_,
		others_dots_covered_by_this.non_zero_overlap_
	);

	this_dots_covered_by_other.increment_count_for_some(
		this_covered_by_other_small_proxy,
		this_covered_by_other_large_proxy,
		get_num_atoms() );

	others_dots_covered_by_this.increment_count_for_some(
		other_covered_by_this_small_proxy,
		other_covered_by_this_large_proxy,
		other.get_num_atoms() );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::increment_both_and_cache
///
/// @brief
/// Add rotamer coverage counts for dots on this object only, leaving rhs
/// unaltered.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerDots::increment_this_and_cache
(
	RotamerDots const & other,
	RotamerDotsCache & this_overlap_on_other
)
{
	static RotamerDotsCache this_dots_covered_by_other;

	get_overlap_cache( other,
		this_overlap_on_other,
		this_dots_covered_by_other);

	increment_from_cached( this_dots_covered_by_other );
}

void
RotamerDots::increment( Ligand & ligand )
{
	using namespace fullatom_sasa;

	static RotamerDotsCache this_dots_covered_by_ligand;
	static FArray2D_ubyte this_covered_by_ligand_small( nbytes, param::MAX_ATOM());
	static FArray2D_ubyte this_covered_by_ligand_large( nbytes, param::MAX_ATOM());

	this_covered_by_ligand_small = static_cast< ubyte > (0);
	this_covered_by_ligand_large = static_cast< ubyte > (0);
	this_dots_covered_by_ligand.zero();

	get_residue_ligand_overlap(
		coords_,
		this_covered_by_ligand_small,
		this_covered_by_ligand_large,
		this_dots_covered_by_ligand.non_zero_overlap_,
		ligand
	);

	this_dots_covered_by_ligand.increment_count_for_some(
		this_covered_by_ligand_small,
		this_covered_by_ligand_large,
		get_num_atoms()
	);

	increment_from_cached( this_dots_covered_by_ligand );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::increment_from_cached
///
/// @brief
/// Increments the dot coverage count for this rotamer from a coverage cache
///
/// @detailed
/// If this input rotamer dots cache object represents a partial copy of
/// another, then the method retrieves the data from the original
///
/// @param
/// cache - [in] - the RotamerDotsCache object representing sphere overlaps
///   with a single other residue
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerDots::increment_from_cached( RotamerDotsCache const & cache )
{
	if ( incomplete_copy_ignoring_doubly_counted_spheres_ )
	{
		increment_by_ = &cache;
	}

	score_is_current_ = false;
	if ( cache.partial_copy_ )
	{
		for (int ii = 1; ii <= cache.num_atoms_in_partial_copy_; ++ii)
		{
			if ( cache.original_->non_zero_overlap_( ii ) && small_sphere_needs_updating( ii ))
			{	atom_counts_small_probe_(ii) += cache.original_->atom_counts_small_probe_(ii);}
		}
		for (int ii = 1; ii <= cache.num_atoms_in_partial_copy_; ++ii)
		{
			if ( cache.original_->non_zero_overlap_( ii ) && large_sphere_needs_updating( ii ) )
			{	atom_counts_large_probe_(ii) += cache.original_->atom_counts_large_probe_(ii);}
		}

		for (int ii = cache.num_atoms_in_partial_copy_ + 1; ii <= get_num_atoms(); ++ii)
		{
			if ( cache.non_zero_overlap_( ii ) && small_sphere_needs_updating( ii ) )
			{	atom_counts_small_probe_(ii) += cache.atom_counts_small_probe_(ii);}
		}
		for (int ii = cache.num_atoms_in_partial_copy_ + 1; ii <= get_num_atoms(); ++ii)
		{
			if ( cache.non_zero_overlap_( ii ) && large_sphere_needs_updating( ii ) )
			{	atom_counts_large_probe_(ii) += cache.atom_counts_large_probe_(ii);}
		}
	}
	else
	{
		for (int ii = 1; ii <= get_num_atoms(); ++ii)
		{
			if ( cache.non_zero_overlap_( ii ) && small_sphere_needs_updating( ii ) )
			{	atom_counts_small_probe_(ii) += cache.atom_counts_small_probe_(ii);}
		}
		for (int ii = 1; ii <= get_num_atoms(); ++ii)
		{
			if ( cache.non_zero_overlap_( ii ) && large_sphere_needs_updating( ii ) )
			{	atom_counts_large_probe_(ii) += cache.atom_counts_large_probe_(ii);}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::increment_self_overlap
///
/// @brief
/// computes and stores self-induced dot coverage
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerDots::increment_self_overlap()
{
	using namespace fullatom_sasa;
	using namespace param;

	static FArray2D_ubyte self_overlap_small( nbytes, MAX_ATOM());
	static FArray2D_ubyte self_overlap_large( nbytes, MAX_ATOM());

	//call get_self_res_overlapp
	get_residue_self_coverage( coords_, self_overlap_small, self_overlap_large);

	for( int ii = 1; ii <= get_num_atoms(); ++ii)
	{
		FArray1Da_ubyte atom_mask_small( self_overlap_small( 1, ii), nbytes);
		FArray1Da_ubyte atom_mask_large( self_overlap_large( 1, ii), nbytes);

		atom_counts_small_probe_( ii ).increment_count( atom_mask_small );
		atom_counts_large_probe_( ii ).increment_count( atom_mask_large );
	}


	score_is_current_ = false;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::decrement_from_cached
///
/// @brief
/// decrements the dot coverage count for this by the coverage stored
/// in the input RotamerDotsCache object
///
/// @detailed
///
/// @param
/// cache - [in] -  the RotamerDotsCache object representing sphere overlaps
///   with a single other residue
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// I'm not sure decrement_from_cached is ever called when using a partial copy
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerDots::decrement_from_cached( RotamerDotsCache const & cache )
{
	if ( incomplete_copy_ignoring_doubly_counted_spheres_ )
	{
		decrement_by_ = &cache;
	}

	if ( cache.partial_copy_ )
	{

		for (int ii = 1; ii <= cache.num_atoms_in_partial_copy_; ++ii)
		{
			if ( cache.original_->non_zero_overlap_( ii ) && small_sphere_needs_updating( ii ) )
			{	atom_counts_small_probe_(ii) -= cache.original_->atom_counts_small_probe_(ii);}
		}
		for (int ii = 1; ii <= cache.num_atoms_in_partial_copy_; ++ii)
		{
			if ( cache.original_->non_zero_overlap_( ii ) && large_sphere_needs_updating( ii ) )
			{	atom_counts_large_probe_(ii) -= cache.original_->atom_counts_large_probe_(ii);}
		}

		for (int ii = cache.num_atoms_in_partial_copy_ + 1; ii <= get_num_atoms(); ++ii)
		{
			if ( cache.non_zero_overlap_( ii ) && small_sphere_needs_updating( ii ))
			{	atom_counts_small_probe_(ii) -= cache.atom_counts_small_probe_(ii);}
		}
		for (int ii = cache.num_atoms_in_partial_copy_ + 1; ii <= get_num_atoms(); ++ii)
		{
			if ( cache.non_zero_overlap_( ii ) && large_sphere_needs_updating( ii ))
			{	atom_counts_large_probe_(ii) -= cache.atom_counts_large_probe_(ii);}
		}
	}
	else
	{
		for (int ii = 1; ii <= get_num_atoms(); ++ii)
		{
			if ( cache.non_zero_overlap_( ii ) && small_sphere_needs_updating( ii ))
			{	atom_counts_small_probe_(ii) -= cache.atom_counts_small_probe_(ii);}
		}
		for (int ii = 1; ii <= get_num_atoms(); ++ii)
		{
			if ( cache.non_zero_overlap_( ii ) && large_sphere_needs_updating( ii ))
			{	atom_counts_large_probe_(ii) -= cache.atom_counts_large_probe_(ii);}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::get_score
///
/// @brief
///Report the score given the current dot counts.
///
/// @detailed
/// This method does not do any work figuring out who is overlapping
/// with this rotamer.  It assumes that work has been done.  Instead, it
/// return the score that reflects the dot counts currently held.
/// If the dot counts have not changed since the last time get_score() is called
/// then score_is_current_ will be true.  The method simply returns the member
/// variable score_.  If the dot counts have changed, score_ will not be current:
/// the method then iterates across all atoms and sum their scores.
/// It stores the score in score_ and set score_is_current_ to true.
/// It then returns score_.
/// The reason both score_ and score_is_current_ are "mutable" is so that they
/// can be modified inside this const function.
///
/// @param
///
/// @global_read
/// Wsasa
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

int RotamerDots::get_small_uncovered( int atom ) const
{
	if ( incomplete_copy_ignoring_doubly_counted_spheres_ && spheres_doubly_covered_(1, atom) )
		return 0;
	else
		return atom_counts_small_probe_( atom ).get_num_uncovered();
}

int RotamerDots::get_large_uncovered( int atom ) const
{
	if ( incomplete_copy_ignoring_doubly_counted_spheres_ && spheres_doubly_covered_(2, atom) )
		return 0;
	else
		return atom_counts_large_probe_( atom ).get_num_uncovered();
}

void
RotamerDots::double_counts()
{
	for (int ii = 1; ii <= num_atoms_; ++ii)
	{
		atom_counts_small_probe_( ii ).double_counts();
		atom_counts_large_probe_( ii ).double_counts();
	}
}

float
RotamerDots::get_score() const
{
	using namespace void_ns;
	using namespace numeric::constants::f;

	if( score_is_current_ ){
		return score_;
	}

	int small_uncovered, large_uncovered;
	float atom_radius, total_surface_area, fraction_small_uncovered,
	 fraction_large_uncovered, expose_small, expose_large, residue_sasa_small,
	 residue_sasa_large, residue_sasa_large_probe;
	float const four_pi = 4.0f * pi;
	int residue_type = coords_.get_residue();

	residue_sasa_large = 0.0f;
	residue_sasa_small = 0.0f;
	residue_sasa_large_probe = 0.0f;

	//assert( residue_type != 0 );
	if ( residue_type == 0 ) return 0.0f; // residue_type 0 means unassigned rotamer.

	for( int ii = 1; ii <= coords_.get_num_atoms(); ++ii){

		small_uncovered = get_small_uncovered( ii );
		large_uncovered = get_large_uncovered( ii );

		fraction_small_uncovered = static_cast< float >( small_uncovered ) /
		  atom_counts_small_probe_( ii ).get_total_dots();
		fraction_large_uncovered = static_cast< float >( large_uncovered ) /
			atom_counts_large_probe_( ii ).get_total_dots();

		//std::cerr << "atom: " << ii << " frac_small_uncov: " << fraction_small_uncovered;
		//std::cerr << ", frac_large_uncov: " << fraction_large_uncovered;
		//std::cerr << ", sm_un: " << small_uncovered;
		//std::cerr << ", lg_un: " << large_uncovered << std::endl;

		//if (ii == 17)
		//	std::cerr << "large:" << std::endl << atom_counts_large_probe_(ii) << std::endl << "small: " << std::endl << atom_counts_small_probe_(ii);
		atom_radius = coords_.get_atom_radius( ii );

		total_surface_area = four_pi * ( atom_radius * atom_radius );
		expose_small = fraction_small_uncovered * total_surface_area;
		expose_large = fraction_large_uncovered * total_surface_area;

		//std::cerr << " es: " << expose_small << " ";

		residue_sasa_small += expose_small;
		residue_sasa_large += expose_large;

		float atom_radius_probe = atom_radius + void_ns::probe_radius_large; //glb-temp
		total_surface_area = four_pi * ( atom_radius_probe * atom_radius_probe );
		float expose_large_probe = fraction_large_uncovered * total_surface_area;

		residue_sasa_large_probe += expose_large_probe;
		assert ( expose_large_probe >= 0 );

		assert ( expose_small >= 0 );
		assert ( expose_large >= 0 );
	}

	//Look up the average sasa5 for this range of sasa14 given the amino acid type
	//float offset = -1.0;
	//Objexx:SGM Unassigned elements accessed assuming they are zero
	//for ( int jj = 1; jj <= max_sasa_offset_bins; ++jj ) {
	//	if ( residue_sasa_large >= sasa_offset(1,jj,residue_type)
	//	 && residue_sasa_large < sasa_offset(2,jj,residue_type) ) {
	//		offset = sasa_offset(3,jj,residue_type);
	//		//break;
	//	}
	//}

	//if ( offset == -1.0 ) {
	//	std::cout << "sasa_large out of bounds!!" << SS( residue_sasa_large ) << std::endl;
	//	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	//}
	//score_ = residue_sasa_small - residue_sasa_large - offset;
	score_ = calc_residue_sasa_prob(
		residue_type,
		residue_sasa_small,
		residue_sasa_large,
		residue_sasa_large_probe);

	//take neg. log of sasaprob, apply cut off for low end
	score_ = (score_ < 0.00001 ? 5  : -std::log( score_ ) ) * param_pack::pack_wts.Wsasa();
	score_is_current_ = true;

	//std::cerr << "(o: " << offset << " rss " << residue_sasa_small << " rsl " << residue_sasa_large << ") ";
	return score_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDots::get_num_shared_atoms
///
/// @brief
/// returns the number of shared atoms between this rotamer and another
///
/// @detailed
///
/// @param
/// other - [in] - the other RotamerDots object
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
RotamerDots::get_num_shared_atoms( RotamerDots const & other ) const
{
	return coords_.get_num_shared_atoms( other.coords_ );
}


void
RotamerDots::write_dot_kinemage( std::ofstream & kinfile )
{
	using namespace void_ns;

	for (int ii = 1; ii <= get_num_atoms(); ++ii)
	{
		write_dotlist_header( kinfile, "1.4 exposed dots", "red");
		for (int jj = 1; jj <= DotSphere::NUM_DOTS_TOTAL; ++jj)
		{
			if ( ! atom_counts_large_probe_( ii ).get_dot_covered( jj ) )
			{
				write_dot( kinfile, ii, jj, probe_radius_large );
			}
		}

		write_dotlist_header( kinfile, "SA", "red");
		for (int jj = 1; jj <= DotSphere::NUM_DOTS_TOTAL; ++jj)
		{
			if ( ! atom_counts_large_probe_( ii ).get_dot_covered( jj ) )
			{
				write_dot( kinfile, ii, jj, 0 );
			}
		}

		write_dotlist_header( kinfile, "0.5 exposed dots", "green" );
		for (int jj = 1; jj <= DotSphere::NUM_DOTS_TOTAL; ++jj)
		{
			if ( ! atom_counts_small_probe_( ii ).get_dot_covered( jj ) )
			{
				write_dot( kinfile, ii, jj, probe_radius_small );
			}
		}

		write_dotlist_header( kinfile, "1.0 A probe Accessible", "green");
		for (int jj = 1; jj <= DotSphere::NUM_DOTS_TOTAL; ++jj)
		{
			if ( ! atom_counts_large_probe_( ii ).get_dot_covered( jj ) )
			{
				write_dot( kinfile, ii, jj, 0 );
			}
		}

		write_dotlist_header( kinfile, "void surface", "blue");
		for (int jj = 1; jj <= DotSphere::NUM_DOTS_TOTAL; ++jj)
		{
			if ( atom_counts_large_probe_( ii ).get_dot_covered( jj )  &&
				! atom_counts_small_probe_( ii ).get_dot_covered( jj ) )
			{
				write_dot( kinfile, ii, jj, 0 );
			}
		}

		//write_dotlist_header( kinfile, "all dots", "white");
		//for (int jj = 1; jj <= DotSphere::NUM_DOTS_TOTAL; ++jj)
		//{
		//	write_dot( kinfile, ii, jj, 0 );
		//}
	}
}

void
RotamerDots::write_dotlist_header(
	std::ofstream & kinfile,
	std::string master_name,
	std::string color )
{
	kinfile << "@dotlist master= {" << master_name << "} color= " << color << "\n";
}

void
RotamerDots::write_dot(
	std::ofstream & kinfile,
	int atom,
	int dot,
	float radius
)
{
	static numeric::xyzVector< float > coord;
	coord = coords_.get_atom_coords_xyz( atom );
	coord += (radius + coords_.get_atom_radius( atom )) * get_dot_coord(dot);

	//int fullatom_atom_id = trie_order_2_fullatom(atom, get_residue(), 1);
	write_dot( kinfile, coord, "dot" );
}

void
RotamerDots::write_dot(
	std::ofstream & kinfile,
	numeric::xyzVector< float > const & coord,
	std::string atname
)
{
	static std::string last_atname = "";
	if ( last_atname == atname )
	{
		kinfile << "{\"} P " << coord.x() << " " << coord.y() << " " << coord.z() << "\n";
	}
	else
	{
		kinfile << "{" << atname << "} P " << coord.x() << " " << coord.y() << " " << coord.z() << "\n";
		last_atname = atname;
	}
}

numeric::xyzVector< float > const &
RotamerDots::get_dot_coord( int dot_id )
{
	if ( ! dot_sphere_coordinates_initialized_ )
	{
		initialize_dot_sphere_coordinates_from_file();
	}
	return dot_sphere_coordinates_( dot_id );
}

void
RotamerDots::initialize_dot_sphere_coordinates_from_file()
{
	dot_sphere_coordinates_initialized_ = true;
	dot_sphere_coordinates_.dimension( DotSphere::NUM_DOTS_TOTAL );

	std::ifstream dotfile("sphere.txt");

	for (int ii = 1; ii <= DotSphere::NUM_DOTS_TOTAL; ++ii)
	//for (int ii = DotSphere::NUM_DOTS_TOTAL; ii >= 1; --ii )
	{
		float x, y, z;
		dotfile >> x >> y >> z;
		dot_sphere_coordinates_( ii ) = -1 * numeric::xyzVector< float >( x,y,z );
	}
	dotfile.close();
}

void
RotamerDots::zero()
{
	for ( int ii = 1; ii <= get_num_atoms(); ++ii )
	{
		atom_counts_large_probe_( ii ).zero();
		atom_counts_small_probe_( ii ).zero();
	}
	spheres_doubly_covered_ = false;
	incomplete_copy_ignoring_doubly_counted_spheres_ = false;
	original_ = 0;
	decrement_by_ = 0;
	increment_by_ = 0;
	score_is_current_ = false;
}

void
RotamerDots::make_first_two_atoms_overlap()
{
	using namespace numeric;
	using namespace fullatom_sasa;
	using namespace void_ns;

	FArray1D_ubyte at1_large( fullatom_sasa::nbytes, static_cast<ubyte> (0) );
	FArray1D_ubyte at2_large( fullatom_sasa::nbytes, static_cast<ubyte> (0) );
	FArray1D_ubyte at1_small( fullatom_sasa::nbytes, static_cast<ubyte> (0) );
	FArray1D_ubyte at2_small( fullatom_sasa::nbytes, static_cast<ubyte> (0) );
	float sqr_dist;

	get_2_way_atom_atom_coverage(
		coords_.get_atom_coords( 1 ), coords_.get_atom_type( 1 ),
		coords_.get_atom_coords( 2 ), coords_.get_atom_type( 2 ),
		at1_small, at2_small,
		at1_large, at2_large,
		sqr_dist);

	atom_counts_large_probe_( 1 ).increment_count( at1_large );
	atom_counts_large_probe_( 2 ).increment_count( at2_large );
	atom_counts_small_probe_( 1 ).increment_count( at1_small );
	atom_counts_small_probe_( 2 ).increment_count( at2_small );

	for (int ii = 1; ii <= DotSphere::NUM_DOTS_TOTAL; ++ii)
	{
		xyzVector< float > dotcoord = (coords_.get_atom_radius(1) + probe_radius_large)
			* get_dot_coord(ii) + coords_.get_atom_coords_xyz( 1 );
		if ( distance_squared( dotcoord, coords_.get_atom_coords_xyz( 2 ) )
			< std::pow( coords_.get_atom_radius( 2 ) + probe_radius_large , 2 ) )
		{
			std::cerr << "1.4 -- 1 " << ii << " "<< dotcoord.x() << ", " << dotcoord.y() << ", " << dotcoord.z() << std::endl;
		}
	}

	std::cerr << "------\nAccording To Input File:\n";
	for (int ii = 1; ii <= DotSphere::NUM_DOTS_TOTAL; ++ii)
	{
		if (atom_counts_large_probe_( 1 ).get_dot_covered( ii ) )
		{
			xyzVector< float > dotcoord = (coords_.get_atom_radius(1) + probe_radius_large)
				* get_dot_coord(ii) + coords_.get_atom_coords_xyz( 1 );
			std::cerr << "1.4 -- 1: " << ii << " " << dotcoord.x() << ", " << dotcoord.y() << ", " << dotcoord.z() << std::endl;
		}
	}

}


//----------------------------------------------------------------------------//
//------------------------- Rotamer Dots Cache Class -------------------------//
//----------------------------------------------------------------------------//


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDotsCache::RotamerDotsCache
///
/// @brief
/// default constructor
///
/// @detailed
/// allocates enough space for any amino acid
///
/// @param
///
/// @global_read
/// param::MAX_ATOM
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerDotsCache::RotamerDotsCache()
:
	atom_counts_large_probe_( param::MAX_ATOM() ),
	atom_counts_small_probe_( param::MAX_ATOM() ),
	non_zero_overlap_( param::MAX_ATOM(), false ),
	partial_copy_( false ),
	num_atoms_in_partial_copy_( -1 ),
	original_( 0 )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDotsCache::~RotamerDotsCache
///
/// @brief
/// destructor
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerDotsCache::~RotamerDotsCache(){}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDotsCache::RotamerDotsCache
///
/// @brief
/// copy constructor
///
/// @detailed
///
/// @param
/// rhs - [in] - the cache to copy
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerDotsCache::RotamerDotsCache( RotamerDotsCache const & rhs )
:
	atom_counts_large_probe_( rhs.atom_counts_large_probe_ ),
	atom_counts_small_probe_( rhs.atom_counts_small_probe_ ),
	non_zero_overlap_( rhs.non_zero_overlap_),
	partial_copy_( rhs.partial_copy_ ),
	num_atoms_in_partial_copy_( rhs.num_atoms_in_partial_copy_ ),
	original_( rhs.original_ )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDotsCache::operator =
///
/// @brief
/// assignment operator
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
RotamerDotsCache const &
RotamerDotsCache::operator = ( RotamerDotsCache const & rhs )
{
	copy(rhs);
	return *this;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDotsCache::copy
///
/// @brief
/// copies the contents of one cache into another
///
/// @detailed
/// if the other cache object represents a partial copy of this cache object,
/// then some work has been saved.  A partial copy may be assigned only once
/// and retaint its pointer to the original.  At the end of this method,
/// both this and rhs have their partial_copy_ flags/pointers set to false/null
///
/// @param
/// rhs - [in] - the cache to copy
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerDotsCache::copy( RotamerDotsCache const & rhs )
{
	if ( ! rhs.partial_copy_ || rhs.original_ != this )
	{
		atom_counts_large_probe_ = rhs.atom_counts_large_probe_;
		atom_counts_small_probe_ = rhs.atom_counts_small_probe_;
		non_zero_overlap_ = rhs.non_zero_overlap_;
	}
	else
	{
		int const natoms = atom_counts_large_probe_.size1();
		for (int ii = rhs.num_atoms_in_partial_copy_ + 1; ii <= natoms; ++ii)
		{
			atom_counts_large_probe_( ii ) = rhs.atom_counts_large_probe_( ii );
			atom_counts_small_probe_( ii ) = rhs.atom_counts_small_probe_( ii );
			non_zero_overlap_( ii ) = rhs.non_zero_overlap_( ii );
		}
	}

	original_ = 0;
	partial_copy_ = false;

	rhs.original_ = 0;
	rhs.partial_copy_ = false;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDotsCache::copy_some_and_zero
///
/// @brief
/// makes this a partial copy of the cache rhs.  Zeros the portions
/// of this not included in the partial copy.
///
/// @detailed
///
/// @param
/// rhs - [in] - the cache to copy
/// num_to_copy - [in] - the number of atoms to partially copy
/// num_atoms - [in] - the number of atoms total in this rotamer; the method
///   zeros the DotSphere counts for the atoms in the range
///   [num_to_copy + 1 .. num_atoms ]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerDotsCache::copy_some_and_zero
(
	RotamerDotsCache & rhs,
	int num_to_copy,
	int num_atoms
)
{
	assert( num_to_copy <= num_atoms );
	partial_copy_ = true;
	original_ = & rhs;
	num_atoms_in_partial_copy_ = num_to_copy;

	//TEMP
	//for (int ii = 1; ii <= num_to_copy; ++ii)
	//{
	//	atom_counts_large_probe_( ii ) = rhs.atom_counts_large_probe_( ii );
	//	atom_counts_small_probe_( ii ) = rhs.atom_counts_small_probe_( ii );
	//	non_zero_overlap_( ii ) = rhs.non_zero_overlap_( ii );
	//}
	for (int ii = num_to_copy + 1; ii <= num_atoms; ++ii)
	{
		atom_counts_large_probe_( ii ).zero();
		atom_counts_small_probe_( ii ).zero();
		non_zero_overlap_( ii ) = false;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDotsCache::increment_count_for_some
///
/// @brief
/// increments the dot coverage counts for some of the atoms in the cache
///
/// @detailed
/// increments coverage counts for atoms in the range
/// [num_to_ignore + 1 .. num_atoms ].  Default num_to_ignore parameter
/// is defined 0 in RotamerDots.h
///
/// @param
/// covered_small - [in] - compact ubyte array of dots; '1' for any covered dot
///   for the vdw + 0.5 A sphere
/// covered_large - [in] - compact ubyte array of dots; '1' for any covered dot
///   for the vdw + 1.4 A sphere
/// num_atoms - [in] - the number of atoms in this rotamer.  Arguably, num_atoms
///   should be a member of the class
/// num_to_ignore - [in] - the number of atoms whose count should not be
///   incremented.  The first num_to_ignore atoms are those atoms.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
 RotamerDotsCache::increment_count_for_some
 (
	FArray2DB< ubyte > & covered_small,
	FArray2DB< ubyte > & covered_large,
	int num_atoms,
	int num_to_ignore
)
{
	assert( ! partial_copy_ || num_to_ignore == num_atoms_in_partial_copy_ );
	for (int ii = num_to_ignore + 1; ii <= num_atoms; ++ii )
	{
		if ( ! non_zero_overlap_( ii )) continue;

		FArray1Da_ubyte atom_mask_small( covered_small( 1, ii), fullatom_sasa::nbytes);
		FArray1Da_ubyte atom_mask_large( covered_large( 1, ii), fullatom_sasa::nbytes);

		atom_counts_small_probe_( ii ).increment_count( atom_mask_small );
		atom_counts_large_probe_( ii ).increment_count( atom_mask_large );

	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerDotsCache::zero
///
/// @brief
/// sets the dot counts to zero for all atoms
///
/// @detailed
/// if the cache already knows the atom's counts are uniformly 0, it skips it
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl glb
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerDotsCache::zero()
{
	int num_atoms = static_cast< int > (atom_counts_large_probe_.size1() );
	for (int ii = 1; ii <= num_atoms; ++ii)
	{
		if ( non_zero_overlap_( ii ) )
		{
			atom_counts_small_probe_(ii).zero();
			atom_counts_large_probe_(ii).zero();
		}
	}
	non_zero_overlap_ = false;
	partial_copy_ = false;
	original_ = 0;
}

void
RotamerDotsCache::write_to_compact_array( FArray3D_ubyte & compact ) const
{
	assert( compact.size3() == 2 );
	int num_atoms_to_write = compact.size2();
	for (int ii = 1; ii <= num_atoms_to_write; ++ii)
	{
		FArray1Da_ubyte small_compact( compact(1, ii, 1 ) );
		atom_counts_small_probe_( ii ).write_to_compact_array( small_compact );

		FArray1Da_ubyte large_compact( compact(1, ii, 2 ) );
		atom_counts_large_probe_( ii ).write_to_compact_array( large_compact );
	}
}

bool
RotamerDotsCache::any_overlap() const
{
	return true; //unimplemented
}

} //end namespace pack


