// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1 $
//  $Date: 2006/10/19 20:16:48 $
//  $Author: plato $

#ifndef ROTAMER_DOTS_H
#define ROTAMER_DOTS_H


//Rosetta Headers
#include "aaproperties_pack.h"
#include "fullatom_sasa_ns.h"
#include "ligand_ns.h"

//Objexx Headers
#include <ObjexxFCL/ObjexxFCL.Project.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>

//Utilitiy Headers
#include <numeric/xyzVector.hh>

//C++ Headers
#include <iosfwd>


namespace pack {

class RotamerCoords;
class DotSphere;
class RotamerDots;
class RotamerDotsCache;


///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
typedef struct
{
	float coord_[3];
	int atom_type_;
	float atom_radius_;
	float subtree_interaction_sphere_radius_;
} AtomInfoForSASA;


class RotamerCoords{
private:
	FArray1D< AtomInfoForSASA > coords_;
	//FArray1D_float subtree_sphere_radii_; //apl For 2nd Implementation
	int aa_;
	int aa_variant_;
	int natoms_;

	void initialize_coords_trie_order( FArray2DB_float const & coords );

public:
	RotamerCoords();
	~RotamerCoords();
	RotamerCoords( RotamerCoords const & rhs);

	//main constructor
	RotamerCoords( int aa, int aav, FArray2DB_float const & coords );

	void
	set_coords( int aa, int aav, FArray2DB_float const & coords );

	inline
	int get_num_atoms() const
	{
		return natoms_;
	}

	int get_residue() const {return aa_;}
	int get_aav() const {return aa_variant_;}

	inline
	FArray1Da_float const get_atom_coords( int atom ) const
	{
		assert( atom > 0 && atom <= get_num_atoms() );
		return FArray1Da_float( coords_(atom).coord_[0], 3);
	}

	inline
	numeric::xyzVector< float > const get_atom_coords_xyz( int atom ) const
	{
		assert( atom > 0 && atom <= get_num_atoms() );
		return numeric::xyzVector< float >( coords_(atom).coord_[0],
			coords_(atom).coord_[1], coords_(atom).coord_[2] );
	}

	inline
	int get_atom_type( int atom ) const {return coords_( atom ).atom_type_;}

	inline
	float get_atom_radius( int atom ) const {return coords_( atom ).atom_radius_;}

	inline
	float get_subtree_interaction_sphere_radius( int atom ) const
	{return coords_( atom ).subtree_interaction_sphere_radius_; }

	inline
	void copy( RotamerCoords const & rhs )
	{
		aa_ = rhs.aa_;
		aa_variant_ = rhs.aa_variant_ ;
		natoms_ = rhs.natoms_;
		coords_ = rhs.coords_;
	}

	inline
	RotamerCoords const & operator = ( RotamerCoords const & rhs )
	{
		copy( rhs );
		return *this;
	}

	int get_num_shared_atoms( RotamerCoords const & other ) const;

};

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
class DotSphere{
private:
	//1 char for each dot
	//Each char represents a count of the number of whole residues
	//that overlap with a particular dot on this atom.
	//ASSUMPTION: a single atom will never be covered by more than 255
	//residues.
	//FArray1D< unsigned char > dots_coverage_count_;

	//mirrors extern nbytes - but const in header can't read from .cc file?
	static int const NUM_BYTES_IN_DOT_SPHERE_OVERLAP_ARRAYS = 21;
	static int const NUM_COUNTS_TO_ALLOCATE = NUM_BYTES_IN_DOT_SPHERE_OVERLAP_ARRAYS * 8;

	unsigned char dots_coverage_count_[ NUM_COUNTS_TO_ALLOCATE ];
	mutable int num_covered_;
	mutable bool num_covered_current_;
	mutable bool all_dot_counts_two_or_more_;

public:
	inline
	void count_num_covered() const;

private:
	void invert();

public:
	DotSphere();
	~DotSphere();
	DotSphere( DotSphere const & rhs );

	inline
	void copy( DotSphere const & rhs )
	{
		memcpy(dots_coverage_count_, rhs.dots_coverage_count_, NUM_COUNTS_TO_ALLOCATE);
		num_covered_ = rhs.num_covered_;
		num_covered_current_ = rhs.num_covered_current_;
		all_dot_counts_two_or_more_ = rhs.all_dot_counts_two_or_more_;
	}


	inline
	DotSphere const & operator = ( DotSphere const & rhs )
	{
		copy( rhs );
		return *this;
	}

	inline
	void zero();

	void non_inlined_zero();

	inline
	int get_total_dots() const;
	//get and set count for each atom
	//Input: bit representation for atom dots covered by a single other residue

	inline
	void increment_count( FArray1DB_ubyte const & );

	void non_inlined_increment_count( FArray1DB_ubyte const & bytearray );

	inline
	int get_num_uncovered() const;

	inline
	int get_num_covered() const;

	inline
	DotSphere const & operator -= ( DotSphere const & rhs );

	inline
	DotSphere const & operator += ( DotSphere const & rhs );
	void print( std::ostream & os) const;

	bool get_dot_covered( int dot_index );

	bool get_all_counts_two_or_more() {
		//if ( ! num_covered_current_ ){ count_num_covered(); }
		return all_dot_counts_two_or_more_;
		//return false;
	}

	void write_to_compact_array( FArray1DB_ubyte & compact ) const;

	void
	double_counts();

	static int const NUM_DOTS_TOTAL = 162;
};

std::ostream & operator << ( std::ostream & os, DotSphere const & ds );

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
class RotamerDots{
private:
	RotamerCoords coords_;
	int num_atoms_;
	FArray1D< DotSphere > atom_counts_large_probe_;
	FArray1D< DotSphere > atom_counts_small_probe_;
	FArray2D_bool spheres_doubly_covered_;
	mutable bool incomplete_copy_ignoring_doubly_counted_spheres_;
	mutable RotamerDots * original_;
	mutable RotamerDotsCache const * decrement_by_;
	mutable RotamerDotsCache const * increment_by_;

	mutable float score_;
	mutable bool score_is_current_;

	int get_small_uncovered( int atom ) const;
	int get_large_uncovered( int atom ) const;
	inline
	bool
	small_sphere_needs_updating( int atom ) const
	{	return ! ( incomplete_copy_ignoring_doubly_counted_spheres_ && spheres_doubly_covered_(1, atom ) ); }
	inline
	bool
	large_sphere_needs_updating( int atom ) const
	{	return ! ( incomplete_copy_ignoring_doubly_counted_spheres_ && spheres_doubly_covered_(2, atom ) ); }

	void copy_non_procrastinated_dot_spheres( RotamerDots const & rhs );
	void complete_procrastinated_decrement( RotamerDots const & rhs );
	void complete_procrastinated_increment( RotamerDots const & rhs );
	void detect_double_covereage_in_procrastinated_spheres( RotamerDots const & rhs );

  void get_overlap_cache
	(
		RotamerDots const & other,
		RotamerDotsCache & others_dots_covered_by_this,
		RotamerDotsCache & this_dots_covered_by_other
	) const;



	void
	write_dotlist_header(
		std::ofstream & kinfile,
		std::string master_name,
		std::string color );

	void write_dot( std::ofstream & kinfile, int atom, int dot, float radius );
	void write_dot( std::ofstream & kinfile,
		numeric::xyzVector< float > const & coord,
		std::string atname );
	numeric::xyzVector< float > const & get_dot_coord( int dot_id );
	void initialize_dot_sphere_coordinates_from_file();

	static bool dot_sphere_coordinates_initialized_;
	static FArray1D< numeric::xyzVector< float > > dot_sphere_coordinates_;

public:

	RotamerDots();
	~RotamerDots();
	RotamerDots( RotamerDots const & rhs );

	void
	set_coords( RotamerCoords const & coords );

	inline
	int
	get_num_atoms() const
	{
		return num_atoms_;
	}

	void
	print() const;

	void copy(RotamerDots const & rhs);

	void copy_standard(RotamerDots const & rhs);

	//invokes copy, returns this
	inline
	RotamerDots const & operator = ( RotamerDots const & rhs )
	{
		copy( rhs );
		return *this;
	}

	void
	copy_spheres_not_doubly_covered( RotamerDots * other );

	//Do two rotamers have any sphere overlaps?
	bool overlaps( RotamerDots const & other ) const;

	//Add rotamer coverage counts for dots on both this and other.
	//sets score_is_current_ to false on both this and rhs
	void increment_both(
		RotamerDots & other
	);

	//Add rotamer coverage counts for dots on both this and other.
	//Cache's the overlap this and other have with each other for greater efficiency.
	//The second parameter are the dots on the surface of other_rotamer that this overlaps.
	//The third parameter are the dots on the surface of this that other_rotamer overlaps.
	//sets score_is_current_ to false on both this and rhs
	inline
	void increment_both_and_cache(
		RotamerDots & other_rotamer,
		RotamerDotsCache & others_dots_covered_by_this,
		RotamerDotsCache & this_dots_covered_by_other
	)
	{
		get_overlap_cache( other_rotamer,
			others_dots_covered_by_this,
			this_dots_covered_by_other);

		increment_from_cached( this_dots_covered_by_other );
		other_rotamer.increment_from_cached( others_dots_covered_by_this );
	}

	void
	increment_both_and_cache(
		RotamerDots & other_rotamer,
		RotamerDotsCache & others_dots_covered_by_this,
		RotamerDotsCache & this_dots_covered_by_other,
		FArray3DB< short > & alt_atom_overlap_mask_inds,
		FArray2DB< ubyte > & mask_this_covered_by_other_small,
		FArray2DB< ubyte > & mask_other_covered_by_this_small,
		FArray2DB< ubyte > & mask_this_covered_by_other_large,
		FArray2DB< ubyte > & mask_other_covered_by_this_large,
		int num_atoms_same
	);

	//Add rotamer coverage counts for dots on this object only, leaving other unaltered
	//sets score_is_current_ to false
	//void increment_this( RotamerDots const & other );

	void increment_this_and_cache(
		RotamerDots const & other,
		RotamerDotsCache & this_overlap_on_other
	);

	void increment( Ligand & ligand );

	//Add rotamer coverage counts produced by self overlap
	//sets score_is_current_ to false
	void increment_self_overlap();

	void decrement_from_cached( RotamerDotsCache const & cached_dot_overlap );
	void increment_from_cached( RotamerDotsCache const & cached_dot_overlap );

	void
	force_rescore() const {score_is_current_ = false;}

	void
	double_counts();


	//Report the score given the current dot counts.
	//That is - this method does not do any work figuring out who is overlapping
	//with this rotamer.  It assumes that work has been done.  Instead, it
	//return the score that reflects the dot counts currently held.
	//If the dot counts have not changed since the last time get_score() is called
	//then score_is_current_ will be true.  The method simply returns the member
	//variable score_.  If the dot counts have changed, score_ will not be current:
	//the method then iterates across all atoms and sum their scores.
	//It stores the score in score_ and set score_is_current_ to true.
	//It then returns score_.
	//The reason both score_ and score_is_current_ are "mutable" is so that they
	//can be modified inside this const function.
	float get_score() const;
	int get_num_shared_atoms( RotamerDots const & other) const;

	void write_dot_kinemage( std::ofstream & kinfile );
	void zero();
	void make_first_two_atoms_overlap();

};


///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
class RotamerDotsCache
{
public:
	RotamerDotsCache();
	~RotamerDotsCache();
	RotamerDotsCache( RotamerDotsCache const & rhs );
	RotamerDotsCache const & operator = ( RotamerDotsCache const & rhs );

	void copy( RotamerDotsCache const & rhs );

	void copy_some_and_zero(
		RotamerDotsCache & rhs,
		int num_to_copy,
		int num_atoms );

	void
	increment_count_for_some(
		FArray2DB< ubyte > & covered_small,
		FArray2DB< ubyte > & covered_large,
		int num_atoms,
		int num_to_ignore = 0);

	//set all counts to 0 for all atoms
	void zero();

	inline
	void set_atom_has_overlap( int atom )
	{	non_zero_overlap_( atom ) = true; }

	FArray1D_bool const &
	get_non_zero_overlap() const { return non_zero_overlap_;}

	void
	set_non_zero_overlap( FArray1D_bool const & nzo )
	{	non_zero_overlap_ = nzo;}

	void write_to_compact_array( FArray3D_ubyte & compact ) const;
	bool any_overlap() const;

	friend class RotamerDots;
private:
	FArray1D< DotSphere > atom_counts_large_probe_;
	FArray1D< DotSphere > atom_counts_small_probe_;
	FArray1D_bool non_zero_overlap_;

	mutable bool partial_copy_;
	int num_atoms_in_partial_copy_;
	mutable RotamerDotsCache * original_;
};


} // namespace pack

#endif
