// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7739 $
//  $Date: 2006-03-23 14:43:49 -0500 (Thu, 23 Mar 2006) $
//  $Author: monica $
//
//  @begin SplicedProtein.cc
//
//  @brief
//  Creates a map with protein insertion information
//
//  @detailed:
//  The map shows where the original pdb information came from and how it corresponds in to the
//  information in the new pdb of a combined pose that will be output.
//
//  @authors: Monica Berrondo
//
//  @last_modified: April 07 2006


// Rosetta Headers
#include "SplicedProtein.h"
#include "pose.h"
#include "domins_ns.h"

// C++ Headers
#include <iostream>

// ObjexxFCL Headers

void SplicedProtein::reset_spliced_protein_map()
{
	using namespace domins_ns;

	if ( !spliced_protein_res_info.empty() ) spliced_protein_res_info.clear();
}
//////////////////////////////////////////////////////////////////////////////
/// @begin SplicedProtein::pushback_segment
///
/// @brief
/// inserts information of a segment into a map
///
/// @detailed
/// loops through the residues of a segment and sets the information
/// just uses a default repeated residue value of 0
/// must call set_repetition to change
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo
///
/// @last_modified: April 07 2006
////////////////////////////////////////////////////////////////////////////////
void SplicedProtein::pushback_segment(
		std::string const name,
		std::string const type,
		int const protein_start,
		int const segment_start,
		int const segment_size,
		pose_ns::Pose const & pose
 )
 {
		using namespace domins_ns;
		using namespace pose_ns;

		int residue_repetition (0);

		std::cout << "size of spliced_map: " << spliced_protein_res_info.size() << std::endl;
		for (int i=segment_start, k=protein_start; i<segment_size+segment_start; ++i, ++k) {
			spliced_protein_res_info.push_back(SplicedProtein(k, name, type, i, pose.pdb_info().pdb_res_num(i), residue_repetition));
		 }
 }

////////////////////////////////////////////////////////////////////////////
void SplicedProtein::set_repetition(
		int const protein_start,
		int const segment_start,
		int const segment_size,
		std::string const type,
		std::vector <int> repeated_residues,
		int const repetition
 )
 {
		using namespace domins_ns;
		using namespace pose_ns;

		int residue_repetition (0);

		Pose pose;
		if (type == "host") {
			pose = host_protein_pose;
		} else if (type == "insert") {
			pose = insert_protein_pose;
		}

		for (int i=protein_start, k=segment_start; k<segment_size+segment_start; ++i, ++k) {
			for (int j=0; j<(int)repeated_residues.size(); ++j) {
				if (pose.pdb_info().pdb_res_num(k) == repeated_residues.at(j)) {
					residue_repetition = repetition;
					break;
				}
				else
					residue_repetition = 0;
			}
			spliced_protein_res_info.at(i).residue_repetition = residue_repetition;
		 }
 }

//////////////////////////////////////////////////////////////////////////////
/// @begin SplicedProtein::get_combined_residue_from_pdb()
///
/// @brief
///		gets the residue number of the combined protein from the information of
//the initial proteins
///
/// @detailed
///		By passing in the name of the protein and the original protein's pdb residue
//number, return the new residue number
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo
///
/// @last_modified: April 07 2006
////////////////////////////////////////////////////////////////////////////////
void SplicedProtein::get_combined_residue_from_pdb(
		int &num_out,
		std::string name_in,
		int num_in,
		int start,
		int end,
		int resrep_in
 )
 {
		using namespace domins_ns;

		std::string name;
		int num, resrep;

		for (int i=start; i<end; ++i) {
			name = spliced_protein_res_info.at(i).original_pdb_filename;
			num = spliced_protein_res_info.at(i).original_protein_pdb_res_num;
			resrep = spliced_protein_res_info.at(i).residue_repetition;
			if ( name == name_in && num == num_in && resrep == resrep_in ) {
				num_out =  spliced_protein_res_info.at(i).fusion_protein_res_num;
				break;
			}
		}
 }

//////////////////////////////////////////////////////////////////////////////
/// @begin SplicedProtein::get_combined_residue()
///
/// @brief
///		gets the residue number of the combined protein from the information of
//the initial proteins
///
/// @detailed
///		By passing in the name of the protein and the original protein's residue
//number, return the new residue number
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo
///
/// @last_modified: April 07 2006
////////////////////////////////////////////////////////////////////////////////
void SplicedProtein::get_combined_residue(
		int &num_out,
		std::string name_in,
		int num_in,
		int start,
		int end,
		int resrep_in
 )
 {
		using namespace domins_ns;

		std::string name;
		int num, resrep;

		for (int i=start; i<end; ++i) {
			name = spliced_protein_res_info.at(i).original_pdb_filename;
			num = spliced_protein_res_info.at(i).original_protein_res_num;
			resrep = spliced_protein_res_info.at(i).residue_repetition;
			if ( name == name_in && num == num_in && resrep == resrep_in ) {
				num_out =  spliced_protein_res_info.at(i).fusion_protein_res_num;
				break;
			}
		}
 }

//////////////////////////////////////////////////////////////////////////////
/// @begin operator<< (ostream, SplicedProtein)
///
/// @brief
///		invokes the print on an SplicedProtein object
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo
///
/// @last_modified: April 08 2006
////////////////////////////////////////////////////////////////////////////////
std::ostream &operator<< ( std::ostream &os, SplicedProtein const &inmap )
{
	os << "\t " << inmap.fusion_protein_res_num << "\t\t\t" << inmap.original_pdb_filename << "\t\t" << inmap.original_protein_type << "\t\t" << inmap.original_protein_res_num << "\t\t\t" << inmap.original_protein_pdb_res_num << "\t\t" << inmap.residue_repetition << std::endl;
	return os;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin SplicedProtein::print_map()
///
/// @brief
///		print the contents of the insertion map
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo
///
/// @last_modified: April 08 2006
////////////////////////////////////////////////////////////////////////////////
void SplicedProtein::print_map()
{
	using namespace domins_ns;

	std::cout << "Spliced Protein Map:\n" << std::endl;
	std::cout << "map contents: fusion residue number | original pdb name | original protein type | original residue number | orignal pdb residue number | residue repetition" << std::endl;

	for ( signed int i=1; i<(int)spliced_protein_res_info.size(); ++i ) {
		std::cout << "map contents: " << spliced_protein_res_info.at(i);
	}
}
