// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 08:39:36 +0200 (Sun, 18 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "assemble_domains.h"
#include "after_opts.h"
#include "maps.h"
#include "maps_ns.h"
#include "files_paths.h"
#include "fold_abinitio.h"
#include "fragments.h"
#include "misc.h"
#include "monte_carlo.h"
#include "namespace_assemble_options.h"
#include "param.h"
#include "refold.h"
#include "relax_structure.h"
#include "cenlist.h"
#include "interface.h"
#include "runlevel.h"
#include "rotamer_trials.h"
#include "score_ns.h"
#include "score.h"
#include "random_numbers.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>


//Utility Headers
#include <utility/basic_sys_util.hh>

////////////////////////////////////////////////////////////////////////////////
/// @begin get_assemble_flag
///
/// @brief (logical) function that read status of assemble_flag
///
/// @detailed
///
/// @global_read
///              assemble_flag (assemble_domains.h)
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_assemble_flag()
{
	return assemble_options::assemble_flag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin extend_region
///
/// @brief make stretches of protein chain extended
///
/// @detailed
///
/// @global_read
///              insert_map : for knowing where to change phi/psi values ext.
///
/// @global_write
///              phi, psi, omerga arrays: change values to ext. values
///
/// @remarks
///              from an earlier, never checked-in version of Rosetta.
///              Part of the assemble mode for bencharking purposes mainly.
///              Given a native structure, extend the linker region.
///
/// @references
///
/// @authors
///              caz : Alex Zanghellini
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
extend_regions()
{
	using namespace misc;
	using namespace param;

	int total_insert;
	FArray1D_int insert_map( MAX_RES()() );

	std::cout << "extend_regions:: in extend mode" << std::endl;

	retrieve_insertmap(insert_map,total_insert);

	if ( total_insert == 0 ) {
		std::cout << "extend_regions::WARNING: no regions defined, " <<
		 "no residues to extend" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int j = 1; j <= total_insert; ++j ) {
		int const k = insert_map(j);
		phi(k) = init_phi;
		psi(k) = init_psi;
		omega(k) = init_omega;
	}

	refold(1,total_residue);
	monte_carlo_reset();
}


////////////////////////////////////////////////////////////////////////////////
/// @rethread_regions
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///   Andrew Wollacott
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
rethread_regions()
{
  using namespace misc;
	using namespace param;

  int total_insert;
  FArray1D_int insert_map( MAX_RES()() );

	std::cout << "rethread_regions: in rethread mode" << std::endl;

	retrieve_insertmap(insert_map, total_insert);

	if (total_insert == 0) {
		std::cout << "rethread_regions::WARNING: no regions defined," << std::endl;
		utility::exit(EXIT_FAILURE, __FILE__, __LINE__);
	}

	refold(1,total_residue);
	monte_carlo_reset();
}



////////////////////////////////////////////////////////////////////////////////
/// @begin perturbe_starting
///
/// @brief perturb linker positions of starting structure  a little bit
///        and output the structure (needed for tests prupose)
///
/// @detailed
///       each perturbation consists of linker_size /2 random choice of a position,
///       and for each a small angular increment is applied to the phi angle.
///
/// @global_read
///       insert_map, phi, psi tables
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///              caz : Alex Zanghellini
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
perturbe_starting()
{
	using namespace misc;
	using namespace param;
	using namespace scorefxns;

	int total_insert;
	int N_perturb = 0;
	int k;
	float phi_tmp, psi_tmp;
	float score, base_score=0.0;
	FArray1D_int insert_map( MAX_RES()() );

	std::cout << "perturbe_native():: in native perturbation mode" << std::endl;

	retrieve_insertmap(insert_map,total_insert);
	N_perturb = (int) total_residue/2;

	FArray1D_float phi_bkp( MAX_RES()() );
	FArray1D_float psi_bkp( MAX_RES()() );

	for (int i=1; i <= MAX_RES()(); ++i) {
		phi_bkp(i) = phi(i);
		psi_bkp(i) = psi(i);
	}

	base_score = score4();

	// perturb structure and avoid too many clashes
	for (int i=1; i <= MAX_ITER; ++i) {
		for (int j=1; j <= total_insert; ++j) {
			k = insert_map(j);
			phi(k) = phi_bkp(k);
			psi(k) = psi_bkp(k);
		}

		for (int j=1; j <= N_perturb; ++j) {
			int random_index = static_cast< int >( total_insert * ran3() )+1;

			k = insert_map(random_index);

			phi_tmp = phi(k) + ( ran3()-0.5 ) * 0.5 * (1- (float) i/MAX_ITER) ;
			psi_tmp = psi(k) + ( ran3()-0.5 ) * 0.5 * (1- (float) i/MAX_ITER);

			if( phi_tmp > 180.0 ){
				phi_tmp -= 360.0;
			}
			if( psi_tmp > 180.0 ){
				psi_tmp -= 360.0;
			}

			if( phi_tmp < -180.0 ){
				phi_tmp += 360.0;
			}
			if( psi_tmp < -180.0 ){
				psi_tmp += 360.0;
			}

			phi(k) = phi_tmp;
			psi(k) = psi_tmp;

		}

		refold(1,total_residue);

		score = score4();

		std::cout << "score after perturbation = " << score << std::endl;

		if( score < 0.0 || score - base_score < MAX_CLASH ){ break; }
	}

	monte_carlo_reset();

}


////////////////////////////////////////////////////////////////////////////////
/// @begin assemble_detect_interf_res
///
/// @brief set up domain interface residue list
///
/// @detailed
///     This function detects domain-interface residues, based on linker
///     definition and using the 8 A cut-off definition of interface.
///
///     This function is a modified copy of docking's docking_movement.cc
///
/// @global_read cendist array precalculated in cenlist.h
///
/// @global_write interface array in interface.h
///
/// @remarks
///
/// WARNING: as a the other functions in domain assembly, this function
///          assumes that there is only *one* linker and thus 2 domains.
///          Although more than 2 domains can of course be treated in a
///          recursive manner, a more elegant treatement would allow
///          multiple domains to be assembled simultaneously.
///
/// @references
///
/// @authors
/// Alex Z, from original function from Jeff Gray ?
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
assemble_detect_interf_res()
{
	using namespace cenlist_ns;
	using namespace interface;
	using namespace runlevel_ns;
	using namespace misc;
	using namespace param;
	using namespace assemble_options;

	int total_insert;
	FArray1D_int insert_map( MAX_RES()() );

	if (!bDetectInterface) {
		return;
	}

	int_res8 = false;
	for( int i=1; i <= max_docking_sites; ++i ) {
		int_pair_list8(i).clear();
		int_res_list8(i).clear();
	}

	if (! relax_repack_flag ) return;

	retrieve_insertmap(insert_map, total_insert);
	int min = insert_map(1);
	int max = insert_map(total_insert);

	//rhiju Before an error was triggered if min=1 or max=total_residues, but
	//rhiju we're facing some non-traditional cases where the "linker" goes to the end of the protein.
	if (min < 1 || max > total_residue) {
		std::cout << "WARNING: error with domain boundaries" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//rhiju Changed the code here to allow assembly of multiple domains.
	//rhiju The logic appears to be that we want any residues with
	//rhiju centroid distance less than 8 Angstroms to another residue to be
	//rhiju repacked -- unless they only make such pairwise interactions with
	//rhiju other residues within the same domain.

	//rhiju One more thing (remove this comment in the future, andrew or alex):
	//rhiju the previous code appears to add residue pairs from the linker *twice*
	//rhiju to int_pair_list8. Is that OK? I don't see any modes except for dock_symmetry
	//rhiju that really use int_pair_list8. Anyway, this double counting does not
	//rhiju occur in the new code.
// 	for (int i=1; i <= max; ++i) {
// 		for (int j=min; j <= total_residue; ++j) {
// 			if (cendist(i,j) < 64.0) {
// 				if (!int_res8(j)) {
// 					int_res_list8(2).push_back(j);
// 				}
// 				int_res8(i) = true;
// 				int_res8(j) = true;

// 				int_pair_list8(1).push_back(i);
// 				int_pair_list8(2).push_back(j);
// 			}
// 		}

// 		if (int_res8(i)) {
// 			int_res_list8(1).push_back(i);
// 		}

// 	}

	for (int i=1; i <= total_residue; ++i){
		// Scan through rest of protein until we get past this residue's domain to the next linker.
		int next_domain_start;
		for ( next_domain_start = i; next_domain_start <= total_residue && (! protein_maps::allow_insert(next_domain_start)); ++next_domain_start);
		//		std::cout << i << " " << next_domain_start << std::endl; // Yes this has been checked reasonably carefully.

		// Pretty similar to before.
		for (int j = next_domain_start; j <= total_residue; ++j){
 			if (cendist(i,j) < 64.0) {
 				if (!int_res8(j)) {
 					int_res_list8(2).push_back(j);
 				}
 				int_res8(i) = true;
 				int_res8(j) = true;

 				int_pair_list8(1).push_back(i);
 				int_pair_list8(2).push_back(j);
 			}
 		}

 		if (int_res8(i)) {
 			int_res_list8(1).push_back(i);
 		}
	}


}



////////////////////////////////////////////////////////////////////////////////
/// @begin assemble_update_allow_repack
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///     Andrew Wollacott
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
assemble_update_allow_repack()
{
	using namespace param;
	using namespace misc;
	using namespace protein_maps;
	using namespace interface;

	int total_insert;
	FArray1D_int insert_map( total_residue );

	// retrieve domain interface residues
	for (int i=1; i <= total_residue; ++i) {
		if (int_res8(i)) {
			allow_repack(i) = true;
		} else {
			allow_repack(i) = false;
		}
	}

	// repack linker region in all cases
	retrieve_insertmap(insert_map, total_insert);
	for (int i=insert_map(1); i <= insert_map(total_insert); ++i) {
		allow_repack(i) = true;
	}
}



void
assemble_set_allow_rottrial()
{
	using namespace param;
	using namespace misc;
	using namespace protein_maps;
	using namespace interface;
	using namespace assemble_options;

	if (!bDetectInterface) {
		return;
	}

	FArray1D_bool assemble_rottrial( total_residue );
	int total_insert;
	FArray1D_int insert_map( MAX_RES()() );

	for (int i=1; i <= total_residue; ++i) {
		if (int_res8(i) ) {
			assemble_rottrial(i) = true;
		} else {
			assemble_rottrial(i) = false;
		}
	}

	retrieve_insertmap(insert_map, total_insert);
	if (total_insert == 0) {
		std::cout << "WARNING: regiosn file not defined" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for (int i=insert_map(1); i <= insert_map(total_insert); ++i) {
		assemble_rottrial(i) = true;
	}

	set_allow_rotamer_trials(assemble_rottrial, total_residue);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin assemble_domains
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///      rewritten by Alex. Zanghellini on 04/15/04
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
assemble_domains()
{
	using namespace files_paths;
	using namespace assemble_options;

//az special submode: extend
	if ( truefalseoption("extend" ) ) {
		extend_regions();
		return;
	}

	if ( ! truefalseoption("regions") ) {
		std::cout << "calling assemble domains without -regions is meaningless" << std::endl;
		std::cout << "use -abinito instead for folding a complete chain" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( truefalseoption("thread") ) {
		rethread_regions();
		return;
	}

	if ( truefalseoption("perturbe_starting") ) {
		perturbe_starting();
		return;
	}

//rhiju abrelax is ab initio followed by relax
	bool const abrelax_mode = truefalseoption("abrelax");

//az calls standard ab_initio if no -relax option for speedup, cycles numbers
//az in abinitio should be set to around 2*(total_insert)*max_top_N_frag

	if (  (! truefalseoption("relax")) || abrelax_mode ) {
		choose_frag_set_check_ss(false);
		fold_abinitio();
	}

	if ( truefalseoption("relax") || abrelax_mode ){
//az if relax is provided (only in full_atom mode), the assembler relaxes the
//az linker (specific protocol, with some minimization) in current decoy

		if ( ! input_fa ) {
			std::cout <<
			 "Not in full-atom mode, centroid relax of the linker region only" << std::endl;
			centroid_relax();
		} else {
			std::cout <<
			 "In full-atom mode, centroid+fullatom relax of the linker region only" << std::endl;

			relax_repack_flag = true;
			score_enable_rotamer_trials(true);
			fullatom_relax();
			relax_repack_flag = false;
		}
	}

}



void
should_detect_interface(bool bDetect) {
	using namespace assemble_options;
	bDetectInterface = bDetect;
}
