// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7212 $
//  $Date: 2006-01-04 07:49:01 -0800 (Wed, 04 Jan 2006) $
//  $Author: pbradley $


// Rosetta Headers
#include "atom_tree_routines.h"
#include "after_opts.h"
#include "aaproperties_pack.h"
#include "disulfides.h"
#include "fullatom.h"
#include "pose.h"
#include "pose_io.h"
#include "read_aaproperties.h"
#include "score.h" //Possu? score12?
#include "util_vector.h"

// Numeric Headers
#include <numeric/xyz.functions.hh> // for dihedral()
#include <numeric/trig.functions.hh>


typedef numeric::xyzVector_float Vec;


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// Several experimental routines for addressing the degrees of freedom
// in Phil's atom trees. At some point, it might be worth making these
// member functions of the atom tree object!
//
// Also, some of the stuff with allowing degrees of freedom to move
// might best belong in an AllowMove object (lin may have a start on
// this).
//
// Rhiju, March 2007
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

bool
get_vary_bond_geometry_no_hydrogens()
{
	static bool init = {false};
	static bool vary_bond_geometry_no_hydrogens;

	if (!init) {
		vary_bond_geometry_no_hydrogens = truefalseoption("vary_bond_geometry_no_hydrogens");
		init = true;
	}

	return vary_bond_geometry_no_hydrogens;
}

//////////////////////////////////////////////////////////////////////////////
bool
get_tether_to_ideal_geometry_flag()
{
	static bool init = {false};
	static bool tether_to_ideal_geometry;

	if (!init){
		tether_to_ideal_geometry = truefalseoption("tether_to_ideal_geometry");
		init = true;
	}

	return tether_to_ideal_geometry;
}


///////////////////////////////////////////////////////////////////////////////
void
set_sidechain_bond_angle_allow_move(
	pose_ns::Pose & pose
)
{
	using namespace kin;
	int const nres( pose.total_residue() );
	for ( int i=1; i<= nres; ++i ) {
		if ( !param_aa::is_protein(pose.res(i)) || pose.res(i) == param_aa::aa_gly ||
				 !pose.get_allow_chi_move(i) ) continue;

		// c-beta atom
		Atom_id cbeta( 5, i );

		pose.set_allow_move( cbeta, PHI, true );
		pose.set_allow_move( cbeta, THETA, true );

		// cgamma
		int const cgamma
			( LookupByName(pose.res(i),pose.res_variant(i)," CG " ) );
		if ( cgamma >= 1) {
			pose.set_allow_move( Atom_id( cgamma, i ), THETA, true );
		}
	}

	pose.set_vary_bond_geometry_flag( true );
}


///////////////////////////////////////////////////////////////////////////////

void
add_ring_constraint(
		pose_ns::Pose & pose,
		cst_set_ns::Cst_set & cst_set,
		kin::Atom_id const ring_atom1_id,
		kin::Atom_id const ring_atom2_id,
		float const ring_tether
)
{
	using namespace numeric;
	using namespace kin;

	Coords_FArray_const coords( pose.full_coord());

	//	Add one distance constraint
	add_distance_constraint_from_2_atoms(
					 ring_atom1_id,
					 ring_atom2_id,
					 ring_tether, coords, cst_set);

	xyzVector_float const xyz1( coords.get_xyz( ring_atom1_id) );
	xyzVector_float const xyz2( coords.get_xyz( ring_atom2_id) );
	float const current_val = (xyz1 - xyz2).length();
	cst_set.add_atompair_constraint( ring_atom1_id, ring_atom2_id,
																	 current_val, ring_tether, false /*verbose*/ );

	//One dihedral constraint
	Atom_tree const * atom_tree = pose.atom_tree();
	Atom_id const ring_atom1_parent_id = atom_tree->atom(ring_atom1_id)->input_stub_atom0();
	Atom_id const ring_atom2_parent_id = atom_tree->atom(ring_atom2_id)->input_stub_atom0();
	add_dihedral_cst_with_dihedral_from_4_atoms(
					 ring_atom1_parent_id,
					 ring_atom1_id,
					 ring_atom2_id,
					 ring_atom2_parent_id,
					 ring_tether, coords, cst_set);

	//Two bond angle constraints.
	add_angle_cst_with_theta_from_3_atoms(
					 ring_atom1_parent_id,
					 ring_atom1_id,
					 ring_atom2_id,
					 ring_tether, coords, cst_set);

	add_angle_cst_with_theta_from_3_atoms(
					 ring_atom1_id,
					 ring_atom2_id,
					 ring_atom2_parent_id,
					 ring_tether, coords, cst_set);

}

//////////////////////////////////////////////////////////////////////////////////////////////
void
find_and_add_ring_constraints(
		kin::Atom_tree const * atom_tree,
		cst_set_ns::Cst_set * cst_set_p,
		pose_ns::Pose & pose_with_desired_geometry,
		int const i,
		int const aa,
		int const aav )
{
	using namespace cst_set_ns;
	using namespace kin;
	using namespace aaproperties_pack;

	static float const ring_tether( realafteroption("ring_tether", 200.0) );

	FArray3D_float const & full_coord( pose_with_desired_geometry.full_coord());

	std::map< std::pair<int,int>, bool >  bond_in_tree;
	float const BOND_LENGTH_CUTOFF( 2.0 );

	for ( int j = 1; j <= nheavyatoms( aa, aav ) ; ++j ) {
		Vec  xyz1( &full_coord(1,j,i) );
		Atom_id current_atom1_id( j, i );
		const Atom * current_atom1 = atom_tree->atom(current_atom1_id);

		for ( int k = j+1; k <= nheavyatoms( aa, aav ) ; ++k ) {
			Vec  xyz2( &full_coord(1,k,i) );
			Atom_id current_atom2_id( k, i );
			const Atom * current_atom2 = atom_tree->atom( current_atom2_id);

			//Are these atoms "bonded"?
			float const bond_length = (xyz1 - xyz2).length();

			if (bond_length < BOND_LENGTH_CUTOFF ){
				if ( current_atom1->input_stub_atom0() == current_atom2_id  ||
						 current_atom2->input_stub_atom0() == current_atom1_id ) continue;

				//				std::cout << "RING CLOSE? " << param_aa::aa_name3(aa) << " "  <<
				//					atom_name(j,aa,1) <<  " " << atom_name(k,aa,1) << std::endl;

				add_ring_constraint(pose_with_desired_geometry, *cst_set_p,
														current_atom1_id, current_atom2_id, ring_tether );

			}
		}
	}

}



void
add_all_bond_constraints(
												 pose_ns::Pose & pose
												 )
{
	using namespace cst_set_ns;
	using namespace kin;
	using namespace aaproperties_pack;

  // Dynamically allocate a cst_set, which will be pointed to by a pose.
	// The cst_set will delete itself when the pose no longer points to it, but this
	// requires that we call "remove_ptr_reference()" once the pose pointer is setup (below)
	Cst_set * cst_set_p;
	cst_set_p = new Cst_set();

	static float const ring_tether( realafteroption("ring_tether", 200.0) );

	FArray3D_float const & full_coord( pose.full_coord());

	float const BOND_LENGTH_CUTOFF( 2.0 );

	Atom_tree const * atom_tree = pose.atom_tree(); //Need the atom tree to see where jumps are.

	int const nres = pose.total_residue();

	for (int i = 1; i <= nres; ++i) {
		int const aa  = pose.res(i);
		int const aav = pose.res_variant(i);

		for ( int j = 1; j <= nheavyatoms( aa, aav ) ; ++j ) {
			Vec  xyz1( &full_coord(1,j,i) );
			Atom_id current_atom1_id( j, i );
			const Atom * current_atom1 = atom_tree->atom(current_atom1_id);

			for ( int k = j+1; k <= nheavyatoms( aa, aav ) ; ++k ) {
				Vec  xyz2( &full_coord(1,k,i) );
				Atom_id current_atom2_id( k, i );
				const Atom * current_atom2 = atom_tree->atom( current_atom2_id);

				//Are these atoms "bonded"?
				float const bond_length = (xyz1 - xyz2).length();

				if (bond_length < BOND_LENGTH_CUTOFF ){
					if ( current_atom1->input_stub_atom0() == current_atom2_id  ||
							 current_atom2->input_stub_atom0() == current_atom1_id ) continue;

					//					std::cout << "RING CLOSE? " << param_aa::aa_name3(aa) << " "  <<
					//						atom_name(j,aa,1) <<  " " << atom_name(k,aa,1) << std::endl;

					add_ring_constraint(pose, *cst_set_p,
															current_atom1_id, current_atom2_id, ring_tether );

				}
			}
		}
	}

	if (pose.constraints_exist()){
		cst_set_p -> collect( pose.constraints() );
	}
	pose.set_constraints( *cst_set_p );

	// this was allocated by "new" (above), and will cause a memory leak otherwise
	cst_set_p->remove_ptr_reference();

}

///////////////////////////////////////////////////////////////////////////////
void
create_totally_new_ideal_pose( pose_ns::Pose &  ideal_pose, pose_ns::Pose & pose )
{
	using namespace pose_ns;

	// setup ideal pose
	int nres = pose.total_residue();
	Fold_tree f = pose.fold_tree();
	ideal_pose.simple_fold_tree( nres );
	ideal_pose.set_fullatom_flag( true, false /*repack*/ );

	for (int i=1; i<= nres; ++i ) {
		ideal_pose.set_res         ( i, pose.res(i) );
		ideal_pose.set_res_variant ( i, pose.res_variant(i) );
		ideal_pose.set_phi         ( i, pose.phi(i) );
		ideal_pose.set_psi         ( i, pose.psi(i));
		ideal_pose.set_omega       ( i, pose.omega(i) );
		int const nchi( aaproperties_pack::nchi( pose.res(i), pose.res_variant(i) ) );
		for ( int chino=1; chino<= nchi; ++chino ) {
			ideal_pose.set_chi( chino, i, pose.chi(chino, i) );
		}
	}

	ideal_pose.copy_to_misc(); //force a refold.

	//	pose_set_termini( ideal_pose );
	ideal_pose.set_fold_tree( f );
	ideal_pose.setup_atom_tree();
}

/////////////////////////////////////////////////////////////////////////////////////
bool
check_torsion_deviation_OK(
														kin::Torsion_id & this_bond_torsion,
														float const bond_torsion_tether,
														pose_ns::Pose & pose,
														pose_ns::Pose & pose_with_desired_geometry)
{
	// If the deviation between the current value and the ideal value is too big,
	// maybe we don't want to enforce the constraint!

	float const current_torsion( pose.get_atom_tree_torsion(this_bond_torsion));
	float const ideal_torsion  ( pose_with_desired_geometry.get_atom_tree_torsion(this_bond_torsion));
	float const deviation = current_torsion - ideal_torsion;
	float const deviation_energy = deviation*deviation*bond_torsion_tether;

	if (deviation_energy > 20.0) {
		std::cout << "Torsion deviation too big? " << this_bond_torsion <<  current_torsion << " " << ideal_torsion << std::endl;
		return false;
	}
	return true;
}

/////////////////////////////////////////////////////////////////////////////////////
void
set_all_bond_distance_allow_move(
																 pose_ns::Pose & pose
																 ){
	pose_ns::Pose ideal_pose;
	create_totally_new_ideal_pose( ideal_pose, pose );
	set_all_bond_distance_allow_move( pose, ideal_pose );
}

void
set_all_bond_distance_allow_move(
																 pose_ns::Pose & pose,
																 pose_ns::Pose & pose_with_desired_geometry
)
{
	using namespace cst_set_ns;
	using namespace kin;

  // Dynamically allocate a cst_set, which will be pointed to by a pose.
	// The cst_set will delete itself when the pose no longer points to it, but this
	// requires that we call "remove_ptr_reference()" once the pose pointer is setup (below)
	Cst_set * cst_set_p;
	cst_set_p = new Cst_set();

	static float const stretch_factor( realafteroption("stretch_factor", 1.0));
	// Empirically, a spring constant of 10.0 gives a variation of
	// +/-0.02 Angstroms. That variation is on the same scale as
	// the empirically observed variation (0.01-0.03 Angstroms). See
	// Engh and Huber, Acta Cryst (1991), A47, p. 392-400.
	// The constant 10.0 seems awfully low, but this might be
	// due to some scaling of distance d.o.f. in the atom tree minimizer.
	static float const bond_length_tether( realafteroption("bond_length_tether", 10.0) );

	Atom_tree const * atom_tree = pose_with_desired_geometry.atom_tree(); //Need the atom tree to see where jumps are.

	int const nres( pose_with_desired_geometry.total_residue() );
	for (int i = 1; i <= nres; ++i) {
		int const aa  = pose_with_desired_geometry.res(i);
		int const aav = pose_with_desired_geometry.res_variant(i);
		int const natoms( aaproperties_pack::natoms( aa, aav ) );

		//Most of the distance constraints can be implemented as
		// atom tree constraints... very efficient.
		for ( int j = 1; j <= natoms; ++j ) {
			//			if (i==1 && j==1) continue; //first atom in fold tree has no bond.
			if (aaproperties_pack::fullatom_type(j,aa,aav) > 21 &&
					get_vary_bond_geometry_no_hydrogens()) continue;

			Atom_id current_atom_id( j, i );

			bool const this_is_a_jump = atom_tree->atom( current_atom_id )->is_jump();

			if (!this_is_a_jump) {

				Torsion_id this_bond_length( current_atom_id, D );

				if (!check_torsion_deviation_OK( this_bond_length, bond_length_tether,
																				 pose, pose_with_desired_geometry)) continue;

				float const current_val( pose_with_desired_geometry.get_atom_tree_torsion(this_bond_length));
				cst_set_p -> add_kin_torsion_constraint( this_bond_length, stretch_factor * current_val,
																								 bond_length_tether );
				pose.set_allow_move( this_bond_length, true );

			}
		}

		// Also add in ring closure! These are the bonds that aren't stored as
		// "torsions" in the atom tree. Need atompair and angle constraints...
		// The most general way to do this is to go through the source pose
		// and find atoms that look bonded, but are not connected within the
		// atom tree.
		find_and_add_ring_constraints( atom_tree, cst_set_p,
																	 pose,
																	 i, aa, aav );
	}

	if (pose.constraints_exist()){
		cst_set_p -> collect( pose.constraints() );
	}
	pose.set_constraints( *cst_set_p );

	// this was allocated by "new" (above), and will cause a memory leak otherwise
	cst_set_p->remove_ptr_reference();

	pose.set_vary_bond_geometry_flag( true );
}


///////////////////////////////////////////////////////////////////////////////
/// Is this a phi, psi, omega, or chi angle?
/// Following is inefficient (we could set up a boolean mask),
/// but this won't get called too many times.
bool is_this_phi_psi_omega_chi(
	 pose_ns::Pose & pose,
	 int pos,
	 int atomnum
)
{
	using namespace kin;
	Atom_tree const * atom_tree = pose.atom_tree();

	using aaproperties_pack::nchi;
	int const aa  = pose.res( pos );
	int const aav = pose.res_variant( pos );
	int const max_bb_tor( param_aa::is_protein( aa ) ? 3 :
												( param_aa::is_DNA( aa ) || param_aa::is_RNA( aa ) ? 6 : 0 ) );

	Atom_id current_atom_id( atomnum, pos );
	const kin::Atom* current_atom( atom_tree->atom( current_atom_id ) );

	// see e.g., get_allow_bb_move()
	//Check residue before and after, too...
	for (int offset=-1; offset <= 1; ++offset ) {

		if ( pos+offset < 1) continue;
		if ( pos+offset > pose.total_residue()) continue;

		for ( int tor=1; tor<= max_bb_tor; ++tor ) {
			const kin::Atom* atom( atom_tree->get_rosetta_torsion_atom( tor, pos + offset ) );
			if ( atom && atom == current_atom) return true;
		}

	}

	int const max_chi( param_aa::is_protein( aa ) ? nchi(aa,aav):
										 ( param_aa::is_DNA( aa ) ? 1 :
											 ( param_aa::is_RNA( aa ) ? 4 : 0 )));
	for ( int chino=1; chino<= max_chi; ++chino ) {
		const kin::Atom* atom( atom_tree->get_rosetta_torsion_atom(max_bb_tor+chino,
																												 pos ));
		if ( atom && atom == current_atom )	return true;
	}

	return false;
}


/////////////////////////////////////////////////////////////////////////////////////
void
set_all_bond_angle_allow_move(
															pose_ns::Pose & pose
															){
	pose_ns::Pose ideal_pose;
	create_totally_new_ideal_pose( ideal_pose, pose );
	set_all_bond_angle_allow_move( pose, ideal_pose );
}

/////////////////////////////////////////////////////////////////////////////////////
void
set_all_bond_angle_allow_move(
															pose_ns::Pose & pose,
															pose_ns::Pose & pose_with_desired_geometry
)
{
	using namespace cst_set_ns;
	using namespace kin;

  // Dynamically allocate a cst_set, which will be pointed to by a pose.
	// The cst_set will delete itself when the pose no longer points to it, but this
	// requires that we call "remove_ptr_reference()" once the pose pointer is setup (below)
	Cst_set * cst_set_p;
	cst_set_p = new Cst_set();

	static float const stretch_factor( realafteroption("stretch_factor", 1.0));
	// Empirically, a spring constant of 10.0 gives a variation of
	// +/- 2 degrees. That variation is on the same scale as
	// the empirically observed variation ( 1-3 degrees). See
	// Engh and Huber, Acta Cryst (1991), A47, p. 392-400.
	// The constant 10.0 seems awfully low, but this might be
	// due to some scaling in the minimizer.
	static float const bond_angle_tether( realafteroption("bond_angle_tether", 10.0) );

	Atom_tree const * atom_tree = pose_with_desired_geometry.atom_tree(); //Need the atom tree to see where jumps are.

	int const nres( pose_with_desired_geometry.total_residue() );
	for (int i = 1; i <= nres; ++i) {
		int const aa  = pose_with_desired_geometry.res(i);
		int const aav = pose_with_desired_geometry.res_variant(i);
		int const natoms( aaproperties_pack::natoms( aa, aav ) );

		//Most of the distance constraints can be implemented as atom
		// tree constraints... very efficient.
		for ( int j = 1; j <= natoms; ++j ) {
			if (aaproperties_pack::fullatom_type(j,aa,aav) > 21 &&
					get_vary_bond_geometry_no_hydrogens()) continue;
			Atom_id current_atom_id( j, i );

			bool const this_is_a_jump = atom_tree->atom( current_atom_id )->is_jump();

			if (!this_is_a_jump) {

				Torsion_id this_bond_angle( current_atom_id, THETA );

				if (!check_torsion_deviation_OK( this_bond_angle, bond_angle_tether,
																				 pose, pose_with_desired_geometry)) continue;

				float const current_angle( pose_with_desired_geometry.get_atom_tree_torsion(this_bond_angle));
				cst_set_p -> add_kin_torsion_constraint( this_bond_angle, stretch_factor * current_angle,
																								 bond_angle_tether );
				pose.set_allow_move( this_bond_angle, true );

				Torsion_id this_bond_torsion( current_atom_id, PHI );

				if (!check_torsion_deviation_OK( this_bond_torsion, bond_angle_tether,
																				 pose, pose_with_desired_geometry)) continue;

				float const current_torsion( pose_with_desired_geometry.get_atom_tree_torsion(this_bond_torsion));
				if ( is_this_phi_psi_omega_chi(pose_with_desired_geometry, i, j)) continue;

				float const sin_theta = std::sin( current_angle );
				cst_set_p -> add_kin_torsion_constraint( this_bond_torsion,
																								 stretch_factor * current_torsion,
																								 bond_angle_tether * sin_theta * sin_theta );
				pose.set_allow_move( this_bond_torsion, true );

			}

		// Also add in ring closure! These are the bonds that aren't stored as
		// "torsions" in the atom tree. Need atompair and angle constraints...
		// The most general way to do this is to go through the source pose
		// and find atoms that look bonded, but are not connected within the
		// atom tree.
			find_and_add_ring_constraints( atom_tree, cst_set_p,
																				 pose,
																				 i, aa, aav );
		}
	}

	if (pose.constraints_exist()){
		cst_set_p -> collect( pose.constraints() );
	}
	pose.set_constraints( *cst_set_p );

	// this was allocated by "new" (above), and will cause a memory leak otherwise
	cst_set_p->remove_ptr_reference();

	pose.set_vary_bond_geometry_flag( true );
}

/////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////
bool is_backbone( int const aa, int const atomnum){
	using namespace param_aa;

	//There's something similar to this in kin_util, but it doesn't have *all*
	// the backbone atoms, just the ones used to define phi, psi, and omega.
	//Currently the following is only set up for proteins AND ONLY FOR NON-HYDROGENS!!!
	// To do: add an "is_backbone" flag in read_aa_properties?
	if (is_protein(aa)){
		if ( atomnum <= 5 ) return true;
	}

	return false;
}

bool is_sidechain( int const aa, int const atomnum ){
	return (!is_backbone( aa, atomnum ));
}

/////////////////////////////////////////////////////////////////////////////////////
bool
has_kin_torsion_constraint( pose_ns::Pose & pose,
														kin::Torsion_id & some_torsion ){
	if (! pose.constraints_exist()) return false;

	cst_set_ns::Cst_set const cst_set( pose.constraints() );
	return cst_set.has_kin_torsion_constraint( some_torsion);

}

/////////////////////////////////////////////////////////////////////////////////////
void
atom_tree_set_allow_move(
	 pose_ns::Pose & pose,
	 bool const allow_move_backbone,
	 bool const allow_move_sidechain,
	 bool const allow_move_jump
)
{
	using namespace cst_set_ns;
	using namespace kin;

	Atom_tree const * atom_tree = pose.atom_tree(); //Need the atom tree to see where jumps are.

	int const nres( pose.total_residue() );
	for (int i = 1; i <= nres; ++i) {
		int const aa  = pose.res(i);
		int const aav = pose.res_variant(i);
		int const natoms( aaproperties_pack::natoms( aa, aav ) );

		for ( int j = 1; j <= natoms; ++j ) {
			Atom_id current_atom_id( j, i );

			bool allow_move = allow_move_backbone;
			if (is_sidechain(aa,j)) allow_move = allow_move_sidechain;

			bool const this_is_a_jump = atom_tree->atom( current_atom_id )->is_jump();
			if (!this_is_a_jump) {
				Torsion_id this_bond_angle( current_atom_id, THETA );
				pose.set_allow_move( this_bond_angle, false );
				if (has_kin_torsion_constraint( pose, this_bond_angle )){
					pose.set_allow_move( this_bond_angle, allow_move );
				}

				Torsion_id this_bond_length( current_atom_id, D );
				pose.set_allow_move( this_bond_length, false );
				if( has_kin_torsion_constraint( pose, this_bond_length )){
					pose.set_allow_move( this_bond_length, allow_move );
				}

				Torsion_id this_bond_torsion( current_atom_id, PHI );
				pose.set_allow_move( this_bond_torsion, false );
				if (this_is_a_jump || is_this_phi_psi_omega_chi(pose, i, j) ||
						has_kin_torsion_constraint( pose, this_bond_angle )){
					pose.set_allow_move( this_bond_torsion, allow_move );
				}
			} //is jump

		} //j
	}//i

	pose.set_allow_jump_move( allow_move_jump );

}

/////////////////////////////////////////////////////////////////////////////////////
void
add_heavyatom_coordinate_constraints( pose_ns::Pose & pose ) {
	using namespace cst_set_ns;
	using namespace kin;
	using namespace aaproperties_pack;

  // Dynamically allocate a cst_set, which will be pointed to by a pose.
	// The cst_set will delete itself when the pose no longer points to it, but this
	// requires that we call "remove_ptr_reference()" once the pose pointer is setup (below)
	Cst_set * cst_set_p;
	cst_set_p = new Cst_set();

	Coords_FArray_const coords( pose.full_coord());

	int const nres( pose.total_residue() );
	for (int i = 1; i <= nres; ++i) {
		int const aa  = pose.res(i);
		int const aav = pose.res_variant(i);
		int const natoms( aaproperties_pack::nheavyatoms( aa, aav ) );

		//Most of the distance constraints can be implemented as
		// atom tree constraints... very efficient.
		for ( int j = 1; j <= natoms; ++j ) {

			Atom_id current_atom_id( j, i );

			numeric::xyzVector_float const xyz1( coords.get_xyz( current_atom_id) );

			cst_set_p -> add_coordinate_constraint(current_atom_id, xyz1 );

		}
	}


	if (pose.constraints_exist()){
		cst_set_p -> collect( pose.constraints() );
	}

	pose.set_constraints( *cst_set_p );

	// this was allocated by "new" (above), and will cause a memory leak otherwise
	cst_set_p->remove_ptr_reference();

}

/////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////
// Constraint setup helper routines.
/////////////////////////////////////////////////////////////////////////////////////
void add_angle_cst_with_theta_from_3_atoms(
			kin::Atom_id const & atom1,
			kin::Atom_id const & atom2,
			kin::Atom_id const & atom3,
			float const weight,
			kin::Coords_FArray_const const & coords,
			cst_set_ns::Cst_set & cst_set)
{
	using namespace numeric;
	using numeric::sin_cos_range;

	xyzVector_float v12( coords.get_xyz( atom1 ) - coords.get_xyz( atom2 ) );
	xyzVector_float v32( coords.get_xyz( atom3 ) - coords.get_xyz( atom2 ) );
	float n1_n2 =  v12.length()*v32.length();
	if ( n1_n2 < 1e-9 ) {
		std::cout << "ERROR pose_constraints want v1,v2 angle but v1 or v2 has length 0" << std::endl;
		//		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else {
		float const x = dot( v12, v32) / ( n1_n2 );
		float const theta = std::acos( sin_cos_range( x )); //arccos( x )
		//std::cout << "theta " << theta << std::endl;
		static float const angle_stretch_factor( realafteroption("angle_stretch_factor", 1.0));
		cst_set.add_atom_angle_constraint( atom1,atom2,atom3,theta * angle_stretch_factor,weight );
	}
}


///////////////////////////////////////////////////////////////////////////////
void
add_distance_constraint_from_2_atoms(
			kin::Atom_id const & atom1,
			kin::Atom_id const & atom2,
			float const weight,
			kin::Coords_FArray_const const & coords,
			cst_set_ns::Cst_set & cst_set) {

	Vec const xyz1( coords.get_xyz( atom1) );
	Vec const xyz2( coords.get_xyz( atom2) );
	float const current_val = (xyz1 - xyz2).length();
	cst_set.add_atompair_constraint( atom1, atom2,
																	 current_val, weight,
																	 false /*verbose*/);

}

///////////////////////////////////////////////////////////////////////////////
void
add_dihedral_cst_with_dihedral_from_4_atoms(
			kin::Atom_id const & atom1,
			kin::Atom_id const & atom2,
			kin::Atom_id const & atom3,
			kin::Atom_id const & atom4,
			float const weight,
			kin::Coords_FArray_const const & coords,
			cst_set_ns::Cst_set & cst_set) {

	using namespace numeric;

	Vec p1( coords.get_xyz( atom1 ) );
	Vec p2( coords.get_xyz( atom2 ) );
	Vec p3( coords.get_xyz( atom3 ) );
	Vec p4( coords.get_xyz( atom4 ) );

	Vec v1( p1-p2 );
	Vec v2( p2-p3 );
	Vec v3( p3-p4 );

	Vec v12( cross( v1, v2 ));
	Vec v23( cross( v2, v3 ));

	float const n12_n23( v12.length() * v23.length() );
	float const deg2rad( numeric::conversions::radians(1.0) );

	if ( n12_n23 < 1e-9 ) {
		std::cout << "ERROR pose_constraints want dihedral but v12 or v23 has length 0" << std::endl;
		//		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else {
		// psh comment out the next two lines and use the dihedral function instead
		//	float const x = dot( v12, v23) / ( n12_n23 );
		//	float const theta = std::acos( sin_cos_range( x )); //arccos( x )
		float const theta (deg2rad * dihedral( coords.get_xyz( atom1 ),coords.get_xyz( atom2 ),
																					 coords.get_xyz( atom3 ),coords.get_xyz( atom4 ) ) );
		cst_set.add_atom_torsion_constraint(atom1,atom2,atom3,atom4,theta,weight);
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// psh setup disulfide constraints
///////////////////////////////////////////////////////////////////////////////

void add_disulf_pose_constraints( pose_ns::Pose & pose){

  // Make a new constraints object:
  using namespace cst_set_ns;
  using namespace kin;
	using numeric::xyzVector_float;
	using numeric::sin_cos_range;
	typedef numeric::xyzVector_float Vec;

  // Dynamically allocate a cst_set, which will be pointed to by a pose.
	// The cst_set will delete itself when the pose no longer points to it, but this
	// requires that we call "remove_ptr_reference()" once the pose pointer is setup (below)
	Cst_set * cst_set_p;
	cst_set_p = new Cst_set();

	// need to score the starting pose first!! this is to make sure the disulf
	// are identified
  pose.score(score12);
  std::cout << "Setting up disulfide constraints: number of disulfides identified is "
      << disulfides::BOUNDARY::get_n_disulf_centroid() << std::endl;
  //In following loop need to add distance constraints
  // cst_set.add_atompair_constraint( atom1, atom2, cst );
  int const n_disulf = disulfides::BOUNDARY::get_n_disulf_centroid();
  FArray1D_int natro_positions( 2*(n_disulf),0);
  int n_natro = 0;
  // collect all the disulf residue positions into an array
  // arrange it so it's parta1, partb1, parta2, partb2,...
  for ( int i_disulf = 1; i_disulf <= n_disulf; i_disulf++ ) {
    //cys1 from partner_a
    int const cys1_id = disulfides::BOUNDARY::get_disulf_partner_a( i_disulf );
    int const cys1_seqpos = disulfides::BOUNDARY::get_cys( cys1_id );
    natro_positions( ++n_natro ) = cys1_seqpos;
    //cys2 from partner_b
    int const cys2_id = disulfides::BOUNDARY::get_disulf_partner_b( i_disulf );
    int const cys2_seqpos = disulfides::BOUNDARY::get_cys( cys2_id );
    natro_positions( ++n_natro ) = cys2_seqpos;
    std::cout << "adding disulf to natro: " << i_disulf << " "
              << pdb::pdb_res_num( natro_positions( n_natro-1 ) ) << " "
              << pdb::pdb_res_num( natro_positions( n_natro ) )
              << " at n_natro i,i+1: " << n_natro-1 << " "
              << n_natro << std::endl;
  }
	FArray3D_float const & full_coord( pose.full_coord());
	kin::Coords_FArray_const coords( pose.full_coord());


  // Add disulfide to cst_set_p.
  // only loop through the disulfide positions from natro_positions array
  // so residue i vs residue j is written as partA vs partB
	float const cst_wt_disulf_distance =
		realafteroption("cst_wt_disulf_distance",10000.0);
	float const cst_wt_disulf_dihedral =
		realafteroption("cst_wt_disulf_dihedral",100.0);
	float const cst_wt_disulf_angle    =
		realafteroption("cst_wt_disulf_angle",10000.0);
  for ( int i=1; i<= 2*n_disulf; i=i+2 ) {
		//distance  only tie the SG together for now
		int partA = natro_positions(i);
		int partB = natro_positions(i+1);
		kin::Atom_id atom1_sg(6,partA); // SG, partner a
		kin::Atom_id atom2_sg(6,partB); // SG, partner b
		float r ( std::sqrt( vec_dist2( full_coord( 1, 6, partA ),
										full_coord( 1, 6, partB ) ) ) );
		if (get_tether_to_ideal_geometry_flag()){
			r = 2.02;
		}
		//totally boost the SG distance weight
		Cst cst(r, cst_wt_disulf_distance);
		cst_set_p->add_atompair_constraint(atom1_sg, atom2_sg, cst );

		//angles
		//First the disulf dihedral CB-SG-SG-CB

		kin::Atom_id atom1(5,partA);//CB i
		kin::Atom_id atom2(6,partA);//SG i
		kin::Atom_id atom3(6,partB);//SG j
		kin::Atom_id atom4(5,partB);//CB j

		Vec p1( coords.get_xyz( atom1 ) );
		Vec p2( coords.get_xyz( atom2 ) );
		Vec p3( coords.get_xyz( atom3 ) );
		Vec p4( coords.get_xyz( atom4 ) );

		Vec v1( p1-p2 );
		Vec v2( p2-p3 );
		Vec v3( p3-p4 );

		Vec v12( cross( v1, v2 ));
		Vec v23( cross( v2, v3 ));


		float const n12_n23( v12.length() * v23.length() );
		float const deg2rad( numeric::conversions::radians(1.0) );


		if ( n12_n23 < 1e-9 ) {
			std::cout << "ERROR pose_constraints want dihedral but v12 or v23 has length 0" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		} else {
		// psh comment out the next two lines and use the dihedral function instead
		//	float const x = dot( v12, v23) / ( n12_n23 );
		//	float const theta = std::acos( sin_cos_range( x )); //arccos( x )
			float theta (deg2rad * dihedral( coords.get_xyz( atom1 ),coords.get_xyz( atom2 ),
		                            coords.get_xyz( atom3 ),coords.get_xyz( atom4 ) ) );
			if (get_tether_to_ideal_geometry_flag()){
				if (theta > 0){
					theta = deg2rad * 90;
				}
				else if (theta < 0){
					theta = deg2rad * -90;
				}
				else {
					std::cout << "ERROR disulfide dihedral equals 0" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
			cst_set_p->add_atom_torsion_constraint(atom1,atom2,atom3,atom4,theta,cst_wt_disulf_dihedral);
		}


//Now add constraints for bond angles CBi-Si-Sj and CBj-Sj-Si
//psh due to the behavior of Atom_id class, the two directions are set
//explicitly instead of going through loops

		kin::Atom_id atom_Cb_i(5,partA);//Cb i
		kin::Atom_id atom_SG_i(6,partA);//SG i
		kin::Atom_id atom_Cb_j(5,partB);//Cb j
		kin::Atom_id atom_SG_j(6,partB);//SG j
		if (get_tether_to_ideal_geometry_flag()) {
			float theta = deg2rad * 103.4;
			//Cb-S-S for i->j
			cst_set_p->add_atom_angle_constraint(atom_Cb_i, atom_SG_i, atom_SG_j, theta, cst_wt_disulf_angle);
			//Cb-S-S for j->i
			cst_set_p->add_atom_angle_constraint(atom_Cb_j, atom_SG_j, atom_SG_i, theta, cst_wt_disulf_angle);
		}
		else {
			//Cb-S-S for i->j
			add_angle_cst_with_theta_from_3_atoms(atom_Cb_i, atom_SG_i, atom_SG_j, cst_wt_disulf_angle, coords, *cst_set_p);
			//Cb-S-S for j->i
			add_angle_cst_with_theta_from_3_atoms(atom_Cb_j, atom_SG_j, atom_SG_i, cst_wt_disulf_angle, coords, *cst_set_p);
		}
	} //check for disulf

	// see if constraints existed in pose
	if (pose.constraints_exist()){
		cst_set_p -> collect( pose.constraints() );
  }

	//psh cst debug flag
	//cst_set_ns::debug_output = true;
	//std::cout << "pose constraints" << pose.constraints() << std::endl;

	// finaly add the constraints to pose
  pose.set_constraints( *cst_set_p );

	// this was allocated by "new" (above), and will cause a memory leak otherwise
	cst_set_p->remove_ptr_reference();

}

