// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $

// Utility Headers
// Must be included first to avoid name resolution conflicts when using VC++
#include <utility/io/izstream.hh>

// Rosetta Headers
#include "barcode_classes.h"
#include "force_barcode.h"
#include "pose_rna_jumping.h" // Maybe this is the wrong style...
#include "random_numbers.h"
//#include "param.h"
#include "misc.h"
//#include "tether_ns.h"
#include "after_opts.h"

#include "fragments_ns.h"
#include "fragments.h"
#include "flexibility.h"
#include "param.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/orstream.hh>

// C++ Headers
#include <cstdlib>
#include <cstdio>
#include <fstream>
#include <iostream>
// #include <iomanip>
#include <utility>
#include <sstream>
#include <string>
#include <iomanip>


////////////////////////////////////////////////////////////////////////////////
///
/// Barcode is a way to parse the conformational space into discrete bins.
/// Currently we only use torsion angle barcodes to describe protein
/// conformation space. For example, if we bin psi angles into 10 degree
/// bins, then for each residue psi angle we have 36 possible codes. The list
/// of such codes for all or a subset of residues in a protein is called the
/// barcode of that protein.
///
/// barcode classes contain an array of torsion angle barcodes, including
/// backbone torsion angle codes, sidechain chi angle codes, rotamer number
/// codes, etc.
///
/// The main usage of barcodes in rosetta are:
/// 1, Defining conformational space used in taboo search, where the barcodes
///    of previously visited conformations are remembered and used to modify
///    the behaviour of minimization process.
/// 2, Used as constraints to restrict the sampling of torsion angles to
///    specified rigions of values.
///
/////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////////
/// we need all the sc constraints scored by apply_chi_tether to
/// be in a row, so that when we sort all the constraints,
/// the sc_begin and sc_end maps will work
/// in other words, anything wiht sc_constraint() = true should be in
/// a single continuos block of torsion values
/////////////////////////////////////////////////////////////////////////////////

namespace barcode_classes {
namespace barcode_param {
	int const PHI_TORSION = 1;
	int const PSI_TORSION = 2;
	int const OMEGA_TORSION = 3;
	int const CHI1_TORSION = 4; // marks the beginning of the chi,rot torsions
	int const ROT1_TORSION = 8; //  = CHI1_TORSION+4
	int const SC_BIN_TORSION = 12; //  = CHI1_TORSION+8
	int const BB_BIG_BIN_TORSION = 13;
	int const SS_TORSION = 14;
	int const BB_SMALL_BIN_TORSION = 15;
	int const BB_CLUSTER_TORSION = 16;
	int const SSPAIR_TORSION = 17;
	int const CONTACT_TORSION = 18;
	int const DISULF_TORSION = 19;

	int const CONTACT_ENERGY_TORSION = 30;
	int const BETA_PAIR_ENERGY_TORSION = 31;
	int const SS_ENERGY_TORSION = 32;
	int const BIG_BIN_ENERGY_TORSION = 33;
	int const DSSP_ENERGY_TORSION = 34;
	int const BBLUM_ENERGY_TORSION = 35;
	int const ENV_ENERGY_TORSION = 36;

	int const RNA_BP_JUMP_TORSION = 50;

}

/////////////////////////////////////////////////////////////////////////////////
///
/// a stream extractor for constraints:
/// failure is signaled by setting the streams failbit
/// this routine handles the converting of user-friendly input files
/// into internal magic torsion numbers.
///
/// it's not a great setup, since we need to keep this synced with some other
/// stuff
///
/// each cst starts with a tag-string which determines the rest of the parsing
/// current tags are:

/// PHI <rsd> <weight> <fvalue>
/// PSI <rsd> <weight> <fvalue>
/// OMEGA <rsd> <weight> <fvalue>
/// CHI1 <rsd> <weight> <fvalue>
/// ROT1 <rsd> <weight> <ivalue>
/// BB_BIG_BIN <rsd> <weight> <cvalue>
/// SS <rsd> <weight> <cvalue>
/// BB_SMALL_BIN <RSD> <weight> <fvalue> <fvalue> <fvalue> <fvalue> ***1
/// SC_BIN <RSD> <weight> <fvalue> <fvalue> <fvalue> <fvalue> ***2
/// BB_CLUSTER <RSD> <weight> <fvalue> <fvalue> <fvalue> <fvalue> ***3
/// SSPAIR <weight> <orientation> <strand1_begin> <s1_end> <s2_begin> <s2_end>

/// CHI1 could be replaced with CHI2,CHI3,CHI4, same with ROT1
/// fvalue means floating point, ivalue means int, cvalue means char

/// New (as of May 2007):
///  The way the barcodes are enforced for most of the features above is
///   a little complex. Here are some ways to just add the user-inputted
///   energies to the score when each feature is seen.
///
///  There is some overlap with other pathways (e.g., for environment, for contacts)
///    to calculate energies in Rosetta.
///
/// Centroid distance based contact. Does this overlap with CONTACT_TORSION?
/// CONTACT_ENERGY   <energy> <i> <j> <distance cutoff>
//
/// Precise beta pairing, as inferred from a DSSP calculation:
/// BETA_PAIR_ENERGY <energy> <i> <j> <orient.: 1,2> <pleat: 1,2>
///
/// Secondary structure inherited from fragments:
/// SS_ENERGY        <energy> <i> <E, H, L>
///
/// Region of phi,psi,omega in the Ramachandran plot:
/// BIG_BIN_ENERGY   <energy> <i> <A, B, G, E, or O>
///
/// Secondary structure assigned by DSSP (looks at hydrogen bonding pattern!)
/// DSSP_ENERGY      <energy> <i> <B, E, I, T, G, H, or  S>
///
/// Ben Blum's classification, separating edge strands (e) and beta bulges (b).
/// BBLUM_ENERGY     <energy> <i> <H, L, e, E, b, B>
///
/// Number of neighbors (centroid distance within 10 A):
/// ENV_ENERGY     <energy> <i> <num neighbors>
///
/// DANGER DANGER DANGER DANGER:
///
/// ***1 the order of the values should be: min_phi,max_phi,min_psi,max_psi
/// ***2 the order of the values should be: min_chi1,max_chi1,min_chi2,max_chi2
/// ***3 the order of the values should be: phi,psi,omega,thresh
/// BUT: if the bin spans the 180 divider, then min_phi should be greater than
///      max_phi,etc; ie the check becomes: (phi >= min_phi || phi <= max_phi)
///      rather than (phi >= min_phi && phi <= max_phi)

/// the format for the constraints file is: (ignore the leading "// ")

/// <feature1-tag> <flavor1-freq> <constraint1> {<constraint2> <constraint3> ... }
/// <feature1-tag> <flavor2-freq> <constraint1> {<constraint2> <constraint3> ... }
/// ...
/// <feature2-tag> <flavor1-freq> <constraint1> {<constraint2> <constraint3> ... }
/// <feature2-tag> <flavor2-freq> <constraint1> {<constraint2> <constraint3> ... }
/// ...

/// here each feature-tag is just a unique, non-white-space string that uniquely
/// identifies a common set of flavors

/// <constraint1> means insert the relevant info for constraint1, eg
/// <constraint1> could be "PHI 12 1.0 46.7" or "SS 24 1.0 E"

/////////////////////////////////////////////////////////////////////////////////

std::istream & operator >>(std::istream & is, constraint & cst) {
	using namespace barcode_param;
	std::string tag;
	is >> tag;
	if ( is.fail() ) {
		return is;
	} else if ( tag.size() == 0) {
		is.setstate( std::ios_base::failbit ); // signal a problem
	} else if ( tag == "PHI" ) { // PHI
		cst.torsion = PHI_TORSION;
		is >> cst.residue >> cst.weight >> cst.float_value;
	} else if ( tag == "PSI" ) { // PSI
		cst.torsion = PSI_TORSION;
		is >> cst.residue >> cst.weight >> cst.float_value;
	} else if ( tag == "OMEGA" ) { // OMEGA
		cst.torsion = PSI_TORSION;
		is >> cst.residue >> cst.weight >> cst.float_value;
	} else if ( tag.substr(0,3) == "CHI" ) { // CHI
		std::istringstream t( tag.substr(3,1) );
		int chi_number;
		t >> chi_number;
		if ( t.fail() || chi_number < 1 || chi_number > 4 ) {
			is.setstate( std::ios_base::failbit );
		}
		cst.torsion = CHI1_TORSION - 1 + chi_number;
		is >> cst.residue >> cst.weight >> cst.float_value;
	} else if ( tag == "SC_BIN") { // SC_BIN
		cst.torsion = SC_BIN_TORSION;
		is >> cst.residue >> cst.weight >> cst.float_value >> cst.float_value1 >>
			cst.float_value2 >> cst.float_value3;
	} else if ( tag.substr(0,3) == "ROT" ) { // ROT
		std::istringstream t( tag.substr(3,1) );
		int nchi;
		t >> nchi;
		if ( t.fail() || nchi < 1 || nchi > 4 ) {
			is.setstate( std::ios_base::failbit );
		}
		cst.torsion = ROT1_TORSION - 1 + nchi;
		is >> cst.residue >> cst.weight;
		FArray1D_float rot(4);
		for ( int chino=1; chino<= nchi; ++chino ) {
			is >> rot(chino);
			if ( std::abs( rot(chino) - std::floor( rot(chino) + 0.5 ) > 0.1 )) {
				std::cout << "WARNING:: bad format for rot constraint: " <<
					" cst values should be integers!!!!" << std::endl;
				is.setstate( std::ios::failbit );
			}
		}
		cst.float_value  = rot(1);
		cst.float_value1 = rot(2); // not nec used
		cst.float_value2 = rot(3); // ditto
		cst.float_value3 = rot(4);
	} else if ( tag == "BB_BIG_BIN" ) { // BB_BIG_BIN
		cst.torsion = BB_BIG_BIN_TORSION;
		is >> cst.residue >> cst.weight >> cst.char_value;
		if ( cst.char_value != 'A' && cst.char_value != 'B' && cst.char_value != 'G' &&
				 cst.char_value != 'E' && cst.char_value != 'O') {
			is.setstate( std::ios_base::failbit );
		}
	} else if ( tag == "SS" ) { // SS
		cst.torsion = SS_TORSION;
		is >> cst.residue >> cst.weight >> cst.char_value;
		if ( cst.char_value != 'H' && cst.char_value != 'E' && cst.char_value != 'L' ) {
			is.setstate( std::ios_base::failbit );
		}
  } else if ( tag == "BB_SMALL_BIN") { // BB_SMALL_BIN
		cst.torsion=BB_SMALL_BIN_TORSION;
		is >> cst.residue >> cst.weight >> cst.float_value >> cst.float_value1 >>
			cst.float_value2 >> cst.float_value3;
  } else if ( tag == "BB_CLUSTER") { // BB_SMALL_BIN
		cst.torsion=BB_CLUSTER_TORSION;
		is >> cst.residue >> cst.weight >> cst.float_value >> cst.float_value1 >>
			cst.float_value2 >> cst.float_value3; // phi,psi,omega,thresh
	} else if ( tag == "SSPAIR" ) {
		cst.torsion = SSPAIR_TORSION;
		char orientation;
		int strand1_begin, strand1_end, strand2_begin, strand2_end;
		is >> cst.weight >> orientation >> strand1_begin >> strand1_end >>
			strand2_begin >> strand2_end;
		cst.residue = 0; // nonsense
		if ( orientation == 'A' ) {
			cst.int_value = constraint::ANTIPARALLEL;
		} else if ( orientation == 'P' ) {
			cst.int_value = constraint::PARALLEL;
		} else {
			std::cout << "SSPAIR orientation should be 'A' or 'P'" << std::endl;
			is.setstate( std::ios_base::failbit );
		}
		cst.int_value1 = strand1_begin;
		cst.int_value2 = strand1_end;
		cst.int_value3 = strand2_begin;
		cst.int_value4 = strand2_end;
	} else if ( tag == "CONTACT" ) {
		// CONTACT weight threshold rsd1 rsd2 rsd3 rsd4
		cst.torsion = CONTACT_TORSION;
		float threshold;
		int range1_begin, range1_end, range2_begin, range2_end;
		is >> cst.weight >> threshold >> range1_begin >> range1_end >>
			range2_begin >> range2_end;
		cst.residue = 0; // nonsense
		cst.float_value  = threshold;
		cst.int_value    = range1_begin;
		cst.float_value1 = range1_end;
		cst.float_value2 = range2_begin;
		cst.float_value3 = range2_end;
	} else if ( tag == "DISULF" ) {
		cst.torsion = DISULF_TORSION;
		int res1, res2;
		is >> res1 >> res2;
		cst.residue = res1;
		cst.int_value = res2;
	} else if ( tag == "CONTACT_ENERGY" ) {
		cst.torsion = CONTACT_ENERGY_TORSION;
		int res1, res2;
		is >> cst.weight >> res1 >> res2 >> cst.float_value1;
		cst.residue = res1;
		cst.int_value = res2;
	} else if ( tag == "BETA_PAIR_ENERGY" ) {
		cst.torsion = BETA_PAIR_ENERGY_TORSION;
		int res1, res2, o, p;
		is >> cst.weight >> res1 >> res2 >> o >> p;
		cst.residue = res1; // just so it isn't junk and doesn't cause an error
		cst.int_value = res2; // just so it isn't junk

		cst.int_value1 = res1;
		cst.int_value2 = res2;
		cst.int_value3 = o;
		cst.int_value4 = p;
	} else if ( tag == "SS_ENERGY" ) {
		cst.torsion = SS_ENERGY_TORSION;
		is >> cst.weight >> cst.residue >> cst.char_value;
	} else if ( tag == "BIG_BIN_ENERGY" ) {
		cst.torsion = BIG_BIN_ENERGY_TORSION;
		is >> cst.weight >> cst.residue >> cst.char_value;
	} else if ( tag == "DSSP_ENERGY" ) {
		cst.torsion = DSSP_ENERGY_TORSION;
		is >> cst.weight >> cst.residue >> cst.char_value;
	} else if ( tag == "BBLUM_ENERGY" ) {
		cst.torsion = BBLUM_ENERGY_TORSION;
		is >> cst.weight >> cst.residue >> cst.char_value;
	} else if ( tag == "ENV_ENERGY" ) {
		cst.torsion = ENV_ENERGY_TORSION;
		is >> cst.weight >> cst.residue >> cst.int_value;
	} else if ( tag == "RNA_BP_JUMP" ) {
		cst.torsion = RNA_BP_JUMP_TORSION;
		is >> cst.int_value1 /*res1*/ >> cst.int_value2 /*res2*/ >> cst.residue /*cutpoint*/;
		is >> cst.char_value1 /*edge1*/ >> cst.char_value2 /*edge2*/ >> cst.char_value /*orientation*/;
	} else {
		std::cout << "unrecognized torsion tag: " << tag << std::endl;
		is.setstate( std::ios_base::failbit );
	}
	if ( is.fail() ) {
		std::cout << "error parsing barcode constraint file!!" << std::endl;
		std::cout << "bad constraint: " << cst << std::endl;
	}
	return is;
}

// for displaying a constraint:
std::ostream & operator <<(std::ostream & os, constraint const & c )
{
	os << "barcode_cst: torsion= " << c.torsion
	 << " residue= " << c.residue
	 << " cval= " << c.char_value
	 << " ival= " << c.int_value
	 << " ival1= " << c.int_value1
	 << " ival2= " << c.int_value2
	 << " ival3= " << c.int_value3
	 << " fval= " << c.float_value
	 << " fval1= " << c.float_value1
	 << " fval2= " << c.float_value2
	 << " fval3= " << c.float_value3;
	return os;
}

std::string constraint::to_string() const {
	using namespace barcode_param;

	std::ostringstream out;
	int s1, s1end, s2, s2end;
	char antipar;

	switch(torsion) {
		case BB_BIG_BIN_TORSION:
			out << "Tor_" << residue << '.' << char_value;
			break;

		case SSPAIR_TORSION:
			s1 = int_value1;
			s1end = int_value2;
			s2 = int_value3;
			s2end = int_value4;
			antipar = (int_value == 1 ? 'A' : 'P');
			if(antipar == 'A') {
				s2++;
				s2end++;
			}
			out << "SSPair_" << s1 << '_' << s1end << '_' << s2 << '_' << s2end << '_' << antipar;
			break;

		case BBLUM_ENERGY_TORSION:
			out << "SS_" << residue << '.' << char_value;
			break;

		case DSSP_ENERGY_TORSION:
		  out << "DSSP_" << residue << '.' << char_value;
			break;

		case SS_TORSION:
		  out << "FragSS_" << residue << '.' << char_value;
			break;

		case BETA_PAIR_ENERGY_TORSION:
			out << "Bct_" << int_value1 << '_' << int_value2 << '_' << (int_value3 == 1 ? 'A' : 'P') << '_' << int_value4 << ".B";
			break;

		default:
			out << "BC" << torsion << '_' << residue << '_' << int_value << '_' << char_value;
			break;
	}

	return out.str();
}

// so that we can sort a vector of constraints
bool operator <( constraint const & a, constraint const & b) {
	return ( ( a.residue < b.residue ) ||
					 ( a.residue == b.residue && a.torsion < b.torsion ) );
}


std::ostream & operator<<(std::ostream & out, const flavor & flav) {
	if(flav.constraints.size() == 0) {
		out << 'X';
	} else {
		for(constraint_const_iterator it = flav.constraints.begin();
				it != flav.constraints.end();
				it++) {
			out << it->to_string();
			if(it+1 != flav.constraints.end())
				out << ':';
		}
	}
	return out;
}


std::ostream & operator<<(std::ostream & out, const feature& feat) {
	out << feat.flavors[feat.current_flavor_index];
	return out;
}

//////////// feature_set methods: ////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::update_active_constraints
///
/// @brief: add constraints to active_constraints
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors:
///
/// @last_modified:
/////////////////////////////////////////////////////////////////////////////////
void
feature_set::update_active_constraints()
{

	active_constraints.clear();
	for ( feature_iterator it = features.begin();
				it != features.end(); ++it ) {
		if ( mode == 4 ) {
			it->second.add_all_constraints( active_constraints );
		} else {
			it->second.add_active_constraints( active_constraints );
		}
	}

	// sort the list of active constraints
	std::sort ( active_constraints.begin(), active_constraints.end() );

	//  now setup the SC maps
	for ( int i=1; i<= nres; ++i ) {
		sc_begin_map[i] = active_constraints.end();
		sc_end_map[i] = active_constraints.end();
	}

	for ( constraint_iterator it = active_constraints.begin();
				it != active_constraints.end();
				++it ) {

		int residue = it->residue;
		bool in_sc = it->sc_constraint();

		if ( in_sc && sc_begin_map[ residue ] == active_constraints.end() ) {
			sc_begin_map[ residue ] = it;
			sc_end_map[ residue ] = it;
		}

		if ( in_sc ) ++sc_end_map[ residue ];
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::loop_update_active_constraints
///
/// @brief: add active constraints for loop modeling
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors:
///
/// @last_modified:
/////////////////////////////////////////////////////////////////////////////////
void
feature_set::loop_update_active_constraints(
	int const & loop_begin,
	int const & loop_end,
	std::vector< int > const & free_res,
	int const & loop_begin_extension,
	int const & loop_end_extension
)
{

	active_constraints.clear();

	for ( int pos  = loop_begin - loop_begin_extension;
						pos <= loop_end   + loop_end_extension;
						++pos ){
		if ( std::find( free_res.begin(), free_res.end(), pos )
									== free_res.end() &&
				pos >= 1 &&
				pos <= misc::total_residue &&
				(pos < loop_begin ||
				pos > loop_end) )
		{
			constraint newcst;
			newcst.torsion = barcode_param::BB_CLUSTER_TORSION;
			newcst.residue = pos;
			newcst.weight = 1.0;
			newcst.float_value = misc::phi( pos );
			newcst.float_value1 = misc::psi( pos );
			newcst.float_value2 = misc::omega( pos );
			newcst.float_value3 = 30;
			active_constraints.push_back(newcst);
			std::cout << "loop_stab_cst "<< newcst << std::endl;
		}
	}

	// now add loop features into the active_constraints
	for ( feature_iterator it = features.begin();
				it != features.end(); ++it ) {
		// go through all the flavors

		bool use = false; // indicate whether or not to use this feature

		std::vector<flavor>::iterator this_flavor = it->second.begin();
		++this_flavor; //first flavor is a dummy
		for ( this_flavor  = this_flavor;
					this_flavor != it->second.end();
					++this_flavor ) {
			std::vector<constraint>::iterator this_cst;
			for ( this_cst  = this_flavor->constraints.begin();
						this_cst != this_flavor->constraints.end();
						++this_cst ) {
//				if ( this_cst->bb_cluster_constraint() || this_cst->bb_big_bin_constraint() ) {
					if ( this_cst->residue <= loop_end && this_cst->residue >= loop_begin ){
						use = true;
//						this_cst->float_value3 = 30;
					}
//				}
			}
		}

		if ( use )
			it->second.add_all_constraints( active_constraints );
	}

	// sort the list of active constraints
	std::sort ( active_constraints.begin(), active_constraints.end() );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::clear
///
/// @brief: clears the feature set, not the same as reset
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors:
///
/// @last_modified:
/////////////////////////////////////////////////////////////////////////////////
void
feature_set::clear() {
	features.clear();
} // feature_set::clear

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::read_file
///
/// @brief: read constraints from a file
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors:
///
/// @last_modified:
/////////////////////////////////////////////////////////////////////////////////
void
feature_set::read_file(
	std::string const & filename,
	int const total_residue,
	int const mode_in,
	bool const verbose
)
{
	using namespace std;

	nres = total_residue; // store in private data member
	mode = mode_in;

	// <PM>
	//features.clear();
	// </PM>
	// read list of constraints from the file
	utility::io::izstream data( filename );

	if ( ! data().good() ) {
		std::cout << "Cant open db_constraint file " << data.filename() << std::endl;
		data.close();
		data.clear();
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	string line;

	// phil fix all this:

	// fill in features; at the end, call update_active_constraints,
	// reset all features ??
	features.clear();

	while ( std::getline( data(), line ) ) {
//		cout << "read line (" << line << ")\n";

		istringstream line_stream ( line );

		flavor f; // each line corresponds to a new flavor
		string tag; // tag tells us which feature to add this flavor to

		line_stream >> tag >> f.frequency;

		if ( line_stream.fail() ) {
			cout << "badline:" << line << endl;
			continue;
		}

		while (true) {
			constraint cst;
			line_stream >> cst;
			if ( line_stream.fail() ) break;
			if ( cst.residue > nres ) {
				std::cout << "bad cst residue: " << cst.residue << " nres: "
									<< nres << std::endl;
				continue;
			}
			f.constraints.push_back( cst );
		}

		if ( f.constraints.size() <= 0 ) {
			cout << "badline:" << line << endl;
			continue;
		}

		features[ tag]. new_flavor(f); // calls constructor for new tag?
	}
	data.close();
	data.clear();

	if (verbose) show_data(std::cout);

	reset(); // turns all features to off-state
}

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::get_bb_code
///
/// @brief: get the backbone code
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors:
///
/// @last_modified:
/////////////////////////////////////////////////////////////////////////////////

class BbCode {
public:
	int res_num;
	int H,E,L,A,G,O,B,EE;
	int total;

	int sum() const {
		return H + E + L + A + G + O + B + EE;
	} // sum

	BbCode() {
		res_num = 0;
		H = E = L = A = G = O = B = EE = 0;
		total = 0;
	} // BbCode

	void add(float phi, float psi, float omega, const char secstruct) {

		total++;

		periodic_range( phi  , 360.0 );  //does this get applied to phi??
		periodic_range( psi  , 360.0 );
		periodic_range( omega, 360.0 );

		if ( std::abs( omega ) < 90 ) {
			O++;
		} else if ( phi >= 0.0 ) {
			if ( -100 < psi && psi <= 100 ) {
				G++; // alpha-L
			} else {
				EE++; // E
			}
		} else {
			if ( -125 < psi && psi <= 50 ) {
				A++; // helical
			} else {
				B++; // extended
			}
		}

		if ( secstruct == 'E'){
			E++;
		} else if ( secstruct == 'H') {
			H++;
		} else {
			L++;
		}

	} // add

	static void add_flav(std::vector<flavor>& v, const int res_num, const int type, const char c, const float freq ) {
		if( freq != 0 ) {
			flavor f;
			f.frequency = freq;
			constraint cst;
			cst.residue = res_num;
			cst.weight  = 1;
			cst.torsion = type;
			cst.char_value = c;
			f.constraints.push_back(cst);
			v.push_back(f);
		}
	} // big_bin_flav

	const std::vector<flavor> get_all_flavors(const float loop_bias, const float sum_loop_bias) const {
		std::vector<flavor> rval;

		//feature_len + 1; // stupid unused parameter warning

		using namespace barcode_param;

		// reweighting sampling - 2006-02-03

// 		add_flav(rval,res_num,SS_TORSION,'H',H/s);
// 		add_flav(rval,res_num,SS_TORSION,'E',E/s);
// 		add_flav(rval,res_num,SS_TORSION,'L',L/s);
// 		add_flav(rval,res_num,BB_BIG_BIN_TORSION,'A',A/s);
// 		add_flav(rval,res_num,BB_BIG_BIN_TORSION,'B',B/s);
// 		add_flav(rval,res_num,BB_BIG_BIN_TORSION,'G',G/s);
// 		add_flav(rval,res_num,BB_BIG_BIN_TORSION,'E',EE/s);
// 		add_flav(rval,res_num,BB_BIG_BIN_TORSION,'O',O/s);

		//float s = sum() * feature_len; // XXX is this correct for the final feature block?

		//float s = loop_bias/(sum() * feature_len); // XXX is this correct for the final feature block?

		float s = static_cast<float>(loop_bias) / sum_loop_bias / sum();

		std::cout << "BbCode::get_all_flavors, loop_bias=" << loop_bias << ", sum_loop_bias=" << sum_loop_bias << ", s=" << s << std::endl;

 		add_flav(rval,res_num,SS_TORSION,'H',H*s);
 		add_flav(rval,res_num,SS_TORSION,'E',E*s);
 		add_flav(rval,res_num,SS_TORSION,'L',L*s);
 		add_flav(rval,res_num,BB_BIG_BIN_TORSION,'A',A*s);
 		add_flav(rval,res_num,BB_BIG_BIN_TORSION,'B',B*s);
 		add_flav(rval,res_num,BB_BIG_BIN_TORSION,'G',G*s);
 		add_flav(rval,res_num,BB_BIG_BIN_TORSION,'E',EE*s);
 		add_flav(rval,res_num,BB_BIG_BIN_TORSION,'O',O*s);

		return rval;

	} // get_all_flavors

// 	const char get_most_common_ss() const {

// 		if( H >= E && H >= L ) return 'H';
// 		if( E >= H && E >= L ) return 'E';
// 		if( L >= H && L >= E ) return 'L';

// 	} // get_most_common_ss

	bool ss_is_above_threshold(char ss, double threshold ) const {

		double f = 0;

		switch( ss ) {
		case 'H': f = H; break;
		case 'E': f = E; break;
		case 'L': f = L; break;
		default:  f = L; break;
		}

		return f/(H+E+L) > threshold;

	} // ss_is_above_threshold

	static bool ss_is_above_threshold(const std::map<int,BbCode>& mBbCodes, int res_num, char ss, double threshold ) {
		std::map<int,BbCode>::const_iterator i = mBbCodes.find(res_num);

		if( i == mBbCodes.end() ) {
			return true;
		}
		else {
			return i->second.ss_is_above_threshold(ss,threshold);
		}

	} // ss_is_above_threshold

}; // class BbCode

std::ostream& operator<<(std::ostream& out, const BbCode& bc ) {
	out << "res_num = " << bc.res_num
			<< "\tH=" << bc.H
			<< "\tE=" << bc.E
			<< "\tL=" << bc.L
			<< "\tA=" << bc.A
			<< "\tG=" << bc.G
			<< "\tO=" << bc.O
			<< "\tB=" << bc.B
			<< "\tEE=" << bc.EE;
	return out;
} // operator<<

const std::map<int,BbCode> get_bb_codes(int const total_residue) {

	using namespace std;
	using namespace fragments;
	using namespace param;

	//cout << "PAUL: get_bb_codes" << endl;

	//const int feature_length = 10;
	const int frag_size = 3;
	const int size_bin = get_index_by_frag_size(frag_size);

	std::map<int,BbCode> rval;

	for( int i = 1; i <= total_residue; ++i ) {

		//cout << "PAUL: i=" << i << endl;

		BbCode bc;
		bc.res_num = i;

		int first = std::max(1,i-frag_size+1);
		int last = std::min(i,total_residue-frag_size+1);
 		//int offset = i-frag_begin; // position of i in fragment

		for ( int frag_begin = first; frag_begin <= last; ++frag_begin ) {

			int offset = i-frag_begin; // position of i in fragment
			for ( int neighbor = 1, max_neigh = MAX_NEIGH(); neighbor <= max_neigh; ++neighbor ) {

				float phi   = align_phi(frag_begin,neighbor,offset,size_bin);
				float psi   = align_psi(frag_begin,neighbor,offset,size_bin);
				float omega = align_omega(frag_begin,neighbor,offset,size_bin);
				char secstruct = ss_type(frag_begin,neighbor,offset,size_bin);

				bc.add(phi,psi,omega,secstruct);
			}

		} // frag_begin

		rval[bc.res_num] = bc;

	} // i

	//cout << "PAUL: done with get_bb_codes" << endl;

	return rval;

} // get_bb_code

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::analyze_fragments
///
/// @brief: impose constraints determined by an analysis of the fragments
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors:
///
/// @last_modified:
/////////////////////////////////////////////////////////////////////////////////

void
feature_set::analyze_fragments(
	int const total_residue,
	int const mode_in
)
{
	using namespace std;
	using namespace barcode_param;
	using namespace flexibility;

	cout << "feature_set::analyze_fragments: analyzing fragments to generate consistent sampling barcode" << endl;

	nres = total_residue; // store in private data member
	mode = mode_in;

	const map<int,BbCode> mBbCode = get_bb_codes(total_residue);

	for(map<int,BbCode>::const_iterator i = mBbCode.begin(); i != mBbCode.end(); ++i ) {
		std::cout << (*i).second << std::endl;
	}

	// <PM>
	// features.clear();
	// this isn't compatible with read_file
	// </PM>

	//const int feature_len = 10;

	int feature_len;
	intafteroption("barcode_from_fragments_length",10,feature_len);

	// 	bool use_loop_bias = true_false_option("barcode_from_fragments_use_loop_bias")
	// 	truefalseoption("barcode_from_fragments_use_loop_bias",false,use_loop_bias);

	float loop_bias_score;
	realafteroption("barcode_from_fragments_loop_bias_score",2.0,loop_bias_score);

	const int num_features = static_cast<int>(ceil(static_cast<float>(total_residue)/feature_len));

	for( int k = 0; k < num_features; ++k ) {

		ostringstream feature_num;
		feature_num.str("");
		feature_num << "bff_" << setw(3) << setfill('0') << k;

		// <2006-02-03> upweighting flavor frequency near loop regions
		// => should change some of these to be command line parameters, done

		map<int,float> loop_bias;
		float sum_loop_bias = 0;

		for( int i = 1 + k*feature_len; i < min(total_residue,1 + (k+1)*feature_len); ++i ) {

			bool loop_prev_2 = BbCode::ss_is_above_threshold(mBbCode,i-2,'L',0.5);
			bool loop_prev_1 = BbCode::ss_is_above_threshold(mBbCode,i-1,'L',0.5);
			bool loop_curr   = BbCode::ss_is_above_threshold(mBbCode,i  ,'L',0.5);
			bool loop_next_1 = BbCode::ss_is_above_threshold(mBbCode,i+1,'L',0.5);
			bool loop_next_2 = BbCode::ss_is_above_threshold(mBbCode,i+2,'L',0.5);

			if( loop_prev_2 || loop_prev_1 || loop_curr || loop_next_1 || loop_next_2 ) {
				loop_bias[i] = loop_bias_score;
			}
			else {
				loop_bias[i] = 1;
			}

			sum_loop_bias += loop_bias[i];
		} // i

		// </2006-02-03>

		for( int i = 1 + k*feature_len; i < min(total_residue,1 + (k+1)*feature_len); ++i ) {

			// for each of H=595   E=2     L=3     A=597   G=0     O=0     B=3     EE=0

			map<int,BbCode>::const_iterator bb_code = mBbCode.find(i);
			assert( bb_code != mBbCode.end() );

			vector<flavor> vFlavors = bb_code->second.get_all_flavors(loop_bias[i],sum_loop_bias);
			features[feature_num.str()].new_flavors(vFlavors);

		} // i
	} // k


// 			flavor f;

// 			f.frequency = // something;

// 			constraint cst;
// 			cst.torsion = BB_BIG_BIN_TORSION;
// 			cst.weight = 1;
// 			cst.residue = 15;
// 			cst.char_value = 'E';


// 			f.constraints.push_back(cst);

// 	features[tag].new_flavor(f);


	//fragment_diversity()

	show_data(std::cout);

	reset(); // turns all features to off-state

}

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::show_data
///
/// @brief: show the feature, flavor data
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors:
///
/// @last_modified: 11/23/04
/////////////////////////////////////////////////////////////////////////////////
void feature_set::show_data(std::ostream& out) {
	// show all the data:
	for ( feature_iterator it=features.begin();
				it!= features.end(); ++it ) {
		out << "Feature: " << it->first << std::endl; // show the tag
		for ( flavor_iterator it2=it->second.begin();
					it2 != it->second.end(); ++it2) {
			out << "Flavor " << it2->frequency << std::endl;
			for ( constraint_iterator it3=it2->constraints.begin();
						it3!=it2->constraints.end();++it3) {
				out << (*it3) << std::endl;
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::increment_counters
///
/// @brief: increase counter for indication of active constraint
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors:
///
/// @last_modified: 11/23/04
/////////////////////////////////////////////////////////////////////////////////
void
feature_set::increment_counters()
{

// reuse this variable throughout
	feature_iterator it;

// modes:
// 0 = use only value[0],
// 1= full combinatorics,
// 2=vary one at a time,
// others off

	if ( mode == 0 ) {
		for ( it = features.begin(); it != features.end(); ++it ) {
			it->second.reset(); // turns off
			it->second.increment(); // sets to state # 1
		}
	} else if ( mode == 1 ) { // full combinatorics
		it = features.begin();
		while ( it->second.increment() ) {
			++it;
			if ( it == features.end() ) {
				it = features.begin();
				break; // now in the all-off state
			}
		}
	} else if ( mode == 2 ) { // one at a time, never hit the all-off state?
		if ( current_feature->second.increment() ) {
			++current_feature;
			if ( current_feature == features.end() )
				current_feature = features.begin();
			current_feature->second.increment(); // remove this call to hit the all-off state
		}
	} else if ( mode == 3 ) {
		for ( it = features.begin(); it != features.end(); ++it ) {
			feature & f = it->second;
			f.reset();
			float total_freq = f.current_frequency(); // the frequency of the off-state
			float const randm = ran3();
			while ( total_freq < randm ) {
				f.increment();
				total_freq += f.current_frequency();
			}
		}
	} else if ( mode == 4) {
		// nothing to do
	} else {
		std::cout << "ERROR:: Barcode unsupported mode:" << SS(mode) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::cout << "Current barcode settings: " << concise_flavor_output() << std::endl;
	update_active_constraints();


	// barcode_initialize_decoy will now call: cst_list.fill_bb_tether_arrays()
}

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::fill_bb_tether_arrays
///
/// @brief: called from barcode_initialize_decoy. Fill in bb_tether arrays
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors:
///
/// @last_modified:
/////////////////////////////////////////////////////////////////////////////////
void
feature_set::fill_bb_tether_arrays(
	FArray1Da_float tether_angle_res_weight,
	FArray1Da_float phi_tether,
	FArray1Da_float psi_tether
)
{
	tether_angle_res_weight.dimension( nres );
	phi_tether.dimension( nres );
	psi_tether.dimension( nres );

	for ( int i = 1; i <= nres; ++i ) {
		tether_angle_res_weight(i)=0.;
	}

	for ( constraint_iterator it = begin(), it_end = end();
	 it != it_end; ++it ) {
		if ( !it->bb_tether_constraint() ) continue;

		int const res_num = it->residue;
		int const torsion = it->torsion;
		float const value = it->float_value;

		tether_angle_res_weight( res_num ) = it->weight; // assumes psi/psi occur together

		if ( torsion == 1 ) {
			phi_tether( res_num ) = value;
		} else if ( torsion == 2) {
			psi_tether( res_num ) = value;
		} else {
			std::cout << "unsupported bb_tether torsion:" << SS( torsion) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
}


/////////////////////////////////////////////////////////////////////////////////
void
feature_set::fill_barcode_energy_arrays()
{
	using namespace barcode_classes::barcode_param;
	using namespace barcode_energy;

	//Clear out energy arrays.
	barcode_contact_energy   = 0.0;
	barcode_contact_distance = 0.0;
	barcode_beta_pair_energy = 0.0;
	barcode_big_bin_energy   = 0.0;
	barcode_ss_energy        = 0.0;
	barcode_dssp_energy      = 0.0;
	barcode_bblum_energy     = 0.0;
	barcode_env_energy       = 0.0;

	ignore_beta_pair_barcode_in_score = truefalseoption("ignore_beta_pair_barcode_in_score");

	need_DSSP = false;

	for ( constraint_iterator it = begin(), it_end = end();
	 it != it_end; ++it ) {

		int const constraint_type = it->torsion;

		switch (constraint_type) {
		case CONTACT_ENERGY_TORSION:
			barcode_contact_energy  ( it->residue, it->int_value ) += it->weight;
			barcode_contact_distance( it->residue, it->int_value ) = (it->float_value1) * (it->float_value1);
			break;
		case BETA_PAIR_ENERGY_TORSION:
			barcode_beta_pair_energy( it->int_value1, it->int_value2, it->int_value3, it->int_value4 ) += it->weight;
			if(!ignore_beta_pair_barcode_in_score)
  			need_DSSP = true;
			break;
		case SS_ENERGY_TORSION:
			barcode_ss_energy( it->residue, get_secstruct_state( it->char_value ) ) += it->weight;
			break;
		case BIG_BIN_ENERGY_TORSION:
			barcode_big_bin_energy( it->residue, get_big_bin_state( it->char_value ) ) += it->weight;
			break;
		case DSSP_ENERGY_TORSION:
			barcode_dssp_energy( it->residue, get_dssp_state( it->char_value ) ) += it->weight;
			need_DSSP = true;
			break;
		case BBLUM_ENERGY_TORSION:
			barcode_bblum_energy( it->residue, get_bblum_state( it->char_value ) ) += it->weight;
			need_DSSP = true;
			break;
		case ENV_ENERGY_TORSION:
			barcode_env_energy( it->residue, it->int_value )   += it->weight;
			break;
		default: break;
		}

	}

}

/////////////////////////////////////////////////////////////////////////////
// RNA jumping. This is a bit spread out between barcode_classes.cc,
//  and pose_rna_jumping.cc. Maybe a little confusing!
void
feature_set::update_pairing_list_RNA()
{
	using namespace barcode_classes::barcode_param;

	clear_barcode_pairing_list_RNA();

	for ( constraint_iterator it = begin(), it_end = end();
				it != it_end; ++it ) {

		int const constraint_type = it->torsion;

		if (constraint_type == RNA_BP_JUMP_TORSION){
			add_to_barcode_pairing_list_RNA( it->int_value1, it->int_value2, it->residue,
																			 it->char_value1, it->char_value2, it->char_value );
		}
	}

}

/////////////////////////////////////////////////////////////////////////////////////
void
feature_set::show_constraints( utility::io::orstream & out )
{
	out << "feature_set_nres:" << SS(nres) << '\n';

	for ( constraint_iterator it = begin(), it_end = end(); it != it_end; ++it ) {
		out << *it << '\n';
	}

	// show by sc maps
	for ( int i = 1; i <= nres; ++i ) {
		for ( constraint_iterator it = sc_begin( i ), it_end = sc_end( i ); it != it_end; ++it ) {
			out << "sc_cst:" << *it << '\n';
		}
	}
}

std::string
feature_set::concise_flavor_output()
{
	feature_iterator it;
	std::stringstream output_string;

	for (it = features.begin(); it != features.end(); ++it){
		feature & f = it->second;

		if (it == features.begin()){
			output_string << f.current_index();
		} else {
			output_string << ',';
			output_string << f.current_index();
		}
	}

	return output_string.str();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::flavordist
///
/// @brief: angle distance of a phipsi array to a flavor
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: scheffer, bqian
///
/// @last_modified: 11/23/04
/////////////////////////////////////////////////////////////////////////////////
float
feature_set::flavordist(
	flavor f,
	FArray1D_float const & phi,
	FArray1D_float const & psi
)
{
	std::vector< constraint >::iterator this_cst;
	float distance = 0;
	float num_cst = 0;
	for ( this_cst = f.constraints.begin();
					this_cst != f.constraints.end();
					++this_cst ) {
		if ( this_cst->bb_cluster_constraint() ) {
			float phidist = subtract_degree_angles( this_cst->float_value,
			 phi(this_cst->residue) );
			float psidist = subtract_degree_angles( this_cst->float_value1,
			 psi(this_cst->residue) );
			distance +=  phidist * phidist + psidist * psidist;
			++num_cst;
		} else {
			std::cout
					<< "WARNING: feature_set::get_closest_barcode "
					<< "ignoring non-BB_CLUSTER constraint!!" << std::endl;
		}
	}
	return std::sqrt(distance/num_cst);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::get_closest_barcode
///
/// @brief: get the flavor code for a phipsi array
///
/// @detailed: if this is a new flavor, add it to the flavor list
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: scheffer, bqian
///
/// @last_modified: 11/23/04
/////////////////////////////////////////////////////////////////////////////////
std::map< std::string, int >
feature_set::get_closest_barcode() {
	return get_closest_barcode( misc::phi,misc::psi );
}

std::map< std::string, int >
feature_set::get_closest_barcode(
	FArray1D_float const & phi,
	FArray1D_float const & psi
)
{
	std::map< std::string, int > barcode;
	std::map< std::string, feature >::iterator this_feature;

	for ( this_feature = features.begin();
	 this_feature != features.end();
	 ++this_feature ) {

		std::vector< flavor >::iterator this_flavor = this_feature->second.begin();

		++this_flavor; // first flavor is a dummy

		float min_dist = 9999999;
		int this_flavor_code = -1;
		int counter = 0;
		int original_code = 0;

		bool found_code = false;
		float thresh = this_flavor->constraints.begin()->float_value3;
		for ( this_flavor=this_flavor;
		 this_flavor != this_feature->second.end();
		 ++this_flavor ) {
			float tmp_dist = flavordist( *this_flavor, phi, psi );
			if ( tmp_dist < min_dist ) {
				min_dist = tmp_dist;
				this_flavor_code = counter;
				// for flavors added by increment_barcode (marked by frequency==0.0)
				// use the first one found within threshhold as the code
				if ( tmp_dist < thresh ) {
					if ( this_flavor->frequency > 0.0 )	{
						found_code = true;
						original_code = this_flavor_code;
					} else { break; }
				}
		}
			++counter;
		}

		if ( found_code ) {
			this_flavor_code = original_code;
		}
		if ( min_dist > thresh ) {// unknown flavor
			this_flavor_code = -1;
		}

		barcode[ this_feature->first ] = this_flavor_code;

	}

	return barcode;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::add new flavor
///
/// @brief: add a new flavor to a feature set
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: bqian
///
/// @last_modified: 11/23/04
/////////////////////////////////////////////////////////////////////////////////
void
feature_set::add_new_bbcluster_flavor(
	FArray1D_float const & phi,
	FArray1D_float const & psi
)
{
	std::map< std::string, feature >::iterator this_feature;

	for ( this_feature = features.begin();
	 this_feature != features.end();
	 ++this_feature ) {

		std::vector< flavor >::iterator this_flavor = this_feature->second.begin();
		++this_flavor; // first flavor is a dummy

		bool flavor_exist = false;
		float thresh = this_flavor->constraints.begin()->float_value3;
		for ( this_flavor=this_flavor;
		 this_flavor != this_feature->second.end();
		 ++this_flavor ) {
			float tmp_dist = flavordist( *this_flavor, phi, psi );
			if ( tmp_dist < thresh ) {
				flavor_exist = true;
				break; // found existing flavor
			}
		}

		if ( flavor_exist ) continue; // next feature

		// didn't find existing flavor
		this_flavor = this_feature->second.begin();
		++this_flavor; //first flavor is a dummy

		// construct a new flavor according to phi,psi
		flavor newflavor;
		newflavor.frequency = 0.0;

		std::vector< constraint >::iterator this_cst;
		for ( this_cst = this_flavor->constraints.begin();
				this_cst != this_flavor->constraints.end();
				++this_cst ) {
			if ( this_cst->bb_cluster_constraint() ) {
				constraint newcst = *this_cst;
				newcst.float_value  = periodic_range( phi(this_cst->residue), 360.0 );
				newcst.float_value1 = periodic_range( psi(this_cst->residue), 360.0 );
//				newcst.float_value3 = min_dist - thresh;
				newcst.weight = 0.0; //set the newly found cst weight 0.0
				newflavor.constraints.push_back( newcst );
			} else {
				std::cout
					<< "WARNING: feature_set::add_new_bbcluster_flavor "
					<< "ignoring non-BB_CLUSTER constraint!" << std::endl;
			}
		}
		this_feature->second.new_flavor( newflavor );

	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_set::dump_barcode_file
///
/// @brief: add a new flavor to a feature set
///
/// @detailed:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
///   The output has been modified to use ostringstreams instead of snprintf.
///   This may be produce differences in precision as the meaning of 'precision'
///   under sprintf is not the same as under iostreams:
///     snprintf: %.[X]f means X digits AFTER the decimal point.
///     ostreams: setprecision(X) means X significant digits before AND after
///       the decimal point.
///
/// @references
///
/// @authors: bqian, ion
///
/// @last_modified: 08/18/05
/////////////////////////////////////////////////////////////////////////////////
void
feature_set::dump_barcode_file()
{
	using std::cout;
	using std::endl;
	using std::fixed;             // I/O manipulator
	using std::map;
	using std::ostringstream;
	using std::setprecision;      //  I/O manipulator
	using std::setw;              //  I/O manipulator
	using std::string;
	using std::vector;

	map< string, feature >::iterator this_feature;
	for ( this_feature = features.begin(); this_feature != features.end();
        ++this_feature ) {

		vector< flavor >::iterator this_flavor = this_feature->second.begin();
		for ( ++this_flavor; // first flavor is a dummy
					this_flavor != this_feature->second.end();
					++this_flavor ) {

			ostringstream outline;

			outline << this_feature->first << "  "
							<< fixed << setprecision(2)
							<< setw(3) << this_flavor->frequency;

			vector< constraint >::iterator this_constraint;
			for ( this_constraint = this_flavor->constraints.begin();
						this_constraint != this_flavor->constraints.end();
						++this_constraint ) {

				outline << "BB_CLUSTER "
								<< setw(4) << this_constraint->residue << " "
								<< setprecision(2)
								<< setw(5) << this_constraint->weight << " "
								<< setprecision(1)
								<< setw(6) << this_constraint->float_value << " "
								<< setw(6) << this_constraint->float_value1 << " "
								<< setw(6) << this_constraint->float_value2 << " "
								<< setw(6) << this_constraint->float_value3 << "  ";
			}

			cout << "newbarcodes " << outline.str() << endl;

		}
	}

}


void
feature_set::set_force_rotamer( const bool setting )
{
	force_rotamer = setting;
}


bool
feature_set::get_force_rotamer() const
{
	return force_rotamer;
}


///////////////////////////////////////////////////////////////////////////////
bool
feature_set::sspair_constraints_exist() const
{
	bool exist( false );
	for ( constraint_const_iterator it = begin(), it_end = end();
				it != it_end; ++it ) {
		if ( it->sspair_constraint() ) {
			exist = true;
			break;
		}
	}
	return exist;
}

///////////////////////////////////////////////////////////////////////////////
int
feature_set::num_bonus_sspair_constraints() const
{
	int num_bonus_sspair = 0;
	for ( constraint_const_iterator it = begin(), it_end = end();
				it != it_end; ++it ) {
		if ( it->sspair_constraint_bonus() )    num_bonus_sspair++;
	}
	return num_bonus_sspair;
}

///////////////////////////////////////////////////////////////////////////////
float
feature_set::sspair_constraint_score(
	int const pos1,
	int const pos2,
	float const theta
) const
{
	float score(0.0);
	// will this be too slow? called inside sspair score
	// for each dimer pair... well -- actually only for dimer_pairs
	// with favorable dimer scores... probably not too many then
	for ( constraint_const_iterator it = begin(), it_end = end();
				it != it_end; ++it ) {
		if ( it->sspair_constraint() ) {
			score += it->sspair_constraint_score( pos1, pos2, theta );
		}
	}
	return score;
}

///////////////////////////////////////////////////////////////////////////////
bool
feature_set::disulf_constraints_exist() const
{
	bool exist( false );
	for ( constraint_const_iterator it = begin(), it_end = end();
				it != it_end; ++it ) {
		if ( it->disulf_constraint() ) {
			exist = true;
			break;
		}
	}
	return exist;
}


///////////////////////////////////////////////////////////////////////////////
bool
feature_set::beta_pair_constraints_exist() const
{
	bool exist( false );
	for ( constraint_const_iterator it = begin(), it_end = end();
				it != it_end; ++it ) {
		if ( it->beta_pair_constraint() ) {
			exist = true;
			break;
		}
	}
	return exist;
}

///////////////////////////////////////////////////////////////////////////////
int
feature_set::num_bonus_beta_pair_constraints() const
{
	int num_bonus_beta_pair = 0;
	for ( constraint_const_iterator it = begin(), it_end = end();
				it != it_end; ++it ) {
		if ( it->beta_pair_constraint_bonus() )    num_bonus_beta_pair++;
	}
	return num_bonus_beta_pair;
}

} // namespace barcode_classes

