// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15424 $
//  $Date: 2007-06-12 21:26:56 +0300 (Tue, 12 Jun 2007) $
//  $Author: bcorreia $


// Rosetta Headers
#include "barcode_stats.h"
#include "after_opts.h"
#include "barcode_stats_classes.h"
#include "cenlist.h"
#include "current_pose.h"
#include "force_barcode.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "fragments_pose.h"
#include "initialize.h" // native routines
#include "jumping_pairings.h"
#include "jumping_util.h"
#include "misc.h"
#include "pose.h"
#include "pose_io.h"
#include "pose_rms.h"
#include "pose_vdw.h"
#include "random_numbers.h"
#include "refold.h"
#include "score.h" // score_set_new_pose
#include "silent_input.h"
#include "structure.h"
#include "taboo_search.h"
#include "util_vector.h"
#include "pose_idealize.h"
#include "vdw.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cassert>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <set>
#include <string>
#include <utility>
#include <vector>


// local counters namespace
namespace strand_pairings_ns {
	bool store_ss ( false );
	//FArray2D_int pairing_count;
	FArray2D_int decoy_pairing_count;
	FArray2D_float decoy_pairing_score;
	FArray1D_int strand_begin;
	FArray1D_int strand_end;
}


// Using
using namespace silent_io;
using namespace barcode_stats_ns; // gives us the classes


///////////////////////////////////////////////////////////////////////////////
// this file contains routines for the mode -barcode stats
// which collects barcode data on a set of decoys
//


///////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_stats
///
/// @brief collect barcode stats on silent-file of decoys
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_barcode_stats()
{
	using namespace misc;
	using namespace fragments;

	bool const count_hairpins ( truefalseoption("count_hairpins") );

	// the counters:
	hairpin_counter
	 frag_hairpin_count( total_residue ),
	 decoy_hairpin_count( total_residue ),
	 native_hairpin_count( total_residue );

	//// first get the fragment hairpin frequencies
	if ( count_hairpins ) barcode_stats_fragments( frag_hairpin_count );

	////////////////////////////////// now look at the decoys /////////////////
	// read the silent-file
	Silent_file_data silent_data( stringafteroption( "silent_file" ) );
	if ( silent_data.size() <= 0 ) {
		std::cout << "STOP: empty silent-file" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( silent_data.nres() != total_residue ) {
		std::cout << "length mismatch in silent file!\n"
							<< "silent file nres= " << silent_data.nres() << '\n'
							<< "total_residue= " << total_residue << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// read barcode csts
	barcode_initialize_start( total_residue );

	// process the decoys //////////////
	barcode_stats_decoys( decoy_hairpin_count, silent_data, count_hairpins );
	if ( get_increment_barcode() )
		dump_barcode();

	// get native stats
	if ( get_native_exists() ) {
		retrieve_native_centroid_pose();
		if ( count_hairpins ) count_decoy_hairpins( native_hairpin_count );
	}

	if ( count_hairpins ) {
		int const total_decoys = silent_data.size();

		for ( int pos = 1; pos <= total_residue; ++pos ) {
			for ( int loop_length = 1; loop_length <= barcode_hairpins::MAX_LOOP_LENGTH;
			 ++loop_length ) {
				// now show these as fractions
				float const actual = float( decoy_hairpin_count.get_count( pos ,loop_length) ) /
					total_decoys;
				float const expected = float( frag_hairpin_count.get_count( pos, loop_length) ) /
					barcode_hairpins::MAX_NN;

				std::cout << "hairpins: pos= " << pos << " loop_length= " << loop_length
				 << " native= " << native_hairpin_count.get_count( pos, loop_length )
				 << " decoy= " << actual << " frag_expected= " << expected << std::endl;
			}
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_stats_decoys
///
/// @brief collect barcode stats on silent-file of decoys
///
/// @detailed this routine counts hairpins in decoys
/// it also identifies decoy-pairing features
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
barcode_stats_decoys(
	hairpin_counter & decoy_hairpin_count,
	Silent_file_data & silent_data,
	bool const count_hairpins
)
{
	using namespace misc;
	using namespace strand_pairings_ns;

	// params
	int const   big_threshold(6);
	int const small_threshold(6); // should be 4?

	int const total_decoys = silent_data.size();


	// initialize counts
	// local
	FArray2D_int pairing_count( total_residue, total_residue, 0 );

	// in a global namespace
	decoy_pairing_count.dimension( total_residue, total_residue);
	decoy_pairing_score.dimension( total_residue, total_residue);

	// loop over all the structures from the silent file
	int decoy_counter=0;
	for ( Silent_file_data::const_iterator it = silent_data.begin(),
	 it_end = silent_data.end(); it != it_end; ++it) {
		std::string tag = it->first;
		Silent_structure const * structure = it->second; // pointer to the data

		if ( ++decoy_counter%20 == 0 ) {
			std::cout << "rescore decoys: " << decoy_counter << ' ' << tag << std::endl;
		}

		// copy into the current arrays
		for ( int i = 1; i <= total_residue; ++i ) {
			phi      (i) = structure->phi      (i);
			psi      (i) = structure->psi      (i);
			omega    (i) = structure->omega    (i);
			secstruct(i) = structure->secstruct(i);
		}

		// refold
		refold(1,total_residue);

		// fills some useful arrays
		barcode_stats_score_ss();

		if ( barcode_exist() ) {
			if ( get_increment_barcode() ) {
				add_barcode( misc::phi, misc::psi );
			}

			std::cout << "flavor code: " << tag << ' '
								<< map_to_string( get_closest_barcode() )
								<< std::endl;
		}

		if ( count_hairpins ) count_decoy_hairpins( decoy_hairpin_count );

		for ( int i = 1; i<= total_residue; ++i ) {
			for ( int j =1; j<= total_residue; ++j ) {
				if ( decoy_pairing_count(i,j) > 0 ) {
					++pairing_count(i,j);
				}
			}
		}

	} // iterate over the structures in the silent-file

	/////////////////////////
	// identify beta features

	// typedef std::vector< Beta_feature> Feature_list;
	Feature_list feature_list;
	identify_beta_features( total_residue, pairing_count, decoy_counter,
		feature_list );


	///////////////////////////////////////////////////// rescore
	// now rescore the decoys to assign features
	// loop over all the structures from the silent file
	decoy_counter=0;

	// keep a count of how many decoys have each feature:
	std::map< Beta_feature, int > feature_count;

	for ( Silent_file_data::const_iterator it = silent_data.begin(),
					it_end = silent_data.end(); it != it_end; ++it) {
		std::string tag = it->first;
		Silent_structure* structure = it->second; // pointer to the data

		if ( ++decoy_counter%20 == 0 ) {
			std::cout << "rescore decoys: " << decoy_counter << ' ' << tag << std::endl;
		}

		// copy into the current arrays
		for ( int i = 1; i <= total_residue; ++i ) {
			phi      (i) = structure->phi      (i);
			psi      (i) = structure->psi      (i);
			omega    (i) = structure->omega    (i);
			secstruct(i) = structure->secstruct(i);
		}

		// refold
		refold(1,total_residue);

		// fills some useful arrays
		barcode_stats_score_ss();

		Feature_list decoy_features;

		get_decoy_features( feature_list, decoy_pairing_count,
		 big_threshold, small_threshold, decoy_features );

		// show the features, and increment the feature_count counter
		std::cout << "DECOY_FEATURES " << tag;
		for ( Feature_list::iterator it = decoy_features.begin(),
		 it_end = decoy_features.end(); it != it_end; ++it ) {
			std::cout << ' ' << *it;
			++feature_count[*it];
		}
		std::cout << std::endl;

	} // iterate over the structures in the silent-file

	// output the frequencies for each feature
	for ( Feature_list::iterator it = feature_list.begin(),
					it_end = feature_list.end(); it != it_end; ++it ) {
		std::cout << "FEATURE_FREQUENCY " << *it
		 << SS( float( feature_count[*it] ) / total_decoys ) << std::endl;
	}

	// what features does the native have?
	if ( get_native_exists() ) {
		retrieve_native_centroid_pose();

		barcode_stats_score_ss();

		Feature_list native_features;

		get_decoy_features( feature_list, decoy_pairing_count,
		 big_threshold, small_threshold, native_features );

		// show the features, and increment the feature_count counter
		std::cout << "NATIVE_FEATURES ";
		for ( Feature_list::iterator it = native_features.begin(),
		 it_end = native_features.end(); it != it_end; ++it ) {
			std::cout << ' ' << *it;
		}
		std::cout << std::endl;
	} // native exists?
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_decoy_features
///
/// @brief fill a list of the features that match a particular decoy
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
// first arg is the list of all features
// adds the decoy's features to the second argument IF NOT ALREADY PRESENT!!!!!
// ie -- we're assuming that the second arg is an empty list to begin with
// and this function is called to identify the list of features present.

void
get_decoy_features(
	Feature_list const & feature_list,
	FArray2D_int const & dec_pairing_count, // shadowing namespace: decoy_pairing_count
	int const big_threshold,
	int const small_threshold,
	Feature_list & decoy_features
)
{
	// debugging:
	static int total_pairings(0);
	static int matched_pairings(0);
	static int total_function_calls(0);

	if ( ++total_function_calls%100 == 0 && total_pairings>0 ) {
		// show coverage of total pairings by the set of features
		std::cout << "total_pairings " << total_pairings
		 << " matched_pairings " << matched_pairings
		 << " fraction " << float(matched_pairings) / total_pairings << std::endl;
	}

	for ( int i = 1; i <= misc::total_residue; ++i ) {
		for ( int j = 1; j <= misc::total_residue; ++j ) {
			if ( dec_pairing_count(i,j) > 0 ) {
				++total_pairings;

				int const pos1 = std::min(i,j);
				int const pos2 = std::max(i,j);
				int const o ( ( i<j ) ? 1 : 2 );
				Beta_feature const f(pos1,pos2,o);
				Feature_list::const_iterator it
					( get_closest_feature( f,feature_list, big_threshold, small_threshold ) );

				if ( it != feature_list.end() ) { // found a match
					++matched_pairings;

					if ( std::find( decoy_features.begin(), decoy_features.end(), *it ) ==
							 decoy_features.end() ) {
						decoy_features.push_back( *it);
					}
				} // matches a feature
			} // has pairing
		} // j
	} // i
}


/////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin get_decoy_features
///
/// @brief fill a list of the features that match a particular decoy
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
void
get_decoy_features_by_score(
	Feature_list const & feature_list,
	FArray2D_int const & dec_pairing_count,
	FArray2D_float const & dec_pairing_score,
	int const big_threshold,
	int const small_threshold,
	float const max_score,
	Feature_list & decoy_features
)
{
	std::map< Beta_feature, float > feature_score;

	for ( int i = 1; i <= misc::total_residue; ++i ) {
		for ( int j = 1; j <= misc::total_residue; ++j ) {
			if ( dec_pairing_count(i,j) > 0 ) {
				int const pos1 = std::min(i,j);
				int const pos2 = std::max(i,j);
				int const o ( ( i<j ) ? 1 : 2 );
				Beta_feature const f(pos1,pos2,o);
				Feature_list::const_iterator it
					( get_closest_feature( f,feature_list, big_threshold, small_threshold ) );

				if ( it != feature_list.end() ) { // found a match
					feature_score[ *it ] += dec_pairing_score( i, j );
				} // matches a feature
			} // has pairing
		} // j
	} // i

	for ( std::map< Beta_feature, float >::const_iterator
					it=feature_score.begin(), it_end = feature_score.end(); it != it_end;
				++it ) {
		if ( it->second <= max_score ) { // negative!
			decoy_features.push_back( it->first );
		} else {
			std::cout << "skip feature: score= " << it->second << " > " <<
				max_score << std::endl;
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_stats_score_ss
///
/// @brief SS-score the current structure, fill some arrays
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
// we assume that Eposition, centroid reflect the structure to be scored
// partial score calculation: calls update_cendist, identify_ss, sspair_score

void
barcode_stats_score_ss()
{
	using namespace misc;
	using namespace structure::BOUNDARY; // total_strands, strand_strand_score
	using namespace strand_pairings_ns; // local to this file

	// zero the per-decoy pairing_count array, in strand_pairings_ns
	// this count is incremented by store_residue_pairing, which is called
	// by SSpair_score

	decoy_pairing_count = 0;
	decoy_pairing_score = 0;

	score_set_new_pose();
	update_cendist(total_residue, centroid);
	identify_ss();

	int const lowstrand = 0; // as in score4
	int const cutoff = 6; // ditto; should consider making smaller??
	float tmp_ss_score, tmp_rsigma_score; // not used

	// tell sspair_score to record extra stuff
	set_store_ss_scores( true ); // actually, we can see this variable

	// call SSpair score!!
	SSpair_score(lowstrand,cutoff,tmp_ss_score,tmp_rsigma_score);
	set_store_ss_scores( false );

	int total_strands = get_total_strands();
	int total_SS_dimer = get_total_SS_dimer();
	FArray1D_int const & SS_resnum = get_SS_resnum();
	FArray1D_int const & SS_strand = get_SS_strand();
	FArray2D_int const & SS_strand_end = get_SS_strand_end();

	// figure out where the strands begin and end
	// should probably be done in identify_ss but I dont think it is
	// these arrays are indexed by strand_number!
	// also, they point to first or last E residue in the strand
	// this is different from SS_strand_end's behavior

	// these arrays are in strand_pairings_ns namespace

	strand_begin.dimension(total_strands);
	strand_end.dimension(total_strands);
	strand_begin = 0; // must be 0'd out
	strand_end = 0;

	for ( int i = 1; i <= total_SS_dimer; ++i ) {
		int const pos = SS_resnum(i); // the first residue of the EE dimer
		int const strand = SS_strand(i);
		if ( !strand_begin(strand) ) strand_begin(strand) = pos;
		strand_end(strand) = pos+1; // dimers are two E residues long!!!!!!
		//std::cout << "tmp1" << SS(pos) << SS(strand_begin(strand))
		//					<< SS(strand_end(strand)) << std::endl;
	}

	// debug:
	{ // for scope
		int last_pos = -1; //debug:
		for ( int i = 1; i <= total_SS_dimer; ++i ) {
			int const pos = SS_resnum(i);
			int const strand = SS_strand(i);
			//std::cout << "tmp2" << SS(pos) << SS(strand)
			//					<< strand_begin( strand) <<  SS_strand_end(1,i)+1
			//					<< strand_end  ( strand) <<  SS_strand_end(2,i)-1 << std::endl;
			if ( strand_begin( strand) > 1 ) {
				assert( strand_begin( strand) == SS_strand_end(1,i)+1 );
			} else {
				assert( strand_begin( strand) == SS_strand_end(1,i) );
			}
			if ( strand_end( strand) < total_residue) {
				assert( strand_end  ( strand) == SS_strand_end(2,i)-1 );
			} else {
				assert( strand_end  ( strand) == SS_strand_end(2,i) );
			}
			// just for debugging the ordering of dimers
			assert( last_pos < pos );
			last_pos = pos;
		}
	}
}


//////////////////////////////////////////////////////////////////////////////
/// @begin count_decoy_hairpins
///
/// @brief collect hairpin stats on current pose
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
// we assume that Eposition, centroid reflect the structure to be scored
// partial score calculation: calls update_cendist, identify_ss, sspair_score

void
count_decoy_hairpins( hairpin_counter & counter )
{
	int total_strands = structure::BOUNDARY::get_total_strands();
	FArray2D_float const & strand_strand_score = structure::BOUNDARY::get_strand_strand_score();

	for ( int strand1 = 1; strand1 < total_strands; ++strand1 ) {
		int const strand2 = strand1 + 1;
		// since strand1<strand2 this accesses the side of this array
		// devoted to anti-parallel pairings:
		float const hp_score = strand_strand_score( strand1, strand2 );
		if ( hp_score < 0.0 ) { // still, may not count, depending on score threshold
			int const loop_begin = strand_pairings_ns::strand_end  (strand1) + 1;
			int const loop_end   = strand_pairings_ns::strand_begin(strand2) - 1;
			counter.store_decoy_hairpin( loop_begin, loop_end, hp_score);
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_stats_fragments
///
/// @brief collect hairpin stats on fragments
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified

void
barcode_stats_fragments( hairpin_counter & frag_hairpin_count )
{
	using namespace misc;
	using namespace fragments;
	using namespace barcode_hairpins; // some params

	bool const local_verbose = false;

	// read the fragment definitions from the vall:
	vall_hairpins_type vall_hairpins;
	read_vall_hairpins( vall_hairpins );

	// now go through the frags, counting hairpins ///////////////////////////////
	int const size = 9;
	int const size_bin = get_index_by_frag_size(size);
	int const window_pos = 4; // central residue of 9mer (0...8)

	// the counts:

	for ( int start=1; start <= total_residue - size + 1; ++start ) {
		assert( align_depth(start,size_bin) >= MAX_NN );
		for ( int nn = 1; nn <= MAX_NN; ++nn ) {

			// is the central residue in a hairpin?
			std::string const id( trimmed( align_name( start, nn, size_bin ) )
			 + align_chain( start, nn, size_bin) );

			// what is the vall rsd number of the central residue of this fragment
			// look at the silly way that the residue numbers are read in from the
			// fragment file. align_resseq will be the residue number of the last
			// residue of the fragment.

			int const vall_rsd_number = align_resseq( start, nn, size_bin)
				- size + 1 + window_pos;
			int const rsd_number = start + window_pos;

			if ( local_verbose ) {
				std::cout << SS(start) << SS(nn) << "id: (" << id << ")"
									<< SS(vall_rsd_number) << SS(rsd_number) << std::endl;
			}

			// now look-up inside vall_hairpins map
			vall_hairpins_type::iterator hp_data = vall_hairpins.find( id );
			if ( hp_data == vall_hairpins.end() ) {
				std::cout << "cant find id in hairpin list file: " << id << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			std::map< int, int >::iterator iter = hp_data->second.find( vall_rsd_number);
			if ( iter != hp_data->second.end() ) { // its a hairpin
				int const loop_length = iter->second;
				if ( local_verbose) {
					std::cout << "its a hairpin! " << id << SS(vall_rsd_number)
										<< SS(loop_length) << std::endl;
				}
				// one overloaded version of store_hairpin
				frag_hairpin_count.store_frag_hairpin( rsd_number, loop_length);
			}
		} // nn=1; nn <= MAX_NN; ++nn
	} // start = 1; start <= total_residue-size+1; ++start
}


///////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin read_vall_hairpins
///
/// @brief which residues in vall are in hairpins?
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified

void
read_vall_hairpins( vall_hairpins_type & vall_hairpins )
{

	//// first get the fragment hairpin frequencies
	utility::io::izstream vall_data ( "/data/pbradley/diversity/vall.hairpins_maxsep8.2001-02-02");
	std::string line;
	while ( std::getline( vall_data(), line ) ) {
		std::istringstream line_stream (line);
		std::string id;
		line_stream >> id;
		assert( id.size() == 5 );
		std::vector< int > hp_list;

		while ( true ) {
			int pos;
			line_stream >> pos;
			if ( line_stream.fail() ) break;
			hp_list.push_back( pos );
		}

		std::map< int, int > hp_length;
		int const list_length = hp_list.size();
		if ( list_length > 0 ) {
			// calculate the hairpin lengths
			std::sort( hp_list.begin(), hp_list.end() ); // ensure that they are in order

			int start_pos = hp_list[0];
			int end_pos = start_pos;
			for ( int i = 1; i <= list_length; ++i ) {
				if ( i == list_length || hp_list[i] != end_pos + 1 ) {
					// either a new loop or the end of the list
					int const loop_length = end_pos - start_pos + 1;
					for ( int pos = start_pos; pos <= end_pos; ++pos ) {
						hp_length[pos] = loop_length;
					}
					if ( i < list_length ) { // setup for the next loop
						start_pos = hp_list[i];
					}
				}

				if ( i < list_length ) end_pos = hp_list[i];
			}
		}

		vall_hairpins[ id ] = hp_length;
	}
  vall_data.close();
  vall_data.clear();
}

///////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin get_store_ss_scores
///
/// @brief access function: should we store pairings
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
get_store_ss_scores()
{
	return strand_pairings_ns::store_ss;
}
///////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin set_store_ss_scores
///
/// @brief access function
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_store_ss_scores( bool const value )
{
	strand_pairings_ns::store_ss = value;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin store_residue_pairing
///
/// @brief called by SS-pair score if get_store_ss_scores() == true
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// store the pairing between dimers:
// EE dimers are: dimer_pos1,dimer_pos1+1 AND dimer_pos2,dimer_pos2+1
//
// called from sspair_score if get_store_ss_scores() returns true

void
store_residue_pairing(
	int const dimer_pos1,
	int const dimer_pos2,
	float const dimer_score,
	int const nres // total_residue
)
{
	using namespace strand_pairings_ns; // decoy_pairing_*

	static float const MAX_DISTANCE_SQUARED( 36.0 ); // guess

	assert( dimer_pos1 +1 < dimer_pos2 );

	if ( int(decoy_pairing_count.size1()) < nres ) {
		std::cout <<"WARNING: decoy_pairing_count not initialized!" << std::endl;
		decoy_pairing_count.dimension( nres, nres );
		decoy_pairing_count = 0;
		decoy_pairing_score.dimension( nres, nres );
		decoy_pairing_score = 0.0;
	}

	// for storing the score: divide evenly among the matched pairings
	std::vector< std::pair< int, int > > pairings;

	// look for good geometry between triplets centered on dimer heads or tails
	for ( int off1=0; off1<2; ++off1 ) {
		for ( int off2=0; off2<2; ++off2 ) {
			// pos1,pos2 are central! residue of triplet
			int const pos1 = dimer_pos1 + off1;
			int const pos2 = dimer_pos2 + off2;

			if ( pos1<=1 || pos1>=nres || pos2<=1 || pos2>=nres ) continue;

			// are these C-alpha's closer than +-2 shifts?
			float const distance_squared( vec_dist2( misc::Eposition(1,2,pos1),
																							 misc::Eposition(1,2,pos2)) );


			float min_dis2 (1000.0);
			for ( int tmp_pos1 = pos1-2; tmp_pos1 <= pos1+2; tmp_pos1 +=2 ) {
				for ( int tmp_pos2 = pos2-2; tmp_pos2 <= pos2+2; tmp_pos2 +=2 ) {
					if ( tmp_pos1 == pos1 && tmp_pos2 == pos2 ||
							 tmp_pos1 != pos1 && tmp_pos2 != pos2 ||
							 tmp_pos1 < 1 || tmp_pos1 > nres ||
							 tmp_pos2 < 1 || tmp_pos2 > nres ||
							 tmp_pos2 <= pos1+1 || tmp_pos1 >= pos2-1 ) continue;
					min_dis2 = vec_dist2( misc::Eposition(1,2,tmp_pos1),
																misc::Eposition(1,2,tmp_pos2) );
					if ( min_dis2 < distance_squared ) {
						break;
					}
				}
				if ( min_dis2 < distance_squared) break; // gotcha!
			}


			if ( distance_squared > MAX_DISTANCE_SQUARED ||
					 distance_squared > min_dis2 ) continue;

			float orientation, pleating1, pleating2;

			get_pairing_geometry( pos1, pos2,	orientation, pleating1, pleating2 );

			if ( pleating1 * pleating2 > 0 ) {
				if ( orientation < 0 ) {
					++decoy_pairing_count( pos1, pos2 );
					pairings.push_back( std::make_pair( pos1, pos2) );
				} else {
					++decoy_pairing_count( pos2, pos1 );
					pairings.push_back( std::make_pair( pos2, pos1 ) );
				}
			}
		}
	}

	int const npairs( pairings.size() );

	for ( int i=0; i<npairs; ++i ) {
		int const pos1( pairings[i].first ); // could be < or > than pos2
		int const pos2( pairings[i].second );
		decoy_pairing_score( pos1, pos2 ) += dimer_score / npairs;
	}
}


///////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin get_CA_vectors
///
/// @brief setup C-alpha,C-alpha vectors for calculating pairing geometry
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_CA_vectors(
	FArray2D_float const & ca1, // pass by reference, so no tricks:: 3x3
	FArray2D_float const & ca2, // pass by reference, so no tricks:: 3x3
	FArray1D_float & a,
	FArray1D_float & b,
	FArray1D_float & c
)
{
//    double a1[3],b1[3],b2[3],b3[3],c1[3];
	static FArray1D_float a1(3), b1(3), b2(3), b3(3), c1(3);

/*       a goes from c-alpha #1 to c-alpha #3 */
  subvec(ca1(1,3),ca1(1,1),a1);
//    subvec(ca1+6,ca1,a1);
  unitvec(a1,a);

/*       b gives direction of pleat for ca1 c-alphas */
  subvec(ca1(1,2),ca1(1,1),b1);
  subvec(ca1(1,2),ca1(1,3),b2);
//    subvec(ca1+3,ca1,b1);
//    subvec(ca1+3,ca1+6,b2);
  vector_sum(b1,b2,b3);
  unitvec(b3,b);

/*       c goes from ca1 triple to ca2 triple (central alpha-carbons) */
  subvec(ca2(1,2),ca1(1,2),c1);
//    subvec(ca2+3,ca1+3,c1);
  unitvec(c1,c);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_pairing_geometry
///
/// @brief get geometry of residue pairing
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// pos1,pos2 are central residues of triplets
void
get_pairing_geometry(
	int const pos1,
	int const pos2,
	float & orientation,
	float & pleating1,
	float & pleating2
)
{
	assert( pos1>1 && pos1<misc::total_residue &&
	 pos2>pos1 && pos2<misc::total_residue);

	static FArray2D_float ca1(3,3), ca2(3,3);
	static FArray1D_float a1(3), b1(3), c1(3), a2(3), b2(3), c2(3), ab1(3), ab2(3);

	// fill c-alpha positions
	for ( int i = 1; i <= 3; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			ca1(k,i) = misc::Eposition(k,2,pos1+i-2);
			ca2(k,i) = misc::Eposition(k,2,pos2+i-2);
		}
	}

	get_CA_vectors( ca1,ca2,a1,b1,c1);
	get_CA_vectors( ca2,ca1,a2,b2,c2);

	orientation = dotprod( a1,a2);

	cros(a1,b1,ab1);
	cros(a2,b2,ab2);

	float const d1 = dotprod(ab1,c1);
	float const d2 = dotprod(ab2,c2);

	pleating1 = d1;

	if ( orientation < 0 ) {
		pleating2 =  d2; // antiparallel
	} else {
		pleating2 = -d2;
	}

}

///////////////////////////////////////////////////////////////////////////////
bool
same_side(
	int const s1_pos1,
	int const s2_pos,
	int const s1_pos2,
	int const s3_pos
)
{
	{ // debug
		for ( int i=std::min(s1_pos1,s1_pos2), i_end=std::max(s1_pos1,s1_pos2);
					i<= i_end; ++i ) {
			if ( misc::secstruct(i) != 'E' ) {
				std::cout <<"WARNING:: same_side(): non-E residue between s1_pos1 " <<
					"and s1_pos2: " << i << std::endl;
			}
		}
	}

	int const sep( s1_pos1 - s1_pos2 );
	int const p1( get_pleating( s1_pos1, s2_pos ) );
	int const p2( get_pleating( s1_pos2, s3_pos ) );
	return ( sep%2 == 0 && p1 == p2 ||
					 sep%2 == 1 && p1 != p2 );
}

///////////////////////////////////////////////////////////////////////////////
int
get_pleating(
	int const pos1,
	int const pos2
)
{

	//Why did this have to get so complicated?
	// Its actually a pretty simple concept!
	//
	// For some reasons, get_pairing_geometry flips
	// pleating2 depending on the orientation --
	// in ideal strand pairs, pleating1 and pleating2 then have the same sign.
	//
	// But for some twisted strand pairs (see, e.g, 22,48 in 1brs.pdb),
	// the numbers get a little crazy...
	//

	float orientation, pleating1, pleating2;
	if ( pos1 < pos2 ) {
		get_pairing_geometry( pos1, pos2, orientation,pleating1,pleating2);

		//This isn't always quite true...
		//		assert( pleating1 * pleating2 > 0.0 );
		return ( (pleating1+pleating2) < 0 ? 1 : 2 );
	} else {
		get_pairing_geometry( pos2, pos1, orientation,pleating1,pleating2);


		//This isn't always quite true...
		//		assert( pleating1 * pleating2 > 0.0 );

		if ( orientation < 0 ) {
			// pleating for anti-parallel pairings is preserved when we
			// interchange positions
			return ( (pleating1+pleating2) < 0 ? 1 : 2 );
		} else {
			// pleating for parallel pairings is reversed when we
			// interchange positions
			return ( (pleating1+pleating2) < 0 ? 2 : 1 );
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin local_max
///
/// @brief is the frequency at i,j,o a local max of the pairing distribution?
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
local_max(
	int const i,
	int const j,
	int const o,
	FArray2D_float const & freq
)
{
	assert( i < j && ( o == 1 || o == 2 ) );

	float const f( (o==1) ? freq(i,j) : freq(j,i) );

	bool is_max = true;

	for ( int ii = i-1; ii <= i+1; ++ii ) {
		if ( !is_max ) break;
		for ( int jj = j-1; jj <= j+1; ++jj ) {
			if ( jj > ii && ii >= 1 && jj <= misc::total_residue ) {
				float const ff( (o==1) ? freq(ii,jj) : freq(jj,ii) );
				if ( ff > f || ( ff==f && ( ii>i || ( ii==i && jj>j ) ) ) ) {
					is_max = false;
					break;
				}
			}
		}
	}
	return is_max;
}


///////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin get_closest_feature
///
/// @brief get the closest feature in feature_list to a given pairing
///
/// @detailed returns feature_list.end() if no match found
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
Feature_list::const_iterator
get_closest_feature(
	Beta_feature const & f,
	std::vector< Beta_feature > const & feature_list,
	int const big_threshold,
	int const small_threshold
)
{
	int const a( f.long_axis() );
	int const b( f.short_axis() );
	int const o( f.orientation );

	// signal failure by returning feature_list.end()

	std::pair< float, Feature_list::const_iterator > best_match ( 1000.0, feature_list.end() );

	for ( Feature_list::const_iterator it = feature_list.begin(),
	 it_end = feature_list.end(); it!= it_end; ++it ) {
		if ( it->orientation == o ) {

			int const adev ( std::abs( a- it-> long_axis() ));
			int const bdev ( std::abs( b- it->short_axis() ));

			int const dev ( small_threshold * adev + big_threshold * bdev );

			if ( adev <= big_threshold && bdev <= small_threshold &&
					 dev < best_match.first ) {
				best_match.first = dev;
				best_match.second = it;
			}
		}
	}
	return best_match.second;
}

///////////////////////////////////////////////////////////////////////////////
// ss-scores all the decoys
//
// identifies features
//
// classifies features by features of the intervening loops
//
void
ss_analyze_silent_data(
	silent_io::Silent_file_data const & data
)
{
	using namespace pose_ns;
	using namespace silent_io;
	using namespace strand_pairings_ns;


	// params
	int const big_threshold( 6 );
	int const small_threshold( 4 );
	float const max_feature_score( -1.0 ); // guess
	float const strand_extension_min_frequency( 0.4 );
	float const barcode_kill_weight( 4.0 );
	float const barcode_kill_frequency( 0.5 );
	float const min_frequency_for_killing( 0.05 );
	int const min_resample_features(3);
	float const min_resample_frequency(0.01);
	int const max_resample_pairings(2000);

	bool const make_barcode_file( truefalseoption("barcode_file") );
	bool const kill_hairpins( truefalseoption("kill_hairpins") );
	bool const kill_bab( truefalseoption("kill_bab") );
	bool const exclude_hairpin_clips( true );
	bool const make_pairing_file( truefalseoption("pairing_file") );

	std::string const resample_mode( make_pairing_file ?
		stringafteroption("resample_mode") : "nil"  );
	//bool const sandwich( false );
	//bool const alpha_plus_beta( true );

	// output filestream
	std::ofstream out;

	// silent-data
	int const nres( data.nres() );
	int const total_decoys( data.size() );

	// local
	float tmp_ss_score, tmp_rsigma_score; // reuse these guys
	int const A(1), P(2);

	// in global namespace
	decoy_pairing_count.dimension( nres, nres);
	decoy_pairing_score.dimension( nres, nres);

	// tell sspair score to store residue pairings:
	set_store_ss_scores( true );

	//////////////////////
	// get native pairings
	Feature_list native_pairings;
	Pose native_pose;
	bool const native_exists( truefalseoption("n") );
	FArray1D_int native_mapping; // goes from native seq to decoy seq
	int native_mapping_count(0);
	if ( native_exists ) {
		pose_from_pdb( native_pose, stringafteroption("n"), false, false );
		int const nres_nat( native_pose.total_residue() );

		//
		native_mapping.dimension( nres_nat );
		if ( native_pose.sequence() != data.sequence() ) {
			std::cout << "native sequence mismatch: looking for align_file" <<
				std::endl;
			bool const fail( mapping_from_file( stringafteroption("align" ),
				native_pose.sequence(), data.sequence(), native_mapping ));
			if ( fail ) {
				std::cout << "STOP: unable to map native sequence." << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		} else {
			for ( int i=1; i<= nres_nat; ++i ) {
				native_mapping(i) = i;
			}
		}

		native_mapping_count=0;
		for ( int i=1; i<= nres_nat; ++i ) {
			if ( native_mapping(i) >= 1 ) ++native_mapping_count;
		}

		// zero out the global ss arrays
		decoy_pairing_count = 0;
		decoy_pairing_score = 0;

		// sspair score this decoy:
		score_set_current_pose( native_pose );
		structure::BOUNDARY::identify_ss();
		structure::BOUNDARY::SSpair_score( 0, 6 , tmp_ss_score, tmp_rsigma_score ); // lowstrand,cutoff
		score_reset_current_pose();

		for ( int i = 1; i<= nres_nat; ++i ) {
			for ( int j = 1; j<= nres_nat; ++j ) {
				if ( decoy_pairing_count(i,j) > 0 ) {
					int const ii( native_mapping(i) );
					int const jj( native_mapping(j) );
					if ( ii<1 || jj <1 ) continue;
					if ( i<j ) {
						native_pairings.push_back
							( Beta_feature(ii,jj,A));
					} else {
						native_pairings.push_back
							( Beta_feature(jj,ii,P));
					}
				}
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// score all the decoys once to identify beta-pairing features:
	//
	// initialize counts over the set of decoys:
	FArray2D_int pairing_count( nres, nres, 0 );
	FArray1D_int paired_count( nres, 0 );
	FArray1D_int ss_E_count( nres, 0 );


	Pose pose;

	int decoy_counter(0);
	for ( Silent_file_data::const_iterator decoy_iter = data.begin(),
					decoy_end = data.end(); decoy_iter != decoy_end; ++decoy_iter ) {
		std::string const & decoy_tag( decoy_iter->first );
		if ( (++decoy_counter)%500 == 0 ) 	std::cout << "score: " <<
																					decoy_counter << std::endl;
		decoy_iter->second->fill_pose( pose );

		// zero out the global ss arrays
		decoy_pairing_count = 0;
		decoy_pairing_score = 0;

		// sspair score this decoy:
		score_set_current_pose( pose );
		structure::BOUNDARY::identify_ss();
		structure::BOUNDARY::SSpair_score( 0, 6 , tmp_ss_score, tmp_rsigma_score ); // lowstrand,cutoff
		score_reset_current_pose();

		// increment the counters:
		// pairing counter:
		FArray1D_bool paired( nres, false );
		for ( int i = 1; i<= nres; ++i ) {
			if ( pose.secstruct(i) == 'E' ) ++ss_E_count(i);
			for ( int j =1; j<= nres; ++j ) {
				if ( decoy_pairing_count(i,j) > 0 ) {
					++pairing_count(i,j);
					if ( !paired(i) ) {
						paired(i) = true;
						++paired_count(i);
					}
					if ( !paired(j) ) {
						paired(j) = true;
						++paired_count(j);
					}
				}
			}
		}

		if ( native_exists ) {
			// compare to native
			float const rmsd( CA_rmsd_by_mapping( native_pose, pose,native_mapping));
			int const nali( CA_maxsub_by_mapping( native_pose, pose,native_mapping));

			// count native pairings
			int native_pairing_count(0);
			for ( Feature_list::const_iterator it= native_pairings.begin(),
							it_end= native_pairings.end(); it != it_end; ++it ) {
				assert( it->pos1 < it->pos2 );
				if ( it->antiparallel() &&
						 decoy_pairing_count( it->pos1, it->pos2 ) > 0 ) {
					++native_pairing_count;
				} else if ( it->parallel() &&
										decoy_pairing_count( it->pos2, it->pos1 ) > 0 ) {
					++native_pairing_count;
				}
			}
			std::cout << "SCORE: " <<
				native_pairing_count << ' ' <<
				decoy_iter->second->total_score << ' ' <<
				rmsd << ' ' <<
				nali << ' ' <<
				native_mapping_count << ' ' <<
				decoy_tag << std::endl;
		}
	} // loop over decoys

	////////////////////////////////////////////////////////////////
	// compare frequency of a residue being paired with the fraction
	// of E fragments
	if ( truefalseoption("frag_name" ) ) {
		// read standard rosetta fragment file
		read_fragments_simple( stringafteroption("frag_name"), nres );
		FArray1D_float frag_E( calc_frag_E() );

		for ( int i=1; i<= nres; ++i ) {
			float const decoy_E
				( static_cast< float >( paired_count(i) ) / total_decoys );
			float const decoy_ss_E
				( static_cast< float >( ss_E_count(i) ) / total_decoys );
			float ratio(0.0);
			if ( std::abs( frag_E(i) ) > 0.001 ) {
				ratio = decoy_E / frag_E(i);
			}
			std::cout << "FRAG_E " << F(6,3,frag_E(i)) << F(6,3,decoy_E) <<
				F(6,3,decoy_ss_E) << F(6,3,ratio ) << std::endl;
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	// identify the features
	Feature_list feature_list;
	identify_beta_features( nres, pairing_count, decoy_counter,	feature_list );


	/////////////////////////////////////////////////////////////////////////////
	// re-score the decoys to identify topologies, get overall feature
	// frequencies
	//
	// also want to classify features, eg as hairpins, beta-alpha-betas,
	// i->i+3
	// so keep counters for the deltas between paired strands:
	//
	// number of E,H,L residues
	// number of helices- not right now, actually
	// number of Rosetta strands
	//

	// to do: identify approximate residue ranges for pairings,
	// in order to make the barcode topology file

	std::map< Beta_feature, int > match_count, E_count, H_count, L_count,
		strand_count, helix_count;

	std::map< Beta_feature, std::map< Silly_pairing_type, int > > big_p_map;

	std::map< Beta_feature, FArray1D_int > ss_count;

	std::map< Beta_feature, std::map< Beta_feature, int > > big_pairing_map;

	decoy_counter = 0;
	for ( Silent_file_data::const_iterator decoy_iter = data.begin(),
					decoy_end = data.end(); decoy_iter != decoy_end; ++decoy_iter ) {
		std::string const & decoy_tag( decoy_iter->first );
		if ( (++decoy_counter)%500 == 0 ) 	std::cout << "rescore: " <<
																					decoy_counter << std::endl;
		decoy_iter->second->fill_pose( pose );

		decoy_pairing_count = 0;
		decoy_pairing_score = 0;
		score_set_current_pose( pose );
		structure::BOUNDARY::identify_ss();
		structure::BOUNDARY::SSpair_score( 0, 6 , tmp_ss_score, tmp_rsigma_score ); // lowstrand,cutoff
		score_reset_current_pose();

		// get strands:
		FArray1D_char const & ss( pose.secstruct() );
		FArray1D_int strand_num( nres, 0 );
		{
			int strand(0);
			bool in_strand( false );
			for ( int i=1; i<= nres; ++i ) {
				if ( ss(i) == 'E' ) {
					if ( !in_strand ) {
						in_strand = true;
						++strand;
					}
					strand_num(i) = strand;
				} else {
					in_strand = false;
				}
			}
		}

		// assign features:
		std::map< Beta_feature, Feature_list > pairing_map;
		std::map< Beta_feature, float > feature_score;


		// identify matched features: sum the sspair score for each
		for ( int i = 1; i <= nres; ++i ) {
			for ( int j = 1; j <= nres; ++j ) {
				if ( decoy_pairing_count(i,j) > 0 ) {
					int const pos1 = std::min(i,j);
					int const pos2 = std::max(i,j);
					int const o ( ( i<j ) ? A : P );
					int const pleating( get_pleating( pos1,pos2 ) );
					Beta_feature const f( pos1, pos2, o, pleating );
					Feature_list::const_iterator it
						( get_closest_feature( f, feature_list, big_threshold,
																	 small_threshold ) );
					if ( it != feature_list.end() ) {
						// found a match
						pairing_map[ *it ].push_back( f );
						feature_score[ *it ] += decoy_pairing_score( i, j );
					} // matches a feature
				} // has pairing
			} // j
		} // i


		// find matched features: sspair score > threshold
		//
		for ( std::map< Beta_feature, float >::const_iterator
						it=feature_score.begin(), it_end = feature_score.end();
					it != it_end; ++it ) {
			if ( it->second <= max_feature_score ) {
				// feature match:
				Beta_feature const & f( it->first );

				bool const f_parallel( f.parallel() );

				++match_count[f]; // total decoys matching f

				if ( match_count[f] == 1 ) {
					// first decoy match to f
					FArray1D_int count( nres, 0 );
					ss_count.insert( std::make_pair( f, count ) );
					big_pairing_map[f]; // create entry
					//std::map< Beta_feature, int > tmp_map;
					//big_pairing_map.insert( f, tmp_map ); // so we can call find below
				}
				assert( ss_count.count(f) );
				FArray1D_int & ss_count_f( ss_count.find(f)->second );
				for ( int i=1; i<= nres; ++i ) if ( ss(i) == 'E' ) ++ss_count_f(i);

				Feature_list l( pairing_map.find(f)->second );
				std::sort( l.begin(), l.end() );
				int const l_size( l.size() );
				if ( l_size > 1 ) assert( l[0].pos1 <= l[1].pos1 );

				Beta_feature central_pairing( l[ l_size/2 ] );

				// what are the bounds of the paired strands
				int min_pos1(1000), min_pos2(1000), max_pos1(0), max_pos2(0);
				std::map< Beta_feature, int > & big_pairing_map_f
					( big_pairing_map.find(f)->second );
				for ( int i=0; i< l_size; ++i ) {
					Beta_feature const & p(l[i]);
					min_pos1 = std::min( min_pos1, p.pos1 );
					max_pos1 = std::max( max_pos1, p.pos1 );
					min_pos2 = std::min( min_pos2, p.pos2 );
					max_pos2 = std::max( max_pos2, p.pos2 );
					//++(big_pairing_map[f][ p ]);
					++big_pairing_map_f[p];
				}
				assert( max_pos1 < min_pos2 );

				// analyze the loop between the two strands:
				int pos1( max_pos1 );
				int pos2( min_pos2 );
				assert( ss(pos1) == 'E' && ss(pos2) == 'E' );

				// how many strands separate pos1 from pos2?
				int const strand1( strand_num( pos1 ) );
				int const strand2( strand_num( pos2 ) );
				int const strand_sep( strand2 - strand1 - 1 );
				strand_count[ f ] += strand_sep;

				// identify the loop separating these two strands
				while ( pos1 < pos2 && ss(pos1) == 'E' ) ++pos1;
				while ( pos1 < pos2 && ss(pos2) == 'E' ) --pos2;
				if ( ss(pos1) == 'E' ) {
					std::cout << "Funny pairing: " << f << ' ' << max_pos1 << ' ' <<
						min_pos2 << ' ' << pos1 << ' ' << pos2 << std::endl;
				} else {
					int H(0),E(0),L(0);
					std::map< Silly_pairing_type, Feature_list > p_map;
					bool in_helix( false );
					int helices(0);
					for ( int i= pos1; i<= pos2; ++i ) {
						if ( ss(i) != 'H' ) in_helix = false;
						if ( ss(i) == 'L') ++L;
						else if ( ss(i) == 'H' ) {
							++H;
							if (!in_helix ) {
								in_helix = true;
								++helices;
							}
						} else if ( ss(i) == 'E' ) {
							++E;

							int const strand3 = strand_num(i);
							assert( strand3 > strand1 && strand3 < strand2 );

							// analyze the partners of this strand:
							for ( int j=1; j<= nres; ++j ) {
								if ( decoy_pairing_count(i,j) ||
										 decoy_pairing_count(j,i) ) {
									bool const parallel
										( ( i<j && decoy_pairing_count(j,i) ) ||
											( i>j && decoy_pairing_count(i,j) ) );
									int const strand4 = strand_num(j);

									Beta_feature new_p( std::min(i,j), std::max(i,j),
										( parallel ? P : A ));


									if ( strand4 < strand1 ) {
										if ( !f_parallel && !parallel ) {
											// interleaved
											p_map[IL_P].push_back( new_p );
											// cross-beta ?
											if ( strand4 == strand1 -1 && strand3 == strand2-1) {
												p_map[CB_P].push_back( new_p );
											}
										}
									} else if ( strand4 == strand1 ) {
										int const s1_pos1 = central_pairing.pos1; // strand1
										int const s2_pos = central_pairing.pos2; // strand2
										int const s1_pos2 = j; // strand1
										int const s3_pos = i; // strand3
										if ( same_side( s1_pos1, s2_pos, s1_pos2, s3_pos ) ) {
											p_map[BAD_P].push_back( new_p );
										} else {
											p_map[GOOD_P].push_back( new_p );
										}
									} else if ( strand4 < strand2 ) {
										p_map[INT_P].push_back( new_p );
									} else if ( strand4 == strand2 ) {
										int const s1_pos1 = central_pairing.pos2; // strand2
										int const s2_pos = central_pairing.pos1; // strand1
										int const s1_pos2 = j; // strand2
										int const s3_pos = i; // strand3
										if ( same_side( s1_pos1, s2_pos, s1_pos2, s3_pos ) ) {
											p_map[BAD_P].push_back( new_p );
										} else {
											p_map[GOOD_P].push_back( new_p );
										}
									} else {
										assert ( strand4 > strand2 );
										if ( !f_parallel && !parallel ) {
											// interleaved
											p_map[IL_P].push_back( new_p );
											// cross-beta ?
											if ( strand3 == strand1+1 && strand4 == strand2+1 ) {
												p_map[CB_P].push_back( new_p );
											}
										}
									}
								}
							} // j -- possible partner of E residue in the loop
						} else {
							std::cout << "Non EHL ss: " << ss(i) << std::endl;
							utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
						}
					} // i = pos1...pos2

					// increment counters that describe the intervening loops
					H_count[f]+=H;
					E_count[f]+=E;
					L_count[f]+=L;

					// now that we've figured out how many helices are in the loop:
					helix_count[f]+=helices;


					// Figure out how many topologically uniqe pairings we have in each
					// of the pairing_type classes:
					for ( std::map< Silly_pairing_type, Feature_list >::const_iterator
									it=p_map.begin(), it_end=p_map.end(); it != it_end; ++it ) {
						Silly_pairing_type const p_type( it->first );

						// how many topolog uniqe pairings?
						Feature_list const & ll( it->second );
						int const ll_size( ll.size() );
						int n_strand_pairs( 0 );
						for ( int i=0; i< ll_size; ++i ) {
							bool new_strand_pair( true );
							for ( int j=0; j<i; ++j ) {
								if ( ( strand_num( ll[i].pos1 ) == strand_num( ll[j].pos1 ) &&
											 strand_num( ll[i].pos2 ) == strand_num( ll[j].pos2 ) )||
										 ( strand_num( ll[i].pos2 ) == strand_num( ll[j].pos1 ) &&
											 strand_num( ll[i].pos1 ) == strand_num( ll[j].pos2 ) )){
									new_strand_pair = false;
									//std::cout <<"Same: " << ll[i] << ' ' << ll[j] << std::endl;
									break;
								}
							}
							if ( new_strand_pair ) {
								//std::cout << "new nsp: " << decoy_tag << ' ' << f << ' ' <<
								//	ll[i] << ' ' << p_type << std::endl;
								++n_strand_pairs;
							}
						}
						big_p_map[f][ p_type ] += n_strand_pairs;
						if ( false && n_strand_pairs > 0 ) {
							std::cout << "nsp: " << decoy_tag << ' ' << f << ' ' <<
								p_type << ' ' << n_strand_pairs << std::endl;
						}
					}
				} // funny pairing??
			} else {
				//std::cout << "skip feature: score= " << it->second << " > " <<
				//	max_feature_score << std::endl;

			} // check if the sspair score is good enough for this feature

		} // loop over the map from beta_features to total sspair score

	} // loop over silent-file decoys


	/////////////////////////////////////////////////////////////////////////////
	// classify features:
	if ( make_barcode_file ) {
		out.open( stringafteroption("barcode_file").c_str() );
	}

	std::vector< std::pair< float, Beta_feature > > resample_list;
	for ( std::map< Beta_feature, int >::const_iterator it=match_count.begin(),
					it_end = match_count.end(); it != it_end; ++it ) {
		Beta_feature f( it->first );
		int const count( it->second );
		float const freq( static_cast< float >( count )/ total_decoys );
		float const avg_E( static_cast< float >( E_count[f] ) / count );
		float const avg_H( static_cast< float >( H_count[f] ) / count );
		float const avg_L( static_cast< float >( L_count[f] ) / count );
		float const sep( avg_E + avg_H + avg_L );
		float const avg_strands( static_cast< float >( strand_count[f] ) / count );
		float const avg_helices( static_cast< float >( helix_count[f] ) / count );

		float const avg_int_p ( static_cast<float>( big_p_map[f][ INT_P ])/count );
		float const avg_good_p( static_cast<float>( big_p_map[f][ GOOD_P])/count );
		float const avg_bad_p ( static_cast<float>( big_p_map[f][ BAD_P ])/count );
		float const avg_il_p  ( static_cast<float>( big_p_map[f][ IL_P  ])/count );
		float const avg_cb_p  ( static_cast<float>( big_p_map[f][ CB_P  ])/count );


		// look at strand lengths
		int strand1_begin, strand1_end, strand2_begin, strand2_end;
		{
			FArray1D_int const & ss_count_f( ss_count[f] );
			FArray1D_float E_frac( nres );
			for ( int i=1; i<= nres; ++i ) {
				E_frac(i) = float( ss_count_f( i ) ) / count;
			}

			int pos1( f.pos1 );
			if ( f.pos1 < nres && E_frac( f.pos1 + 1 ) > E_frac( pos1 ) )
				pos1 = f.pos1 + 1;
			if ( f.pos1 >    1 && E_frac( f.pos1 - 1 ) > E_frac( pos1 ) )
				pos1 = f.pos1 - 1;

			int pos2( f.pos2 );
			if ( f.pos2 < nres && E_frac( f.pos2 + 1 ) > E_frac( pos2 ) )
				pos2 = f.pos2 + 1;
			if ( f.pos2 >    1 && E_frac( f.pos2 - 1 ) > E_frac( pos2 ) )
				pos2 = f.pos2 - 1;


			//for ( int m=2; m<= 8; ++m ) {
			//	float const min_freq( float(m)/10.0 );
			float const min_freq( strand_extension_min_frequency );

			// boundaries of first strand
			strand1_begin = pos1 ;
			strand1_end = pos1;
			while ( strand1_begin > 0 && E_frac( strand1_begin ) > min_freq )
				--strand1_begin;
			++strand1_begin;
			while ( strand1_end <= nres && E_frac( strand1_end ) > min_freq )
				++strand1_end;
			--strand1_end;

			// boundaries of second strand
			strand2_begin = pos2;
			strand2_end = pos2;
			while ( strand2_begin > 0 && E_frac( strand2_begin ) > min_freq )
				--strand2_begin;
			++strand2_begin;
			while ( strand2_end <= nres && E_frac( strand2_end ) > min_freq )
				++strand2_end;
			--strand2_end;

			if ( false ) {
			std::cout << "f strand bounds: " << f << ' ' << min_freq << ' ' <<
				strand1_begin << ' ' << strand1_end << ' ' <<
				strand1_end - strand1_begin + 1 << ' ' <<
				strand2_begin << ' ' << strand2_end << ' ' <<
				strand2_end - strand2_begin + 1 << ' ' <<
				E_frac( pos1 ) << ' ' << ss_count_f( pos1 ) << ' ' <<
				E_frac( pos2 ) << ' ' << ss_count_f( pos2 ) << std::endl;
			}
			//}
		}

		// calc native distance
		int nat_dist(1000);
		for ( Feature_list::const_iterator it= native_pairings.begin(),
						it_end = native_pairings.end(); it != it_end; ++it ) {
			nat_dist = std::min( nat_dist, distance( *it, f ) );
		}

		// classify:
		bool const no_intervening_strand( avg_E < 1.2 || avg_strands < 0.5 );
		bool const parallel( f.parallel() );

		std::string tag;
		if ( sep < 20 ) {
			tag = ( parallel ? "SR-P" : "SR-A" );
		} else {
			tag = ( parallel ? "LR-P" : "LR-A" );
		}

		if ( !parallel && sep <= 21 && no_intervening_strand ) {
		//if ( !parallel && sep <= 21 && no_intervening_strand && avg_H <= 5.0 ) {
			// HAIRPIN
			tag = "HP";
		} else if ( parallel && sep <= 40 && no_intervening_strand &&
								avg_H >= 1.0 ) {
			// BAB
			tag = "BAB";
		} else if ( parallel && sep <= 20 && avg_H <= 1.0 && avg_strands < 1.3 &&
								( avg_E >= 1.0 || avg_strands >= 0.5 ) ) {
			// PAPER CLIP
			tag = "CLIP";
		} else if ( !parallel && avg_strands > 1.5 && avg_strands < 3 ) {
			// i,i+3 ??
			tag = "i->i+3";
		}


		std::cout << "FEATURE: " << f <<
			" freq= " <<  F(6,3,freq) <<
			" E= " << F(6,2,avg_E) <<
			" H= " << F(6,2,avg_H) <<
			" L= " << F(6,2,avg_L) <<
			" str= " << F(6,2,avg_strands) <<
			" hel= " << F(6,2,avg_helices) <<
			" p: " <<
			F(6,2,avg_int_p) <<
			F(6,2,avg_good_p) <<
			F(6,2,avg_bad_p) <<
			F(6,2,avg_il_p) <<
			F(6,2,avg_cb_p) <<
			" sep= " << F(6,2,sep) <<
			" nat= " << I(5,nat_dist) << ' ' <<
			tag << std::endl;

		//////////////////////
		// add to pairing file
		if ( make_pairing_file ) {

			// first try at simple logic for beta-sandwich proteins
			if ( resample_mode == "sandwich" ) {
				if ( !f.parallel() && sep >= 20 &&
						 sep <= 45 && tag != "HP" ) {
					std::cout << "resample: " << f << '\n';
					resample_list.push_back( std::make_pair( freq, f ) );
				}
			} else if ( resample_mode == "alpha_plus_beta" ) {
				if ( !f.parallel() && sep >= 20 && tag != "HP" ) {
					std::cout << "resample: " << f << '\n';
					resample_list.push_back( std::make_pair( freq, f ) );
				}
			} else if ( resample_mode == "all" ) {
				std::cout << "resample: " << f << '\n';
				resample_list.push_back( std::make_pair( freq, f ) );
			} else if ( resample_mode == "rhiju" ) {
				if ( sep >= 20 && tag != "HP" && tag != "BAB" ) {
					std::cout << "resample: " << f << '\n';
					resample_list.push_back( std::make_pair( freq, f ) );
				}
			}	else if ( resample_mode == "long_range_antiparallel" ) {
				if ( !f.parallel() && sep >= 20 &&
						 tag != "HP" ) {
					std::cout << "resample: " << f << '\n';
					resample_list.push_back( std::make_pair( freq, f ) );
				}
			}
		}


		//////////////////////
		// add to barcode file
		if ( make_barcode_file ) {
			char const o( f.parallel() ? 'P' : 'A' );
			if ( kill_hairpins && tag == "HP" && freq >= min_frequency_for_killing ) {
				// this depends on there being no whitespace in the stream inserter
				// for Beta_features
				out << f << '(' << nat_dist << ") " << barcode_kill_frequency <<
					" SSPAIR " <<
					barcode_kill_weight << ' ' << o << ' ' << strand1_begin << ' ' <<
					strand1_end << ' ' << strand2_begin << ' ' << strand2_end;

				if ( exclude_hairpin_clips ) {
					// no parallel pairings involving the first strand, and within 20
					// rsds of the end of the second strand
					int const end( std::min( nres, strand2_end + 20 ) );
					out << " SSPAIR " <<
						barcode_kill_weight << " P " << strand1_begin << ' ' <<
						strand1_end << ' ' << strand1_begin << ' ' << end;

					// no parallel pairings involving the second strand, and within 20
					// rsds of the beginning of the first strand
					int const begin( std::max( 1, strand1_begin - 20 ) );
					out << " SSPAIR " <<
						barcode_kill_weight << " P " << strand2_begin << ' ' <<
						strand2_end << ' ' << begin << ' ' << strand2_end << '\n';

				} else out << '\n';
			}
			if ( kill_bab && tag == "BAB" && freq >= min_frequency_for_killing ) {
				// this depends on there being no whitespace in the stream inserter
				// for Beta_features
				out << f << '(' << nat_dist << ") " << barcode_kill_frequency <<
					" SSPAIR " <<
					barcode_kill_weight << ' ' << o << ' ' << strand1_begin << ' ' <<
					strand1_end << ' ' << strand2_begin << ' ' << strand2_end << '\n';
			}
		}
	}

	if ( make_barcode_file ) {
		out.close();
	}

	////////////////////
	// make pairing file
	if ( make_pairing_file ) {
		std::sort( resample_list.begin(), resample_list.end() );
		std::reverse( resample_list.begin(), resample_list.end() );

		// loop through the features, cull a subset for resampling
		// sum the total frequency for this subset
		float freq_sum(0.0);
		for ( int i=0, i_end = resample_list.size(); i< i_end; ++i ) {
			float const freq( resample_list[i].first );
			Beta_feature const & f( resample_list[i].second );
			if ( i >= min_resample_features &&
					 freq < min_resample_frequency ) break;
			freq_sum += freq;
			std::cout << "resample: " << i << ' ' << f << ' ' << freq <<
				' ' << freq_sum << std::endl;
		}

		// generate the pairing file
		out.open( stringafteroption("pairing_file").c_str() );
		for ( int ii=0, ii_end = resample_list.size(); ii< ii_end; ++ii ) {
			float const freq( resample_list[ii].first );
			Beta_feature const & f( resample_list[ii].second );

			if ( ii >= min_resample_features &&
					 freq < min_resample_frequency ) break;

			int const npairs( static_cast< int >(
				max_resample_pairings * ( freq /freq_sum)));

			std::map< Beta_feature, int > const & pairing_map
				( big_pairing_map.find( f )->second );
			Feature_list pairing_list;
			while( int(pairing_list.size()) < npairs ) {
				for ( std::map< Beta_feature, int >::const_iterator
								it=pairing_map.begin(), it_end=pairing_map.end();
							it!= it_end; ++it ) {
					Beta_feature const & pairing( it->first );
					int const count( it->second );
					for ( int j=0; j<count; ++j ) pairing_list.push_back( pairing );
				}
				std::cout << f << " allowed pairs: " << npairs << ' ' <<
					pairing_list.size() << std::endl;
			}

			std::random_shuffle( pairing_list.begin(), pairing_list.end() );

			for ( int j=0; j<npairs; ++j ) {
				Beta_feature const & p( pairing_list[j] );
				assert( p.pleating == 1 || p.pleating == 2);
				out << p.pos1 << ' ' << p.pos2 << ' ' << p.orientation << ' ' <<
					p.pleating << '\n';
// 				out << p.pos1 << ' ' << p.pos2 << ' ' << p.o() << ' ' <<
// 					p.pleating << '\n';
			}
		} // loop over features
		out.close();

	} // if ( make_pairing_file )

	set_store_ss_scores( false );
}



///////////////////////////////////////////////////////////////////////////////
void
main_test_interleaved()
{
	pose_ns::Pose pose;

	// decoy test, list of outfiles
	if ( true ) {
		std::ifstream data( stringafteroption("l").c_str() );
		assert( data.good() );
		std::string filename;

		int const ndecoys( intafteroption("ndecoys") );

		while ( getline( data, filename ) ) {

			silent_io::Silent_file_data data( filename );

			int decoy_counter(0);
			for ( silent_io::Silent_file_data::const_iterator it = data.begin(),
							it_end = data.end(); it != it_end; ++it) {
				if ( (++decoy_counter)%50 == 0 ) 	std::cout << "score: " <<
																						decoy_counter << std::endl;
				it->second->fill_pose( pose );
				find_interleaved_pairings( pose, it->first, false, false );
				//find_interleaved_pairings( pose, it->first, true, false );

				if ( decoy_counter >= ndecoys ) break;
			}
		}
		data.close();
	}


	// decoy test
	if ( false ) {
		silent_io::Silent_file_data data( stringafteroption("s") );

		int decoy_counter(0);
		for ( silent_io::Silent_file_data::const_iterator it = data.begin(),
						it_end = data.end(); it != it_end; ++it) {
			if ( (++decoy_counter)%50 == 0 ) 	std::cout << "score: " <<
																					decoy_counter << std::endl;
			it->second->fill_pose( pose );
			find_interleaved_pairings( pose, it->first, true );
		}
	}


	// native test
	if ( false ) {
		std::ifstream data( stringafteroption("l").c_str() );
		std::string line;
		while ( getline( data, line ) ) {
			bool ok = pose_from_pdb( pose, line, false, false );
			if ( !ok ) continue;
			find_interleaved_pairings( pose, line, false );
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
void
find_interleaved_pairings(
	pose_ns::Pose & pose,
	std::string const & tag,
	bool const update_ss,
	bool const verbose // = true
)
{
	std::cout << "start: " << tag << std::endl;
	using namespace pose_ns;
	using namespace strand_pairings_ns;

	int const nres( pose.total_residue() );

	// score to get ss pairings
	decoy_pairing_count.dimension(nres,nres);
	decoy_pairing_score.dimension(nres,nres);
	decoy_pairing_count = 0;
	decoy_pairing_score = 0;
	score_set_current_pose( pose );
	structure::BOUNDARY::identify_ss();
	float tmp_ss_score, tmp_rsigma_score;
	set_store_ss_scores( true );
	structure::BOUNDARY::SSpair_score( 0, 4 , tmp_ss_score, tmp_rsigma_score ); // lowstrand,cutoff
	set_store_ss_scores( false );
	score_reset_current_pose();

	// recalculate to define new secondary structure
	// E residues are defined as residues with a pairing
	//
	// also define strand numbering system
	std::map< Beta_feature, Feature_list > feature_map;
	FArray1D_char ss( pose.secstruct() ); // make a copy
	FArray1D_int old_strand_num( nres, 0 ); // strand nbrs wrt initial ss
	FArray1D_int old_strand( nres, 0 ); // mapping to old strand numbers
	FArray1D_int strand_num( nres, 0 ); // strand nbrs wrt new ss

	std::ofstream out;
	if ( verbose ) out.open( (tag+"_pairs.pdb").c_str() );
	else out.open("/dev/null");
	out << "load pdb inline\n";
	{
		// old numbers
		int strand(0);
		bool in_strand( false );
		for ( int i=1; i<= nres; ++i ) {
			if ( ss(i) == 'E' ) {
				if ( !in_strand ) {
					in_strand = true;
					++strand;
				}
				old_strand_num(i) = strand;
			} else {
				in_strand = false;
			}
		}

		// new numbers
		strand = 0;
		in_strand = false;
		for ( int i=1; i<= nres; ++i ) {

			bool paired( update_ss ? false : ss(i) == 'E' );
			if ( paired ) {
				if ( !in_strand ) {
					in_strand = true;
					++strand;
				}
				strand_num(i) = strand;
			}

			for ( int j=1; j<= nres; ++j ) {
				if ( decoy_pairing_count(i,j) || decoy_pairing_count(j,i) ) {
					paired = true;
					if ( !in_strand ) {
						assert( update_ss );
						in_strand = true;
						++strand;
					}
					strand_num(i) = strand;
					if ( old_strand( strand ) ) {
						assert( old_strand( strand ) == old_strand_num( i ) );
					}
					old_strand( strand ) = old_strand_num(i);
					if ( j < i ) {
						int orientation( decoy_pairing_count(j,i) ? 1 : 2 );
						Beta_feature p( j, i, orientation );
						Beta_feature f( strand_num(j), strand_num(i), orientation );
						feature_map[f].push_back( p );

						{
							int const i_atom( (i-1) * 5 + 2 );
							int const j_atom( (j-1) * 5 + 2 );
							out << "monitor " << i_atom << ' ' << j_atom << '\n';
						}
					}
				}
			}

			if ( !paired ) {
				in_strand = false;
				if ( ss(i) == 'E' ) {
					ss(i) = 'L';
				}
			} else if ( paired ) {
				assert( ss(i) == 'E' );
			}
		}
	}



	{
		// show start and stop of ss elements
		bool in_helix = false;
		int helix = 0;
		bool in_strand = false;
		int strand = 0;
		int helix_begin(0), strand_begin(0);

		typedef std::vector< std::pair< int, int > > IP_list;
		IP_list strands, helices;
		for ( int i=1; i<= nres; ++i ) {
			if ( in_strand && ss(i) != 'E') {
				in_strand = false;
				strands.push_back( std::make_pair( strand_begin, i-1) );
			}
			if ( in_helix && ss(i) != 'H') {
				in_helix = false;
				helices.push_back( std::make_pair( helix_begin, i-1) );
			}

			if ( ss(i) == 'H' && !in_helix ) {
				in_helix = true;
				++helix;
				helix_begin = i;
			}

			if ( ss(i) == 'E' && !in_strand ) {
				in_strand = true;
				++strand;
				strand_begin = i;
			}
		} // i

		if ( in_strand ) {
			strands.push_back( std::make_pair( strand_begin, nres ) );
		}
		if ( in_helix ) {
			helices.push_back( std::make_pair( helix_begin, nres ) );
		}


		out << "select\ncolor cpk\nwireframe off\nbackbone 75\n";
		for ( int r=1; r<=2; ++r ) {
			std::string const hs( r==1 ? "STRAND" : "HELIX" );
			IP_list const & l( r==1 ? strands : helices );
			for ( int i=0, l_size=l.size(); i< l_size; ++i ) {
				int const begin( l[i].first );
				int const end  ( l[i].second );
				std::cout << hs << ' ' << tag << ' ' << i+1 << ' ' << begin <<
					' ' << end << std::endl;
				std::string color( r==1 ? "yellow" : "pink" );
				out << "select " << begin << "-" << end << "\ncolor " << color << '\n';
			}
		}
		out << "pause\n";
	}

	// compare ss
	if ( verbose ) {
	std::cout << "SS: " << tag << ' ';
	for ( int i=1; i<= nres; ++i ) {
		std::cout << (i/10)%10;
	} std::cout << std::endl;
	std::cout << "SS: " << tag << ' ';
	for ( int i=1; i<= nres; ++i ) {
		std::cout << (i%10);
	} std::cout << std::endl;
	std::cout << "SS: " << tag << ' ';
	for ( int i=1; i<= nres; ++i ) {
		std::cout << pose.secstruct(i);
	} std::cout << std::endl;
	std::cout << "SS: " << tag << ' ';
	for ( int i=1; i<= nres; ++i ) {
		std::cout << ss(i);
	} std::cout << std::endl;
	}

	// look for interleaved strand pairs
	// i-j and k-l paired with i<k<j
	//
	// have to check for extra pairings: i-k, i-l, j-k, j-l:
	//  dont want these pairings
	//
	// look for sheet packing between pairings in i-j and k-l pairs
	// also look for packing involving residue pairs on the other side
	// of the core strands
	Feature_list decoy_features;
	for ( std::map< Beta_feature, Feature_list >::const_iterator
					it = feature_map.begin(), it_end = feature_map.end();
				it != it_end; ++it ) {
		decoy_features.push_back( it->first );
	}
	int const total_npairs( decoy_features.size() );

	// show all the pairings
	std::cout << "PAIRS: " << tag;
	for ( int s1=0; s1< total_npairs; ++s1 ) {
		Beta_feature const & f1( decoy_features[s1] );
		std::cout << I(5,f1.pos1) << I(5,f1.pos2);
		if ( f1.orientation == 1 ) {
			std::cout << " A ";
		} else {
			std::cout << " P ";
		}
	}
	std::cout << std::endl;

	for ( int s1=0; s1< total_npairs; ++s1 ) {
		Beta_feature const & f1( decoy_features[s1] );
		for ( int s2=0; s2< total_npairs; ++s2 ) {
			Beta_feature const & f2( decoy_features[s2] );
			if ( f2.pos1 <= f1.pos1 ) continue;


			// strands in sequence order: 1-3 and 2-4 are paired
			int const strand1( f1.pos1 );
			int const strand2( f2.pos1 );
			int const strand3( f1.pos2 );
			int const strand4( f2.pos2 );


			//check for interleaved:
			assert( strand1 < strand3 );
			assert( strand2 < strand4 );
			if ( strand3 <= strand2 ||
					 strand3 >= strand4 ) continue;


			if ( old_strand( strand1 ) == old_strand( strand2 ) ||
					 old_strand( strand2 ) == old_strand( strand3 ) ||
					 old_strand( strand3 ) == old_strand( strand4 ) ) {
				continue;
			}

			// check for a pairing between these two features:
			bool paired( false );
			for ( int s3=0; s3< total_npairs; ++s3 ) {
				Beta_feature f3( decoy_features[s3] );
				if ( f3.pos1 == strand1 && ( f3.pos2==strand2 || f3.pos2==strand4 ) ||
						 f3.pos1 == strand2 && f3.pos2 == strand3 ||
						 f3.pos1 == strand3 && f3.pos2 == strand4 ) {
					paired = true;
					break;
				}
			}

			if ( paired ) continue; // skip!

			std::cout << "interleaved: " << tag << ' ' <<
				f1 << ' ' << f2 << std::endl;

			if ( !verbose ) continue;

			{
				/////////////////////
				// script file output

				Feature_list const & l1( feature_map[ f1 ] );
				Feature_list const & l2( feature_map[ f2 ] );
				int const n1( l1.size() );
				int const n2( l2.size() );

				static int counter(0);
				std::string ptag( "cross" + string_of(++counter));
				out << "select none\n";
				for ( int i=0; i< n1; ++i ) {
					out << "select selected or " << l1[i].pos1 << " or " <<
						l1[i].pos2 << '\n';
				}
				for ( int i=0; i< n2; ++i ) {
					out << "select selected or " << l2[i].pos1 << " or " <<
						l2[i].pos2 << '\n';
				}
				out << "define " << ptag << " selected\n";
				out << "select\nbackbone 75\nwireframe off\ncolor group\nselect not "
						<< ptag << "\ncolor cpk\npause\n";
			}

			// now look for packing between any pairings sharing at least one
			// strand with these interleaved pairings.

			std::ostringstream os;
			os << tag << ' ' << f1 << '-' << f2;
			for ( int ss1=0; ss1< total_npairs; ++ss1 ) {
				Beta_feature ff1( decoy_features[ss1] );

				if ( ff1.pos1 != strand1 && ff1.pos2 != strand1 &&
						 ff1.pos1 != strand3 && ff1.pos2 != strand3 ) continue;

				Feature_list const & ll1( feature_map[ ff1 ] );
				int const n1( ll1.size() );

				for ( int ss2=0; ss2< total_npairs; ++ss2 ) {
					Beta_feature ff2( decoy_features[ss2] );

					if ( ff2.pos1 != strand2 && ff2.pos2 != strand2 &&
							 ff2.pos1 != strand4 && ff2.pos2 != strand4 ) continue;

					Feature_list const & ll2( feature_map[ decoy_features[ss2] ] );
					int const n2( ll2.size() );

					for ( int i1=0; i1< n1; ++i1 ) {
						Beta_feature const & p1( ll1[i1] );
						for ( int i2=0; i2< n2; ++i2 ) {
							Beta_feature const & p2( ll2[i2] );
							evaluate_sheet_packing( p1, p2, pose.Eposition(), 15.0,
																			os.str() );
						}
					}
				}
			}


		} // s2: f2
	} // s1: f1


	if ( verbose ) {
		out << "exit\n";
		dump_Eposition( pose, out );
		out.close();
	}
}

///////////////////////////////////////////////////////////////////////////////
void
evaluate_sheet_packing(
	barcode_stats_ns::Beta_feature const & f1,
	barcode_stats_ns::Beta_feature const & f2,
	FArray3D_float const & Epos,
	float const MAX_DISTANCE,
	std::string const & tag
)
{
	using numeric::xyzVector_float;

	xyzVector_float n1,v1,c1,n2,v2,c2;
	get_pairing_vectors( f1, Epos, n1, v1, c1 );
	get_pairing_vectors( f2, Epos, n2, v2, c2 );

	assert( n1.is_normalized(0.01) && v2.is_normalized(0.01) );

	xyzVector_float c12( c1-c2 );
	float const distance( c12.length() );
	c12.normalize();
	if ( distance < MAX_DISTANCE ) {
		float const n1_dot( dot( n1, c12 ) );
		float const n2_dot( dot( n2, c12 ) );

		if ( std::abs( n1_dot ) > 0.5 && std::abs( n2_dot ) > 0.5 ) {
			std::cout << "SHEET_PACKING: " << tag << ' ' << f1 << ' ' << f2 << ' ' <<
				F( 9, 3, n1_dot ) <<  F( 9, 3, n2_dot ) <<
				F( 9, 3, dot( n1, n2 ) ) << F( 9, 3, dot( v1, v2 ) ) <<
				F( 9, 3, distance ) << std::endl;
		}
	}
}


///////////////////////////////////////////////////////////////////////////////
void
get_pairing_vectors(
	barcode_stats_ns::Beta_feature const & f,
	FArray3D_float const & Epos,
	numeric::xyzVector_float & n,
	numeric::xyzVector_float & v,
	numeric::xyzVector_float & c
)
{
	numeric::xyzVector_float
		a1( & ( Epos(1,2,f.pos1-1) ) ),
		a2( & ( Epos(1,2,f.pos1  ) ) ),
		a3( & ( Epos(1,2,f.pos1+1) ) ),
		b1( & ( Epos(1,2,f.pos2-1) ) ),
		b2( & ( Epos(1,2,f.pos2  ) ) ),
		b3( & ( Epos(1,2,f.pos2+1) ) );

	n = ( 2.0f * ( a2 + b2 ) - a1 - a3 - b1 - b3 ).normalized();
	c = 0.5f * ( a2 + b2 );
	v = ( b2 - a2 ).normalized();
}


///////////////////////////////////////////////////////////////////////////////
void
identify_beta_features(
	int const total_residue,
	FArray2D_int const & pairing_count,
	int const total_decoys,
	Feature_list & feature_list
)
{
	const int  LONG_SMEAR(4);
	const int SHORT_SMEAR(2);
	const float MIN_FREQ = 0.001;

	// first smear the counts, then take local maxima
	typedef std::vector<std::pair<int,int> > Pair_list;

	std::map<int,Pair_list> smearList;
	for ( int o=1; o<= 2; ++o ) { // o=1: antiparallel, o=2: parallel
		Pair_list l;
		for ( int i=-20; i<= 20; ++i ) {
			for ( int j=-20; j<=20; ++j ) {
				const Beta_feature f(i,j,o);
				const int a (  f.long_axis() );
				const int b ( f.short_axis() );
				if (  -LONG_SMEAR <= a && a <=  LONG_SMEAR &&
						 -SHORT_SMEAR <= b && b <= SHORT_SMEAR ) {
					l.push_back( std::make_pair( i,j) );
				}
			}
		}
		smearList[o] = l;
	} // o=1,2

	// now calculate the smeared frequencies, store anti-parallel and parallel
	// in same matrix: i<j ==> anti i>j ==> parallel
	FArray2D_float freq( total_residue, total_residue, 0.0);

	for ( int o=1; o<= 2; ++o ) {
		const Pair_list::iterator it_start = smearList[o].begin();
		const Pair_list::iterator it_stop  = smearList[o].end();
		for ( int i=1; i<= total_residue; ++i) {
			for ( int j=i+1; j<= total_residue; ++j ) {
				for ( Pair_list::iterator it = it_start; it != it_stop; it++ ) {
					const int ii = i + it->first;
					const int jj = j + it->second;
					if ( jj>ii && ii >= 1 && jj <= total_residue ) {
						if ( o==1 ) {
							freq(i,j) += ( float( pairing_count( ii,jj ) ) / total_decoys);
						} else {
							freq(j,i) += ( float( pairing_count( jj,ii ) ) / total_decoys);
						}
					}
				}
			}
		}
	}

	// identify features, sort by frequencies:
	// typedef std::list< std::pair< float, Beta_feature > > Feature_sorter;
	Feature_sorter feature_sorter;

	for ( int o=1; o<=2 ; ++o ) {
		for ( int i=1; i<= total_residue; ++i) {
			for ( int j=i+1; j <= total_residue; ++j ) {
				const float f( (o==1) ? freq(i,j) : freq(j,i) );
				if ( f < MIN_FREQ ) continue;
				if ( local_max( i,j,o,freq ) ) {
					const float f( (o==1) ? freq(i,j) : freq(j,i) );
					feature_sorter.push_back( std::make_pair( f, Beta_feature( i,j,o ) ) );
				}
			}
		}
	}

	feature_sorter.sort();
	feature_sorter.reverse();

	// now kill features that are too close: //////////////////////////

	for ( Feature_sorter::iterator it = feature_sorter.begin(),
					it_end = feature_sorter.end(); it != it_end; it++ ) {
		bool skip = false;

		for ( Feature_list::iterator it2 = feature_list.begin(),
						it2_end = feature_list.end(); it2 != it2_end; it2++ ) {
			if ( it->second.too_close( *it2 ) ) {
				skip = true;
				break;
			}
		}

		if ( skip ) continue;
		std::cout << "New feature: " << it->second << std::endl;
		feature_list.push_back( it->second );
	}
}
///////////////////////////////////////////////////////////////////////////////
bool
Same_strand(
	barcode_stats_ns::Feature_list const & l1,
	barcode_stats_ns::Feature_list const & l2,
	FArray1D_char const & ss
)
{
	bool same_strand( false);

	int const n1( l1.size() );
	int const n2( l2.size() );
	for ( int i=0; i<n1; ++i ) {
		Beta_feature const & f1( l1[i] );
		for ( int j=0; j<n2; ++j ) {
			Beta_feature const & f2( l2[j] );
			if ( Same_strand( f1.pos1, f2.pos1, ss ) ||
					 Same_strand( f1.pos2, f2.pos1, ss ) ||
					 Same_strand( f1.pos1, f2.pos2, ss ) ||
					 Same_strand( f1.pos2, f2.pos2, ss ) ) {
				same_strand = true;
				break;
			}
		}
		if ( same_strand ) break;
	}
	return same_strand;

}

///////////////////////////////////////////////////////////////////////////////
bool
Same_strand(
	int const pos1,
	int const pos2,
	FArray1D_char const & ss
)
{
	assert( ss(pos1) == 'E' && ss(pos2) == 'E' );
	bool same_strand( true );
	for ( int i=std::min(pos1,pos2); i<=std::max(pos1,pos2); ++i ) {
		if ( ss(i) != 'E' ) {
			same_strand = false;
			break;
		}
	}
	return same_strand;

}
///////////////////////////////////////////////////////////////////////////////
void
pose_barcode_stats()
{

	silent_io::Silent_file_data data( stringafteroption("s") );
	if ( truefalseoption( "score_filter" ) ) {
		data.score_filter( realafteroption( "score_filter" ) );
	}
	ss_analyze_silent_data( data );
	utility::exit( 0, __FILE__, __LINE__);
}

///////////////////////////////////////////////////////////////////////////////
void
get_misc_pairings(
	std::vector< barcode_stats_ns::Beta_feature > & pairings
)
{
	using namespace strand_pairings_ns;

	int const nres( misc::total_residue );
	int const ANTIPARALLEL( 1 );
	int const PARALLEL( 2 );

	if ( nres != int( decoy_pairing_count.size1() ) ) {
		decoy_pairing_count.dimension( nres, nres );
		decoy_pairing_score.dimension( nres, nres );
	}

	decoy_pairing_count = 0;
	decoy_pairing_score = 0;

	update_cendist(misc::total_residue, misc::centroid);
	structure::BOUNDARY::identify_ss();

	int const lowstrand = 0; // as in score4
	int const cutoff = 6; // ditto; should consider making smaller??
	float tmp_ss_score, tmp_rsigma_score; // not used

	// call SSpair score!!
	set_store_ss_scores( true );
	structure::BOUNDARY::SSpair_score(lowstrand,cutoff,tmp_ss_score,tmp_rsigma_score);
	set_store_ss_scores( false );

	for ( int i = 1; i<= nres; ++i ) {
		for ( int j = 1; j<= nres; ++j ) {
			if ( decoy_pairing_count(i,j) > 0 ) {
				if ( i<j ) {
					pairings.push_back
						( Beta_feature(i,j,ANTIPARALLEL));
				} else {
					pairings.push_back
						( Beta_feature(j,i,PARALLEL));
				}
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
namespace topology_info_ns {
	bool init( false );
	bool output_info( false );
	std::string topology_info_line;
}

///////////////////////////////////////////////////////////////////////////////
bool
output_topology_info()
{
	using namespace topology_info_ns;
	if ( !init ) {
		init = true;
		output_info = truefalseoption("topology_info");
	}
	return output_info;
}

///////////////////////////////////////////////////////////////////////////////
std::string const &
retrieve_topology_info()
{
	using namespace topology_info_ns;
	return topology_info_line;
}

/////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin fix_unpaired_strand_ss
///
/// @brief fixes problems with Rosetta's assignment of secondary structure
///
/// @detailed
/// Quick and dirty routine to redefine secondary structure of strands. The
/// basic problem is that Rosetta takes secondary structure from fragments, and
/// sometimes strand-labeled fragments do not form strands in decoys. This
/// routine attempts to fix that problem using the secstruct_ns::strands::SS_strand_end
/// array.
///
/// @global_read
/// structure::secstruct_ns::strands::SS_strand_end
/// structure::secstruct_ns::strand_strand_score
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors tex
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
fix_unpaired_strand_ss( float score_threshold ) {

	int total_strands = structure::BOUNDARY::get_total_strands();
	FArray2D_int const & SS_strand_end = structure::BOUNDARY::get_SS_strand_end();
	FArray2D_float const & strand_strand_score = structure::BOUNDARY::get_strand_strand_score();

	// first, fix the SS_strand_end FArray.
	int new_index = 1;
	FArray2D_int new_SS_strand_end( 2, misc::total_residue );
	new_SS_strand_end( 1, 1 ) = SS_strand_end( 1, 1 );
	new_SS_strand_end( 2, 1 ) = SS_strand_end( 2, 1 );
	for ( int m = 2; m < misc::total_residue; m++ ) {
		if ( SS_strand_end(1,m) != SS_strand_end(1,m-1) ) {
			new_index++;
			for ( int n = 1; n <= 2; n++ ) {
				new_SS_strand_end(n,new_index) = SS_strand_end(n,m);
			}
		}
	}

	// Find the strand boundaries.
	std::set< std::pair< int, int > > strand_bounds;
	for ( int i=1; i<= total_strands; ++i ) {
		for ( int j=1; j<= total_strands; ++j ) {
			int beg1 = 0, beg2 = 0, end1 = 0, end2 = 0;
			if ( strand_strand_score( i, j ) < score_threshold ) {
				// these numbers are zero-based, while the secstruct FArray is
				// one-based.
				beg1 = new_SS_strand_end( 1, i ) + 1;
				end1 = new_SS_strand_end( 2, i ) + 1;
				beg2 = new_SS_strand_end( 1, j ) + 1;
				end2 = new_SS_strand_end( 2, j ) + 1;
				strand_bounds.insert( std::make_pair( beg1, end1 ) );
				strand_bounds.insert( std::make_pair( beg2, end2 ) );
			}
		}
	}


	using misc::secstruct;

	// change all E's to L's.
	for ( int i = 1; i < misc::total_residue; i++ ) {
		if ( secstruct(i) == 'E' ) {
			secstruct(i) = 'L';
		}
	}

	// Iterate over the strands in the set, set the secstruct appropriately.
	std::set< std::pair< int, int > >::iterator iter;
	for ( iter = strand_bounds.begin(); iter != strand_bounds.end(); iter++ ) {
		int beg = iter->first, end = iter->second;
		for ( int i = beg; i <= end; i++ ) {
			secstruct(i) = 'E';
		}
	}

} // fix_unpaired_strand_ss()


///////////////////////////////////////////////////////////////////////////////
// update topology info on current misc:: array structure
//
// separate routine from retrieve_topology_info so that the latter can
// be called when output file is already opened... this routine is called
// during prep phase
//
void
update_topology_info()
{
	float const score_threshold( -0.001 );

	update_cendist(misc::total_residue, misc::centroid);
	structure::BOUNDARY::identify_ss();
	set_store_ss_scores( true );
	int const lowstrand = 0; // as in score4
	int const cutoff = 6; // ditto; should consider making smaller??
	float tmp_ss_score, tmp_rsigma_score; // not used
	structure::BOUNDARY::SSpair_score(lowstrand,cutoff,tmp_ss_score,tmp_rsigma_score);
	set_store_ss_scores( false );

	// identify ss segments
	std::string segments;
	{
		using misc::secstruct;
		bool in_segment( false );
		for ( int i=1; i<= misc::total_residue; ++i ) {
			if ( i==1 || secstruct(i) == 'L' || secstruct(i) != secstruct(i-1) ) {
				in_segment = false;
			}
			if ( !in_segment && ( secstruct(i) == 'E' || secstruct(i) == 'H' ) ) {
				in_segment = true;
				segments += secstruct(i);
			}
		}
	}

	// identify strand-strand pairs
	std::string strand_pairs;
	std::string motifs;
	{
		// NOTE: strand_strand_score(i,j) < 0 ==> strands i and j are paired
		//
		// if i<j then pairing is anti-parallel
		// if i>j then pairing is parallel

		int total_strands = structure::BOUNDARY::get_total_strands();
		FArray2D_float const & strand_strand_score = structure::BOUNDARY::get_strand_strand_score();

		for ( int i=1; i<= total_strands; ++i ) {
			for ( int j=1; j<= total_strands; ++j ) {
				if ( strand_strand_score( i, j ) < score_threshold ) {
					strand_pairs += " "+string_of(i)+","+string_of(j);
				}
			}
		}


		// cross-beta
		for ( int i=1; i<= total_strands-1; ++i ) {
			for ( int j=i+3; j<= total_strands-1; ++j ) {
				if ( strand_strand_score(i,j) < score_threshold &&
						 strand_strand_score(i+1,j+1) < score_threshold ) {
					motifs += " cross-beta:"+string_of(i)+","+string_of(j);
				}
			}
		}

		// doubly-wound
		for ( int i=1; i<= total_strands - 3; ++i ) {
			if ( strand_strand_score(i+1,i  ) < score_threshold &&
					 strand_strand_score(i+2,i+1) < score_threshold &&
					 strand_strand_score(i+3,i  ) < score_threshold ) {
				motifs += " doubly-wound:"+string_of(i);
			}
		}

		// ferredoxin-type
		for ( int i=1; i<= total_strands - 3; ++i ) {
			if ( strand_strand_score(i  ,i+2) < score_threshold &&
					 strand_strand_score(i+1,i+2) < score_threshold &&
					 strand_strand_score(i  ,i+3) < score_threshold ) {
				motifs += " ferredoxin:"+string_of(i);
			}
		}

		// greek-key
		for ( int i=1; i<= total_strands - 3; ++i ) {
			if ( strand_strand_score(i  ,i+3) < score_threshold &&
					 strand_strand_score(i  ,i+1) < score_threshold &&
					 strand_strand_score(i+1,i+2) < score_threshold ) {
				motifs += " greek-key:"+string_of(i);
			}
		}

		// 2134
		for ( int i=1; i<= total_strands - 3 ; ++i ) {
			if ( strand_strand_score(i+1,i  ) < score_threshold &&
					 strand_strand_score(i+2,i  ) < score_threshold &&
					 strand_strand_score(i+3,i+2) < score_threshold ) {
				motifs += " 2134:"+string_of(i);
			}
		}

		// protein L
		for ( int i=1; i<= total_strands - 3; ++i ) {
			if ( strand_strand_score(i  ,i+1 ) < score_threshold &&
					 strand_strand_score(i+2,i+3) < score_threshold &&
					 strand_strand_score(i+3,i  ) < score_threshold ) {
				motifs += " prot-L:"+string_of(i);
			}
		}
	}

	topology_info_ns::topology_info_line =
		"TOP: " + segments + strand_pairs + motifs;
}


///////////////////////////////////////////////////////////////////////////////
char
torsion2big_bin(
	float const phi,
	float const psi,
	float const omega
)
{
	if ( std::abs( omega ) < 90 ) {
		return 'O'; // cis-omega
	} else if ( phi >= 0.0 ) {
		if ( -100 < psi && psi <= 100 ) {
			return 'G'; // alpha-L
		} else {
			return 'E'; // E
		}
	} else {
		if ( -125 < psi && psi <= 50 ) {
			return 'A'; // helical
		} else {
			return 'B'; // extended
		}
	}
	return 'X';
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
