// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 08:39:36 +0200 (Sun, 18 Mar 2007) $
//  $Author: stuartm $

#ifndef INCLUDED_barcode_stats_classes
#define INCLUDED_barcode_stats_classes

// Rosetta Headers

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/formatted.o.hh>

//Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <list>
#include <vector>


namespace barcode_stats_ns {
	enum Silly_pairing_type {
		INT_P,
		GOOD_P,
		BAD_P,
		IL_P,
		CB_P
	};


	// hairpin params:
	namespace barcode_hairpins {
		extern int const MAX_LOOP_LENGTH;
		extern float const SCORE_THRESHOLD;
		extern int const MAX_NN;
	}

	// some classes

	class hairpin_counter {
	private:
		int nres;
		FArray2D_int count;

	public:
		// constructor: zero initialize the counts
		hairpin_counter( int nres_in ):
			nres( nres_in),
			count( barcode_hairpins::MAX_LOOP_LENGTH, nres_in, 0)
		{}

		// increment counters, decoy-style
		void store_decoy_hairpin( int const loop_begin, int const loop_end, float const score )
		{
			if ( loop_begin < 1 || loop_begin > loop_end || loop_end > nres ) {
				std::cout << "bad hairpin data" << SS(nres)
									<< SS(loop_begin) << SS(loop_end) << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			int const loop_length = loop_end - loop_begin + 1;
			if ( score <= barcode_hairpins::SCORE_THRESHOLD &&
					 loop_length <= barcode_hairpins::MAX_LOOP_LENGTH ) {
				for ( int i = loop_begin; i <= loop_end; ++i ) {
					++count( loop_length, i);
				}
			}
		}

		// increment counters, fragment-style
		void store_frag_hairpin( int const pos, int const loop_length) {
			if ( loop_length < 1 || pos < 1 || pos > nres ) {
				std::cout << "bad hairpin data" << SS(nres)
									<< SS(pos) << SS(loop_length) << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			if ( loop_length <= barcode_hairpins::MAX_LOOP_LENGTH ) {
				++count( loop_length, pos);
			}
		}

		// access the counts
		int get_count( int const pos, int const loop_length ) {
			if ( loop_length < 1 || loop_length > barcode_hairpins::MAX_LOOP_LENGTH ||
					 pos < 1 || pos > nres ) {
				std::cout << "bad hairpin data" << SS(nres)
									<< SS(pos) << SS(loop_length) << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			return count( loop_length, pos);
		}
	};

	//////////////////////////////////////////
	class Beta_feature {
	public:
		static int const ANTIPARALLEL = 1;
		static int const PARALLEL = 2;
	public:
		int pos1;
		int pos2;
		int orientation;
		int pleating;

		Beta_feature ( int const i, int const j, int const o ):
			pos1(i),
			pos2(j),
			orientation(o),
			pleating(0)
		{ assert( orientation == PARALLEL || orientation == ANTIPARALLEL ); }

		Beta_feature ( int const i, int const j, int const o, int const p ):
			pos1(i),
			pos2(j),
			orientation(o),
			pleating(p)
		{
			assert( orientation == PARALLEL || orientation == ANTIPARALLEL );
			assert( pleating == 1 || pleating == 2);
		}

		char o() const {return ( orientation == PARALLEL ? 'P' : 'A' );}
		bool parallel() const { return orientation == PARALLEL; }
		bool antiparallel() const { return orientation == ANTIPARALLEL; }

		inline bool too_close( Beta_feature const & b) const {
			return ( orientation == b.orientation &&
							 ( ( pos1 - b.pos1) * ( pos1 - b.pos1 ) +
									( pos2 - b.pos2) * ( pos2 - b.pos2 ) <= 9 ) );
		}

		inline int long_axis() const {
			if ( orientation==ANTIPARALLEL ) {
				return pos1-pos2;
			} else {
				return pos1+pos2;
			}
		}

		inline int short_axis() const {
			if ( orientation==ANTIPARALLEL ) {
				return pos1+pos2;
			} else {
				return pos1-pos2;
			}
		}

		friend
		inline bool operator ==( Beta_feature const &a, Beta_feature const &b) {
			return ( a.pos1 == b.pos1 && a.pos2 == b.pos2 && a.orientation == b.orientation );
		}

		friend
		inline bool operator <( Beta_feature const &a, Beta_feature const &b) {
			return ( ( a.pos1 < b.pos1 ) ||
							 ( a.pos1 == b.pos1 && a.pos2 < b.pos2 ) ||
							 ( a.pos1 == b.pos1 && a.pos2 == b.pos2 && a.orientation < b.orientation ) );
		}

		friend
		inline int distance( Beta_feature const &a, Beta_feature const &b) {
			return ( a.orientation != b.orientation ? 1000 :
							 std::abs( a.pos1 - b.pos1 ) + std::abs( a.pos2 - b.pos2 ) );
		}
	};

	typedef std::list< std::pair< float, Beta_feature > > Feature_sorter;
	typedef std::vector< Beta_feature > Feature_list;

	// stream inserter
	std::ostream & operator <<( std::ostream & os, Beta_feature const & c );



} // namespace barcode_stats_ns


#endif
