// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 16880 $
//  $Date: 2007-09-01 14:21:33 +0300 (Sat, 01 Sep 2007) $
//  $Author: yab $


// Rosetta Headers
#include "bonds_class.h"
#include "input_pdb.h" // dihedral !!
#include "jumping_util.h"
#include "param.h"
#include "refold.h"
#include "refold_ns.h"
#include "util_basic.h"
#include "util_vector.h"

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <fstream>


namespace bonds_class {
	// call after changes to nres_
	void Bonds::redimension()
	{
		cb_phi_offset_.dimension( DRange( -1, 1 ), nres_ );
		ox_psi_offset_.dimension( DRange( -1, 1 ), nres_ );
		D_.dimension( 5, DRange( -1, 1 ), DRange( 0, nres_+1 ) );
		cT_.dimension( 5, DRange( -1, 1 ), DRange( 0, nres_+1 ) );
		sT_.dimension( 5, DRange( -1, 1 ), DRange( 0, nres_+1 ) );
		idealize_main_chain(); // for safety, b/c of pose.set_res side-effects
		idealize_termini();
	}

	void
	Bonds::set_nres(
		int const nres_in
	)
	{
		nres_ = nres_in;
		redimension();
	}

	// fill in the ends
	void Bonds::idealize_main_chain()
	{
		int const n2c( refold_ns::n2c );
		int const c2n( refold_ns::c2n );
		init_fold();
		for ( int i=0; i<=nres_+1; ++i ) {
			for ( int r=1; r<=2; ++r ) {
				const int dir( r==1 ? n2c : c2n );
				for ( int j = 1; j <= param::MAX_POS; ++j ) {
					D_ (j,dir,i) = refold_ns::D (j,dir);
					cT_(j,dir,i) = refold_ns::cT(j,dir);
					sT_(j,dir,i) = refold_ns::sT(j,dir);
				}
				if ( i>=1 && i<=nres_ ) {
					// offsets
					cb_phi_offset_(dir,i) = refold_ns::cb_phi_offset(dir);
					ox_psi_offset_(dir,i) = refold_ns::ox_psi_offset(dir);
				}
			}
		}
	}

	// fill in the ends
	void Bonds::idealize_termini()
	{
		int const n2c( refold_ns::n2c );
		int const c2n( refold_ns::c2n );
		init_fold();
		for ( int ii=1; ii<=2; ++ii ) {
			const int i ( ii==1 ? 0 : nres_+1 );
			for ( int r=1; r<=2; ++r ) {
				const int dir( r==1 ? n2c : c2n );
				for ( int j = 1; j <= param::MAX_POS; ++j ) {
					D_ (j,dir,i) = refold_ns::D (j,dir);
					cT_(j,dir,i) = refold_ns::cT(j,dir);
					sT_(j,dir,i) = refold_ns::sT(j,dir);
				}
			}
		}
		// add on extra angles:
		int const n( refold_ns::n );
		int const c( refold_ns::c );

		D_ (n,n2c,1) = refold_ns::D (n,n2c);
		cT_(n,n2c,1) = refold_ns::cT(n,n2c);
		sT_(n,n2c,1) = refold_ns::sT(n,n2c);

		D_ (c,c2n,nres_) = refold_ns::D (c,c2n);
		cT_(c,c2n,nres_) = refold_ns::cT(c,c2n);
		sT_(c,c2n,nres_) = refold_ns::sT(c,c2n);
	}

	// operator=
	Bonds & Bonds::operator =( Bonds const & src ) {
		if ( nres_ != src.nres_ ) {
			nres_ = src.nres_;
			redimension();
		}
		// operator= for FArrays:
		cb_phi_offset_ = src.cb_phi_offset_;
		ox_psi_offset_ = src.ox_psi_offset_;
		D_ = src.D_;
		cT_ = src.cT_;
		sT_ = src.sT_;
		return *this;
	}

	// calculate the lengths and angles from a coordinate array
	void Bonds::calculate(
		int const nres_in,
		FArray3D_float const & Eposition,
		FArray1D_float const & phi,
		FArray1D_float const & psi,
		FArray1D_float const & omega
	)
	{
		nres_ = nres_in;
		redimension();
		calculate_bond_info( nres_, Eposition, phi, psi, omega,
		 cb_phi_offset_, ox_psi_offset_,
		 D_, cT_, sT_ );
	}

	// copy ideal bond angles and lengths into a stretch of the backbone
	// everything's easy except we have to decide who gets the omega
	// bond length
	//
	// right now, attached to the n-terminal residue
	// ie -- the same way we do the omega angle
	//
	// only trick is that some relevant bond angles/lengths are numbered
	// outside the range (start,stop); these are added at the end
	// and have to skip an equal number of irrelevant bond angles/lengths
	void Bonds::idealize(
		int const start,
		int const stop
	)
	{
		// stolen from refold_ns:
		int const n( refold_ns::n );
		//int const ca( refold_ns::ca );
		//int const cb( refold_ns::cb );
		int const c( refold_ns::c );
		//int const ox( refold_ns::ox );

		int const n2c( refold_ns::n2c );
		int const c2n( refold_ns::c2n );

		assert( start <= stop && start >= 1 && stop <= nres_ );

		int const begin( start );
		int const size( stop - start + 1 );

		bool warned(false);
		for ( int repeat = 1; repeat <= 2; ++repeat ) {
			// insert bond lengths,angles,cb/ox offsets
			int const dir = ( repeat == 1 ? n2c : c2n );

			for ( int i = 1; i <= size; ++i ) {
				const int seqpos( begin+i-1 );

				// bond lengths:
				for ( int j = 1; j <= param::MAX_POS; ++j ) {
					if ( j != n || i != 1 || dir != n2c )
						D_(j,dir,seqpos) = refold_ns::D(j,dir);
				}

				// bond angles:
				for ( int j = 1; j <= param::MAX_POS; ++j ) {
					if ( ( j != n || i != 1 || dir != n2c ) &&
							 ( j != c || i != size || dir != c2n ) ) {
						cT_(j,dir,seqpos) = refold_ns::cT(j,dir);
						sT_(j,dir,seqpos) = refold_ns::sT(j,dir);
					}
				}               // j

				// offsets
				cb_phi_offset_(dir,seqpos) = refold_ns::cb_phi_offset(dir);
				ox_psi_offset_(dir,seqpos) = refold_ns::ox_psi_offset(dir);
			}                  // i
		}                     // repeat
		if ( warned ) std::cout << std::endl;

		// extra bond length:
		D_(n,n2c,begin+size) = refold_ns::D(n,n2c);

		// extra bond angles:
		cT_(c,c2n,begin-1) = refold_ns::cT(c,c2n);
		sT_(c,c2n,begin-1) = refold_ns::sT(c,c2n);

		cT_(n,n2c,begin+size) = refold_ns::cT(n,n2c);
		sT_(n,n2c,begin+size) = refold_ns::sT(n,n2c);

// 		if ( include_edges ) {
// 			// idealize any lengths/angles that depend on the atoms
// 			// in this stretch

// 			// I think these calls are a little slow...

// 			idealize_chainbreak( start-1 );
// 			idealize_chainbreak( stop    );
// 		}
	}

	bool Bonds::is_ideal(
		int const start,
		int const stop
	) const
	{
		// stolen from refold_ns:
		int const n( refold_ns::n );
		//int const ca( refold_ns::ca );
		//int const cb( refold_ns::cb );
		int const c( refold_ns::c );
		//int const ox( refold_ns::ox );

		int const n2c( refold_ns::n2c );
		int const c2n( refold_ns::c2n );

		assert( start <= stop && start >= 1 && stop <= nres_ );

		int const begin( start );
		int const size( stop - start + 1 );

		float total_dev(0.0);

		bool warned(false);
		for ( int repeat = 1; repeat <= 2; ++repeat ) {
			// insert bond lengths,angles,cb/ox offsets
			int const dir = ( repeat == 1 ? n2c : c2n );

			for ( int i = 1; i <= size; ++i ) {
				const int seqpos( begin+i-1 );

				// bond lengths:
				for ( int j = 1; j <= param::MAX_POS; ++j ) {
					if ( j != n || i != 1 || dir != n2c )
						total_dev += std::abs( D_(j,dir,seqpos) - refold_ns::D(j,dir) );
				}

				// bond angles:
				for ( int j = 1; j <= param::MAX_POS; ++j ) {
					if ( ( j != n || i != 1 || dir != n2c ) &&
							 ( j != c || i != size || dir != c2n ) ) {
						total_dev += std::abs( cT_(j,dir,seqpos) - refold_ns::cT(j,dir) );
						total_dev += std::abs( sT_(j,dir,seqpos) - refold_ns::sT(j,dir) );
					}
				}               // j

				// offsets
				total_dev += std::abs( cb_phi_offset_(dir,seqpos) - refold_ns::cb_phi_offset(dir) );
				total_dev += std::abs( ox_psi_offset_(dir,seqpos) - refold_ns::ox_psi_offset(dir) );
			}                  // i
		}                     // repeat
		if ( warned ) std::cout << std::endl;

		// extra bond length:
		total_dev += std::abs( D_(n,n2c,begin+size) - refold_ns::D(n,n2c) );

		// extra bond angles:
		total_dev += std::abs( cT_(c,c2n,begin-1) - refold_ns::cT(c,c2n) );
		total_dev += std::abs( sT_(c,c2n,begin-1) - refold_ns::sT(c,c2n) );

		total_dev += std::abs( cT_(n,n2c,begin+size) - refold_ns::cT(n,n2c) );
		total_dev += std::abs( sT_(n,n2c,begin+size) - refold_ns::sT(n,n2c) );

// 		if ( include_edges ) {
// 			// idealize any lengths/angles that depend on the atoms
// 			// in this stretch

// 			// I think these calls are a little slow...

// 			idealize_chainbreak( start-1 );
// 			idealize_chainbreak( stop    );
// 		}

		total_dev /= ( stop - start + 1 );

		//std::cout << "total_dev: " << total_dev << std::endl;

		return ( total_dev < 1e-1 );



	}

	/////////////////////////////////////////////////////////////////////////////

	FArray1D_bool
	Bonds::flag_nonideal(
		float const tolerance // = 1.0 ??
	)
	{
		using numeric::conversions::radians;
		FArray1D_bool nonideal( nres_, false );

		// stolen from refold_ns:
		int const n( refold_ns::n );
		//int const ca( refold_ns::ca );
		//int const cb( refold_ns::cb );
		int const c( refold_ns::c );
		//int const ox( refold_ns::ox );

		int const n2c( refold_ns::n2c );
		int const c2n( refold_ns::c2n );

		for ( int repeat = 1; repeat <= 2; ++repeat ) {
			int const dir = ( repeat == 1 ? n2c : c2n );

			for ( int seqpos = 1; seqpos <= nres_; ++seqpos ) {

				// bond lengths:
				float bond_dev(0.0);
				for ( int j = 1; j <= param::MAX_POS; ++j ) {
					int pos;
					if ( j==n && dir == n2c ) {
						pos = seqpos+1;
					} else {
						pos = seqpos;
					}
					bond_dev += std::abs( D_(j,dir,pos) - refold_ns::D(j,dir) );
				}

				// bond angles:
				float angle_dev(0.0);
				for ( int j = 1; j <= param::MAX_POS; ++j ) {
					int pos;
					if ( j == n && dir == n2c ) {
						pos = seqpos+1;
					} else if ( j == c && dir == c2n ) {
						pos = seqpos-1;
					} else {
						pos = seqpos;
					}

					float x( D_(j,dir,pos) * cT_(j,dir,pos) );
					float y( D_(j,dir,pos) * sT_(j,dir,pos) );
					float idl_x( refold_ns::D(j,dir) * refold_ns::cT(j,dir) );
					float idl_y( refold_ns::D(j,dir) * refold_ns::sT(j,dir) );
					angle_dev += std::sqrt( (x-idl_x)*(x-idl_x) + (y-idl_y)*(y-idl_y) );
				}               // j

				// offsets
				float offset_dev(0.0);
				offset_dev += std::abs( radians( subtract_degree_angles(
					cb_phi_offset_(dir,seqpos), refold_ns::cb_phi_offset(dir) ) ));
				offset_dev += std::abs( radians( subtract_degree_angles(
					ox_psi_offset_(dir,seqpos), refold_ns::ox_psi_offset(dir) ) ));

				float dev( bond_dev + angle_dev/3.0 + offset_dev/2.0 );
				nonideal(seqpos) = ( dev > tolerance );
				std::cout << "nonideal: " << nonideal(seqpos) << ' ' <<
					dev << ' ' << tolerance << ' ' <<
					bond_dev << ' ' << angle_dev << ' ' << offset_dev << std::endl;

			}                  // i
		}                     // repeat
		return nonideal;
	}

	/////////////////////////////////////////////////////////////////////////////
	// this is a very sneaky routine...
	//
	// among other things, the carbonyl oxygen psi_offsets are set to
	// default values (180), which means that the oxygen will move
	// unless the psi angle is also reset to one calculated from the
	// oxygen


	void Bonds::idealize_chainbreak(
		int const cutpoint
	)
	{
		using numeric::conversions::radians;
		using numeric::conversions::degrees;

		// stolen from refold_ns:
		int const n( refold_ns::n );
		int const ca( refold_ns::ca );
		//int const cb( refold_ns::cb );
		int const c( refold_ns::c );
		int const ox( refold_ns::ox );

		int const n2c( refold_ns::n2c );
		int const c2n( refold_ns::c2n );

		assert( cutpoint >= 1 && cutpoint < nres_ );

		//////////////////
		// one bond length
		// c->n
		D_(n,n2c,cutpoint+1) = refold_ns::D(n,n2c);
		D_(c,c2n,cutpoint  ) = refold_ns::D(c,c2n);

		//////////////////
		// two bond angles
		// c-n-ca
		cT_(ca,n2c,cutpoint+1) = refold_ns::cT(ca,n2c);
		sT_(ca,n2c,cutpoint+1) = refold_ns::sT(ca,n2c);
		cT_(c ,c2n,cutpoint  ) = refold_ns::cT(c ,c2n);
		sT_(c ,c2n,cutpoint  ) = refold_ns::sT(c ,c2n);
		// ca-c-n
		cT_(n ,n2c,cutpoint+1) = refold_ns::cT(n ,n2c);
		sT_(n ,n2c,cutpoint+1) = refold_ns::sT(n ,n2c);
		cT_(ca,c2n,cutpoint  ) = refold_ns::cT(ca,c2n);
		sT_(ca,c2n,cutpoint  ) = refold_ns::sT(ca,c2n);

		////////////////
		// ox-psi-offset: sets planar geometry at C(cutpoint)
		ox_psi_offset_(n2c,cutpoint) = refold_ns::ox_psi_offset(n2c);
		ox_psi_offset_(c2n,cutpoint) = refold_ns::ox_psi_offset(c2n);

		// very tricky: geometry for building the carbonyl oxygen c2n
		// have to make this agree with bond angles already defined
		// not so hard since the carbonyl carbon geometry is planar
		float angle1,angle2,angle3;
		// ca - c - n
		angle1 = 180.0 - degrees( std::atan2( sT_(n,n2c,cutpoint+1),
																					cT_(n,n2c,cutpoint+1) ) );

		assert( std::abs( angle1 - refold_ns::angle(n,n2c) ) < 0.01 );

		// ca - c - o
		angle2 = 180.0 - degrees( std::atan2( sT_(ox,n2c,cutpoint),
																					cT_(ox,n2c,cutpoint) ) );
		// n - c - o
		angle3 = 360.0 - angle1 - angle2;

//		std::cout << "angle_dev: " <<
//			std::abs( angle1 - refold_ns::angle(n ,n2c) ) << ' ' <<
//			std::abs( angle2 - refold_ns::angle(ox,n2c) ) << ' ' <<
//			std::abs( angle3 - refold_ns::angle(ox,c2n) ) << std::endl;

		cT_(ox,c2n,cutpoint ) = std::cos( radians( 180.0 - angle3 ) );
		sT_(ox,c2n,cutpoint ) = std::sin( radians( 180.0 - angle3 ) );
	}

	/////////////////////////////////////////////////////////////////////////////
	// this routine is setup just like the previous one
	//
	// copy src bond angles and lengths into a stretch of the backbone
	// everything's easy except we have to decide who gets the omega
	// bond length
	//
	// right now, attached to the n-terminal residue
	// ie -- the same way we do the omega angle
	//
	// only trick is that some relevant bond angles/lengths are numbered
	// outside the range (start,stop); these are added at the end
	// and have to skip an equal number of irrelevant bond angles/lengths
	void Bonds::copy_segment(
		int const size,
		Bonds const & src,
		int const begin,
		int const src_begin
	)
	{
		// stolen from refold_ns:
		int const n( refold_ns::n );
		int const c( refold_ns::c );

		int const n2c( refold_ns::n2c );
		int const c2n( refold_ns::c2n );

		assert( begin >= 1 && begin+size-1 <= nres_ && src_begin+size-1 <= src.nres_ );

		for ( int repeat = 1; repeat <= 2; ++repeat ) {
			// insert bond lengths,angles,cb/ox offsets
			int const dir = ( repeat == 1 ? n2c : c2n );

			for ( int i = 1; i <= size; ++i ) {
				const int seqpos( begin+i-1 );
				const int src_seqpos( src_begin+i-1 );

				// bond lengths:
				for ( int j = 1; j <= param::MAX_POS; ++j ) {
					if ( j != n || i != 1 || dir != n2c )
						D_(j,dir,seqpos) = src.D_(j,dir,src_seqpos);
				}

				// bond angles:
				for ( int j = 1; j <= param::MAX_POS; ++j ) {
					if ( ( j != n || i != 1 || dir != n2c ) &&
							 ( j != c || i != size || dir != c2n ) ) {
						cT_(j,dir,seqpos) = src.cT_(j,dir,src_seqpos);
						sT_(j,dir,seqpos) = src.sT_(j,dir,src_seqpos);
					}
				}               // j

				// offsets
				cb_phi_offset_(dir,seqpos) = src.cb_phi_offset_(dir,src_seqpos);
				ox_psi_offset_(dir,seqpos) = src.ox_psi_offset_(dir,src_seqpos);
			}                  // i
		}                     // repeat

		// extra bond length:
		D_(n,n2c,begin+size) = src.D_(n,n2c,src_begin+size);

		// extra bond angles:
		cT_(c,c2n,begin-1) = src.cT_(c,c2n,src_begin-1);
		sT_(c,c2n,begin-1) = src.sT_(c,c2n,src_begin-1);

		cT_(n,n2c,begin+size) = src.cT_(n,n2c,src_begin+size);
		sT_(n,n2c,begin+size) = src.sT_(n,n2c,src_begin+size);

		// keep the terminal angles/bonds ideal and in sync
		if ( begin == 1 || begin+size == nres_+1 ) {
			idealize_termini();
		}
	}

	std::ostream & operator <<( std::ostream & os, const Bonds & b )
	{
		for ( int i=0; i<= b.nres_+1; ++i ) {
			for( int j=1; j<= 5; ++j  ) {
				for ( int r=1; r<= 2; ++r ) {
					const int dir( r==1 ? -1 : 1 );
					std::cout << "D_cT_sT: " << I(5,i) << I(5,j) << I(5,dir);
					std::cout << F( 9, 3, b.D_ (j,dir,i) );
					std::cout << F( 9, 3, b.cT_(j,dir,i) );
					std::cout << F( 9, 3, b.sT_(j,dir,i) ) << std::endl;
				}
			}
		}
		return os;
	}

	/////////////////////////////////////////////////////////////////////////////
	// dump bond info to a stream:
	// this should match exactly the input format in the next routine
	void Bonds::dump(
		std::ostream & out
	) const
	{
		out << "nres_= " << nres_ << '\n';
		for ( int i=0; i<= nres_+1; ++i ) {
			bool const extra_rsd( i==0 || i == nres_ + 1 );
			for ( int r=1; r<= 2; ++r ) {
				int const dir( r==1 ? 1 : -1 );
				out << i << ' ' << dir << ' ';
				if ( !extra_rsd ) {
					out << cb_phi_offset_( dir, i ) << ' ';
					out << ox_psi_offset_( dir, i ) << ' ';
				}
				for ( int j=1; j<= 5; ++j ) {
					out << D_ ( j, dir, i ) << ' ';
					out << cT_( j, dir, i ) << ' ';
					out << sT_( j, dir, i ) << ' ';
				}
				out << '\n';
			}
		}
	}

	// not error tolerant right now: ie will die if bad formating
	void Bonds::read(
		utility::io::izstream & data
	)
	{
		std::string line, tag;
		getline(data,line);
		std::istringstream line_stream( line );
		line_stream >> tag >> nres_;
		if ( tag != "nres_=" ) {
			std::cout << "bad format in bonds-file: " << line << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		// allocate space if necessary to match the setting of nres_
		redimension();

		int fail(0);
		for ( int i=0; i<= nres_+1; ++i ) {
			bool const extra_rsd( i==0 || i == nres_ + 1 );
			for ( int r=1; r<= 2; ++r ) {
				if ( !getline( data, line ) ) {
					fail = 1;
					break;
				}
				//std::cout <<"read bonds: line=" << line << std::endl;
				line_stream.clear();
				line_stream.str( line );

				int const dir( r==1 ? 1 : -1 );
				int tmp1, tmp2;
				line_stream >> tmp1 >> tmp2;
				if ( tmp1 != i || tmp2 != dir ) {
 					fail = 2;
					break;
				}
				if ( !extra_rsd ) {
					line_stream >> cb_phi_offset_( dir, i );
					line_stream >> ox_psi_offset_( dir, i );
				}
				for ( int j=1; j<= 5; ++j ) {
					line_stream >> D_ ( j, dir, i );
					line_stream >> cT_( j, dir, i );
					line_stream >> sT_( j, dir, i );
				}
			}
			if ( line_stream.fail() ) {
				fail = 3;
				break;
			}
		} // i

		if ( fail ) {
			std::cout << "bad format in bonds file: fail= " << fail <<
				std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

} // namespace bonds_class


///////////////////////////////////////
// used to be in shotgun_bonds.cc
void
calculate_bond_info(
	int const total_residue,
	FArray3D_float const & Eposition,
	FArray1D_float const & phi,
	FArray1D_float const & psi,
	FArray1D_float const & omega,
	FArray2D_float & st_cb_phi_offset,
	FArray2D_float & st_ox_psi_offset,
	FArray3D_float & st_D,
	FArray3D_float & st_cT,
	FArray3D_float & st_sT
)
{
	using namespace param;
	using namespace refold_ns;
	using numeric::conversions::radians;

// input
//	bool coord_fail; // true if there is no coordinates read in
//pb   more input:: usually taken from common block in start.h

// local:
	FArray3D_float st_angle( 5, DRange( -1, 1 ), DRange( 0, total_residue+1 ) );
	 // !this is local!
	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );
	float tmp;
	FArray1D_float pseudo_n( 3 );
	FArray1D_float pseudo_ca( 3 );
	FArray1D_float pseudo_c( 3 );

	init_fold(); // need cT and sT

	int i = 0; // add a pseudo position before N-terminal
	for ( int j = 1; j <= 5; ++j ) {
		st_D(j,n2c,i) = D(j,n2c);
		st_D(j,c2n,i) = D(j,c2n);
		st_angle(j,n2c,i) = angle(j,n2c);
		st_angle(j,c2n,i) = angle(j,c2n);
	}
//chu   n2c direction  the first residue is special
	i = 1;
	subvec(Eposition(1,ca,i),Eposition(1, c,i),X(1,ps));
	subvec(Eposition(1, n,i),Eposition(1,ca,i),X(1,ph));
	refold_coord_sys(X(1,ps),X(1,ph),M(1,1),M(1,2),M(1,3));
	build_atom(M,Eposition(1,n,1),cT(c,c2n),sT(c,c2n),phi(i), D(c,c2n),pseudo_c,
	 X(1,om)); // generate a pseduo C at position zero

// bond_length
	st_D( n,n2c,i) = D(n,n2c); // use template D
	st_D(ca,n2c,i) = vec_dist(Eposition(1,ca,i),Eposition(1, n,i));
	st_D(cb,n2c,i) = vec_dist(Eposition(1,cb,i),Eposition(1,ca,i));
	st_D( c,n2c,i) = vec_dist(Eposition(1, c,i),Eposition(1,ca,i));
	st_D(ox,n2c,i) = vec_dist(Eposition(1,ox,i),Eposition(1, c,i));

// bond_angle
	st_angle( n,n2c,i) = angle( n,n2c); // use template angle
	st_angle(ca,n2c,i) = angle(ca,n2c); // use template angle
	st_angle(cb,n2c,i) = vec_angle(Eposition(1, n,i), Eposition(1,ca,i),
	 Eposition(1,cb,i));
	st_angle( c,n2c,i) = vec_angle(Eposition(1, n,i), Eposition(1,ca,i),
	 Eposition(1, c,i));
	st_angle(ox,n2c,i) = vec_angle(Eposition(1,ca,i), Eposition(1, c,i),
	 Eposition(1,ox,i));

// offsets
	st_cb_phi_offset(n2c,i) = dihedral( pseudo_c, Eposition(1, n,i),
	 Eposition(1,ca,i), Eposition(1,cb,i) ) - phi(i);
	st_ox_psi_offset(n2c,i) = dihedral( Eposition(1, n,i), Eposition(1,ca,i),
	 Eposition(1, c,i), Eposition(1,ox,i) ) - psi(i);

//chu   the rest residues
	for ( int i = 2; i <= total_residue; ++i ) {
// bond_length
		st_D( n,n2c,i) = vec_dist(Eposition(1, n,i),Eposition(1, c,i-1));
		st_D(ca,n2c,i) = vec_dist(Eposition(1,ca,i),Eposition(1, n,i));
		st_D(cb,n2c,i) = vec_dist(Eposition(1,cb,i),Eposition(1,ca,i));
		st_D( c,n2c,i) = vec_dist(Eposition(1, c,i),Eposition(1,ca,i));
		st_D(ox,n2c,i) = vec_dist(Eposition(1,ox,i),Eposition(1, c,i));

// bond_angle
		st_angle( n,n2c,i) = vec_angle(Eposition(1,ca,i-1), Eposition(1, c,i-1),
		 Eposition(1, n,i));
		st_angle(ca,n2c,i) = vec_angle(Eposition(1, c,i-1), Eposition(1, n,i),
		 Eposition(1,ca,i));
		st_angle(cb,n2c,i) = vec_angle(Eposition(1, n,i), Eposition(1,ca,i),
		 Eposition(1,cb,i));
		st_angle( c,n2c,i) = vec_angle(Eposition(1, n,i), Eposition(1,ca,i),
		 Eposition(1, c,i));
		st_angle(ox,n2c,i) = vec_angle(Eposition(1,ca,i), Eposition(1, c,i),
		 Eposition(1,ox,i));

// offsets
		st_cb_phi_offset(n2c,i) = dihedral( Eposition(1, c,i-1), Eposition(1, n,i),
		 Eposition(1,ca,i), Eposition(1,cb,i) ) - phi(i);
		st_ox_psi_offset(n2c,i) = dihedral( Eposition(1, n,i), Eposition(1,ca,i),
		 Eposition(1, c,i), Eposition(1,ox,i) ) - psi(i);
	}

	i = total_residue + 1; // add a pseudo position after C-terminal
	for ( int j = 1; j <= 5; ++j ) {
		st_D(j,n2c,i) = D(j,n2c);
		st_D(j,c2n,i) = D(j,c2n);
		st_angle(j,n2c,i) = angle(j,n2c);
		st_angle(j,c2n,i) = angle(j,c2n);
	}

//chu   c2n direction, the last residue is special
	i = total_residue;

	subvec(Eposition(1,ca,i),Eposition(1, n,i),X(1,ph));
	subvec(Eposition(1, c,i),Eposition(1,ca,i),X(1,ps));
	refold_coord_sys(X(1,ph),X(1,ps),M(1,1),M(1,2),M(1,3));
	build_atom(M,Eposition(1,c,1),cT(n,n2c),sT(n,n2c),psi(i), D(n,n2c),pseudo_n,
	 X(1,om)); // generate a pseudo N at total_res+1
	refold_coord_sys(X(1,ps),X(1,om),M(1,1),M(1,2),M(1,3));
	build_atom(M,pseudo_n,cT(ca,n2c),sT(ca,n2c),omega(i), D(ca,n2c),pseudo_ca,
	 X(1,ph)); // generate a pseudo CA at total_res+1
// save them in the default template
//	vector_copy(Eposition(1,c,i),st_default_template(1,1,c2n));
//	vector_copy(pseudo_n ,st_default_template(1,2,c2n));
//	vector_copy(pseudo_ca,st_default_template(1,3,c2n));

	st_D( c,c2n,i) = D( c,c2n); // use template
	st_D(ox,c2n,i) = vec_dist(Eposition(1,ox,i),Eposition(1, c,i));
	st_D(ca,c2n,i) = vec_dist(Eposition(1,ca,i),Eposition(1, c,i));
	st_D(cb,c2n,i) = vec_dist(Eposition(1,cb,i),Eposition(1,ca,i));
	st_D( n,c2n,i) = vec_dist(Eposition(1, n,i),Eposition(1,ca,i));

	st_angle( c,c2n,i) = angle( c,c2n); // use template
	st_angle(ca,c2n,i) = vec_angle(pseudo_n,Eposition(1,c,i), Eposition(1,ca,i));
	st_angle(ox,c2n,i) = vec_angle(pseudo_n,Eposition(1,c,i), Eposition(1,ox,i));
	st_angle(cb,c2n,i) = vec_angle(Eposition(1, c,i),Eposition(1,ca,i),
	 Eposition(1,cb,i));
	st_angle( n,c2n,i) = vec_angle(Eposition(1, c,i),Eposition(1,ca,i),
	 Eposition(1, n,i));

	st_cb_phi_offset(c2n,i) = dihedral( Eposition(1,cb,i), Eposition(1,ca,i),
	 Eposition(1,c,i), pseudo_n ) - psi(i);
	st_ox_psi_offset(c2n,i) = dihedral( Eposition(1,ox,i), Eposition(1, c,i),
	 pseudo_n,pseudo_ca ) - omega(i);

	for ( int i = total_residue-1; i >= 1; --i ) {
		st_D( c,c2n,i) = vec_dist(Eposition(1, c,i),Eposition(1, n,i+1));
		st_D(ox,c2n,i) = vec_dist(Eposition(1,ox,i),Eposition(1, c,i));
		st_D(ca,c2n,i) = vec_dist(Eposition(1,ca,i),Eposition(1, c,i));
		st_D(cb,c2n,i) = vec_dist(Eposition(1,cb,i),Eposition(1,ca,i));
		st_D( n,c2n,i) = vec_dist(Eposition(1, n,i),Eposition(1,ca,i));

		st_angle( c,c2n,i) = vec_angle(Eposition(1,ca,i+1), Eposition(1, n,i+1),
		 Eposition(1, c,i));
		st_angle(ca,c2n,i) = vec_angle(Eposition(1, n,i+1), Eposition(1, c,i),
		 Eposition(1,ca,i));
		st_angle(ox,c2n,i) = vec_angle(Eposition(1, n,i+1), Eposition(1, c,i),
		 Eposition(1,ox,i));
		st_angle(cb,c2n,i) = vec_angle(Eposition(1, c,i), Eposition(1,ca,i),
		 Eposition(1,cb,i));
		st_angle( n,c2n,i) = vec_angle(Eposition(1, c,i), Eposition(1,ca,i),
		 Eposition(1, n,i));

		st_cb_phi_offset(c2n,i) = dihedral( Eposition(1,cb,i), Eposition(1,ca,i),
		 Eposition(1,c,i), Eposition(1,n,i+1) ) - psi(i);
		st_ox_psi_offset(c2n,i) = dihedral( Eposition(1,ox,i), Eposition(1, c,i),
		 Eposition(1,n,i+1), Eposition(1,ca,i+1) ) - omega(i);
	}


// this was a separate function:

	for ( int i = 1; i <= total_residue; ++i ) {
		tmp = radians( 180.0 - st_angle(n,n2c,i) );
		st_cT(n,n2c,i) = std::cos(tmp);
		st_sT(n,n2c,i) = std::sin(tmp);

		tmp = radians( 180.0 - st_angle(ca,n2c,i) );
		st_cT(ca,n2c,i) = std::cos(tmp);
		st_sT(ca,n2c,i) = std::sin(tmp);

		tmp = radians( 180.0 - st_angle(c,n2c,i) );
		st_cT(c,n2c,i) = std::cos(tmp);
		st_sT(c,n2c,i) = std::sin(tmp);

		tmp = radians( 180.0 - st_angle(ox,n2c,i) );
		st_cT(ox,n2c,i) = std::cos(tmp);
		st_sT(ox,n2c,i) = std::sin(tmp);

		tmp = radians( 180.0 - st_angle(cb,n2c,i) );
		st_cT(cb,n2c,i) = std::cos(tmp);
		st_sT(cb,n2c,i) = std::sin(tmp);

		tmp = radians( 180.0 - st_angle(n,c2n,i) );
		st_cT(n,c2n,i) = std::cos(tmp);
		st_sT(n,c2n,i) = std::sin(tmp);

		tmp = radians( 180.0 - st_angle(ca,c2n,i) );
		st_cT(ca,c2n,i) = std::cos(tmp);
		st_sT(ca,c2n,i) = std::sin(tmp);

		tmp = radians( 180.0 - st_angle(c,c2n,i) );
		st_cT(c,c2n,i) = std::cos(tmp);
		st_sT(c,c2n,i) = std::sin(tmp);

		tmp = radians( 180.0 - st_angle(ox,c2n,i) );
		st_cT(ox,c2n,i) = std::cos(tmp);
		st_sT(ox,c2n,i) = std::sin(tmp);

		tmp = radians( 180.0 - st_angle(cb,c2n,i) );
		st_cT(cb,c2n,i) = std::cos(tmp);
		st_sT(cb,c2n,i) = std::sin(tmp);
	}

	i = 0;
	for ( int j = 1; j <= 5; ++j ) {
		st_cT(j,n2c,i) = cT(j,n2c);
		st_sT(j,c2n,i) = sT(j,c2n);
	}

	i = total_residue + 1;
	for ( int j = 1; j <= 5; ++j ) {
		st_cT(j,n2c,i) = cT(j,n2c);
		st_sT(j,c2n,i) = sT(j,c2n);
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin build_centroid_n2c
///
/// @brief
///
/// @detailed
///
/// @param  aa - [in/out]? -
/// @param  phi - [in/out]? -
/// @param  c_xyz - [in/out]? -
/// @param  n_xyz - [in/out]? -
/// @param  ca_xyz - [in/out]? -
/// @param  can_xyz - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
build_centroid_n2c(
	int const aa,
	float const phi,
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float cen_xyz
)
{
	using namespace refold_ns;
	using numeric::conversions::degrees;

	c_xyz.dimension( 3 );
	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );
	cen_xyz.dimension( 3 );

// local:
	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );
	FArray1D_float junk( 3 );

	subvec(n_xyz, c_xyz,X(1,om));
	subvec(ca_xyz,n_xyz,X(1,ph));

	refold_coord_sys(X(1,om),X(1,ph),M(1,1),M(1,2),M(1,3));
	build_atom( M, ca_xyz, centroid_parm(3,aa,n2c), centroid_parm(4,aa,n2c),
							phi + degrees( centroid_parm(2,aa,n2c) ),
							centroid_parm(1,aa,n2c), cen_xyz, junk );
}
///////////////////////////////////////////////////////////////////////////////
// just use backbone from a single residue:

void
build_centroid_n2c_ncac(
	int const aa,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float c_xyz,
	FArray1Da_float cen_xyz
)
{
	c_xyz.dimension( 3 );
	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );
	cen_xyz.dimension( 3 );

	float const phi(0.0);
	FArray1D_float prev_c_xyz(3);
	build_ideal_C_coords( phi, n_xyz, ca_xyz, c_xyz, prev_c_xyz );
	build_centroid_n2c( aa, phi, prev_c_xyz, n_xyz, ca_xyz, cen_xyz );
}



///////////////////////////////////////////////////////////////////////////////

