// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 08:39:36 +0200 (Sun, 18 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "chuck.h"
#include "angles.h"
#include "elliptic_msd.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "loops.h"
#include "maps.h"
#include "maps_ns.h"
#include "misc.h"
#include "namespace_cold.h"
#include "param.h"
#include "random_numbers.h"
#include "refold.h"
#include "refold_ns.h"
#include "rotate.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cstdlib>
#include <iostream>


////////////////////////////////////////////////////////////////////////////////
/// @begin choose_chuck
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? -
/// @param  min_begin - [in/out]? -
/// @param  max_begin - [in/out]? -
/// @param  best_frag - [in/out]? -
/// @param  cost - [in/out]? -
/// @param  frag_begin - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
choose_chuck(
	int size,
	float & cutoff_max,
	int min_begin,
	int max_begin,
	int & best_frag,
	float & cost,
	int & frag_begin
)
{
	using namespace param;

	FArray1D_float cut_cost( MAX_NEIGH()() );
	FArray1D_int cut_frag( MAX_NEIGH()() );
	int cut_count;


	find_chuck_moves(size,cutoff_max,min_begin,max_begin,cut_cost,cut_frag,
	 cut_count,frag_begin);

	if ( cut_count <= 0 ) {
		best_frag = -1; // no frags found
		cost = cut_cost(1); // for diagnostics only return lowest cost
		return;
	}

// randomly choose one of the fragments in this set
	best_frag = static_cast< int >(cut_count*(ran3())+1);
	cost = cut_cost(best_frag);
	best_frag = cut_frag(best_frag);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin find_chuck_moves
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff - [in/out]? -
/// @param  min_begin - [in/out]? -
/// @param  max_begin - [in/out]? -
/// @param  cut_cost - [in/out]? -
/// @param  cut_frag - [in/out]? -
/// @param  cut_count - [in/out]? -
/// @param  frag_begin - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
find_chuck_moves(
	int size,
	float & cutoff,
	int min_begin,
	int max_begin,
	FArray1Da_float cut_cost,
	FArray1Da_int cut_frag,
	int & cut_count,
	int & frag_begin
)
{
	using namespace cold;
	using namespace fragments;
	using namespace misc;
	using namespace param;

	cut_cost.dimension( MAX_NEIGH() );
	cut_frag.dimension( MAX_NEIGH() );

//car cutoff < 0 indicates cutoff should be set automatically
//car cutoff is altered and set to the actual value used,
//car the sign is retained.
//car min_begin, max_begin are min and max value of frag_begin

//car return a list of frags below cutoff
//car actual cutoff = passed cutoff/4   if >=3 frags under cutoff/4
//car                 passed cutoff/2   if  <3 frags under cutoff/4 &
//car                                      >=3 frags under cutoff/2
//car                 passed cutoff     if  <3 frags under cutoff/2

//car recommended cutoff is 60.0 if chuck move followed by wobble


	int size_bin,best_frag;
	float best_cost;
	FArray2D_float list_cost( MAX_NEIGH()(), 2 );
	FArray1D_int list_frag( MAX_NEIGH()() );
	int list_count;

	int loop_end,loop_begin;
	int msd_start; // first residue in msd list for chuck_move
	int num_atoms; // num atoms in msd_list for chuck_move
	int loop_num;
	float cutoff_max;

	FArray2D_float temp_pos( 3, MAX_POS );
	 // theoretical template position at loopend+1 if contig.
	FArray2D_float pos2( 3, MAX_POS );
	 // position2 for chuckmove, end of fragment insertion


	size_bin = get_index_by_frag_size(size);
	chuck_choose_insertion(min_begin,max_begin,frag_begin,size,cutoff,cutoff_max);

	cutoff = cutoff_max; // send actual cutoff back out of function

	int frag_end = frag_begin + size - 1;
	for ( int j = 1; j <= 3; ++j ) {
		for ( int k = 1; k <= 5; ++k ) {
			pos2(j,k) = Ebest_position(j,k,frag_end+1);
		}
	}

// decide which part of the protein we will try to keep stationary
// the number of unfavorable clashes will scale
//  with the atom count of the smaller "half" on either side of the
// fold.  thus we want to monitor how much the smaller side moves in rmsd.
//  This is obviously not strictly true, just a rule of thumb.
	if ( total_residue-frag_end < frag_begin-1 ) {
		msd_start = frag_end+1; // min downstream msd
		num_atoms = (total_residue-frag_end)*5;
	} else {
		msd_start = 1; // min upstream msd
		num_atoms = (frag_begin-1)*5;
	}

	if ( get_loop_flag() ) {
		identify_loop(frag_begin,frag_end,loop_begin,loop_end,loop_num);
		if ( loop_begin != 1 && loop_end != total_residue ) { // try to close loop
			if ( loop_get_refold_dir(loop_num) == 1 ) {
				msd_start = loop_end+1;
				num_atoms = 5*(total_residue-loop_end);
				loop_get_best_overlap(loop_num,2,temp_pos);
				loop_halfsplice_reverse(temp_pos,Ebest_position(1,1,loop_end+1),pos2,1);
			} else {
				msd_start = 1;
				num_atoms = 5*(loop_begin-1);
				loop_get_best_overlap(loop_num,1,temp_pos);
				loop_halfsplice_reverse(temp_pos,Ebest_position(1,1,loop_begin-1),pos2,1);
			}
		}
	}

	chuck_move(size,frag_begin,align_depth(frag_begin,size_bin),
	 Ebest_position(1,1,frag_begin),best_phi(frag_begin),pos2,best_phi(frag_end+1),
	 /* note +1 */ Ebest_position(1,1,msd_start),
	 Eatom_weights(1,msd_start),num_atoms,1e9, /* return all frags */
	 best_frag,best_cost,list_cost,list_frag,list_count);

	cutoff_list(cutoff_max,list_cost,list_frag,list_count,cut_cost,cut_frag,
	 cut_count);
	cutoff = cutoff_max;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin cutoff_list
///
/// @brief
///
/// @detailed
///
/// @param  cutoff_max - [in/out]? - cutoffs for output list
/// @param  list_cost - [in/out]? - cost in input list
/// @param  list_frag - [in/out]? - frag # in input list
/// @param  list_count - [in/out]? - # of frags in input list
/// @param  cut_cost - [in/out]? - cost in returned list
/// @param  cut_frag - [in/out]? - fragment number in returned list
/// @param  count - [in/out]? - number of frags in returned list
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
cutoff_list(
	float & cutoff_max, // cutoffs for output list
	FArray2Da_float list_cost, // cost in input list
	FArray1Da_int list_frag, // frag # in input list
	int & list_count, // # of frags in input list
	FArray1Da_float cut_cost, // cost in returned list
	FArray1Da_int cut_frag, // fragment number in returned list
	int & count // number of frags in returned list
)
{
	list_cost.dimension( list_count, 2 );
	list_frag.dimension( list_count );
	cut_cost.dimension( list_count );
	cut_frag.dimension( list_count );

// filters a list using a specified max value, and a set min value

	float const MIN_MSD = { .0001 }; // minimum MSD; ie min move size

	float cutoff = cutoff_max/4.0;
	count = 0;
	for ( int i = 1; i <= list_count; ++i ) {
		if ( list_cost(i,1) <= cutoff && list_cost(i,1) > MIN_MSD ) {
			++count;
			cut_cost(count) = list_cost(i,1);
			cut_frag(count) = list_frag(i);
		}
	}
	if ( count >= 3 ) {
//$$$   std::cout << "cutoff=" << cutoff << " count=" << count << std::endl;
		cutoff_max = cutoff;
		return;
	}

	cutoff = cutoff_max/2.0;
	count = 0;
	for ( int i = 1; i <= list_count; ++i ) {
		if ( list_cost(i,1) <= cutoff && list_cost(i,1) > MIN_MSD ) {
			++count;
			cut_cost(count) = list_cost(i,1);
			cut_frag(count) = list_frag(i);
		}
	}
	if ( count >= 3 ) {
//$$$   std::cout << "cutoff=" << cutoff << " count=" << count << std::endl;
		cutoff_max = cutoff;
		return;
	}

	cutoff = cutoff_max;
	count = 0;
	for ( int i = 1; i <= list_count; ++i ) {
		if ( list_cost(i,1) <= cutoff && list_cost(i,1) > MIN_MSD ) {
			++count;
			cut_cost(count) = list_cost(i,1);
			cut_frag(count) = list_frag(i);
		}
	}
//$$$      std::cout << "cutoff=" << cutoff << " count=" << count << std::endl;

	if ( count > 0 ) return;

//car no good frags, save the best one for diagnostics
	int best = 1;
	for ( int i = 2; i <= list_count; ++i ) {
		if ( list_cost(i,1) < list_cost(best,1) ) best = i;
	}

	cut_frag(1) = list_frag(best);
	cut_cost(1) = list_cost(best,1);

}


////////////////////////////////////////////////////////////////////////////////
/// @begin insert_frag
///
/// @brief
/// copies a frag into the current phi,psi,omega,secstruct,name arrays
///
/// @detailed
///
/// @param  length - [in/out]? -
/// @param  size_bin - [in/out]? -
/// @param  frag_begin - [in/out]? -
/// @param  best_frag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
insert_frag(
	int length,
	int & size_bin,
	int & frag_begin,
	int & best_frag
)
{
	using namespace fragments;
	using namespace misc;

	for ( int i = 0; i < length; ++i ) {
		phi(i+frag_begin) = align_phi(frag_begin,best_frag,i,size_bin);
		// warning: retarded array -1 offset on align_psi
		psi(i+frag_begin) = align_psi(frag_begin,best_frag,i,size_bin);

		omega(i+frag_begin) = align_omega(frag_begin,best_frag,i,size_bin);

		secstruct(i+frag_begin) = ss_type(frag_begin,best_frag,i,size_bin);
		name(i+frag_begin) = align_name(frag_begin,best_frag,size_bin);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_msd_best
///
/// @brief
///
/// @detailed
///
/// @param  begin - [in/out]? -
/// @param  end - [in/out]? -
/// @param  sum - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_msd_best(
	int begin,
	int end,
	float & sum
)
{
	using namespace misc;

	double sum_val = 0.0;
	for ( int i = begin; i <= end; ++i ) {
		for ( int k = 1; k <= 5; ++k ) {
			for ( int j = 1; j <= 3; ++j ) {
				double const position_dif = Ebest_position(j,k,i) - Eposition(j,k,i);
				sum_val += position_dif * position_dif;
			}
		}
	}
	sum_val /= ( end - begin + 1.0 ) * 5.0;
	sum = sum_val;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin chuck_move
///
/// @brief
///
/// @detailed
///
/// @param  length - [in/out]? -
/// @param  frag_begin - [in/out]? -
/// @param  n_depth - [in/out]? -
/// @param  position1 - [in/out]? -
/// @param  phi1 - [in/out]? -
/// @param  position2 - [in/out]? -
/// @param  phi2 - [in/out]? -
/// @param  position3 - [in/out]? -
/// @param  atom_weights - [in/out]? -
/// @param  n_atom - [in/out]? -
/// @param  cutoff - [in/out]? -
/// @param  best_frag - [in/out]? - index to best frag in align list
/// @param  cost - [in/out]? - cost of best frag
/// @param  list_cost - [in/out]? -
/// @param  list_frag - [in/out]? -
/// @param  count - [in/out]? - total frags in list
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
chuck_move(
	int length,
	int & frag_begin,
	int & n_depth,
	FArray2Da_float position1,
	float const phi1,
	FArray2Da_float position2,
	float const phi2,
	FArray2Da_float position3,
	FArray1Da_float atom_weights,
	int & n_atom,
	float cutoff,
	int & best_frag, // index to best frag in align list
	float & cost, // cost of best frag
	FArray2Da_float list_cost,
	FArray1Da_int list_frag,
	int & count // total frags in list
)
{
	using namespace fragments::chuck;
	using namespace misc;
	using namespace param;

	position1.dimension( 3, MAX_POS );
	position2.dimension( 3, MAX_POS );
	position3.dimension( 3, n_atom );
	atom_weights.dimension( n_atom );
	list_cost.dimension( n_depth, 2 );
	list_frag.dimension( n_depth );

//  score  a list of frags at insertion position by
//  the rmsd change they would cause if inserted
//  chuck needs to given a list of atoms over which the rmsd is to be
//  computed (position3) and
//  the positions and phi angles of the first and last residues of the
//  fragment being replaced :position1 and position2
//  the fragments tested come from the align_frag arrays
// (which we precompute the rotation matrices)
//  we test the top N_DEPTH frags, or if the user has set top_N_frags
// in choose_frag to a smaller number, this is used instead.

//car always returns at least one frag

// local
	int r_depth;
	float rmsd_el,rmsd_el_off;
	FArray1D_float EL( 10 );
	FArray1D_float EL2( 10 );
	FArray2D_float C_EL( 3, 3 );

// decide how deep in fraglist to test:
	choose_frag_get_top_N_frags(r_depth);
	r_depth = std::min(r_depth,n_depth);

//car if no atoms in msd list, then return all frags as acceptable
	if ( n_atom == 0 ) {    // all frags okay
		for ( int i = 1; i <= r_depth; ++i ) {
			list_cost(i,1) = 0.0;
			list_cost(i,2) = 0.0;
			list_frag(i) = i;
		}
		count = r_depth;
		cost = 0;
		best_frag = 1; // arbitrarily say first frag is best
		return;
	}

	int size_bin = get_index_by_frag_size(length);
	precompute_frag_movement(length,size_bin,total_residue);

// first compute EL matrix and A2,U2
	FArray2D_float ZtC( 3, 3 );
	FArray1D_float O2( 3 );
	FArray2D_float A2( 3, 3 );
	FArray1D_float U2( 3 );
	compute_frag_move(position1,phi1,position2,phi2,A2,U2,ZtC,O2);
	eliptic_cross_moments(position3,n_atom,atom_weights,A2,U2,O2,C_EL,EL2,EL,
	 ZtC );
//$$$      std::cout << "chuck cross moments" << std::endl;
//$$$      debug_mat("EL*",EL);
//$$$      debug_mat("EL2*",EL2);
//$$$      debug_mat("C_EL*",C_EL);
//$$$      debug_3("U2 O2 etc2*",U2,O2,frag_off(1,i,frag_begin,size_bin));
//$$$      debug_mat("A2*",A2);
//$$$      debug_mat("ZtC*",ZtC);

// loop over all (depth) fragments.  keep track of "best" one
	cost = 1e9;
	best_frag = 1;
	count = 0;
	for ( int i = 1; i <= r_depth; ++i ) {
		eliptic_rmsd_2(frag_rot( 1, 1, i, frag_begin, size_bin ),
		 frag_off(1,i,frag_begin,size_bin), EL, EL2, C_EL, cutoff, rmsd_el, rmsd_el_off );
		if ( rmsd_el <= cutoff ) {

// keep all results in a matrix
			++count;
			list_cost(count,1) = rmsd_el;
			list_cost(count,2) = rmsd_el_off;
			list_frag(count) = i;
// track "best" one too
			if ( rmsd_el < cost ) {
				cost = rmsd_el;
				best_frag = i;
			}
		} else if ( count == 0 ) { // haven't found one that passes cutoff
			if ( rmsd_el < cost ) { // keep the best one in the list
				list_cost(1,1) = rmsd_el;
				list_cost(1,2) = rmsd_el_off;
				list_frag(1) = i;
				cost = rmsd_el;
				best_frag = i;
			}
		}
	}
	if ( count == 0 ) count = 1; // none pass cutoff, return best

}


////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_coord_sys
///
/// @brief
///       initializes internal coord system variables and returns Z_mat
///
/// @detailed
///car old_refold (angles.cc)
/// what it does is copy part of the refold_template
/// being good programmers we dont violate refolds encapsulation.
/// instead we use refold's get_template method.  Then copy the results into
/// out own package globals. The only thing I dont like about this is
/// that we have to set the dims correctly
/// locally. To solve this we should move this rotuine to be part of refold.
///
/// new refold:
///  default coord system for chuck is n1,ca1,c1; we can compute this from
///  the refold default stub (c0,n1,ca1) by folding one more atom with
///  phi=0
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
FArray2DB_float const &
initialize_coord_sys()
{
	using namespace refold_ns;

	FArray2D_double Bxyz2( 3, 8 );
//KMa phospho_ser
	FArray2D_double Bcentroids2( 3, param::MAX_AA_PLUS() );

	FArray2D_float ni_ca_cc( 3, 3 );

	static bool init = { false };
	static FArray2D_float z_mat( 3, 3, 0.0f );

	if ( init ) return z_mat;

//CAR for new refold,  get default stub, fold one more atom
//CAR and use this as a template
//car variables for new_refold:
	FArray1D_float c0( 3 );
	FArray1D_float n1( 3 );
	FArray1D_float ca1( 3 );
	FArray1D_float c1( 3 );
	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );
	float phi;
//chu chuck_move now is compatible with use_input_bond flag.
//chu However, precompute_chuck still use idealized bonds to calculate
//chu downstream msd anyway and only when peptide is refolded, it is built
//chu with non-idealized bonds. So people should judge by themselves whether
//chu this is a right thing to do.

//car retrieve default stub:
//chu copy only from the idealized template
	refold_default_template_stub(c0,n1,ca1);

//car define initial bond vectors:
	subvec(n1,c0,X(1,om));
	subvec(ca1,n1,X(1,ph));

//car build next c atom
	refold_coord_sys(X(1,om),X(1,ph),M(1,1),M(1,2),M(1,3));
	phi = 0.0;
	build_atom(M,ca1,cT(c,n2c),sT(c,n2c),phi,D(c,n2c),c1,X(1,ps));

//car get coord system for this triplet--note that we angles type coord system!!
	for ( int i = 1; i <= 3; ++i ) {
		ni_ca_cc(i,1) = n1(i);
		ni_ca_cc(i,2) = ca1(i);
		ni_ca_cc(i,3) = c1(i);
	}
	angles_coord_sys(ni_ca_cc(1,1),ni_ca_cc(1,2),ni_ca_cc(1,3),z_mat);

	init = true;
	return z_mat;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin compute_frag_move
///
/// @brief
/// compute the rotation and offset of a fragment that starts at position1
/// and terminates at position2-1
///
/// @detailed
///
/// @param  position1 - [in/out]? - residue at the begining of the insert fragment
/// @param  phi1 - [in/out]? -
/// @param  position2 - [in/out]? - residue ONE PAST the insert fragment.
//         NOTE: is is NOT the last residue of the insertion fragment.
/// @param  phi2 - [in/out]? -
/// @param  G_mat - [in/out]? - The G matrix and U2 vector are :
///        a standardized orientation matrix and offset between two residues
///        the matrix A2 will orient the standard stub located at position 1 into
///        the standard stub at position2. i.e. stub2 = mover(stub1,A2,U2)
/// @param  U2 - [in/out]? - the offset vector in Z-Coordinate system i
///        (default orientation)
/// @param  ZtC_mat - [in/out]? - rotates the stub at position1 to the default
///        coordinate system
/// @param  O2 - [in/out]? - O2 is the offset to move the stub to the origin
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car actually ZtC may be the transpose of this rotation matrix
///car and O2 may be the negative of the offset
///car z_mat is the default or standard coordinate system
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
compute_frag_move(
	FArray2Da_float position1,
	float const phi1,
	FArray2Da_float position2,
	float const phi2,
	FArray2Da_float G_mat,
	FArray1Da_float U2,
	FArray2Da_float ZtC_mat,
	FArray1Da_float O2
)
{
	using namespace param;

	position1.dimension( 3, MAX_POS );
	position2.dimension( 3, MAX_POS );
	G_mat.dimension( 3, 3 );
	U2.dimension( 3 );
	ZtC_mat.dimension( 3, 3 );
	O2.dimension( 3 );

// local vars
	FArray1D_float cc( 3 );
	FArray1D_float cc2( 3 );
	FArray2D_float temp( 3, 3 );
	FArray2D_float c_mat( 3, 3 );

	FArray2DB_float const & z_mat( initialize_coord_sys() );  //self-initializes, retrieve Z_mat

// get stub for begin_residue
	chuck_recover_stub(position1(1,1),position1(1,2),position1(1,4),phi1,cc);

// find its coord_sys
	angles_coord_sys(position1(1,1),position1(1,2),cc,c_mat);

// compute rotation matrix from coord_sys c_mat back to coord_sys z_mat
	mat_multiply_transpose3(z_mat,c_mat,ZtC_mat);

// recover stub (cc2) for last residue '
// (using next residue's position info and phi)
	chuck_recover_stub(position2(1,1),position2(1,2),position2(1,4),phi2,cc2);

// rotate this into Z-mat coordinate system
	rotate(ZtC_mat,position2(1,1),temp(1,1));
	rotate(ZtC_mat,position2(1,2),temp(1,2));
	rotate(ZtC_mat,cc2,           temp(1,3));

// compute coord_system of above three points
	angles_coord_sys( temp(1,1), temp(1,2), temp(1,3), G_mat );

// compute origin and offset between Calpha of first and last residue of
// fragment
	for ( int i = 1; i <= 3; ++i ) {
		O2(i) = position1(i,2); // origin of rotation is Calpha of first residue
		U2(i) = position2(i,2) - O2(i); // offset relative to Origin
	}

// rotate offest vector into z_mat coordinate system
	rotate_in_place(ZtC_mat,U2);

}

//////////////////////////////////////////////////////////////////////////////
/// @begin compute_stub_move
///
/// @brief
/// same as compute_frag_move, but the stub1 and stub2 triplets
/// of points are not shifted to remove the effect of phi
///
/// @detailed
///
/// @param stub1 - [in/out]? -
/// @param stub2 - [in/out]? -
/// @param Gmat - [in/out]? -
/// @param U2 - [in/out]? -
/// @param ZtC_mat - [in/out]? -
/// @param O2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
compute_stub_move(
	FArray2Da_float stub1,
	FArray2Da_float stub2,
	FArray2Da_float G_mat,
	FArray1Da_float U2,
	FArray2Da_float ZtC_mat,
	FArray1Da_float O2
)
{
	stub1.dimension( 3, 3 );
	stub2.dimension( 3, 3 );
	G_mat.dimension( 3, 3 );
	U2.dimension( 3 );
	ZtC_mat.dimension( 3, 3 );
	O2.dimension( 3 );

// inputs
//	float phi1, phi2;
// local vars
//	FArray2D_float position1(3,5), position2(3,5);

	FArray2D_float temp( 3, 3 );
	FArray2D_float c_mat( 3, 3 );

	FArray2DB_float const & z_mat( initialize_coord_sys() );  //self-initializes, retrieve Z_mat

// get stub for begin_residue
//	chuck_recover_stub(position1(1,1),position1(1,2),position1(1,4),phi1,cc);

// find its coord_sys
//	angles_coord_sys(position1(1,1),position1(1,2),cc,c_mat);

	angles_coord_sys(stub1(1,1),stub1(1,2),stub1(1,3),c_mat);

// compute rotation matrix from coord_sys c_mat back to coord_sys z_mat
	mat_multiply_transpose3(z_mat,c_mat,ZtC_mat);

// recover stub (cc2) for last residue
// (using next residue's position info and phi)
//	chuck_recover_stub(position2(1,1),position2(1,2),position2(1,4),phi2,cc2)

// rotate this into Z-mat coordinate system
//	rotate(ZtC_mat,position2(1,1),temp(1,1));
//	rotate(ZtC_mat,position2(1,2),temp(1,2));
//	rotate(ZtC_mat,cc2,           temp(1,3));
	rotate(ZtC_mat,stub2(1,1),temp(1,1));
	rotate(ZtC_mat,stub2(1,2),temp(1,2));
	rotate(ZtC_mat,stub2(1,3),temp(1,3));

// compute coord_system of above three points
	angles_coord_sys( temp(1,1), temp(1,2), temp(1,3), G_mat );

// compute origin and offset between Calpha of first and last residue of fragment
	for ( int i = 1; i <= 3; ++i ) {
//	O2 (i)= position1(i,2); // origin of rotation is Calpha of first residue
//	u2(i) = position2(i,2) - O2(i); // offset relative to Origin
		O2 (i)= stub1(i,2); // origin of rotation is Calpha of first residue
		U2(i) = stub2(i,2) - O2(i); // offset relative to Origin
	}

// rotate offest vector into Z_mat coordinate system
	rotate_in_place(ZtC_mat,U2);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin chuck_recover_stub
///
/// @brief
///     regenerates the nitro,calpha,carbon default stub from a
///     current set of nitro,calpha,c-carbon positions plus the
///     phi angle used to generate them originally.
///
/// @detailed
///
/// @param  ni2 - [in/out]? -
/// @param  ca2 - [in/out]? -
/// @param  cc2 - [in/out]? -
/// @param  ph - [in/out]? -
/// @param  cc_out - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
chuck_recover_stub(
	FArray1Da_float ni2,
	FArray1Da_float ca2,
	FArray1Da_float cc2,
	float const ph,
	FArray1Da_float cc_out
)
{
	using numeric::conversions::radians;

	ni2.dimension( 3 );
	ca2.dimension( 3 );
	cc2.dimension( 3 );
	cc_out.dimension( 3 );

//     Note that, the nitro and calpha positions are the same
//     as the stub.  its the C carbon we have to compute:
//     outputs
//     modifies input value of cc   //car: cc no longer overwritten?
//     Note: this is not useful in refold because ph will be overwritten
//     before refold is called and hence is unknown.  It is however useful in
//     double moves where ph is known.
//     namespace of precomputed rotation matrix coeficients

	FArray1D_double Ni( 3 );
	FArray1D_double Ca( 3 );
	FArray1D_double CC( 3 );

	FArray1D_double vec( 3 );
	FArray2D_double mat( 3, 3 );

	for ( int i = 1; i <= 3; ++i ) {
		Ni(i) = ni2(i);
		Ca(i) = ca2(i);
		CC(i) = cc2(i);
	}

	double tmp = -radians( ph ); // backwards rotation to undo phi
	getrot( Ni, Ca, tmp, mat, vec ); // mat,vec rotate tmp rad about n-ca bond
	mover( CC, mat, vec ); // apply reverse rotation to c' position

	for ( int i = 1; i <= 3; ++i ) {
		cc_out(i) = CC(i);
	}

}
////////////////////////////////////////////////////////////////////////////////
/// @begin chuck_choose_insertion
///
/// @brief
///car select an insetion point for a chuck move, set the cutoff if it's
///car negative (ie to be set automatically)
///
/// @detailed
///
/// @param  min_begin - [in/out]? -
/// @param  max_begin - [in/out]? -
/// @param  begin - [in/out]? -
/// @param  size - [in/out]? -
/// @param  cutoff - [in/out]? -
/// @param  cutoff_out - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
chuck_choose_insertion(
	int min_begin,
	int max_begin,
	int & begin,
	int size,
	float cutoff,
	float & cutoff_out
)
{
	using namespace misc;
	//	using namespace protein_maps;
	using namespace param;

	int loop_begin,loop_end,loop_num;
	float msd;
	int max_begin_local;
	float lower_limit;

//------------------------------------------------------------------------------// dont use protein_maps version directly, for pose compatibility
	FArray1D_bool allow_insert( MAX_RES()() );
	FArray1D_int insert_map( MAX_RES()() );
	int total_insert;

	retrieve_allow_insert(allow_insert,total_residue);
	retrieve_insertmap(insert_map,total_insert);

	int ntries = 0;

	max_begin_local = std::min(max_begin,insert_map(total_insert)-size+1);
	if ( max_begin_local < min_begin ) {
		std::cout << "error in chuck_choose_insertion " << min_begin << ' ' <<
		 max_begin << ' ' << size << ' ' << total_insert << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

L10:

	++ntries;
	if ( ntries > 1000 ) {
		std::cout << "Hey moron, I can't find a frag! " << min_begin << ' ' <<
		 max_begin << ' ' << total_insert << ' ' << size << ' ' << begin << std::endl;
		for ( int i = 1; i <= total_insert; ++i ) {
			std::cout << SS( i ) << SS( insert_map(i) ) << std::endl;
		}
#ifdef BOINC
		return;
#endif
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( max_begin - min_begin > total_insert - size ) {
		begin = insert_map( static_cast< int >( ran3() * ( total_insert - size + 1 ) ) + 1 );
	} else {
		begin = static_cast< int >( ran3() * ( max_begin_local - min_begin + 1 ) ) + min_begin;
	}

	for ( int i = begin; i < begin + size; ++i ) {
		if ( !allow_insert(i) ) goto L10;
	}

	if ( begin < min_begin ) goto L10;
	if ( begin > max_begin ) goto L10;

	cutoff_out = cutoff;
	if ( cutoff > 0 ) return;

	if ( get_loop_flag() ) {
		identify_loop(begin,begin,loop_begin,loop_end,loop_num);
		if ( loop_begin == 1 || loop_end == total_residue ) {
			cutoff_out = std::abs(cutoff);
		} else {
			loop_get_best_splicemsd(loop_num,msd);
			float const tmp = 0.25*(loop_end-loop_begin+1);
			lower_limit = tmp * tmp;
			cutoff_out = std::max(lower_limit,msd*1.2f);
			cutoff_out = std::max(cutoff_out,std::abs(cutoff));
		}
	} else {
		cutoff_out = std::abs(cutoff);
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin precompute_chuck
///
/// @brief
/// compute chuck and variables for a given fragment and store
///
/// @detailed
///
/// @param  frag_start - [in] -
/// @param  frag_depth - [in] -
/// @param  size_bin - [in] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     assumes that fragment has just been refolded by refold
///      and stub has been generated
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
precompute_chuck(
	int const frag_start,
	int const frag_depth,
	int const size_bin,
	FArray1Da_float ca1
)
{
	using namespace fragments::chuck;

	ca1.dimension( 3 );

	FArray1D_float ni2( 3 );
	FArray1D_float ca2( 3 );
	FArray1D_float cc2( 3 );

	// we make the VITAL assumption that the reference orientation of
	// the reference plane (i.e. phi=0) of the first three ni-ca-cc atoms
	// in refold is always(!) the same
	// on every call to refold.  otherwise what follows is incorrect.

	// get coord_system for final stub
	angles_get_final_stub(ni2,ca2,cc2);
	angles_coord_sys(ni2,ca2,cc2,frag_rot(1,1,frag_depth,frag_start,size_bin));

	// compute origin and offset between Calpha of first and last residue of frag
	for ( int k = 1; k <= 3; ++k ) {
		// offset relative to res 1 c_alpha
		frag_off(k,frag_depth,frag_start,size_bin) = ca2(k) - ca1(k);
		//CAR position(1,2)= ca1 =  0,0,0 by definition of how refold works
		//CAR but subtract anyway to be safe
	}
}
