// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7523 $
//  $Date: 2006-02-20 15:10:59 -0800 (Mon, 20 Feb 2006) $
//  $Author: jiangl $

//////////////////////////////////////////////

// Rosetta Headers
#include "aaproperties_pack.h"
#include "cst_countpair.h"
#include "enzyme.h"
#include "enzyme_ns.h"
#include "files_paths.h"
#include "misc.h"
#include "pairenergy.h"
#include "pose.h"
#include "runlevel.h"

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <iostream>
#include <sstream>
#include <fstream>
#include <string>

namespace cst_countpair_ns {

  /////////////////////////////////
	// single line constraint format
	////////////////////////////////
	// single line format constraint setup for pose type minimization
	// Lin Jiang
	//
  ///////////////////////////////////////////////////////////////////////////////////////////////////
  //lin read distance, angle, torsion constraint directly from single line format
	//lin base on the input constraints:
	//lin          add the constraints into Cst_set class
	//lin          setup allow move for pose
	//lin          detect the covalent bond and set up count pair
	//
	// single line constraint format:
	//
	// reading begins with "CST::BEGIN" and end with "CST::END"
	// CST::BEGIN
	// #distance_cst  Atom1(atomno res) Atom2(atomno res)   d  delta_d  weight  is_covalent
	//    DISTANCE:           3    248          9     50   1.31    0.2   100.0      TRUE
	//
	// #angle_cst     Atom1(atomno res) Atom2(atomno res) Atom3(atomno res) angle delta_angle weight periodic_value(the angle parameter by degree )
	//    ANGLE:              2    248          3    248           9    50  125.0     5.0       0.2        360.0
	//
	// #torsion_cst   Atom1(atomno res) Atom2(atomno res) Atom3(atomno res) Atom4(atomno res) angle delta_angle weight periodic_value (the angle parameter by degree )
	//    TORSION:            1    248          2    248           3    248          9    50   81.7     10.0      0.2     180.0
	// CST::END

	bool cst_from_file(
				std::string const & filename,
				pose_ns::Pose & pose,
				cst_set_ns::Cst_set & pose_cst,
				packer_cst_ns::Packer_cst_set & packer_cst,
				cst_allow_move_ns::Allow_move_set & allow_move  ){

    using namespace files_paths;

    float const deg2rad( numeric::conversions::radians(1.0) );
		float const pi( numeric::constants::d::pi );

    std::ifstream file;
    std::istringstream line_stream;
    file.clear();
    file.open( filename.c_str() );

    if ( !file ) {
      std::cout << "read_cst_from_file - unable to open cstfile " << filename  << "\n";
      return false;
    } else {
      std::cout << "cst_mode: read constraint information from file: "<< filename  << "\n";
    }

    std::string line, tag("");
    bool start( false);

    while(file) {
      tag = "";
      getline(file, line);
      line_stream.clear();
      line_stream.str(line);
      line_stream.seekg( std::ios_base::beg );
      line_stream >> tag;

      if( tag == "CST::BEGIN" ) {
        start=true;
      }
      if( tag == "CST::END" ) {
        start=false;
      }

      if( !start ) continue;

			//cleanup
			std::string val_name(""), add_tag("");
			int atomno,rsd, is_covalent;
			float val, val_sd, val_k, val_max;
			float conf_val, conf_sd;
			int conf_N, ang_N;
			kin::Atom_id a1,a2,a3,a4;
			kin::Fullatom_id p_a1,p_a2,p_a3,p_a4;

			if( tag == "DISTANCE:") {
				line_stream >> atomno >> rsd;
				get_pose_cst_atom_id( pose, atomno, rsd, a1, p_a1 );
				line_stream >> atomno >> rsd;
				get_pose_cst_atom_id( pose, atomno, rsd, a2, p_a2 );
				line_stream >> val >> val_sd >> val_k >> is_covalent
										>> add_tag >> conf_val >> conf_sd >> conf_N ;
				if( val_k != 0 ) {
					std::cout<<"Distance: "<<a1<<" "<<a2<<" "<<val
					<<" "<<val_sd<<" "<<val_k<<"\n";
					float weight_scale (1.0);
					if( is_covalent ) weight_scale=enzyme::covalent_cst_scale;
					cst_set_ns::Cst distance_cst( val, val_sd, weight_scale*val_k, val_k );
					pose_cst.add_atompair_constraint( a1, a2, distance_cst );
					packer_cst.add_distance_constraint
						( p_a1, p_a2, val, val_sd, val_k );
				}
				if( is_covalent ) {
					covalent_bonds.push_back( std::make_pair(std::min(p_a1,p_a2), std::max(p_a1,p_a2)) );
				}
				if( add_tag == "Conformer:" ) {
					std::cout<<"Distance(conf): "<<a1<<" "<<a2<<" "<<conf_val
					<<" "<<conf_sd<<" "<<conf_N<<"\n";
					allow_move.add_distance_atoms( p_a1, p_a2, conf_val, conf_sd, conf_N );
				} else {
					allow_move.add_distance_atoms( p_a1, p_a2, 0, 0, 0 );
				}
			}

			if( tag == "ANGLE:") {
				line_stream >> atomno >> rsd;
				get_pose_cst_atom_id( pose, atomno, rsd, a1, p_a1 );
				line_stream >> atomno >> rsd;
				get_pose_cst_atom_id( pose, atomno, rsd, a2, p_a2 );
				line_stream >> atomno >> rsd;
				get_pose_cst_atom_id( pose, atomno, rsd, a3, p_a3 );
				line_stream >> val >> val_sd >> val_k >> val_max
										>> add_tag >> conf_val >> conf_sd >> conf_N >> ang_N;
				if( val_k != 0 ) {
					std::cout<<"Angle: "<<a1<<" "<<a2<<" "<<a3<<" "
									 <<val<<" "<<val_sd<<" "<<val_k<<" "<<val_max<<"\n";
					cst_set_ns::Angle_cst angle_cst( p_a1, p_a2, p_a3,
																					 val == 999.0 ? val : std::abs( periodic_range( val*deg2rad, 2.0*pi ) ),
																					 std::abs( periodic_range( val_sd*deg2rad, 2.0*pi ) ),
																					 val_max*deg2rad,
																					 val_k );
					pose_cst.add_atom_angle_constraint( angle_cst );
					packer_cst.add_angle_constraint( p_a1, p_a2, p_a3,
																					 val == 999.0 ? val : std::abs( periodic_range( val*deg2rad, 2.0*pi ) ),
																					 std::abs( periodic_range( val_sd*deg2rad, 2.0*pi ) ),
																					 val_max*deg2rad,
																					 val_k );
				}
				if( add_tag == "Conformer:" ) {
					std::cout<<"Angle(conf): "<<a1<<" "<<a2<<" "<<a3<<" "<<conf_val
					<<" "<<conf_sd<<" "<<conf_N<<" "<<ang_N<<"\n";
					allow_move.add_angle_atoms( p_a1, p_a2, p_a3,
																			periodic_range( conf_val*deg2rad, 2.0*pi) ,
																			periodic_range( conf_sd*deg2rad, 2.0*pi),
																			conf_N, ang_N );
				} else {
					allow_move.add_angle_atoms( p_a1, p_a2, p_a3, 0, 0, 0, 0 );
				}
			}

			if( tag == "TORSION:") {
				line_stream >> atomno >> rsd;
				get_pose_cst_atom_id( pose, atomno, rsd, a1, p_a1 );
				line_stream >> atomno >> rsd;
				get_pose_cst_atom_id( pose, atomno, rsd, a2, p_a2 );
				line_stream >> atomno >> rsd;
				get_pose_cst_atom_id( pose, atomno, rsd, a3, p_a3 );
				line_stream >> atomno >> rsd;
				get_pose_cst_atom_id( pose, atomno, rsd, a4, p_a4 );
				line_stream >> val >> val_sd >> val_k >> val_max >> val_name
 										>> add_tag >> conf_val >> conf_sd >> conf_N >> ang_N;
				if( val_k != 0 ) {
					std::cout<<"Torsion: "<<a1<<" "<<a2<<" "<<a3
									 <<" "<<a4<<" "<<val<<" "<<val_sd<<" "<<val_k
									 <<" "<<val_max<<" "<<val_name<<"\n";
					if( val_name != "PERIODIC" ) val_name = "HARMONIC";
					cst_set_ns::Torsion_cst torsion_cst( p_a1, p_a2, p_a3, p_a4,
																						 val == 999.0 ? val : periodic_range( val*deg2rad, 2.0*pi ),
																						 periodic_range( val_sd*deg2rad, 2.0*pi ),
																						 val_max*deg2rad,
																						 val_k, val_name );
					pose_cst.add_atom_torsion_constraint( torsion_cst );
					packer_cst.add_torsion_constraint( p_a1, p_a2, p_a3, p_a4,
																						 val == 999.0 ? val : periodic_range( val*deg2rad, 2.0*pi ),
																						 periodic_range( val_sd*deg2rad, 2.0*pi ),
																						 val_max*deg2rad,
																						 val_k, val_name );
				}
				if( add_tag == "Conformer:" ) {
					std::cout<<"Torsion(conf): "<<a1<<" "<<a2<<" "<<a3<<" "<<a4<<" "<<conf_val
					<<" "<<conf_sd<<" "<<conf_N<<" "<<ang_N<<"\n";
					allow_move.add_torsion_atoms( p_a1, p_a2, p_a3, p_a4,
																				periodic_range( conf_val*deg2rad, 2.0*pi) ,
																				periodic_range( conf_sd*deg2rad, 2.0*pi),
																				conf_N, ang_N );
				} else {
					allow_move.add_torsion_atoms( p_a1, p_a2, p_a3, p_a4, 0, 0, 0, 0 );
				}
			}
		}
		packer_cst.clean_up();
		return true;
	}

  /////////////////////////////////////////////////////////////////////////////
	//lin data for holding the count pair information
  float clash_Erep_cutoff = DEFAULT_CLASH_LJENERGY;  // lj rep energy cutoff if covalent bond
	bool covalent_from_cstfile = false; // if take covalent bond from cstfile
	std::vector< int > countpair_poslist;  // seqpos list involving the countpair, for speedup
  std::vector< Fullatom_pair > covalent_bonds; // defining covalent bonds
	std::vector< std::pair< int, int > > atompair_list; // count pair for defined pose

  /////////////////////////////////////////////////////////////////////////////
	// check if atom pair between A and B is covalent bond by giving distance AB
	bool is_covalent_bond(
			kin::Fullatom_id const & A, kin::Fullatom_id const & B, float const & distanceAB ){
    using namespace aaproperties_pack;

    float atr, rep;
    get_lj_energy( distanceAB, fullatom_type( A.atomno(), A.aa(), A.aav() ),
                   fullatom_type( B.atomno(), B.aa(), B.aav() ), atr, rep );
    return ( rep >= clash_Erep_cutoff );
	}

  /////////////////////////////////////////////////////////////////////////////
	//lin fill the data for count pair calculation
	void fill_cst_count_pair( ) {

    atompair_list.clear();
    countpair_poslist.clear();

    for(int ii=0,ie=covalent_bonds.size(); ii<ie; ii++){
			fill_cst_count_pair( covalent_bonds[ii] );
		}
	}

  /////////////////////////////////////////////////////////////////////////////
	void fill_cst_count_pair(
    pose_ns::Pose & pose ) {

    atompair_list.clear();
    countpair_poslist.clear();

    for(int ii=0,ie=covalent_bonds.size(); ii<ie; ii++){
      //check the current aa,aav array match
      if( pose.match_aa_aav( covalent_bonds[ii].first ) &&
          pose.match_aa_aav( covalent_bonds[ii].second ) ) {
				fill_cst_count_pair( covalent_bonds[ii] );
			}
		}
	}

  /////////////////////////////////////////////////////////////////////////////
	void fill_cst_count_pair( Fullatom_pair const & bonds) {
    using namespace aaproperties_pack;

		// method for count pair information
    int atm1,res1,aa1,aav1,atm2,res2,aa2,aav2;
		bonds.first.get_values(res1,atm1,aa1,aav1);
		bonds.second.get_values(res2,atm2,aa2,aav2);

		if( find( countpair_poslist.begin(), countpair_poslist.end(), res1 ) ==
				countpair_poslist.end() ) countpair_poslist.push_back( res1 );
		if( find( countpair_poslist.begin(), countpair_poslist.end(), res2 ) ==
				countpair_poslist.end() ) countpair_poslist.push_back( res2 );

		int a1 ( bonds.first.key() );
		int a2 ( bonds.second.key() );

		assert( a1 < a2 );
		//{
		  //FArray4D_int temp( param::MAX_ATOM()(), param::MAX_AA()(), param::MAX_AA_VARIANTS()(), param::MAX_RES()() );
			//std::cout<<"DEBUG: "<< a1 << " " <<temp.index(atm1,aa1,aav1,res1) <<"\n";
		//}
		if( find( atompair_list.begin(), atompair_list.end(), std::make_pair( a1, a2 ) )
				== atompair_list.end() ) {
			atompair_list.push_back( std::make_pair( a1, a2 ) );
		}
		for ( int bnd = 1, bnde = nbonded_neighbors(atm1, aa1, aav1); bnd <= bnde; ++bnd ) {
			int ba1 ( kin::fullatom_index(res1, bonded_neighbor(bnd, atm1, aa1, aav1), aa1, aav1) );
			if( find( atompair_list.begin(), atompair_list.end(), std::make_pair( ba1, a2 ) )
					== atompair_list.end() ) {
				atompair_list.push_back( std::make_pair( ba1, a2 ) );
			}
			for( int kk=1, ke=nhydrogens_on_atm(atm2,aa2,aav2); kk<=ke; ++kk ) {
				int ha2 ( kin::fullatom_index(res2, hydrogens_on_atm(kk,atm2,aa2,aav2), aa2, aav2) );
// 				kin::Fullatom_id temp(ha2);
// 				std::cout<<"tag1 ha2 :" <<temp<<"\n";
				if( find(atompair_list.begin(),atompair_list.end(),std::make_pair( ba1, ha2 ))
						== atompair_list.end() ) {
					atompair_list.push_back( std::make_pair( ba1, ha2 ) );
				}
			}
			for( int jj=1, je=nhydrogens_on_atm(bonded_neighbor(bnd,atm1,aa1,aav1),aa1,aav1); jj<=je; ++jj ) {
				int ha1( kin::fullatom_index(res1, hydrogens_on_atm(jj,bonded_neighbor(bnd,atm1,aa1,aav1),
																														aa1,aav1), aa1, aav1 ) );
				if( find( atompair_list.begin(), atompair_list.end(), std::make_pair( ha1, a2 ) )
						== atompair_list.end() ) {
					atompair_list.push_back( std::make_pair( ha1, a2 ) );
				}
				for( int kk=1, ke=nhydrogens_on_atm(atm2,aa2,aav2); kk<=ke; ++kk ) {
					int ha2( kin::fullatom_index(res2, hydrogens_on_atm(kk,atm2,aa2,aav2), aa2, aav2 ) );
					if( find( atompair_list.begin(), atompair_list.end(), std::make_pair( ha1, ha2 ) )
							== atompair_list.end() ) {
						atompair_list.push_back( std::make_pair( ha1, ha2 ) );
					}
				}
			}
		}
		for ( int bnd = 1, bnde = nbonded_neighbors(atm2, aa2, aav2); bnd <= bnde; ++bnd ) {
			int ba2( kin::fullatom_index(res2, bonded_neighbor(bnd, atm2, aa2, aav2), aa2, aav2) );
			if( find( atompair_list.begin(), atompair_list.end(), std::make_pair( a1, ba2 ) )
					== atompair_list.end() ) {
				atompair_list.push_back( std::make_pair( a1, ba2 ) );
			}
			for( int kk=1, ke=nhydrogens_on_atm(atm1,aa1,aav1); kk<=ke; ++kk ) {
				int ha1( kin::fullatom_index(res1, hydrogens_on_atm(kk,atm1,aa1,aav1), aa1, aav1) );
				if( find( atompair_list.begin(), atompair_list.end(), std::make_pair( ha1, ba2 ) )
						== atompair_list.end() ) {
					atompair_list.push_back( std::make_pair( ha1, ba2 ) );
				}
			}
			for( int jj=1, je=nhydrogens_on_atm(bonded_neighbor(bnd, atm2, aa2, aav2),aa2,aav2); jj<=je; ++jj ) {
				int ha2( kin::fullatom_index(res2, hydrogens_on_atm(jj,bonded_neighbor(bnd, atm2, aa2, aav2),aa2,aav2),
																		 aa2, aav2) );
				if( find( atompair_list.begin(), atompair_list.end(), std::make_pair( a1, ha2 ) )
						== atompair_list.end() ) {
					atompair_list.push_back( std::make_pair( a1, ha2 ) );
				}
				for( int kk=1, ke=nhydrogens_on_atm(atm1,aa1,aav1); kk<=ke; ++kk ) {
					int ha1( kin::fullatom_index(res1, hydrogens_on_atm(kk,atm1,aa1,aav1), aa1, aav1) );
					if( find( atompair_list.begin(), atompair_list.end(), std::make_pair( ha1, ha2 ) )
							== atompair_list.end() ) {
						atompair_list.push_back( std::make_pair( ha1, ha2 ) );
					}
				}
			}
		}
	}

  /////////////////////////////////////////////////////////////////////////////
	//lin output count pair information
	void show_cst_count_pair() {

    //check the count pair
		std::cout << "Covalently bonding information:\n";
		if( covalent_bonds.empty() ) {
			std::cout <<"\tZERO bond is detected"<<"\n";
		} else {
			for(int ii=0,ie=covalent_bonds.size(); ii<ie; ii++){
				std::cout <<"\tBonding pair < "<<covalent_bonds[ii].first<<" --- "<<covalent_bonds[ii].second<<" >"<<"\n";
			}

			std::cout<<"\tcountpair_poslist: ";
			for( int ii=0, ie=countpair_poslist.size(); ii<ie; ii++ ) std::cout<<countpair_poslist[ii]<<" ";
			std::cout<<"\n";
			if( runlevel_ns::runlevel > runlevel_ns::standard ) {
				std::cout<<"\tatompair_list:"<<"\n";
				for( int ii=0, ie=atompair_list.size(); ii<ie; ii++ ) {
					kin::Fullatom_id a1(atompair_list[ii].first);
					kin::Fullatom_id a2(atompair_list[ii].second);
					std::cout<<"\t\t[ "<<a1<<" , "<<a2<<" ]" <<"\n";
				}
			}
		}
	}

  /////////////////////////////////////////////////////////////////////////////
	//lin set up internal residue minimization by KIN_1D_CST (torsion/bond/angle tether)
	void set_cst_pose_by_rsd(
		pose_ns::Pose & pose,
		cst_set_ns::Cst_set & cst_set,
		int const rsd, int const root_atomno,
		bool const minimize_bonds,
		bool const minimize_angles,
		bool const minimize_torsions
		) {

		using namespace kin;
		using numeric::conversions::degrees;

		//assert( param_aa::is_ligand(pose.res(rsd)) );

// 		//lin initialize the pose cst
// 		if( !pose.constraints_exist() ) {
// 			cst_set_ns::Cst_set pose_cst;
// 			pose.set_constraints( pose_cst );
// 		}

		// params
		// note that bond angles and torsions are stored in the atom_tree
		// as radians, hence the degrees(1.0) term in the tether weights
		// below -- this converts the tether score to degrees, which
		// is easier to think about
		float const bond_length_tether( 10000.0 );
		float const bond_angle_tether( degrees(1.0) * degrees(1.0) * 10.0 );
		float const bond_torsion_tether( degrees(1.0) * degrees(1.0) );

		int const natoms( aaproperties_pack::natoms( pose.res(rsd),
																								 pose.res_variant(rsd) ) );
		for ( int i=1; i<= natoms; ++i ) {
			if ( i == root_atomno ) continue;

			Atom_id atom( i, rsd );

			if ( minimize_bonds ) { // bond length
				Torsion_id this_bond_length( atom, D );
				float const current_val( pose.get_atom_tree_torsion(this_bond_length));
				//float const eps( current_val*0.1 );
				cst_set.add_kin_torsion_constraint( this_bond_length, current_val,
																						bond_length_tether );
				pose.set_allow_move( this_bond_length, true );
			}

			if ( minimize_angles ) { // bond angle
				Torsion_id this_bond_angle( atom, THETA );
				float const current_val( pose.get_atom_tree_torsion( this_bond_angle));
				//float const eps( current_val*0.1 );
				cst_set.add_kin_torsion_constraint( this_bond_angle, current_val,
																						bond_angle_tether );
				pose.set_allow_move( this_bond_angle, true );
			}

			if ( minimize_torsions ) { // torsion angle or torsion-offset
				Torsion_id this_bond_torsion( atom, PHI );
				float const current_val(pose.get_atom_tree_torsion(this_bond_torsion));
				//float const eps( current_val*0.1 );
				cst_set.add_kin_torsion_constraint( this_bond_torsion, current_val,
																						bond_torsion_tether );
				pose.set_allow_move( this_bond_torsion, true );
			}
		}
	}

	void
	get_pose_cst_atom_id(
    pose_ns::Pose & pose,
		int & atomno,
		int & rsd,
		kin::Atom_id & a1,
		kin::Fullatom_id & p_a1 ) {
		int aa,aav;
		if( rsd == 0 ) {
			rsd = pose.total_residue() + 1;
			aa = param_aa::ligand_aa_vector[1];
			aav = 1;
		} else if( rsd > pose.total_residue() && rsd <= pose.total_residue() + enzyme::ligaaN ) {
			assert( get_enable_ligaa_flag() );
			aa = param_aa::ligand_aa_vector[1] - 1 + rsd - pose.total_residue();
			aav = 1;
		} else {
			aa = pose.res(rsd);
			aav = pose.res_variant(rsd);
		}
		a1.atomno()=atomno;
		a1.rsd()=rsd;
		p_a1.assign(rsd,atomno,aa,aav);
	}
}
