// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 10231 $
//  $Date: 2006-09-07 12:59:56 -0700 (Thu, 07 Sep 2006) $
//  $Author: jiangl $


//////////////////////////////////////////////

// Rosetta Headers
#include "cst_packer.h"
#include "atom_is_backbone.h"
#include "param.h"
#include "pose.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3D.hh>

namespace packer_cst_ns {

	/////////////////////////////////////////////////////////////////////////////
	float Group_rsd_rsd_cst::get_best_score_cst(
            kin::Coords const & coords,
						std::vector<Rsd_rsd_cst*> & best_cst_plist,
						bool const if_score ) const {
		//lin only allow to pick up one value from the rsd_cst_list
		//lin get the minimum value from the rsd_cst_list
		float Emin ( 5000.0 );
		float cstE;
		int min_id ( 0 );

		assert( !group_cst_plist_.empty() );

		if( if_score || group_cst_plist_.size() > 1 ) {
			for ( int ii=0, ie=group_cst_plist_.size(); ii<ie; ++ii ) {
				cstE=0;
				for( int jj=0, je=group_cst_plist_[ii].size(); jj<je; ++jj ) {
					Rsd_rsd_cst* const it ( group_cst_plist_[ii][jj] );
					cstE+=it->score(coords);
				}
				if( cstE < Emin || ii == 0 ) {
					Emin = cstE;
					min_id = ii;
				}
			}
		}

		best_cst_plist = group_cst_plist_[min_id] ;
		return Emin;
	}

	/////////////////////////////////////////////////////////////////////////////
	float Group_rsd_rsd_cst::best_packer_score(
					int const aa1, int const aav1,
					int const aa2, int const aav2,
					FArray3D_float const & coord,
					bool const bb1, bool const sc1,
					bool const bb2, bool const sc2
					) const	 {
		//lin only allow to pick up one value from the rsd_cst_list
		//lin get the minimum value from the rsd_cst_list
		float Emin ( 5000.0 );
		float cstE;
		Rsd_rsd_cst* it;
		bool is_bb;

		for ( int ii=0, ie=group_cst_plist_.size(); ii<ie; ++ii ) {
			cstE=0;
			for( int jj=0, je=group_cst_plist_[ii].size(); jj<je; ++jj ) {
        //Rsd_rsd_cst const & it ( *group_cst_plist_[ii][jj] );
				it = group_cst_plist_[ii][jj] ;
// 				std::cout<< it<<" "<<group_cst_plist_[ii].size()<<" "
// 								 <<ii<<" "<<jj<<" "<<cstE<<" "<<it->atomno1()<<" "
// 								 <<atom_is_backbone( it->atomno1(),aa1,aav1 )<<" "
// 								 <<it->atomno2()<<" "<<atom_is_backbone( it->atomno2(),aa2,aav2)<<" ";
				is_bb = atom_is_backbone( it->atomno1(),aa1,aav1 );
				if ( !bb1 && is_bb ) continue;
				if ( !sc1 && !is_bb ) continue;
				is_bb = atom_is_backbone( it->atomno2(),aa2,aav2);
				if ( !bb2 && is_bb ) continue;
				if ( !sc2 && !is_bb ) continue;
				cstE+=it->score(coord);
			}
			if( cstE < Emin || ii == 0 ) Emin = cstE;
		}
		return Emin;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool Group_rsd_rsd_cst::is_ambiguous_cst() const
	{ return ( group_cst_plist_.size() > 1 ); }

	/////////////////////////////////////////////////////////////////////////////
	void Group_rsd_rsd_cst::insert( std::vector<Rsd_rsd_cst*> const & rsd_cst_plist, bool ambiguous_cst )
	{
		if( !ambiguous_cst ) {
			for ( int ii=0, ie=rsd_cst_plist.size(); ii<ie; ++ii ) {
				insert( rsd_cst_plist[ii] );
			}
		} else {
			group_cst_plist_.push_back( rsd_cst_plist );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void Group_rsd_rsd_cst::insert( Rsd_rsd_cst* const  & rsd_cst_p )
	{
		if( group_cst_plist_.size() == 0 ) {
			Rsd_cst_plist cst_plist_tmp;
			cst_plist_tmp.push_back( rsd_cst_p );
			rsd_cst_p->owner_count++;
			group_cst_plist_.push_back( cst_plist_tmp );
		} else {
			if( find( group_cst_plist_.back().begin(), group_cst_plist_.back().end(), rsd_cst_p ) ==
					group_cst_plist_.back().end() ) {
				group_cst_plist_.back().push_back( rsd_cst_p );
				rsd_cst_p->owner_count++;
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	Group_rsd_rsd_cst & Group_rsd_rsd_cst::collect( Group_rsd_rsd_cst const & other_cst )
	{
		assert( !this->is_ambiguous_cst() && !other_cst.is_ambiguous_cst() );

		for ( int ii=0, ie=other_cst.group_cst_plist_[0].size(); ii<ie; ++ii ) {
			insert( other_cst.group_cst_plist_[0][ii] );
		}
		return *this;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Packer_cst_set::clear()
	{
		for ( Cst_rsd_map_map::const_iterator it1 = rsd_cst_set.begin(),
						it1_end = rsd_cst_set.end(); it1!= it1_end; ++it1 ) {
			Cst_rsd_map const & cst_map( it1->second );
			for ( Cst_rsd_map::const_iterator it2 = cst_map.begin(),
							it2_end = cst_map.end(); it2!= it2_end; ++it2 ) {
				Group_cst_plist const & group_csts ( it2->second );
				for( Group_cst_plist::const_iterator it3 = group_csts.begin(),
							it3_end = group_csts.end(); it3!= it3_end; ++it3 ) {
					(*it3)->clear();
				}
			}
		}
		rsd_cst_set.clear();
	}

	/////////////////////////////////////////////////////////////////////////////
	float
	Packer_cst_set::get_cstE( pose_ns::Pose const & pose ) const
	{
		kin::Coords_FArray_const coords( pose.full_coord() );
		float score( 0.0 );

		for ( Cst_rsd_map_map::const_iterator it1 = rsd_cst_set.begin(),
						it1_end = rsd_cst_set.end(); it1!= it1_end; ++it1 ) {
			Cst_rsd_map const & cst_map( it1->second );
			for ( Cst_rsd_map::const_iterator it2 = cst_map.begin(),
							it2_end = cst_map.end(); it2!= it2_end; ++it2 ) {
				Group_cst_plist const & group_csts ( it2->second );
				for( Group_cst_plist::const_iterator it3 = group_csts.begin(),
							it3_end = group_csts.end(); it3!= it3_end; ++it3 ) {
					std::vector<Rsd_rsd_cst*> cst_tmp;
					score+= (*it3)->get_best_score_cst(coords,cst_tmp,true);
				}
			}
		}
		return score;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Packer_cst_set::get_cst_set(
                pose_ns::Pose const & pose,
								cst_set_ns::Cst_set & cst,
								bool keep_cst_sd, //lin default is true, and keep cst_sd
								bool keep_cst //lin default is false, and erase old atom cst
	) const {

		int res1, res2, aa1, aa2, aav1, aav2;
		Aa_aav_pair pose_aa_pair;

		kin::Coords_FArray_const coords( pose.full_coord() );

		if( !keep_cst ) cst.erase_atom_constraints();

		for ( Cst_rsd_map_map::const_iterator it1 = rsd_cst_set.begin(),
						it1_end = rsd_cst_set.end(); it1!= it1_end; ++it1 ) {
			res1 = it1->first.first; res2 = it1->first.second;
			aa1 = pose.res( res1 ) ; aav1 = pose.res_variant( res1 );
      aa2 = pose.res( res2 ) ; aav2 = pose.res_variant( res2 );
			pose_aa_pair = std::make_pair( std::make_pair(aa1,aav1),
																     std::make_pair(aa2,aav2) );
			for ( Cst_rsd_map::const_iterator it2 = it1->second.begin(),
							it2_end = it1->second.end(); it2!= it2_end; ++it2 ) {
				Aa_aav_pair const & aa_pair ( it2->first );
				if( aa_pair == pose_aa_pair ) {
					Group_cst_plist const & group_csts ( it2->second );
					for( Group_cst_plist::const_iterator it3 = group_csts.begin(),
							it3_end = group_csts.end(); it3!= it3_end; ++it3 ) {
						std::vector<Rsd_rsd_cst*> cst_plist;
						(*it3)->get_best_score_cst(coords,cst_plist,false);
						//go through cst_tmp list to fill the cst set
						for( int ii=0, ie=cst_plist.size(); ii<ie; ii++ ) {
							switch ( cst_plist[ii]->cst_type() ) {
								case DISTANCE_CST:
								{
									Rsd_rsd_distance_cst* distance_cst_p =
										static_cast<Rsd_rsd_distance_cst*> ( cst_plist[ii] );
									cst.add_atompair_constraint( distance_cst_p->atom1, distance_cst_p->atom2, distance_cst_p->cst_set );
									break;
								}
								case ANGLE_CST:
								{
									Rsd_rsd_angle_cst* angle_cst_p =
										static_cast<Rsd_rsd_angle_cst*> ( cst_plist[ii] );
									cst.add_atom_angle_constraint( angle_cst_p->cst_set );
									break;
								}
								case TORSION_CST:
								{
									Rsd_rsd_torsion_cst* torsion_cst_p =
										static_cast<Rsd_rsd_torsion_cst*> ( cst_plist[ii] );
                  cst.add_atom_torsion_constraint( torsion_cst_p->cst_set );
									break;
								}
							}
						}
					}
				}
			}
		}

		if( !keep_cst_sd ) cst.zero_atompair_cst_sd();
	}

	void
	Packer_cst_set::cst_take_input_value( pose_ns::Pose & pose ) {
		for ( Cst_rsd_map_map::const_iterator it1 = rsd_cst_set.begin(),
						it1_end = rsd_cst_set.end(); it1!= it1_end; ++it1 ) {
			Cst_rsd_map const & cst_map( it1->second );
			for ( Cst_rsd_map::const_iterator it2 = cst_map.begin(),
							it2_end = cst_map.end(); it2!= it2_end; ++it2 ) {
				Group_cst_plist const & group_csts ( it2->second );
				for( Group_cst_plist::const_iterator it3 = group_csts.begin(),
							it3_end = group_csts.end(); it3!= it3_end; ++it3 ) {
					for ( int ii=0, ie=(*it3)->cst_list().size(); ii<ie; ++ii ) {
						for( int jj=0, je=((*it3)->cst_list())[ii].size(); jj<je; ++jj ) {
							Rsd_rsd_cst* it4 ( ((*it3)->cst_list())[ii][jj] );
							it4->take_input_value(pose);
						}
					}
				}
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// method to assume another packer constraints
	Packer_cst_set &
	Packer_cst_set::collect( Packer_cst_set const & other_cst ) {

		if( !rsd_cst_set.empty() ) {
			std::cout<<"danger!!!!"<<"\n";
			//assert( false );
		}

		assert( packer_cst_weight == other_cst.packer_cst_weight );

		for ( Cst_rsd_map_map::const_iterator it1 = other_cst.rsd_cst_set.begin(),
						it1_end = other_cst.rsd_cst_set.end(); it1!= it1_end; ++it1 ) {
			// add to the map by residue pairs
			if ( !rsd_cst_set.count( it1->first ) ) {
				rsd_cst_set.insert( *it1 );
			} else {
				for ( Cst_rsd_map::const_iterator it2 = it1->second.begin(),
							it2_end = it1->second.end(); it2!= it2_end; ++it2 ) {
					// add to the map by atom pair (aa,aav)
					if ( !rsd_cst_set.find( it1->first )->second.count( it2->first ) ){
						rsd_cst_set.find( it1->first )->second.insert( *it2 );
					} else {
						for( std::vector<Group_rsd_rsd_cst*>::const_iterator it3 = it2->second.begin(),
									 it3_end = it2->second.end(); it3!= it3_end; ++it3 ) {
						rsd_cst_set.find( it1->first )->second.find( it2->first )->second.push_back( *it3 );
						}
					}
				}
			}
		}
		//clean_up();

		return *this;
	}

	/////////////////////////////////////////////////////////////////////////////
  void Packer_cst_set::clean_up( ) {
		for ( Cst_rsd_map_map::iterator it1 = rsd_cst_set.begin(),
						it1_end = rsd_cst_set.end(); it1!= it1_end; ++it1 ) {
			for ( Cst_rsd_map::iterator it2 = it1->second.begin(),
							it2_end = it1->second.end(); it2!= it2_end; ++it2 ) {
				Group_rsd_rsd_cst* base_cst_p ( 0 );
				for( std::vector<Group_rsd_rsd_cst*>::iterator it3 = it2->second.begin(),
							 it3_end = it2->second.end(); it3!= it3_end; ++it3 ) {
					if( ! (*it3)->is_ambiguous_cst() ){
						if( base_cst_p ==  0 ) {
							base_cst_p = *it3;
						} else {
							base_cst_p->collect( **(it3) );
							(*it3)->clear();
							it2->second.erase(it3);
							it3--;
							it3_end--;
						}
					}
				}
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	float
	Packer_cst_set::get_res_res_cstE(
		int const res1,
		int const res2,
		int const aa1,
		int const aav1,
		int const aa2,
		int const aav2,
		FArray2Da_float coord1,
		FArray2Da_float coord2,
		bool const bb1,
		bool const sc1,
		bool const bb2,
		bool const sc2
	) const
	{

		//std::cout<<"call get_res_res_cstE"<<"\n";
		if( packer_cst_weight == 0. ) return 0;
		if( res1 == res2 && !( aa1==aa1 && aav1==aav2 ) ) return 0;

		Rsd_pair pos_pair( std::min(res1,res2), std::max(res1,res2) );
		float energy (0);

		//lin go for the order res1 < res2
		if( res1 == pos_pair.first ) {
			energy = get_rsd_pair_cstE
					(pos_pair,aa1,aav1,aa2,aav2,coord1,coord2,bb1,sc1,bb2,sc2);
		} else {
			energy = get_rsd_pair_cstE
					(pos_pair,aa2,aav2,aa1,aav1,coord2,coord1,bb2,sc2,bb1,sc1);
		}
		//std::cout<<"res_res_cstE: " << aa1 << " " << res1 << " " << aa2 << " " << res2 << " "<<energy<<"\n";
		return energy;
	}

	/////////////////////////////////////////////////////////////////////////////
	float
	Packer_cst_set::get_rsd_pair_cstE(
		Rsd_pair const &  rsd_pair,
		int const aa1,
		int const aav1,
		int const aa2,
		int const aav2,
		FArray2Da_float coord1,
		FArray2Da_float coord2,
		bool const bb1,
		bool const sc1,
		bool const bb2,
		bool const sc2
	) const
	{

		//lin combine the coord1 and coord2 to coord( 3, MAX_ATOM(), 2 )
		coord1.dimension( 3, param::MAX_ATOM() );
		coord2.dimension( 3, param::MAX_ATOM() );


		float score( 0.0 );
		if( rsd_cst_set.count( rsd_pair ) ) {

			Aa_aav_pair aa_pair ( std::make_pair( std::make_pair(aa1,aav1),
																						std::make_pair(aa2,aav2) ) );

			if ( rsd_cst_set.find( rsd_pair )->second.count( aa_pair ) ){
				FArray3D_float coord( 3, param::MAX_ATOM(), 2 );

				//speed up the following code
				for( int l=coord1.index(1,1), le=coord1.index(3,param::MAX_ATOM());
						 l<=le; l++ ) {
					coord[l]=coord1[l]; coord[l+le+1]=coord2[l];
				}
// 				for( int ii=1, ie=param::MAX_ATOM(); ii<=ie; ii++)
// 					for (int jj=1; jj<=3; jj++ ) {
// 						coord(jj,ii,1) = coord1(jj,ii);
// 						coord(jj,ii,2) = coord2(jj,ii);
// 					}

				Group_cst_plist const & cst_list
					( rsd_cst_set.find( rsd_pair )->second.find( aa_pair )->second );
				for ( Group_cst_plist::const_iterator it=cst_list.begin(),
								it_end = cst_list.end(); it != it_end; ++it ) {
					score += (*it)->best_packer_score
						(aa1,aav1,aa2,aav2,coord,bb1,sc1,bb2,sc2);
				}
				//std::cout<<"total: aa_aav_pair " << aa1 << " " << aa2 <<" res_pair: "
				//				 <<rsd_pair.first<<" "<<rsd_pair.second<< " score: "<< score<<"\n";
			} else {
				score = 5000.;
			}
		}
		return packer_cst_weight*score;
	}

  /////////////////////////////////////////////////////////////////////////////
  // add a new atompair constraint into rsd_rsd_cst
  void
  Packer_cst_set::add_distance_constraint(
      kin::Fullatom_id const & atom1,
      kin::Fullatom_id const & atom2,
      float const r0_in,
			float const r_sd_in,
			float const weight_in ) {

		if ( atom1.key() == atom2.key() ) {
			std::cout<<"WARNING: Packer_cst_set::add_distance_constraint atom1 == atom2, skipping it... "<<"\n";
			return;
		}

		// add the distance constraint
		cst_set_ns::Cst cst( r0_in, r_sd_in, weight_in );
		Rsd_rsd_cst* rr_cst_p;
		rr_cst_p = new Rsd_rsd_distance_cst( atom1, atom2, cst );
		add_rr_constraint( atom1, atom2, rr_cst_p );
	}

	void
  Packer_cst_set::add_angle_constraint(
			kin::Fullatom_id const & atom1,
			kin::Fullatom_id const & atom2,
			kin::Fullatom_id const & atom3,
			float const theta0_in,
			float const theta_sd_in,
			float const periodic_value_in,
			float const weight_in ) {

		//safte check atom2 should share the same aa,aav with either atom1 or atom3
		if( !(atom2 == atom1) && !(atom2 == atom3) ) { // == is comparing only seqpos,aa,aav
			std::cout<<"WARNING: Packer_cst_set::add_angle_constraint atom2 "<<atom2
							 <<" doesnt match wtih atom1 " << atom1 <<" or atom3 "<<atom3
							 <<" , skipping it..."<<"\n";
			return;
		}

		// add the angle constraint
		cst_set_ns::Angle_cst cst(atom1, atom2, atom3, theta0_in,
															theta_sd_in, periodic_value_in, weight_in );
		Rsd_rsd_cst* rr_cst_p;
		rr_cst_p = new Rsd_rsd_angle_cst( cst );
		add_rr_constraint( atom1, atom3, rr_cst_p );
	}

	void
	Packer_cst_set::add_torsion_constraint(
	    kin::Fullatom_id const & atom1,
			kin::Fullatom_id const & atom2,
			kin::Fullatom_id const & atom3,
			kin::Fullatom_id const & atom4,
			float const theta0_in,
			float const theta_sd_in,
			float const periodic_value_in,
			float const weight_in,
			std::string const & func_name ) {//default is HARMONIC

		//safte check atom2 or atom3 should share the same aa,aav with either atom1 or atom4
		if( !(atom2 == atom1) && !(atom2 == atom4) ) {// == is comparing only seqpos,aa,aav
			std::cout<<"WARNING: Packer_cst_set::add_torsion_constraint atom2 " << atom2
							 <<" doesnt match wtih atom1 " << atom1 << " or atom4 " << atom4
							 <<" , shipping it..."<<"\n";
			return;
		}
		if( !(atom3 == atom1) && !(atom3 == atom4) ) {// == is comparing only seqpos,aa,aav
			std::cout<<"WARNING: Packer_cst_set::add_torsion_constraint atom3 "<< atom3
							 <<" doesnt match wtih atom1 "<< atom1 <<" or atom4 " << atom4
							 <<" , skipping it..."<<"\n";
			return;
		}

		// add the torsion constraint
		cst_set_ns::Torsion_cst cst(atom1, atom2, atom3, atom4, theta0_in,
															theta_sd_in, periodic_value_in, weight_in, func_name );
		Rsd_rsd_cst* rr_cst_p;
		rr_cst_p = new Rsd_rsd_torsion_cst( cst );
		add_rr_constraint( atom1, atom4, rr_cst_p );
	}

  /////////////////////////////////////////////////////////////////////////////
  // add a new atompair constraint into rsd_rsd_cst
  void
  Packer_cst_set::add_distance_constraint(
			pose_ns::Pose const & pose,
      kin::Atom_id const & atom1,
      kin::Atom_id const & atom2,
			cst_set_ns::Cst const & cst ) {

		kin::Fullatom_id const & p_a1( pose.get_fullatom_id( atom1 ) );
		kin::Fullatom_id const & p_a2( pose.get_fullatom_id( atom2 ) );

		if ( p_a1.key() == p_a2.key() ) {
			std::cout<<"ERROR: Packer_cst_set::add_distance_constraint atom1 == atom2, skipping it... "<<"\n";
			return;
		}

		// add the distance constraint
		Rsd_rsd_cst* rr_cst_p;
		rr_cst_p = new Rsd_rsd_distance_cst( p_a1, p_a2, cst );
		add_rr_constraint( p_a1, p_a2, rr_cst_p );
	}

	void
  Packer_cst_set::add_angle_constraint(
			pose_ns::Pose const & pose,
			cst_set_ns::Angle_cst cst ) {

		kin::Atom_id atom1, atom2, atom3;
		cst.extract_atom_ids(atom1,atom2,atom3);

		kin::Fullatom_id const & p_a1( pose.get_fullatom_id( atom1 ) );
		kin::Fullatom_id const & p_a2( pose.get_fullatom_id( atom2 ) );
		kin::Fullatom_id const & p_a3( pose.get_fullatom_id( atom3 ) );

		//saftey check atom2 should share the same aa,aav with either atom1 or atom3
		if( p_a2 == p_a1 || p_a2 == p_a3 ) { // == is comparing only seqpos,aa,aav
			// add the angle constraint
			Rsd_rsd_cst* rr_cst_p;
			rr_cst_p = new Rsd_rsd_angle_cst( cst );
			add_rr_constraint( p_a1, p_a3, rr_cst_p );
		} else {//angle involve > 2 residue
			std::cout<<" agnle cst atom involves > 2 residues, skipping it"<<"\n";
		}
	}

	void
	Packer_cst_set::add_torsion_constraint(
			pose_ns::Pose const & pose,
			cst_set_ns::Torsion_cst const & cst ) {

		kin::Atom_id atom1, atom2, atom3, atom4;
		cst.extract_atom_ids(atom1,atom2,atom3,atom4);

		kin::Fullatom_id const & p_a1( pose.get_fullatom_id( atom1 ) );
		kin::Fullatom_id const & p_a2( pose.get_fullatom_id( atom2 ) );
		kin::Fullatom_id const & p_a3( pose.get_fullatom_id( atom3 ) );
		kin::Fullatom_id const & p_a4( pose.get_fullatom_id( atom4 ) );

		//saftey check atom2 or atom3 should share the same aa,aav with either atom1 or atom4
		if( ( p_a2 == p_a1 || p_a2 == p_a4 )
				&& ( p_a3 == p_a1 || p_a3 == p_a4 ) ) {

			// add the torsion constraint
			Rsd_rsd_cst* rr_cst_p;
			rr_cst_p = new Rsd_rsd_torsion_cst( cst );
			add_rr_constraint( p_a1, p_a4, rr_cst_p );
		}else {//angle involve > 2 residue
			std::cout<<" torsion cst atom involves > 2 residues, skipping it"<<"\n";
		}

	}

  void
  Packer_cst_set::add_rr_constraint(
      kin::Fullatom_id const & atom1,
      kin::Fullatom_id const & atom2,
			Rsd_rsd_cst* const & rr_cst_p,
			bool cst_ambiguous_group //lin default is false, if cst is ambiguous group, use true
  )
  {
		assert( atom1.key() != atom2.key() );
	  Rsd_pair pos_pair;
		Aa_aav_pair aa_pair;

		//lin make pair atom a, and a.first< a.second
		if( atom1.key() < atom2.key() ) {
			pos_pair = std::make_pair( atom1.rsd(), atom2.rsd() );
			aa_pair = std::make_pair( std::make_pair( atom1.aa(), atom1.aav() ),
																	std::make_pair( atom2.aa(), atom2.aav() ) );
		}else{
			pos_pair = std::make_pair( atom2.rsd(), atom1.rsd() );
			aa_pair = std::make_pair( std::make_pair( atom2.aa(), atom2.aav() ),
																	std::make_pair( atom1.aa(), atom1.aav() ) );
		}

    // add to the map by residue pairs
    if ( !rsd_cst_set.count( pos_pair ) ) {
			Cst_rsd_map tmp;
			rsd_cst_set.insert( std::make_pair( pos_pair, tmp ) );
		}
		// add to the map by atom pair (aa,aav)
		if ( !rsd_cst_set.find( pos_pair )->second.count( aa_pair ) ){
			rsd_cst_set.find(pos_pair)->second.insert
					( std::make_pair( aa_pair, std::vector< Group_rsd_rsd_cst* >() ) );
		}

		// add the constraint
		if( cst_ambiguous_group &&
				!rsd_cst_set.find( pos_pair )->second.find( aa_pair )->second.empty() ) {
			//lin if cst is the same group as the previous one
			//lin insert the cst into the last group_rsd_rsd_cst
			Group_rsd_rsd_cst* group_cst_p
				( rsd_cst_set.find( pos_pair )->second.find( aa_pair )->second.back() );
			Rsd_cst_plist cst_plist_tmp;
			cst_plist_tmp.push_back( rr_cst_p );
			group_cst_p->insert( cst_plist_tmp, true );
		} else {
			Group_rsd_rsd_cst* group_cst_p;
			group_cst_p = new Group_rsd_rsd_cst;
			group_cst_p->insert( rr_cst_p );
			rsd_cst_set.find( pos_pair )->second.find( aa_pair )
				->second.push_back( group_cst_p );
		}
  }

  /////////////////////////////////////////////////////////////////////////////
  // add a new constraint list into rsd_rsd_cst
  void
  Packer_cst_set::add_rr_constraint_list(
      kin::Fullatom_id const & atom1,
      kin::Fullatom_id const & atom2,
			std::vector< Rsd_rsd_cst* > const & cst_plist,
			bool cst_ambiguous_group //lin default is false, if cst is ambiguous group, use true
			) {

		assert( atom1.key() != atom2.key() );
	  Rsd_pair pos_pair;
		Aa_aav_pair aa_pair;

		//lin make pair atom a, and a.first< a.second
		if( atom1.key() < atom2.key() ) {
			pos_pair = std::make_pair( atom1.rsd(), atom2.rsd() );
			aa_pair = std::make_pair( std::make_pair( atom1.aa(), atom1.aav() ),
																	std::make_pair( atom2.aa(), atom2.aav() ) );
		}else{
			pos_pair = std::make_pair( atom2.rsd(), atom1.rsd() );
			aa_pair = std::make_pair( std::make_pair( atom2.aa(), atom2.aav() ),
																	std::make_pair( atom1.aa(), atom1.aav() ) );
		}

    // add to the map by residue pairs
    if ( !rsd_cst_set.count( pos_pair ) ) {
			Cst_rsd_map tmp;
			rsd_cst_set.insert( std::make_pair( pos_pair, tmp ) );
		}
		// add to the map by atom pair (aa,aav)
		if ( !rsd_cst_set.find( pos_pair )->second.count( aa_pair ) ){
			rsd_cst_set.find(pos_pair)->second.insert
					( std::make_pair( aa_pair, std::vector< Group_rsd_rsd_cst* >() ) );
		}

		// add the constraint
		if( cst_ambiguous_group &&
				!rsd_cst_set.find( pos_pair )->second.find( aa_pair )->second.empty() ) {
			//lin if cst is the same group as the previous one
			//lin insert the cst into the last group_rsd_rsd_cst
			Group_rsd_rsd_cst* group_cst_p
				( rsd_cst_set.find( pos_pair )->second.find( aa_pair )->second.back() );
			group_cst_p->insert( cst_plist, true );
		} else {
			Group_rsd_rsd_cst* group_cst_p;
			group_cst_p = new Group_rsd_rsd_cst;
			group_cst_p->insert( cst_plist, false );
			rsd_cst_set.find( pos_pair )->second.find( aa_pair )
				->second.push_back( group_cst_p );
		}
	}

  /////////////////////////////////////////////////////////////////////////////
  // lin methods for rsd_rsd_cst
	int Rsd_rsd_distance_cst::atomno1() const {
		if( index1_ == 1 ) {
			return atomno1_;
		} else if ( index2_ == 1 ) {
			return atomno2_;
		} else {
			std::cout<<"ERROR: cant find the key atom1 "<<"\n";
			assert(false);
			return 0;
		}
	}

	int Rsd_rsd_distance_cst::atomno2() const {
		if( index1_ == 2 ) {
			return atomno1_;
		} else if ( index2_ == 2 ) {
			return atomno2_;
		} else {
			std::cout<<"ERROR: cant find the key atom1 "<<"\n";
			assert(false);
			return 0;
		}
	}

	void Rsd_rsd_distance_cst::assign(
       kin::Atom_id const & a1,
			 kin::Atom_id const & a2,
			 cst_set_ns::Cst const & cst_in ) {

		atom1 = a1 ;
		atom2 = a2 ;
		cst_set = cst_in ;

		assert( a1 != a2 );

		atomno1_ = a1.atomno();
		atomno2_ = a2.atomno();

		// key atom1 < atom2
		if( a1 < a2 ) {
			index1_ = 1 ; index2_ = 2 ;
		} else {
			index1_ = 2 ; index2_ = 1 ;
		}
	}

	int Rsd_rsd_angle_cst::atomno1() const {
		if( index1_ == 1 ) {
			return atomno1_;
		} else if ( index3_ == 1 ) {
			return atomno3_;
		} else {
			std::cout<<"ERROR: cant find the key atom1 "<<"\n";
			assert(false);
			return 0;
		}
	}

	int Rsd_rsd_angle_cst::atomno2() const {
		if( index1_ == 2 ) {
			return atomno1_;
		} else if ( index3_ == 2 ) {
			return atomno3_;
		} else {
			std::cout<<"ERROR: cant find the key atom1 "<<"\n";
			assert(false);
			return 0;
		}
	}

	Rsd_rsd_angle_cst::Rsd_rsd_angle_cst( cst_set_ns::Angle_cst const & cst_set_in ):
		cst_set( cst_set_in ) {

		kin::Atom_id a1, a2, a3;
		cst_set_in.extract_atom_ids( a1, a2, a3 );
		assert( a1 != a3 );

		atomno1_=a1.atomno();
		atomno2_=a2.atomno();
		atomno3_=a3.atomno();

		if( a1 < a3 ) {
			index1_ = 1 ; index3_ = 2 ;
		} else {
			index1_ = 2 ; index3_ = 1 ;
		}

		//lin decide a2 belong to a1.res or a3.res
		if( a1.rsd() == a2.rsd() ) {
			index2_=index1_;
		} else if ( a3.rsd() == a2.rsd() ) {
			index2_=index3_;
		} else {
			std::cout<<"Error!!Rsd_rsd_angle_cst atom2 matches with neither atom1 nor atom3"<<"\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	int Rsd_rsd_torsion_cst::atomno1() const {
		if( index1_ == 1 ) {
			return atomno1_;
		} else if ( index4_ == 1 ) {
			return atomno4_;
		} else {
			std::cout<<"ERROR: cant find the key atom1 "<<"\n";
			assert(false);
			return 0;
		}
	}

	int Rsd_rsd_torsion_cst::atomno2() const {
		if( index1_ == 2 ) {
			return atomno1_;
		} else if ( index4_ == 2 ) {
			return atomno4_;
		} else {
			std::cout<<"ERROR: cant find the key atom1 "<<"\n";
			assert(false);
			return 0;
		}
	}

	Rsd_rsd_torsion_cst::Rsd_rsd_torsion_cst(cst_set_ns::Torsion_cst const & cst_set_in):
		cst_set( cst_set_in ) {

		kin::Atom_id a1, a2, a3, a4;
		cst_set_in.extract_atom_ids( a1, a2, a3, a4 );
		assert( a1 != a4 );

		atomno1_=a1.atomno();
		atomno2_=a2.atomno();
		atomno3_=a3.atomno();
		atomno4_=a4.atomno();

		if( a1 < a4 ) {
			index1_ = 1 ; index4_ = 2 ;
		} else {
			index1_ = 2 ; index4_ = 1 ;
		}

		//lin decide a2 belong to a1.res or a4.res
		if( a2.rsd() == a1.rsd() ) {
			index2_=index1_;
		} else if ( a2.rsd() == a4.rsd() ) {
			index2_=index4_;
		} else {
			std::cout<<"Error!!Rsd_rsd_angle_cst atom2 matches with neither atom1 nor atom4"<<"\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		if( a3.rsd() == a1.rsd() ) {
			index3_=index1_;
		} else if ( a3.rsd() == a4.rsd() ) {
			index3_=index4_;
		} else {
			std::cout<<"Error!!Rsd_rsd_angle_cst atom3 matches with neither atom1 nor atom4"<<"\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

	}

  /////////////////////////////////////////////////////////////////////////////
  //lin  rsd_rsd constraint score method

  /////////////////////////////////////////////////////////////////////////////
	//lin  score method for pose
	//lin   distance constraint rsd_rsd score
  float Rsd_rsd_distance_cst::score( kin::Coords const & coords ) const
	{
		return cst_set.score( coords.get_xyz(atom1), coords.get_xyz(atom2) );
	}
  /////////////////////////////////////////////////////////////////////////////
  //lin  angle rsd_rsd constraint score method
  float Rsd_rsd_angle_cst::score( kin::Coords const & coords ) const
	{
		return cst_set.score( coords );
	}
  /////////////////////////////////////////////////////////////////////////////
  //lin  torsion rsd_rsd constraint score method
  float Rsd_rsd_torsion_cst::score( kin::Coords const & coords ) const
	{
		return cst_set.score( coords );
	}

  /////////////////////////////////////////////////////////////////////////////
	//lin  score method for packer
  /////////////////////////////////////////////////////////////////////////////
	//lin    distance constraint rsd_rsd score
  float Rsd_rsd_distance_cst::score( FArray3D_float const & coord ) const
	{
    numeric::xyzVector_float xyz1( &coord(1,atomno1_,index1_) );
    numeric::xyzVector_float xyz2( &coord(1,atomno2_,index2_) );

    return cst_set.score( xyz1, xyz2 );
	}

  /////////////////////////////////////////////////////////////////////////////
  //lin  angle rsd_rsd constraint score method
	float Rsd_rsd_angle_cst::score( FArray3D_float const & coord ) const
	{
    numeric::xyzVector_float xyz1( &coord(1,atomno1_,index1_) );
    numeric::xyzVector_float xyz2( &coord(1,atomno2_,index2_) );
    numeric::xyzVector_float xyz3( &coord(1,atomno3_,index3_) );

    return cst_set.score( xyz1, xyz2, xyz3 );
	}

  /////////////////////////////////////////////////////////////////////////////
  //lin  torsion rsd_rsd constraint score method
	float Rsd_rsd_torsion_cst::score( FArray3D_float const & coord ) const
	{
		numeric::xyzVector_float xyz1( &coord(1,atomno1_,index1_) );
    numeric::xyzVector_float xyz2( &coord(1,atomno2_,index2_) );
    numeric::xyzVector_float xyz3( &coord(1,atomno3_,index3_) );
    numeric::xyzVector_float xyz4( &coord(1,atomno4_,index4_) );

		return cst_set.score( xyz1, xyz2, xyz3, xyz4 );
	}

	/////////////////////////////////////////////////////////////////////////////
  void Rsd_rsd_distance_cst::show( std::ostream & s ) const {
    s << "\t\t\t\trsd_rsd_cst:" << cst_type() << "\n";
		s << "\t\t\t\t\tcst_set: " <<cst_set;
		s << "\t\t\t\t\t\tDISTANCE_ids: "<< atom1 <<" - "<<atom2<<"\n";
		s << "\t\t\t\t\tatomno: "<<atomno1_<<" "<<atomno2_;
		s << " index: "<<index1_<<" "<<index2_;
		s << " key atom: "<< atomno1() << " "<<atomno2()<<"\n";
	}

	/////////////////////////////////////////////////////////////////////////////
  void Rsd_rsd_angle_cst::show( std::ostream & s ) const {
    s << "\t\t\t\trsd_rsd_cst:" << cst_type() << "\n";
		s << "\t\t\t\t\t"<<cst_set;
		s << "\t\t\t\t\tatomno: "<<atomno1_<<" "<<atomno2_<<" "<<atomno3_;
		s << " index: "<<index1_<<" "<<index2_<<" "<<index3_;
		s << " key atom: "<< atomno1() << " "<<atomno2()<<"\n";
	}

	/////////////////////////////////////////////////////////////////////////////
  void Rsd_rsd_torsion_cst::show( std::ostream & s ) const {
    s << "\t\t\t\trsd_rsd_cst:" << cst_type() << "\n";
		s << "\t\t\t\t\tcst_set: "<<cst_set;
		s << "\t\t\t\t\tatomno: "<<atomno1_<<" "<<atomno2_<<" "<<atomno3_<<" "<<atomno4_;
		s << " index: "<<index1_<<" "<<index2_<<" "<<index3_<<" "<<index4_;
		s << " key atom: "<< atomno1() << " "<<atomno2()<<"\n";
	}

	/////////////////////////////////////////////////////////////////////////////
  void Group_rsd_rsd_cst::show( std::ostream & s ) const {
    s << "\t\t\t\tGroup_rsd_rsd_cst:" << "\n";
		for ( int ii=0, ie=group_cst_plist_.size(); ii<ie; ++ii ) {
			s << "\t\t\t\t\tgroup_cst_plist[ " << ii <<" ]" <<"\n";
			for( int jj=0, je=group_cst_plist_[ii].size(); jj<je; ++jj ) {
				Rsd_rsd_cst* const it ( group_cst_plist_[ii][jj] );
				it->show(s);
			}
		}
	}

	////////////////////////////////////////////////////////////////////////////
  std::ostream& operator<< ( std::ostream& s, Packer_cst_set const & d ) {
    s << "PACKER CST SET INFORMATION:" << "\n";
		s << "\t packer_cst_weight = " << d.packer_cst_weight <<"\n";
		s << "\t cst_rsd_map_map: " <<"\n";
		for ( Cst_rsd_map_map::const_iterator it1 = d.rsd_cst_set.begin(),
						it1_end = d.rsd_cst_set.end(); it1!= it1_end; ++it1 ) {
			s << "\t\t rsd_pair< "<<it1->first.first<<" , "
				<<it1->first.second<<" >"<<"\n";
			for ( Cst_rsd_map::const_iterator it2 = it1->second.begin(),
							it2_end = it1->second.end(); it2!= it2_end; ++it2 ) {
			s << "\t\t\t aa_aav_pair< ("<<it2->first.first.first<<","
				<<it2->first.first.second<<")  ("<<it2->first.second.first<<","
				<<it2->first.second.second<<") >"<<"\n";
				for( std::vector<Group_rsd_rsd_cst*>::const_iterator it3 = it2->second.begin(),
							 it3_end = it2->second.end(); it3!= it3_end; ++it3 ) {
					(*it3)->show(s);
				}
			}
		}
    return s;
	}

	/////////////////////////////////////////////////////////////////////////////
	void Rsd_rsd_distance_cst::take_input_value( pose_ns::Pose & pose ) {
		if( cst_set.use_input_value() ) {
			float dis ( pose.get_distance( atom1, atom2 ) );
			cst_set.set_value( dis );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void Rsd_rsd_angle_cst::take_input_value( pose_ns::Pose & pose ) {
		if( cst_set.use_input_value() ) {
			kin::Atom_id atom1, atom2, atom3;
			cst_set.extract_atom_ids( atom1, atom2, atom3 );
			float theta ( pose.get_angle( atom1, atom2, atom3 ) );
			cst_set.set_value( theta );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void Rsd_rsd_torsion_cst::take_input_value( pose_ns::Pose & pose ) {
		if( cst_set.use_input_value() ) {
			kin::Atom_id atom1, atom2, atom3, atom4;
			cst_set.extract_atom_ids( atom1, atom2, atom3, atom4 );
			float theta ( pose.get_torsion_angle( atom1, atom2, atom3, atom4 ) );
			cst_set.set_value( theta );
		}
	}

}

