// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14367 $
//  $Date: 2007-04-19 12:08:54 +0300 (Thu, 19 Apr 2007) $
//  $Author: yab $

// Rosetta Headers
#include "aa_name_conversion.h"
#include "decoystats.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "are_they_neighbors.h"
#include "atom_is_backbone.h"
#include "decoystats_classes.h"
#include "decoystats_interface.h"
#include "decoystats_ns.h"
#include "design.h"
#include "files_paths.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "fullatom_sasa.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "initialize.h"
#include "make_pdb.h"
#include "maxsub_threshold.h"
#include "misc.h"
#include "options.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack.h"
#include "param.h"
#include "param_aa.h"
#include "pdbstatistics_pack.h"
#include "pose.h"
#include "pose_io.h"
#include "template_pack.h"
#include "util_basic.h"
#include "util_vector.h"
#include "void.h"
#include "void_ns.h"
#include "water.h"
#include "packing_measures.h"
//KMa phospho_ser
#include "add_pser.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/irstream.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <sstream>
#include <string>
#include <list>

// Namespaces

// namespace used by decoystats_analyze_fragments, local_bb_counter,
// output_top_local_bb to calculate frequencies for the native combination in
// the fragments, then the values are erased and used to hold the decoy counts.

namespace bb_count_common {
	using namespace param;
	bool got_native_counters = { false };
	int const max_bb = { 5 };
	FArray2D_int bb_count1( max_bb, MAX_RES() );
	FArray3D_int bb_count2( max_bb, max_bb, MAX_RES() );
	FArray4D_int bb_count3( max_bb, max_bb, max_bb, MAX_RES() );
	FArray2D_int total_count( MAX_RES(), 3 );
	FArray1D_int native_bb( MAX_RES() );
}

namespace decoystats_flag_common {
	bool decoystats_flag = { false };
	bool ds_outpdbonly_flag = { false };
}

namespace native_is_stored {
	bool stored = { false };
}

namespace ds_compare_to_native_init {
	bool init = { false };
}

namespace fa_atr_counter_init {
	bool init = { false };
}

namespace hb_counter_init {
	bool init = { false };
}

namespace local_bb_counter_init {
	bool init = { false };
}

namespace store_output_line_common {
	int const max_output_lines = { 10000 };
	int n_lines;
	FArray1D_string output_lines( max_output_lines );
}

namespace use_big_polar_h_common {
	bool use_big_polar_h = { false };
}

// functions for comparing decoys to native(s)

// currently only works with -score mode
// where we are getting stats on
// a bunch of re-scored decoys/pdbs
//
// the stats are currently written to stderr

//////////////////////////////////////////////////////////////////////////////
/// @begin setup_decoystats
///
/// @brief
///
/// @detailed
///
/// @param  mode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
setup_decoystats()
{
	using namespace decoystats;
	using namespace decoystats_flag_common;
	using namespace files_paths;

	if ( truefalseoption("interface_ds") ) {
		set_interface_ds_flag(true);
		enable_multi_chain();
		if ( truefalseoption("find_CBCB") ) {
			find_interface_CBCB = true;
			std::ofstream cbcbOUT("cbcb.txt");
			cbcbOUT.close();
		}

	} else if ( truefalseoption("decoystats") ) {
		set_decoystats_flag(true);
	} else {
		set_decoystats_flag(false);
		return;
	}

	if ( truefalseoption("ds_outpdb_only") ) {
		set_ds_outpdbonly_flag(true);
	}

	if ( truefalseoption("tight_core_analysis") ) {
		tight_core_analysis=true;
	}

// don't trim the filenames
	scorefile_output_full_filename = true; // files_paths.h

	reset_counters();

	dimension_big_decoystats_arrays();
}

void
dimension_big_decoystats_arrays()
{
	static bool init( false );
	if ( init ) return;
	init = true;

	using namespace param;
	using namespace decoystats;
	using namespace decoystats::decoystats_atom_set_common;
	using namespace decoystats::decoystats_native_common;
	using namespace decoystats::decoystats_counter_common;

	// dimension big arrays
	// these dont appear to be used anymore:
	//native_hbondE.dimension( MAX_RES(), MAX_RES(), 4 );
	//native_lj_atr.dimension( MAX_RES(), MAX_RES() );

	atom_set.dimension( MAX_ATOM(), MAX_RES(), n_atom_set );

	rsd_pair_hbondE.dimension( MAX_RES(), MAX_RES(), 4 );
	pair_count.dimension( MAX_RES(), MAX_RES(), max_pair_count );
	atom_count.dimension( MAX_ATOM(), MAX_RES(), max_atom_count );


}

//////////////////////////////////////////////////////////////////////////////
/// @begin set_decoystats_flag
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_decoystats_flag( bool setting )
{
	using namespace decoystats_flag_common;

	std::cout << "set_decoystats_flag: from,to" <<
	 SS( decoystats_flag ) << SS( setting ) << std::endl;
	decoystats_flag = setting;
	if ( setting ) dimension_big_decoystats_arrays();

}

//////////////////////////////////////////////////////////////////////////////
/// @begin set_ds_outpdbonly_flag
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_ds_outpdbonly_flag( bool setting )
{
	using namespace decoystats_flag_common;

	ds_outpdbonly_flag = setting;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_decoystats_flag
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
get_decoystats_flag()
{
	using namespace decoystats_flag_common;

	return decoystats_flag;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_ds_outpdbonly_flag
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
get_ds_outpdbonly_flag()
{
	using namespace decoystats_flag_common;

	return ds_outpdbonly_flag;
}


//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
// END OF INTERFACE
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

//////////////////////////////////////////////////////////////////////////////
/// @begin reset_decoy_scores
///
/// @brief
///
/// @detailed
///
/// @param  tag - [in/out]? -
/// @param  sc - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
reset_decoy_scores()
{
	using namespace decoystats;

	n_score_tags = 0;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin add_decoy_score
///
/// @brief
///
/// @detailed
///
/// @param  tag - [in/out]? -
/// @param  sc - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
add_decoy_score(
	std::string const & tag,
	float const sc
)
{
	using namespace decoystats;

	++n_score_tags;
	if ( n_score_tags > max_score_tags ) {
		std::cout << "STOP:: too many score tags!" << SS( n_score_tags ) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	score_tags(n_score_tags) = tag;
	decoystats_score(n_score_tags) = sc;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_decoy_score
///
/// @brief return score for a given tag id
///
/// @detailed
///
/// @param[in] - tag - string id of the score you want
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @refrences
///
/// @authors Glenn Butterfoss
///
/// @last_modified 24 June 2005
////////////////////////////////////////////////////////////////////////////////
float
get_decoy_score(
	std::string const & tag
)
{
	using namespace decoystats;

	if ( !get_decoystats_flag() ) return 0.0;

	for ( int i = 1; i <= n_score_tags; ++i ) {
		if ( tag == score_tags(i) ) {
			return decoystats_score(i);
		}
	}

	std::cout << "STOP:: tag not found " << tag << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

	return 0.0;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin decoystats_make_pdb_scores
///
/// @brief
///
/// @detailed
///
/// @param[in]  iunit -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
decoystats_make_pdb_scores( std::ostream & iunit )
{
	using namespace decoystats;

	for ( int i = 1; i <= n_score_tags; ++i ) {
		iunit << A( 36, score_tags(i) ) << ':' <<
		 F( 11, 2, decoystats_score(i) ) << '\n';
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin decoystats_score_header_output
///
/// @brief
///
/// @detailed
///
/// @param[in]  iunit -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
decoystats_score_header_output( std::ostream & iunit )
{
	using namespace decoystats;

	for ( int i = 1; i <= n_score_tags; ++i ) {
		iunit << ' ' << A( 8, score_tags(i) );
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin decoystats_score_output
///
/// @brief
///
/// @detailed
///
/// @param[in]  iunit
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
decoystats_score_output( std::ostream & iunit )
{
	using namespace decoystats;

	for ( int i = 1; i <= n_score_tags; ++i ) {
		iunit << ' ' << F( 8, 2, decoystats_score(i) );
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin decoystats_analyze_fragments
///
/// @brief
///
/// @detailed
///
/// @param  nres - [in/out]? - total_residue
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
decoystats_analyze_fragments( int const nres /* total_residue */ )
{
	using namespace bb_count_common;
	using namespace fragments;

	if ( ! get_decoystats_flag() ) return;

// local
	int pos;
	FArray1D_int bb( DRange( 0, 2 ) );

// initialize the counts
	for ( int i = 1; i <= nres; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			total_count(i,j) = 0;
		}
		for ( int bb1 = 1; bb1 <= max_bb; ++bb1 ) {
			bb_count1(bb1,i) = 0;
			for ( int bb2 = 1; bb2 <= max_bb; ++bb2 ) {
				bb_count2(bb1,bb2,i) = 0;
				for ( int bb3 = 1; bb3 <= max_bb; ++bb3 ) {
					bb_count3(bb1,bb2,bb3,i) = 0;
				}
			}
		}
	}

	int const size = 3; // use the 3mer fragments
	int const bin = get_index_by_frag_size(size);
	for ( int i = 1, ie = nres-size+1; i <= ie; ++i ) {
		for ( int j = 1, je = align_depth(i,bin); j <= je; ++j ) {
// calculate bb state at the 3 frag positions
			for ( int k = 0; k <= 2; ++k ) {
				bb(k) = ppo_to_int( align_phi(i,j,k,bin), align_psi(i,j,k,bin),
				 align_omega(i,j,k,bin));
			}

// now increment the counters:
			for ( int k = 0; k <= 2; ++k ) { // 1mer
				pos = i+k;
				bb_count1(bb(k),pos) = bb_count1(bb(k),pos) + 1;
				++total_count(pos,1);
			}

			for ( int k = 0; k <= 1; ++k ) { // 2mer
				pos = i+k;
				bb_count2(bb(k),bb(k+1),pos) = bb_count2(bb(k),bb(k+1),pos) + 1;
				++total_count(pos,2);
			}

			pos = i; // 3mer
			bb_count3(bb(0),bb(1),bb(2),pos) = bb_count3(bb(0),bb(1),bb(2),pos) + 1;
			++total_count(pos,3);

		}                  // j = frag number
	}                     // i = frag start position

// output the frequency of the most popular combination
// for each position

	total_count(nres-1,3) = 1;
	total_count(nres,2) = 1;
	total_count(nres,3) = 1;

	output_top_local_bb("FRG");
}

//////////////////////////////////////////////////////////////////////////////
/// @begin ppo_to_int
///
/// @brief
///
/// @detailed
/// 1-- helical
/// 2-- extended
/// 3-- alpha-L
/// 4-- E
/// 5-- cis omega
///
/// @param ph - [in/out]? -
/// @param ps - [in/out]? -
/// @param om - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
ppo_to_int(
	float const ph,
	float const ps,
	float const om
)
{
	float const tmp_phi   = periodic_range(ph,360.);
	float const tmp_psi   = periodic_range(ps,360.);
	float const tmp_omega = periodic_range(om,360.);

	int ppo_to_int; // Return value
	if ( std::abs(tmp_omega) < 90 ) {
		ppo_to_int = 5; // cis-omega
	} else if ( tmp_phi >= 0.0 ) {
		if ( -100 < tmp_psi && tmp_psi <= 100 ) {
			ppo_to_int = 3; // alpha-L
		} else {
			ppo_to_int = 4; // E
		}
	} else {
		if ( -125 < tmp_psi && tmp_psi <= 50 ) {
			ppo_to_int = 1; // helical
		} else {
			ppo_to_int = 2; // extended
		}
	}
	return ppo_to_int;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin decoystats_store_native
///
/// @brief
/// this function will only be called if native exists and query_defined
/// currently:: assume that we have just called fullatom_score_position
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
decoystats_store_native()
{
	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace files_paths;
	using namespace fullatom_energies; // for atr_pair
	using namespace misc;
	using namespace param;

	//	if ( ! get_fullatom_flag() || ! get_decoystats_flag() ) return;
	if( ! output_fa || ! get_decoystats_flag() ) return;

	if ( native_is_stored::stored ) return;

	int total_contacts,aa,aav,highest_chi;

	FArray1D_float rsd_sasa( MAX_RES()() );
	FArray2D_float atom_sasa( MAX_ATOM()(), MAX_RES()() );
	FArray1D_float dummy_phi( MAX_RES()() );
	FArray1D_float dummy_psi( MAX_RES()() );
	float sasa_fraction;

	bool buried_residue,atom_in_set;

	start_file = "native";

// initialize the per-atom logicals for calculating rms-like scores
	calc_per_atom_sasa( atom_sasa, rsd_sasa, 1.4f, false, false );
	// water probe radius, real sasa,small polar H
	for ( int i = 1, max_atom = MAX_ATOM(); i <= total_residue; ++i ) {
		aa = res(i);
		aav = res_variant(i);

		for ( int j = 1; j <= max_atom; ++j ) {
			for ( int k = 1; k <= n_atom_set; ++k ) {
				atom_set(j,i,k) = false;
			}
		}

		sasa_fraction = std::max( 0.0f, std::min( 1.0f, rsd_sasa(i) / rsd_exposed_sasa(aa) ) );

		if ( sasa_fraction < buried_residue_threshold ) {
			buried_residue = true;
		} else {
			buried_residue = false;
		}

		for ( int j = 1, je = nheavyatoms(aa,aav); j <= je; ++j ) {
			atom_set(j,i,all_heavyatoms) = true;

			if ( buried_residue ) atom_set(j,i,native_buried_rsd_heavyatoms) = true;

			if ( atom_sasa(j,i) <= 0.0 ) atom_set(j,i,native_buried_heavyatoms) = true;
		}
// define the atom sets by chi-dependence
		for ( int chi_start = 1; chi_start <= 3; ++chi_start ) {
			for ( int j = 1, je = nheavyatoms(aa,aav); j <= je; ++j ) {
				atom_in_set = true;
				for ( int chino = chi_start; chino <= MAX_CHI; ++chino ) {
					if ( chi_required(chino,j,aa,aav) ) atom_in_set = false;
				}
				atom_set(j,i,chi_atom_set_number(chi_start)) = atom_in_set;
			}
		}
	}

// atom set diagnostics
	if ( false ) {
		for ( int i = 1; i <= n_atom_set; ++i ) {
			set_bfactor_by_atom_set(atom_set(1,1,i));
			dump_fullatom_pdb( code + protein_name + protein_chain + "atom_set" +
			 lead_zero_string_of( i, 1 ) + ".pdb" );
		}

// transmit chi_required info to decoystats.py
		for ( int i = 1; i <= total_residue; ++i ) {
			aa = res(i);
			aav = res_variant(i);
			for ( int j = 1, je = natoms(aa,aav); j <= je; ++j ) {
				highest_chi = 0;
				for ( int chino = 1; chino <= MAX_CHI; ++chino ) {
					if ( chi_required(chino,j,aa,aav) ) highest_chi = chino;
				}
				make_pdb_ns::bfactor(j,i) = highest_chi;
			}
		}
		dump_fullatom_pdb( code + protein_name + protein_chain + ".chi_required.pdb" );
	}

// store native aa, aav, full_coord
	for ( int i = 1; i <= total_residue; ++i ) {
		int const aa = res(i);
		int const aav = res_variant(i);
		native_aa(i) = aa;
		native_aav(i) = aav;
		for ( int j = 1, je = natoms(aa,aav); j <= je; ++j ) {
			for ( int k = 1; k <= 3; ++k ) {
				native_full_coord(k,j,i) = full_coord(k,j,i);
			}
		}
	}

// store the native all-atom contacts
	get_allatom_contacts(full_coord, atom_set(1,1,all_heavyatoms), total_contacts,
	 native_contacts);

	// store native chi angles
	get_chi_and_rot_from_coords(total_residue,res,res_variant,full_coord,native_chi,native_rot);

	decoy_is_native = true;
	decoystats_store_decoy();
	decoy_is_native = false;

	native_is_stored::stored = true;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin copy_hbenergies
///
/// @brief
/// fill the residue-residue pairwise hbenergy arrays
///
/// uses the arrays in hbonds.h indexed by hbond-number
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
copy_hbenergies()
{
	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace files_paths;
	using namespace hbonds;
	using namespace misc;
	using namespace param;

	dimension_big_decoystats_arrays();

	int dres,ares,dhatm,aatm,dres_type,ares_type,datm,abase_atm,atype,dtype;
	int daav,aaav;
	FArray1D_float dhatm_xyz( 3 ), aatm_xyz( 3 ), datm_xyz( 3 ), abase_xyz( 3 );
	float hbE, dis_ha, dis_dh, dis_ad, psi_hb, phi_hb;

	for ( int i = 1, max_atom = MAX_ATOM(); i <= total_residue; ++i ) { // initialize stuff
		for ( int j = 1; j <= total_residue; ++j ) {
			for ( int k = 1; k <= 4; ++k ) {
				rsd_pair_hbondE(i,j,k) = 0.0;
			}
		}
		for ( int j = 1; j <= max_atom; ++j ) {
			atom_hbondE(j,i) = 0.0;
		}

	}

	bool const require_allowed(true);

	for ( int i = 1; i <= hbond_set.nhbonds(); ++i ) {

		if ( ( ! require_allowed ) || ( allow_hbond_by_idx(i) ) ) {
			// loop over all hbonds, even those forbidden by allow_hbond()
			// jk no, don't allow those forbidden by allow_hbond!

			dres = hbond_set.hbdon_res(i);
			ares = hbond_set.hbact_res(i);
			dres_type = res(dres);
			ares_type = res(ares);
			daav=res_variant(dres);
			aaav=res_variant(ares);
			dhatm = hbond_set.hbdonh_atm(i);
			aatm = hbond_set.hbact_atm(i);
			datm = atom_base(dhatm,dres_type,daav);
			abase_atm = atom_base(aatm,ares_type,aaav);
			atype = fullatom_type(aatm,ares_type,aaav);
			dtype = fullatom_type(datm,dres_type,daav);

			for ( int j = 1; j <= 3; ++j ) {
				dhatm_xyz(j) = full_coord(j,dhatm,dres);
				aatm_xyz(j) = full_coord(j,aatm,ares);
				datm_xyz(j) = full_coord(j,datm,dres);
				abase_xyz(j) = full_coord(j,abase_atm,ares);
			}

			hbE = hbond_set.hbenergies(i);
			atom_hbondE(dhatm,dres) += hbE;
			atom_hbondE(aatm,ares) += hbE;

			int k = get_hbondE_key(atom_is_backbone(dhatm,res(dres),res_variant(dres)),
		   atom_is_backbone(aatm,res(ares),res_variant(ares)));

			rsd_pair_hbondE(dres,ares,k) += hbE;

			generate_hbond_info(dhatm_xyz,aatm_xyz,datm_xyz,abase_xyz,dis_ha,dis_dh,
		   dis_ad,psi_hb,phi_hb);

			if( get_decoystats_flag() ) {
				dump_hbond_info(start_file,dres,dres_type,datm,dtype,ares,ares_type,aatm,
											  atype,dis_ha,dis_dh,dis_ad,psi_hb,phi_hb,i);
			}

		}
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_hbondE_key
///
/// @brief
///
/// @detailed
///
/// @param  d_is_bb - [in/out]? -
/// @param  a_is_bb - [in/out]? -
///
/// @return get_hbondE_key
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
get_hbondE_key(
	bool const d_is_bb,
	bool const a_is_bb
)
{
	int get_hbondE_key; // Return value

	if ( d_is_bb && a_is_bb ) {
		get_hbondE_key = 1;
	} else if ( d_is_bb ) {
		get_hbondE_key = 2;
	} else if ( a_is_bb ) {
		get_hbondE_key = 3;
	} else {
		get_hbondE_key = 4;
	}
	return get_hbondE_key;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin decoystats_store_output_line
///
/// @brief
/// this function exists so that we can echo output in make_pdb
///
/// @detailed
///
/// @param[in]  line -
/// @param[in]  line_length -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
decoystats_store_output_line( std::string const & line )
{
	using namespace store_output_line_common;

	++n_lines;
	if ( n_lines > max_output_lines ) {
		std::cout << "too many output lines!!!!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	output_lines( n_lines ) = line;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin decoystats_reset_output_lines
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
decoystats_reset_output_lines()
{
	using namespace store_output_line_common;

	n_lines = 0;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin decoystats_write_output_lines
///
/// @brief
///
/// @detailed
///
/// @param[in]  iunit -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
decoystats_write_output_lines( std::ostream & iunit )
{
	using namespace store_output_line_common;

	for ( int i = 1; i <= n_lines; ++i ) {
		iunit << output_lines(i) << '\n';
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin find_unsatisfied_hbonds
///
/// @brief
/// find buried, unsatisfied donors and acceptors
///
/// @detailed
///
/// previously used the arrays in hbonds.h indexed by hbond-number,
/// which assumes these have been updated by a call to fill_hbond_arrays
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
find_unsatisfied_hbonds(
	std::list < unsatisfied_buried_polar > & uns_list,
	std::list < unsatisfied_buried_group > & group_uns_list
)
{
	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace files_paths;   // for output_file
	using namespace hbonds;
	using namespace misc;
	using namespace param;
	using namespace param_aa;

// local
	FArray1D_float rsd_sasa( MAX_RES()() );
	FArray2D_float atom_sasa( MAX_ATOM()(), MAX_RES()() );
	FArray1D_float rsd_sasa10( MAX_RES()() );
	FArray2D_float atom_sasa10( MAX_ATOM()(), MAX_RES()() );
	FArray1D_float rsd_sasa7( MAX_RES()() );
	FArray2D_float atom_sasa7( MAX_ATOM()(), MAX_RES()() );

	bool const local_debug = { false };

	uns_list.clear();
	group_uns_list.clear();

// constants for hbonds and burial
	float const probe_radius = { 1.4 };
	float const burial_threshold = { 0.01 };
	float const hbond_energy_threshold = { -0.01 }; // per-atom total hbondenergy

// calc sasa with big polar hydrogens
	bool setting = get_use_big_polar_h();
	set_use_big_polar_h(true);
	calc_per_atom_sasa( atom_sasa, rsd_sasa, probe_radius, false, false );
	calc_per_atom_sasa( atom_sasa10, rsd_sasa10, 1.0f, false, false );
	calc_per_atom_sasa( atom_sasa7, rsd_sasa7, 0.7f, false, false );
	set_use_big_polar_h(setting);

	if ( ! get_interface_ds_flag() ) {
		// diagnostics
		for ( int i = 1, max_atom = MAX_ATOM(); i <= total_residue; ++i ) {
			for ( int j = 1; j <= max_atom; ++j ) {
				make_pdb_ns::bfactor(j,i) = atom_sasa(j,i);
			}
		}
	}

	if ( false ) {
		std::string filename1, filename2;
		strip_path( start_file, filename1 );
		std::cout << "filename1: " << filename1 << std::endl;
		filename2 = "ds_dump" + output_file + ".sasapdb";
		std::cout << "filename2: " << filename2 << std::endl;
		dump_fullatom_pdb(filename2);
	}

// look for buried, unsatisfied donors/acceptors:

	for ( int i = 1; i <= total_residue; ++i ) {
		int const aa = res(i);
		int const aav = res_variant(i);
		for ( int hnum = 1, hnume = nH_polar(aa,aav); hnum <= hnume; ++hnum ) {
			int const dhatm = Hpos_polar(hnum,aa,aav);
			float hbE = atom_hbondE(dhatm,i);
			bool const is_donor = true;
			bool const is_backbone = atom_is_backbone(dhatm,res(i),res_variant(i));

			if ( hbE > hbond_energy_threshold
			 && atom_sasa(dhatm,i) <= burial_threshold ) {
				// check for water hbE
				hbE += get_sc_atm_h2oE( i, dhatm );
			}

			bool is_satisfied = false;
			if ( hbE < hbond_energy_threshold ) {
				is_satisfied = true;
			}

			if ( local_debug ||
					 ( ( ! is_satisfied ) && atom_sasa(dhatm,i) <= burial_threshold ) ) {

				// jk This is an UNS: add it to the list
				unsatisfied_buried_polar uns;
				// jk Get the 1-letter sequence id from the aa,
				// jk because design doesn't always update the "residue1" array
				uns.set_all(is_donor, is_backbone, is_satisfied, start_file,
				 aa_name3(aa), i,	dhatm, hbE, atom_name(dhatm,aa,aav),
				 atom_sasa(dhatm,i), atom_sasa10(dhatm,i), atom_sasa7(dhatm,i));
				uns_list.push_back(uns);

			}
		}

		for ( int anum = 1, anume = nacceptors(aa,aav); anum <= anume; ++anum ) {
			int const aatm = accpt_pos(anum,aa,aav);
			float hbE = atom_hbondE(aatm,i);
			bool const is_donor = false;
			bool const is_backbone = atom_is_backbone(aatm,res(i),res_variant(i));
			if ( hbE > hbond_energy_threshold
			 && atom_sasa(aatm,i) <= burial_threshold ) {
				// check for water hbE
				hbE += get_sc_atm_h2oE( i, aatm );
			}

			bool is_satisfied = false;
			if ( hbE < hbond_energy_threshold ) {
				is_satisfied = true;
			}

			if ( local_debug ||
					 ( ( ! is_satisfied ) && atom_sasa(aatm,i) <= burial_threshold ) ) {

				// jk This is an UNS: add it to the list
				unsatisfied_buried_polar uns;
				// jk Get the 1-letter sequence id from the aa,
				// jk because design doesn't always update the "residue1" array
				uns.set_all(is_donor, is_backbone, is_satisfied, start_file,
				 aa_name3(aa), i,	aatm, hbE, atom_name(aatm,aa,aav), atom_sasa(aatm,i),
				 atom_sasa10(aatm,i), atom_sasa7(aatm,i));
				uns_list.push_back(uns);

			}

		}
	}

	// jk sort the uns list before returning it
	uns_list.sort();
	collect_group_uns(uns_list,group_uns_list);

	return;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin report_unsatisfied_hbonds
///
/// @brief
/// report buried, unsatisfied donors and acceptors
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
report_unsatisfied_hbonds(
	std::string tag,
	std::list < unsatisfied_buried_polar > & uns_list )
{

	int bb_unsatisfied=0;
	int sc_unsatisfied=0;
	float uns_score=0;

	if ( ! get_ds_outpdbonly_flag() ) {
		std::cout << "ID type state pdb  res  pos  pdb  CH   atm      " <<
	 "hbE name      sasa    sasa10     sasa7" << std::endl;
	}

	for( std::list<unsatisfied_buried_polar>::iterator uns = uns_list.begin();
			uns != uns_list.end(); ++uns ) {

		uns_score+=uns->weight();

		std::string uns_type = decide_uns_type( *uns );

		if ( ! get_ds_outpdbonly_flag() ) uns->write_to_cout( tag, uns_type );
		uns->write_to_PDB( tag, uns_type );

		// jk require burial to a 1.0 A probe if explicit water is not used,
		// jk otherwise relax the criterion to "buried to a 1.4 A probe"
		if ( ! uns->satisfied() ) {
			if ( ( uns->buried10() ) || ( ( design::explicit_h2o ) && uns->buried() ) ) {
				if ( uns->backbone() ) {
					++bb_unsatisfied;
				} else {
					++sc_unsatisfied;
				}
			}
		}

	}

	if ( tag != "" ) tag = tag + "_";
	add_decoy_score(tag+"bbhb_uns",float(bb_unsatisfied));
	add_decoy_score(tag+"schb_uns",float(sc_unsatisfied));
	add_decoy_score(tag+"uns_score",uns_score);

	return;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin report_unsatisfied_hbonds
///
/// @brief
/// report buried, unsatisfied donors and acceptors
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
report_unsatisfied_hbonds(
	std::string tag,
	std::list < unsatisfied_buried_polar > & uns_list,
	std::list < unsatisfied_buried_group > & group_uns_list
)
{

	// jk Overloading: if group_uns is passed in do it after doing the uns as per previous subroutine
	report_unsatisfied_hbonds(tag, uns_list);

	int num_group_uns=0;
	float group_uns_score=0;

	if ( ! get_ds_outpdbonly_flag() ) {
		std::cout << "ID res  type  pos  pdb " << std::endl;
	}

	for( std::list<unsatisfied_buried_group>::iterator group_uns = group_uns_list.begin();
			group_uns != group_uns_list.end(); ++group_uns ) {

		++num_group_uns;
		group_uns_score+=group_uns->weight();

		if ( ! get_ds_outpdbonly_flag() ) group_uns->write_to_cout(tag);
		group_uns->write_to_PDB(tag);

	}

	if ( tag != "" ) tag = tag + "_";
	add_decoy_score(tag+"group_uns",float(num_group_uns));
	add_decoy_score(tag+"group_uns_score",group_uns_score);

	return;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin dump_uns_list
///
/// @brief
/// write the contents of an uns_list to STDOUT, for debugging
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
dump_uns_list(
	std::string tag,
	std::list < unsatisfied_buried_polar > & uns_list
)
{

	for( std::list<unsatisfied_buried_polar>::iterator uns = uns_list.begin();
			uns != uns_list.end(); ++uns ) {

		std::string uns_type = decide_uns_type( *uns );
		uns->write_to_cout( tag, uns_type );
	}

	return;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin collect_group_uns
///
/// @brief
/// given a list of UNS, build a list of unsatisfied groups
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void collect_group_uns(
	std::list < unsatisfied_buried_polar > & uns_list,
	std::list < unsatisfied_buried_group > & group_uns_list
)
{

	group_uns_list.clear();

	for( std::list<unsatisfied_buried_polar>::iterator uns = uns_list.begin();
			uns != uns_list.end(); ++uns ) {

		if ( uns->uns_group(uns_list) ) {
			const int seqpos = uns->get_seqpos();
			const int group_type = uns->group_type();
			unsatisfied_buried_group group_uns(seqpos,group_type);
			group_uns_list.push_back(group_uns);
		}

	}

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin read_uns_from_pdb
///
/// @brief
/// read a list of uns from an input PDB
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
read_uns_from_pdb(
	utility::io::izstream & pdb_stream,
	std::list < unsatisfied_buried_polar > & uns_list,
	std::list < unsatisfied_buried_group > & group_uns_list
)
{

	// reposition to top of pdb file
	pdb_stream.seek_beg();

	// clear the lists, in case there were previous entries
	uns_list.clear();
	group_uns_list.clear();

	std::string line;
	std::istringstream line_stream;
	while ( true ) {

		// read the next line
		pdb_stream.getline( line );
		if ( pdb_stream.eof() ) {
			pdb_stream.close();
			pdb_stream.clear();
			return;

		} else if ( line.find("UNS") != std::string::npos ) {

			line_stream.clear();
			line_stream.str( line );
			line_stream.seekg( std::ios_base::beg );

			bool is_backbone(false), is_donor(false), is_satisfied(false);
			std::string junk, tag, start_file, atom_name;
			std::string res_type;
			int seqpos, atm;
			float hbE, sasa, sasa10, sasa7;

			line_stream >> junk >> tag >> start_file >> res_type >> seqpos;
			line_stream >> atm >> hbE >> atom_name >> sasa >> sasa10 >> sasa7;

			if ( tag.find("BB") != std::string::npos ) is_backbone=true;
			if ( tag.find("DON") != std::string::npos ) is_donor=true;
			if ( tag.find("SAT") != std::string::npos ) is_satisfied=true;

			unsatisfied_buried_polar uns;
			uns.set_all( is_donor, is_backbone, is_satisfied, start_file, res_type,
			 seqpos, atm, hbE, atom_name, sasa, sasa10, sasa7);
			uns_list.push_back(uns);

		} else if ( line.find("GU") != std::string::npos ) {

			line_stream.clear();
			line_stream.str( line );
			line_stream.seekg( std::ios_base::beg );

			std::string junk, aa, group_type;
			int seqpos;
			line_stream >> junk;
			if ( junk != "GU" ) line_stream >> junk;
			line_stream >> aa >> group_type >> seqpos;
			unsatisfied_buried_group group_uns(seqpos,group_type);
			group_uns_list.push_back(group_uns);
		}

	}

	return;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin find_tight_core_packing
///
/// @brief
/// for atoms which are buried to a 1.4 A probe, put their
/// exposure to a 0.5 A probe in the B-factor field. For all
/// others, set it to 25.0
///
/// @detailed
///   Note: does NOT use hydrogens in SASA calculations!
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
find_tight_core_packing( bool const interface_flag )
{
	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace misc;
	using namespace param;
	using namespace param_aa;

// local
	FArray1D_float rsd_sasa14( MAX_RES()() );
	FArray2D_float atom_sasa14( MAX_ATOM()(), MAX_RES()() );
	FArray1D_float rsd_sasa5( MAX_RES()() );
	FArray2D_float atom_sasa5( MAX_ATOM()(), MAX_RES()() );

	// jk return value - number of tightly packed atoms
	int num_tight = 0;
	tightly_packed = false;

	calc_per_atom_sasa( atom_sasa14, rsd_sasa14, 1.4f, true, interface_flag );
	calc_per_atom_sasa( atom_sasa5, rsd_sasa5, 0.5f, true, interface_flag );

	// jk define "buried" atoms
	float const buried_thres = { 0.2 };

	// jk 25.0 is about the maximum for a buried atom
	float const max_sasa5=50.;

	for ( int i = 1; i <= total_residue; ++i ) {

		int const aa = res(i);
		int const aav = res_variant(i);
		int max_heavyatom=nheavyatoms(aa,aav);
		// Use the value of atom_sasa5 if buried to 1.4 A probe
		for ( int j = 1; j <= max_heavyatom; ++j ) {
			int const atomtype = fullatom_type(j,aa,aav);
			if ( ( atom_sasa14(j,i) <= buried_thres ) &&
					 ( atomtype == 4 ) || ( atomtype == 5 ) || ( atomtype == 6 ) ||
					 ( ( atomtype == 16 ) && ( aa == aa_met) ) ) {
				// jk count only CH2, CH3, Caro, or Met SD
				float this_sasa5=atom_sasa5(j,i);
				make_pdb_ns::bfactor(j,i) = this_sasa5;
				if ( this_sasa5 <= tight_core_thres ) {
					tightly_packed(j,i)=true;
					++num_tight;
				}
			} else {
				make_pdb_ns::bfactor(j,i) = max_sasa5;
			}
		}

		// Fill all hydrogens with max_sasa5
		int max_atom=natoms(aa,aav);
		for ( int j = max_heavyatom+1; j <= max_atom; ++j ) {
			make_pdb_ns::bfactor(j,i) = max_sasa5;
		}

	}

	return num_tight;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin decoystats_store_decoy
///
/// @brief
///  assumes we have just called fullatom score w/ evaluate all terms=true
///
///  write out some per-decoy diagnostic info to stats file
///  increment counters and average values
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
decoystats_store_decoy()
{
	using namespace decoystats;
	using namespace files_paths;   // query_defined
	using namespace misc;          // total_residue,res

	//	if ( ! get_fullatom_flag() || ! get_decoystats_flag() ) return;
	if( ! output_fa || ! get_decoystats_flag() ) return;

	reset_decoy_scores();
	decoystats_reset_output_lines();

	add_decoy_score("sasapack",retrieve_sasa_pack_score());

	if ( ! get_ds_outpdbonly_flag() ) {
		std::cout << "DS START_FILE " << start_file << std::endl;
		// these two calls should come right after the preceding output statement::
		dump_torsions();
		dump_rsd_energies(total_residue,res);
	}

	if ( ! get_interface_ds_flag() ) {
		copy_hbenergies();
		std::list < unsatisfied_buried_polar > uns_list;
		std::list < unsatisfied_buried_group > group_uns_list;
		find_unsatisfied_hbonds(uns_list, group_uns_list);
		report_unsatisfied_hbonds("",uns_list,group_uns_list);
		//	report_unsatisfied_hbonds( "", uns_list );
	}

	if ( get_interface_ds_flag() && ! decoy_is_native ) {
		pose_ns::Pose pose;
		pose_from_misc( pose, true, false, true ); // coords_init
		interface_ds interface(&pose);
		if ( find_interface_CBCB ) {
			interface.find_CBCB();
		}
		interface.write_all_quantifiers_to_PDB();
		// jk write a list of interface residues
		//		interface.list_interface_res();
		// jk write a list of interface atoms
		//		interface.list_interface_atoms();

	} else if ( tight_core_analysis ) {
		// jk max SASA5 which still qualifies as tight packing
		realafteroption("tight_core_thres",4.0,tight_core_thres);
		add_decoy_score("tight_core_atoms",find_tight_core_packing(false));
	}

	if ( ! query_defined ) return; //------------------------

	increment_counters();

	if ( ! get_native_exists() ) return;

	if ( decoy_is_native ) return;

	compare_to_native();

}

//////////////////////////////////////////////////////////////////////////////
/// @begin dump_torsions
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
dump_torsions()
{
	using namespace misc;
	using namespace param;

	FArray2D_int decoy_rot( MAX_CHI, MAX_RES()() );
	FArray1D_float dummy_phi( MAX_RES()() );
	FArray1D_float dummy_psi( MAX_RES()() );
	FArray2D_float decoy_chi( MAX_CHI, MAX_RES()() );

	// chi angles
	get_chi_and_rot_from_coords(total_residue,res,res_variant,full_coord,decoy_chi,decoy_rot);

	for ( int i = 1; i <= total_residue; ++i ) {
		std::cout << "DS_TORSIONS " << I( 4, i ) << ' ' << residue3(i) << ' ' <<
		 ' ' << F( 6, 1, phi(i) ) <<
		 ' ' << F( 6, 1, psi(i) ) <<
		 ' ' << F( 6, 1, omega(i) ) <<
		 ' ' << F( 6, 1, decoy_chi(1,i) ) << ' ' << I( 2, decoy_rot(1,i) ) <<
		 ' ' << F( 6, 1, decoy_chi(2,i) ) << ' ' << I( 2, decoy_rot(2,i) ) <<
		 ' ' << F( 6, 1, decoy_chi(3,i) ) << ' ' << I( 2, decoy_rot(3,i) ) <<
		 ' ' << F( 6, 1, decoy_chi(4,i) ) << ' ' << I( 2, decoy_rot(4,i) ) << std::endl;
	}

}


//////////////////////////////////////////////////////////////////////////////
/// @begin e_names_initializer
///
/// @brief
/// initialize e_names array
///
/// @detailed
///
/// @param  e_names - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
e_names_initializer( FArray1D_string & e_names )
{
	assert( ( e_names.l() == 1 ) && ( e_names.u() == 8 ) );

	e_names(1) = "atr ";
	e_names(2) = "rep ";
	e_names(3) = "sol ";
	e_names(4) = "aa  ";
	e_names(5) = "dun ";
	e_names(6) = "hbnd";
	e_names(7) = "pair";
	e_names(8) = "res ";
}


//////////////////////////////////////////////////////////////////////////////
/// @begin dump_rsd_energies
///
/// @brief
/// modelled on output_residueE in make_pdb.cc
/// same include statements (dont know what comes from what!!)
///
/// @detailed
///
/// @param  total_residue - [in/out]? -
/// @param  res - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
dump_rsd_energies(
	int const total_residue,
	FArray1Da_int res
)
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace fullatom_energies;
	using namespace param;
	using namespace pdbstatistics_pack;
	using namespace template_pack;

	res.dimension( total_residue );

	int nb,aa;
	int const n_energies = { 8 };
	static FArray1D_string const e_names( n_energies, e_names_initializer );
	FArray2D_float rsd_e( n_energies, MAX_RES()() );
	FArray2D_float rsd_avg_e( n_energies, MAX_RES()() );

	for ( int i = 1; i <= total_residue; ++i ) {
		nb = neighbors(i);
		aa = res(i);

		rsd_e(1,i) = atrenergy(i);
		rsd_avg_e(1,i) = atr_avg(aa,nb);

		rsd_e(2,i) = repenergy(i);
		rsd_avg_e(2,i) = rep_avg(aa,nb);

		rsd_e(3,i) = solenergy(i);
		rsd_avg_e(3,i) = sol_avg(aa,nb);

		rsd_e(4,i) = probenergy(i);
		rsd_avg_e(4,i) = one_avg(aa,nb);

		rsd_e(5,i) = dunenergy(i);
		rsd_avg_e(5,i) = dun_avg(aa,nb);

		rsd_e(6,i) = hbenergy(i);
		rsd_avg_e(6,i) = hb_avg(aa,nb);

		rsd_e(7,i) = pair_energy(i);
		rsd_avg_e(7,i) = pair_avg(aa,nb);

		rsd_e(8,i) = resenergy(i);
		rsd_avg_e(8,i) = tot_avg(aa,nb);
	}

	std::cout << "DS_E_NAMES ";
	for ( int i = 1; i <= n_energies; ++i ) {
		std::cout << ' ' << e_names(i);
	} std::cout << std::endl;

	std::string res3;
	for ( int i = 1; i <= total_residue; ++i ) {
		name_from_num( res(i), res3 );
		std::cout << "DS_E " << I( 4, i ) << ' ' << res3;
		for ( int j = 1; j <= n_energies; ++j ) {
			std::cout << ' ' << F( 6, 1, rsd_e(j,i) ) <<
			 ' ' << F( 6, 1, rsd_avg_e(j,i) );
		} std::cout << std::endl;
	}

}



//------------------------------------------------------------------------------
// these functions increment running totals over all the
// decoys:
//------------------------------------------------------------------------------

//////////////////////////////////////////////////////////////////////////////
/// @begin increment_counters
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
increment_counters()
{
	using namespace decoystats;    // total_decoys

	if ( ! decoy_is_native ) ++total_decoys;

	hbond_counter();
	fa_atr_counter();
	local_bb_counter();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin hbtags_initialize
///
/// @brief
///
/// @detailed
///
/// @param  hbtags - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
hbtags_initializer( FArray1D_string & hbtags )
{
	int i = 0;
	hbtags( ++i ) = "HB-BB-BB";
	hbtags( ++i ) = "HB-BB-SC";
	hbtags( ++i ) = "HB-SC-BB";
	hbtags( ++i ) = "HB-SC-SC";
}

//////////////////////////////////////////////////////////////////////////////
/// @begin hbond_counter
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
hbond_counter()
{
	using namespace decoystats;
	using namespace misc;

	float const hbondE_threshold = { -0.25 };

	static FArray1D_string const hbtags( 4, hbtags_initializer );
	 // this should sync with get_hbondE_key

	if ( ! hb_counter_init::init ) {
		for ( int k = 1; k <= 4; ++k ) {
			new_counter( hbtags(k), pair_type );
		}
		hb_counter_init::init = true;
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= total_residue; ++j ) {
			for ( int k = 1; k <= 4; ++k ) {
				if ( rsd_pair_hbondE(i,j,k) <= hbondE_threshold ) {
					increment_counter( hbtags(k), pair_type, i, j );
				}
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin fa_atr_threshold_initializer
///
/// @brief
///
/// @detailed
///
/// @param  fa_atr_threshold - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
fa_atr_threshold_initializer( FArray1D_float & fa_atr_threshold )
{
	int i = 0;
	fa_atr_threshold( ++i ) = -0.75;
	fa_atr_threshold( ++i ) = -1.5;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin fa_atr_counter
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
fa_atr_counter()
{
	using namespace decoystats;
	using namespace fullatom_energies; // atr_pair
	using namespace misc;

	static FArray1D_float const fa_atr_threshold( 2,
	 fa_atr_threshold_initializer );

	std::string tag;

	if ( ! fa_atr_counter_init::init ) {
		for ( int k = 1; k <= 2; ++k ) {
			tag = "FA_ATR-" + lead_zero_string_of( k, 1 );
			new_counter(tag,pair_type);
		}
		fa_atr_counter_init::init = true;
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= total_residue; ++j ) {
			for ( int k = 1; k <= 2; ++k ) {
				tag = "FA_ATR-" + lead_zero_string_of( k, 1 );
				if ( atr_pair(i,j) <= fa_atr_threshold(k) ) {
					increment_counter(tag,pair_type,i,j);
				}
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin local_bb_counter
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
local_bb_counter()
{
	using namespace bb_count_common;
	using namespace decoystats;
	using namespace misc;
	using namespace param;

	FArray1D_int local_bb( MAX_RES()() );
	int native_count;

	// jump out if we don't have/want fragments
	if ( ! files_paths::require_frags ) return;

	for ( int i = 1; i <= total_residue; ++i ) { // calc bb states
		local_bb(i) = ppo_to_int(phi(i),psi(i),omega(i));
	}

	if ( ! local_bb_counter_init::init )
		decoystats_analyze_fragments( total_residue );

	if ( decoy_is_native ) { // evaluate native frequencies for fragments
		for ( int i = 1; i <= total_residue; ++i ) { // calc native bb
			native_bb(i) = local_bb(i);
		}

		for ( int i = 1; i <= total_residue; ++i ) {
			native_count = bb_count1(native_bb(i),i);
			DS_RSD_output("BBFRG-N1",i, float(native_count)/total_count(i,1));

			if ( i < total_residue ) {
				native_count = bb_count2(native_bb(i),native_bb(i+1),i);
				DS_RSD_output("BBFRG-N2",i, float(native_count)/total_count(i,2));
			} else {
				DS_RSD_output("BBFRG-N2",i,0.0);
			}

			if ( i < total_residue-1 ) {
				native_count = bb_count3(native_bb(i),native_bb(i+1), native_bb(i+2),i);
				DS_RSD_output("BBFRG-N3",i, float(native_count)/total_count(i,3));
			} else {
				DS_RSD_output("BBFRG-N3",i,0.0);
			}
		}
 		got_native_counters = true;
		return;
	}                     // decoy_is_native

	if ( ! got_native_counters ) return;

	if ( ! local_bb_counter_init::init ) {
		local_bb_counter_init::init = true;
// initialize the counts
		for ( int i = 1; i <= total_residue; ++i ) {
			for ( int j = 1; j <= 3; ++j ) {
				total_count(i,j) = 0;
			}
			for ( int bb1 = 1; bb1 <= max_bb; ++bb1 ) {
				bb_count1(bb1,i) = 0;
				for ( int bb2 = 1; bb2 <= max_bb; ++bb2 ) {
					bb_count2(bb1,bb2,i) = 0;
					for ( int bb3 = 1; bb3 <= max_bb; ++bb3 ) {
						bb_count3(bb1,bb2,bb3,i) = 0;
					}
				}
			}
		}
		new_counter("BBDEC-N1",rsd_type);
		new_counter("BBDEC-N2",rsd_type);
		new_counter("BBDEC-N3",rsd_type);
	}           // if not init ---------------------


	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			++total_count(i,j);
		}

		int const bb1 = local_bb(i);
		++bb_count1(bb1,i);
		if ( bb1 == native_bb(i) ) increment_counter("BBDEC-N1",rsd_type,i,0);

		if ( i < total_residue ) {
			int const bb2 = local_bb(i+1);
			++bb_count2(bb1,bb2,i);
			if ( bb1 == native_bb(i) && bb2 == native_bb(i+1) )
			 increment_counter("BBDEC-N2",rsd_type,i,0);

			if ( i < total_residue-1 ) {
				int const bb3 = local_bb(i+2);
				++bb_count3(bb1,bb2,bb3,i);
				if ( bb1 == native_bb(i) && bb2 == native_bb(i+1) && bb3 == native_bb(i+2) )
				 increment_counter("BBDEC-N3",rsd_type,i,0);
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin bb_name_initializer
///
/// @brief
///
/// @detailed
///
/// @param  bb_name - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
bb_name_initializer( FArray1D_char & bb_name )
{
	int i = 0;
	bb_name( ++i ) = 'A';
	bb_name( ++i ) = 'B';
	bb_name( ++i ) = 'G';
	bb_name( ++i ) = 'E';
	bb_name( ++i ) = 'O';
}

//////////////////////////////////////////////////////////////////////////////
/// @begin output_top_local_bb
///
/// @brief
/// xxx is either DEC or FRG
///
/// @detailed
///
/// @param[in]  xxx -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
output_top_local_bb( std::string const & xxx )
{
	using namespace bb_count_common;
	using namespace misc;

// local:
	static FArray1D_char const bb_name( max_bb, bb_name_initializer );
	FArray1D_int best( 3 );
	std::string tag;

// output the frequency of the most popular combination for each position

	for ( int i = 1; i <= total_residue; ++i ) {

		for ( int j = 1; j <= 3; ++j ) {
			best(j) = 0;
		}

		for ( int bb1 = 1; bb1 <= max_bb; ++bb1 ) {
			best(1) = std::max(best(1), bb_count1(bb1,i));
			for ( int bb2 = 1; bb2 <= max_bb; ++bb2 ) {
				best(2) = std::max(best(2), bb_count2(bb1,bb2,i));
				for ( int bb3 = 1; bb3 <= max_bb; ++bb3 ) {
					best(3) = std::max(best(3), bb_count3(bb1,bb2,bb3,i));
				}
			}
		}

		DS_RSD_output( "BB" + xxx + "-T1", i, float(best(1) )/total_count(i,1));
		DS_RSD_output( "BB" + xxx + "-T2", i, float(best(2) )/total_count(i,2));
		DS_RSD_output( "BB" + xxx + "-T3", i, float(best(3) )/total_count(i,3));

		for ( int j = 1; j <= max_bb; ++j ) {
			tag = "BB" + xxx + '-' + bb_name(j) + ' ';
			DS_RSD_output(tag,i,float(bb_count1(j,i))/total_count(i,1));
			if ( ( native_bb(i) == j ) && ( ! get_ds_outpdbonly_flag() ) ) {
				DS_RSD_BOX(tag,i);
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin reset_counters
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
reset_counters()
{
	using namespace decoystats;

	ntags = 0;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin new_counter
///
/// @brief
///
/// @detailed
///
/// @param[in]  tag -
/// @param[in]  type -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
new_counter(
	std::string const & tag, // input
	int const type // input
)
{
	using namespace decoystats;
	using namespace misc;
	using namespace param;

	dimension_big_decoystats_arrays();

	if ( type <= 0 || type > 3 ) { // debug
		std::cout << "bad counter type" << SS( type ) << std::endl;
		return;
	}

	int const n = ntags(type) + 1;

	if ( ! get_ds_outpdbonly_flag() ) {
		std::cout << "new_counter:" << tag << SS( type ) << SS( n ) << std::endl;
	}

	if ( type == pair_type ) { // pair_type
		if ( n > max_pair_count ) {
			std::cout << "STOP:: too many pair tags!" << tag << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for ( int i = 1; i <= total_residue; ++i ) {
			for ( int j = 1; j <= total_residue; ++j ) {
				pair_count(i,j,n) = 0;
			}
		}
		tags(n,type) = tag;
	} else if ( type == rsd_type ) { // rsd_type
		if ( n > max_rsd_count ) {
			std::cout << "STOP:: too many rsd tags!" << tag << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for ( int i = 1; i <= total_residue; ++i ) {
			rsd_count(i,n) = 0;
		}
		tags(n,type) = tag;
	} else if ( type == atm_type ) { // atm_type
		if ( n > max_atom_count ) {
			std::cout << "STOP:: too many atom tags!" << tag << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for ( int i = 1, max_atom = MAX_ATOM(); i <= total_residue; ++i ) {
			for ( int j = 1; j <= max_atom; ++j ) {
				atom_count(j,i,n) = 0;
			}
		}
		tags(n,type) = tag;
	}

	++ntags(type);

}

//////////////////////////////////////////////////////////////////////////////
/// @begin increment_counter
///
/// @brief
///
/// @detailed
/// DANGER DANGER DANGER:: if type==atm_type: i is residue and j is atomno
/// j is a dummy if type==rsd_type
///
/// @param[in]  tag -
/// @param  type - [in/out]? -
/// @param  i - [in/out]? -
/// @param  j - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
increment_counter(
	std::string const & tag, // input
	int const type,
	int const i,
	int const j
)
{
	using namespace decoystats;

	if ( type <= 0 || type > 3 ) { // debug
		std::cout << "bad counter type" << SS( type ) << std::endl;
		return;
	}

	if ( ! get_ds_outpdbonly_flag() ) {
		if ( decoy_is_native ) {
			if ( type == pair_type ) // only using boxes for pairs right now
				DS_PAIR_BOX(tag,i,j);
			return; // dont increment counters
		}
	}

	int tag_num;
	int const ntags_type = ntags( type );
	for ( tag_num = 1; tag_num <= ntags_type; ++tag_num ) { // find the tag
		if ( tags(tag_num,type) == tag ) {
			goto L10;
		}
	}
	std::cout << "couldn't find the tag!" << tag << SS( type ) << std::endl;
	return;
L10:

	if ( type == pair_type ) { // increment the counter
		++pair_count(i,j,tag_num);
	} else if ( type == rsd_type ) {
		++rsd_count(i,tag_num);
	} else if ( type == atm_type ) {
// DANGER DANGER:: note the order: j is the atom number!!
		++atom_count(j,i,tag_num);
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin output_decoystats
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
output_decoystats()
{
	using namespace decoystats;

	output_counters(total_decoys);
	output_top_local_bb("DEC");
}

//////////////////////////////////////////////////////////////////////////////
/// @begin output_counters
///
/// @brief
///
/// @detailed
///
/// @param[in]  total
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
output_counters( int const total )
{
	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace misc;

	for ( int type = 1; type <= 3; ++type ) {
		for ( int n = 1, ne = ntags(type); n <= ne; ++n ) {
			if ( type == pair_type ) {
				for ( int i = 1; i <= total_residue; ++i ) {
					for ( int j = 1; j <= total_residue; ++j ) {
						if ( pair_count(i,j,n) > 0 ) {
							DS_PAIR_output(tags(n,type),i,j,float(pair_count(i,j,n))/total);
						}
					}
				}
			} else if ( type == rsd_type ) {
				for ( int i = 1; i <= total_residue; ++i ) {
					DS_RSD_output(tags(n,type),i,float(rsd_count(i,n))/total);
				}
			} else if ( type == atm_type ) {
				for ( int i = 1; i <= total_residue; ++i ) {
					int const aa = res(i);
					int const aav = res_variant(i);
					for ( int j = 1, je = nheavyatoms(aa,aav); j <= je; ++j ) {
						DS_ATOM_output(tags(n,type),i,j,atom_name(j,aa,aav),
						 float(atom_count(j,i,n))/total);
					}
				}
			}
		}                  // n
	}                     // type
}

//////////////////////////////////////////////////////////////////////////////
/// @begin DS_PAIR_output
///
/// @brief
/// so that other functions can make equivalent output
///
/// @detailed
///
/// @param[in]  tag -
/// @param  i - [in/out]? -
/// @param  j - [in/out]? -
/// @param  f - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
DS_PAIR_output(
	std::string const & tag, // input
	int const i,
	int const j,
	float const f
)
{
	std::cout << "DS_PAIR " << tag << ' ' << I( 4, i ) << ' ' << I( 4, j ) <<
	 ' ' << F( 12, 6, f ) << std::endl;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin DS_PAIR_BOX
///
/// @brief
/// so that other functions can make equivalent output
///
/// @detailed
///
/// @param[in]  tag -
/// @param  i - [in/out]? -
/// @param  j - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
DS_PAIR_BOX(
	std::string const & tag, // input
	int const i,
	int const j
)
{
	std::cout << "DS_PAIR_BOX " << tag << ' ' << I( 4, i ) << ' ' << I( 4, j ) <<
	 std::endl;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin DS_RSD_output
///
/// @brief
/// so that other functions can make equivalent output
///
/// @detailed
///
/// @param[in]  tag -
/// @param[in]  i -
/// @param[in]  f -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
DS_RSD_output(
	std::string const & tag, // input
	int const i, // input
	float const f
)
{
	if ( ! get_ds_outpdbonly_flag() ) {
		std::cout << "DS_RSD " << tag << ' ' << I( 4, i ) << ' ' << F( 12, 6, f ) <<
			std::endl;
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin DS_RSD_BOX
///
/// @brief
/// so that other functions can make equivalent output
///
/// @detailed
///
/// @param[in]  tag -
/// @param[in]  i -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
DS_RSD_BOX(
	std::string const & tag, // input
	int const i // input
)
{
	std::cout << "DS_RSD_BOX " << tag << ' ' << I( 4, i ) << std::endl;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin DS_ATOM_output
///
/// @brief
/// so that other functions can make equivalent output
///
/// @detailed
///
/// @param[in]  tag -
/// @param  i - [in/out]? -
/// @param  j - [in/out]? -
/// @param[in]  name -
/// @param  f - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
DS_ATOM_output(
	std::string const & tag, // input
	int const i,
	int const j,
	std::string const & name, // input
	float const f
)
{
	std::cout << "DS_ATOM " << tag << ' ' << I( 4, i ) << ' ' << I( 4, j ) <<
	 ' ' << name << ' ' << F( 12, 6, f ) << std::endl;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin sc_rms_thresholds_initializer
///
/// @brief
///
/// @detailed
///
/// @param  sc_rms_thresholds - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
sc_rms_thresholds_initializer( FArray1D_float & sc_rms_thresholds )
{
	int i = 0;
	sc_rms_thresholds( ++i ) = 0.5;
	sc_rms_thresholds( ++i ) = 1.0;
	sc_rms_thresholds( ++i ) = 1.5;
}



//////////////////////////////////////////////////////////////////////////////
/// @begin compute_environment_changes
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void compute_environment_changes() {

	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace misc;
	using namespace param;
	using namespace void_ns;

	// jk The plan is to loop over all nearby atoms, and compute the difference
	// jk between the distance to each neighbor in the native and in the decoy

	float maxval(-99999.);
	float minval(99999.);

	FArray1D_float res_env_change( MAX_RES(), 0. );
	FArray1D_int num_contacts( total_residue, 0 );

	for ( int i = 1; i <= total_residue; ++i ) {
		int const aai = res(i);
		int const aavi = res_variant(i);

		for ( int j = i+5; j <= total_residue; ++j ) {
			int const aaj = res(j);
			int const aavj = res_variant(j);

			bool native_neighbors(false);
			bool decoy_neighbors(false);
			float dis2(0.);
			are_they_neighbors(res(i),res(j),native_full_coord(1,1,i),
												 native_full_coord(1,1,j), dis2, native_neighbors);
			are_they_neighbors(res(i),res(j),full_coord(1,1,i),
												 full_coord(1,1,j), dis2, decoy_neighbors);

			if ( native_neighbors || decoy_neighbors ) {
				float currval(0.);
				for ( int ia = 1, iae = nheavyatoms(aai,aavi); ia <= iae; ++ia ) {
					for ( int ja = 1, jae = nheavyatoms(aaj,aavj); ja <= jae; ++ja ) {

						// jk Note: we shouldn't really be comparing atom-by-atom without
						//          flipping symmetric atoms (eg. Phe) - this needs to be fixed...

						// jk Initially, try using the difference between r^-2 for each pair
						float native_dis2(0.), decoy_dis2(0.);
						distance2_bk(native_full_coord(1,ia,i),native_full_coord(1,ja,j),native_dis2);
						distance2_bk(full_coord(1,ia,i),full_coord(1,ja,j),decoy_dis2);

						if ( ( native_dis2 < 36. ) || ( decoy_dis2 < 36. ) ) {
							//							float const newval = std::abs( ( 1. / sqrt(native_dis2) ) - ( 1. / sqrt(decoy_dis2) ) );
							//							float const newval = std::abs( ( 1. / native_dis2 ) - ( 1. / decoy_dis2 ) );
							//							float const newval = std::abs( ( 1. / ( native_dis2 * native_dis2 ) ) - ( 1. / ( decoy_dis2 * decoy_dis2 ) ) );
							float const newval = std::abs( sqrt(native_dis2) - sqrt(decoy_dis2) );
							//							float const newval = 100.;

							currval += newval;

							num_contacts(i) += 1;
							num_contacts(j) += 1;
						}
					}
				}

				res_env_change(i) += currval;
				res_env_change(j) += currval;
			} // if are_neighbors

		}

		if ( num_contacts(i) > 0 ) {
			res_env_change(i) = sqrt(res_env_change(i));
			//			res_env_change(i) /= float(num_contacts(i));
			res_env_change(i) /= float(nheavyatoms(aai,aavi));
			res_env_change(i) /= sqrt(float(num_contacts(i)));
			if ( res_env_change(i) > maxval ) maxval = res_env_change(i);
			if ( res_env_change(i) < minval ) minval = res_env_change(i);
		}

	}

	// Scale all the values from 0->10, set the termini to the minimum
	float const newmax(10.);
	float const newmin(0.);
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( num_contacts(i) > 0 ) {
			res_env_change(i) = newmin + ( ( res_env_change(i) - minval ) *
																		 ( newmax - newmin ) / ( maxval - minval ) );
		} else {
			res_env_change(i) = newmin;
		}
	}
	res_env_change(1) = newmin;
	res_env_change(2) = newmin;
	res_env_change(total_residue-1) = newmin;
	res_env_change(total_residue) = newmin;

	std::ofstream sasaprobOUT("allostery_vs_sasaprob.txt",std::ios_base::out);
	std::ofstream LJatrVsPdbOUT("allostery_vs_LJatrVsPDB.txt",std::ios_base::out);
	std::ofstream BallSasaOUT("allostery_vs_ballsasa.txt",std::ios_base::out);

	FArray1D_float rsd_sasa( MAX_RES()() );
	FArray2D_float atom_sasa( MAX_ATOM()(), MAX_RES()() );
	calc_per_atom_sasa( atom_sasa, rsd_sasa, 1.4f, false, false );

	packing_ns::ProteinSasa ps;
	ps.compute_atom_bsasa_score();

	for ( int i = 1; i <= total_residue; ++i ) {
		int const aa = res(i);
		int const aav = res_variant(i);
		float env_change_val = res_env_change(i);
		for ( int j = 1, je = aaproperties_pack::natoms(aa,aav); j <= je; ++j ) {
			make_pdb_ns::bfactor(j,i) = env_change_val;
		}
		if ( ( i > 2 ) && ( i < (total_residue-1) ) ) {
			float const sasa_fraction = std::max( 0.0f, std::min( 1.0f, rsd_sasa(i) / rsd_exposed_sasa(aa) ) );
			if ( sasa_fraction < 0.25 ) {
				float const LJatrVsPdb = fullatom_energies::atrenergy(i)-
					pdbstatistics_pack::atr_avg(aa,template_pack::neighbors(i));
				LJatrVsPdbOUT << env_change_val << ' ' << LJatrVsPdb << ' ' << i << std::endl;
				sasaprobOUT << env_change_val << ' ' << void_ns::rsd_sasaprob(i) << ' ' << i << std::endl;
				float avg_atom_bsasa_weighted(0.);
				int natoms_buried(0);
				for ( int atom1 = 1, atom1e = aaproperties_pack::natoms(aa,aav);  atom1 <= atom1e; ++atom1 ) {
					float const val = ps.atom_bsasa_score_weighted(atom1,i);
					if ( val > 0. ) {
						avg_atom_bsasa_weighted += val;
						++natoms_buried;
					}
				}
				if ( natoms_buried > 0 ) {
					avg_atom_bsasa_weighted /= float(natoms_buried);
				} else {
					avg_atom_bsasa_weighted=0;
				}
				BallSasaOUT << env_change_val << ' ' << avg_atom_bsasa_weighted << ' ' << i << std::endl;
			}
		}
	}
	sasaprobOUT.close();
	LJatrVsPdbOUT.close();
	BallSasaOUT.close();

	return;
}



//////////////////////////////////////////////////////////////////////////////
/// @begin compare_to_native
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
compare_to_native()
{
	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace files_paths;
	using namespace misc;
	using namespace param;

	if ( ! get_native_exists() ) return;

	static int const n_sc_rms_thresholds = { 3 };
	static FArray1D_float const sc_rms_thresholds( n_sc_rms_thresholds,
	 sc_rms_thresholds_initializer );

	int const n_maxsub = { 3 };

	FArray2D_int decoy_rot( MAX_CHI, MAX_RES()() );
	int maxsub_length, aa, aav;
	FArray1D_float residue_rms( MAX_RES()() );
	FArray1D_float dummy_phi( MAX_RES()() );
	FArray1D_float dummy_psi( MAX_RES()() );
	FArray2D_float decoy_chi( MAX_CHI, MAX_RES()() );
	float fa_rms, sc_only_rms, touch_score, rms, logeval, rmsd_threshold;
	FArray1D_bool flipped( MAX_RES()() );
	bool correct_rot;
	FArray1D_bool maxsub_alignment( MAX_RES()() );
	std::string tag;


	if ( ! ds_compare_to_native_init::init ) {
		ds_compare_to_native_init::init = true;

		for ( int i = 1; i <= n_maxsub; ++i ) {
			tag = "MAXSUB-" + lead_zero_string_of( i, 1 );
			new_counter(tag,pair_type);
			new_counter(tag,rsd_type);
		}

		for ( int i = 1; i <= n_sc_rms_thresholds; ++i ) {
			tag = "SC-RMS-" + lead_zero_string_of( i, 1 );
			new_counter(tag,rsd_type);
		}

		tag = "NAT-ROT";
		new_counter( tag, atm_type );
	}

// flip symmetric sidechains to minimize sc-rms to native
	flip_side_chains( full_coord, native_full_coord, flipped );

// rmsd variants
	FArray2D_bool curr_atom_set( MAX_ATOM(), MAX_RES() );
	for ( int nset = 1; nset <= n_atom_set; ++nset ) {

		// for RMS comparisons to native, exclude all residues which have changed sequence (or aav)
		for ( int i = 1; i <= total_residue; ++i ) {
			if ( ( res(i) != native_aa(i) ) || ( res_variant(i) != native_aav(i) ) ) {
				for ( int j = 1; j <= MAX_ATOM(); ++j ) {
					curr_atom_set(j,i) = false;
				}
			} else {
				for ( int j = 1; j <= MAX_ATOM(); ++j ) {
					curr_atom_set(j,i) = atom_set(j,i,nset);
				}
			}
		}
		fa_rms = allatom_rms(full_coord, native_full_coord, curr_atom_set);
		sc_only_rms = sc_rms(full_coord, native_full_coord, curr_atom_set,
		 residue_rms);

		float frac_nat_contacts=0.;
		int total_contacts=0;
		touch_score = allatom_touch_score(full_coord, atom_set(1,1,nset),
		 native_contacts,frac_nat_contacts,total_contacts);

		if ( ! get_ds_outpdbonly_flag() ) {
			std::cout << "DS_DECOY RMS " << start_file <<
				' ' << I( 4, nset ) <<
				' ' << F( 9, 3, fa_rms ) <<
				' ' << F( 9, 3, sc_only_rms ) <<
				' ' << F( 9, 3, touch_score ) <<
				' ' << F( 9, 3, frac_nat_contacts) << std::endl;
		}

		if ( nset == all_heavyatoms ) {

			for ( int i = 1; i <= n_sc_rms_thresholds; ++i ) {
				tag = "SC-RMS-" + lead_zero_string_of( i, 1 );
				for ( int j = 1; j <= total_residue; ++j ) {
					if ( residue_rms(j) <= sc_rms_thresholds(i) ) {
						increment_counter( tag, rsd_type, j, 0 );
					}
				}
			}
			add_decoy_score("touch_sc",touch_score);
			add_decoy_score("Fnatcont",frac_nat_contacts);
			add_decoy_score("tot_cont",total_contacts);
		} else if ( nset == native_buried_rsd_heavyatoms ) {
			add_decoy_score("brsd_rms",fa_rms);
		} else if ( nset == native_buried_heavyatoms ) {
			add_decoy_score("batm_rms",fa_rms);
		} else if ( nset >= chi_atom_set_number(1) && nset <= chi_atom_set_number(3) ) {
			tag = "chi" + lead_zero_string_of( nset - chi_atom_set_number(1), 1 ) + "_rms";
			add_decoy_score( tag, fa_rms );
		}
	}

// maxsub at different thresholds
	for ( int i = 1; i <= n_maxsub; ++i ) {
		tag = "MAXSUB-" + lead_zero_string_of( i, 1 );
		rmsd_threshold = float(i);
		maxsub_native_threshold(position,maxsub_length,rms,logeval, rmsd_threshold,
		 maxsub_alignment);

		if ( i < 3 )            /* only 1,2 */
		 add_decoy_score( tag, float(maxsub_length) );

		if ( ! get_ds_outpdbonly_flag() ) {
			std::cout << "DS MAXSUB " << start_file <<
				' ' << I( 4, i ) << ' ' << I( 4, maxsub_length ) <<
				' ' << F( 9, 3, static_cast< float >( maxsub_length )/total_residue ) <<
				' ' << F( 9, 3, rms ) << std::endl;
		}

		for ( int j = 1; j <= total_residue; ++j ) {
			if ( maxsub_alignment(j) ) {
				increment_counter( tag, rsd_type, j, 0 );

				for ( int k = j+1; k <= total_residue; ++k ) {
					if ( ! decoy_is_native && maxsub_alignment(k) ) {
						increment_counter( tag, pair_type, j, k );
						increment_counter( tag, pair_type, k, j );
					}
				}
			}
		}
	}

	unflip_side_chains( full_coord, flipped);

	// chi angles
	get_chi_and_rot_from_coords(total_residue,res,res_variant,full_coord,decoy_chi,decoy_rot);

	tag = "NAT-ROT";
	for ( int i = 1; i <= total_residue; ++i ) {
		aa = res(i);
		aav = res_variant(i);
		for ( int j = 1, je = nheavyatoms(aa,aav); j <= je; ++j ) {
			correct_rot = true;
			for ( int chi = 1; chi <= MAX_CHI; ++chi ) {
				if ( ! chi_required(chi,j,aa,aav) ) goto L10;
				if ( decoy_rot(chi,i) != native_rot(chi,i) ) {
					correct_rot = false;
					goto L10;
				}
			}
L10:
			if ( correct_rot ) {
				increment_counter( tag, atm_type, i, j );
			}
		}                  // j
	}                     // i

	bool measure_changed_environment(false);
	if ( truefalseoption("measure_changed_environment") ) measure_changed_environment=true;
	if ( measure_changed_environment ) {
		compute_environment_changes();
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin flip_side_chains
///
/// @brief
/// align each residue n-ca-c to native, calculate rms of selected atoms
///
/// @detailed
///
/// @param[in,out]  coord -
/// @param[in]  nat_coord -
/// @param[out]  flipped -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
flip_side_chains(
	FArray3Da_float coord, // input/output
	FArray3Da_float nat_coord, // input
	FArray1D_bool & flipped // output
)
{
	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace misc;
	using namespace param;
	using namespace param_aa;

	coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	nat_coord.dimension( 3, MAX_ATOM(), MAX_RES() );

// local
	int dummy,chi_to_flip;
	float old_sc_msd, new_sc_msd;
	FArray1D_bool use_atom( MAX_ATOM()(), true );

	for ( int i = 1; i <= total_residue; ++i ) {
		int const aa = res(i);
		int const aav = res_variant(i);
		flipped(i) = false;

		// jk jump out if the sequence differs from the native
		if ( ( aa != native_aa(i) ) || ( aav != native_aav(i) ) )
			continue;

		if ( aa == aa_phe || aa == aa_tyr || aa == aa_asp || aa == aa_glu ) {

			if ( aa == aa_glu ) {
				chi_to_flip = 3;
			} else {
				chi_to_flip = 2;
			}

			calc_sc_msd(coord(1,1,i),nat_coord(1,1,i),aa,aav, use_atom,old_sc_msd,
			 dummy);

			flip_sc_chi_angle(coord(1,1,i),aa,aav,chi_to_flip);

			calc_sc_msd(coord(1,1,i),nat_coord(1,1,i),aa,aav, use_atom,new_sc_msd,
			 dummy);

			if ( ! get_ds_outpdbonly_flag() ) {
				std::cout << "msd before/after flipping:" <<
					SS( old_sc_msd ) << SS( new_sc_msd ) << std::endl;
			}

			if ( old_sc_msd <= new_sc_msd ) {
				flip_sc_chi_angle(coord(1,1,i),aa,aav,chi_to_flip); // undo it
			} else {
				if ( ! get_ds_outpdbonly_flag() ) {
					std::cout << "FLIPPING::" << SS( i ) << SS( residue1(i) ) <<
						SS( old_sc_msd ) << SS( new_sc_msd ) << std::endl;
				}
				flipped(i) = true;
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin unflip_side_chains
///
/// @brief
///
/// @detailed
///
/// @param[in,out]  coord -
/// @param[in]  flipped -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
unflip_side_chains(
	FArray3Da_float coord, // input/output
	FArray1D_bool & flipped // input
)
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;
	using namespace param_aa;

	coord.dimension( 3, MAX_ATOM(), MAX_RES() );

	for ( int i = 1; i <= total_residue; ++i ) {
		int const aa = res(i);
		int const aav = res_variant(i);

		if ( flipped(i) ) {

			int chi_to_flip;
			if ( aa == aa_glu ) {
				chi_to_flip = 3;
			} else {
				chi_to_flip = 2;
			}

			flip_sc_chi_angle(coord(1,1,i),aa,aav,chi_to_flip);
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin flip_sc_chi_angle
///
/// @brief
///
/// @detailed
///
/// @param[in,out]  coord
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  chi_to_flip - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
flip_sc_chi_angle(
	FArray2Da_float coord, // input/output
	int const aa,
	int const aav,
	int const chi_to_flip
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using numeric::conversions::radians;

	if ( chi_to_flip > nchi(aa,aav) || chi_to_flip == 0 ) return;

	coord.dimension( 3, MAX_ATOM() );

	//bk retrieve atoms that determine chi angle
	int const c1 = chi_atoms(1,chi_to_flip,aa,aav);
	int const c2 = chi_atoms(2,chi_to_flip,aa,aav);
	int const c3 = chi_atoms(3,chi_to_flip,aa,aav);
	int const c4 = chi_atoms(4,chi_to_flip,aa,aav);

	//bk calculate initial chi angle
	float old_chi;
	dihedral_bk( coord(1,c1), coord(1,c2), coord(1,c3), coord(1,c4), old_chi );

	float const new_chi = old_chi-180.0f;

	FArray2D_float mat( 3, 3 );
	FArray1D_float vec( 3 );

//bk rotate angle by new-initial degrees
	float const rot = radians( new_chi - old_chi );

//bk generate rotation vector and matrix
	getrot_bk( coord(1,c2), coord(1,c3), rot, mat, vec );

//bk move atoms
	for ( int i = 1, ie = natoms( aa, aav ); i <= ie; ++i ) {
		if ( chi_required( chi_to_flip, i, aa, aav ) ) {
			move_bk( coord(1,i), mat, vec );
		}
	}

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin sc_rms
///
/// @brief
/// align each residue n-ca-c to native, calculate rms of selected atoms
///
/// @detailed
///
/// @param  coord1 - [in/out]? -
/// @param  coord2 - [in/out]? -
/// @param[in]  use_atom -
/// @param[out]  residue_rms  -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
sc_rms(
	FArray3Da_float coord1,
	FArray3Da_float coord2,
	FArray2Da_bool use_atom, // input
	FArray1Da_float residue_rms // output
)
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;

	coord1.dimension( 3, MAX_ATOM(), MAX_RES() );
	coord2.dimension( 3, MAX_ATOM(), MAX_RES() );
	use_atom.dimension( MAX_ATOM(), MAX_RES() );
	residue_rms.dimension( MAX_RES() );

// local
	int sc_npoints;
	float sc_msd;

	float msd = 0.0;
	int npoints = 0;

	for ( int i = 1; i <= total_residue; ++i ) {

		calc_sc_msd( coord1(1,1,i),coord2(1,1,i),res(i),res_variant(i),
		 use_atom(1,i),sc_msd, sc_npoints);

		msd += sc_msd;
		npoints += sc_npoints;
		if ( sc_npoints > 0 ) {
			residue_rms(i) = std::sqrt(sc_msd/sc_npoints);
		} else {
			residue_rms(i) = 0.0;
		}
	}

	return ( npoints > 0 ? std::sqrt( msd / npoints ) : 0.0f );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin calc_sc_msd
///
/// @brief
///
/// @detailed
///
/// @param  coord1 - [in/out]? -
/// @param  coord2 - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param[in]  use_atom -
/// @param[out]  sc_msd -
/// @param[out]  sc_npoints -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
calc_sc_msd(
	FArray2Da_float coord1,
	FArray2Da_float coord2,
	int const aa,
	int const aav,
	FArray1Da_bool use_atom, // input
	float & sc_msd, // output
	int & sc_npoints // output
)
{
	using namespace aaproperties_pack;
	using namespace param;

	coord1.dimension( 3, MAX_ATOM() );
	coord2.dimension( 3, MAX_ATOM() );
	use_atom.dimension( MAX_ATOM() );

// local
	FArray2D_float m1( 3, 3 );
	FArray1D_float z1( 3 );
	FArray2D_float m2( 3, 3 );
	FArray1D_float z2( 3 );

	sc_rms_coord_sys(coord1,m1,z1);
	sc_rms_coord_sys(coord2,m2,z2);

	sc_msd = 0.0;
	sc_npoints = 0;

	for ( int j = 5, je = nheavyatoms(aa,aav); j <= je; ++j ) { // dont use  N,CA,C,O
		if ( use_atom(j) ) {
			++sc_npoints;
			sc_msd += calc_sc_atom_msd( coord1(1,j),m1,z1, coord2(1,j),m2,z2 );
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin sc_rms_coord_sys
///
/// @brief
///
/// @detailed
///
/// @param[in]  p - full_coord(1,1,pos): N,CA,C
/// @param  m - [in/out]? -
/// @param  z - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
sc_rms_coord_sys(
	FArray2Da_float p, // input: full_coord(1,1,pos): N,CA,C
	FArray2Da_float m,
	FArray1Da_float z
)
{
	p.dimension( 3, 3 );
	m.dimension( 3, 3 );
	z.dimension( 3 );

	FArray1D_float a1( 3 );
	FArray1D_float a2( 3 );
	FArray1D_float a3( 3 );

	for ( int k = 1; k <= 3; ++k ) {
		z(k) = p(k,2); // C-alpha
	}

	subvec(p(1,1),z,a1); // CA -> N
	subvec(p(1,3),z,a2); // CA -> CO
	unitvec(a1,m(1,1));
	cros(a1,a2,a3);
	unitvec(a3,m(1,3));
	cros(m(1,3),m(1,1),m(1,2));
}

//////////////////////////////////////////////////////////////////////////////
/// @begin calc_sc_atom_msd
///
/// @brief
///
/// @detailed
///
/// @param  coord1 - [in/out]? -
/// @param  m1 - [in/out]? -
/// @param  z1 - [in/out]? -
/// @param  coord2 - [in/out]? -
/// @param  m2 - [in/out]? -
/// @param  z2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
calc_sc_atom_msd(
	FArray1Da_float coord1,
	FArray2Da_float m1,
	FArray1Da_float z1,
	FArray1Da_float coord2,
	FArray2Da_float m2,
	FArray1Da_float z2
)
{
	coord1.dimension( 3 );
	m1.dimension( 3, 3 );
	z1.dimension( 3 );
	coord2.dimension( 3 );
	m2.dimension( 3, 3 );
	z2.dimension( 3 );

	FArray1D_float v( 3 );
	float x1, x2;

	float msd = 0.0;
	for ( int i = 1; i <= 3; ++i ) {
		subvec(coord1,z1,v);
		x1 = dotprod(v,m1(1,i));

		subvec(coord2,z2,v);
		x2 = dotprod(v,m2(1,i));

		msd += square( x1 - x2 );
	}
	return msd;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin allatom_rms
///
/// @brief
///
/// @detailed
///
/// @param  coord1 - [in/out]? -
/// @param  coord2 - [in/out]? -
/// @param  use_atom - [in/out]? - which atoms to use
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
allatom_rms(
	FArray3Da_float coord1,
	FArray3Da_float coord2,
	FArray2Da_bool use_atom // which atoms to use
)
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;

	coord1.dimension( 3, MAX_ATOM(), MAX_RES() );
	coord2.dimension( 3, MAX_ATOM(), MAX_RES() );
	use_atom.dimension( MAX_ATOM(), MAX_RES() );

// local:
	FArray2D_double p1a( 3, MAX_ATOM()() * MAX_RES()() );
	FArray2D_double p2a( 3, MAX_ATOM()() * MAX_RES()() );
	FArray1D_double ww( MAX_ATOM()() * MAX_RES()() );
	FArray2D_double uu( 3, 3 );

	int npoints = 0;
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1, je = natoms(res(i),res_variant(i)); j <= je; ++j ) {
			if ( use_atom(j,i) ) {
				++npoints;
				ww(npoints) = 1.0;
				for ( int k = 1; k <= 3; ++k ) {
					p1a(k,npoints) = coord1(k,j,i);
					p2a(k,npoints) = coord2(k,j,i);
				}
			}
		}
	}

	float fast_rms = 0.0; // Return value
	if ( npoints > 0 ) {
		double ctx;
		findUU(p1a,p2a,ww,npoints,uu,ctx);
		calc_rms_fast(fast_rms,p1a,p2a,ww,npoints,ctx);
	}

	return fast_rms;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_allatom_contacts
///
/// @brief
///
/// @detailed
///
/// @param  coord - [in/out]? -
/// @param  use_atom - [in/out]? -
/// @param[out]  use_atom -
/// @param  contact - [in/out]? - list: res1,atom1,res2,atom2
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_allatom_contacts(
	FArray3Da_float coord,
	FArray2Da_bool use_atom,
	int & num_contacts, // output
	FArray2Da_int contacts // list: res1,atom1,res2,atom2
)
{
	using namespace aaproperties_pack;
	using namespace decoystats;       // max_contacts
	using namespace misc;
	using namespace param;

	coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	use_atom.dimension( MAX_ATOM(), MAX_RES() );
	contacts.dimension( 4, max_contacts );

// constants
	int const min_sep = { 5 }; // min sequence separation btw contacts

// local
	bool nbrs;
	float tmp_dis2;

	num_contacts = 0;

	for ( int i1 = 1; i1 <= total_residue-min_sep; ++i1 ) {
		for ( int i2 = i1 + min_sep; i2 <= total_residue; ++i2 ) {
			are_they_neighbors(res(i1),res(i2),coord(1,1,i1),coord(1,1,i2),tmp_dis2,
			 nbrs);
			if ( nbrs ) {
				for ( int j1 = 1, j1e = nheavyatoms(res(i1),res_variant(i1));
				 j1 <= j1e; ++j1 ) {
					if ( use_atom(j1,i1) ) {
						for ( int j2 = 1, j2e = nheavyatoms(res(i2),res_variant(i2));
						 j2 <= j2e; ++j2 ) {
							if ( use_atom(j2,i2) &&
							 atoms_in_contact(coord(1,j1,i1),coord(1,j2,i2)) ) {
								++num_contacts;
								if ( num_contacts >= max_contacts ) {
									std::cout << "too many contacts!!" << std::endl;
									utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
								}
								contacts(1,num_contacts) = i1;
								contacts(2,num_contacts) = j1;
								contacts(3,num_contacts) = i2;
								contacts(4,num_contacts) = j2;
							}
						}
					}
				}
			}
		}
	}

	contacts(1,num_contacts+1) = 0; // signal the end of the list
	contacts(2,num_contacts+1) = 0;
	contacts(3,num_contacts+1) = 0;
	contacts(4,num_contacts+1) = 0;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin allatom_touch_score
///
/// @brief
///
/// @detailed
/// this is totally inefficient!!
/// but should only be called once or twice per decoy
///
/// @param  coord - [in/out]? -
/// @param  use_atom - [in/out]? -
/// @param  correct_contacts - [in/out]? - list: res1,atom1,res2,atom2
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
allatom_touch_score(
	FArray3Da_float coord,
	FArray2Da_bool use_atom,
	FArray2Da_int correct_contacts, // list: res1,atom1,res2,atom2
	float & frac_nat_contacts,
	int & total_contacts
)
{
	using namespace decoystats;
	using namespace param;

	coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	use_atom.dimension( MAX_ATOM(), MAX_RES() );
	correct_contacts.dimension( 4, max_contacts );

// local:
	FArray2D_int dummy_contacts( 4, max_contacts );
	int i1, i2, j1, j2, total_correct_contacts, num_nat_contacts;
	frac_nat_contacts=0.0;

// count total contacts
	get_allatom_contacts(coord, use_atom, total_contacts, dummy_contacts);

// count correct contacts
	total_correct_contacts = 0;
	num_nat_contacts = 0;
	for ( int ii = 1; ii <= max_contacts; ++ii ) {
		i1 = correct_contacts(1,ii);
		j1 = correct_contacts(2,ii);
		i2 = correct_contacts(3,ii);
		j2 = correct_contacts(4,ii);
		if ( i1 == 0 ) goto L10; // end of list
		if ( use_atom(j1,i1) && use_atom(j2,i2) ) {
			++num_nat_contacts;
			if ( atoms_in_contact( coord(1,j1,i1), coord(1,j2,i2) ) )
			++total_correct_contacts;
		}
	}
L10:; // escape for the end of the list

	float allatom_touch_score; // Return value
	if ( total_contacts > 0 ) {
		allatom_touch_score =
		 static_cast< float >( total_correct_contacts ) / total_contacts;
	} else {
		allatom_touch_score = 0.0;
	}
	if ( num_nat_contacts > 0 ) {
		frac_nat_contacts =
		 static_cast< float >( total_correct_contacts ) / num_nat_contacts;
	} else {
		frac_nat_contacts  = 0.0;
	}

	return allatom_touch_score;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin atoms_in_contact
///
/// @brief
/// right now Im assuming only heavyatoms
///
/// @detailed
///
/// @param  v - [in/out]? -
/// @param  w - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
atoms_in_contact(
	FArray1Da_float v,
	FArray1Da_float w
)
{
	v.dimension( 3 );
	w.dimension( 3 );

	float const threshold = { 20.25 }; // 4.5**2; should make this atom-specific??

	float const dist2 =
	 square( v(1) - w(1) ) + square( v(2) - w(2) ) + square( v(3) - w(3) );

	bool atoms_in_contact; // Return value
	if ( dist2 <= threshold ) {
		atoms_in_contact = true;
	} else {
		atoms_in_contact = false;
	}
	return atoms_in_contact;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_use_big_polar_h
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
get_use_big_polar_h()
{
	using namespace use_big_polar_h_common;

	return use_big_polar_h;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin set_use_big_polar_h
///
/// @brief
///
/// @detailed
///
/// @param[in]  setting -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_use_big_polar_h( bool const setting )
{
	using namespace use_big_polar_h_common;

	use_big_polar_h = setting;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin set_bfactor_by_atom_set
///
/// @brief
///
/// @detailed
///
/// @param[in]  use_atom
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_bfactor_by_atom_set( FArray2Da_bool use_atom )
{

	using namespace misc;
	using namespace param;

	use_atom.dimension( MAX_ATOM(), MAX_RES() );

	for ( int i = 1, max_atom = MAX_ATOM(); i <= total_residue; ++i ) {
		for ( int j = 1; j <= max_atom; ++j ) {
			if ( use_atom(j,i) ) {
				make_pdb_ns::bfactor(j,i) = 50.0;
			} else {
				make_pdb_ns::bfactor(j,i) = 0.0;
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin exposed_rsd_sasa_initializer
///
/// @brief
///
/// @detailed
///
/// @param  exposed_rsd_sasa - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
exposed_rsd_sasa_initializer( FArray1D_float & exposed_rsd_sasa )
{
	using namespace param;
	using namespace param_aa;

	exposed_rsd_sasa( aa_ala ) = 115.594; // 1 A
	exposed_rsd_sasa( aa_cys ) = 136.441; // 2 C
	exposed_rsd_sasa( aa_asp ) = 166.109; // 3 D
	exposed_rsd_sasa( aa_glu ) = 187.047; // 4 E
	exposed_rsd_sasa( aa_phe ) = 227.126; // 5 F
	exposed_rsd_sasa( aa_gly ) =  95.379; // 6 G
	exposed_rsd_sasa( aa_his ) = 174.877; // 7 H
	exposed_rsd_sasa( aa_ile ) = 175.089; // 8 I
	exposed_rsd_sasa( aa_lys ) = 215.766; // 9 K
	exposed_rsd_sasa( aa_leu ) = 178.404; // 10 L
	exposed_rsd_sasa( aa_met ) = 187.310; // 11 M
	exposed_rsd_sasa( aa_asn ) = 142.086; // 12 N
	exposed_rsd_sasa( aa_pro ) = 146.215; // 13 P
	exposed_rsd_sasa( aa_gln ) = 180.624; // 14 Q
	exposed_rsd_sasa( aa_arg ) = 216.548; // 15 R
	exposed_rsd_sasa( aa_ser ) = 125.275; // 16 S
	exposed_rsd_sasa( aa_thr ) = 138.319; // 17 T
	exposed_rsd_sasa( aa_val ) = 153.885; // 18 V
	exposed_rsd_sasa( aa_trp ) = 229.518; // 19 W
	exposed_rsd_sasa( aa_tyr ) = 215.013; // 20 Y
//KMa phospho_ser 2006-01
	if ( add_pser() )
	exposed_rsd_sasa( aa_sep ) = 125.275; // 20 Y
}


//////////////////////////////////////////////////////////////////////////////
/// @begin rsd_exposed_sasa
///
/// @brief
///
/// @detailed
///
/// @param  rsd - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
rsd_exposed_sasa( int const rsd )
{
	using namespace param;
	using namespace param_aa;
//KMa phospho_ser 2006-01
	static FArray1D_float const exposed_rsd_sasa( MAX_AA_PLUS(),
	 exposed_rsd_sasa_initializer );

	float rsd_exposed_sasa; // Return value
//KMa phospho_ser 2006-01
	if ( rsd <= 0 || rsd > MAX_AA_PLUS() ) {
		rsd_exposed_sasa = 999.0;
	} else {
		rsd_exposed_sasa = exposed_rsd_sasa(rsd);
	}

	return rsd_exposed_sasa;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin generate_hbond_info
///
/// @brief
///
/// @detailed
///
/// @param  dhatm_xyz - [in/out]? -
/// @param  aatm_xyz - [in/out]? -
/// @param  datm_xyz - [in/out]? -
/// @param  abase_xyz - [in/out]? -
/// @param  dis_ha - [in/out]? -
/// @param  dis_ad - [in/out]? -
/// @param  psi_hb - [in/out]? -
/// @param  phi_hb - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
generate_hbond_info(
	FArray1Da_float dhatm_xyz,
	FArray1Da_float aatm_xyz,
	FArray1Da_float datm_xyz,
	FArray1Da_float abase_xyz,
	float & dis_ha,
	float & dis_dh,
	float & dis_ad,
	float & psi_hb,
	float & phi_hb
)
{
	dhatm_xyz.dimension( 3 );
	aatm_xyz.dimension( 3 );
	datm_xyz.dimension( 3 );
	abase_xyz.dimension( 3 );

	distance_bk(dhatm_xyz(1),aatm_xyz(1),dis_ha);
	distance_bk(datm_xyz(1),dhatm_xyz(1),dis_dh);
	distance_bk(datm_xyz(1),aatm_xyz(1),dis_ad);
	angle_bk(datm_xyz(1),dhatm_xyz(1), aatm_xyz(1),dhatm_xyz(1),psi_hb);
	numeric::conversions::to_degrees(psi_hb);
	dihedral_bk(datm_xyz(1),dhatm_xyz(1), aatm_xyz(1),abase_xyz(1),phi_hb);
}


//////////////////////////////////////////////////////////////////////////////
/// @begin dump_hbond_info
///
/// @brief
///
/// @detailed
///
/// @param  start_file - [in/out]? -
/// @param  dres - [in/out]? -
/// @param  dres_type - [in/out]? -
/// @param  datm - [in/out]? -
/// @param  dtype - [in/out]? -
/// @param  ares - [in/out]? -
/// @param  ares_type - [in/out]? -
/// @param  aatm - [in/out]? -
/// @param  atype - [in/out]? -
/// @param  dis_ha - [in/out]? -
/// @param  dis_dh - [in/out]? -
/// @param  dis_ad - [in/out]? -
/// @param  psi_hb - [in/out]? -
/// @param  phi_hb - [in/out]? -
/// @param  hb_num - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
dump_hbond_info(
	std::string const & start_file,
	int const dres,
	int const dres_type,
	int const datm,
	int const dtype,
	int const ares,
	int const ares_type,
	int const aatm,
	int const atype,
	float const dis_ha,
	float const dis_dh,
	float const dis_ad,
	float const psi_hb,
	float const phi_hb,
	int const hb_num
)
{
	using namespace aaproperties_pack;
	using namespace hbonds;
	using namespace template_pack;

	std::string dres_name, ares_name;
	std::string datm_name, aatm_name, aatm_type, datm_type;
	std::string hbond_status;
	std::string hb_type;

	name_from_num(ares_type, ares_name);
	name_from_num(dres_type, dres_name);

	int don_nb = neighbors( dres );
	int act_nb = neighbors( ares );

	datm_name = atom_name( datm, dres_type, 1 );
	aatm_name = atom_name( aatm, ares_type, 1 );

	aatm_type = atom_type_name( atype );
	datm_type = atom_type_name( dtype );

	if ( hbond_set.hbond_allowed(hb_num) ) {
		hbond_status = "ALLOWED";
	} else {
		hbond_status = "NT_ALWD";
	}

//cmd datm_state and aatm_state reflect the charge state of the acceptors
//cmd and donors in the hydrogen bond (1=charged; 2=neutral)

	int aatm_state = 2;
	int datm_state = 2;
	if ( atype == 15 ) {
		aatm_state = 1;
	}
	if ( dtype == 10 ) {
		datm_state = 1;
	} else if ( dtype == 11 ) {
		if ( datm != 8 ) {
			datm_state = 1;
		}
	}

	if ( aatm_state == 2 ) {
		if ( datm_state == 2 ) {
			hb_type = "NEUT_NEUT";
		} else if ( datm_state == 1 ) {
			hb_type = "CHRG_NEUT";
		} else {
			std::cout << "Invalid charge state for hbond " << SS( hb_num ) << std::endl;
		}
	} else if ( aatm_state == 1 ) {
		if ( datm_state == 2 ) {
			hb_type = "CHRG_NEUT";
		} else if ( datm_state == 1 ) {
			hb_type = "CHRG_CHRG";
		} else {
			std::cout << "Invalid charge state for hbond " << SS( hb_num ) << std::endl;
		}
	} else {
		std::cout << "Invalid charge state for hbond " << SS( hb_num ) << std::endl;
	}

	if ( ! get_ds_outpdbonly_flag() ) {
		std::cout << A( 5, start_file ) << "  " << I( 3, dres ) << ' ' << dres_name <<
			' ' << I( 2, don_nb ) << ' ' << datm_name << ' ' << datm_type <<
			"  " << I( 3, ares ) << ' ' << ares_name << ' ' << I( 2, act_nb ) <<
			' ' << aatm_name << ' ' << aatm_type << "  " << F( 6, 3, dis_ha ) <<
			' ' << F( 6, 3, dis_dh ) << ' ' << F( 6, 3, dis_ad ) <<
			' ' << F( 9, 3, psi_hb ) << ' ' << F( 9, 3, phi_hb ) <<
			' ' << hb_type << ' ' << hbond_status << std::endl;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin uns_type
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
std::string
decide_uns_type( unsatisfied_buried_polar & uns )
{
	using namespace aaproperties_pack;
	using namespace param_aa;

	std::string uns_type;

	int aa;
	num_from_res3( uns.get_residue_type(), aa );

	if ( is_DNA(aa) ) {
		int atm = uns.get_atm();
		//ja this is a map of vectors of bools
		if ( major_groove_hb_atom[ na_gua ][ atm ] ) uns_type = "MAJOR";
		else if ( atm <= dna_bb_atoms ) uns_type = "PHOSB";
		else uns_type = "OTHER";
	} else if ( uns.backbone() ) {
		if ( uns.donor() )	uns_type ="BBDON";
		else uns_type ="BBACC";
	} else {
		if ( uns.donor() )	uns_type ="SCDON";
		else uns_type ="SCACC";
	}
	return uns_type;
}



//------------------------------------------------------------------------------
//-----------------------------------END----------------------------------------
