// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: sheffler $


// Rosetta Headers
#include "decoystats_interface_classes.h"
#include "decoystats.h"
#include "docking_ns.h"
#include "misc.h"
#include "param_aa.h"
#include "template_pack.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <string>



////////////////////////////////////////////////////////////////
//////  STARTING METHODS FOR interface_energy_quantifier  //////
////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_energy_quantifier::reset
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_energy_quantifier::reset() {

	include_delta = false;
	include_interface_sum = false;
	include_VsPDB = false;
	include_max = false;
	include_min = false;
	include_maxVsPDB = false;
	include_minVsPDB = false;

	delta_val = 0.;
	interface_sum = 0.;
	core_interface_sum = 0.;
	interface_sum_VsPDB = 0.;
	core_interface_sum_VsPDB = 0.;
	interface_max = -999.;
	core_interface_max = -999.;
	interface_min = 999.;
	core_interface_min = 999.;
	interface_max_VsPDB = -999.;
	core_interface_max_VsPDB = -999.;
	interface_min_VsPDB = 999.;
	core_interface_min_VsPDB = 999.;

	initialized = false;

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_energy_quantifier::reset_energies
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_energy_quantifier::reset_energies() {

	delta_val = 0.;
	interface_sum = 0.;
	core_interface_sum = 0.;
	interface_sum_VsPDB = 0.;
	core_interface_sum_VsPDB = 0.;
	interface_max = -999.;
	core_interface_max = -999.;
	interface_min = 999.;
	core_interface_min = 999.;
	interface_max_VsPDB = -999.;
	core_interface_max_VsPDB = -999.;
	interface_min_VsPDB = 999.;
	core_interface_min_VsPDB = 999.;

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_energy_quantifier::allow_min
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_energy_quantifier::allow_min() {

	if ( include_interface_sum ) {
		include_min=true;
	} else {
		std::cerr << "Cannot allow_min unless interface sums are computed";
		std::exit( EXIT_FAILURE );
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_energy_quantifier::allow_max
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_energy_quantifier::allow_max() {

	if ( include_interface_sum ) {
		include_max=true;
	} else {
		std::cerr << "Cannot allow_max unless interface sums are computed";
		std::exit( EXIT_FAILURE );
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_energy_quantifier::allow_min_VsPDB
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_energy_quantifier::allow_min_VsPDB() {

	if ( include_VsPDB ) {
		include_minVsPDB=true;
	} else {
		std::cerr << "Cannot allow_min_VsPDB unless interface sums VsPDB are computed";
		std::exit( EXIT_FAILURE );
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_energy_quantifier::allow_max_VsPDB
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_energy_quantifier::allow_max_VsPDB() {

	if ( include_VsPDB ) {
		include_maxVsPDB=true;
	} else {
		std::cerr << "Cannot allow_max_VsPDB unless interface sums VsPDB are computed";
		std::exit( EXIT_FAILURE );
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_energy_quantifier::compute_energies
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_energy_quantifier::compute_energies(
  FArray1DB_bool & interface_res,
	FArray1DB_bool & core_interface_res
) {

	using namespace docking;
	using namespace misc;
	using namespace template_pack;

	if ( ! initialized ) {
		std::cerr << "Cannot compute energies for uninitialized quantifier";
		std::exit( EXIT_FAILURE );
	}

	reset_energies();

	if ( include_delta ) {
		for ( int i = part_begin(1); i <= part_end(1); ++i ) {
			for ( int j = part_begin(2); j <= part_end(2); ++j ) {
				delta_val += (*ene_pair)(i,j);
			}
		}
	}

	if ( include_interface_sum || include_VsPDB || include_min ||
			 include_max || include_minVsPDB || include_maxVsPDB ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			if ( interface_res(i) ) {
				int const nb = neighbors(i);
				int const aa = res(i);

				float const val = (*ene_res)(i);
				if ( include_interface_sum ) {
					interface_sum += val;
					if ( core_interface_res(i) ) {
						core_interface_sum += val;
					}

					if ( include_min ) {
						if ( val < interface_min ) {
							interface_min = val;
						}
						if ( ( val < core_interface_min ) && core_interface_res(i) ) {
							core_interface_min = val;
						}
					}

					if ( include_max ) {
						if ( val > interface_max ) {
							interface_max = val;
						}
						if ( ( val > core_interface_max ) && core_interface_res(i) ) {
							core_interface_max = val;
						}
					}

				}

				if ( include_VsPDB ) {
					float const val_VsPDB = val - ((*ene_VsPDB)(aa,nb));
					interface_sum_VsPDB += val_VsPDB;
					if ( core_interface_res(i) ) core_interface_sum_VsPDB += val_VsPDB;

					if ( include_minVsPDB ) {
						if ( val_VsPDB < interface_min_VsPDB ) {
							interface_min_VsPDB = val_VsPDB;
						}
						if ( ( val_VsPDB < core_interface_min_VsPDB ) && ( core_interface_res(i) ) ) {
							core_interface_min_VsPDB = val_VsPDB;
						}
					}

					if ( include_maxVsPDB ) {
						if ( val_VsPDB > interface_max_VsPDB ) {
							interface_max_VsPDB = val_VsPDB;
						}
						if ( ( val_VsPDB > core_interface_max_VsPDB ) && ( core_interface_res(i) ) ) {
							core_interface_max_VsPDB = val_VsPDB;
						}
					}
				}

			}
		}
	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin interface_energy_quantifier::write_to_PDB
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_energy_quantifier::write_to_PDB( int num_interface_res, int num_core_interface_res ) {

	if ( ! initialized ) {
		std::cerr << "Cannot write to PDB for uninitialized quantifier";
		std::exit( EXIT_FAILURE );
	}

	if ( include_delta ) {
		std::ostringstream name_stream;
		name_stream << "delta_" << name;
		add_decoy_score(name_stream.str(), delta_val);
	}

	if ( include_interface_sum ) {

		std::ostringstream name_stream;
		name_stream << "interface_" << name;
		add_decoy_score(name_stream.str(), interface_sum);
		std::ostringstream ave_name_stream;
		ave_name_stream << "interface_ave_" << name;
		add_decoy_score(ave_name_stream.str(),
		 interface_sum / static_cast<float>(num_interface_res));

		if ( num_core_interface_res > 0 ) {
			std::ostringstream core_name_stream;
			core_name_stream << "core_interface_" << name;
			add_decoy_score(core_name_stream.str(), core_interface_sum);
			std::ostringstream core_ave_name_stream;
			core_ave_name_stream << "core_interface_ave_" << name;
			add_decoy_score(core_ave_name_stream.str(),
		   core_interface_sum / static_cast<float>(num_core_interface_res));
		}

		if ( include_min ) {
			std::ostringstream name_stream;
			name_stream << "interface_min_" << name;
			add_decoy_score(name_stream.str(), interface_min);
			if ( num_core_interface_res > 0 ) {
				std::ostringstream core_name_stream;
				core_name_stream << "core_interface_min_" << name;
				add_decoy_score(core_name_stream.str(), core_interface_min);
			}
		}

		if ( include_max ) {
			std::ostringstream name_stream;
			name_stream << "interface_max_" << name;
			add_decoy_score(name_stream.str(), interface_max);
			if ( num_core_interface_res > 0 ) {
				std::ostringstream core_name_stream;
				core_name_stream << "core_interface_max_" << name;
				add_decoy_score(core_name_stream.str(), core_interface_max);
			}
		}
	}

	if ( include_VsPDB ) {
		std::ostringstream name_stream;
		name_stream << "interface_" << name << "_VsPDB";
		add_decoy_score(name_stream.str(), interface_sum_VsPDB);
		std::ostringstream ave_name_stream;
		ave_name_stream << "interface_ave_" << name << "_VsPDB";
		add_decoy_score(ave_name_stream.str(),
		 interface_sum_VsPDB / static_cast<float>(num_interface_res));

		if ( num_core_interface_res > 0 ) {
			std::ostringstream core_name_stream;
			core_name_stream << "core_interface_" << name << "_VsPDB";
			add_decoy_score(core_name_stream.str(), core_interface_sum_VsPDB);
			std::ostringstream core_ave_name_stream;
			core_ave_name_stream << "core_interface_ave_" << name << "_VsPDB";
			add_decoy_score(core_ave_name_stream.str(),
		   core_interface_sum_VsPDB / static_cast<float>(num_core_interface_res));
		}

		if ( include_minVsPDB ) {
			std::ostringstream name_stream;
			name_stream << "interface_min_" << name << "_VsPDB";
			add_decoy_score(name_stream.str(), interface_min_VsPDB);
			if ( num_core_interface_res > 0 ) {
				std::ostringstream core_name_stream;
				core_name_stream << "core_interface_min_" << name << "_VsPDB";
				add_decoy_score(core_name_stream.str(), core_interface_min_VsPDB);
			}
		}
		if ( include_maxVsPDB ) {
			std::ostringstream name_stream;
			name_stream << "interface_max_" << name << "_VsPDB";
			add_decoy_score(name_stream.str(), interface_max_VsPDB);
			if ( num_core_interface_res > 0 ) {
				std::ostringstream core_name_stream;
				core_name_stream << "core_interface_max_" << name << "_VsPDB";
				add_decoy_score(core_name_stream.str(), core_interface_max_VsPDB);
			}
		}

	}

	return;
}


/////////////////////////////////////////////////////////////////
//////  DONE WITH METHODS FOR interface_energy_quantifier  //////
/////////////////////////////////////////////////////////////////

