// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7973 $
//  $Date: 2006-04-20 23:55:51 -0700 (Thu, 20 Apr 2006) $
//  $Author: johnk $

#ifndef INCLUDED_disulfides
#define INCLUDED_disulfides


// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray3Dp.hh>

namespace disulfides {

///////////////
/// Globals ///
///////////////
namespace options { // command lines options for disulfides
	extern bool find_disulf; // cys-cys pairing not known,
		// find & score max # of disulfide bonds
	extern bool fix_disulf; // cys-cys pairing is known, score those disulfs
	extern bool use_disulf_logfile; // output physical parameters of disulfides
		// to a user-defined filename
	extern bool norepack_disulf; // should packer or rotamer trials alter disulfs?
	extern bool minimize_disulf_chi;
}

namespace constants {
	// counting
	extern int const max_cysteines;

	extern int const max_disulfides;

	extern float const disulf_cendist_cutoff; // angstroms^2 it was 49

	extern float const max_native_disulf_cendist;
	 // max cendist seen in native disulf (approx.20 ang^2)

	extern float const min_native_disulf_cendist;
	 // min cendist seen in native disulf (approx.2 ang^2)

	extern float const disulf_cbdist_cutoff; // angstroms^2

	extern float const min_native_disulf_cbdist;
	 // min cbdist seen in native disulf (approx.10 ang^2)

	extern float const max_native_disulf_cbdist;
	 // max cbdist seen in native disulf (approx.20 ang^2)

	extern float const disulf_ssdist_cutoff; // angstroms

	extern int const disulf_min_sequence_separation; // residues

	//bs ** Note: physical parameters of disulfide bonds are given
	//bs in function score_this_disulfide_fa in disulfides.cc
}


////////////////
/// Boundary ///
////////////////
// yab:
// functions here act as an interface between the classic monolithic
// Rosetta world and the new object world until misc removal is
// complete and Rosetta's computational pipeline is detangled.
namespace BOUNDARY {

void
setup_disulfide_filter();

bool
get_disulf_flag();

bool
get_minimize_disulf_chi();

void
get_disulfide_options_and_init_module();

bool
get_norepack_disulf();

void
evaluate_disulfides_centroid(); // mirrors public methods

void
output_disulf_stats_to_log(
	std::string & filename,
	std::string const & tag,
	bool const fullatom
); // mirrors public methods

void
identify_cys(); // mirrors public methods

bool
cys_pair_in_disulf(
	int res1,
	int res2
); // mirrors public methods

void
disulf_apply_filter(
	bool & accepted,
	std::string & tag
); // mirrors public methods

bool
cys_res_in_disulf( int resnum ); // mirrors public methods

float
getTotWtdScThisDisulf_pack(
	int const & cys1,
	FArray2Da_float cys1coord, // fullatom coords for cys1
	int const & cys2,
	FArray2Da_float cys2coord // fullatom coords for cys2
); // mirrors public methods

void
disulf_get_tot_wtd_cent_score( float & disulf_cent_score /* total weighted centroid score for entire protein */
); // mirrors public methods

void
disulf_eval_fa_scores( float & disulf_fa_score /* total weighted fullatom score for entire protein */
); // mirrors public methods

bool
AtomVdwDisulfNativeClash(
	int atom1_in,
	int atom2_in
); // mirrors public methods

void
eval_disulf_dist_deriv(
	int res1,
	int atom1,
	int res2,
	int atom2,
	bool fullatom,
	float & deriv
); // mirrors public methods

void
eval_disulf_fullatom_energy(); // mirrors public methods

int
get_n_disulf_cent_natcut(); // mirrors public methods

int
get_n_disulf_centroid(); // mirrors public methods

int
get_n_disulf_fa(); // mirrors public methods

int
get_n_known_disulfides(); // mirrors public methods

int
get_cys(
	int pos
); // mirrors public methods

int
get_disulf_partner_a(
	int pos
); // mirrors public methods

int
get_disulf_partner_b(
	int pos
); // mirrors public methods

float
get_fullatom_disulf_dist_sc(); // mirrors public methods

float
get_fullatom_disulf_csang_sc(); // mirrors public methods

float
get_fullatom_disulf_dihedral_sc(); // mirrors public methods

float
get_fa_disulf_ca_dihed_sc(); // mirrors public methods

}

/////////////
/// Class ///
/////////////
class Disulfides {

	public: // Creation/Destruction
		Disulfides(
			FArray3Dp_float const * Eposition,
			FArray2D_float const * centroid,
			FArray3D_float const * full_coord,
			FArray1D_float const * phi,
			FArray1D_float const * psi,
			FArray1D_int const * res,
			int const * total_residue
		);

		Disulfides();
		~Disulfides();


	public: // Methods

		void
		evaluate_disulfides_centroid();

		void
		identify_cys();

		bool
		cys_pair_in_disulf(
			int res1,
			int res2
		);

		bool
		cys_res_in_disulf( int resnum );

		void
		disulf_apply_filter(
			bool & accepted,
			std::string & tag
		);

		bool
		get_use_disulf_logfile();

		float
		getTotWtdScThisDisulf_pack(
			int const & cys1,
			FArray2Da_float cys1coord, // fullatom coords for cys1
			int const & cys2,
			FArray2Da_float cys2coord // fullatom coords for cys2
		);

		void
		disulf_get_tot_wtd_cent_score( float & disulf_cent_score /* total weighted centroid score for entire protein */
		);

		void
		disulf_eval_fa_scores( float & disulf_fa_score /* total weighted fullatom score for entire protein */
		);

		bool
		AtomVdwDisulfNativeClash(
			int atom1_in,
			int atom2_in
		);

		void
		eval_disulf_dist_deriv(
			int res1,
			int atom1,
			int res2,
			int atom2,
			bool fullatom,
			float & deriv
		);

		void
		eval_disulf_fullatom_energy();

		int
		get_cys(
			int pos
		);

		int
		get_disulf_partner_a(
			int pos
		);

		int
		get_disulf_partner_b(
			int pos
		);

		int
		get_n_disulf_cent_natcut();

		int
		get_n_disulf_centroid();

		int
		get_n_disulf_fa();

		int
		get_n_known_disulfides();

		float
		get_fullatom_disulf_dist_sc();

		float
		get_fullatom_disulf_csang_sc();

		float
		get_fullatom_disulf_dihedral_sc();

		float
		get_fa_disulf_ca_dihed_sc();

		void
		read_disulf_connectivity( bool const fullatom );

		void
		output_disulf_stats_to_log(
			std::string & filename,
			std::string const & tag,
			bool const fullatom
		);

		void
		output_cys_chi_to_logfile();

	private: // Methods

		void
		initialize_disulfide_globals();

		void
		count_disulfides( bool fullatom );

		void
		find_and_count_disulfides( bool fullatom );

		void
		update_cyscys_cendist();

		void
		put_disulfides_in_ordered_lists();

		void
		get_disulf_chis(
			FArray2Da_float coord1,
			FArray2Da_float coord2,
			float & chi_1,
			float & chi_2
		);

		void
		get_cys_neighbors(
			FArray1Da_int n_neighbors, // number of other cys within cutoff distance
			FArray2Da_float neighbor_distances, // distances to neighbors within cutoff
			bool fullatom
		);

		void
		pick_disulf_from_neighbors(
			FArray1Da_int n_neighbors,
			 // number of other cys within cutoff distance,gets zeroed in algorithm.
			FArray2Da_float neighbor_distances, // distances to neighbors within cutoff
			bool fullatom
		);

		int
		which_disulf_pair(
			int res1,
			int res2
		);

		bool
		disulf_filter();

		void
		score_disulf_cendist();

		void
		score_disulf_cbdist();

		void
		score_disulf_cacbcb();

		void
		score_disulf_cacbcbca();

		void
		score_disulf_bb_dih();

		void
		score_this_disulfide_fa(
		FArray2Da_float coord1, // xyz coords of cys 1
		FArray2Da_float coord2, // xyz coords of cys 2
		float & distance_score_this_disulfide,
		float & csangles_score_this_disulfide,
		float & dihedral_score_this_disulfide,
		float & ca_dihedral_sc_this_disulf,
		bool & truefalse_fa_disulf
		);

		void
		score_this_disulfide_fa_alt(
			FArray2Da_float coord1, // xyz coords of cys 1
			FArray2Da_float coord2, // xyz coords of cys 2
			float & distance_score_this_disulfide,
			float & csangles_score_this_disulfide,
			float & dihedral_score_this_disulfide,
			float & ca_dihedral_sc_this_disulf,
			bool & truefalse_fa_disulf
		);

		void
		set_cent_disulf_scores_to_zero();

		void
		set_fa_disulfide_scores_to_zero();

		void
		update_disulf_fa_scores(
			bool & truefalse_fa_disulf, // did it satisfy fa criteria
			float & dist_sc_this_disulf,
			float & csang_sc_this_disulf,
			float & dihedral_sc_this_disulf,
			float & ca_dihedral_sc_this_disulf,
			float & cbs_dist_sc_this_disulf
		);

		float
		get_disulf_cendist_deriv(
			int res1,
			int atom1,
			int res2,
			int atom2,
			int disulf_pair_i,
			bool fullatom
		);


		float
		get_disulf_cbdist_deriv(
			int res1,
			int atom1,
			int res2,
			int atom2,
			int disulf_pair_i,
			bool fullatom
		);


		float
		get_disulf_ssdist_deriv(
			int res1,
			int atom1,
			int res2,
			int atom2,
			int disulf_pair_i,
			bool fullatom
		);

		void
		eval_disulf_ang_deriv(
			int const i,
			int const ii,
			int const j,
			int const jj,
			int const aai,
			int const aaj,
			int const aavi,
			int const aavj,
			bool const fullatom,
			FArray1Da_float F1_disulfE,
			FArray1Da_float F2_disulfE
		);

		void
		get_disulf_csang_deriv(
			FArray1Da_float Cbxyz, // cys1 - analog to acceptor base
			FArray1Da_float Sxyz, // cys1 - analog to acceptor
			FArray1Da_float Cb2xyz, // cys2 - analog to donor
			FArray1Da_float S2xyz, // cys2 - analog to H
			float & energy,
			FArray2Da_float deriv // xyz,f1/f2
		);

		void
		disulf_csang_compute_energy(
			float const SS2dis,
			float const xS,
			float const xS2,
			float & E_r,
			float & E_xS,
			float & E_xS2,
			float & dE_dr,
			float & dE_dxS,
			float & dE_dxS2
		);

		float
		get_disulf_ssdist_deriv_II( float const ssdist );

		void
		get_disulfide_params(
			FArray2Da_float coord1, // coords of cys 1
			FArray2Da_float coord2, // coords of cys 2
			float & dist_between_sulfurs, // dist between cys sulfurs
			float & cs_bond_angle_1,
			float & cs_bond_angle_2,
			float & disulf_dihedral_angle, // dihedral (torsion) angle, cbeta-s-s-cbeta
			float & disulf_ca_dihedral_angle_1,
			 // dihedral (torsion) angle, calpha1-cbeta1-s1-s2
			float & disulf_ca_dihedral_angle_2 // dihedral (torsion) angle, calpha2-cbeta2-s2-s1
		);

		void
		get_disulf_dist_use_globl_crds(
			int & cys_1,
			int & cys_2,
			float & dist_between_sulfurs // dist between cys sulfurs
		);

		void
		get_disulf_bb_dih(
			FArray2Da_float cys1coord, // coords of cys 1 from full_coord
			FArray2Da_float cys2coord, // coords of cys 2 from full_coord
			float & disulf_bb_dih // backbone dihedral N_1-Calpha_1-Calpha_2-Cprime_2
		);

		void
		get_disulf_cacbcbca_dih(
			FArray2Da_float cys1coord, // coords of cys 1 full_coord numbering
			FArray2Da_float cys2coord, // coords of cys 2 full_coord numbering
			float & disulf_cacbcbca_dih
		);

		void
		get_disulf_cacbcb_angles(
			FArray2Da_float cys1coord, // coords of cys 1 full_coord numbering
			FArray2Da_float cys2coord, // coords of cys 2 full_coord numbering
			float & disulf_cacbcb_ang_1,
			float & disulf_cacbcb_ang_2
		);

		void
		get_disulf_cbs_dist(
			FArray2Da_float cys1coord,
			 // coords of cys 1 from full_coord (include sidechain)
			FArray2Da_float cys2coord,
			 // coords of cys 2 from full_coord (include sidechain)
			float & dist_cbs_1,
			float & dist_cbs_2
		);

		void
		get_cbs_sc_this_disulf(
			FArray2Da_float cys1coord,
			 // coords of cys 1 from full_coord (include sidechain)
			FArray2Da_float cys2coord,
			 // coords of cys 2 from full_coord (include sidechain)
			float & cbs_sc_this_disulf
		);

		float
		get_disulf_cendist_sc();

		int
		get_disulf_Ncen_sc();

		int
		get_disulf_Ncb_sc();

		int
		get_disulf_Nfa_sc();

		float
		get_fa_disulf_cbs_dist_sc();

		void
		open_disulf_logfile();

	private: // Methods, Data Initializers
		static
		void
		bb_dih_cnrlx_score_initializer(
			FArray1D_float & bb_dih_cnrlx_score
		);

		static
		void
		bb_dih_cnrlx_score_sm_initializer(
			FArray1D_float & bb_dih_cnrlx_score_sm
		);

		static
		void
		bb_dih_foldcst_score_sm_initializer(
			FArray1D_float & bb_dih_foldcst_score_sm
		);

		static
		void
		bb_dih_native_histogram_sm_initializer(
			FArray1D_float & bb_dih_native_histogram_sm
		);

		static
		void
		bb_dih_native_histogram_unsm_initializer(
			FArray1D_float & bb_dih_native_histogram_unsm
		);

		static
		void
		cacbcb_cnrlx_score_initializer(
			FArray1D_float & cacbcb_cnrlx_score
		);

		static
		void
		cacbcb_native_histogram_initializer(
			FArray1D_float & cacbcb_native_histogram
		);

		static
		void
		cacbcbca_cnrlx_score_initializer(
			FArray1D_float & cacbcbca_cnrlx_score
		);

		static
		void
		cacbcbca_cnrlx_score_sm_initializer(
			FArray1D_float & cacbcbca_cnrlx_score_sm
		);

		static
		void
		cacbcbca_foldcst_score_sm_initializer(
			FArray1D_float & cacbcbca_foldcst_score_sm
		);

		static
		void
		cacbcbca_native_histogram_sm_initializer(
			FArray1D_float & cacbcbca_native_histogram_sm
		);

		static
		void
		cacbcbca_native_histogram_unsm_initializer(
			FArray1D_float & cacbcbca_native_histogram_unsm
		);

		static
		void
		cbdist_cnrlx_score_initializer(
			FArray1D_float & cbdist_cnrlx_score
		);

		static
		void
		cbdist_foldcst_score_initializer(
			FArray1D_float & cbdist_foldcst_score
		);

		static
		void
		cbdist_native_histogram_initializer(
			FArray1D_float & cbdist_native_histogram
		);

		static
		void
		cendist_score_interp300_initializer(
			FArray1D_float & cendist_score_interp300
		);

		static
		void
		deriv_cbdist_score_initializer(
			FArray1D_float & deriv_cbdist_score
		);

		static
		void
		deriv_cendist_interp300_initializer(
			FArray1D_float & deriv_cendist_interp300
		);

		static
		void
		deriv_ssdist_sc_interp300_initializer(
			FArray1D_float & deriv_ssdist_sc_interp300
		);

		static
		void
		farlx_csang_score_2_initializer(
			FArray1D_float & farlx_csang_score_2
		);

		static
		void
		farlx_csang_score_initializer(
			FArray1D_float & farlx_csang_score
		);

		static
		void
		farlx_dihed_score_2_initializer(
			FArray1D_float & farlx_dihed_score_2
		);

		static
		void
		farlx_dihed_score_initializer(
			FArray1D_float & farlx_dihed_score
		);

		static
		void
		farlx_ssdist_sc_interp300_initializer(
			FArray1D_float & farlx_ssdist_sc_interp300
		);

		static
		void
		reorder_cys_initializer(
			FArray1D_int & reorder_cys
		);


	private: // Methods, Math
		static
		float
		gauss(
			float const x, // x value at which gaussian will be evaluated
			float const base,
			float const amp,
			float const mean,
			float const width
		);

		static
		float
		dbl_gauss(
			float const x, // x value at which gaussian will be evaluated
			float const base,
			float const amp1,
			float const amp2,
			float const mean1,
			float const mean2,
			float const width1,
			float const width2
		);

		static
		float
		triple_gauss(
			float const x, // x value at which gaussian will be evaluated
			float const base,
			float const amp1,
			float const amp2,
			float const amp3,
			float const mean1,
			float const mean2,
			float const mean3,
			float const width1,
			float const width2,
			float const width3
		);

		static
		float
		d_triple_gauss_d_x(
			float const x, // x value at which gaussian will be evaluated
			float const base,
			float const amp1,
			float const amp2,
			float const amp3,
			float const mean1,
			float const mean2,
			float const mean3,
			float const width1,
			float const width2,
			float const width3
		);

		static
		float
		gauss2D(
			float const x,
			float const y,
			float const z0, // offset = value of 2D gauss at (x,y) = (0,0)
			float const A, // amplitude
			float const x0,
			float const xwidth,
			float const y_0,
			float const ywidth,
			float const cor // cross-correlation term
		);

		static
		float
		d_gauss2D_d_x(
			float const x,
			float const y,
			float const z0, // offset = value of 2D gauss at (x,y) = (0,0)
			float const A, // amplitude
			float const x0,
			float const xwidth,
			float const y_0,
			float const ywidth,
			float const cor // cross-correlation term
		);

		static
		float
		d_double_gauss_d_x(
			float const x, // x value at which gaussian will be evaluated
			float const base,
			float const amp1,
			float const amp2,
			float const mean1,
			float const mean2,
			float const width1,
			float const width2
		);

		static
		bool
		even_integer( int & num );

	private: // Methods, Class Initialization
		void
		initialize();

	private: // Pointers, Mirrors
		// yab:
		// These are an interim solution, these will probably change once misc is gone
		// and Disulfides is fully integrated.
		// In the current code we don't use these pointers directly.  Instead we
		// create references from the pointers.  This decreases the amount of changes
		// we have to do to the code in misc removal + class wrapping.
		FArray3Dp_float const * Eposition_;
		FArray2D_float const * centroid_;
		FArray3D_float const * full_coord_;
		FArray1D_float const * phi_;
		FArray1D_float const * psi_;
		FArray1D_int const * res_;
		int const * total_residue_;

	private: // Data
		/* connectivity */
		FArray2D_int disulfide_pairs;
		// 2-d array of disulf connectivity, = 1 if paired, else = 0.
		FArray1D_int disulf_partner_a;
		// 1-d arrays of disulf connectivity, _a(i) paired to _b(i)
		FArray1D_int disulf_partner_b;
		// stores cys index, e.g. if _a(i)=2 and _b(i)=5,
		// then the ith disulf has 2nd cys paired to 5th cys.

		/* counts */
		int num_cysteines;
		FArray1D_int cys;
		// residue numbers of the cysteines
		int n_disulfides_centroid;
		// # of disulf with cendist less than disulf_cendist_cutoff
		int n_known_disulfides;
		// # of disulf read in from disulf connectivity file (only if fix_disulf=true)
		int n_disulf_cent_natcut;
		// # of disulf with cendist less than max_native_disulf_cendist
		int n_disulf_cb_natcut;
		// # of disulf with cbdist less than max_native_disulf_cbdist
		FArray2D_int track_disulf_cen_cb;
		// i=1(cen)=2(cb), j=disulfide index, track_disulf_cen_cb = 0/1 if fail/satisfy criteria
		FArray1D_float cb_scale_factor;

		/* fullatom scores */
		int n_disulfides_fa; // num of disulf that satisfy s-s distance criteria
		float disulfide_distance_score;
		float disulfide_csangles_score;
		float disulfide_dihedral_score;
		float disulfide_ca_dihedral_score;
		float disulfide_cbs_dist_score;

};

}

#endif
