// -*- mode:c++;tab-width:1;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15655 $
//  $Date: 2007-06-26 10:16:38 -0700 (Tue, 26 Jun 2007) $
//  $Author: ashworth $

#include "dna_classes.h"
#include "dna.h" // na_partner

#include "aa_name_conversion.h" // num_from_res1
#include "after_opts.h" // truefalseoption
#include "param.h" // 'MAX' dimensions
#include "param_aa.h" // aa_name3
#include "pdb.h" // pdb_res_num

#include <ObjexxFCL/formatted.o.hh> // F(), I(), A()...
#include <utility/basic_sys_util.hh> // utility::exit
#include <ObjexxFCL/FArray1Ds.hh>

#include <vector>
#include <cstdio>
#include <iostream>

////////////////////////////////////////////////////////////////////////////////
// storage classes for DNA modes
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// @begin DNAPose::DnaOptions
//
// @brief
// Reader/container of some of the command-line DNA design options
//
// @authors
// ashworth
//
////////////////////////////////////////////////////////////////////////////////
DnaOptions::DnaOptions()
{
	std::cout << std::endl << "Reading command-line DNA options:" << std::endl;

	// bool options
	verbose = truefalseoption( "dna_verbose" );
	design_by_base = truefalseoption( "design_by_base" );
	dna_minimize = truefalseoption( "dna_minimize" );
	no_design = truefalseoption( "no_design" );
	dna_scan = truefalseoption( "dna_scan" );
	revert = truefalseoption( "revert" );
	all_interface_stats = truefalseoption( "all_interface_stats" );
	decoystats = truefalseoption( "dna_decoystats" );
	ddG = truefalseoption( "dna_ddG" );

	// bool options with parameters
	probe_specificity = truefalseoption( "probe_specificity" );
	if ( probe_specificity ) {
		intafteroption( "probe_specificity", 3, num_packs );
		normalize_Boltzmann = true; // on by default
	}
	user_seq = truefalseoption( "user_seq" );
	if ( user_seq ) stringafteroption( "user_seq", "", userseq );
	smm = truefalseoption( "smm" );
	if ( smm ) realafteroption( "smm", 0., ms_offset );
	smm_exhaustive = truefalseoption( "smm_exhaustive" );
	dna_bb_moves = truefalseoption( "dna_bb_moves" );
	if ( dna_bb_moves ) intafteroption( "dna_bb_moves", 1, outer_runs );

	// parameters
	realafteroption( "dz_cutoff", 6.0, dz_cutoff );
	intafteroption( "runindex_offset", 0, runindex_offset );
	realafteroption( "Boltz_temp", 1.5, Boltz_temp );
}

////////////////////////////////////////////////////////////////////////////////
// Could use a faster search method, but the length is short and this step will not be a bottleneck.
DnaPosInfo &
DnaSeqInfo::operator[] (
	int const pos
)
{
	for ( std::vector< DnaPosInfo >::iterator napos( dna.begin() );
	      napos != dna.end(); ++napos ) {
		if ( pos == napos->fwdpos() ) return *napos;
		if ( napos->paired() && pos == napos->rvspos() ) return *napos;
	}
	std::cerr << "Error, position not found in the DNA sequence." << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );
	return dna.front();
}

////////////////////////////////////////////////////////////////////////////////
DnaPosInfo const &
DnaSeqInfo::operator[] (
	int const pos
) const
{
	for ( std::vector< DnaPosInfo >::const_iterator napos( dna.begin() );
	      napos != dna.end(); ++napos ) {
		if ( pos == napos->fwdpos() ) return *napos;
		if ( napos->paired() && pos == napos->rvspos() ) return *napos;
	}
	std::cerr << "Error, position not found in the DNA sequence." << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );
	return dna.front();
}

////////////////////////////////////////////////////////////////////////////////
// construct a DnaSeqInfo from a sequence string, with nonsense sequence indices
DnaSeqInfo::DnaSeqInfo( std::string const seqstr )
{
	for ( unsigned i(0); i < seqstr.size(); ++i ) {
		// convert char to na number
		int na;
		// force lower-case codes to ensure DNA-only sequence
		num_from_res1( tolower(seqstr[i]), na );
		if ( !param_aa::is_DNA(na) ) {
			std::cerr << "ERROR: seqstr specifies non-DNA sequence!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );
		}
		dna.push_back( DnaPosInfo( 0, na, 0, na_partner(na) ) );
	}
}

////////////////////////////////////////////////////////////////////////////////
// update existing DNA sequence information from a 'res' array
void
DnaSeqInfo::update_from_res( ObjexxFCL::FArray1DB_int const & res )
{
	for ( std::vector< DnaPosInfo >::iterator napos( dna.begin() );
	      napos != dna.end(); ++napos ) {
		if ( !napos->paired() ) napos->set_type( res(napos->fwdpos()) );
		else napos->set_type( res(napos->fwdpos()), res(napos->rvspos()) );
	}
}

////////////////////////////////////////////////////////////////////////////////
// update a 'res' array from this DNA sequence info
ObjexxFCL::FArray1D_int
DnaSeqInfo::res_from_seq( ObjexxFCL::FArray1DB_int const & res ) const
{
	ObjexxFCL::FArray1D_int newres( res );
	for ( std::vector< DnaPosInfo >:: const_iterator napos( begin() );
	      napos != end(); ++napos ) {
		newres( napos->fwdpos() ) = napos->fwdtype();
		if ( !napos->paired() ) continue;
		newres( napos->rvspos() ) = napos->rvstype();
	}
	return newres;
}

////////////////////////////////////////////////////////////////////////////////
// return a subset sequence representing the designable positions of parent
DnaSeqInfo
DnaSeqInfo::design_set() const
{
	DnaSeqInfo seq;
	for ( std::vector< DnaPosInfo >::const_iterator napos( begin() );
	      napos != end(); ++napos ) {
		if ( napos->design() ) seq.push_back( *napos );
	}
	return seq;
}

////////////////////////////////////////////////////////////////////////////////
// return a subset sequence with arbitrary nucleotides mapped onto the designable positions of parent
DnaSeqInfo
DnaSeqInfo::target_set( std::string const seqstr ) const
{
	DnaSeqInfo seq;
	unsigned i(0);
	for ( std::vector< DnaPosInfo >::const_iterator napos( begin() );
	      napos != end(); ++napos ) {
		if ( !napos->design() ) continue;

		if ( i == seqstr.size() ) {
			std::cerr << "ERROR: seqstr didn't specify enough positions" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );
		}
		// convert char to na number
		int na;
		// force lower-case codes to ensure DNA-only sequence
		num_from_res1( tolower( seqstr[i] ), na );
		if ( !param_aa::is_NA(na) ) {
			std::cerr << "ERROR: seqstr specifies non-DNA sequence" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );
		}
		if ( !napos->paired() ) seq.push_back( DnaPosInfo( napos->fwdpos(), na ) );
		else seq.push_back( DnaPosInfo( napos->fwdpos(), na,
		                                napos->rvspos(), na_partner(na) ) );
		++i;
	}
	if ( i != seqstr.size() ) {
		std::cerr << "WARNING: more DNA positions specified than are designable" <<
		 std::endl;
	}
	return seq;
}

////////////////////////////////////////////////////////////////////////////////
// check for the existence a residue sequence index
bool
DnaSeqInfo::contains(
	int const pos
) const
{
	for ( std::vector< DnaPosInfo >::const_iterator napos( dna.begin() );
	      napos != dna.end(); ++napos ) {
		if ( pos == napos->fwdpos() ) return true;
		if ( napos->paired() && ( pos == napos->rvspos() ) ) return true;
	}
	return false;
}

// check for the existence a residue sequence index with specific type
bool
DnaSeqInfo::contains(
	int const pos,
	int const na
) const
{
	for ( std::vector< DnaPosInfo >::const_iterator napos( dna.begin() );
	      napos != dna.end(); ++napos ) {

		if ( ( pos == napos->fwdpos() ) && ( na == napos->fwdtype() ) ) return true;
		if ( ( napos->paired() ) &&
			   ( pos == napos->rvspos() ) && ( na == napos->rvstype() ) ) return true;
	}
	return false;
}
////////////////////////////////////////////////////////////////////////////////
std::vector<int> const
DnaSeqInfo::design_positions() const
{
	std::vector<int> positions;

	if ( empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );

	for ( std::vector< DnaPosInfo >::const_iterator napos( dna.begin() );
	     napos != dna.end(); ++napos ) {
		if ( napos->design() ) positions.push_back( napos->fwdpos() );
	}

	if ( positions.size() == 0 ) {
		std::cerr << "ERROR: There are no DNA design positions!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );
	}

/*
	std::cout << "DNA design positions are: ";
	for ( std::vector<int>::const_iterator napos( positions.begin() );
	      napos != positions.end(); ++napos ) {
		std::cout << pdb::pdb_res_num(*napos) << ", ";
	}
	std::cout << std::endl;
*/

	return positions;
}

////////////////////////////////////////////////////////////////////////////////
//
std::list< DnaSeqInfo >
DnaSeqInfo::single_mutant_list( bool const verbose ) const
{
	std::list< DnaSeqInfo > seqs;
	seqs.push_back( *this );

	for ( std::vector< DnaPosInfo >::const_iterator na( begin() );
	      na != end(); ++na ) {
		for ( int type(1); type <= param::MAX_AA(); ++type ) {
			if ( !param_aa::is_DNA(type) || type == na->fwdtype() ) continue;
			DnaSeqInfo mutantseq( *this );
			mutantseq[ na->fwdpos() ].set_type( type, na_partner(type) );
			seqs.push_back( mutantseq );
		}
	}
	if ( verbose ) {
		std::cout << "DNA sequence population:" << std::endl;
		debug_print_seqs( seqs );
	}

	return seqs;
}

////////////////////////////////////////////////////////////////////////////////
std::string const
DnaSeqInfo::str() const
{
	std::string seq_string;
	// the map specifies both positive and negative DNA strands, but here only the sequence of one strand is returned (positions specifies only one strand)
	for ( std::vector< DnaPosInfo >::const_iterator napos( dna.begin() );
	      napos != dna.end(); ++napos ) {
		seq_string += param_aa::aa_name1( napos->fwdtype() );
	}
	return seq_string;
}

////////////////////////////////////////////////////////////////////////////////
void
DnaSeqInfo::print( std::ostream & out ) const
{
	out << std::endl << "DNA Sequence:" << std::endl;
	for ( std::vector< DnaPosInfo >::const_iterator napos( dna.begin() );
	      napos != dna.end(); ++napos ) {

		int const fwdpos( napos->fwdpos() );
		out << param_aa::aa_name3( napos->fwdtype() ) << " " <<
		       I( 4, pdb::pdb_res_num( fwdpos ) ) << " " <<
		       pdb::res_chain( fwdpos );

		if ( !napos->paired() ) { out << " (unpaired)" << std::endl; continue; }

		int const rvspos( napos->rvspos() );
		out << " <=> " << param_aa::aa_name3( napos->rvstype() ) << " " <<
		                  I( 4, pdb::pdb_res_num( rvspos ) ) << " " <<
		                  pdb::res_chain( rvspos ) << std::endl;
	}
	out << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
std::ostream & operator<< ( std::ostream & out, DnaSeqInfo const & seq )
{
	seq.print( out ); return out;
}

////////////////////////////////////////////////////////////////////////////////
// contents will be assigned by vector index--must pre-dimension
void
DnaNeighbors::dimension( int const nres )
{
	neighbors.clear();
	neighbors.assign( nres, std::vector< DnaNeighbor >() );

	for ( std::vector< std::vector< DnaNeighbor > >::iterator
	      it( neighbors.begin() ); it!= neighbors.end(); ++it ) {
		it->assign( nres, DnaNeighbor() );
	}
}

