// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 12053 $
//  $Date: 2007-01-16 07:46:34 +0200 (Tue, 16 Jan 2007) $
//  $Author: yab $


// Rosetta Headers
#include "dock_loops.h"
#include "dock_loops_ns.h"
#include "docking_score.h"
#include "fold_loops.h"
#include "loops.h"
#include "loops_ns.h"
#include "misc.h"
#include "monte_carlo.h"
#include "recover.h"
#include "refold.h"
#include "status.h"
#include "docking_ns.h"
#include "docking_minimize.h"
#include "docking_movement.h"
#include "runlevel.h"
#include "score.h"
#include "torsion_bbmove_trials.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>

// C++ Headers
#include <iostream>


// JJG
// 3/18/204
//
// Routines to support docking with flexible loops

////////////////////////////////////////////////////////////////////////////////
/// @begin get_dock_loopmove_flag
///
/// @brief
/// the dock_loopmove_flag indicates that a docking mode run is moving a loop -- this is needed
/// in low-level functions to indicate when it is appropriate to behave like loop mode instead
/// of dock mode
///
/// @detailed
/// as of 4/20/04, dock_loops mode is capable of starting with an input
/// structure which is missing loops (must have a template pdb in loop
/// mode format) and building those loops and optimizing them before
/// proceeding to the regular docking search.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_dock_loopmove_flag()
{
	using namespace dockloop_private;

	return dock_loopmove_flag;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin set_dock_loopmove_flag
///
/// @brief
///
/// @detailed
///
/// @param[in]   m - in
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
///
void
set_dock_loopmove_flag( bool const m /* input */ )
{
	using namespace dockloop_private;

	// not loop moves
	dock_loopmove_flag = m;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin get_dock_loop_rmsd
///
/// @brief
/// the dock_loopmove_flag indicates that a docking mode run is moving a loop -- this is needed
/// in low-level functions to indicate when it is appropriate to behave like loop mode instead
//// of dock mode
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
get_dock_loop_rmsd()
{
	using namespace dockloop_private;

	return dock_loop_rmsd;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin set_dock_loop_rmsd
///
/// @brief
///
/// @detailed
///
/// @param[in]   r - in
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_dock_loop_rmsd( float const r /* input */ )
{
	using namespace dockloop_private;

	dock_loop_rmsd = r;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_loop_trials_flag
///
/// @brief if true ... allow loop trials during rigid body docking stages
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Glenn Butterfoss
///
/// @last_modified 2 Nov 2004
/////////////////////////////////////////////////////////////////////////////////
bool
get_docking_loop_trials_flag()
{
	return docking::docking_flagflags::docking_loop_trials;
}

//////////////////////////////////////////////////////////////////////////////
///@begin get_dock_mcm_loop_min_flag
///
///@brief
///
///@detailed
///
///@param[out] bool value of dock_mcm_loop_min_flag
///
///@global_read
///
///@global_write
///
///@remarks
///
///@refrences
///
///@authors Glenn Butterfoss
///
///@last_modified 3 March 2005
////////////////////////////////////////////////////////////////////////////////
bool
get_dock_mcm_loop_min_flag()
{
	return docking::docking_flagflags::dock_mcm_loop_min_flag;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin set_docking_loop_trials_flag
///
/// @brief if true ... allow loop trials during rigid body docking stages
///
/// @detailed
///
/// @param[in]   m - in
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Glenn Butterfoss
///
/// @last_modified 2 Nov 2004
/////////////////////////////////////////////////////////////////////////////////
///
void
set_docking_loop_trials_flag( bool const m /* input */ )
{
	using namespace docking;

	docking_loop_trials = m;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dock_flexible_loops_setup
///
/// @brief
///
/// @detailed Setup functions from fold_loops() inserted into the docking protocol
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 3/18/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dock_flexible_loops_setup()
{
	using namespace loops_ns;
	using namespace misc;

	if ( ! get_loop_flag() ) return;
	if ( get_docking_loop_trials_flag() ) return; //glb won't remember new loops in trials
	if ( get_dock_mcm_loop_min_flag() ) return; //glb note: perhaps make these work togther

//jg local
	bool exist;

	int const all_loops = { -1 }; // invalid loop number

//car NOTE: these cutoffs are consistent with checks in main_wobble
	int const cutoff3 = { 7 }; // no 3mer insertions in anything shorter than this
	int const cutoff9 = { 17 }; // no 9mer insertions in anything shorter than this

	float const splicerms_cutoff = { 0.5 }; // loops<cutoff are considered closed
	// msd still contributes to score


	set_dock_loopmove_flag(true);

	set_allow_insert_by_looplength(total_residue+1,true,exist);
	select_starting_loops();
	refold(1,total_residue);
	save_status_info("select_loops",0,0);
	monte_carlo_reset(); // save as low structure

//car update trimming
	set_allow_insert_by_looplength(total_residue+1,true,exist);
// if ( trim ) trim_template();

//car extended regions
	fold_extended_loop(trim,all_loops);

//car frag moves in long,internal loops
	build_loop(cutoff3,cutoff9,fast,all_loops);

//car min_moves in loops
	optimize_loop(cutoff3,splicerms_cutoff,fast,all_loops);

//car close all (all loops open by more than splicerms_cutoff)
	close_loop(splicerms_cutoff,fast,all_loops);

	set_dock_loopmove_flag(false);

	recover_LOW(score4d);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dock_loop_trials
///
/// @brief iterate through (closed) loop library to find best combination of loops
///
/// @detailed
/// Called after rigid body move but before monte_carlo (in main_rigid_body_trial()).
/// Step up to search through, in series,  all structures in .loops file
///
/// The loops are directly relpaced in the structure without any optimiziation
/// so it is critical to use a loop library that contains only already closed structures.
/// The loops are searched in series so this method my have problems when some
/// of the loop structures of loop A contact contact structures of loop B
/// To get a closed loop library, you can search a .loops.screened file for a
/// small gap score (5th column I think) or I should make a loops option to write
/// out a closed loop file.
///
/// call on command line with "-loops -loop_trials"
///
/// @param[in]  scoring_function - that would be the scoring_function to use
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// uses docking_hijack_best_pose() to refold the new loops into the current
/// structure ... this should be changed ... be carefull about concurrent hijacks
/// elsewhere.
/// Only use during rigid body steps ... do not call in a
/// monte carlo + minimiziation cycle.
///
/// @references
///
/// @authors Glenn Butterfoss
///
/// @last_modified 2 Nov 2004
/////////////////////////////////////////////////////////////////////////////////
void
dock_loop_trials(
	Scoring_Function scoring_function
)
{
	using namespace misc;
	using namespace loops_ns;
	using namespace runlevel_ns;

	using namespace mc_global_track::mc_score; // yab: misc removal

	if ( ! get_docking_loop_trials_flag() ) return;
	if ( ! get_loop_flag() ) return;
	if ( runlevel >= verbose ) std::cout << "entering dock_swap_loops()" << std::endl;

	//glb local
	bool exist,err;
	int best_depth_index = 1;
	float best_score_here = 999;

	//score_set_loop_weight(1.0); //assume loops are closed
	set_dock_loopmove_flag(true);

	// now, need to have current in "best" for refold to work ...
	docking_hijack_best_pose(); //copy structure and vectors, best_score doesn't change

 	for ( int loop_index = 1; loop_index <= num_loop; ++loop_index ) {
 		for ( int depth_index = 1, depth_index_end = loop_depth( loop_index );
		 depth_index <= depth_index_end; ++depth_index ) {

			set_allow_insert_by_loopnum( loop_index, true, exist );
			loop_insert( loop_index, depth_index, err ); //put #=depth loop in loop = loop_num
			refold( 1, total_residue ); // coord updated with refold

			score = scoring_function();

			if ( score <= best_score_here ) {
				best_depth_index = depth_index;
				best_score_here = score;

				if ( runlevel > chat ) {
					std::cout << "accepted: loop " << loop_index << " depth " << depth_index << " score " << score << std::endl;
				}

			}

		}

		// put in best loop for current loop_index
		if ( runlevel > standard ) {
			std::cout << "best: loop " << loop_index << " depth " << best_depth_index << " score " << score << std::endl;
		}
		set_allow_insert_by_loopnum(loop_index,true,exist);
		loop_insert(loop_index,best_depth_index,err);
		refold(1,total_residue);
		best_depth_index = 1; //reset for next cycle
	}

	//   best <- save for future monte carlo
	docking_hijack_best_pose_rlease(); //vectors and structure

	//reset loop options
	set_allow_insert_by_looplength(total_residue+1,false,exist); //true or false?
	set_dock_loopmove_flag(false);

}

//////////////////////////////////////////////////////////////////////////////
///@begin docking_loop_minimize
///
///@brief loop minimiziation option for dock_mcm protocol
//
///
///@detailed
///
///@param[in] iteration - number to feed to output in dMCM.dat for this step
///@param[in] scoring_fn - score function to minimize loops with
///
///@global_read
///
///@global_write
///
///@remarks  1. Make sure the scoring function you feed this uses a loop weights
///          2. Needs modification if you call from somewhere other than dock_mcm
///
///@refrences
///
///@authors Glenn Butterfoss
///
///@last_modified 3 March 2005
////////////////////////////////////////////////////////////////////////////////
void
docking_loop_minimize(
	int iteration,
	Scoring_Function scoring_fn
)
{
  if ( ! get_dock_mcm_loop_min_flag() ) return;

	using namespace docking;
	using namespace misc;
	using namespace runlevel_ns;
	using namespace loops_ns;

 	bool exist;

	// set up loop stuff
	score_set_loop_weight(600);
	new_score_function(scoring_fn);
	set_allow_insert_by_looplength(total_residue+1,true,exist);
	//glb select_starting_loops();
	set_dock_loopmove_flag(true);

	main_minimize_trial(scoring_fn,"dfpmin");

	if ( runlevel > inform ) dMCM_output(iteration,6); //6 is "loop_min" dMCM.dat

	//set_allow_insert_by_looplength(total_residue+1,false,exist);
	score_set_loop_weight( 0 );
	set_dock_loopmove_flag(false);
}
