// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $


// Rosetta Headers
#include "docking.h"
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "cenlist.h"
#include "design.h"
#include "dock_fab.h"
#include "dock_pivot.h"
#include "dock_structure.h"
#include "docking_constraints.h"
#include "docking_minimize.h"
#include "docking_movement.h"
#include "docking_score.h"
#include "docking_scoring.h"
#include "docking_unbound.h"
#include "docking_ns.h"
#include "dock_loop_ensemble_ns.h"
#include "files_paths.h"
#include "filters.h"
#include "fullatom.h"
#include "hbonds_ns.h"
#include "initialize.h"
#include "input_pdb.h"
#include "interface.h"
#include "jumping_util.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "monte_carlo.h"
#include "minimize_chi.h"
#include "misc.h"
#include "monte_carlo.h"
#include "native.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack.h"
#include "param.h"
#include "param_aa.h"
#include "pdb.h"
#include "pose_io.h"
#include "read_aaproperties.h"
#include "recover.h"
#include "rotamer_functions.h"
#include "runlevel.h"
#include "score_ns.h"
#include "silent_input.h"
#include "start.h"
#include "score.h"
#include "start.h"
#include "SurfaceMode.h"
#include "util_vector.h"
#include "vdw.h"
#include "status.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.io.hh>
#include <numeric/xyzMatrix.io.hh>
#include "FArray_xyz_functions.h"

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cassert>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <sstream>

//     docking.cc: public-access functions for docking calculations
//
//     Jeff Gray April 2001
//
//


//     bool get_docking_flag
//             set_docking_flag
//     bool get_docking_fullatom_flag
//     int get_docking_part2_begin
//     float    get_docking_rmsd()
//     float    get_docking_translation_dist()
//     float    get_rotation_angle()
//             docking_output_geometry(unit)

//     docking_compare functions at the end:
//      function docking_store_native_interface()
//      function docking_print_native_interface()
//       (later, add functions to calculate # of native contacts achieved)

//chu   this is added now 07/14/2003
//
//     function docking_store_native_contact_map()
//     function calc_docking_frac_native_contact()
//     function get_docking_frac_nat_contact()
//     function get_docking_frac_nonnat_contact()
//     function calc_docking_interf_rmsd()
//     function set_unbound_start_flag()
//     function get_unbound_start_flag()
//
//ora  calculation of number of contacts: added 03/26/2004
//     function calc_docking_number_of_contacts()
//     function get_countcont()
//
//     docking_init initialization functions for docking
//
//     Jeff Gray April 2001
//
//

//     docking_store_residues   -- set docking part1/part2 begin/end
//     initialize_docking_start -- find the centroids and other geometric info
//     initialize_docking_decoy -- copy start position to decoy
//     docking_store_native     -- store the docking variables of the native




// ============== Public Access Routines ==============


//==modes

////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_flag
///
/// @brief retrieve docking mode flag
///
/// @detailed
///
/// @return  bool, true if rosetta docking mode is called
///
/// @global_read  docking_flag in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_docking_flag()
{
	using namespace docking;

	return ( docking_flag || get_ligand_flag() );
}

bool
get_pose_docking_flag()
{
	using namespace docking;

	return ( pose_docking_flag && docking_flag );
}

bool
get_pose_symmetry_docking_flag()
{
	using namespace docking;

	return ( docking_pose_symmetry );
}

bool
get_flexbb_docking_flag()
{
	using namespace docking;

	return ( flexbb_docking_flag && docking_flag );
}

bool
get_docking_silent_input_flag()
{
	using namespace docking;
	return docking_silent_input;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_docking_flag
///
/// @brief set docking mode flag
///
/// @detailed
///
/// @param  m - [in/out]? - bool, how the docking mode flag will be set
///
/// @global_read
///
/// @global_write docking_flag in docking.h
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_docking_flag( bool & m )
{
	using namespace docking;

//      std::cout << "docking mode set to " << m << std::endl;
	docking_flag = m;
}

void
set_pose_docking_flag( bool & m )
{
	using namespace docking;

	pose_docking_flag = m;
}////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_fullatom_flag
///
/// @brief retrieve docking_fullatom flag
///
/// @detailed
/// retrieve the flag indicating whether full-atom mode is used in docking
/// command-line option is 'dockFA' and docking protocol will only repack the
/// interface residues after centroid monte carlo search if no other flags are
/// turned on.
///
/// @return  bool, true if fullatom mode is used in docking
///
/// @global_read docking_fullatom_flag in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_docking_fullatom_flag()
{
	using namespace docking;

	return docking_fullatom_flag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_FAsearch_flag
///
/// @brief retrieve docking_fullatom_search_flag
///
/// @detailed
/// docking_full_atom search is one of docking sub-modes where fullatom monte
/// carlo search is carried out. The command line flag to choose this mode is
/// 'FAsearch', but this is not used any more.
///
/// @return  bool, true if this sub-mode is chosen
///
/// @global_read docking_fullatom_search_flag in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_docking_FAsearch_flag()
{
	using namespace docking;

	return docking_fullatom_search_flag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_minimize_flag
///
/// @brief retrieve docking_minimize flag
///
/// @detailed
/// docking_minimize is one of docking sub-modes where a rigid-body
/// minimization is carried out following interface side chain repacking as
/// explained in 'dockFA' mode above. Command-line flag is 'dock_min'
///
/// @return  bool, true if docking_minimize mode is chosen
///
/// @global_read docking_minimize_flag in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_docking_minimize_flag()
{
	using namespace docking;

	return docking_minimize_flag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_mcm_flag
///
/// @brief retrieve docking_mcm flag
///
/// @detailed
/// docking_mcm(monte carlo minimization) is one of docking sub-modes where
/// a total of 50 cycles of monte carlo rigid-body minimization are carried
/// out following 'dock_min' mode. Between any two cycles, interface side chains
/// will be either repacked or rotamer_trialed.
///
/// @return  bool, true if this mode is selected
///
/// @global_read docking_mcm_flag in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_docking_mcm_flag()
{
	using namespace docking;

	return docking_mcm_flag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_output_position_hetatm_flag
///
/// @brief retrieve docking_output_position_hetatm flag
///
/// @detailed
///
/// @return  bool, true if hetatms will be printed
///
/// @global_read docking_output_position_hetatm in docking.h
///
/// @global_write
///
/// @remarks
///
/// @refrences
///
/// @authors J. Karanicolas 06/02/04
///
/// @last_modified Chu Wang 03/16/06
/////////////////////////////////////////////////////////////////////////////////
bool
get_docking_output_position_hetatm_flag()
{
	using namespace docking;

	return docking_output_position_hetatm;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_prepack_mode
///
/// @brief retrieve docking_prepack flag
///
/// @detailed
/// docking_prepack is one of docking sub-modes in order to prepack starting
/// structures. In this mode, two parterners are separated and all their
/// side-chains are fully repacked. Then they are brought back to the original
/// positions to be used as the docking starting structure. Command-line option
/// is 'prepack'
///
/// @return  bool, true if docking_prepack mode is selected
///
/// @global_read prepack_mode in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_docking_prepack_mode()
{
	using namespace docking;

	return prepack_mode;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_unboundrot_flag
///
/// @brief retrieve unboundrot flag in docking
///
/// @detailed
/// unboundrot is the flag indicationg whether extra rotamers from the native
/// docking unbound structures will be included in side-chain repacking
///
/// @return  bool, true if those extra rotamers will be used
///
/// @global_read unboundrot in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_unboundrot_flag()
{
	using namespace docking;

	return unboundrot;
}

bool
get_unbound_start_flag()
{
	using namespace docking;

	return unbound_start;
}

void
set_unbound_start_flag( bool m )
{
	using namespace docking;

	unbound_start = m;
}

bool
get_docking_local_refine()
{
	using namespace docking;

	return docking_local_refine;
}

void
set_docking_use_mapped_rmsd( bool m )
{
	using namespace docking_unbound;

	use_mapped_rmsd = m;
}
//==info

////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_part2_begin
///
/// @brief retrieve the sequence number of the beginning residue of docking
///       partner 2
///
/// @detailed
///
/// @return  integer, sequence number(defined in rosetta) of starting residue
///        of partner 2
///
/// @global_read part2_begin in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
get_docking_part2_begin()
{
	using namespace docking;

	return part_begin(2);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_rmsd
///
/// @brief retrieve rmsd of a docking decoy
///
/// @detailed rmsd is calcuated compared to the docking native structure, backbone
///      CA only
///
/// @return  float, the value of rmsd
///
/// @global_read docking_rmsd in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
get_docking_rmsd()
{
	using namespace docking;

	return docking_rmsd;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_translation_dist
///
/// @brief retrieve docking_translation vector distance
///
/// @detailed
/// This vector is from the geometric center of the native ligand to the
/// decoy ligand.
///
/// @return  float, value of this distance
///
/// @global_read docking_translation_distance
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
get_docking_translation_dist()
{
	using namespace docking;

	return docking_translation_distance;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_rotation_angle
///
/// @brief retrieve docking_rotation angle
///
/// @detailed
/// a rigid body rotation angle required to superimpose decoy ligand to its
/// native counterpart
///
/// @return  float, value of that angle in degree
///
/// @global_read docking_rotation_angle in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
get_docking_rotation_angle()
{
	return docking::docking_rotation_angle;
}

float
get_docking_frac_nat_contact()
{
	return interface::frac_native_contact;
}

float
get_docking_frac_nonnat_contact()
{
	return interface::frac_nonnat_contact;
}

int
get_countcont()
{
	return interface::n_contacts;
}

float
get_docking_interf_rmsd()
{
	return docking::docking_interf_rmsd;
}

float
get_docking_interf_energy()
{
	return docking::docking_interf_energy;
}
//==============================================================================

// jumpout protocol to stop searching as soon as minimum criteria are met
// 9/4/1

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_jumpout_check
///
/// @brief check if docking can jump out of monte carlo cyle in centroid stage
///
/// @detailed
/// This function is checking several contraints filters to see if the current
/// struture has been good enough to satisfy all of them.
///
/// @return  bool, true if the structure is good and can jump out monte carlo
///
/// @global_read
///
/// @global_write
///
/// @remarks Centroid mode use only
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
docking_jumpout_check()
{
	using namespace dock_fab;
	using namespace docking;
	using namespace scores;

	bool docking_jumpout_check = false; // Return value

//mj jump out if ligand flag
	if ( get_ligand_flag() ) {
		std::cout << "ERROR: Called centroid function with ligand flag:" <<
		 " docking_jumpout_check in dock.cc" << std::endl;
		return docking_jumpout_check;
	}

//	std::cout << "jumpout: " << L( docking_jumpout_mode ) <<
//	 SS( docking_contact_score ) << SS( docking_vdw_score ) <<
//	 SS( docking_fab_score ) << std::endl;

	if ( !docking_jumpout_mode ) return docking_jumpout_check;

	float const docking_contact_score_filter = realafteroption( "docking_contact_score_filter", 10.0);
	if ( docking_contact_score >= docking_contact_score_filter ) return docking_jumpout_check;

	float const docking_vdw_score_filter = realafteroption( "docking_vdw_score_filter", 1.0);
	if (docking_vdw_score >= docking_vdw_score_filter ) return docking_jumpout_check;

	if ( (fab1 || fab2) && docking_fab_score >= -4.0 ) return docking_jumpout_check;

	if ( !docking_dist_constr_filter() ) return docking_jumpout_check;
	if ( !docking_site_constr_filter() ) return docking_jumpout_check;
	if ( !score_filter() ) return docking_jumpout_check;

	docking_jumpout_check = true;
	std::cout << "jumpout criteria met, exiting search" << std::endl;
	std::cout << "contact (10) " << SS( docking_contact_score ) << " vdw (1) " <<
	 SS( docking_vdw_score ) << " fab (-4) " << SS( docking_fab_score ) << std::endl;

	return docking_jumpout_check;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_output_geometry
///
/// @brief output geometry information of a docking structure
///
/// @detailed
///
/// @param  iunit   [in/out]? - I/O port to output
///
/// @global_read geometrical variables in docking.h
///
/// @global_write
///
/// @remarks see docking.h for detailed definition of those parameters
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_output_geometry( std::ostream & iunit )
{
	using namespace docking;

	iunit << std::endl;
	iunit << "Centroids:" << std::endl;
	for ( int i = 1; i <= n_monomers; ++i ) {
		vector_write_line(iunit,part_centroid(1,i));
	}
	if ( n_monomers == 2 ) {
		iunit << std::endl;
		iunit << "Centroid bond:" << std::endl;
		vector_write_line(iunit,centroid_bond);
	}

	iunit << std::endl;
	iunit << "Docking basis vectors:" << std::endl;
	vector_write_line(iunit,docking_normal);
	vector_write_line(iunit,docking_parallel1);
	vector_write_line(iunit,docking_parallel2);
	iunit << "Contact point:" << std::endl;
	vector_write_line(iunit,contact_point);
	iunit << std::endl;
	iunit << "----------------------------------------------" << std::endl;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin output_docking_position
///
/// @brief output docking geometry into the pdb file for each decoy
///
/// @detailed
///
/// @param  iunit - [in/out]? - I/O unit
///
/// @global_read geometric variables in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_docking_position( std::ostream & iunit )
{
//     output position info right into the pdb
	using namespace docking;

	iunit << "Docking position: " << '\n';
	for ( int i = 1; i <= n_monomers; ++i ) {
		iunit << "docking part" << I( 1, i ) << "_centroid ";
		vector_write_line(iunit,part_centroid(1,i));
	}
	iunit << "docking centroid_bond  ";
	vector_write_line(iunit,centroid_bond);
	iunit << "docking contact_point  ";
	vector_write_line(iunit,contact_point);

	iunit << "docking_normal         ";
	vector_write_line(iunit,docking_normal);
	iunit << "docking_parallel1      ";
	vector_write_line(iunit,docking_parallel1);
	iunit << "docking_parallel2      ";
	vector_write_line(iunit,docking_parallel2);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_docking_position_hetatm
///
/// @brief output docking geometric information as HETATM in pdb file
///
/// @detailed
///
/// @param  iunit - [in/out]? - I/O unit
///
/// @global_read geometric information in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_docking_position_hetatm( std::ostream & iunit )
{
//     output position info right into the pdb as hetatm coords
	using namespace docking;

	 // j and chain are used to get equal position/chain definition as before
	char chain;

	for ( int i = 1; i <= n_monomers; ++i ) {
		int j = i;
		if ( i == 1 ) {
			chain = 'Y';
		} else if ( i == 2 ) {
			chain = 'Z';
		} else {
			chain = 'T';
			j = 7;
		}

		iunit << "HETATM" << I( 5, j ) << " CEN" << I( 1, i ) <<
		 " DOC " << chain << I( 4, j ) << "    " <<
		 F( 8, 3, part_centroid(1,i) ) <<
		 F( 8, 3, part_centroid(2,i) ) <<
		 F( 8, 3, part_centroid(3,i) ) << '\n';
	}

	iunit << "HETATM" << I( 5, 3 ) << " CONT" << " DOC " << "X" <<
	 I( 4, 3 ) << "    " <<
	 F( 8, 3, contact_point(1) ) <<
	 F( 8, 3, contact_point(2) ) <<
	 F( 8, 3, contact_point(3) ) << '\n';
	iunit << "HETATM" << I( 5, 4 ) << "  AXU" << " DOC " << "U" <<
	 I( 4, 4 ) << "    " <<
	 F( 8, 3, contact_point(1)+5.0*docking_normal(1) ) <<
	 F( 8, 3, contact_point(2)+5.0*docking_normal(2) ) <<
	 F( 8, 3, contact_point(3)+5.0*docking_normal(3) ) << '\n';
	iunit << "HETATM" << I( 5, 5 ) << "  AXV" << " DOC " << "V" <<
	 I( 4, 5 ) << "    " <<
	 F( 8, 3, contact_point(1)+5.0*docking_parallel1(1) ) <<
	 F( 8, 3, contact_point(2)+5.0*docking_parallel1(2) ) <<
	 F( 8, 3, contact_point(3)+5.0*docking_parallel1(3) ) << '\n';
	iunit << "HETATM" << I( 5, 6 ) << "  AXW" << " DOC " << "W" <<
	 I( 4, 6 ) << "    " <<
	 F( 8, 3, contact_point(1)+5.0*docking_parallel2(1) ) <<
	 F( 8, 3, contact_point(2)+5.0*docking_parallel2(2) ) <<
	 F( 8, 3, contact_point(3)+5.0*docking_parallel2(3) ) << '\n';

//                     23    CA    GLY   L  99 A      xyz      occ    bfactor
}

//==============================================================================
// docking_compare.cc: functions to compare the structure to native

// currently, we only detect the native residues and print them out
// future: produce a %-native contacts diagnostic

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_store_native_interface
///
/// @brief save interface information for docking native structure
///
/// @detailed
///
///
/// @global_read int_res array in interface.h, precalulated by 6A
///             centroid-centroid distance definition
///
/// @global_write native_site array in interface.h
///
/// @remarks native(1,x) stores the sequence number of xth interface residue in
///       partner1, and it should be terminated as 0. Similar for native(2,x)
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_store_native_interface()
{
	using namespace docking;
	using namespace interface;
	using namespace ligand;
	using namespace misc;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;

//     calculate cendist array
	update_cendist(total_residue,centroid);

//     detect the interface residues
	docking_detect_interf_res();

//chu   use 8A defintion to define native interface residues for interface
//     for later I_rmsd calculation(this is not true for ligand flag)

//mj ligand check
	if (get_ligand_flag()) {
		for ( int i = 1; i <= total_residue; ++i ) {
			if ( int_res(i) ) {
				native_site_residues(1).push_back(i);
			}
		}
		return;
	}

	for ( int i = part_begin(1), iend = part_end(1); i <= iend; ++i ) {
		if ( int_res8(i) ) {
			native_site_residues(1).push_back(i);
		}
	}
	// native_site_residues(2,j) is used for all further monomers
	// (not only monomer2)
	for ( int n = 2; n <= n_monomers; ++n ) {
		for ( int i = part_begin(n), iend = part_end(n); i <= iend; ++i ) {
			if ( int_res8(i) ) {
				native_site_residues(2).push_back(i);
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_print_native_interface
///
/// @brief print out interface information for the docking native structure
///
/// @detailed
///
/// @global_read native_site(x,x) in interface.h
///             pdb_res_num(x) etc in pdb.h
///             residue3(x) in misc.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_print_native_interface()
{
	using namespace interface;
	using namespace misc;
	using namespace pdb;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;

	std::cout << "\nNative interface residues:\n" << std::endl;

	std::string buffer;
	std::string aa_name3;

	for ( int i = 1; i <= max_docking_sites; ++i ) {
		std::cout << "Site " << i << std::endl;
		for( int j = 1; j <= int( native_site_residues(i).size() ); ++j ) {
			int r = native_site_residues(i)[j];
			aa_name3 = residue3(r);
			std::cout << space( 5 ) <<
			 aa_name3 << I( 4, pdb_res_num(r) ) << res_chain(r) <<
			 " (Rosetta" << I( 4, r ) << ")" << std::endl;
			buffer += right_string_of( pdb_res_num(r), 4 ) + res_chain(r) + ',';
		}

		if ( int(native_site_residues(i).size()) > 0 ) {
			buffer.erase( buffer.length() - 1 ); // Erase last comma character.
			std::cout << "      so select " << buffer << "\n" << std::endl;
		}

		if ( i == 1 ) buffer.clear();
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_store_residues
///
/// @brief save where each docking partner begins and ends in the global array
///
/// @detailed
///
/// @global_read domain information in misc.h
///
/// @global_write part1_begin,...part2_end in docking.h
///
/// @remarks This requires a TER put at the end of each partner (not each chain)
///       in the input pdb file
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_store_residues()
{
	// this function is called by the pdb loader
	// part_begin(2) is not set by the pdb loader
// adapted to multimers

	using namespace docking;
	using namespace files_paths;
	using namespace ligand;
	using namespace misc;

	if ( !multi_chain || design::dna_interface ) return;

	if (get_ligand_flag()) {
		std::cout << "Protein - Ligand docking partners detected: Residues " <<
		 I( 2, 1 ) << '-' << I( 4, total_residue ) << " and Hetero atoms " <<
		 I( 4, 1 ) << '-' << I( 4, ligand::ligand_one->atom_vector.size() ) << std::endl;
		//car initialize
		part_begin(1) = 1;
		part_begin(2) = total_residue + 1;
		part_end(1) = total_residue;
		part_end(2) = total_residue;
		n_monomers = 2;
		return;
	}

	part_begin(1) = 1;
	for ( int i = 2; i <= n_monomers; ++i ) {
		part_begin(i) = domain_end(i-1)+1;
		part_end(i-1) = part_begin(i) - 1;
	}
	part_end(n_monomers) = total_residue;

	std::cout << "Docking partners detected: Residues" << std::endl;
	for ( int i = 1; i <= n_monomers; ++i ) {
		std::cout << SS( part_begin(i) ) << '-' << SS( part_end(i) ) << std::endl;
	}

	if ( ( (total_residue < part_begin( n_monomers )) && !get_ligand_flag() && !prepack_mode ) || part_begin(2) == 0 || part_end(1) == -1 ) {
		std::cout << "Docking partners are goofy....this simulation's hosed. Did you forget to put in a TER statement?" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_docking_start
///
/// @brief initialize docking_start structure
///
/// @detailed
/// after reading in docking starting structure, mark the boundary for each
/// partner; derive the geometric information and remember them; read in
/// constraints file if there is any. fullatom mode is turned off temperarily.
///
/// @global_read position array in misc.h
///
/// @global_write geometric variables in docking.h
///
/// @remarks
///  - align partner1 with native (not yet necessary or implemented)
///  - find the centroids and other geometric info necessary for docking search
///  - perturb the native to a starting position
///  - fill the start_ variables for resetting later
///
///  called from initialize_start
///  4/17: no fullatom information is needed in this function (JJG 3/7/1)
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_docking_start()
{
	using namespace docking;
	using namespace files_paths;
	using namespace misc;

//glb call docking_store_residues if multi_chain to set part_begin & part_end
	if ( !multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;

//-----Set bookkeeping parameters (query info)

	docking_store_residues();

//mj check ligand flag
	if ( get_ligand_flag() ) {
		chain_centroid_CA(position,1,total_residue,part_centroid(1,1));
		ligand_center(part_centroid(1,2),(*ligand::ligand_one));
	} else {
		std::cout << "----------------------------------------------\n" << std::endl;
		std::cout << "Docking partners: Residues" << std::endl;
		for ( int i = 1; i <= n_monomers; ++i ) {
			std::cout << part_begin(i) << '-' << part_end(i) << std::endl;
		}

//-----Calculate geometric information

		for ( int i = 1; i <= n_monomers; ++i ) {
			chain_centroid_CA(position,part_begin(i),part_end(i),part_centroid(1,i));
		}
	}
	docking_derive_coord_frame();
	docking_output_geometry( std::cout );
	bool const orig_fa_flag =	get_fullatom_flag();
	set_fullatom_flag(false);

//-----remember start position
	for ( int i = 2; i <= n_monomers; ++i ) {
		vector_copy( part_centroid(1,i), start_part_centroid(1,i) );
	}
	vector_copy( docking_normal   , start_docking_normal );
	vector_copy( docking_parallel1, start_docking_parallel1 );
	vector_copy( docking_parallel2, start_docking_parallel2 );
	if ( n_monomers == 2) vector_copy( centroid_bond, start_centroid_bond );
	vector_copy( contact_point, start_contact_point );

//-----read constraints
	if ( !get_ligand_flag() ) {
		docking_read_site_constraints();
		docking_print_site_constraints();

		docking_read_dist_constraint();
		docking_print_dist_constraint();
		docking_fab_read_sites();
	}

	set_fullatom_flag(orig_fa_flag);
	docking_interf_energy = 0.0 ;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_derive_coord_frame
///
/// @brief derive geometic information for a docking structure
///
/// @detailed
///
/// @global_read part1_centroid, part2_centroid in docking.h
///
/// @global_write contact_point, centroid_bond, docking_normal,
///              docking_parallel1, docking_parallel2 in docking.h
///
/// @remarks simple vector math
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_derive_coord_frame()
{
//     derive centroid_bond, contact_point,
//            docking_normal, docking_parallel1, docking_parallel2
//     from part1/2_centroids and a seeded docking_parallel1

	using namespace docking;

	subvec(part_centroid(1,1),part_centroid(1,2),centroid_bond);
	// cb=p1c-p2c
	vector_midpoint(part_centroid(1,1),part_centroid(1,2),contact_point);
	// later, get more sophisticated w/contact_point calculation

	vector_copy(centroid_bond,docking_normal);
	vector_normalize(docking_normal);

//-----need an arbitrary starting vector to project normal to the centroid_bond
	docking_parallel1(1) = 1.;
	docking_parallel1(2) = 0.;
	docking_parallel1(3) = 0.;
	if ( docking_normal(2) == 0.0 && docking_normal(3) == 0.0 ) docking_parallel1(2) = 1.;
	 // avoid rare case of the seed being parallel

	vector_project_normal(docking_normal,docking_parallel1);
	vector_normalize(docking_parallel1);
	cros(docking_normal,docking_parallel1,docking_parallel2); // vectorcross

}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_docking_decoy
///
/// @brief initialize variables for a docking_decoy
///
/// @detailed
/// copy from docking_start all the geometric information in preparation
/// for generating a docking decoy. Also define which residues are allowed to
/// change their side-chain conformations by rotamer_trial
///
/// @global_read starting geometric variables in docking.h
///
/// @global_write current geometric variables in docking.h
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_docking_decoy()
{
	using namespace docking;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;

	for ( int i = 2; i <= n_monomers; ++i ) {
		vector_copy( start_part_centroid(1,i), part_centroid(1,i) );
	}
	vector_copy(start_docking_normal   ,docking_normal );
	vector_copy(start_docking_parallel1,docking_parallel1);
	vector_copy(start_docking_parallel2,docking_parallel2);
	vector_copy(start_centroid_bond    ,centroid_bond );
	vector_copy(start_contact_point    ,contact_point );

	docking_detect_interf_res();

	docking_derive_coord_frame();
	docking_set_allow_rottrial();
	docking_interf_energy = 0.0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_store_native
///
/// @brief save native geometric information
///
/// @detailed
///  - store the centroid of partner 2
///  - later, add storing of orientation
///  - that should be all we need to measure distance from correct position
///  (JJG 4/13/1)
///
/// @global_read position array in misc.h
///
/// @global_write native geometric information in docking.h
/// also, docking_translation_distance and docking_rotation_angle initialzed here
///
/// @remarks called only when native structure is in current position array.
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified Aroop Sircar 04/23/2007
/////////////////////////////////////////////////////////////////////////////////
void
docking_store_native()
{
	using namespace docking;
	using namespace misc;

	if ( !files_paths::multi_chain || design::dna_interface ||
			 files_paths::antibody_modeler )
		return;

	docking_translation_distance = 0.0;
	docking_rotation_angle = 0.0;

//mj check ligand flag
	if ( get_ligand_flag() ) {
		chain_centroid_CA(position,1,total_residue,native_part_centroid(1,1));
		ligand_center(native_part_centroid(1,2),(*ligand::ligand_one));

//mj set zero since not use and senseless
		for ( int i = 1; i <= 3; ++i ) {
			native_part2_orientation(i) = 0.0;
		}
		return;
	}

	for ( int i = 2; i <= n_monomers; ++i ) {
		chain_centroid_CA(position,part_begin(i),part_end(i),
		native_part_centroid(1,i));
	}

	subvec(Eposition(1,2,part_end(2)),Eposition(1,2,part_begin(2)),
	native_part2_orientation);

	for ( int i = 1; i <= n_monomers; ++i ) {
		native_part_begin(i) = part_begin(i);
		native_part_end(i) = part_end(i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_init_query
///
/// @brief initialize variabls for a docking native structure
///
/// @detailed  derive geometric information; store and print out native interface
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_init_query()
{
	using namespace docking;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler) return;

	initialize_docking_start();
	initialize_pivot_residues();
	docking_store_native_interface();
	docking_print_native_interface();
	docking_store_nat_contact_map();
	docking_mapping_native();
	set_docking_use_mapped_rmsd(false);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calculate_interface_centroid
///
/// @brief define the geometric center of docking interface
///
/// @detailed
/// update cendist array and determine the interface residue list
/// calculate the center of mass of interface residues (CA only)
///
///the center of the interface is an appropriate point to rotate
///about when minimizing.  this function finds that point, using CAs
///
/// @param[out]   interface_centroid - out - coord of geometric center of interface
///
/// @global_read
///
/// @global_write
///
/// @remarks
///      currently not used in centroid mode
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified Aroop Sircar 07/18/2007
/////////////////////////////////////////////////////////////////////////////////
void
calculate_interface_centroid( FArray1Da_float interface_centroid )
{
	using namespace docking;
	using namespace interface;
	using namespace misc;

	interface_centroid.dimension( 3 );

	if ( !files_paths::multi_chain || design::dna_interface ||
			 (files_paths::antibody_modeler && !antibody_ns::snug_fit) ) return;

//     calculate cendist array
	if ( ! pose_flag() ) { //hack for pose
		update_cendist(total_residue,centroid);
	}

//     detect the interface residues
	docking_detect_interf_res();

//     initialize
	int atom_count = 0;
	interface_centroid(1) = 0.0;
	interface_centroid(2) = 0.0;
	interface_centroid(3) = 0.0;

//mj ckeck ligand flag
	int startid,endid;
	if ( get_ligand_flag() ) {
		startid = 1;
		endid = total_residue;
	} else {
		startid = part_begin(1);
		endid = part_end(2);
	}

//     accumulate
	for ( int i = startid; i <= endid; ++i ) {
		if ( int_res(i) ) {
			interface_centroid(1) += Eposition(1,2,i);
			interface_centroid(2) += Eposition(2,2,i);
			interface_centroid(3) += Eposition(3,2,i);
			++atom_count;
		}
	}

//     calculate the centroid
	if ( atom_count != 0 ) {
		float const one_atom_count = 1.0 / atom_count;
		interface_centroid(1) *= one_atom_count;
		interface_centroid(2) *= one_atom_count;
		interface_centroid(3) *= one_atom_count;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin chain_centroid
///
/// @brief Calculate the geometrical center of a chain of residues,
///       weighting each atom equally
/// @detailed
///
/// @param[in]   xyz - in - coordinates of a chain of residues
/// @param[in]   aan - in - sequence of a chain of residues
/// @param[in]   aav - in - aa variable for each residues
/// @param[in]   start_res - in - sequence number of the starting residues in xyz
/// @param[in]   end_res - in -sequence number of the end residue in xyz
/// @param[out]   ch_centroid - out - coordinate of geometric center
///
/// @global_read natoms in aaproperties.h
///
/// @global_write
///
/// @remarks all atoms are considered, including H
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
chain_centroid(
	FArray3Da_float xyz, // atom position array, in full_coord format
	FArray1Da_int aan, // amino acid numbers
	FArray1Da_int aav, // amino acid variants
	int & start_res,
	int & end_res,
	FArray1Da_float ch_centroid // the chain centroid
)
{
	using namespace aaproperties_pack;
	using namespace param;

	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );
	aan.dimension( MAX_RES() );
	aav.dimension( MAX_RES() );
	ch_centroid.dimension( 3 );

//     initialize
	int atom_count = 0; // total number of atoms in the chain
	ch_centroid(1) = 0.0;
	ch_centroid(2) = 0.0;
	ch_centroid(3) = 0.0;

//     loop over all atoms
	for ( int i = start_res; i <= end_res; ++i ) { // residues
		for ( int j = 1, je = natoms( aan(i), aav(i) ); j <= je; ++j ) { // atoms in each residue
			ch_centroid(1) += xyz( 1, j, i );
			ch_centroid(2) += xyz( 2, j, i );
			ch_centroid(3) += xyz( 3, j, i );
			++atom_count;
		}
	}

//     calculate the centroid
	float const one_atom_count = 1.0 / atom_count;
	ch_centroid(1) *= one_atom_count;
	ch_centroid(2) *= one_atom_count;
	ch_centroid(3) *= one_atom_count;

// debug
//	std::cout << "Centroid is ";
//	vector_write( std::cout,ch_centroid)

}

////////////////////////////////////////////////////////////////////////////////
/// @begin chain_centroid_CA
///
/// @brief Calculate the geometrical center of a chain of residues,
///       weighting each residue equally using C-alpha positions
/// @detailed
///
/// @param[in]   xyz - in - coordinates of a chain of residues
/// @param[in]   start_res - in - sequence number of the starting residue in xyz
/// @param[in]   end_res - in -sequence number of the end residue in xyz
/// @param[out]   ch_centroid - out - coordinate of geometric center
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
chain_centroid_CA(
	FArray3Da_float xyz, // atom position array, in Eposition format
	int start_res,
	int end_res,
	FArray1Da_float ch_centroid // the chain centroid
)
{
	using namespace param;

	xyz.dimension( 3, MAX_POS, MAX_RES() );
	ch_centroid.dimension( 3 );

	// initialize
	ch_centroid = 0.0;

	// loop over all atoms
	for ( int i = start_res; i <= end_res; ++i ) { // residues
		ch_centroid(1) += xyz(1,2,i);
		ch_centroid(2) += xyz(2,2,i);
		ch_centroid(3) += xyz(3,2,i);
	}

	// calculate the centroid
	assert( ( end_res >= start_res ) );
	ch_centroid /= end_res - start_res + 1; // Divide by atom count

// debug
//	std::cout << "Centroid is ";
//	vector_write( std::cout,ch_centroid);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin vector_centroid
///
/// @brief Calculate the geometrical center of a chain of residues,
///       weighting each residue equally using C-alpha positions
/// @detailed
///		same as chain_centroid_CA function for use with a vector of xyzVectors
///		instead of with an xyz FArray3Da_float
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references Chu Wang, previous function
///
/// @authors Monica Berrondo
///
/// @last_modified April 12 2006
/////////////////////////////////////////////////////////////////////////////////
numeric::xyzVector< double >
vector_centroid(
	std::vector< numeric::xyzVector< double > > const xyz // atom position array, in Eposition format
)
{
	// initialize
	numeric::xyzVector< double > ch_centroid(0.0);

	// loop over all atoms
	for ( int i = 0; i < (int)xyz.size(); ++i ) { // residues
		ch_centroid += xyz.at(i);
	}

	// calculate the centroid
	ch_centroid /= xyz.size(); // Divide by atom count
	return ch_centroid;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_score_input
///
/// @brief score native/input structures
///
/// @detailed
/// score docking native/input structure; check the docking submode and also
/// score their descendents in those modes, ie nat_repacked, nat_min, nat_mcm...
/// dump out pdb files for these structures if command line flag is specified.
/// double check native/input structure does not fail any filters.
///
/// @param[in]   filename - in - structure to score
/// @param[in]   label_in - in - native or input?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     docking mode needs to output repacked and minimized scores. do not
///     save the coordinates (save_START) as would happen with the
///     repack_input flag. also, we have here the option to output the pdb
///     files of the start or native with the flags output_input_pdb and
///     output_native_pdb.
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_score_input(
	std::string const & filename,
	std::string const & label // 'native' or 'input'
)
{
	using namespace docking;
	using namespace files_paths;
	using namespace runlevel_ns;

	if ( !multi_chain || design::dna_interface || antibody_modeler) return;
	if ( !docking_fullatom_flag ) return;
	if ( docking::docking_fake_native ) return;
	if ( surface::mode.flag.is_enabled() ) return;

	docking_store_initial_rms();

	bool const output_pdb = truefalseoption( "output_" + label + "_pdb" );

	std::string unique_file_name;
	if ( label == "input" ) {
		unique_file_name = code + protein_name + '_' + start_file + '_' + label;
	} else {
		unique_file_name = code + protein_name + '_' + label;
	}

	if ( output_pdb ) make_named_pdb( unique_file_name + ".pdb", true );

//chu in score mode, just score the current without further repacking
	if ( docking_score_norepack ) return;

	FArray1D_bool const save_use_filter( use_filter );

	disable_all_filters();

	std::string const label3( label.substr( 0, 3 ) );
	if ( scorefile_entry_present(filename,label3+"_repacked",true) ) {
		if ( runlevel >= inform ) std::cout << label3 <<
		 "_repacked already in scorefile, skipping." << std::endl;
	} else {
		fullatom_score_position(true,include_inputchi);
		scorefile_output( filename, label3 + "_repacked", true );
		if ( output_pdb )
		 make_named_pdb( unique_file_name + "_repacked.pdb", true );

		if ( docking_minimize_flag ) {
			if ( scorefile_entry_present( filename, label3 + "_min", true ) ) {
				if ( runlevel >= inform ) std::cout << label3 <<
				 "_min already in scorefile, skipping." << std::endl;
			} else {
				docking_fullatom_min_score_pose(false);
				scorefile_output(filename,label3+"_min",true);
				if ( output_pdb )
				 make_named_pdb( unique_file_name + "_min.pdb", true );

				if ( docking_mcm_flag ) {
					if ( scorefile_entry_present( filename, label3 + "_mcm", true ) ) {
						if ( runlevel >= inform ) std::cout << label3 <<
						 "_mcm already in scorefile, skipping." << std::endl;
					} else {
						docking_fullatom_mcm_score_pose(false);
						scorefile_output( filename, label3 + "_mcm", true );
						if ( output_pdb )
						 make_named_pdb( unique_file_name + "_mcm.pdb", true );
					} // mcm
				}
			} // min
		}
	} // rep

	use_filter = save_use_filter;
	docking_reset_tracking();

//     make sure the native doesn't fail the filters!!
//     (except score filter, that's legit)
//     for now dont die in ligand mode
	if ( use_filter(dock_type) && get_docking_rmsd() < 5.0 &&
			 !docking_fake_native && !get_ligand_flag()) {
		bool accepted;
		std::string tag;
		apply_filters(accepted,tag); // rms_save => accepted is T
		if ( ( ! has_prefix( tag, "pass" ) ) &&
		 ( ! has_prefix( tag, "rms_save_score_failure" ) ) ) {
			user_x.clear();
			user_x.open( "CRASH_REPORT" );
			if ( !user_x ) {
				std::cout << "Open failed for file: " << "CRASH_REPORT" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			pad( tag, TAG_MAX_LENGTH );
			user_x << tag << label << std::endl; // for /dev/null runs
			user_x.close();
			user_x.clear();
			std::cout << tag << label << std::endl;
			error_stop("FATAL: native fails filters!");
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_init_unboundrot_query
///
/// @brief read in and initialize the docking unbound structure
///
/// @detailed
///     This function is to read unbound pdb files and save coord of native
///     side chain rotamers. Further, native chi angles will be calculated and
///     included when packing.
///     Alternatively, if bound structures are provided
///     in paths.txt:/pdb2, native rotamers in the complex will be added.
///
///     Futher modified to set up the sequence alignment map when unbound pdb is
///     different from native in sequence and only those rotamers at the seqpos
///     where AAs match each other will be added for side-chain packing. Also,
///     the seq align map will be used for rmsd calculation when starting from
///     an unbound backbone.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     docking native will always be the cocrystalized structure.
///     docking start can be either bound backbone (docking native) or
///     unbound backbone (as defined here), and a third sequence other than
///     these two will not be supported.
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_init_unboundrot_query()
{
	using namespace docking;
	using namespace files_paths;

	if ( !multi_chain || design::dna_interface || antibody_modeler) return;
	if ( get_ligand_flag() ) return;
	if ( !unboundrot && !unbound_start ) return;

	std::string filename( pdb_path_1 + protein_name_prefix + protein_name + ".unbound.pdb" );
	std::cout << "Searching for pdb...: " << filename << std::endl;
	pdb_x.clear();
	pdb_x.open( filename );
	if ( !pdb_x ) {
		filename = pdb_path_2 + protein_name_prefix + protein_name + ".unbound.pdb";
		std::cout << "Searching for pdb...: " << filename << std::endl;
		pdb_x.clear();
		pdb_x.open( filename );
		if ( !pdb_x ) {
			std::cout << "either unbound_start or unboundrot has been turned on..." << std::endl;
			std::cout << "an unbound structure is required" << std::endl;
			std::cout << "no unbound structure for " << pdb_x.filename() << std::endl;
			pdb_x.clear();
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	bool fail;
	input_pdb(pdb_x,use_fasta,input_fa,fail); // read corrd of unbound
	pdb_x.close();
	pdb_x.clear();

	if ( fail ) {
		std::cout << "Failed to input unbound: " << protein_name_prefix << protein_name << ".unbound.pdb" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	docking_save_unbound();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_save_unbound
///
/// @brief save information for docking unbound structure
///
/// @detailed copy coordinates read in from current array to unbound array and
///      calculate the torsion angles based on the coordinates
///
/// @global_read coordinate array in misc.h
///
/// @global_write unbound array in unbound_pose namespace
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_save_unbound()
{
//chu   save current information in UNBOUND arrays and determine chi angles

	using namespace docking_unbound;
	using namespace misc;
	using namespace param;

	unbound_total_res = total_residue;

	for ( int j = 1, max_atom = MAX_ATOM(); j <= total_residue; ++j ) {
		unbound_res(j) = res(j);
		unbound_res_variant(j) = res_variant(j);
		unbound_phi(j) = phi(j);
		unbound_psi(j) = psi(j);
		unbound_omega(j) = omega(j);

		for ( int i = 1; i <= max_atom; ++i ) {
			for ( int k = 1; k <= 3; ++k ) {
				unbound_fullcoord(k,i,j) = full_coord(k,i,j);
			}
		}
	}

	for ( int j = 1; j <= total_residue; ++j ) { // determine action center of each side chain
		int aa = unbound_res(j);
		put_wcentroid(unbound_fullcoord(1,1,j), unbound_actcoord(1,j),aa);
	}

	// determine chi angles
	get_chi_and_rot_from_coords(total_residue,res,res_variant,full_coord,unbound_chiarray,unbound_rotarray);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_unbound_rotamers
///
/// @brief get extra rotamers from unbound structure when packing
///
/// @detailed
///
/// @param[out]   unbound_rot - out - rotamer conformation number for each side chain
/// @param[out]   unbound_chi - out - chi angles for each side chain
///
/// @global_read unbound array in namespace
///
/// @global_write
///
/// @remarks current structure share the same sequence as unbound structure
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_unbound_rotamers(
	FArray2Da_int unbound_rot,
	FArray2Da_float unbound_chi
)
{
	using namespace docking_unbound;
	using namespace misc;
	using namespace param;
	using namespace runlevel_ns;

	unbound_rot.dimension( MAX_CHI, MAX_RES() );
	unbound_chi.dimension( MAX_CHI, MAX_RES() );

	if ( get_ligand_flag() ) return;
//chu   unbound_map_set = true : native and unbound have different sequences
//     use_mapped_rmsd = false: native sequence in current arrays
//     so check sequence match before retrieving additional rotamers under
//     this situation

	int counter = 0;

	if ( unbound_map_set && !use_mapped_rmsd ) {
		for ( int j = 1; j <= total_residue; ++j ) {
			int jj = unbound_seq_align_map(j);
			if ( jj != 0 && res(j) == unbound_res(jj) ) {
				for ( int i = 1; i <= MAX_CHI; ++i ) {
					unbound_rot(i,j) = unbound_rotarray(i,jj);
					unbound_chi(i,j) = unbound_chiarray(i,jj);
				}
				++counter;
			} else {
				for ( int i = 1; i <= MAX_CHI; ++i ) {
					unbound_rot(i,j) = 0;
					unbound_chi(i,j) = 0.0;
				}
			}
		}
	} else {
		for ( int j = 1; j <= total_residue; ++j ) {
			for ( int i = 1; i <= MAX_CHI; ++i ) {
				unbound_rot(i,j) = unbound_rotarray(i,j);
				unbound_chi(i,j) = unbound_chiarray(i,j);
			}
			++counter;
		}
	}

	if ( runlevel > inform ) std::cout << "retrieve unbound rotamers for " <<
	 I( 4, counter ) << " postions" << std::endl;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin get_size_from_docking_map
///
/// @brief
///
/// @detailed
/// read number of residues in sequence alignment map
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// returns 0 if map file is not read or empty
///
/// @references
///
/// @authors Stuart Mentzer 2005/03/25  (based on docking_mapping_native)
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_size_from_docking_map()
{
	using namespace files_paths;
	using namespace param;

	if ( !multi_chain || design::dna_interface || antibody_modeler ) return;
	if ( get_ligand_flag() ) return;
	if ( !get_unboundrot_flag() && !get_unbound_start_flag() ) return;

	// local
	std::string native3, unbound3;

	// Open the map file and read the number of residues
	utility::io::izstream map_stream( seq_path + protein_name_prefix + protein_name + ".map" );
	if ( ! map_stream ) return;
	int alignment_res = 0;
	while ( map_stream ) {
		map_stream >> skip( 7 ) >> bite( native3 ) >> skip( 11 ) >> bite( unbound3 ) >> skip;
		if ( map_stream ) ++alignment_res;
	}
	map_stream.close();
	map_stream.clear();

	// Set MAX_RES if map file read successfully
	if ( alignment_res > 0 ) MAX_RES_assign_res( alignment_res ); // Set MAX_RES
}


//////////////////////////////////////////////////////////////////////////////
/// @begin docking_mapping_native
///
/// @brief
///
/// @detailed
///chu  read in sequence alignment map of native bound and native unbound
///     convert it into format of "seqpos vs seqpos" as defined in rosetta
///     then set up the map so that given the seqpos in native bound structure,
///     look up its equivalent in native unbound structure.
///
///     1         11   alignment_res
///     ACDGEF-YN-L    nat_seq_char
///     |||||||||||
///     -CD-EY-YNDI    unbound_seq_char
///
///     12345607809    nat_seq_pos
///     |||||||||||
///     01203405678    unbound_seq_pos
///
///     align_map(1) ==> 0
///     align_map(2) ==> 1
///     align_map(3) ==> 2
///     align_map(4) ==> 0
///     ..........
///     align_map(9) ==> 8
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_mapping_native()
{
	using namespace docking_unbound;
	using namespace files_paths;
	using namespace misc;
	using namespace param;

	// assume both native and unbound have been input correctly
	std::cout << "MAP routine" << std::endl;
	if ( !multi_chain || design::dna_interface || antibody_modeler ) return;
	if ( get_ligand_flag() ) return;
	if ( !get_unboundrot_flag() && !get_unbound_start_flag() ) return;

	// local
	FArray1D_int nat_seq_aa( MAX_RES()() );
	FArray1D_int unbound_seq_aa( MAX_RES()() );
	FArray1D_int nat_seq_pos( MAX_RES()() );
	FArray1D_int unbound_seq_pos( MAX_RES()() );
	std::string native3, unbound3;

	// check if they have the same sequence
	if ( total_residue != unbound_total_res ) goto L10;
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( res(i) != unbound_res(i) ) goto L10;
	}
	std::cout << "Native and unbound structures share the same sequence." << std::endl;
	std::cout << "Mapping is not needed and will not be set up.\n" << std::endl;
	return;

L10:

	std::string const filename( seq_path + protein_name_prefix + protein_name + ".map" );
	std::cout << "Searching for docking sequence alignment map...:" <<
	 filename << std::endl;

  utility::io::izstream seq_zx( filename );

	if ( !seq_zx ) {
		std::cout << "Having trouble in opening docking native map file: " <<
		 seq_zx.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//     read the alignment
//$$$      int i = 1;
//$$$      while ( seq_zx ) {
//$$$
//$$$					seq_zx >> skip( 16 );
//$$$					for ( int j = i; j <= i+59; ++j ) {
//$$$						seq_zx >> bite( nat_seq_char(j) );
//$$$					} seq_zx >> skip;
//$$$					if ( !seq_zx ) goto L200;
//$$$					seq_zx >> skip( 16 );
//$$$					for ( int j = i; j <= i+59; ++j ) {
//$$$						seq_zx >> bite( unbound_seq_char(j) );
//$$$					} seq_zx >> skip;
//$$$					if ( !seq_zx ) goto L200;
//$$$
//$$$					for ( int j = i; j <= i+59; ++j ) {
//$$$						std::cout << nat_seq_char(j);
//$$$					} std::cout << std::endl;
//$$$					for ( int j = i; j <= i+59; ++j ) {
//$$$						std::cout << unbound_seq_char(j);
//$$$					} std::cout << std::endl;
//$$$
//$$$         i += 60;
//$$$      }
//$$$
//$$$ L200:
//$$$			 seq_zx.close();
//$$$
//$$$      if ( i == 1 ) {
//$$$         std::cout << "no sequence alignment maps found" << std::endl;
//$$$					for ( int j = i; j <= i+59; ++j ) {
//$$$						std::cout << nat_seq_char(j);
//$$$					} std::cout << std::endl;
//$$$					for ( int j = i; j <= i+59; ++j ) {
//$$$						std::cout << unbound_seq_char(j);
//$$$					} std::cout << std::endl;
//$$$         utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
//$$$      }
//$$$
//$$$      alignment_res = i-1;

	// read the alignment
	int alignment_res = 0;
	while ( seq_zx ) {
		seq_zx >> skip( 7 ) >> bite( 3, native3 ) >> skip( 11 ) >> bite( 3, unbound3 ) >> skip;
		if ( seq_zx ) {
			++alignment_res;
			num_from_name( native3, nat_seq_aa(alignment_res) );
			num_from_name( unbound3, unbound_seq_aa(alignment_res) );
		}
	}
	seq_zx.close();
	seq_zx.clear();

	// check the alignment
	if ( alignment_res == 0 ) {
		std::cout << "no sequence alignment maps found" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	check_docking_native_map(alignment_res,nat_seq_aa,nat_seq_pos);
	check_docking_unbound_map(alignment_res,unbound_seq_aa,unbound_seq_pos);

	// set up alignment map in form of seqpos num: native ==> unbound
	int counter1 = 0;
	int counter2 = 0;
	for ( int i = 1; i <= alignment_res; ++i ) {
		if ( nat_seq_pos(i) != 0 ) {
			unbound_seq_align_map(nat_seq_pos(i)) = unbound_seq_pos(i);
			if ( unbound_seq_pos(i) != 0 ) {
				++counter1;
				if ( nat_seq_aa(i) == unbound_seq_aa(i) ) {
					++counter2;
				}
			}
		}
	}

	unbound_map_set = true;

	std::cout << "************************************************" << std::endl;
	std::cout << "Unbound sequence map onto native set up" << std::endl;
	std::cout << "Native: " << total_residue << " residues" << std::endl;
	std::cout << "Unbound: " << unbound_total_res << " residues" << std::endl;
	std::cout << "Number of aligned positions: " << counter1 << std::endl;
	std::cout << "Number of identical positions: " << counter2 << std::endl;
	std::cout << "************************************************" << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin check_docking_native_map
///
/// @brief
///chu   check if the read-in map matches the docking native (bound) sequence
///     saved in misc.h. also convert the map from char to num
///
/// @detailed
///
/// @param  align - [in/out]? - num of characters read in
/// @param  unbound_aa - [in/out]? - seq to be aligned
/// @param  unbound_pos - [in/out]? - seqpos num in rosetta
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
check_docking_native_map(
	int const align, // num of characters read in
	FArray1Da_int native_aa, // seq to be aligned
	FArray1Da_int native_pos // seqpos num in rosetta
)
{
	using namespace misc;
	using namespace param;

	native_aa.dimension( MAX_RES() );
	native_pos.dimension( MAX_RES() );


	int pdb_res = 0;
	for ( int i = 1; i <= align; ++i ) {
		if ( native_aa(i) == 0 ) {
			native_pos(i) = 0;
			goto L11; // cycle
		}
		++pdb_res;
//$$$c     convert to upper case
//$$$         if ( native_seq(i) >= 'a' && native_seq(i) <= 'z' )
//$$$          native_seq(i) = CHAR(ICHAR(native_seq(i))-ICHAR('a')+ICHAR("A"));
//$$$         res1_from_num(res(pdb_res),res_name);
		if ( native_aa(i) != res(pdb_res) ) {
			std::cout << "Docking bound pdb map misaligned:" << std::endl;
			std::cout << native_aa(i) << ' ' << i << " in .map file" << std::endl;
			std::cout << res(pdb_res) << ' ' << pdb_res << " in .pdb file" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		native_pos(i) = pdb_res;
L11:;
	}
	if ( pdb_res != total_residue ) {
		std::cout << "Docking bound pdb map misaligned:" << std::endl;
		std::cout << pdb_res << " residues in total in native map" << std::endl;
		std::cout << total_residue << " residues in total in native pdb" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin check_docking_unbound_map
///
/// @brief
///chu   check if the read-in map matches the docking unbound sequence
///     also convert the map from char to num
///
/// @detailed
///
/// @param  align - [in/out]? - num of characters read in
/// @param  unbound_aa - [in/out]? - seq to be aligned
/// @param  unbound_pos - [in/out]? - seqpos num in rosetta
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////////
void
check_docking_unbound_map(
	int const align, // num of characters read in
	FArray1Da_int unbound_aa, // seq to be aligned
	FArray1Da_int unbound_pos // seqpos num in rosetta
)
{
	using namespace docking_unbound;
	using namespace param;

	unbound_aa.dimension( MAX_RES() );
	unbound_pos.dimension( MAX_RES() );


	int pdb_res = 0;
	for ( int i = 1; i <= align; ++i ) {
		if ( unbound_aa(i) == 0 ) {
			unbound_pos(i) = 0;
			goto L11; // cycle
		}
		++pdb_res;
//     convert to upper case
//$$$         if ( unbound_seq(i) >= 'a' && unbound_seq(i) <= 'z' )
//$$$          unbound_seq(i) = CHAR(ICHAR(unbound_seq(i))-ICHAR('a')+ICHAR("A"));
//$$$         res1_from_num(unbound_res(pdb_res),res_name);
		if ( unbound_aa(i) != unbound_res(pdb_res) ) {
			std::cout << "Docking unbound pdb map misaligned:" << std::endl;
			std::cout << unbound_aa(i) << ' ' << i << " in unbound map file" << std::endl;
			std::cout << unbound_res(pdb_res) << ' ' << pdb_res << " in unbound pdb file" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		unbound_pos(i) = pdb_res;
L11:;
	}
	if ( pdb_res != unbound_total_res ) {
		std::cout << "Docking unbound pdb map misaligned:" << std::endl;
		std::cout << pdb_res << " residues in total in .map file" << std::endl;
		std::cout << unbound_total_res << " residues in total in .pdb file" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///
/// @detailed
///chu   contact definition: heavy-atom to heavy-atom less than 4.0A
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_store_nat_contact_map()
{

	using namespace aaproperties_pack;
	using namespace docking;
	using namespace interface;
	using namespace misc;
	using namespace param;

	float const dis2_cutoff = { 16.0 };

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;
	if ( !docking_fullatom_flag ) return;
	if ( get_ligand_flag() ) return;

	// initialize
	docking_native_contact_map = false;
	n_native_contact = 0;

	for ( int seqpos1 = part_begin(1), seqpos1e = part_end(1), seqpos2e = part_end(2);
	 seqpos1 <= seqpos1e; ++seqpos1 ) {
		int res1 = res(seqpos1);
		int res1_variant = res_variant(seqpos1);
		for ( int seqpos2 = part_begin(2), atom1e = nheavyatoms( res1, res1_variant );
		 seqpos2 <= seqpos2e; ++seqpos2 ) {
			int res2 = res( seqpos2 );
			int res2_variant = res_variant( seqpos2 );
			for ( int atom1 = 1, atom2e = nheavyatoms( res2, res2_variant ); atom1 <= atom1e; ++atom1 ) {
				for ( int atom2 = 1; atom2 <= atom2e; ++atom2 ) {
					if ( vec_dist2( full_coord(1,atom1,seqpos1), full_coord(1,atom2,seqpos2) ) <= dis2_cutoff ) {
						docking_native_contact_map( seqpos1, seqpos2 ) = true;
						docking_native_contact_map( seqpos2, seqpos1 ) = true;
						++n_native_contact;
						goto L100;
					}
				}            // atom2
			}               // atom1
L100:;
		}                  // res2
	}                     // res1

	if ( n_native_contact == 0 ) {
		std::cout << "WARNING: ###docking_store_native_contact_map###" << std::endl;
		std::cout << "n_native_contact=0, no contacts in native!!!" << std::endl;
		return;
	}

	std::cout << "There are " << n_native_contact <<
	 " contacts in native structure!" << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin convert_docking_nat_contact_map
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
convert_docking_nat_contact_map()
{

	using namespace docking;
	using namespace docking_unbound;
	using namespace interface;
	using namespace param;


	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler) return;
	if ( get_ligand_flag() ) return;
	if ( !unbound_start ) return; // bound backbone
	if ( !unbound_map_set ) return; // same sequence

	set_docking_use_mapped_rmsd(true); // mapping before calc rmsd
	if ( n_native_contact == 0 ) return;
	if ( docking::docking_fake_native ) return;
	if ( surface::mode.flag.is_enabled() ) return;

	FArray2D_bool tmp( MAX_RES()(), MAX_RES()() );
	int const ibegin = native_part_begin(1), iend = native_part_end(1);
	int const jbegin = native_part_begin(2), jend = native_part_end(2);

	for ( int i = ibegin; i <= iend; ++i ) {
		for ( int j = jbegin; j <= jend; ++j ) {
			tmp(i,j) = docking_native_contact_map(i,j);
			tmp(j,i) = docking_native_contact_map(j,i);
		}
	}

	for ( int i = ibegin; i <= iend; ++i ) {
		for ( int j = jbegin; j <= jend; ++j ) {
			docking_native_contact_map(i,j) = false;
			docking_native_contact_map(j,i) = false;
		}
	}
	n_native_contact = 0;

	for ( int i = ibegin; i <= iend; ++i ) {
		for ( int j = jbegin; j <= jend; ++j ) {
			int ii = unbound_seq_align_map(i);
			int jj = unbound_seq_align_map(j);
			if ( ii != 0 && jj != 0 ) {
				docking_native_contact_map(ii,jj) = tmp(i,j);
				docking_native_contact_map(jj,ii) = tmp(j,i);
				if ( tmp(i,j) ) ++n_native_contact;
			}
		}
	}

	std::cout << "There are" << n_native_contact <<
	 " contacts in mapped structure!" << std::endl;



}

////////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////////
void
calc_docking_frac_nat_contact()
{

	using namespace aaproperties_pack;
	using namespace docking;
	using namespace interface;
	using namespace misc;
	using namespace param;

	float const dis2_cutoff = { 16.0 };

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;
	if ( !docking_fullatom_flag ) return;
	if ( get_ligand_flag() ) {
		frac_native_contact = 0.0;
		frac_nonnat_contact = 0.0;
		return;
	}
	if ( !get_native_exists() || n_native_contact == 0 ) {
		std::cout << "WARNING: ###calc_docking_frac_native_contact###" << std::endl;
		std::cout << "No native exists or no contacts in the native structure" << std::endl;
		std::cout << "fraction of native contacts will not be evaluated!!!" << std::endl;
		frac_native_contact = 0.0;
		frac_nonnat_contact = 0.0;
		return;
	}

//     initilaize
	int n_decoy_contact = 0;
	FArray2D_bool docking_decoy_contact_map( MAX_RES()(), MAX_RES()(), false );

//     set up decoy contact map
	for ( int seqpos1 = part_begin(1), seqpos1e = part_end(1), seqpos2e = part_end(2);
	 seqpos1 <= seqpos1e; ++seqpos1 ) {
		int res1 = res(seqpos1);
		int res1_variant = res_variant(seqpos1);
		for ( int seqpos2 = part_begin(2), atom1e = nheavyatoms(res1,res1_variant);
		 seqpos2 <= seqpos2e; ++seqpos2 ) {
			int res2 = res(seqpos2);
			int res2_variant = res_variant(seqpos2);
			for ( int atom1 = 1, atom2e = nheavyatoms(res2,res2_variant); atom1 <= atom1e; ++atom1 ) {
				for ( int atom2 = 1; atom2 <= atom2e; ++atom2 ) {
					if ( vec_dist2( full_coord(1,atom1,seqpos1), full_coord(1,atom2,seqpos2) ) <= dis2_cutoff ) {
						docking_decoy_contact_map( seqpos1, seqpos2 ) = true;
						docking_decoy_contact_map( seqpos2, seqpos1 ) = true;
						++n_decoy_contact;
						goto L100;
					}
				} // atom2
			} // atom1
L100:;
		} // res2
	} // res1

	if ( n_decoy_contact == 0 ) {
		if ( !get_docking_prepack_mode() ) {
			std::cout << "WARNING: ###calc_docking_frac_native_contact###"
								<< std::endl;
			std::cout << "n_decoy_contact=0, no contacts in decoy!!!" << std::endl;
			std::cout << "fraction of native contacts will not be evaluated!!!"
								<< std::endl;
		}
		frac_native_contact = 0.0;
		frac_nonnat_contact = 0.0;
		return;
	}

//     calculate fraction of native contact in decoy
	int n_correct_native_contact = 0;
	int n_false_positive_contact = 0;
	for ( int seqpos1 = part_begin(1), seqpos1e = part_end(1), seqpos2e = part_end(2);
	 seqpos1 <= seqpos1e; ++seqpos1 ) {
		for ( int seqpos2 = part_begin(2); seqpos2 <= seqpos2e; ++seqpos2 ) {
			if ( docking_decoy_contact_map(seqpos1,seqpos2) ) {
				if ( docking_native_contact_map(seqpos1,seqpos2) ) {
					++n_correct_native_contact;
				} else {
					++n_false_positive_contact;
				}
			} // contact in decoy
		} // seqpos2
	} // seqpos1

	frac_native_contact =
	 static_cast< float >( n_correct_native_contact ) / n_native_contact;
	frac_nonnat_contact =
	 static_cast< float >( n_false_positive_contact ) / n_decoy_contact;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_docking_number_of_contacts
///
/// @brief
/// calculation of ncont: number of interface atom pairs.
/// the  number of contacts in the interface is defined as the number of
/// heavy atom pairs within 5A (as in John's script)
///
/// @detailed
/// currently the loop is over all atoms, but a cutoff on the residue level could
/// precede this
///
/// currently ncont is set to 0 for ligand_flag
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors ora
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_docking_number_of_contacts()
{

	using namespace aaproperties_pack;
	using namespace docking;
	using namespace files_paths;
	using namespace interface;
	using namespace misc;

	int res1, res2, res1_variant, res2_variant;

	float const dis5_cutoff = { 25.0 };

	if ( !multi_chain || design::dna_interface || antibody_modeler ) return;
	if ( ! docking_fullatom_flag ) return;
	if ( part_begin(1) == 0 ) return;

	n_contacts = 0;
	if ( get_ligand_flag() ) {
		calc_number_of_ligand_contacts( n_contacts, (*ligand::ligand_one));
		return;
	}

	for ( int seqpos1 = part_begin(1), seqpos1e = part_end(1), seqpos2e = part_end(2);
	 seqpos1 <= seqpos1e; ++seqpos1 ) {
		res1 = res(seqpos1);
		res1_variant = res_variant(seqpos1);
		for ( int seqpos2 = part_begin(2), atom1e = nheavyatoms(res1,res1_variant);
		 seqpos2 <= seqpos2e; ++seqpos2 ) {
			res2 = res(seqpos2);
			res2_variant = res_variant(seqpos2);
			for ( int atom1 = 1, atom2e = nheavyatoms(res2,res2_variant); atom1 <= atom1e; ++atom1 ) {
				for ( int atom2 = 1; atom2 <= atom2e; ++atom2 ) {
					if ( vec_dist2( full_coord(1,atom1,seqpos1),
					 full_coord(1,atom2,seqpos2) ) <= dis5_cutoff ) ++n_contacts;
				} // atom2
			} // atom1
		} // res2
	} // res1

	if ( n_contacts == 0 && !get_docking_prepack_mode()) {
		std::cout << "WARNING: ###calc_docking_number_of_contacts###" << std::endl;
		std::cout << "n_contacts=0, no contacts in decoy!!!" << std::endl;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_docking_interf_rmsd
///
/// @brief
///chu   Identify interface res in the native structure(8A definition).
///chu   Superimpose interface CAs onto their equivalents in the decoy and
///chu   calculate rmsd.  NOT adapted yet for symmetry mode!!
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_docking_interf_rmsd()
{


	using namespace docking;
	using namespace docking_unbound;
	using namespace interface;
	using namespace misc;
	using namespace native;
	using namespace param;

	FArray2D_double native_interf_ca( 3, MAX_RES()() );
	FArray2D_double decoy_interf_ca( 3, MAX_RES()() );

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler) return;
	if ( get_ligand_flag() ) {
		docking_interf_rmsd = 0.0;
	}

	if ( !get_native_exists() ) {
		std::cout << "WARNING: ###calc_docking_interf_rmsd###" << std::endl;
		std::cout << "No native exists !!!" << std::endl;
		std::cout << "docking_interf_rmsd will not be evaluated!!!" << std::endl;
		docking_interf_rmsd = 0.0;
		return;
	}

	int n_interf_res = 0;
	if ( !use_mapped_rmsd ) {
		for ( int j = 1; j <= max_docking_sites; ++j ) {
			for ( int i=1; i<= int( native_site_residues(j).size() ); ++i) {
				++n_interf_res;
				int const nat_seqpos = native_site_residues(j)[i];
				for ( int k = 1; k <= 3; ++k ) {
					native_interf_ca(k,n_interf_res) = native_ca(k,nat_seqpos);
					decoy_interf_ca(k,n_interf_res) = Eposition(k,2,nat_seqpos);
				}
			}
		}
	} else {
		for ( int j = 1; j <= max_docking_sites; ++j ) {
			for ( int i=1; i<=int( native_site_residues(j).size() ); ++i ){
				int const nat_seqpos = native_site_residues(j)[i];
				int const start_seqpos = unbound_seq_align_map(nat_seqpos);
				if ( start_seqpos == 0 ) continue;
				++n_interf_res;
				for ( int k = 1; k <= 3; ++k ) {
					native_interf_ca(k,n_interf_res) = native_ca(k,nat_seqpos);
					decoy_interf_ca(k,n_interf_res) = Eposition(k,2,start_seqpos);
				}
			}
		}
	}
	if ( n_interf_res < 2 && ! get_docking_prepack_mode() ) {
		std::cout << "WARNING:###calc_docking_interf_rmsd###" << std::endl;
		std::cout << " # of interf res is less than 2" << std::endl;
		std::cout << " docking_interf_rmsd will not be evaluated " << std::endl;
		docking_interf_rmsd = 0.0;
		return;
	}

	fast_rms_ca(native_interf_ca,decoy_interf_ca,n_interf_res,
	 docking_interf_rmsd);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin calc_docking_hb_stats()
///
/// @brief output info on interfacial hbonds and hbond networks
///       to standard error.
/// @detailed only active with command-line option "-docking_hb_stats"
///
/// @return  nothing
///
/// @global_read docking_hb_stats in docking.h
///             start_file in files_paths.h
///             pdb_res_num,res_chain in pdb.h
///             hbond arrays in hbonds.h
/// @global_write nothing
///
/// @remarks  only called in update_scorefile_info in status.f
///
/// @references
///
/// @authors Bill Schief 4/20/04
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

//------------------------------------------------------------------------------
//bs function used to output info on interfacial hbonds and hbond networks
//bs stats are output to standard error.
//bs only call to this is in update_scorefile_info in status.f

void
calc_docking_hb_stats()
{
	using namespace aaproperties_pack;
	using namespace files_paths;
	using namespace docking;
	using namespace hbonds;
	using namespace misc;
	using namespace param;
	using namespace pdb;

	if ( ! docking_hb_stats ) return;

	bool output_orig_pdb_resnum = true;

	if ( !multi_chain || design::dna_interface || antibody_modeler ) return;
	if ( ! docking_fullatom_flag ) return;
	if ( get_ligand_flag() ) {
		std::cout << "no docking hbonds for ligands yet" << std::endl;
		return;
	}

	float const dis2_cutoff = { 16.0 };
	int res1, res2, res1_variant, res2_variant;
	FArray2D_bool docking_decoy_contact_map( MAX_RES()(), MAX_RES()(), false );
	FArray1D_bool docking_decoy_res_w_contact( MAX_RES()(), false );
//	int n_correct_native_contact, n_false_positive_contact;
	int dres,ares,dhatm,aatm;
	float hbE;
	std::string linetag;
	char dchain, achain;

// initialize
	int n_decoy_contact = 0;

// set up decoy contact map
	for ( int seqpos1 = part_begin(1), seqpos1e = part_end(1), seqpos2e = part_end(2);
	 seqpos1 <= seqpos1e; ++seqpos1 ) {
		res1 = res(seqpos1);
		res1_variant = res_variant(seqpos1);
		for ( int seqpos2 = part_begin(2), atom1e = nheavyatoms(res1,res1_variant);
		 seqpos2 <= seqpos2e; ++seqpos2 ) {
			res2 = res(seqpos2);
			res2_variant = res_variant(seqpos2);
			for ( int atom1 = 1, atom2e = nheavyatoms(res2,res2_variant);
			 atom1 <= atom1e; ++atom1 ) {
				for ( int atom2 = 1; atom2 <= atom2e; ++atom2 ) {
					if ( vec_dist2(full_coord(1,atom1,seqpos1),
					 full_coord(1,atom2,seqpos2) ) <= dis2_cutoff ) {
						docking_decoy_contact_map(seqpos1,seqpos2)=true;
						docking_decoy_contact_map(seqpos2,seqpos1)=true;
						docking_decoy_res_w_contact(seqpos1)=true;
						docking_decoy_res_w_contact(seqpos2)=true;
						++n_decoy_contact;
						goto L100;
					}
				} // atom2
			} // atom1
L100:;
		} // res2
	} // res1

	if ( n_decoy_contact == 0 ) {
		std::cout << "WARNING: ###calc_docking_frac_native_contact###" << std::endl;
		std::cout << "n_decoy_contact=0, no contacts in decoy!!!" << std::endl;
		return;
	}

// calculate interface-crossing hbonds in decoy
//bs calc all hbonds _across_ interface
	std::cout << "INTERFACE_HB: " << "  " << "DATM" << "  " << "DAA" << "   " <<
	 "DRES" << "  " << "AATM" << "  " << "AAA" << "   " << "ARES" << "  " <<
	 "  HB_E  " << "  " << "START_FILE" << std::endl;

//bs find donor/acceptor atoms in interface-crossing hbonds

	std::string resname1, resname2;
	std::string atomname1, atomname2;
	for ( int i = 1; i <= hbond_set.nhbonds(); ++i ) {
		dres = hbond_set.hbdon_res(i);
		ares = hbond_set.hbact_res(i);
		dhatm = hbond_set.hbdonh_atm(i);
		aatm = hbond_set.hbact_atm(i);
		hbE = hbond_set.hbenergies(i);

// crossing hbonds

		if ( docking_decoy_contact_map(dres,ares) ) {

			linetag = "INTERFACE_HB: ";
			goto L120;

		} else if ( docking_decoy_res_w_contact(dres) ||
		 docking_decoy_res_w_contact(ares) ) {

// hbonds within part 1 only (but still involving at least one interface residue)
			if ( dres >= part_begin(1) && dres <= part_end(1) &&
			 ares >= part_begin(1) && ares <= part_end(1) ) {

				linetag = "ADJACENT__HB: ";
				goto L120;

			} else if ( dres >= part_begin(2) && dres <= part_end(2) &&
			 ares >= part_begin(2) && ares <= part_end(2) ) {

				linetag = "ADJACENT__HB: ";
				goto L120;

			} else {
				goto L130;
			}
		} else {
			goto L130;
		}

//bs output info
L120:

		res1 = res(dres);
		res2 = res(ares);
		name_from_num(res1,resname1);
		name_from_num(res2,resname2);
		atom_name_from_atom_num( dhatm,res1,res_variant(dres),atomname1);
		atom_name_from_atom_num( aatm,res2,res_variant(ares),atomname2);

		if ( output_orig_pdb_resnum ) {
			dchain = res_chain(dres);
			achain = res_chain(ares);
			std::cout << linetag << "  " << atomname1 << "  " <<
			 resname1 << "  " << I( 4, pdb_res_num(dres) ) << dchain << "  " <<
			 atomname2 << "  " << resname2 << "  " << I( 4, pdb_res_num(ares) ) <<
			 achain << "  " << F( 8, 3, hbE ) << "  " <<
			 start_file << std::endl;
		} else {
			dchain = ' ';
			achain = ' ';
			std::cout << linetag << "  " << atomname1 << "  " <<
			 resname1 << "  " << I( 4, dres ) << dchain << "  " <<
			 atomname2 << "  " << resname2 << "  " << I( 4, ares ) <<
			 achain << "  " << F( 8, 3, hbE ) << "    " <<
			 I( 4, pdb_res_num(dres) ) << res_chain(dres) << "  " <<
			 I( 4, pdb_res_num(ares) ) << res_chain(ares) << "  " <<
			 start_file << std::endl;
		}

//bs not output anything
L130:;

	}

}
////////////////////////////////////////////////////////////////////////////////
/// @begin bias_this_rot_to_native()
///
/// @brief is this rotamer similiar to the native one or not?
///
/// @detailed decide whether the current rotamer has the same rotamer number as
///           the native rotamer in the docking unbound structure. If true, we
///           will assign it a lower dun energy to bias the native rotamer
///
/// @return  true or false
///
/// @global_read param.h, misc.h, docking_unbound.h
///
/// @global_write nothing
///
/// @remarks  only called in get_rotamer_probability() in fullatom_energy.cc
///
/// @references
///
/// @authors Chu Wang 2004/10/15
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
bias_this_rot_to_native(
	int aa,
	int const seqpos,
	FArray1DB_int const & rot //rot numbersunbound_res
	)
{
	using namespace param;
	using namespace docking_unbound;
	using namespace runlevel_ns;

	if ( !get_native_exists() ) return false;
	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return false;
	if ( get_ligand_flag() ) return false;
	if ( !get_unboundrot_flag() ) return false;

	if ( unbound_map_set && !use_mapped_rmsd ) {
		int jj = unbound_seq_align_map(seqpos);
		if ( jj == 0 || aa != unbound_res(jj) ) return false;
		for ( int i = 1; i <= MAX_CHI; ++i ) {
			if ( rot(i) != unbound_rotarray(i,jj) ) return false;
		}
	} else {
		for ( int i = 1; i <= MAX_CHI; ++i ) {
			if ( rot(i) != unbound_rotarray(i,seqpos)) return false;
		}
	}

	return true;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_dle_flag
///
/// @brief retrieve dle_flag
///
/// @detailed
/// retrieve the flag indicating whether there should be backbone flexibility
/// while carrying out docking rigid body moves
///
/// @return  bool, true if flexible backbone is used in docking
///
/// @global_read dle_flag in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop Sircar 02/22/2006
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_dle_flag()
{
	using namespace docking;

	return dle_flag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_dle_loops_flag
///
/// @brief retrieve dle_loops_flag
///
/// @detailed retrieve the flag indicating whether there should be loop building
/// while carrying out docking rigid body moves
///
/// @return  bool, true if loop building is used in docking
///
/// @global_read dle_loops_flag in docking.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop Sircar 06/15/2006
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_dle_loops_flag()
{
	using namespace docking;

	return dle_loops_flag;
}

/////////////////////////////////////////////////////////////////////////////////// @begin translate_subunits
///
/// @brief
///   Move docking partners apart by the specified number of angstroms,
///   in the direction separating their centers of mass
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void translate_subunits( float const translation_dist ) {

	// jk Note: the second partner moves, the first is fixed
	// jk Note: translation is along the direction connecting the partner centriods

	using namespace docking;
	using namespace misc;
	using namespace param;

	// jk Find the translation needed
	FArray1D_float translation_bond( 3 );
	float centroid_bond_mag=0.;
	for ( int k = 1; k <= 3; ++k ) {
		centroid_bond_mag += centroid_bond(k) * centroid_bond(k);
	}
	float const f = translation_dist / sqrt(centroid_bond_mag);
	for ( int k = 1; k <= 3; ++k ) {
		translation_bond(k) = f * centroid_bond(k);
	}

	// Move the second partner away from the first (ie. loop over the full_coord array)
	for ( int i = (domain_end(1)+1), max_atom = MAX_ATOM(); i <= domain_end(2); ++i ) {
		for ( int j = 1; j <= max_atom; ++j ) {
			for ( int k = 1; k <= 3; ++k ) {
				full_coord(k,j,i) -= translation_bond(k);
			}
		}
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
void
docking_initialize_silent_input_file()
{
	using namespace files_paths;
	using namespace misc;
	using namespace docking;


	if ( ! docking_silent_input ) return;

	// determine the input file name
	docking_silent_input_file  = code + protein_name + ".out";
	stringafteroption( "docking_silent_input_file", docking_silent_input_file,
										 docking_silent_input_file );
	if ( ! has_suffix( docking_silent_input_file, ".out" ) )
		docking_silent_input_file += ".out";

	// open the file
	utility::io::izstream file_stream;
	file_stream.clear();
	file_stream.open( docking_silent_input_file.c_str() );
	if ( ! file_stream ) {
		std::cout << "WARNING:: Unable to open docking_silent_input_file: "
							<< docking_silent_input_file << "\n";
		docking_silent_input_file = start_path + docking_silent_input_file ;
		std::cout << "Looking for " << docking_silent_input_file << "\n";
		file_stream.clear();
		file_stream.open( docking_silent_input_file.c_str() );
		if ( ! file_stream ) {
			std::cout << "ERROR:: Unable to open docking_silent_input_file: "
								<< docking_silent_input_file << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	std::string line, word;
	std::string::size_type tag_location;
	/* -- check the sequence -- */

	// read the first line
	getline( file_stream, line );
	std::istringstream line_stream( line );
	// break line into white-space-delimited words
	line_stream >> word;
	if ( word == "SEQUENCE:" ) {
		line_stream >> word;
		for ( int i = 0; i < total_residue; ++i ) {
			if ( residue1[i] != word[i] ) {
				std::cout << "ERROR:: sequence mismatch, residue: " << i+1 << "\n";
				std::cout << "residue1 " << residue1[i] << " vs silent_file "
									<< word[i] << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
	} else {
		std::cout << " ERROR:: docking silent_input_file format error\n"
							<< " the first line does not start with SEQUENCE:\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	/* -- check sequence done -- */

	/* -- check fa_silent_input begin -- */
	getline( file_stream, line );
	if ( line.substr(0,6) != "SCORE:" ) {
		std::cout << " ERROR:: docking silent_input_file format error\n"
							<< " the second line does not start with SCORE:\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	docking_fa_silent_input =
		( line.find(" bk_tot ") == std::string::npos ? false : true );

	if ( ! docking_fullatom_flag && docking_fa_silent_input ) {
		std::cout << "WARNING:: docking_fullatom_flag is false, "
							<< "but docking_fa_silent_input is true\n"
							<< "fullatom sidechain info in docking_silent_input_file will "
							<< "not be used!!!" << std::endl;
		docking_fa_silent_input = false;
	}

	// support -l <list> option to extract only a subset from silent_input file
	std::string listname;
	stringafteroption("l", "none", listname );
	if ( listname == "none" ) {
		std::cout << "docking_silent_input: use ALL entries in "
							<< docking_silent_input_file << std::endl;
		while( ! file_stream.eof() ) {
			getline( file_stream, line );
			if ( line.substr(0,6) != "SCORE:" ) continue; // not score line
			tag_location = line.rfind("S_");
			if ( tag_location == std::string::npos ) continue; // not find tag
			line = line.substr( tag_location, line.length()-tag_location+1 );
			std::istringstream line_stream ( line );
			line_stream >> word;
			docking_silent_input_name_map.push_back(word);
		}
	} else {
		std::cout << "docking_silent_input: use ONLY entries listed in "
							<< listname << std::endl;
		// open the file
		std::ifstream list_stream;
		list_stream.clear();
		list_stream.open( listname.c_str() );
		if ( ! list_stream ) {
			std::cout << "WARNING:: Unable to open docking_silent_input list: "
								<< listname << "\n";
			listname = start_path + listname;
			std::cout << "Looking for " << listname << "\n";
			list_stream.clear();
			list_stream.open( listname.c_str() );
			if ( ! list_stream ) {
				std::cout << "ERROR:: Unable to open docking_silent_input list: "
									<< listname << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
		while( ! list_stream.eof() ) {
			getline(list_stream,line);
			tag_location = line.rfind("S_");
			if ( tag_location == std::string::npos ) continue; // not find tag
			line = line.substr( tag_location, line.length()-tag_location+1 );
			std::istringstream line_stream ( line );
			line_stream >> word;
			docking_silent_input_name_map.push_back(word);
		}
	}
	int number_of_output;
	intafteroption("nstruct",default_nstruct,number_of_output);
	if ( int(docking_silent_input_name_map.size()) != number_of_output ) {
		std::cout << "ERROR:: error in docking_silent_input list: "
							<< "nstruct does not match the number of list entries "
							<< number_of_output << " vs. "
							<< docking_silent_input_name_map.size() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// in pose_docing make out.bonds and out.rot_template files if there is none
	if ( ! get_pose_docking_flag() ) return;
	bool const ideal_pose = { false };
	bool const coords_init = { true };
	pose_ns::Pose start_pose;
	silent_io::Silent_out silent_out;
	pose_from_misc(start_pose, docking_fa_silent_input, ideal_pose, coords_init);
	if ( docking_fa_silent_input )
		silent_out.store_rotamers( start_pose );
	silent_out_write_nonideal_geometry( start_pose, silent_out,
		docking_silent_input_file );
	return;
}
///////////////////////////////////////////////////////////////
void
docking_read_silent_input( bool & fail )
{
	using namespace param;
	using namespace misc;
	using namespace files_paths;
	using namespace docking;

	if ( ! docking::docking_silent_input ) return;

	set_fullatom_flag( get_docking_fullatom_flag() );
	bool found_it = false;

	utility::io::izstream file_stream;
	file_stream.clear();
	file_stream.open( docking_silent_input_file.c_str() );
	if ( ! file_stream ) {
		std::cout << " ERROR:: Unable to open docking_silent_input_file: "
							<< docking_silent_input_file << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string tag_name = docking_silent_input_get_current_tag( true );

	std::string line, word, number;
	std::string::size_type tag_location = 0;
	while ( ! file_stream.eof() ) {
		getline( file_stream, line );
		if ( line.substr(0,6) != "SCORE:" ) continue;
		tag_location = line.rfind( tag_name );
		if ( tag_location != std::string::npos ) { // find the tag_name
			found_it = true;
			break;
		}
	}

	fail = ! found_it;
	if ( fail ) {
		std::cout << "WARNING::docking_read_silent_input: Unable to find "
							<< tag_name << " in " << docking_silent_input_file << std::endl;
		return;
	}

	getline( file_stream, line );
	std::istringstream matrix_stream( line );
	std::string dummy_tag;
	// read in translation
  matrix_stream >> dummy_tag
								>> docking_silent_translation(1)
								>> docking_silent_translation(2)
								>> docking_silent_translation(3);
	// read in rotation by column
	for ( int i = 1; i <= 3; ++i ) {
		matrix_stream >> docking_silent_rotation(1,i)
									>> docking_silent_rotation(2,i)
									>> docking_silent_rotation(3,i);
	}

	if ( dummy_tag != "MATRIX:" ) {
		std::cout << " ERROR: no MATRIX line following SCORE line" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::cout << "docking_silent_translation\n" << docking_silent_translation
						<< std::endl;
	std::cout << "docking_silent_rotation\n" << docking_silent_rotation
						<< std::endl;

	docking_apply_silent_matrix( get_docking_fullatom_flag() );

	if ( docking_fa_silent_input ) {
		FArray2D_float silent_chi( MAX_CHI, MAX_RES()());
		FArray1D_bool ideal_sc( MAX_RES()(), false );
		int seqpos; char sc_tag;
		std::string decoy_tag;
		for ( int i = 1; i <= total_residue; ++i ) {
			getline( file_stream, line );
			std::istringstream line_stream( line );
			line_stream >> seqpos >> silent_chi(1,i) >> silent_chi(2,i)
									>> silent_chi(3,i) >> silent_chi(4,i) >> sc_tag >> decoy_tag;
			// safety check
			if (seqpos != i || decoy_tag != tag_name ) {
				std::cout << "WARNING::docking_slient_input_file error\n"
									<< "line: " << line << std::endl;
				fail = true;
				return;
			}
			// sc_tag
			if ( sc_tag == 'S' ) {
				ideal_sc(i) = false;
			} else if ( sc_tag == 'I' ) {
				ideal_sc(i) = true;
			} else {
				std::cout << "WARNING::docking_silent_input_file error\n"
									<< "seqpos/sc_tag: " << i << "/" << sc_tag << std::endl;
				fail = true;
				return;
			}
		}
		docking_refold_silent_chi( silent_chi, ideal_sc );
	}

	monte_carlo_reset();

}
///////////////////////////////////////////////////////////////////////
void
docking_calculate_silent_matrix()
{
	using namespace param;
	using namespace misc;
	using namespace docking;
	using namespace start;

	/* use the N-CA-C triplet of the last residue as the internal reference frame
		 with CA as the origin. Let SP and DP be the same atom P in partner_2 in the
		 starting and current pose respectively, and S0 and D0 be the origin of the
		 internal frames. Given the fact that a rigid-body move does not change the
		 coordinates of P in the internal frame of partner_2, the following equation
		 stands:

		 Mstart^T x (SP-S0) = Mdecoy^T x (DP-D0) = P's internal coodinates

		 ==> DP = D0 + Mdecoy x Mstart^T x (SP-S0)

		 docking_silent_translation = D0;
		 docking_silent_rotation = Mdecoy x Mstart^T

		 later on, we can reconstruct DP from SP/S0 based on the above formula */


	//FArray2D_double tmp(3,3), Mstart(3,3), Mdecoy(3,3);
	numeric::xyzMatrix_double Mstart, Mdecoy, tmp;
	get_ncac( Estart_position(1,1,total_residue), tmp );
	get_coordinate_system( tmp, Mstart );
	get_ncac( Eposition(1,1,total_residue), tmp );
	get_coordinate_system( tmp, Mdecoy );

	//for ( int i = 1; i <= 3; ++i ) {
	//	docking_silent_translation(i,1) = Eposition(i,2,total_residue); // CA of the last residue
	//}
	//matrix_transpose( Mstart, tmp );
	//matrix_multiply( Mdecoy, tmp,  docking_silent_rotation ); // Mdecoy x Mstart^T
	docking_silent_translation = &Eposition(1,2,total_residue);
	docking_silent_rotation = Mdecoy * Mstart.transposed();
}
/////////////////////////////////////////////////////////////
void
docking_apply_silent_matrix( bool const fullatom )
{
	using namespace misc;
	using namespace start;
	using namespace docking;

	if ( ! docking_silent_input ) return;

	FArray1D_double P(3), PT(3);
	for ( int i = 1; i <= 3; ++i ) {
		P(i) = static_cast<double>( Estart_position(i,2,total_residue) );
		PT(i) = docking_silent_translation(i);
	}
	FArray2D_double R(3,3);
	copy_to_FArray( docking_silent_rotation, R );
	for ( int i = part_begin(2), ie = part_end(2); i <= ie; ++i ) {
		// y = R x (x -P) + PT
		// in silent matrix definition ( see docking_calculate_silent_matrix )
		// P is S0 and PT is D0
		apply_rb_to_vector(centroid(1,i), start_centroid(1,i), R, P, PT );

		for ( int j = 1; j <= 5; ++j) {
			apply_rb_to_vector( Eposition(1,j,i), Estart_position(1,j,i), R, P, PT );
		}

		if ( fullatom ) {
			for ( int j = 1, je= aaproperties_pack::natoms(res(i),res_variant(i));
						j <= je; ++j ) {
				apply_rb_to_vector( full_coord(1,j,i), start_fullcoord(1,j,i), R, P,
					PT );
			}
		}
	}

	apply_rb_to_vector( part_centroid(1,2), start_part_centroid(1,2), R, P, PT );

	docking_derive_coord_frame();

}

/////////////////////////////////////////////////////////////
void
docking_write_silent_matrix( std::ostream & iunit )
{
	using namespace docking;

	// translation
	for ( int i = 1; i <= 3; ++i ) {
		iunit << F( 15, 8, docking_silent_translation(i) );
	}

	// rotation, output by column
	for ( int j = 1; j <= 3; ++j ) {
		for ( int i = 1; i <= 3 ; ++i ) {
			iunit << F( 15, 8, docking_silent_rotation(i,j) );
		}
	}
}
///////////////////////////////////////////////////////////////////////////////
void
docking_flag_ideal_sidechain(
	FArray1D_bool & ideal_sc,
	FArray2D_float & decoy_chi
)
{
	using namespace param;
	using namespace misc;
	using namespace start;


	assert( get_docking_fullatom_flag() );

	float const tol = { 0.01 };
	FArray1D_bool change_chi( MAX_CHI, true );
	float rms1( 0.0 ), rms2( 0.0 );
	FArray2D_double p1(3,MAX_ATOM()()), p2(3,MAX_ATOM()());

	for ( int i = 1; i <= total_residue; ++i ) {
		int const aa = res(i);
		int const aav = res_variant(i);
		int nheavyatoms = aaproperties_pack::nheavyatoms(aa,aav);
		FArray2D_float rotcoord( 3, MAX_ATOM()(), 0.0 );
		// Gly exception
		if ( nheavyatoms <= 4 ) {
			ideal_sc(i) = false;
			continue;
		}
		// ideal sc bond first
		get_rot_coord( full_coord, aa, aav, decoy_chi(1,i), rotcoord, i );
		for ( int j = 1; j <= nheavyatoms; ++j ) {
			for ( int k = 1; k <= 3; ++k ) {
				p1(k,j) = full_coord(k,j,i);
				p2(k,j) = rotcoord(k,j);
			}
		}
		fast_rms_atoms( p1, p2, nheavyatoms, rms1 );
		// start sc bond next
		for ( int j = 1; j <= MAX_ATOM()(); ++j ) {
			for ( int k = 1; k <= 3; ++k ) {
				rotcoord(k,j) = start_fullcoord(k,j,i);
			}
		}
		refold_new_chi_residue( rotcoord, aa, aav, decoy_chi(1,i), change_chi );
		for ( int j = 1; j <= nheavyatoms; ++j ) {
			for ( int k = 1; k <= 3; ++k ) {
				p2(k,j) = rotcoord(k,j);
			}
		}
		fast_rms_atoms( p1, p2, nheavyatoms, rms2 );

		if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "seqpos/aa/nheavyatms/rms2ideal/rms2start/: "
								<< i << "/" << param_aa::aa_name3(aa) << pdb::pdb_res_num(i)
								<< pdb::res_chain(i) << "/" << nheavyatoms << "/"
								<< rms1 << "/" << rms2 << std::endl;
		}
		// make the decision
		if ( rms1 > tol && rms2 > tol ) {
			std::cout << "ERROR: docking_flag_ideal_sidechain\n"
								<< "sc bonds for neither ideal nor start"
								<< "rms_to_ideal: " << rms1 << " rms_to_start: "
								<< rms2 << std::endl << "seqpos/aa/nheavyatms/"
								<< i << "/" << param_aa::aa_name3(aa) << pdb::pdb_res_num(i)
								<< pdb::res_chain(i) << "/" << nheavyatoms << std::endl
								<< "chi: " << decoy_chi(1,i) << " " << decoy_chi(2,i)
								<< " " << decoy_chi(3,i) << " " << decoy_chi(4,i) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		} else if ( rms1 <= rms2 ) {
			ideal_sc(i) = true;
		} else {
			ideal_sc(i) = false;
		}
	}
	return;
}
///////////////////////////////////////////////////////////////////////////////
void
docking_refold_silent_chi(
	FArray2D_float & decoy_chi,
	FArray1D_bool & ideal_sc
)
{
	using namespace param;
	using namespace misc;
	using namespace start;

	assert( get_docking_fullatom_flag() );

	FArray2D_float rotcoord( 3, MAX_ATOM()(), 0.0 );
	FArray1D_bool change_chi( MAX_CHI, true );

	for ( int i = 1; i <= total_residue; ++i ) {
		int const aa = res(i);
		int const aav = res_variant(i);
		// Gly exception
		if ( aa == param_aa::aa_gly ) continue;
		if ( ideal_sc(i) ) {
			get_rot_coord(full_coord, aa, aav, decoy_chi(1,i), rotcoord, i);
			for ( int lr = 0, lxyz = full_coord.index(1,1,i);
						lr < 3*MAX_ATOM()(); ++lr, ++lxyz ) {
				full_coord[ lxyz ] = rotcoord[ lr ];
			}
		} else {
			refold_new_chi_residue( full_coord(1,1,i), aa, aav, decoy_chi(1,i),
				change_chi );
		}
	}
	return;
}
///////////////////////////////////////////////////////////////////////////////
std::string
docking_silent_input_get_current_tag( bool setting )
{
	static bool map_init;
	static int decoy_num_int;

	if ( ! map_init ) {
		map_init = setting;
	}

	get_decoy_number( decoy_num_int );
	return ( map_init ? docking::docking_silent_input_name_map[decoy_num_int]
		: "S_dummy_tag" );

}
///////////////////////////////////////////////////////////////////////////////
void
calc_docking_interf_energy( bool repack_away )
{
	// WARNING: this function calls monte_carlo_reset twice.
	// so it is only safe to call in output_decoy() where the
	// structure is final and no changes will occur.
	using namespace param;
	using namespace docking;

  using namespace scores;

	if ( ! get_docking_flag() ) return;
	if ( get_ligand_flag() ) return;
	if ( docking::pose_docking_flag ) return; // pose uses another function

	const float away_dist = { 99.0 };
	const float zero_1 ( 0.0f );
	const FArray1D_float zero_3( 3, 0.0f );

	FArray1D_float T( 3 );
	float init_score, mid_score, final_score;

	score_set_evaluate_all_terms( true );
	init_score = scorefxn();
	monte_carlo_reset();
	//save complex interface before pulling it apart
	FArray1D_bool allow_repack( MAX_RES() );
	for ( int i=1; i<=misc::total_residue; ++i ) {
		allow_repack(i) = interface::int_res8(i);
	}

	//pull the complex apart
	create_docking_trans_vector( T, -away_dist, zero_1, zero_1 );
	apply_rigid_body( T, zero_3 );
	mid_score = scorefxn();
	monte_carlo_reset();

  fa_d_elec_score = fa_elec_score;

	if ( repack_away ) {
		select_rotamer_set( "large" );
		//overwrite int_res8 array by previously saved allow_repack to repack
		//the subset of residues when two proteins are apart
		for ( int i=1; i<=misc::total_residue; ++i ) {
			interface::int_res8(i) = allow_repack(i);
		}
		docking_repack( true );
		mid_score = scorefxn();

    fa_d_elec_score = fa_elec_score;

		// No monte_carlo_reset after repacking because we don't want to
		// remember anything from the away-repacked structure except the score.
		retrieve_best_pose(); //ensure using sidechains before repacking
		score_set_new_pose(); //ensure a fresh score calculation
	}

	//restore
	create_docking_trans_vector( T,  away_dist, zero_1, zero_1 );
	apply_rigid_body( T, zero_3 );
	final_score = scorefxn();
	monte_carlo_reset();

  fa_d_elec_score = fa_elec_score - fa_d_elec_score;

	docking_interf_energy = final_score - mid_score;

	if ( runlevel_ns::runlevel > runlevel_ns::standard )
		std::cout << "calc_docking_interf_energy() -- init/mid/final/delta: "
							<< init_score << " " << mid_score << " " << final_score << " "
							<< docking_interf_energy << std::endl;
}
///////////////////////////////////////////////////////////////////////////////
float
calc_docking_partner_rmsd(
	int const partner
)
{
	using namespace docking;
	// calculate the bacbkone ca rmsd for each docking partner after alignment
	assert( partner > 0 && partner <= max_monomers );

	int start = native_part_begin( partner );
	int end = native_part_end( partner );
	int size = 0;
	float rmsd = 0.0;

	FArray2D_double decoy_ca( 3, param::MAX_RES()() ), native_ca( 3, param::MAX_RES()() );

	if ( docking_unbound::use_mapped_rmsd ) {
		for ( int ii = start; ii <= end; ++ii ) {
			int const unbound_ii = docking_unbound::unbound_seq_align_map(ii);
			if ( unbound_ii == 0 ) continue;
			++size;
			for ( int k = 1; k <= 3; ++k ) {
				native_ca( k, size ) = native::native_ca( k, ii );
				decoy_ca( k, size ) = misc::Eposition( k, 2, unbound_ii );
			}
		}
	} else {
		for ( int ii = start; ii <= end; ++ii ) {
			++size;
			for ( int k = 1; k <= 3; ++k ) {
				native_ca( k, size ) = native::native_ca( k, ii );
				decoy_ca( k, size ) = misc::Eposition( k, 2, ii );
			}
		}
	}

	fast_rms_ca( native_ca, decoy_ca, size, rmsd );

	return rmsd;
}
