// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 18466 $
//  $Date: 2007-11-16 23:05:47 +0200 (Fri, 16 Nov 2007) $
//  $Author: yab $


// Rosetta Headers
#include "docking_minimize.h"
#include "aaproperties_pack.h"
#include "atom_is_backbone.h"
#include "cenlist.h"
#include "counters.h"
#include "count_pair.h"
#include "diagnostics_rosetta.h"
#include "dock_loops.h"
#include "dock_structure.h"
#include "docking.h"
#include "docking_minimize_ns.h"
#include "docking_movement.h"
#include "docking_score.h"
#include "docking_scoring.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "fragments.h"
#include "fullatom.h"
#include "fullatom_setup.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "interface.h"
#include "ligand.h"
#include "maps.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "namespace_fullatom_flag.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_pack.h"
#include "pdbstatistics_pack.h"
#include "prof.h"
#include "random_numbers.h"
#include "read_aaproperties.h"
#include "recover.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "status.h"
#include "template_pack.h"
#include "vdw.h"
#include "SurfaceMode.h"//KMa surface 2006-02

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>

//     these are functions to minimize a score function through rigid body
//     minimization of the position of the ligand in a docking simulation
//
//     the fullatom scoring function is used (minus the sasa calc)
//
//     jjg 1/21/2002



//Utility Headers
#include <utility/basic_sys_util.hh>

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_mcm_protocol
///
/// @brief main entrance to do monte_carlo minimization
///
/// @detailed
///      a total of 50 cyles of monte-carlo minimization will be
///      carried out if the minimized structure can pass the filter
///      after the first and fifth cycle. Then it is rigid-body minimized
///      to a stringent tolerance. This protocol is following the interface
///      repacking in dockFA.
///
/// @global_read  repacked structure in current array
///
/// @global_write minimized structure in current array
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_mcm_protocol()
{
// a short protocol for minimizing efficiently

	using namespace docking;
	using namespace runlevel_ns;

	const float rotmag = 0.05 /* ~2.86 degree*/;

	const float tranmag = 0.1; /* angstrom */
	// minimize only when current_score < best_score + threshold
	const float minimization_threshold = 15.0; /* score unit */
	// in docking we use absolute tolerance
	const float loose_func_tol = 1.0;
	const float strict_func_tol = 0.02;

	if ( runlevel > inform ) dMCM_open_file();
	docking_store_cenmode_rms();

//     -be sure the structure passes (non-score-related) filters
	if ( docking_mcm_filter(delta_before_mcm,score_before_mcm) ) {
//     -structure has been repacked in dock_structure()
		docking_rigidbdy_minimize_trial("dfpmin",score10d_min,loose_func_tol);

//     -be sure the structure passes filters
		if ( docking_mcm_filter(delta_after_one_min,score_after_one_min) ) {
      docking_score10d_min_after1 = mc_global_track::mc_score::score; // SJF For Ora's FunHunt

//     -minimize a few steps
			docking_monte_carlo_minimize(4,"dfpmin",score10d_min,tranmag,rotmag,
																	 minimization_threshold,loose_func_tol);

//     -be sure the structure passes filters
			if ( docking_mcm_filter(delta_after_five_mcm,score_after_five_mcm) ) {
        docking_score10d_min_after5 = mc_global_track::mc_score::score; // SJF For Ora's FunHunt
//     -minimize many steps
				if ( !benchmark ) {
					docking_monte_carlo_minimize(45,"dfpmin",score10d_min,tranmag,rotmag,
																			 minimization_threshold,loose_func_tol);
				}

//     -minimize to a strict tolerance
				docking_rigidbdy_minimize_trial("dfpmin",score10d_min,strict_func_tol);

			}
		}
	}

	if ( runlevel > inform ) dMCM_close_file();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_mcm_filter
///
/// @brief apply docking score filters in monte-carlo minimization
///
/// @detailed
///     the idea here is to stop minimizing structures that are going to be bad
///     also intermediate scores are saved into corresponding variables
///
/// @param[in]   margin - in - margin to pass the score filter
///                       the larger margin, the less stringent criteria
///
/// @param[out]   score_storage - out - pass the current score into common variables
///
/// @return  bool, true if it can pass the filter
///
/// @global_read scores for the structure in the low array
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
docking_mcm_filter(
	float margin, // points to allow for passing scorefilter, bkrepfilter
	float & score_storage // variable to store the current score
)
{
	using namespace runlevel_ns;
	using namespace scores;

	bool docking_mcm_filter; // Return value

//     local
	std::string tag;
	bool pass = true;

//     This should be a score that calculates contact, fab, vdw, site_constr
	/* in pose mode, monte_carlo is an object now and recover_LOW won't work,
		 user needs to ensure all the score components are updated - chutmp */
	if ( ! pose_flag() ) {
		score_set_evaluate_all_terms(true);
		recover_LOW(score10d);
		score_set_evaluate_all_terms(false);
	}
	score_storage = mc_global_track::mc_score::score;

	mc_global_track::mc_score::score -= margin; // margin to pass scorefilter
	fa_rep_score -= margin; // margin to pass bkrep filter
	docking_apply_filters( pass, tag );
	mc_global_track::mc_score::score += margin;
	fa_rep_score += margin;

	if ( !pass && runlevel > standard ) std::cout <<
	 "rejected before extensive refinement, score = " << mc_global_track::mc_score::score <<
	 " tag=" << tag << std::endl;

	docking_mcm_filter = pass;
	return docking_mcm_filter;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_rigidbdy_minimize_trial
///
/// @brief main entrance for normal rigid-body minimization
///
/// @detailed retrieve the structure in the low array and do the normal minimization
///      by calling docking_minimize_position to optimize the score according
///      the scorefxn chosen.
///
/// @param[in]   type - in - which type of mimization, i.e. dfpmin, linmin...
/// @param[in]   scoring_fn - in - which scorefxn to minimize, score10d_min...
/// @param[in]   func_tol - in - the tolerance for the minimization
///
/// @global_read  structure in the low array
///
/// @global_write the minimized structure into the best and low array
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_rigidbdy_minimize_trial(
	std::string const & type, // type of minimization to perform
	Scoring_Function scoring_fn, // which score to use
	float func_tol // tolerance on the minimization function
)
{
//     translate/rotate one partner in a docking complex
//     this function minimizes the current position, and automatically
//     accepts that position, since the score will be lower
//     03/01 JJG

	using namespace runlevel_ns;

	if ( runlevel > standard ) std::cout << "==docking minimization--" << type <<
	 "==" << std::endl;
	recover_LOW(scoring_fn);
	if ( runlevel > inform ) dMCM_output(0,0);
	if ( runlevel > inform ) dMCM_output(0,2);

	docking_minimize_position( type, scoring_fn, func_tol );
	mc_global_track::mc_score::score = scoring_fn();

	monte_carlo(1);
	if ( runlevel > inform ) dMCM_output(0,3);
	 //ora: call AFTER mc, so best and low arrays are updated

}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_monte_carlo_minimize
///
/// @brief main functioning function for docking monte-carlo minimization
///
/// @detailed
///    retrieve the structure in the low array
///    for each cycle {
///       generate a random move for the current structure
///       refine the interface side-chains by either full repacking
///              (every 8 cyles) or rotamer_trial
///       if it is not too bad, do a rigid-body minimization on this structure
///       score it and decide whether to accept it by monte-carlo criterion
///    }
///
/// @param[in]   cycles - in - how many cylce of monte-carlo minimizations to be done
/// @param[in]   type - in - which type of rigid-body minimization, dfpmin...
/// @param[in]   scoring_fn - in - which scorefxn to be minimized...
/// @param[in]   trans_magnitude - in - translation magnitude for MCM move
/// @param[in]   rot_magnitude - in - rotation magnitude for MCM move
/// @param[in]   minimization_threshold - in - energy difference (score - best_score) needed for minimization
/// @param[in]   func_tol - in - the tolerance for rigid-body minimization
///
/// @global_read structure in low array
///
/// @global_write minimized structure in best and low array if accepted
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_monte_carlo_minimize(
	int cycles, // number of monte carlo minimizations
	std::string const & type, // type of minimization to perform
	Scoring_Function scoring_fn, // which score to use
	float trans_magnitude, // size of translation moves
	float rot_magnitude, // size of rotation moves
	float minimization_threshold, // energy difference (score - best_score) needed for minimization
	float func_tol // tolerance on the minimization function
)
{
//     this function coordinates a monte-carlo search in fullatom mode,
//     minimizing each position
//     05/01 JJG

	using namespace docking;
	using namespace runlevel_ns;
	using namespace mc_global_track::mc_score; // yab: misc removal

//   local

	if ( runlevel > standard ) {
		std::cout << "==============================" << std::endl;
		std::cout << "docking Monte Carlo Minimization" << std::endl;
		std::cout << "           cycles:" << SS( cycles ) << std::endl;
		std::cout << "translation moves:" << SS( trans_magnitude ) << " A" << std::endl;
		std::cout << "   rotation moves:" << SS( rot_magnitude ) << " rads" << std::endl;
		std::cout << "     minimization: " << type << std::endl;
		std::cout << "   func tolerance:" << SS( func_tol ) << std::endl;
		std::cout << "==============================" << std::endl;
	}

	//recover_LOW(scoring_fn); glb - changed to new_score_function
	new_score_function(scoring_fn);

//      if ( runlevel > inform ) dMCM_output(-1,0);

	for ( int i = 1; i <= cycles; ++i ) {


		docking_MCM_move(trans_magnitude,rot_magnitude);

		if ( runlevel > inform ) score = scoring_fn(); ////
		if ( runlevel > inform ) dMCM_output(i,1);
		docking_MCM_pack_side_chains(i,scoring_fn);

		if ( score-best_score < minimization_threshold ) {
			docking_minimize_position( type, scoring_fn, func_tol );
			score = scoring_fn();
		}

		docking_loop_minimize(i,scoring_fn);

		save_status_info("dMCM",part_begin(2),0);
		monte_carlo(i);
		increment_trial_counters(i);
		if ( runlevel > inform ) dMCM_output(i,3);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dMCM_open_file
///
/// @brief open a file for outputing monte carlo minimization log
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dMCM_open_file()
{
//     stuff for outputting progress

	using namespace files_paths;

	user_x.clear();
	user_x.open( "dMCM.dat" );
	if ( !user_x ) {
		std::cout << "Open failed for file: " << "dMCM.dat" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	user_x << A( 9, "cycle" ) << ' ' << A( 9, "mc" ) << ' ' <<
	 A( 9, "score" ) << ' ' << A( 9, "best" ) << ' ' << A( 9, "low" ) << ' ' <<
	 A( 9, "rms" ) << ' ' << A( 9, "best_rms" ) << ' ' <<
	 A( 9, "low_rms" ) << ' ' << A( 9, "func_calls" ) << ' ' <<
	 A( 9, "description" ) << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dMCM_close_file
///
/// @brief close file for outputing monte carlo minimization log
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dMCM_close_file()
{
	using namespace files_paths;

	user_x.close();
	user_x.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dMCM_output
///
/// @brief output the log for each monte carlo cycle
///
/// @detailed useful for tracking down the trajectory of the minimization
///
/// @param[in]   icycle - in - number of the current cycle
/// @param[in]   state - in - whether the structure has been repacked or minimized
///            in the current cyle, not = 0, repacked = 1, minimized = 2
///
/// @global_read
///
/// @global_write rmsd of the current structure will updated
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dMCM_output(
	int icycle, // cycle number
	int state // whether the structure has been repacked or minimized
)
{
//     stuff for outputting progress

	using namespace docking;
	using namespace files_paths;
	using namespace mc_global_track::mc_score; // yab: misc removal
	using namespace mc_global_track::diagnose; // yab: misc removal
	using namespace runlevel_ns;

//ora      // not = 0, moved = 1, repacked = 2, minimized = 3, rotamer_trial = 5
// local
	int monte_carlo_accept;

	std::string tag;

	update_scorefile_info(); // calc rms
	float fcycle = float(icycle) + float(state-3)/3.0;
	if ( icycle == -1 ) {
		tag = "start";
	} else if ( state == 0 ) {
		user_x << std::endl;
		return;
	} else if ( state == 1 ) {
		tag = "moved";
	} else if ( state == 2 ) {
		tag = "repacked";
	} else if ( state == 3 ) {
		tag = "minimized";
	} else if ( state == 4 ) {
		tag = "montecarloed";
	} else if ( state == 5 ) {
		tag = "rot_trial";
		fcycle = float(icycle) + float(2-3)/3.0; // same fcycle as for repacked
	} else if ( state == 6 ) {
		tag = "loop min";
	} else {
		tag = "unknown-dMCM";
	}

	monte_carlo_accept=get_monte_carlo_accept();
	user_x <<
		F( 9, 2, fcycle ) << ' ' <<
		I( 9, monte_carlo_accept ) << ' ' <<
		F( 9, 3, score ) << ' ' <<
		F( 9, 3, best_score ) << ' ' <<
		F( 9, 3, low_score ) << ' ' <<
		F( 9, 3, rms_err ) << ' ' <<
		F( 9, 3, best_rms ) << ' ' <<
		F( 9, 3, low_rms ) << ' ' <<
		I( 9, get_func_evals() ) << ' ' <<
		tag << std::endl;
	if ( runlevel > chat ) std::cout <<
		F( 9, 2, fcycle ) << ' ' <<
		I( 9, monte_carlo_accept ) << ' ' <<
		F( 9, 3, score ) << ' ' <<
		F( 9, 3, best_score ) << ' ' <<
		F( 9, 3, low_score ) << ' ' <<
		F( 9, 3, rms_err ) << ' ' <<
		F( 9, 3, best_rms ) << ' ' <<
		F( 9, 3, low_rms ) << ' ' <<
		I( 9, get_func_evals() ) << ' ' <<
		tag << std::endl;

	save_status_info( tag, 0, 0 );
	output_status_file(0,0.0);

	if ( runlevel > chat ) make_named_numbered_pdb( tag, icycle, true );

//      compare_fullatom_energy();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_MCM_move
///
/// @brief move a structure in monte-carlo minimization
///
/// @detailed
///     retrieve the last accepted structure(best array), generate the
///     translation and rotation matrix with the magnitude specified and
///     apply it. Notify rosetta that a move has been made.
///
/// @param[in]   Tmag - in - translation magnitude
/// @param[in]   Rmag - in - rotation magnitude
/// @param  icycle - [in/out]? - not used any more
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_MCM_move(
	float Tmag, // in Angstroms
	float Rmag // in radians
)
{
	using namespace misc;

//     local
	FArray1D_float delta_TR( 6 ); // the translation/rotation perturbation


	retrieve_best_pose(); // necessary for small move through pack/unpack

	docking_pack_best_position(delta_TR);
	docking_generate_MCM_move(delta_TR,Tmag,Rmag);
	docking_unpack_position(delta_TR);
//     need to rescore everything, since the best rotamers have been restored
//      CAR:: if the _best_ rotamers have been restored, then there is
// no need to rescore anything!! If new rotamers are on the backbone
// use maps_set_new_rotamer. Note that the packer and rotamer_trials
// handle this, so you likely don't need to do it.
// score_set_new_pose() is surely overkill here, but guarantees correctness.

	score_set_new_pose();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_generate_MCM_move
///
/// @brief use random number generator to generate a trans-rotation matrix
///
/// @detailed
///
/// @param[out]   delta_TR - out - matrix generated
/// @param[in]   Tmag - in - magnitude for translation
/// @param[in]   Rmag - in - magnitude for rotation
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_generate_MCM_move(
	FArray1Da_float delta_TR,
	float Tmag, // in Angstroms
	float Rmag // in radians
)
{
	delta_TR.dimension( 6 );

//     generate random translation and rotation

	delta_TR(1) = gaussian() * Tmag;
	delta_TR(2) = gaussian() * Tmag;
	delta_TR(3) = gaussian() * Tmag;
	delta_TR(4) = gaussian() * Rmag;
	delta_TR(5) = gaussian() * Rmag;
	delta_TR(6) = gaussian() * Rmag;

}




////////////////////////////////////////////////////////////////////////////////
/// @begin docking_MCM_pack_side_chains
///
/// @brief pack side chains during the docking MC-minimization loop
///
/// @detailed includes rotamer trials or full repack (every 8 cycles) or
///           rotamer-trial minimization according to run flags
///
/// @param[in]   i - cycle number
///
/// @global_read protein pose
///
/// @global_write new side chain positions and appropriate scores
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03, Jeff Gray 7/12/05 separated routine
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_MCM_pack_side_chains(
  int i, // cycle number in MCM loop
	Scoring_Function scoring_fn // which score to use
)
{
	using namespace docking;
	using namespace runlevel_ns;

		bool save_pack_interface = get_docking_interface_pack();

		// try rotamers (T) or do full pack (F)?  A: pack every 8 cycles
		bool mcm_try_rotamers = ( mod(i,8) != 0 );

		if ( mcm_try_rotamers ) {
			set_docking_interface_pack(false);
			score_set_try_rotamers(true);
			mc_global_track::mc_score::score = scoring_fn();
			score_set_try_rotamers(false);
			if ( runlevel > inform ) dMCM_output(i,5); //ora: 5 = rotamer trial
		} else { // full interface repack
			update_cendist(misc::total_residue,misc::centroid);
			set_docking_interface_pack(true);
			docking_detect_interf_res(); //chu: update interface
			PROF_START( prof::DOCK_REPACK );
			docking_repack(true);
			PROF_STOP( prof::DOCK_REPACK );
			if (dock_rtmin) {
				score_set_try_rotamers(true);
				score_set_minimize_rot(true);
				mc_global_track::mc_score::score = scoring_fn();
				score_set_minimize_rot(false);
				score_set_try_rotamers(false);
			} else {
				score_enable_rotamer_trials(false);
				mc_global_track::mc_score::score = scoring_fn();
				score_enable_rotamer_trials(true);
			}
			if ( runlevel > inform ) dMCM_output(i,2); //ora: 2 = repack
		}

		set_docking_interface_pack(save_pack_interface);
		return;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin docking_minimize_position
///
/// @brief main functioning function for docking_rigid_body_minimization
///
/// @detailed
///    the relative orientation of two docking partners is represented by
///    TR(6). Given a TR, a docking conformation can be defined and a score
///    can be evaluated by the scorefxn. Here, we are minimizing the scorefxn
///    by finding the best TR(6).
///
/// @param[in]   type - in - minimization type, dfpmin...
/// @param[in]   scoring_fn - in - scorefxn to be minimized, score10d_min(no gsolt)...
/// @param[in]   func_tol - in - a tolerance for minimization.
///
/// @global_read
///
/// @global_write current array should have the minimized structure
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_minimize_position(
	std::string const & type, // type of minimization to perform:
	Scoring_Function scoring_fn, // which score to use
	float func_tol // tolerance for the minimize functions
)
{
//     this function minimizes the current position
//     it will hijack the best array, to use best as a reference point
//     for the minimization transforms

	using namespace docking;
	using namespace runlevel_ns;

//   local
	int const nfree = { 6 }; // degrees of freedom in the minimization
	FArray1D_float TR( nfree ); // the translation/rotation independent
	// variables to minimize on
	FArray1D_float dE_dTR( nfree );
	 // derivatives of energy wrt transl'n/rotat'n
	float start_func = 0.0,end_func;

//   variables passed to minimizer
	float func_return_val; // return value of the minimizing function
	bool lstat; // indicates success status of minimize function
	int iter; // number of iterations, returned by minimizer

	if ( !get_fullatom_flag() )
	 error_stop("must be in fullatom mode for docking minimization");

//      make_named_pdb("dmin.start.pdb",true)
	lstat = true;

//     setup minimize functions for docking
	minimize_set_func(4); // set minimize functions in docking mode
	brent_set_abs_tolerance(func_tol);

//     create the list of residues at the interface for quicker dfunc eval

//      docking_detect_interf_res()  // no need: func called always before dfunc

//      docking_make_interface_list()
//      docking_make_interf_close_list()

	docking_hijack_best_pose();
	docking_pack_best_position(TR);

//     set_no_eval // this would save time by skipping a scoring step
	score_enable_rotamer_trials(false);
	score_set_use_subset_energy(true);
	mc_global_track::mc_score::score = score_no_sasa(scoring_fn); // be sure the weights are turned on
	// and the sasa calc is off
//     set_yes_eval

	if ( runlevel > standard ) start_func = func(lstat,TR);

	if ( type == "linmin" ) {
		dfunc(TR,dE_dTR,nfree); // calculate derivatives
		linmin(TR,dE_dTR,nfree,func_return_val,lstat); // linear minim.
	} else if ( type == "dfpmin" ) { // quasi-Newton minimization
		dfpmin_atol(TR,nfree,func_tol,iter,func_return_val,lstat);
	} else if ( type == "frpmin" ) { // conjugate gradient
		frprmn_atol(TR,nfree,func_tol,iter,func_return_val,lstat);
	} else {
		std::cout << "minimization type not defined in function minimize" << std::endl;
		std::cout << "type: " << type << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( lstat ) { // minimization succeeded

		docking_unpack_position(TR);

		if ( runlevel > standard ) {
			end_func = func_return_val;
			std::cout << "dock_min: bef/aft/del" <<
			 SS( start_func ) << SS( end_func ) << SS( end_func-start_func ) <<
			 " iters " << iter << " tol" << SS( func_tol ) <<
			 " func_evals " << get_func_evals() << std::endl;
		}

		mc_global_track::mc_score::score = score_no_sasa(scoring_fn);

		save_status_info("rigid_body_min",part_begin(2),0);

	} else {
		std::cout << "gfrag is false!!??" << std::endl; // don't know what if else???
		std::cout << "TR: ";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << SS( TR(i) );
		} std::cout << std::endl;
		std::cout << "dE_dTR: ";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << SS( dE_dTR(i) );
		} std::cout << std::endl;
		std::cout << "func: " << SS( func_return_val ) << std::endl;
		std::cout << "docking_minimize_position failed, "
							<< "restore to the state before minimization" << std::endl;
		TR = 0.0;
		docking_unpack_position(TR);
		//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	docking_hijack_best_pose_rlease();
	score_set_use_subset_energy(false);
	score_enable_rotamer_trials(true);
//      make_named_pdb("dmin.final.pdb",true)

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_pack_best_position
///
/// @brief  initialize TR(6) and calculate center of rotation
///
/// @detailed  this function basically stores the center of rotation
///       check that THE BEST_ ARRAYS MATCH THE CURRENT POSITION
///
/// @param  TR - [in/out]? - initialized translation and rotation matrix
///
/// @global_read current and best array
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_pack_best_position( FArray1Da_float TR )
{
	using namespace docking_min;
	using namespace misc;

	TR.dimension( 6 );

//     this function basically stores the center of rotation
//     THE BEST_ ARRAYS SHOULD MATCH THE CURRENT POSITION

//     test
	if ( Eposition(3,1,total_residue) != Ebest_position(3,1,total_residue) )
	 error_stop("position does not match best in docking_pack");

//     calculate the interface center to use as the center_of_rotation
	calculate_interface_centroid(center_of_rotation);

//     set TR to zeros
	TR = 0.0;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_unpack_position
///
/// @brief apply trans/rotation matrix to generate a new structure
///
/// @detailed
///jjg     this function takes the translation/rotation vector TR, unpacks it
///jjg     into the translation vector and rotation matrix that it
///jjg     represents, and uses the center_of_rotation common variable to
///jjg     apply the transformation to the docking ligand in misc.h
///
/// @param[in]   TR - in - trans/rotation matrix to be applied
///
/// @global_read best array in misc.h
///
/// @global_write current array in misc.h
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_unpack_position( FArray1Da_float TR )
{
	using namespace docking_min;
	using namespace runlevel_ns;

	TR.dimension( 6 );

//     local
	FArray1D_float translation( 3 );
	FArray2D_float R( 3, 3 );

	if ( runlevel >= yap ) std::cout << "unpack TR: " <<
	 F( 8, 4, TR(1) ) << ' ' << F( 8, 4, TR(2) ) << ' ' <<
	 F( 8, 4, TR(3) ) << ' ' << F( 8, 4, TR(4) ) << ' ' <<
	 F( 8, 4, TR(5) ) << ' ' << F( 8, 4, TR(6) ) << std::endl;

//     unload TR vector
	translation(1) = TR(1);
	translation(2) = TR(2);
	translation(3) = TR(3);
	create_three_rotations_matrix(R,TR(4),TR(5),TR(6));

//      matrix_name_write( std::cout,R,"R");
//      make_named_pdb("unpackA",true);

	apply_rigid_body_matrix(translation,R,center_of_rotation);

//      make_named_pdb("unpackB",true);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin func_dvdw
///
/// @brief func for docking_rigid_body minimization
///
/// @detailed
///     called by minimization recipes as the fuction to be minimized in docking.
///     apply TR matrix to current and score the new structure, return the full
///     score as the function value.
///
/// @param[in]   TR - in - trans/rotation matrix, it is the input parameter for func
/// @param[out]   gfrag - out - status of calling func, true if no error occurs
///
/// @return  float, fullatom score for the current structure
///
/// @global_read  best array
///
/// @global_write current array
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
func_dvdw(
	FArray1Da_float TR,
	bool & gfrag // indicates success of this function
)
{
	using namespace runlevel_ns;

	TR.dimension( 6 );

	float func_dvdw; // Return value

//     this function evaluates the fullatom energy of a docking position
//     with translation/rotation as encoded in the TR vector

//     output -> the function value = the score of the current position

	for ( int i = 1; i <= 6; ++i ) {
		if ( TR(i) != TR(i) ) { // NaN detected (works for IEEE floating point)
			func_dvdw = 0.;
			gfrag = false;
			std::cout << "func_dvdw ERROR" << std::endl;
			std::cout << "variable_type: " << i << std::endl;
			std::cout << "value:      " << TR(i) << std::endl;
			return func_dvdw;
		}
	}
	gfrag = true;

	docking_unpack_position(TR);

	func_dvdw = scorefxn();

	gfrag = ( interface::int_res_list(1).size() > 0 );
	if ( !gfrag ) std::cout << "the complex is out of contact " << std::endl;

	if ( runlevel >= verbose ) {
		score_output( std::cout, "func_dvdw", true );
		std::cout << std::endl;
	} else if ( runlevel >= yap ) {
		std::cout << "func_dvdw: " << func_dvdw << std::endl;
	}

	return func_dvdw;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin dfunc_dvdw
///
/// @brief dfunc for docking rigid body minimization
///
/// @detailed
///       this is the main entrance to cacluate the function gradients
///       , called by minimize recipes.
///
/// @param[in]   TR - in - function parameters, trans/rotation matrix
/// @param[out]   dE_dTR - out - function gradients with respect to these parameters
/// @param[in]   nfree - in - number of parameters
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dfunc_dvdw(
	FArray1Da_float TR,
	FArray1Da_float dE_dTR,
	int nfree
)
{
	TR.dimension( nfree );
	dE_dTR.dimension( nfree );

//mj jump out if ligand docking
	if ( get_ligand_flag() ) {//kwk docking loop for multiple ligands
		derivative_ligand_pos(TR,dE_dTR,nfree,(*ligand::ligand_one));
	} else {
		derivative_protein_pos(TR,dE_dTR,nfree);
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin derivative_protein_pos
///
/// @brief calculate function derivatives for TR
///
/// @detailed
///     this function evaluates the derivatives of the van der Waals
///     energy relative to the variables in the translation rotation
///     vector TR. Interface interaction only. Check numerically if necessary.
///
/// @param[in]   TR - in - trans/rotation vector
/// @param[out]   dE_dTR - out - derivatives relative to TR
/// @param[in]   nfree - in - number of freedom in TR
///
/// @global_read current array, interface array, neighborlist
///
/// @global_write
///
/// @remarks
/// in addition, the derivatives of the solvation energy, pair energy,
/// electrostatics and hbond energy are evaluated.
/// see documentation in specific functions.
///
/// cheap elec replaced by coulomb energy with dielec=r
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
derivative_protein_pos(
	FArray1Da_float TR,
	FArray1Da_float dE_dTR,
	int nfree
)
{
	using namespace aaproperties_pack;
	using namespace interface;
	using namespace misc;
	using namespace param;
	using namespace param_pack;
	using namespace runlevel_ns;
	using namespace scorefxns;

	TR.dimension( nfree );
	dE_dTR.dimension( nfree );

//     local
	int ires,jres; // residue numbers
	int irtype,jrtype; // residue type numbers
	int irvar,jrvar; // amino acid variants
	int iatype,jatype; // atom type numbers

	float dE_dr; // derivative of energy with respect to atom-atom distance
	FArray1D_float dr_dx( 3 );
	 // derivative of atom-atom distance along each coord. axis
	FArray1D_float q( 3 ); // position of j-atom relative to center of rotation

	float fa_pair_factor;
	float cp_weight;

	FArray2D_bool jneighborlist( MAX_RES()(), MAX_RES()() );

	docking_unpack_position(TR);

// note that neighborlist was updated in func (by calling fullatom_energy part)
	docking_get_brians_neighborlist(jneighborlist);

	// initialize the derivative vector
	for ( int i = 1; i <= 6; ++i ) {
		dE_dTR(i) = 0.0;
	}

//ora: this loop evaluates the derivatives of the LJ and solvation energies
	// loop over residues in the interface
	for ( int i = 1, ie = int_res_list8(1).size(); i <= ie; ++i ) { // loop over residues in receptor
		ires = int_res_list8(1)[i];
//        for ( ires = part1_begin; ires <= part1_end; ++ires ) {
		irtype = res(ires);
		irvar = res_variant(ires);
		for ( int j = 1, je = int_res_list8(2).size(); j <= je; ++j ) { // loop over residues in ligand
			jres = int_res_list8(2)[j];
//        for ( jres = part2_begin; jres <= part2_end; ++jres ) {
			jrtype = res(jres);
			jrvar = res_variant(jres);
			if ( jneighborlist(ires,jres) ) {
//            if ( cendist(ires,jres) < 144 ) {
				// loop over atoms in residues
				for ( int iat = 1, iate = natoms(irtype,irvar); iat <= iate; ++iat ) {
					iatype = fullatom_type(iat,irtype,irvar);
					for ( int jat = 1, lf = fullatom_type.index(jat,jrtype,jrvar),
					 jate = natoms(jrtype,jrvar); jat <= jate; ++jat, ++lf ) {
						if ( count_pair(ires,iat,irtype,irvar,jres,jat,jrtype,jrvar,true,
						 cp_weight) ) {
							jatype = fullatom_type[ lf ]; // fullatom_type(jat,jrtype,jrvar);
							// evaluate derivatives
							eval_dE_dr_docking(ires,iat,jres,jat,iatype,jatype,dE_dr,dr_dx,q,
							 cp_weight);
							// sum derivatives
							accumulate_dE_dTR(dE_dTR,dE_dr,dr_dx,q);
						}
					}
				}
			}
		}
	}
//ora end of evaluation of derivatives of LJ and solvation energies

//ora evaluation of derivatives of pair energy
	fa_pair_factor = pack_wts.Wpair()*fa_pair_weight;
	if ( fa_pair_factor != 0. ) {
		get_total_pair_deriv_docking(dE_dTR,jneighborlist);
	}

//ora  evaluation of derivative of electrostatic term (dielectric = r)
// updated to new electrostatic function
	if ( docking_warshel_elec_weight != 0.0 ) {
		get_total_elec_deriv_docking(dE_dTR);
	}

//ora evaluation of derivative of hbond energy
//ora/jss (note that in docking_mode, hb_energy_deriv should be passed an optional parameter to
//          accumulate derivatives in array deriv. Then deriv is weighted and summed up
//          according to allow_hbond, and added to dE_dTR
	if ( pack_wts.hbond_wts_nonzero() ) {
		get_total_hb_deriv_docking(dE_dTR,nfree,jneighborlist);
	}


	if ( runlevel > verbose ) {
//     check derivatives numerically: moved to separate function
		check_derivatives_numerically(TR,dE_dTR,nfree);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_get_brians_neighborlist
///
/// @brief retrieve neighborlist information in packer
///
/// @detailed
///       the reason is because template_pack.h and misc.h can not stay together
///
/// @param[out]   nl_out - out - identical to the packer neighborlist
///
/// @global_read packer neighborlist in template_pack.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_get_brians_neighborlist( FArray2Da_bool nl_out )
{
	using namespace docking;
	using namespace param;
	using namespace template_pack;

	nl_out.dimension( MAX_RES(), MAX_RES() );

	for ( int i = part_begin(1), iend = part_end(1), jend = part_end(2); i <= iend; ++i ) {
		for ( int j = part_begin(2); j <= jend; ++j ) {
			nl_out(i,j) = neighborlist(i,j);
		}
	}
}

//------------------------------------------------------------------------------
void
docking_get_brians_neighbors( FArray1Da_int n_out )
{
	using namespace param;
	using namespace misc;
	using namespace template_pack;

	n_out.dimension( MAX_RES() );

	for ( int i = 1; i <= total_residue; ++i ) {
		n_out(i) = neighbors(i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_dE_dr_docking
///
/// @brief evaluate vdw (and solvation) derivative relative to distance between two atoms
///
/// @detailed
///     this function differs from standard-Rosetta's because in docking,
///     the movement is rigid-body, not rotation of phi angles.  Also,
///     there is no atom-checking here; the calling function should send
///     all necessary atom pairs across the interface.  Finally, we always
///     use the full-atom mode.
///     JJG 2/5/2
///
/// @param[in]   ires - in - residue i
/// @param[in]   iat - in - atom i
/// @param[in]   jres - in - residue j
/// @param[in]   jat - in - atom j
/// @param[in]   itype - in - fullatom type of atom i
/// @param[in]   jtype - in - fullatom type of atom j
/// @param  dE_dr - [in/out]? - derivative of energy relative to atom-atom distance
/// @param  dr_dx - [in/out]? - deriv of atom-atom distance along each coord. axis
/// @param  q - [in/out]? - position of j-atom relative to center of rotation
///
/// @global_read lj derivative tables
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_dE_dr_docking(
	int & ires,
	int & iat,
	int & jres,
	int & jat,
	int & itype,
	int & jtype,
	float & dE_dr, // derivative of energy with respect to atom-atom distance
	FArray1Da_float dr_dx,
	 // derivative of atom-atom distance along each coord. axis
	FArray1Da_float q, // position of j-atom relative to center of rotation
	float const cp_weight
)
{
	using namespace docking_min;
	using namespace misc;
	using namespace param_pack;
	using namespace scorefxns;

	dr_dx.dimension( 3 );
	q.dimension( 3 );

//     local
	FArray1D_float ixyz( 3 );
	FArray1D_float jxyz( 3 );
	FArray1D_float dxyz( 3 ); // vector between atoms, pointing from i --> j
	float dsq; // squared distance between atoms
	float one_r; // 1/distance

	float dlj,datr,drep,dsol; // energy derivative terms

//      float ljE,sE1,sE2,atrE,repE;     // testing

	ixyz(1) = full_coord(1,iat,ires); // get position of atom i
	ixyz(2) = full_coord(2,iat,ires);
	ixyz(3) = full_coord(3,iat,ires);

	jxyz(1) = full_coord(1,jat,jres); // get position of atom j
	jxyz(2) = full_coord(2,jat,jres);
	jxyz(3) = full_coord(3,jat,jres);

	dxyz(1) = jxyz(1) - ixyz(1); // calculate ij vector
	dxyz(2) = jxyz(2) - ixyz(2);
	dxyz(3) = jxyz(3) - ixyz(3);

	dsq = ( dxyz(1) * dxyz(1) ) + ( dxyz(2) * dxyz(2) ) + ( dxyz(3) * dxyz(3) );
	 // calculate distance squared

	dE_dr = 0.0;
	if ( dsq >= 64.0 ) return;

	// lookup and interpolate the derivative(s)
	pairderiv(ixyz,jxyz,itype,jtype,dlj,datr,drep,dsol);

	dE_dr = cp_weight * (
	 drep * pack_wts.Wrep() * fa_rep_weight +
	 datr * pack_wts.Watr() * fa_atr_weight +
	 dsol * pack_wts.Wsol() * fa_solv_weight );

	// calculate directional components x_i/r
	one_r = 1.0/std::sqrt(dsq);
	dr_dx(1) = dxyz(1)*one_r;
	dr_dx(2) = dxyz(2)*one_r;
	dr_dx(3) = dxyz(3)*one_r;

	// calculate the distance from the pivot point
	q(1) = jxyz(1) - center_of_rotation(1);
	q(2) = jxyz(2) - center_of_rotation(2);
	q(3) = jxyz(3) - center_of_rotation(3);

}


////////////////////////////////////////////////////////////////////////////////
/// @begin accumulate_dE_dTR
///
/// @brief convert dE_dr correctly into dE_dTR
///
/// @detailed
///       after the derivative relative to atom-atom distance is done,
///       it needs to be packed correctly to derivatives relative to translation
///       and rotation vectors. The conversion simply follows math.
///
/// @param[out]   dE_dTR - out - derivatives relative to translation/rotation vector
/// @param[in]   dE_dr - in - derivative of energy with respect to atom-atom distance
/// @param[in]   dr_dx - in - derivative of atom-atom distance along each coord. axis
/// @param[in]   q - in - position of j-atom relative to center of rotation
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
accumulate_dE_dTR(
	FArray1DB_float & dE_dTR,
	float const dE_dr, // derivative of energy with respect to atom-atom distance
	FArray1DB_float const & dr_dx,
	// derivative of atom-atom distance along each coord. axis
	FArray1DB_float const & q // position of j-atom relative to center of rotation
	)
{
	if ( dE_dr == 0.0 ) return;

//     assemble derivatives using the chain rule
	dE_dTR(1) += dE_dr * dr_dx(1); // dE_dx
	dE_dTR(2) += dE_dr * dr_dx(2); // dE_dy
	dE_dTR(3) += dE_dr * dr_dx(3); // dE_dz
	dE_dTR(4) += dE_dr * (dr_dx(3)*q(2) - dr_dx(2)*q(3) ); // dE_dx-rotation
	dE_dTR(5) += dE_dr * (dr_dx(1)*q(3) - dr_dx(3)*q(1) ); // dE_dy-rotation
	dE_dTR(6) += dE_dr * (dr_dx(2)*q(1) - dr_dx(1)*q(2) );  // dE_dz-rotation

//$$$      if ( dE_dr != 0.0 ) {
//$$$         std::cout << "------------accumulate dEdTR-----------" << std::endl;
//$$$         vector_name_write_line( std::cout,dr_dx,"dr_dx")
//$$$         vector_name_write_line( std::cout,dE_dTR,"dE_dT")
//$$$         vector_name_write_line( std::cout,dE_dTR(4:6),"dE_dtheta")
//$$$      }

}


////////////////////////////////////////////////////////////////////////////////
/// @begin pairderiv
///
/// @brief vdw and solvation deriv relative atom-atom distance between two atoms
///
/// @detailed
///
/// @param[in]   atom1 - in -
/// @param[in]   atom2 - in -
/// @param[in]   attype1 - in - fullatom type of atom1
/// @param[in]   attype2 - in - fullatom type of atom2
/// @param[out]   ljderiv - out -  overall vdw derivative
/// @param[out]   attract - out -  attractive part
/// @param[out]   repulse - out - repulsive part
/// @param[out]   solvate - out - solvation derivative
///
/// @global_read data block of derivative look-up table
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
pairderiv(
	FArray1Da_float atom1,
	FArray1Da_float atom2,
	int const attype1,
	int const attype2,
	float & ljderiv,
	float & attract,
	float & repulse,
	float & solvate
)
{
	using namespace pdbstatistics_pack;

//jg     uses lookup table to find the pairwise derivatives between two atoms,
//jg     currently calculate lennard jones only
//ora    solvation added (as well as other gradients in additional functions)

	float const a_1 = atom1[ 0 ] - atom2[ 0 ]; // atom1(1) - atom2(1)
	float const a_2 = atom1[ 1 ] - atom2[ 1 ]; // atom1(2) - atom2(2)
	float const a_3 = atom1[ 2 ] - atom2[ 2 ]; // atom1(3) - atom2(3)
	float const d2 = ( ( a_1 * a_1 ) + ( a_2 * a_2 ) + ( a_3 * a_3 ) );

	if ( d2 >= safe_max_dis2 || d2 == 0.0 ) {
		ljderiv = 0.0;
		attract = 0.0;
		repulse = 0.0;
		solvate = 0.0;
		return;
	}

//ctsa
//ctsa  get bins and interpolation fraction
//ctsa

	float const d2_bin = d2 * fa_bins_per_A2;
	int const disbin1 = static_cast< int >(d2_bin)+1;
//	int disbin2 = disbin1 + 1;
	float const frac = d2_bin - float(disbin1-1);
	float const one_minus_frac = 1.0f - frac;

//ctsa
//ctsa   tables have been hacked so that if disbin2 = lastbin, all values = 0.
//ctsa
//chu     a bug was fixed. Not ljatr but dljatr should be read.

//Objexx: The following speed mod's assume arrays having the same dimensions!!!
	int const l1 = pCurrentEtable->dljatr.index(disbin1,attype2,attype1), l2 = l1 + 1;
	float attract1 = pCurrentEtable->dljatr[ l1 ]; // dljatr(disbin1,attype2,attype1);
	float attract2 = pCurrentEtable->dljatr[ l2 ]; // dljatr(disbin2,attype2,attype1);

	float repulse1 = pCurrentEtable->dljrep[ l1 ]; // dljrep(disbin1,attype2,attype1);
	float repulse2 = pCurrentEtable->dljrep[ l2 ]; // dljrep(disbin2,attype2,attype1);

	float solvate1 = pCurrentEtable->dsolv[ l1 ]; // dsolv(disbin1,attype2,attype1);
	float solvate2 = pCurrentEtable->dsolv[ l2 ]; // dsolv(disbin2,attype2,attype1);

	attract = one_minus_frac * attract1 + frac * attract2;
	repulse = one_minus_frac * repulse1 + frac * repulse2;
	ljderiv = attract + repulse;
	solvate = one_minus_frac * solvate1 + frac * solvate2;

}

//------------------------------------------------------------------------------
void
get_total_pair_deriv_docking(
	FArray1DB_float & dE_dTR,
	FArray2DB_bool & jneighborlist
)
{
	using namespace docking;
	using namespace misc;
	using namespace param;

//ora  calculates derivatives of pairscore: based on get_total_pair_deriv

//ora input: nfree: degrees of freedom (here 6: 3 translation, 3 rotation)
//ora        jneighborlist: bool array that indicates which pairs of residues
//ora                       (across the interface) are neighbors
//ora input/output: dE_dTR: array of derivatives: is updated here by the contribution from the pair energy term

//ora: it is assumed that the following are updated:
//ora             jneighborlist, neighbors
//ora             nbin
//ora             actcoord

//ora: jneighborlist is apparently not really necessary: neighborlist might be used as well
//ora: note that ALL neighbors between the partners are considered, not only those involving IF residues

//     local
	int irtype,jrtype; // residue type numbers

	float dpairE_dr; // derivative of energy with respect to atom-atom distance
	static FArray1D_float dr_dx( 3 );
	 // derivative of atom-atom distance along each coord. axis
	static FArray1D_float q( 3 ); // position of j-atom relative to center of rotation

//      test_if_updated(full_coord)

// loops over ALL neighbors between the 2 partners
// (not only if residues contribute apparently, as the comp to the
// numerically evaluated derivative shows)
	for ( int ires = part_begin(1), irese = part_end(1), jrese = part_end(2);
	 ires <= irese; ++ires ) {
		irtype = res(ires);
		for ( int jres = part_begin(2); jres <= jrese; ++jres ) {
			jrtype = res(jres);
			if ( jneighborlist(ires,jres) ) {
				calc_pair_deriv_dock(ires,irtype,jres,jrtype,dpairE_dr,dr_dx,q);
				accumulate_dE_dTR(dE_dTR,dpairE_dr,dr_dx,q);
			}
		}
	}

}
//------------------------------------------------------------------------------

void
calc_pair_deriv_dock(
	int ires,
	int aa1,
	int jres,
	int aa2,
	float & dpairE_dr, // derivative of energy with respect to atom-atom distance
	FArray1DB_float & dr_dx, // derivative of atom-atom distance along each coord. axis
	FArray1DB_float & q // position of j-atom relative to center of rotation
)
{
	using namespace docking_min;
	using namespace param_pack;
	using namespace scorefxns;
	using namespace template_pack;

//ora
// based on calc_and_sum_pair_deriv, but for residue pairs across interface
// sum is done separately
// disulfide case needs to be added

//     parameters  //Objexx: static for speed
	static FArray1D_float ixyz( 3 );
	static FArray1D_float jxyz( 3 );

//     local
	float tmp1;

	FArray1D_float dxyz( 3 ); // vector between atoms, pointing from i --> j
	float dsq; // squared distance between atoms
	float one_r; // 1/distance
	float dpair;

	ixyz(1) = actcoord(1,ires); // get position of atom i
	ixyz(2) = actcoord(2,ires);
	ixyz(3) = actcoord(3,ires);

	jxyz(1) = actcoord(1,jres); // get position of atom j
	jxyz(2) = actcoord(2,jres);
	jxyz(3) = actcoord(3,jres);

	get_pairtermE(aa1,aa2,ires,jres,ixyz,jxyz,tmp1,dpair);
	dpairE_dr = 0.0;
	if ( dpair == 0. ) return;

	dpairE_dr = pack_wts.Wpair() * fa_pair_weight * dpair;

	dxyz(1) = jxyz(1) - ixyz(1); // calculate ij vector
	dxyz(2) = jxyz(2) - ixyz(2);
	dxyz(3) = jxyz(3) - ixyz(3);

	dsq = ( dxyz(1) * dxyz(1) ) + ( dxyz(2) * dxyz(2) ) + ( dxyz(3) * dxyz(3) );
	 // calculate distance squared

	// calculate directional components x_i/r
	one_r = 1.0/std::sqrt(dsq);
	dr_dx(1) = dxyz(1)*one_r;
	dr_dx(2) = dxyz(2)*one_r;
	dr_dx(3) = dxyz(3)*one_r;

	// calculate the distance from the pivot point
	q(1) = jxyz(1) - center_of_rotation(1);
	q(2) = jxyz(2) - center_of_rotation(2);
	q(3) = jxyz(3) - center_of_rotation(3);

}

//-----------electrostatics part (dielectric constant=r)-begin-------------
void
get_total_elec_deriv_docking(
	FArray1DB_float & dE_dTR
)
{
	using namespace aaproperties_pack;
	using namespace cenlist_ns;
	using namespace interface;
	using namespace misc;

//ora: electrostatics deriv added: based on docking_warshel_electrostatics
//chu   (currently e(r) = r, so called "inverse r" model)
//ora: it is assumed that the following are updated:
//ora             int_res_list8
//ora             cendist
//ora this goes over loops, the calcs are done in calc_and_accumulate_elec_deriv

	if ( !get_fullatom_flag() ) return;

//     local
//ora generic charges    int i,j,ires,jres,iaa,jaa,iaav,jaav,iatom,jatom,iatype,jatype;
	int ires,jres,iaa,jaa,iaav,jaav;
	float iacharge,jacharge;
	float d2;
	static float const min_d2 = 1.5 * 1.5;

//     go through the list of residues at the interface...
	for ( int i = 1, ie = int_res_list8(1).size(), je = int_res_list8(2).size();
				i <= ie; ++i ) {
	 // loop over residues in receptor
		ires = int_res_list8(1)[i];
		iaa = res(ires);
		iaav = res_variant(ires);
		int const iatome = natoms(iaa,iaav);
		int const lai0 = atomic_charge.index(1,iaa,iaav);
		for ( int j = 1; j <= je; ++j ) { // loop over residues in ligand
			jres = int_res_list8(2)[j];
			jaa = res(jres);
			jaav = res_variant(jres);
			int const jatome = natoms(jaa,jaav);
			int const laj0 = atomic_charge.index(1,jaa,jaav);
			if ( cendist(ires,jres) < cen_dist_cutoff2 ) {
				for ( int iatom = 1, lai = lai0; iatom <= iatome; ++iatom, ++lai ) {
//ora: generic partial charges
					iacharge = atomic_charge[ lai ]; // atomic_charge(iatom,iaa,iaav);
					float & full_coord_i( full_coord(1,iatom,ires) );
					for ( int jatom = 1, laj = laj0; jatom <= jatome; ++jatom, ++laj ) {
//ora: generic partial charges
						jacharge = atomic_charge[ laj ]; // atomic_charge(jatom,jaa,jaav);
						float & full_coord_j( full_coord(1,jatom,jres) );
						distance2_bk(full_coord_i,full_coord_j,d2);

						if ( d2 < min_d2 ) d2 = min_d2; // flatten coulombic energy
						calc_and_accumulate_elec_deriv(iacharge,jacharge,d2,
						 full_coord_i,full_coord_j,dE_dTR);
					}         // jatom
				}            // iatom
			} // cendis_cutoff
		} // int8_2
	} // int8_1

}

//------------------------------------------------------------------------------
void
calc_and_accumulate_elec_deriv(
	float q1,
	float q2,
	float r2,
	FArray1Da_float atomi,
	FArray1Da_float atomj,
	FArray1DB_float & dE_dTR
)
{
	using namespace docking_min;
	using namespace scorefxns;

	if ( q1 == 0.0 || q2 == 0.0 ) return; // skip if any one of the atoms neutral

	static float const cutoff = 1.5 * 1.5;
	if ( r2 <= cutoff ) return; // 0 derivative if distance smaller than 1.5

	atomi.dimension( 3 );
	atomj.dimension( 3 );

//ora: based on inverse_r_elec

//     local  //Objexx: static arrays for speed
	static FArray1D_float dr_dx( 3 );
	 // derivative of atom-atom distance along each coord. axis
	static FArray1D_float q( 3 ); // position of j-atom relative to center of rotation
	static FArray1D_float dxyz( 3 ); // ij vector


	float E = 322.0637 * q1 * q2 / r2;
	if ( E == 0.0 ) return;

	float one_r = 1.0/std::sqrt(r2); // 1/r
	float dE_dr = -2*one_r*E;
	if ( dE_dr == 0.0 ) return;

// weight!!
	dE_dr *= docking_warshel_elec_weight;
//     // calculate vector ij
	dxyz(1) = atomj(1) - atomi(1);
	dxyz(2) = atomj(2) - atomi(2);
	dxyz(3) = atomj(3) - atomi(3);
	// calculate directional components x_i/r
	dr_dx(1) = dxyz(1)*one_r;
	dr_dx(2) = dxyz(2)*one_r;
	dr_dx(3) = dxyz(3)*one_r;
	// calculate the distance from the pivot point
	q(1) = atomj(1) - center_of_rotation(1);
	q(2) = atomj(2) - center_of_rotation(2);
	q(3) = atomj(3) - center_of_rotation(3);

	accumulate_dE_dTR(dE_dTR,dE_dr,dr_dx,q); // sum derivatives
}

//------------------------------------------------------------------------------
//SJF The electric potential was smoothed so that it hits zero with zero derivative
//    at a cutoff separation of 6Angstroms. The changes in the derivative computation
//    are marked below with //SJF
void
calc_and_accumulate_elec_deriv_smooth(
  float q1,
  float q2,
  float r2,
  FArray1Da_float atomi,
  FArray1Da_float atomj,
  FArray1DB_float & dE_dTR
)
{
  using namespace docking_min;
  using namespace scorefxns;

  if ( q1 == 0.0 || q2 == 0.0 ) return; // skip if any one of the atoms neutral

  static float const locutoff = 1.5 * 1.5;
  static float const hicutoff3 = std::pow(5.5,3.0); // SJF the distance cutoff for E is 5.5Ang
  static float const hicutoff2 = std::pow(5.5,2.0);

  if ( r2 <= locutoff || r2 >= hicutoff2) return; // 0ra derivative if distance smaller than 1.5 // SJF or greater than 5.5Ang

  atomi.dimension( 3 );
  atomj.dimension( 3 );

//ora: based on inverse_r_elec

//     local  //Objexx: static arrays for speed
  static FArray1D_float dr_dx( 3 );
   // derivative of atom-atom distance along each coord. axis
  static FArray1D_float q( 3 ); // position of j-atom relative to center of rotation
  static FArray1D_float dxyz( 3 ); // ij vector


 float r = std::sqrt(r2); // r
 float prefactor = 322.0637 * q1 * q2;

//SJF Previous formulation:   float dE_dr = -2*one_r*E;
  float dE_dr = 2 * prefactor * ( -1/r/r2 + 1/hicutoff3 ); // = 2*prefactor * (-1/r^3 + 1/hicutoff^3)  if ( dE_dr == 0.0 ) return;
//SJF  The smoothed function is ramped up so that the average electric energy on 1000 decoys of a global run on barnase-barstar matches that seen without smoothing
  dE_dr *= 1.590;

//SJF below this point there are no significant changes

// weight!!
  dE_dr *= docking_warshel_elec_weight;
//     // calculate vector ij
  dxyz(1) = atomj(1) - atomi(1);
  dxyz(2) = atomj(2) - atomi(2);
  dxyz(3) = atomj(3) - atomi(3);
  // calculate directional components x_i/r
  dr_dx(1) = dxyz(1)/r;
  dr_dx(2) = dxyz(2)/r;
  dr_dx(3) = dxyz(3)/r;
  // calculate the distance from the pivot point
  q(1) = atomj(1) - center_of_rotation(1);
  q(2) = atomj(2) - center_of_rotation(2);
  q(3) = atomj(3) - center_of_rotation(3);

  accumulate_dE_dTR(dE_dTR,dE_dr,dr_dx,q); // sum derivatives
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_fullatom_min_score_pose
///
/// @brief rigid-body minimize a structure
///
/// @detailed
///jg like fullatom_score_position, this function scores the structure in
///jg the position array and then restores the fullatom flag to its
///jg previous value.  packing is an option.
///jg the difference is, the structure is minimized first.
///
///   called from initialize.cc to score native/input
///
/// @param[in]   repack - in - true if repacking the structure after minimization
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_fullatom_min_score_pose( bool repack )
{
	using namespace fullatom_flag;
	using namespace runlevel_ns;

	bool save_fullatom_state;

	if ( runlevel > standard ) std::cout << "==docking_fullatom_min_score_pose==" << std::endl;

	save_fullatom_state = full_atom;
	full_atom = true;
	save_status_info("init_min_start",0,0);
	monte_carlo_reset(); // fill best arrays to enable rigid body moves
	score_set_evaluate_all_terms(false);
	docking_minimize_position("dfpmin",score10d_min,0.02);
	if ( repack ) main_repack(false); // don't include current rotamers
	score_enable_rotamer_trials(false);
	score_set_default_function(true);
	score_set_evaluate_all_terms(true);
	score_set_new_pose(); // needed b/c subset zeros-out the dunbracks
	mc_global_track::mc_score::score = scorefxn();
	calc_docking_interf_energy( docking::flexbb_docking_flag );
	score_enable_rotamer_trials(true);
	full_atom = save_fullatom_state;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_fullatom_mcm_score_pose
///
/// @brief monte-carlo minimize a structure
///
/// @detailed
///jg   like docking_fullatom_min_score_pose, except we run the mcm
///jg   protocol instead of the minimization protocol
///
/// @param[in]   repack - in - true if repacking the structure after mcm
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_fullatom_mcm_score_pose( bool repack )
{
	using namespace fullatom_flag;
	using namespace runlevel_ns;

	bool save_fullatom_state;

	if ( runlevel > standard ) std::cout << "==docking_fullatom_mcm_score_pose==" << std::endl;

	save_fullatom_state = full_atom;
	full_atom = true;
	save_status_info("init_mcm_start",0,0);
	monte_carlo_reset(); // fill best arrays to enable rigid body moves
	score_set_evaluate_all_terms(false);
	docking_mcm_protocol();
	if ( repack ) main_repack(false); // don't include current rotamers
	score_enable_rotamer_trials(false);
	score_set_default_function(true);
	score_set_evaluate_all_terms(true);
	score_set_new_pose(); // needed b/c subset zeros-out the dunbracks
	mc_global_track::mc_score::score = scorefxn();
	calc_docking_interf_energy( docking::flexbb_docking_flag );
	score_enable_rotamer_trials(true);
	full_atom = save_fullatom_state;
}


//=============== testing functions =====================

//------------------------------------------------------------------------------
void
check_derivatives_numerically(
	FArray1Da_float TR,
	FArray1Da_float dE_dTR,
	int nfree
)
{
	using namespace param_pack; // for hbond weights
	using namespace scorefxns;
	using namespace scores;

	TR.dimension( nfree );
	dE_dTR.dimension( nfree );

//     check derivatives numerically
//     the idea is to evaluate func, which is the entire composite score,
//     as well as the fullatom repulsive energy term (alone), for TRs
//     that are perturbed slightly.  We can then calculate the derivative
//     numerically.  The repulsive derivative should match dE_dTR, and
//     the composite score behavior will be an amusement.

//     for numerical derivative
	bool gfrag( true ); // success indicator
	FArray1D_float TR_pert( 6 ); // TR vector with a perturbation
	float f_plus, f_minus; // values of func with a perturbation
	float lj_plus, lj_minus; // values of LJ energy with a perturbation
	float solv_plus,solv_minus; // values of solv energy with a perturbation (ora)
	float pair_plus,pair_minus;
	 // values of pair term energy with a perturbation (ora)
//$$$      float eleca_plus,eleca_minus; // values of cheap elec energy atr with a perturbation (ora)
//$$$      float elecr_plus,elecr_minus; // values of cheap elec energy rep with a perturbation (ora)
	float elec_plus,elec_minus;
	 // values for elec energy with a perturbation (ora)
	float hb_plus,hb_minus; // values of hb energy with a perturbation (ora)

	FArray1D_float deriv( 6 ); // numerically computed derivative, cf dE_dTR
	FArray1D_float lj_deriv( 6 );
	 // numerically computed derivative of LJ score
	FArray1D_float solv_deriv( 6 );
	 // numerically computed derivative of solvation score
	FArray1D_float pair_deriv( 6 );
	 // numerically computed derivative of pair term score
//$$$      FArray1D_float eleca_deriv(6);       // numerically computed derivative of elec score
//$$$      FArray1D_float elecr_deriv(6);       // numerically computed derivative of elec score
	FArray1D_float elec_deriv( 6 );
	 // numerically computed derivative of elec score
	FArray1D_float hb_deriv( 6 );
	 // numerically computed derivative of hb score

	float epsilon; // small perturbation
//      test_elec_atr_deriv();

	for ( int i = 1; i <= nfree; ++i ) {
		TR_pert(i) = TR(i);
	}

	f_minus = func(gfrag,TR_pert);
//         std::cout << "f = " << f_minus << std::endl;

	for ( int i = 1; i <= 6; ++i ) {
		epsilon = 0.001;
		if ( i >= 4 ) epsilon = 0.0001;
		TR_pert(i) = TR(i) - epsilon;
		f_minus = func(gfrag,TR_pert);
		lj_minus = fa_rep_weight * fa_rep_score + fa_atr_weight * fa_atr_score;
		solv_minus = fa_solv_weight * fa_solv_score;
		pair_minus = fa_pair_weight * fa_pair_score;
//$$$         eleca_minus = docking_elec_atr_weight *    elec_atr_score +
//$$$          docking_lr_elec_atr_weight * lr_elec_atr_score;
//$$$         elecr_minus = docking_elec_rep_weight *    elec_rep_score +
//$$$          docking_lr_elec_rep_weight * lr_elec_rep_score;
		elec_minus = docking_warshel_elec_weight * docking_warshel_elec_score;

		hb_minus = pack_wts.Whb_lrbb() * hb_lrbb_score + pack_wts.Whb_srbb() * hb_srbb_score +
			pack_wts.Whb_sc() * hb_sc_score;

		TR_pert(i) = TR(i) + epsilon;
		f_plus = func(gfrag,TR_pert);
		lj_plus = fa_rep_weight * fa_rep_score+ fa_atr_weight * fa_atr_score;
		solv_plus = fa_solv_weight * fa_solv_score;
		pair_plus = fa_pair_weight * fa_pair_score;
//$$$         eleca_plus = docking_elec_atr_weight *    elec_atr_score +
//$$$          docking_lr_elec_atr_weight * lr_elec_atr_score;
//$$$         elecr_plus = docking_elec_rep_weight *    elec_rep_score +
//$$$          docking_lr_elec_rep_weight * lr_elec_rep_score;
		elec_plus = docking_warshel_elec_weight * docking_warshel_elec_score;

		hb_plus = pack_wts.Whb_lrbb() * hb_lrbb_score + pack_wts.Whb_srbb() * hb_srbb_score +
			pack_wts.Whb_sc() * hb_sc_score;

		deriv(i)      =  0.5 * (f_plus  - f_minus)  / epsilon;
		lj_deriv(i)   =  0.5 * (lj_plus - lj_minus) / epsilon;
		solv_deriv(i) =  0.5 * (solv_plus - solv_minus) / epsilon;
		pair_deriv(i) =  0.5 * (pair_plus - pair_minus) / epsilon;
//$$$         eleca_deriv(i) = 0.5 * (eleca_plus - eleca_minus) / epsilon;
//$$$         elecr_deriv(i) = 0.5 * (elecr_plus - elecr_minus) / epsilon;
		elec_deriv(i) = 0.5 * (elec_plus - elec_minus) / epsilon;
		hb_deriv(i)   = 0.5 * (hb_plus - hb_minus) / epsilon;

		TR_pert(i) = TR(i);
	}
	std::cout << "        TR:";
	for ( int i = 1; i <= 6; ++i ) {
		std::cout << F( 12, 4, TR(i) );
	} std::cout << std::endl;
	std::cout << "    dE_dTR:";
	for ( int i = 1; i <= 6; ++i ) {
		std::cout << F( 12, 4, dE_dTR(i) );
	}
	std::cout << F( 12, 4, vmax(dE_dTR,6)/vmin(dE_dTR,6) ) <<
	 "eps=" << F( 10, 4, epsilon ) << std::endl;
	std::cout << "func_deriv:";
	for ( int i = 1; i <= 6; ++i ) {
		std::cout << F( 12, 4, deriv(i) );
	}
	std::cout << F( 12, 4, vmax(deriv,6)/vmin(deriv,6) ) <<
	 F( 12, 4, f_plus ) << std::endl;
	std::cout << "   f_ratio:";
	for ( int i = 1; i <= 6; ++i ) {
		std::cout << F( 12, 4, dE_dTR(i)/deriv(i) );
	} std::cout << std::endl;

	if ( fa_rep_weight > 0.0 || fa_atr_weight > 0.0 ) {
		std::cout << "  lj_deriv:";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << F( 12, 4, lj_deriv(i) );
		}
		std::cout << F( 12, 4, vmax(lj_deriv,6)/vmin(lj_deriv,6) ) <<
		 F( 12, 4, lj_plus ) << std::endl;
		std::cout << "  lj_ratio:";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << F( 12, 4, dE_dTR(i)/lj_deriv(i) );
		} std::cout << std::endl;
	}

	if ( fa_solv_weight > 0.0 ) {
		std::cout << "solv_deriv:";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << F( 12, 4, solv_deriv(i) );
		}
		std::cout << F( 12, 4, vmax(solv_deriv,6)/vmin(solv_deriv,6) ) <<
		 F( 12, 4, solv_plus ) << std::endl;
		std::cout << "solv_ratio:";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << F( 12, 4, dE_dTR(i)/solv_deriv(i) );
		} std::cout << std::endl;
	}

	if ( fa_pair_weight > 0.0 ) {
		std::cout << "pair_deriv:";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << F( 12, 4, pair_deriv(i) );
		}
		std::cout << F( 12, 4, vmax(pair_deriv,6)/vmin(pair_deriv,6) ) <<
		 F( 12, 4, pair_plus ) << std::endl;
		std::cout << "pair_ratio:";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << F( 12, 4, dE_dTR(i)/pair_deriv(i) );
		} std::cout << std::endl;
	}

//$$$	if ( docking_elec_atr_weight > 0.0 || docking_lr_elec_atr_weight > 0.0 ) {
//$$$		std::cout << "elat_deriv:";
//$$$		for ( int i = 1; i <= 6; ++i ) {
//$$$			std::cout << F( 12, 4, eleca_deriv(i) );
//$$$		}
//$$$		std::cout << F( 12, 4, vmax(eleca_deriv,6)/vmin(eleca_deriv,6) ) <<
//$$$		 F( 12, 4, eleca_plus ) << std::endl;

//$$$		std::cout << "elat_ratio:";
//$$$		for ( int i = 1; i <= 6; ++i ) {
//$$$			std::cout << F( 12, 4, dE_dTR(i)/eleca_deriv(i) );
//$$$		} std::cout << std::endl;
//$$$	}

//$$$	if ( docking_elec_rep_weight > 0.0 || docking_lr_elec_rep_weight > 0.0 ) {
//$$$		std::cout << "elre_deriv:";
//$$$		for ( int i = 1; i <= 6; ++i ) {
//$$$			std::cout << F( 12, 4, elecr_deriv(i) );
//$$$		}
//$$$		std::cout << F( 12, 4, vmax(elecr_deriv,6)/vmin(elecr_deriv,6) ) <<
//$$$		 F( 12, 4, elecr_plus ) << std::endl;
//$$$		std::cout << "elre_ratio:";
//$$$		for ( int i = 1; i <= 6; ++i ) {
//$$$			std::cout << F( 12, 4, dE_dTR(i)/elecr_deriv(i) );
//$$$		} std::cout << std::endl;
//$$$	}
	if ( docking_warshel_elec_weight > 0.0 ) {
		std::cout << "elec_deriv:";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << F( 12, 4, elec_deriv(i) );
		}
		std::cout << F( 12, 4, vmax(elec_deriv,6)/vmin(elec_deriv,6) ) <<
		 F( 12, 4, elec_plus ) << std::endl;
		std::cout << "elec_ratio:";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << F( 12, 4, dE_dTR(i)/elec_deriv(i) );
		} std::cout << std::endl;
	}

	if ( param_pack::pack_wts.hbond_wts_nonzero() ) {
		std::cout << "  hb_deriv:";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << F( 12, 4, hb_deriv(i) );
		}
		std::cout << F( 12, 4, vmax(hb_deriv,6)/vmin(hb_deriv,6) ) <<
		 F( 12, 4, hb_plus ) << std::endl;
		std::cout << "  hb_ratio:";
		for ( int i = 1; i <= 6; ++i ) {
			std::cout << F( 12, 4, dE_dTR(i)/hb_deriv(i) );
		} std::cout << std::endl;
	}
}
////////////////////////////////////////////////////////////////////////////////
/// @begin vmax
///
/// @brief find the largest abs(element) of a vector
///
/// @detailed
///
/// @param[in]   v - in  - the vector
/// @param[in]   n - in  - dimension of that vector
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
vmax(
	FArray1Da_float v,
	int n
)
{
	v.dimension( n );

	float vmax = 1.e-9; // Return value
	for ( int i = 1; i <= n; ++i ) {
		if ( std::abs(v(i) ) > vmax) vmax = std::abs(v(i));
	}
	return vmax;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin vmin
///
/// @brief find the smallest abs(element) of a vector
///
/// @detailed
///
/// @param[in]   v - in - the vector
/// @param[in]   n - in - the dimension of that vector
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
vmin(
	FArray1Da_float v,
	int n
)
{
	v.dimension( n );

	float vmin = 9.e9; // Return value
	for ( int i = 1; i <= n; ++i ) {
		if ( std::abs(v(i) ) < vmin) vmin = std::abs(v(i));
	}
	return vmin;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin docking_test_packunpack
///
/// @brief testing function for pack&unpack in docking minimization
///
/// @detailed
///
/// @param  TR - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_test_packunpack( FArray1Da_float TR )
{
	TR.dimension( 6 );

	float epsilon = 0.001;
	make_named_pdb("start.pdb",true);
	docking_pack_best_position(TR);
	make_named_pdb("TR0.pdb",true);
	TR(1) = epsilon;
	docking_unpack_position(TR);
	make_named_pdb("TR1.pdb",true);
	TR(1) = 0.0;
	TR(2) = epsilon;
	docking_unpack_position(TR);
	make_named_pdb("TR2.pdb",true);
	TR(2) = 0.0;
	TR(3) = epsilon;
	docking_unpack_position(TR);
	make_named_pdb("TR3.pdb",true);
	TR(3) = 0.0;
	TR(4) = epsilon;
	docking_unpack_position(TR);
	make_named_pdb("TR4.pdb",true);
	TR(4) = 0.0;
	TR(5) = epsilon;
	docking_unpack_position(TR);
	make_named_pdb("TR5.pdb",true);
	TR(5) = 0.0;
	TR(6) = epsilon;
	docking_unpack_position(TR);
	make_named_pdb("TR6.pdb",true);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_test_lj_deriv
///
/// @brief testing lj derivatives
///
/// @detailed
///
/// @param[in]   itype - in - atom type of i
/// @param[in]   jtype - in - atom type of j
/// @param[in]   dsq - in - atom-atom distance
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/20/03
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_test_lj_deriv(
	int & itype,
	int & jtype,
	float & dsq
)
{
	using namespace pdbstatistics_pack;

	int dsqbin;
	float d1,d2,r2;
	static int mismatches = { 0 };
	static int binning = { 0 };
	static int eight = { 0 };
	static int zero = { 0 };

	dsqbin = nint(20.0*dsq)+1;

	if ( /* ljrep(dsqbin,jtype,itype) <= 0.0 && */
	 pCurrentEtable->ljatr(dsqbin,jtype,itype) <= 0.0 && /* dljrep(dsqbin,jtype,itype) <= 0.0 && */
	 pCurrentEtable->dljatr(dsqbin,jtype,itype) <= 0.0 ) return;
	std::cout << "----" << std::endl;
	std::cout << "rep " << SS( pCurrentEtable->ljrep(dsqbin,jtype,itype) ) <<
	 SS( pCurrentEtable->ljatr(dsqbin,jtype,itype) ) << std::endl;
	std::cout << "rep+" << SS( pCurrentEtable->ljrep(dsqbin+1,jtype,itype) ) <<
	 SS( pCurrentEtable->ljatr(dsqbin+1,jtype,itype) ) << std::endl;
	std::cout << "rep-" << SS( pCurrentEtable->ljrep(dsqbin-1,jtype,itype) ) <<
	 SS( pCurrentEtable->ljatr(dsqbin-1,jtype,itype) ) << std::endl;
	d1 = (pCurrentEtable->ljrep(dsqbin+1,jtype,itype)-pCurrentEtable->ljrep(dsqbin-1,jtype,itype))/
	 (std::sqrt(float(dsqbin)/20)-std::sqrt(float(dsqbin-2)/20));
	d2 = (pCurrentEtable->ljatr(dsqbin+1,jtype,itype)-pCurrentEtable->ljatr(dsqbin-1,jtype,itype))/
	 (std::sqrt(float(dsqbin)/20)-std::sqrt(float(dsqbin-2)/20));
	std::cout << "d   " << SS( d1 ) << SS( d2 ) << std::endl;
	std::cout << "drep" << SS( pCurrentEtable->dljrep(dsqbin,jtype,itype) ) <<
	 SS( pCurrentEtable->dljatr(dsqbin,jtype,itype) ) << std::endl;
	r2 = d2/pCurrentEtable->dljatr(dsqbin,jtype,itype);
	if ( r2 < 0.9 || r2 > 1.1 ) {
		++mismatches;
		std::cout << "********** RATIO MISMATCH ********** " << mismatches << std::endl;
	}
	if ( pCurrentEtable->dljatr(dsqbin,jtype,itype) != 0.0 && d2 == 0.0 ) {
		++binning;
		std::cout << "********** BINNING ********** " << binning << std::endl;
	}
	if ( dsqbin >= 1280 ) {
		++eight;
		std::cout << "********** EIGHT ANGSTROM LIMIT ********** " << eight << std::endl;
	}
	if ( pCurrentEtable->ljatr(dsqbin-1,jtype,itype) == 0.0 &&
			 pCurrentEtable->ljatr(dsqbin+1,jtype,itype) != 0.0 ) {
		++zero;
		std::cout << "********** REP/ATR SWITCH ********** " << zero << std::endl;
	}
}

//------------------------------------------------------------------------------


// for eval_dE_dr_docking debugging:

//$$$
//$$$      if ( dE_dr != 0.0 ) {
//$$$         std::cout << "------------eval dEdR-----------" << std::endl;
//$$$         write_fullcoord_atom_coords( std::cout, ires, iat );
//$$$         vector_name_write_line( std::cout,ixyz,"ixyz");
//$$$         write_fullcoord_atom_coords( std::cout, jres, jat );
//$$$         vector_name_write_line( std::cout,jxyz,"jxyz");
//$$$         vector_name_write_line( std::cout,dxyz,"dxyz");
//$$$         std::cout << "dsq " << dsq << " r " << std::sqrt(dsq) <<
//$$$         " ------>dE_dr " << dE_dr << std::endl;
//$$$         vector_name_write_line( std::cout,dr_dx,"dr_dx");
//$$$
//$$$c        more testing:
//$$$         pairenergy(ixyz,jxyz,itype,jtype,ljE,sE1,sE2,atrE,repE);
//$$$         std::cout << "current: " << repE << ' ' << atrE << ' ' << ljE << std::endl;
//$$$         jxyz(1) += 0.1;
//$$$         vector_name_write_line( std::cout,jxyz,"jxyz");
//$$$         dxyz(1) = ixyz(1) - jxyz(1); // calculate ij vector
//$$$         dxyz(2) = ixyz(2) - jxyz(2);
//$$$         dxyz(3) = ixyz(3) - jxyz(3);
//$$$         dsq = ( dxyz(1) * dxyz(1) ) + ( dxyz(2) * dxyz(2) ) + ( dxyz(3) * dxyz(3) );
//$$$          // calculate distance square
//$$$         std::cout << "dsq " << dsq << " r " << std::sqrt(dsq) << std::endl;
//$$$         pairenergy(ixyz,jxyz,itype,jtype,ljE,sE1,sE2,atrE,repE);
//$$$         std::cout << "  moved: " << repE << ' ' << atrE << ' ' << ljE << std::endl;
//$$$
//$$$      }

//$$$
//$$$c-----------------------------------------
//$$$c debugging
//$$$
//$$$      remember_best()
//$$$      include 'param.h'
//$$$      include 'misc.h'
//$$$      include 'fullatom_energies.h'
//$$$      int i,ii,iii,j;
//$$$
//$$$      float remember_coord(3,MAX_ATOM(),MAX_RES())
//$$$      float remember_atr(MAX_RES())
//$$$      float remember_atr_pair(MAX_RES(),MAX_RES())
//$$$      common /remember/ remember_coord,remember_atr,remember_atr_pair
//$$$
//$$$c     remember best position and scores
//$$$
//$$$c     position
//$$$      for ( i = 1, max_atom = MAX_ATOM(); i <= total_residue; ++i ) {
//$$$         for ( ii = 1; ii <= max_atom; ++ii ) {
//$$$            for ( iii = 1; iii <= 3; ++iii ) {
//$$$               remember_coord(iii,ii,i) = best_full_coord(iii,ii,i);
//$$$            }
//$$$         }
//$$$      }
//$$$
//$$$c     single energy
//$$$      for ( i = 1; i <= total_residue; ++i ) {
//$$$         remember_atr(i) = best_atrenergy(i);
//$$$      }
//$$$
//$$$c     pair energy
//$$$      for ( i = 1; i <= total_residue; ++i ) {
//$$$         for ( j = i+1; j <= total_residue; ++j ) {
//$$$            remember_atr_pair(i,j) = best_atr_pair(i,j);
//$$$         }
//$$$      }
//$$$
//$$$      return
//$$$      end
//$$$
//$$$
//$$$      compare_best()
//$$$      include 'param.h'
//$$$      include 'misc.h'
//$$$      include 'fullatom_energies.h'
//$$$      int i,ii,iii,j;
//$$$
//$$$      float remember_coord(3,MAX_ATOM(),MAX_RES())
//$$$      float remember_atr(MAX_RES())
//$$$      float remember_atr_pair(MAX_RES(),MAX_RES())
//$$$      common /remember/ remember_coord,remember_atr,remember_atr_pair
//$$$
//$$$      std::cout << "compairing remember best" << std::endl;
//$$$
//$$$      std::cout << "1-2 is best,current" << SS( best_atr_pair(1,2) ) <<
//$$$       SS( atr_pair(1,2) ) << std::endl;
//$$$c     check that they haven't changed
//$$$
//$$$c     position
//$$$      for ( i = 1, max_atom = MAX_ATOM(); i <= total_residue; ++i ) {
//$$$         for ( ii = 1; ii <= max_atom; ++ii ) {
//$$$            for ( iii = 1; iii <= 3; ++iii ) {
//$$$               if ( remember_coord(iii,ii,i) != best_full_coord(iii,ii,i) )
//$$$                std::cout << "remember position:" << SS( iii ) << SS( ii ) << SS( i ) <<
//$$$                SS( remember_coord(iii,ii,i) ) << SS( best_full_coord(iii,ii,i) ) << std::endl;
//$$$            }
//$$$         }
//$$$      }
//$$$
//$$$c     single energy
//$$$      for ( i = 1; i <= total_residue; ++i ) {
//$$$         if ( remember_atr(i) != best_atrenergy(i) )
//$$$          std::cout << "remember single energy:" << SS( i ) <<
//$$$          SS( remember_atr(i) ) << SS( best_atrenergy(i) ) << std::endl;
//$$$      }
//$$$
//$$$c     pair energy
//$$$      for ( i = 1; i <= total_residue; ++i ) {
//$$$         for ( j = i+1; j <= total_residue; ++j ) {
//$$$            if ( remember_atr_pair(i,j) != best_atr_pair(i,j) )
//$$$             std::cout << "remember pair energy:" << SS( i ) << SS( j ) <<
//$$$             SS( remember_atr_pair(i,j) ) << SS( best_atr_pair(i,j) ) << std::endl;
//$$$         }
//$$$      }
//$$$
//$$$
//$$$      return
//$$$      end
//$$$
//$$$
//$$$      compare_12( std::string const & label )
//$$$      include 'param.h'
//$$$      include 'misc.h'
//$$$      include 'fullatom_energies.h'
//$$$
//$$$      std::cout << label << " subr1-2 is best,current" <<
//$$$       SS( best_atr_pair(1,2) ) << SS( atr_pair(1,2) ) << std::endl;
//$$$      return
//$$$      end
//$$$c----------------------------------------------------------------------------------------------
//$$$      part2_centroid_debug(cycles)
//$$$
//$$$      include 'param.h'
//$$$      include 'misc.h'
//$$$      include 'docking.h'
//$$$
//$$$      int cycles,i
//$$$      float calc_part2_centroid(3),calc_best_part2_centroid(3)
//$$$
//$$$      chain_centroid_CA(Eposition,part2_begin,part2_end,
//$$$       calc_part2_centroid);
//$$$      chain_centroid_CA(Ebest_position,part2_begin,part2_end,
//$$$       calc_best_part2_centroid);
//$$$      std::cout << "MCM_cycle: " << cycles << std::endl;
//$$$      std::cout << "Current best_part2_centroid:";
//$$$				for ( int i = 1; i <= 3; ++i ) {
//$$$					std::cout << SS( best_part2_centroid(i) );
//$$$				} std::cout << std::endl;
//$$$      std::cout << "Calcul. best_part2_centroid:";
//$$$				for ( int i = 1; i <= 3; ++i ) {
//$$$					std::cout << SS( calc_best_part2_centroid(i) );
//$$$				} std::cout << std::endl;
//$$$      std::cout << "Current part2_centroid:";
//$$$				for ( int i = 1; i <= 3; ++i ) {
//$$$					std::cout << SS( part2_centroid(i) );
//$$$				} std::cout << std::endl;
//$$$      std::cout << "Calcul. part2_centroid:";
//$$$				for ( int i = 1; i <= 3; ++i ) {
//$$$					std::cout << SS( calc_part2_centroid(i) );
//$$$				} std::cout << std::endl;
//$$$      std::cout << std::endl;
//$$$
//$$$      return
//$$$      end
//--------------------------------------------
void
test_if_updated( FArray3Da_float fcoord )
{
	using namespace misc;
	using namespace param;
	using namespace template_pack;

	fcoord.dimension( 3, MAX_ATOM(), MAX_RES() );


	bool test_ok = true;

	if ( !test_ok ) {
		FArray2D_bool neighborlist_tmp( neighborlist );

		make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);

		for ( int i = 1, e = MAX_RES(); i <= e; ++i ) {
			for ( int j = 1; j <= e; ++j ) {
				if ( neighborlist_tmp(i,j) != neighborlist(i,j) ) {
					std::cout << "neighborlist not correctly updated " <<
					 i << ' ' << j << ' ' <<
					 L( neighborlist_tmp(i,j) ) << ' ' << L( neighborlist(i,j) ) << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}
	}

	test_ok = true;
	if ( !test_ok ) {
		std::cout << "testing actcoord update:" << std::endl;
		FArray2D_float actcoord_tmp( actcoord );

		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			int aa1 = res(seqpos);
			if ( aa1 == 6 ) {
				for ( int j = 1; j <= 3; ++j ) {
					actcoord(j,seqpos) = fcoord(j,2,seqpos);
				}
			} else {
				put_wcentroid(fcoord(1,5,seqpos),actcoord(1,seqpos),aa1);
			}
		}

		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			for ( int j = 1; j <= 3; ++j ) {
				if ( actcoord_tmp(j,seqpos) != actcoord(j,seqpos) ) {
					std::cout << "actcoord not correctly updated" <<
					 SS( seqpos ) << SS( j ) << SS( actcoord_tmp(j,seqpos) ) <<
					 SS( actcoord(j,seqpos) ) << " fullcoord j 5 seqpos" <<
					 SS( fcoord(j,5,seqpos) ) << std::endl;

					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}
	}

}

///////////////////////////////////////////////////////////////////////////////////////
// end of test_if_updated
//------------------------------------------------------------------------------
