// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $


// Rosetta Headers
#include "docking_movement.h"
#include "dock_loops.h"
#include "aaproperties_pack.h"
#include "angles.h"
#include "cenlist.h"
#include "diagnostics_rosetta.h"
#include "dock_structure.h"
#include "docking.h"
#include "design.h"
#include "docking_score.h"
#include "docking_unbound.h"
#include "docking_ns.h"
#include "FArray_xyz_functions.h"
#include "files_paths.h"
#include "fullatom.h"
#include "initialize.h"
#include "interface.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "maps.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "native.h"
#include "orient_rms.h"
#include "pack_fwd.h"
#include "PackerTask.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "pdbstatistics_pack.h"
#include "pose.h"
#include "pose_io.h"
#include "current_pose.h"
#include "pose_rotamer_trials.h"
#include "random_numbers.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "score_ns.h"
#include "status.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/constants.hh>
#include <numeric/conversions.hh>
//#include <numeric/xyzVector.io.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <numeric/trig.functions.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <iostream>


//     docking_movement.cc: functions for moving docking partners
//
//     Jeff Gray April 2001
//
//


//     main_rigid_body_trial -- do a monte carlo rigid body move
//
//     choose_rigid_body  -- pick a motion for partner 2
//     apply_rigid_body   -- move the coordinates of one protein
//
//     apply_rb_to_vector -- move one vector
//     rotate_tilt_rotate -- create transformation matrix for rotations
//     setup_rotation_matrix
//
//     calc_docking_rmsd
//
//     docking_mc_accept_best  -- monte carlo copy functions
//     docking_mc_accept_low

//car   docking_retrieve_best  --return to last accepted move (null move)
//     docking_retrieve_low

//     docking_repack          -- select interface residues for repack


//     july 27 2001: creating these functions to start a docking partner
//     from a random orientation.  This requires three coordinates.  Two
//     coordinates (azimuth phi and declination theta) specify a point on
//     the surface of the sphere, and a third coordinate (direction
//     delta) specifies the spin about that point.

//     dock_random_orient
//     dock_slide_into_contact
//     create_random_orientation_matrix
//     create_orientation_matrix
//     xrotation, yrotation, zrotation
//     docking_axis_spin



////////////////////////////////////////////////////////////////////////////////
/// @begin main_rigid_body_trial
///
/// @brief translate/rotate one partner in a docking complex
///
/// @detailed partner2 is moved. Then the new conformation is scored, and accepted
///      according to mc criterion
///
/// @param[in]   scoring_function - input (external) - which score to use
/// @param[out]   cycle_number     - input - number of cycle (for labelling output in
///         monte_carlo function)
/// @param[in]   repack_flag - input - bool: repack side chains after move?
/// @param[in]   move_label  - input - usually 'rigid_body' or 'rigid_body_FA' (used
///         for save_status)
/// @param[in]   trans_magnitude - input - magnitude of rigid-body translation
/// @param[in]   rot_magnitude   - input - magnitude of rigid-body rotation
///
/// @global_read
///   part2_begin (in docking.h) - used for save_status
/// @global_write
///   score (in misc.h) -  evaluated after move
///
/// @remarks
///       currently used only in the low-resolution step (centroid mode)
///       for the high-resolution step, minimization has been included: ( see
///       docking_mcm_protocol).
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_rigid_body_trial(
	Scoring_Function scoring_function, // which score to use
	int const cycle_number, // for labelling output
	bool const repack_flag, // repack sidechains after the move?
	std::string const & move_label, // usually "rigid_body" or "rigid_body_FA"
	float const trans_magnitude,
	float const rot_magnitude
)
{
	using namespace docking;

//   translate/rotate one partner in a docking complex
//   03/01 JJG

//   local
	FArray1D_float translation( 3 );
	FArray1D_float rotation( 3 );

//  std::cout << "main_rigid_body_trial " << cycle_number << ' ' << move_label << std::endl;

	//                 norm,paral,rot
	choose_rigid_body(translation,rotation,trans_magnitude,trans_magnitude,
	 rot_magnitude); // 0.5,0.5,pi_2*0.01)

	apply_rigid_body(translation,rotation);

	if ( repack_flag ) docking_repack(true); // true means only repack interface

	mc_global_track::mc_score::score = scoring_function();
//$$$      std::cout << "score" << SS( score ) << SS( best_score ) << std::endl;

	//glb iterate through structures in .loops file, needs -loop_trials flag
	if ( 0 == mod(cycle_number,3) ) dock_loop_trials(scoring_function);

	save_status_info( move_label, part_begin(2), 0 );
	monte_carlo(cycle_number);
	increment_trial_counters(cycle_number);

}




////////////////////////////////////////////////////////////////////////////////
/// @begin main_rigid_body_test
///
/// @brief test rigid body rotation in docking mode
///
/// @detailed apply the following rotations to a structure:
///   spin=0,36,72,...,360 degrees,
///   tilt=90 degrees, and
///   tilt direction=0,90,180,270,360 degrees.
/// and write the resulting structures out.
///
/// @global_read
///
/// @global_write
///
/// @remarks currently not called
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_rigid_body_test()
{
	using namespace numeric::constants::f;
//   test the rotations
//   03/02 JJG

//   local
	FArray1D_float translation( 3, 0.0 );
	FArray1D_float rotation( 3 );

//	int nstruct = 1;
	float const pi_2_over_ten = pi_2 / 10.0f;
	for ( int i = 0; i <= 10; ++i ) {
		for ( int j = 0; j <= 3; ++j ) {

//$$$     spin = radians( rot_mag * gaussian() );
//$$$
//$$$     tilt = radians( rot_mag * gaussian() );
//$$$     tilt_direction = pi_2 * ran3();
//$$$
//$$$     rotation(1) = spin;
//$$$     rotation(2) = tilt;
//$$$     rotation(3) = tilt_direction;

			rotation(1) = i * pi_2_over_ten;
			rotation(2) = pi_over_2; // float(j)
			rotation(3) = pi_over_2 * j;

			apply_rigid_body(translation,rotation);

			vector_write( std::cout,rotation);
			std::cout << i << ' ' << j << std::endl;
			make_named_numbered_pdb("T",i+j*11,false);
//		++nstruct;
		}
	}

}




//------------------------------------------------------------------------------
// choose_rigid_body: pick a random translation/rotation rigid body move
//
//     input: magnitude of std deviation of translation and rotation
//     output: translation and rotation vectors
//
//     rotation is centered around part2_centroid and centroid_bond (best)
//
// JJG 3/8/1

////////////////////////////////////////////////////////////////////////////////
/// @begin choose_rigid_body
///
/// @brief  pick a random translation/rotation rigid body move
///
/// @detailed
///
/// @param[out]   translation - output - translation vector (in Angstroms)
/// @param[out]   rotation    - output - rotation vector (in radians): spin, tilt,
///         tilt_direction
/// @param[in]   trans_normal_mag - input - magnitude of std deviation of
///         translation along normal axis (in Angstroms)
/// @param[in]   trans_parall_mag - input - magnitude of std deviation of
///         translation along parallel axes (in Angstroms)
/// @param[in]   rot_mag          - input - magnitude of std deviation of
///         rotations (in degrees)
///
/// @global_read
///
/// @global_write
///
/// @remarks usually followed by apply_rigid_body to make the move specified here
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
choose_rigid_body(
	FArray1Da_float translation,
	FArray1Da_float rotation,
	float const trans_normal_mag, // magnitudes of movement
	float const trans_parall_mag,
	float const rot_mag // rotation magnitude in degrees
)
{
	using namespace numeric::constants::f;
	using numeric::conversions::radians;
	using numeric::sin_cos_range;

	translation.dimension( 3 );
	rotation.dimension( 3 );

//   local
	float mag1, mag2, mag3; // translation magnitudes along each direction
	float spin, tilt, tilt_direction; // rotation specifiers

//   Random translation

	mag1 = gaussian() * trans_normal_mag;
	mag2 = gaussian() * trans_parall_mag;
	mag3 = gaussian() * trans_parall_mag;

	create_docking_trans_vector(translation,mag1,mag2,mag3);

//   Random rotations
//mj check random flag and rot_mag, if true => ensure random orientation

	if ( get_ligand_flag() && rot_mag >= 90.0 ) {
		spin = pi_2 * ran3();
		 //cmj spin between 0..360 equally distributet
		tilt = std::acos( sin_cos_range( 2 * ran3() - 1 ) );
		 //cmj tilt between 0..180 equally distributed on sphere
		tilt_direction = pi_2 * ran3();
		 //cmj tilt_direction between 0..360 equally distributet
	} else {
		spin = radians( rot_mag * gaussian() );
		tilt = radians( rot_mag * gaussian() );
		tilt_direction = pi_2 * ran3();
	}

	rotation(1) = spin;
	rotation(2) = tilt;
	rotation(3) = tilt_direction;

//	std::cout << "translation magnitudes: " <<
//   F( 8, 3, mag1 ) << ' ' << F( 8, 3, mag2 ) << ' ' << F( 8, 3, mag3 ) << " ; ";

}

////////////////////////////////////////////////////////////////////////////////
/// @begin create_docking_trans_vector
///
/// @brief create a translation vector from the magnitudes normal and
///     parallel to the docking centroid vector
///
/// @detailed
///
/// @param[out]   translation - output - translation vector (in Angstroms)
/// @param[in]   m_norm - input - magnitude of std deviation of translation
///                          along normal axis (in Angstroms)
/// @param[in]   m_par1 - input - magnitude of std deviation of translation
///                          along parallel axis 1 (in Angstroms)
/// @param[in]   m_par2 - input - magnitude of std deviation of  translation
///                          along parallel axis 2 (in Angstroms)
///
/// @global_read
/// docking_best block in docking.h:
/// best_docking_normal,best_docking_parallel1,best_docking_parallel2
///
/// @global_write
///
/// @remarks
///   translation part of choose_rigid_body
///   usually followed by apply_rigid_body to make the move specified here
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_docking_trans_vector(
	FArray1Da_float translation,
	float m_norm,
	float m_par1,
	float m_par2
)
{
	using namespace docking;

	translation.dimension( 3 );

//     create a translation vector from the magnitudes normal and
//     parallel to the docking centroid vector

	for ( int i = 1; i <= 3; ++i ) {
		translation(i) = m_norm*best_docking_normal(i) + m_par1*
		best_docking_parallel1(i) + m_par2*best_docking_parallel2(i);
	}

}



//------------------------------------------------------------------------------
// apply_rigid_body: move the coordinates of partner2
//
//     input: translation and rotation vectors
//
//     rotation is centered around part2_centroid and centroid_bond (best)
//
//     we move several arrays: centroid, position, full_coord, and
//     the docking_geometry namespace (part2_centroid, centroid_bond,
//     contact_point)
//
//     actually, we move these arrays using the *best* arrays as
//     our starting reference position.  This makes this move 'undoable'
//     in the monte_carlo setup. 4/17/1
//
// JJG 3/8/1
////////////////////////////////////////////////////////////////////////////////
/// @begin apply_rigid_body
///
/// @brief  move the coordinates of partner2
///
/// @detailed apply rigid body movement (translation and rotation) to partner2.
///      first creates rotation matrix based on rotation input. Then applies
///      translation vector and rotation matrix to coordinates
///
/// @param[in]   translation - input - translation vector (in Angstroms)
/// @param[in]   rotation    - input - rotation vector (in radians): spin, tilt,
///         tilt_direction
///
/// @global_read
/// docking_best block in docking.h:
/// best_docking_normal,best_docking_parallel1,best_docking_parallel2,best_part2_centroid
///
/// @global_write
///
/// @remarks
///     rotation is centered around part2_centroid and centroid_bond
///
///     we move several arrays: centroid, position, full_coord, and
///     the docking_geometry namespace (part2_centroid, centroid_bond,
///     contact_point)
///
///     actually, we move these arrays using the *best* arrays as
///     our starting reference position.  This makes this move 'undoable'
///     in the monte_carlo setup. 4/17/1
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
apply_rigid_body(
	FArray1DB_float const & translation,
	FArray1DB_float const & rotation
)
{
	using namespace docking;
	using numeric::conversions::degrees;

//   local
	float spin,tilt,tilt_direction; // rotation specifiers
	static FArray2D_float R( 3, 3 ); // rotation matrix

//   Prepare rotation and translation intermediates
	spin = rotation(1);
	tilt = rotation(2);
	tilt_direction = rotation(3);

	setup_rotation_matrix(tilt_direction,tilt,(-tilt_direction+spin),
	 best_docking_normal,best_docking_parallel1,best_docking_parallel2,R); // -> R

	apply_rigid_body_matrix(translation,R,best_part_centroid(1,2));

//	std::cout << "fun: best_part_centroid(1,2) " <<
//	 SS( best_part_centroid(1,2) ) << std::endl;
//   Echo motions

//	std::cout << " translating: ";
//	vector_write( std::cout, translation );

//	std::cout << "and rotating: ";
//	std::cout << "spin=" << F( 8, 4, spin ) <<
//	 "(" << F( 5, 1, degrees( spin ) ) << "d); tilt=" <<
//	 F( 8, 4, tilt ) << "(" << F( 5, 1, degrees( tilt ) ) << "d)" <<
//	 "in direction" << F( 8, 4, tilt_direction ) << "(" <<
//	 F( 5, 1, degrees( tilt_direction ) ) << "d)" << std::endl;

}

//------------------------------------------------------------------------------
// apply_rigid_body: move the coordinates of partner2
//
//     input: translation and rotation vectors
//
//     rotation is centered around part2_centroid and centroid_bond
//
//     we move several arrays: centroid, position, full_coord, and
//     the docking_geometry namespace (part2_centroid, centroid_bond,
//     contact_point)
//
//     actually, we move these arrays using the *best* arrays as
//     our starting reference position.  This makes this move 'undoable'
//     in the monte_carlo setup. 4/17/1
//
// JJG 3/8/1
////////////////////////////////////////////////////////////////////////////////
/// @begin apply_rigid_body_matrix
///
/// @brief move the coordinates as specified by a translation vector, a rotation
///       matrix, and a rotation center.
///
/// @detailed
///
/// @param[in]   translation - input - translation vector
/// @param[in]   R           - input - rotation matrix
/// @param[in]   rot_center  - input - rotation center
///
/// @global_read
/// docking geometry:
///  PART2_BEGIN, PART2_END (in docking.h) - range of partner2
///  BEST_PART2_CENTROID (in docking.h)
/// best_pose block in misc.h (best_centroid,Ebest_position,best_full_coord)
/// hetero_atoms block in ligand.h (see there for more info):
///  ligand_flag,best_hetero_atom_coord,
///  recompute_ligand_interface
/// natoms array (in aaproperties_pack.h) - number of atoms for each aa type
///
/// @global_write
/// PART2_CENTROID (in docking.h) - centroid of partner2 moved
/// current_pose block in misc.h - coordinates of partner2 moved
///   centroid,Eposition,full_coord
/// hetero_atom_coord (in ligand.h) - coordinates of ligand moved
///
/// @remarks
///    By specifying the rotation center, one can effectively randomize
///    either docking partner (and keep partner1 fixed).
/// @references
///
/// @authors Ora 08/20/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
apply_rigid_body_matrix(
	FArray1DB_float const & translation,
	FArray2DB_float const & R, // rotation matrix
	FArray1Da_float rot_center // rotation center
)
{
	using namespace aaproperties_pack;
	using namespace docking;
	using namespace ligand;
	using namespace misc;
	using namespace design;

	rot_center.dimension( 3 );

//   local
	static FArray1D_float P( 3 ); // rotation origin
	static FArray1D_float PT( 3 ); // P+translation


	static FArray1D_float het_coord( 3 ); // these arrays should disappear
	static FArray1D_float b_het_coord( 3 ); // as the conversion to xyzVector takes place


	vector_copy(rot_center,P); // -> P
	vector_sum(P,translation,PT); // -> PT

//      vector_name_write_line( std::cout,P,"P");
//      vector_name_write_line( std::cout,PT,"PT");
//      matrix_name_write( std::cout,R,"R");

//mj check ligand flag
	if ( get_ligand_flag() ) {
		if(!get_ligand_flexible_flag()){
			for ( size_t i=0; i<(*ligand::ligand_one).atom_vector.size();
					 ++i ) {

				copy_to_FArray((*ligand::ligand_one).best_coord[i], b_het_coord);

				apply_rb_to_vector(het_coord(1),b_het_coord(1),R,P,PT);

				(*ligand::ligand_one).atom_vector[i]->set_coordinates(
						numeric::xyzVector<double>(het_coord(1),het_coord(2),het_coord(3)));
			}
			(*ligand::ligand_one).recompute_ligand_interface = true;
		}else{
			// if ligand is flexible performs translation and rotation on anchor point
			// and then rebuilds based on internal coordinates.

			copy_to_FArray(ligand::ligand_one->anchor_first_best, b_het_coord);
			apply_rb_to_vector(het_coord(1),b_het_coord(1),R,P,PT);
			ligand::ligand_one->anchor_first=
					numeric::xyzVector<double>(het_coord(1),het_coord(2),het_coord(3));
			copy_to_FArray(ligand::ligand_one->anchor_second_best, b_het_coord);
			apply_rb_to_vector(het_coord(1),b_het_coord(1),R,P,PT);
			ligand::ligand_one->anchor_second=
					numeric::xyzVector<double>(het_coord(1),het_coord(2),het_coord(3));
			copy_to_FArray(ligand::ligand_one->anchor_third_best, b_het_coord);
			apply_rb_to_vector(het_coord(1),b_het_coord(1),R,P,PT);
			ligand::ligand_one->anchor_third=
					numeric::xyzVector<double>(het_coord(1),het_coord(2),het_coord(3));

			ligand::ligand_one->anchor_state++;
			ligand::ligand_one->refold_ligand();
			(*ligand::ligand_one).recompute_ligand_interface = true;
		}
	} else {
		bool const fullatom = get_fullatom_flag();
//   Loop over all atoms
		for ( int i = part_begin(2), ie = part_end(2); i <= ie; ++i ) { // residues
//     move centroids
			apply_rb_to_vector(centroid(1,i),best_centroid(1,i),R,P,PT);
//     move backbone
			for ( int j = 1; j <= 5; ++j ) {
				apply_rb_to_vector(Eposition(1,j,i),Ebest_position(1,j,i),R,P,PT);
			}
			if ( fullatom || design_dock_pert ) {
//     move full-coordinate atoms
				for ( int j = 1, je = natoms(res(i),res_variant(i)); j <= je; ++j ) {
				 // atoms in each residue -- full_coord
					apply_rb_to_vector(full_coord(1,j,i),best_full_coord(1,j,i),R,P,PT);
				}               // loop over atoms in each residue
			}
		}                  // loop over part2 residues
	}

//   Move docking vectors -- part2_centroid
	apply_rb_to_vector(part_centroid(1,2),best_part_centroid(1,2),R,P,PT);

//   Derive centroid_bond,contact_point,docking_normal/parallel1/2
	docking_derive_coord_frame();

	if ( get_ligand_flag() ) return;

	//   Let other functions know which residues have moved
	maps_set_new_rigidbody(part_begin(2),total_residue);

}




////////////////////////////////////////////////////////////////////////////////
/// @begin apply_rb_to_vector
///
/// @brief  y --> R*(x-P) + (PT)
///
/// @detailed   apply a rotation matrix R centered at P, and a translation T = PT-P
///     to a generic vector x(3), return the modified value in y
///
/// @param  y   [in/out]? - output position vector
/// @param[in]   x - input -  input position vector
/// @param[in]   R - input -  rotation matrix
/// @param[in]   P - input -  offset for rotation center
/// @param  PT - [in/out]? -P + translation vector
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ora 08/20/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
apply_rb_to_vector(
	FArray1Da_float y, // output position vector
	FArray1Da_float x, // input position vector
	FArray2DB_float const & R, // rotation matrix
	FArray1DB_float const & P, // offset for rotation center
	FArray1DB_float const & PT // P + translation vector
)
{
//Objexx: Use linear zero-based indexing for these to save cost of dimension
//	y.dimension( 3 );
//	x.dimension( 3 );

//     apply a rotation matrix R centered at P, and a translation T = PT-P
//     to a generic vector x(3), return the modified value in y

//     local  //Objexx: static for speed
	static FArray1D_float xP( 3 );
	static FArray1D_float RxP( 3 );

	xP(1) = x[0] - P(1);
	xP(2) = x[1] - P(2);
	xP(3) = x[2] - P(3);

	RxP(1) = R[0] * xP(1) + R[3] * xP(2) + R[6] * xP(3);
	RxP(2) = R[1] * xP(1) + R[4] * xP(2) + R[7] * xP(3);
	RxP(3) = R[2] * xP(1) + R[5] * xP(2) + R[8] * xP(3);

//Objexx: Replace by above for speed
//	RxP(1) = R(1,1) * xP(1) + R(1,2) * xP(2) + R(1,3) * xP(3);
//	RxP(2) = R(2,1) * xP(1) + R(2,2) * xP(2) + R(2,3) * xP(3);
//	RxP(3) = R(3,1) * xP(1) + R(3,2) * xP(2) + R(3,3) * xP(3);

	y[0] = RxP(1) + PT(1);
	y[1] = RxP(2) + PT(2);
	y[2] = RxP(3) + PT(3);

}
//////////////////////////////////////////////////////////////
// double precision version for R, P, PT
void
apply_rb_to_vector(
	FArray1Da_float y, // output position vector
	FArray1Da_float x, // input position vector
	FArray2DB_double const & R, // rotation matrix
	FArray1DB_double const & P, // offset for rotation center
	FArray1DB_double const & PT // P + translation vector
)
{
//Objexx: Use linear zero-based indexing for these to save cost of dimension
//	y.dimension( 3 );
//	x.dimension( 3 );

//     apply a rotation matrix R centered at P, and a translation T = PT-P
//     to a generic vector x(3), return the modified value in y

//     local  //Objexx: static for speed
	static FArray1D_double xP( 3 );
	static FArray1D_double RxP( 3 );

	xP(1) = x[0] - P(1);
	xP(2) = x[1] - P(2);
	xP(3) = x[2] - P(3);

	RxP(1) = R[0] * xP(1) + R[3] * xP(2) + R[6] * xP(3);
	RxP(2) = R[1] * xP(1) + R[4] * xP(2) + R[7] * xP(3);
	RxP(3) = R[2] * xP(1) + R[5] * xP(2) + R[8] * xP(3);

//Objexx: Replace by above for speed
//	RxP(1) = R(1,1) * xP(1) + R(1,2) * xP(2) + R(1,3) * xP(3);
//	RxP(2) = R(2,1) * xP(1) + R(2,2) * xP(2) + R(2,3) * xP(3);
//	RxP(3) = R(3,1) * xP(1) + R(3,2) * xP(2) + R(3,3) * xP(3);

	y[0] = RxP(1) + PT(1);
	y[1] = RxP(2) + PT(2);
	y[2] = RxP(3) + PT(3);

}
////////////////////////////////////////////////////////////////////////////////
/// @begin rotate_tilt_rotate
///
/// @brief Create a transformation matrix to rotate by phi around the x-axis,
///     then by theta around the y-axis, then by psi around the x-axis.
///
/// @detailed
///     We spin a protein around an axis (x), tilt off that axis by theta, {
///     spin around the original axis to a new angle.
///              phi=tilt_angle;   theta=tilt;   psi+phi=spin
///
/// @param[in]   phi   - input - 1.rotation: around x axis
/// @param[in]   theta - input - 2.rotation: around y axis
/// @param[in]   psi   - input - 3.rotation: around x axis again
/// @param[out]   M    - output - transformation matrix
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     This is written for the dock module to rotate the ligand protein
///     around the centroid-bond axis.  if theta is small and phi+psi is small,
///     then this transformation becomes a small-move with a spin and
///     a tilt.
///
/// @references
///
/// @authors Ora 08/20/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rotate_tilt_rotate(
	float phi,
	float theta,
	float psi,
	FArray2DB_float & M
)
{
//     Create a transformation matrix to rotate by phi around the x-axis,
//     then by theta around the y-axis, then by psi around the x-axis.
//
//     In this way, we can spin a protein around an axis (x), tilt off that
//     axis by theta, then spin around the original axis to a new angle.
//
//              phi=tilt_angle;   theta=tilt;   psi+phi=spin
//
//     This is written for the dock module to rotate the ligand protein
//     around the centroid-bond axis.  if theta is small and phi+psi is small,
//     then this transformation becomes a small-move with a spin and
//     a tilt

//     local
	float cphi,sphi,ctheta,stheta,cpsi,spsi;

	cphi =   std::cos(phi);
	ctheta = std::cos(theta);
	cpsi =   std::cos(psi);
	sphi =   std::sin(phi);
	stheta = std::sin(theta);
	spsi =   std::sin(psi);

	M(1,1) = ctheta;
	M(1,2) = -(stheta*spsi);
	M(1,3) = -(stheta*cpsi);

	M(2,1) = -(sphi*stheta);
	M(2,2) = cphi*cpsi - (sphi*ctheta*spsi);
	M(2,3) = -(cphi*spsi)-(sphi*ctheta*cpsi);

	M(3,1) = cphi*stheta;
	M(3,2) = sphi*cpsi+cphi*ctheta*spsi;
	M(3,3) = -(sphi*spsi)+cphi*ctheta*cpsi;

}



////////////////////////////////////////////////////////////////////////////////
/// @begin setup_rotation_matrix
///
/// @brief
///
/// @detailed
///
/// @param  phi - [in/out]? -
/// @param  theta - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  u - [in/out]? -
/// @param  v - [in/out]? -
/// @param  w - [in/out]? -
/// @param  R - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_rotation_matrix(
	float phi,
	float theta,
	float psi,
	FArray1DB_float const & u,
	FArray1DB_float const & v,
	FArray1DB_float const & w,
	FArray2DB_float & R
)
{
//     The docking rotations are defined in the docking coordinate system
//     (u,v,w) which is based on the centroids of the two proteins.  This
//     function creates a rotation matrix to be applied in *cartesian*
//     coordinates, given the rotation angles phi,theta, and psi, and the
//     docking-frame basis vectors, u,v,w.

//     local
	static FArray2D_float STS( 3, 3 );
	 // matrix that does the rotations through phi,theta,psi
	// in docking coordinate system

	rotate_tilt_rotate(phi,theta,psi,STS); // get STS
	rotation_coord_transform(u,v,w,STS,R); // transform to cartesian

//      std::cout << "phi,theta,psi:" << SS( phi ) << SS( theta ) << SS( psi ) << std::endl;
//      matrix_name_write( std::cout,B,"B");
//      matrix_name_write( std::cout,STS,"STS");
//      matrix_name_write( std::cout,BSTS,"BSTS");

}


////////////////////////////////////////////////////////////////////////////////
/// @begin rotation_coord_transform
///
/// @brief
///
/// @detailed
///
/// @param  u - [in/out]? -
/// @param  v - [in/out]? -
/// @param  w - [in/out]? -
/// @param  Rin - [in/out]? -
/// @param  Rout - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rotation_coord_transform(
	FArray1DB_float const & u,
	FArray1DB_float const & v,
	FArray1DB_float const & w,
	FArray2DB_float const & Rin,
	FArray2DB_float & Rout
)
{
//     transform a rotation in the (u,v,w) coordinates (docking frame)
//     to a rotation in (x,y,z) coordinates (cartesian frame)
//
//     revised & tested for randomizing and rb moves 3/02 jjg

//     local
	static FArray2D_float B( 3, 3 );
	 // coordinate transformation matrix from x,y,z to u,v,w
	 // where u,v,w are in the docking framework
	static FArray2D_float BTRin( 3, 3 ); // intermediate result, B*Rin

	B(1,1) = u(1); // are the indices on B correct????
	B(1,2) = u(2);
	B(1,3) = u(3);
	B(2,1) = v(1);
	B(2,2) = v(2);
	B(2,3) = v(3);
	B(3,1) = w(1);
	B(3,2) = w(2);
	B(3,3) = w(3);

	BlankMatrixMult4(B,3,3,1,Rin,3,0,BTRin); // B^T*Rin -> BTRin
	BlankMatrixMult4(BTRin,3,3,0,B,3,0,Rout); // BTRin*B -> Rout

//       Rout = B^T * R * B

//      std::cout << "phi,theta,psi:" << SS( phi ) << SS( theta ) << SS( psi ) << std::endl;
//      matrix_name_write( std::cout,B,"B");
//      matrix_name_write( std::cout,Rin,"Rin");
//      matrix_name_write( std::cout,BRin,"BRin");

}



//------------------------------------------------------------------------------

//
//     RMSD: In docking, partner 1 stays fixed and sets the coordinate
//     frame, then the rmsd is calculated over partner 2.  Here, we
//     consider only CA atoms.
//
//     Also calculated is the total translation of the centroid of part2
//     and the rotation of part2 relative to the native
//
////////////////////////////////////////////////////////////////////////////////
/// @begin calc_docking_rmsd
///
/// @brief calculation of the rmsd of partner2 relative to native (CA only).
///
/// @detailed  in docking, partner1 stays fixed and sets the coordinate
///     frame. the rmsd is calculated over partner2 only.
///     Also calculated is the total translation of the centroid of part2
///     and the rotation of part2 relative to the native.
///
/// @global_read
/// Eposition (in misc.h)
/// native block in native.h: native_ca,NATIVE_HETERO_ATOM_COORD
///
/// @global_write
///  docking_result block in docking.h:
///     docking_rmsd,docking_rotation_angle,docking_translation_distance,
///     docking_translation_vector
///  rms_err (in misc.h) -  last evaluated rms set to 0 if no native around
///
/// @remarks
///
/// @references
///
/// @authors Ora 08/20/2003
///
/// @last_modified Aroop 04/23/2007
/////////////////////////////////////////////////////////////////////////////////
void
calc_docking_rmsd()
{
	using namespace docking;
	using namespace docking_unbound;
	using namespace ligand;
	using namespace misc;
	using namespace native;
	using numeric::conversions::degrees;
	using numeric::sin_cos_range;

//     local
	float tsd,dis2; // total square distance, accumulated
	FArray1D_float current_orientation( 3 );
	 // current orientation of part2(see docking.h)

	if ( !get_native_exists() || files_paths::antibody_modeler) {
		mc_global_track::diagnose::rms_err = 0.0;
		docking_rmsd = 0.0;
		docking_rotation_angle = 0.0;
		docking_translation_distance = 0.0;
		return;
	}

//-1-  translation distance
	subvec(part_centroid(1,2),native_part_centroid(1,2),
	 docking_translation_vector);
	docking_translation_distance = vector_norm(docking_translation_vector);

//mj ligand flag
	if ( get_ligand_flag() ) {

//mj angle - its definition makes no sense at all, set it therefore to 0.0 for the moment
//mj there is actually a clean definition for this angle
		docking_rotation_angle = 0.0;

//mj rmsd over all hetero atoms
		tsd = 0.0;
		if((*ligand::ligand_one).atom_vector.size()==native_hetero_atom_coord.size()){
			for ( size_t i = 0;
					i < (*ligand::ligand_one).atom_vector.size();
					++i ) {
						dis2 = distance_squared((*ligand::ligand_one).atom_vector[i]->get_coordinates(),native_hetero_atom_coord[i]);
						tsd += dis2;
			}
			docking_rmsd = std::sqrt(tsd/float((*ligand::ligand_one).atom_vector.size()));
		}else{std::cerr << "Diferent number of atoms cooridinates in atom_vector" << std::endl;
				std::cerr << "than in native."  << std::endl;
		}
		return;
	}
// following only for non-ligand
//-2-  Calculate the orientational difference from native
	subvec(Eposition(1,2,part_end(2)),Eposition(1,2,part_begin(2)),
	 current_orientation);
//      vector_name_write_line( std::cout,native_part2_orientation,"native")
//      vector_name_write_line( std::cout,current_orientation,"current")
	docking_rotation_angle = degrees( std::acos( sin_cos_range(
	 dotprod( native_part2_orientation, current_orientation ) /
	 std::sqrt( dotprod( current_orientation, current_orientation ) *
	 dotprod( native_part2_orientation, native_part2_orientation ) ) ) ) );
//8-27 actually, if the decoy is spun around the begin-end vector,
//     docking_rotation_angle will be zero when it is not....how to
//     remedy this??

//-3-  rmsd averaged over all CA's in part2
	if ( !use_mapped_rmsd ) {
		tsd = 0.0;
		for ( int i = part_begin(2), iend = part_end(2); i <= iend; ++i ) {
			float const dif1 = Eposition(1,2,i) - native_ca(1,i);
			float const dif2 = Eposition(2,2,i) - native_ca(2,i);
			float const dif3 = Eposition(3,2,i) - native_ca(3,i);
			tsd += ( dif1 * dif1 ) + ( dif2 * dif2 ) + ( dif3 * dif3 );
		}
		docking_rmsd = std::sqrt(tsd/float(part_end(2)-part_begin(2)+1));
	} else {
		tsd = 0.0;
		int counter = 0;
		for ( int i = native_part_begin(2), iend = native_part_end(2); i <= iend; ++i ) {
			int const k = unbound_seq_align_map(i);
			if ( k != 0 ) { // k==0 => no equivalent in unbound structure
				float const dif1 = Eposition(1,2,k) - native_ca(1,i);
				float const dif2 = Eposition(2,2,k) - native_ca(2,i);
				float const dif3 = Eposition(3,2,k) - native_ca(3,i);
				tsd += ( dif1 * dif1 ) + ( dif2 * dif2 ) + ( dif3 * dif3 );
				++counter;
			}
		}
		docking_rmsd = std::sqrt(tsd/counter);
	}

}




////////////////////////////////////////////////////////////////////////////////
/// @begin docking_hijack_best_pose
///
/// @brief save the "best" information in while the best arrays are hijacked
///       (for minimization, JJG 5/02)
///
/// @detailed
///
/// @global_read  docking best array
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_hijack_best_pose()
{
	using namespace docking;
	using namespace docking_hijack_geometry;

	hijack_best_pose();

	for ( int i = 2; i <= n_monomers; ++i ) {
		vector_copy(best_part_centroid(1,i),save_part_centroid(1,i));
	}

	vector_copy(best_docking_normal   ,save_docking_normal );
	vector_copy(best_docking_parallel1,save_docking_parallel1);
	vector_copy(best_docking_parallel2,save_docking_parallel2);
	vector_copy(best_centroid_bond    ,save_centroid_bond );
	vector_copy(best_contact_point    ,save_contact_point );
//chu copy current to best, otherwise coordinate and geometry will not match
	for ( int i = 2; i <= n_monomers; ++i ) {
		vector_copy(part_centroid(1,i),best_part_centroid(1,i));
	}
	vector_copy(docking_normal,best_docking_normal);
	vector_copy(docking_parallel1,best_docking_parallel1);
	vector_copy(docking_parallel2,best_docking_parallel2);
	vector_copy(centroid_bond,best_centroid_bond);
	vector_copy(contact_point,best_contact_point);

}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_hijack_best_pose_rlease
///
/// @brief restore the "best" information after the best arrays were hijacked
///       (for minimization, JJG 5/02)
///
/// @detailed
///
/// @global_read
///
/// @global_write best array in docking
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_hijack_best_pose_rlease()
{
	using namespace docking;
	using namespace docking_hijack_geometry;

//     restore the "best" information after the best arrays were hijacked
//     (for minimization, JJG 5/02)

	hijack_best_pose_release();

	for ( int i = 2; i <= n_monomers; ++i ) {
		vector_copy(save_part_centroid(1,i),best_part_centroid(1,i));
	}

	vector_copy(save_docking_normal   ,best_docking_normal );
	vector_copy(save_docking_parallel1,best_docking_parallel1);
	vector_copy(save_docking_parallel2,best_docking_parallel2);
	vector_copy(save_centroid_bond    ,best_centroid_bond );
	vector_copy(save_contact_point    ,best_contact_point );

}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_mc_accept_best
///
/// @brief accept docking_positons of a best structure
///
/// @detailed
///     This function copies the current docking configuration
///     into the 'best' namespace, meaning that a monte carlo move
///     has been accepted and will be saved to start from in the future
///c
/// @global_read current docking positions
///
/// @global_write docking best arrays
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_mc_accept_best()
{
	using namespace docking;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;

// I don't think this is really necessary...
//     correct for roundoff errors when accepting best
//      chain_centroid_CA(position,part2_begin,part2_end,part2_centroid)
//      docking_derive_coord_frame()

	for ( int i = 2; i <= n_monomers; ++i ) {
		vector_copy(part_centroid(1,i),best_part_centroid(1,i));
	}

	vector_copy(docking_normal   ,best_docking_normal );
	vector_copy(docking_parallel1,best_docking_parallel1);
	vector_copy(docking_parallel2,best_docking_parallel2);
	vector_copy(centroid_bond    ,best_centroid_bond );
	vector_copy(contact_point    ,best_contact_point );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_mc_accept_low
///
/// @brief accept docking_positions of a low structure
///
/// @detailed
///     This function copies the current docking configuration
///     into the 'low' namespace, meaning that a monte carlo move
///     has been accepted and will be saved to start from in the future
///
/// @global_read docking current positions
///
/// @global_write dockiing low arrays
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_mc_accept_low()
{
	using namespace docking;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;
	for ( int i = 2; i <= n_monomers; ++i ) {
		vector_copy(part_centroid(1,i),low_part_centroid(1,i));
	}
	vector_copy(docking_normal   ,low_docking_normal );
	vector_copy(docking_parallel1,low_docking_parallel1);
	vector_copy(docking_parallel2,low_docking_parallel2);
	vector_copy(centroid_bond    ,low_centroid_bond );
	vector_copy(contact_point    ,low_contact_point );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_retrieve_best
///
/// @brief retrieve docking parameters of the last accepted structure
///
/// @detailed
///car Copy the BEST pose docking parameters into the current pose
///car This is a null move and returns one to the last accepted move
///
/// @global_read  docking_best_array
///
/// @global_write docking_current_array
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_retrieve_best()
{
//car Copy the BEST pose docking parameters into the current pose
//car This is a null move and returns one to the last accepted move

	using namespace docking;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;
	for ( int i = 2; i <= n_monomers; ++i ) {
		vector_copy(best_part_centroid(1,i),part_centroid(1,i));
	}
	vector_copy(best_docking_normal   ,docking_normal );
	vector_copy(best_docking_parallel1,docking_parallel1);
	vector_copy(best_docking_parallel2,docking_parallel2);
	vector_copy(best_centroid_bond    ,centroid_bond );
	vector_copy(best_contact_point    ,contact_point );

}
////////////////////////////////////////////////////////////////////////////////
/// @begin docking_hose_pose
///
/// @brief a test function
///
/// @detailed
///jg Put nonsense numbers into the docking position to see if they are used
///jg somewhere
///jg 5/23: put this in resetphipsi and nothing changed (this is good)
///jg that means all moves are based off of the BEST arrays, not the current,
///jg as is rightc
///
/// @global_read
///
/// @global_write docking position arrays in docking.h
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_hose_pose()
{
	using namespace docking;

	FArray1D_float crap( 3 );
	crap(1) = 1e9;
	crap(2) = 1e9;
	crap(3) = 1e9;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;
	for ( int i = 2; i <= n_monomers; ++i ) {
		vector_copy(crap,part_centroid(1,i));
	}
	vector_copy(crap,docking_normal );
	vector_copy(crap,docking_parallel1);
	vector_copy(crap,docking_parallel2);
	vector_copy(crap,centroid_bond );
	vector_copy(crap,contact_point );

}
////////////////////////////////////////////////////////////////////////////////
/// @begin docking_retrieve_low
///
/// @brief retrieve docking parameters in low array
///
/// @detailed
///     This function copies the 'low' namespace into the current
///     docking configuration, meaning that a monte carlo move
///     has been accepted and will be saved to start from in the future
///                             --JJG
///
/// @global_read part of low arrays in docking.h
///
/// @global_write current array in docking.h
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_retrieve_low()
{
	using namespace docking;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;
	for ( int i = 2; i <= n_monomers; ++i ) {
		vector_copy(low_part_centroid(1,i)   ,part_centroid(1,i));
	}
	vector_copy(low_docking_normal   ,docking_normal );
	vector_copy(low_docking_parallel1,docking_parallel1);
	vector_copy(low_docking_parallel2,docking_parallel2);
	vector_copy(low_centroid_bond    ,centroid_bond );
	vector_copy(low_contact_point    ,contact_point );

}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_docking_interface_pack
///
/// @brief set a global flag to indicate repacking only the interface or not
///
/// @detailed
///
/// @param[in]   set_pack_interface - in - setting
///
/// @global_read
///
/// @global_write pack_interface flag in namespace
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_docking_interface_pack( bool set_pack_interface )
{
	using namespace docking_repack_private;
	using namespace runlevel_ns;

	if ( runlevel > standard ) std::cout << "set_docking_interface_pack: " <<
	 L( set_pack_interface ) << std::endl;
	pack_interface = set_pack_interface;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin get_docking_interface_pack
///
/// @brief get a global flag to indicate repacking only the interface or not
///
/// @detailed
///
/// @param[in]
///
/// @global_read pack_interface flag in namespace
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 02/09/2005
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_docking_interface_pack()
{
	using namespace docking_repack_private;

	return pack_interface;
}
//------------------------------------------------------------------------------
//$$$
//$$$      print_docking_interface_pack()
//$$$      bool pack_interface, set_pack_interface
//$$$      common /docking_repack_private/ pack_interface
//$$$      std::cout << "print_docking_interface_pack: " << pack_interface << std::endl;
//$$$      return
//$$$      end

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_set_allow_rottrial
///
/// @brief set which positions are allowd to change side-chain in rotamer_trial
///
/// @detailed check the various norepack flags in dock mode to be sure a given
///      position is supposed to be packed.  norepack flags can be for a
///      particular partner or for cysteines in disulphide bonds
///
/// @global_read various norepack flags
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_set_allow_rottrial()
{
	using namespace docking;
	using namespace docking_repack_private;
	using namespace interface;
	using namespace ligand;
	using namespace misc;
	using namespace param;

//car local
	FArray1D_bool docking_repack_position( MAX_RES()() );

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;

//mj ligand flag
	if ( get_ligand_flag() ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			docking_repack_position(i) = true && !(norepack1);
		}
	} else {  // protein-protein docking
		if ( pack_interface ) { // only interface res
			for ( int i = 1; i <= total_residue; ++i ) {
				docking_repack_position(i) = int_res8(i) && !( norepack1 && i <= part_end(1) )
				 && !( norepack2 && i >= part_begin(2) );
			}
		} else {
			for ( int i = 1; i <= total_residue; ++i ) {
				docking_repack_position(i) = true && !( norepack1 && i <= part_end(1) )
				 && !( norepack2 && i >= part_begin(2) );
			}
		}
	}

	set_allow_rotamer_trials(docking_repack_position,total_residue);
	// pose_rotamer_trials has two ways to control which pos to be rottrial:
	// by_deltaE or by_allow. Currently these two ways can not be combined.
	// So if we have decided earlier to set_by_deltaE, we need to make sure
	// it is not overwritten here. Otherwise, we just use set_by_allow.
	if( get_pose_docking_flag() && !get_rotamer_trials_by_deltaE() )
		set_rt_pos_list_by_allow( docking_repack_position,total_residue, 1 );
}
////////////////////////////////////////////////////////////////////////////////
/// @begin docking_repack
///
/// @brief repack side-chains in docking
///
/// @detailed
///     !!!  This function can be merged with main_repack   !!!  7/11/1
///
///    this function is based on main_repack, but I have changed the
///    allow_repack array which is given to pack to indicate which residues
///    to repack
///
///    residues chosen for repacking are: interface residues (computed in
///    docking_evaluate_all_scores, those residues with centroids within 6
///    Angstroms of another across the interface) and their immediate
///    neighbors
///
///    JJG 5/23
///
///    Aug 2001: incorported the 'norepack' flags which prevent repacking
///    of either interface, for the case when one bound complex is known
///
///    chu 2002-07-09
///    use 8A definition to identify interface residues to be repacked
///
///    extra rotamers, such as rotamers from unbound structures can
///    be included in repacking now.
///
/// @param[in] - rotamers_exist - in - true if including current rotamers in packing
///
/// @global_read
///            flags indicating which partner is allowed to be repacked.
///            flag indicating if only repack the interface.
///
/// @global_write
/// current positions array passed back from packer
///
/// @remarks
///
///@refrence
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
docking_repack( bool const rotamers_exist /* passed to pack */ )
{
	using namespace docking;
	using namespace docking_repack_private;
	using namespace interface;
	using namespace ligand;
	using namespace misc;
	using namespace param;
	using namespace runlevel_ns;

	FArray1D_bool allow_repack( MAX_RES()() );
	int i,j,n,startres,endres;
	FArray2D_int unbound_rot( MAX_CHI, MAX_RES()() );
	FArray2D_float unbound_chi( MAX_CHI, MAX_RES()() );
	//	float dsq;
	//yl flags for pose_from_misc
	bool const fullatom( true );
	bool const ideal_pose( false ); // non-ideal backbone geometry
  bool const coords_init( true );

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;

	if ( !get_fullatom_flag() ) return;
	//	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;

	if ( !rotamers_exist ) initialize_fullcoord_array(Eposition,full_coord,
	 total_residue,res,res_variant);

// set loop limits

//mj check ligand flag
	if ( get_ligand_flag() ) {
		if ( !norepack1 ) {
			startres = 1;
			endres = total_residue;
		} else {
			if ( runlevel > standard ) std::cout <<
			 "something is wrong in docking_repack, nothing to repack" << std::endl;
			startres = 1;
			endres = 0;
		}
	} else {
		if ( !norepack1 ) {
			startres = 1;
			if ( !norepack2 ) {
				endres = total_residue;
			} else {
				endres = part_end(1);
			}
		} else {
			if ( !norepack2 ) {
				startres = part_begin(2);
				endres = part_end(2);
			} else {
				if ( runlevel > standard ) std::cout <<
				 "something is wrong in docking_repack, nothing to repack" << std::endl;
				startres = 1;
				endres = 0;
			}
		}
	}

	if ( !pack_interface ) {

// Pack everything betweeen startres and endres, but that's all!

		for ( i = 1; i < startres; ++i ) { // not between startres and endres
			allow_repack(i) = false;
		}
		for ( i = endres+1; i <= total_residue; ++i ) {
		 // not between startres and endres
			allow_repack(i) = false;
		}

		n = 0;
		for ( i = startres; i <= endres; ++i ) {
		 // must initialize CB coordinates so
			for ( j = 1; j <= 3; ++j ) { // packer can build a neighbor list
				full_coord(j,5,i) = Eposition(j,3,i);
			}
			allow_repack(i) = true;
			++n;
		}
		if ( runlevel > standard ) std::cout << "docking_repack: repacking ALL " <<
		 n << " residues " << !norepack1 << !norepack2 << !pack_interface << std::endl;

	} else { // don't pack everything...

// determine which residues to pack:

// 1-pack all interface residues within repack limits
		n = 0;
//mj check ligand flag
		if ( get_ligand_flag() ) {
			for ( i = startres; i <= endres; ++i ) {
				allow_repack(i) = int_res(i);
				if ( int_res(i) ) ++n;
			}
		} else {
			for ( i = startres; i <= endres; ++i ) {
				allow_repack(i) = int_res8(i);
				if ( int_res8(i) ) ++n;
			}
		}

		for ( i = 1; i < startres; ++i ) { // not between startres and endres
			allow_repack(i) = false;
		}
		for ( i = endres+1; i <= total_residue; ++i ) {
		 // not between startres and endres
			allow_repack(i) = false;
		}
		if ( runlevel > standard ) std::cout <<
		 "docking_repack: " << I( 4, n ) << " interface residues " <<
		 L( 1, !norepack1 ) << L( 1, !norepack2 ) << std::endl;

//     2-pack all interface-residue neighbors
//         n = 0;
//         for ( i = startres; i <= endres; ++i ) { // loop over non-interface residues
//            if ( !int_res(i) && res(i) != 6 ) { // non-iface res,not Gly
//               for ( j = 1; j <= total_residue; ++j ) {
//                  if ( i != j && int_res(j) ) {
//$$$                        distance_bk_sq(full_coord(1,5,i),
//$$$                         full_coord(1,5,j), dsq);
//$$$                        if ( dsq < paircutoff_sq(res(i),res(j)) )
//                     if ( cendist(i,j) <= 36 ) {
//                        allow_repack(i) = true;
//                        ++n;
//                        goto L300; // next i
//                     }
//                  }
//               }            // j res
//            }               // not gly,check distance to varying residues
// L300:;  }                  // i res
//         if ( runlevel > standard ) std::cout << "plus " << n << " neighbors" << std::endl;

	} // end determing residues

// which residues are we packing?
	if ( runlevel >= verbose ) {
		for ( i = 1; i <= total_residue; ++i ) {
			if ( allow_repack(i) ) {
				write_res( std::cout, i );
				std::cout << std::endl;
			}
		}
	}

	std::string packmode( "packrot" );
	bool const make_output_file = false;

	if ( unboundrot ) retrieve_unbound_rotamers(unbound_rot,unbound_chi);

	pose_from_misc(pose,fullatom, ideal_pose, coords_init);

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task( pose );
	Task.set_task(packmode, make_output_file, allow_repack, rotamers_exist, unboundrot, unbound_rot, unbound_chi);
	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();
	pack_rotamers( pose, Task );
	pose.copy_to_misc();

//	if ( runlevel > standard ) std::cout << res(i) << ' ' << allow_repack(i) << std::endl;
//	if ( runlevel > standard ) std::cout << full_coord(1,6,i) << std::endl;

	save_status_info("docking_repack",1,total_residue);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dock_random_orient
///
/// @brief  randomly orient a docking partner
///
/// @detailed   orient the docking ligand in a random direction.
///        by specifying the rotation center, one can effectively randomize
///        either docking partner. --JJG
///
/// @param[in]   npartner - in - which partner to be randomized
/// @param[in]   rotation_center - in - rotation center
///
/// @global_read
///
/// @global_write current position array in misch.h
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dock_random_orient(
	int npartner, // which partner to rotate, 1 or 2?
	FArray1Da_float rotation_center
)
{
	using namespace docking;
	using namespace runlevel_ns;

	rotation_center.dimension( 3 );

//     local
	static FArray2D_float R1( 3, 3 ); // rotation matrix in cartesian frame
	static FArray2D_float R2( 3, 3 ); // rotation matrix in docking frame
	static FArray1D_float const zero( 3, 0.0f ); // translation vector of zero

//-----define transformation
//      docking_output_geometry( std::cout )

	create_random_orientation_matrx(R1,npartner);
	rotation_coord_transform(docking_normal,docking_parallel1,docking_parallel2,
	 R1,R2);
	apply_rigid_body_matrix(zero,R2,rotation_center);

	if ( runlevel > standard ) {
		matrix_name_write( std::cout, R1, "rotation in docking coords:" );
		matrix_name_write( std::cout, R2, "rotation in cartesian coords:" );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dock_slide_into_contact
///
/// @brief
///     slide the ligand along the centroid bond until the docking_vdw
///     score starts to kick in
///     JJG 7/19/2
///
/// @detailed
//       docking_vdw_score is the criterion. if it is equal to zero, the
///      current structure has no contact, so put two partners together
///      by 1.0 A; if it is larger than zero, the structure are in contact
///      so separate two partners by 1.0 A. Only when the vdw score
///      of the last step is larger than zero (in contact) and
///      the current one is zero, we know that two partners are starting to
///      kick in.
///
///
/// @global_read docking_vdw_score in scores.h
///
/// @global_write current position array in misc.h
///
/// @remarks
///        variable fun is not used and should be removed
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dock_slide_into_contact()
{
	using namespace runlevel_ns;
	using namespace scores;

//     local
	static FArray1D_float const zero( 3, 0.0f );
	FArray1D_float T( 3 );
	float d,last_dvdw;

//mj jump out if ligand flag
//      if ( get_ligand_flag() ) {
//         std::cout << "ERROR: Called centroid function with ligand flag:" <<
//         " dock_slide_into_contact in docking_movement.cc" << std::endl;
//         return;
//      }

//mj if ligand flag do jump out in case of contact => some times a ligand sits in a hole and
//mj only a small move is necesarry to get him in an non-bumping position. when sliding away
//mj the hole is hard to find again

	if ( get_ligand_flag() ) {
		mc_global_track::mc_score::score = score4d();
		if ( docking_vdw_score != 0.0 ) return;
	}

//mj end

	d = 0.0;
	last_dvdw = -1.0;
//     go until the last location has bumps, but not the current one
	while ( last_dvdw <= 0.0 || docking_vdw_score != 0.0 ) {

		create_docking_trans_vector(T,d,0.0,0.0);

		if ( runlevel > standard) std::cout << "vdw=" << SS( docking_vdw_score ) <<
		 ", sliding: " << d << " Angstroms" << std::endl;
		apply_rigid_body(T,zero);

		last_dvdw = docking_vdw_score;
		mc_global_track::mc_score::score = score4d();

		if ( docking_vdw_score == 0.0 ) {
			d += 1.0; // slide together
		} else {
			d -= 1.0; // slide apart
		}

	}

	if ( runlevel > standard) std::cout << "vdw=" << SS( docking_vdw_score ) <<
	 ", slid: " << d+1 << " Angstroms" << std::endl;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin create_random_orientation_matrx
///
/// @brief
//        randomly generate psi and theta and create a matrix representing
///       a random orientation.
///
/// @detailed
//       around z-axis, psi --> [0,2pi]
///      around y-axis, theta --> [-pi/2, pi/2]
///
/// @param[out]   R - out - matrix generated
/// @param[in]   npartner - in - which partner to be rotated
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_random_orientation_matrx(
	FArray2Da_float R,
	int npartner // which partner to rotate -- changes order of operations
)
{
	using namespace runlevel_ns;
	using namespace numeric::constants::f;
	using numeric::conversions::degrees;
	using numeric::sin_cos_range;

	R.dimension( 3, 3 );

//     return a random orientation matrix, evenly sampled on the surface
//     of the sphere

//$$$c-----this is a good test for -randomize1
//$$$c     it should produce an even distribution of points around the receptor
//$$$c     for example, by looking at the DOC2 atom
//$$$      static int i = 0, j = 0;
//$$$
//$$$      psi = pi_2 * float(j)/10.0;
//$$$      theta = std::acos(sin_cos_range(1.0-2.0*float(i)/12.0));
//$$$      std::cout << i << ' ' << j << ' ' << psi << ' ' << theta << std::endl;
//$$$      ++j;
//$$$      if ( j == 11 ) ++i;
//$$$      if ( j == 11 ) j = 0;
//$$$      if ( i == 13 ) i = 0;
//$$$c      psi = pi;
//$$$c      theta = pi_over_2;
//$$$c-----end test code

	float psi = pi_2 * ran3(); // returns 0:2pi uniformly distributed
	float theta = std::acos( sin_cos_range( 1.0 - 2.0*ran3() ) ); // returns 0:pi distributed on the sphere

	create_orientation_matrix(R,psi,theta,npartner);

	if ( runlevel > standard ) {
		std::cout << "Random orientation:" << std::endl;
		std::cout << "  azimuth " << degrees( psi ) << std::endl;
		std::cout << "  declination " << degrees( theta ) << std::endl;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin create_orientation_matrix
///
/// @brief given psi and theta,create an matrix representing rotation
///       around y and z axis to sample the surface of the whole sphere
///
/// @detailed
///     for npartner=1:
///     return a rotation matrix R to rotate by (pi/2-theta) around the
///     y-axis then by -psi around the z-axis.  in this manner, a point at
///     (1,0,0) is rotated to any location on the surface of the sphere
///     for npartner=2:
///     just the opposite: the z-rotation first of 0-2pi, then the y-rotation
///     of -pi/2 to pi/2, to move any evenly-distributed location on the
///     sphere to (-1,0,0)
///     revised jjg 3/02
///
/// @param[out]   R - out - matrix generated
/// @param[out]   psi - out - radian angle rotation around z-axis
/// @param[out]   theta - out - radian angle rotation around y-axis
/// @param[out]   npartner - out - which partner to be rotated
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_orientation_matrix(
	FArray2Da_float R,
	float & psi,
	float & theta,
	int npartner // which partner to rotate -- changes order of operations
)
{
	using namespace numeric::constants::f;

	R.dimension( 3, 3 );

	FArray2D_float psiR( 3, 3 );
	FArray2D_float thetaR( 3, 3 );

	zrotation( psiR, -psi );
	yrotation( thetaR, pi_over_2 - theta ); // pi/2 - theta

	if ( npartner == 1 ) {
		mat_multiply3( thetaR, psiR, R ); // tempR = psiR * thetaR
	} else if ( npartner == 2 ) {
		mat_multiply3( psiR, thetaR, R ); // tempR = thetaR * psiR
	} else {
		std::cout << "wacky partner choice" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin create_three_rotations_matrix
///
/// @brief creat an overall rotation matrix around three axes
///
/// @detailed
///     create an orientation matrix R which performs a rotation of a
///     radians around the x-axis, b radians around the y-axis, and
///     z-radians around the z-axis - JJG
///
/// @param[out]   R - out - the overall rotation matrix generated
/// @param[in]   a - in - radian angle to be rotated around x-axis
/// @param[in]   b - in - radian angle to be rotated around y-axis
/// @param[in]   c - in - radian angle to be rotated around z-axis
///
/// @global_read
///
/// @global_write
///
/// @remarks
///      It seems to me that if the order of three rotations change, the R matrix
///      will be different.
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_three_rotations_matrix(
	FArray2Da_float R,
	float & a,
	float & b,
	float & c
)
{
	R.dimension( 3, 3 );

//     local
	FArray2D_float Rx( 3, 3 );
	FArray2D_float Ry( 3, 3 );
	FArray2D_float Rz( 3, 3 );
	FArray2D_float Rxy( 3, 3 );

	xrotation(Rx,a);
	yrotation(Ry,b);
	zrotation(Rz,c);

	mat_multiply3(Rx,Ry,Rxy); // Rxy = Ry * Rx
	mat_multiply3(Rxy,Rz,R); // R = Rz * Ry * Rx
}

//------------------------------------------------------------------------------
// Single-axis rotation primitives

////////////////////////////////////////////////////////////////////////////////
/// @begin xrotation
///
/// @brief return a rotation matrix R to rotate alpha radians about the x-axis
///
/// @detailed
///
/// @param[out]   R - out - rotation matrix generated
/// @param[in]  alpha - in - alpha radians to be rotated about x-axis
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
xrotation(
	FArray2Da_float R,
	float alpha
)
{
	R.dimension( 3, 3 );

//     return a rotation matrix R to rotate alpha radians about the x-axis

	float cosA = std::cos(alpha);
	float sinA = std::sin(alpha);

	R(1,1) = 1.0;
	R(1,2) = 0.0;
	R(1,3) = 0.0;

	R(2,1) = 0.0;
	R(2,2) = cosA;
	R(2,3) = -sinA;

	R(3,1) = 0.0;
	R(3,2) = sinA;
	R(3,3) = cosA;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin yrotation
///
/// @brief return a rotation matrix R to rotate alpha radians about the y-axis
///
/// @detailed
///
/// @param[out]   R - out - rotation matrix generated
/// @param[in]   alpha - in - alpha radians to be rotated around y-axis
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
yrotation(
	FArray2Da_float R,
	float alpha
)
{
	R.dimension( 3, 3 );

//     return a rotation matrix R to rotate alpha radians about the y-axis

	float cosA = std::cos(alpha);
	float sinA = std::sin(alpha);

	R(1,1) = cosA;
	R(1,2) = 0.0;
	R(1,3) = sinA;

	R(2,1) = 0.0;
	R(2,2) = 1.0;
	R(2,3) = 0.0;

	R(3,1) = -sinA;
	R(3,2) = 0.0;
	R(3,3) = cosA;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin zrotation
///
/// @brief return a rotation matrix R to rotate alpha radians about the z-axis
///
/// @detailed
///
/// @param[out]   R - out - rotation maxtrix generated
/// @param[in]   alpha - in - alpha radians to be rotated around z-axis
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/25/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
zrotation(
	FArray2Da_float R,
	float alpha
)
{
	R.dimension( 3, 3 );

//     return a rotation matrix R to rotate alpha radians about the z-axis

	float cosA = std::cos(alpha);
	float sinA = std::sin(alpha);

	R(1,1) = cosA;
	R(1,2) = -sinA;
	R(1,3) = 0.0;

	R(2,1) = sinA;
	R(2,2) = cosA;
	R(2,3) = 0.0;

	R(3,1) = 0.0;
	R(3,2) = 0.0;
	R(3,3) = 1.0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_axis_spin
///
/// @brief
///     spin along the centroid-bond axis to randomize the starting
///     orientation relative to each other
///
/// @detailed
///
/// @global_read
/// docking_best block in docking.h
/// standard,runlevel (in runlevel.h) - documentation is runlevel dependent
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ora 08/20/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_axis_spin()
{
//     spin along the centroid-bond axis to randomize the starting
//     orientation relative to each other

	using namespace docking;
	using namespace runlevel_ns;
	using namespace numeric::constants::f;
	using numeric::conversions::degrees;

//   local
	float delta;
	static FArray2D_float R( 3, 3 ); // rotation matrix
	static FArray1D_float const zero( 3, 0.0f ); // translation vector of zero

//   Prepare rotation and translation intermediates

	delta = pi_2 * ran3(); // returns 0:2pi uniformly distributed

	if ( runlevel > standard ) std::cout << "Spinning around axis by " <<
	 SS( degrees( delta ) ) << std::endl;

	setup_rotation_matrix(0.0,0.0,delta,best_docking_normal,
	 best_docking_parallel1,best_docking_parallel2,R); // -> R

	apply_rigid_body_matrix( zero, R, best_part_centroid(1,2) );

//   Echo motions

//      std::cout << " translating: ";
//      vector_write( std::cout,translation)

//      std::cout << "and rotating: ";
//      std::cout << "spin=" << F( 8, 4, spin ) <<
//       "(" << F( 5, 1, degrees( spin ) ) << "d); tilt=" <<
//			 F( 8, 4, tilt ) << "(" << F( 5, 1, degrees( tilt ) ) << "d)" <<
//			 "in direction" << F( 8, 4, tilt_direction ) << "(" <<
//			 F( 5, 1, degrees( tilt_direction ) ) << "d)" << std::endl;
}
