// Rosetta Headers
#include "domins_fullatom.h"
#include "cenlist.h"
#include "current_pose.h"
#include "docking_minimize.h"
#include "docking_movement.h"
#include "docking_score.h"
#include "DomainInsertionMode.h"
#include "domins_ns.h"
#include "domins_score.h"
#include "files_paths.h"
#include "jumping_loops.h"
#include "jumping_util.h"
#include "jumping_refold.h"
#include "loop_relax.h"
#include "minimize.h"
#include "misc.h"
#include "nblist.h"
#include "pose.h"
#include "pose_design.h"
#include "pose_docking.h"
#include "pose_io.h"
#include "pose_loops.h"
#include "pose_vdw.h"
#include "prof.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "smallmove.h"
#include "torsion_bbmove_trials.h"

// ObexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <fstream>

//////////////////////////////////////////////////////////////////////////////
/// @begin insertion_fullatom()
///
/// @brief
///		does all the fullatom portions of domain_insertion mode
///   adds sidechains, sets up the fullatom variables
///		repacks and/or does a domins fast_relax
///
/// @detailed
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified August 08 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_fullatom_protocol(
		pose_ns::Pose & pose,
		pose_ns::Pose & save_fullatom_pose,
		bool const input
		)
{
	using namespace pose_ns;

	// params
	float const temperature ( 0.8 );
	bool use_nblist ( true );

	// ensure that the information in pose is the same as that in misc
	assert(misc_in_sync(pose));

	// setup
	score_set_try_rotamers(false);
	set_smallmove_size(2.0,2.0,3.0); // helix,strand,other
	pose_setup_minimizer( 1.0 );

	// monte_carlo object
	//Score_weight_map weight_map (score12di);
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
	Monte_carlo mc (pose, weight_map, temperature);

	///////////////////////////////////////////////////////
	// packer params:
	pose_setup_packer();


	///////////////////////////////////////////////////////
	// recover the sidechains from the prepacked version of
	// the insert and parent proteins
	if(!input) {
		// recovers sidechains from the prepacked version
		recover_all_sidechains( pose );
	} else {
		// adds on sidechains
		pose.recover_sidechain( save_fullatom_pose );
	}

	// interface and loop repack
	FArray1D_bool allow_repack( pose.total_residue() );
	domins_select_loop_and_interface( pose, use_nblist, allow_repack );
	pose.repack( allow_repack, true );
	mc.boltzmann( pose, "ljramp_repack" );

	if ( !domins_ns::repack_relax ) {
		PROF_START( prof::DOMINS_RELAX );
		insertion_fast_relax(pose, weight_map, mc);
		PROF_STOP( prof::DOMINS_RELAX );
		domins_select_loop_and_interface( pose, use_nblist, allow_repack );
		pose.repack( allow_repack, true );
		mc.boltzmann( pose, "ljramp_repack" );
	}

	// recover low
	pose = mc.low_pose();

	mc.show_counters();
	prof::show();

	// now score final structure
	pose = mc.low_pose();
	score_set_evaluate_all_terms(true);
	pose.score(weight_map);
	score_set_evaluate_all_terms(false);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin insertion_fast_relax()
///
/// @brief
///		initial repack with lowered repulsive
///		slowly ramps up the repulsives as it does a rigid body mcm minimization
///   then it does a small_moves and shear_moves on the linker loops
///   finally it ensures that the loops are closed and runs a repack
///
/// @detailed
///		couple of cycles of:
///		repack, minimize, small trials with rotamer-trials on
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified August 08 2006
////////////////////////////////////////////////////////////////////////////////
void
insertion_fast_relax(
		pose_ns::Pose & pose,
		pose_ns::Score_weight_map & weight_map,
		pose_ns::Monte_carlo & mc
		)
{
	using namespace pose_ns;
	using namespace runlevel_ns;
	using namespace domins_ns;

	// params
	float const start_rep_weight ( 0.02 );
	float const end_rep_weight ( 1.0 );
	int const lj_ramp_cycles ( benchmark ? 2 : 6 );
	int lj_ramp_inner_cycles ( benchmark ? 2 : 10 );
	int const nmoves ( 1 );
	bool const use_nblist ( true );
	bool const include_current ( true );
	int small_accepted(0), shear_accepted(0), n_accepted(0), n_trial(0); // diagnostic

	FArray1D_bool allow_repack( pose.total_residue() );

	//////////////////////////////////////
	// setup ramping of the repulsive term
	// geometric
	// rep_delta^(lj_ramp_cycles-1) = end_rep_weight / start_rep_weight

	assert ( lj_ramp_cycles > 1 );
	float const rep_delta
		( std::exp( std::log( end_rep_weight / start_rep_weight ) /
								(lj_ramp_cycles-1) ) );
	float rep_weight = start_rep_weight / rep_delta;

	// lj_ramp_cycles
	std::cout << "lj_ramp_cycles: " << lj_ramp_cycles << std::endl;
	for ( int n=1; n<= lj_ramp_cycles; ++n ) {
		std::cout << "ramp_cycle: " << n << std::endl;

		rep_weight *= rep_delta;
		weight_map.set_weight( FA_REP, rep_weight );
		mc.set_weight_map( weight_map );

		if ( n == 2 || n == 3 ) lj_ramp_inner_cycles = 5;
		if ( n == 4 || n == 5 ) lj_ramp_inner_cycles = 7;

		// lower minimizer tolerance on final pass
		if ( n == lj_ramp_cycles ) {
			minimize_set_tolerance(0.0001);
			lj_ramp_inner_cycles = (benchmark ? 2 : 10);
		}

		// recover low
		pose = mc.low_pose();

		std::cout << "lj_ramp_inner_cycles: " << lj_ramp_inner_cycles << std::endl;
		for ( int m=1; m<=lj_ramp_inner_cycles; ++m) {
			n_trial++;

			if (domins_ns::rb_relax) {
				std::cout << "rb_relax" << std::cout;
				PROF_START ( prof::DOMINS_MCM );
				rigid_body_mcm_protocol( pose, n_accepted, n_trial, mc );
				PROF_STOP ( prof::DOMINS_MCM );
				show_decoy_status(pose, "rigid_body");
				PROF_START ( prof::DOMINS_FLOP );
				domins_insertion_flop(pose, lj_ramp_inner_cycles, weight_map, mc);
				PROF_STOP ( prof::DOMINS_FLOP );
				show_decoy_status(pose, "insertion_flop");
			}
			PROF_START ( prof::DOMINS_SM );
			loop_refinement_move(pose, weight_map, nmoves, "small_move");
			PROF_STOP ( prof::DOMINS_SM );
			if( mc.boltzmann( pose, "ljramp_small_min" )) {small_accepted++; }
			show_decoy_status(pose, "small_move");
			PROF_START ( prof::DOMINS_SHM );
			loop_refinement_move(pose, weight_map, nmoves, "shear_move");
			PROF_STOP ( prof::DOMINS_SHM );
			if( mc.boltzmann( pose, "ljramp_shear_min" )) {shear_accepted++; }
			show_decoy_status(pose, "shear_move");

			for( Loops::const_iterator it=loops.begin(), it_end=loops.end();
					it != it_end; ++it ) {
				int const loop_begin = it->start();
				int const loop_end = it->stop();
				int const cutpoint = it->cut();
				ccd_moves(5, pose, loop_begin, loop_end, cutpoint);
			}

			//repack and score
			if(m % 10 == 0 || m == lj_ramp_inner_cycles) {
				domins_select_loop_and_interface( pose, use_nblist, allow_repack );
				std::cout << "cycle: " << m << std::endl;
				pose.repack( allow_repack, include_current );
				pose.score( weight_map );
				mc.boltzmann( pose, "small/shear_repack" );
			}
		} // inner cycles

	} // outer cycles
	std::cout << "fast_relax -- accepted/trial/ratio:\n"
						<< small_accepted<< "/" << n_trial << "/"
						<< float(small_accepted)/float(n_trial) << "for small moves\n"
						<< shear_accepted << "/" << n_trial << "/"
						<< float(shear_accepted)/float(n_trial) << "for shear moves"
						<< std::endl;
}

//////////////////////////////////////////////////////////////////////////////
// Monica: copied pose_docking_monte_carlo_minimize and customized for domins mode
void
rigid_body_mcm_protocol(
		pose_ns::Pose & docking_pose,
		int & n_accepted,
		int const cycle,
		pose_ns::Monte_carlo & mc
		)
{
	// MC move
	const float trans_magnitude = 0.1; /* angstrom */
	const float rot_magnitude = 5.0; /* degree */
	// rb minimization
	const float loose_func_tol = 1.0; /* absolute tolerance */
	const float tight_func_tol = 0.02; /*absolute tolerance */
	float func_tol = 0.0;
	const std::string min_type = "dfpmin_atol";
	// monte carlo
	const float minimization_threshold = 15.0; /* score unit */
	FArray1D_bool interface_repack( docking_pose.total_residue() );
	// scorefxn to minimize
	pose_ns::Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score10d_min );

	std::cout << std::endl;
	std::cout << "*******rigid_body_mcm*******************" << std::endl;

	using namespace pose_ns;

	// set up a monte_carlo object
	mc.set_weight_map( weight_map );

	// start MCM
	bool save_docking_interface_pack = get_docking_interface_pack();
	const bool interface_only = true;
	const bool include_current_sidechain = true;

	pose_docking_set_rb_center( docking_pose );
	pose_docking_gaussian_rigid_body_move( docking_pose, 1/*jump number*/, trans_magnitude, rot_magnitude );


	if ( cycle % 10 == 0 ) {
		// repacking with rtmin
		std::cout << "Repacking with rtmin....." << std::endl;
		docking_pose.set_allow_chi_move( true );
		pose_docking_repack( docking_pose, interface_only, include_current_sidechain );
		set_docking_interface_pack( interface_only );
		score_set_try_rotamers( true );
		score_set_minimize_rot( true );
		docking_pose.score( weight_map );
		score_set_minimize_rot( false );
		score_set_try_rotamers( false );
	} else {
		// rotamer_trials for the whole protein
		std::cout << "Rotamer_trials for the whole protein..." << std::endl;
		docking_pose.set_allow_chi_move(true);
		set_docking_interface_pack( !interface_only );
		score_set_try_rotamers( true );
		docking_pose.score( weight_map );
		score_set_try_rotamers( false );
	}

	set_docking_interface_pack( save_docking_interface_pack );

	const float current_score ( docking_pose.get_0D_score( SCORE ) );
	const float best_score ( mc.best_pose().get_0D_score( SCORE ) );
	if (cycle < 50) { func_tol = loose_func_tol; } else { func_tol = tight_func_tol; }

	if ( current_score - best_score < minimization_threshold ) {
		pose_docking_minimize_rb_position( docking_pose, min_type, weight_map, func_tol );
	}

	if (mc.boltzmann( docking_pose, "rigid_body" ) ) { ++n_accepted; }

	mc.show_scores();

	std::cout << "rigid_body_mcm -- " << n_accepted << " out of " << cycle
						<< " monte carlo cycles accepted " << std::endl;

	docking_pose = mc.low_pose();

	std::cout << "******rigid_body_mcm*******************" << std::endl;
	std::cout << std::endl;
}

void
loop_refinement_move(
		pose_ns::Pose & pose,
		pose_ns::Score_weight_map & weight_map,
		int const nmoves,
		std::string const move_type
		)
{
	using namespace pose_ns;
	using namespace domins_ns;

	Loops one_loop;
	one_loop.add_loop( loops.one_random_loop() );
	pose_loops_set_allow_bb_move( pose, one_loop );
	if ( move_type == "small_move" ) {
		pose_small_moves(pose, nmoves);
	} else if ( move_type == "shear_move" ) {
		pose_shear_moves(pose, nmoves);
	}
	pose_ccd_close_loops( pose, one_loop);
	pose_loops_set_allow_bb_move( pose, loops );
	pose.main_minimize( weight_map, "dfpmin" );
}

void
domins_select_loop_and_interface(
		pose_ns::Pose & pose,
		bool const use_nblist,
		FArray1D_bool & allow_repack
		)
{
	using namespace domins_ns;

	FArray1D_bool interface_repack( pose.total_residue() ), loop_repack( pose.total_residue() );
	pose_docking_calc_interface( pose, interface_repack );
	pose_loops_select_loop_residues( pose, loops, use_nblist, loop_repack );

	if ( runlevel_ns::runlevel > runlevel_ns::yap ) {
		std::cout << "sele interface, resi ";
		for (int i=1; i<pose.total_residue(); ++i ) {
			if ( interface_repack(i) == true ) {
				std::cout << i << '+';
			}
		}
		std::cout << std::endl;
		std::cout << "sele loops, resi ";
		for (int i=1; i<pose.total_residue(); ++i ) {
			if ( loop_repack(i) == true ) {
				std::cout << i << '+';
			}
		}
		std::cout << std::endl;
	}

	for (int i=1; i<pose.total_residue(); ++i ) {
		if (interface_repack(i) == true || loop_repack(i) == true) {
			allow_repack(i) = true;
		} else {
			allow_repack(i) = false;
		}
	}
	pose.set_allow_chi_move( allow_repack );
}

// pose namespace functions
// stored in the namespace in order to be able to access the private
// functions of pose
void
recover_all_sidechains(
		pose_ns::Pose & pose
		)
{
	using namespace domins_ns;
	using namespace pose_ns;

	if ( !prepack_read ) {
		domins_read_prepacked();

		assert( pose.fullatom() && insert_ppk_pose.fullatom() && host_ppk_pose.fullatom() );
		assert( pose.total_residue() == (insert_ppk_pose.total_residue() + host_ppk_pose.total_residue() ) );
	}

	FArray3D_float insert_full_coord( insert_ppk_pose.full_coord() );
	FArray3D_float host_full_coord( host_ppk_pose.full_coord() );
	assert( check_xyz( insert_full_coord ) );
	assert( check_xyz( host_full_coord ) );

	for ( int i=1; i<= pose.total_residue(); ++i ) {
		std::string name = spliced_protein_res_info.at(i).original_protein_type;
		int m = spliced_protein_res_info.at(i).original_protein_res_num;
		if (name == "host_protein") {
			assert( pose.res(i) == host_ppk_pose.res(m) );
			assert( pose.res_variant(i) == host_ppk_pose.res_variant(m) );
			pose.copy_sidechain( i, pose.res(i), pose.res_variant(i), host_full_coord(1,1,m) );
		} else if (name == "insert_protein") {
			assert( pose.res(i) == insert_ppk_pose.res(m) );
			assert( pose.res_variant(i) == insert_ppk_pose.res_variant(m) );
			pose.copy_sidechain( i, pose.res(i), pose.res_variant(i), insert_full_coord(1,1,m) );
		}
	}
}

// read in the prepacked structures to get the sidechains from
void
domins_read_prepacked()
{
	using namespace domins_ns;
	using namespace files_paths;
	using namespace pose_ns;

	std::string filename;

	std::string hostp = params["hostp"].c_str();
	if ( has_suffix( hostp, ".pdb" ) ) hostp.erase( hostp.length() - 4 );
	filename = start_path + hostp + ".ppk.pdb";
	pose_from_pdb(host_ppk_pose, filename, true, false);
	std::string insertp = params["insertp"].c_str();
	if ( has_suffix( insertp, ".pdb" ) ) insertp.erase( insertp.length() - 4 );
	filename = start_path + insertp + ".ppk.pdb";
	pose_from_pdb(insert_ppk_pose, filename, true, false);
}

