// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 10040 $
//  $Date: 2006-08-29 00:01:52 -0700 (Tue, 29 Aug 2006) $
//  $Author: rhiju $

#include "dummy_model.h"
#include "after_opts.h"
#include "files_paths.h"
#include "input_pdb.h"
#include "misc.h"
#include "param.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzMatrix.io.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzVector.io.hh>
#include <numeric/xyz.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <list>
#include <vector>

#define A_BIG_FLOAT 9.0e10

/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
//
//
// Following routines enable Rosetta to compare its decoys to
//   a model of dummy atoms, as could be obtained from cryo-electron
//   microscopy or from a solution x-ray scattering experiment.
//
// The model needs to be in PDB format with the atoms at C-alpha positions.
// Its read in with the flag
//          -dummy_model <dummy_model.pdb>
//
//
// Its OK if the model has fewer or more C-alphas than the Rosetta decoy.
//
// The comparison of decoys to the model is with the normalized
// spatial discrepancy (NSD) measure. See "Automated matching of high-
// and low-resolution structural models", Kozin and Svergun, J. Appl. Cryst.,
// 2001 34:33-41.
//
// This NSD measure has the property that it is zero for identical models,
// and around 1 for models that are consistent given the "fineness"
// (resolution of model). Its also symmetric.
//
// I've also tried to include a score term to help drive ab initio
// simulations towards decoys with similar shape to the dummy model.
// Activate with:
//     -dummy_model_score
//
// Its currently a silly power law of the form -depth/nsd^power. By
// default the depth and power are 2.0 and 4, but can be changed with:
//
//  -dummy_model_depth <float> -dummy_model_power <float>
//
// (In calculating the score, if nsd is less than 0.2, its set to 0.2,
//  to prevent an infinite bonus.)
//
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////

// Place to store dummy model, read in at beginning of rosetta run,
// in initialize_query.
namespace dummy_model{
	using namespace param;
  int total_dummy_atoms = 0;
  FArray2D_float dummy_atom(3, MAX_RES(), 0.0f);
} //dummy_model


/////////////////////////////////////////////////////////////////////
void
eval_dummy_model_score( float & dummy_model_score, float & nsd){
	dummy_model_score = 0.0;
	if (!get_dummy_model_flag()) return;
	if (!get_dummy_model_score_flag()) return;
	read_dummy_model(); //returns if already done.

	float NSD = calc_NSD_to_dummy_model();
	nsd = NSD;

	float depth = get_dummy_model_depth();
	double power  = get_dummy_model_power();
	// The power law thing below dips to minus infinity as you
	// go to zero -- should cut it off a some typical NSD between a
	// protein and its cryoEM model, here estimated to be 0.2.
	if (NSD < 0.2) NSD = 0.2;
	dummy_model_score = static_cast< float >( -1.0 * depth * std::pow( static_cast< double > (1.0/ NSD), power) );
	return;
}

/////////////////////////////////////////////////////////////////////
float
calc_NSD_to_dummy_model() {
	using namespace misc;
	using namespace dummy_model;
	if (!get_dummy_model_flag()) return 0.0;

	FArray2D_float decoy_calpha_temp( 3, total_residue, 0.0f);
  FArray2D_float decoy_calpha     ( 3, total_residue, 0.0f);
  for (int i = 1; i <= total_residue; i++){
    for(int j = 1; j <= 3; j++){
      decoy_calpha_temp( j, i) = Eposition( j, 2, i);
    }
  }


	//First use the moment-of-inertia tensor to align to x,y,z axes.
	align_to_axes_via_inertia_tensor( decoy_calpha_temp, total_residue, decoy_calpha );
	//Silly consistency check -- should get rotation matrix to be identity.
	//	align_to_axes_via_inertia_tensor( decoy_calpha, total_residue, decoy_calpha_temp );
	//	std::cout << "DUMMY_MODEL: " << decoy_calpha(1, total_residue) <<  " " << decoy_calpha_temp(1, total_residue) << std::endl;


	// Just need to calculate fineness once -- its the average distance
	// to the nearest neighbor dummy atom, and shouldn't change (much)
	// for the decoy during the Rosetta run.
	static bool init = { false };
	static float fineness_dummy = {1.0};
	static float fineness_decoy = {1.0};

	if (!init){
		init = true;
		fineness_dummy = calculate_fineness( dummy_atom  , total_dummy_atoms);
		fineness_decoy = calculate_fineness( decoy_calpha, total_residue    );
		//		std::cout << "DUMMY_MODEL Fineness dummy " << fineness_dummy << std::endl;
		//		std::cout << "DUMMY_MODEL Fineness decoy " << fineness_decoy << std::endl;
	}


	// Momentum of inertia based alignment doesn't totally decide
	// on the orientation.
	//Check the four possible flips.
	float NSD_best = A_BIG_FLOAT;
	for (int i = -1; i <= 1; i += 2){
		for (int res = 1; res <= total_residue; res++) decoy_calpha_temp(1, res) = decoy_calpha(1, res) * i;

		for (int j = -1; j <= 1; j += 2){
			for (int res = 1; res <= total_residue; res++) decoy_calpha_temp(2, res) = decoy_calpha(2, res) * j;

			int k = i*j;
			assert (i*j*k == 1); //can't violate parity.
			for (int res = 1; res <= total_residue; res++) decoy_calpha_temp(3, res) = decoy_calpha(3, res) * k;

			float NSD_temp =
				calculate_NSD( dummy_atom, decoy_calpha_temp, total_dummy_atoms, total_residue, fineness_dummy, fineness_decoy );
			//			std::cout << "DUMMY_MODEL NSD " << i << " " << j << " " << k << ": " << NSD_temp << std::endl;
			if (NSD_temp < NSD_best){
				NSD_best = NSD_temp;
			}

		}
	}

	return NSD_best;
}

/////////////////////////////////////////////////////////////////////
float
calculate_fineness( FArray2D_float const pos, int total_residue){
	using namespace numeric;

	FArray2D_float dist_matrix( total_residue, total_residue, 0.0);
	float temp = 0.0;
	for (int i = 1; i <= total_residue; i++){
		for (int j = i+1; j <= total_residue; j++){
			xyzVector_float coord1( &pos(1,i) );
			xyzVector_float coord2( &pos(1,j) );
//Svergun defines this based on distance, not distance_squared.
			temp = distance( coord1, coord2 );
			dist_matrix(i, j) = temp;
			dist_matrix(j, i) = temp;
		}
	}

	float avg_distance_to_nearest_neighbor = 0.0;
	for (int i = 1; i <= total_residue; i++){
		float dist_min = A_BIG_FLOAT;
		for (int j = 1; j <= total_residue; j++){
			if (i==j) continue;
			if (dist_matrix(i,j) < dist_min) dist_min = dist_matrix(i,j);
		}
		avg_distance_to_nearest_neighbor += dist_min;
	}
	avg_distance_to_nearest_neighbor /= total_residue;

	return avg_distance_to_nearest_neighbor;

}

/////////////////////////////////////////////////////////////////////
float
calculate_NSD(
							FArray2D_float const pos1,
							FArray2D_float const pos2,
							int const total_residue1,
							int const total_residue2
)
{
	float fineness1 = calculate_fineness( pos1, total_residue1 );
	float fineness2 = calculate_fineness( pos2, total_residue2 );
	return calculate_NSD( pos1, pos2, total_residue1, total_residue2, fineness1, fineness2);
}

/////////////////////////////////////////////////////////////////////
float
calculate_NSD(
							FArray2D_float const pos1,
							FArray2D_float const pos2,
							int const total_residue1,
							int const total_residue2,
							float const fineness1,
							float const fineness2
){
	using namespace numeric;

	FArray2D_float dist_matrix( total_residue1, total_residue2, 0.0);
	for (int i = 1; i <= total_residue1; i++){
		for (int j = 1; j <= total_residue2; j++){
			xyzVector_float coord1( &pos1(1,i) );
			xyzVector_float coord2( &pos2(1,j) );
			dist_matrix(i, j) = distance_squared( coord1, coord2 );
		}
	}

	float	NSD_1_to_2 = 0.0;
	for (int i = 1; i <= total_residue1; i++){
		float dist_min = A_BIG_FLOAT;
		for (int j = 1; j <= total_residue2; j++){
			if (dist_matrix(i, j) < dist_min) dist_min = dist_matrix(i, j);
		}
		NSD_1_to_2 += dist_min;
	}
	NSD_1_to_2 *= 1.0 / (total_residue1 * fineness2 * fineness2);


	float	NSD_2_to_1 = 0.0;
	for (int j = 1; j <= total_residue2; j++){
		float dist_min = A_BIG_FLOAT;
		for (int i = 1; i <= total_residue1; i++){
			if (dist_matrix(i, j) < dist_min) dist_min = dist_matrix(i, j);
		}
		NSD_2_to_1 += dist_min;
	}
	NSD_2_to_1 *= 1.0 / (total_residue2 * fineness1 * fineness1);

	float NSD = 0.5 * (NSD_1_to_2 + NSD_2_to_1);

	return NSD;
}

/////////////////////////////////////////////////////////////////////
void
align_to_axes_via_inertia_tensor(
																 FArray2D_float const pos,
																 int total_residue,
																 FArray2D_float & pos_align
){
	using namespace numeric;
	FArray2D_float pos_recenter( 3, total_residue, 0.0);
	FArray2D_double m_moment( 3, 3 );

	for (int j = 1; j <= 3; j++){
		float pos_center = 0.0;
		for (int i = 1; i <= total_residue; i++) pos_center += pos( j, i);
		pos_center /= float( total_residue );
		for (int i = 1; i <= total_residue; i++) pos_recenter(j, i) = pos( j, i) - pos_center;
	}

	// Generate inertia tensor:
	for ( int k = 1; k <= 3; ++k ) {
		for ( int j = 1; j <= 3; ++j ) {
			float temp1 = 0.0;
			for ( int i = 1; i <= total_residue; ++i ) {
				temp1 += pos_recenter(j, i) * pos_recenter(k, i);
			}
			m_moment(k,j) = temp1;
		}
	}

	xyzMatrix_double xyz_rr_moment( xyzMatrix_double::cols( &m_moment( 1,1 ) ) );
	xyzVector_double xyz_w_w;
	xyzMatrix_double xyz_eVec;

	xyz_w_w = eigenvector_jacobi( xyz_rr_moment, 1E-9, xyz_eVec );

	//	std::cout << "DUMMY_MODEL xyz_ww: " << xyz_w_w << std::endl;
	//	std::cout << "DUMMY_MODEL xyz_eVec:\n " << xyz_eVec;
	xyz_eVec.transpose();

	//Shoot, eigenvectors are not ordered by eigenvalue. That's annoying. Following copied from orient_rms.cc
	FArray1D_int sort(3, 0);
	for ( int i = 1; i <= 3; ++i ) {
		sort(i) = i;
	}
	if ( xyz_w_w(1) < xyz_w_w(2) ) {
		sort(2) = 1;
		sort(1) = 2;
	}
	if ( xyz_w_w(sort(2)) < xyz_w_w(3) ) {
		sort(3) = sort(2);
		sort(2) = 3;
		if ( xyz_w_w(sort(1)) < xyz_w_w(3) ) {
			sort(2) = sort(1);
			sort(1) = 3;
		}
	}

	xyzMatrix_double xyz_eVec_reorder;
	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			xyz_eVec_reorder( i, j )= xyz_eVec( sort(i), j);
		}
	}

	float determinant = xyz_eVec.det() > 0 ? 1 : -1;
	//	std::cout << "DUMMY_MODEL Determinant: " << xyz_eVec.det() << " "  << determinant << std::endl;

	xyzVector_double new_coord;
	for (int i = 1; i <= total_residue; i++) {
		xyzVector_double old_coord( &pos_recenter( 1, i) );
		new_coord = xyz_eVec_reorder * old_coord;
		for ( int j=1; j<=3; j++) pos_align(j,i) = new_coord[ j-1 ];
		pos_align(3,i) *= determinant;
	}


}


/////////////////////////////////////////////////////////////////////
void
read_dummy_model( ) {
  using namespace misc;
  using namespace dummy_model;
	using namespace files_paths;

	static bool init = {false};

	if (init) return;
	init = true;

	if (!get_dummy_model_flag()) return;

  // Assume dummy model is in PDB format, with atoms at CA positions.
  bool fail = true;
	std::string filename;
	stringafteroption("dummy_model","hey",filename);
	utility::io::izstream dummy_model_stream( filename );


	if ( !dummy_model_stream ) {
		std::cerr << "Couldn't find file " << filename << " for reading in dummy model." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return;
	}

	std::cout << "Ignore the warning message that follows .. dummy model just needs C-alpha's." << std::endl;
	bool const skip_missing_orig  = skip_missing;
	bool const allow_missing_orig = allow_missing;
	skip_missing  = false;
	allow_missing = true;
  input_pdb( dummy_model_stream, false /* seq_defined */, false /* fullatom */, fail );
	skip_missing  = skip_missing_orig;
	allow_missing = allow_missing_orig;
  total_dummy_atoms = total_residue;
	std::cout << "Dummy model read in: " << filename << std::endl;
	std::cout << " Number of dummy atoms: " << total_dummy_atoms <<  std::endl;


  FArray2D_float dummy_atom_temp(3, total_dummy_atoms, 0.0f);
  for (int i = 1; i <= total_dummy_atoms; i++){
    for(int j = 1; j <= 3; j++){
      dummy_atom_temp( j, i) = Eposition( j, 2, i); //C-alphas
    }
  }

	//Now we know how big to make this dummy_atom coordinate array.
	dummy_atom.redimension( 3, total_dummy_atoms );

	align_to_axes_via_inertia_tensor( dummy_atom_temp, total_dummy_atoms, dummy_atom );
	//Check -- did we rotate properly?
//	align_to_axes_via_inertia_tensor( dummy_atom, total_dummy_atoms, dummy_atom_temp );
}

/////////////////////////////////////////////////////////////////////
bool
get_dummy_model_flag() {
  static bool init = {false};
  static bool dummy_model_flag;

  if (!init) {
    dummy_model_flag = truefalseoption("dummy_model");
    init = true;
  }

  return dummy_model_flag;
}
/////////////////////////////////////////////////////////////////////
bool
get_dummy_model_score_flag() {
  static bool init = {false};
  static bool dummy_model_score_flag;

  if (!init) {
    dummy_model_score_flag = truefalseoption("dummy_model_score");
    init = true;
  }

  return dummy_model_score_flag;
}
/////////////////////////////////////////////////////////////////////
float get_dummy_model_depth(){
  static bool init = {false};
  static float dummy_model_depth;
  if (!init) {
    realafteroption("dummy_model_depth", 2.0, dummy_model_depth );
    init = true;
  }
  return dummy_model_depth;
}
/////////////////////////////////////////////////////////////////////
double get_dummy_model_power(){
  static bool init = {false};
  static double dummy_model_power;
  if (!init) {
    Drealafteroption("dummy_model_power", 4.0, dummy_model_power );
    init = true;
  }
  return dummy_model_power;
}

