// -*-
// mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t
// -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 8705 $
//  $Date: 2006-06-06 16:15:21 -0700 (Tue, 06 Jun 2006) $
//  $Author: pbradley $


// Rosetta++ functions
#include "electron_density.h"
#include "param.h"
#include "pose_sse.h"
#include "pose_symmetric_docking.h"
#include "jumping_util.h"


//For testing scoring -- remove?
#include "after_opts.h"
#include "pose_io.h"
#include "score.h"
#include "files_paths.h"


// Objexx headers
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray1D.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyz.functions.hh>

// C++ Headers
#include <fstream>

// Utility Headers
#include <utility/basic_sys_util.hh>

// ccp4 stuff
#ifdef CCP4_EDENSITY
extern "C" {
#include "ccp4/solomon.h"
}
#endif

// This namespace contains the native density for scoring and display
namespace electron_density_map{
  bool density_map_initialized = {false};
  bool density_map_present     = {false};
  ElectronDensity native_electron_density;
  numeric::xyzVector_float center_of_mass;
  numeric::xyzMatrix_float principal_components;
  int axis1_flip;
  int axis2_flip;
  int axis3_flip;
}

// Set the origin of the cell in integral units
void
ElectronDensity::set_cell_origin( FArray1DB_int const & origin )
{
  cell_origin_.dimension(3);
  cell_origin_ = origin;
}

// Set the number of grid points. This is generally only used to calculate
// the grid spacing
void
ElectronDensity::set_grid_dimension( FArray1DB_int const & grid_size )
{
  grid_dimension_.dimension(3);
  grid_dimension_ = grid_size;
}

// The number of data points in the data set
void
ElectronDensity::set_data_point_size( FArray1DB_int const & data_size )
{
  data_point_size_.dimension(3);
  data_point_size_ = data_size;
}

// Dimensions in angstrom of the unit cell. Is used together with the grid
// dimension to calculate the grid spacing
void
ElectronDensity::set_cell_dimension( FArray1DB_float const & cell_dimension )
{
  cell_dimension_.dimension(3);
  cell_dimension_ = cell_dimension;
}

// Cell angles of the unit cell: alpha, beta and gamma. Differs from 90 degrees
// for most unit cells in X-ray crystallography. For cryoEM this should be 90
// for all angles
void
ElectronDensity::set_cell_angle( FArray1DB_float const & angles )
{
  cell_angle_.dimension(3);
  cell_angle_ = angles;
}

// Calculate the grid spacing
void
ElectronDensity::set_grid_step( FArray1DB_float const & step )
{
  grid_step_.dimension(3);
  grid_step_ = step;
}

// This is the minimum density in the data set. Does not make much sense for
// cryoEM density where most datapoints are 0
void
ElectronDensity::set_min( float const & min)
{
  min_ = min;
}

// Maximum density in the data set
void
ElectronDensity::set_max( float & max )
{
  max_ = max;
}

// Mean density in the data set. Not meaningful for cryoEM data
void
ElectronDensity::set_mean( float & mean )
{
  mean_ = mean;
}

// RMS of density in the data set. Does not make sense for cryoEM data
void
ElectronDensity::set_rms( float & rms )
{
  rms_ = rms;
}

// This is the threshold over which data is displayed and overlap is calculated
// in the scoring
void
ElectronDensity::set_threshold( float & threshold )
{
  threshold_ = threshold;
}

// The actual density data
void
ElectronDensity::set_density( FArray3DB_float const & density )
{
  density_.dimension(data_point_size_(1), data_point_size_(2), data_point_size_(3));
  density_ = density;
}

// Set the principal components of the density
void
ElectronDensity::set_principal_component(numeric::xyzMatrix_float & principal_components)
{
  principal_component_ = principal_components;
}

// The center of mass of the data points
void
ElectronDensity::set_center_of_mass(numeric::xyzVector_float & center_of_mass)
{
  center_of_mass_ = center_of_mass;
}

FArray3D_float const &
ElectronDensity::density() const
{
  return density_;
}

FArray4D_float const &
ElectronDensity::grid_xyz() const
{
  return grid_xyz_;
}

FArray1D_int const &
ElectronDensity::data_point_size() const
{
  return data_point_size_;
}

FArray1D_int const &
ElectronDensity::cell_origin() const
{
  return cell_origin_;
}

FArray1D_float const &
ElectronDensity::cell_dimension() const
{
  return cell_dimension_;
}

FArray1D_float const &
ElectronDensity::cell_angle() const
{
  return cell_angle_;
}

FArray1D_int const &
ElectronDensity::grid_dimension() const
{
  return grid_dimension_;
}

FArray1D_float const &
ElectronDensity::grid_step() const
{
  return grid_step_;
}

float const &
ElectronDensity::threshold() const
{
  return threshold_;
}

float const &
ElectronDensity::min() const
{
  return min_;
}

float const &
ElectronDensity::max() const
{
  return max_;
}

float const &
ElectronDensity::mean() const
{
  return mean_;
}

float const &
ElectronDensity::rms() const
{
  return rms_;
}

numeric::xyzMatrix_float const &
ElectronDensity::principal_component() const
{
  return principal_component_;
}

numeric::xyzVector_float const &
ElectronDensity::center_of_mass() const
{
  return center_of_mass_;
}

// Print out the content of the ElectronDensity class
void
ElectronDensity::print()
{
  std::cout << "ElectronDensity object is initialized with: " << std::endl <<
    "Number of data points: " << data_point_size()(1) << " " << data_point_size()(2) <<
    " " << data_point_size()(3) << std::endl <<
    "Origin " << cell_origin()(1) << " " << cell_origin()(2) << " " << cell_origin()(3)
	    << std::endl <<
    "Grid points: " << grid_dimension()(1) << " " << grid_dimension()(2) << " " <<
    grid_dimension()(3) << std::endl <<
    "Cell_angles: " << cell_angle()(1) << " " << cell_angle()(2) << " " << cell_angle()(3)
	    << std::endl <<
    "Grid steps: " << grid_step()(1) << " " << grid_step()(2) << " " << grid_step()(3) <<
    std::endl <<
    "Min density: " << min() << std::endl <<
    "Max density: " << max() << std::endl <<
    "Mean density: " << mean() << std::endl <<
    "RMS density: " << rms() << std::endl <<
    "Density threshold: " << threshold() << std::endl;

}

// Map the grid index to the actual xyz coordinates. This complicated
// calculation is necessary for cases when the unit cell is no perfectly cubic
FArray1D_float const
ElectronDensity::index_to_xyz(
			      int const i,
			      int const j,
			      int const k
			      ) const
{

  using numeric::conversions::radians;

  FArray1D_float xyz(3);
  float const alpha = cell_angle_(1);
  float const beta = cell_angle_(2);
  float const gamma = cell_angle_(3);
  float const step_x = grid_step_(1);
  float const step_y = grid_step_(2);
  float const step_z = grid_step_(3);

  float const cg = cos(radians(gamma));
  float const sg = sin(radians(gamma));
  float const	cb = cos(radians(beta));
  float const ca = cos(radians(alpha));
  float const c1 = (ca - cb*cg)/sg;
  float const c2 = sqrt(1 - cb*cb - c1*c1);

  xyz(1) = step_x*i + step_y*cg*j + step_z*cb*k;
  xyz(2) = step_y*sg*j + step_z*c1*k;
  xyz(3) = step_z*c2*k;
  return xyz;
}

// Calculate the xyz coordinates from the grid indices
void
ElectronDensity::set_grid_xyz()
{
  int const size1 = data_point_size()(1);
  int const size2 = data_point_size()(2);
  int const size3 = data_point_size()(3);

  grid_xyz_.dimension(3,size1,size2,size3);
  for (int i=1; i<= size1; ++i) {
    for (int j=1; j<=size2; ++j) {
      for (int k=1; k<=size3; ++k) {
	int const ip (i - cell_origin()(1) );
	int const jp (j - cell_origin()(2) );
	int const kp (k - cell_origin()(3) );
	FArray1D_float grid_value( index_to_xyz(ip,jp,kp) );
	grid_xyz_(1,i,j,k) = grid_value(1);
	grid_xyz_(2,i,j,k) = grid_value(2);
	grid_xyz_(3,i,j,k) = grid_value(3);
      }
    }
  }
}

// Find the center of the density in grid space
void
ElectronDensity::find_center()
{
  int const size1 (data_point_size()(1) );
  int const size2 (data_point_size()(2) );
  int const size3 (data_point_size()(3) );

  FArray1D_int center(3,0);
  int num_points(0);
  for (int i = 1; i <= size1; i++){
    for (int j = 1; j <= size2; j++){
      for (int k = 1; k <= size3; k++){
				if (  fabs( density()(i,j,k) ) > threshold() ){
					center(1) += i;
					center(2) += j;
					center(3) += k;
					num_points++;
				}
      }
    }
  }
  FArray1D_int origin(3);
  origin(1) = static_cast<int> ( center(1)/num_points ); //*grid_step()(1) );
  origin(2) = static_cast<int> ( center(2)/num_points ); //*grid_step()(2) );
  origin(3) = static_cast<int> ( center(3)/num_points ); //*grid_step()(3) );
  std::cout << "new center: " << origin(1) << " " << origin(2) << " " << origin(3) << std::endl;
  set_cell_origin(origin);
}

// Rotate the density so that the axis with the smallest moment
// of inertia is aligned with the z-axis. This is the direction
// where the "mass" is most stretched out...
void
ElectronDensity::rotate_to_principal_frame()
{
  using namespace numeric;

  std::cout << "rotate to principal frame " << std::endl;
  int const size1 = data_point_size()(1);
  int const size2 = data_point_size()(2);
  int const size3 = data_point_size()(3);

  // We switch the order of the axes's so that the smallest monoment
  // of inertia align with z
  xyzVector_float axis1 (&principal_component()(3,1));
  xyzVector_float axis2 (&principal_component()(2,1));
  xyzVector_float axis3 (&principal_component()(1,1));

  axis1.normalize();
  axis2.normalize();
  axis3.normalize();


  //orthonormalize using the Gram-Schmidt method
  axis2 = axis2 - dot(axis2,axis1)*axis1;
  axis2.normalize();
  xyzVector_float x_axis (cross(axis1,axis2) );
  xyzMatrix_float R( xyzMatrix_float::rows( x_axis, axis2, axis1 ) );

  //	std::cout << R(1,1) << " "  << R(2,1) << " " << R(3,1) << std::endl
  //					<< R(1,2) << " "  << R(2,2) << " " << R(3,2) << std::endl
  //					<< R(1,3) << " "  << R(2,3) << " " << R(3,3) << std::endl;

  for (int i=1; i<= size1; ++i) {
    for (int j=1; j<=size2; ++j) {
      for (int k=1; k<=size3; ++k) {
	xyzVector_float xyz(&grid_xyz_(1,i,j,k));
	xyzVector_float rotated_xyz( R*xyz );
	grid_xyz_(1,i,j,k) = rotated_xyz(1);
        grid_xyz_(2,i,j,k) = rotated_xyz(2);
        grid_xyz_(3,i,j,k) = rotated_xyz(3);
      }
    }
  }
}

// This function reads and density map in ccp4 format and initializes the
// ElectronDensity class. Assign the native density in the end of the function.
// Some function need inialize data in order not to produce rubbish. Need
// to add asserts that check that data is initialized.
#ifdef CCP4_EDENSITY
void
read_electron_density_map()
{
  using namespace electron_density_map;

  if (density_map_initialized) return;

  static bool const electron_density_file_flag = get_electron_density_file_flag();
  if (!electron_density_file_flag) return;

  std::cout << "Reading an electron density file: " << std::endl;
  ElectronDensity ED;
  MAP *cryoMap;
  std::string filename = stringafteroption("electron_density_file","2zta.ccp4");
  std::string filename_out = "out_"+filename;
  char *c_filename = strdup( filename.c_str() );
  char *c_filename_out = strdup( filename_out.c_str() );
  std::cout << "c_filename " << c_filename << " and dump it out translated with the name "
						<< c_filename_out << std::endl;
  cryoMap = ReadMap(c_filename);
  set_data_point_size(ED,cryoMap->ext);
  set_cell_origin(ED,cryoMap->ori);
  set_grid_dimension(ED,cryoMap->grid);
  set_cell_dimension(ED,cryoMap->cell);
  set_grid_step(ED);
  fill_density(ED,cryoMap->map);
  set_min_max_mean_rms(ED,cryoMap->min, cryoMap->max, cryoMap->mean, cryoMap->rms);
  float threshold (ED.max()/2);
  ED.set_threshold( threshold );
  ED.find_center();
  ED.set_grid_xyz();
  ED.principal_moments_of_inertia_electron_density();
  //		ED.rotate_to_principal_frame();
  ED.print();
  cryoMap->ori[0] = - ED.cell_origin()(1);
  cryoMap->ori[1] = - ED.cell_origin()(2);
  cryoMap->ori[2] = - ED.cell_origin()(3);

  WriteMap(cryoMap,c_filename_out);
  native_electron_density = ED;

  density_map_initialized = true;
  density_map_present = true;

  return;

}

#else
void
read_electron_density_map()
{
	return;
}
#endif

void
set_cell_origin(
		ElectronDensity & ED,
		int *origin
		)
{
  FArray1D_int ori(3);
  for (int i=1; i<= 3; ++i)
    {
      ori(i) = origin[i-1];
    }
  ED.set_cell_origin(ori);
}

void
set_grid_dimension(
		   ElectronDensity & ED,
		   int *grid
		   )
{
  FArray1D_int grid_dim(3);
  for (int i=1; i<= 3; ++i)
    {
      grid_dim(i) = grid[i-1];
    }
  ED.set_grid_dimension(grid_dim);
}

void
set_data_point_size(
		    ElectronDensity & ED,
		    int *size
		    )
{
  FArray1D_int data_size(3);
  for (int i=1; i<= 3; ++i)
    {
      data_size(i) = size[i-1];
    }
  ED.set_data_point_size(data_size);
}

void
set_cell_dimension(
		   ElectronDensity & ED,
		   float *cell_data
		   )
{
  FArray1D_float cell_data_length(3);
  FArray1D_float cell_data_angle(3);
  for (int i=1; i<= 3; ++i)
    {
      cell_data_length(i) = cell_data[i-1];
    }
  for (int i=1; i<= 3; ++i)
    {
      cell_data_angle(i) = cell_data[i+2];
    }

  ED.set_cell_dimension(cell_data_length);
  ED.set_cell_angle(cell_data_angle);
}

void
set_grid_step(
	      ElectronDensity & ED
	      )
{
  assert ( ED.grid_dimension().size() > 0 &&
	   ED.grid_dimension().size() == ED.cell_dimension().size() );

  FArray1D_float step(3);
  for (int i=1; i<= 3; ++i)
    {
      step(i) = ED.cell_dimension()(i)/ED.grid_dimension()(i);
    }
  ED.set_grid_step(step);
}

void
fill_density(
	     ElectronDensity & ED,
	     float ***map
	     )
{
  int const size1 = ED.data_point_size()(1);
  int const size2 = ED.data_point_size()(2);
  int const size3 = ED.data_point_size()(3);
  assert ( size1 > 0 && size2 > 0 && size3 > 0 );

  FArray3D_float density(size1,size2,size3);

  for (int i=1; i<= size1; ++i) {
    for (int j=1; j<=size2; ++j) {
      for (int k=1; k<=size3; ++k) {
	density(i,j,k) = map[i-1][j-1][k-1];
      }
    }
  }
  ED.set_density(density);
}

void
set_min_max_mean_rms(
		     ElectronDensity & ED,
		     float min,
		     float max,
		     float mean,
		     float rms
		     ){
  ED.set_min(min);
  ED.set_max(max);
  ED.set_mean(mean);
  ED.set_rms(rms);
}

// This function calculates the principal moments of inertia for the electron
// density. Uses the jacobi method to find eigenvectors
void
ElectronDensity::principal_moments_of_inertia_electron_density()
{
  using namespace numeric;
  xyzMatrix_float I( 0.0 );
  xyzVector_float center_of_mass_density(0);
  xyzMatrix_float principal_components(0);
  xyzVector_float eigenvalue(0.0);
  float const tol( 0.000001);
  // calculate moment of inertia tensor I
  get_moment_of_inertia(center_of_mass_density, I );
  // Find eigenvalues and eigenvectors
  eigenvalue = eigenvector_jacobi(I, tol, principal_components);
  principal_components.transpose(); //Why is this necessary?
  sort_eigenvectors(eigenvalue, principal_components);

	// Swith the order so that z is along the smallest component. Not
	// neccessary...
  xyzVector_float axis1 (&principal_components(3,1));
  xyzVector_float axis2 (&principal_components(2,1));
  xyzVector_float axis3 (&principal_components(1,1));
  axis1.normalize();
  axis2.normalize();
  axis3.normalize();

  //orthonormalize using the Gram-Schmidt method. Just to make sure its really
	//orthonormal
  axis2 = axis2 - dot(axis2,axis1)*axis1;
  axis2.normalize();
  axis3 = cross(axis1,axis2);
  axis2 = cross(axis1,axis3);

  xyzMatrix_float PC( xyzMatrix_float::rows( axis1, axis2, axis3 ) );
  set_principal_component(PC);
  set_center_of_mass(center_of_mass_density);

}

// Find the moment of inertia for a pose
void
principal_moments_of_inertia(
			     pose_ns::Pose & pose,
			     numeric::xyzVector_float & center_of_mass,
			     numeric::xyzMatrix_float & principal_components
			     )
{

  int const total_residue (pose.total_residue_for_scoring() );
  FArray3D_float const Epos( pose.Eposition() );
  principal_moments_of_inertia(Epos, total_residue,center_of_mass,principal_components);

}

void
principal_moments_of_inertia(
			     pose_ns::Pose const & pose,
			     numeric::xyzVector_float & center_of_mass,
			     numeric::xyzMatrix_float & principal_components
			     )
{

  int const total_residue (pose.total_residue_for_scoring() );
  FArray3D_float const Epos( pose.Eposition() );
  principal_moments_of_inertia(Epos, total_residue,center_of_mass,principal_components);

}

// Find the moment of inertia for a pose
void
principal_moments_of_inertia(
			     FArray3Da_float Epos,
			     int const nres,
			     numeric::xyzVector_float & center_of_mass,
			     numeric::xyzMatrix_float & principal_components
			     )
{
  using namespace numeric;
  xyzMatrix_float I( 0.0 );
  xyzVector_float eigenvalue(0.0);
  // Calculate the moment of inertia tensor
  get_moment_of_inertia( Epos, nres, center_of_mass, I );
  float const tol( 0.000001);
  // Find the eigenvalues and eigenvectors
  eigenvalue = eigenvector_jacobi(I, tol, principal_components);
  principal_components.transpose(); //Why is this necessary?

  // Sort the eigenvectors in descending order of the eigenvalues
  sort_eigenvectors(eigenvalue, principal_components);

}

// Sort the eigenvalues and eigenvectors in descending order of the
// eigenvalues. Uses straightforward insertion which is slow for larger
// matrices
void
sort_eigenvectors(numeric::xyzVector_float & eigenvalues, numeric::xyzMatrix_float & eigenvectors)
{
  // Sort eigenvectors using insertion in descending order according to the size of the eigenvalues. Do we need to
  // care about the sign?
  using namespace numeric;
  int k;
  float val_temp;
  xyzVector_float vec_temp(0);
  for (int i=1; i <= 3; ++i){
    val_temp = eigenvalues(k=i);
    for (int j=i; j<= 3; ++j){
      if ( eigenvalues(j) > val_temp ) val_temp=eigenvalues(k=j);
    }
    if ( k != i ) {
      eigenvalues(k) = eigenvalues(i);
      eigenvalues(i) = val_temp;
      xyzVector_float vec_temp (&eigenvectors(k,1));
      for (int l=1; l<=3; ++l) {
	eigenvectors(k,l)	= eigenvectors(i,l);
	eigenvectors(i,l) = vec_temp(l);
      }
    }
  }
}

// Calculate the "moment of inertia" and "center of mass" for the density
void
ElectronDensity::get_moment_of_inertia(
				       numeric::xyzVector_float & center_of_mass,
				       numeric::xyzMatrix_float & I )
{

  using namespace numeric;

  FArray2D_float m_moment( 3, 3 );
  FArray1D_float center(3);
  int num_points(0);

  int const size1 (data_point_size()(1) );
  int const size2 (data_point_size()(2) );
  int const size3 (data_point_size()(3) );

  for (int i = 1; i <= size1; i++){
    for (int j = 1; j <= size2; j++){
      for (int k = 1; k <= size3; k++){
	if (fabs (density()(i,j,k) ) > threshold() )
	  {
	    num_points++;
	    FArray1Dp_float xyz(grid_xyz()(1,i,j,k), 3 );
	    center += xyz;
	  }
      }
    }
  }
  center /= float( num_points);

  center_of_mass(1) = center(1);
  center_of_mass(2) = center(2);
  center_of_mass(3) = center(3);

  FArray2D_float xyz_recenter(3,num_points);
  int point(0);
  for (int i = 1; i <= size1; i++){
    for (int j = 1; j <= size2; j++){
      for (int k = 1; k <= size3; k++){
        if (fabs (density()(i,j,k) ) > threshold() )
	  {
	    point++;
            FArray1Dp_float xyz(grid_xyz()(1,i,j,k), 3 );
            xyz_recenter(1,point) = xyz(1) - center(1);
	    xyz_recenter(2,point) = xyz(2) - center(2);
	    xyz_recenter(3,point) = xyz(3) - center(3);
	  }
      }
    }
  }

  // Generate inertia tensor:
  for ( int k = 1; k <= 3; ++k ) {
    for ( int j = 1; j <= 3; ++j ) {
      float cross_term = 0.0;
      for ( int i = 1; i <= num_points; ++i ) {
	cross_term -= xyz_recenter(j, i) * xyz_recenter(k, i);
      }
      m_moment(k,j) = cross_term;
    }
  }


  float diagonal_term( 0.0 );
  for (int  i = 1; i <= num_points; ++i ) {
    for ( int j = 1; j <= 3; ++j ) {
      diagonal_term += xyz_recenter(j, i) * xyz_recenter(j, i);
    }
  }
  for ( int j = 1; j <= 3; ++j ) {
    m_moment(j,j) += diagonal_term;
  }

  I =  xyzMatrix_float::cols( &m_moment( 1,1 ) );

  I /= num_points;

}
void
transform_to_principal_frame(
			     FArray2Da_float Epos,
			     int const nres,
			     numeric::xyzVector_float & center_of_mass,
			     numeric::xyzMatrix_float & principal_component,
			     int const i,
			     int const j,
			     int const k
			     )
{
  using namespace numeric;
  using namespace electron_density_map;

  xyzMatrix_float pc (principal_component);

  // We switch the order of the axes's so that the smallest monoment
  // of inertia align with z
  xyzVector_float axis1 (&pc(3,1));
  xyzVector_float axis2 (&pc(2,1));
  xyzVector_float axis3 (&pc(1,1));

  axis1.normalize();
  axis2.normalize();
  axis3.normalize();

  //orthonormalize using the Gram-Schmidt method. Uneccessary really, just to make
	//sure.
  axis2 = axis2 - dot(axis2,axis1)*axis1;
  axis2.normalize();
  axis3 = cross(axis1,axis2);
  axis2 = cross(axis1,axis3);

  xyzMatrix_float D(xyzMatrix_float::rows(i*axis1,j*axis2,k*axis3));
  xyzMatrix_float N = native_electron_density.principal_component();
  N.transpose();
  xyzMatrix_float R (N*D);

  float determinant = R.det() > 0 ? 1.0 : -1.0;
  R(3,3) *= determinant;

  /*	std::cout << "Transformation matrix " << std::endl;
	std::cout << R(1,1) << " "  << R(2,1) << " " << R(3,1) << std::endl
	<< R(1,2) << " "  << R(2,2) << " " << R(3,2) << std::endl
	<< R(1,3) << " "  << R(2,3) << " " << R(3,3) << std::endl;
	std::cout << "test axis1 maps onto itself? " << R*axis1 << " vs axis1 " << axis1 << std::endl;
	std::cout << " Determinant is " << R.det() << std::endl;*/

  for (int l=1; l<= nres; ++l) {
    xyzVector_float xyz(&Epos(1,l));
    xyzVector_float xyz_trans(R*(xyz - center_of_mass));
    for ( int m=1; m<= 3; ++m ) Epos(m,l) = xyz_trans(m);
  }
}


void
transform_to_principal_frame(
			     pose_ns::Pose & pose
			     ){
  using namespace electron_density_map;

  if (!density_map_present) return;

  eval_electron_density_score( pose ); //sets center of mass, etc.

#ifdef CCP4_EDENSITY
  transform_to_principal_frame( pose,
				center_of_mass,
				principal_components,
				axis1_flip,
				axis2_flip,
				axis3_flip );
#endif
}



// This function takes a (symmetrical) pose and align it so that the smallest
// eigenvector corresponding to the smallest moment of inertia is align with
// the z-axis. This orientation can then be used in scoring in an overlap
// calculation
void
transform_to_principal_frame(
			     pose_ns::Pose & pose,
			     numeric::xyzVector_float & center_of_mass,
			     numeric::xyzMatrix_float & principal_component,
			     int const i,
			     int const j,
			     int const k
			     )
{
  using namespace numeric;
  using namespace pose_ns;
  using namespace electron_density_map;

  xyzMatrix_float pc (principal_component );

  // We switch the order of the axes's so that the smallest monoment
  // of inertia align with z. This procedure is not necessary...
  xyzVector_float axis1 (&pc(3,1));
  xyzVector_float axis2 (&pc(2,1));
  xyzVector_float axis3 (&pc(1,1));

  axis1.normalize();
  axis2.normalize();
  axis3.normalize();

  //orthonormalize using the Gram-Schmidt method
  axis2 = axis2 - dot(axis2,axis1)*axis1;
  axis2.normalize();
  axis3 = cross(axis1,axis2);
  axis2 = cross(axis1,axis3);

  xyzMatrix_float D(xyzMatrix_float::rows(i*axis1,j*axis2,k*axis3));
  xyzMatrix_float N = native_electron_density.principal_component();
  // What we do here is N-1*D, for a rotation matrix the transpose is equal to
	// the inverse.
	N.transpose();
  xyzMatrix_float R (N*D);
	//  Make sure the determinant is 1 othervise we have a rotoinversion.
  float determinant = R.det() > 0 ? 1.0 : -1.0;
  R(3,3) *= determinant;

  /*	std::cout << "Transformation matrix pose rotation" << std::endl;
	std::cout << R(1,1) << " "  << R(2,1) << " " << R(3,1) << std::endl
	<< R(1,2) << " "  << R(2,2) << " " << R(3,2) << std::endl
	<< R(1,3) << " "  << R(2,3) << " " << R(3,3) << std::endl;*/

  xyzVector_float trans ( -center_of_mass );

  pose.transform_Ax_plus_b(Z_rot(0.0),trans);
  pose.transform_Ax_plus_b(R,xyzVector_float (0.0));

}

float
eval_electron_density_score(
			    pose_ns::Pose const & pose
			    )
{
  using namespace electron_density_map;

  if (!density_map_initialized) read_electron_density_map();

  if (!density_map_present) return 0.0;

  int const total_residue (pose.total_residue());
  FArray2D_float Eca ( 3, total_residue, 0.0f);
  for (int i = 1; i <= total_residue; i++){
    for(int j = 1; j <= 3; j++){
      Eca( j, i) = pose.Eposition()( j, 2, i);
    }
  }
  // Principal components
  numeric::xyzVector_float center_of_mass(0);
  numeric::xyzMatrix_float PC(0);
  principal_moments_of_inertia(pose,center_of_mass,PC);

	// We need to test for all directions of the principal components. That would
	// be 8 calculations but we can discard 4 of those since we don't want mirror
	// images, do we.
  float score (0);
  float score_save (0);
  int flip1(1);
  int flip2(1);
  int flip3(1);
  for (int i = -1; i <= 1; i += 2){
    for (int j = -1; j <= 1; j += 2){
      int k = i*j;
      assert (i*j*k == 1); //can't violate parity.
      FArray2D_float Eca_temp ( Eca );
      transform_to_principal_frame(Eca_temp,total_residue,center_of_mass,PC,i,j,k);
      score = score_electron_density(Eca_temp,total_residue);
      if ( score < score_save ) {
	score_save = score;
	flip1 = i;
	flip2 = j;
	flip3 = k;
      }
    }
  }
	// Save flips in namespac so that the pose can be rotated into the density
	// outside the scoring. We can't rotate the pose in scoring since its a
	// const. There is a trick around that though...
  electron_density_map::center_of_mass = center_of_mass;
  electron_density_map::principal_components = PC;
  electron_density_map::axis1_flip = flip1;
  electron_density_map::axis2_flip = flip2;
  electron_density_map::axis3_flip = flip3;
  return score_save;
}

// The score is evaluated in this function. Brutally simple at this time. Just
// add up density for all Ca coordinates.
float
score_electron_density(
		       FArray2Da_float Epos_ca,
		       int const nres
		       )
{
  using namespace electron_density_map;

  float score(0);
  int x_cell, y_cell, z_cell;
  int const size1 ( native_electron_density.data_point_size()(1) );
  int const size2 ( native_electron_density.data_point_size()(1) );
  int const size3 ( native_electron_density.data_point_size()(1) );

  for (int i=1; i<= nres; ++i) {
    x_cell = (int)( ( Epos_ca(1,i) - native_electron_density.grid_xyz()(1,1,1,1) )/
									native_electron_density.grid_step()(1));
    y_cell = (int)( ( Epos_ca(2,i) - native_electron_density.grid_xyz()(2,1,1,1) )/
									native_electron_density.grid_step()(2));
    z_cell = (int)( ( Epos_ca(3,i) - native_electron_density.grid_xyz()(3,1,1,1) )/
									native_electron_density.grid_step()(3)) ;
    if (x_cell > 0 && y_cell > 0 && z_cell > 0 && x_cell <= size1 && y_cell <= size2 && z_cell <= size3 ) {
      score -= native_electron_density.density()(x_cell,y_cell,z_cell);
    }
  }
  return score;
}



///////////////////////////////////////////////////////////////////////////////
void
electron_density_score_test()
{
  using namespace silent_io;
  using namespace pose_ns;
  using namespace electron_density_map;

	param::MAX_RES_assign_res(2000);
  //What we are testing:
  read_electron_density_map();

  static const bool fullatom = truefalseoption( "fullatom" );
  Pose pose;

  std::string silent_file_name= stringafteroption("s");
  std::string start_pdb_name;

  // read silent file
  Silent_file_data decoys( silent_file_name, fullatom );
  if ( !decoys.size() ) {
    std::cout << "STOP:: couldnt open silent-file!! " << std::endl;
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }

  // setup tag list
  std::vector< std::string > tag_list;
  if ( truefalseoption("t") ) {
    tag_list.push_back( stringafteroption("t") );
  } else if ( truefalseoption("all") ) {
    tag_list = decoys.tags();
  } else if ( truefalseoption("l") ) {
    std::ifstream data( stringafteroption("l").c_str() );
    if ( !data.good() ) {
      std::cout << "STOP:: cant open tags file: " << std::endl;
      utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }
    std::string tag;
    while ( getline(data,tag) ) {
      tag_list.push_back(tag);
    }
    data.close();
  }

  // silly:
  std::string prefix( files_paths_pdb_out_prefix() );
  if ( files_paths::protein_chain == '-' ||
       files_paths::protein_chain == '_' ) {
    // who likes hyphens or underscores in their pdbs ?
    files_paths::protein_chain = ' ';
  }

  //Outfile, or scorefile in this case.
  std::string const scorefilename = stringafteroption("scorefile","rescore.sc");
  Silent_out out( scorefilename );


  //Set up native.
  Pose native_pose;
  bool native_exists( false );
  if (truefalseoption("n")){
    native_exists = true;
    std::string native_file_name = stringafteroption("n","blah.pdb");
    bool success = pose_from_pdb( native_pose, native_file_name,
				  true, false, true );
    if (!success){
      std::cout << "Had trouble with native pdb " << native_file_name << std::endl;
      utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }
    pose_to_native( native_pose );

    native_pose.score( score12 );
    native_pose.set_native_pose( native_pose );
    calc_rms( native_pose );

    out.write_scores( "NATIVE", native_pose );
 /*   	numeric::xyzVector_float center_of_mass (0);
	numeric::xyzMatrix_float PC(0);
	principal_moments_of_inertia( native_pose,center_of_mass, PC);
	native_pose.dump_pdb("before_pc.pdb");
	transform_to_principal_frame(native_pose, center_of_mass, PC,axis1_flip,axis2_flip,axis3_flip);
	native_pose.dump_pdb("after_pc.pdb");
	std::cout << "calculate pc again " << std::endl;
	principal_moments_of_inertia( native_pose,center_of_mass, PC);
	std::exit(0);*/

    //		native_pose.dump_pdb( "native.pdb" );
  }


	Score_weight_map weight_map (score4);
  // loop through the tag list
  for ( std::vector< std::string >::const_iterator it=tag_list.begin(),
	  it_end = tag_list.end(); it != it_end; ++it ) {

    std::string const & tag( *it );

    if ( ! decoys.has_key( tag ) ) {
      std::cout << "couldnt find tag in silent-file: " << tag << std::endl;
      continue;
    }

    // get the data
    Silent_structure const & decoy( decoys.get_structure( tag ) );

    //Fresh pose for each decoy. (Necessary for RNA stuff, which sets up atom trees.)
    Pose pose;

    // fill the pose
    decoy.fill_pose( pose, true );

    static const int monomers = intafteroption("monomers",1);
    create_full_symmetric_pose(pose,monomers);

    if (fullatom) {
      pose.score( score12 );
    } else {
      pose.score( weight_map );
    }

    float electron_density_score (0);
    electron_density_score = eval_electron_density_score( pose );

    if (native_exists){
      pose.set_native_pose( native_pose );
      calc_rms( pose );
    }

    transform_to_principal_frame( pose );

    //Ingemar: "I present to you: EDENSITY!".

    //    pose.set_extra_score( "EDENSITY", electron_density_score);

    static const bool extract = truefalseoption("extract");
    if (extract) pose.dump_pdb( prefix+tag+".pdb" );
    out.write_scores( tag, pose );

  }

}

void
ElectronDensity::find_n_fold_axis(int const N, numeric::xyzVector_float & symm_axis)
{
	using namespace numeric;

	xyzVector_float rot_axis (&principal_component()(3,1));
	xyzVector_float axis (&principal_component()(1,1));

	xyzVector_float origin(center_of_mass());


	int const size1 (data_point_size()(1) );
  int const size2 (data_point_size()(2) );
  int const size3 (data_point_size()(3) );
	int x_cell, y_cell, z_cell;
	int v_step ( std::max( std::max(size1,size2), size3) );
	float save_density(0);
	float sum_density(0);
	xyzVector_float vec_coord(0);
	int center(0);
	xyzVector_float axis_rotated(0.0);
	xyzVector_float  axis_rotated_n_fold(0);

	for (int i=1; i<= v_step; ++i) {
		sum_density = 0;
		for (int j=-3; j<= 3; ++j){
			xyzVector_float vec_coord ( origin + (i+j)*axis);
			x_cell = (int)( ( vec_coord(1) - grid_xyz()(1,1,1,1) )/grid_step()(1));
			y_cell = (int)( ( vec_coord(2) - grid_xyz()(2,1,1,1) )/grid_step()(2));
			z_cell = (int)( ( vec_coord(3) - grid_xyz()(3,1,1,1) )/grid_step()(3));
			if (x_cell > 0 && y_cell > 0 && z_cell > 0 && x_cell <= size1 && y_cell <= size2 && z_cell <= size3 ) {
				sum_density += fabs(density()(x_cell,y_cell,z_cell));
			}
			sum_density /=7;
		}
		if ( sum_density > save_density ) {
			save_density=sum_density;
			center = i;
		}
	}

	float alpha (360/N);
	save_density = 0;
	for (int deg=0; deg<360; ++deg) {
		sum_density = 0;
		xyzMatrix_float M ( rotation_matrix(rot_axis,(float)deg) );
		axis_rotated = M*axis;
		for (int i=1; i<=N; ++i) {
			xyzMatrix_float R ( rotation_matrix(rot_axis,(float)(N*alpha)) );
			axis_rotated = R*axis_rotated;
			vec_coord = origin + center*axis_rotated;
			x_cell = (int)( ( vec_coord(1) - grid_xyz()(1,1,1,1) )/grid_step()(1));
      y_cell = (int)( ( vec_coord(2) - grid_xyz()(2,1,1,1) )/grid_step()(2));
      z_cell = (int)( ( vec_coord(3) - grid_xyz()(3,1,1,1) )/grid_step()(3));
      if (x_cell > 0 && y_cell > 0 && z_cell > 0 && x_cell <= size1 && y_cell <= size2 && z_cell <= size3 ) {
				sum_density += fabs(density()(x_cell,y_cell,z_cell));
			}
		}

			std::cout << "angle, sum_density " << deg << " " << sum_density << std::endl;
    if ( sum_density > save_density ) {
			save_density=sum_density;
			symm_axis = axis_rotated;
		}
	}
	std::cout << "12-fold axis is found at " << symm_axis << std::endl;
}

///////////////////////////////////////////////////////////////////
bool
get_electron_density_file_flag()
{
  static bool init( false );
  static bool electron_density_file_flag ( false );

  if ( !init ){
    electron_density_file_flag = truefalseoption("electron_density_file");
    init = true;
  }

  return electron_density_file_flag;

}
