// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14080 $
//  $Date: 2007-04-09 23:55:07 +0300 (Mon, 09 Apr 2007) $
//  $Author: yab $


// Rosetta Headers
#include "elliptic_msd.h"
#include "angles.h"
#include "chuck.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "loops.h"
#include "misc.h"
#include "namespace_cold.h"
#include "param.h"
#include "rotate.h"
#include "recover.h"
#include "refold.h"
#include "util_vector.h"
#include "wobble.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <algorithm>
#include <cstdlib>
#include <iostream>


//Utility Headers
#include <utility/basic_sys_util.hh>

////////////////////////////////////////////////////////////////////////////////
/// @begin double_move
///
/// @brief
///car inserts two fragments, returns updated position array
///
/// @detailed
///car sets total_begin and total_end to the values associated with the
///car entire change in the position array
///car if double_move fails, total_begin and total_end are set to 0
///car and angles are reset
///
/// @param  size - [in/out]? -
/// @param  cutoff_in - [in/out]? -
/// @param  tag1 - [in/out]? - minimum separation between frags
/// @param  tag2 - [in/out]?  max separation between frags
/// @param  total_begin - [in/out]? -
/// @param  total_end - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
double_move(
	int size,
	float cutoff_in,
	int tag1,
	int tag2,
	int & total_begin,
	int & total_end
)
{

	using namespace cold;
	using namespace fragments;
	using namespace misc;
	using namespace param;

//car local
	int i,j,k;
	int size_bin;
	int frag2_begin,frag2_end,frag_end,frag_begin;
	int double_begin, double_end;
	float chuck_cost,cost1,cost2;
	float cutoff_max;
	FArray2D_float list_cost( MAX_NEIGH()(), 2 );
	 // (frag#;1=rmsd_el, 2=rmsd_el_off)
	FArray1D_int list_frag( MAX_NEIGH()() );
	int best_frag, list_count;
	int increment;
	int min_frag2_begin,max_frag2_begin;
	int msd_start,msd_atoms;
	float scan_cost;
	int scan_frag,scan_begin;
	int dir_x;
	float phi1,phi2;
	FArray2D_float pos1( 3, MAX_POS );
	FArray2D_float pos2( 3, MAX_POS );
	float rama_score,dist_score;
	bool gfrag( true );

// diagnostic: these costs are all directly comparable
// cost1,cost2 and dist_score are not
//$$$      float frag1_cost,frag2_cost,wobble_cost;

	cutoff_max = cutoff_in;
	if ( get_loop_flag() ) {
		std::cout << "no double moves allowed for loops " << std::endl;
		goto L100;
	}

	size_bin = get_index_by_frag_size(size);
	scan_cost = 4.0e9;
	scan_frag = 0;
	scan_begin = 0;
// first frag:
	choose_chuck( size, cutoff_max, 5, total_residue-size-5, best_frag,
	 chuck_cost, frag_begin );
	if ( best_frag < 0 ) goto L100; // change, used to return
	insert_frag( size, size_bin, frag_begin, best_frag );
	frag_end = frag_begin + size - 1;
	refold( frag_begin, frag_end );
	refold_get_dir(dir_x);

// as in wobble, calculate msd over shortest part of chain that could move
// # of interactions scale with smaller half
	min_frag2_begin = std::max(5,frag_begin-size-tag2);
	max_frag2_begin = std::min(frag_end+tag2,total_residue-size-5);

	if ( dir_x == 1 ) {      // fix N-term
		msd_start = max_frag2_begin + size - 1;
		msd_atoms = ( total_residue - msd_start ) * 5;
	} else {                      // fix C-term
		msd_start = 1;
		msd_atoms = (min_frag2_begin-1)*5;
	}

// get an estimate of the cost of the first move
	calc_msd_best(msd_start,msd_start+msd_atoms/5,cost1);
//$$$      frag1_cost = cost1

// scan along in quarter fragment length steps looking for a good place
// to insert a fragment Cterm to first frag
// first look at frag2's Nterm to frag1, so fold with Cterm fixed
	refold_set_direction(-1);
	refold(frag_begin,frag_end);

	increment = std::max(static_cast< int >(size/4),1);
	for ( frag2_begin = min_frag2_begin; frag2_begin <= max_frag2_begin;
	 frag2_begin += increment ) {
		frag2_end = frag2_begin + size - 1;

		if ( frag2_begin == frag_begin ) {  // need to switch fold dir
			refold_set_direction(1); // for frag2's Cterm to frag1
			refold(frag_begin,frag_end);
		}

		if ( std::abs(frag_begin-frag2_end) < tag1 ) goto L200; // too close
		if ( std::abs(frag2_begin-frag_end) < tag1 ) goto L200;
		if ( std::abs(frag_begin-frag2_begin) < size ) goto L200; // overlap

//car pos1,pos2 describe the endpoints of the ideal fragment
//car here, reconnect position and best_position
		if ( frag2_begin > frag_begin ) { // frag2 cterm
			for ( j = 1; j <= 3; ++j ) {
				for ( k = 1; k <= 5; ++k ) {
					pos1(j,k) = Eposition(j,k,frag2_begin);
					pos2(j,k) = Ebest_position(j,k,frag2_end+1);
				}
			}
			phi1 = phi(frag2_begin);
			phi2 = best_phi(frag2_end+1);
		} else {
			for ( j = 1; j <= 3; ++j ) {
				for ( k = 1; k <= 5; ++k ) {
					pos1(j,k) = Ebest_position(j,k,frag2_begin);
					pos2(j,k) = Eposition(j,k,frag2_end+1);
				}
			}
			phi1 = best_phi(frag2_begin);
			phi2 = phi(frag2_end+1);
		}

		chuck_move(size,frag2_begin,align_depth(frag2_begin,size_bin),pos1,phi1,
		 pos2,phi2,Ebest_position(1,1,msd_start),Eatom_weights(1,msd_start),
		 msd_atoms,4.0*cutoff_max,best_frag,chuck_cost,list_cost,list_frag,
		 list_count);

		if ( list_count > 0 ) {    // if we found some frags
			cost2 = list_cost(1,1);
			best_frag = list_frag(1);
			for ( i = 2; i <= list_count; ++i ) {
				if ( list_cost(i,1) < cost2 ) { // best frag in this scan
					cost2 = list_cost(i,1);
					best_frag = list_frag(i);
				}
			}
			if ( cost2 < scan_cost ) { // better than other scans?
				scan_cost = cost2;
				scan_frag = best_frag;
				scan_begin = frag2_begin;
			}
		}
L200:;
	}                     // possible frag2 positions

	cost2 = scan_cost; // recover the best frag
	frag2_begin = scan_begin;
	frag2_end = scan_begin + size - 1;
	best_frag = scan_frag;
	refold_set_direction(0);

	if ( cost2 > cost1 ) goto L100; // bail

	double_begin = std::min(frag_begin,frag2_begin);
	double_end = std::max(frag_end,frag2_end);
	insert_frag(size,size_bin,frag2_begin,best_frag);
	refold(double_begin,double_end); // refold the doubly inserted region

//$$$      calc_msd_best(msd_start,msd_start+msd_atoms/5,frag2_cost);

	if (total_begin < 1); return //warning sign!
	add_fast_wobble(double_begin,double_end-double_begin+1,2,0,rama_score,
	 dist_score,gfrag,total_begin,total_end);

	if ( !gfrag ) return;
	if ( rama_score > 7.0 || dist_score > 30.0 ) goto L100;

//$$$      calc_msd_best(msd_start,msd_start+msd_atoms/5,wobble_cost);
//$$$      std::cout << "dbl" <<
//$$$       F( 8, 2, cost1 ) << F( 8, 2, cost2 ) << F( 8, 2, dist_score ) <<
//$$$       I( 4, dir_x ) << I( 4, frag_begin-frag2_begin ) <<
//$$$       I( 4, frag_begin ) << I( 4, frag_end ) << I( 4, frag2_begin ) <<
//$$$       I( 4, frag2_end ) << F( 8, 2, frag1_cost ) <<
//$$$       F( 8, 2, frag2_cost ) << F( 8, 2, wobble_cost ) << std::endl;

	return;

L100:; // no frag insertion
	resetphipsi();
	total_begin = 0;
	total_end = 0;
}

//------------------------------------------------------------------------------

////////////////////////////////////////////////////////////////////////////////
/// @begin eliptic_rmsd_2
///
/// @brief
///
/// @detailed
///
/// @param  A1 - [in/out]? -
/// @param  U1 - [in/out]? -
/// @param  EL - [in/out]? -
/// @param  EL2 - [in/out]? -
/// @param  C_EL - [in/out]? -
/// @param  cutoff - [in/out]? -
/// @param  rmsd - [in/out]? -
/// @param  rcM - [in/out]? -
///
/// @detailed
///
///car  explanations for:
///car  elliptic_rmsd, elliptic_rmsd_2
///car         (functions for wobble, chuck and double moves)
///car  dfunc_wobble
///car         (derivative of the elliptic_rmsd function)
///
///db object is to minimize the mean square displacement of the chain caused by
///db a fragment insertion.  the function minimized is the msd between the best
///db position coords and the coords given the current values of the torsion
///db angles in the "splint" region which follows the fragment insertion. The
///db msd is only computed over the residues downstream of the splint.
///db
///db Notation:
///db
///db (V is an arbitrary vector, A represents a matrix and U an offset)
///db
///db A2(V-U2) is the transformation which maps the stub of the residue
///db following the splint onto the stub of the first residue in the splint
///db which is at the origin in the standard orientation.the splint is in the
///db original conformation (that of the best position array).  thus this
///db transformation effectively "undoes" the translation and rotation
///db corresponding to the original conformation of the splint.
///db
///db A1t V + U1 is the transformation which maps the stub of the first residue
///db at the origin in the standard orientation onto the stub of the residue
///db following the splint.  the splint is in the current conformation specified
///db by the current values of the torsion angles.  thus this transformation
///db describes the rotation and translation corresponding to the current
///db conformation of the splint.
///db
///db
///db suppose an atom downstream from the splint originally has a position V in
///db the
///db coordinate frame where the first residue of the splint is at the origin in
///db standard orientation.  what is the new position of the atom following
///db replacement of the old conformation of the splint with the new
///db conformation?  we first undo the transformation corresponding to the old
///db splint, and then apply the transformation corresponding to the new splint:
///db
///db V' = A1t A2 (V-U2) + U1 = R V + L
///db
///db where R = A1t A2  and L = U1 - R U2.
///db
///db
///db Now, the msd is given by the sum over the atoms i in the splint of
///db (Vi - V'i)**2 (in the following for brevity a sum is implied wherever the
///db subscript i is used).  Charlie's insight is that computing this sum can be
///db greatly sped up by some precomputation, and that the V'i never need to be
///db computed directly.  Charlie wrote two different functions for computing
///db this efficiently.
///db
///db eliptic_rmsd
///db
///db (Vi - V'i)**2 = (Vi - (RVi + L))**2 = 2Vi**2 - 2ViRVi + L**2 + 2(LR-L)Vi
///db to separate out motion associated with the center of mass of the Vi,
///db substitute in Vi = (Vi -Vav + Vav).
///db
///db = 2 * [(Vi - Vav)**2 + Vav**2 - (Vi-Vav)R(Vi-Vav) - Vav R Vav
///db
///db + (LR-L)(Vi-Vav) + (LR-L) Vav] + L**2
///db
///db (note that all the cross terms Vav (Vi - Vav), etc, drop out when the sum
///db is taken)
///db
///db Now, collect terms in Vi(x)**2, Vi(x)Vi(y), etc. :
///db
///db xx      (xi-xav)**2 + (xi-xav)Rxx(xi-xav) - xav Rxx xav
///db
///db xy                    (xi-xav)Rxy(yi-yav) + (yi-yav)Ryx(xi-xav)
///db
///db x                       (LR-L)i xav
///db
///db
///db the sums over the (xi-xav)(xi-xav) and  (xi-xav)(yi-yav) can be
///db precomputed.  they are the eliptical moments (these two are EL(1) and
///db EL(4)).
///db
///db
///db eliptic_rmsd_2,   an even faster way to do the same thing:
///db
///db (V-V')**2 = (V - A1t A2 (V-U2) -U1)**2 =
///db
///db = ( (V-U1) - A1t A2 (V-U2) )**2 =
///db
///db = (V-U1)**2 + (V-U2)**2 -2 (V-U1) A1t A2 (V-U2) =
///db
///db = (V-U1)**2 + (V-U2)**2 -2 A1 (V-U1) A2(V-U2)
///db
///db and we have never even had to compute A1t A2 !  Taking out the center of
///db mass as before gives
///db
///db = (Vi-Vav)**2 + (Vav-U1)**2 + (Vi-Vav-U2)**2 + Vav**2
///db
///db -2 A1 (V-Vav) A2 (V-Vav) -2 A1(Vav - U1) A2 (Vav-U2).
///db
///db Here, Charlie precomputes both the (Vi - Vav) moments and the
///db (Vi - Vav) A2 (Vi-Vav) moments.  the latter are  called C_EL.  The
///db (Vi - Vav - U2) are also precomputed and are called EL2.
///db
///db dfunc_wobble computes the derivative of f=(V-V')**2 with respect to each
///db of the torsion angles in the splint.  Using the chain rule, we have
///db
///db df/d phi = df/dri dri/dphi = 2ri dri/dphi.
///db
///db dri/dphi = Eab x (V-Vb) * (V' - V)/|V-V'|
///db
///db (the first cross produce is the displacement of V upon a rotation dphi
///db around the unit vector Eab, Vb is the coordinates of the second atom in
///db the bond)
///db
///db since | V-V'| = ri,
///db
///db df/ dphi = 2 Eab x (V-Vb) * (V' - V) =
///db
///db  note that Eab and Vb are different for each torsion angle, but V'
///db and V are the same.  rearranging:
///db
///db = -  2 Eab X Vb * (V' - V) - 2 Eab * (V' x V).
///db
///db now we need the averages over all Vi of the difference and the
///db crossproduct of V and V'.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eliptic_rmsd_2(
	FArray2Da_float A1,
	FArray1Da_float U1,
	FArray1Da_float EL,
	FArray1Da_float EL2,
	FArray2Da_float C_EL,
	float cutoff,
	float & rmsd,
	float & rcm
)
{
	A1.dimension( 3, 3 );
	U1.dimension( 3 );
	EL.dimension( 10 );
	EL2.dimension( 10 );
	C_EL.dimension( 3, 3 );

//     this function replaces the following one using some precomputation to
// double the speed.
// computes the effect of replacing one fragment with another using
// a virtual ellipsoid instead of explicit coordinate positions.
// fragment is specified by a rotation matrix plus an offset
// some precomputation has occured to make this calulation fast:
// the EL2 matrix is a prerotated ellipsoid.
// the C_EL is the cross moment of the two ellipoids (rotated and not rotated)
// (speedup over old method: 29 multiplies 28 adds versus >58 multiplies and  >58 adds)
//
// Rcm is the change in center of mass position
// rmsd is the total change: rotational plus com.
// note the com change will be upper bound on total change.
// thus you could imagine splitting this
// calculation into two parts: compute COM first,
// if its less than threshold value then compute
// rotation too, thus saving time.

// local variables              // rms_off is change ignoring displacement
	float rce,a,b,c;

// compute rotational motion about center of mass

//  A1(j,i) * C_EL(i,j)                               // 9 multiplies
	rce =
	 A1(1,1)*C_EL(1,1) + A1(2,1)*C_EL(1,2) + A1(3,1)*C_EL(1,3) +
	 A1(1,2)*C_EL(2,1) + A1(2,2)*C_EL(2,2) + A1(3,2)*C_EL(2,3) +
	 A1(1,3)*C_EL(3,1) + A1(2,3)*C_EL(3,2) + A1(3,3)*C_EL(3,3);

	rmsd = -2*rce + EL2(1) + EL2(2) + EL2(3) + EL(1) + EL(2) + EL(3);
	// 1 muliply 6 adds

// short circuit further calculation if over limit
	if ( rmsd > cutoff ) return; // note: RCM value is undefined if true

// compute center of mass motion

	a = EL(7) - U1(1); // three adds
	b = EL(8) - U1(2);
	c = EL(9) - U1(3);

// 12 muliplies 6 adds  ( A1(1,i)*a + A1(2,i)*b + A1(3,i)*c ) * EL2(6+i)
	rcm =
	 ( A1(1,1)*a + A1(2,1)*b + A1(3,1)*c ) * EL2(7) +
	 ( A1(1,2)*a + A1(2,2)*b + A1(3,2)*c ) * EL2(8) +
	 ( A1(1,3)*a + A1(2,3)*b + A1(3,3)*c ) * EL2(9);

// could precompute sum //  7 muliplies 6 adds
	rcm = -2*rcm + ( EL2(7) * EL2(7) ) + ( EL2(8) * EL2(8) ) +
	 ( EL2(9) * EL2(9) ) + ( a * a ) + ( b * b ) + ( c * c );

// combine the scores
	rmsd += rcm; // 1 add

}

////////////////////////////////////////////////////////////////////////////////
/// @begin eliptic_rmsd
///
/// @brief
/// computes the effect of replacing one fragment with another on
/// a virtual ellipsoid.
/// each fragment is specified by a rotation matrix plus an offset
///
/// @detailed
///
/// @param  A1 - [in/out]? -
/// @param  U1 - [in/out]? -
/// @param  A2 - [in/out]? -
/// @param  U2 - [in/out]? -
/// @param  EL - [in/out]? - the inertial ellipsoid
/// @param  rmsd - [in/out]? -
/// @param  rms_off - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eliptic_rmsd(
	FArray2Da_float A1,
	FArray1Da_float U1,
	FArray2Da_float A2,
	FArray1Da_float U2,
	FArray1Da_float EL, // the inertial ellipsoid
	float & rmsd,
	float & rms_off
)
{
	A1.dimension( 3, 3 );
	U1.dimension( 3 );
	A2.dimension( 3, 3 );
	U2.dimension( 3 );
	EL.dimension( 10 );

// local variables                  // rms_off is change ignoring displacement
	float Drmsd,Drms_off;
// local variables
	FArray2D_float R( 3, 3 );
	FArray1D_float L( 3 );
	FArray1D_float T( 3 );
	FArray1D_float RTL( 3 );
// at some point need to rewrite this to remove the calls to transpose.
// but slow step currently is the mat multiply.
//       double precision does not seem to help, I removed it.
	mat_multiply_transpose3(A1,A2,R); // R = A1^transpose * A2
	rotate(R,U2,T);
	subvec(U1,T,L); // L = U1 - R U2
	transpose3(R);
	rotate(R,L,RTL); //  RTL = R^t L

	Drmsd = EL(1) * (1.0-R(1,1)); // XX
	Drmsd += EL(2) * (1.0-R(2,2)); // YY
	Drmsd += EL(3) * (1.0-R(3,3)); // ZZ

	rmsd = Drmsd - EL(4) * (R(1,2)+R(2,1)); // XY
	rmsd -= EL(5) * (R(1,3)+R(3,1)); // XZ
	rmsd -= EL(6) * (R(2,3)+R(3,2)); // YZ

	Drms_off = EL(7) * ( RTL(1) - L(1) );
	 // X      change in COM due to rotation only
	Drms_off += EL(8) * ( RTL(2) - L(2) ); // Y
	Drms_off += EL(9) * ( RTL(3) - L(3) ); // Z

	Drms_off = 2*Drms_off + L(1)*L(1) + L(2)*L(2) + L(3)*L(3);
	 // 1   change in COM due to translation only, same as T^2

	rmsd = 2*rmsd + Drms_off;
	rms_off = 2*Drmsd;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin eliptic_moments
///
/// @brief
///car not in use, subset of eliptic_cross_moments function
///
/// @detailed
///
/// @param  atoms - [in/out]? -
/// @param  N_atoms - [in/out]? - (logical) number of atoms in list
/// @param  atom_weights - [in/out]? -
/// @param  origin - [in/out]? - offset for atom coordinate system
/// @param  EL - [in/out]? - the inertial moments XX,YY,ZZ,XY,XZ,YZ,X,Y,Z in that order
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eliptic_moments(
	FArray2Da_float atoms,
	int N_atoms, // (logical) number of atoms in list
	FArray1Da_float atom_weights,
	FArray1Da_float origin, // offset for atom coordinate system
	FArray1Da_float EL // the inertial moments XX,YY,ZZ,XY,XZ,YZ,X,Y,Z in that order
)
{
	atoms.dimension( 3, N_atoms );
	atom_weights.dimension( N_atoms );
	origin.dimension( 3 );
	EL.dimension( 10 );

	EL = 0.0; // initialize summation

	for ( int i = 1; i <= N_atoms; ++i ) {
		float const weight_i = atom_weights(i);
		float const a1o1 = atoms(1,i) - origin(1);
		float const a2o2 = atoms(2,i) - origin(2);
		float const a3o3 = atoms(3,i) - origin(3);

		EL(1) += weight_i * ( a1o1 * a1o1 );
		EL(2) += weight_i * ( a2o2 * a2o2 );
		EL(3) += weight_i * ( a3o3 * a3o3 );

		EL(4) += weight_i * ( a1o1 * a2o2 );
		EL(5) += weight_i * ( a1o1 * a3o3 );
		EL(6) += weight_i * ( a2o2 * a3o3 );

		EL(7) += weight_i * a1o1;
		EL(8) += weight_i * a2o2;
		EL(9) += weight_i * a3o3;
		EL(10) += weight_i;
	}

	for ( int i = 1; i <= 9; ++i ) { // normalize
		EL(i) /= EL(10);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin eliptic_cross_moments
///
/// @brief
///
/// @detailed
/// computes various moment matricies as follows:
/// everything is referecened to default coordinate system (ZtC)
/// EL(1-6) are the second order moments about the center of mass
/// in order the entries are:  XX  YY ZZ ZY XZ YZ
/// EL(7-9) are the center of mass coordinates
///  in order:  X Y Z
/// EL2 is the ellipsoid rotated by A2 and offset by U2.
/// EL2(1-6) are second order moments ABOUT CENTER OF MASS (no offset)
/// in order the entries are:  XX YY ZZ XY XZ YZ
/// EL2(7-9) are center of mass of eipsoid
/// in order:  X Y Z
///
///
/// C_EL(3,3) is the cross moments between these two elipsoids.
/// again these moments are computed between both elpses centered at the origin.
/// the order of the entries is different:
///  X1X2 X1Y2 X1Z2
///  Y1X2 Y1Y2 Y1Z2
///  Z1X2 Z1Y2 Z1Z2
///  (or maybe its the transpose, I fergit)
///
/// @param  atoms - [in/out]? -
/// @param  N_atoms - [in/out]? - (logical) number of atoms in list
/// @param  atom_weights - [in/out]? -
/// @param  A2 - [in/out]? -
/// @param  U2 - [in/out]? -
/// @param  origin - [in/out]? - offset for atom coordinate system
/// @param  C_EL - [in/out]? - the inertial cross moments
/// @param  EL2 - [in/out]? -
/// @param  EL - [in/out]? -
/// @param  ZtC - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eliptic_cross_moments(
	FArray2Da_float atoms,
	int N_atoms, // (logical) number of atoms in list
	FArray1Da_float atom_weights,
	FArray2Da_float A2,
	FArray1Da_float U2,
	FArray1Da_float origin, // offset for atom coordinate system
	FArray2Da_float C_EL, // the inertial cross moments
	FArray1Da_float EL2,
	FArray1Da_float EL,
	FArray2Da_float ZtC
)
{

	if ( N_atoms == 0 ) {
		std::cout << "ERROR:  N_atoms=0 in eliptic_cross_moments" << std::endl;
		std::cout << "Stopping..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	atoms.dimension( 3, N_atoms );
	atom_weights.dimension( N_atoms );
	A2.dimension( 3, 3 );
	U2.dimension( 3 );
	origin.dimension( 3 );
	C_EL.dimension( 3, 3 );
	EL2.dimension( 10 );
	EL.dimension( 10 );
	ZtC.dimension( 3, 3 );

	EL = 0.0; // initialize summation

	double norm = 0.0; // init sum

// compute center of mass position
	for ( int i = 1; i <= N_atoms; ++i ) {
		float const weight_i = atom_weights(i);
		EL(7) += weight_i * atoms(1,i);
		EL(8) += weight_i * atoms(2,i);
		EL(9) += weight_i * atoms(3,i);
		norm += weight_i;
	}
	for ( int j = 7; j <= 9; ++j ) { // normalize summation
		EL(j) /= norm;
	}
	EL(10) = norm;

// compute the 2nd order moments about the center of mass
	FArray1D_float temp( 3 );
	for ( int i = 1; i <= N_atoms; ++i ) {
		float const weight_i = atom_weights(i);

		subvec(atoms(1,i),EL(7),temp); // remove mean.  could do this later

		EL(1) += weight_i * ( temp(1) * temp(1) );
		EL(2) += weight_i * ( temp(2) * temp(2) );
		EL(3) += weight_i * ( temp(3) * temp(3) );

		EL(4) += weight_i * temp(1) * temp(2);
		EL(5) += weight_i * temp(1) * temp(3);
		EL(6) += weight_i * temp(2) * temp(3);
	}
	for ( int j = 1; j <= 6; ++j ) { // NORMalize summation
		EL(j) /= norm;
	}

// compute center of mass relative to origin of rotation in protein
	subvec(EL(7),origin,EL(7)); // offest by origin
// rotate ellipse to default coordinate system (Z)
	eliptic_rotate(ZtC,EL,EL); // this rotates both quadratic and com

// copy quadratic moments into a symmetric matrix
	FArray2D_float C_temp( 3, 3 );
	C_temp(1,1) = EL(1); //   XX XY XZ
	C_temp(2,2) = EL(2); //   XY YY YZ
	C_temp(3,3) = EL(3); //   XZ YZ ZZ
	C_temp(1,2) = EL(4);
	C_temp(2,1) = EL(4);
	C_temp(1,3) = EL(5);
	C_temp(3,1) = EL(5);
	C_temp(2,3) = EL(6);
	C_temp(3,2) = EL(6);

// offset it:  to produce moments between elipse and offset elipse.
// But it turns out the moments are unchanged by offset operation! so do nothing
// next rotate it to produce cross moment between rotated and offset elipse and original.
	mat_multiply3(A2,C_temp,C_EL);
	transpose3(C_EL);

// generate moments of EL2 (ie rotate and offset EL moments)
// copy moments again
// rotate them
	eliptic_rotate(A2,EL,EL2);

// compute com of offest and rotated ellipse

	subvec(EL(7),U2,EL2(7)); // new com is offset
	rotate_in_place(A2,EL2(7)); // new com is rotated

}

////////////////////////////////////////////////////////////////////////////////
/// @begin eliptic_rotate
///
/// @brief
/// applies rotation matrix A1 to eliptic moments EL.  results returned in
/// new matrix (can be in place)
/// output  ( can be inplace)
///
///
/// @detailed
///
/// @param  A1 - [in/out]? -
/// @param  EL - [in/out]? -
/// @param  EL_out - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eliptic_rotate(
	FArray2Da_float A1,
	FArray1Da_float EL,
	FArray1Da_float EL_out
)
{
	A1.dimension( 3, 3 );
	EL.dimension( 10 );
	EL_out.dimension( 10 );

// local
	FArray2D_float TM( 3, 3 );
	FArray2D_float TM2( 3, 3 );
	FArray1D_float T( 3 );
	FArray1D_float T2( 3 );

// copy quadratic moments into a symmetric matrix
	TM(1,1) = EL(1); //   XX XY XZ
	TM(2,2) = EL(2); //   XY YY YZ
	TM(3,3) = EL(3); //   XZ YZ ZZ
	TM(1,2) = EL(4);
	TM(2,1) = EL(4);
	TM(1,3) = EL(5);
	TM(3,1) = EL(5);
	TM(2,3) = EL(6);
	TM(3,2) = EL(6);

//        compute A1 TM A1^T, where ^T means transpose

	mat_multiply3(A1,TM,TM2); //  TM2 = A1 TM
	transpose3(TM2); //  TM2 = (A1 TM) ^T  = TM^T  A1^T  = TM A1^T
	mat_multiply3(A1,TM2,TM); //  TM = A1 TM2 = A1 TM A1^T   as desired


// copy symmetric matrix back into eliptic moments list
	EL_out(1) = TM(1,1);
	EL_out(2) = TM(2,2);
	EL_out(3) = TM(3,3);
	EL_out(4) = TM(1,2);
	EL_out(5) = TM(1,3);
	EL_out(6) = TM(2,3);


// now do CM components
	T(1) = EL(7); // single precision
	T(2) = EL(8);
	T(3) = EL(9);

	rotate(A1,T,T2);

	EL_out(7) = T2(1);
	EL_out(8) = T2(2);
	EL_out(9) = T2(3);

//car what about EL(10)? need to copy?
}

////////////////////////////////////////////////////////////////////////////////
/// @begin eliptic_rotate_cross_moments
///
/// @brief
/// applies rotation matrix A1 to eliptic moments EL.  results returned in place
///
/// @detailed
///
/// @param  A1 - [in/out]? -
/// @param  C_EL - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eliptic_rotate_cross_moments(
	FArray2Da_float A1,
	FArray2Da_float C_EL
)
{
	A1.dimension( 3, 3 );
	C_EL.dimension( 3, 3 );

// local
	FArray2D_float TM( 3, 3 );
	FArray2D_float TM2( 3, 3 );
// debug
	FArray2D_float crap( 3, 3 );

//       copy the A1 matrix
	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			TM(i,j) = A1(j,i); // transposed copy
			crap(i,j) = C_EL(i,j);
		}
	}

//        compute A1 C_EL A1^T, where ^T means transpose

	mat_multiply3(crap,TM,TM2);
	mat_multiply3(A1,TM2,C_EL);
	debug_mat("ercm non-transpoded*", C_EL);
	mat_multiply3(A1,crap,TM2);
	mat_multiply3(TM2,TM,C_EL);
	debug_mat("ercm non-transpoded*", C_EL);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin eliptic_translate
///
/// @brief
/// applies translation U1 to eliptic moments EL.  results returned in place
/// note moments must be normalized to unit-mass
///
/// @detailed
///
/// @param  U1 - [in/out]? -
/// @param  EL - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eliptic_translate(
	FArray1Da_float U1,
	FArray1Da_float EL
)
{
	U1.dimension( 3 );
	EL.dimension( 10 );

//  quadratic moments

	EL(1) += 2.0*EL(7)*U1(1) + ( U1(1) * U1(1) ); // XX
	EL(2) += 2.0*EL(8)*U1(2) + ( U1(2) * U1(2) ); // YY
	EL(3) += 2.0*EL(9)*U1(3) + ( U1(3) * U1(3) ); // ZZ

	EL(4) += EL(7)*U1(1) + EL(8)*U1(2) + U1(1)*U1(2); // XY
	EL(5) += EL(7)*U1(1) + EL(9)*U1(3) + U1(1)*U1(3); // XZ
	EL(6) += EL(8)*U1(2) + EL(9)*U1(3) + U1(2)*U1(3); // YZ
// linear moments
	EL(7) += U1(1); // X
	EL(8) += U1(2); // Y
	EL(9) += U1(3); // Z

}

////////////////////////////////////////////////////////////////////////////////
/// @begin frag_effect
///
/// @brief
/// computes rotation A1,   needed align P1,P2,P3 into Q1,Q2,Q3
/// computes delta D1 = Q1-P1
///
/// @detailed
///
/// @param  P1 - [in/out]? -
/// @param  P2 - [in/out]? -
/// @param  P3 - [in/out]? -
/// @param  Q1 - [in/out]? -
/// @param  Q2 - [in/out]? -
/// @param  Q3 - [in/out]? -
/// @param  A1 - [in/out]? -
/// @param  D1 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
frag_effect(
	FArray1Da_float P1,
	FArray1Da_float P2,
	FArray1Da_float P3,
	FArray1Da_float Q1,
	FArray1Da_float Q2,
	FArray1Da_float Q3,
	FArray2Da_float A1,
	FArray1Da_float D1
)
{
	P1.dimension( 3 );
	P2.dimension( 3 );
	P3.dimension( 3 );
	Q1.dimension( 3 );
	Q2.dimension( 3 );
	Q3.dimension( 3 );
	A1.dimension( 3, 3 );
	D1.dimension( 3 );

	angles_align_transform(P1,P2,P3,Q1,Q2,Q3,A1);

	D1(1) = Q2(1) - P2(1);
	D1(2) = Q2(2) - P2(2);
	D1(3) = Q2(3) - P2(3);

	// is this what I want?

}

////////////////////////////////////////////////////////////////////////////////
/// @begin debug_mat
///
/// @brief
///
/// @detailed
///
/// @param  name - [in/out]? -
/// @param  mat - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
debug_mat(
	std::string const & name,
	FArray2Da_float mat
)
{
	mat.dimension( 3, 3 );
	std::string::size_type const k = name.find( '*' );

	if ( k != std::string::npos ) std::cout << name.substr( 0, k ) << std::endl;
	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			std::cout << SS( mat(j,i) );
		} std::cout << std::endl;
	}
	std::cout << " ______________" << std::endl;
}

void
debug_vec(
	std::string const & name,
	FArray1Da_float vec
)
{
	vec.dimension( 3 );
	std::string::size_type const k = name.find( '*' );

	if ( k != std::string::npos ) std::cout << name.substr( 0, k ) << std::endl;
	for ( int i = 1; i <= 3; ++i ) {
		std::cout << SS( vec(i) );
	} std::cout << std::endl;
	std::cout << " ______________" << std::endl;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin Ddebug_mat
///
/// @brief
///
/// @detailed
///
/// @param  name - [in/out]? -
/// @param  mat - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Ddebug_mat(
	std::string const & name,
	FArray2Da_double mat
)
{
	mat.dimension( 3, 3 );
	std::string::size_type const k = name.find( '*' );

	if ( k != std::string::npos ) std::cout << name.substr( 0, k ) << std::endl;
	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			std::cout << SS( mat(j,i) );
		} std::cout << std::endl;
	}
	std::cout << " ______________" << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin debug_3
///
/// @brief
///
/// @detailed
///
/// @param  name - [in/out]? -
/// @param  a1 - [in/out]? -
/// @param  b1 - [in/out]? -
/// @param  c1 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
debug_3(
	std::string const & name,
	FArray1Da_float a1,
	FArray1Da_float b1,
	FArray1Da_float c1
)
{
	a1.dimension( 3 );
	b1.dimension( 3 );
	c1.dimension( 3 );

	std::string::size_type const k = name.find( '*' );

	if ( k != std::string::npos ) std::cout << name.substr( 0, k ) << std::endl;

	for ( int i = 1; i <= 3; ++i ) {
		std::cout << SS( a1(i) ) << SS( b1(i) ) << SS( c1(i) ) << std::endl;
	}
	std::cout << " ______________" << std::endl;
}
