// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: sheffler $


// Rosetta Headers
#include "etable_class.h"
#include "after_opts.h"
#include "etable.h"
#include "files_paths.h"
#include "fullatom_setup.h"
#include "hbonds_ns.h"
#include "namespace_fa_lj_rep_slope_setting.h"
#include "namespace_fullatom_init.h"
#include "namespace_lj_etable_settings.h"
#include "param.h"
#include "param_pack.h"
#include "pdbstatistics_pack.h"
#include "read_paths.h"
#include "spline.h"
#include "weights_manager.h"
#include "water_ns.h"
#include "score.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <iostream>


////////////////////////////////////////////////////////////////////////////////
/// @begin Etable::setup_Etable
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void Etable::setup_Etable( bool const standard_weights, float const lj_rep, float const Wrad ){

	using namespace param;
	using namespace pdbstatistics_pack;

	if ( done_setup_ ) return;

	etable::highres_lj_lin_sig = lj_rep;
	etable::fa_Wradius = Wrad;

	ljatr.dimension(  MAX_ETABLE_DISBINS(), MAX_ATOMTYPES(), MAX_ATOMTYPES() );
	ljrep.dimension(  MAX_ETABLE_DISBINS(), MAX_ATOMTYPES(), MAX_ATOMTYPES() );
	solv1.dimension(  MAX_ETABLE_DISBINS(), MAX_ATOMTYPES(), MAX_ATOMTYPES() );
	solv2.dimension(  MAX_ETABLE_DISBINS(), MAX_ATOMTYPES(), MAX_ATOMTYPES() );
	dljatr.dimension( MAX_ETABLE_DISBINS(), MAX_ATOMTYPES(), MAX_ATOMTYPES() );
	dljrep.dimension( MAX_ETABLE_DISBINS(), MAX_ATOMTYPES(), MAX_ATOMTYPES() );
	dsolv.dimension(  MAX_ETABLE_DISBINS(), MAX_ATOMTYPES(), MAX_ATOMTYPES() );

  elec_rep.dimension( MAX_ETABLE_DISBINS(), 1, 1 );
  d_elec_rep.dimension( MAX_ETABLE_DISBINS(), 1, 1 );
  elec_atr.dimension( MAX_ETABLE_DISBINS(), 1, 1 );
  d_elec_atr.dimension( MAX_ETABLE_DISBINS(), 1, 1 );

	// Apply weights and build the energy tables
	RetrieveWeightsToCurrent( PW_STANDARD );
	apply_packer_weights( standard_weights );

	// Call select_lj_switchpoint, which calls fa_make_pairenergy_table
	lj_etable_settings::lj_lsr_setting = "undef"; // force "select_lj_switchpoint" to be active
	select_lj_switchpoint( fa_lj_rep_slope_setting::fa_lj_rep_slope );

	done_setup_ = true;
	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin Etable::read_Etable
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void Etable::read_Etable(){

	using namespace param;
	using namespace pdbstatistics_pack;
	using namespace std;

	map< string, FArray3D_float* > etables;
	etables["ljatr"]  = &ljatr;
	etables["ljrep"]  = &ljrep;
	etables["dljatr"] = &dljatr;
	etables["dljrep"] = &dljrep;
	etables["solv1"]  = &solv1;
	etables["solv2"]  = &solv2;
	etables["dsolv"]  = &dsolv;

	if ( truefalseoption("mod_etable") ) {
		for (map<string,FArray3D_float*>::iterator i = etables.begin();
				 i != etables.end(); i++) {
			string ename = i->first;
			if( truefalseoption("input_etable_"+ename) ) {
				string fname;
				stringafteroption("input_etable_"+ename,"",fname);
				utility::io::izstream & input( open_data_file( fname.c_str() ) );
				input_etable(*(i->second),ename,input);
				input.close();
			}
		}
	}

	if ( truefalseoption("input_etables") ) {
		string tag = stringafteroption("input_etables");
		for (map<string,FArray3D_float*>::iterator i = etables.begin();
				 i != etables.end(); i++) {
			string ename = i->first;
			string fname = tag+"."+ename+".etable";
			utility::io::izstream & input( open_data_file( fname.c_str() ) );
			input_etable(*(i->second),ename,input);
			input.close();
		}
	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin Etable::write_Etable
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void Etable::write_Etable(){

	using namespace param;
	using namespace pdbstatistics_pack;
	using namespace std;

	map< string, FArray3D_float* > etables;
	etables["ljatr"]  = &ljatr;
	etables["ljrep"]  = &ljrep;
	etables["dljatr"] = &dljatr;
	etables["dljrep"] = &dljrep;
	etables["solv1"]  = &solv1;
	etables["solv2"]  = &solv2;
	etables["dsolv"]  = &dsolv;
  etables["elec_rep"] = &elec_rep;
  etables["elec_atr"] = &elec_atr;
  etables["delec_rep"] = &d_elec_rep;
  etables["delec_atr"] = &d_elec_atr;


	if ( truefalseoption("output_etables") ) {
		cerr << "OUTPUT ETABLE" << endl;
		string header;
		stringafteroption("output_etables","",header);
		if( header[0]=='-' )
			header="";
		header = files_paths::pdb_out_path+header;
		for (map<string,FArray3D_float*>::iterator i = etables.begin();
				 i != etables.end(); i++) {
			string ename = i->first;
			string fname = header+"."+ename+".etable";
			cerr << "output_etable: writing etable: " << ename << " to " << fname << endl;
			ofstream out(fname.c_str());
			output_etable(*(i->second),ename,out);
			out.close();
		}
	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin select_lj_switchpoint
///
/// @brief sets up the strength of close-contact lj repulsion
///
/// @detailed
///
/// for each atomtype pair, lj repulsive strength below an (atom
/// separation distance)/sigma ratio defined by
/// fa_lj_switch_dis2sigma is set to a value derived from a
/// linear extension of the slope of the true lj value at the switchover point
///
/// @param[in]   setting - in - char string in set ("lowres","highres","undef") ;
///
/// @global_read
///
/// etable.h: lowres_lj_lin_sig, highres_lj_lin_sig
///
/// @global_write
///
/// etable.h: fa_lj_switch_dis2sigma
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void Etable::select_lj_switchpoint( std::string const & setting ) {

//car namespace from initialize_fullatom

	using namespace etable;
	using namespace lj_etable_settings;

	if ( setting == lj_lsr_setting && lj_lsr_setting != "undef" ) return; // no change

	lj_lsr_setting = setting;
	if ( lj_lsr_setting == "lowres" ) {
		fa_lj_switch_dis2sigma = lowres_lj_lin_sig;
	} else if ( lj_lsr_setting == "highres" ) {
		fa_lj_switch_dis2sigma = highres_lj_lin_sig;
	} else {
		std::cout << "ERROR:: undefined setting for " <<
		 "fa_lj_switch_dis2sigma : " << setting << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	fa_make_pairenergy_table();

}


////////////////////////////////////////////////////////////////////////////////
/// @begin Etable::fa_make_pairenergy_table
///
/// @brief calculate fast lookup arrays for vdw and solvation energy
///
/// @detailed
///
/// Several energies are precomputed when fullatom mode is initialized and
/// stored in lookup tables to speed fullatom calculation. Currently
/// pre-computed values are the Lennard-Jones van der Waals approximation
/// (lj) and the Lazaridis-Karplus implicit solvation function (lk). For
/// each of these energies the derivative w.r.t. atom pair separation distance is
/// calculated and stored as well. Note that the lj energy is artificially
/// divided into atractive and repulsive components.
///
/// @global_read
///
/// pdbstatistics_pack.h: several threshold distances
/// energy.h: just about everything should be used in the tree of
///           etable functions
///
/// @global_write
///
/// the etable: ljatr,dljatr,ljrep,dljrep,solvE
///
/// @remarks
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void Etable::fa_make_pairenergy_table() {

	using namespace etable;
	using namespace param;
	using namespace pdbstatistics_pack;

	//  locals
	double dis2,dis,dis2_step;
	float atrE,d_atrE,repE,d_repE,solvE1,solvE2,dsolvE;


	//  pre-calculated coefficients for high-speed potential calculation
	//    computed in initialization function
	//  Note: +1 atom type is a disulfide-bonded cysteine
	//        (only if you are using the old disulfide hack)
	FArray2D_double fa_lj_sigma( MAX_ATOMTYPES()() + 1, MAX_ATOMTYPES()() + 1 );
	FArray2D_double fa_lj_r6_coeff( MAX_ATOMTYPES()() + 1, MAX_ATOMTYPES()() + 1 );
	FArray2D_double fa_lj_r12_coeff( MAX_ATOMTYPES()() + 1, MAX_ATOMTYPES()() + 1 );
	FArray2D_double fa_lj_switch_intercept( MAX_ATOMTYPES()() + 1, MAX_ATOMTYPES()() + 1 );
	FArray2D_double fa_lj_switch_slope( MAX_ATOMTYPES()() + 1, MAX_ATOMTYPES()() + 1 );
	FArray1D_double fa_lk_inv_lambda2( MAX_ATOMTYPES()() );
	FArray2D_double fa_lk_coeff( MAX_ATOMTYPES()(), MAX_ATOMTYPES()() );
	FArray2D_double fa_lk_min_dis2sigma_value( MAX_ATOMTYPES()() + 1, MAX_ATOMTYPES()() + 1 );

	// parameters to calculate the damping at max_dis range
	float damping_thresh_dis2;
	int damping_disbins, normal_disbins;
	float dljatr_damp, dljrep_damp, dsolv1_damp, dsolv2_damp;
	float intercept_ljatr_damp, intercept_ljrep_damp;
	float intercept_solv1_damp, intercept_solv2_damp;

	//  etable parameters
	std::cout << "Starting energy table calculation" << std::endl;

	std::cout << "Energy table parameter set: " << fa_etable_label << ' ' <<
	 fa_etable_revision.substr( 1, fa_etable_revision.length() - 3 ) << ' ' <<
	 fa_etable_date.substr( 1, fa_etable_date.length() - 3 ) << std::endl;

	// ctsa - values precomputed from etable parameters:
	//   these were all originally constants but need
	//   to be calculated here b/c the switch value:
	//   fa_lj_switch_dis2sigma
	//   is set at runtime
	fa_lj_switch_sigma2dis = 1.0/fa_lj_switch_dis2sigma;

	// ctsa - value of the lennard-jones potential at the linear
	//   switch point divided by the wdepth (note that this
	//   coefficient is independent of atomtype)
	fa_lj_switch_value2wdepth = std::pow( fa_lj_switch_sigma2dis, 12 ) -
	 2.0 * std::pow( fa_lj_switch_sigma2dis, 6 );

	// ctsa - slope of the lennard-jones potential at the linear
	//   switch point times sigma divided by wdepth (note that this
	//   coefficient is independent of atomtype)
	fa_lj_switch_slope_sigma2wdepth = -12.0 * (
	 std::pow( fa_lj_switch_sigma2dis, 13 ) -
	 std::pow( fa_lj_switch_sigma2dis, 7 ) );

	//  initialize non-distance dependent coefficients
	fa_precalc_etable_coefficients(fa_lj_sigma, fa_lj_r6_coeff, fa_lj_r12_coeff,
	 fa_lj_switch_intercept, fa_lj_switch_slope, fa_lk_inv_lambda2, fa_lk_coeff,
	 fa_lk_min_dis2sigma_value );

	//  calc distance**2 step per bin
	dis2_step = 1.0 / fa_bins_per_A2;

	//  get the number of damping disbins
	if ( fa_add_long_range_damping ) {
		float const fa_dif = fa_max_dis - fa_long_range_damping_length;
		damping_thresh_dis2 = fa_max_dis2 - ( fa_dif * fa_dif );
		damping_disbins = static_cast< int >( damping_thresh_dis2*fa_bins_per_A2 );
		normal_disbins = etable_disbins-damping_disbins;
	} else {
		normal_disbins = etable_disbins;
	}

	//  ctsa - step through distance**2 bins and calculate potential
	for ( int atype1 = 1, atype_end = MAX_ATOMTYPES(); atype1 <= atype_end; ++atype1 ) {
		for ( int atype2 = 1; atype2 <= atype_end; ++atype2 ) {
			//  ctsa - normal bins have their lj and lk values
			//    calculated analytically
			for ( int disbin = 1; disbin <= normal_disbins; ++disbin ) {
				dis2 = ( disbin - 1 ) * dis2_step;

				fa_calc_etable_value(dis2,atype1,atype2,atrE,d_atrE,repE,d_repE,solvE1,
				 solvE2,dsolvE,fa_lj_sigma, fa_lj_r6_coeff,fa_lj_r12_coeff,
				 fa_lj_switch_intercept,fa_lj_switch_slope, fa_lk_inv_lambda2,
				 fa_lk_coeff, fa_lk_min_dis2sigma_value);

				ljatr(disbin,atype2,atype1) = atrE;
				dljatr(disbin,atype2,atype1) = d_atrE;
				ljrep(disbin,atype2,atype1) = repE;
				dljrep(disbin,atype2,atype1) = d_repE;
				solv1(disbin,atype2,atype1) = solvE1;
				solv2(disbin,atype2,atype1) = solvE2;
				dsolv(disbin,atype2,atype1) = dsolvE;
			}

			if ( !fa_add_long_range_damping ) goto L456;

			// ctsa - remaining bins damp to 0. on a linear path
			dljatr_damp = -ljatr(normal_disbins,atype2,atype1) /
			 fa_long_range_damping_length;
			dljrep_damp = -ljrep(normal_disbins,atype2,atype1) /
			 fa_long_range_damping_length;
			dsolv1_damp = -solv1(normal_disbins,atype2,atype1) /
			 fa_long_range_damping_length;
			dsolv2_damp = -solv2(normal_disbins,atype2,atype1) /
			 fa_long_range_damping_length;

			intercept_ljatr_damp = -dljatr_damp*fa_max_dis;
			intercept_ljrep_damp = -dljrep_damp*fa_max_dis;
			intercept_solv1_damp = -dsolv1_damp*fa_max_dis;
			intercept_solv2_damp = -dsolv2_damp*fa_max_dis;

			for ( int disbin = normal_disbins+1; disbin <= etable_disbins; ++disbin ) {
				dis2 = ( disbin - 1 ) * dis2_step;
				dis = std::sqrt(dis2);

				ljatr(disbin,atype2,atype1) = intercept_ljatr_damp + dis * dljatr_damp;
				ljrep(disbin,atype2,atype1) = intercept_ljrep_damp + dis * dljrep_damp;
				solv1(disbin,atype2,atype1) = intercept_solv1_damp + dis * dsolv1_damp;
				solv2(disbin,atype2,atype1) = intercept_solv2_damp + dis * dsolv2_damp;

				dljatr(disbin,atype2,atype1) = dljatr_damp;
				dljrep(disbin,atype2,atype1) = dljrep_damp;
				dsolv(disbin,atype2,atype1)  = dsolv1_damp + dsolv2_damp;
			}

L456:

			//  ctsa - set last bin of all values to zero
			ljatr(etable_disbins,atype2,atype1) = 0.0;
			ljrep(etable_disbins,atype2,atype1) = 0.0;
			solv1(etable_disbins,atype2,atype1) = 0.0;
			solv2(etable_disbins,atype2,atype1) = 0.0;
		}
	}

// SJF ***** precomputing the distance-dependent dielectric coefficient
  for ( int disbin = 1; disbin <= etable_disbins; ++disbin ) {
    static float prefactor = 322.0637 * 1.590; // 1.590 was computed to match the mean elec energy of a benchmark of 47 docked structures having ~20k decoys in total compared to the mean obtained by a non-smoothed w_elec computed by Ora & Chu
    static float cutoff3 = std::pow( 5.5, 3);
    static float cutoff2 = std::pow( 5.5, 2);

// The repulsive potential is defined differently from the attractive in order
// to reduce the effects of the electrostatic potential on the optimal
// distances between hbonding atoms. Since the hbonding optimum is below 2.5Ang
// the attractive potential levels at this point, whereas the repulsive
// potential increases until 2.25

    dis2 = ( disbin - 1) * dis2_step;

// the lower cutoff for the attractive potential is set higher so as not to change the minimum
// for the hbonding potential (~1.8Ang).
    float rep_lowercutoff = 1.5;
    float rep_lowercutoff2 = rep_lowercutoff * rep_lowercutoff;

    float atr_lowercutoff = 2.25;
    float atr_lowercutoff2 = atr_lowercutoff * atr_lowercutoff;

    if ( dis2 <= rep_lowercutoff2 )
      dis2 = rep_lowercutoff2;
    dis  = std::sqrt( dis2 );

    elec_rep( disbin, 1, 1 ) = prefactor *
       ( 1.0 / dis2 + // distance dependent dielectric ( 1 / r)
         2.0 * dis / cutoff3 - 3.0 / cutoff2 ); // smoothing to make the potential and its derivative drop to zero at the 6angstrom cutoff

   if ( dis <= rep_lowercutoff )  d_elec_rep( disbin, 1, 1) = 0.0;
   else d_elec_rep( disbin, 1, 1) = prefactor * ( -2.0 / dis2 / dis + 2.0 / cutoff3 );

   if ( dis <= atr_lowercutoff) { // lower cutoff for attractive potential
      dis2 = atr_lowercutoff2;
      dis  = atr_lowercutoff;
      d_elec_atr( disbin, 1, 1) = 0.0;
    }
    else d_elec_atr( disbin, 1, 1) = d_elec_rep( disbin, 1, 1);

    elec_atr( disbin, 1, 1 ) = prefactor *
       ( 1.0 / dis2 +
         2.0 * dis / cutoff3 - 3.0 / cutoff2 );
  }
  elec_atr( etable_disbins, 1, 1 ) = 0.0;
  elec_rep( etable_disbins, 1, 1 ) = 0.0;
  d_elec_atr( etable_disbins, 1, 1) = 0.0;
  d_elec_rep( etable_disbins, 1, 1) = 0.0;

// SJF
// item (2) of modify_pot below is commented out because the electric potential is supposed to
// take care of the underlying problem

//db  the following function call modifies the potential in three ways:
//db     (1) the solvation energy for nonpolar atoms is held constant below
//db     4.2A to avoid shifting the minimum in the LJ potential.
//db     (2) a short range repulsion is added between backbone oxygens which are
//db     otherwise brought too close together by the LJatr.
//db     (3) the range of the repulsive interaction between non polar hydrogens is
//db     increased slightly.  (this is currently commented out because the effects
//db     on design have not been tested)

//db  all three modifications are based on inspection of the atom pair distributions
//db  after extensive refinement.

	modify_pot();

	std::cout << "Finished calculating energy tables." << std::endl;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin modify_pot
///
/// @brief modify Etable to better treat 0-0, C-C, and H-H interactions
///
/// @detailed
///$$$ the Etables are  modified in three ways:
///$$$ (1) the LK solvation energy is set to a constant below 4.2A to avoid shifting the position
///$$$ of the minimum on the LJatr potential.  in refined decoys the peak in the C C pair
///$$$ distribution function shifts to around 3.8 A from ~4.0A in native structures; the LJatr
///$$$ potential has a minimum at 4.0A but this shifts towards smaller values because of the LK
///$$$ solvation term, which became increasingly favorable at shorter distances
///$$$ (2) the backbone carbonyl oxygen-carbonyl oxygen LJrep term has been modified to become
///$$$ moderately repulsive at distances less than 3.6A.  this is to counteract the favorable
///$$$ LJatr between the atoms (which have radii of ~1.4A and so a minimum at ~2.8A; very few
///$$$ counts are observed in the pdb until around 3.2A) which leads to a significant shift in the
///$$$ O O pair distribution function towards smaller values in refined decoys.  the repulsion is
///$$$ a temporary proxy for the lack of explicit electrostatic repulsion in the current force
///$$$ field.
///$$$ (3) a third soft repulsion between non polar hydrogens that was also based on comparison of
///$$$ refined decoy to native pdf's is currently commented out as the effects on packing have not
///$$$ been tested.  it was observed that the protons tend to pile up at just beyond the point
///$$$ where the repulsion becomes strong, perhaps due to a general tendency to overcontraction
///$$$ because of long range LJatr interactions not compensated by interactions with solvent
///$$$ (which are of course missing)
///
/// @global_read
/// pdbstatistics_pack.h:  ljatr,dljatr,ljrep, dljrep, solv1,solv2,dsolv
///
/// @global_write
/// pdbstatistics_pack.h:  ljatr,dljatr,ljrep, dljrep, solv1,solv2,dsolv
///
/// @remarks
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void Etable::modify_pot() {

	using namespace pdbstatistics_pack;
	using namespace std;

	bool mod_hhrep      = truefalseoption("mod_hhrep");
	bool SMOOTH_ETABLES = param_pack::packer_logical::smooth_etable;
	float const h = fullatom_setup_ns::mod_hhrep_height;
	float const c = fullatom_setup_ns::mod_hhrep_center;
	float const w = fullatom_setup_ns::mod_hhrep_width;
	float const e = fullatom_setup_ns::mod_hhrep_exponent;

	if(mod_hhrep) {
		std::cout << "fullatom_setup: modifying h-h repulsion "
							<< " hhrep center "   << c
							<< " hhrep height "   << h
							<< " hhrep width "    << w
							<< " hhrep exponent " << e
							<< std::endl;
	}

	{

  		float const bin = ( 4.2 * 4.2 / .05 ) + 1.0; //SGM Off-by-1 bug fix: Add + 1.0: Index 1 is at distance^2==0
		int const ibin( static_cast< int >( bin ) );
 		for ( int k = 1; k <= etable_disbins; ++k ) {
 			float const dis = std::sqrt( ( k - 1 ) * .05f ); //SGM Off-by-1 bug fix: k -> ( k - 1 )


			if( !SMOOTH_ETABLES ) {
				if ( dis < 4.2 ) {
					for ( int kk = 3; kk <= 6; ++kk ) {
						for ( int j = 3; j <= 6; ++j ) {
							solv1(k,j,kk) = solv1(ibin,j,kk);
							solv1(k,kk,j) = solv1(ibin,kk,j);
							solv2(k,j,kk) = solv2(ibin,j,kk);
							solv2(k,kk,j) = solv2(ibin,kk,j);
							dsolv(k,j,kk) = 0.0;
							dsolv(k,kk,j) = 0.0;
						}
					}
				}
			}

//   if ( !get_simple_elec() ) { // SJF No need to account for carbonyl oxygen
// repulsion especially, if electrostatics is on

//   push carbonyl oxygens (in beta sheets) apart.  a proxy for the missing
//   electrostatic repulsion needed to counteract the LJatr which pulls the oxyens
//   together
			if ( dis <= 3.6 ) {
				float const fac = std::max( dis - 3.6f, -1.5f );
				ljrep(k,20,20) += 2 * ( fac * fac );
				dljrep(k,20,20) += 4 * fac;
			}
//   }
//  the following gives peak at 2.4 in 22 and 23 interactions. maybe push out a
//  bit further.  (this is commented out because effects on design have not been
//  tested)

			// use one half of a single term polynomial
			// as the repulsive term for apolar hydrogens (types 23 & 24)

			if( mod_hhrep ) {
				if( dis < c ) {
					for ( int j = 23; j <= 24; ++j ) {
						for ( int kk = 23; kk <= 24; ++kk ) {
							ljrep(k,j,kk)  = h * pow( min(0.0f, (dis-c)/w ), e );//  +
							dljrep(k,j,kk) = h * e / w * pow( min(0.0f, dis-c)/w, e-1 ) ;//  +
						}
					}
				}
			}

		} // end  for ( int k = 1; k <= etable_disbins; ++k ) {


	} //Objexx:SGM Extra {} scope is a VC++ work-around

	if( SMOOTH_ETABLES ) {
		smooth_etables();
	}

}


void Etable::smooth_etables() {

	using namespace pdbstatistics_pack;
	using namespace std;

	int   const SWITCHBIN = 500;
	float const SWITCHGAP = 0.5;
	int   const SOLV_LOW_SWITCH_SMOOTH = 20;

	cout << "fullatom_setup.cc: smoothing etables" << endl;
	//using namespace decoy_features_ns;
	FArray1D_float fa_dist(etable_disbins);
	for ( int kk = 1; kk <= etable_disbins; ++kk ){
		fa_dist(kk) = std::sqrt( ( kk - 1 ) * .05f ); //SGM Off-by-1 bug fix: kk -> ( kk - 1 )
	}
	for ( int at1 = 1; at1 <= 25; ++at1 ) {
		for ( int at2 = 1; at2 <= 25; ++at2 ) {
			{
				/////////////////////////////////////////////////////////
				// for LJ, smooth from inflection point with spline
				/////////////////////////////////////////////////////////
				float maxd = -999;
				int  imaxd = -1;
				for( int kk = 1; kk <= 499; ++kk ) {//etable_disbins; ++kk ) {
					if( dljatr(kk,at1,at2) > maxd) {
						maxd = dljatr(kk,at1,at2);
						imaxd = kk;
					}
				}
				if( maxd > 0 ) {
					for( int kk = imaxd; kk <= etable_disbins; ++kk ) {
						ljatr(kk,at1,at2) = -9e9;
					}
					ljatr(etable_disbins,at1,at2)  = 0;
					dljatr(etable_disbins,at1,at2) = 0;
					//cout << "ljatr " << at1 << ' ' << at2 << ' ';
					//std::cout<< "fadist.size1 " << fa_dist.size1() << " ljatrSize " << ljatr.size1() << std::endl;
					spline_fill_marked_gaps( fa_dist, ljatr(1,at1,at2), dljatr(1,at1,at2) , -9e9 );
				}
			}

			/////////////////////////////////////////////////////
			// smooth solvation terms by scaling to make
			// smooth junction with e = 0 at 5.5A
			////////////////////////////////////////////////////
			FArray1D_float dsolv1(etable_disbins,-9e9);
			FArray1D_float dsolv2(etable_disbins,-9e9);
			{
				//now solv1 terms
				float const de = ( solv1(SWITCHBIN,at1,at2)-solv1(SWITCHBIN-1,at1,at2) )/
					( fa_dist(SWITCHBIN)-fa_dist(SWITCHBIN-1) );
				//float const x = fa_dist(SWITCHBIN);
				float const y = solv1(SWITCHBIN,at1,at2);
				float const ytarget = -0.5 * SWITCHGAP * de;
				if( y < 0 ) {
					float mn = solv1(1,at1,at2);
					float const mult = ( ytarget-mn ) / ( y-mn );
					for(int ii=1; ii<= SWITCHBIN; ii++) {
						float const e = solv1(ii,at1,at2);
						solv1(ii,at1,at2) = ( e - mn ) * mult + mn;
					}
				} else if(y > 0) {
					float mx = solv1(1,at1,at2);
					float const mult = ( mx-ytarget ) / ( mx-y );
					for(int ii=1; ii<= SWITCHBIN; ii++) {
						float const e = solv1(ii,at1,at2);
						solv1(ii,at1,at2) = ( e - mx ) * mult + mx;
					}
				} else { // y==0
					//continue; //SGM Bug fix: Intent was not to break out of loop
				}
				int cutoff = 2;
				while(solv1(cutoff,at1,at2) == solv1(1,at1,at2) )
					cutoff++;
				for(int ii=max(1,cutoff-SOLV_LOW_SWITCH_SMOOTH);
						 ii <= min(etable_disbins,cutoff+SOLV_LOW_SWITCH_SMOOTH); ii++)
					solv1(ii,at1,at2) = -9e9;
				for(int ii=SWITCHBIN+1; ii <= etable_disbins; ii++)
					solv1(ii,at1,at2) = -9e9;
				//cout << "solv1 " << at1 << ' ' << at2 << ' ' << cutoff-10 << ' '<< cutoff+10;
				spline_fill_marked_gaps( fa_dist, solv1(1,at1,at2), dsolv1 , -9e9);
			}
			{
				//now solv2 terms
				float const de = ( solv2(SWITCHBIN,at1,at2)-solv2(SWITCHBIN-1,at1,at2) )/
					( fa_dist(SWITCHBIN)-fa_dist(SWITCHBIN-1) );
				//float const x = fa_dist(SWITCHBIN);
				float const y = solv2(SWITCHBIN,at1,at2);
				float const ytarget = -0.5 * SWITCHGAP * de;
				if( y < 0 ) {
					float mn = solv2(1,at1,at2);
					float const mult = ( ytarget-mn ) / ( y-mn );
					for(int ii=1; ii<= SWITCHBIN; ii++) {
						float const e = solv2(ii,at1,at2);
						solv2(ii,at1,at2) = ( e - mn ) * mult + mn;
					}
				} else if(y > 0) {
					float mx = solv2(1,at1,at2);
					float const mult = ( mx-ytarget ) / ( mx-y );
					for(int ii=1; ii<= SWITCHBIN; ii++) {
						float const e = solv2(ii,at1,at2);
						solv2(ii,at1,at2) = ( e - mx ) * mult + mx;
					}
				} else { // y==0
					//continue; //SGM Bug fix: Intent was not to break out of loop
				}
				int cutoff = 2;
				while(solv2(cutoff,at1,at2) == solv2(1,at1,at2) )
					cutoff++;
				for(int ii=max(1,cutoff-SOLV_LOW_SWITCH_SMOOTH);
						 ii <= min(etable_disbins,cutoff+SOLV_LOW_SWITCH_SMOOTH); ii++)
					solv2(ii,at1,at2) = -9e9;
				for(int ii=SWITCHBIN+1; ii <= etable_disbins; ii++)
					solv2(ii,at1,at2) = -9e9;
				//cout << "solv2 " << at1 << ' ' << at2 << ' ' << cutoff-10 << ' '<< cutoff+10;
				spline_fill_marked_gaps( fa_dist, solv2(1,at1,at2), dsolv2 , -9e9);
			}

			for(int ii = 1; ii <= etable_disbins; ii++) {
				dsolv(ii,at1,at2) = dsolv1(ii) + dsolv2(ii);
			}

			for(int ii = 1; ii <= etable_disbins; ii++) {
				ljatr(ii,at1,at2) *= fullatom_setup_ns::smooth_etable_ljweight;
				dljatr(ii,at1,at2) *= fullatom_setup_ns::smooth_etable_ljweight;
				ljrep(ii,at1,at2) *= fullatom_setup_ns::smooth_etable_ljweight;
				dljrep(ii,at1,at2) *= fullatom_setup_ns::smooth_etable_ljweight;
				solv1(ii,at1,at2) *= fullatom_setup_ns::smooth_etable_solvweight;
				solv2(ii,at1,at2) *= fullatom_setup_ns::smooth_etable_solvweight;
				dsolv(ii,at1,at2) *= fullatom_setup_ns::smooth_etable_solvweight;
			}

		}
	}

}





// HELPER FUNCTIONS BELOW HERE....







////////////////////////////////////////////////////////////////////////////////
/// @begin fa_precalc_etable_coefficients
///
/// @brief precalculate non-distance dependent coefficients of energy functions
///
/// @detailed
///
/// @param[out]   fa_lj_sigma - out - for atomtypes i and j: (radius_i+radius_j)
/// @param[out]   fa_lj_r6_coeff - out - precalced coefficient on the (1/dis)**6 term in lj
/// @param[out]   fa_lj_r12_coeff - out - precalced coefficient on the (1/dis)**12 term in lj
/// @param[out]   fa_lj_switch_intercept - out -
///            for close contacts calculate lj from a line with this intercept
/// @param[out]   fa_lj_switch_slope - out -
///            for close contacts calculate lj from a line with this slope
/// @param[out]   fa_lk_inv_lambda2 - out -
///            surprise! it's the 1/(lambda)**2 term in the lk equation
/// @param[out]   fa_lk_coeff - out - precalculation of all non-distance dependent terms
///            outside of the exponential in the lk equation
/// @param[out]   fa_lk_min_dis2sigma_value - out - below the min dis2sigma ratio for lk,
///            this value is assigned to the solvation
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void fa_precalc_etable_coefficients(
	FArray2DB_double & fa_lj_sigma,
	FArray2DB_double & fa_lj_r6_coeff,
	FArray2DB_double & fa_lj_r12_coeff,
	FArray2DB_double & fa_lj_switch_intercept,
	FArray2DB_double & fa_lj_switch_slope,
	FArray1DB_double & fa_lk_inv_lambda2,
	FArray2DB_double & fa_lk_coeff,
	FArray2DB_double & fa_lk_min_dis2sigma_value
) {

	using namespace etable;
	using namespace param;
	using namespace water;
  using namespace hbonds;

	// locals
	double sigma,sigma6,sigma12,wdepth;
	double inv_lambda;
	FArray1D_double lk_coeff_tmp( MAX_ATOMTYPES()() );
	double thresh_dis,inv_thresh_dis2,x_thresh;
	int dstype;
	int const atype_sulfur = { 16 };
	double const fa_inv_neg2_tms_pi_sqrt_pi = { -0.089793561062583294 };
	// coefficient for lk solvation

	// include follows locals so that data statements can initialize included arrays
	for ( int i = 1, e = MAX_ATOMTYPES(); i <= e; ++i ) {
		inv_lambda = 1.0/fa_lk_lambda(i);
		fa_lk_inv_lambda2(i) = inv_lambda * inv_lambda;
		lk_coeff_tmp(i) = fa_inv_neg2_tms_pi_sqrt_pi * fa_lk_dgfree(i) * inv_lambda;
	}

	for ( int i = 1, e = MAX_ATOMTYPES(); i <= e; ++i ) {
		for ( int j = i; j <= e; ++j ) {

			sigma = fa_Wradius * ( fa_lj_radius(i) + fa_lj_radius(j) );
//jjh temporary fix to prevent division by zero below
			sigma = ( sigma < 1.0e-9 ? 1.0e-9 : sigma );

			// (bk) modify sigma for hbond donors and acceptors
			//   ctsa -- should these scale down by fa_Wradius as well?

      // pb specific sigma correction for pairs between charged oxygen acceptors (15)
      // pb and hydroxyl oxygen donors (13). sigma correction for all polar H and charged oxygen
      // pb acceptor. Combinations of these corrections allow better prediction of both
      // pb hydroxyl O donor/charged O acceptor and charged NH donor/charged O acceptor
      // pb distances.

      if ( fa_lj_use_hbond_radii ) {
        if ( ( fa_acceptor(i) && fa_donor(j) ) ||
         ( fa_donor(i) && fa_acceptor(j) ) ) {
          sigma = fa_lj_hbond_dis;
        } else if ( ( fa_acceptor(i) && fa_hbondH(j) ) ||
         ( fa_hbondH(i) && fa_acceptor(j) ) ) {
          sigma = fa_lj_hbond_hdis;
        }
      }

//lin   modify sigma for water and hbond donors/acceptors
			if ( fa_lj_use_water_radii ) {
				if ( ( ( fa_acceptor(i) || fa_donor(i) ) && ( j == type_h2o ) ) ||
				 ( ( fa_acceptor(j) || fa_donor(j) ) && ( i == type_h2o ) ) ) {
					sigma = fa_lj_water_dis;
				} else if ( ( fa_hbondH(i) && j == type_h2o ) ||
				 ( fa_hbondH(j) && i == type_h2o ) ) {
					sigma = fa_lj_water_hdis;
				}
			}

			sigma6  = std::pow( sigma, 6 );
			sigma12 = sigma6 * sigma6;
			wdepth = std::sqrt(fa_lj_wdepth(i)*fa_lj_wdepth(j));

			fa_lj_sigma(i,j) = sigma;
			fa_lj_sigma(j,i) = fa_lj_sigma(i,j);

			fa_lj_r6_coeff(i,j) = -2. * wdepth * sigma6;
			fa_lj_r6_coeff(j,i) = fa_lj_r6_coeff(i,j);

			fa_lj_r12_coeff(i,j) = wdepth * sigma12;
			fa_lj_r12_coeff(j,i) = fa_lj_r12_coeff(i,j);

			// ctsa - create coefficients for linear projection of lj repulsive used
			//  for low distance values
			if ( fa_lj_use_lj_deriv_slope ) {

				// ctsa - use the slope of the true lj repulsive at the
				//  linear switch point to create a linear projection of
				//  lj for low distances

				//  slope = wdepth/sigma *
				//          (slope@switch_point*sigma/wdepth)
				fa_lj_switch_slope(i,j) = (wdepth/sigma)*
				 fa_lj_switch_slope_sigma2wdepth;
				fa_lj_switch_slope(j,i) = fa_lj_switch_slope(i,j);

				// intercept = wdepth*(lj@switch_point/wdepth)
				//             - slope*switch_point_distance
				fa_lj_switch_intercept(i,j) = wdepth*fa_lj_switch_value2wdepth -
				 fa_lj_switch_slope(i,j)*sigma*fa_lj_switch_dis2sigma;
				fa_lj_switch_intercept(j,i) = fa_lj_switch_intercept(i,j);
			} else {

				// ctsa - create a linear projection of lj for low distances which
				//  is defined by a constant y intercept and the true lj repulsive
				//  value at the linear switch point
				fa_lj_switch_slope(i,j) = -(1./sigma)*fa_lj_switch_sigma2dis*
				 (fa_lj_slope_intercept-wdepth*fa_lj_switch_value2wdepth);
				fa_lj_switch_slope(j,i) = fa_lj_switch_slope(i,j);

				fa_lj_switch_intercept(i,j) = fa_lj_slope_intercept;
				fa_lj_switch_intercept(j,i) = fa_lj_switch_intercept(i,j);
			}

			// ctsa - precalculated lk solvation coefficients
			fa_lk_coeff(i,j) = lk_coeff_tmp(i) * fa_lk_volume(j);
			fa_lk_coeff(j,i) = lk_coeff_tmp(j) * fa_lk_volume(i);

			// ctsa - when dis/sigma drops below fa_lk_min_dis2sigma,
			//   a constant lk solvation value equal to the value at the
			//   switchover point is used. That switchover-point value
			//   is calculated here and stored in fa_lk_min_dis2sigma_value
			thresh_dis = fa_lk_min_dis2sigma*sigma;
			inv_thresh_dis2 = 1./( thresh_dis * thresh_dis );
			double dis_rad = thresh_dis - fa_lj_radius(i);
			x_thresh = ( dis_rad * dis_rad ) * fa_lk_inv_lambda2(i);
			fa_lk_min_dis2sigma_value(i,j) = std::exp(-x_thresh) * fa_lk_coeff(i,j) *
			 inv_thresh_dis2;

			dis_rad = thresh_dis - fa_lj_radius(j);
			x_thresh = ( dis_rad * dis_rad ) * fa_lk_inv_lambda2(j);
			fa_lk_min_dis2sigma_value(j,i) = std::exp(-x_thresh) * fa_lk_coeff(j,i) *
			 inv_thresh_dis2;

		}
	}

	// ctsa - calculate disulfide coefficients
	dstype = MAX_ATOMTYPES()() + 1;
	sigma = fa_Wradius * fa_disulfide_radius * 2;
	sigma6 = std::pow( sigma, 6 );
	sigma12 = sigma6 * sigma6;
	wdepth = fa_lj_wdepth(atype_sulfur);

	fa_lj_sigma(dstype,dstype) = sigma;

	fa_lj_r6_coeff(dstype,dstype) = -2. * wdepth * sigma6;
	fa_lj_r12_coeff(dstype,dstype) = wdepth * sigma12;

	if ( fa_lj_use_lj_deriv_slope ) {
		fa_lj_switch_slope(dstype,dstype) = (wdepth/sigma) *
		 fa_lj_switch_slope_sigma2wdepth;

		fa_lj_switch_intercept(dstype,dstype) = wdepth * fa_lj_switch_value2wdepth -
		 fa_lj_switch_slope(dstype,dstype) * fa_lj_sigma(dstype,dstype) *
		 fa_lj_switch_dis2sigma;
	} else {
		fa_lj_switch_slope(dstype,dstype) = -(1./sigma) * fa_lj_switch_sigma2dis *
		 (fa_lj_slope_intercept-wdepth*fa_lj_switch_value2wdepth);

		fa_lj_switch_intercept(dstype,dstype) = fa_lj_slope_intercept;
	}

	thresh_dis = fa_lk_min_dis2sigma * sigma;
	inv_thresh_dis2 = 1./( thresh_dis * thresh_dis );
	double const dis_rad = thresh_dis - fa_lj_radius(atype_sulfur);
	x_thresh = ( dis_rad * dis_rad ) * fa_lk_inv_lambda2(atype_sulfur);
	fa_lk_min_dis2sigma_value(dstype,dstype) = std::exp(-x_thresh) *
	 fa_lk_coeff(atype_sulfur,atype_sulfur) * inv_thresh_dis2;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin fa_calc_etable_value
///
/// @brief calc all etable values given a distance and atom-type pair
///
/// @detailed
///
/// given a pair of atom types and the squared inter-atomic separation
/// distance (and a whole bunch of pre-computed coeffecients), this returns
/// the value of the lennard-jones and lk solvation potentials and
/// their derivatives w.r.t. the separation distance
///
///
/// @param[in]   dis2 - in - atomic separation distance squared
/// @param[in]   atype1 - in - chemical type of atom 1
/// @param[in]   atype2 - in - chemical type of atom 2
/// @param[out]   atrE - out - atractive lj energy
/// @param[out]   d_atrE - out - d(atrE)/d(dis)
/// @param[out]   repE - out - repulsive lj energy
/// @param[out]   d_repE - out - d(repE)/d(dis)
/// @param[out]   solvE1 - out - lk solvation energy
/// @param[out]   solvE2 - out - lk solvation energy
/// @param[out]   dsolvE - out - d(solvE1+solvE2)/d(dis)
/// @param[in]   fa_lj_sigma - in - for atomtypes i and j: (radius_i+radius_j)
/// @param[in]   fa_lj_r6_coeff - in - precalced coefficient on the (1/dis)**6 term in lj
/// @param[in]   fa_lj_r12_coeff - in - precalced coefficient on the (1/dis)**12 term in lj
/// @param[in]   fa_lj_switch_intercept - in -
///            for close contacts calculate lj from a line with this intercept
/// @param[in]   fa_lj_switch_slope - in -
///            for close contacts calculate lj from a line with this slope
/// @param[in]   fa_lk_inv_lambda2 - in -
///            surprise! it's the 1/(lambda)**2 term in the lk equation
/// @param[in]   fa_lk_coeff - in - precalculation of all non-distance dependent terms
///            outside of the exponential in the lk equation
/// @param[in]   fa_lk_min_dis2sigma_value - in - below the min dis2sigma ratio for lk,
///            this value is assigned to the solvation
///
/// @global_read
///
/// pdbstatistics_pack.h
/// etable.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void fa_calc_etable_value(
	double & dis2,
	int & atype1,
	int & atype2,
	float & atrE,
	float & d_atrE,
	float & repE,
	float & d_repE,
	float & solvE1,
	float & solvE2,
	float & dsolvE,
	FArray2DB_double & fa_lj_sigma,
	FArray2DB_double & fa_lj_r6_coeff,
	FArray2DB_double & fa_lj_r12_coeff,
	FArray2DB_double & fa_lj_switch_intercept,
	FArray2DB_double & fa_lj_switch_slope,
	FArray1DB_double & fa_lk_inv_lambda2,
	FArray2DB_double & fa_lk_coeff,
	FArray2DB_double & fa_lk_min_dis2sigma_value
)
{
	using namespace etable;
	using namespace param;
	using namespace pdbstatistics_pack;

	// locals
	double ljE,d_ljE,x1,x2;
	double dis;
	double inv_dis,inv_dis2,inv_dis6,inv_dis7,inv_dis12,inv_dis13;
	double dsolvE1,dsolvE2;
	double dis2sigma;
	int xtra_atype1,xtra_atype2;
	int const atype_sulfur = { 16 };

	// include after local variables to allow data statements to initialize
	atrE = 0.;
	d_atrE = 0.;
	repE = 0.;
	d_repE = 0.;
	solvE1 = 0.;
	solvE2 = 0.;
	dsolvE = 0.;

	//  ctsa - epsilon allows final bin value to be calculated
	if ( dis2 > fa_max_dis2 + epsilon ) return;

	if ( dis2 < fa_min_dis2 ) dis2 = fa_min_dis2;

	dis = std::sqrt(dis2);
	inv_dis = 1.0/dis;
	inv_dis2 = inv_dis * inv_dis;



	//  ctsa - switch to disulfide bonded atom types
	//    when conditions are met
	if ( ( atype1 == atype_sulfur && atype2 == atype_sulfur ) &&
	 dis < fa_disulfide_dis_thresh ) {
		xtra_atype1 = MAX_ATOMTYPES()() + 1;
		xtra_atype2 = MAX_ATOMTYPES()() + 1;
	} else {
		xtra_atype1 = atype1;
		xtra_atype2 = atype2;
	}


	dis2sigma = dis / fa_lj_sigma(xtra_atype1,xtra_atype2);


	if ( dis2sigma < fa_lj_switch_dis2sigma ) {
		//  ctsa - use linear ramp instead of lj when the dis/sigma
		//    ratio drops below theshold
		d_ljE = fa_lj_switch_slope(xtra_atype1,xtra_atype2);
		ljE = dis*d_ljE + fa_lj_switch_intercept(xtra_atype1,xtra_atype2);
	} else {
		//  ctsa - calc regular lennard-jones
		inv_dis6  = inv_dis2 * inv_dis2 * inv_dis2;
		inv_dis7  = inv_dis6 * inv_dis;
		inv_dis12 = inv_dis6 * inv_dis6;
		inv_dis13 = inv_dis12 * inv_dis;

		ljE = fa_lj_r12_coeff(xtra_atype1,xtra_atype2) * inv_dis12 +
		 fa_lj_r6_coeff(xtra_atype1,xtra_atype2) * inv_dis6;

		d_ljE = -12.*fa_lj_r12_coeff(xtra_atype1,xtra_atype2) * inv_dis13-6. *
		 fa_lj_r6_coeff(xtra_atype1,xtra_atype2) * inv_dis7;
	}

	if ( ljE < 0. ) {
		atrE = ljE;
		d_atrE = d_ljE;

		// cbk  don't give hydrogens or water attractive lennard-jones
		// cbk  this is so very short range cut-offs can be used
		if ( ( xtra_atype1 >= 22 && xtra_atype1 <= 26 ) ||
		 ( xtra_atype2 >= 22 && xtra_atype2 <= 26 ) ) {
			atrE = 0.0;
			d_atrE = 0.0;
		}

	} else {
		repE = ljE;
		d_repE = d_ljE;
	}


	// ctsa - calc lk
	if ( dis2sigma < fa_lk_min_dis2sigma ) {
		// ctsa - solvation is constant when the dis/sigma ratio
		//   falls below minimum threshold
		solvE1 = fa_lk_min_dis2sigma_value(xtra_atype1,xtra_atype2);
		solvE2 = fa_lk_min_dis2sigma_value(xtra_atype2,xtra_atype1);
		dsolvE = 0.0;

	} else {

		double dis_rad = dis - fa_lj_radius(atype1);
		x1 = ( dis_rad * dis_rad ) * fa_lk_inv_lambda2(atype1);
		dis_rad = dis - fa_lj_radius(atype2);
		x2 = ( dis_rad * dis_rad ) * fa_lk_inv_lambda2(atype2);

		solvE1 = std::exp(-x1) * fa_lk_coeff(atype1,atype2) * inv_dis2;
		solvE2 = std::exp(-x2) * fa_lk_coeff(atype2,atype1) * inv_dis2;

		// ctsa - get d(lk_E)/dr
		dsolvE1 = -2.0 * solvE1 *
		 (((dis-fa_lj_radius(atype1))*fa_lk_inv_lambda2(atype1))+inv_dis);
		dsolvE2 = -2.0 * solvE2 *
		 (((dis-fa_lj_radius(atype2))*fa_lk_inv_lambda2(atype2))+inv_dis);

		dsolvE = dsolvE1 + dsolvE2;

	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin output_etable
///
/// @brief output an etable data file in the same format used in input_etable
///
/// @detailed
///$$$ file first line is <etable> <etable_disbins>
///$$$ other lines are <atom type 1> <atomtype 1> <eval bin 1> <eval bin 2>...
///
/// @global_read
/// pdbstatistics_pack.h:  ljatr, dljatr, ljrep, dljrep, solv1,solv2,dsolv
///
///
/// @remarks
///
/// @references
///
/// @authors sheffler mar 19 2006
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void output_etable(FArray3D_float & etable, std::string label, std::ostream & out) {
	using namespace param;
	using namespace pdbstatistics_pack;
	using namespace std;

	out << label << " " << etable_disbins << endl;
	for(int at1 = 1; at1 <= MAX_ATOMTYPES()(); at1++) {
		for(int at2 = 1; at2 <= MAX_ATOMTYPES()(); at2++) {
			out << I(4,at1) << " "
					<< I(4,at2) << " ";
			for(int bin = 1; bin <= etable_disbins; bin++) {
				float evalue = etable(bin,at1,at2);
				out << evalue << ' ';
			}
			out << endl;
		}
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin input_etable
///
/// @brief read in etable from a datafile
///
/// @detailed
///$$$ file first line is <etable> <etable_disbins>
///$$$ other lines are <atom type 1> <atomtype 1> <eval bin 1> <eval bin 2>...
///
/// @global_read
/// pdbstatistics_pack.h:  ljatr, dljatr, ljrep, dljrep, solv1,solv2,dsolv
///
/// @global_write
/// pdbstatistics_pack.h:  ljatr, dljatr, ljrep, dljrep, solv1,solv2,dsolv
///
/// @remarks
///
/// @references
///
/// @authors sheffler mar 19 2006
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void input_etable(FArray3D_float & etable, const std::string label, std::istream & in) {
	using namespace param;
	using namespace pdbstatistics_pack;
	using namespace std;
	cout << "input_etable: reading etable... " << label << endl;
	istringstream intmp;
	string lblin;
	int numdisbin;
	char buf[100000];

	// check header
	in.getline(buf,100000);
	intmp.str(buf);
	if( !(intmp >> lblin >> numdisbin) ) {
		cout << "input_etable: WARNING bad etable header " << buf << endl;
		return;
	}
	if( lblin != label || etable_disbins != numdisbin ) {
		cerr << "input_etable: WARNING etable types don't match! "<< endl;
		cerr << "              expected " << label << "," << etable_disbins
				 << " got " << lblin << ',' << numdisbin<< endl;
		exit(0);
	} else {
		cout << "input_etable expected etable " << label << " of size " << etable_disbins
				 << ", got " << lblin << ',' << numdisbin<< endl;

	}

	// read in etable
	int at1,at2,count=0,scount=0;
	float evalue;
	while( in.getline(buf,100000) ) {
		//cout << "ASDFF buf  " << buf << endl;
		count++;
		intmp.clear();
		intmp.str(buf);
		if( ! (intmp >> at1 >> at2 ) ) {
			cerr << "input_etable: error reading etable line: " << buf << endl;
		} else {
			for(int bin = 1; bin <=numdisbin; bin++) {
				if( !(intmp >> evalue) ) {
					cerr << "input_etable: not enough bins on etable line: " << buf << endl;
					exit(-1);
				}
				//cout << "ASDFF read " << lblin << ' ' << at1 << ' ' << at2 << ' ' << bin << ' ' << evalue << endl;
				//cout << "ASDFF " << lblin << at1 << at2 << evalue;
				etable(bin,at1,at2) = evalue;
				etable(bin,at2,at1) = evalue;
			}
			scount++;
		}
	}
	cout << "              read " << scount << " of " << count << " lines" << endl;
}


