// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 8705 $
//  $Date: 2006-06-06 16:15:21 -0700 (Tue, 06 Jun 2006) $
//  $Author: pbradley $

// Rosetta Headers
#include "fibril.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "are_they_neighbors.h"
#include "atom_tree_routines.h"
#include "barcode_stats.h"
#include "cst_set.h"
#include "decoystats.h"
#include "decoystats_ns.h"
#include "decoystats_classes.h"
#include "design.h"
#include "docking_scoring.h"
#include "files_paths.h"
#include "fragments.h"
#include "fragments_pose.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "fullatom_setup.h"
#include "gb_elec.h" // HACK!!
//#include "gb_elec_ns.h" // HACK!!
#include "hbonds.h"
#include "hbonds_ns.h"
#include "jumping_util.h"
#include "jumping_pairings.h" //IA debug
#include "jumping_minimize.h" //IA debug
#include "ligand.h"
#include "minimize.h"
#include "misc.h"
#include "make_pdb.h"
#include "map_sequence.h"
#include "maxsub.h"
#include "monte_carlo.h"
#include "nblist.h"
#include "pack_fwd.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pdbstatistics_pack.h"
#include "pose.h"
#include "pose_abinitio.h"
#include "pose_benchmark.h"
#include "pose_design.h"
#include "pose_io.h"
#include "pose_rms.h"
#include "prof.h"
#include "read_aa_ss.h"
#include "read_aaproperties.h"
#include "refold.h"
#include "random_numbers.h"
#include "rotamer_trials.h"
#include "pose_rotamer_trials.h"
#include "rms.h"
//#include "rotamers_pack.h"
#include "score.h"
#include "silent_input.h"
#include "smallmove.h"
#include "ssblocks.h"
//#include "SpaceGroup.h"
#include "symmetric_design.h"
#include "torsion_bbmove_trials.h"
#include "util_basic.h"
#include "pose_vdw.h"

// Docking
#include "docking.h"
#include "docking_score.h"
#include "docking_ns.h"
#include "dock_structure.h"
#include "docking_minimize.h"
#include "docking_movement.h"

// ObjexxFCL Headers
#include <ObjexxFCL/DimensionExpressions.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>

// BOINC
#ifdef BOINC
#include "boinc_rosetta_util.h"
#include "counters.h"
#include "trajectory.h"
#endif

typedef numeric::xyzVector_float  Vec;
typedef numeric::xyzMatrix_float  Mat;
typedef numeric::xyzVector_double DVec;
typedef numeric::xyzMatrix_double DMat;


///////////////////////////////////////////////////////////////////////////////

namespace fibril {
	int hbparam(0);
}



///////////////////////////////////////////////////////////////////////////////
// silly old code

int
get_nres_monomer(
	pose_ns::Pose const & pose,
	int & N
)
{
	pose_ns::Symmetry_info const & s( pose.symmetry_info() );
	N = s.num_bb_clones() + 1;
	int const nres_protein( pose.total_residue() - s.npseudo() );
	assert(  nres_protein%N == 0 );
	return ( nres_protein/N );
}


///////////////////////////////////////////////////////////////////////////////
void
fibril_set_allow_move(
											pose_ns::Pose & pose,
											bool const bb_move,
											bool const chi_move,
											bool const jump_move
											)
{
	pose.set_allow_bb_move( false );
	pose.set_allow_chi_move( false );
	pose.set_allow_jump_move( false );

	pose_ns::Symmetry_info const & s( pose.symmetry_info() );
	for ( int i=1; i<= pose.total_residue_for_scoring(); ++i ){
		if ( s. bb_independent(i) ) pose.set_allow_bb_move ( i, bb_move );
		if ( s.chi_independent(i) ) pose.set_allow_chi_move( i, chi_move );
	}

	for ( int i=1; i<= pose.num_jump(); ++i ){
		if ( s.jump_independent(i) ) pose.set_allow_jump_move( i, jump_move );
	}
}


///////////////////////////////////////////////////////////////////////////////
void
get_independent_jumps(
											pose_ns::Pose const & pose,
											int & jump1,
											int & jump2
											)
{
	jump1 = 0;
	jump2 = 0;
	for ( int i=1; i<= pose.num_jump(); ++i ) {
		if ( pose.symmetry_info().jump_independent(i) ) {
			if ( !jump1 ) jump1 = i;
			else if ( !jump2 ) jump2 = i;
			else assert( false );
		}
	}
	assert( !pose.pseudo( pose.fold_tree().get_jump_point()(1,jump1) ) &&
					 pose.pseudo( pose.fold_tree().get_jump_point()(2,jump1) ) &&
					 pose.pseudo( pose.fold_tree().get_jump_point()(1,jump2) ) &&
					 pose.pseudo( pose.fold_tree().get_jump_point()(2,jump2) ) );
}

///////////////////////////////////////////////////////////////////////////////
void
get_aligned_anchors(
										pose_ns::Pose const & pose,
										int & anchor_pos1,
										int & anchor_pos2
										)
{
	using namespace pose_ns;
	anchor_pos1 = 0;
	anchor_pos2 = 0;

	Symmetry_info const & s( pose.symmetry_info() );
	Fold_tree const & f( pose.fold_tree() );

	int pseudo_pos1(0), pseudo_pos2(0), pseudo_pos3(0);

	while ( !anchor_pos2 ) { // this is a freakin nuisance!
		for ( Fold_tree::const_iterator edge = f.begin(); edge != f.end(); ++edge){
			if ( !edge->is_jump() ) continue;

			if ( pose.pseudo( edge->start ) && !pose.pseudo( edge->stop) &&
					 s.bb_independent( edge->stop ) ) {
				pseudo_pos1 = edge->start;
				anchor_pos1 = edge->stop;
			}
			if ( edge->start == pseudo_pos1 && pose.pseudo( edge->stop ) ) {
				pseudo_pos2 = edge->stop;
			}
			if ( edge->start == pseudo_pos2 && pose.pseudo( edge->stop ) ) {
				pseudo_pos3 = edge->stop;
			}
			if ( edge->start == pseudo_pos3 && !pose.pseudo( edge->stop ) ) {
				anchor_pos2 = edge->stop;
			}
		}
	}
	assert( anchor_pos1 && anchor_pos2 );
}


///////////////////////////////////////////////////////////////////////////////
void
get_anchor_pseudo_pos(
	pose_ns::Pose const & pose,
	int & anchor_pos,
	int & pseudo_pos
)
{
	int const num_jump( pose.num_jump() );
	FArray2D_int const & jump_point( pose.fold_tree().get_jump_point() );

	anchor_pos = 0;
	pseudo_pos = 0;
	for ( int i=1; i<= num_jump; ++i ) {
		if ( pose.symmetry_info().bb_independent( jump_point(1,i) ) &&
				 !pose.pseudo( jump_point(1,i) ) &&
				 pose.pseudo( jump_point(2,i) ) ) {
			anchor_pos = jump_point(1,i);
			pseudo_pos = jump_point(2,i);
			break;
		}
	}

	if ( pseudo_pos == 0 /*|| pseudo_pos != nres_protein + N/2 + 1 */ ) {
		std::cout << "Failed to get anchor/pseudo pos! " <<
			anchor_pos << ' ' << pseudo_pos << ' ' << pose.fold_tree() <<
			std::endl;
		std::exit( EXIT_FAILURE );
	}
}

///////////////////////////////////////////////////////////////////////////////
void
setup_beta_jump_pose(
	pose_ns::Jump const & beta_jump,
	pose_ns::Pose & pose
)
{
	// first the pseudo->monomer jump
	// make silly two rsd pose
	pose.reset_bonds();
	pose.set_fullatom_flag( false );
	pose.one_jump_tree( 2, 1, 2, 1 );
	for ( int i=1; i<=2; ++i ) {
		pose.set_res(i,param_aa::aa_gln);
		pose.set_res_variant(i,1);
	}
	insert_init_frag( pose,1,2 );
	pose.set_jump( 1, beta_jump );

}


///////////////////////////////////////////////////////////////////////////////
//
// this will remove any rotational component
//
void
get_fibril_jumps(
	pose_ns::Jump beta_jump, // make copy
	float const xpos,
	float const ypos,
	pose_ns::Jump & monomer_jump,
	pose_ns::Jump & pseudo_jump,
	bool const inward_pointing // c-beta of anchor pos
	)
{
	using namespace pose_ns;

	// modify our copy to ensure no twist
	beta_jump.fold_in_rb_deltas();
	beta_jump.set_rotation( DMat::identity() );

	double const step( 0.5 * beta_jump.get_translation().length() );



	Pose tmp_pose;
	setup_beta_jump_pose( beta_jump, tmp_pose );

	// now want to reorient gln backbone so that:
	// 1 -- translation vector is parallel to z-axis
	// 2 -- ca-cb vector is in (-x)--z plane
	// 3 -- ca z-coord = 0.0
	Vec
		ca1( &tmp_pose.Eposition()(1,2,1) ),
		ca2( &tmp_pose.Eposition()(1,2,2) ),
		cb ( &tmp_pose.Eposition()(1,3,1) );

	Vec z( (ca2 - ca1).normalized() ),x( (ca1 - cb).normalized() ),y;
	if ( !inward_pointing ) x *= -1.0f;
	x = ( x - z * dot(x,z) ).normalized();
	y = cross(z,x);

	Mat M( Mat::rows(x,y,z) );

	// create a new Epos
	Vec offset( xpos, ypos, 0.0 /* zpos */ );
	FArray2D_float Epos(3,param::MAX_POS);
	for ( int j=1; j<= 5; ++j ) {
		Vec v( &(tmp_pose.Eposition()(1,j,1)));
		v = M * ( v - ca1 ) + offset;
		for ( int k=1; k<=3; ++k ) Epos(k,j) = v(k);
	}

	// create temporary pseudo-Eposition array
	FArray2D_float pseudo_Epos( 3, param::MAX_POS, 0.0 );
	pseudo_Epos( 1, 1 ) = 1.0; // N
	pseudo_Epos( 2, 4 ) = 1.0; // C

	// setup jump:
	Jump tmp_jump1( Epos, pseudo_Epos );
	monomer_jump = tmp_jump1;


	// setup the pseudo-pseudo jump
	Jump pj;
	pj.set_rotation( Z_rot( 180.0 ) );
	pj.set_translation( DVec( 0.0, 0.0, step ) );

	pseudo_jump = pj;


}


///////////////////////////////////////////////////////////////////////////////
void
setup_fibril_fold_tree(
	pose_ns::Pose & pose,
	int const N,
	std::string const & monomer_sequence,
	int const anchor_pos
)
{
	using namespace pose_ns;

	// the order that the sequence segments come in the 3D structure
	// 0th segment is in the middle
	//
	assert( N%2 == 1 );
	int const central_monomer( (N+1) / 2 ); // 3 if N == 5
	std::vector< int > order; // EG: < 1 2 0 3 4 >
	for ( int i=1; i<= N; ++i ) {
		if ( i < central_monomer ) order.push_back(i);
		else if ( i == central_monomer ) order.push_back(0);
		else order.push_back( i-1 );
	}

	//////////////////////
	//////////////////////
	int const nres_monomer( monomer_sequence.size() );
	int const nres( nres_monomer * N + N );
	int const nres_protein( nres_monomer * N );

	// ensure ideal & centroid
	pose.set_fullatom_flag( false );
	pose.reset_bonds();

	////////////////////
	//// setup fold_tree
	Fold_tree f;
	{
		int const num_jump( 2*N-1 );
		FArray2D_int jumps(2,num_jump);
		FArray1D_int cuts(num_jump);

		// 1-N jumps from pseudo-residues to peptides
		for ( int i=1; i<= N; ++i ) {
			int const peptide_pos( anchor_pos + order[i-1] * nres_monomer );
			int const pseudo_pos( nres_protein + i );
			int const jump_number( order[i-1] + 1 );
			jumps(1,jump_number) = peptide_pos;
			jumps(2,jump_number) = pseudo_pos;
			cuts(i) = i*nres_monomer;
		}

		// N+1 -> N+(N-1) jumps between pseudo-residues
		// note that the first jump is by default the one that will get packed
		// b/c of setup of jump_clones in Symmetry_info
		//
		// if we move to setting the domain_map so that only interactions with
		// a single peptide (the first) are scored, we need jump N+1 to run from
		// the first peptide to one of its neighbors...
		for ( int i=1; i<N; ++i ) {
			cuts( N+i ) = nres_protein+i;
			// jump_number will be 1 for jump that goes from the first peptides
			// pseudo residue to the one above it
			int const jump_number( N+1+order[i-1] );
			int const pseudo_pos1( nres_protein+i );
			int const pseudo_pos2( nres_protein+i+1 );
			jumps(1,jump_number) = pseudo_pos1;
			jumps(2,jump_number) = pseudo_pos2;
// 			std::cout << "jumps: " << jump_number << ' ' << pseudo_pos1 << ' ' <<
// 				pseudo_pos2 << ' ' << std::endl;
		}
		f.tree_from_jumps_and_cuts( nres, num_jump, jumps, cuts );

		// root at first pseudo rsd
		f.reorder( nres_protein + 1 );
		pose.set_fold_tree( f );
	}



	// fill sequence, ss, torsion information
	insert_init_frag( pose, 1, nres );
	for ( int k=0; k<N; ++k ) {
		int const pseudo_pos( nres_protein + k+1 );
		pose.set_res( pseudo_pos, param_aa::aa_gly );
		pose.set_res_variant( pseudo_pos, 1 );

		for ( int i=1; i<= nres_monomer; ++i ) {
			int const aa( aa2int( monomer_sequence[i-1] ) );
			int const aav(1);
			int const seqpos( i + k*nres_monomer );
			pose.set_res( seqpos, aa );
			pose.set_res_variant( seqpos, aav );
		}
	}
}



///////////////////////////////////////////////////////////////////////////////
// returns an ideal pose
//
void
setup_fibril_start(
									 pose_ns::Pose & pose,
									 int const N,
									 std::string const & monomer_sequence,
									 int const anchor_pos,
									 bool const inward_pointing,
									 pose_ns::Jump beta_jump, // make copy
									 float const xpos,
									 float const ypos
									 )
{
	using namespace pose_ns;


	// setup tree and sequence
	setup_fibril_fold_tree( pose, N, monomer_sequence, anchor_pos );

	// declare symmetric
	int const nres_monomer( monomer_sequence.size() );
	pose.setup_symm_info( nres_monomer, 0, N, "simple" );

	// fill in the jumps
	Jump monomer_jump, pseudo_jump;
	get_fibril_jumps( beta_jump, xpos, ypos, monomer_jump, pseudo_jump,
										inward_pointing );
	pose.set_jump(1,monomer_jump);
	pose.set_jump(N+1,pseudo_jump);
	pose.set_extra_score( "xpos_init", xpos );
	pose.set_extra_score( "ypos_init", ypos );


	{ // debugging
		int pos1,pos2;
		get_aligned_anchors( pose, pos1, pos2 );
		assert( pos1 <= nres_monomer );
		Jump bjump
			( pose.Eposition()(1,1,pos1 ),
				pose.Eposition()(1,1,pos2 ) );

		Jump tmp_jump;
		tmp_jump = beta_jump;
		tmp_jump.set_rotation( DMat::identity() );

		std::cout << "bjump-dev: " << distance( bjump, beta_jump ) << ' ' <<
			distance( bjump, tmp_jump ) << std::endl;
	}
}



///////////////////////////////////////////////////////////////////////////////
void
setup_twist(
	pose_ns::Pose & pose,
	pose_ns::Jump const & beta_jump
)
{
	using namespace pose_ns;
	using namespace cst_set_ns;

	// try to add a little twist to improve the fit to the
	// beta-jump transform

	int anchor_pos1, anchor_pos2;
	get_aligned_anchors( pose, anchor_pos1, anchor_pos2 );

	//int const N( pose.symmetry_info().num_bb_clones() + 1 );

	// make a temporary pose
	Pose beta_pose;
	setup_beta_jump_pose( beta_jump, beta_pose );

	// setup a constraint set
	Cst_set cst_set;
	for ( int ii=1; ii<= 3; ++ii ) {
		Vec ii_xyz( &beta_pose.full_coord()(1,ii,1) );
		for ( int jj=1; jj<= 3; ++jj ) {
			Vec jj_xyz( &beta_pose.full_coord()(1,jj,2) );
			float const r( distance( ii_xyz, jj_xyz ) );
			cst_set.add_atompair_constraint( kin::Atom_id( ii, anchor_pos1 ),
																			 kin::Atom_id( jj, anchor_pos2 ),Cst(r));
		}
	}


	// save starting constraints
	Cst_set const * save_cst_set( 0 );
	if ( pose.constraints_exist() ) {
		save_cst_set = &pose.constraints();
	}

	// set the new atompair constraints
	pose.set_constraints( cst_set );


	Score_weight_map w;
	w.set_weight( ATOMPAIR_CST, 100.0 );

	pose.score( w );

	// degrees of freedom?
	pose.set_allow_bb_move( false );
	pose.set_allow_chi_move( false );
	pose.set_allow_jump_move( false );

	int jump1, jump2;
	get_independent_jumps( pose, jump1, jump2 );
	pose.set_allow_jump_move( jump1, true );
	pose.set_allow_jump_move( jump2, true );

	pose.set_allow_rb_move( 1, jump2, false );
	pose.set_allow_rb_move( 2, jump2, false );
	pose.set_allow_rb_move( 3, jump2, true ); // z-translation
	pose.set_allow_rb_move( 4, jump2, false );
	pose.set_allow_rb_move( 5, jump2, false );
	pose.set_allow_rb_move( 6, jump2, true ); // z-rotation

	// minimize
	pose_setup_minimizer( 1e-6 );
	pose.score(w);


	{ // debugging
		int pos1,pos2;
		get_aligned_anchors( pose, pos1, pos2 );
		Jump bjump
			( pose.Eposition()(1,1,pos1 ),
				pose.Eposition()(1,1,pos2 ) );

		Jump bjump2
			( beta_pose.Eposition()(1,1,1 ),
				beta_pose.Eposition()(1,1,2 ) );

		Jump tmp_jump;
		tmp_jump = beta_jump;
		tmp_jump.set_rotation( DMat::identity() );

		std::cout << "bjump-dev-before: " << distance( bjump, beta_jump ) << ' ' <<
			distance( bjump, tmp_jump ) << ' ' <<
			distance( beta_jump, bjump2 ) << std::endl;
	}

	//pose.dump_pdb("before_twist.pdb");
	pose.score( w );
	std::cout << pose.show_scores() << std::endl;
	pose.main_minimize( w, "dfpmin" );
	std::cout << pose.show_scores() << std::endl;

	pose.score( w );

	{ // debugging
		int pos1,pos2;
		get_aligned_anchors( pose, pos1, pos2 );
		Jump bjump
			( pose.Eposition()(1,1,pos1 ),
				pose.Eposition()(1,1,pos2 ) );

		Jump tmp_jump;
		tmp_jump = beta_jump;
		tmp_jump.set_rotation( DMat::identity() );

		std::cout << "bjump-dev-after: " << distance( bjump, beta_jump ) << ' ' <<
			distance( bjump, tmp_jump ) << std::endl;
	}

	//pose.dump_pdb("after_twist.pdb");

	// restore original constraints
	if ( save_cst_set ) pose.set_constraints( *save_cst_set );
	else pose.reset_constraints();

}


///////////////////////////////////////////////////////////////////////////////
//
// stolen from fast_relax_pose but with some fibril-specific mods

void
fibril_relax(
	pose_ns::Pose & pose,
	int const lj_ramp_cycles,
	int const cycles,
	bool const zipper_relax, // = false
	bool const space_group_relax, // = false
	bool const abeta_relax // = false
)
{
	using namespace pose_ns;

	//symm_check( pose );
	int const N( pose.symmetry_info().num_bb_clones() + 1 );

	// params
	float const start_rep_weight( 0.02 );
	float const end_rep_weight( 1.0 );
	float const temperature ( N * 0.2 ); // N~20 for p212121
// 	float const temperature ( 0.8 );
	int const lj_ramp_inner_cycles( 10 );
	float const energycut(0.001); // for rotamer_trials
// 	float const energycut(0.1); // for rotamer_trials
	bool const use_nblist( true );
// 	bool const use_nblist( param_pack::gen_born ? false : true );
// 	if ( !use_nblist ) {
// 		std::cout << "WARNING:: fibril_relax: not using nblist b/c " <<
// 			"gen_born is TRUE!!!!!" << std::endl;
// 	}
	int const nsmallmoves( 3 ), nshearmoves( 5 );
	//float const squash_x( 0.25 ), squash_y( 0.5 );
	//rotamers::packer_nloop = 20;


	// ensure that packer params have been setup
	//check_packer_params();
	//setup_symmetric_packing( pose.total_residue_for_scoring(),
	//												 pose.symmetry_info() );

	// setup
	score_set_try_rotamers(false);
	set_use_nblist( use_nblist );
	minimize_set_local_min(false,5); // no local min
	minimize_exclude_sstype(false,false);
	minimize_set_tolerance(0.001);
	minimize_set_vary_phipsi(true);
	minimize_set_vary_omega(false);
	minimize_set_vary_chi(false);
	if ( abeta_relax ) {
		set_smallmove_size(4.0,4.0,6.0); // helix,strand,other
	} else {
		set_smallmove_size(6.0,6.0,9.0); // helix,strand,other
	}
// 	set_smallmove_size(10.0,10.0,15.0); // helix,strand,other
// 	set_smallmove_size(2.0,2.0,3.0); // helix,strand,other


	// monte-carlo object
	Score_weight_map weight_map( score12 );
	Monte_carlo mc( pose, weight_map, temperature );

	// profiling:
	prof::reset( true );

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// first phase: ramping the repulsive NOTE: this is only the repulsive
	// in scoring not repacking
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////

	if ( lj_ramp_cycles ) {
		// no rb-relax here
		pose.set_allow_jump_move( false );

		// geometric ramping
		// rep_delta^(lj_ramp_cycles-1) = end_rep_weight / start_rep_weight
		float const rep_delta
			( std::exp( std::log( end_rep_weight / start_rep_weight ) /
									(lj_ramp_cycles-1) ) );
		float rep_weight = start_rep_weight / rep_delta;

		// now ramp the repulsive weight
		for ( int n=1; n<= lj_ramp_cycles; ++n ) {

			rep_weight *= rep_delta;
			weight_map.set_weight( FA_REP, rep_weight );
			mc.set_weight_map( weight_map );

			// lower minimizer tolerance on final pass
			if ( n == lj_ramp_cycles ) {
				minimize_set_tolerance(0.0001);
			}

			// recover low
			mc.recover_low( pose );

			///////////////////////
			// start with a repack:
			if ( n==1 ) {
				// first time through, may have to put sidechains on
				if ( !pose.fullatom() ) {
					pose.set_fullatom_flag( true, true /* repack rotamers */);
					mc.reset( pose );
				}
			} else {
				pose.full_repack( true );
				mc.boltzmann( pose, "ljramp_repack" );
			}

			///////////////
			// now minimize
			set_use_nblist( false );
			pose.main_minimize( weight_map, "dfpmin" );
			set_use_nblist( use_nblist );
			mc.boltzmann( pose, "ljramp_minimize" );

			//////////////
			// small moves
			for ( int m=1; m<= lj_ramp_inner_cycles; ++m ) {
				pose_small_min_trial( pose, mc, weight_map, "linmin", nsmallmoves,
															true, energycut );
			}
		}
		// recover low
		mc.recover_low( pose );

	} else {
		// no lj-ramp cycles
		if ( !pose.fullatom() ) {
			std::cout << "WARNING:: no ljramp cycles but the pose was not fullatom!"
								<< std::endl;
			pose.set_fullatom_flag( true, true );
			mc.reset( pose );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// round 2: small/shear moves, w dfpmin
	// lower tolerance, still no varying chi
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////

	// params
	minimize_set_tolerance(.0001);

	if ( zipper_relax ) {
		// let in-out and up-down vary:
		pose.set_allow_jump_move( false );
		pose.set_allow_jump_move( 1, true );

		pose.set_allow_rb_move( 1, 1, true );
		pose.set_allow_rb_move( 2, 1, true );
		pose.set_allow_rb_move( 3, 1, false ); // z-translation
		pose.set_allow_rb_move( 4, 1, false );
		pose.set_allow_rb_move( 5, 1, false );
		pose.set_allow_rb_move( 6, 1, true ); // z-rotation
	} else if ( abeta_relax ) {
		// let in-out and up-down vary:
		int jump1, jump2;
		get_independent_jumps( pose, jump1, jump2 );
		pose.set_allow_jump_move( false );
		pose.set_allow_jump_move( jump1, true );

		pose.set_allow_rb_move( 1, jump1, true );
		pose.set_allow_rb_move( 2, jump1, true );
		pose.set_allow_rb_move( 3, jump1, false ); // z-translation
		pose.set_allow_rb_move( 4, jump1, false );
		pose.set_allow_rb_move( 5, jump1, false );
		pose.set_allow_rb_move( 6, jump1, true ); // z-rotation
	} else if ( space_group_relax ) {
		pose.set_allow_jump_move( false );
		pose.set_allow_jump_move( 1, true );
	}

	// repack
	pose.full_repack( true );
	mc.boltzmann( pose, "repack" );

	// minimize
	minimize_set_tolerance(.0001);

	pose.main_minimize( weight_map, "dfpmin" );
	mc.boltzmann( pose, "minimize" );

	mc.recover_low( pose );


	// cycles:
	for ( int n=1; n<= cycles; ++n ) {
		pose_small_min_trial( pose, mc, weight_map, "dfpmin", nsmallmoves ,true,
													energycut);
		pose_shear_min_trial( pose, mc, weight_map, "dfpmin", nshearmoves, true,
													energycut);

// 		if ( zipper_relax ) {
// 			// try a squash move!
// 			squash_move( pose, squash_x, squash_y );
// 			score_set_try_rotamers( true );
// 			set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut,
// 												 pose.total_residue(),
// 												 mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
// 			pose.main_minimize( weight_map, "dfpmin" );
// 			score_set_try_rotamers( false );
// 			mc.boltzmann( pose, "squash_min" );
// 		}

	}

	mc.recover_low( pose );

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// round 3: small/shear moves, w dfpmin
	// now vary chi
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////

	// params:
	minimize_set_vary_chi(true); // chi min!
	minimize_set_tolerance(.0001);

	if ( zipper_relax ) {
		// now allow all 5 dof's
		pose.set_allow_jump_move( false );

		pose.set_allow_jump_move( 1, true );
		pose.set_allow_jump_move( N+1, true );

		pose.set_allow_rb_move( 1, 1, true );
		pose.set_allow_rb_move( 2, 1, true );
		pose.set_allow_rb_move( 3, 1, false ); // z-translation
		pose.set_allow_rb_move( 4, 1, true );
		pose.set_allow_rb_move( 5, 1, true );
		pose.set_allow_rb_move( 6, 1, true ); // z-rotation

		pose.set_allow_rb_move( 1, N+1, false );
		pose.set_allow_rb_move( 2, N+1, false );
		pose.set_allow_rb_move( 3, N+1, true ); // z-translation
		pose.set_allow_rb_move( 4, N+1, false );
		pose.set_allow_rb_move( 5, N+1, false );
		pose.set_allow_rb_move( 6, N+1, false ); // z-rotation
	} else if ( abeta_relax ) {
		// now allow all 6 dof's
		int jump1, jump2;
		get_independent_jumps( pose, jump1, jump2 );
		pose.set_allow_jump_move( false );

		pose.set_allow_jump_move( jump1, true );
		pose.set_allow_jump_move( jump2, true );

		pose.set_allow_rb_move( 1, jump1, true );
		pose.set_allow_rb_move( 2, jump1, true );
		pose.set_allow_rb_move( 3, jump1, false ); // z-translation
		pose.set_allow_rb_move( 4, jump1, true );
		pose.set_allow_rb_move( 5, jump1, true );
		pose.set_allow_rb_move( 6, jump1, true ); // z-rotation

		pose.set_allow_rb_move( 1, jump2, false );
		pose.set_allow_rb_move( 2, jump2, false );
		pose.set_allow_rb_move( 3, jump2, true ); // z-translation
		pose.set_allow_rb_move( 4, jump2, false );
		pose.set_allow_rb_move( 5, jump2, false );
		pose.set_allow_rb_move( 6, jump2, true ); // z-rotation --> twist

	}

	// repack
	pose.full_repack( true );
	mc.boltzmann( pose, "repack" );

	// minimize
	pose.main_minimize( weight_map, "dfpmin" );
	mc.boltzmann( pose, "minimize" );

	mc.recover_low( pose );

	// cycles
	for ( int n=1; n<= cycles; ++n ) {
		pose_small_min_trial( pose, mc, weight_map,"dfpmin", nsmallmoves, true,
													energycut);
		pose_shear_min_trial( pose, mc, weight_map,"dfpmin", nshearmoves, true,
													energycut);
// 		if ( zipper_relax ) {
// 			// try a squash move!
// 			squash_move( pose, squash_x, squash_y );
// 			score_set_try_rotamers( true );
// 			set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut,
// 																		pose.total_residue(),
// 												 mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
// 			pose.main_minimize( weight_map, "dfpmin" );
// 			score_set_try_rotamers( false );
// 			mc.boltzmann( pose, "squash_min" );
// 		}

	}

	mc.recover_low( pose );

	// repack
	pose.full_repack( true );
	mc.boltzmann( pose, "repack" );

	// minimize
	minimize_set_tolerance(.00005);
	pose.main_minimize( weight_map, "dfpmin" );
	mc.boltzmann( pose, "minimize" );


	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// status, final score evaluation
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////

	// status info
	prof::show();
	mc.show_counters();

	mc.recover_low( pose );
	score_set_evaluate_all_terms(true);
	pose.score(weight_map);
	score_set_evaluate_all_terms(false);
}








///////////////////////////////////////////////////////////////////////////////
void
simple_cluster(
	int const ndecoys, /* input */
	int const natoms,
	FArray3D_float const & coords,
	std::map< int, int > decoy2tag,
	int min_cluster_size, /* params */
	int min_top_cluster_size,
	int try_top_cluster_size,
	int max_top_cluster_size,
	float const min_threshold,
	float const max_threshold,
	float & threshold, /* output */
	int & top_cluster_size,
	std::vector< std::vector< int > > & cluster_members
)
{
	assert( int(coords.size1()) == 3 &&
					int(coords.size2()) >= natoms &&
					int(coords.size3()) >= ndecoys );

	if ( ndecoys < 2 ) {
		std::cout << "Cant cluster a single decoy!" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	max_top_cluster_size = std::min( max_top_cluster_size, ndecoys );
	try_top_cluster_size = std::min( try_top_cluster_size, max_top_cluster_size);
	min_top_cluster_size = std::min( min_top_cluster_size, try_top_cluster_size);
	min_cluster_size     = std::min(     min_cluster_size, min_top_cluster_size);

	// dont want cluster size<2
	max_top_cluster_size = std::max( max_top_cluster_size, 2 );
	try_top_cluster_size = std::max( try_top_cluster_size, 2 );
	min_top_cluster_size = std::max( min_top_cluster_size, 2 );


	int const max_nbrs( max_top_cluster_size - 1 );

	// setup nbr heaps
	int const max_heap( max_nbrs + 3 ); // need extra space
	FArray2D_int heap  ( max_heap, ndecoys );
	FArray2D_float coheap( max_heap, ndecoys );
	for ( int i=1; i<= ndecoys; ++i ) {
		heap_init( heap(1,i), coheap(1,i), max_nbrs );
	}

	// calculate rmsds
	for ( int i=1; i<= ndecoys; ++i ) {
		if ( i%100 == 0 ) std::cout << "rms: " << i << std::endl;
		std::map< int, int >::const_iterator it( decoy2tag.find(i));
		int const itag( it == decoy2tag.end() ? 0 : it->second );
		for ( int j=i+1; j<= ndecoys; ++j ) {
			std::map< int, int >::const_iterator jt( decoy2tag.find(j));
			int const jtag( jt == decoy2tag.end() ? 0 : jt->second );
			float rms;
			if ( itag != jtag ) {
				rms = 999.9;
			} else {
				rms = simple_rms( natoms, coords(1,1,i), coords(1,1,j) );
			}
			bool err;
			heap_insert(heap(1,i), coheap(1,i), j, -rms, err );
			heap_insert(heap(1,j), coheap(1,j), i, -rms, err );
		}
	}

	// extract rmsds
	FArray2D_float nbr_rmsd( max_nbrs, ndecoys );
	FArray2D_int   nbrs    ( max_nbrs, ndecoys );
	for ( int i=1; i<= ndecoys; ++i ) {
		float prev_rmsd(1000.0);
		for ( int j=max_nbrs; j>= 1; --j ) {
			float rmsd;
			bool err;
			heap_extract(heap(1,i), coheap(1,i), nbrs(j,i), rmsd, err );
			rmsd *= -1.0;
			assert( rmsd <= prev_rmsd );
			nbr_rmsd(j,i) = rmsd;
			prev_rmsd = rmsd;
		}
	}

	// get threshold
	int size( try_top_cluster_size );
	int center(0);
	float old_threshold(0.0);
	while ( true ) {
		assert( size>1 );
		threshold = 1000.0;
		for ( int i=1; i<= ndecoys; ++i ) {
			float const t( nbr_rmsd(size-1, i ) );
			if ( t < threshold ) {
				threshold = t;
				center = i;
			}
		}

		std::cout << "size: " << size << ' ' << threshold << std::endl;

		if ( threshold > max_threshold ) {
			if ( size <= min_top_cluster_size ) break;
			--size;
		} else if ( threshold < min_threshold ) {
			if ( size >= max_top_cluster_size ||
					 old_threshold > max_threshold ) break;
			++size;
		} else break;
		old_threshold = threshold;
	}

	top_cluster_size = size;

	// get clusters
	FArray1D_bool clustered( ndecoys, false );
	cluster_members.clear();

	while ( true ) {
		clustered( center ) = true;

		// get members
		std::vector< int > members;
		members.push_back( center );
		for ( int ii=1; ii< top_cluster_size; ++ii ) {
			if ( nbr_rmsd(ii,center) > threshold ) break;
			int const i( nbrs(ii,center) );
			if ( clustered(i) ) continue;
			clustered(i) = true;
			members.push_back(i);
		}

		assert( int(members.size()) == size );

		cluster_members.push_back( members );

		// now get center of next cluster
		size = 0;
		for ( int i=1; i<= ndecoys; ++i ) {
			if ( clustered(i) ) continue;
			int my_size(1);
			for ( int ii=1; ii<top_cluster_size; ++ii ) {
				if ( nbr_rmsd(ii,i) > threshold ) break;
				if ( !clustered( nbrs(ii,i) ) ) ++my_size;
			}
			if ( my_size > size ) {
				size = my_size;
				center = i;
			}
		}

		if ( size < min_cluster_size ) break;
	}
	// all done!
}

///////////////////////////////////////////////////////////////////////////////
// takes in full_coord arrays
//
float
superimpose(
	int const nres,
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray3Da_float x1, // unmodified
	FArray3Da_float x2, // unmodified
	FArray2D_bool const & atom_mask,
	FArray3Da_float x2_out // transformed to superimpose on x1
) // returns rmsd over alignment
{
	using namespace param;

	// dimension the args
	x1.dimension(3,MAX_ATOM()(),nres);
	x2.dimension(3,MAX_ATOM()(),nres);
	x2_out.dimension(3,MAX_ATOM()(),nres);


	// pack coords into local arrays:
	FArray2D_double xx1(3,MAX_ATOM()()*nres);
	FArray2D_double xx2(3,MAX_ATOM()()*nres);
	FArray1D_double wt(MAX_ATOM()()*nres);

	int natoms=0; // total number of atoms
	int nsup=0; // number of superimposed atoms

	for (int i=1; i <= nres; i++) {
		int const aa ( res        (i) );
		int const aav( res_variant(i) );
		for (int j=1; j<= aaproperties_pack::natoms(aa,aav); ++j ) {
			++natoms;
			for ( int k=1;k<=3;k++ ) {
				xx1(k, natoms) = x1(k,j,i);
				xx2(k, natoms) = x2(k,j,i);
			}

			if ( atom_mask(j,i) ) {
				wt(natoms) = 1.0;
				nsup++;
			} else {
				wt(natoms) = 0.0;
			}
		}
	}
	int const total_atoms( natoms );

	// calculate starting center of mass (COM):
	FArray1D_double COM(3);
	COMAS(xx1,wt,natoms,COM(1),COM(2),COM(3));

	// superimpose:: shifts xx1, shifts and transforms xx2;
	double rms;
	rmsfitca2(natoms,xx1,xx2,wt,nsup,rms);

	if ( false ) { // debug:
		double tmp1,tmp2,tmp3;
		COMAS(xx1,wt,natoms,tmp1,tmp2,tmp3); // store xcen,ycen,zcen vals for later
		std::cout << "zero??: " << std::abs(tmp1) + std::abs(tmp2) + std::abs(tmp3)
							<< std::endl;
	}

	// translate xx2 by COM and fill x2_out array
	natoms = 0;
	for (int i=1; i <= nres; i++) {
		int const aa ( res        (i) );
		int const aav( res_variant(i) );
		for (int j=1; j<= aaproperties_pack::natoms(aa,aav); ++j ) {
			natoms++;
			for ( int k=1;k<=3;k++ ) {
				x2_out(k,j,i) = xx2( k, natoms ) + COM(k);
			}
		}
	}
	assert ( natoms == total_atoms );

	return ( static_cast <float> (rms) );
}

///////////////////////////////////////////////////////////////////////////////
inline
Vec
get_tetrad_vector(
									pose_ns::Pose const & pose,
									int const seqpos
									)
{
	Vec
		n ( &pose.full_coord()(1,1,seqpos)),
		ca( &pose.full_coord()(1,2,seqpos)),
		c ( &pose.full_coord()(1,3,seqpos)),
		cb( &pose.full_coord()(1,5,seqpos));

	return ( n-ca + c-ca + cb -ca ).normalized();
}


///////////////////////////////////////////////////////////////////////////////
bool
abeta_filter_new(
	pose_ns::Pose & pose
)
{
	static bool const skip_abeta_filter
		( truefalseoption("skip_abeta_filter") );

	if ( skip_abeta_filter ) return true;

	using namespace pose_ns;

	int const nres_monomer( pose.symmetry_info().nres_monomer_bb() );
	int const trim( 40 - nres_monomer );
	int const phe_pos( 19 -trim );
	int const asp_pos( 23 -trim );
	int const lys_pos( 28 -trim );
	int const ile_pos( 32 -trim );
	int const leu_pos( 34 -trim );
	int const val_pos( 36 -trim );

	assert( pose.res( asp_pos ) == param_aa::aa_asp &&
					pose.res( phe_pos ) == param_aa::aa_phe &&
					pose.res( ile_pos ) == param_aa::aa_ile &&
					pose.res( lys_pos ) == param_aa::aa_lys );


	// look at c-alpha tetrad orientations
	int const pseudo_pos
		( pose.total_residue() - pose.total_pseudo_residue() + 1 );
	assert( pose.pseudo( pseudo_pos ) );
	pose.center( pseudo_pos );

	Vec
		asp( &(pose.centroid()(1,asp_pos))),
		lys( &(pose.centroid()(1,lys_pos))),
		phe( &(pose.centroid()(1,phe_pos))),
		ile( &(pose.centroid()(1,ile_pos))),
		leu( &(pose.centroid()(1,leu_pos))),
		val( &(pose.centroid()(1,val_pos)));

	float const d2a
		( std::sqrt( square( phe(1) - ile(1) ) + square( phe(2) - ile(2) ) ) );

	float const d2b
		( std::sqrt( square( phe(1) - leu(1) ) + square( phe(2) - leu(2) ) ) );

	float const d2c
		( std::sqrt( square( phe(1) - val(1) ) + square( phe(2) - val(2) ) ) );

	float const d2( std::min( std::min( d2a, d2b ), d2c ) );

	// figure out the target orientation
	float const dot1( get_tetrad_vector( pose, phe_pos )(3) );
	float const dot2( get_tetrad_vector( pose, asp_pos )(3) );
	float const dot3( get_tetrad_vector( pose, lys_pos )(3) );
	float const dot4( get_tetrad_vector( pose, ile_pos )(3) );

	// now use only the four dots
	std::cout << "abeta_filter_new: phe19_dist= " << F(9,3,d2) <<
		" dots= " << F(9,3,dot1) << F(9,3,dot2) << F(9,3,dot3) << F(9,3,dot4) <<
		std::endl;

	return ( d2 < 9.0 &&
					 ( ( dot1 < 0 && dot2 < 0 && dot3 < 0 && dot4 < 0 ) ||
						 ( dot1 > 0 && dot2 > 0 && dot3 > 0 && dot4 > 0 ) ) );
}


///////////////////////////////////////////////////////////////////////////////
bool
abeta_filter(
	pose_ns::Pose /*const*/ & pose,
	std::string & turn_conf
)
{
	using namespace pose_ns;

	static bool init( false ), find_unsat, rescore;
	static int unsat_threshold;
	if ( !init ) {
		init = true;
		find_unsat = truefalseoption("find_unsat");
		rescore = truefalseoption("rescore");
		if ( find_unsat ) unsat_threshold = intafteroption("find_unsat");
	}

	bool passed( true );

	int const asp_pos( 14 );
	int const lys_pos( 19 );

	int N;
	int const nres_monomer( get_nres_monomer( pose, N ) );



	/////////////////////////////////////////////////////////////////////////////
	// check D->K salt bridge
	{

		// min cbeta-cbeta distance
		float min_d(999.9), dot1(0.0), dot2(0.0);
		int const pos1( asp_pos ); // in 1st monomer
		for ( int i=0; i< N; ++i ) {
			int const pos2( lys_pos + i * nres_monomer );
			Vec
				ca1( &(pose.Eposition()(1,2,pos1) )),
				cb1( &(pose.Eposition()(1,3,pos1) )),
				ca2( &(pose.Eposition()(1,2,pos2) )),
				cb2( &(pose.Eposition()(1,3,pos2) ));

			float const d( distance( cb1, cb2 ) );
			if ( d<min_d ) {
				min_d = d;
				dot1 = dot( (cb1-ca1).normalized(), (cb2-ca1).normalized() );
				dot2 = dot( (cb2-ca2).normalized(), (cb1-ca2).normalized() );
			}
		}
		std::cout << "min_d= " << min_d << " min_dot1= " << dot1 <<
			" min_dot2= " << dot2 << std::endl;

		if ( min_d > 10 || dot1 < 0 || dot2 < 0 ) passed = false;
		float const DK_min_d( min_d ),
			DK_dot1( dot1 ), DK_dot2( dot2 );

		///////////////////////////////////////////////////////////////////////////
		if ( pose.fullatom() ) {
			using namespace param_aa;
			// check functional group distances
			int const asp_pos( 14 );
			int const lys_pos( asp_pos + 5 );
			assert( pose.res( asp_pos ) == param_aa::aa_asp );

			// look at c-alpha tetrad orientations
			int const pseudo_pos
				( pose.total_residue() - pose.total_pseudo_residue() + 1 );
			assert( pose.pseudo( pseudo_pos ) );
			pose.center( pseudo_pos );

			// figure out the target orientation
			int const phe_pos( asp_pos -4 );
			int const ile_pos( lys_pos +4 );
			float const dot1( get_tetrad_vector( pose, phe_pos )(3) );
			float const dot2( get_tetrad_vector( pose, asp_pos )(3) );
			float const dot3( get_tetrad_vector( pose, lys_pos )(3) );
			float const dot4( get_tetrad_vector( pose, ile_pos )(3) );
			std::cout << "dots: " <<
				dot1 << ' ' << dot2 << ' ' << dot3 << ' ' << dot4 << ' ' <<
				DK_min_d << ' ' << DK_dot1 << ' ' << DK_dot2 << std::endl;

			// now use only the four dots
			passed = ( ( dot1 < 0 && dot2 < 0 && dot3 < 0 && dot4 < 0 ) ||
								 ( dot1 > 0 && dot2 > 0 && dot3 > 0 && dot4 > 0 ) );

			//////// old:
			if ( false ) {
				assert( pose.res( asp_pos ) == aa_asp &&
								pose.res( lys_pos ) == aa_lys );
				int const atomno1( LookupByName( aa_asp, pose.res_variant( asp_pos ),
																				 " CG " ) );
				int const atomno2( LookupByName( aa_lys, pose.res_variant( lys_pos ),
																				 " NZ " ) );
				int const pos1( asp_pos ); // in 1st monomer
				float min_fun_d(999);
				for ( int i=0; i< N; ++i ) {
					int const pos2( lys_pos + i * nres_monomer );
					Vec
						cg1( &(pose.full_coord()(1,atomno1,pos1))),
						nz2( &(pose.full_coord()(1,atomno2,pos2)));

					float const d( distance( cg1, nz2 ) );
					if ( d<min_fun_d ) {
						min_fun_d = d;
					}
				}
				std::cout << "fun_d: " << passed << ' ' << min_fun_d << std::endl;
				passed = passed && ( min_fun_d < 6.0 );
				passed = ( min_fun_d < 7.5 );
			}

		}
	}

	/////////////////////////////////////////////////////////////////////////////
	if ( true /*passed*/ ) {
		// look at contacts: for F19 (10) between copies of 23,25,27

		// hard-code the current tree
		assert( N == 9 );

		int const Nrungs( 5 );
		FArray1D_int order( Nrungs);
		order(1) = 1;
		order(2) = 3;
		order(3) = 0;
		order(4) = 6;
		order(5) = 8;

		int const pos1( 10 );
		Vec xyz1( &pose.Eposition()(1,3,pos1));


		float min_total_d( 999.9 );
		int stag( -1 );
		for ( int i=1; i<= Nrungs-1; ++i ) {
			float total_d( 0.0 );
			for ( int nn=0; nn< 2; ++nn ) {
				int const n( order( i+nn ) );
				float min_d( 999 );
				for ( int jj=23; jj<= 27; jj+= 2 ) {
					int const pos2( jj + nres_monomer * n );
					Vec xyz2( &pose.Eposition()(1,3,pos2));
					float const d( distance( xyz2,xyz1) );
					if ( d<min_d ) {
						min_d = d;
					}
				}
				total_d += min_d;
			}
			total_d /= 2;
			if ( total_d < min_total_d ) {
				min_total_d = total_d;
				stag = i;
			}
		}

		std::cout << "closest: " << passed << ' ' << stag << ' ' <<
			order(stag) << "-" << order(stag+1) << ' ' <<
			F(9,3,min_total_d ) << std::endl;

		passed = passed && ( min_total_d < 10.0 );
	}


	/////////////////////////////////////////////////////////////////////////////
	{ // look at the turn conformations
		std::string full_ss,ss,seq;
		for ( int i=2; i<= nres_monomer-1; ++i ) {
			full_ss += torsion2big_bin( pose.phi(i), pose.psi(i), pose.omega(i) );
			if ( i>= asp_pos && i<= lys_pos ) {
				ss += torsion2big_bin( pose.phi(i), pose.psi(i), pose.omega(i) );
				seq += param_aa::aa_name1( pose.res(i) );
			}
		}
		std::cout << "turn_conf: " << passed << ' ' << ss << ' ' << seq <<
			std::endl;

		std::cout << "full_conf: " << passed << ' ' << full_ss << std::endl;

		turn_conf = ss;
	}


	///////////////////////////////////////////////////////////////////////////
	{ // look at turn torsion angles vs experimental data

		float const dev24
			( std::min(std::abs(subtract_degree_angles(pose.phi(asp_pos+1),-145))+
								 std::abs(subtract_degree_angles(pose.psi(asp_pos+1), 115)),
								 std::abs(subtract_degree_angles(pose.phi(asp_pos+1), 145))+
								 std::abs(subtract_degree_angles(pose.psi(asp_pos+1),-115))));

		float const dev25
			( std::min(std::abs(subtract_degree_angles(pose.phi(asp_pos+2), -70))+
								 std::abs(subtract_degree_angles(pose.psi(asp_pos+2), -40)),
								 std::abs(subtract_degree_angles(pose.phi(asp_pos+2),  70))+
								 std::abs(subtract_degree_angles(pose.psi(asp_pos+2),  40))));

		float const dev26
			( std::min(std::abs(subtract_degree_angles(pose.phi(asp_pos+3),  68))+
								 std::abs(subtract_degree_angles(pose.psi(asp_pos+3), -65)),
								 std::abs(subtract_degree_angles(pose.phi(asp_pos+3), -68))+
								 std::abs(subtract_degree_angles(pose.psi(asp_pos+3),  65))));

		float const dev29
			( std::min(std::abs(subtract_degree_angles(pose.phi(asp_pos+6),-120))+
								 std::abs(subtract_degree_angles(pose.psi(asp_pos+6),-125)),
								 std::abs(subtract_degree_angles(pose.phi(asp_pos+6), 120))+
								 std::abs(subtract_degree_angles(pose.psi(asp_pos+6), 125))));

		float const dev30
			( std::min(std::abs(subtract_degree_angles(pose.phi(asp_pos+7),-165))+
								 std::abs(subtract_degree_angles(pose.psi(asp_pos+7), 133)),
								 std::abs(subtract_degree_angles(pose.phi(asp_pos+7), 165))+
								 std::abs(subtract_degree_angles(pose.psi(asp_pos+7),-133))));
		std::cout << "turn_dev: " << passed << ' ' << turn_conf << ' ' <<
			F(9,3,dev24) << F(9,3,dev25) << F(9,3,dev26) <<
			F(9,3,dev29) << F(9,3,dev30) << std::endl;
	}

	return abeta_filter_new( pose );


	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// simple test for unsatisfied hbonds:
	// min hbondE per amide plane
	if ( passed && find_unsat ) {
		pose.score( score12 );

		copy_hbenergies();

		float hbE_threshold( -0.1 );

		int unsat_count( 0 );
		{ // HACK
			using namespace decoystats;
			using namespace aaproperties_pack;
			for ( int i=asp_pos-2; i<= lys_pos+1; ++i ) {
				// consider the amide plane between i and i+1
				// OC of i
				// HN of i+1

				bool ok( false );
				for ( int r=1; r<= 2; ++r ) {
					int const seqpos( r == 1 ? i : i + 1 );
					int const atomno
						( r == 1 ? 4 : HNpos(pose.res(seqpos),pose.res_variant(seqpos)));
					if ( atom_hbondE( atomno, seqpos ) < hbE_threshold ) ok = true;
				}
				if ( !ok ) ++unsat_count;
			}
		}
		if ( unsat_count > unsat_threshold ) passed = false;

		std::cout << "simple_unsat: " << unsat_count << std::endl;
	}


	return passed;
}

///////////////////////////////////////////////////////////////////////////////
void
fibril_cluster()
{
	using namespace pose_ns;
	using namespace silent_io;

	//int const fibril_N(5);

	//// parse args
	int min_cluster_size, min_top_cluster_size, try_top_cluster_size,
		max_top_cluster_size;
	float min_threshold, max_threshold;
	std::vector< int > rsd_list;

	{
		std::string sizes( stringafteroption("z"));
		std::string thresholds( stringafteroption("t"));

		{
			std::replace( sizes.begin(), sizes.end(), ',', ' ' );
			std::istringstream is( sizes );
			is >> min_cluster_size >> min_top_cluster_size >> try_top_cluster_size >>
				max_top_cluster_size;
			if ( is.fail() ) {
				std::cout << "parse error: " << sizes << std::endl;
				std::exit( EXIT_FAILURE );
			}
		}

		{
			std::replace( thresholds.begin(), thresholds.end(), ',', ' ' );
			std::istringstream is( thresholds );
			is >> min_threshold >> max_threshold;
			if ( is.fail() ) {
				std::cout << "parse error: " << thresholds << std::endl;
				std::exit( EXIT_FAILURE );
			}
		}

		if ( truefalseoption("r") ) {
			std::string residues( stringafteroption("r"));
			std::replace( residues.begin(), residues.end(), ',', ' ' );
			std::istringstream is( residues );
			while ( true ) {
				int rsd;
				is >> rsd;
				if ( is.fail() ) break;
				rsd_list.push_back(rsd);
			}
		}
	}

	std::string target_turn_conformation;
	stringafteroption( "turn", "", target_turn_conformation );

	std::string const prefix( stringafteroption("p") );

	// output the command
	{
		std::ofstream out( ( prefix+".info" ).c_str());
		out << "command executed: " << rosetta_command_string() << std::endl;
		out.close();
	}

	// read in a list of fullatom silent files, score filter?
	// read silent files
	Silent_file_data decoys;

	bool const fullatom( !truefalseoption("centroid") );

	if ( truefalseoption("s") ) {
		decoys.read_file( stringafteroption("s"), fullatom, true );
	} else { // read multiple silent files
		// all should have the same sequence
		std::ifstream data( stringafteroption("l").c_str() );
		std::string filename;
		while ( getline( data, filename ) ) {
			std::cout << "Read file: "<< filename << std::endl;
			decoys.read_file( filename, fullatom, true/*prepend-filename*/);
		}
		data.close();
	}

	if ( !decoys.size() ) {
		std::cout << "couldnt open silent-files!!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( truefalseoption("f") ) {
		decoys.score_filter( realafteroption("f") );
	}

	int ndecoys( decoys.size() );
	std::cout << "Found " << decoys.size() << " decoys." << std::endl;

	int nres(0), nres_protein(0);

	Symmetry_info s;
	if ( decoys.get_symm_info() ) {
		s = *decoys.get_symm_info();
		nres = s.N_chi() * s.nres_monomer_chi();
		nres_protein = nres - s.npseudo();
	} else {
		std::cout << "WARNING:: no symm_info in silent file, assuming fibril " <<
			"with N=5" << std::endl;
		std::exit( EXIT_FAILURE );
// 		nres = decoys.nres();
// 		N = fibril_N;
// 		nres_protein = nres - N;
// 		nres_monomer = nres_protein / N;
	}

	bool const cluster_monomer1( truefalseoption("monomer1") );

	bool const nobb( truefalseoption("nobb"));

	std::string const & sequence( decoys.sequence() );

	std::vector< std::pair< int, int > > atoms;
	FArray2D_bool cluster_mask( param::MAX_ATOM()(), nres, false );
	for ( int i=1; i<= nres_protein; ++i ) {
		if ( cluster_monomer1 && !s.chi_independent(i) ) continue;
		int const aa ( aa2int( sequence[i-1] ) );
		int const aav( 1 );
		//int const ii( (i-1)%nres_monomer + 1 );
		for ( int j=1; j<= aaproperties_pack::nheavyatoms(aa,aav); ++j ) {
			if ( ( j <= 4 && !nobb ) ||
					 ( fullatom &&
						 std::find( rsd_list.begin(), rsd_list.end(), i ) !=
						 rsd_list.end())) {
				atoms.push_back( std::make_pair(j,i) );
				cluster_mask(j,i) = true;
				std::cout << "cluster-atom: " << I(4,atoms.size() ) <<
					I(3,j) << I(4,i) << /*I(4,ii) <<*/ std::endl;
			}
		}
// 		for ( int j=1; j<= aaproperties_pack::natoms(aa,aav); ++j ) {
// 			if ( cluster_mask(j,i) ) {
// 				atoms.push_back( std::make_pair(j,i) );
// 			}
// 		}
	}
	int const natoms( atoms.size() );
	std::cout << "total atoms: " << natoms << std::endl;

	// storage for the coordinates we are going to cluster:
	FArray3D_float coords( 3, natoms, ndecoys );
	FArray1D< std::string > decoy_name( ndecoys );

	// run through all the decoys, reconstruct and fill in the clustering atoms
	Silent_out out;
	bool const make_outfile( truefalseoption("outfile") );
	if ( make_outfile ) {
		out.open( stringafteroption("outfile") );
	}



	int decoy(0);
	std::map< int, int > decoy2tag;
	std::map< int, std::string > tag2string;
	std::map< std::string, int > string2tag;

	bool const turn_cluster( truefalseoption( "turn_cluster" ) );

	for ( Silent_file_data::const_iterator it = decoys.begin(),
					it_end = decoys.end(); it != it_end; ++it) {
		// copy from silent-data
		Pose pose;
		it->second->fill_pose( pose );

		std::string turn;
		bool passed( abeta_filter( pose, turn ) );
		std::cout << it->first << std::endl;

		if ( target_turn_conformation.size() && target_turn_conformation != turn )
			passed = false;

		if ( !passed ) continue;

		++decoy;
		if ( decoy%100 == 0 ) {
			std::cout << "fill-coords: " << decoy << ' ' << it->first << std::endl;
		}
		decoy_name( decoy ) = it->first;

		// cluster by turn
		if ( turn_cluster ) {
			int tag(0);
			if ( !string2tag.count( turn ) ){
				tag = string2tag.size();
				string2tag[ turn ] = tag;
				tag2string[ tag ] = turn;
			}
			decoy2tag[ decoy ] = string2tag[ turn ];
		}


		std::cout << "decoy_name: " << decoy << ' ' << decoy_name( decoy ) <<
			std::endl;
		if ( make_outfile ) {
			out.write( string_of( decoy ), pose );
		}

		// fill coords
		FArray3D_float const & fcoord( pose.full_coord() );
		for ( int ii=1; ii<= natoms; ++ii ) {
			int const j( atoms[ii-1].first  );
			int const i( atoms[ii-1].second );
			for ( int k=1; k<= 3; ++k ) {
				coords(k,ii,decoy) = fcoord(k,j,i);
			}
		}
	}
	ndecoys = decoy;
	std::cout << "now have " << ndecoys << " decoys." << std::endl;

	// do the clustering
	std::vector< std::vector< int > > cluster_members;
	float threshold;
	int top_cluster_size;

	simple_cluster( ndecoys, natoms, coords, decoy2tag, min_cluster_size,
									min_top_cluster_size, try_top_cluster_size,
									max_top_cluster_size, min_threshold, max_threshold,
									threshold, top_cluster_size, cluster_members );

	make_pdb_ns::occupancy = 1.;
	make_pdb_ns::bfactor = 0.;
	for ( int i=1; i<= nres; ++i ) {
		for ( int j=1; j<= param::MAX_ATOM()(); ++j ) {
			if ( cluster_mask(j,i) ) make_pdb_ns::bfactor(j,i) = 1.0;
		}
	}

	// output the clusters
	int const nclusters( cluster_members.size() );
	FArray3D_float center_fcoord( 3, param::MAX_ATOM()(), nres ),
		aligned_fcoord( 3, param::MAX_ATOM()(), nres );
	for ( int c=0; c< nclusters; ++c ) {

		std::vector< int > const & members( cluster_members[c] );
		int const size( members.size() );

		// make an outfile
		std::string extra_tag;
		if ( turn_cluster ) {
			std::string const center_turn_conf
				( tag2string[ decoy2tag[ members[0] ] ] );
			extra_tag = center_turn_conf + ".";
		}
		std::string const filename
			( prefix+"."+lead_zero_string_of(c,2)+"."+extra_tag+
				lead_zero_string_of(size,3)+".pdb" );
		std::cout << filename << std::endl;
		utility::io::ozstream out( filename.c_str() );


		// write cluster members
		for ( int i=0; i<size; ++i ) {
			out << "REMARK " << i+1 << ' ' << decoy_name( members[i] ) << '\n';
		}


		// output center
		Pose pose;
		int const center( members[0] );
		decoys.get_structure( decoy_name(center) ).fill_pose( pose );
		center_fcoord = pose.full_coord();

		out << "MODEL" << I(9,1) << '\n';
		if ( fullatom ) {
			output_fullatom_pdb( center_fcoord, pose.res(), pose.res_variant(),
													 nres_protein, 'A', 1, nres, out );
		} else {
			output_CA_pdb( nres_protein, pose.res(), pose.res_variant(),
										 center_fcoord, out );
		}
		out << "ENDMDL\n";

		// output superimposed members
		for ( int i=1; i<size; ++i ) {
			int const decoy( members[i] );
			decoys.get_structure( decoy_name(decoy) ).fill_pose( pose );

			// fill
			superimpose( nres, pose.res(), pose.res_variant(), center_fcoord,
									 pose.full_coord(), cluster_mask, aligned_fcoord );

			out << "MODEL" << I(9,i+1) << '\n';
			if ( fullatom ) {
				output_fullatom_pdb( aligned_fcoord, pose.res(), pose.res_variant(),
														 nres_protein, 'A', 1, nres, out );
			} else {
				output_CA_pdb( nres_protein, pose.res(), pose.res_variant(),
											 aligned_fcoord, out );
			}
			out << "ENDMDL\n";

		}
		out.close();

	}

}


///////////////////////////////////////////////////////////////////////////////
void
abeta_AI(
	pose_ns::Pose & pose
)
{
	using namespace pose_ns;
	std::cout <<"abeta_AI!" << std::endl;

	float const arate( 0.5 );

	int const N( pose.symmetry_info().N_bb() );
	int const nres_monomer( pose.symmetry_info().nres_monomer_bb() );
	int const trim( intafteroption("nterm_trim", 9 ));
	//int const trim( 40 - nres_monomer );

 	// define region of fragment insertions
	int monomer_begin(0), monomer_end(0);
	for ( int i=1; i<= pose.total_residue_for_scoring(); ++i ) {
		if ( pose.symmetry_info().bb_independent(i) ) {
			monomer_begin = i;
			monomer_end = i + nres_monomer -1;
			break;
		}
	}
	assert( monomer_begin );
	int const frag_offset( 1-monomer_begin );

	// these are unused if turn_begin and turn_end are present on the
	// command line
	int const asp_pos( 23 - trim );
	int const lys_pos( 28 - trim );
	// extra fragment insertions in this region:
	int const turn_begin
		( monomer_begin-1 + intafteroption("turn_begin", asp_pos - 2 ) );
	int const turn_end
		( monomer_begin-1 + intafteroption("turn_end"  , lys_pos + 2 ) );

	fibril_set_allow_move( pose, true, true, false );

	pose.set_fullatom_flag( false );

	prof::reset(true);
	Score_weight_map w;
	w.set_weight( VDW, 1.0 );

	float const temperature( 1.0 * N );
	Monte_carlo mc( pose, w, temperature );
	mc.set_autotemp( true, temperature );

	/////////////////////////////////////////////////////////////////////////////
	// STAGE1
	//
	// in this first stage: want no clashes, DK dots, PHE19 contacts and
	//   low-res hbonds in nbhood of turn
	//
	// ramping the EXTRA score weight
	//
	fibril::hbparam = 1; // D-1 all the way to K+1

	prof::reset( true );
	int const outer_cycles1( intafteroption("outer_cycles1",5) );
	int const outer_cycles2( intafteroption("outer_cycles2",5) );
	int const outer_cycles3( intafteroption("outer_cycles3",10) );
	int const inner_cycles1( intafteroption("inner_cycles1",1000) );
	int const inner_cycles2( intafteroption("inner_cycles2",1000) );
	int const inner_cycles3( intafteroption("inner_cycles3",500) );

	for ( int ii=0; ii< outer_cycles1; ++ii ) {
		w.set_weight( EXTRA, 0.2 * ii );
		mc.set_weight_map( w );

		for ( int i=1; i<= inner_cycles1; ++i ) {
			if ( i%2 )
				choose_offset_frag( 3, pose, monomer_begin, monomer_end, frag_offset );
			else
				choose_offset_frag( 3, pose, turn_begin, turn_end, frag_offset );
			mc.boltzmann( pose, "frag1" );
		}
		mc.recover_low( pose );
	}
	prof::show();
	mc.show_counters();

	if ( abeta_filter_new( pose ) ) {

		///////////////////////////////////////////////////////////////////////////
		// STAGE2
		// now extend the hb-constraints, add env, pair
		//
		// ramping the EXTRA score weight
		//
		w.set_weight( ENV, 1.0 );
		w.set_weight( PAIR, 1.0 );

		prof::reset();
		for ( int ii=1; ii<= outer_cycles2; ++ii ) {
			fibril::hbparam = ii;

			w.set_weight( EXTRA, 1 + 0.2 * ii );
			mc.set_weight_map( w );

			for ( int i=1; i<= inner_cycles2; ++i ) {
				if ( i%2 )
					choose_offset_frag( 3, pose, monomer_begin, monomer_end,frag_offset);
				else
					choose_offset_frag( 3, pose, turn_begin, turn_end, frag_offset );
				mc.boltzmann( pose, "frag2" );
			}
			mc.recover_low( pose );
		}
		prof::show();
		mc.show_counters();

		if ( abeta_filter_new( pose ) &&
				 score_filter( pose.get_0D_score( SCORE ), "frag2", arate ) ) {

			/////////////////////////////////////////////////////////////////////////
			// STAGE3
			// now include explicit backbone hbonds
			//
			// also use hbcst on all residues
			//
			fibril::hbparam = 1000;
			w.set_weight( HB_LRBB, 1.0 );

			mc.set_weight_map( w );
			mc.set_autotemp( false, 0 );
			mc.set_temperature( 0.2 * N );

			prof::reset();
			for ( int i=1; i<= outer_cycles3; ++i ) {
				for ( int i=1; i<= inner_cycles3; ++i ) {
					pose_shear_trial( pose, mc, 6 );
					pose_small_trial( pose, mc, 3 );
				}
				mc.recover_low( pose );
			}
			prof::show();
			mc.show_counters();

		} // frag2-filter
	} // frag1-filter
}


///////////////////////////////////////////////////////////////////////////////
void
abeta40_test()
{
	using namespace pose_ns;

	int const N( 9 );

	float const min_x(  2.25 );
	float const max_x(  5.50 );
	float const min_y( -1.00 );
	float const max_y(  1.00 );

	//float const ai_cycles_factor( 1.0 );

	float const seq_weight( 0.1 );
	float const ss_weight ( 1.0 );
	float const bb_weight ( 0.0 );

	int const nstruct( 25000 );

	int const nterm_trim( 9 ); // only fold rsds 10-40

	// setup monomer
	std::string const full_sequence("DAEFRHDSGYEVHHQKLVFFAEDVGSNKGAIIGLMVGGVV");
	std::string const full_ss      ("LLLLLLLLLEEEEEEEEEEEEE..LLL..EEEEEEEEEEE");
	std::string const full_bb      ("........................................");
	//                               1234567890123456789012345678901234567890


	std::string const monomer_sequence( full_sequence.substr(nterm_trim));
	std::string const monomer_ss      ( full_ss      .substr(nterm_trim));
	std::string const monomer_bb      ( full_bb      .substr(nterm_trim));
	int const nres_monomer( monomer_sequence.size() );

	// read fragments
	param::MAX_RES_assign_res( nres_monomer * N + N );
	if ( false ){
	get_vall_frags( monomer_sequence, monomer_ss, monomer_bb,
									seq_weight, ss_weight, bb_weight,
									1, nres_monomer, 9, 200,
									true /*excl-gly*/, true /*excl-pro*/, true/*excl-cyspep*/,2);
	}

	get_vall_frags( monomer_sequence, monomer_ss, monomer_bb,
									seq_weight, ss_weight, bb_weight,
									1, nres_monomer, 3, 200,
									true /*excl-gly*/, true /*excl-pro*/, true/*excl-cyspep*/,1);


	// open silent outfile
	silent_io::Silent_out out( files_paths_pdb_out_prefix()+"abeta40_FA.out" );


	for ( int n=1; n<= nstruct; ++n ) {

		// check if model exists
		std::string const decoy_name( string_of( n ) );
		if ( !out.start_decoy( decoy_name ) ) continue;

		Pose pose;
		{ // setup the starting model

			// choose anchor-rsd

			// choose beta-sheet jump
			Jump beta_jump;
			while ( true ) {
				using numeric::conversions::degrees;
				int const orientation( 2 );
				int const pleating( 2 );

				Jump jump( get_random_beta_sheet_jump( orientation, pleating ) );

				// check for rotation component:
				DVec t( jump.get_translation() );
				DMat R( jump.get_rotation() );

				// RADIANS baby!
				double theta;
				DVec axis( ( rotation_axis( R, theta ) ).normalized() );
				if ( dot( axis,t) < 0.0 ) {
					axis *= -1.0;
					theta *= -1.0;
				}

				// alpha is the angle between the rotation axis and the translation t
				double const alpha
					( std::acos( dot( t.normalized(), axis ) ) );
				float const step( t.length() );
				double const cross_dev
					( cross( theta * axis, t.normalized() ).length() );

				DMat const new_R( rotation_matrix( t, theta ) );
				double R_dev;
				rotation_axis( R * new_R.transposed(), R_dev );

				std::cout << "jump_stats: " <<
					" theta= " << F(9,3,degrees(theta)) <<
					" step= " << F(9,3,step) <<
					" alpha= " << F(9,3,degrees(alpha)) <<
					" Rdev-angle= " << F(9,3,degrees(R_dev)) <<
					" cross-norm= " << F(9,3,cross_dev) << std::endl;


				// might want to make this depend on theta? or the dev param
				if ( std::abs( degrees( R_dev ) ) < 100000.0  ) { // !!!!!!!!!!!!!!!!
					jump.fold_in_rb_deltas();
					jump.set_rotation( new_R );
					beta_jump = jump;
					break;
				}
			} // while( true )

			float const xpos(  min_x + ran3() * ( max_x - min_x ) );
			float const ypos(  min_y + ran3() * ( max_y - min_y ) );
			int const anchor_pos( 34 - nterm_trim ); // LEU34
			setup_fibril_start( pose, N, monomer_sequence, anchor_pos, false,
													beta_jump, xpos, ypos );

			//pose.dump_pdb("test1.pdb");
			setup_twist( pose, beta_jump );

			{
				int const pseudo_pos( pose.total_residue_for_scoring() + 1 );
				pose.center( pseudo_pos );
				assert( pose.fold_tree().begin()->start == pseudo_pos );
			}
		} // scope


		{ // ABinitio
			prof::reset( true );
			pose.set_allow_bb_move( false );
			pose.set_allow_chi_move( false );
			pose.set_allow_jump_move( false );
			for ( int i=1; i<= nres_monomer; ++i ) {
				pose.set_allow_bb_move( i, true );
				pose.set_allow_chi_move( i, true );
			}

			abeta_AI( pose );
		}

		if ( abeta_filter_new( pose ) ) {
			// relax
			/////////////////////////////////////////////////////////////////////////
			// BEGIN ELECTROSTATICS HACKING /////////////////////////////////////////
			initialize_fullatom(); // ensure gen_born is false the first time called
 			param_pack::pack_wts.set_Wgb_elec(1.0); //fibril::Wgb;
 			param_pack::gen_born = true;
			gb_set_use_simple_electrostatics( true );
 			//gb_elec::hijack = true;
 			pose.new_score_pose();
			// END ELECTROSTATICS HACKING ///////////////////////////////////////////
			/////////////////////////////////////////////////////////////////////////

			pose_setup_packer();
			fibril_relax( pose, 6, 30, false, false, true );
			out.write( decoy_name, pose );
		}


	} // nstruct

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// TRY TO RUN THIS ONE ON BOINC
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//
void
abeta_double()
{
	using namespace pose_ns;

	int const N( intafteroption("N",12 ) );

	float const min_x(  2.75 ); // push out farther since aligned positions
	float const max_x(  6.50 ); // are outward pointing
	float const min_y( -1.00 );
	float const max_y(  1.00 );

	//float const ai_cycles_factor( 1.0 );

	float const seq_weight( 0.1 );
	float const ss_weight ( 1.0 );
	float const bb_weight ( 0.0 );

	int nstruct( intafteroption( "nstruct", 25000 ));

	// for abeta40 -- only fold rsds 10-40
	int const nterm_trim( intafteroption("nterm_trim", 9 ));
	int const anchor_pos( intafteroption("anchor_pos", 34 - nterm_trim ));//LEU34

	// setup monomer
	std::string const full_sequence
		( stringafteroption( "full_sequence",
												 "DAEFRHDSGYEVHHQKLVFFAEDVGSNKGAIIGLMVGGVV") );
	std::string const full_ss
		( stringafteroption( "full_ss",
												 "LLLLLLLLLEEEEEEEEEEEEE..LLL..EEEEEEEEEEE") );
	std::string const full_bb
		( stringafteroption( "full_bb",
												 "........................................") );
	//                      1234567890123456789012345678901234567890


	std::string const monomer_sequence( full_sequence.substr(nterm_trim));
	std::string const monomer_ss      ( full_ss      .substr(nterm_trim));
	std::string const monomer_bb      ( full_bb      .substr(nterm_trim));
	int const nres_monomer( monomer_sequence.size() );

	files_paths::mode_title = "Fibril mode";

	// read fragments
	param::MAX_RES_assign_res( nres_monomer * N + N );

	if ( truefalseoption("vall") ) {
		get_vall_frags( monomer_sequence, monomer_ss, monomer_bb,
										seq_weight, ss_weight, bb_weight,
										1, nres_monomer, 3, 200,
										true /*excl-gly*/, true /*excl-pro*/, true/*excl-cyspep*/,1);
	} else {
		read_frags_new( stringafteroption("frags"), 3 /*size*/, 1 /*size_bin*/ );
	}

	if ( truefalseoption("dump_frags") ) {
		dump_frags( stringafteroption("dump_frags"), 3, 1, nres_monomer-3+1 );
	}

	// open silent outfile
	silent_io::Silent_out out( files_paths_pdb_out_prefix()+"abeta_double.out" );


	for ( int n=1; n<= nstruct; ++n ) {

		// check if model exists
		std::string const decoy_name( string_of( n ) );
		if ( !out.start_decoy( decoy_name ) ) continue;

		Pose relax_pose;
		while ( true ) { // keep looping until we pass the filter

			Pose pose;

			int const nres( N * nres_monomer + N );
			//setup pdb info.
			Pdb_info pdb_info;
			pdb_info.dimension( nres );
			for ( int i=1; i<= N*nres_monomer; ++i ) {
				pdb_info.set_pdb_occ( i, 1.0 );
			}
			pose.set_pdb_information( pdb_info );


		//////////////////////////////
		{ // setup the starting model

			int const nres_protein( N * nres_monomer );
			int const num_jump( 2*N-1 );

			/////////////////////////
			{ // setup the fold_tree
				Fold_tree f;
				FArray1D_int cuts( num_jump );
				FArray2D_int jumps( 2, num_jump );

				for ( int i=1; i<= N; ++i ) {
					int const pos( anchor_pos + (i-1)*nres_monomer );
					int const pseudo_pos( nres_protein + i );

					jumps(1,i) = pos;
					jumps(2,i) = pseudo_pos;

					cuts(i) = i*nres_monomer;

					if ( i<N ) {
						jumps(1,N+i) = pseudo_pos;
						jumps(2,N+i) = pseudo_pos+1;

						cuts(N+i) = nres_protein+i;
					}
				}
				f.tree_from_jumps_and_cuts( nres, num_jump, jumps, cuts );
				f.reorder( nres_protein + 1 );
				pose.set_fold_tree( f );
			}




			//////////////////
			// setup symm_info
			Symmetry_info s;
			s.npseudo() = N;

			int const base_monomer_bb( 5 );
			int const base_jump1( base_monomer_bb );
			int const base_jump2( base_monomer_bb + N );
			for ( int i=1; i<= nres_monomer; ++i ) {
				for ( int j=1; j<= N; ++j ) {
					if ( j != base_monomer_bb ) {
						int const      pos( i + (               j - 1 )*nres_monomer );
						int const base_pos( i + ( base_monomer_bb - 1 )*nres_monomer );
						s.add_bb_clone( base_pos, pos );
					}

					int const base_monomer_chi( base_monomer_bb + 2*(( (j-1)/2 )%2) );
					if ( j != base_monomer_chi ) {
						int const      pos( i + (                j - 1 )*nres_monomer );
						int const base_pos( i + ( base_monomer_chi - 1 )*nres_monomer );
						s.add_chi_clone( base_pos, pos );
					}
				}
			}

			for ( int j=1; j<= N; ++j ) {
				int const jump1( j ), jump2( j+N );

				if ( jump1 != base_jump1 ) {
					s.add_jump_clone( base_jump1, jump1 );
				}

				if ( jump2 != base_jump2 && jump2 <= num_jump ) {
					s.add_jump_clone( base_jump2, jump2 );
				}
			}
			pose.setup_symm_info( s );

			////////////////////////
			// now fill in the data
			// choose beta-sheet jump
			Jump beta_jump;
			while ( true ) {
				using numeric::conversions::degrees;
				int const orientation( 2 );
				int const pleating( 2 );

				Jump jump( get_random_beta_sheet_jump( orientation, pleating ) );

				// check for rotation component:
				DVec t( jump.get_translation() );
				DMat R( jump.get_rotation() );

				// RADIANS baby!
				double theta;
				DVec axis( ( rotation_axis( R, theta ) ).normalized() );
				if ( dot( axis,t) < 0.0 ) {
					axis *= -1.0;
					theta *= -1.0;
				}

				// alpha is the angle between the rotation axis and the translation t
				double const alpha
					( std::acos( dot( t.normalized(), axis ) ) );
				float const step( t.length() );
				double const cross_dev
					( cross( theta * axis, t.normalized() ).length() );

				DMat const new_R( rotation_matrix( t, theta ) );
				double R_dev;
				rotation_axis( R * new_R.transposed(), R_dev );

				std::cout << "jump_stats: " <<
					" theta= " << F(9,3,degrees(theta)) <<
					" step= " << F(9,3,step) <<
					" alpha= " << F(9,3,degrees(alpha)) <<
					" Rdev-angle= " << F(9,3,degrees(R_dev)) <<
					" cross-norm= " << F(9,3,cross_dev) << std::endl;


				// might want to make this depend on theta? or the dev param
				if ( std::abs( degrees( R_dev ) ) < 100000.0  ) { // !!!!!!!!!!!!!!!!
					jump.fold_in_rb_deltas();
					jump.set_rotation( new_R );
					beta_jump = jump;
					break;
				}
			} // while( true )

			Jump monomer_jump, pseudo_jump;
			bool const inward_pointing( truefalseoption("inward_pointing"));
			float const xpos(  min_x + ran3() * ( max_x - min_x ) );
			float const ypos(  min_y + ran3() * ( max_y - min_y ) );
			get_fibril_jumps( beta_jump, xpos, ypos, monomer_jump, pseudo_jump,
												inward_pointing );
			pose.set_jump( base_jump1, monomer_jump );
			pose.set_jump( base_jump2, pseudo_jump );

			insert_init_frag( pose, 1, nres );
			for ( int k=0; k<N; ++k ) {
				int const pseudo_pos( nres_protein + k+1 );
				pose.set_res( pseudo_pos, param_aa::aa_gly );
				pose.set_res_variant( pseudo_pos, 1 );

				for ( int i=1; i<= nres_monomer; ++i ) {
					int const aa( aa2int( monomer_sequence[i-1] ) );
					int const aav(1);
					int const seqpos( i + k*nres_monomer );
					pose.set_res( seqpos, aa );
					pose.set_res_variant( seqpos, aav );
				}
			}

			setup_twist( pose, beta_jump );
		}

		fibril_set_allow_move( pose, true, true, false );

		abeta_AI( pose );

		if ( abeta_filter_new( pose ) ) {
			relax_pose = pose;
			break;
		}

		} // while ( true )

		// relax
		/////////////////////////////////////////////////////////////////////////
		// BEGIN ELECTROSTATICS HACKING /////////////////////////////////////////
		initialize_fullatom(); // ensure gen_born is false the first time called
		param_pack::pack_wts.set_Wgb_elec(1.0); //fibril::Wgb;
		param_pack::gen_born = true;
		gb_set_use_simple_electrostatics( true );
		//gb_elec::hijack = true;
		relax_pose.new_score_pose();
		// END ELECTROSTATICS HACKING ///////////////////////////////////////////
		/////////////////////////////////////////////////////////////////////////

		pose_setup_packer();
		fibril_relax( relax_pose, 6, 15, false, false, true );
		out.write( decoy_name, relax_pose );


// BOINC STUFF
#ifdef BOINC
		out.append_to_list( decoy_name ); // mark as done
		store_low_info(); // for trajectory plotting.
		clear_trajectory();
		counters::monte_carlo_ints::ntrials = 0;
		int farlx_stage = 0;
		bool const ready_for_boinc_end
			( boinc_checkpoint_in_main_loop(n, n, nstruct, farlx_stage) );
		if ( ready_for_boinc_end ) return; // Go back to main, which will then go to BOINC_END and shut off BOINC.
#endif


	} // n=1,nstruct
	exit(0);
}


///////////////////////////////////////////////////////////////////////////////
//

void
abeta40_rerelax()
{
	using namespace pose_ns;
	using namespace silent_io;

	Silent_file_data decoys( stringafteroption("s"), true );

	for ( Silent_file_data::const_iterator it = decoys.begin(),
					it_end = decoys.end(); it != it_end; ++it) {
		// copy from silent-data
		Pose pose;
		it->second->fill_pose( pose );

		if ( ! abeta_filter_new( pose ) ) continue;

		// minimize w/gb
		initialize_fullatom();

		param_pack::pack_wts.set_Wgb_elec(1.0); //fibril::Wgb;
		param_pack::gen_born = true;
		gb_set_use_simple_electrostatics( true );


		fibril_set_allow_move( pose, true, true, false );

		pose_deriv_check( pose );
		pose.dump_pdb(it->first+".pdb");
	}
}


///////////////////////////////////////////////////////////////////////////////
void
abeta_turn_test()
{
	using namespace pose_ns;

	int const N( 5 );

	float const seq_weight( 0.1 );
	float const ss_weight ( 1.0 );
	float const bb_weight ( 10.0 );

	int const nstruct( 100000 );

	set_smallmove_size(15,15,30);
// 	set_smallmove_size(5,5,10);

	// setup monomer
	std::string const full_sequence("DAEFRHDSGYEVHHQKLVFFAEDVGSNKGAIIGLMVGGVV");
	std::string const full_ss      ("LLLLLLLLLEEEEEEEEEEEEE..LLL..EEEEEEEEEEE");
	std::string const full_bb      ("......................BABBGB............");
//std::string const full_bb      ("......................BBEBGB............");
	//                               1234567890123456789012345678901234567890

	int const turn_start( 10 );
	int const turn_stop ( full_sequence.size() );
// 	int const turn_start( 19 );
// 	int const turn_stop ( 32 );
	int const nres_monomer( turn_stop - turn_start + 1 );

	std::string const monomer_sequence
		( full_sequence.substr( turn_start-1, nres_monomer ) );
	std::string const monomer_ss
		( full_ss.substr( turn_start-1, nres_monomer ) );
	std::string const monomer_bb
		( full_bb.substr( turn_start-1, nres_monomer ) );


	// read fragments
	param::MAX_RES_assign_res( nres_monomer * N + N );
	get_vall_frags( monomer_sequence, monomer_ss, monomer_bb,
									seq_weight, ss_weight, bb_weight,
									1, nres_monomer, 3, 200,
									true /*excl-gly*/, true /*excl-pro*/, true/*excl-cyspep*/,1);


	// open silent outfile
	silent_io::Silent_out out( files_paths_pdb_out_prefix()+"abeta40_FA.out" );


	// setup low-res backbone HBOND constraints
	//
	// from N -> O of monomer1 and monomer2
	//
	for ( int i=1; i< nres_monomer; ++i ) {

	}


	for ( int n=1; n<= nstruct; ++n ) {

		// check if model exists
		std::string const decoy_name( string_of( n ) );
		//if ( !out.start_decoy( decoy_name ) ) continue;

		Pose pose;
		{ // setup the starting model

			// choose anchor-rsd

			// choose beta-sheet jump
			Jump beta_jump;
			while ( true ) {
				using numeric::conversions::degrees;
				int const orientation( 2 );
				int const pleating( 2 );

				Jump jump( get_random_beta_sheet_jump( orientation, pleating ) );

				// check for rotation component:
				DVec t( jump.get_translation() );
				DMat R( jump.get_rotation() );

				// RADIANS baby!
				double theta;
				DVec axis( ( rotation_axis( R, theta ) ).normalized() );

				std::cout << "theta= " << theta << std::endl;

				// might want to make this depend on theta? or the dev param
				if ( std::abs( degrees( theta ) ) < 20 ) {
					jump.fold_in_rb_deltas();
					jump.set_rotation( DMat::identity() ); // no twist!!!!!!!!!
					beta_jump = jump;
					break;
				}
			} // while( true )


			// setup pose:
			int const nres( N * nres_monomer );
			int const anchor_pos( 19 - turn_start + 1 ); //PHE19
			int const num_jump( N-1 );
			int const middle_jump( (N+1)/2 );

			{ // setup fold_tree
				Fold_tree f;
				FArray2D_int jumps(2,num_jump);
				FArray1D_int cuts(num_jump);
				for ( int i=1; i<= num_jump; ++i ) {
					jumps(1,i) = anchor_pos + (i-1)*nres_monomer;
					jumps(2,i) = anchor_pos +     i*nres_monomer;
					cuts(i) = i*nres_monomer;
				}
				// fix middle jump
				jumps(1,middle_jump) = anchor_pos;
				jumps(2,middle_jump) = anchor_pos + (N-1)*nres_monomer;
				f.tree_from_jumps_and_cuts( nres, num_jump, jumps, cuts );
				f.reorder( anchor_pos );
				pose.set_fold_tree( f );
			}
			std::string seq;
			for ( int i=1; i<= N; ++i ) seq += monomer_sequence;
			pose.set_sequence( seq );
			pose.set_res_variant_all( 1 );
			insert_init_frag( pose, 1, nres );
			for ( int i=1; i<= num_jump; ++i ) {
				if ( i== middle_jump ) {
					Jump j;
					j = beta_jump;
					j.reverse();
					pose.set_jump( i, j );
				} else pose.set_jump( i, beta_jump );
			}
			pose.setup_symm_info( nres_monomer, 0, N, "no_pseudo" );
			insert_random_frags(3, pose, 1, nres_monomer, 0 );
			pose.set_allow_bb_move( false );
			pose.set_allow_chi_move( false );
			for ( int i=1; i<= nres_monomer; ++i ) {
				pose.set_allow_bb_move( i,true);
				pose.set_allow_chi_move( i,true);
			}
			pose.set_allow_jump_move( false );

			assert( symm_check( pose ) );

			{
				FArray3D_float Epos( pose.Eposition() );
				// transform so that:
				//
				// C-alpha of 1 is at the origin
				// fibril axis is along +z
				// N->C vector is along +x

				Vec x,y,z,
					ca1( &Epos(1,2,anchor_pos) ),
					c  ( &Epos(1,4,anchor_pos) ),
					n  ( &Epos(1,1,anchor_pos) ),
					ca2( &Epos(1,2,anchor_pos+nres_monomer ));


				z = ( ca2 - ca1 ).normalized();

				x = ( c - n );
				x = ( x - z * dot( x,z) ).normalized();
				y = cross( z,x );

				Mat M( Mat::rows(x,y,z) );
				Vec b( -1.0f * M * ca1 );

				pose.transform_Ax_plus_b( M, b );
				assert( std::abs( pose.Eposition()(1,2,anchor_pos) ) +
								std::abs( pose.Eposition()(2,2,anchor_pos) ) +
								std::abs( pose.Eposition()(3,2,anchor_pos) ) < 1e-2 );

			}
		}

		abeta_AI( pose );

		float const arate( 0.5 );

		if ( score_filter( pose.get_0D_score( SCORE ), "final", arate ) ) {
			pose_setup_packer();

			fibril_relax( pose, 5, 30 );

			pose.dump_pdb(files_paths_pdb_out_prefix()+
										"abeta_turn_FA."+string_of(n)+".pdb");
		}
	}
}





///////////////////////////////////////////////////////////////////////////////
float
eval_extra_score(
	pose_ns::Pose const & pose
)
{
	static float const phe_dist_weight
		( realafteroption("extra_score_phe_dist_weight",1.0 ));
	static float const hb_weight
		( realafteroption("extra_score_hb_weight",1.0 ));
	static float const asp_lys_weight
		( realafteroption("extra_score_asp_lys_weight",1.0 ));

	static int const trim( intafteroption("nterm_trim",9));
	int const nres_monomer( pose.symmetry_info().nres_monomer_bb() );


	/////////////////////////////////////////////////////////////////////////////
	// abeta phe19 contacts
	float phe_dist_score( 0.0 );
	if ( std::abs( phe_dist_weight ) > 1e-2 ) {
		int const phe_pos( 19 -trim );
		int const ile_pos( 32 -trim );
		int const leu_pos( 34 -trim );
		int const val_pos( 36 -trim );

		Vec
			phe( &(pose.centroid()(1,phe_pos))),
			ile( &(pose.centroid()(1,ile_pos))),
			leu( &(pose.centroid()(1,leu_pos))),
			val( &(pose.centroid()(1,val_pos)));

		float const phe_dist1
			( std::sqrt( square( phe(1) - ile(1) ) + square( phe(2) - ile(2) ) ) );

		float const phe_dist2
			( std::sqrt( square( phe(1) - leu(1) ) + square( phe(2) - leu(2) ) ) );

		float const phe_dist3
			( std::sqrt( square( phe(1) - val(1) ) + square( phe(2) - val(2) ) ) );

		float const phe_dist( std::min( std::min( phe_dist1, phe_dist2 ),
																		phe_dist3 ) );
		phe_dist_score = ( phe_dist < 9.0 ? 0.0 : square( phe_dist-9.0 ) );
	}


	/////////////////////////////////////////////////////////////////////////////
	// asp-lys inward pointing
	float asp_lys_score( 0.0 );
	if ( std::abs( asp_lys_weight ) > 1e-2 ) {
		int const phe_pos( 19 -trim );
		int const asp_pos( 23 -trim );
		int const lys_pos( 28 -trim );
		int const ile_pos( 32 -trim );

		// hbond orientations
		float const dot1( get_tetrad_vector( pose, phe_pos )(3) );
		float const dot2( get_tetrad_vector( pose, asp_pos )(3) );
		float const dot3( get_tetrad_vector( pose, lys_pos )(3) );
		float const dot4( get_tetrad_vector( pose, ile_pos )(3) );


		// now use only the four dots
		if ( ( dot1 < 0 && dot2 < 0 && dot3 < 0 && dot4 < 0 ) ||
				 ( dot1 > 0 && dot2 > 0 && dot3 > 0 && dot4 > 0 ) ) {
			asp_lys_score = 0.0;
		} else {
			asp_lys_score = 10.0;
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	// low-res parallel hbonds
	//
	float hb_score(0.0);
	if ( std::abs( hb_weight ) > 1e-2 ) { // simple hbond
		int const asp_pos( 23 - trim );
		int const lys_pos( 28 - trim );

		static int const turn_begin
			( intafteroption("turn_begin", asp_pos ) );
		static int const turn_end
			( intafteroption("turn_end"  , lys_pos ) );

		int offset1, offset2;
		{
			int pos1,pos2;
			get_aligned_anchors( pose, pos1, pos2 );
			assert( pose.symmetry_info().bb_independent(pos1) );
			offset1 = pos1 - ( ( pos1 - 1)%nres_monomer +1);
			offset2 = pos2 - ( ( pos2 - 1)%nres_monomer +1);
			assert( offset1%nres_monomer == 0 );
		}
		for ( int i=1; i< nres_monomer; ++i ) {
			if ( turn_begin - i - 1 > fibril::hbparam ||
					 i - turn_end       > fibril::hbparam ) continue;
			assert( pose.symmetry_info(). bb_independent( i+offset1 ) &&
							pose.symmetry_info().chi_independent( i+offset2 ) );
			Vec
				o1( &(pose.Eposition()(1,5,i   + offset1 ))),
				o2( &(pose.Eposition()(1,5,i   + offset2 ))),
				n1( &(pose.Eposition()(1,1,i+1 + offset1 ))),
				n2( &(pose.Eposition()(1,1,i+1 + offset2 )));

			float const d( std::min( distance( n1, o2 ), distance( n2, o1 ) ));
			hb_score += ( d < 2.7 ? 0.0 : d - 2.7 );

			// ensure correct orientation
			assert( (o1-o2).length() < 6.0 /* should be about 4.8 */ );
			//assert( std::abs( (o1-o2)(1) ) + std::abs( (o1-o2)(2) ) < 1e-2 );

		}
	}

	return N * ( phe_dist_weight * phe_dist_score +
							 asp_lys_weight * asp_lys_score +
							 hb_weight * hb_score );
}

///////////////////////////////////////////////////////////////////////////////
void
fibril_test()
{
	using namespace pose_ns;

	if ( truefalseoption("cluster") ) {
		fibril_cluster();
		exit(0);

	} else if ( truefalseoption("abeta40" ) ) {
		abeta40_test();
		exit(0);

	} else if ( truefalseoption("abeta_turn" ) ) {
		abeta_turn_test();
		exit(0);

	} else if ( truefalseoption("abeta_double" ) ) {

		abeta_double();

	} else {

		//abeta40_rerelax();

		//setup_simple_electrostatics();
		//exit(0);

		std::cout << "STOP: fibril mode undefined!" << std::endl;
		std::exit( EXIT_FAILURE );
	}

}



///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

