// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 08:39:36 +0200 (Sun, 18 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "filters.h"
#include "after_opts.h"
#include "cenlist.h"
#include "disulfides.h"
#include "files_paths.h"
#include "initialize.h"
#include "misc.h"
#include "param.h"
#include "param_aa.h"
#include "runlevel.h"
#include "sheet_filter.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/char.functions.hh>
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <sstream>

// Namespaces
namespace filters {
	float co_cutoff = { -1.0 };
	float rg_standard;
	float rms_cutoff;
}

//// yab: misc removal
//// originally in misc, moved to here because it's only used in filters.cc
namespace score_contact {
	bool score_contact_flag = { false };
	float score_contact_weight = { 1.0 };
	float score_contact_threshold = { 0.5 };
	int score_contact_seq_sep = { 2 };
	int total_contact = { 0 };
//	FArray1D_int aa1( MAX_RES() * MAX_RES() );
//	FArray1D_int aa2( MAX_RES() * MAX_RES() );
//	FArray1D_float prob( MAX_RES() * MAX_RES() );
//	FArray1D_float contactdist( MAX_RES() * MAX_RES() );
	FArray1D_int aa1;  // now initialized upon read_contact_prediction()
	FArray1D_int aa2;  // now initialized upon read_contact_prediction()
	FArray1D_float prob; // now initialized upon read_contact_prediction()
	FArray1D_float contactdist; // now initialized upon read_contact_prediction()
	bool score_contact_usecalpha  = { false };
	float score_contact_defaultdistance = { 8.0 };
	bool score_contact_readindist = { false };
}
//// yab: misc removal

////////////////////////////////////////////////////////////////////////////////
/// @begin contact_order
///
/// @brief
///
/// @detailed
///
/// @param  co - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
contact_order(
	float & co,
	FArray1D_int const & res,
	int const & total_residue
)
{
	using namespace cenlist_ns;
	using namespace param_aa;

	co = 0.0;
	int nco = 0;
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( is_protein(res(i)) || is_nonnatural(res(i)) ) {
			for ( int kk = 1, kke = cen12up(i); kk <= kke; ++kk ) {
				int j = cen_list(kk,i);
				if ( cendist(i,j) < 64.0 && std::abs(j-i) > 2 ) {
					co += std::abs(j-i);
					++nco;
				}
			}
		}
	}
	if ( nco > 0 ) co /= static_cast< float >( nco );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin contact_order_cutoff
///
/// @brief
///
/// @detailed
/// here are the co 5%,50%,5% dividing lines for native (70-180res) proteins
///------------------------------------------------------------------------------
///        alpha:
///                upper: 0.220 * len + 5.50
///                mid  : 0.176 * len + 2.07
///                lower: 0.150 * len - 3.0
///
///        beta :
///                upper: 0.340 * len + 2.00
///                mid  : 0.238 * len + 6.00
///                lower: 0.145 * len + 8.00
///
///        a/b  :
///                upper: 0.250 * len + 8.75
///                mid  : 0.187 * len + 7.36
///                lower: 0.137 * len + 3.25
///------------------------------------------------------------------------------
///
/// @param  ss_type - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  co_cutoff - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
contact_order_cutoff(
	int & ss_type,
	int & total_residue,
	float & co_cutoff
)
{
	using namespace files_paths;

	co_cutoff = 0.0;
	if ( ss_type == 2 ) {    //  alpha beta
		co_cutoff = ( 0.137f * total_residue ) + 3.25f;
	} else if ( ss_type == 3 ) { // all beta
		co_cutoff = ( 0.145f * total_residue ) + 7.50f;
	} else {
		use_filter(co_type) = false; // no co filtering of all alpha
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_jones_1state
///
/// @brief
///
/// @detailed
///
/// @param  iunit - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  seq - [in/out]? - sequence in 1 letter code
/// @param  ss_jon - [in/out]? -
/// @param  fail - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_jones_1state(
	utility::io::izstream & iunit,
	int & total_residue,
	FArray1Da_char seq, // sequence in 1 letter code
	FArray1Da_char ss_jon,
	bool & fail
)
{
	seq.dimension( total_residue );
	ss_jon.dimension( total_residue );

//car local
	FArray1D_char seq_jon( 60, ' ' );
	std::string indicator;

//------------------------------------------------------------------------------

	int line = 0;
L666:
	indicator = "abcdefgh";
	while ( indicator.substr(0,5) != "Conf:" ) {
		++line;
		iunit >> bite( indicator ) >> skip;
		if ( iunit.eof() ) goto L201;
	}
	if ( llt( indicator[6], '0' ) || lgt( indicator[6], '9' ) ) goto L666;
	iunit.seek_beg();
	for ( int i = 1; i < line; ++i ) {
		iunit >> skip;
	}

	for ( int i = 1; i <= total_residue; i += 60 ) {
		iunit >> skip;
		iunit >> skip( 6 );
		for ( int j = i, je = std::min( i+59, total_residue ); j <= je; ++j ) {
			iunit >> bite( ss_jon(j) );
		}
		iunit >> skip;
		iunit >> skip( 6 );
		for ( int j = 1; j <= 60; ++j ) {
			iunit >> bite( seq_jon(j) );
		}
		iunit >> skip;

		if ( std::min(i+59,total_residue) < total_residue ) {
			iunit >> skip;
			iunit >> skip;
			iunit >> skip;
		}
		for ( int j = 1; j <= 60; ++j ) {
			int const k = i + j - 1; // residue number
			if ( k < total_residue ) {
				if ( seq(k) != seq_jon(j) ) {
					std::cout << "jones mismatch: " << k << ' ' << seq(k) << " jon: " <<
					 seq_jon(j) << std::endl;
					iunit.close();
					iunit.clear();
					fail = true;
					return;
				}
				if ( ss_jon(j) == 'C' ) ss_jon(j) = 'L';
				if ( ss_jon(j) == ' ' ) ss_jon(j) = 'L';
				if ( ss_jon(j) != 'H' && ss_jon(j) != 'L' && ss_jon(j) != 'E' ) {
					std::cout << "ERROR reading jones file!" << std::endl;
					std::cout << "unrecognized ss type: " << k << ' ' << ss_jon(j) << std::endl;
					iunit.close();
					iunit.clear();
					fail = true;
					return;
				}
			}
		}
	}
	iunit.close();
	iunit.clear();
	fail = false;
	return;

L201:
	std::cout << "unable to find start of jones file" << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_jones_3state
///
/// @brief
///
/// @detailed
///
/// @param  iunit - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  seq - [in/out]? - sequence in 1 letter code
/// @param  ss_jon - [in/out]? -
/// @param  error - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_jones_3state(
	utility::io::izstream & iunit,
	int & total_residue,
	FArray1Da_char seq, // sequence in 1 letter code
	FArray1Da_char ss_jon,
	bool & error
)
{
	using namespace runlevel_ns;

	seq.dimension( total_residue );
	ss_jon.dimension( total_residue );

//car local
	float jonL, jonH, jonE;
	char seq_jon;
	std::string line;

//car v2.01 has no header lines
//car v2.3 does
	while ( line.substr(0,4) != "   1" ) {
		iunit >> bite( line ) >> skip;
		if ( iunit.eof() ) goto L201;
	}
	{ // Scope
		std::istringstream line_stream( line );
		line_stream >> skip( 5 ) >>
		 bite( seq_jon ) >> skip( 1 ) >>
		 bite( ss_jon(1) ) >> skip( 3 ) >>
		 bite( 5, jonL ) >> skip( 2 ) >>
		 bite( 5, jonH ) >> skip( 2 ) >>
		 bite( 5, jonE );
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( i > 1 ) {
			iunit >> skip( 5 ) >>
			 bite( seq_jon ) >> skip( 1 ) >>
			 bite( ss_jon(i) ) >> skip( 3 ) >>
			 bite( 5, jonL ) >> skip( 2 ) >>
			 bite( 5, jonH ) >> skip( 2 ) >>
			 bite( 5, jonE ) >> skip;
			if ( iunit.eof() ) {
				goto L101;
			} else if ( iunit.fail() ) {
				goto L201;
			}
		}
		if ( ss_jon(i) == 'C' ) ss_jon(i) = 'L';
		if ( runlevel > inform )
		 std::cout << "ss_jon(i) " << i << ' ' << ss_jon(i) << std::endl;
		if ( ss_jon(i) != 'L' && ss_jon(i) != 'H' && ss_jon(i) != 'E' ) {
			std::cout << "unrecognized ss type " << ss_jon(i) << ' ' << i << std::endl;
			goto L301;
		}
		if ( seq_jon != seq(i) ) {
			std::cout << "jones mismatch: " << i << ' ' << seq(i) << " jones: " <<
			 seq_jon << std::endl;
			goto L301;
		}
	}

L101:
	iunit.close();
	iunit.clear();
	error = false;
	return;

L201:
	std::cout << "unable to read jones 3-state (psipred_ss2) file" << std::endl;

L301:
	iunit.close();
	iunit.clear();
	error = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_filters
///
/// @brief
///car setup parameters for filtering decoys before output
///car see files_paths.h for existing filter types
///
/// @detailed
///car note that filters are pretty much hardwired for a single
///car query (ie this setup function is designed to be called only
///car 1 time per run; self-initializing filters have no mechanism
///car to reinitialize for a new query sequence)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_filters()
{
	using namespace files_paths;
	using namespace filters;
	using namespace misc;

//car local
	int type;

	if ( disable_filters ) disable_all_filters();

	if ( total_residue < 0 ) {
		std::cout << "WARNING:: total_residue is not yet defined" << std::endl;
		std::cout << "default cutoffs for rg and co filters cannot be determined" << std::endl;
		use_filter(rg_type) = false;
		use_filter(co_type) = false;
	}

//car now set up all filters that can be activated by command line:

//car ss-dependent behavior: rg, co, sheet filters
	std::string protein_sstype = get_protein_sstype(); //chu setup occurs earlier

//car  rg filter initializtion
	if ( use_filter(rg_type) ) {
		rg_standard = ( 3.0 * std::pow(
		 static_cast< float >( total_residue ), ( 1.0f / 3 ) ) ) + 2;
		std::cout << "RG cutoff: " << SS( rg_standard ) << std::endl;
	}

//car  co filter initialization
	realafteroption("co",-1.0,co_cutoff);
	if ( co_cutoff == -1.0 ) {
		if ( use_filter(co_type) ) {
			type = 0;
			if ( protein_sstype == "a" ) type = 1;
			if ( protein_sstype == "ab" ) type = 2;
			if ( protein_sstype == "b" ) type = 3;
			if ( type == 0 ) {
				std::cout << "WARNING::unable to determine protein" << std::endl;
				std::cout << "         ss_type!! Assuming a/b" << std::endl;
				type = 2;
			}
			contact_order_cutoff(type,total_residue,co_cutoff);
		}
	} else {
		use_filter(co_type) = true;
	}
	if ( use_filter(co_type) ) std::cout << "Contact order cutoff: " <<
	 SS( co_cutoff ) << std::endl;

//car sheet filter self initializes
//car knot filter self initializes

//car docking filters self initialize?
	disulfides::BOUNDARY::setup_disulfide_filter();

//bk rms filter
//car this filter can only be activated by command line option
	realafteroption( "rms", -1.0, rms_cutoff );
	if ( rms_cutoff > 0.0 ) {
		std::cout << "RMS filter cutoff: " << SS( rms_cutoff ) << std::endl;
		if ( get_native_exists() ) {
			use_filter(rms_type) = true;
		} else {
			std::cout << "WARNING:: " <<
			 "native structure unknown; disabling rms filtering" << std::endl;
			use_filter(rms_type) = false;
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin disable_all_filters
///
/// @brief  disables all filters
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write    use_filter      files_paths.h
///
/// @remarks
///
/// @references
///
/// @authors car 9/3/2003
///
/////////////////////////////////////////////////////////////////////////////////
void
disable_all_filters()
{
	using namespace files_paths;

	std::cout << "disabling all filters" << std::endl;
	for ( int i = 1; i <= MAX_FILTERS; ++i ) {
		use_filter(i) = false;
	}
}
////////////////////////////////////////////////////////////////////////////////
/// @begin get_protein_sstype
///
/// @brief get the ss_type of the query protein
///
/// @detailed
///
/// @param
///
/// @global_read: misc::ss_type
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 2005/08/24/
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
std::string get_protein_sstype() {

	return misc::sstype::sstype;

}
////////////////////////////////////////////////////////////////////////////////
/// @begin setup_protein_sstype
///
/// @brief read in psipred files and determine query protein ss_type
///
/// @detailed
///
/// @param  seq_undefined - [in] - query sequence defined or not ?
///
/// @global_read:
///
/// @global_write: misc::ss_type
///
/// @remarks
/// note that this function should be called only once in the initialize_query stage
/// and requires the query sequence has been defined already, i.e., a fasta file has
/// been read in. As the names of psipred files are associated with the query name,
/// it does not work with multiple queries( not supported by Rosetta anyway ).
/// Once ss_type is determined, it is saved globally and can be retrieved through
/// fucntion get_protein_sstype().
///
/// @references
///
/// @authors
///
/// @last_modified Chu Wang 2005/08/24
/////////////////////////////////////////////////////////////////////////////////
void
setup_protein_sstype( bool seq_undefined )
{
	using namespace files_paths;
	using namespace misc;
	using namespace param;

	if ( ! query_defined ) return;

	if ( seq_undefined ) {
		std::cout << "setup_protein_sstype: query sequence not defined !!!" << std::endl;
		std::cout << "fasta file needed to read in psipred prediction !!!" << std::endl;
		std::cout << "query sstype will not be defined in this run !!!" << std::endl;
		return;
	}

//car local
	bool error;
	int alpha,beta;
	float beta_ratio;
	int helix_length;
	int max_helix_length;
	float max_helix_fraction;
	std::string protein_sstype;
	FArray1D_char ss_jon( MAX_RES(), ' ' );

	//chu read psipred files
 	utility::io::izstream seq_zx;

	std::string filename;

//car look for 3-state jones file:
	filename = seq_path + protein_name_prefix + protein_name + protein_chain + ".psipred_ss2";
	std::cout << "Looking for psipred file: " << filename << std::endl;
	seq_zx.open( filename );

	if ( seq_zx ) goto L200; // success

	filename = ss_path + protein_name_prefix + protein_name + protein_chain + ".psipred_ss2";
	std::cout << "Looking for psipred file: " << filename << std::endl;
	seq_zx.clear();
	seq_zx.open( filename );
	if ( seq_zx ) goto L200; // success

//car look for 1-state jones file:

L50:
	filename = seq_path + protein_name_prefix + protein_name + protein_chain + ".jones";
	std::cout << "Looking for psipred file: " << filename << std::endl;
	seq_zx.clear();
	seq_zx.open( filename );
	if ( seq_zx ) goto L100; // success

	filename = seq_path + protein_name_prefix + protein_name + protein_chain + ".psipred";
	std::cout << "Looking for psipred file: " << filename << std::endl;
	seq_zx.clear();
	seq_zx.open( filename );
	if ( seq_zx ) goto L100; // success

	filename = ss_path + protein_name_prefix + protein_name + protein_chain + ".jones";
	std::cout << "Looking for psipred file: " << filename << std::endl;
	seq_zx.clear();
	seq_zx.open( filename );
	if ( seq_zx ) goto L100;

	filename = ss_path + protein_name_prefix + protein_name + protein_chain + ".psipred";
	std::cout << "Looking for psipred file: " << filename << std::endl;
	seq_zx.clear();
	seq_zx.open( filename );
	if ( seq_zx ) goto L100;

	std::cout << "WARNING :: Unable to find psipred file!" << std::endl;
	return;

L100:
	read_jones_1state(seq_zx,total_residue,residue1,ss_jon,error);
	if ( error ) return;
	goto L300;

L200:
	read_jones_3state(seq_zx,total_residue,residue1,ss_jon,error);
	if ( error ) goto L50;

L300:

	alpha = 0;
	beta = 0;
	helix_length = 0;
	max_helix_length = 0;
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( ss_jon(i) == 'H' ) {
			++alpha;
			++helix_length;
			if ( helix_length > max_helix_length ) {
				max_helix_length = helix_length;
			}
		} else if ( ss_jon(i) == 'E' ) {
			++beta;
			helix_length = 0;
		} else {
			helix_length = 0;
		}
	}
	max_helix_fraction = static_cast< float >( max_helix_length ) / total_residue;

//car protein_ss_type
	beta_ratio = static_cast< float >( beta ) / ( alpha + beta );
	if ( beta_ratio >= 0.8 ) {
		std::cout << "Protein type: all beta  Fraction beta: " <<
		 SS( beta_ratio ) << std::endl;
		protein_sstype = 'b';
	} else if ( beta_ratio > 0.2 && beta >= 10 ) {
		std::cout << "Protein type: alpha/beta  Fraction beta: " <<
		 SS( beta_ratio ) << std::endl;
		protein_sstype = "ab";
	} else {
		std::cout << "Protein type: all alpha  Fraction beta: " <<
		 SS( beta_ratio ) << "nbeta" << SS( beta ) << std::endl;
		protein_sstype = 'a';
	}
	misc::sstype::sstype = protein_sstype;

//car rg filter disable
	if ( max_helix_length > 25 || max_helix_fraction > 0.22 ) rg_filter_disable();

//car sheet_filter disable
	if ( beta < 20 || beta_ratio <= 0.2 ) sheet_filter_disable();

}

void
setup_protein_sstype_from_input()
{
	using namespace files_paths;
	using namespace misc;
	using namespace param;

	int alpha,beta;
	float beta_ratio;
	int helix_length;
	int max_helix_length;
	float max_helix_fraction;
	std::string protein_sstype;
	FArray1D_char ss_jon( MAX_RES(), ' ' );

	alpha = 0;
	beta = 0;
	helix_length = 0;
	max_helix_length = 0;
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( secstruct(i) == 'H' ) {
			++alpha;
			++helix_length;
			if ( helix_length > max_helix_length ) {
				max_helix_length = helix_length;
			}
		} else if ( secstruct(i) == 'E' ) {
			++beta;
			helix_length = 0;
		} else {
			helix_length = 0;
		}
	}
	max_helix_fraction = static_cast< float >( max_helix_length ) / total_residue;

//car protein_ss_type
	beta_ratio = static_cast< float >( beta ) / ( alpha + beta );
	if ( beta_ratio >= 0.8 ) {
		std::cout << "Protein type: all beta  Fraction beta: " <<
		 SS( beta_ratio ) << std::endl;
		protein_sstype = 'b';
	} else if ( beta_ratio > 0.2 && beta >= 10 ) {
		std::cout << "Protein type: alpha/beta  Fraction beta: " <<
		 SS( beta_ratio ) << std::endl;
		protein_sstype = "ab";
	} else {
		std::cout << "Protein type: all alpha  Fraction beta: " <<
		 SS( beta_ratio ) << "nbeta" << SS( beta ) << std::endl;
		protein_sstype = 'a';
	}
	misc::sstype::sstype = protein_sstype;

	use_filter(rg_type) = true;
	use_filter(sheet_type) = true;

//car rg filter disable
	if ( max_helix_length > 25 || max_helix_fraction > 0.22 ) rg_filter_disable();

//car sheet_filter disable
	if ( beta < 20 || beta_ratio <= 0.2 ) sheet_filter_disable();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin co_filter
///
/// @brief
///
/// @detailed
///
/// @param  co - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
co_filter( float & co )
{
	using namespace files_paths;
	using namespace filters;

	bool co_filter; // Return value

//car returns true if passes the filter

	co_filter = true;
	if ( !use_filter(co_type) ) return co_filter;
	if ( co < co_cutoff ) co_filter = false;

	return co_filter;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rg_filter
///
/// @brief
///
/// @detailed
///
/// @param  rg - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
rg_filter( float & rg )
{
	using namespace files_paths;
	using namespace filters;

	bool rg_filter; // Return value

//car returns true if passes the filter

	rg_filter = true;
	if ( !use_filter(rg_type) ) return rg_filter;
	if ( rg > rg_standard ) rg_filter = false;

	return rg_filter;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rms_filter
///
/// @brief
///
/// @detailed
///
/// @param[in]   rms - in - rms to native
///
/// @return  true if structure passes filter (rms < cutoff)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  car 10/7/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
rms_filter( float & rms )
{
	using namespace files_paths;
	using namespace filters;

//car returns true if passes the filter

	if ( !use_filter(rms_type) ) return true;
	return ( rms <= rms_cutoff );
}
////////////////////////////////////////////////////////////////////////////////
/// @begin rg_filter_disable
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rg_filter_disable()
{
	using namespace files_paths;

	if ( use_filter(rg_type) ) std::cout << "disabling rg filter" << std::endl;
	use_filter(rg_type) = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_score_contact_optons
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_score_contact_options()
{
	if ( truefalseoption("score_contact_flag") ||
			truefalseoption("score_contact_weight") ||
			truefalseoption("score_contact_threshold")) {
		score_contact::score_contact_flag = true;
		realafteroption("score_contact_weight",float( 1.0),score_contact::score_contact_weight);
		realafteroption("score_contact_threshold",float( 0.5),score_contact::score_contact_threshold);
		intafteroption("score_contact_seq_sep", int( 1), score_contact::score_contact_seq_sep);
		score_contact::score_contact_seq_sep = std::max( 1, score_contact::score_contact_seq_sep);


		//rhiju
		if ( truefalseoption("score_contact_calpha") ){
			score_contact::score_contact_usecalpha = true;
			realafteroption("score_contact_distance",float( 11.0),
											score_contact::score_contact_defaultdistance);
		}
		else {
			realafteroption("score_contact_distance",float( 8.0),
											score_contact::score_contact_defaultdistance);
		}
		if ( truefalseoption("score_contact_readindist") )
				score_contact::score_contact_readindist = true;
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin read_contact_prediction
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @return  fraction native contacts
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  mj 03/28/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_contact_prediction(
	int const & total_residue
)
{
	using namespace cenlist_ns;
	using namespace param_aa;
	using namespace files_paths;
	using namespace param;

	// return if flag is not set
	if ( !score_contact::score_contact_flag ) return;

	// reset contacts
	score_contact::total_contact = 0;

	// initialize arrays
	score_contact::aa1 = FArray1D_int( MAX_RES()() * MAX_RES()() );
	score_contact::aa2 = FArray1D_int( MAX_RES()() * MAX_RES()() );
	score_contact::prob = FArray1D_float( MAX_RES()() * MAX_RES()() );
	score_contact::contactdist = FArray1D_float( MAX_RES()() * MAX_RES()() );

	// make filename
	std::string const default_filename( constraints_path + protein_name_prefix + protein_name + protein_chain + ".contact" );
	std::string filename;
	stringafteroption("score_contact_file",default_filename,filename);


	// open file
  utility::io::izstream cnstr_zx( filename );

	if ( !cnstr_zx ) {
		std::cerr << "contact prediction: score_contact_flag is true but file " << cnstr_zx.filename() << " with predicted contact was not found!" << std::endl;
		std::cerr << "provide file of format <aa1> <aa2> <probability (0..1)>" << std::endl;
		std::cerr << "if the aa1 centroid - aa2 centroid distance in the decoy is below 8A" << std::endl;
		std::cerr << "and abs( aa2 - aa1 ) >= score_contact_seq_sep its energy will be decreased by" << std::endl;
		std::cerr << "(probability - score_contact_threshold) * score_contact_weight/ total residue" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return;
	}

	// read data
	int aa1, aa2;
	float prob;
	float whatdist;
	int const MAX_RES_squared = MAX_RES()() * MAX_RES()();
	while ( cnstr_zx >> aa1 && score_contact::total_contact < MAX_RES_squared ) {
		cnstr_zx >> aa2 >> prob ;

		if (score_contact::score_contact_readindist){
			cnstr_zx >> whatdist >> skip;
		}
		else {
			whatdist = score_contact::score_contact_defaultdistance;
			cnstr_zx >> skip;
		}


		if ( 0 < aa1 && aa1 <= total_residue && 0 < aa2 && aa2 <= total_residue && std::abs( aa2 - aa1) >= score_contact::score_contact_seq_sep ) {
			++score_contact::total_contact;
			score_contact::aa1(  score_contact::total_contact ) = aa1;
			score_contact::aa2(  score_contact::total_contact ) = aa2;
			score_contact::prob( score_contact::total_contact ) = prob - score_contact::score_contact_threshold;
			score_contact::contactdist( score_contact::total_contact ) = whatdist*whatdist;
		}
	}

	std::cout << "contact prediction: " << score_contact::total_contact << " contacts read from file " << cnstr_zx.filename() << std::endl;

	//rhiju -- Calpha-Calpha contacts
  if (score_contact::score_contact_usecalpha)
 		std::cout << "Flag score_contact_calpha:  Using Calpha-Calpha instead of centroid-centroid distances." << std::endl;

	//close file
	cnstr_zx.close();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin score_contact_prediction
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @return  fraction native contacts
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  mj 03/28/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score_contact_prediction(
	FArray3Dp_float const & Eposition,
	int const & total_residue
)
{
	using namespace param;
	using namespace cenlist_ns;
	using namespace param_aa;
	using numeric::xyzVector_float;

	//return if flag is not set
	if ( !score_contact::score_contact_flag ) return 0.0f;

	// read contacts if not yet done
	if ( score_contact::total_contact == 0 )
		read_contact_prediction(total_residue);

	// compute score
	float contact_score = 0.0;
	int contact_count = 0;
	int i,j;
	float alpha_alpha = 0.0;

	//Added by rhiju
	// Look at calpha-calpha distances... useful for integrating low resolution
	// information, e.g. from fold recognition servers, that only have Calpha's.
	if (score_contact::score_contact_usecalpha)
		for ( int k = 1; k <= score_contact::total_contact; ++k ) {
			i = score_contact::aa1( k);
			j = score_contact::aa2( k);
			xyzVector_float const Epos_2i( &Eposition(1,2,i) );
			xyzVector_float const Epos_2j( &Eposition(1,2,j) );
			alpha_alpha = distance_squared( Epos_2i, Epos_2j );
			//Default contactdist = 11.0*11.0 in calpha mode.
			if ( alpha_alpha < score_contact::contactdist( k)) {
				contact_score -= score_contact::prob( k);
				++contact_count;
			}
		}
	else
		//Default -- use centroid-centroid separations to evaluate contacts.
		for ( int i = 1; i <= score_contact::total_contact; ++i ) {
			//Default contactdist = 8.0*8.0 in centroid mode.
			if ( cendist( score_contact::aa1( i), score_contact::aa2( i)) < score_contact::contactdist( i)) {
				contact_score -= score_contact::prob( i);
				++contact_count;
			}
		}

	//std::cout << "contact_score: " << contact_score << " | contact_count:  " << contact_count << std::endl;

	//return
	return 200.0f * score_contact::score_contact_weight * contact_score / total_residue;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin sheet_filter_disable
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
sheet_filter_disable()
{
	using namespace files_paths;

	if ( use_filter(sheet_type) ) std::cout << "disabling sheet filter" << std::endl;
	use_filter(sheet_type) = false;
}
