// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $


// Rosetta Headers
#include "constraints.h"
#include "counters.h"
#include "diagnostics_rosetta.h"
#include "files_paths.h"
#include "fold_abinitio.h"
#include "fold_loops.h"
#include "fragments.h"
#include "fullatom.h"
#include "idealize.h"
#include "jumping_loops.h"
#include "knots.h"
#include "loops.h"
#include "loops_ns.h"
#include "make_pdb.h"
#include "maps.h"
#include "maps_ns.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "nblist.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "param.h"
#include "param_pack.h"
#include "pose_loops.h"
#include "recover.h"
#include "refine_structure.h"
#include "refold.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "smallmove.h"
#include "status.h"
#include "tether.h"
#include "torsion_bbmove_trials.h"

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <vector>

//Utility Headers
#include <utility/basic_sys_util.hh>

////////////////////////////////////////////////////////////////////////////////
/// @begin fold_loops
///
/// @brief
///
/// @detailed
///car main protocol for folding loops.
///car note that recoverLOW's only refold from angles the regions that
///car are variable.  Should always do a recover_LOW or a monte_carlo_reset
///car before making any regions fixed. (ie best_phi and low_phi should
///car agree when changing allow_insert)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fold_loops()
{

	using namespace loops_ns;
	using namespace misc;
	using namespace runlevel_ns;

	int const all_loops = { -1 }; // invalid loop number

//car NOTE: these cutoffs are consistent with checks in main_wobble
	int const cutoff3 = { 7 }; // no 3mer insertions in anything shorter than this
	int const cutoff9 = { 17 };
	 // no 9mer insertions in anything shorter than this

	float const splicerms_cutoff = { 0.5 }; // loops < cutoff are considered closed
	// msd still contributes to score

	bool exist;

	monte_carlo_set_simannealing(true);
	classical_constraints::BOUNDARY::set_max_seqSep(total_residue);
	minimize_exclude_sstype(false,false);
	set_use_nblist(false);

	if ( fold_with_dunbrack ) {
		if ( get_pose_loop_flag() ) {
			pose_loops_main();
			return;
		}
		// this is a separate loop modeling protocol which calls dunbrack loop
		// closure everytime after backbone is perturbed.
		fold_loops_with_dunbrack_closure();
		return;
	}
//car protocols set loop weights and select regions to operate on
	if ( screen ) screen_loops();

//car select starting loops
	if ( permute || fold ) {
		recover_LOW(score0L);
		set_allow_insert_by_looplength(total_residue+1,true,exist);
		select_starting_loops();
		refold(1,total_residue);
		save_status_info("select_loops",0,0);
		monte_carlo_reset(); // save as low structure
	}

//vds screen loop library for compatible, optimal extensions
	if ( grow ) {
		recover_LOW(score0L);
		set_allow_insert_by_looplength(total_residue+1,true,exist);
		refold(1,total_residue);
		save_status_info("select_loops",0,0);
		monte_carlo_reset(); // save as low structure
		grow_tail ();
	}

	if ( permute ) {
		set_allow_insert_by_looplength(total_residue+1,true,exist);
		if ( loops_ns::loop_bool::trim ) trim_template();
		if ( fast ) {
			permute_loops();
		} else {
			permute_minimize_loops();
			close_loop(splicerms_cutoff,fast,all_loops);
		}
		if ( loops_ns::loop_bool::trim ) trim_template_reset();
	}

	if ( fold ) {

//car fold loops:
		if ( sequential ) {

//car fold loops one at a time:
			for ( int irank = 1; irank <= num_loop; ++irank ) {
				int iloop = loop_rank(irank);
				std::cout << "folding loop " << iloop << " rank " << irank << std::endl;

//car update trimming now that loop is selected
				set_allow_insert_by_loopnum(iloop,true,exist);
				if ( loops_ns::loop_bool::trim ) trim_template();

				fold_extended_loop(loops_ns::loop_bool::trim,iloop);

//car  check that this is an okay conformation, otherwise repeat?
				build_loop(cutoff3,cutoff9,fast,iloop);
				optimize_loop(cutoff3,splicerms_cutoff,fast,iloop);
				close_loop(splicerms_cutoff,fast,iloop); // do for all loops?
				if ( loops_ns::loop_bool::trim ) trim_template_reset();
			}

		} else {

//car update trimming
			set_allow_insert_by_looplength(total_residue+1,true,exist);
			if ( loops_ns::loop_bool::trim ) trim_template();

			fold_extended_loop(loops_ns::loop_bool::trim,all_loops);

//car frag moves in long,internal loops
			build_loop(cutoff3,cutoff9,fast,all_loops);

//car min_moves in loops
			optimize_loop(cutoff3,splicerms_cutoff,fast,all_loops);

//car close all  (all loops open by more than splicerms_cutoff)
			close_loop(splicerms_cutoff,fast,all_loops);

			if ( loops_ns::loop_bool::trim ) trim_template_reset();

		}         // sequential/simultaneous
	}           // fold option

	if ( !knot_filter() ) return; // try again?, caught in output decoy

	if ( fa_refine ) {
		set_fullatom_flag(true); //chu build fullatom coord before trimming
//chu trim_template only changes the current pose,  but not the starting pose.
//chu If the fullatom_coord is built after trimming, ie, in refine_loops,
//chu starting side-chains will not be added to the current backbone correctly

		if ( loops_ns::loop_bool::trim ) trim_template();
		refine_loops(fast,fix_natsc);
		if ( loops_ns::loop_bool::trim ) trim_template_reset();
	}

	if ( idl_breaks ) idealize_chainbreaks();
	// note that idealize_chainbreak leaves refolding set up for
	// a complete idealized structure with no breaks, but loops are still
	// defined and the loop_flag is ON!!!  This is the desired setup
	// to end a run through output_decoy, but probalby not if another
	// protocol is called.

}
////////////////////////////////////////////////////////////////////////////////
/// @begin fold_extended_loop
///
/// @brief call fold_abinitio to fold one or more  extended regions
///
/// @detailed
///
/// @param[in]   trim - in - use template trimming?
/// @param  loop_num - [in/out]? - non-valid loop number means all extended regions
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  car 9/4/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fold_extended_loop(
	bool trim,
	int loop_num // non-valid loop number means all extended regions
)
{
	using namespace misc;

//car local
	bool exist;

	if ( !loopnum_is_valid(loop_num) ) {
		set_allow_insert_by_extended(true,exist);
	} else {
		if ( !loop_is_extended(loop_num) ) return;
		set_allow_insert_by_loopnum(loop_num,true,exist);
	}
	if ( !exist ) return;
	set_use_nblist(false);

	check_secstruct(secstruct,total_residue);
	score_set_loop_weight(0.0);
	fold_abinitio();

//car update trimming now that loops are folded
	if ( trim ) {
		trim_template_reset();
		trim_template();
	}

}
////////////////////////////////////////////////////////////////////////////////
/// @begin build_loop
///
/// @brief
///
/// @detailed
///
/// @param[in]   cutoff3 - in - length cutoff for 3mer insertions
/// @param[in]   cutoff9 - in - length cutoff for 3mer insertions
/// @param[in]   fast - in - fast protocol?
/// @param[in]   loop_num - in - operate on this loop number, return if not valid
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     no splice rms escape
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_loop(
	int cutoff3,
	int cutoff9,
	bool fast,
	int loop_num // loop number to operate on
)
{
	using namespace runlevel_ns;

	using namespace mc_global_track::mc_score; // yab: misc removal

//car local
	int length; // loop length
	int nloop;
	int cycles;
	float gamma, final_temp,init_temp,temperature;
	float loop_weight = 0.0;
	int num_loop;
	FArray1D_float splice_rms( 1000 );
	bool exist;
	int total_size;

//------------------------------------------------

	if ( loopnum_is_valid(loop_num) ) {
		set_allow_insert_by_loopnum(loop_num,true,exist);
	} else {
		set_allow_insert_by_looplength(cutoff3,false,exist);
	}
	insert_map_get_max_contig_size(length,total_size);

//car figure out if 9mers can be inserted
	if ( length < cutoff3 ) return;
	int size = 3; // largest fragment that can be inserted
	if ( length >= cutoff9 ) size = 9;

	std::cout << "fragment insertions in loop" << std::endl;
	set_use_nblist(true);

	recover_LOW(score4L);
	std::cout << "score:" << SS( score ) << std::endl;

	nloop = 5;
	cycles = std::min(1000,total_size*20);
	if ( fast ) cycles = std::min(500,total_size*2);
	if ( benchmark ) {
		nloop = 1;
		cycles = 100;
	}
	init_temp = 1.0;
	final_temp = 0.5;
	gamma = std::pow( (final_temp/init_temp), (1.0f/(nloop*cycles)) );
	temperature = init_temp;
	monte_carlo_set_temp(temperature);

	reset_trial_counters();
	choose_frag_set_top_N_frags(200);
	for ( int jk = 1; jk <= nloop; ++jk ) {
		loop_weight = jk*jk*jk*jk/200.0;
		score_set_loop_weight(loop_weight);
		recover_LOW(score4L);

//--diagnostics
		std::cout << "jk= " << SS( jk-1 ) << " score= " <<
		 SS( score-loop_weight*get_loop_splicemsd() ) << std::endl;
		loop_get_all_splicerms(splice_rms,num_loop);
		std::cout << "    weight " <<
		 F( 6, 1, loop_weight ) << " splice_rms ";
		for ( int j = 1; j <= num_loop; ++j ) {
			std::cout << ' ' << F( 6, 2, splice_rms(j) );
		} std::cout << std::endl;
		output_trial_counters( std::cout );
//--end diagnostics

		for ( int j = 1; j <= cycles; ++j ) {
			temperature *= gamma;
			monte_carlo_set_temp(temperature);

			if ( size >= 9 ) {
				if ( !loopnum_is_valid(loop_num) )
				 set_allow_insert_by_looplength(cutoff9-1,false,exist);
				main_wobble_trial(9,-60.0f,scorefxn,j,1,-1);
				if ( !loopnum_is_valid(loop_num) )
				 set_allow_insert_by_looplength(cutoff3-1,false,exist);
			}
			main_wobble_trial(3,-60.0f,scorefxn,j,1,-1);
		}
	}                     // jk

//$ diagnostics
	score_set_loop_weight(0.0);
	recover_LOW(score4L);
	std::cout << "jk= " << nloop << std::endl;
	std::cout << "score: " << SS( score ) << std::endl;
	loop_get_all_splicerms(splice_rms,num_loop);
	std::cout << "    weight " <<
	 F( 6, 1, loop_weight ) << " splice_rms ";
	for ( int j = 1; j <= num_loop; ++j ) {
		std::cout << ' ' << F( 6, 2, splice_rms(j) );
	} std::cout << std::endl;
	output_trial_counters( std::cout );
	std::cout << "-----------------------------------------------------" << std::endl;
//$ end diagnostics

}


////////////////////////////////////////////////////////////////////////////////
/// @begin optimize_loop
///
/// @brief
///
/// @detailed
///
/// @param[in]   cutoff3 - in - no frag insertions in regions shorter than this
/// @param[in]   splicerms_cutoff - in - no optimization of loops more closed than this
/// @param  fast - [in/out]? - run fast protocol?
/// @param[in]   loop_num -in - loop to operate on, if invalid, use all
///
/// @global_read
///
/// @global_write
///
/// @remarks
///  min moves, splice rms escape
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
optimize_loop(
	int cutoff3,
	float splicerms_cutoff,
	bool fast,
	int loop_num
)
{
	using namespace misc;
	using namespace runlevel_ns;

	using namespace mc_global_track::mc_score; // yab: misc removal

//car local
	float loop_weight = 0.0;
	bool exist;
	int num_loop;
	FArray1D_float splice_rms( 1000 );
	int length,total_size;

	if ( loopnum_is_valid(loop_num) ) {
		set_allow_insert_by_loopnum(loop_num,true,exist);
		fix_lowrms_loops(splicerms_cutoff,true,exist);
	} else {
		set_allow_insert_by_looprms(splicerms_cutoff,false,exist);
	}
	if ( !exist ) return;

	set_use_nblist(true);
	insert_map_get_max_contig_size(length,total_size);
	bool tryfrags = false;
	if ( length > cutoff3 ) tryfrags = true;
	std::cout << "optimize loops" << std::endl;


	recover_LOW(score4L);
	int nloop = 5;
	int cycles = std::min(1000,total_size*20);
	if ( fast ) cycles = std::min(400,total_size*2);
	if ( benchmark ) {
		nloop = 1;
		cycles = 100;
	}
	float init_temp = 1.0;
	float final_temp = 0.5;
	float gamma = std::pow( (final_temp/init_temp), (1.0f/(nloop*cycles)) );
	float temperature = init_temp;
	monte_carlo_set_temp(temperature);

	for ( int jk = 1; jk <= nloop; ++jk ) {
		loop_weight = jk*jk*jk*jk/10.0;
		score_set_loop_weight(loop_weight);

		recover_LOW(score4L);
		fix_lowrms_loops(splicerms_cutoff,true,exist);
		if ( !exist ) goto L100;

//$$$ diagnostics
		std::cout << "jk= " << jk << " score= " <<
		 SS( score-loop_weight*get_loop_splicemsd() ) << std::endl;
		loop_get_all_splicerms(splice_rms,num_loop);
		std::cout << "    weight " <<
		 F( 6, 1, loop_weight ) << " splice_rms ";
		for ( int j = 1; j <= num_loop; ++j ) {
			std::cout << ' ' << F( 6, 2, splice_rms(j) );
		} std::cout << std::endl;
//$$$ end diagnostics

		for ( int j = 1; j <= cycles; ++j ) {
			temperature *= gamma;
			monte_carlo_set_temp(temperature);
//car try frags if possible
			if ( !loopnum_is_valid(loop_num) ) {
				set_allow_insert_by_looplength(cutoff3-1,false,exist);
				fix_lowrms_loops(splicerms_cutoff,true,tryfrags);
			}
			if ( tryfrags ) main_wobble_min_trial(3,-60.,scorefxn,j,1,-1,"linmin");
			if ( !loopnum_is_valid(loop_num) ) {
				set_allow_insert_by_looprms(splicerms_cutoff,false,exist);
				if ( !exist ) goto L100;
			}
//car small/shear
			main_small_min_trial(1,scorefxn,j*jk,"linmin");
			main_shear_min_trial(1,scorefxn,j*jk,"linmin");
		}
	}
	recover_LOW(scorefxn);

L100:
//$$$ diagnostics
	std::cout << "end wobble 3mers, score= " <<
	 SS( score-loop_weight*get_loop_splicemsd() ) << std::endl;
	loop_get_all_splicerms(splice_rms,num_loop);
	std::cout << "    weight " <<
	 F( 6, 1, loop_weight ) << " splice_rms ";
	for ( int j = 1; j <= num_loop; ++j ) {
		std::cout << ' ' << F( 6, 2, splice_rms(j) );
	} std::cout << std::endl;
//$$$ end diagnostics

}

////////////////////////////////////////////////////////////////////////////////
/// @begin close_loop
///
/// @brief
///
/// @detailed
///
/// @param[in]   splicerms_cutoff - in - no optimization of loops more closed than this
/// @param[in]   fast - in -  run fast protocol?
/// @param[in]   loop_num - in - loop number to operate on
///
/// @global_read
///
/// @global_write
///
/// @remarks
///   small/shear min moves; splice rms escape
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
close_loop(
	float splicerms_cutoff,
	bool fast,
	int loop_num
)
{
	using namespace mc_global_track::mc_score; // yab: misc removal
	using namespace mc_global_track::diagnose; // yab: misc removal
	using namespace runlevel_ns;

//car local
	float rms;
	int num_loop;
	int length,total_size;
	FArray1D_float splice_rms( 1000 );
	bool exist;


	recover_LOW(score4L);
	if ( loopnum_is_valid(loop_num) ) {
		loop_get_one_splicerms(loop_num,rms);
		if ( rms < splicerms_cutoff ) return;
	} else {
		set_allow_insert_by_looprms(splicerms_cutoff,false,exist);
		if ( !exist ) return;
	}

	set_use_nblist(true);
	std::cout << "close loops..." << std::endl;
	std::cout << "score:" << SS( scorefxn() ) << std::endl;
	calc_rms();
	if ( rms_err > 0.0 ) std::cout << "rms:" << SS( rms_err ) << std::endl;

	insert_map_get_max_contig_size(length,total_size);
	int nloop = 6;
	int cycles = std::min(1000,total_size*20);
	if ( fast ) cycles = std::min(400,total_size*10);
	if ( benchmark ) {
		nloop = 1;
		cycles = 100;
	}

	float init_temp = 1.0;
	float final_temp = 0.5;
	float gamma = std::pow( ( final_temp / init_temp ),
	 ( 1.0f / ( nloop * cycles * 2 ) ) );
	float temperature = init_temp;
	monte_carlo_set_temp(temperature);
	set_smallmove_size(2.0,5.0,6.0); // helix,strand,other

	reset_trial_counters();

	for ( int jk = 3; jk <= nloop+2; ++jk ) {
		float loop_weight = jk*jk*jk*jk/10.0;
		score_set_loop_weight(loop_weight);
		recover_LOW(score4L);
		std::cout << "jk= " << jk << " score= " <<
		 SS( score - loop_weight * get_loop_splicemsd() ) << std::endl;
		loop_get_all_splicerms(splice_rms,num_loop);
		std::cout << "    weight " <<
		 F( 6, 1, loop_weight ) << " splice_rms ";
		for ( int j = 1; j <= num_loop; ++j ) {
			std::cout << ' ' << F( 6, 2, splice_rms(j) );
		} std::cout << std::endl;
		fix_lowrms_loops(splicerms_cutoff,true,exist);
		if ( !exist ) goto L100;

		for ( int j = 1; j <= cycles; ++j ) {
			temperature *= gamma;
			monte_carlo_set_temp(temperature);
			main_small_min_trial(1,scorefxn,j*jk,"linmin");
			main_shear_min_trial(1,scorefxn,j*jk,"linmin");
		}
	}                     // jk

L100:
	recover_LOW(scorefxn);
	main_minimize_trial(scorefxn,"dfpmin");

}
////////////////////////////////////////////////////////////////////////////////
/// @begin screen_extensions
///
/// @brief
///vds iterate through loops and screen by vdw and rg
///vds attach loop and minimize junction (aka anchor or stub) residue
///vds dump pdb only if vdw and rg conditions met (set on command line)
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
grow_tail()
{
	using namespace files_paths;
	using namespace loops_ns;
	using namespace misc;
	using namespace param;
	using namespace scores;

	using namespace mc_global_track::mc_score; // yab: misc removal

//loop variables
	bool exist, err;
	int loop_num = 1;
//minimization variables
	std::string const move_type( "minimize" );
	std::string const min_type( "linmin" );
	int begin = loop_begin(1); int end = loop_end(1);
	bool gfrag( true );
//junction move variables
	int junction_res1(0);
	int junction_res2(0);
	int const cycles = 1000;
	std::vector< int > min_res;

	score_set_loop_weight(1.0);
	set_allow_insert_by_loopnum( loop_num, true, exist );
	if ( !exist ) return;

	if ( begin == 1 ) { //nterminal
		//minimize_include_one_residue( end );
		junction_res1 = end;
		min_res.push_back( junction_res1 );
	} else if ( end == total_residue ) {//cterminal
		//minimize_include_one_residue( begin );
		junction_res1 = begin;
		min_res.push_back( junction_res1 );
	} else { //non-terminal loop:
		//minimize_include_one_residue( end );
		//minimize_include_one_residue( begin );
		junction_res1 = begin;
		junction_res2 = end;
		min_res.push_back( junction_res1 );
		min_res.push_back( junction_res2 );

		//make protein a single segment for refold
		FArray1D_int new_seg_begin(1), new_seg_end(1), dir(1);
		new_seg_begin(1) = 1;
		new_seg_end(1) = total_residue;
		dir(1) = 1;
		refold_set_segments( 1, 1, new_seg_begin, new_seg_end, dir );

		//allow loop and everything after to refold
		set_allow_insert_by_region(begin, total_residue, true, exist );
	}
	minimize_only_these_residues( min_res );

	for ( int depth_index = 1, depth_index_end = loop_depth(1);
				depth_index <= depth_index_end; ++depth_index ) {
		loop_insert( loop_num, depth_index, err );
		//refold( begin, end );
		refold( 1, total_residue );
		minimize_only_these_residues( min_res );
		minimize(min_type, move_type, scorefxn, begin, end, gfrag);
		score = score3L();
		monte_carlo_reset();
		if ( loops_ns::wiggle_jxn ) {
			for ( int j = 1; j <= cycles; ++j ) {
				main_wiggle_junction_min_trial( junction_res1, scorefxn, j, min_type );
				if(junction_res2){
					main_wiggle_junction_min_trial( junction_res2, scorefxn, j, min_type );
				}
			}
			minimize_only_these_residues( min_res );
			minimize(min_type, move_type, scorefxn, begin, end, gfrag);
		}
		recover_LOW(score3L);
		std::string filename(files_paths::pdb_out_path + files_paths::code + loops_ns::loop_name(depth_index, 1) + "_" + lead_zero_string_of(depth_index,4) + ".pdb");
		if (scores::vdw_score <= vdw_max && scores::rg <= rg_max ) {
			dump_pdb( filename );
		}
	}
}
////////////////////////////////////////////////////////////////////////////////
/// @begin permute_loops
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
permute_loops()
{
	using namespace counters;
	using namespace misc;

	using namespace mc_global_track::mc_score; // yab: misc removal
	using namespace mc_global_track::diagnose; // yab: misc removal

	int cycles;
	int num_loop;
	FArray1D_float splice_rms( 99 );
	bool exist;

	set_allow_insert_by_looplength(total_residue,true,exist);
	if ( !exist ) return;

	score_set_loop_weight(1.0);
	recover_LOW(score3L);

	permute_loops_get_cycles(cycles);
	std::cout << "permute loops, cycles=" << SS( cycles ) << std::endl;

	int nloop = 10;

	float temperature = 1.0;
	monte_carlo_set_temp(temperature);

	reset_trial_counters();
	std::cout << "jk= " << 0 << " score=" << SS( score ) << std::endl;
	loop_get_all_splicerms(splice_rms,num_loop);
	std::cout << "splice_rms ";
	for ( int j = 1; j <= num_loop; ++j ) {
		std::cout << ' ' << F( 6, 2, splice_rms(j) );
	} std::cout << std::endl;

	for ( int jk = 1; jk <= nloop; ++jk ) {
		for ( int j = 1; j <= cycles; ++j ) {
//      main_loop_min_trial(scorefxn,j*jk,"linmin");
			main_loop_trial(scorefxn,j*jk);
		}

//car diagnostics
		retrieve_best_pose();
		calc_rms();
		std::cout << "relaxed score..." << std::endl;
		std::cout << "jk" << ' ' << F( 8, 3, best_score ) << ' ' <<
		 F( 8, 3, low_score ) << ' ' << F( 8, 3, rms_err ) << ' ' <<
		 I( 4, naccept ) << ' ' << I( 4, jk ) << ' ' <<
		 F( 6, 3, temperature ) << std::endl;
		loop_get_all_splicerms(splice_rms,num_loop);
		std::cout << "splice_rms ";
		for ( int j = 1; j <= num_loop; ++j ) {
			std::cout << F( 7, 2, splice_rms(j) );
		} std::cout << std::endl;
//car end diagnositics
	}                     // jk
	main_minimize_trial(scorefxn,"dfpmin");

	output_trial_counters( std::cout );

	std::cout << "------------------------------------------------" << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin permute_loops_get_cycles
///
/// @brief
///car just sets number of permute cycles to the number of loops to permute
///car really should be higher, but until robetta is faster....
///
/// @detailed
///
/// @param  cycles - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
permute_loops_get_cycles( int & cycles )
{
	using namespace loops_ns;
	using namespace protein_maps;


	cycles = 0;
	for ( int i = 1; i <= num_loop; ++i ) {
		if ( allow_insert(loop_begin(i)) ) cycles += loop_depth(i);
	}
//	cycles *= 10;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin permute_minimize_loops
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
permute_minimize_loops()
{
	using namespace counters;

	using namespace mc_global_track::mc_score; // yab: misc removal
	using namespace mc_global_track::diagnose; // yab: misc removal

	int num_loop;
	FArray1D_float splice_rms( 99 );

	std::cout << "permute minimize loops" << std::endl;

	score_set_loop_weight(10.0);
	recover_LOW(score4L);

	int nloop = 1000;
	int cycles = 100;

	float temperature = 1.0;
	monte_carlo_set_temp(temperature);

	reset_trial_counters();
	std::cout << "jk= " << 0 << " score= " << score << std::endl;
	loop_get_all_splicerms(splice_rms,num_loop);
	std::cout << "splice_rms ";
	for ( int j = 1; j <= num_loop; ++j ) {
		std::cout << ' ' << F( 6, 2, splice_rms(j) );
	} std::cout << std::endl;

	for ( int jk = 1; jk <= nloop; ++jk ) {
		recover_LOW(scorefxn);
		for ( int j = 1; j <= cycles; ++j ) {
			main_loop_min_trial(scorefxn,j*jk,"linmin");
			main_small_min_trial(1,scorefxn,j*jk,"linmin");
		}

//car diagnostics
		retrieve_best_pose();
		calc_rms();
		std::cout << "relaxed score..." << std::endl;
		std::cout << "jk" << ' ' << F( 8, 3, best_score ) << ' ' <<
		 F( 8, 3, low_score ) << ' ' << F( 8, 3, rms_err ) << ' ' <<
		 I( 4, naccept ) << ' ' << I( 4, jk ) << ' ' <<
		 F( 6, 3, temperature ) << std::endl;
		loop_get_all_splicerms(splice_rms,num_loop);
		std::cout << "splice_rms ";
		for ( int j = 1; j <= num_loop; ++j ) {
			std::cout << F( 7, 2, splice_rms(j) );
		} std::cout << std::endl;
//car end

	}            // jk

	output_trial_counters( std::cout );

	std::cout << "-----------------------------------------------------" << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin screen_loops
///
/// @brief
///car loops < length_cutoff are screened by rmsd and vdw
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
screen_loops()
{

	using namespace files_paths;
	using namespace loops_ns;
	using namespace misc;
	using namespace scores;
	using namespace runlevel_ns;


//car local
	bool err;
	float globalrms; // rms of loops in template frame
	float splicerms; // sum of splice distances
	int length;
	float bump_score;
	float start_splicerms,min_splicerms;
	bool knots;
	bool exist;

	float const loop_weight = 20;

//car set filename for output
	std::string const fullname( pdb_out_path + libraryname + ".screened" );
	user_x.clear();
	user_x.open( fullname.c_str() );
	if ( !user_x ) {
		std::cout << "Open failed for file: " << fullname << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::cout << "screening loops" << std::endl;

//car move trial and scoring parameters
	set_smallmove_size(2.0,5.0,6.0);
	monte_carlo_set_temp(0.001);
	score_set_loop_weight(loop_weight);
	set_use_nblist(true);
	bool minimize = true;
	std::string min_type( "linmin" );


//select fullatom or centroid evaluation
	if (output_fa) {
		set_fullatom_flag(true);
		score_set_try_rotamers(true);
		if ( !fix_natsc ) fullatom_set_packall(false);
		recover_LOW(score9L);
		minimize = false;
		min_type = "dfpmin";
	} else {
		recover_LOW(score0L);
	}


	for ( int j = 1; j <= num_loop; ++j ) {

		if ( runlevel > standard ) std::cout << "LOOP # " << j << std::endl;

		length = loop_end(j)-loop_begin(j)+1;
		bool terminal = loop_is_terminal(j);
		int cycles = length*2;
		if ( output_fa ) cycles = length;

//car  operate on one loop at a time (setup for trim_template)
//car  could trim here? after inserting first conformation
//car would give constant trim set for a loop
//car  changes:  score before reset
//car            loop j becomes loop 1 in inner loop.
		set_allow_insert_by_loopnum(j, true, exist);


		for ( int i = 1; i <= loop_depth(j); ++i ) {
			loop_insert(j,i,err);

//car skip extended loops
			if ( loop_is_extended(j) ) {
				globalrms = -999.0;
				splicerms = -999.0;
				vdw_score = -999.0;
				knots = false;
				write_loop_library(user_x,j,i,splicerms,vdw_score,knots,globalrms);
				continue;
			}

//car only library loop conformations left
			refold(loop_begin(j),loop_end(j));
			loop_get_one_splicerms(j,start_splicerms);
			reset_trial_counters();

//car trim template to reduce protein size, keeping only loop to screen
			trim_template();   //only 1 loop exists now, ie: loop(j) ==> loop(i)

//car try dunbrack closure; skip terminal loops.
			if ( dunbrack_close && !terminal ) {
				float forward_deviation,backward_deviation;
				scored_ccd_loop_closure(loop_begin(1),loop_end(1),loop_end(1),100,scorefxn,forward_deviation,backward_deviation,0,0);
			}

			main_minimize_trial(scorefxn,"dfpmin");
			loop_get_one_splicerms(1,min_splicerms); // only 1 loop exists now

			if ( !fast && !dunbrack_close ) {
				for ( int jj = 1; jj <= cycles; ++jj ) {
					main_small_min_trial( 1, scorefxn, jj, min_type );
					main_shear_min_trial( 1, scorefxn, jj, min_type );
				}
				if (minimize) {
					recover_LOW(scorefxn);
					main_minimize_trial(scorefxn,"dfpmin");
				}
			}

//restore position array
			trim_template_reset();

//collect final scores for output  (after reset so that all loops scored
//with common trim set (ie all residues)
			loop_get_one_splicerms(j,splicerms);
			loop_global_rms(j,Eposition,globalrms);
			bump_score = vdw_score;
			if (output_fa) {
				score_set_loop_weight(0.0);
				bump_score = score9L();
				score_set_loop_weight(loop_weight);
				score9L();
			}
			knots = !knot_filter();

//save optimized loop torsions for output
			int jj = loop_begin(j);
			for ( int ii = 1; ii <= length; ++ii ) {
				loop_phi(ii,i,j) = phi(jj);
				loop_psi(ii,i,j) = psi(jj);
				loop_omega(ii,i,j) = omega(jj);
				++jj;
			}
			write_loop_library(user_x,j,i,splicerms,bump_score,knots,globalrms);

			if ( runlevel > standard) std::cout << "screen" << SS( start_splicerms )
			 << SS( min_splicerms ) << SS( splicerms ) << SS( loop_name(i,j) ) <<
       SS( vdw_score ) << SS( globalrms ) << std::endl;
			if ( runlevel > inform ) output_trial_counters( std::cout );
		}
	}
	user_x.close();
	user_x.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin refine_loops
///
/// @brief
///car  refinement via wobble and chuck moves, centroid and full atom
///car assumes all arrays are initialized, position, best, low, low_sc
///car alreay occupied.
///
/// @detailed
///
/// @param  fast - [in/out]? -
/// @param  fix_natsc - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refine_loops(
	bool fast,
	bool fix_natsc
)
{

	using namespace misc;

//car local

	bool exist;
	int length,total_size;
	FArray1D_float splice_rms( 1000 );
	int num_loop;
//-----------------------------------------------

	set_allow_insert_by_looplength(total_residue+1,true,exist);
	if ( !exist ) return;

	std::cout << "refine loops..." << std::endl;

	//chu put set_fullatom before trim
	//set_fullatom_flag(true);
	set_use_nblist(true);

//car pack interface only, unless all template scs are fixed
	if ( !fix_natsc ) fullatom_set_packall(false);
	main_repack(true); // repack interface
	score_enable_rotamer_trials(false);
	scorefxn(); //chu update score info before monte_carlo_reset
	score_enable_rotamer_trials(true);
	monte_carlo_reset();

	score_set_loop_weight(20.0);
	minimize_set_local_min(false,total_residue);
	set_smallmove_size(2.0,2.0,2.0);
	minimize_exclude_sstype(true,false);
	score_set_try_rotamers(true);
	insert_map_get_max_contig_size(length,total_size);

	int nloop = 6;
	int cycles = std::min(200,10*total_size);
	if (fast) cycles = std::min(50,total_size/4);  //cdy reduce even further?

	float init_temp = 1.5;
	float final_temp = 0.5;
	float gamma = std::pow( (final_temp/init_temp), (1.0f/(nloop*cycles)) );
	float temperature = init_temp;
	monte_carlo_set_temp(temperature);


	std::cout << "-----------------------------------------------------" << std::endl;
	std::cout << "-----------------------------------------------------" << std::endl;
	std::cout << "dfpmin moves..." << std::endl;
	set_smallmove_size(2.0,2.0,2.0);
	recover_LOW(score12);
	logfile_output(0);

	main_minimize_trial(scorefxn,"dfpmin");
	logfile_output(0);

	reset_trial_counters();
	for ( int jk = 3; jk <= nloop; ++jk ) {
		float loop_weight = jk*jk*jk*jk/10.0;
		score_set_loop_weight(loop_weight);
		recover_LOW(score12);
		for ( int j = 1; j <= cycles; ++j ) {
			temperature *= gamma;
			monte_carlo_set_temp(temperature);
			main_small_min_trial(1,scorefxn,j*jk,"dfpmin");
			main_shear_min_trial(1,scorefxn,j*jk,"dfpmin");
		}
		loop_get_all_splicerms(splice_rms,num_loop);
		std::cout << "splice_rms ";
		for ( int j = 1; j <= num_loop; ++j ) {
			std::cout << ' ' << F( 6, 2, splice_rms(j) );
		} std::cout << std::endl;
		logfile_output(jk);
	}                     // jk

	output_trial_counters( std::cout );

	recover_LOW(score12);// recover before restoring the trimmed region

}

////////////////////////////////////////////////////////////////////////////////
/// @begin idealize_chainbreaks
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// note that idealize_chainbreak leaves refolding set up for
/// a complete idealized structure with no breaks, but loops are still
/// defined and the loop_flag is ON!!!  This is the desired setup
/// to end a run through output_decoy, but probalby not if another
/// protocol is called.
///
/// @references
///
/// @authors  car
///
/// @last_modified  10.1.2004
/////////////////////////////////////////////////////////////////////////////////
void
idealize_chainbreaks()
{
	using namespace files_paths;
	using namespace loops_ns;
	using namespace misc;

//car local
	int save_num_loop,first,last;
	float fret;
	bool gfrag( true );
	std::string filename;

	recover_LOW(score3);
	retrieve_decoy_name(filename);
	scorefile_output( filename, "pre_idealize", output_fa );
	minimize_set_func(3);
	minimize_exclude_sstype(false,false);
	set_use_nblist(false); // cannot tether with nblist on

//car setup weights for idealization
	tether_reset();
	scorefxn_set_tether_weight(1.0); // tether coords
	scorefxn_set_torsion_weight(1.0,2.0); // tether phipsi,omega

//car modify weights at loop junctions
	for ( int i = 1; i <= num_loop; ++i ) {
		if ( loop_begin(i) == 1 ) goto L100;
		if ( loop_end(i) == total_residue ) goto L100;
		if ( loop_get_refold_dir(i) == c2n ) {
			first = std::max(1,loop_begin(i)-3);
			last = std::min(loop_begin(i)+5, total_residue);
		} else {
			first = std::max(1,loop_end(i)-5);
			last = std::min(loop_end(i)+3, total_residue);
		}
		for ( int j = first; j <= last; ++j ) {
			tether_set_res_weight(i,0.2);
		}

		first = loop_begin(i);
		last = loop_end(i);
		for ( int j = first; j <= last; ++j ) {
			tether_set_angle_res_weight(i,0.5);
			tether_set_res_weight(i,0.5);
		}
L100:;
	}
//car modify weights in loop regions
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( secstruct(i) == 'L' )  tether_set_angle_res_weight(i,0.3);
	}

//car exit loop mode
	save_num_loop = num_loop;
	set_loop_flag(false);
	idealized_structure = true;
	initialize_maps();
	minimize_reset_exclude_list();

//car idealize
	idealize_position(false,fret,gfrag,true); // global idealization

	set_loop_flag(true); // turn on again for output formating
	num_loop = save_num_loop;
	reset_loops();
	// but inform refold that there is only 1 segment
	refold_set_segments(1,1,1,total_residue,0);
		//and all residues are insertable
	bool exist;
	set_allow_insert_by_region(1,total_residue,true,exist);


	if ( gfrag ) {
		save_status_info("idl_ch_breaks",0,0);
		monte_carlo_reset();
	} else {
		std::cout << "idealization FAILED!!" << std::endl;
	}

//car reset idealization weights
	tether_reset();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score0L
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score0L()
{
	using namespace scorefxns;

	score_reset_weights();

	vdw_weight = 1.0;
	pc_weight = 1.0;
	splicemsd_weight = loop_weight;

	pc_mode = 4;

	return scorefxn();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score3L
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score3L()
{
//car AUTOKSYNC relies on this scorefxn for -loops -permute
//car vdw,env,pair,ss,hs,sheet,pc,rg,rsigma
//car pc w/ low weight
//car loop closure
//car score3 has hb weights, disulfides

	using namespace scorefxns;

	score_reset_weights();
	vdw_weight = 1.0;
	env_weight = 1.0;
	pair_weight = 1.0;
	cb_weight = 1.0;
	sheet_weight = 1.0;
	ss_weight = 1.0;
	hs_weight = 1.0;
	rsigma_weight = 1.0;
	rg_weight = 1.0;
	pc_weight = 1.0;
	dipolar_weight = 1.0;
	param_pack::pack_wts.set_Whb_srbb(0.2);
	param_pack::pack_wts.set_Whb_lrbb(0.2);
	ramachandran_weight = 0.1;
	splicemsd_weight = loop_weight;

	pc_mode = cst_mode;
	ss_lowstrand = 0;
	ss_cutoff = 6;

	return scorefxn();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score4L
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score4L()
{
//car centroid function suitable for minimization

	using namespace scorefxns;

	score_reset_weights();
	vdw_weight = 1.0;
	env_weight = 1.0;
	pair_weight = 1.0;
	cb_weight = 1.0;
	pc_weight = 1.0;
	dipolar_weight = 1.0;
	param_pack::pack_wts.set_Whb_srbb(1.0);
	param_pack::pack_wts.set_Whb_lrbb(1.0);
	ramachandran_weight = 0.1;
	splicemsd_weight = loop_weight;

	pc_mode = cst_mode;
	ss_lowstrand = 0;
	ss_cutoff = 6;

	return scorefxn();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score4Lc
///
/// @brief score4L with loop contact score
///
/// @detailed all the score weights are exactly the same as in score4L. The only
///           difference is the incorporation of a contact score that rewards a
///           long loop if it makes contacts with the main body of the protein.
///           The cutoff is set at 5A. The actual calculation for this term can
///           be found in antibody_modeler. Antibody_modeler is the only mode that
///           presently uses this score
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks Developed when one found that 25 residue linkers for Fv  region were
///          just floating all over the place and not lying down in hydrophobic patches
///          in the protein surface facing the loop
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 01/31/2007
/////////////////////////////////////////////////////////////////////////////////
float
score4Lc()
{
//car centroid function suitable for minimization

	using namespace scorefxns;

	loop_scorefxn = true;

	score_reset_weights();
	vdw_weight = 1.0;
	env_weight = 1.0;
	pair_weight = 1.0;
	cb_weight = 1.0;
	pc_weight = 1.0;
	dipolar_weight = 1.0;
	param_pack::pack_wts.set_Whb_srbb(1.0);
	param_pack::pack_wts.set_Whb_lrbb(1.0);
	ramachandran_weight = 0.1;
	splicemsd_weight = loop_weight;

	pc_mode = cst_mode;
	ss_lowstrand = 0;
	ss_cutoff = 6;

	loop_contact_weight = 2; // aroop: contact weight defined in antibody_modeler

	return scorefxn();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin score9L
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///  mixed centroid-fullatom function
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score9L()
{

	using namespace scorefxns;

	score_reset_weights();
	fa_scorefxn = true;

	vdw_weight = 1.0;
	env_weight = 1.0;
	pair_weight = 1.0;
	cb_weight = 1.0;
	sheet_weight = 1.0;
	ss_weight = 1.0;
	hs_weight = 1.0;
	rsigma_weight = 1.0;
	rg_weight = 3.0;
	pc_weight = 1.0;
	dipolar_weight = 1.0;
	splicemsd_weight = loop_weight;

	pc_mode = cst_mode;
	ss_lowstrand = 1;
	ss_cutoff = 6;

	param_pack::pack_wts.set_Whb_srlrsc( 0.2 + lj_weight, 0.2 + lj_weight, lj_weight);
	ramachandran_weight = 0.1;

	fa_atr_weight = lj_weight;
	fa_rep_weight = lj_weight * rep_reduce;
	fa_dun_weight = lj_weight;

	return scorefxn();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score10L
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score10L()
{
//car intended for full atom optimization; similar to score12

	using namespace scorefxns;

	score_reset_weights();
	fa_scorefxn = true;

	splicemsd_weight = loop_weight;
	pc_weight = cst_weight;
	dipolar_weight = dpl_weight;
	ramachandran_weight = 0.05;

	param_pack::pack_wts.set_Whb_srlrsc(1.0, 1.0, 1.0);

	fa_atr_weight = 1.0;
	fa_rep_weight = 1.0 * rep_reduce;
	fa_dun_weight = 1.0;
  fa_elec_weight = 1.0;

	fa_pair_weight = 1.0;
	fa_solv_weight  = 1.0;
	fa_ref_weight = 1.0;

	fa_h2o_weight = 1.0;

	disulf_cendist_weight = 5.0;
	disulf_bb_dih_weight = 3.0;
	disulf_cbdist_weight = 5.0;
	disulf_cacbcb_weight = 1.0;
	disulf_cacbcbca_weight = 3.0;

	return scorefxn();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score11L
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score11L()
{
	using namespace scorefxns;

//car fullatom scoring function for selecting alignments
//car AUTOKSYNC relies on this scorefxn

	score_reset_weights();
	fa_scorefxn = true;
	use_mc_prescore = false;

	splicemsd_weight = loop_weight;
// preferred form as of 2003-04-03   dylan autoksync rescore5
	env_weight = 1.0;
	pair_weight = 0.0;

	param_pack::pack_wts.set_Whb_srlrsc(1.0, 1.0, 1.0);
	fa_atr_weight = 1.0;
	fa_rep_weight = 0.1;
	fa_dun_weight = 1.0;
	fa_pair_weight = 1.0;
	fa_solv_weight = 1.0;
	fa_ref_weight = 1.0;
  fa_elec_weight = 1.0;

	return scorefxn();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score12L
///
/// @brief
///car fullatom scoring function for final scoring of loops
///car SCOREFXN for model compleetion in robetta
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score12L()
{
	using namespace scorefxns;

	score_reset_weights();
	fa_scorefxn = true;
	use_mc_prescore = false;

	splicemsd_weight = loop_weight;
	ramachandran_weight = 0.1;

	fa_atr_weight = 1.0;
	fa_rep_weight = 0.1;
	vdw_weight = 1.0;

	param_pack::pack_wts.set_Whb_srlrsc(1.0, 1.0, 1.0);
	fa_dun_weight = 1.0;
	fa_pair_weight = 1.0;
	fa_solv_weight = 1.0;
	fa_ref_weight = 1.0;
	ex_gsolt_weight = 2.0;
  fa_elec_weight = 1.0;

	return scorefxn();
}
///////////////////////////////////////////////////////////////////////////////
void
fold_loops_with_dunbrack_closure()
{
	using namespace loops_ns;
	using namespace misc;
	using namespace runlevel_ns;

	if ( loop_int::num_loop != 1 ) {
		std::cout << "in fold_loops_with_dunbrack_closure" << std::endl;
		std::cout << "it currently does not work for multiple loops" << std::endl;
		std::cout << "exit ..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	const int loop_num = { 1 }; // this should be changed if with multi loops

	const int cutoff9 = { 16 }; // no 9 mer insertion if less than it
	const int cutoff3 = { 6 }; // no 3 mer insertion if less than it
	const int cutoff1 = { 3 }; // no 1 mer insertion if less than it

	// centroid mode frag_insert, dunbrack closure and minimization
	perturb_loop_with_dunbrack_closure( cutoff9, cutoff3, cutoff1, fast,
																			loop_num );
	// fullatom
	set_fullatom_flag( true );

	if ( loops_ns::loop_bool::trim ) trim_template();
	// repacking, small/shear min
	refine_loops_with_dunbrack_closure( fast, fix_natsc );

	if ( loops_ns::loop_bool::trim ) trim_template_reset();

	main_repack_trial( score12, 0 /*cycle_number*/ );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin perturb_loop
///
/// @brief generate variety of loop conformations by frag insertion
///
/// @detailed
///
/// @param[in]   cutoff1 - in - no 1mer insertions in regions shorter than this
/// @param[in]   cutoff3 - in - no 3mer insertions in regions shorter than this
/// @param[in]   fast - in - run fast protocol?
/// @param[in]   loop_num -in - loop to operate on, if invalid, use all
///
/// @global_read
///
/// @global_write
///
/// @remarks
///  insert frags, close loops and minimize
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
perturb_loop_with_dunbrack_closure(
	int cutoff9,
	int cutoff3,
	int cutoff1,
	bool fast,
	int loop_num
)
{
	using namespace misc;
	using namespace runlevel_ns;

	using namespace mc_global_track::mc_score; // yab: misc removal

	//car local
	bool exist;
	int length,total_size;
	int num_loop;
	FArray1D_float splice_rms(1000);

	if ( loopnum_is_valid(loop_num) ) {
		set_allow_insert_by_loopnum(loop_num,true,exist);
	} else {
		set_allow_insert_by_looplength(total_residue,true,exist);
	}

	if ( !exist ) return;

	insert_map_get_max_contig_size(length,total_size);

	if ( length < cutoff1 ) {
		std::cout << "WARNING: in perturb_loop_with_dunbrack_closure" << std::endl;
		std::cout << "too short to insert a fragment, " << "length: " << length
							<< "; cutoff: " << cutoff1 << std::endl;
		return; // too short to insert a fragment
	}

	std::cout << "perturb loops in centroid with dunbrack_closure" << std::endl;

	//chutmp set_use_nblist(true);
	recover_LOW(score4L);
	int nloop =6; //starting from 3
	int cycles = std::min(1000,total_size*20);
	if ( fast ) {
		nloop = 5;
		cycles = std::min(400,total_size*5);
	} else if ( benchmark ) {
		nloop = 3;
		cycles = std::min(30, total_size*2);
	}
	float init_temp = 2.0;
	float final_temp = 1.0;
	float gamma = std::pow( (final_temp/init_temp), (1.0f/(nloop*cycles)) );
	float temperature = init_temp;
	monte_carlo_set_temp(temperature);

	score_set_loop_weight( 20.0 ); // keep a modest loop closure weight

	reset_trial_counters();
	for ( int jk = 3; jk <= nloop; ++jk ) {

		// float loop_weight;
// 		loop_weight = jk*jk*jk*jk/10.0;
// 		score_set_loop_weight(loop_weight);

		recover_LOW(score4L);

		//$$$ diagnostics
		if ( benchmark || runlevel >= inform ) {
			loop_get_all_splicerms(splice_rms,num_loop);
			std::cout << "jk= " << jk << " score= " << SS( score ) << std::endl;
			std::cout << "weight " << F( 6, 1, 20.0) << " splice_rms ";
			for ( int j = 1; j <= num_loop; ++j ) {
				std::cout << ' ' << F( 6, 2, splice_rms(j) );
			} std::cout << std::endl;
		}
		//$$$ end diagnostics

		for ( int j = 1; j <= cycles; ++j ) {
			temperature *= gamma;
			monte_carlo_set_temp(temperature);
//chu try frags
			// 9 mer if possible
			if ( length >= cutoff9 ) {
				if ( !loopnum_is_valid(loop_num) ) {
					set_allow_insert_by_looplength(cutoff9-1,false,exist);
				}
				if ( exist ) main_frag_min_trial(9,score4L,j*jk,"limin");
			}
			// 3 mer if possible
			if ( length >= cutoff3 ) {
				if ( !loopnum_is_valid(loop_num) ) {
					set_allow_insert_by_looplength(cutoff3-1,false,exist);
				}
				if ( exist ) main_frag_min_trial(3,score4L,j*jk,"linmin");
			}
			// 1 mer if possible
			{
				if ( !loopnum_is_valid(loop_num) ) {
					set_allow_insert_by_looplength(cutoff1-1,false,exist);
				}
				if ( exist ) main_frag_min_trial(1,score4L,j*jk,"linmin");
			}
		} // inner loop with fixed temperature
	} // outer loop

	output_trial_counters(std::cout);
	recover_LOW(score4L);

}
///////////////////////////////////////////////////////////////////////////////
void
refine_loops_with_dunbrack_closure(
	bool fast,
	bool fix_natsc
)
{

	using namespace misc;
	using namespace scorefxns;
	using namespace runlevel_ns;

	using namespace mc_global_track::mc_score; // yab: misc removal

//car local
	bool exist;
	int length,total_size;
	FArray1D_float splice_rms( 1000 );
	int num_loop;
//-----------------------------------------------
	set_allow_insert_by_looplength(total_residue+1,true,exist);
	if ( !exist ) return;

	std::cout << "refine loops in fullatom with dunbrack_closure" << std::endl;

	//chu put set_fullatom before trim
	//set_fullatom_flag(true);
	//set_use_nblist(true);

//car pack interface only, unless all template scs are fixed
	if ( !fix_natsc ) fullatom_set_packall(false);
	main_repack(true); // repack interface
	score_enable_rotamer_trials(false);
	scorefxn(); //chu update score info before monte_carlo_reset
	score_enable_rotamer_trials(true);
	monte_carlo_reset();

	score_set_loop_weight(20.0);
	minimize_set_local_min(false,total_residue);
	set_smallmove_size(2.0,2.0,2.0);
	minimize_exclude_sstype(true,false);
	score_set_try_rotamers(true);
	insert_map_get_max_contig_size(length,total_size);

	int nloop = 6; //start from 4
	int cycles = std::min(200,10*total_size);
	if ( fast ) cycles = total_size*2;
	if ( benchmark ) {
		nloop = 4;
		cycles = total_size;
	}
	float init_temp = 1.5;
	float final_temp = 0.5;
	float gamma = std::pow( (final_temp/init_temp), (1.0f/(nloop*cycles)) );
	float temperature = init_temp;
	monte_carlo_set_temp(temperature);

	//std::cout << "dfpmin moves..." << std::endl;

	//recover_LOW(score12);
	//logfile_output(0);

	//chutmp this seems to blow up the structure
	//chutmp main_minimize_trial(scorefxn,"dfpmin");
	//chutmp logfile_output(0);

	reset_trial_counters();
	for ( int jk = 4; jk <= nloop; ++jk ) {
		float loop_weight = jk*jk*jk*jk/10.0;
		score_set_loop_weight(loop_weight);

		recover_LOW(score12);
		// diagnostics
		if ( benchmark || runlevel >= inform ) {
			loop_get_all_splicerms(splice_rms,num_loop);
			std::cout << "jk= " << jk << " score= " << SS(score) << std::endl;
			std::cout << "weight:" << F( 6, 1, loop_weight ) << " ; "
								<< " splice_rms:";
			for ( int j = 1; j <= num_loop; ++j ) {
				std::cout << ' ' << F( 6, 2, splice_rms(j) );
			} std::cout << std::endl;
			logfile_output(jk);
		}
		// end diagnostics

		for ( int j = 1; j <= cycles; ++j ) {
			temperature *= gamma;
			monte_carlo_set_temp(temperature);
			//main_wobble_min_trial(1,-60.,scorefxn,j*jk,1,-1,"dfpmin");
			main_small_min_trial(1,scorefxn,j*jk,"dfpmin");
			main_shear_min_trial(1,scorefxn,j*jk,"dfpmin");
		}
	} // jk

	output_trial_counters( std::cout );

	recover_LOW(score12);// recover before restoring the trimmed region

}
