// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 25807 $
//  $Date: 2008-11-21 02:40:02 +0200 (Fri, 21 Nov 2008) $
//  $Author: yab $


// Rosetta Headers
#include "files_paths.h"				// For frag_sizes
#include "fragments_ns.h"
#include "param.h"

// ObjexxFCL Headers
#include <ObjexxFCL/Dimension.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/StaticIndexRange.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <iostream>


namespace fragments {

using namespace param;

int fragments_nres( 0 );


namespace neigh_names {
	FArray3D_string align_name( MAX_RES(), MAX_NEIGH(), n_frag_sizes(), std::string( 4, ' ' ) );
	FArray3D_char align_chain( MAX_RES(), MAX_NEIGH(), n_frag_sizes(), ' ' );
	FArray4D_char align_res_id( MAX_RES(), MAX_NEIGH(), DRange( 0, MAX_LEN() ), n_frag_sizes(), ' ' );
	FArray3D_int align_resseq( MAX_RES(), MAX_NEIGH(), n_frag_sizes() );
}


//cems says- note mentally ill order of array subscripts


namespace neighbors {

	//These maps can store pointers to arrays analogous to the "align_<angle>_<X>" arrays
	//used for 1,3 and 9mers, so that memory for new arrays can be assigned only when it's needed
	//
	//It would make sense to use these arrays for all fragment sizes, but I want to see how this operates
	//in the long run before building it into everyone's main protocol.
	//-Robert Dec 12, 2007
	std::map< int, FArray3D_float *> phi_alternate_fragment_sizes;
	std::map< int, FArray3D_float *> psi_alternate_fragment_sizes;
	std::map< int, FArray3D_float *> omega_alternate_fragment_sizes;

	FArray1D_bool fragsize_initialized(MAX_RES(), false);

	//Makes an FArray pointer, assigns memory to the Farray, then appends it to the pointer map based on fragment size -RV
	void initialize_alternate_fragment_size(int fragment_size){

		if ( !phi_alternate_fragment_sizes.count(fragment_size) ) {
			FArray3D_float * Phi_FArray3D_p;
			FArray3D_float * Psi_FArray3D_p;
			FArray3D_float * Omega_FArray3D_p;

			Phi_FArray3D_p = new FArray3D_float(MAX_RES(), MAX_NEIGH(), SRange(0, fragment_size));
			Psi_FArray3D_p = new FArray3D_float(MAX_RES(), MAX_NEIGH(), SRange(0, fragment_size));
			Omega_FArray3D_p = new FArray3D_float(MAX_RES(), MAX_NEIGH(), SRange(0, fragment_size));

			phi_alternate_fragment_sizes[fragment_size] = Phi_FArray3D_p;
			psi_alternate_fragment_sizes[fragment_size] = Psi_FArray3D_p;
			omega_alternate_fragment_sizes[fragment_size] = Omega_FArray3D_p;

			fragsize_initialized(fragment_size) = true;
		}
	}


	void reset_alternate_fragment_size( int fragment_size ) {
		std::map< int, FArray3D_float * >::iterator phi, psi, omega;

		phi = phi_alternate_fragment_sizes.find( fragment_size );
		psi = psi_alternate_fragment_sizes.find( fragment_size );
		omega = omega_alternate_fragment_sizes.find( fragment_size );

		if ( phi != phi_alternate_fragment_sizes.end() ) {
			*(phi->second) = 0.0f;
		}
		if ( psi != psi_alternate_fragment_sizes.end() ) {
			*(psi->second) = 0.0f;
		}
		if ( omega != omega_alternate_fragment_sizes.end() ) {
			*(omega->second) = 0.0f;
		}
	}


	void reset_all_alternate_fragment_sizes() {
		for ( std::map< int, FArray3D_float * >::iterator i = phi_alternate_fragment_sizes.begin(), ie = phi_alternate_fragment_sizes.end(); i != ie; ++i ) {
			*(i->second) = 0.0f;
		}

		for ( std::map< int, FArray3D_float * >::iterator i = psi_alternate_fragment_sizes.begin(), ie = psi_alternate_fragment_sizes.end(); i != ie; ++i ) {
			*(i->second) = 0.0f;
		}

		for ( std::map< int, FArray3D_float * >::iterator i = omega_alternate_fragment_sizes.begin(), ie = omega_alternate_fragment_sizes.end(); i != ie; ++i ) {
			*(i->second) = 0.0f;
		}

	}


	void delete_alternate_fragment_size( int fragment_size ) {
		std::map< int, FArray3D_float * >::iterator phi, psi, omega;

		phi = phi_alternate_fragment_sizes.find( fragment_size );
		psi = psi_alternate_fragment_sizes.find( fragment_size );
		omega = omega_alternate_fragment_sizes.find( fragment_size );

		if ( phi != phi_alternate_fragment_sizes.end() ) {
			delete phi->second;
			phi_alternate_fragment_sizes.erase( phi );
		}
		if ( psi != psi_alternate_fragment_sizes.end() ) {
			delete psi->second;
			psi_alternate_fragment_sizes.erase( psi );
		}
		if ( omega != omega_alternate_fragment_sizes.end() ) {
			delete omega->second;
			omega_alternate_fragment_sizes.erase( omega );
		}

		fragsize_initialized( fragment_size ) = false;
	}


	void delete_all_alternate_fragment_sizes() {
		for ( std::map< int, FArray3D_float * >::iterator i = phi_alternate_fragment_sizes.begin(), ie = phi_alternate_fragment_sizes.end(); i != ie; ++i ) {
			if ( i->second != NULL ) {
				delete i->second;
			}
		}

		for ( std::map< int, FArray3D_float * >::iterator i = psi_alternate_fragment_sizes.begin(), ie = psi_alternate_fragment_sizes.end(); i != ie; ++i ) {
			if ( i->second != NULL ) {
				delete i->second;
			}
		}

		for ( std::map< int, FArray3D_float * >::iterator i = omega_alternate_fragment_sizes.begin(), ie = omega_alternate_fragment_sizes.end(); i != ie; ++i ) {
			if ( i->second != NULL ) {
				delete i->second;
			}
		}

		phi_alternate_fragment_sizes.clear();
		psi_alternate_fragment_sizes.clear();
		omega_alternate_fragment_sizes.clear();

		fragsize_initialized = false;
	}


	//First takes a reference FArray out of the pointer map
	//and then returns a reference to the specified field
	//
	//There are enough ampersands and asterisks around here I figured I should spell that out -RV
 	float & align_phi_v(int residue, int neighbor, int fragment, int fragment_size){
		FArray3D_float & my_phi_FArray3D = *(phi_alternate_fragment_sizes[fragment_size]);

		return my_phi_FArray3D(residue,neighbor,fragment);
	}

	/// @brief Phi values for fragments of size 1.
	/// @warning You cannot access this directly -- use #align_phi().
	/// @see #align_phi().
	FArray3D_float static align_phi_1 (MAX_RES(), MAX_NEIGH(), SRange(0, 1));

	/// @brief Phi values for fragments of size 3.
	/// @warning You cannot access this directly -- use #align_phi().
	/// @see #align_phi().
	FArray3D_float static align_phi_3 (MAX_RES(), MAX_NEIGH(), SRange(0, 3));

	/// @brief Phi values for fragments of size 9.
	/// @warning You cannot access this directly -- use #align_phi().
	/// @see #align_phi().
	FArray3D_float static align_phi_9 (MAX_RES(), MAX_NEIGH(), SRange(0, 9));

	/// @detailed
	///
	/// Dispatch between the @c align_phi_* arrays of various fragment
	/// sizes.
	///
	/// This function uses the <em>function-as-array idiom</em> -- it
	/// returns an @e lvalue, which means its results can be assigned to.
	///
	/// @param[in] residue       The index of the residue in #full_coords.
	/// @param[in] neighbor      The index of the neighbor in #neighbors.
	/// @param[in] fragment      The index of the fragment.
	/// @param[in] fragment_bin  The index of the original fragment bin requested.
	///   This is the index into into #files_paths::frag_sizes, and is not
	///   meaningful outside that context.
	/// @return                  The alignment of the phi angle of the given
	///   residue.  <em>Note that this is modifiable</em>.
	///
	/// @warning This should work properly when either accessing a value from
	///   the @c align_phi pseudo-array, or modifying one, but it changes the
	///   behavior when assigning the FArray to an argument FArray (in this
	///   case @c #ObjexxFCL::FArray3Da_float), and should be carefully
	///   examined in cases where it is used that way.

	float & align_phi(int residue, int neighbor, int fragment, int fragment_bin)
	{
		using files_paths::frag_sizes;
		using std::cerr;
		using std::endl;

		int const fragment_size = frag_sizes(fragment_bin);

		switch (fragment_size)
		{
  		case 1:
				return align_phi_1(residue, neighbor, fragment);
  		case 3:
				return align_phi_3(residue, neighbor, fragment);
  		case 9:
				return align_phi_9(residue, neighbor, fragment);
  		default:
				if (!fragsize_initialized(fragment_size)){
					initialize_alternate_fragment_size(fragment_size);
				}
				return align_phi_v(residue, neighbor, fragment, fragment_size);

		}
	}

	//First takes a reference FArray out of the pointer map
	//and then returns a reference to the specified field
	//
	//There are enough ampersands and asterisks around here I figured I should spell that out -RV
	float & align_psi_v(int residue, int neighbor, int fragment, int fragment_size)
	{
		FArray3D_float & my_psi_FArray3D = *(psi_alternate_fragment_sizes[fragment_size]);

		return my_psi_FArray3D(residue,neighbor,fragment);
	}


	/// @brief Psi values for fragments of size 1.
	/// @warning You cannot access this directly -- use #align_psi().
	/// @see #align_psi().
	FArray3D_float static align_psi_1 (MAX_RES(), MAX_NEIGH(), SRange(0, 1));

	/// @brief Psi values for fragments of size 3.
	/// @warning You cannot access this directly -- use #align_psi().
	/// @see #align_psi().
	FArray3D_float static align_psi_3 (MAX_RES(), MAX_NEIGH(), SRange(0, 3));

	/// @brief Psi values for fragments of size 9.
	/// @warning You cannot access this directly -- use #align_psi().
	/// @see #align_psi().
	FArray3D_float static align_psi_9 (MAX_RES(), MAX_NEIGH(), SRange(0, 9));

	/// @detailed
	///
	/// Dispatch between the @c align_psi_* arrays of various fragment
	/// sizes.
	///
	/// This function uses the <em>function-as-array idiom</em> -- it
	/// returns an @e lvalue, which means its results can be assigned to.
	///
	/// @param[in] residue       The index of the residue in #full_coords.
	/// @param[in] neighbor      The index of the neighbor in #neighbors.
	/// @param[in] fragment      The index of the fragment.
	/// @param[in] fragment_bin  The index of the original fragment bin requested.
	///   This is the index into into #files_paths::frag_sizes, and is not
	///   meaningful outside that context.
	/// @return                  The alignment of the psi angle of the given
	///   residue.  <em>Note that this is modifiable</em>.
	///
	/// @warning This should work properly when either accessing a value from
	///   the @c align_phi pseudo-array, or modifying one, but it changes the
	///   behavior when assigning the FArray to an argument FArray (in this
	///   case @c #ObjexxFCL::FArray3Da_float), and should be carefully
	///   examined in cases where it is used that way.

	float & align_psi(int residue, int neighbor, int fragment, int fragment_bin)
	{
		using files_paths::frag_sizes;
		using std::cerr;
		using std::endl;

		int const fragment_size = frag_sizes(fragment_bin);

		switch (fragment_size)
		{
  		case 1:
				return align_psi_1(residue, neighbor, fragment);
  		case 3:
				return align_psi_3(residue, neighbor, fragment);
  		case 9:
				return align_psi_9(residue, neighbor, fragment);
  		default:
				if (!fragsize_initialized(fragment_size)){
					initialize_alternate_fragment_size(fragment_size);
				}
				return align_psi_v(residue, neighbor, fragment, fragment_size);

		}
	}

	//First takes a reference FArray out of the pointer map
	//and then returns a reference to the specified field
	//
	//There are enough ampersands and asterisks around here I figured I should spell that out -RV
	float & align_omega_v(int residue, int neighbor, int fragment, int fragment_size)
	{
		FArray3D_float & my_omega_FArray3D = *(omega_alternate_fragment_sizes[fragment_size]);

		return my_omega_FArray3D(residue,neighbor,fragment);
	}


	/// @brief Omega values for fragments of size 1.
	/// @warning You cannot access this directly -- use #align_omega().
	/// @see #align_omega().
	FArray3D_float static align_omega_1 (MAX_RES(), MAX_NEIGH(), SRange(0, 1));

	/// @brief Omega values for fragments of size 3.
	/// @warning You cannot access this directly -- use #align_omega().
	/// @see #align_omega().
	FArray3D_float static align_omega_3 (MAX_RES(), MAX_NEIGH(), SRange(0, 3));

	/// @brief Omega values for fragments of size 9.
	/// @warning You cannot access this directly -- use #align_omega().
	/// @see #align_omega().
	FArray3D_float static align_omega_9 (MAX_RES(), MAX_NEIGH(), SRange(0, 9));

	/// @detailed
	///
	/// Dispatch between the @c align_omega_* arrays of various fragment
	/// sizes.
	///
	/// This function uses the <em>function-as-array idiom</em> -- it
	/// returns an @e lvalue, which means its results can be assigned to.
	///
	/// @param[in] residue       The index of the residue in #full_coords.
	/// @param[in] neighbor      The index of the neighbor in #neighbors.
	/// @param[in] fragment      The index of the fragment.
	/// @param[in] fragment_bin  The index of the original fragment bin requested.
	///   This is the index into into #files_paths::frag_sizes, and is not
	///   meaningful outside that context.
	/// @return                  The alignment of the omega angle of the given
	///   residue.  <em>Note that this is modifiable</em>.
	///
	/// @warning This should work properly when either accessing a value from
	///   the @c align_phi pseudo-array, or modifying one, but it changes the
	///   behavior when assigning the FArray to an argument FArray (in this
	///   case @c #ObjexxFCL::FArray3Da_float), and should be carefully
	///   examined in cases where it is used that way.

	float & align_omega(int residue, int neighbor, int fragment, int fragment_bin)
	{
		using files_paths::frag_sizes;
		using std::cerr;
		using std::endl;

		int const fragment_size = frag_sizes(fragment_bin);


		switch (fragment_size)
		{
  		case 1:
				return align_omega_1(residue, neighbor, fragment);
  		case 3:
				return align_omega_3(residue, neighbor, fragment);
  		case 9:
				return align_omega_9(residue, neighbor, fragment);
  		default:
				if (!fragsize_initialized(fragment_size)){
					initialize_alternate_fragment_size(fragment_size);
				}
				return align_omega_v(residue, neighbor, fragment, fragment_size);

		}
	}

	FArray2D_int align_depth( MAX_RES(), n_frag_sizes(), 0 );
}


namespace neighbors_int {
	FArray4D_char ss_type( MAX_RES(), MAX_NEIGH(), DRange( 0, MAX_LEN() ),
	 n_frag_sizes(), ' ' );
}


namespace choose_frag_parm {
	int top_N_frags = { 25 };
  bool check_ss = { true };
}


namespace frag_pointer {
	FArray3D_int block_frag_pointer( MAX_NEIGH(), MAX_RES(), 3 );
	FArray2D_int block_depth( MAX_RES(), 3, 0 );
}


namespace frag_precompute {
	FArray1D_bool initialized( n_frag_sizes(), false );
}


namespace chuck {
	FArray5D_float frag_rot( 3, 3, MAX_NEIGH(), MAX_RES(), n_frag_sizes() );
	FArray4D_float frag_off( 3, MAX_NEIGH(), MAX_RES(), n_frag_sizes() );
}


void
reset_fragment_arrays_used_by_Vall()
{
	align_depth = 0;

	align_phi_1 = 0.0f;
	align_phi_3 = 0.0f;
	align_phi_9 = 0.0f;

	align_psi_1 = 0.0f;
	align_psi_3 = 0.0f;
	align_psi_9 = 0.0f;

	align_omega_1 = 0.0f;
	align_omega_3 = 0.0f;
	align_omega_9 = 0.0f;

	fragments::neighbors::delete_all_alternate_fragment_sizes();

	fragments::neighbors_int::ss_type = ' ';

	// need to reset MAX_LEN as well
	// the following will mark the Dimension as uninitialized and zero the value
	param::MAX_LEN().clear();
}

} // namespace fragments
