// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: sheffler $


// Rosetta Headers
#include "geometric_solvation.h"
#include "aaproperties_pack.h"
#include "count_pair.h"
#include "current_pose.h" //Make sure we're in atom tree minimize
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "namespace_fullatom_flag.h"
#include "geometric_solvation_ns.h"
#include "hbonds_geom.h"
#include "hbonds_ns.h"
#include "make_pdb.h"
#include "maps.h"
#include "misc.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pose.h"
#include "water_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <string>

// Numeric Headers
#include <numeric/xyzVector.hh>

//Utility Headers
#include <utility/basic_sys_util.hh>


//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
// Geometric solvation -- computing a penalty proportional to
// the log odds of seeing a water at the location of an occluding atom
//
// Derivatives for atom_tree_minimize added on Sep. 8 , 2007; talk to
//  johnk or rhiju to implement in other minimizes.
//
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

//Forward reference
float
occluded_water_hbond_penalty(
														 bool const is_donor,
														 hbonds::HBEvalType const hbond_eval_type, // which hbond score to use
														 FArray1Da_float occ_atm, // coors of potentially occluded atm
														 FArray1Da_float base_atm, // coors of base atm
														 FArray1Da_float water_pos, // coors of occluding atm, which will be replaced with water
														 bool const update_deriv,
														 FArray2DB_float & deriv = NODERIV2D
														 );


// move this to geometric_solvation_ns.cc?
namespace geom_sol_deriv {
	FArray3D_double F1_geom_sol( 3, param::MAX_ATOM(), param::MAX_RES(), 0.0f );
	FArray3D_double F2_geom_sol( 3, param::MAX_ATOM(), param::MAX_RES(), 0.0f );
}


//////////////////////////////////////////////////////////////////////////////
/// @begin fill_geo_sol_arrays
///
/// @brief
/// loop over all Hbond donors/acceptors, compute a solvation energy
///
/// @detailed
///
/// @param  mode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
fill_geo_sol_arrays(
	bool const fullatom,
  FArray1DB_int const & aa,
	FArray1DB_int const & aav,
	FArray3DB_float const & xyz,
	int const nres,
	FArray2DB_bool const & neighborlist,
	bool const update_deriv // = false
){

	if ( ! fullatom ) {
		std::cout << "geometric solvation requires fullatom mode" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( update_deriv ) {

		if ( !minimize_check_current_pose() || !minimize_get_current_pose().atom_tree() ) {
			std::cout << "Geometric solvation derivatives only implemented for atom_tree_minimize!" << std::endl;
			std::cout << " Wouldn't be hard to implement elsewhere ... talk to rhiju or johnk." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		geom_sol_deriv::F1_geom_sol = 0.0;
		geom_sol_deriv::F2_geom_sol = 0.0;
	}

	//rhiju In jk's original formulation, geometric solvation always replaced lk solvation.
	//rhiju It was always recomputed, never cached.
	//rhiju For some applications to RNA, though, I'd like to cache the scores, and actually
	//rhiju add in a little bit of the usual Lazaridus-Karplus solvation too.
	//rhiju Requires the assumption of pairwise independence!
	if ( !geometric_sol::we_are_caching_solvation_scores_and_assuming_pairwise_independent ){
		fullatom_energies::solenergy = 0.0;
		fullatom_energies::sol_pair  = 0.0;
	}

	//Not actually used if we are not caching...
	FArray2DB_bool const & pair_moved( retrieve_pair_moved() );

// loop over all pairs of neighboring residues
	int const xyz_size12 = xyz.size1() * xyz.size2();
	int const neighborlist_size1 = neighborlist.size1();

  bool const include_polar_bb = true;
  bool const include_occ_bb = true;
  bool const include_polar_sc = true;
  bool const include_occ_sc = true;

	float totE( 0.0 );

	for ( int res1 = 1; res1 <= nres; ++res1 ) {
		int const aa1 = aa(res1);
		int const aav1 = aav(res1);
		float const & xyz1( xyz(1,1,res1) );

		for ( int res2 = res1+1,
		 ln = neighborlist.index(res1,res2), lxyz = xyz.index(1,1,res2);
		 res2 <= nres; ++res2, ln+=neighborlist_size1, lxyz+=xyz_size12 ) {

			// jk According to bk (taken from hbonds.cc)....
//bk pair_moved indicates which residues have changed relative position,
//bk and therefore need new energies calculated.  This is tricky though, because
//bk hbonds are not truly pairwise additive because sidechain-backbone hbonds
//bk are not allowed in cases where backbone-backbone hbond is already present.
//bk So for right now all hbonds will be calculated to be on the safe side.
			if ( !geometric_sol::we_are_caching_solvation_scores_and_assuming_pairwise_independent ||
					 pair_moved(res1,res2) ) {

				if ( neighborlist[ ln ] ) { // neighborlist(res1,res2)

					int const aa2 = aa(res2);
					int const aav2 = aav(res2);
					float geo_solE = res_res_geometric_sol(res1,res2,aa1,aav1,aa2,aav2,xyz1,xyz[lxyz], // [lxyz]==(1,1,res2)
																							 include_polar_bb,include_occ_bb,include_polar_sc,include_occ_sc,
																								 update_deriv);
					geo_solE += res_res_geometric_sol(res2,res1,aa2,aav2,aa1,aav1,xyz[lxyz], // [lxyz]==(1,1,res2)
																						xyz1,include_polar_bb,include_occ_bb,include_polar_sc,include_occ_sc,
																						update_deriv);

					totE += geo_solE;

				// jk add this to the running count of the solvation term,
				// jk without adding anything to atrE or repE
					fill_solE_array(res1,res2,geo_solE);

				} //neighborlist
			} //pair_moved

		}
	}

	// Just a diagnostic.
	//	std::cout << "jk TOT "  << totE << std::endl;

	return;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin res_res_geometric_sol
///
/// @brief
/// loop over all Hbond donors/acceptors in this residue, compute the solvation energy
///
/// @detailed
///
///    DANGER: uses "hbchk", which assumes that "fill_hbond_arrays" has been called
///      since the last time the backbone moved
///
/// @param  mode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
float
res_res_geometric_sol(
	int const polar_res,
	int const occ_res,
	int const polar_aa,
	int const polar_aav,
	int const occ_aa,
	int const occ_aav,
	FArray2Da_float polar_coord,
	FArray2Da_float occ_coord,
	bool const include_polar_bb,
	bool const include_occ_bb,
	bool const include_polar_sc,
	bool const include_occ_sc,
	bool const update_deriv /* = false */
	){

	using namespace aaproperties_pack;
	using namespace hbonds;
	using namespace param_aa;
	using namespace param;
	using namespace water;
	using namespace geom_sol_deriv;

	polar_coord.dimension( 3, MAX_ATOM() );
	occ_coord.dimension( 3, MAX_ATOM() );

	// jk require that we're in fullatom mode
	assert( fullatom_flag::full_atom );

	// jk ensure this subroutine was called to actually do something
	assert( include_polar_bb || include_polar_sc );
	assert( include_occ_bb || include_occ_sc );

	// jk debug mode
	bool const debug( false );

	// no Hbonds longer than sqrt of this (the square)
	float const dist_cut2 = 27.;   // 5.2*5.2

	// the backbone NH and CO are donor and acceptor #1
	// skip these if sidechain-only desired, do only these if backbone-only desired
	int polar_start = 1;

	if ( ! include_polar_bb ) ++polar_start;
	int polar_don_end = 1;
	int polar_acc_end = 1;

	// proline doesn't have a donating backbone NH
	if ( polar_aa == aa_pro ) {
		polar_don_end = 0;
	}

	if ( include_polar_sc ) {
		polar_don_end = nH_polar(polar_aa,polar_aav);
		polar_acc_end = nacceptors(polar_aa,polar_aav);
	}

	// atoms 1-4 are backbone, so start at 5 if only considering backbone occluding atoms
	int occ_start = 1;
	if ( ! include_occ_bb ) occ_start = 5;
	int occ_end = 4;
	if ( include_occ_sc ) occ_end = nheavyatoms(occ_aa,occ_aav);

	// loop over all Hbond donors and acceptors
	float res_solE = 0.;  // running sum of solvation energy for this residue

	//rhiju yuck, derivatives
	FArray2D_float deriv( 3, 2, 0.0 );

	for ( int hnum = polar_start; hnum <= polar_don_end; ++hnum ) {

		// For all protein donors, water is the acceptor
		bool const is_donor = true;
		int const dhatm = Hpos_polar(hnum,polar_aa,polar_aav);
		int const datm = atom_base(dhatm,polar_aa,polar_aav);

		for ( int occ_atm = occ_start; occ_atm <= occ_end; ++occ_atm ) {

			// the base atom isn't allowed to occlude solvent
			if ( ( polar_res == occ_res ) && ( occ_atm == dhatm ) ) continue;

			// if a backbone donor participates in a backbone-backbone Hbond,
			// nothing is allowed to occlude solvent except a backbone acceptor
			if ( ( datm == 1 ) && hbond_set.BBhbToNH_exists(polar_res) && ( occ_atm != 4 ) ) continue;

			// if the distance is > 5.2 A, from the base atom, it doesn't occlude solvent
			float base_dis2;
			distance2_bk(polar_coord(1,datm),occ_coord(1,occ_atm),base_dis2);
			if ( base_dis2 > dist_cut2 ) continue;

			// if distance to base atom is greater than distance to donor, it doesn't occlude solvent
			float hdis2;
			distance2_bk(polar_coord(1,dhatm),occ_coord(1,occ_atm),hdis2);
			if ( hdis2 > base_dis2 ) continue;

			// For a backbone CO occluding a backbone NH, use the backbone-backbone (linear) geometry
			// to compute solvation penalty (only really matters for the CO acceptor's geometry, but
			// do it here as well for consistency)
			HBEvalType hbe = ( ( datm == 1 ) && ( occ_atm == 4 ) ) ?
				hbond_evaluation_type(datm,polar_aa,polar_aav, occ_atm,occ_aa,occ_aav) : hbe_SP3SC;

			// jk Compute the Hbond energy as if this was a water
			// jk Add the water Hbond energy to a running total sum, as well as to the residue sum
			deriv = 0.0f;
			float occ_solE = occluded_water_hbond_penalty(is_donor, hbe,
																										polar_coord(1,dhatm),polar_coord(1,datm),occ_coord(1,occ_atm),
																										update_deriv, deriv);

			res_solE += occ_solE;
			//							ds_bfactor(dhatm,polar_res) += occ_solE;

			if (update_deriv){
				for ( int k = 1; k <= 3; ++k ) {

					F1_geom_sol(k, dhatm,  polar_res) += deriv(k,1);
					F2_geom_sol(k, dhatm,  polar_res) += deriv(k,2);

					F1_geom_sol(k, occ_atm,  occ_res) += -1.0f * deriv(k,1);
					F2_geom_sol(k, occ_atm,  occ_res) += -1.0f * deriv(k,2);

				}
			}


			if ( debug && ( occ_solE > 0.0 ) ) {
				std::cout <<"jk DON res "<< aa_name1( polar_aa) << I(3,polar_res)<<
					" atom "<< atom_name( dhatm, polar_aa, polar_aav)<<" is occluded by occ_res " <<
					aa_name1( occ_aa )<< I(3, occ_res) <<
					" atom "<< atom_name( occ_atm, occ_aa, occ_aav ) <<
					"  (HBEvalType " <<  I(2,hbe) << ") " <<
					" with energy "<< F(8,3,occ_solE)<<"."<<std::endl;
			}
		}
	}

	for ( int anum = polar_start; anum <= polar_acc_end; ++anum ) {
		int const aatm = accpt_pos(anum,polar_aa,polar_aav);
		bool const is_donor = false;
		int const base_atm = atom_base(aatm,polar_aa,polar_aav);

		for ( int occ_atm = occ_start; occ_atm <= occ_end; ++occ_atm ) {

			// the base atom isn't allowed to occlude solvent
			if ( ( polar_res == occ_res ) && ( occ_atm == base_atm ) ) continue;

			// if a backbone acceptor participates in a backbone-backbone Hbond,
			// nothing is allowed to occlude solvent except a backbone donor
			if ( ( aatm == 4 ) && hbond_set.BBhbToO_exists(polar_res) && ( occ_atm != 1 || occ_aa == aa_pro) ) continue;

			// an atom directly bound to the acceptor isn't allowed to occlude solvent
			if ( polar_res == occ_res ) {
				bool occludes_solvent=true;
				for ( int nb = 1, nnb = nbonded_neighbors(aatm,polar_aa,polar_aav); nb <= nnb; ++nb ) {
					if ( bonded_neighbor(nb,aatm,polar_aa,polar_aav) == occ_atm ) {
						occludes_solvent=false; break;
					}
				}
				if (!occludes_solvent) continue;
			}

			// if the distance is > 5.2 A, from the acceptor, it doesn't occlude solvent
			float acc_dis2;
			distance2_bk(polar_coord(1,aatm),occ_coord(1,occ_atm),acc_dis2);
			if ( acc_dis2 > dist_cut2 ) continue;

			// if distance to base atom is greater than distance to donor, it doesn't occlude solvent
			float base_dis2;
			distance2_bk(polar_coord(1,base_atm),occ_coord(1,occ_atm),base_dis2);
			if ( acc_dis2 > base_dis2 ) continue;


			// For a backbone NH occluding a backbone CO, use the backbone-backbone (linear) geometry
			// to compute solvation penalty

			HBEvalType hbe = ( ( aatm == 4 ) && ( occ_atm == 1 ) && (occ_aa != aa_pro) )  // if backbone C=O to NH occludes water
				? hbond_evaluation_type(occ_atm, occ_aa, occ_aav, aatm, polar_aa, polar_aav)
				//				? hbond_evaluation_type(aatm,polar_aa,polar_aav, occ_atm,occ_aa,occ_aav)
				: hbond_evaluation_type_HOHdonor(aatm,polar_aa,polar_aav);


			// jk Compute the Hbond energy as if this was a water
			// jk Add the water Hbond energy to a running total sum, as well as to the residue sum
			deriv = 0.0f;
			float occ_solE = occluded_water_hbond_penalty(is_donor, hbe,
																										polar_coord(1,aatm),polar_coord(1,base_atm),occ_coord(1,occ_atm),
																										update_deriv, deriv);

			res_solE += occ_solE;
			//							ds_bfactor(aatm,polar_res) += occ_solE;

			if (update_deriv){
				for ( int k = 1; k <= 3; ++k ) {

					F1_geom_sol(k,    aatm,polar_res) += deriv(k,1);
					F2_geom_sol(k,    aatm,polar_res) += deriv(k,2);

					F1_geom_sol(k, occ_atm,  occ_res) += -1.0f * deriv(k,1);
					F2_geom_sol(k, occ_atm,  occ_res) += -1.0f * deriv(k,2);
				}
			}

			if ( debug && ( occ_solE > 0.0 ) ) {
				std::cout<<"jk ACC res "<< aa_name1( polar_aa ) << I(3, polar_res)<<
					" atom "<< atom_name(aatm, polar_aa, polar_aav)<<" is occluded by occ_res "<<
					aa_name1( occ_aa )<< I(3, occ_res) <<
					" atom "<< atom_name( occ_atm, occ_aa, occ_aav ) <<
					"  (HBEvalType " <<  I(2,hbe) << ") " <<
					" with energy "<< F(8,3,occ_solE)<<"."<<std::endl;
			}
		}
	}

	return res_solE;
}

//////////////////////////////////////////////////////////////////////////////
// Helper function for creating a mock H or O for the mock water.
// Assume it is in the same plane as that defined by the other
// atoms involved in the hydrogen bond.
// There are probably (better) examples of this function elsewhere in the code.
//
void
set_water_base_atm( FArray1Da_float base_pos,
										FArray1Da_float atom_pos,
										FArray1Da_float water_pos,
										FArray1Da_float water_base_pos,
										float const xH /*cos(theta)*/,
										float const bond_length) {

		numeric::xyzVector_float base_v( & base_pos(1) );
		numeric::xyzVector_float atom_v( & atom_pos(1) );
		numeric::xyzVector_float water_v( & water_pos(1) );

		numeric::xyzVector_float x,y,z,water_base_v, direction;

		//Define coordinate system.
		z = cross( (water_v - atom_v),  (atom_v - base_v) );
		z.normalize();
		y = water_v - atom_v;
		y.normalize();
		x = cross( y, z );

		// Plop the atom down
		direction = y * xH  + x * std::sqrt( 1 - (xH * xH) );
		water_base_v = water_v + bond_length * direction;

		for (int k = 1; k <= 3; k++) water_base_pos(k) = water_base_v(k);

}

//////////////////////////////////////////////////////////////////////////////
/// @begin geometric_sol_water_hbond_energy
///
/// @brief
///    Compute a water-protein Hbond energy. The water geometry is assumed ideal.
///
/// @detailed
///    Works by calling "hbond_compute_energy", making this a simplified version
///    of "hb_energy_deriv".
///
/// @param  mode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
occluded_water_hbond_penalty(
  bool const is_donor,
	hbonds::HBEvalType const hbond_eval_type,
	FArray1Da_float occ_atm, // coors of potentially occluded atm
	FArray1Da_float base_atm, // coors of base atm
	FArray1Da_float water_pos, // coors of occluding atm, which will be replaced with water
	bool const update_deriv,
	FArray2DB_float & deriv /* = NODERIV2D */
){

	using namespace geometric_sol;
	using namespace hbonds;
	using namespace param_pack;

	occ_atm.dimension( 3 );
	base_atm.dimension( 3 );
	water_pos.dimension( 3 );

	deriv = 0.0f;

	// jumpout criteria copied from hb_energy_deriv in hbonds.cc

	// Compute geometry
	float AHdis,xD,xH,energy( 0.0 );

	//Craziness... create an artifical atom to complete "water molecule".
	FArray1D_float water_base_atm( 3, 0.0 );
	static float const water_O_H_distance = 0.958;

	if ( is_donor ) {

		// water is the acceptor, give it perfect geometry
		xH = 1./3.;  // perfect geometry is cos( 180 - 109.5 degrees), which is 1/3

		// compute the distance to the accepting water
		float AHdis2;
		distance2_bk(occ_atm,water_pos,AHdis2);
		if ( AHdis2 > MAX_R2 ) return 0.;
		if ( AHdis2 < MIN_R2 ) return 0.;
		AHdis = std::sqrt(AHdis2);

		// find the cosine of the base-proton-water angle (xD)
		xD = get_water_cos(base_atm,occ_atm,water_pos);
		if ( xD < MIN_xD ) return 0.;
		if ( xD > MAX_xD ) return 0.;

		//rhiju, testing alternative calculation that will give derivative.
		if (update_deriv) {
			set_water_base_atm( base_atm, occ_atm, water_pos, water_base_atm, xH, water_O_H_distance );
			hb_energy_deriv(hbond_eval_type, base_atm, occ_atm,
											water_pos,water_base_atm,water_base_atm,
											energy,hbderiv_ABE_GO_NO_xH,deriv);
			//		std::cout << "DERIV ENERGY DONOR:  " << energy;
		}


	} else {

		// water is the donor, give it perfect geometry
		xD = 0.9999;

		// compute the distance to the accepting water proton
		// subtract the water's OH distance to get the AHdis,
		// since the distance computed was from the acceptor to the water oxygen
		// note: water proton lies on the line between the acceptor and the water oxygen
		distance_bk(occ_atm,water_pos,AHdis);
		AHdis -= water_O_H_distance; // water O-H distance
		float const AHdis2 = AHdis * AHdis;
		if ( AHdis2 > MAX_R2 ) return 0.;
		if ( AHdis2 < MIN_R2 ) return 0.;

		// find cosine of the base-acceptor-water_proton angle (xH)
		// note: this is the same as the base-acceptor-water_oxygen angle
		xH = get_water_cos(base_atm,occ_atm,water_pos);
		if ( xH < MIN_xH ) return 0.;
		if ( xH > MAX_xH ) return 0.;

		//rhiju, testing alternative calculation that will give derivative.

		if ( update_deriv ) {
			set_water_base_atm( base_atm, occ_atm, water_pos, water_base_atm, -xD, water_O_H_distance );
			hb_energy_deriv(hbond_eval_type, water_pos, water_base_atm,
											occ_atm,base_atm,base_atm,
											energy,hbderiv_ABE_GO_NO_xD,deriv);
			//		std::cout << "DERIV ENERGY ACCPT: " << energy;
		}


	}

	hbond_compute_energy(hbond_eval_type, AHdis, xD, xH, energy);

	//	std::cout << "  JK ENERGY: " << energy << std::endl;

	float sol_penalty = -energy;


	// JK THIS NEEDS TO BE FIT MORE RIGOROUSLY LATER...
	// Apply a scaling factor (effectively a weight), tying the weight of the
	// solvation term to the Hbond term (rather than to the LK weight)
	// Note: chose the bb-sc Hbond weight, because they're all about the same
	float const sol_weight = geometric_sol_weight * pack_wts.Whbond_bb_sc() / pack_wts.Wsol();
	sol_penalty *= sol_weight;

	if ( update_deriv ){
		deriv *= -1.0 * sol_weight;
		if ( !is_donor ) deriv *= -1.0; // had to flip reference frame to get HB energy and derivative.
	}

	// this is a penalty, don't return a negative number
	if (sol_penalty < 0.) {
		if ( update_deriv ) deriv = 0.0f;
		return 0.;
	}

	//	if (sol_penalty > 0. ) {
		//		std::cout << F(7,3,occ_atm(1)) << " "  << F(7,3,base_atm(1)) <<  " " << F(7,3,water_pos(1) ) << std::endl;
		//		std::cout << "[AHdis " << F(7,3,AHdis) << "; xD " << F(7,3,xD) << ";  xH " << F(7,3,xH) << ", e " << F(7,3,energy)  <<  "]" ;
	//	}

	return sol_penalty; // return a positive number (this is a penalty)

}



//////////////////////////////////////////////////////////////////////////////
/// @begin get_water_cos
///
/// @brief
///    Compute the cosine required for calling water Hbond energies
///
/// @detailed
///
/// @param  mode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
get_water_cos(
	FArray1Da_float atmA,
	FArray1Da_float atmB,
	FArray1Da_float atmC
){

	atmA.dimension( 3 );
	atmB.dimension( 3 );
	atmC.dimension( 3 );

// A->B unit vector
	FArray1D_float ABunit( 3 );
	ABunit(1) = atmB(1) - atmA(1);
	ABunit(2) = atmB(2) - atmA(2);
	ABunit(3) = atmB(3) - atmA(3);
	float const ABdis2 = ABunit(1) * ABunit(1) + ABunit(2) * ABunit(2) + ABunit(3) * ABunit(3);
	float const inv_ABdis = 1.0f / std::sqrt(ABdis2);
	ABunit(1) *= inv_ABdis;
	ABunit(2) *= inv_ABdis;
	ABunit(3) *= inv_ABdis;

// B->C unit vector
	FArray1D_float BCunit( 3 );
	BCunit(1) = atmC(1) - atmB(1);
	BCunit(2) = atmC(2) - atmB(2);
	BCunit(3) = atmC(3) - atmB(3);
	float const BCdis2 = BCunit(1) * BCunit(1) + BCunit(2) * BCunit(2) + BCunit(3) * BCunit(3);
	float const inv_BCdis = 1.0f / std::sqrt(BCdis2);
	BCunit(1) *= inv_BCdis;
	BCunit(2) *= inv_BCdis;
	BCunit(3) *= inv_BCdis;

	float const cosval = ABunit(1) * BCunit(1) + ABunit(2) * BCunit(2) + ABunit(3) * BCunit(3);

	return cosval;

}


////////////////////////////////////////////////////////////////////////////////
void
compute_atom_geom_sol_deriv( int const atomno1, int const rsd1,
														 numeric::xyzVector_float & F1,
														 numeric::xyzVector_float & F2
)
{
	using namespace geom_sol_deriv;

	float const Wsol = param_pack::pack_wts.Wsol();

	for (int k = 1; k <= 3; k++ ){
		F1(k) += Wsol * F1_geom_sol(k, atomno1, rsd1 );
		F2(k) += Wsol * F2_geom_sol(k, atomno1, rsd1 );
	}

}


