// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 10626 $
//  $Date: 2006-10-10 00:57:03 -0400 (Tue, 10 Oct 2006) $
//  $Author: chu $

// Rosetta Headers
#include "after_opts.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "fullatom.h"
#include "grid_docking.h"
#include "initialize.h"
#include "job_distributor.h"
#include "jumping_refold.h"
#include "jumping_util.h"
#include "misc.h"
#include "pack_fwd.h"
#include "param.h"
#include "pose.h"
#include "pose_design.h"
#include "pose_docking.h"
#include "pose_io.h"
#include "pose_param.h"
#include "runlevel.h"
#include "start.h"
#include "status.h"
#include "util_basic.h"
#include "util_vector.h"

// Numeric Headers
#include <numeric/xyzVector.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/byte.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/formatted.io.hh>

// Utility Headers
#include <utility/vector1.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <cstdlib>
#include <fstream>
#include <iostream>

namespace grid_dock {
	std::vector<GridSearchParam> grid_moves;
	std::vector<GridSearchParam> input_grid_moves;

	std::map< int, std::vector<float> >  grid_move_map; //store all moves
	bool init_progressive(false);	
	const unsigned int max_grid_moves( 6 );	
}

/////////////////////////////////////////////////////////////////////////////
/// grid docking
///    explore a set of positions defined by translation and rotation axes
///    (given in a .axes file)
///
/// two interface functions
/// grid_dock_output_all_pdbs:
///    dumps pdbs of all grid positions ans exits
///
/// progressive_grid_dock_move:
///    designed to be called with "nstruct" and move to new grid positions
///    at the start of each run
///
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
/// @begin grid_dock_output_all_pdbs
///
/// @brief dump all structs defined by grid coords
/////////////////////////////////////////////////////////////////////////////
void
grid_dock_output_all_pdbs(
	pose_ns::Pose & pose,
	int const rb_cutpoint
)
{

	using namespace files_paths;
	using namespace grid_dock;
	using namespace pose_ns;

	int const nres( pose.total_residue() );
	pose_docking_build_simple_tree( pose, nres, rb_cutpoint );
	Pose pose_save;
	pose_save  = pose;

	//tabulate moves
	build_grid_move_map();

	//cycle through all moves and write pdb
	for ( unsigned int ii = 0; ii < grid_move_map.size(); ii++) {


		std::vector<float> this_move_vector = grid_move_map[ii];
		std::string filename( files_paths::code );

		for ( unsigned int jj = 0; jj < this_move_vector.size(); jj++) {

			grid_search_basic_move( pose, jj, this_move_vector[jj] );
			filename.append( "_" + string_of(this_move_vector[jj]) );


		}

		filename.append( ".pdb" );
		pose.dump_pdb( filename );

		//reset to starting axes and position
		grid_moves = input_grid_moves;
		pose = pose_save;

	}

	exit(1);
	return;

}

/////////////////////////////////////////////////////////////////////////////
/// @begin progressive_grid_dock_move
///
/// @brief apply a single move from map and increment counter
/////////////////////////////////////////////////////////////////////////////
void
progressive_grid_dock_move_from_misc()
{

	//using namespace files_paths;
	using namespace grid_dock;
	using namespace misc;
	using namespace pose_ns;

	if ( total_residue == docking::part_end(1) ) {
		std::cout << "ERROR: calling grid docking with only 1 monomer" << std::endl;
	}

	pose_ns::Pose docking_pose;

	fullatom_nonideal_initialized_pose_from_misc( docking_pose );
	docking_pose.set_fullatom_flag(true);
	docking_pose.set_allow_bb_move(true);
	docking_pose.set_allow_chi_move(true);
	docking_pose.refold_sidechains_from_chi();
	set_pose_flag( true );
	pose_docking_build_simple_tree( docking_pose, total_residue, docking::part_end(1) );

	progressive_grid_dock_move(docking_pose);

	docking_pose.copy_to_misc();  // make sure complex is in misc
	assert( misc_in_sync( docking_pose ));

	//std::string filename( "xx_" + string_of(main_job_distributor::jd->get_curr_outnum()) + ".pdb");
	//docking_pose.dump_pdb( filename );

	set_pose_flag(false);

}

/////////////////////////////////////////////////////////////////////////////
/// @begin progressive_grid_dock_move
///
/// @brief apply a single move from map and increment counter
///        NOTE: pose_docking_build_simple_tree must already be set for the pose
/////////////////////////////////////////////////////////////////////////////
void
progressive_grid_dock_move(
	pose_ns::Pose & pose
)
{

	using namespace files_paths;
	using namespace grid_dock;
	using namespace pose_ns;

	//tabulate moves
	build_grid_move_map();

	//reset to starting axes
	grid_moves = input_grid_moves;
	
	unsigned int grid_structure_index = main_job_distributor::jd->get_curr_outnum() - 1;

	//if you reach the end ... start over
	if ( grid_structure_index >= grid_move_map.size() ) {
		grid_structure_index = grid_structure_index % grid_move_map.size();
	}

	//get this set of moves
	std::vector<float> this_move_vector = grid_move_map[grid_structure_index];

	for ( unsigned int ii = 0; ii < this_move_vector.size(); ii++) {
		grid_search_basic_move( pose, ii, this_move_vector[ii] );
	}

}

/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
/// @begin build_grid_move_map
///
/// @brief store map of all moves
/////////////////////////////////////////////////////////////////////////////
void
build_grid_move_map()
{

	using namespace grid_dock;
	using namespace files_paths;
	using namespace grid_dock;

	if ( init_progressive ) { return; }


	//get data from .axes file
	read_GridSearchParam();

	//get how many sets of moves are being tried
	if ( grid_moves.size() < 1 ){
		std::cout << "ERROR: calling grid docking without any defined moves" << std::cout;
		exit(1);
	}

	//get how many sets of moves are being tried
	if ( grid_moves.size() > max_grid_moves ){
		std::cout << "ERROR: calling grid docking with too many moves (more than " << max_grid_moves << ")" << std::cout;
		exit(1);
	}

	float min1, max1, step1, reset_perturb1, perturb1;
	float min2, max2, step2, reset_perturb2, perturb2;
	float min3, max3, step3, reset_perturb3, perturb3;
	float min4, max4, step4, reset_perturb4, perturb4;
	float min5, max5, step5, reset_perturb5, perturb5;
	float min6, max6, step6, reset_perturb6, perturb6;

	const int move_index1 = 0;
	const int move_index2 = 1;
	const int move_index3 = 2;
	const int move_index4 = 3;
	const int move_index5 = 4;
	const int move_index6 = 5;

	int conformation_index = 0;
	std::vector< float >  move_vector;

	//begin looping through all moves and putting them in the map
	GridSearchParam grid1 = grid_moves[move_index1];  //first element of vector
	min1 = grid1.get_min();
	max1 = grid1.get_max();
	step1 = grid1.get_step_size();
	reset_perturb1 = 0.0;;
	perturb1 = 0.0;

	for (float dim1 = min1; dim1 <= max1; dim1 += step1) {
	
		move_vector.clear();
		//
		move_vector.push_back( dim1 );
		grid_move_map[conformation_index] =  move_vector;

		if ( 1 == grid_moves.size() ){ conformation_index++; }
		else {

			GridSearchParam grid2 = grid_moves[move_index2];
			min2 = grid2.get_min();
			max2 = grid2.get_max();
			step2 = grid2.get_step_size();
			reset_perturb2 = 0.0;;
			perturb2 = 0.0;

			for (float dim2 = min2; dim2 <= max2; dim2 += step2) {

				//
				move_vector.push_back( dim2 );
				grid_move_map[conformation_index] = move_vector;

				if ( 2 == grid_moves.size() ){ conformation_index++; }
				else {

					GridSearchParam grid3 = grid_moves[move_index3];
					min3 = grid3.get_min();
					max3 = grid3.get_max();
					step3 = grid3.get_step_size();
					reset_perturb3 = 0.0;;
					perturb3 = 0.0;

					for (float dim3 = min3; dim3 <= max3; dim3 += step3) {

						//
						move_vector.push_back( dim3 );
						grid_move_map[conformation_index] = move_vector;

						if ( 3 == grid_moves.size() ){ conformation_index++; }
						else {

							GridSearchParam grid4 = grid_moves[move_index4];
							min4 = grid4.get_min();
							max4 = grid4.get_max();
							step4 = grid4.get_step_size();
							reset_perturb4 = 0.0;;
							perturb4 = 0.0;

							for (float dim4 = min4; dim4 <= max4; dim4 += step4) {

								//
								move_vector.push_back( dim4 );
								grid_move_map[conformation_index] =  move_vector;

								if ( 4 == grid_moves.size() ){ conformation_index++; }
								else {

									GridSearchParam grid5 = grid_moves[move_index5];
									min5 = grid5.get_min();
									max5 = grid5.get_max();
									step5 = grid5.get_step_size();
									reset_perturb5 = 0.0;;
									perturb5 = 0.0;

									for (float dim5 = min5; dim5 <= max5; dim5 += step5) {

										//
										move_vector.push_back( dim5 );
										grid_move_map[conformation_index] = move_vector;

										if ( 5 == grid_moves.size() ){ conformation_index++; }
										else {

											GridSearchParam grid6 = grid_moves[move_index6];
											min6 = grid6.get_min();
											max6 = grid6.get_max();
											step6 = grid6.get_step_size();
											reset_perturb6 = 0.0;;
											perturb6 = 0.0;

											for (float dim6 = min6; dim6 <= max6; dim6 += step6) {

												//
												move_vector.push_back( dim6 );
												grid_move_map[conformation_index] = move_vector;
												conformation_index++;

												move_vector.pop_back(); //6th element
											}
										}
										move_vector.pop_back(); //5th element
									}
								}
								move_vector.pop_back(); //4th element
							}
						}
						move_vector.pop_back(); //3th element
					}
				}
				move_vector.pop_back(); //2nd element
			}
		}
	}

	//for(int ii = 0; ii < conformation_index; ii++){
	//	std::vector<float> temp_vec = grid_move_map[ii];
	//	std::cout << "vector index: " << ii << ": ";
	//	for(int jj = 0; jj < temp_vec.size(); jj++){
	//		std::cout << "[" << jj << "] " << temp_vec[jj] << " ";
	//	}
	//	std::cout << std::endl;
	//}

	init_progressive = true;	

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin grid_search_basic_move
///
/// @brief one trans or rot move of size perturb and update required centers
///        and axes
///
//////////////////////////////////////////////////////////////////////////////
void
grid_search_basic_move(
	pose_ns::Pose & pose,
	int grid_move_index,
	float perturb
){

	using namespace grid_dock;
	using namespace pose_ns;

	int const dock_jump( 1 );
	pose_ns::Jump perturbed_jump( pose.get_jump( dock_jump ) );
	int n_moves = grid_moves.size();

	int const pos1( pose.fold_tree().get_jump_point()(1, dock_jump) );
	const FArray3D_float & Epos( pose.Eposition() );

	GridSearchParam this_move = grid_moves[grid_move_index];

	if ( true ){
		std::cout << "grid_search_basic_move: " << this_move.get_move_type() <<" by " << perturb << std::endl;
	}

	if( this_move.get_move_type() ==  "translation" ){
		perturbed_jump.translation_along_axis( Epos(1,1,pos1),this_move.get_axis(),perturb );

		//if move is a translation:
		//   translate rotation centers for other movements
		//if move is a rotation:
		//   rotate rotation centers
		//   also rotate rotation axes of "floating rotations"

		for( int ii = 0; ii < n_moves; ii++ ){
			GridSearchParam other_move = grid_moves[ii];
			if( other_move.get_move_type() ==  "rotation" ) {

				//tranlate rotation centers
				other_move.translate_rot_center( this_move.get_axis(),perturb );

				//update
				grid_moves[ii] = other_move;

			}
		}

	}else if ( this_move.get_move_type() ==  "rotation" ){

		perturbed_jump.rotation_by_axis( Epos(1,1,pos1),this_move.get_axis(),this_move.get_center(),perturb );

		for( int ii = 0; ii < n_moves; ii++ ){
			GridSearchParam other_move = grid_moves[ii];
			if( other_move.get_move_type() ==  "rotation" && grid_move_index != ii ) {

				//rotate other rotation centers
				other_move.rotate_center_by_axis( this_move.get_axis(),this_move.get_center(),perturb );

				if( !(other_move.is_fixed_axis()) ) {
					//rotate other rotation axis
					other_move.rotate_axis_by_axis( this_move.get_axis(),perturb );
				}

				//update
				grid_moves[ii] = other_move;
			}
		}
	}

	pose.set_jump(dock_jump, perturbed_jump);

}

//////////////////////////////////////////////////////////////////////////////
/// @begin read_GridSearchParam
///
/// @brief reads file defining grid moves
///
/// @detailed
///
/// sample file:
/// (note the 1 space indentations and center and "fixed axis"
/// are needed for rotation moves)
///
///number of motions: 3
///move type: translation
/// axis: 3.21 54.08 16.50 30.81 48.51 0.99
/// step size: 1
/// min displacement: -1
/// max displacement: 1
///move type: rotation
/// axis: 15.728 56.757 10.733 23.210 34.081 6.500
/// center: 15.099  56.647  24.910
/// fixed axis: true
/// step size: 45
/// min displacement: -90
/// max displacement: 90
///move type: rotation
/// axis: 31.451  24.739 -11.827 15.099  56.647  24.910
/// center: 16.549 48.454 14.536
/// fixed axis: false
/// step size: 90
/// min displacement: -180
/// max displacement: 180
///
//////////////////////////////////////////////////////////////////////////////
void
read_GridSearchParam() {
	using namespace files_paths;
	using namespace grid_dock;

	data_x.clear();

	//open file
	if ( true ) { data_x.open( start_path + start_file + ".axes" ); }

	if ( !data_x ) {
		std::cout << "Open failed for file: " << data_x.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	GridSearchParam gridsearch;
	std::string tag;
	int n_motions = 0;
	bool error_in_file = false;

	//number of motions: 3
	data_x >> bite( 18, tag ) >> skip(1) >> n_motions >> skip(1) >> skip;

	std::cout <<  "Grid Docking: reading " << n_motions << " motions" << std::endl;

	for (int ii = 1; ii <= n_motions; ii++){

		//move type: rotation
		std::string move_type;
		std::string move_flag;
		data_x >> bite( 10, tag ) >> skip(1) >> bite( 3, move_type ) >> skip;
		if(tag != "move type:" && ( move_type != "rot" || move_type != "tra") ){ error_in_file = true; }
		if(move_type == "rot"){ move_flag = "rotation"; }
		if(move_type == "tra"){ move_flag = "translation"; }
		gridsearch.set_move_type( move_flag );

		//axis: X1 Y1 Z1 X2 Y2 Z2
		double x1_axis(0.0), y1_axis(0.0), z1_axis(0.0), x2_axis(0.0), y2_axis(0.0), z2_axis(0.0);
		data_x >> bite( 6, tag ) >> skip(1) >> x1_axis >> skip(1) >> y1_axis >> skip(1) >> z1_axis
		 >> skip(1) >> x2_axis >> skip(1) >> y2_axis >> skip(1) >> z2_axis >> skip;
		if(tag != " axis:"){ error_in_file = true; }
		numeric::xyzVector_double axis(double(x1_axis - x2_axis), double(y1_axis - y2_axis),
		 double(z1_axis - z2_axis) );
		gridsearch.set_axis( axis );

		if(move_flag == "rotation"){
			//center: 1.0 3.0 5.0
			double x_center(0.0), y_center(0.0), z_center(0.0);
			data_x >> bite( 8, tag ) >> skip(1) >> x_center >> skip(1) >> y_center >> skip(1) >> z_center >> skip;
			if(tag != " center:"){ error_in_file = true; }
			numeric::xyzVector_double c_coord(x_center, y_center, z_center);
			gridsearch.set_center( c_coord );

			//fixed axis: true
			bool fixed_axis = true;
			std::string mv_axis_flag;
			data_x >> bite( 12, tag ) >> skip(1) >> bite(1, mv_axis_flag) >> skip;
			if(tag != " fixed axis:"){ error_in_file = true; }
			if(mv_axis_flag == "t"){ fixed_axis = true;  }
			if(mv_axis_flag == "f"){ fixed_axis = false;  }
			gridsearch.set_fixed_axis(fixed_axis);
		}

		//step size: 1.5
		float step_size(0.0);
		data_x >> bite( 11, tag ) >> skip(1) >> step_size >> skip;
		if(tag != " step size:"){ error_in_file = true; }
		gridsearch.set_step_size( step_size );

		//min displacement: -10
		float min_dsp(0.0);
		data_x >> bite( 18, tag ) >> skip(1) >> min_dsp >> skip;
		if(tag != " min displacement:"){ error_in_file = true; }
		gridsearch.set_min( min_dsp );

		//max displacement: 15
		float max_dsp(0.0);
		data_x >> bite( 18, tag ) >> skip(1) >> max_dsp >> skip;
		if(tag != " max displacement:"){ error_in_file = true; }
		gridsearch.set_max( max_dsp );

		//put into grid param stuff
		gridsearch.dump_info();

		// put param into grid_moves vector
		grid_moves.push_back(gridsearch);
		input_grid_moves.push_back(gridsearch);

		//make start to stop  == range * dispalcement
		float range_fit = ( max_dsp - min_dsp ) / step_size;
		if ( std::fmod(range_fit,1) ){
			std::cout << "Problem reading axes file: " << data_x.filename() << std::endl;
			std::cout << "range cannot be evenly split by step size:" << std::endl;
			std::cout << " min: " << min_dsp << " max: " << max_dsp << " step: " << step_size << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );
		}

	}

	if(error_in_file){
		std::cout << "Problem reading axes file: " << data_x.filename() << std::endl;
		std::cout << "format not correct" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );
	}

	return;
}


///////////////////////////////////////////////////////////////////////////////////
/// GridSearchParam
///
/// holds and maniplulates information for a grid move type
///////////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////
/// @begin GridSearchParam::set_fixed_axis
///
/// @brief set whether rotation axis should move with the
///        the molecule or stay fixed in the target molecule frame
///////////////////////////////////////////////////////
void
GridSearchParam::set_fixed_axis(
	bool at
) {
		if (move_type_ != "rotation") {
		std::cout << "Grid Docking Error: can only set axis type for rotations.";
		std::cout << "You are using: " << move_type_ << std::endl;
	}
	fixed_axis_ = at;
}

///////////////////////////////////////////////////////
/// @begin GridSearchParam::translate_rot_center
///////////////////////////////////////////////////////
void
GridSearchParam::translate_rot_center(
	numeric::xyzVector_double ext_axis,
	float translate_by
){

	float axis_distance = std::sqrt(ext_axis(1)*ext_axis(1) + ext_axis(2)*ext_axis(2)
	 + ext_axis(3)*ext_axis(3) );
	float scale = translate_by / axis_distance;

	std::cout << " Translating center by: " << translate_by << std::endl;
	std::cout << " start: " << center_(1) << " " << center_(2) << " " << center_(3) << std::endl;

	center_(1) += scale * ext_axis(1);
	center_(2) += scale * ext_axis(2);
	center_(3) += scale * ext_axis(3);

	std::cout << " end: " << center_(1) << " " << center_(2) << " " << center_(3) << std::endl;
}

///////////////////////////////////////////////////////
/// @begin GridSearchParam::rotate_axis_by_axis
///
/// @brief rotate a "floating" rotation axis with a given rotation
///
/// @detailed
/// This is for cases where a partner has a rotation axis tied to
/// its current orientation
///
/// rotation code attributed to Ronald Goldman:
/// http://local.wasp.uwa.edu.au/~pbourke/geometry/rotate/source.c
///////////////////////////////////////////////////////
void
GridSearchParam::rotate_axis_by_axis(
	numeric::xyzVector_double ext_axis,
	float theata
){

	float rad_theata = numeric::conversions::radians(theata);
	float cos_theta = std::cos (rad_theata);
	float sin_theta = std::sin (rad_theata);
	numeric::xyzVector_double old_axis = axis_;
	ext_axis.normalize();

	std::cout << " Rotating axis by: " << theata << " " << cos_theta << std::endl;
	std::cout << "  start: " << axis_(1) << " " << axis_(2) << " " << axis_(3) << std::endl;

	//ext_axis = external axis (axis you are rotation this one by)
	//old_axis = starting orientation of this axis

	axis_(1) = (cos_theta + (1 - cos_theta) * ext_axis(1) * ext_axis(1)) * old_axis(1)
	 + ((1 - cos_theta) * ext_axis(1) * ext_axis(2) - ext_axis(3) * sin_theta) * old_axis(2)
	 + ((1 - cos_theta) * ext_axis(1) * ext_axis(3) + ext_axis(2) * sin_theta) * old_axis(3);

	axis_(2) = ((1 - cos_theta) * ext_axis(1) * ext_axis(2) + ext_axis(3) * sin_theta) * old_axis(1)
	 + (cos_theta + (1 - cos_theta) * ext_axis(2) * ext_axis(2)) * old_axis(2)
	 + ((1 - cos_theta) * ext_axis(2) * ext_axis(3) - ext_axis(1) * sin_theta) * old_axis(3);

	axis_(3) = ((1 - cos_theta) * ext_axis(1) * ext_axis(3) - ext_axis(2) * sin_theta) * old_axis(1)
	 + ((1 - cos_theta) * ext_axis(2) * ext_axis(3) + ext_axis(1) * sin_theta) * old_axis(2)
	 + (cos_theta + (1 - cos_theta) * ext_axis(3) * ext_axis(3)) * old_axis(3);

	std::cout << "  end: " << axis_(1) << " " << axis_(2) << " " << axis_(3) << std::endl;

}

///////////////////////////////////////////////////////
/// @begin GridSearchParam::rotate_center_by_axis
///
/// @brief rotate a rotation center with by a given axis
///
/// @detailed
/// This allows a partner to carry its starting center with it through
/// multiple rotations
///
/// rotation code attributed to Ronald Goldman:
/// http://local.wasp.uwa.edu.au/~pbourke/geometry/rotate/source.c
///////////////////////////////////////////////////////
void
GridSearchParam::rotate_center_by_axis(
	numeric::xyzVector_double ext_axis,
	numeric::xyzVector_double ext_center,
	float theata
){

	float rad_theata = numeric::conversions::radians(theata);
	float cos_theta = std::cos (rad_theata);
	float sin_theta = std::sin (rad_theata);
	numeric::xyzVector_double old_center = center_;
	ext_axis.normalize();

	std::cout << " Rotating center by: " << theata << " " << cos_theta << std::endl;
	std::cout << "  ext: " << ext_center(1) << " " << ext_center(2) << " " << ext_center(3) << std::endl;
	std::cout << "  start: " << center_(1) << " " << center_(2) << " " << center_(3) << std::endl;

	// 1. translate 2nd center so 1st center is at origin
	old_center(1) = center_(1) - ext_center(1);
	old_center(2) = center_(2) - ext_center(2);
	old_center(3) = center_(3) - ext_center(3);

	// 2. rotate 2nd center
	// old_center = starting position for this center
	// center_ = output center
	// ext_axis = external axis

	center_(1) = (cos_theta + (1 - cos_theta) * ext_axis(1) * ext_axis(1)) * old_center(1)
	 + ((1 - cos_theta) * ext_axis(1) * ext_axis(2) - ext_axis(3) * sin_theta) * old_center(2)
	 + ((1 - cos_theta) * ext_axis(1) * ext_axis(3) + ext_axis(2) * sin_theta) * old_center(3);

	center_(2) = ((1 - cos_theta) * ext_axis(1) * ext_axis(2) + ext_axis(3) * sin_theta) * old_center(1)
	 + (cos_theta + (1 - cos_theta) * ext_axis(2) * ext_axis(2)) * old_center(2)
	 + ((1 - cos_theta) * ext_axis(2) * ext_axis(3) - ext_axis(1) * sin_theta) * old_center(3);

	center_(3) = ((1 - cos_theta) * ext_axis(1) * ext_axis(3) - ext_axis(2) * sin_theta) * old_center(1)
	 + ((1 - cos_theta) * ext_axis(2) * ext_axis(3) + ext_axis(1) * sin_theta) * old_center(2)
	 + (cos_theta + (1 - cos_theta) * ext_axis(3) * ext_axis(3)) * old_center(3);

	// 3. translate back
	center_(1) += ext_center(1);
	center_(2) += ext_center(2);
	center_(3) += ext_center(3);

	std::cout << "  end: " << center_(1) << " " << center_(2) << " " << center_(3) << std::endl;

}

///////////////////////////////////////////////////////
/// @begin GridSearchParam::dump_info
///////////////////////////////////////////////////////
void
GridSearchParam::dump_info() {

	std::cout << "GridSearchParam:" <<std::endl;
	std::cout << "  move type: "<< move_type_ << std::endl;
	std::cout << "  axis: "<< axis_(1) << " " << axis_(2) << " " << axis_(3) << std::endl;

	if (move_type_ == "rotation") {
		std::cout << "  center: "<< center_(1) << " " << center_(2) << " " << center_(3) << std::endl;
		std::cout << "  fixed axis: "<< fixed_axis_ << std::endl;
	}

	std::cout << "  step size: "<< step_size_ << std::endl;
	std::cout << "  min displacement: "<< min_displacement_ << std::endl;
	std::cout << "  max displacement: "<< max_displacement_ << std::endl;

}

