// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 11796 $
//  $Date: 2006-12-14 17:40:12 -0800 (Thu, 14 Dec 2006) $
//  $Author: snoeyink $

// Rosetta Headers
#include "atom_is_backbone.h"
#include "docking_ns.h"
#include "docking_minimize_ns.h"
#include "docking_minimize.h" // for accumulate_dE_dTR
#include "hbonds_geom.h"
#include "hbonds_ns.h"
#include "minimize_ns.h"
#include "runlevel.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <cmath>
#include <iostream>

//Utility Headers
#include <utility/basic_sys_util.hh>
using namespace hbonds;

////////////////////////////////////////////////////////////////////////////////
/// @begin classify_BB_by_separation
///
/// @brief
///  Returns the evaluation type of a backbone/backbone hydrogen bond ---
///  which geometric function is applied -- based on sequence separation.
/// @param  dres - [in] -
/// @param  ares - [in] -
/// @remarks
///  Used only for backbone atoms with residues on the same protein sequence.
///
/// @authors Jack Snoeyink Dec 2006
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
hbonds::HBEvalType
hbe_classify_BB_by_separation(int const don_res, int const acc_res) {
	HBEvalType hbe;

	// JSS switch statements are implemented by table lookup, so are better than
	// if-else for cases like this.  Do not forget the "break;" statements, however!

	switch (don_res-acc_res) {
		case 0: // This probably means that someone called this without specifying residues
					hbe =	hbe_NONE;
					if (runlevel_ns::benchmark || runlevel_ns::runlevel > runlevel_ns::standard)
						std::cout << "Warning: donor/acceptor on same residue " << acc_res
											<< ".  Probably someone forgot to specify ares,dres for hbond_evaluation_type" << std::endl;
					break;
		case -2: // short separation formerly disallowed by allow_bond; now permitted. jss 23 jan 07
		case -1:
		case  1:
		case  2:
		case -3: // hbe = hbe_BBMINUS3; break;
		case  3: hbe = hbe_BBTURN; break;
		case -4: // hbe = hbe_BBMINUS4; break; // Suggestion for later types; put types used into HBEvalType enum
		case  4: hbe = hbe_BBHELIX; break;
		default: hbe = hbe_BBOTHER; break;
	}
	return hbe;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin hbond_evaluation_type
///
/// @brief
///  This function returns the evaluation type of a hydrogen bond ---
///  which determines which geometric hbond scoring function is to be applied.
///
/// @detailed
/// The type of a hydrogen bond determine which polynomials or scoring functions are applied
/// An hbond scoring function uses parameters  r=AHdis, xD=theta, xH=psi (historical names)
/// a burial number, and chi and returns energy and optional derivative information.
/// don_res and acc_res may be omitted for non-BB bonds.
///
/// Two variants (hbond_evaluation_type_HOHacceptor & hbond_evaluation_type_HOHdonor)
/// allow bonds donated or accepted by HOH that are not actually in the file;
/// these are used by geometric solvation.
///
/// @param don_atomtype - [in] - donor atomtype from fullatom_type
/// @param acc_atomtype - [in] - donor atom
/// @param don_res - [in] - donor res #    - to refine backbone-backbone
/// @param acc_res - [in] - acceptor res # -   classification by separation
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

hbonds::HBEvalType
hbond_evaluation_type(
											int const don_atomtype,
											int const acc_atomtype,
											int const don_res,
											int const acc_res
											)
{
  using namespace hbonds;
	using namespace aaproperties_pack;
	HBEvalType hbe;

	hbe = HBeval_lookup(HBaccchemtype(acc_atomtype), HBdonchemtype(don_atomtype)); // reverses the usual don/acc convention. Sorry.
	if (hbe == hbe_BB) { // refine backbone/backbone bond types by separation
		hbe = hbe_classify_BB_by_separation(don_res, acc_res);
	}

	return hbe;
}

hbonds::HBEvalType
hbond_evaluation_type_AH( // temporary for rotamer trie - use H instead of heavy atom.
											int const don_atomtype,
											int const acc_atomtype,
											int const don_res,
											int const acc_res
											)
{
  using namespace hbonds;
	using namespace aaproperties_pack;
	HBEvalType hbe;
	int don_type;
	if (don_atomtype==25)
		don_type = hbdon_BBN;
	else if (don_atomtype == 22)
		don_type = hbdon_SC;
	else
		don_type = hbdon_NO;

	hbe = HBeval_lookup(HBaccchemtype(acc_atomtype), don_type); // reverses the usual don/acc convention. Sorry.
	if (hbe == hbe_BB) { // refine backbone/backbone bond types by separation
		hbe = hbe_classify_BB_by_separation(don_res, acc_res);
	}
	return hbe;
}

hbonds::HBEvalType
hbond_evaluation_type_HOHdonor(int const acc_atomtype)
{
  using namespace hbonds;
	using namespace aaproperties_pack;
	HBEvalType hbe;

	int const don_type = 26; // HOH - not in file, but postulated as ideal by geometric solvation
	hbe = HBeval_lookup(HBaccchemtype(acc_atomtype), HBdonchemtype(don_type)); // reverses the usual don/acc convention. Sorry.
	return hbe;
}

HBEvalType
hbond_evaluation_type_HOHacceptor(int const don_atomtype)
{
  using namespace hbonds;
	using namespace aaproperties_pack;
	HBEvalType hbe;

	int const acc_type = 26; // HOH - not in file, but postulated as ideal by geometric solvation
	hbe = HBeval_lookup(HBaccchemtype(acc_type), HBdonchemtype(don_atomtype)); // reverses the usual don/acc convention. Sorry.
	return hbe;
}

// Fade intervals for current hydrogen bond parameters
// These specify the ranges where the corresponding polynomial is linearly interpolated to zero
// Eventually, these will be attached to the polynomials.
static FadeInterval const fade_rBB(MIN_R, MIN_R, R_INTERP_EDGE, MAX_R); // used to adjust xD,xH
static FadeInterval const fade_rshort(MIN_R, MIN_R, INTERP_MIN, INTERP_MAX);
static FadeInterval const fade_rlong(INTERP_MIN, INTERP_MAX, INTERP_MAX, MAX_R);
static FadeInterval const fade_xD(MIN_xD, ANGLE_INTERP_EDGE, 1, 1); // xD=theta should fade r,xH sooner!
static FadeInterval const fade_xH(MIN_xH, ANGLE_INTERP_EDGE, 1, 1); // fades r,xD
// Since C++ does not yet have unnamed (lambda) functions in the standard,
// I use #define to set up named hbond scoring polynomials that are called
// as name(/*in*/ double x, /*out*/ double &value, double &deriv);
// The idea is to auto-generate the create_poly#() by matlab scripts that
// fit polynomials; if you enter coodinates by hand and have typos,
// the error message may be cryptic becaue of the #defines (but at least
// you'll get them -- the old fortran-style allowed disagreements between
// degree and # of coefficients without complaint.)
//
// Polynomial coefficients are listed from most to least significant:
// e.g., create_poly8(poly_name, xmin, xmax, coefs on x^7..x^0 );
// This is the same order as MATLAB so polynomials can be made by polyfit
// and displayed by x=xmin:0.01:xmax; plot(x,polyval(coeffs, x));
//
// Each score polynomial is clipped to an interval [xmin,xmax], which
// is currently important only to the distance polynomials.
// Outside the interval, the derivative = 0.0 and value = POLY_CLIPVAL;
// If you want the polynomial to always be non-positive, you can find the
// interval by combining roots(coeffs) information with MATLAB plots.
//
// Formerly there was a scale and offset applied at evaluation time;
// those must be built into the polynomial now (WYSIWYG),
// but this is easily done with the MATLAB polynomial fitting scripts.

// define polynomials for use in hbond score calculation
create_poly8(POLY_AHdisBBHelix, MIN_R, 2.8, // 1.78218633, 2.6757, // these are the roots to eliminate positive values.
             12.93768086,-221.0155722,1604.391304,-6409.335773,15200.86425,-21375.00216,16475.98811,-5361.55644)
create_poly8(POLY_AHdisBBOther, MIN_R, 2.745, // 1.6971523, 2.679339, // non-positive interval
             13.58980244,-224.0452428,1568.933094,-6044.257847,13820.1498,-18730.96076,13912.92238,-4361.995425)
create_poly5(POLY_AHdisSP2, MIN_R, 2.5, // 1.6941, 2.5, // non-positive interval
             10.98727738,-100.2401419,340.9733405,-511.6111233,285.0061262)
create_poly5(POLY_AHdisSP3, MIN_R, 2.5, // 1.755, 2.521385, // non-positive interval
             7.011735538,-68.99968829,251.820931,-403.3593133,238.7378958)
create_poly8(POLY_xDBBHelix, MIN_xD, MAX_xD, // 0.3746, 1.04,
             223.5268153,-757.7254095,1019.593508,-689.2232431,240.1436064,-37.84119583,0.85868904,0.278181985)
create_poly8(POLY_xDBBOther, MIN_xD, MAX_xD, // 0.76, 1.09,
             111.9877946,-380.3066184,514.7650204,-352.4092342,124.6219703,-19.94401946,0.149314979,0.635771774)
create_poly3(POLY_xDSP2short, MIN_xD, MAX_xD, // 0.7071, 1.01,
             -0.562582503,-0.746682668,0.809265171)
create_poly3(POLY_xDSP2long, MIN_xD, MAX_xD, // 0.0, 1.01,
             0.094962885,-0.254313172,0.0)
create_poly3(POLY_xDSP3short, MIN_xD, MAX_xD, // 0.61566, 1.01,
             -0.100140144,-1.139139041,0.739279186)
create_poly3(POLY_xDSP3long, MIN_xD, MAX_xD, // 0.0, 1.01,
             0.089380221,-0.207503776,0.0)
create_poly8(POLY_xHBBHelix, MIN_xH, MAX_xH, // 0.156, 1.03,
             54.80664331,-196.8196655,295.9418886,-232.105602,96.99124565,-20.60918361,1.573169816,0.000745458)
create_poly8(POLY_xHBBOther, MIN_xH, MAX_xH, // 0.61566, 1.07,
             43.94483847,-144.3836033,193.5865176,-132.4469355,47.28137288,-8.945888012,-0.227035135,0.791902995)
create_poly3(POLY_xHSP2short, MIN_xH, MAX_xH, // 0.0, 1.08,
             1.720984644,-1.855254573,0.0)
create_poly3(POLY_xHSP2long, MIN_xH, MAX_xH, // 0.0, 1.01,
             0.439598249,-0.444673076,0.0)
create_poly3(POLY_xHSP3, MIN_xH, MAX_xH, // 0.0, 1.06,
             1.761487842,-1.876959406,0.0)
create_poly7(POLY_xHRing, MIN_xH, MAX_xH, // 0.7608, 1.089,
             37.744316,-117.731674,143.0759275,-86.2258835,26.7448175,-4.4699705,0.6458455)

////////////////////////////////////////////////////////////////////////////////
/// @begin hbond_compute_energy
///
/// @brief
///js   This function takes a distance and cosines of 2 angles, and returns the
///js   energy a of hydrogen bond, and the derivatives of that energy with
///js   respect to the 3 variables.
///
/// @detailed
/// JSS the parameters r=AHdis, xD=theta, xH=psi (historical names)
/// are actually combined as follows,
/// where Pa is a polynomial defined for param a; these can vary by hbond type
/// and Fa is the fading function defined for param a
/// E = Pr*FxD*FxH + Fr*PxD*FxH + Fr*FxD*PxH;
/// dE/dr = dPr/dr *FxD*FxH + dFr/dr *(PxD*FxH+FxD*PxH)
/// dE/dxD & dE/dxH are similar.
/// For sidechains, the PxD,PxH split into short and long, so
/// PxD = PSxD*FSr + PLxD*FLr; PxH = PSxH*FSr + PLxH*FLr;
/// complicating the derivative w.r.t. r
/// dE/dr = dPr/dr *FxD*FxH + dFr/dr *(PxD*FxH+FxD*PxH) +
///              Fr*(FxH*(PSxD*dFSr/dr+PLxD*dFLr/dr)+FxD*(PSxH*dFSr/dr+PLxH*dFLr/dr))
///
/// This "fading" was Jack Schonbrun's, although he had not identified it
/// as such, and had some minor problems with the derivatives in overlapping fade regions.
/// It may be easier to define the polynomials so that the energy can be
/// E = Pr*PxD*PxH
/// dE/dr = dPr/dr *PxD*PxH; etc.
/// Or to define the polynomials so that they live in wells whose boundaries have
/// positive values that outweigh the negative contributions from the other two terms.
/// Positive energy values are treated as "no bond" when this function returns.
/// This is not done yet, so as to reproduce the current rosetta hb_energy functions.
///
/// @param  hbe - [in] - hbond_evaluation_type
/// @param  AHdis - [in] - acceptor proton distance
/// @param  xD - [in] - -cos(180-theta), where theta is defined by Tanja K.
/// @param  xH - [in] - cos(180-phi), where phi is defined by Tanja K.
/// @param  energy - [out] -
/// @param  dE_dr - [out (optional)] -
/// @param  dE_dxD - [out (optional)] -
/// @param  dE_dxH - [out (optional)] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
hbond_compute_energy(  HBEvalType hbe,
											 float const AHdis, // acceptor proton distance
											 float const xD, // -cos(180-theta), where theta is defined by Tanja K.
											 float const xH, // cos(180-phi), where phi is defined by Tanja K.
											 float & energy,
											 float & dE_dr,
											 float & dE_dxD,
											 float & dE_dxH
											 ) {
	using namespace hbonds;

	energy = MAX_HB_ENERGY + 1.0f;
	dE_dr = dE_dxD = dE_dxH = 0.0;

	if ( std::abs(xD) > 1.0 || std::abs(xH) > 1.0 ) {
		if (runlevel_ns::benchmark || runlevel_ns::runlevel > runlevel_ns::standard)
			std::cout << "WARNING:: invalid angle value in hbond_compute_energy:" <<
				" xH = " << SS( xH ) << " xD = " << SS( xD ) << std::endl;
		return;
	}
	if ( AHdis > MAX_R || AHdis < MIN_R || xH < MIN_xH || xD < MIN_xD ||
			 xH > MAX_xH || xD > MAX_xD ) {
		return;
	}

  double const dAHdis = static_cast<double>(AHdis);
  double const dxD = static_cast<double>(xD);
  double const dxH = static_cast<double>(xH);

	double  FSr,  FLr(0.0),  FxD,  FxH; // values of fading intervals
	double dFSr, dFLr(0.0), dFxD, dFxH; // derivatives of fading intervals
	double  Pr(0.0),  PSxD(0.0),  PSxH(0.0),  PLxD(0.0),  PLxH(0.0); // values of polynomials
	double dPr(0.0), dPSxD(0.0), dPSxH(0.0), dPLxD(0.0), dPLxH(0.0); // derivatives of polynomials

  // fade functions handle the interpolation at edges and for AHdis in [INTERP_MIN,INTERP_MAX]
	fade_xD.value_deriv(xD, FxD, dFxD);
	fade_xH.value_deriv(xH, FxH, dFxH);
	if ( hbe_is_BB_type(hbe) ) { // if we are a BB/BB type
		fade_rBB.value_deriv(AHdis, FSr, dFSr);
		// long range is initialized to zero
	}	else{
		fade_rshort.value_deriv(AHdis, FSr, dFSr);
		fade_rlong.value_deriv(AHdis, FLr, dFLr);
	}

  // here we evaluate the polynomials that pertain to the hbond_evaluation_type hbe
	switch (hbe) {
		case hbe_NONE: break;
		case hbe_BBHELIX:  //hbstat = 1, polys 1,5,11
      POLY_AHdisBBHelix(dAHdis, Pr,dPr);
			POLY_xDBBHelix(dxD, PSxD,dPSxD);
			POLY_xHBBHelix(dxH, PSxH,dPSxH);
      break;

		case hbe_BBOTHER: //hbstat = 2, polys 2,6,11
			POLY_AHdisBBOther(dAHdis, Pr,dPr);
			POLY_xDBBOther(dxD, PSxD,dPSxD);
			POLY_xHBBOther(dxH, PSxH,dPSxH);
			break;

		case hbe_BB:  //hbstat = 3, polys 3,7,13,8,14
		case hbe_BBTURN:
		case hbe_BSC:
		case hbe_SP2B:
		case hbe_SP2SC:
			POLY_AHdisSP2(dAHdis,Pr,dPr);
			POLY_xDSP2short(dxD, PSxD,dPSxD);
			POLY_xHSP2short(dxH, PSxH,dPSxH);
			POLY_xDSP2long(dxD, PLxD,dPLxD);
			POLY_xHSP2long(dxH, PLxH,dPLxH);
      break;

		case hbe_SP3B: //hbstat = 4, polys 4,9,15,10,15
		case hbe_SP3SC:
			POLY_AHdisSP3(dAHdis, Pr,dPr);
			POLY_xDSP3short(dxD, PSxD,dPSxD);
			POLY_xHSP3(dxH, PSxH,dPSxH);
			POLY_xDSP3long(dxD, PLxD,dPLxD);
			PLxH = PSxH; dPLxH = dPSxH;
			break;

		case hbe_RINGB: //hbstat = 5, polys 3,7,16,8,16
		case hbe_RINGSC:
			POLY_AHdisSP2(dAHdis, Pr,dPr);
			POLY_xDSP2short(dxD, PSxD,dPSxD);
			POLY_xHRing(dxH, PSxH,dPSxH);
			POLY_xDSP2long(dxD, PLxD,dPLxD);
      PLxH = PSxH; dPLxH = dPSxH;
			break;
	}

	////////////////////////////
	//	FxD = 1.0; FxH = 1.0; dFxD = 0.0; dFxH = 0.0;

	energy =  Pr*FxD*FxH +  FSr*(PSxD*FxH + FxD*PSxH) +  FLr*(PLxD*FxH + FxD*PLxH);
	if (&dE_dxH == &NODERIV) return;  // NOTE: if any deriv parameter omitted, we don't compute derivatives.

	dE_dr =  dPr*FxD*FxH + dFSr*(PSxD*FxH + FxD*PSxH) + dFLr*(PLxD*FxH + FxD*PLxH);
	dE_dxD = dFxD*(Pr*FxH + FLr*PLxH + FSr*PSxH) + FxH*(FSr*dPSxD + FLr*dPLxD);
	dE_dxH = dFxH*(Pr*FxD + FLr*PLxD + FSr*PSxD) + FxD*(FSr*dPSxH + FLr*dPLxH);
}



////////////////////////////////////////////////////////////////////////////////
/// @begin hb_energy_deriv
///
/// @brief
///car Evaluate the energy and derivative components for a hydrogen bond
///
/// @detailed
///car Energy is assumed to be a sum of independent functions of distance,
///car angle at the donor atom, and angle at the acceptor atom. This form
///car of the hydrogen bond potential was selected by Tanja Kortemme
///car The math for computing the derivative of the angular components of
///car the potential was derived by Bill Wedemeyer.
///
///ora used also to calculate derivatives for RB movements if docking_flag T
///ora important NOTE: if in docking mode minimization is done NOT in TR space, this
///ora       should be specified and the docking_specific part should be skipped
///
/// @param  hbe_type - [in] - hydrogen bond evaluation type from hbonds_ns.h
/// @param  donor_res - [in] -
/// @param  acceptor_res - [in] -
/// @param  Dxyz - [in/out] - donor
/// @param  Hxyz - [in/out] - proton
/// @param  Axyz - [in/out] - acceptor
/// @param  Bxyz - [in/out] - acceptor base
/// @param  B2xyz - [in/out] - 2nd acceptor base for ring acceptors
/// @param  energy - [out] -
/// @param  deriv - [out (optional)] - xyz,f1/f2
/// @param  deriv_type [in (optional)] - deriv is NORMAL(default), DOCK_ACC_DON, DOCK_DON_ACC
///
/// @global_read
///   When docking derivatives are computed (DOCK_ACC_DON or DOCK_DON_ACC), center_of_rotation MUST BE DEFINED!
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

void
hb_energy_deriv(
	hbonds::HBEvalType const hbe_type, // hb evalation type
	FArray1Da_float Hxyz, // proton coords
	FArray1Da_float HDunit, // unit vector toward donor
	FArray1Da_float Axyz, // acceptor coords
	FArray1Da_float BAunit, // unit vector towards base
	float & energy, // returned energy
  HBDerivType const deriv_type, //= NONE // optional derivative type for docking
	FArray2DB_float & deriv //= NODERIV2D // optional returned derivativs: xyz w.r.t. dist, theta, psi
)
{
	using namespace hbonds;

	Hxyz.dimension( 3 );
	HDunit.dimension( 3 );
	Axyz.dimension( 3 );
	BAunit.dimension( 3 );

	//  angle definitions:  JSS the angle names are really bad. A-H-D = xD and B-A-H = xH
	//   cos(180-theta) = cos(thetaD) = xD    angle to donor
	//   cos(180-psi) = cos(thetaH) = xH       angle to proton
	//   raw angle in radians for ring nitrogen improper dihedral

	//    energy  - total energy from this hbond
	//    dE_dr   - deriv w/respect to distance
	//    dE_dxD  - deriv w/respect to cos(thetaD)
	//    dE_dxH  - deriv w/respect to cos(thetaH)

	energy = MAX_HB_ENERGY + 1.0f;
	deriv = 0.0f;

	//Objexx: Local arrays declared static for speed
	//car A->H unit vector, distance
	static FArray1D_float AH( 3 );
	AH(1) = Hxyz(1) - Axyz(1);
	AH(2) = Hxyz(2) - Axyz(2);
	AH(3) = Hxyz(3) - Axyz(3);

	float const AHdis2 = AH(1) * AH(1) + AH(2) * AH(2) + AH(3) * AH(3);

	//CAR AFTER FUNCTION EVALUATION IS WRITTEN, CHECK THAT THIS IS CORRECT
//	if (AHdis2 > 20.0) return; // jss debug
//	if (runlevel_ns::benchmark) // jss debug
//		std::cout<<"hbed1 "<<F(10,4,AHdis2)  <<
////     " t:"<< SS(hbe_type) <<
//      "  H" << F(9,2,Hxyz(1))<<","<<F(9,2,Hxyz(2))<<","<<F(9,2,Hxyz(3))<<
//      "  A"<<F(9,2,Axyz(1))<<","<<F(9,2,Axyz(2))<<","<<F(9,2,Axyz(3))<<
//  std::endl;

  if ( AHdis2 > MAX_R2 ) return;
	if ( AHdis2 < MIN_R2 ) return;
	float const AHdis = std::sqrt(AHdis2);
	float const inv_AHdis = 1.0f / AHdis;
	static FArray1D_float AHunit( 3 );
	AHunit(1) = AH(1) * inv_AHdis;
	AHunit(2) = AH(2) * inv_AHdis;
	AHunit(3) = AH(3) * inv_AHdis;


	//BW cosines of angle at donor, proton
	float const xD =            /* cos(180-theta) = cos(thetaD) */
		AHunit(1) * HDunit(1) + AHunit(2) * HDunit(2) + AHunit(3) * HDunit(3);

	if ( xD < MIN_xD ) return;
	if ( xD > MAX_xD ) return;

	float const xH =            /* cos(180-psi) = cos(thetaH) */
		BAunit(1) * AHunit(1) + BAunit(2) * AHunit(2) + BAunit(3) * AHunit(3);

	if ( xH < MIN_xH ) return;
	if ( xH > MAX_xH ) return;

  if ( deriv_type == hbderiv_NONE ) {
    hbond_compute_energy(hbe_type, AHdis,xD,xH,energy); // NOTE: early return with energy if no derivatives
    return;
  }

	//JSS the rest happens only if we want deriviative information
	float dE_dxH, dE_dxD, dE_dr;
	hbond_compute_energy(hbe_type, AHdis,xD,xH,energy,dE_dr,dE_dxD,dE_dxH);

  if (false&&runlevel_ns::benchmark && energy < 0.0) { // jss debug
		std::cout<<"Deriv "<<I(3,deriv_type)<<" d:"<<F(10,4,AHdis)  <<
    "\n xH:" << F(10,4, xH ) << " xD:" << F(10,4, xD ) << " e:"<< F(14,6,energy) <<
    "dE_dr:"<< F(10,4,dE_dr) <<"dE_dxD:"<< F(10,4,dE_dxD) <<"dE_dxH:" << F(10,4,dE_dxH) <<
    "  H" << F(9,2,Hxyz(1))<<","<<F(9,2,Hxyz(2))<<","<<F(9,2,Hxyz(3))<<" HD"<<F(9,2,HDunit(1))<<","<<F(9,2,HDunit(2))<<","<<F(9,2,HDunit(3))<<
    "  A"<<F(9,2,Axyz(1))<<","<<F(9,2,Axyz(2))<<","<<F(9,2,Axyz(3))<< " BA"<<F(9,2,BAunit(1))<<","<<F(9,2,BAunit(2))<<","<<F(9,2,BAunit(3))<<
    std::endl;
}

  if (energy >= MAX_HB_ENERGY) return;

  if (deriv_type == hbderiv_ABE_GO || deriv_type == hbderiv_ABE_GO_NO_xD || deriv_type == hbderiv_ABE_GO_NO_xH ) {
		//car distance-dependent gradient component.
		//car see also comments in minimize.cc
		//car  dr/dphi = Eab x (V-Vb) . (V' - V)/|V-V'|
		//db  (first cross product is the displacement of V upon a rotation dphi
		//db   around the unit vector Eab, Vb is the coordinates of the second atom
		//db   in the bond)

		//car  dEr/dphi = dEr/dr * (Eab  x (V-Vb) . (V' - V)] / |V-V'|
		//car  dEr/dphi = dEr/dr * (Eab  x (V-Vb) . (V' - V)] / r
		//car  rearrange...
		//car  dEr/dphi = dEr/dr * [Eab X Vb . (V' - V) + Eab . (V' x V)] / r

		//car Eab and Eab X Vb are calulated in dfunc_vdw and dependent on the torison
		//car angle with respect to which the derivative is being taken
		//car f1 and f2 are independent of the torsion angle and here we're precomputing
		//car increments to f1 and f2 for each hbond
		//car f1 = dEr/dr * (V'xV) / r
		//car f2 = dEr/dr * (V'-V) / r
		//car here, V' = H
		//car       V  = A
		//car       dE/dr * 1/r = prefactor

		//car Eab and Eab X Vb are calulated in dfunc_vdw, here
    assert ( minimize_ns::func_switch::switch_value != 4 ); //chu's test -- make sure we aren't in docking mode


		//car V'x V
		static FArray1D_float HxA( 3 );
		HxA(1) = Hxyz(2) * Axyz(3) - Hxyz(3) * Axyz(2);
		HxA(2) = Hxyz(3) * Axyz(1) - Hxyz(1) * Axyz(3);
		HxA(3) = Hxyz(1) * Axyz(2) - Hxyz(2) * Axyz(1);

		// in/decrements to the f1 and f2 of the angle moving donor/acceptor
		float prefactor = inv_AHdis * dE_dr;
		deriv[0] = prefactor * HxA(1); // deriv(1,1) // f1
		deriv[1] = prefactor * HxA(2); // deriv(2,1)
		deriv[2] = prefactor * HxA(3); // deriv(3,1)

		deriv[3] = prefactor * AH(1);  // deriv(1,2) // f2
		deriv[4] = prefactor * AH(2);  // deriv(2,2)
		deriv[5] = prefactor * AH(3);  // deriv(3,2)


		if (deriv_type != hbderiv_ABE_GO_NO_xD ){
			//car gradient component for xD (theta)
			//car (see comments below for xH gradient)
			static FArray1D_float BD( 3 );
			prefactor = inv_AHdis * dE_dxD;
			BD(1) = prefactor * ( HDunit(1) - xD * AHunit(1) );
			BD(2) = prefactor * ( HDunit(2) - xD * AHunit(2) );
			BD(3) = prefactor * ( HDunit(3) - xD * AHunit(3) );

			static FArray1D_float BDxA( 3 );
			BDxA(1) = BD(2) * Axyz(3) - BD(3) * Axyz(2);
			BDxA(2) = BD(3) * Axyz(1) - BD(1) * Axyz(3);
			BDxA(3) = BD(1) * Axyz(2) - BD(2) * Axyz(1);


			//BW in/decrements to the f1 and f2 of the angle moving donor/acceptor
			deriv[0] += BDxA(1); // deriv(1,1) // f1,theta
			deriv[1] += BDxA(2); // deriv(2,1)
			deriv[2] += BDxA(3); // deriv(3,1)

			deriv[3] += BD(1);   // deriv(1,2) // f2,theta
			deriv[4] += BD(2);   // deriv(2,2)
			deriv[5] += BD(3);   // deriv(3,2)
		}

		if (deriv_type != hbderiv_ABE_GO_NO_xH ){
			//BW gradient component for xH (psi)
			//car   (from code and paper by Bill Wedemeyer)
			//car  xH = (BAunit . AH)/AHdis
			//car  dxH/dphi = 1/AHdis * (BAunit . dAH/dphi - xH *dAHdis/dphi)
			//car    (note dBAunit/dphi = 0)
			//car
			//car  dAHdis/dphi = AHunit . dAH/dphi
			//car
			//car  substituting and rearranging....
			//car  dxH/dphi = 1/AHdis * dAH/dphi . (BAunit - xH*AHunit)
			//car           = 1/AHdis * dAH/dphi . BH
			//car
			//car note: BH = (BAunit - xH*AHunit) = component of BAunit that is
			//car       perpendicular to AHunit
			//car
			//car dAH/dphi = Eab x (V-Vb)  . (V' - V)/|V-V'|   from above
			//car dAH/dphi = Eab x (H-Vb)  . AHunit
			//car
			//car dExH/dphi = dExH/dxH * dxH/dphi
			//car           = dExH/dxH * 1/AHdis * dAH/dphi . BH
			//car           = dExH/dxH * 1/AHdis * (Eab x (H-Vb) .(H-A)/AHdis) . BH
			//car
			//car rearrange as with dEr/dr above to get f1 and f2 component

			//car f1 = dE/dxH *(1/AHdis) * BH x H
			//car f2 = dE/dxH *(1/AHdis) * BH

			static FArray1D_float BH( 3 );
			prefactor = inv_AHdis * dE_dxH;
			BH(1) = prefactor * ( BAunit(1) - xH * AHunit(1) );
			BH(2) = prefactor * ( BAunit(2) - xH * AHunit(2) );
			BH(3) = prefactor * ( BAunit(3) - xH * AHunit(3) );

			static FArray1D_float BHxH( 3 );
			BHxH(1) = BH(2) * Hxyz(3) - BH(3) * Hxyz(2);
			BHxH(2) = BH(3) * Hxyz(1) - BH(1) * Hxyz(3);
			BHxH(3) = BH(1) * Hxyz(2) - BH(2) * Hxyz(1);

			//BW in/decrements to the f1 and f2 of the angle moving donor/acceptor
			deriv[0] += BHxH(1); // deriv(1,1)
			deriv[1] += BHxH(2); // deriv(2,1)
			deriv[2] += BHxH(3); // deriv(3,1)

			deriv[3] += BH(1);   // deriv(1,2)
			deriv[4] += BH(2);   // deriv(2,2)
			deriv[5] += BH(3);   // deriv(3,2)
		}

	} else { // deriv_type != NORMAL, so we are in a docking mode
    using namespace docking;
    using namespace docking_min;
		//ora
		//calculates derivatives of hb relative to docking degrees of freedom
		// stored in dE_hb_dTR, that is added to dE_dTR in dfunc_dvdw
		//
		// note that dr_dx in this case is in direction A->H, and dE_dr is relative to this direction
    assert ( minimize_ns::func_switch::switch_value == 4 ); //chu's test -- make sure we are in docking mode

		//local
    static FArray1D_float tmp( 3 );
    float prefactor;
    static FArray1D_float BD( 3 );
    static FArray1D_float BH( 3 );
    static FArray1D_float dr_dx( 3 );
		// derivative of atom-atom distance along each coord. axis
    static FArray1D_float q( 3 ); // position of j-atom relative to center of rotation
    static FArray1D_float dE_dTR_hb( 6, 0.0 );
		// array in format of dE_dTR: its contents are copied to deriv

		//     calculate the distance from the pivot point (same for all 3)
    if ( deriv_type == hbderiv_DOCK_ACC_DON ) {      // second molecule has donor
      q(1) = Hxyz(1) - center_of_rotation(1);
      q(2) = Hxyz(2) - center_of_rotation(2);
      q(3) = Hxyz(3) - center_of_rotation(3);
      dr_dx(1) = AHunit(1);
      dr_dx(2) = AHunit(2);
      dr_dx(3) = AHunit(3);
      prefactor = inv_AHdis;  // for angle dependent part
    } else {                                // second molecule has acceptor
      assert(deriv_type == hbderiv_DOCK_DON_ACC);
      q(1) = Axyz(1) - center_of_rotation(1);
      q(2) = Axyz(2) - center_of_rotation(2);
      q(3) = Axyz(3) - center_of_rotation(3);
      dr_dx(1) = -AHunit(1);
      dr_dx(2) = -AHunit(2);
      dr_dx(3) = -AHunit(3);
      prefactor = -inv_AHdis;
    }
		// 1) dE_dr:
    accumulate_dE_dTR(dE_dTR_hb,dE_dr,dr_dx,q);

		// angle dependent part:
		// 2) dE_dxD:
		// dxD/dTR = d AH(vector) /dTR   . BD
		//
		// BD = 1/AHdis * [HDunit - xD*AHunit]
		//
		// so  dxD/dTR(1)  = dxD/dx = d AH(vector) /dx .BD  = BDx
		// and dxD/dTR(4)  = dxD/dthetax = d AH(vector) /dthetax .BD  = (-qz*BDy + qy*BDz)
		//     so dxD/dTR(4-6) = [q x BD]

    BD(1) = HDunit(1) - xD*AHunit(1);
    BD(2) = HDunit(2) - xD*AHunit(2);
    BD(3) = HDunit(3) - xD*AHunit(3);

    dr_dx(1) = BD(1) * prefactor;
    dr_dx(2) = BD(2) * prefactor;
    dr_dx(3) = BD(3) * prefactor;

    accumulate_dE_dTR(dE_dTR_hb,dE_dxD,dr_dx,q);

		// 3) dE_dxH: same as for dE_dxD
		// dxH/dTR = d AH(vector) /dTR   . BH
		//
		// BH = 1/AHdis * [BAunit - xH*AHunit]
		//
		// so  dxH/dTR(1)  = dxH/dx = d AH(vector) /dx .BH  = BHx
		//     so dxH/dTR(4-6) = [q x BH]
		//
    BH(1) = BAunit(1) - xH*AHunit(1);
    BH(2) = BAunit(2) - xH*AHunit(2);
    BH(3) = BAunit(3) - xH*AHunit(3);

    dr_dx(1) = BH(1) * prefactor;
    dr_dx(2) = BH(2) * prefactor;
    dr_dx(3) = BH(3) * prefactor;

    accumulate_dE_dTR(dE_dTR_hb,dE_dxH,dr_dx,q);

		// additional terms:
		// if donor is in partner 1, for dxH/dTR(4-6)
		//    dBAunit/dTR(4-6) is NOT = 0,
		//    and the term HAunit . dABunit/dTR(4-6) does NOT cancel out.
		//    HAunit . dABunit/dTR(4-6) = ABunit x HAunit = BAunit x AHunit
		//
		// if donor is in partner 2, for dxD/dTR(4-6)
		//    dHDunit/dTR(4-6) is NOT = 0,
		//    and the term AHunit . dHDunit/dTR(4-6) does NOT cancel out.
		//    AHunit . dHDunit/dTR(4-6) = HDunit x AHunit
		//

    if ( deriv_type == hbonds::hbderiv_DOCK_DON_ACC ) { // donor in partner 1
      tmp(1) = AHunit(3)*BAunit(2) - AHunit(2)*BAunit(3);
      tmp(2) = AHunit(1)*BAunit(3) - AHunit(3)*BAunit(1);
      tmp(3) = AHunit(2)*BAunit(1) - AHunit(1)*BAunit(2);

      dE_dTR_hb(4) += dE_dxH*tmp(1);
      dE_dTR_hb(5) += dE_dxH*tmp(2);
      dE_dTR_hb(6) += dE_dxH*tmp(3);
    } else {                           // donor in partner 2
      tmp(1) = AHunit(3)*HDunit(2) - AHunit(2)*HDunit(3);
      tmp(2) = AHunit(1)*HDunit(3) - AHunit(3)*HDunit(1);
      tmp(3) = AHunit(2)*HDunit(1) - AHunit(1)*HDunit(2);

      dE_dTR_hb(4) += dE_dxD*tmp(1);
      dE_dTR_hb(5) += dE_dxD*tmp(2);
      dE_dTR_hb(6) += dE_dxD*tmp(3);
    }

		// finally update deriv
    deriv(1,1) = dE_dTR_hb(1); deriv(2,1) = dE_dTR_hb(2); deriv(3,1) = dE_dTR_hb(3);
    deriv(1,2) = dE_dTR_hb(4); deriv(2,2) = dE_dTR_hb(5); deriv(3,2) = dE_dTR_hb(6);
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin hb_energy_deriv
///
/// @remarks See comments on helper function above.
///
////////////////////////////////////////////////////////////////////////////////

void
hb_energy_deriv(
  HBEvalType const hbe_type, // hbond evaluation type -- determines what scoring function to use
  FArray1Da_float Dxyz, // donor coords
  FArray1Da_float Hxyz, // proton
  FArray1Da_float Axyz, // acceptor
  FArray1Da_float Bxyz, // acceptor base
  FArray1Da_float B2xyz, // 2nd acceptor base for ring & SP3 acceptors
  float & energy,
  HBDerivType const deriv_type, //= NONE // optional derivative type for docking
  FArray2DB_float & deriv // = NODERIV2D// xyz,f1/f2
)
{
	using namespace hbonds;

	Dxyz.dimension( 3 ); //JSS can I omit these?
	Hxyz.dimension( 3 );
	Axyz.dimension( 3 );
	Bxyz.dimension( 3 );
	B2xyz.dimension( 3 );

//  angle definitions:  JSS the angle names are really bad. A-H-D = xD and B-A-H = xH
//   cos(180-theta) = cos(thetaD) = xD    angle to donor
//   cos(180-psi) = cos(thetaH) = xH       angle to proton
//   raw angle in radians for ring nitrogen improper dihedral

//    energy  - total energy from this hbond
//    dE_dr   - deriv w/respect to distance
//    dE_dxD  - deriv w/respect to cos(thetaD)
//    dE_dxH  - deriv w/respect to cos(thetaH)

//Objexx: Local arrays declared static for speed
//JSS all early exits are in helper above, so this version of the function is deprecated.
//These unit vectors are invariant for hbonded pairs and can be precalculated.
//car  H->D unit vector, dis2
	static FArray1D_float HDunit( 3 );
	HDunit(1) = Dxyz(1) - Hxyz(1);
	HDunit(2) = Dxyz(2) - Hxyz(2);
	HDunit(3) = Dxyz(3) - Hxyz(3);
	float const HDdis2 =
    HDunit(1) * HDunit(1) + HDunit(2) * HDunit(2) + HDunit(3) * HDunit(3);

  if ( HDdis2 < 0.64 || HDdis2 > 1.5625 ) { // .8 to 1.25A
    if (runlevel_ns::benchmark || runlevel_ns::runlevel > runlevel_ns::standard)
      std::cout << "Warning: hb_energy_deriv has H(" << Hxyz(1)<<","<<Hxyz(2)<<","<<Hxyz(3)<<") D(" << Dxyz(1)<<","<<Dxyz(2)<<","<<Dxyz(3)<<")  distance out of range " << std::sqrt(HDdis2)<<std::endl;
    energy = 0.0;
    deriv = 0.0;
		return;
  }

	float const inv_HDdis = 1.0f / std::sqrt( HDdis2 );
	HDunit(1) *= inv_HDdis;
	HDunit(2) *= inv_HDdis;
	HDunit(3) *= inv_HDdis;

//car  B->A unit vector
	static FArray1D_float BAunit( 3 );
	make_hbBasetoAcc_unitvector(hbe_type, Axyz, Bxyz, B2xyz, BAunit);
  hb_energy_deriv(hbe_type, Hxyz, HDunit, Axyz, BAunit, energy, deriv_type, deriv); // do the work
}



void
make_hbBasetoAcc_unitvector( // construct for a hydrogen bond Acceptor, the unit vector from Base
						HBEvalType const hbe_type, // the BA unit vector depends on the hb evaluation type
						FArray1Da_float const Axyz,
						FArray1Da_float const Bxyz,
						FArray1Da_float const B2xyz,
						FArray1Da_float BAunit)
{
  switch (hbe_type) {
    case hbe_RINGSC:
    case hbe_RINGB: // RING uses midpoint between base and base2
  		BAunit(1) = Axyz(1) - 0.5f * ( Bxyz(1) + B2xyz(1) );
      BAunit(2) = Axyz(2) - 0.5f * ( Bxyz(2) + B2xyz(2) );
      BAunit(3) = Axyz(3) - 0.5f * ( Bxyz(3) + B2xyz(3) );
      break;
    case hbe_SP3SC:
    case hbe_SP3B: // SP3 uses base2
      BAunit(1) = Axyz(1) - B2xyz(1);
      BAunit(2) = Axyz(2) - B2xyz(2);
      BAunit(3) = Axyz(3) - B2xyz(3);
      break;
    default: // SP2 uses base
      BAunit(1) = Axyz(1) - Bxyz(1);
      BAunit(2) = Axyz(2) - Bxyz(2);
      BAunit(3) = Axyz(3) - Bxyz(3);
  }
  float const BAdis = BAunit(1) * BAunit(1) + BAunit(2) * BAunit(2) + BAunit(3) * BAunit(3);
  //assert(BAdis > 1e-3); //JSS base and base2 shouldn't be the same atoms
  if (BAdis < 1e-2) //JSS base and base2 shouldn't be the same atoms
		if (runlevel_ns::benchmark || runlevel_ns::runlevel > runlevel_ns::standard)
			std::cout << "WARNING: hb_energy_deriv() has base and acceptor atoms at distance "<<BAdis<<std::endl;
	float const inv_BAdis = 1.0f / std::sqrt(BAdis);
	BAunit(1) *= inv_BAdis;
	BAunit(2) *= inv_BAdis;
	BAunit(3) *= inv_BAdis;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin hbond_polynomial
///
/// @brief evaluate a polynomial and its derivative.
///
/// @detailed
///
/// @param  variable - [in] - evaluate polynomial(value)
/// @param  table - [in] - table of parameters -- JSS backwards: low to high
/// @param  value - [out] - returned output
/// @param  deriv - [out] - returned output
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// No longer used as of changeset 11880 or so; I leave it in as an example of
/// how to use horner's rule to compute a polynomial and its derivative together.
/// This is the template for the create_poly# macros in hbonds_ns.h
///
/// @references
///
/// @authors
///
/// @last_modified Jack Snoeyink
/////////////////////////////////////////////////////////////////////////////////
//void
//hbond_polynomial(
//								 double const variable,
//								 int const table, // table of parameters
//								 double & value, // returned output
//								 double & deriv // returned output
//								 )
//{
//  using namespace hbonds;
//
//  double const internal_weight = { .5 };
//
//  // Polynomial value and derivative using Horner's rule
//  // value = Sum_(i = 1,...,N) [ coeff_i * variable^(i-1) ]
//  // deriv = Sum_(i = 2,...,N) [ ( i - 1 ) * coeff_i * variable^(i-2) ]
//  // JSS: Horner's rule for evaluating polynomials is based on rewriting the polynomial as:
//  // JSS: p(x)  = a0 + x*(a1 + x*(a2 + x*(...  x*(aN)...)))
//  // JSS: or value_k = a_k + x*value_k+1 for k = N-1 to 0
//  // JSS: and the derivative is
//  // JSS: deriv_k = value_k+1 + deriv_k+1 for k = N-1 to 1
//  int N = poly_degree(table); // offset for arrays
//  assert( N > 0 );
//  int l = coeff.index( N, table );
//  value = coeff[ l ];
//  deriv = 0.0;
//  while ( --N > 0 ) {
//    ( deriv *= variable ) += value;
//    ( value *= variable ) += coeff[ --l ];
//  }
//
//  value -= offset(table);  // JSS: normalization should be applied to the original polynomial.
//  double const normalize = internal_weight / scale(table);
//  value *= normalize;
//  deriv *= normalize;
//}
