// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 11796 $
//  $Date: 2006-12-14 17:40:12 -0800 (Thu, 14 Dec 2006) $
//  $Author: snoeyink $

#ifndef INCLUDED_hbonds_geom
#define INCLUDED_hbonds_geom
#include "aaproperties_pack.h"
#include "hbonds_ns.h"

using namespace hbonds;

// class declarations

/////////////////////////////
/// class FadeInterval
/// stores an "fading interval" [a b c d] with a <= b <= c <= d
/// and computes the fraction of containment for x, which is defined to be
/// 0 if x is outside of (a,d), 1 if x is inside of [b,c],
/// and a continuous linear ramp otherwise.
/// i.e. (x-a)/(b-a) for x in (a,b), and (d-x)/(d-c) for x in (c,d)
/// This is used to ensure that hbond scoring as a sum Er + ExH + ExD goes to zero at the edges.
////////////////////////////
class FadeInterval {
private:
	float const min0_;
	float const fmin_;
	float const fmax_;
	float const max0_;
	double const dfade_min_;
	double const dfade_max_;

public:
	/// Constructor

	FadeInterval(float const min0,
							 float const fmin,
							 float const fmax,
							 float const max0) : min0_(min0), fmin_(fmin), fmax_(fmax), max0_(max0),
																	 dfade_min_(1.0/static_cast<double>(fmin-min0)),
																	 dfade_max_(1.0/static_cast<double>(max0-fmax)) {
		assert(min0 <= fmin && fmin <= fmax && fmax <= max0); // make sure dfades are not negative
		// if they are 1.0/0.0, they won't be used...
	};

	double value(float const x) const {  //JSS  5 intervals --a-b---c-d--
		if (x < fmax_) {
			if (x >= fmin_) return 1.0;      // in [fmin, fmax]
			if (x <= min0_) return 0.0;      // <= min0
			return (x - min0_) * dfade_min_; // in (min0,fmin)
		} else {
			if (x >= max0_) return 0.0;      // >= max0
			return (max0_ - x) * dfade_max_; // in (fmax,max0)
		}
	}

	void value_deriv(float const x, double &val, double &deriv) const {  //JSS  5 intervals --a-b---c-d--
		if (x < fmax_) {
			if (x >= fmin_) { val = 1.0; deriv = 0.0; return; }  // in [fmin, fmax]
			if (x <= min0_) { val = deriv = 0.0; return; }       // <= min0
			deriv = dfade_min_; val = (x - min0_) * dfade_min_;  // in (min0,fmin)
		} else {
			if (x >= max0_) { val = deriv = 0.0; return; }       // >= max0
			deriv = -dfade_max_; val = (max0_ - x) * dfade_max_; // in (fmax,max0)
		}
	}
};


hbonds::HBEvalType
hbond_evaluation_type_AH(  // determine evalType from acceptor & hydrogen -- temporary for rotamer trie
											int const don_atomtype,
											int const acc_atomtype,
											int const don_res = 0,
											int const acc_res = 999
											);

hbonds::HBEvalType
hbond_evaluation_type(
											int const don_atomtype,
											int const acc_atomtype,
											int const don_res = 0,
											int const acc_res = 999
											);

inline hbonds::HBEvalType
hbond_evaluation_type(
											int const don_atm,
											int const don_aa,
											int const don_aav,
											int const acc_atm,
											int const acc_aa,
											int const acc_aav,
											int const don_res = 0,
											int const acc_res = 999
											)
{
	using namespace aaproperties_pack;

	int const don_type = fullatom_type(don_atm, don_aa, don_aav);
	int const acc_type = fullatom_type(acc_atm, acc_aa, acc_aav);
	return hbond_evaluation_type(don_type, acc_type, don_res, acc_res);
}

HBEvalType
hbond_evaluation_type_HOHdonor(int const acc_atomtype);

HBEvalType
hbond_evaluation_type_HOHacceptor(int const don_atomtype);

inline HBEvalType
hbond_evaluation_type_HOHdonor(int const acc_atm, int const acc_aa, int const acc_aav = 1) {
	using namespace aaproperties_pack;
	
	return hbond_evaluation_type_HOHdonor(fullatom_type(acc_atm, acc_aa, acc_aav));
}

inline HBEvalType
hbond_evaluation_type_HOHacceptor(int const don_atm, int const don_aa, int const don_aav = 1) {
	using namespace aaproperties_pack;
	
	return hbond_evaluation_type_HOHacceptor(don_atm, don_aa, don_aav);
}

hbonds::HBEvalType
hbe_classify_BB_by_separation(int const don_res, int const acc_res);


void
hbond_compute_energy(  HBEvalType hbe, // used internally & by geometric solvation
  float const AHdis, // acceptor proton distance
  float const xD, // -cos(180-theta), where theta is defined by Tanja K.
  float const xH, // cos(180-phi), where phi is defined by Tanja K.
  float & energy,
  float & dE_dr = hbonds::NODERIV,
  float & dE_dxD = hbonds::NODERIV,
  float & dE_dxH = hbonds::NODERIV
  );

void
hb_energy_deriv(
  HBEvalType const hbe_type, // hbond evaluation type -- determines what scoring function to use
  FArray1Da_float Hxyz, // proton
  FArray1Da_float HDunit, // donor
  FArray1Da_float Axyz, // acceptor
  FArray1Da_float BAunit, // unit vector towards base
  float & energy,
  HBDerivType const deriv_type = hbderiv_NONE, // optional derivative type NONE/ABE_GO/or docking
  FArray2DB_float & deriv = NODERIV2D // xyz,f1/f2
  );

void
hb_energy_deriv(
  HBEvalType const hbe_type, // hbond evaluation type -- determines what scoring function to use
  FArray1Da_float Dxyz, // donor coords
  FArray1Da_float Hxyz, // proton
  FArray1Da_float Axyz, // acceptor
  FArray1Da_float Bxyz, // acceptor base
  FArray1Da_float B2xyz, // 2nd acceptor base for ring & SP3 acceptors
  float & energy,
  HBDerivType const deriv_type = hbderiv_NONE, // optional derivative type NONE/ABE_GO/or docking
  FArray2DB_float & deriv = NODERIV2D // xyz,f1/f2
);


void
make_hbBasetoAcc_unitvector( // construct for a hydrogen bond Acceptor, the unit vector from Base
						HBEvalType const hbe_type, // the BA unit vector depends on the hb evaluation type
						FArray1Da_float const Axyz,
						FArray1Da_float const Bxyz,
						FArray1Da_float const B2xyz,
						FArray1Da_float BAunit);


#endif
